/*
 * This is an example of a server that closes when hitting the /close route.
 */

// Importing some console colors
import { bold, cyan, green, yellow } from "jsr:@std/fmt@0.223/colors";

import { Application, type Context, Router, Status } from "../mod.ts";

function notFound(context: Context) {
  context.response.status = Status.NotFound;
  context.response.body =
    `<html><body><h1>404 - Not Found</h1><p>Path <code>${context.request.url}</code> not found.`;
}

const controller = new AbortController();

const router = new Router();
router
  .get("/", (context) => {
    context.response.body = "Hello world!";
  })
  .get("/close", (context) => {
    context.response.body = "Bye!";
    // This will cause the application to stop listening and stop processing
    // requests
    controller.abort();
  });

const app = new Application();

// Logger
app.use(async (context, next) => {
  await next();
  const rt = context.response.headers.get("X-Response-Time");
  console.log(
    `${green(context.request.method)} ${cyan(context.request.url.pathname)} - ${
      bold(
        String(rt),
      )
    }`,
  );
});

// Response Time
app.use(async (context, next) => {
  const start = Date.now();
  await next();
  const ms = Date.now() - start;
  context.response.headers.set("X-Response-Time", `${ms}ms`);
});

// Use the router
app.use(router.routes());
app.use(router.allowedMethods());

// A basic 404 page
app.use(notFound);

app.addEventListener("listen", ({ hostname, port, serverType }) => {
  console.log(
    bold("Start listening on ") + yellow(`${hostname}:${port}`),
  );
  console.log(bold("  using HTTP server: " + yellow(serverType)));
});

// Utilise the signal from the controller
const { signal } = controller;
await app.listen({ hostname: "127.0.0.1", port: 8000, signal });
console.log(bold("Finished."));
