---
title: Meta Tags
description: 'Learn how to migrate from Nuxt 2 to Nuxt Bridge new meta tags.'
---

If you need to access the component state with `head`, you should migrate to using [`useHead`](/docs/api/composables/use-head) .

If you need to use the Options API, there is a `head()` method you can use when you use `defineNuxtComponent`.

## Migration

### Set `bridge.meta`

```js
import { defineNuxtConfig } from '@nuxt/bridge'
export default defineNuxtConfig({
  bridge: {
    meta: true,
    nitro: false // If migration to Nitro is complete, set to true
  }
})
```

### Update head properties

In your `nuxt.config`, rename `head` to `meta`. (Note that objects no longer have a `hid` key for deduplication.)

::code-group

```ts [Nuxt 2]
export default {
  head: {
    titleTemplate: '%s - Nuxt',
    meta: [
      { charset: 'utf-8' },
      { name: 'viewport', content: 'width=device-width, initial-scale=1' },
      { hid: 'description', name: 'description', content: 'Meta description' }
    ]
  }
}
```

```ts [Nuxt 3]
export default defineNuxtConfig({
  app: {
    head: {
      titleTemplate: '%s - Nuxt',
      meta: [
        { charset: 'utf-8' },
        { name: 'viewport', content: 'width=device-width, initial-scale=1' },
        { name: 'description', content: 'Meta description' }
      ]
    }
  }
})
```

::

## `useHead` Composables

Nuxt Bridge provides a new Nuxt 3 meta API that can be accessed with a new [`useHead`](/docs/api/composables/use-head) composable.

```vue
<script setup lang="ts">
useHead({
  title: 'My Nuxt App',
})
</script>
```

::tip
This [`useHead`](/docs/api/composables/use-head) composable uses `@unhead/vue` under the hood (rather than `vue-meta`) to manipulate your `<head>`.
::

::warning
We recommend not using the native Nuxt 2 `head()` properties in addition to [`useHead`](/docs/api/composables/use-head) , as they may conflict.
::

For more information on how to use this composable, see [the docs](/docs/getting-started/seo-meta).

## Options API

```vue
<script>
// if using options API `head` method you must use `defineNuxtComponent`
export default defineNuxtComponent({
  head (nuxtApp) {
    // `head` receives the nuxt app but cannot access the component instance
    return {
      meta: [{
        name: 'description',
        content: 'This is my page description.'
      }]
    }
  }
})
</script>
```

::warning
Possible breaking change: `head` receives the nuxt app but cannot access the component instance. If the code in your `head` tries to access the data object through `this` or `this.$data`, you will need to migrate to the `useHead` composable.
::

## Title Template

If you want to use a function (for full control), then this cannot be set in your nuxt.config, and it is recommended instead to set it within your `/layouts` directory.

```vue [layouts/default.vue]
<script setup lang="ts">
useHead({
  titleTemplate: (titleChunk) => {
    return titleChunk ? `${titleChunk} - Site Title` : 'Site Title';
  }
})
</script>
```
