---
title: Plugins
description: Nuxt Kit provides a set of utilities to help you create and use plugins. You can add plugins or plugin templates to your module using these functions.
links:
  - label: Source
    icon: i-simple-icons-github
    to: https://github.com/nuxt/nuxt/blob/main/packages/kit/src/plugin.ts
    size: xs
---

Plugins are self-contained code that usually add app-level functionality to Vue. In Nuxt, plugins are automatically imported from the `plugins` directory. However, if you need to ship a plugin with your module, Nuxt Kit provides the `addPlugin` and `addPluginTemplate` methods. These utils allow you to customize the plugin configuration to better suit your needs.

## `addPlugin`

Registers a Nuxt plugin and to the plugins array.

::tip{icon="i-ph-video" to="https://vueschool.io/lessons/injecting-plugins?friend=nuxt" target="_blank"}
Watch Vue School video about addPlugin.
::

### Type

```ts
function addPlugin (plugin: NuxtPlugin | string, options: AddPluginOptions): NuxtPlugin

interface NuxtPlugin {
  src: string
  mode?: 'all' | 'server' | 'client'
  order?: number
}

interface AddPluginOptions { append?: boolean }
```

### Parameters

#### `plugin`

**Type**: `NuxtPlugin | string`

**Required**: `true`

A plugin object or a string with the path to the plugin. If a string is provided, it will be converted to a plugin object with `src` set to the string value. If a plugin object is provided, it must have the following properties:

- `src` (required)

  **Type**: `string`

  Path to the plugin.

- `mode` (optional)

  **Type**: `'all' | 'server' | 'client'`

  **Default**: `'all'`

  If set to `'all'`, the plugin will be included in both client and server bundles. If set to `'server'`, the plugin will only be included in the server bundle. If set to `'client'`, the plugin will only be included in the client bundle. You can also use `.client` and `.server` modifiers when specifying `src` option to use plugin only in client or server side.

- `order` (optional)

  **Type**: `number`

  **Default**: `0`

  Order of the plugin. This allows more granular control over plugin order and should only be used by advanced users. Lower numbers run first, and user plugins default to `0`. It's recommended to set `order` to a number between `-20` for `pre`-plugins (plugins that run before Nuxt plugins) and `20` for `post`-plugins (plugins that run after Nuxt plugins).

::warning
Don't use `order` unless you know what you're doing. For most plugins, the default `order` of `0` is sufficient. To append a plugin to the end of the plugins array, use the `append` option instead.
::

#### `options`

**Type**: `AddPluginOptions`

**Default**: `{}`

Options to pass to the plugin. If `append` is set to `true`, the plugin will be appended to the plugins array instead of prepended.

### Examples

::code-group

```ts [module.ts]
import { createResolver, defineNuxtModule, addPlugin } from '@nuxt/kit'

export default defineNuxtModule({
  setup() {
    const resolver = createResolver(import.meta.url)

    addPlugin({
      src: resolver.resolve('runtime/plugin.js'),
      mode: 'client'
    })
  }
})
```

```ts [runtime/plugin.js]
// https://github.com/nuxt/nuxters
export default defineNuxtPlugin((nuxtApp) => {
  const colorMode = useColorMode()

  nuxtApp.hook('app:mounted', () => {
    if (colorMode.preference !== 'dark') {
      colorMode.preference = 'dark'
    }
  })
})
```

::

## `addPluginTemplate`

Adds a template and registers as a nuxt plugin. This is useful for plugins that need to generate code at build time.

::tip{icon="i-ph-video" to="https://vueschool.io/lessons/injecting-plugin-templates?friend=nuxt" target="_blank"}
Watch Vue School video about addPluginTemplate.
::

### Type

```ts
function addPluginTemplate (pluginOptions: NuxtPluginTemplate, options: AddPluginOptions): NuxtPlugin

interface NuxtPluginTemplate<Options = Record<string, any>> {
  src?: string,
  filename?: string,
  dst?: string,
  mode?: 'all' | 'server' | 'client',
  options?: Options,
  getContents?: (data: Options) => string | Promise<string>,
  write?: boolean,
  order?: number
}

interface AddPluginOptions { append?: boolean }

interface NuxtPlugin {
  src: string
  mode?: 'all' | 'server' | 'client'
  order?: number
}
```

### Parameters

#### `pluginOptions`

**Type**: `NuxtPluginTemplate`

**Required**: `true`

A plugin template object with the following properties:

- `src` (optional)

  **Type**: `string`

  Path to the template. If `src` is not provided, `getContents` must be provided instead.

- `filename` (optional)

  **Type**: `string`

  Filename of the template. If `filename` is not provided, it will be generated from the `src` path. In this case, the `src` option is required.

- `dst` (optional)

  **Type**: `string`

  Path to the destination file. If `dst` is not provided, it will be generated from the `filename` path and nuxt `buildDir` option.

- `mode` (optional)

  **Type**: `'all' | 'server' | 'client'`

  **Default**: `'all'`

  If set to `'all'`, the plugin will be included in both client and server bundles. If set to `'server'`, the plugin will only be included in the server bundle. If set to `'client'`, the plugin will only be included in the client bundle. You can also use `.client` and `.server` modifiers when specifying `src` option to use plugin only in client or server side.

- `options` (optional)

  **Type**: `Options`

  Options to pass to the template.

- `getContents` (optional)

  **Type**: `(data: Options) => string | Promise<string>`

  A function that will be called with the `options` object. It should return a string or a promise that resolves to a string. If `src` is provided, this function will be ignored.

- `write` (optional)

    **Type**: `boolean`

    If set to `true`, the template will be written to the destination file. Otherwise, the template will be used only in virtual filesystem.

- `order` (optional)

  **Type**: `number`

  **Default**: `0`

  Order of the plugin. This allows more granular control over plugin order and should only be used by advanced users. Lower numbers run first, and user plugins default to `0`. It's recommended to set `order` to a number between `-20` for `pre`-plugins (plugins that run before Nuxt plugins) and `20` for `post`-plugins (plugins that run after Nuxt plugins).

::warning
Don't use `order` unless you know what you're doing. For most plugins, the default `order` of `0` is sufficient. To append a plugin to the end of the plugins array, use the `append` option instead.
::

#### `options`

**Type**: `AddPluginOptions`

**Default**: `{}`

Options to pass to the plugin. If `append` is set to `true`, the plugin will be appended to the plugins array instead of prepended.

### Examples

::code-group

```ts [module.ts]
// https://github.com/vuejs/vuefire
import { createResolver, defineNuxtModule, addPluginTemplate } from '@nuxt/kit'

export default defineNuxtModule({
  setup() {
    const resolver = createResolver(import.meta.url)

    addPluginTemplate({
      src: resolve(templatesDir, 'plugin.ejs'),
      filename: 'plugin.mjs',
      options: {
        ...options,
        ssr: nuxt.options.ssr,
      },
    })
  }
})
```

```ts [runtime/plugin.ejs]
import { VueFire, useSSRInitialState } from 'vuefire'
import { defineNuxtPlugin } from '#imports'

export default defineNuxtPlugin((nuxtApp) => {
  const firebaseApp = nuxtApp.$firebaseApp

  nuxtApp.vueApp.use(VueFire, { firebaseApp })

  <% if(options.ssr) { %>
  if (import.meta.server) {
    nuxtApp.payload.vuefire = useSSRInitialState(undefined, firebaseApp)
  } else if (nuxtApp.payload?.vuefire) {
    useSSRInitialState(nuxtApp.payload.vuefire, firebaseApp)
  }
  <% } %>
})
```

::
