---
title: "Context"
description: Nuxt Kit provides a set of utilities to help you work with context.
links:
  - label: Source
    icon: i-simple-icons-github
    to: https://github.com/nuxt/nuxt/blob/main/packages/kit/src/context.ts
    size: xs
---

Nuxt modules allow you to enhance Nuxt's capabilities. They offer a structured way to keep your code organized and modular. If you're looking to break down your module into smaller components, Nuxt offers the `useNuxt` and `tryUseNuxt` functions. These functions enable you to conveniently access the Nuxt instance from the context without having to pass it as argument.

::note
When you're working with the `setup` function in Nuxt modules, Nuxt is already provided as the second argument. This means you can directly utilize it without needing to call `useNuxt()`. You can look at [Nuxt Site Config](https://github.com/harlan-zw/nuxt-site-config) as an example of usage.
::

## `useNuxt`

Get the Nuxt instance from the context. It will throw an error if Nuxt is not available.

### Type

```ts
function useNuxt(): Nuxt

interface Nuxt {
  options: NuxtOptions
  hooks: Hookable<NuxtHooks>
  hook: Nuxt['hooks']['hook']
  callHook: Nuxt['hooks']['callHook']
  addHooks: Nuxt['hooks']['addHooks']
  ready: () => Promise<void>
  close: () => Promise<void>
  server?: any
  vfs: Record<string, string>
  apps: Record<string, NuxtApp>
}
```

### Examples

::code-group

```ts [setupTranspilation.ts]
// https://github.com/Lexpeartha/nuxt-xstate/blob/main/src/parts/transpile.ts
import { useNuxt } from '@nuxt/kit'

export const setupTranspilation = () => {
  const nuxt = useNuxt()

  nuxt.options.build.transpile = nuxt.options.build.transpile || []

  if (nuxt.options.builder === '@nuxt/webpack-builder') {
    nuxt.options.build.transpile.push(
      'xstate',
    )
  }
}
```

```ts [module.ts]
import { useNuxt } from '@nuxt/kit'
import { setupTranspilation } from './setupTranspilation'

export default defineNuxtModule({
  setup() {
    setupTranspilation()
  }
})
```

::

## `tryUseNuxt`

Get the Nuxt instance from the context. It will return `null` if Nuxt is not available.

### Type

```ts
function tryUseNuxt(): Nuxt | null

interface Nuxt {
  options: NuxtOptions
  hooks: Hookable<NuxtHooks>
  hook: Nuxt['hooks']['hook']
  callHook: Nuxt['hooks']['callHook']
  addHooks: Nuxt['hooks']['addHooks']
  ready: () => Promise<void>
  close: () => Promise<void>
  server?: any
  vfs: Record<string, string>
  apps: Record<string, NuxtApp>
}
```

### Examples

::code-group

```ts [requireSiteConfig.ts]
// https://github.com/harlan-zw/nuxt-site-config/blob/main/test/assertions.test.ts
import { tryUseNuxt } from '@nuxt/kit'

interface SiteConfig {
  title: string
}

export const requireSiteConfig = (): SiteConfig => {
  const nuxt = tryUseNuxt()
  if (!nuxt) {
    return { title: null }
  }
  return nuxt.options.siteConfig
}
```

```ts [module.ts]
import { useNuxt } from '@nuxt/kit'
import { requireSiteConfig } from './requireSiteConfig'

export default defineNuxtModule({
  setup(_, nuxt) {
    const config = requireSiteConfig()
    nuxt.options.app.head.title = config.title
  }
})
```

::
