---
title: "Components"
description: Nuxt Kit provides a set of utilities to help you work with components. You can register components globally or locally, and also add directories to be scanned for components.
links:
  - label: Source
    icon: i-simple-icons-github
    to: https://github.com/nuxt/nuxt/blob/main/packages/kit/src/components.ts
    size: xs
---

Components are the building blocks of your Nuxt application. They are reusable Vue instances that can be used to create a user interface. In Nuxt, components from the components directory are automatically imported by default. However, if you need to import components from an alternative directory or wish to selectively import them as needed, `@nuxt/kit` provides the `addComponentsDir` and `addComponent` methods. These utils allow you to customize the component configuration to better suit your needs.

::tip{icon="i-ph-video" to="https://vueschool.io/lessons/injecting-components-and-component-directories?friend=nuxt" target="_blank"}
Watch Vue School video about injecting components.
::

## `addComponentsDir`

Register a directory to be scanned for components and imported only when used. Keep in mind, that this does not register components globally, until you specify `global: true` option.

### Type

```ts
async function addComponentsDir (dir: ComponentsDir, opts: { prepend?: boolean } = {}): void

interface ComponentsDir {
  path: string
  pattern?: string | string[]
  ignore?: string[]
  prefix?: string
  pathPrefix?: boolean
  enabled?: boolean
  prefetch?: boolean
  preload?: boolean
  isAsync?: boolean
  extendComponent?: (component: Component) => Promise<Component | void> | (Component | void)
  global?: boolean
  island?: boolean
  watch?: boolean
  extensions?: string[]
  transpile?: 'auto' | boolean
}

// You can augment this interface (exported from `@nuxt/schema`) if needed
interface ComponentMeta {
  [key: string]: unknown
}

interface Component {
  pascalName: string
  kebabName: string
  export: string
  filePath: string
  shortPath: string
  chunkName: string
  prefetch: boolean
  preload: boolean
  global?: boolean
  island?: boolean
  mode?: 'client' | 'server' | 'all'
  priority?: number
  meta?: ComponentMeta
}
```

### Parameters

#### `dir`

**Type**: `ComponentsDir`

**Required**: `true`

An object with the following properties:

- `path` (required)

  **Type**: `string`

  Path (absolute or relative) to the directory containing your components.
  You can use Nuxt aliases (~ or @) to refer to directories inside project or directly use an npm package path similar to require.

- `pattern` (optional)

  **Type**: `string | string[]`

  Accept Pattern that will be run against specified path.

- `ignore` (optional)

  **Type**: `string[]`

  Ignore patterns that will be run against specified path.

- `prefix` (optional)

  **Type**: `string`

  Prefix all matched components with this string.

- `pathPrefix` (optional)

  **Type**: `boolean`

  Prefix component name by its path.

- `enabled` (optional)

  **Type**: `boolean`

  Ignore scanning this directory if set to `true`.

- `prefetch` (optional)

  **Type**: `boolean`

  These properties (prefetch/preload) are used in production to configure how components with Lazy prefix are handled by webpack via its magic comments.
  Learn more on [webpack documentation](https://webpack.js.org/api/module-methods/#magic-comments)

- `preload` (optional)

  **Type**: `boolean`

  These properties (prefetch/preload) are used in production to configure how components with Lazy prefix are handled by webpack via its magic comments.
  Learn more on [webpack documentation](https://webpack.js.org/api/module-methods/#magic-comments)

- `isAsync` (optional)

  **Type**: `boolean`

  This flag indicates, component should be loaded async (with a separate chunk) regardless of using Lazy prefix or not.

- `extendComponent` (optional)

  **Type**: `(component: Component) => Promise<Component | void> | (Component | void)`

  A function that will be called for each component found in the directory. It accepts a component object and should return a component object or a promise that resolves to a component object.

- `global` (optional)

  **Type**: `boolean`

  **Default**: `false`

  If enabled, registers components to be globally available.

- `island` (optional)

  **Type**: `boolean`

  If enabled, registers components as islands.

- `watch` (optional)

  **Type**: `boolean`

  Watch specified path for changes, including file additions and file deletions.

- `extensions` (optional)

  **Type**: `string[]`

  Extensions supported by Nuxt builder.

- `transpile` (optional)

  **Type**: `'auto' | boolean`

  Transpile specified path using build.transpile. If set to `'auto'`, it will set `transpile: true` if `node_modules/` is in path.

#### `opts`

**Required**: `false`

- `prepend` (optional)

  **Type**: `boolean`

  If set to `true`, the directory will be prepended to the array with `unshift()` instead of `push()`.

## `addComponent`

Register a component to be automatically imported.

### Type

```ts
async function addComponent (options: AddComponentOptions): void

interface AddComponentOptions {
  name: string,
  filePath: string,
  pascalName?: string,
  kebabName?: string,
  export?: string,
  shortPath?: string,
  chunkName?: string,
  prefetch?: boolean,
  preload?: boolean,
  global?: boolean,
  island?: boolean,
  mode?: 'client' | 'server' | 'all',
  priority?: number,
}
```

### Parameters

#### `options`

**Type**: `AddComponentOptions`

**Required**: `true`

An object with the following properties:

- `name` (required)

  **Type**: `string`

  Component name.

- `filePath` (required)

  **Type**: `string`

  Path to the component.

- `pascalName` (optional)

  **Type**: `pascalCase(options.name)`

  Pascal case component name. If not provided, it will be generated from the component name.

- `kebabName` (optional)

  **Type**: `kebabCase(options.name)`

  Kebab case component name. If not provided, it will be generated from the component name.

- `export` (optional)

  **Type**: `string`

  **Default**: `'default'`

  Specify named or default export. If not provided, it will be set to `'default'`.

- `shortPath` (optional)

  **Type**: `string`

  Short path to the component. If not provided, it will be generated from the component path.

- `chunkName` (optional)

  **Type**: `string`

  **Default**: `'components/' + kebabCase(options.name)`

  Chunk name for the component. If not provided, it will be generated from the component name.

- `prefetch` (optional)

  **Type**: `boolean`

  These properties (prefetch/preload) are used in production to configure how components with Lazy prefix are handled by webpack via its magic comments.
  Learn more on [webpack documentation](https://webpack.js.org/api/module-methods/#magic-comments)

- `preload` (optional)

  **Type**: `boolean`

  These properties (prefetch/preload) are used in production to configure how components with Lazy prefix are handled by webpack via its magic comments.
  Learn more on [webpack documentation](https://webpack.js.org/api/module-methods/#magic-comments)

- `global` (optional)

  **Type**: `boolean`

  **Default**: `false`

  If enabled, registers component to be globally available.

- `island` (optional)

  **Type**: `boolean`

  If enabled, registers component as island. You  can read more about islands in [`<NuxtIsland/>`](/docs/api/components/nuxt-island#nuxtisland) component description.

- `mode` (optional)

  **Type**: `'client' | 'server' | 'all'`

  **Default**: `'all'`

  This options indicates if component should render on client, server or both. By default, it will render on both client and server.

- `priority` (optional)

  **Type**: `number`

  **Default**: `1`

  Priority of the component, if multiple components have the same name, the one with the highest priority will be used.
