---
title: "Auto-imports"
description: Nuxt Kit provides a set of utilities to help you work with auto-imports. These functions allow you to register your own utils, composables and Vue APIs.
links:
  - label: Source
    icon: i-simple-icons-github
    to: https://github.com/nuxt/nuxt/blob/main/packages/kit/src/imports.ts
    size: xs
---

# Auto-imports

Nuxt auto-imports helper functions, composables and Vue APIs to use across your application without explicitly importing them. Based on the directory structure, every Nuxt application can also use auto-imports for its own composables and plugins. With Nuxt Kit you can also add your own auto-imports. `addImports` and `addImportsDir` allow you to add imports to the Nuxt application. `addImportsSources` allows you to add listed imports from 3rd party packages to the Nuxt application.

::note
These functions are designed for registering your own utils, composables and Vue APIs. For pages, components and plugins, please refer to the specific sections: [Pages](/docs/api/kit/pages), [Components](/docs/api/kit/components), [Plugins](/docs/api/kit/plugins).
::

Nuxt auto-imports helper functions, composables and Vue APIs to use across your application without explicitly importing them. Based on the directory structure, every Nuxt application can also use auto-imports for its own composables and plugins. Composables or plugins can use these functions.

::tip{icon="i-ph-video" to="https://vueschool.io/lessons/expanding-nuxt-s-auto-imports?friend=nuxt" target="_blank"}
Watch Vue School video about Auto-imports Nuxt Kit utilities.
::

## `addImports`

Add imports to the Nuxt application. It makes your imports available in the Nuxt application without the need to import them manually.

### Type

```ts
function addImports (imports: Import | Import[]): void

interface Import {
  from: string
  priority?: number
  disabled?: boolean
  meta?: {
    description?: string
    docsUrl?: string
    [key: string]: any
  }
  type?: boolean
  typeFrom?: string
  name: string
  as?: string
}
```

### Parameters

#### `imports`

**Type**: `Import | Import[]`

**Required**: `true`

An object or an array of objects with the following properties:

- `from` (required)

  **Type**: `string`

  Module specifier to import from.

- `priority` (optional)

  **Type**: `number`

  **Default**: `1`

  Priority of the import, if multiple imports have the same name, the one with the highest priority will be used.

- `disabled` (optional)

  **Type**: `boolean`

  If this import is disabled.

- `meta` (optional)

  **Type**: `object`

  Metadata of the import.

- `meta.description` (optional)

  **Type**: `string`

  Short description of the import.

- `meta.docsUrl` (optional)

  **Type**: `string`

  URL to the documentation.

- `meta[key]` (optional)

  **Type**: `any`

  Additional metadata.

- `type` (optional)

  **Type**: `boolean`

  If this import is a pure type import.

- `typeFrom` (optional)

  **Type**: `string`

  Using this as the from when generating type declarations.

- `name` (required)

  **Type**: `string`

  Import name to be detected.

- `as` (optional)

  **Type**: `string`

  Import as this name.

### Examples

```ts
// https://github.com/pi0/storyblok-nuxt
import { defineNuxtModule, addImports, createResolver } from '@nuxt/kit'

export default defineNuxtModule({
  setup(options, nuxt) {
    const names = [
      "useStoryblok",
      "useStoryblokApi",
      "useStoryblokBridge",
      "renderRichText",
      "RichTextSchema"
    ];

    names.forEach((name) =>
      addImports({ name, as: name, from: "@storyblok/vue" })
    );
  }
})
```

## `addImportsDir`

Add imports from a directory to the Nuxt application. It will automatically import all files from the directory and make them available in the Nuxt application without the need to import them manually.

### Type

```ts
function addImportsDir (dirs: string | string[], options?: { prepend?: boolean }): void
```

### Parameters

#### `dirs`

**Type**: `string | string[]`

**Required**: `true`

A string or an array of strings with the path to the directory to import from.

#### `options`

**Type**: `{ prepend?: boolean }`

**Default**: `{}`

Options to pass to the import. If `prepend` is set to `true`, the imports will be prepended to the list of imports.

### Examples

```ts
// https://github.com/vueuse/motion/tree/main/src/nuxt
import { defineNuxtModule, addImportsDir, createResolver } from '@nuxt/kit'

export default defineNuxtModule({
  meta: {
    name: '@vueuse/motion',
    configKey: 'motion',
  },
  setup(options, nuxt) {
    const resolver = createResolver(import.meta.url)
    addImportsDir(resolver.resolve('./runtime/composables'))
  },
})
```

## `addImportsSources`

Add listed imports to the Nuxt application.

### Type

```ts
function addImportsSources (importSources: ImportSource | ImportSource[]): void

interface Import {
  from: string
  priority?: number
  disabled?: boolean
  meta?: {
    description?: string
    docsUrl?: string
    [key: string]: any
  }
  type?: boolean
  typeFrom?: string
  name: string
  as?: string
}

interface ImportSource extends Import {
  imports: (PresetImport | ImportSource)[]
}

type PresetImport = Omit<Import, 'from'> | string | [name: string, as?: string, from?: string]
```

### Parameters

#### `importSources`

**Type**: `ImportSource | ImportSource[]`

**Required**: `true`

An object or an array of objects with the following properties:

- `imports` (required)

  **Type**: `PresetImport | ImportSource[]`

  **Required**: `true`

  An object or an array of objects, which can be import names, import objects or import sources.

- `from` (required)

  **Type**: `string`

  Module specifier to import from.

- `priority` (optional)

  **Type**: `number`

  **Default**: `1`

  Priority of the import, if multiple imports have the same name, the one with the highest priority will be used.

- `disabled` (optional)

  **Type**: `boolean`

  If this import is disabled.

- `meta` (optional)

  **Type**: `object`

  Metadata of the import.

- `meta.description` (optional)

  **Type**: `string`

  Short description of the import.

- `meta.docsUrl` (optional)

  **Type**: `string`

  URL to the documentation.

- `meta[key]` (optional)

  **Type**: `any`

  Additional metadata.

- `type` (optional)

  **Type**: `boolean`

  If this import is a pure type import.

- `typeFrom` (optional)

  **Type**: `string`

  Using this as the from when generating type declarations.

- `name` (required)

  **Type**: `string`

  Import name to be detected.

- `as` (optional)

  **Type**: `string`

  Import as this name.

### Examples

```ts
// https://github.com/elk-zone/elk
import { defineNuxtModule, addImportsSources } from '@nuxt/kit'

export default defineNuxtModule({
  setup() {
    // add imports from h3 to make them autoimported
    addImportsSources({
      from: 'h3',
      imports: ['defineEventHandler', 'getQuery', 'getRouterParams', 'readBody', 'sendRedirect'] as Array<keyof typeof import('h3')>,
    })
  }
})
```
