---
title: Builder
description: Nuxt Kit provides a set of utilities to help you work with the builder. These functions allow you to extend the webpack and vite configurations.
links:
  - label: Source
    icon: i-simple-icons-github
    to: https://github.com/nuxt/nuxt/blob/main/packages/kit/src/build.ts
    size: xs
---

Nuxt have builders based on [webpack](https://github.com/nuxt/nuxt/tree/main/packages/webpack) and [vite](https://github.com/nuxt/nuxt/tree/main/packages/vite). You can extend the config passed to each one using `extendWebpackConfig` and `extendViteConfig` functions. You can also add additional plugins via `addVitePlugin`, `addWebpackPlugin` and `addBuildPlugin`.

## `extendWebpackConfig`

Extends the webpack configuration. Callback function can be called multiple times, when applying to both client and server builds.

### Type

```ts
function extendWebpackConfig (callback: ((config: WebpackConfig) => void), options?: ExtendWebpackConfigOptions): void

export interface ExtendWebpackConfigOptions {
  dev?: boolean
  build?: boolean
  server?: boolean
  client?: boolean
  prepend?: boolean
}
```

::read-more{to="https://webpack.js.org/configuration" target="_blank" color="gray" icon="i-simple-icons-webpack"}
Checkout webpack website for more information about its configuration.
::

### Parameters

#### `callback`

**Type**: `(config: WebpackConfig) => void`

**Required**: `true`

A callback function that will be called with the webpack configuration object.

#### `options`

**Type**: `ExtendWebpackConfigOptions`

**Default**: `{}`

Options to pass to the callback function. This object can have the following properties:

- `dev` (optional)

  **Type**: `boolean`

  **Default**: `true`

  If set to `true`, the callback function will be called when building in development mode.

- `build` (optional)

  **Type**: `boolean`

  **Default**: `true`

  If set to `true`, the callback function will be called when building in production mode.

- `server` (optional)

  **Type**: `boolean`

  **Default**: `true`

  If set to `true`, the callback function will be called when building the server bundle.

- `client` (optional)

  **Type**: `boolean`

  **Default**: `true`

  If set to `true`, the callback function will be called when building the client bundle.

- `prepend` (optional)

  **Type**: `boolean`

  If set to `true`, the callback function will be prepended to the array with `unshift()` instead of `push()`.

### Examples

```ts
import { defineNuxtModule, extendWebpackConfig } from '@nuxt/kit'

export default defineNuxtModule({
  setup() {
    extendWebpackConfig((config) => {
      config.module?.rules.push({
        test: /\.txt$/,
        use: 'raw-loader'
      })
    })
  }
})
```

## `extendViteConfig`

Extends the Vite configuration. Callback function can be called multiple times, when applying to both client and server builds.

### Type

```ts
function extendViteConfig (callback: ((config: ViteConfig) => void), options?: ExtendViteConfigOptions): void

export interface ExtendViteConfigOptions {
  dev?: boolean
  build?: boolean
  server?: boolean
  client?: boolean
  prepend?: boolean
}
```

::read-more{to="https://vitejs.dev/config" target="_blank" color="gray" icon="i-simple-icons-vite"}
Checkout Vite website for more information about its configuration.
::

### Parameters

#### `callback`

**Type**: `(config: ViteConfig) => void`

**Required**: `true`

A callback function that will be called with the Vite configuration object.

#### `options`

**Type**: `ExtendViteConfigOptions`

**Default**: `{}`

Options to pass to the callback function. This object can have the following properties:

- `dev` (optional)

  **Type**: `boolean`

  **Default**: `true`

  If set to `true`, the callback function will be called when building in development mode.

- `build` (optional)

  **Type**: `boolean`

  **Default**: `true`

  If set to `true`, the callback function will be called when building in production mode.

- `server` (optional)

  **Type**: `boolean`

  **Default**: `true`

  If set to `true`, the callback function will be called when building the server bundle.

- `client` (optional)

  **Type**: `boolean`

  **Default**: `true`

  If set to `true`, the callback function will be called when building the client bundle.

- `prepend` (optional)

  **Type**: `boolean`

  If set to `true`, the callback function will be prepended to the array with `unshift()` instead of `push()`.

### Examples

```ts
// https://github.com/Hrdtr/nuxt-appwrite
import { defineNuxtModule, extendViteConfig } from '@nuxt/kit'

export default defineNuxtModule({
  setup() {
    extendViteConfig((config) => {
      config.optimizeDeps = config.optimizeDeps || {}
      config.optimizeDeps.include = config.optimizeDeps.include || []
      config.optimizeDeps.include.push('cross-fetch')
    })
  }
})
```

## `addWebpackPlugin`

Append webpack plugin to the config.

### Type

```ts
function addWebpackPlugin (pluginOrGetter: PluginOrGetter, options?: ExtendWebpackConfigOptions): void

type PluginOrGetter = WebpackPluginInstance | WebpackPluginInstance[] | (() => WebpackPluginInstance | WebpackPluginInstance[])

interface ExtendWebpackConfigOptions {
  dev?: boolean
  build?: boolean
  server?: boolean
  client?: boolean
  prepend?: boolean
}
```

::tip
See [webpack website](https://webpack.js.org/concepts/plugins) for more information about webpack plugins. You can also use [this collection](https://webpack.js.org/awesome-webpack/#webpack-plugins) to find a plugin that suits your needs.
::

### Parameters

#### `pluginOrGetter`

**Type**: `PluginOrGetter`

**Required**: `true`

A webpack plugin instance or an array of webpack plugin instances. If a function is provided, it must return a webpack plugin instance or an array of webpack plugin instances.

#### `options`

**Type**: `ExtendWebpackConfigOptions`

**Default**: `{}`

Options to pass to the callback function. This object can have the following properties:

- `dev` (optional)

  **Type**: `boolean`

  **Default**: `true`

  If set to `true`, the callback function will be called when building in development mode.

- `build` (optional)

  **Type**: `boolean`

  **Default**: `true`

  If set to `true`, the callback function will be called when building in production mode.

- `server` (optional)

  **Type**: `boolean`

  **Default**: `true`

  If set to `true`, the callback function will be called when building the server bundle.

- `client` (optional)

  **Type**: `boolean`

  **Default**: `true`

  If set to `true`, the callback function will be called when building the client bundle.

- `prepend` (optional)

  **Type**: `boolean`

  If set to `true`, the callback function will be prepended to the array with `unshift()` instead of `push()`.

### Examples

```ts
// https://github.com/nuxt-modules/eslint
import EslintWebpackPlugin from 'eslint-webpack-plugin'
import { defineNuxtModule, addWebpackPlugin } from '@nuxt/kit'

export default defineNuxtModule({
  meta: {
    name: 'nuxt-eslint',
    configKey: 'eslint',
  },
  defaults: nuxt => ({
    include: [`${nuxt.options.srcDir}/**/*.{js,jsx,ts,tsx,vue}`],
    lintOnStart: true,
  }),
  setup(options, nuxt) {
    const webpackOptions = {
      ...options,
      context: nuxt.options.srcDir,
      files: options.include,
      lintDirtyModulesOnly: !options.lintOnStart
    }
    addWebpackPlugin(new EslintWebpackPlugin(webpackOptions), { server: false })
  }
})
```

## `addVitePlugin`

Append Vite plugin to the config.

### Type

```ts
function addVitePlugin (pluginOrGetter: PluginOrGetter, options?: ExtendViteConfigOptions): void

type PluginOrGetter = VitePlugin | VitePlugin[] | (() => VitePlugin | VitePlugin[])

interface ExtendViteConfigOptions {
  dev?: boolean
  build?: boolean
  server?: boolean
  client?: boolean
  prepend?: boolean
}
```

::tip
See [Vite website](https://vitejs.dev/guide/api-plugin.html) for more information about Vite plugins. You can also use [this repository](https://github.com/vitejs/awesome-vite#plugins) to find a plugin that suits your needs.
::

### Parameters

#### `pluginOrGetter`

**Type**: `PluginOrGetter`

**Required**: `true`

A Vite plugin instance or an array of Vite plugin instances. If a function is provided, it must return a Vite plugin instance or an array of Vite plugin instances.

#### `options`

**Type**: `ExtendViteConfigOptions`

**Default**: `{}`

Options to pass to the callback function. This object can have the following properties:

- `dev` (optional)

  **Type**: `boolean`

  **Default**: `true`

  If set to `true`, the callback function will be called when building in development mode.

- `build` (optional)

  **Type**: `boolean`

  **Default**: `true`

  If set to `true`, the callback function will be called when building in production mode.

- `server` (optional)

  **Type**: `boolean`

  **Default**: `true`

  If set to `true`, the callback function will be called when building the server bundle.

- `client` (optional)

  **Type**: `boolean`

  **Default**: `true`

  If set to `true`, the callback function will be called when building the client bundle.

- `prepend` (optional)

  **Type**: `boolean`

  If set to `true`, the callback function will be prepended to the array with `unshift()` instead of `push()`.

### Examples

```ts
// https://github.com/yisibell/nuxt-svg-icons
import { defineNuxtModule, addVitePlugin } from '@nuxt/kit'
import { svg4VuePlugin } from 'vite-plugin-svg4vue'

export default defineNuxtModule({
  meta: {
    name: 'nuxt-svg-icons',
    configKey: 'nuxtSvgIcons',
  },
  defaults: {
    svg4vue: {
      assetsDirName: 'assets/icons',
    },
  },
  setup(options) {
    addVitePlugin(svg4VuePlugin(options.svg4vue))
  },
})
```

## `addBuildPlugin`

Builder-agnostic version of `addWebpackPlugin` and `addVitePlugin`. It will add the plugin to both webpack and vite configurations if they are present.

### Type

```ts
function addBuildPlugin (pluginFactory: AddBuildPluginFactory, options?: ExtendConfigOptions): void

interface AddBuildPluginFactory {
  vite?: () => VitePlugin | VitePlugin[]
  webpack?: () => WebpackPluginInstance | WebpackPluginInstance[]
}

interface ExtendConfigOptions {
  dev?: boolean
  build?: boolean
  server?: boolean
  client?: boolean
  prepend?: boolean
}
```

### Parameters

#### `pluginFactory`

**Type**: `AddBuildPluginFactory`

**Required**: `true`

A factory function that returns an object with `vite` and/or `webpack` properties. These properties must be functions that return a Vite plugin instance or an array of Vite plugin instances and/or a webpack plugin instance or an array of webpack plugin instances.

#### `options`

**Type**: `ExtendConfigOptions`

**Default**: `{}`

Options to pass to the callback function. This object can have the following properties:

- `dev` (optional)

  **Type**: `boolean`

  **Default**: `true`

  If set to `true`, the callback function will be called when building in development mode.

- `build` (optional)

  **Type**: `boolean`

  **Default**: `true`

  If set to `true`, the callback function will be called when building in production mode.

- `server` (optional)

  **Type**: `boolean`

  **Default**: `true`

  If set to `true`, the callback function will be called when building the server bundle.

- `client` (optional)

  **Type**: `boolean`

  **Default**: `true`

  If set to `true`, the callback function will be called when building the client bundle.

- `prepend` (optional)

  **Type**: `boolean`

  If set to `true`, the callback function will be prepended to the array with `unshift()` instead of `push()`.
