---
title: Resolving
description: Nuxt Kit provides a set of utilities to help you resolve paths. These functions allow you to resolve paths relative to the current module, with unknown name or extension.
links:
  - label: Source
    icon: i-simple-icons-github
    to: https://github.com/nuxt/nuxt/blob/main/packages/kit/src/resolve.ts
    size: xs
---

Sometimes you need to resolve a paths: relative to the current module, with unknown name or extension. For example, you may want to add a plugin that is located in the same directory as the module. To handle this cases, nuxt provides a set of utilities to resolve paths. `resolvePath` and `resolveAlias` are used to resolve paths relative to the current module. `findPath` is used to find first existing file in given paths. `createResolver` is used to create resolver relative to base path.

## `resolvePath`

Resolves full path to a file or directory respecting Nuxt alias and extensions options. If path could not be resolved, normalized input path will be returned.

### Type

```ts
async function resolvePath (path: string, options?: ResolvePathOptions): Promise<string>
```

### Parameters

#### `path`

**Type**: `string`

**Required**: `true`

Path to resolve.

#### `options`

**Type**: `ResolvePathOptions`

**Default**: `{}`

Options to pass to the resolver. This object can have the following properties:

- `cwd` (optional)

  **Type**: `string`

  **Default**: `process.cwd()`

  Current working directory.

- `alias` (optional)

  **Type**: `Record<string, string>`

  **Default**: `{}`

  Alias map.

- `extensions` (optional)

  **Type**: `string[]`

  **Default**: `['.js', '.mjs', '.ts', '.jsx', '.tsx', '.json']`

  Extensions to try.

### Examples

```ts
// https://github.com/P4sca1/nuxt-headlessui
import { defineNuxtModule, resolvePath } from '@nuxt/kit'
import { join } from 'pathe'

const headlessComponents: ComponentGroup[] = [
  {
    relativePath: 'combobox/combobox.js',
    chunkName: 'headlessui/combobox',
    exports: [
      'Combobox',
      'ComboboxLabel',
      'ComboboxButton',
      'ComboboxInput',
      'ComboboxOptions',
      'ComboboxOption'
    ]
  },
]

export default defineNuxtModule({
  meta: {
    name: 'nuxt-headlessui',
    configKey: 'headlessui',
  },
  defaults: {
    prefix: 'Headless'
  },
  async setup (options) {
    const entrypoint = await resolvePath('@headlessui/vue')
    const root = join(entrypoint, '../components')

    for (const group of headlessComponents) {
      for (const e of group.exports) {
        addComponent(
          {
            name: e,
            export: e,
            filePath: join(root, group.relativePath),
            chunkName: group.chunkName,
            mode: 'all'
          }
        )
      }
    }
  }
})
```

## `resolveAlias`

Resolves path aliases respecting Nuxt alias options.

### Type
  
```ts
function resolveAlias (path: string, alias?: Record<string, string>): string
```

### Parameters

#### `path`

**Type**: `string`

**Required**: `true`

Path to resolve.

#### `alias`

**Type**: `Record<string, string>`

**Default**: `{}`

Alias map. If not provided, it will be read from `nuxt.options.alias`.

## `findPath`

Try to resolve first existing file in given paths.

### Type

```ts
async function findPath (paths: string | string[], options?: ResolvePathOptions, pathType: 'file' | 'dir'): Promise<string | null>

interface ResolvePathOptions {
  cwd?: string
  alias?: Record<string, string>
  extensions?: string[]
}
```

### Parameters

#### `paths`

**Type**: `string | string[]`

**Required**: `true`

A path or an array of paths to resolve.

#### `options`

**Type**: `ResolvePathOptions`

**Default**: `{}`

Options to pass to the resolver. This object can have the following properties:

- `cwd` (optional)

  **Type**: `string`

  **Default**: `process.cwd()`

  Current working directory.

- `alias` (optional)

  **Type**: `Record<string, string>`

  **Default**: `{}`

  Alias map.

- `extensions` (optional)

  **Type**: `string[]`

  **Default**: `['.js', '.mjs', '.ts', '.jsx', '.tsx', '.json']`

  Extensions to try.

#### `pathType`

**Type**: `'file' | 'dir'`

**Default**: `'file'`

Type of path to resolve. If set to `'file'`, the function will try to resolve a file. If set to `'dir'`, the function will try to resolve a directory.

## `createResolver`

Creates resolver relative to base path.

::tip{icon="i-ph-video" to="https://vueschool.io/lessons/resolving-paths-and-injecting-assets-to-the-app?friend=nuxt" target="_blank"}
Watch Vue School video about createResolver.
::

### Type

```ts
function createResolver (basePath: string | URL): Resolver

interface Resolver {
  resolve (...path: string[]): string
  resolvePath (path: string, options?: ResolvePathOptions): Promise<string>
}

interface ResolvePathOptions {
  cwd?: string
  alias?: Record<string, string>
  extensions?: string[]
}
```

### Parameters

#### `basePath`

**Type**: `string`

**Required**: `true`

Base path to resolve from.

### Examples

```ts
// https://github.com/vuejs/pinia/blob/v2/packages/nuxt
import {
  defineNuxtModule,
  isNuxt2,
  createResolver,
} from '@nuxt/kit'

export default defineNuxtModule({
  setup(options, nuxt) {
    const resolver = createResolver(import.meta.url)

    nuxt.hook('modules:done', () => {
      if (isNuxt2()) {
        addPlugin(resolver.resolve('./runtime/plugin.vue2'))
      } else {
        addPlugin(resolver.resolve('./runtime/plugin.vue3'))
      }
    })
  }
})
```
