---
title: "Modules"
description: Nuxt Kit provides a set of utilities to help you create and use modules. You can use these utilities to create your own modules or to reuse existing modules.
links:
  - label: Source
    icon: i-simple-icons-github
    to: https://github.com/nuxt/nuxt/blob/main/packages/kit/src/module
    size: xs
---

Modules are the building blocks of Nuxt. Kit provides a set of utilities to help you create and use modules. You can use these utilities to create your own modules or to reuse existing modules. For example, you can use the `defineNuxtModule` function to define a module and the `installModule` function to install a module programmatically.

## `defineNuxtModule`

Define a Nuxt module, automatically merging defaults with user provided options, installing any hooks that are provided, and calling an optional setup function for full control.

### Type

```ts
function defineNuxtModule<OptionsT extends ModuleOptions> (definition: ModuleDefinition<OptionsT> | NuxtModule<OptionsT>): NuxtModule<OptionsT>

type ModuleOptions = Record<string, any>

interface ModuleDefinition<T extends ModuleOptions = ModuleOptions> {
  meta?: ModuleMeta
  defaults?: T | ((nuxt: Nuxt) => T)
  schema?: T
  hooks?: Partial<NuxtHooks>
  setup?: (this: void, resolvedOptions: T, nuxt: Nuxt) => Awaitable<void | false | ModuleSetupReturn>
}

interface NuxtModule<T extends ModuleOptions = ModuleOptions> {
  (this: void, inlineOptions: T, nuxt: Nuxt): Awaitable<void | false | ModuleSetupReturn>
  getOptions?: (inlineOptions?: T, nuxt?: Nuxt) => Promise<T>
  getMeta?: () => Promise<ModuleMeta>
}

interface ModuleSetupReturn {
  timings?: {
    setup?: number
    [key: string]: number | undefined
  }
}

interface ModuleMeta {
  name?: string
  version?: string
  configKey?: string
  compatibility?: NuxtCompatibility
  [key: string]: unknown
}
```

### Parameters

#### `definition`

**Type**: `ModuleDefinition<T> | NuxtModule<T>`

**Required**: `true`

A module definition object or a module function.

- `meta` (optional)

  **Type**: `ModuleMeta`

  Metadata of the module. It defines the module name, version, config key and compatibility.

- `defaults` (optional)

  **Type**: `T | ((nuxt: Nuxt) => T)`

  Default options for the module. If a function is provided, it will be called with the Nuxt instance as the first argument.

- `schema` (optional)

  **Type**: `T`

  Schema for the module options. If provided, options will be applied to the schema.

- `hooks` (optional)

  **Type**: `Partial<NuxtHooks>`

  Hooks to be installed for the module. If provided, the module will install the hooks.

- `setup` (optional)

  **Type**: `(this: void, resolvedOptions: T, nuxt: Nuxt) => Awaitable<void | false | ModuleSetupReturn>`

  Setup function for the module. If provided, the module will call the setup function.

### Examples

```ts
// https://github.com/nuxt/starter/tree/module
import { defineNuxtModule } from '@nuxt/kit'

export default defineNuxtModule({
  meta: {
    name: 'my-module',
    configKey: 'myModule'
  },
  defaults: {
    test: 123
  },
  setup (options, nuxt) {
    nuxt.hook('modules:done', () => {
      console.log('My module is ready with current test option: ', options.test)
    })    
  }
})
```

## `installModule`

Install specified Nuxt module programmatically. This is helpful when your module depends on other modules. You can pass the module options as an object to `inlineOptions` and they will be passed to the module's `setup` function.

### Type

```ts
async function installModule (moduleToInstall: string | NuxtModule, inlineOptions?: any, nuxt?: Nuxt)
```

### Parameters

#### `moduleToInstall`

**Type**: `string` | `NuxtModule`

**Required**: `true`

The module to install. Can be either a string with the module name or a module object itself.

#### `inlineOptions`

**Type**: `any`

**Default**: `{}`

An object with the module options to be passed to the module's `setup` function.

#### `nuxt`

**Type**: `Nuxt`

**Default**: `useNuxt()`

Nuxt instance. If not provided, it will be retrieved from the context via `useNuxt()` call.

### Examples

```ts
import { defineNuxtModule, installModule } from '@nuxt/kit'

export default defineNuxtModule({  
  async setup (options, nuxt) {
    // will install @nuxtjs/fontaine with Roboto font and Impact fallback
    await installModule('@nuxtjs/fontaine', {
      // module configuration
      fonts: [
        {
          family: 'Roboto',
          fallbacks: ['Impact'],
          fallbackName: 'fallback-a',
        }
      ]
    })
  }
})
```
