---
title: "<NuxtPage>"
description: The <NuxtPage> component is required to display pages located in the pages/ directory.
links:
  - label: Source
    icon: i-simple-icons-github
    to: https://github.com/nuxt/nuxt/blob/main/packages/nuxt/src/pages/runtime/page.ts
    size: xs
---

`<NuxtPage>` is a built-in component that comes with Nuxt. It lets you display top-level or nested pages located in the [`pages/`](/docs/guide/directory-structure/pages) directory.

::note
`<NuxtPage>` is a wrapper around [`<RouterView>`](https://router.vuejs.org/api/interfaces/RouterViewProps.html#interface-routerviewprops) component from Vue Router. :br
It accepts same `name` and `route` props.
::

::warning
`<NuxtPage>` should be used instead of `<RouterView>` as the former takes additional care on internal states. Otherwise, `useRoute()` may return incorrect paths.
::

## Props

- `name`: tells `RouterView` to render the component with the corresponding name in the matched route record's components option.
  - type: `string`
- `route`: route location that has all of its components resolved.
  - type: `RouteLocationNormalized`
- `pageKey`: control when the `NuxtPage` component is re-rendered.
  - type: `string` or `function`
- `transition`: define global transitions for all pages rendered with the `NuxtPage` component.
  - type: `boolean` or `TransitionProps`
- `keepalive`: control state preservation of pages rendered with the `NuxtPage` component.
  - type: `boolean` or `KeepAliveProps`

::tip
Nuxt automatically resolves the `name` and `route` by scanning and rendering all Vue component files found in the `/pages` directory.
::

## Example

For example, passing `static` key, `NuxtPage` component is rendered only once when it is mounted.

```vue [app.vue]
<template>
  <NuxtPage page-key="static" />
</template>
```

You can also use a dynamic key based on the current route:

```html
<NuxtPage :page-key="route => route.fullPath" />
```

::warning
Don't use `$route` object here as it can cause problems with how `<NuxtPage>` renders pages with `<Suspense>`.
::

Alternatively, `pageKey` can be passed as a `key` value via [`definePageMeta`](/docs/api/utils/define-page-meta) from the `<script>` section of your Vue component in the `/pages` directory.

```vue [pages/my-page.vue]
<script setup lang="ts">
definePageMeta({
  key: route => route.fullPath
})
</script>
```

:link-example{to="/docs/examples/routing/pages"}

## Page's Ref

To get the `ref` of a page component, access it through `ref.value.pageRef`

````vue [app.vue]
<script setup lang="ts">
const page = ref()

function logFoo () {
  page.value.pageRef.foo()
}
</script>

<template>
  <NuxtPage ref="page" />
</template>
````

````vue [my-page.vue]
<script setup lang="ts">
const foo = () => {
  console.log('foo method called')
}

defineExpose({
  foo,
})
</script>
````

## Custom Props

In addition, `<NuxtPage>` also accepts custom props that you may need to pass further down the hierarchy.

These custom props are accessible via `attrs` in the Nuxt app.

```html
<NuxtPage :foobar="123" />
```

For example, in the above example, the value of `foobar` will be available using `$attrs.foobar` in the template or `useAttrs().foobar` in `<script setup>`.

:read-more{to="/docs/guide/directory-structure/pages"}
