---
title: "Lifecycle Hooks"
description: "Nuxt provides a powerful hooking system to expand almost every aspect using hooks."
---

::tip
The hooking system is powered by [unjs/hookable](https://github.com/unjs/hookable).
::

## Nuxt Hooks (Build Time)

These hooks are available for [Nuxt Modules](/docs/guide/going-further/modules) and build context.

### Within `nuxt.config`

```js [nuxt.config]
export default defineNuxtConfig({
  hooks: {
    close: () => { }
  }
})
```

### Within Nuxt Modules

```js
import { defineNuxtModule } from '@nuxt/kit'

export default defineNuxtModule({
  setup (options, nuxt) {
    nuxt.hook('close', async () => { })
  }
})
```

::read-more{to="/docs/api/advanced/hooks#nuxt-hooks-build-time"}
Explore all available Nuxt hooks.
::

## App Hooks (Runtime)

App hooks can be mainly used by [Nuxt Plugins](/docs/guide/directory-structure/plugins) to hook into rendering lifecycle but could also be used in Vue composables.

```js [plugins/test.ts]
export default defineNuxtPlugin((nuxtApp) => {
  nuxtApp.hook('page:start', () => {
    /* your code goes here */
  })
})
```

::read-more{to="/docs/api/advanced/hooks#app-hooks-runtime"}
Explore all available App hooks.
::

## Server Hooks (Runtime)

These hooks are available for [server plugins](/docs/guide/directory-structure/server#server-plugins) to hook into Nitro's runtime behavior.

```js [~/server/plugins/test.ts]
export default defineNitroPlugin((nitroApp) => {
  nitroApp.hooks.hook('render:html', (html, { event }) => {
    console.log('render:html', html)
    html.bodyAppend.push('<hr>Appended by custom plugin')
  })

  nitroApp.hooks.hook('render:response', (response, { event }) => {
    console.log('render:response', response)
  })
})
```

::read-more{to="/docs/api/advanced/hooks#nitro-app-hooks-runtime-server-side"}
Learn more about available Nitro lifecycle hooks.
::

## Additional Hooks

You can add additional hooks by augmenting the types provided by Nuxt. This can be useful for modules.

```ts
import { HookResult } from "@nuxt/schema";

declare module '#app' {
  interface RuntimeNuxtHooks {
    'your-nuxt-runtime-hook': () => HookResult
  }
  interface NuxtHooks {
    'your-nuxt-hook': () => HookResult
  }
}

declare module 'nitro/types' {
  interface NitroRuntimeHooks {
    'your-nitro-hook': () => void;
  }
}
```
