---
title: "Experimental Features"
description: "Enable Nuxt experimental features to unlock new possibilities."
---

The Nuxt experimental features can be enabled in the Nuxt configuration file.

Internally, Nuxt uses `@nuxt/schema` to define these experimental features. You can refer to the [API documentation](/docs/api/configuration/nuxt-config#experimental) or the [source code](https://github.com/nuxt/nuxt/blob/main/packages/schema/src/config/experimental.ts) for more information.

::note
Note that these features are experimental and could be removed or modified in the future.
::

## asyncContext

Enable native async context to be accessible for nested composables in Nuxt and in Nitro. This opens the possibility to use composables inside async composables and reduce the chance to get the `Nuxt instance is unavailable` error.

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    asyncContext: true
  }
})
```

::read-more{icon="i-simple-icons-github" color="gray" to="https://github.com/nuxt/nuxt/pull/20918" target="_blank"}
See full explanation on the GitHub pull-request.
::

## asyncEntry

Enables generation of an async entry point for the Vue bundle, aiding module federation support.

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    asyncEntry: true
  }
})
```

## externalVue

Externalizes `vue`, `@vue/*` and `vue-router` when building.

*Enabled by default.*

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    externalVue: true
  }
})
```

::warning
This feature will likely be removed in a near future.
::

## emitRouteChunkError

Emits `app:chunkError` hook when there is an error loading vite/webpack chunks. Default behavior is to perform a hard reload of the new route when a chunk fails to load.

You can disable automatic handling by setting this to `false`, or handle chunk errors manually by setting it to `manual`.

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    emitRouteChunkError: 'automatic' // or 'manual' or false
  }
})
```

## restoreState

Allows Nuxt app state to be restored from `sessionStorage` when reloading the page after a chunk error or manual [`reloadNuxtApp()`](/docs/api/utils/reload-nuxt-app) call.

To avoid hydration errors, it will be applied only after the Vue app has been mounted, meaning there may be a flicker on initial load.

::important
Consider carefully before enabling this as it can cause unexpected behavior,
and consider providing explicit keys to [`useState`](/docs/api/composables/use-state) as auto-generated keys may not match across builds.
::

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    restoreState: true
  }
})
```

## inlineRouteRules

Define route rules at the page level using [`defineRouteRules`](/docs/api/utils/define-route-rules).

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    inlineRouteRules: true
  }
})
```

Matching route rules will be created, based on the page's `path`.

::read-more{to="/docs/api/utils/define-route-rules" icon="i-ph-function"}
Read more in `defineRouteRules` utility.
::

:read-more{to="/docs/guide/concepts/rendering#hybrid-rendering" icon="i-ph-medal"}

## renderJsonPayloads

Allows rendering of JSON payloads with support for revivifying complex types.

*Enabled by default.*

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    renderJsonPayloads: true
  }
})
```

## noVueServer

Disables Vue server renderer endpoint within Nitro.

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    noVueServer: true
  }
})
```

## payloadExtraction

Enables extraction of payloads of pages generated with `nuxt generate`.

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    payloadExtraction: true
  }
})
```

## clientFallback

Enables the experimental [`<NuxtClientFallback>`](/docs/api/components/nuxt-client-fallback) component for rendering content on the client if there's an error in SSR.

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    clientFallback: true
  }
})
```

## crossOriginPrefetch

Enables cross-origin prefetch using the Speculation Rules API.

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    crossOriginPrefetch: true
  }
})
```

::read-more{icon="i-simple-icons-w3c" color="gray" to="https://wicg.github.io/nav-speculation/prefetch.html" target="_blank"}
Read more about the **Speculation Rules API**.
::

## viewTransition

Enables View Transition API integration with client-side router.

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    viewTransition: true
  }
})
```

:link-example{to="https://stackblitz.com/edit/nuxt-view-transitions?file=app.vue" target="_blank"}

::read-more{icon="i-simple-icons-mdnwebdocs" color="gray" to="https://developer.mozilla.org/en-US/docs/Web/API/View_Transitions_API" target="_blank"}
Read more about the **View Transition API**.
::

## writeEarlyHints

Enables writing of early hints when using node server.

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    writeEarlyHints: true
  }
})
```

## componentIslands

Enables experimental component islands support with [`<NuxtIsland>`](/docs/api/components/nuxt-island) and `.island.vue` files.

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    componentIslands: true // false or 'local+remote'
  }
})
```

:read-more{to="/docs/guide/directory-structure/components#server-components"}

::read-more{icon="i-simple-icons-github" color="gray" to="https://github.com/nuxt/nuxt/issues/19772" target="_blank"}
You can follow the server components roadmap on GitHub.
::

## localLayerAliases

Resolve `~`, `~~`, `@` and `@@` aliases located within layers with respect to their layer source and root directories.

*Enabled by default.*

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    localLayerAliases: true
  }
})
```

## typedPages

Enable the new experimental typed router using [`unplugin-vue-router`](https://github.com/posva/unplugin-vue-router).

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    typedPages: true
  }
})
```

Out of the box, this will enable typed usage of [`navigateTo`](/docs/api/utils/navigate-to), [`<NuxtLink>`](/docs/api/components/nuxt-link), [`router.push()`](/docs/api/composables/use-router) and more.

You can even get typed params within a page by using `const route = useRoute('route-name')`.

::tip{icon="i-ph-video" to="https://www.youtube.com/watch?v=SXk-L19gTZk" target="_blank"}
Watch a video from Daniel Roe explaining type-safe routing in Nuxt.
::

## watcher

Set an alternative watcher that will be used as the watching service for Nuxt.

Nuxt uses `chokidar-granular` by default, which will ignore top-level directories
(like `node_modules` and `.git`) that are excluded from watching.

You can set this instead to `parcel` to use `@parcel/watcher`, which may improve
performance in large projects or on Windows platforms.

You can also set this to `chokidar` to watch all files in your source directory.

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    watcher: 'chokidar-granular' // 'chokidar' or 'parcel' are also options
  }
})
```

## sharedPrerenderData

Enabling this feature automatically shares payload *data* between pages that are prerendered. This can result
in a significant performance improvement when prerendering sites that use `useAsyncData` or `useFetch` and
fetch the same data in different pages.

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    sharedPrerenderData: true
  }
})
```

::tip{icon="i-ph-video" to="https://www.youtube.com/watch?v=1jUupYHVvrU" target="_blank"}
Watch a video from Alexander Lichter about the experimental `sharedPrerenderData` setting.
::

It is particularly important when enabling this feature to make sure that any unique key of your data
is always resolvable to the same data. For example, if you are using `useAsyncData` to fetch
data related to a particular page, you should provide a key that uniquely matches that data. (`useFetch`
should do this automatically for you.)

```ts
// This would be unsafe in a dynamic page (e.g. `[slug].vue`) because the route slug makes a difference
// to the data fetched, but Nuxt can't know that because it's not reflected in the key.
const route = useRoute()
const { data } = await useAsyncData(async () => {
  return await $fetch(`/api/my-page/${route.params.slug}`)
})
// Instead, you should use a key that uniquely identifies the data fetched.
const { data } = await useAsyncData(route.params.slug, async () => {
  return await $fetch(`/api/my-page/${route.params.slug}`)
})
```

## clientNodeCompat

With this feature, Nuxt will automatically polyfill Node.js imports in the client build using [`unenv`](https://github.com/unjs/unenv).

::alert{type=info}
To make globals like `Buffer` work in the browser, you need to manually inject them.

```ts
import { Buffer } from 'node:buffer'

globalThis.Buffer = globalThis.Buffer || Buffer
```
::

## scanPageMeta

This option allows exposing some route metadata defined in `definePageMeta` at build-time to modules (specifically `alias`, `name`, `path`, `redirect`).

This only works with static or strings/arrays rather than variables or conditional assignment. See [original issue](https://github.com/nuxt/nuxt/issues/24770) for more information and context.

It is also possible to scan page metadata only after all routes have been registered in `pages:extend`. Then another hook, `pages:resolved` will be called. To enable this behavior, set `scanPageMeta: 'after-resolve'`.

You can disable this feature if it causes issues in your project.

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    scanPageMeta: false
  }
})
```

## cookieStore

Enables CookieStore support to listen for cookie updates (if supported by the browser) and refresh `useCookie` ref values.

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    cookieStore: true
  }
})
```

::read-more{icon="i-simple-icons-mdnwebdocs" color="gray" to="https://developer.mozilla.org/en-US/docs/Web/API/CookieStore" target="_blank"}
Read more about the **CookieStore**.
::

## buildCache

Caches Nuxt build artifacts based on a hash of the configuration and source files.

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    buildCache: true
  }
})
```

When enabled, changes to the following files will trigger a full rebuild:

```bash [Directory structure]
.nuxtrc
.npmrc
package.json
package-lock.json
yarn.lock
pnpm-lock.yaml
tsconfig.json
bun.lockb
```

In addition, any changes to files within `srcDir` will trigger a rebuild of the Vue client/server bundle. Nitro will always be rebuilt (though work is in progress to allow Nitro to announce its cacheable artifacts and their hashes).

::note
A maximum of 10 cache tarballs are kept.
::

## normalizeComponentNames

Ensure that auto-generated Vue component names match the full component name
you would use to auto-import the component.

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  experimental: {
    normalizeComponentNames: true
  }
})
```

By default, if you haven't set it manually, Vue will assign a component name that matches
the filename of the component.

```bash [Directory structure]
├─ components/
├─── SomeFolder/
├───── MyComponent.vue
```

In this case, the component name would be `MyComponent`, as far as Vue is concerned. If you wanted to use `<KeepAlive>` with it, or identify it in the Vue DevTools, you would need to use this component.

But in order to auto-import it, you would need to use `SomeFolderMyComponent`.

By setting `experimental.normalizeComponentNames`, these two values match, and Vue will generate a component name that matches the Nuxt pattern for component naming.
