---
title: Configuration
description: Nuxt is configured with sensible defaults to make you productive.
navigation.icon: i-ph-gear
---

By default, Nuxt is configured to cover most use cases. The [`nuxt.config.ts`](/docs/guide/directory-structure/nuxt-config) file can override or extend this default configuration.

## Nuxt Configuration

The [`nuxt.config.ts`](/docs/guide/directory-structure/nuxt-config) file is located at the root of a Nuxt project and can override or extend the application's behavior.

A minimal configuration file exports the `defineNuxtConfig` function containing an object with your configuration. The `defineNuxtConfig` helper is globally available without import.

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  // My Nuxt config
})
```

This file will often be mentioned in the documentation, for example to add custom scripts, register modules or change rendering modes.

::read-more{to="/docs/api/configuration/nuxt-config"}
Every option is described in the **Configuration Reference**.
::

::note
You don't have to use TypeScript to build an application with Nuxt. However, it is strongly recommended to use the `.ts` extension for the `nuxt.config` file. This way you can benefit from hints in your IDE to avoid typos and mistakes while editing your configuration.
::

### Environment Overrides

You can configure fully typed, per-environment overrides in your nuxt.config

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  $production: {
    routeRules: {
      '/**': { isr: true }
    }
  },
  $development: {
    //
  }
})
```

::tip{icon="i-ph-video" to="https://www.youtube.com/watch?v=DFZI2iVCrNc" target="_blank"}
Watch a video from Alexander Lichter about the env-aware `nuxt.config.ts`.
::

::note
If you're authoring layers, you can also use the `$meta` key to provide metadata that you or the consumers of your layer might use.
::

### Environment Variables and Private Tokens

The `runtimeConfig` API exposes values like environment variables to the rest of your application. By default, these keys are only available server-side. The keys within `runtimeConfig.public` are also available client-side.

Those values should be defined in `nuxt.config` and can be overridden using environment variables.

::code-group

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  runtimeConfig: {
    // The private keys which are only available server-side
    apiSecret: '123',
    // Keys within public are also exposed client-side
    public: {
      apiBase: '/api'
    }
  }
})
```

```bash [.env]
# This will override the value of apiSecret
NUXT_API_SECRET=api_secret_token
```

::

These variables are exposed to the rest of your application using the [`useRuntimeConfig()`](/docs/api/composables/use-runtime-config) composable.

```vue [pages/index.vue]
<script setup lang="ts">
const runtimeConfig = useRuntimeConfig()
</script>
```

:read-more{to="/docs/guide/going-further/runtime-config"}

## App Configuration

The `app.config.ts` file, located in the source directory (by default the root of the project), is used to expose public variables that can be determined at build time. Contrary to the `runtimeConfig` option, these can not be overridden using environment variables.

A minimal configuration file exports the `defineAppConfig` function containing an object with your configuration. The `defineAppConfig` helper is globally available without import.

```ts [app.config.ts]
export default defineAppConfig({
  title: 'Hello Nuxt',
  theme: {
    dark: true,
    colors: {
      primary: '#ff0000'
    }
  }
})
```

These variables are exposed to the rest of your application using the [`useAppConfig`](/docs/api/composables/use-app-config) composable.

```vue [pages/index.vue]
<script setup lang="ts">
const appConfig = useAppConfig()
</script>
```

:read-more{to="/docs/guide/directory-structure/app-config"}

## `runtimeConfig` vs `app.config`

As stated above, `runtimeConfig` and `app.config` are both used to expose variables to the rest of your application. To determine whether you should use one or the other, here are some guidelines:

- `runtimeConfig`: Private or public tokens that need to be specified after build using environment variables.
- `app.config`: Public tokens that are determined at build time, website configuration such as theme variant, title and any project config that are not sensitive.

Feature                        | `runtimeConfig`  | `app.config`
-------------------------------|------------------|-------------------
Client Side                    | Hydrated         | Bundled
Environment Variables          | ✅ Yes           | ❌ No
Reactive                       | ✅ Yes           | ✅ Yes
Types support                  | ✅ Partial       | ✅ Yes
Configuration per Request      | ❌ No            | ✅ Yes
Hot Module Replacement         | ❌ No            | ✅ Yes
Non primitive JS types         | ❌ No            | ✅ Yes

## External Configuration Files

Nuxt uses [`nuxt.config.ts`](/docs/guide/directory-structure/nuxt-config) file as the single source of truth for configurations and skips reading external configuration files. During the course of building your project, you may have a need to configure those. The following table highlights common configurations and, where applicable, how they can be configured with Nuxt.

Name                                         | Config File               |  How To Configure
---------------------------------------------|---------------------------|-------------------------
[Nitro](https://nitro.unjs.io)               | ~~`nitro.config.ts`~~     | Use [`nitro`](/docs/api/nuxt-config#nitro) key in `nuxt.config`
[PostCSS](https://postcss.org)               | ~~`postcss.config.js`~~   | Use [`postcss`](/docs/api/nuxt-config#postcss) key in `nuxt.config`
[Vite](https://vitejs.dev)                   | ~~`vite.config.ts`~~      | Use [`vite`](/docs/api/nuxt-config#vite) key in `nuxt.config`
[webpack](https://webpack.js.org)            | ~~`webpack.config.ts`~~   | Use [`webpack`](/docs/api/nuxt-config#webpack-1) key in `nuxt.config`

Here is a list of other common config files:

Name                                         | Config File             | How To Configure
---------------------------------------------|-------------------------|--------------------------
[TypeScript](https://www.typescriptlang.org) | `tsconfig.json`         | [More Info](/docs/guide/concepts/typescript#nuxttsconfigjson)
[ESLint](https://eslint.org)                 | `eslint.config.js`      | [More Info](https://eslint.org/docs/latest/use/configure/configuration-files)
[Prettier](https://prettier.io)              | `.prettierrc.json`      | [More Info](https://prettier.io/docs/en/configuration.html)
[Stylelint](https://stylelint.io)            | `.stylelintrc.json`     | [More Info](https://stylelint.io/user-guide/configure)
[TailwindCSS](https://tailwindcss.com)       |  `tailwind.config.js`   | [More Info](https://tailwindcss.nuxtjs.org/tailwind/config)
[Vitest](https://vitest.dev)                 | `vitest.config.ts`      | [More Info](https://vitest.dev/config)

## Vue Configuration

### With Vite

If you need to pass options to `@vitejs/plugin-vue` or `@vitejs/plugin-vue-jsx`, you can do this in your `nuxt.config` file.

- `vite.vue` for `@vitejs/plugin-vue`. Check available options [here](https://github.com/vitejs/vite-plugin-vue/tree/main/packages/plugin-vue).
- `vite.vueJsx` for `@vitejs/plugin-vue-jsx`. Check available options [here](https://github.com/vitejs/vite-plugin-vue/tree/main/packages/plugin-vue-jsx).

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  vite: {
    vue: {
      customElement: true
    },
    vueJsx: {
      mergeProps: true
    }
  }
})
```

:read-more{to="/docs/api/configuration/nuxt-config#vue"}

### With webpack

If you use webpack and need to configure `vue-loader`, you can do this using `webpack.loaders.vue` key inside your `nuxt.config` file. The available options are [defined here](https://github.com/vuejs/vue-loader/blob/main/src/index.ts#L32-L62).

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  webpack: {
    loaders: {
      vue: {
        hotReload: true,
      }
    }
  }
})
```

:read-more{to="/docs/api/configuration/nuxt-config#loaders"}

### Enabling Experimental Vue Features

You may need to enable experimental features in Vue, such as `propsDestructure`. Nuxt provides an easy way to do that in `nuxt.config.ts`, no matter which builder you are using:

```ts twoslash [nuxt.config.ts]
export default defineNuxtConfig({
  vue: {
    propsDestructure: true
  }
})
```

#### experimental `reactivityTransform` migration from Vue 3.4 and Nuxt 3.9

Since Nuxt 3.9 and Vue 3.4, `reactivityTransform` has been moved from Vue to Vue Macros which has a [Nuxt integration](https://vue-macros.dev/guide/nuxt-integration.html).

:read-more{to="/docs/api/configuration/nuxt-config#vue-1"}
