/*
 * Notifo.io
 *
 * @license
 * Copyright (c) Sebastian Stehle. All rights reserved.
 */

import * as React from 'react';
import { useDispatch } from 'react-redux';
import { toast } from 'react-toastify';
import { Button, Col, Label, Row } from 'reactstrap';
import { FormError, Icon, Loader, useEventCallback } from '@app/framework';
import { ChannelTemplateDto } from '@app/service';
import { createSmsTemplate, deleteSmsTemplate, loadSmsTemplates, useApp, useSmsTemplates } from '@app/state';
import { texts } from '@app/texts';
import { SmsTemplateCard } from './SmsTemplateCard';

export const SmsTemplatesPage = () => {
    const dispatch = useDispatch<any>();
    const app = useApp()!;
    const appId = app.id;
    const creating = useSmsTemplates(x => x.creating?.isRunning);
    const creatingError = useSmsTemplates(x => x.creating?.error);
    const deletingError = useSmsTemplates(x => x.deleting?.error);
    const smsTemplates = useSmsTemplates(x => x.templates);

    React.useEffect(() => {
        dispatch(loadSmsTemplates({ appId }));
    }, [dispatch, appId]);

    React.useEffect(() => {
        if (creatingError) {
            toast.error(creatingError.response);
        }
    }, [creatingError]);

    React.useEffect(() => {
        if (deletingError) {
            toast.error(deletingError.response);
        }
    }, [deletingError]);

    const doCreate = useEventCallback(() => {
        dispatch(createSmsTemplate({ appId }));
    });

    const doDelete = useEventCallback((template: ChannelTemplateDto) => {
        dispatch(deleteSmsTemplate({ appId, id: template.id }));
    });

    return (
        <div className='sms-templates'>
            <div className='align-items-center header'>
                <Row className='align-items-center'>
                    <Col xs='auto'>
                        <h2 className='truncate'>{texts.smsTemplates.header}</h2>
                    </Col>
                    <Col>
                        <Loader visible={smsTemplates.isLoading} />
                    </Col>
                    <Col xs='auto'>
                        <Button color='success' onClick={doCreate}>
                            <Loader light small visible={creating} /> <Icon type='add' /> {texts.smsTemplates.create}
                        </Button>
                    </Col>
                </Row>
            </div>

            <FormError error={smsTemplates.error} />

            {smsTemplates.items &&
                <>
                    {smsTemplates.items.map(template => (
                        <SmsTemplateCard key={template.id} template={template}
                            onDelete={doDelete}
                        />
                    ))}
                </>
            }

            {smsTemplates.isLoaded && smsTemplates.items?.length === 0 &&
                <div className='empty-button'>
                    <Label>{texts.smsTemplates.notFound}</Label>

                    <Button size='lg' color='success' disabled={creating} onClick={doCreate}>
                        <Loader light small visible={creating} /> <Icon type='add' /> {texts.smsTemplates.notFoundButton}
                    </Button>
                </div>
            }
        </div>
    );
};
