local t = require('test.testutil')
local n = require('test.functional.testnvim')()
local Screen = require('test.functional.ui.screen')

local clear = n.clear
local command = n.command
local feed = n.feed
local feed_command = n.feed_command
local insert = n.insert
local fn = n.fn
local api = n.api
local split = vim.split
local dedent = t.dedent

describe('multibyte rendering', function()
  local screen
  before_each(function()
    clear()
    screen = Screen.new(60, 6)
    screen:attach({ rgb = true })
  end)

  it('works with composed char at start of line', function()
    insert([[
      ̊
      x]])
    feed('gg')
    -- verify the modifier in fact is alone
    feed_command('ascii')
    screen:expect([[
      ^ ̊                                                           |
      x                                                           |
      {1:~                                                           }|*3
      < ̊> 778, Hex 030a, Octal 1412                               |
    ]])

    -- a char inserted before will spontaneously merge with it
    feed('ia<esc>')
    feed_command('ascii')
    screen:expect([[
      ^å                                                           |
      x                                                           |
      {1:~                                                           }|*3
      <a>  97,  Hex 61,  Octal 141 < ̊> 778, Hex 030a, Octal 1412  |
    ]])
  end)

  it('works with doublewidth char at end of line', function()
    feed('58a <esc>a馬<esc>')
    screen:expect([[
                                                                ^馬|
      {1:~                                                           }|*4
                                                                  |
    ]])

    feed('i <esc>')
    screen:expect([[
                                                                ^ {1:>}|
      馬                                                          |
      {1:~                                                           }|*3
                                                                  |
    ]])

    feed('l')
    screen:expect([[
                                                                 {1:>}|
      ^馬                                                          |
      {1:~                                                           }|*3
                                                                  |
    ]])
  end)

  it('clears left half of double-width char when right half is overdrawn', function()
    feed('o-馬<esc>ggiab ')
    screen:expect([[
      ab ^                                                         |
      -馬                                                         |
      {1:~                                                           }|*3
      {5:-- INSERT --}                                                |
    ]])

    -- check double-width char is temporarily hidden when overlapped
    fn.complete(4, { 'xx', 'yy' })
    screen:expect([[
      ab xx^                                                       |
      - {12: xx             }                                          |
      {1:~ }{4: yy             }{1:                                          }|
      {1:~                                                           }|*2
      {5:-- INSERT --}                                                |
    ]])

    -- check it is properly restored
    feed('z')
    screen:expect([[
      ab xxz^                                                      |
      -馬                                                         |
      {1:~                                                           }|*3
      {5:-- INSERT --}                                                |
    ]])
  end)

  it('no stray chars when splitting left of window with double-width chars', function()
    api.nvim_buf_set_lines(0, 0, -1, true, {
      ('口'):rep(16),
      'a' .. ('口'):rep(16),
      'aa' .. ('口'):rep(16),
      'aaa' .. ('口'):rep(16),
      'aaaa' .. ('口'):rep(16),
    })
    screen:expect([[
      ^口口口口口口口口口口口口口口口口                            |
      a口口口口口口口口口口口口口口口口                           |
      aa口口口口口口口口口口口口口口口口                          |
      aaa口口口口口口口口口口口口口口口口                         |
      aaaa口口口口口口口口口口口口口口口口                        |
                                                                  |
    ]])

    command('20vnew')
    screen:expect([[
      ^                    │口口口口口口口口口口口口口口口口       |
      {1:~                   }│a口口口口口口口口口口口口口口口口      |
      {1:~                   }│aa口口口口口口口口口口口口口口口口     |
      {1:~                   }│aaa口口口口口口口口口口口口口口口口    |
      {3:[No Name]            }{2:[No Name] [+]                          }|
                                                                  |
    ]])
  end)

  it('0xffff is shown as 4 hex digits', function()
    command([[call setline(1, "\uFFFF!!!")]])
    feed('$')
    screen:expect {
      grid = [[
      {18:<ffff>}!!^!                                                   |
      {1:~                                                           }|*4
                                                                  |
    ]],
    }
  end)

  it('works with a lot of unicode (zalgo) text', function()
    screen:try_resize(65, 10)
    api.nvim_buf_set_lines(
      0,
      0,
      -1,
      true,
      split(
        dedent [[
      L̓̉̑̒̌̚ơ̗̌̒̄̀ŕ̈̈̎̐̕è̇̅̄̄̐m̖̟̟̅̄̚ ̛̓̑̆̇̍i̗̟̞̜̅̐p̗̞̜̉̆̕s̟̜̘̍̑̏ū̟̞̎̃̉ḿ̘̙́́̐ ̖̍̌̇̉̚d̞̄̃̒̉̎ò́̌̌̂̐l̞̀̄̆̌̚ȯ̖̞̋̀̐r̓̇̌̃̃̚ ̗̘̀̏̍́s̜̀̎̎̑̕i̟̗̐̄̄̚t̝̎̆̓̐̒ ̘̇̔̓̊̚ȃ̛̟̗̏̅m̜̟̙̞̈̓é̘̞̟̔̆t̝̂̂̈̑̔,̜̜̖̅̄̍ ̛̗̊̓̆̚c̟̍̆̍̈̔ȯ̖̖̝̑̀n̜̟̎̊̃̚s̟̏̇̎̒̚e̙̐̈̓̌̚c̙̍̈̏̅̕ť̇̄̇̆̓e̛̓̌̈̓̈t̟̍̀̉̆̅u̝̞̎̂̄̚r̘̀̅̈̅̐ ̝̞̓́̇̉ã̏̀̆̅̕d̛̆̐̉̆̋ȉ̞̟̍̃̚p̛̜̊̍̂̓ȋ̏̅̃̋̚ṥ̛̏̃̕č̛̞̝̀̂í̗̘̌́̎n̔̎́̒̂̕ǧ̗̜̋̇̂ ̛̜̔̄̎̃ê̛̔̆̇̕l̘̝̏̐̊̏ĩ̛̍̏̏̄t̟̐́̀̐̎,̙̘̍̆̉̐ ̋̂̏̄̌̅s̙̓̌̈́̇e̛̗̋̒̎̏d̜̗̊̍̊̚
      ď̘̋̌̌̕ǒ̝̗̔̇̕ ̙̍́̄̄̉è̛̛̞̌̌i̜̖̐̈̆̚ȕ̇̈̓̃̓ŝ̛̞̙̉̋m̜̐̂̄̋̂ȯ̈̎̎̅̕d̜̙̓̔̋̑ ̞̗̄̂̂̚t̝̊́̃́̄e̛̘̜̞̓̑m̊̅̏̉̌̕p̛̈̂̇̀̐ỏ̙̘̈̉̔r̘̞̋̍̃̚ ̝̄̀̇̅̇ỉ̛̖̍̓̈n̛̛̝̎̕̕c̛̛̊̅́̐ĭ̗̓̀̍̐d̞̜̋̐̅̚i̟̙̇̄̊̄d̞̊̂̀̇̚ủ̝̉̑̃̕n̜̏̇̄̐̋ť̗̜̞̋̉ ̝̒̓̌̓̚ȕ̖̙̀̚̕t̖̘̎̉̂̌ ̛̝̄̍̌̂l̛̟̝̃̑̋á̛̝̝̔̅b̝̙̜̗̅̒ơ̖̌̒̄̆r̒̇̓̎̈̄e̛̛̖̅̏̇ ̖̗̜̝̃́e̛̛̘̅̔̌ẗ̛̙̗̐̕ ̖̟̇̋̌̈d̞̙̀̉̑̕ŏ̝̂́̐̑l̞̟̗̓̓̀ơ̘̎̃̄̂r̗̗̖̔̆̍ẻ̖̝̞̋̅ ̜̌̇̍̈̊m̈̉̇̄̒̀a̜̞̘̔̅̆g̗̖̈̃̈̉n̙̖̄̈̉̄â̛̝̜̄̃ ̛́̎̕̕̚ā̊́́̆̌l̟̙̞̃̒́i̖̇̎̃̀̋q̟̇̒̆́̊ủ́̌̇̑̚ã̛̘̉̐̚.̛́̏̐̍̊
      U̝̙̎̈̐̆t̜̍̌̀̔̏ ̞̉̍̇̈̃e̟̟̊̄̕̕n̝̜̒̓̆̕i̖̒̌̅̇̚m̞̊̃̔̊̂ ̛̜̊̎̄̂a̘̜̋̒̚̚d̟̊̎̇̂̍ ̜̖̏̑̉̕m̜̒̎̅̄̚i̝̖̓̂̍̕n̙̉̒̑̀̔ỉ̖̝̌̒́m̛̖̘̅̆̎ ̖̉̎̒̌̕v̖̞̀̔́̎e̖̙̗̒̎̉n̛̗̝̎̀̂ȉ̞̗̒̕̚ȧ̟̜̝̅̚m̆̉̐̐̇̈,̏̐̎́̍́ ̜̞̙̘̏̆q̙̖̙̅̓̂ủ̇́̀̔̚í̙̟̟̏̐s̖̝̍̏̂̇ ̛̘̋̈̕̕ń̛̞̜̜̎o̗̜̔̔̈̆s̞̘̘̄̒̋t̛̅̋́̔̈ȓ̓̒́̇̅ủ̜̄̃̒̍d̙̝̘̊̏̚ ̛̟̞̄́̔e̛̗̝̍̃̀x̞̖̃̄̂̅e̖̅̇̐̔̃r̗̞̖̔̎̚c̘̜̖̆̊̏ï̙̝̙̂̕t̖̏́̓̋̂ă̖̄̆̑̒t̜̟̍̉̑̏i̛̞̞̘̒̑ǒ̜̆̅̃̉ṅ̖̜̒̎̚
      u̗̞̓̔̈̏ĺ̟̝́̎̚l̛̜̅̌̎̆a̒̑̆̔̇̃m̜̗̈̊̎̚ċ̘̋̇̂̚ơ̟̖̊́̕ ̖̟̍̉̏̚l̙̔̓̀̅̏ä̞̗̘̙̅ḃ̟̎̄̃̕o̞̎̓̓̓̚r̗̜̊̓̈̒ï̗̜̃̃̅s̀̒̌̂̎̂ ̖̗̗̋̎̐n̝̟̝̘̄̚i̜̒̀̒̐̕s̘̘̄̊̃̀ī̘̜̏̌̕ ̗̖̞̐̈̒ư̙̞̄́̌t̟̘̖̙̊̚ ̌̅̋̆̚̚ä̇̊̇̕̕l̝̞̘̋̔̅i̍̋́̆̑̈q̛̆̐̈̐̚ư̏̆̊́̚î̜̝̑́̊p̗̓̅̑̆̏ ̆́̓̔̋̋e̟̊̋̏̓̚x̗̍̑̊̎̈ ̟̞̆̄̂̍ë̄̎̄̃̅a̛̜̅́̃̈ ̔̋̀̎̐̀c̖̖̍̀̒̂ơ̛̙̖̄̒m̘̔̍̏̆̕ḿ̖̙̝̏̂ȍ̓̋̈̀̕d̆̂̊̅̓̚o̖̔̌̑̚̕ ̙̆́̔̊̒c̖̘̖̀̄̍o̓̄̑̐̓̒ñ̞̒̎̈̚s̞̜̘̈̄̄e̙̊̀̇̌̋q̐̒̓́̔̃ư̗̟̔̔̚å̖̙̞̄̏t̛̙̟̒̇̏.̙̗̓̃̓̎
      D̜̖̆̏̌̌ư̑̃̌̍̕i̝̊̊̊̊̄s̛̙̒́̌̇ ̛̃̔̄̆̌ă̘̔̅̅̀ú̟̟̟̃̃t̟̂̄̈̈̃e̘̅̌̒̂̆ ̖̟̐̉̉̌î̟̟̙̜̇r̛̙̞̗̄̌ú̗̗̃̌̎r̛̙̘̉̊̕e̒̐̔̃̓̋ ̊̊̍̋̑̉d̛̝̙̉̀̓o̘̜̐̐̓̐l̞̋̌̆̍́o̊̊̐̃̃̚ṙ̛̖̘̃̕ ̞̊̀̍̒̕ȉ́̑̐̇̅ǹ̜̗̜̞̏ ̛̜̐̄̄̚r̜̖̈̇̅̋ĕ̗̉̃̔̚p̟̝̀̓̔̆r̜̈̆̇̃̃e̘̔̔̏̎̓h̗̒̉̑̆̚ė̛̘̘̈̐n̘̂̀̒̕̕d̗̅̂̋̅́ê̗̜̜̜̕r̟̋̄̐̅̂i̛̔̌̒̂̕t̛̗̓̎̀̎ ̙̗̀̉̂̚ȉ̟̗̐̓̚n̙̂̍̏̓̉ ̙̘̊̋̍̕v̜̖̀̎̆̐ő̜̆̉̃̎l̑̋̒̉̔̆ư̙̓̓́̚p̝̘̖̎̏̒t̛̘̝̞̂̓ȁ̘̆̔́̊t̖̝̉̒̐̎e̞̟̋̀̅̄ ̆̌̃̀̑̔v̝̘̝̍̀̇ȅ̝̊̄̓̕l̞̝̑̔̂̋ĭ̝̄̅̆̍t̝̜̉̂̈̇
      ē̟̊̇̕̚s̖̘̘̒̄̑s̛̘̀̊̆̇e̛̝̘̒̏̚ ̉̅̑̂̐̎c̛̟̙̎̋̓i̜̇̒̏̆̆l̟̄́̆̊̌l̍̊̋̃̆̌ủ̗̙̒̔̚m̛̘̘̖̅̍ ̖̙̈̎̂̕d̞̟̏̋̈̔ơ̟̝̌̃̄l̗̙̝̂̉̒õ̒̃̄̄̚ŕ̗̏̏̊̍ê̞̝̞̋̈ ̜̔̒̎̃̚e̞̟̞̒̃̄ư̖̏̄̑̃ ̛̗̜̄̓̎f̛̖̞̅̓̃ü̞̏̆̋̕g̜̝̞̑̑̆i̛̘̐̐̅̚à̜̖̌̆̎t̙̙̎̉̂̍ ̋̔̈̎̎̉n̞̓́̔̊̕ư̘̅̋̔̚l̗̍̒̄̀̚l̞̗̘̙̓̍â̘̔̒̎̚ ̖̓̋̉̃̆p̛̛̘̋̌̀ä̙̔́̒̕r̟̟̖̋̐̋ì̗̙̎̓̓ȃ̔̋̑̚̕t̄́̎̓̂̋ư̏̈̂̑̃r̖̓̋̊̚̚.̒̆̑̆̊̎ ̘̜̍̐̂̚E̞̅̐̇́̂x̄́̈̌̉̕ć̘̃̉̃̕è̘̂̑̏̑p̝̘̑̂̌̆t̔̐̅̍̌̂ȇ̞̈̐̚̕ű̝̞̜́̚ŕ̗̝̉̆́
      š̟́̔̏̀ȉ̝̟̝̏̅n̑̆̇̒̆̚t̝̒́̅̋̏ ̗̑̌̋̇̚ơ̙̗̟̆̅c̙̞̙̎̊̎c̘̟̍̔̊̊a̛̒̓̉́̐e̜̘̙̒̅̇ć̝̝̂̇̕ả̓̍̎̂̚t̗̗̗̟̒̃ ̘̒̓̐̇́c̟̞̉̐̓̄ȕ̙̗̅́̏p̛̍̋̈́̅i̖̓̒̍̈̄d̞̃̈̌̆̐a̛̗̝̎̋̉t̞̙̀̊̆̇a̛̙̒̆̉̚t̜̟̘̉̓̚ ̝̘̗̐̇̕n̛̘̑̏̂́ō̑̋̉̏́ň̞̊̆̄̃ ̙̙̙̜̄̏p̒̆̋̋̓̏r̖̖̅̉́̚ơ̜̆̑̈̚i̟̒̀̃̂̌d̛̏̃̍̋̚ë̖̞̙̗̓n̛̘̓̒̅̎t̟̗̙̊̆̚,̘̙̔̊̚̕ ̟̗̘̜̑̔s̜̝̍̀̓̌û̞̙̅̇́n̘̗̝̒̃̎t̗̅̀̅̊̈ ̗̖̅̅̀̄i̛̖̍̅̋̂n̙̝̓̓̎̚ ̞̋̅̋̃̚c̗̒̀̆̌̎ū̞̂̑̌̓ĺ̛̐̍̑́p̝̆̌̎̈̚a̖̙̒̅̈̌ ̝̝̜̂̈̀q̝̖̔̍̒̚ư̔̐̂̎̊ǐ̛̟̖̘̕
      o̖̜̔̋̅̚f̛̊̀̉́̕f̏̉̀̔̃̃i̘̍̎̐̔̎c̙̅̑̂̐̅ȋ̛̜̀̒̚a̋̍̇̏̀̋ ̖̘̒̅̃̒d̗̘̓̈̇̋é̝́̎̒̄š̙̒̊̉̋e̖̓̐̀̍̕r̗̞̂̅̇̄ù̘̇̐̉̀n̐̑̀̄̍̐t̟̀̂̊̄̚ ̟̝̂̍̏́m̜̗̈̂̏̚ő̞̊̑̇̒l̘̑̏́̔̄l̛̛̇̃̋̊i̓̋̒̃̉̌t̛̗̜̏̀̋ ̙̟̒̂̌̐a̙̝̔̆̏̅n̝̙̙̗̆̅i̍̔́̊̃̕m̖̝̟̒̍̚ ̛̃̃̑̌́ǐ̘̉̔̅̚d̝̗̀̌̏̒ ̖̝̓̑̊̚ȇ̞̟̖̌̕š̙̙̈̔̀t̂̉̒̍̄̄ ̝̗̊̋̌̄l̛̞̜̙̘̔å̝̍̂̍̅b̜̆̇̈̉̌ǒ̜̙̎̃̆r̝̀̄̍́̕ư̋̊́̊̕m̜̗̒̐̕̚.̟̘̀̒̌̚]],
        '\n'
      )
    )

    -- tests that we can handle overflow of the buffer
    -- for redraw events (4096 bytes) gracefully
    screen:expect {
      grid = [[
      ^L̓̉̑̒̌̚ơ̗̌̒̄̀ŕ̈̈̎̐̕è̇̅̄̄̐m̖̟̟̅̄̚ ̛̓̑̆̇̍i̗̟̞̜̅̐p̗̞̜̉̆̕s̟̜̘̍̑̏ū̟̞̎̃̉ḿ̘̙́́̐ ̖̍̌̇̉̚d̞̄̃̒̉̎ò́̌̌̂̐l̞̀̄̆̌̚ȯ̖̞̋̀̐r̓̇̌̃̃̚ ̗̘̀̏̍́s̜̀̎̎̑̕i̟̗̐̄̄̚t̝̎̆̓̐̒ ̘̇̔̓̊̚ȃ̛̟̗̏̅m̜̟̙̞̈̓é̘̞̟̔̆t̝̂̂̈̑̔,̜̜̖̅̄̍ ̛̗̊̓̆̚c̟̍̆̍̈̔ȯ̖̖̝̑̀n̜̟̎̊̃̚s̟̏̇̎̒̚e̙̐̈̓̌̚c̙̍̈̏̅̕ť̇̄̇̆̓e̛̓̌̈̓̈t̟̍̀̉̆̅u̝̞̎̂̄̚r̘̀̅̈̅̐ ̝̞̓́̇̉ã̏̀̆̅̕d̛̆̐̉̆̋ȉ̞̟̍̃̚p̛̜̊̍̂̓ȋ̏̅̃̋̚ṥ̛̏̃̕č̛̞̝̀̂í̗̘̌́̎n̔̎́̒̂̕ǧ̗̜̋̇̂ ̛̜̔̄̎̃ê̛̔̆̇̕l̘̝̏̐̊̏ĩ̛̍̏̏̄t̟̐́̀̐̎,̙̘̍̆̉̐ ̋̂̏̄̌̅s̙̓̌̈́̇e̛̗̋̒̎̏d̜̗̊̍̊̚     |
      ď̘̋̌̌̕ǒ̝̗̔̇̕ ̙̍́̄̄̉è̛̛̞̌̌i̜̖̐̈̆̚ȕ̇̈̓̃̓ŝ̛̞̙̉̋m̜̐̂̄̋̂ȯ̈̎̎̅̕d̜̙̓̔̋̑ ̞̗̄̂̂̚t̝̊́̃́̄e̛̘̜̞̓̑m̊̅̏̉̌̕p̛̈̂̇̀̐ỏ̙̘̈̉̔r̘̞̋̍̃̚ ̝̄̀̇̅̇ỉ̛̖̍̓̈n̛̛̝̎̕̕c̛̛̊̅́̐ĭ̗̓̀̍̐d̞̜̋̐̅̚i̟̙̇̄̊̄d̞̊̂̀̇̚ủ̝̉̑̃̕n̜̏̇̄̐̋ť̗̜̞̋̉ ̝̒̓̌̓̚ȕ̖̙̀̚̕t̖̘̎̉̂̌ ̛̝̄̍̌̂l̛̟̝̃̑̋á̛̝̝̔̅b̝̙̜̗̅̒ơ̖̌̒̄̆r̒̇̓̎̈̄e̛̛̖̅̏̇ ̖̗̜̝̃́e̛̛̘̅̔̌ẗ̛̙̗̐̕ ̖̟̇̋̌̈d̞̙̀̉̑̕ŏ̝̂́̐̑l̞̟̗̓̓̀ơ̘̎̃̄̂r̗̗̖̔̆̍ẻ̖̝̞̋̅ ̜̌̇̍̈̊m̈̉̇̄̒̀a̜̞̘̔̅̆g̗̖̈̃̈̉n̙̖̄̈̉̄â̛̝̜̄̃ ̛́̎̕̕̚ā̊́́̆̌l̟̙̞̃̒́i̖̇̎̃̀̋q̟̇̒̆́̊ủ́̌̇̑̚ã̛̘̉̐̚.̛́̏̐̍̊   |
      U̝̙̎̈̐̆t̜̍̌̀̔̏ ̞̉̍̇̈̃e̟̟̊̄̕̕n̝̜̒̓̆̕i̖̒̌̅̇̚m̞̊̃̔̊̂ ̛̜̊̎̄̂a̘̜̋̒̚̚d̟̊̎̇̂̍ ̜̖̏̑̉̕m̜̒̎̅̄̚i̝̖̓̂̍̕n̙̉̒̑̀̔ỉ̖̝̌̒́m̛̖̘̅̆̎ ̖̉̎̒̌̕v̖̞̀̔́̎e̖̙̗̒̎̉n̛̗̝̎̀̂ȉ̞̗̒̕̚ȧ̟̜̝̅̚m̆̉̐̐̇̈,̏̐̎́̍́ ̜̞̙̘̏̆q̙̖̙̅̓̂ủ̇́̀̔̚í̙̟̟̏̐s̖̝̍̏̂̇ ̛̘̋̈̕̕ń̛̞̜̜̎o̗̜̔̔̈̆s̞̘̘̄̒̋t̛̅̋́̔̈ȓ̓̒́̇̅ủ̜̄̃̒̍d̙̝̘̊̏̚ ̛̟̞̄́̔e̛̗̝̍̃̀x̞̖̃̄̂̅e̖̅̇̐̔̃r̗̞̖̔̎̚c̘̜̖̆̊̏ï̙̝̙̂̕t̖̏́̓̋̂ă̖̄̆̑̒t̜̟̍̉̑̏i̛̞̞̘̒̑ǒ̜̆̅̃̉ṅ̖̜̒̎̚               |
      u̗̞̓̔̈̏ĺ̟̝́̎̚l̛̜̅̌̎̆a̒̑̆̔̇̃m̜̗̈̊̎̚ċ̘̋̇̂̚ơ̟̖̊́̕ ̖̟̍̉̏̚l̙̔̓̀̅̏ä̞̗̘̙̅ḃ̟̎̄̃̕o̞̎̓̓̓̚r̗̜̊̓̈̒ï̗̜̃̃̅s̀̒̌̂̎̂ ̖̗̗̋̎̐n̝̟̝̘̄̚i̜̒̀̒̐̕s̘̘̄̊̃̀ī̘̜̏̌̕ ̗̖̞̐̈̒ư̙̞̄́̌t̟̘̖̙̊̚ ̌̅̋̆̚̚ä̇̊̇̕̕l̝̞̘̋̔̅i̍̋́̆̑̈q̛̆̐̈̐̚ư̏̆̊́̚î̜̝̑́̊p̗̓̅̑̆̏ ̆́̓̔̋̋e̟̊̋̏̓̚x̗̍̑̊̎̈ ̟̞̆̄̂̍ë̄̎̄̃̅a̛̜̅́̃̈ ̔̋̀̎̐̀c̖̖̍̀̒̂ơ̛̙̖̄̒m̘̔̍̏̆̕ḿ̖̙̝̏̂ȍ̓̋̈̀̕d̆̂̊̅̓̚o̖̔̌̑̚̕ ̙̆́̔̊̒c̖̘̖̀̄̍o̓̄̑̐̓̒ñ̞̒̎̈̚s̞̜̘̈̄̄e̙̊̀̇̌̋q̐̒̓́̔̃ư̗̟̔̔̚å̖̙̞̄̏t̛̙̟̒̇̏.̙̗̓̃̓̎         |
      D̜̖̆̏̌̌ư̑̃̌̍̕i̝̊̊̊̊̄s̛̙̒́̌̇ ̛̃̔̄̆̌ă̘̔̅̅̀ú̟̟̟̃̃t̟̂̄̈̈̃e̘̅̌̒̂̆ ̖̟̐̉̉̌î̟̟̙̜̇r̛̙̞̗̄̌ú̗̗̃̌̎r̛̙̘̉̊̕e̒̐̔̃̓̋ ̊̊̍̋̑̉d̛̝̙̉̀̓o̘̜̐̐̓̐l̞̋̌̆̍́o̊̊̐̃̃̚ṙ̛̖̘̃̕ ̞̊̀̍̒̕ȉ́̑̐̇̅ǹ̜̗̜̞̏ ̛̜̐̄̄̚r̜̖̈̇̅̋ĕ̗̉̃̔̚p̟̝̀̓̔̆r̜̈̆̇̃̃e̘̔̔̏̎̓h̗̒̉̑̆̚ė̛̘̘̈̐n̘̂̀̒̕̕d̗̅̂̋̅́ê̗̜̜̜̕r̟̋̄̐̅̂i̛̔̌̒̂̕t̛̗̓̎̀̎ ̙̗̀̉̂̚ȉ̟̗̐̓̚n̙̂̍̏̓̉ ̙̘̊̋̍̕v̜̖̀̎̆̐ő̜̆̉̃̎l̑̋̒̉̔̆ư̙̓̓́̚p̝̘̖̎̏̒t̛̘̝̞̂̓ȁ̘̆̔́̊t̖̝̉̒̐̎e̞̟̋̀̅̄ ̆̌̃̀̑̔v̝̘̝̍̀̇ȅ̝̊̄̓̕l̞̝̑̔̂̋ĭ̝̄̅̆̍t̝̜̉̂̈̇        |
      ē̟̊̇̕̚s̖̘̘̒̄̑s̛̘̀̊̆̇e̛̝̘̒̏̚ ̉̅̑̂̐̎c̛̟̙̎̋̓i̜̇̒̏̆̆l̟̄́̆̊̌l̍̊̋̃̆̌ủ̗̙̒̔̚m̛̘̘̖̅̍ ̖̙̈̎̂̕d̞̟̏̋̈̔ơ̟̝̌̃̄l̗̙̝̂̉̒õ̒̃̄̄̚ŕ̗̏̏̊̍ê̞̝̞̋̈ ̜̔̒̎̃̚e̞̟̞̒̃̄ư̖̏̄̑̃ ̛̗̜̄̓̎f̛̖̞̅̓̃ü̞̏̆̋̕g̜̝̞̑̑̆i̛̘̐̐̅̚à̜̖̌̆̎t̙̙̎̉̂̍ ̋̔̈̎̎̉n̞̓́̔̊̕ư̘̅̋̔̚l̗̍̒̄̀̚l̞̗̘̙̓̍â̘̔̒̎̚ ̖̓̋̉̃̆p̛̛̘̋̌̀ä̙̔́̒̕r̟̟̖̋̐̋ì̗̙̎̓̓ȃ̔̋̑̚̕t̄́̎̓̂̋ư̏̈̂̑̃r̖̓̋̊̚̚.̒̆̑̆̊̎ ̘̜̍̐̂̚E̞̅̐̇́̂x̄́̈̌̉̕ć̘̃̉̃̕è̘̂̑̏̑p̝̘̑̂̌̆t̔̐̅̍̌̂ȇ̞̈̐̚̕ű̝̞̜́̚ŕ̗̝̉̆́           |
      š̟́̔̏̀ȉ̝̟̝̏̅n̑̆̇̒̆̚t̝̒́̅̋̏ ̗̑̌̋̇̚ơ̙̗̟̆̅c̙̞̙̎̊̎c̘̟̍̔̊̊a̛̒̓̉́̐e̜̘̙̒̅̇ć̝̝̂̇̕ả̓̍̎̂̚t̗̗̗̟̒̃ ̘̒̓̐̇́c̟̞̉̐̓̄ȕ̙̗̅́̏p̛̍̋̈́̅i̖̓̒̍̈̄d̞̃̈̌̆̐a̛̗̝̎̋̉t̞̙̀̊̆̇a̛̙̒̆̉̚t̜̟̘̉̓̚ ̝̘̗̐̇̕n̛̘̑̏̂́ō̑̋̉̏́ň̞̊̆̄̃ ̙̙̙̜̄̏p̒̆̋̋̓̏r̖̖̅̉́̚ơ̜̆̑̈̚i̟̒̀̃̂̌d̛̏̃̍̋̚ë̖̞̙̗̓n̛̘̓̒̅̎t̟̗̙̊̆̚,̘̙̔̊̚̕ ̟̗̘̜̑̔s̜̝̍̀̓̌û̞̙̅̇́n̘̗̝̒̃̎t̗̅̀̅̊̈ ̗̖̅̅̀̄i̛̖̍̅̋̂n̙̝̓̓̎̚ ̞̋̅̋̃̚c̗̒̀̆̌̎ū̞̂̑̌̓ĺ̛̐̍̑́p̝̆̌̎̈̚a̖̙̒̅̈̌ ̝̝̜̂̈̀q̝̖̔̍̒̚ư̔̐̂̎̊ǐ̛̟̖̘̕          |
      o̖̜̔̋̅̚f̛̊̀̉́̕f̏̉̀̔̃̃i̘̍̎̐̔̎c̙̅̑̂̐̅ȋ̛̜̀̒̚a̋̍̇̏̀̋ ̖̘̒̅̃̒d̗̘̓̈̇̋é̝́̎̒̄š̙̒̊̉̋e̖̓̐̀̍̕r̗̞̂̅̇̄ù̘̇̐̉̀n̐̑̀̄̍̐t̟̀̂̊̄̚ ̟̝̂̍̏́m̜̗̈̂̏̚ő̞̊̑̇̒l̘̑̏́̔̄l̛̛̇̃̋̊i̓̋̒̃̉̌t̛̗̜̏̀̋ ̙̟̒̂̌̐a̙̝̔̆̏̅n̝̙̙̗̆̅i̍̔́̊̃̕m̖̝̟̒̍̚ ̛̃̃̑̌́ǐ̘̉̔̅̚d̝̗̀̌̏̒ ̖̝̓̑̊̚ȇ̞̟̖̌̕š̙̙̈̔̀t̂̉̒̍̄̄ ̝̗̊̋̌̄l̛̞̜̙̘̔å̝̍̂̍̅b̜̆̇̈̉̌ǒ̜̙̎̃̆r̝̀̄̍́̕ư̋̊́̊̕m̜̗̒̐̕̚.̟̘̀̒̌̚                     |
      {1:~                                                                }|
                                                                       |
    ]],
    }

    -- nvim will reset the zalgo text^W^W glyph cache if it gets too full.
    -- this should be exceedingly rare, but fake it to make sure it works
    api.nvim__invalidate_glyph_cache()
    screen:expect {
      grid = [[
      ^L̓̉̑̒̌̚ơ̗̌̒̄̀ŕ̈̈̎̐̕è̇̅̄̄̐m̖̟̟̅̄̚ ̛̓̑̆̇̍i̗̟̞̜̅̐p̗̞̜̉̆̕s̟̜̘̍̑̏ū̟̞̎̃̉ḿ̘̙́́̐ ̖̍̌̇̉̚d̞̄̃̒̉̎ò́̌̌̂̐l̞̀̄̆̌̚ȯ̖̞̋̀̐r̓̇̌̃̃̚ ̗̘̀̏̍́s̜̀̎̎̑̕i̟̗̐̄̄̚t̝̎̆̓̐̒ ̘̇̔̓̊̚ȃ̛̟̗̏̅m̜̟̙̞̈̓é̘̞̟̔̆t̝̂̂̈̑̔,̜̜̖̅̄̍ ̛̗̊̓̆̚c̟̍̆̍̈̔ȯ̖̖̝̑̀n̜̟̎̊̃̚s̟̏̇̎̒̚e̙̐̈̓̌̚c̙̍̈̏̅̕ť̇̄̇̆̓e̛̓̌̈̓̈t̟̍̀̉̆̅u̝̞̎̂̄̚r̘̀̅̈̅̐ ̝̞̓́̇̉ã̏̀̆̅̕d̛̆̐̉̆̋ȉ̞̟̍̃̚p̛̜̊̍̂̓ȋ̏̅̃̋̚ṥ̛̏̃̕č̛̞̝̀̂í̗̘̌́̎n̔̎́̒̂̕ǧ̗̜̋̇̂ ̛̜̔̄̎̃ê̛̔̆̇̕l̘̝̏̐̊̏ĩ̛̍̏̏̄t̟̐́̀̐̎,̙̘̍̆̉̐ ̋̂̏̄̌̅s̙̓̌̈́̇e̛̗̋̒̎̏d̜̗̊̍̊̚     |
      ď̘̋̌̌̕ǒ̝̗̔̇̕ ̙̍́̄̄̉è̛̛̞̌̌i̜̖̐̈̆̚ȕ̇̈̓̃̓ŝ̛̞̙̉̋m̜̐̂̄̋̂ȯ̈̎̎̅̕d̜̙̓̔̋̑ ̞̗̄̂̂̚t̝̊́̃́̄e̛̘̜̞̓̑m̊̅̏̉̌̕p̛̈̂̇̀̐ỏ̙̘̈̉̔r̘̞̋̍̃̚ ̝̄̀̇̅̇ỉ̛̖̍̓̈n̛̛̝̎̕̕c̛̛̊̅́̐ĭ̗̓̀̍̐d̞̜̋̐̅̚i̟̙̇̄̊̄d̞̊̂̀̇̚ủ̝̉̑̃̕n̜̏̇̄̐̋ť̗̜̞̋̉ ̝̒̓̌̓̚ȕ̖̙̀̚̕t̖̘̎̉̂̌ ̛̝̄̍̌̂l̛̟̝̃̑̋á̛̝̝̔̅b̝̙̜̗̅̒ơ̖̌̒̄̆r̒̇̓̎̈̄e̛̛̖̅̏̇ ̖̗̜̝̃́e̛̛̘̅̔̌ẗ̛̙̗̐̕ ̖̟̇̋̌̈d̞̙̀̉̑̕ŏ̝̂́̐̑l̞̟̗̓̓̀ơ̘̎̃̄̂r̗̗̖̔̆̍ẻ̖̝̞̋̅ ̜̌̇̍̈̊m̈̉̇̄̒̀a̜̞̘̔̅̆g̗̖̈̃̈̉n̙̖̄̈̉̄â̛̝̜̄̃ ̛́̎̕̕̚ā̊́́̆̌l̟̙̞̃̒́i̖̇̎̃̀̋q̟̇̒̆́̊ủ́̌̇̑̚ã̛̘̉̐̚.̛́̏̐̍̊   |
      U̝̙̎̈̐̆t̜̍̌̀̔̏ ̞̉̍̇̈̃e̟̟̊̄̕̕n̝̜̒̓̆̕i̖̒̌̅̇̚m̞̊̃̔̊̂ ̛̜̊̎̄̂a̘̜̋̒̚̚d̟̊̎̇̂̍ ̜̖̏̑̉̕m̜̒̎̅̄̚i̝̖̓̂̍̕n̙̉̒̑̀̔ỉ̖̝̌̒́m̛̖̘̅̆̎ ̖̉̎̒̌̕v̖̞̀̔́̎e̖̙̗̒̎̉n̛̗̝̎̀̂ȉ̞̗̒̕̚ȧ̟̜̝̅̚m̆̉̐̐̇̈,̏̐̎́̍́ ̜̞̙̘̏̆q̙̖̙̅̓̂ủ̇́̀̔̚í̙̟̟̏̐s̖̝̍̏̂̇ ̛̘̋̈̕̕ń̛̞̜̜̎o̗̜̔̔̈̆s̞̘̘̄̒̋t̛̅̋́̔̈ȓ̓̒́̇̅ủ̜̄̃̒̍d̙̝̘̊̏̚ ̛̟̞̄́̔e̛̗̝̍̃̀x̞̖̃̄̂̅e̖̅̇̐̔̃r̗̞̖̔̎̚c̘̜̖̆̊̏ï̙̝̙̂̕t̖̏́̓̋̂ă̖̄̆̑̒t̜̟̍̉̑̏i̛̞̞̘̒̑ǒ̜̆̅̃̉ṅ̖̜̒̎̚               |
      u̗̞̓̔̈̏ĺ̟̝́̎̚l̛̜̅̌̎̆a̒̑̆̔̇̃m̜̗̈̊̎̚ċ̘̋̇̂̚ơ̟̖̊́̕ ̖̟̍̉̏̚l̙̔̓̀̅̏ä̞̗̘̙̅ḃ̟̎̄̃̕o̞̎̓̓̓̚r̗̜̊̓̈̒ï̗̜̃̃̅s̀̒̌̂̎̂ ̖̗̗̋̎̐n̝̟̝̘̄̚i̜̒̀̒̐̕s̘̘̄̊̃̀ī̘̜̏̌̕ ̗̖̞̐̈̒ư̙̞̄́̌t̟̘̖̙̊̚ ̌̅̋̆̚̚ä̇̊̇̕̕l̝̞̘̋̔̅i̍̋́̆̑̈q̛̆̐̈̐̚ư̏̆̊́̚î̜̝̑́̊p̗̓̅̑̆̏ ̆́̓̔̋̋e̟̊̋̏̓̚x̗̍̑̊̎̈ ̟̞̆̄̂̍ë̄̎̄̃̅a̛̜̅́̃̈ ̔̋̀̎̐̀c̖̖̍̀̒̂ơ̛̙̖̄̒m̘̔̍̏̆̕ḿ̖̙̝̏̂ȍ̓̋̈̀̕d̆̂̊̅̓̚o̖̔̌̑̚̕ ̙̆́̔̊̒c̖̘̖̀̄̍o̓̄̑̐̓̒ñ̞̒̎̈̚s̞̜̘̈̄̄e̙̊̀̇̌̋q̐̒̓́̔̃ư̗̟̔̔̚å̖̙̞̄̏t̛̙̟̒̇̏.̙̗̓̃̓̎         |
      D̜̖̆̏̌̌ư̑̃̌̍̕i̝̊̊̊̊̄s̛̙̒́̌̇ ̛̃̔̄̆̌ă̘̔̅̅̀ú̟̟̟̃̃t̟̂̄̈̈̃e̘̅̌̒̂̆ ̖̟̐̉̉̌î̟̟̙̜̇r̛̙̞̗̄̌ú̗̗̃̌̎r̛̙̘̉̊̕e̒̐̔̃̓̋ ̊̊̍̋̑̉d̛̝̙̉̀̓o̘̜̐̐̓̐l̞̋̌̆̍́o̊̊̐̃̃̚ṙ̛̖̘̃̕ ̞̊̀̍̒̕ȉ́̑̐̇̅ǹ̜̗̜̞̏ ̛̜̐̄̄̚r̜̖̈̇̅̋ĕ̗̉̃̔̚p̟̝̀̓̔̆r̜̈̆̇̃̃e̘̔̔̏̎̓h̗̒̉̑̆̚ė̛̘̘̈̐n̘̂̀̒̕̕d̗̅̂̋̅́ê̗̜̜̜̕r̟̋̄̐̅̂i̛̔̌̒̂̕t̛̗̓̎̀̎ ̙̗̀̉̂̚ȉ̟̗̐̓̚n̙̂̍̏̓̉ ̙̘̊̋̍̕v̜̖̀̎̆̐ő̜̆̉̃̎l̑̋̒̉̔̆ư̙̓̓́̚p̝̘̖̎̏̒t̛̘̝̞̂̓ȁ̘̆̔́̊t̖̝̉̒̐̎e̞̟̋̀̅̄ ̆̌̃̀̑̔v̝̘̝̍̀̇ȅ̝̊̄̓̕l̞̝̑̔̂̋ĭ̝̄̅̆̍t̝̜̉̂̈̇        |
      ē̟̊̇̕̚s̖̘̘̒̄̑s̛̘̀̊̆̇e̛̝̘̒̏̚ ̉̅̑̂̐̎c̛̟̙̎̋̓i̜̇̒̏̆̆l̟̄́̆̊̌l̍̊̋̃̆̌ủ̗̙̒̔̚m̛̘̘̖̅̍ ̖̙̈̎̂̕d̞̟̏̋̈̔ơ̟̝̌̃̄l̗̙̝̂̉̒õ̒̃̄̄̚ŕ̗̏̏̊̍ê̞̝̞̋̈ ̜̔̒̎̃̚e̞̟̞̒̃̄ư̖̏̄̑̃ ̛̗̜̄̓̎f̛̖̞̅̓̃ü̞̏̆̋̕g̜̝̞̑̑̆i̛̘̐̐̅̚à̜̖̌̆̎t̙̙̎̉̂̍ ̋̔̈̎̎̉n̞̓́̔̊̕ư̘̅̋̔̚l̗̍̒̄̀̚l̞̗̘̙̓̍â̘̔̒̎̚ ̖̓̋̉̃̆p̛̛̘̋̌̀ä̙̔́̒̕r̟̟̖̋̐̋ì̗̙̎̓̓ȃ̔̋̑̚̕t̄́̎̓̂̋ư̏̈̂̑̃r̖̓̋̊̚̚.̒̆̑̆̊̎ ̘̜̍̐̂̚E̞̅̐̇́̂x̄́̈̌̉̕ć̘̃̉̃̕è̘̂̑̏̑p̝̘̑̂̌̆t̔̐̅̍̌̂ȇ̞̈̐̚̕ű̝̞̜́̚ŕ̗̝̉̆́           |
      š̟́̔̏̀ȉ̝̟̝̏̅n̑̆̇̒̆̚t̝̒́̅̋̏ ̗̑̌̋̇̚ơ̙̗̟̆̅c̙̞̙̎̊̎c̘̟̍̔̊̊a̛̒̓̉́̐e̜̘̙̒̅̇ć̝̝̂̇̕ả̓̍̎̂̚t̗̗̗̟̒̃ ̘̒̓̐̇́c̟̞̉̐̓̄ȕ̙̗̅́̏p̛̍̋̈́̅i̖̓̒̍̈̄d̞̃̈̌̆̐a̛̗̝̎̋̉t̞̙̀̊̆̇a̛̙̒̆̉̚t̜̟̘̉̓̚ ̝̘̗̐̇̕n̛̘̑̏̂́ō̑̋̉̏́ň̞̊̆̄̃ ̙̙̙̜̄̏p̒̆̋̋̓̏r̖̖̅̉́̚ơ̜̆̑̈̚i̟̒̀̃̂̌d̛̏̃̍̋̚ë̖̞̙̗̓n̛̘̓̒̅̎t̟̗̙̊̆̚,̘̙̔̊̚̕ ̟̗̘̜̑̔s̜̝̍̀̓̌û̞̙̅̇́n̘̗̝̒̃̎t̗̅̀̅̊̈ ̗̖̅̅̀̄i̛̖̍̅̋̂n̙̝̓̓̎̚ ̞̋̅̋̃̚c̗̒̀̆̌̎ū̞̂̑̌̓ĺ̛̐̍̑́p̝̆̌̎̈̚a̖̙̒̅̈̌ ̝̝̜̂̈̀q̝̖̔̍̒̚ư̔̐̂̎̊ǐ̛̟̖̘̕          |
      o̖̜̔̋̅̚f̛̊̀̉́̕f̏̉̀̔̃̃i̘̍̎̐̔̎c̙̅̑̂̐̅ȋ̛̜̀̒̚a̋̍̇̏̀̋ ̖̘̒̅̃̒d̗̘̓̈̇̋é̝́̎̒̄š̙̒̊̉̋e̖̓̐̀̍̕r̗̞̂̅̇̄ù̘̇̐̉̀n̐̑̀̄̍̐t̟̀̂̊̄̚ ̟̝̂̍̏́m̜̗̈̂̏̚ő̞̊̑̇̒l̘̑̏́̔̄l̛̛̇̃̋̊i̓̋̒̃̉̌t̛̗̜̏̀̋ ̙̟̒̂̌̐a̙̝̔̆̏̅n̝̙̙̗̆̅i̍̔́̊̃̕m̖̝̟̒̍̚ ̛̃̃̑̌́ǐ̘̉̔̅̚d̝̗̀̌̏̒ ̖̝̓̑̊̚ȇ̞̟̖̌̕š̙̙̈̔̀t̂̉̒̍̄̄ ̝̗̊̋̌̄l̛̞̜̙̘̔å̝̍̂̍̅b̜̆̇̈̉̌ǒ̜̙̎̃̆r̝̀̄̍́̕ư̋̊́̊̕m̜̗̒̐̕̚.̟̘̀̒̌̚                     |
      {1:~                                                                }|
                                                                       |
    ]],
      reset = true,
    }
  end)

  it('works with even huger zalgo chars', function()
    screen:try_resize(100, 20)
    api.nvim_command 'color default'
    api.nvim_set_hl(0, 'MyHighlight', { bg = '#53246b', fg = '#a4d844' })
    screen:add_extra_attr_ids {
      [100] = { background = tonumber('0x53246b'), foreground = Screen.colors.NvimLightGrey4 },
      [101] = { background = tonumber('0x53246b'), foreground = tonumber('0xa4d844') },
    }
    api.nvim_set_option_value('winhl', 'Normal:MyHighlight', {})
    api.nvim_set_option_value('number', true, {})
    local text =
      'c̯̥̳̮̫̳͔̱̀ͦͩ̄͋̓͘c̯̥̳̮̫̳͔̱̀ͦͩ̄͋̓͘l̶̴̴̨̛͓͚͎̥ͦͤ̑͆͛͢l̶̴̴̨̛͓͚͎̥ͦͤ̑͆͛͢ô̷̤̩ͯͧ͆ͪ̈́́͒̒̐͐̕ô̷̤̩ͯͧ͆ͪ̈́́͒̒̐͐̕s̶̷̢̩̱̠̀ͦ̽ͮ͒ͨ̚͟͠s̶̷̢̩̱̠̀ͦ̽ͮ͒ͨ̚͟͠e͉̯̱̖͕̳̼̽̊̒ͣ̊ͥ̚̚e͉̯̱̖͕̳̼̽̊̒ͣ̊ͥ̚̚ ͇͌̈̄ͬͧͧ͝͠͏̷ͪ̎͟͠ ͇͌̈̄ͬͧͧ͝͠͏̷ͪ̎͟͠t̵̳̅̽͏̵̡̥̲͍͙̹̯ͩ̐t̵̳̅̽͏̵̡̥̲͍͙̹̯ͩ̐o͋͏̬̞̣ͬ́ͬ̎̋̓̽͘͠͡o͋͏̬̞̣ͬ́ͬ̎̋̓̽͘͠͡ ͏̛͙̮̈ͬ̌͐ͤ͒ͧ̎ͤ͜͠ ͏̛͙̮̈ͬ̌͐ͤ͒ͧ̎ͤ͜͠ǫ̛̳͕̦͖̪̀͂͛̅̔ͪ͒͜ǫ̛̳͕̦͖̪̀͂͛̅̔ͪ͒͜v̸̡͖̗̣ͯ̿̔͊̅ͯ̈̓̇ͅv̸̡͖̗̣ͯ̿̔͊̅ͯ̈̓̇ͅe̴̶̢̜̭̠̰̞ͪͨ͂ͤ͆́͗e̴̶̢̜̭̠̰̞ͪͨ͂ͤ͆́͗r̨̖̼̳̳͚̖̒ͯ̊̋̂̑̚͞r̨̖̼̳̳͚̖̒ͯ̊̋̂̑̚͞f̵̮̣̬̾͛̌ͧͦ͘͢͜͜͠ͅf̵̮̣̬̾͛̌ͧͦ͘͢͜͜͠ͅl̡͉̬̳͈̠̏͂ͤ̈ͨ̀ͩ̔͏l̡͉̬̳͈̠̏͂ͤ̈ͨ̀ͩ̔͏ỏ̶̢̻̠͎͇͈̜̈̆ͯ̔ͩ̾ỏ̶̢̻̠͎͇͈̜̈̆ͯ̔ͩ̾ẉ̦̞̼̩̣͙̅̿́̓̉̎̈ͪẉ̦̞̼̩̣͙̅̿́̓̉̎̈ͪi̷̡͍͓͔̝͙͖͖̍ͯͤͬͦ͝i̷̡͍͓͔̝͙͖͖̍ͯͤͬͦ͝n̠̪̰͋ͩ̆͏̶̖̯ͭ̀̿͛͘n̠̪̰͋ͩ̆͏̶̖̯ͭ̀̿͛͘g̴̭͎̞͙̗̯͙͖ͬ́ͧͧ͝ͅg̴̭͎̞͙̗̯͙͖ͬ́ͧͧ͝ͅ ̢͎̬͓̮̹̰̙͍̓̐͋͂͐̚ ̢͎̬͓̮̹̰̙͍̓̐͋͂͐̚ť̴̼̝̗ͩ̍͋̿͏͇ͧ̑̏̚ť̴̼̝̗ͩ̍͋̿͏͇ͧ̑̏̚h̨̨̢͓̲͚͕̦ͨ͛̓ͩ̚͟͠h̨̨̢͓̲͚͕̦ͨ͛̓ͩ̚͟͠ȩ͕̜̥̃̑͋̏͐̎̒͛͊͏͙ȩ͕̜̥̃̑͋̏͐̎̒͛͊͏͙ ̵̨̜̜̠͉̱͎͑ͥ̌͐̽͢͡ ̵̨̜̜̠͉̱͎͑ͥ̌͐̽͢͡r̢̫͓͎̙̭̽ͥͭ͐̂̀̕͟͝r̢̫͓͎̙̭̽ͥͭ͐̂̀̕͟͝e̴̷͓̹̩ͧ́̉̑̈̿ͥ̕͡͝e̴̷͓̹̩ͧ́̉̑̈̿ͥ̕͡͝d̵̡̼̩̠̜͈̯̬͚͛̋̀̆͟d̵̡̼̩̠̜͈̯̬͚͛̋̀̆͟ŕ̻̳̬͏̨̮͚̲ͥ̌͆͗͠ͅŕ̻̳̬͏̨̮͚̲ͥ̌͆͗͠ͅā̸͙̥̤͍͈̣ͪͨ̈͋̈́̈́͜ā̸͙̥̤͍͈̣ͪͨ̈͋̈́̈́͜w̶̦̪͚̤̃ͬ̓͒ͤ̇̇ͮ͢͡w̶̦̪͚̤̃ͬ̓͒ͤ̇̇ͮ͢͡ ̷̡̦̻̻̪͚̳͎ͥ̓ͩͪ͠͝ ̷̡̦̻̻̪͚̳͎ͥ̓ͩͪ͠͝b̳ͮ̒̊̆̒́̈́̏̓͋ͭ̔ͤ̚b̳ͮ̒̊̆̒́̈́̏̓͋ͭ̔ͤ̚u̧̟̫͎͚̭̠͕͂̄̀̒̈̇͜u̧̟̫͎͚̭̠͕͂̄̀̒̈̇͜f̶̸̢͍̑̂̊ͥͫ̈́ͥ͛̈́̃͝f̶̸̢͍̑̂̊ͥͫ̈́ͥ͛̈́̃͝f̵̼̭̮͎ͧ̒̆͊̀ͤ͊̇̕͡f̵̼̭̮͎ͧ̒̆͊̀ͤ͊̇̕͡e̮̪̲̣̞̖͎ͥͤ̐̌̓̐͟͢e̮̪̲̣̞̖͎ͥͤ̐̌̓̐͟͢ŗ̭̘̮̹̖̎̄̆́ͫͭ͢ͅͅŗ̭̘̮̹̖̎̄̆́ͫͭ͢ͅͅ.̪̖͉͚̹̾̉ͮ̔̊ͪ̾̎͟͞.̪̖͉͚̹̾̉ͮ̔̊ͪ̾̎͟͞ ̷̧̺̰͎̣̃͒͗ͮ͑ͪͮ͞ͅ ̷̧̺̰͎̣̃͒͗ͮ͑ͪͮ͞ͅf̛̫͚͚͍̜͎̗̳̂ͬͭ͜͢͝f̛̫͚͚͍̜͎̗̳̂ͬͭ͜͢͝i̴̵̡̛̛͎̤̳̮ͩ̐ͩ͑̇̚i̴̵̡̛̛͎̤̳̮ͩ̐ͩ͑̇̚n̵͇͍͚̖̥̣ͨ̄ͧ̌̂͗͘͝n̵͇͍͚̖̥̣ͨ̄ͧ̌̂͗͘͝ȉ̼̱̫̜̋̓̐͌͆ͨ͘͝͡ͅȉ̼̱̫̜̋̓̐͌͆ͨ͘͝͡ͅs̴̸̝̺̬͚̲̹̘ͪ̆̊ͥ͞͝s̴̸̝̺̬͚̲̹̘ͪ̆̊ͥ͞͝h̴̜̠͇ͦͥ̔̅ͭͭ͋ͩ͟͡͞h̴̜̠͇ͦͥ̔̅ͭͭ͋ͩ͟͡͞ ̶̧̛̻͙̤̘̺̣̻̗͍̓͑͠ ̶̧̛̻͙̤̘̺̣̻̗͍̓͑͠t̠͉̼̬̩͛́ͨ͐̀͛̂ͨ̾͞t̠͉̼̬̩͛́ͨ͐̀͛̂ͨ̾͞h̻̹̝̹̾ͩ̍ͧ͆ͥ̔͘͏̉ͯh̻̹̝̹̾ͩ̍ͧ͆ͥ̔͘͏̉ͯì̷̢̛̺̭͇̟̦̄̓́̓ͪ͟ì̷̢̛̺̭͇̟̦̄̓́̓ͪ͟s̴̡̗͍͕͖̮̟̱̫̎ͣ̀̎̿s̴̡̗͍͕͖̮̟̱̫̎ͣ̀̎̿ ̶͇̟̜̗̗͇͇͐̑̈͋̋̽͟ ̶͇̟̜̗̗͇͇͐̑̈͋̋̽͟e̷̥͙͈̖̤̯̹̯͗̉̈́̽ͨ̕e̷̥͙͈̖̤̯̹̯͗̉̈́̽ͨ̕v̛̝͕̱͙̞̖̒̂̔͆̊̎́ͫv̛̝͕̱͙̞̖̒̂̔͆̊̎́ͫę̴̤̭͖̈̐̿͂ͣ͒̃ͭ̕͟ę̴̤̭͖̈̐̿͂ͣ͒̃ͭ̕͟ṇ̳̭͓̟̠͕̯͑̉ͦ̀ͯ͜͡ṇ̳̭͓̟̠͕̯͑̉ͦ̀ͯ͜͡t̢̞͔̣̄̀͆̂̃ͨͦ͜͜͝͠t̢̞͔̣̄̀͆̂̃ͨͦ͜͜͝͠,̬̳̮͓͉̟͉͓̦͔̑̄ͨ̎͜,̬̳̮͓͉̟͉͓̦͔̑̄ͨ̎͜ ͕͈̠̰̬̬̌͂̏ͥ̀̕̚͢͠ ͕͈̠̰̬̬̌͂̏ͥ̀̕̚͢͠f̺̮̞̈́̏̋͏̺̖̝̓̑̂̚͢f̺̮̞̈́̏̋͏̺̖̝̓̑̂̚͢l̜̪͍̩̩̟͉͓̊̓ͤ̆ͣͫ̌l̜̪͍̩̩̟͉͓̊̓ͤ̆ͣͫ̌u̷̧̞̳̱̜̟̲͑̐͂ͪ͛͘͟u̷̧̞̳̱̜̟̲͑̐͂ͪ͛͘͟s͔͍̽ͣͮ̏̓͛̄ͯ̽͂̐ͯ͜s͔͍̽ͣͮ̏̓͛̄ͯ̽͂̐ͯ͜ḩ̼͕̦͖̼͚̗̃̃ͥ̅̂̈͟ḩ̼͕̦͖̼͚̗̃̃ͥ̅̂̈͟,̵͍̮̮̟͚̝̃ͨ̿ͭ̌ͤ̋̊,̵͍̮̮̟͚̝̃ͨ̿ͭ̌ͤ̋̊ ̷̨̨͈̝̦͂ͦ̒̋̇ͧ̒͟͝ ̷̨̨͈̝̦͂ͦ̒̋̇ͧ̒͟͝a̡̨̲̖̾̂͗̚͢͡͏͈ͤ̉͡a̡̨̲̖̾̂͗̚͢͡͏͈ͤ̉͡nͫͤ̚͜͏̧̛̣̻ͩ̔̍ͦ̕͝nͫͤ̚͜͏̧̛̣̻ͩ̔̍ͦ̕͝d͈̻̗̼̀͡͏̶̵̟̹̻̎̾ͯd͈̻̗̼̀͡͏̶̵̟̹̻̎̾ͯ ̶͙͈̠̜̬̺͛̀̊̂ͪ̔ͮ͑ ̶͙͈̠̜̬̺͛̀̊̂ͪ̔ͮ͑ş̧̡̢͓̠͋ͫ͑͒͊̅̔͜͞ş̧̡̢͓̠͋ͫ͑͒͊̅̔͜͞t̤̭͓̰̣̉̔̎ͫ͛ͦ̑̕͟͞t̤̭͓̰̣̉̔̎ͫ͛ͦ̑̕͟͞a̠͇͇̯̥͍͚̳̒́͐͐̏͋̓a̠͇͇̯̥͍͚̳̒́͐͐̏͋̓r͉͈̻̻͕̩̩̃̅͋͆ͮ͢͢͡r͉͈̻̻͕̩̩̃̅͋͆ͮ͢͢͡t̵̛̝̗̥̙̟̆ͮ̽̏ͧ͜͠ͅt̵̛̝̗̥̙̟̆ͮ̽̏ͧ͜͠ͅ ̷̼͎̦ͫ̈͑̃̽͏̲̪ͣͯ̽ ̷̼͎̦ͫ̈͑̃̽͏̲̪ͣͯ̽á̸̷̴̟̣́̔̌͏̶͕͋ͭͭá̸̷̴̟̣́̔̌͏̶͕͋ͭͭ ̧̧̲͍̘̘͚͔͇͙ͨͬ́̊ͅ ̧̧̲͍̘̘͚͔͇͙ͨͬ́̊ͅn̸̸̩͖͔͚͚̖͗ͦ̓̀̀̈́̈́n̸̸̩͖͔͚͚̖͗ͦ̓̀̀̈́̈́ę̵̧̬̣̦̖̝̹̱͌̃̑ͣ̚ę̵̧̬̣̦̖̝̹̱͌̃̑ͣ̚w͍̥͚̺ͮ̓̈̈́̾̊̆́̚͜͝w͍̥͚̺ͮ̓̈̈́̾̊̆́̚͜͝ ̛̹̲̥̝͙̾ͨ̆̎̃̋͂̓̕ ̛̹̲̥̝͙̾ͨ̆̎̃̋͂̓̕"̴̜̰̰̱̖̙̘̈́͌ͨͪ̐̕͠"̴̜̰̰̱̖̙̘̈́͌ͨͪ̐̕͠g̸̛͇̐͊͂̽͢͏͖̣ͫ͊ͯͅg̸̛͇̐͊͂̽͢͏͖̣ͫ͊ͯͅr̴̨̲͎̹͇̠̐ͤ̇̒ͬ̆ͧ͞r̴̨̲͎̹͇̠̐ͤ̇̒ͬ̆ͧ͞į̝̱̩͔̈ͨ̉͌̋̍̂͜͟͞į̝̱̩͔̈ͨ̉͌̋̍̂͜͟͞d̷̴̷̟̎͌͑͛̈́ͭͨͯ̋ͭ̕d̷̴̷̟̎͌͑͛̈́ͭͨͯ̋ͭ̕_̢̭̙̦̪͇̾̔̆ͬͦ́ͥ͢͡_̢̭̙̦̪͇̾̔̆ͬͦ́ͥ͢͡l̡̢̨̧͔̱̥̹̬̆ͮ̈́̏̅͜l̡̢̨̧͔̱̥̹̬̆ͮ̈́̏̅͜ĩ̢͖̠̩̞͚̰̰̉̋̌͛ͪ͠ĩ̢͖̠̩̞͚̰̰̉̋̌͛ͪ͠n̢̬̜̘̲͉ͮ͆͏̯͕ͭͦ̉̅n̢̬̜̘̲͉ͮ͆͏̯͕ͭͦ̉̅ē̡͈̮̭̰̜͍̗̮͔͌͐͆ͫē̡͈̮̭̰̜͍̗̮͔͌͐͆ͫ"̦̠̟ͣ̽͋͐ͧ̓̂̆̎͒͝ͅ"̦̠̟ͣ̽͋͐ͧ̓̂̆̎͒͝ͅ ̴̡̺̹̖̰̀ͤ͊̊͗̊́͜͠ ̴̡̺̹̖̰̀ͤ͊̊͗̊́͜͠ẻ͎̳̻̯̹͓͊̌̄͑͂̉͜͢ẻ͎̳̻̯̹͓͊̌̄͑͂̉͜͢v̞̬̪̥͖ͤ͐̍́ͤ̇̀̕̕ͅv̞̬̪̥͖ͤ͐̍́ͤ̇̀̕̕ͅȇ̶̱̗̩̠͚͎͊ͤͪͦͫ̂̚ȇ̶̱̗̩̠͚͎͊ͤͪͦͫ̂̚n̢̮̜͉̎ͨͩ̒̓ͬͨ̓ͦ͘͜n̢̮̜͉̎ͨͩ̒̓ͬͨ̓ͦ͘͜t̢͚͉̹͇̺̭ͪ̄̉ͨ̄͐̕ͅt̢͚͉̹͇̺̭ͪ̄̉ͨ̄͐̕ͅ ̵̛̳̱͍̩͓ͣ͋̈́͐ͭͧ̿ͅ ̵̛̳̱͍̩͓ͣ͋̈́͐ͭͧ̿ͅã̰̪̙͉̪͇ͣ͋ͤ̓͋͑̕͘ã̰̪̙͉̪͇ͣ͋ͤ̓͋͑̕͘t̴̴̡̡̛̳̼̻̳̂̽͒̇̚͠t̴̴̡̡̛̳̼̻̳̂̽͒̇̚͠ ̴̙̤̹͔̳̟̽̀̆ͥ̂̕͘͝ ̴̙̤̹͔̳̟̽̀̆ͥ̂̕͘͝t̴͖̼͔̬̦̏ͩ̄́̓̊̔̇͡t̴͖̼͔̬̦̏ͩ̄́̓̊̔̇͡ḫ̷͚͇̫̫̠͒ͮ̍ͣ̃͐͐̍ḫ̷͚͇̫̫̠͒ͮ̍ͣ̃͐͐̍e̳͉̮͖̳̣̎͌̂ͣ͋ͯ̆͜͞e̳͉̮͖̳̣̎͌̂ͣ͋ͯ̆͜͞ ̪̖̉̌̀̽̄̍̓̀̂̋͐̈̚ ̪̖̉̌̀̽̄̍̓̀̂̋͐̈̚c̶̫̜͚̊̿̂̿ͥͭ̓̂̈́͘͡c̶̫̜͚̊̿̂̿ͥͭ̓̂̈́͘͡ṷ̵̷͉̯̜̬̝̑͛ͤ̋ͧͯ̉ṷ̵̷͉̯̜̬̝̑͛ͤ̋ͧͯ̉ŗ̙̬̺̬ͥͤ̓͐̈́ͬ̽̌͡͡ŗ̙̬̺̬ͥͤ̓͐̈́ͬ̽̌͡͡r̛͖̼̣͙̋̇̀̅͒̽ͥ̑ͅͅr̛͖̼̣͙̋̇̀̅͒̽ͥ̑ͅͅě̢̪̦̼͘͏̴̠̞͍̓̐ͭ̇ě̢̪̦̼͘͏̴̠̞͍̓̐ͭ̇ǹ̨̨̛̛̯̍ͭͫ̎̍̃̄̐̍ǹ̨̨̛̛̯̍ͭͫ̎̍̃̄̐̍t̵̳͔̭̮̭̱̰ͤ͐̉̾͗̅͢t̵̳͔̭̮̭̱̰ͤ͐̉̾͗̅͢ ̩͕ͧ̌͂͂́ͫͥ̍͏̬̙͗ͅ ̩͕ͧ̌͂͂́ͫͥ̍͏̬̙͗ͅp̞̞͇͇̯̩̬̜ͥ̃͐̑͋́͗p̞̞͇͇̯̩̬̜ͥ̃͐̑͋́͗o̵͖͈̩̪̥̝̊̒̉̿͋ͩ̆͠o̵͖͈̩̪̥̝̊̒̉̿͋ͩ̆͠s̶͇̺̩̟̺͋̆̒ͫͥ̆͏͔͜s̶͇̺̩̟̺͋̆̒ͫͥ̆͏͔͜i̞̝̭̲͈̝̮̫͚͑̓ͤ̎ͮ̀i̞̝̭̲͈̝̮̫͚͑̓ͤ̎ͮ̀t̷̬͇̗̥̳͔̮͔̎̀ͣ̊̕͢t̷̬͇̗̥̳͔̮͔̎̀ͣ̊̕͢i̶̧̗̠̗̲͑̋ͣͪͦͣ͆̕͢i̶̧̗̠̗̲͑̋ͣͪͦͣ͆̕͢ǫ̰̟̙ͯ͏̬̦͚̝̀̈́ͣ͘͜ǫ̰̟̙ͯ͏̬̦͚̝̀̈́ͣ͘͜n̸̻̩͖͓̠̲̮̐̄̌̇́́͠n̸̻̩͖͓̠̲̮̐̄̌̇́́͠.̧̛͈̙̭̉ͥ̋͛̏͋̂̿͛͞.̧̛͈̙̭̉ͥ̋͛̏͋̂̿͛͞ ̡̲͇̳͔̦̤̹̥ͣ̍ͪͮ̎͡ ̡̲͇̳͔̦̤̹̥ͣ̍ͪͮ̎͡F̼̣͙̳̞͑͒ͧͣ̔̊̇̓̈́͠F̼̣͙̳̞͑͒ͧͣ̔̊̇̓̈́͠ŏ̸̶͎̘̟͙̤̑̒̿͗͛̐̚ŏ̸̶͎̘̟͙̤̑̒̿͗͛̐̚r̵̰͇͚̥̻͚̈́̌̽ͨͥ̕͘͞r̵̰͇͚̥̻͚̈́̌̽ͨͥ̕͘͞ ̵̶̧͖̙̝̻̭̤̒̿͌͗͑͡ ̵̶̧͖̙̝̻̭̤̒̿͌͗͑͡ś̸̬̘̭̼͓̹ͥ͛ͪ̐̈́͆͝ś̸̬̘̭̼͓̹ͥ͛ͪ̐̈́͆͝i͎̺̪̖̻ͫ̀ͤ̾ͦ̽ͭ͒̒͘i͎̺̪̖̻ͫ̀ͤ̾ͦ̽ͭ͒̒͘m̨̩͎̫̫̙̃̈͆ͬ̊͠͏̽͞m̨̩͎̫̫̙̃̈͆ͬ̊͠͏̽͞p̴̢̨͖̼͍̲͍̲ͩ̊̒̌̃ͤp̴̢̨͖̼͍̲͍̲ͩ̊̒̌̃ͤļ͔̱͙̝̟̜͚͎͕̮̹ͣ̀͘ļ͔̱͙̝̟̜͚͎͕̮̹ͣ̀͘ḭ̢̛͈͍̠̟̪̑̎̈́̑̽͜͡ḭ̢̛͈͍̠̟̪̑̎̈́̑̽͜͡c̖̠̥͔̪̼̑̃̆̓ͫ͗ͩͩ̋c̖̠̥͔̪̼̑̃̆̓ͫ͗ͩͩ̋i̴̸̡̘͇͔̹͂̍̐̀ͬͩ̈͘i̴̸̡̘͇͔̹͂̍̐̀ͬͩ̈͘t̵̵̖̫͙͎̒ͭ̔̃̔ͧͫ̽͝t̵̵̖̫͙͎̒ͭ̔̃̔ͧͫ̽͝y̴̘̱͈̳ͯ̈́̍ͮ̔͊͊̚̚͞y̴̘̱͈̳ͯ̈́̍ͮ̔͊͊̚̚͞ ̛̩̤̪͇̬͕̯̞̙̪̮̀̂̕ ̛̩̤̪͇̬͕̯̞̙̪̮̀̂̕ļ̲̰̞̈̌̿͐̉́̋ͬ͟͟͡ļ̲̰̞̈̌̿͐̉́̋ͬ͟͟͡e̡̧̛̬̣̗̣̰͂̐̂͗͛̋̇e̡̧̛̬̣̗̣̰͂̐̂͗͛̋̇a̡̯͉̠̞ͩͮ̉̓́ͮ̔̆̒͘a̡̯͉̠̞ͩͮ̉̓́ͮ̔̆̒͘v͔ͦ͏͎̪̮̝̟̤́̀͊̈̎͟v͔ͦ͏͎̪̮̝̟̤́̀͊̈̎͟ȩ̪͔̤̺͗ͦ̅̓ͭͤͫ̆ͣ͒ȩ̪͔̤̺͗ͦ̅̓ͭͤͫ̆ͣ͒ ̶̢̤͎̰̝̤͂̔͒ͦͦ͂̊̀ ̶̢̤͎̰̝̤͂̔͒ͦͦ͂̊̀p̛̝̪͚͖̫͕̅̍̊́̓͒̂̃p̛̝̪͚͖̫͕̅̍̊́̓͒̂̃l̴̸͉͎̖͕̦̥̾ͨ̌̑ͣ̕͝l̴̸͉͎̖͕̦̥̾ͨ̌̑ͣ̕͝a̰̩͔̼͔̦̩͒ͪͭ̐͡͏̗ͮa̰̩͔̼͔̦̩͒ͪͭ̐͡͏̗ͮč̵̱͇̲̞̉̆̌̄ͧͫ̑̈́͠č̵̱͇̲̞̉̆̌̄ͧͫ̑̈́͠e̶̛̜̬̯̤͔̓ͤͮͦ̂͐͜͜e̶̛̜̬̯̤͔̓ͤͮͦ̂͐͜͜ ̸̧̼̯͕̼̭ͣͣ̿̑̈̎̽͜ ̸̧̼̯͕̼̭ͣͣ̿̑̈̎̽͜f̴̼̞͇̺̊̎̓̑̔̈́͗̊͘͞f̴̼̞͇̺̊̎̓̑̔̈́͗̊͘͞ő̶̶̧̺̤̜̹̫͎̻̏̉̍ͩő̶̶̧̺̤̜̹̫͎̻̏̉̍ͩr̵̙̱̺̮͙̯̼͐̀͗̾͊͟͝r̵̙̱̺̮͙̯̼͐̀͗̾͊͟͝ ̵̪̥͍̩̱ͨ͗̓̎̓͒̚͜͞ ̵̪̥͍̩̱ͨ͗̓̎̓͒̚͜͞t̪̤͙̥̖̹̣̤͎̞̒́̒̽̃t̪̤͙̥̖̹̣̤͎̞̒́̒̽̃ḥ̵̷̷̭̘͙͓ͩ̓ͪͤ̓ͮ̍ḥ̵̷̷̭̘͙͓ͩ̓ͪͤ̓ͮ̍é̡̬̺͉̻̻̀̀͂̽ͬ̕̕͠é̡̬̺͉̻̻̀̀͂̽ͬ̕̕͠ ̵̱̻̣͙̌͊́ͯ́̐̀́͂̕ ̵̱̻̣͙̌͊́ͯ́̐̀́͂̕f̨̪͈̲̬̟̝͎̀̔̋ͨ̀͡ͅf̨̪͈̲̬̟̝͎̀̔̋ͨ̀͡ͅi̵̫̱̝͒̇̏̃ͭ̂̄̄̊̐͠i̵̫̱̝͒̇̏̃ͭ̂̄̄̊̐͠n̵̷͖̻̦͍̻̑̈́̎̓͑ͪͫ͟n̵̷͖̻̦͍̻̑̈́̎̓͑ͪͫ͟a̗̗̠̫̦̻̹͇ͯͦͫ͗̽ͥ̚a̗̗̠̫̦̻̹͇ͯͦͫ͗̽ͥ̚l̵̨̨͙̜͔̘̗̯͌̋͂̑̄͢l̵̨̨͙̜͔̘̗̯͌̋͂̑̄͢ ̵̷̛̺̳͙̲̥̬̊̌̽̓̇͝ ̵̷̛̺̳͙̲̥̬̊̌̽̓̇͝"̸̨̫̟̤̥͉̮̥̀ͤ̐͊̇̑"̸̨̫̟̤̥͉̮̥̀ͤ̐͊̇̑c̵̭̘͍͇̜ͨ̔̍̆͛͌͂͌͛c̵̭̘͍͇̜ͨ̔̍̆͛͌͂͌͛l̸̷̢̡̫̩̃́͐̆͒ͨ̔̄͟l̸̷̢̡̫̩̃́͐̆͒ͨ̔̄͟è̙͎̝̞̠̹͍́́̅͐͌͘ͅè̙͎̝̞̠̹͍́́̅͐͌͘ͅą̶̢̡̘̦͖̰̌̎͌ͩ́̓ͮą̶̢̡̘̦͖̰̌̎͌ͩ́̓ͮȑ̹̻̦͙̠͂̽͆ͫ͛͟͏̺͡ȑ̹̻̦͙̠͂̽͆ͫ͛͟͏̺͡"̣̞͕̇ͯ̓̽ͪ͑́̍̚͘͘ͅ"̣̞͕̇ͯ̓̽ͪ͑́̍̚͘͘ͅ ̛̩̘̗̜̗̥̂͊ͥ̀ͨͫ̾ͧ ̛̩̘̗̜̗̥̂͊ͥ̀ͨͫ̾ͧe̵̡͚͉̤̯̮͛̎̓ͪ̾̉̆͟e̵̡͚͉̤̯̮͛̎̓ͪ̾̉̆͟l̢̩͓̙͓͍̈̊ͫͣ́̅ͧ͛͞l̢̩͓̙͓͍̈̊ͫͣ́̅ͧ͛͞e̠̱͉̯͔̙͔̓ͩ̃͋͌ͬͭ͋e̠̱͉̯͔̙͔̓ͩ̃͋͌ͬͭ͋m̧̡̛̤͕̻̳̽͛̓̈́ͣ̊̚͟m̧̡̛̤͕̻̳̽͛̓̈́ͣ̊̚͟e̯͎̺͔̼̾͂͐̄ͮ͌̍͑̕ͅe̯͎̺͔̼̾͂͐̄ͮ͌̍͑̕ͅṉ̸͔̩̙̙̹͚̭ͩ͛͗̀̾ͅṉ̸͔̩̙̙̹͚̭ͩ͛͗̀̾ͅt̸̢̧̝͉̺͉̓́̇ͯ̕͠͠͝t̸̢̧̝͉̺͉̓́̇ͯ̕͠͠͝ ̳͔̜̹̘̪̅̋̎̉͆̑ͤ͡͞ ̳͔̜̹̘̪̅̋̎̉͆̑ͤ͡͞a̷̳͓̞̱͈̓́̒̔̆ͩͯ̊͠a̷̳͓̞̱͈̓́̒̔̆ͩͯ̊͠s̷̛͙̘̳͙͓̠̞̫̆̓̚̚͡s̷̛͙̘̳͙͓̠̞̫̆̓̚̚͡ ̷̢̦̘̮̰͕̏̀ͩ̓͂ͦ͟ͅ ̷̢̦̘̮̰͕̏̀ͩ̓͂ͦ͟ͅw̮͔͇͕̝͖ͪ͒̏̇ͪ̇̍ͦ͠w̮͔͇͕̝͖ͪ͒̏̇ͪ̇̍ͦ͠ẹ̗͚̮̭̓̍̐̃̍́̐͘̕͘ẹ̗͚̮̭̓̍̐̃̍́̐͘̕͘l̴̡̩̱̞͛ͤ͑͑͂̀ͣ̉̌̕l̴̡̩̱̞͛ͤ͑͑͂̀ͣ̉̌̕l̷̨̘̤̣̯͇̲̝ͨ̍ͦ͝͝ͅl̷̨̘̤̣̯͇̲̝ͨ̍ͦ͝͝ͅ,̷̻̙̖͔͚͋̇̂̑͗̕̕͢ͅ,̷̻̙̖͔͚͋̇̂̑͗̕̕͢ͅ ̴̶̮̣̪̣̟͚̅͊ͧͭ̂͂̓ ̴̶̮̣̪̣̟͚̅͊ͧͭ̂͂̓h̡͈̗̙͚̳͕̤͍̿̄͑ͥ͊̉h̡͈̗̙͚̳͕̤͍̿̄͑ͥ͊̉e̡̞͔̘͐̋͌̅̓̈͆̇̿͟͠e̡̞͔̘͐̋͌̅̓̈͆̇̿͟͠n̤͕͉̣̐͊́̽̍́͂͏̙̾ͅn̤͕͉̣̐͊́̽̍́͂͏̙̾ͅc̨̧̛̜͈͍͓̣̹ͮͧ̊͟͡ͅc̨̧̛̜͈͍͓̣̹ͮͧ̊͟͡ͅȩ̢̛͚̣͓͙ͣͮͩ̌̌́͟͠ȩ̢̛͚̣͓͙ͣͮͩ̌̌́͟͠ ̷̜̀ͪ̅͐͝͏̤̮̄͑̾ͬ͝ ̷̜̀ͪ̅͐͝͏̤̮̄͑̾ͬ͝t̙͔̳͕̝̝͔͐̊̓ͩ̈́ͪ̒͠t̙͔̳͕̝̝͔͐̊̓ͩ̈́ͪ̒͠h̩̖̪̠͂ͪ́͐̿͊ͨ̈̋̃͟h̩̖̪̠͂ͪ́͐̿͊ͨ̈̋̃͟e̋͆ͬ͞͏̵̘̹̙̂ͥ̐̀́ͅe̋͆ͬ͞͏̵̘̹̙̂ͥ̐̀́ͅ ̧̱͌̆̏͘͏̬̠̹̏ͦ̓͋̕ ̧̱͌̆̏͘͏̬̠̹̏ͦ̓͋̕f̸̧͎̣̪̯̻̗̟̎̎͂ͫ̕͝f̸̧͎̣̪̯̻̗̟̎̎͂ͫ̕͝a̟̓̃͒͏̸̧̣̱̎̽̏̐̓ͤa̟̓̃͒͏̸̧̣̱̎̽̏̐̓ͤć̯͚͕͡͏̴̛̜̺̣͙͉̀̍ć̯͚͕͡͏̴̛̜̺̣͙͉̀̍ț̢̹͇͈ͩ̋́̔̇̉ͤ̽͢ͅț̢̹͇͈ͩ̋́̔̇̉ͤ̽͢ͅo͈̥͊ͦ̔̃ͬ̎͛ͧ͌͌͟͟͝o͈̥͊ͦ̔̃ͬ̎͛ͧ͌͌͟͟͝r̸̨͕̙̟̟͈̼͔ͥͮ͋ͫ͒͝r̸̨͕̙̟̟͈̼͔ͥͮ͋ͫ͒͝ ̧̗̝̘͇͉̩̗̅̊̓͂̐͊͡ ̧̗̝̘͇͉̩̗̅̊̓͂̐͊͡o̬̻̪̹̘̥̳̲̫̳̫̳ͬͩ͊o̬̻̪̹̘̥̳̲̫̳̫̳ͬͩ͊f̡̣̜̗͚̙͇̓͑̍ͬ̅̚͞͞f̡̣̜̗͚̙͇̓͑̍ͬ̅̚͞͞ ̵̸͓̱̻̱̌̃ͫ̑̿͐͂ͨ̇ ̵̸͓̱̻̱̌̃ͫ̑̿͐͂ͨ̇2̡̝̥̯͚͔̖̫ͫ̽̔͑̅̚͞2̡̝̥̯͚͔̖̫ͫ̽̔͑̅̚͞ ̨̠̰̯̤͕̗̗̳͆̑͐͌̕͟ ̨̠̰̯̤͕̗̗̳͆̑͐͌̕͟i͇̫̲̲͓̖͙͖̱ͤ͊̎̃ͧ͢i͇̫̲̲͓̖͙͖̱ͤ͊̎̃ͧ͢n̵̢̻̦̭̏̓̂́͏̲̪̙̏͌n̵̢̻̦̭̏̓̂́͏̲̪̙̏͌ ̸̵̛̹͇͚̓̋̎̏̽̚̚͢͜ ̸̵̛̹͇͚̓̋̎̏̽̚̚͢͜t͈͔̤̲̬ͧͩ̔̀̂́͑̂ͭ͘t͈͔̤̲̬ͧͩ̔̀̂́͑̂ͭ͘h̡̻̙̖̪̱̍͗̍́͗́́̅͠h̡̻̙̖̪̱̍͗̍́͗́́̅͠e̶̖̩̜͐ͥͨͪͣ̆͋̋̉͢͡e̶̖̩̜͐ͥͨͪͣ̆͋̋̉͢͡ ̧͈̙̤͉͌ͩ̓͐̌̄ͦ͌ͥ͝ ̧͈̙̤͉͌ͩ̓͐̌̄ͦ͌ͥ͝c̨̨̹̗̬͕̩̈̑̉̃̑́̆͞c̨̨̹̗̬͕̩̈̑̉̃̑́̆͞h̭͚̦̻̘͈͆ͪ̿̌́̏̐͊͠h̭͚̦̻̘͈͆ͪ̿̌́̏̐͊͠ę̙͍͚ͮͦ́ͭͥ̈͑ͧ̕̕͢ę̙͍͚ͮͦ́ͭͥ̈͑ͧ̕̕͢c̝̭͓̹̙̠̄̍ͦ̌̏̉̇͛ͥc̝̭͓̹̙̠̄̍ͦ̌̏̉̇͛ͥk̶̡͚̦̰̣͖̔͌́̋͋̔ͥ̕k̶̡͚̦̰̣͖̔͌́̋͋̔ͥ̕.̛̱̖͓̼͚̲ͪ̆̈́̃̚͜͞ͅ.̛̱̖͓̼͚̲ͪ̆̈́̃̚͜͞ͅ ̶̴̺̹̜̺͇ͮ̉ͯ͋͗͝͝ͅ ̶̴̺̹̜̺͇ͮ̉ͯ͋͗͝͝ͅA͉̺̰̲̟̺͚͙̽́̀̌ͬͩ͠A͉̺̰̲̟̺͚͙̽́̀̌ͬͩ͠l̜͈̟͖̾͑̈́́̇ͮ̐ͦͮ͋̋l̜͈̟͖̾͑̈́́̇ͮ̐ͦͮ͋̋s̡̛̯̜̩̪̤̹̅͛̋̓͂̊͡s̡̛̯̜̩̪̤̹̅͛̋̓͂̊͡o̶͉̱͉̠̫̻ͤͣ̓ͭ͊͑͆ͅo̶͉̱͉̠̫̻ͤͣ̓ͭ͊͑͆ͅ ̢̗͉̝̞͗͛̊́ͦ͛̍̚͜͡ ̢̗͉̝̞͗͛̊́ͦ͛̍̚͜͡i̵̧͍̝̦̬̭̽̎̈̃̓̚͟͝i̵̧͍̝̦̬̭̽̎̈̃̓̚͟͝f̴̛̻̞̬̟͖̙̦̑́̀̍̇͝f̴̛̻̞̬̟͖̙̦̑́̀̍̇͝ ͓͚͇̺ͪ͛ͦͥ̓̎ͨ͒͊̚͟ ͓͚͇̺ͪ͛ͦͥ̓̎ͨ͒͊̚͟t̶̯͙̹̟͖̂̇͒̾ͭ͒͐͛̌t̶̯͙̹̟͖̂̇͒̾ͭ͒͐͛̌ḧ̠̬́͋̕͏̧̙̼͑͂̀̌̉̈ḧ̠̬́͋̕͏̧̙̼͑͂̀̌̉̈e̫̺͔̗̳͋̒͌ͤͬ̔͗̕̕͡e̫̺͔̗̳͋̒͌ͤͬ̔͗̕̕͡r̴̛̭͙͑́ͤ̓̒͊̈ͥ̑ͮ͞r̴̛̭͙͑́ͤ̓̒͊̈ͥ̑ͮ͞ȇ̷̛̝͓̜̮̩͙ͨ̎̎͛̌̽ȇ̷̛̝͓̜̮̩͙ͨ̎̎͛̌̽ ̸̩̤̘̖̳̻̋̈͛͑̈́̌̒͝ ̸̩̤̘̖̳̻̋̈͛͑̈́̌̒͝i̧͖͕̞̩̱̭̎̽̀͏̮ͬͨ͡i̧͖͕̞̩̱̭̎̽̀͏̮ͬͨ͡ś̶͍̞̉ͯ̊ͦͧ̀̆̌͂͞͝ś̶͍̞̉ͯ̊ͦͧ̀̆̌͂͞͝ ̡̛͙͈̜̗̻̮̞͍̝̝̓̅͋ ̡̛͙͈̜̗̻̮̞͍̝̝̓̅͋ȃ̢͍̟̬̳ͣ͏̝̔͑̎ͣ̕ͅȃ̢͍̟̬̳ͣ͏̝̔͑̎ͣ̕ͅ ̖̼͙̯͎̒͒ͩͣ̀̆͗̇͝ͅ ̖̼͙̯͎̒͒ͩͣ̀̆͗̇͝ͅļ̩̪͕͓͖͌̐͛̊ͫͪ̿͊ͤļ̩̪͕͓͖͌̐͛̊ͫͪ̿͊ͤȍ͍̜͓̣͈͍̳̄͊̄ͣ͢͝ͅȍ͍̜͓̣͈͍̳̄͊̄ͣ͢͝ͅt̵̷̛̤͎͕̖̹ͦ̇͒́͂ͮ͘t̵̷̛̤͎͕̖̹ͦ̇͒́͂ͮ͘ ̶̨̳̺̖̼̯͎̯̬̀̈̐ͦ͜ ̶̨̳̺̖̼̯͎̯̬̀̈̐ͦ͜o̷̡̯̼͇̻͙̙̙͒͂ͩ̉̀ͮo̷̡̯̼͇̻͙̙̙͒͂ͩ̉̀ͮf̻̺̖͉̊ͤ̇ͤ̍͌͛̐͟͟͝f̻̺̖͉̊ͤ̇ͤ̍͌͛̐͟͟͝ ̷̴̟̠̹͎̱̈́̈́̆̈́̇̃͘͞ ̷̴̟̠̹͎̱̈́̈́̆̈́̇̃͘͞p̸̸̧̨̦̪͕̮̀͒͒͌͌̑͢p̸̸̧̨̦̪͕̮̀͒͒͌͌̑͢a̯͙ͥͬ̊̓ͯ͌ͮ͊̎̂͒͢͠a̯͙ͥͬ̊̓ͯ͌ͮ͊̎̂͒͢͠ċ̛̫̱̥͕̳̲ͫͤ̎͐̄̕͠ċ̛̫̱̥͕̳̲ͫͤ̎͐̄̕͠k̵̡̹̲͓̾̉͏̶̻́ͩ̎̓̊k̵̡̹̲͓̾̉͏̶̻́ͩ̎̓̊e̵̡̢̢̪͍̲̣͒̒̍ͯͤ͊͑e̵̡̢̢̪͍̲̣͒̒̍ͯͤ͊͑ḑ̘͖̼̌ͭ̐͑ͯ͋ͬ̈́͆̓̚ḑ̘͖̼̌ͭ̐͑ͯ͋ͬ̈́͆̓̚ ̶̶̧̺̰͖̯͇̏̄̿ͤ̔͘͡ ̶̶̧̺̰͖̯͇̏̄̿ͤ̔͘͡c̩̝̯͂͆ͤ̈́͆̏̑̄ͦ͗̿͞c̩̝̯͂͆ͤ̈́͆̏̑̄ͦ͗̿͞e̥̫͌̓̚͏̵̧͙̣̻̞́̌̉e̥̫͌̓̚͏̵̧͙̣̻̞́̌̉l̵̪̣̪̘̙̟̘͊͒̿̏͗͝͠l̵̪̣̪̘̙̟̘͊͒̿̏͗͝͠l̨̮͚̟̺̹ͮ͊ͭ̊ͮ̿͘͘͟l̨̮͚̟̺̹ͮ͊ͭ̊ͮ̿͘͘͟s̩̯͌̊͒͝͏̥͑̈̎ͥ̀͟͞s̩̯͌̊͒͝͏̥͑̈̎ͥ̀͟͞,̡̙̻̭̪̭̖̀̇̒ͮ̈̒̇̕,̡̙̻̭̪̭̖̀̇̒ͮ̈̒̇̕ ̷̢̫̳̺͉̯̳̂̓ͨ͋͛͂͞ ̷̢̫̳̺͉̯̳̂̓ͨ͋͛͂͞p̟̰͖̹̦̲͙̉̑͐͑͗̀͟͢p̟̰͖̹̦̲͙̉̑͐͑͗̀͟͢á̳̜̈̓͛͠͏̢̄ͧ̉ͧͨͅá̳̜̈̓͛͠͏̢̄ͧ̉ͧͨͅs̵̢̥̖̰̥̼̯ͪͭ̿͗͌ͪ͊s̵̢̥̖̰̥̼̯ͪͭ̿͗͌ͪ͊s̘̱͕̓̅͛̆̅ͮ́̅ͧͣ̈̚s̘̱͕̓̅͛̆̅ͮ́̅ͧͣ̈̚ ̡̮̹̹̗͎̝ͣ̇́ͤ̾ͤ͞ͅ ̡̮̹̹̗͎̝ͣ̇́ͤ̾ͤ͞ͅt̡̛̝̯̱̖̠̱ͤ͒͆̍̚͘ͅt̡̛̝̯̱̖̠̱ͤ͒͆̍̚͘ͅh̛̯̻̩̘̤̳̿̓͂̐͐͡͡͠h̛̯̻̩̘̤̳̿̓͂̐͐͡͡͠ë́͏̩̣̗̺̟ͭ̃̾͂̀͘͢͝ë́͏̩̣̗̺̟ͭ̃̾͂̀͘͢͝m̸̜͔̮̥̹̲̖̲̄̈̊̎ͬ͠m̸̜͔̮̥̹̲̖̲̄̈̊̎ͬ͠ ̝͇̏͏̡̟̥͍ͦ̋̎ͤ̕͢͜ ̝͇̏͏̡̟̥͍ͦ̋̎ͤ̕͢͜ó͖̬̩̤̜̫͔͑̌ͫ̈́͡͏̇ó͖̬̩̤̜̫͔͑̌ͫ̈́͡͏̇f̴̶̴̥͔̗͖̬̆͒ͨ͊ͬͅͅf̴̶̴̥͔̗͖̬̆͒ͨ͊ͬͅͅ ̮̦̲̼̞̑ͨ͒̌ͤ̿́̌͐͒ ̮̦̲̼̞̑ͨ͒̌ͤ̿́̌͐͒t̛̮̰͔̋͂͗̓͋̇̅̆̒͊ͅt̛̮̰͔̋͂͗̓͋̇̅̆̒͊ͅo̴̢͚̮̳͙̟̳̐ͬ͑̃̓͡ͅo̴̢͚̮̳͙̟̳̐ͬ͑̃̓͡ͅ ̝̺͖͒ͯͭ̑ͤ͊ͮ͂̚̚͡ͅ ̝̺͖͒ͯͭ̑ͤ͊ͮ͂̚̚͡ͅt̶̨̲̦͉͖̲̤̗ͬ̋̋ͦ̂ͯt̶̨̲̦͉͖̲̤̗ͬ̋̋ͦ̂ͯh̶̨̤͍̜̯͈͇͛ͩ́̀͊ͮͩh̶̨̤͍̜̯͈͇͛ͩ́̀͊ͮͩe̡̢̤̰̰͙̭͈̓ͯͬ͑ͨ̃͢e̡̢̤̰̰͙̭͈̓ͯͬ͑ͨ̃͢ ̶̛̺̦̯̹̞͎̟̥̞̝̐̕ͅ ̶̛̺̦̯̹̞͎̟̥̞̝̐̕ͅU͙̓ͤ͏̟̗̟͕̥̰́͑̆ͧ͟U͙̓ͤ͏̟̗̟͕̥̰́͑̆ͧ͟I̪̩͕͕̼̎͐ͭͦ͐̋̐̅ͯ͢I̪̩͕͕̼̎͐ͭͦ͐̋̐̅ͯ͢ ̢̦͍̰̬͚ͯ́ͫ̿ͨ͘͘͝͡ ̢̦͍̰̬͚ͯ́ͫ̿ͨ͘͘͝͡t̴̨͉̟̥͛͐̇̀̎̒̄ͧ̀̇t̴̨͉̟̥͛͐̇̀̎̒̄ͧ̀̇o̸͇͏̷͍͉̠ͤ̈̾ͤ͛ͥ̚͝o̸͇͏̷͍͉̠ͤ̈̾ͤ͛ͥ̚͝ ̸̣͕̝̯̼́ͨͯ̏̋̂̑́͘ ̸̣͕̝̯̼́ͨͯ̏̋̂̑́͘l̵̨̛̬̱̣̬͚̺̫̳ͨ̽ͫͯl̵̨̛̬̱̣̬͚̺̫̳ͨ̽ͫͯḙ̳͕̝̥̄̆ͮ̄ͦͨ̓̚͏͢ḙ̳͕̝̥̄̆ͮ̄ͦͨ̓̚͏͢t̵̼̲̣̉̀̌ͭ̒̓͌̀͒̀ͅt̵̼̲̣̉̀̌ͭ̒̓͌̀͒̀ͅ ̺͈͙̘̣̩̙͕̋̇͆̀͊̇̂ ̺͈͙̘̣̩̙͕̋̇͆̀͊̇̂i̧̭̲̼̗̥̪̣̭ͬ͑ͨ̓̈̐i̧̭̲̼̗̥̪̣̭ͬ͑ͨ̓̈̐ṱ̴̯̲̱̫̲̥ͭ̀͋̂̚̕͟ṱ̴̯̲̱̫̲̥ͭ̀͋̂̚̕͟ ̖̗͔̦̲̻̌̀̏͆͊ͤ̑ͨ̉ ̖̗͔̦̲̻̌̀̏͆͊ͤ̑ͨ̉s̨̢͔̄̃ͯ̈́̑͐̚̕͢͝͝s̨̢͔̄̃ͯ̈́̑͐̚̕͢͝͝t͚͉͇̳̘̅̇̋͌̀̋́ͨ́͏t͚͉͇̳̘̅̇̋͌̀̋́ͨ́͏â͇͎̺ͨͭ͡͏͚̯ͤͩ͋ͧ͘â͇͎̺ͨͭ͡͏͚̯ͤͩ͋ͧ͘r̷̘̜͍̫̯ͭͪ͑͗̓͆͟͝ͅr̷̘̜͍̫̯ͭͪ͑͗̓͆͟͝ͅṭ̷̢̱͕̎ͬ̿́́͊̐͟͢͜ṭ̷̢̱͕̎ͬ̿́́͊̐͟͢͜ ̟̥̖͕̭͍̲ͩ̆̒̑͊̕̚͡ ̟̥̖͕̭͍̲ͩ̆̒̑͊̕̚͡p̶̴̨̧̜̰̜̘̉̽ͤ́͘͢͜p̶̴̨̧̜̰̜̘̉̽ͤ́͘͢͜ṙ̠̘̟̭͖̜̜͍ͦ̾͒͜͠͞ṙ̠̘̟̭͖̜̜͍ͦ̾͒͜͠͞ǒ̷̶̫͈̫̹ͬ͂ͧ̿̿̑͂͂ǒ̷̶̫͈̫̹ͬ͂ͧ̿̿̑͂͂c̵̸̮͕̮͈̞̼͎͉͍̑́ͭ̀c̵̸̮͕̮͈̞̼͎͉͍̑́ͭ̀e̸̡͔͓̳̞̟̘ͩ̍̽ͪͤͣ͢e̸̡͔͓̳̞̟̘ͩ̍̽ͪͤͣ͢s̨͍̬̖͂̏ͦ̀ͫ̐ͨͪ͡ͅͅs̨͍̬̖͂̏ͦ̀ͫ̐ͨͪ͡ͅͅs̡̻̹̜͍̺͌͛ͮ̓̀̈́̉̿͑s̡̻̹̜͍̺͌͛ͮ̓̀̈́̉̿͑í̤̜ͭ͢͏̶̶̹͍ͤͪ̉ͫ̋í̤̜ͭ͢͏̶̶̹͍ͤͪ̉ͫ̋ņ̵̼̪̭̲̤͙͖̒̑ͨ͂̌͘ņ̵̼̪̭̲̤͙͖̒̑ͨ͂̌͘g̷̢̢̭̘̙̥̖̲̭͛ͣ́̀̍g̷̢̢̭̘̙̥̖̲̭͛ͣ́̀̍ ̷̺̱̙̹͔̗͆ͪ̾ͬͬ̓͘͠ ̷̺̱̙̹͔̗͆ͪ̾ͬͬ̓͘͠t̛̤̻̲̓̓ͩ̇ͫͫ͆ͦ͛̕͞t̛̤̻̲̓̓ͩ̇ͫͫ͆ͦ͛̕͞ḣ̜̘̮̫͍̗͇ͥͯ̽̕͢͢͝ḣ̜̘̮̫͍̗͇ͥͯ̽̕͢͢͝ẹ̵̱̹͚͖̜̞̆̾͌ͬ̀̿͞ẹ̵̱̹͚͖̜̞̆̾͌ͬ̀̿͞m̶̢̧͉̝̜̝̍͑͂̽̌̉̈́ͩm̶̢̧͉̝̜̝̍͑͂̽̌̉̈́ͩ'
    api.nvim_buf_set_lines(0, 0, -1, true, { text })

    screen:try_resize(250, 10)
    screen:expect({
      grid = [[
        {100:  1 }{101:^c̯̥̳̮̫̳͔̱̀ͦͩ̄͋̓͘c̯̥̳̮̫̳͔̱̀ͦͩ̄͋̓͘l̶̴̴̨̛͓͚͎̥ͦͤ̑͆͛͢l̶̴̴̨̛͓͚͎̥ͦͤ̑͆͛͢ô̷̤̩ͯͧ͆ͪ̈́́͒̒̐͐̕ô̷̤̩ͯͧ͆ͪ̈́́͒̒̐͐̕s̶̷̢̩̱̠̀ͦ̽ͮ͒ͨ̚͟͠s̶̷̢̩̱̠̀ͦ̽ͮ͒ͨ̚͟͠e͉̯̱̖͕̳̼̽̊̒ͣ̊ͥ̚̚e͉̯̱̖͕̳̼̽̊̒ͣ̊ͥ̚̚ ͇͌̈̄ͬͧͧ͝͠͏̷ͪ̎͟͠ ͇͌̈̄ͬͧͧ͝͠͏̷ͪ̎͟͠t̵̳̅̽͏̵̡̥̲͍͙̹̯ͩ̐t̵̳̅̽͏̵̡̥̲͍͙̹̯ͩ̐o͋͏̬̞̣ͬ́ͬ̎̋̓̽͘͠͡o͋͏̬̞̣ͬ́ͬ̎̋̓̽͘͠͡ ͏̛͙̮̈ͬ̌͐ͤ͒ͧ̎ͤ͜͠ ͏̛͙̮̈ͬ̌͐ͤ͒ͧ̎ͤ͜͠ǫ̛̳͕̦͖̪̀͂͛̅̔ͪ͒͜ǫ̛̳͕̦͖̪̀͂͛̅̔ͪ͒͜v̸̡͖̗̣ͯ̿̔͊̅ͯ̈̓̇ͅv̸̡͖̗̣ͯ̿̔͊̅ͯ̈̓̇ͅe̴̶̢̜̭̠̰̞ͪͨ͂ͤ͆́͗e̴̶̢̜̭̠̰̞ͪͨ͂ͤ͆́͗r̨̖̼̳̳͚̖̒ͯ̊̋̂̑̚͞r̨̖̼̳̳͚̖̒ͯ̊̋̂̑̚͞f̵̮̣̬̾͛̌ͧͦ͘͢͜͜͠ͅf̵̮̣̬̾͛̌ͧͦ͘͢͜͜͠ͅl̡͉̬̳͈̠̏͂ͤ̈ͨ̀ͩ̔͏l̡͉̬̳͈̠̏͂ͤ̈ͨ̀ͩ̔͏ỏ̶̢̻̠͎͇͈̜̈̆ͯ̔ͩ̾ỏ̶̢̻̠͎͇͈̜̈̆ͯ̔ͩ̾ẉ̦̞̼̩̣͙̅̿́̓̉̎̈ͪẉ̦̞̼̩̣͙̅̿́̓̉̎̈ͪi̷̡͍͓͔̝͙͖͖̍ͯͤͬͦ͝i̷̡͍͓͔̝͙͖͖̍ͯͤͬͦ͝n̠̪̰͋ͩ̆͏̶̖̯ͭ̀̿͛͘n̠̪̰͋ͩ̆͏̶̖̯ͭ̀̿͛͘g̴̭͎̞͙̗̯͙͖ͬ́ͧͧ͝ͅg̴̭͎̞͙̗̯͙͖ͬ́ͧͧ͝ͅ ̢͎̬͓̮̹̰̙͍̓̐͋͂͐̚ ̢͎̬͓̮̹̰̙͍̓̐͋͂͐̚ť̴̼̝̗ͩ̍͋̿͏͇ͧ̑̏̚ť̴̼̝̗ͩ̍͋̿͏͇ͧ̑̏̚h̨̨̢͓̲͚͕̦ͨ͛̓ͩ̚͟͠h̨̨̢͓̲͚͕̦ͨ͛̓ͩ̚͟͠ȩ͕̜̥̃̑͋̏͐̎̒͛͊͏͙ȩ͕̜̥̃̑͋̏͐̎̒͛͊͏͙ ̵̨̜̜̠͉̱͎͑ͥ̌͐̽͢͡ ̵̨̜̜̠͉̱͎͑ͥ̌͐̽͢͡r̢̫͓͎̙̭̽ͥͭ͐̂̀̕͟͝r̢̫͓͎̙̭̽ͥͭ͐̂̀̕͟͝e̴̷͓̹̩ͧ́̉̑̈̿ͥ̕͡͝e̴̷͓̹̩ͧ́̉̑̈̿ͥ̕͡͝d̵̡̼̩̠̜͈̯̬͚͛̋̀̆͟d̵̡̼̩̠̜͈̯̬͚͛̋̀̆͟ŕ̻̳̬͏̨̮͚̲ͥ̌͆͗͠ͅŕ̻̳̬͏̨̮͚̲ͥ̌͆͗͠ͅā̸͙̥̤͍͈̣ͪͨ̈͋̈́̈́͜ā̸͙̥̤͍͈̣ͪͨ̈͋̈́̈́͜w̶̦̪͚̤̃ͬ̓͒ͤ̇̇ͮ͢͡w̶̦̪͚̤̃ͬ̓͒ͤ̇̇ͮ͢͡ ̷̡̦̻̻̪͚̳͎ͥ̓ͩͪ͠͝ ̷̡̦̻̻̪͚̳͎ͥ̓ͩͪ͠͝b̳ͮ̒̊̆̒́̈́̏̓͋ͭ̔ͤ̚b̳ͮ̒̊̆̒́̈́̏̓͋ͭ̔ͤ̚u̧̟̫͎͚̭̠͕͂̄̀̒̈̇͜u̧̟̫͎͚̭̠͕͂̄̀̒̈̇͜f̶̸̢͍̑̂̊ͥͫ̈́ͥ͛̈́̃͝f̶̸̢͍̑̂̊ͥͫ̈́ͥ͛̈́̃͝f̵̼̭̮͎ͧ̒̆͊̀ͤ͊̇̕͡f̵̼̭̮͎ͧ̒̆͊̀ͤ͊̇̕͡e̮̪̲̣̞̖͎ͥͤ̐̌̓̐͟͢e̮̪̲̣̞̖͎ͥͤ̐̌̓̐͟͢ŗ̭̘̮̹̖̎̄̆́ͫͭ͢ͅͅŗ̭̘̮̹̖̎̄̆́ͫͭ͢ͅͅ.̪̖͉͚̹̾̉ͮ̔̊ͪ̾̎͟͞.̪̖͉͚̹̾̉ͮ̔̊ͪ̾̎͟͞ ̷̧̺̰͎̣̃͒͗ͮ͑ͪͮ͞ͅ ̷̧̺̰͎̣̃͒͗ͮ͑ͪͮ͞ͅf̛̫͚͚͍̜͎̗̳̂ͬͭ͜͢͝f̛̫͚͚͍̜͎̗̳̂ͬͭ͜͢͝i̴̵̡̛̛͎̤̳̮ͩ̐ͩ͑̇̚i̴̵̡̛̛͎̤̳̮ͩ̐ͩ͑̇̚n̵͇͍͚̖̥̣ͨ̄ͧ̌̂͗͘͝n̵͇͍͚̖̥̣ͨ̄ͧ̌̂͗͘͝ȉ̼̱̫̜̋̓̐͌͆ͨ͘͝͡ͅȉ̼̱̫̜̋̓̐͌͆ͨ͘͝͡ͅs̴̸̝̺̬͚̲̹̘ͪ̆̊ͥ͞͝s̴̸̝̺̬͚̲̹̘ͪ̆̊ͥ͞͝h̴̜̠͇ͦͥ̔̅ͭͭ͋ͩ͟͡͞h̴̜̠͇ͦͥ̔̅ͭͭ͋ͩ͟͡͞ ̶̧̛̻͙̤̘̺̣̻̗͍̓͑͠ ̶̧̛̻͙̤̘̺̣̻̗͍̓͑͠t̠͉̼̬̩͛́ͨ͐̀͛̂ͨ̾͞t̠͉̼̬̩͛́ͨ͐̀͛̂ͨ̾͞h̻̹̝̹̾ͩ̍ͧ͆ͥ̔͘͏̉ͯh̻̹̝̹̾ͩ̍ͧ͆ͥ̔͘͏̉ͯì̷̢̛̺̭͇̟̦̄̓́̓ͪ͟ì̷̢̛̺̭͇̟̦̄̓́̓ͪ͟s̴̡̗͍͕͖̮̟̱̫̎ͣ̀̎̿s̴̡̗͍͕͖̮̟̱̫̎ͣ̀̎̿ ̶͇̟̜̗̗͇͇͐̑̈͋̋̽͟ ̶͇̟̜̗̗͇͇͐̑̈͋̋̽͟e̷̥͙͈̖̤̯̹̯͗̉̈́̽ͨ̕e̷̥͙͈̖̤̯̹̯͗̉̈́̽ͨ̕v̛̝͕̱͙̞̖̒̂̔͆̊̎́ͫv̛̝͕̱͙̞̖̒̂̔͆̊̎́ͫę̴̤̭͖̈̐̿͂ͣ͒̃ͭ̕͟ę̴̤̭͖̈̐̿͂ͣ͒̃ͭ̕͟ṇ̳̭͓̟̠͕̯͑̉ͦ̀ͯ͜͡ṇ̳̭͓̟̠͕̯͑̉ͦ̀ͯ͜͡t̢̞͔̣̄̀͆̂̃ͨͦ͜͜͝͠t̢̞͔̣̄̀͆̂̃ͨͦ͜͜͝͠,̬̳̮͓͉̟͉͓̦͔̑̄ͨ̎͜,̬̳̮͓͉̟͉͓̦͔̑̄ͨ̎͜ ͕͈̠̰̬̬̌͂̏ͥ̀̕̚͢͠ ͕͈̠̰̬̬̌͂̏ͥ̀̕̚͢͠f̺̮̞̈́̏̋͏̺̖̝̓̑̂̚͢f̺̮̞̈́̏̋͏̺̖̝̓̑̂̚͢l̜̪͍̩̩̟͉͓̊̓ͤ̆ͣͫ̌l̜̪͍̩̩̟͉͓̊̓ͤ̆ͣͫ̌u̷̧̞̳̱̜̟̲͑̐͂ͪ͛͘͟u̷̧̞̳̱̜̟̲͑̐͂ͪ͛͘͟s͔͍̽ͣͮ̏̓͛̄ͯ̽͂̐ͯ͜s͔͍̽ͣͮ̏̓͛̄ͯ̽͂̐ͯ͜ḩ̼͕̦͖̼͚̗̃̃ͥ̅̂̈͟ḩ̼͕̦͖̼͚̗̃̃ͥ̅̂̈͟,̵͍̮̮̟͚̝̃ͨ̿ͭ̌ͤ̋̊,̵͍̮̮̟͚̝̃ͨ̿ͭ̌ͤ̋̊ ̷̨̨͈̝̦͂ͦ̒̋̇ͧ̒͟͝ ̷̨̨͈̝̦͂ͦ̒̋̇ͧ̒͟͝a̡̨̲̖̾̂͗̚͢͡͏͈ͤ̉͡a̡̨̲̖̾̂͗̚͢͡͏͈ͤ̉͡nͫͤ̚͜͏̧̛̣̻ͩ̔̍ͦ̕͝nͫͤ̚͜͏̧̛̣̻ͩ̔̍ͦ̕͝d͈̻̗̼̀͡͏̶̵̟̹̻̎̾ͯd͈̻̗̼̀͡͏̶̵̟̹̻̎̾ͯ ̶͙͈̠̜̬̺͛̀̊̂ͪ̔ͮ͑ ̶͙͈̠̜̬̺͛̀̊̂ͪ̔ͮ͑ş̧̡̢͓̠͋ͫ͑͒͊̅̔͜͞ş̧̡̢͓̠͋ͫ͑͒͊̅̔͜͞t̤̭͓̰̣̉̔̎ͫ͛ͦ̑̕͟͞t̤̭͓̰̣̉̔̎ͫ͛ͦ̑̕͟͞a̠͇͇̯̥͍͚̳̒́͐͐̏͋̓a̠͇͇̯̥͍͚̳̒́͐͐̏͋̓r͉͈̻̻͕̩̩̃̅͋͆ͮ͢͢͡r͉͈̻̻͕̩̩̃̅͋͆ͮ͢͢͡t̵̛̝̗̥̙̟̆ͮ̽̏ͧ͜͠ͅt̵̛̝̗̥̙̟̆ͮ̽̏ͧ͜͠ͅ ̷̼͎̦ͫ̈͑̃̽͏̲̪ͣͯ̽ ̷̼͎̦ͫ̈͑̃̽͏̲̪ͣͯ̽á̸̷̴̟̣́̔̌͏̶͕͋ͭͭá̸̷̴̟̣́̔̌͏̶͕͋ͭͭ ̧̧̲͍̘̘͚͔͇͙ͨͬ́̊ͅ ̧̧̲͍̘̘͚͔͇͙ͨͬ́̊ͅn̸̸̩͖͔͚͚̖͗ͦ̓̀̀̈́̈́n̸̸̩͖͔͚͚̖͗ͦ̓̀̀̈́̈́ę̵̧̬̣̦̖̝̹̱͌̃̑ͣ̚ę̵̧̬̣̦̖̝̹̱͌̃̑ͣ̚w͍̥͚̺ͮ̓̈̈́̾̊̆́̚͜͝w͍̥͚̺ͮ̓̈̈́̾̊̆́̚͜͝ ̛̹̲̥̝͙̾ͨ̆̎̃̋͂̓̕ ̛̹̲̥̝͙̾ͨ̆̎̃̋͂̓̕"̴̜̰̰̱̖̙̘̈́͌ͨͪ̐̕͠"̴̜̰̰̱̖̙̘̈́͌ͨͪ̐̕͠g̸̛͇̐͊͂̽͢͏͖̣ͫ͊ͯͅg̸̛͇̐͊͂̽͢͏͖̣ͫ͊ͯͅr̴̨̲͎̹͇̠̐ͤ̇̒ͬ̆ͧ͞r̴̨̲͎̹͇̠̐ͤ̇̒ͬ̆ͧ͞į̝̱̩͔̈ͨ̉͌̋̍̂͜͟͞į̝̱̩͔̈ͨ̉͌̋̍̂͜͟͞d̷̴̷̟̎͌͑͛̈́ͭͨͯ̋ͭ̕d̷̴̷̟̎͌͑͛̈́ͭͨͯ̋ͭ̕_̢̭̙̦̪͇̾̔̆ͬͦ́ͥ͢͡_̢̭̙̦̪͇̾̔̆ͬͦ́ͥ͢͡l̡̢̨̧͔̱̥̹̬̆ͮ̈́̏̅͜l̡̢̨̧͔̱̥̹̬̆ͮ̈́̏̅͜ĩ̢͖̠̩̞͚̰̰̉̋̌͛ͪ͠ĩ̢͖̠̩̞͚̰̰̉̋̌͛ͪ͠n̢̬̜̘̲͉ͮ͆͏̯͕ͭͦ̉̅n̢̬̜̘̲͉ͮ͆͏̯͕ͭͦ̉̅ē̡͈̮̭̰̜͍̗̮͔͌͐͆ͫē̡͈̮̭̰̜͍̗̮͔͌͐͆ͫ"̦̠̟ͣ̽͋͐ͧ̓̂̆̎͒͝ͅ"̦̠̟ͣ̽͋͐ͧ̓̂̆̎͒͝ͅ ̴̡̺̹̖̰̀ͤ͊̊͗̊́͜͠ ̴̡̺̹̖̰̀ͤ͊̊͗̊́͜͠ẻ͎̳̻̯̹͓͊̌̄͑͂̉͜͢ẻ͎̳̻̯̹͓͊̌̄͑͂̉͜͢v̞̬̪̥͖ͤ͐̍́ͤ̇̀̕̕ͅv̞̬̪̥͖ͤ͐̍́ͤ̇̀̕̕ͅȇ̶̱̗̩̠͚͎͊ͤͪͦͫ̂̚ȇ̶̱̗̩̠͚͎͊ͤͪͦͫ̂̚n̢̮̜͉̎ͨͩ̒̓ͬͨ̓ͦ͘͜n̢̮̜͉̎ͨͩ̒̓ͬͨ̓ͦ͘͜t̢͚͉̹͇̺̭ͪ̄̉ͨ̄͐̕ͅt̢͚͉̹͇̺̭ͪ̄̉ͨ̄͐̕ͅ ̵̛̳̱͍̩͓ͣ͋̈́͐ͭͧ̿ͅ ̵̛̳̱͍̩͓ͣ͋̈́͐ͭͧ̿ͅã̰̪̙͉̪͇ͣ͋ͤ̓͋͑̕͘ã̰̪̙͉̪͇ͣ͋ͤ̓͋͑̕͘t̴̴̡̡̛̳̼̻̳̂̽͒̇̚͠t̴̴̡̡̛̳̼̻̳̂̽͒̇̚͠ ̴̙̤̹͔̳̟̽̀̆ͥ̂̕͘͝ ̴̙̤̹͔̳̟̽̀̆ͥ̂̕͘͝t̴͖̼͔̬̦̏ͩ̄́̓̊̔̇͡t̴͖̼͔̬̦̏ͩ̄́̓̊̔̇͡ḫ̷͚͇̫̫̠͒ͮ̍ͣ̃͐͐̍ḫ̷͚͇̫̫̠͒ͮ̍ͣ̃͐͐̍e̳͉̮͖̳̣̎͌̂ͣ͋ͯ̆͜͞e̳͉̮͖̳̣̎͌̂ͣ͋ͯ̆͜͞ ̪̖̉̌̀̽̄̍̓̀̂̋͐̈̚ ̪̖̉̌̀̽̄̍̓̀̂̋͐̈̚c̶̫̜͚̊̿̂̿ͥͭ̓̂̈́͘͡c̶̫̜͚̊̿̂̿ͥͭ̓̂̈́͘͡ṷ̵̷͉̯̜̬̝̑͛ͤ̋ͧͯ̉ṷ̵̷͉̯̜̬̝̑͛ͤ̋ͧͯ̉ŗ̙̬̺̬ͥͤ̓͐̈́ͬ̽̌͡͡ŗ̙̬̺̬ͥͤ̓͐̈́ͬ̽̌͡͡r̛͖̼̣͙̋̇̀̅͒̽ͥ̑ͅͅr̛͖̼̣͙̋̇̀̅͒̽ͥ̑ͅͅě̢̪̦̼͘͏̴̠̞͍̓̐ͭ̇ě̢̪̦̼͘͏̴̠̞͍̓̐ͭ̇ǹ̨̨̛̛̯̍ͭͫ̎̍̃̄̐̍ǹ̨̨̛̛̯̍ͭͫ̎̍̃̄̐̍t̵̳͔̭̮̭̱̰ͤ͐̉̾͗̅͢t̵̳͔̭̮̭̱̰ͤ͐̉̾͗̅͢ ̩͕ͧ̌͂͂́ͫͥ̍͏̬̙͗ͅ ̩͕ͧ̌͂͂́ͫͥ̍͏̬̙͗ͅp̞̞͇͇̯̩̬̜ͥ̃͐̑͋́͗p̞̞͇͇̯̩̬̜ͥ̃͐̑͋́͗o̵͖͈̩̪̥̝̊̒̉̿͋ͩ̆͠o̵͖͈̩̪̥̝̊̒̉̿͋ͩ̆͠s̶͇̺̩̟̺͋̆̒ͫͥ̆͏͔͜s̶͇̺̩̟̺͋̆̒ͫͥ̆͏͔͜i̞̝̭̲͈̝̮̫͚͑̓ͤ̎ͮ̀i̞̝̭̲͈̝̮̫͚͑̓ͤ̎ͮ̀t̷̬͇̗̥̳͔̮͔̎̀ͣ̊̕͢t̷̬͇̗̥̳͔̮͔̎̀ͣ̊̕͢i̶̧̗̠̗̲͑̋ͣͪͦͣ͆̕͢i̶̧̗̠̗̲͑̋ͣͪͦͣ͆̕͢ǫ̰̟̙ͯ͏̬̦͚̝̀̈́ͣ͘͜ǫ̰̟̙ͯ͏̬̦͚̝̀̈́ͣ͘͜n̸̻̩͖͓̠̲̮̐̄̌̇́́͠n̸̻̩͖͓̠̲̮̐̄̌̇́́͠}|
        {100:    }{101:.̧̛͈̙̭̉ͥ̋͛̏͋̂̿͛͞.̧̛͈̙̭̉ͥ̋͛̏͋̂̿͛͞ ̡̲͇̳͔̦̤̹̥ͣ̍ͪͮ̎͡ ̡̲͇̳͔̦̤̹̥ͣ̍ͪͮ̎͡F̼̣͙̳̞͑͒ͧͣ̔̊̇̓̈́͠F̼̣͙̳̞͑͒ͧͣ̔̊̇̓̈́͠ŏ̸̶͎̘̟͙̤̑̒̿͗͛̐̚ŏ̸̶͎̘̟͙̤̑̒̿͗͛̐̚r̵̰͇͚̥̻͚̈́̌̽ͨͥ̕͘͞r̵̰͇͚̥̻͚̈́̌̽ͨͥ̕͘͞ ̵̶̧͖̙̝̻̭̤̒̿͌͗͑͡ ̵̶̧͖̙̝̻̭̤̒̿͌͗͑͡ś̸̬̘̭̼͓̹ͥ͛ͪ̐̈́͆͝ś̸̬̘̭̼͓̹ͥ͛ͪ̐̈́͆͝i͎̺̪̖̻ͫ̀ͤ̾ͦ̽ͭ͒̒͘i͎̺̪̖̻ͫ̀ͤ̾ͦ̽ͭ͒̒͘m̨̩͎̫̫̙̃̈͆ͬ̊͠͏̽͞m̨̩͎̫̫̙̃̈͆ͬ̊͠͏̽͞p̴̢̨͖̼͍̲͍̲ͩ̊̒̌̃ͤp̴̢̨͖̼͍̲͍̲ͩ̊̒̌̃ͤļ͔̱͙̝̟̜͚͎͕̮̹ͣ̀͘ļ͔̱͙̝̟̜͚͎͕̮̹ͣ̀͘ḭ̢̛͈͍̠̟̪̑̎̈́̑̽͜͡ḭ̢̛͈͍̠̟̪̑̎̈́̑̽͜͡c̖̠̥͔̪̼̑̃̆̓ͫ͗ͩͩ̋c̖̠̥͔̪̼̑̃̆̓ͫ͗ͩͩ̋i̴̸̡̘͇͔̹͂̍̐̀ͬͩ̈͘i̴̸̡̘͇͔̹͂̍̐̀ͬͩ̈͘t̵̵̖̫͙͎̒ͭ̔̃̔ͧͫ̽͝t̵̵̖̫͙͎̒ͭ̔̃̔ͧͫ̽͝y̴̘̱͈̳ͯ̈́̍ͮ̔͊͊̚̚͞y̴̘̱͈̳ͯ̈́̍ͮ̔͊͊̚̚͞ ̛̩̤̪͇̬͕̯̞̙̪̮̀̂̕ ̛̩̤̪͇̬͕̯̞̙̪̮̀̂̕ļ̲̰̞̈̌̿͐̉́̋ͬ͟͟͡ļ̲̰̞̈̌̿͐̉́̋ͬ͟͟͡e̡̧̛̬̣̗̣̰͂̐̂͗͛̋̇e̡̧̛̬̣̗̣̰͂̐̂͗͛̋̇a̡̯͉̠̞ͩͮ̉̓́ͮ̔̆̒͘a̡̯͉̠̞ͩͮ̉̓́ͮ̔̆̒͘v͔ͦ͏͎̪̮̝̟̤́̀͊̈̎͟v͔ͦ͏͎̪̮̝̟̤́̀͊̈̎͟ȩ̪͔̤̺͗ͦ̅̓ͭͤͫ̆ͣ͒ȩ̪͔̤̺͗ͦ̅̓ͭͤͫ̆ͣ͒ ̶̢̤͎̰̝̤͂̔͒ͦͦ͂̊̀ ̶̢̤͎̰̝̤͂̔͒ͦͦ͂̊̀p̛̝̪͚͖̫͕̅̍̊́̓͒̂̃p̛̝̪͚͖̫͕̅̍̊́̓͒̂̃l̴̸͉͎̖͕̦̥̾ͨ̌̑ͣ̕͝l̴̸͉͎̖͕̦̥̾ͨ̌̑ͣ̕͝a̰̩͔̼͔̦̩͒ͪͭ̐͡͏̗ͮa̰̩͔̼͔̦̩͒ͪͭ̐͡͏̗ͮč̵̱͇̲̞̉̆̌̄ͧͫ̑̈́͠č̵̱͇̲̞̉̆̌̄ͧͫ̑̈́͠e̶̛̜̬̯̤͔̓ͤͮͦ̂͐͜͜e̶̛̜̬̯̤͔̓ͤͮͦ̂͐͜͜ ̸̧̼̯͕̼̭ͣͣ̿̑̈̎̽͜ ̸̧̼̯͕̼̭ͣͣ̿̑̈̎̽͜f̴̼̞͇̺̊̎̓̑̔̈́͗̊͘͞f̴̼̞͇̺̊̎̓̑̔̈́͗̊͘͞ő̶̶̧̺̤̜̹̫͎̻̏̉̍ͩő̶̶̧̺̤̜̹̫͎̻̏̉̍ͩr̵̙̱̺̮͙̯̼͐̀͗̾͊͟͝r̵̙̱̺̮͙̯̼͐̀͗̾͊͟͝ ̵̪̥͍̩̱ͨ͗̓̎̓͒̚͜͞ ̵̪̥͍̩̱ͨ͗̓̎̓͒̚͜͞t̪̤͙̥̖̹̣̤͎̞̒́̒̽̃t̪̤͙̥̖̹̣̤͎̞̒́̒̽̃ḥ̵̷̷̭̘͙͓ͩ̓ͪͤ̓ͮ̍ḥ̵̷̷̭̘͙͓ͩ̓ͪͤ̓ͮ̍é̡̬̺͉̻̻̀̀͂̽ͬ̕̕͠é̡̬̺͉̻̻̀̀͂̽ͬ̕̕͠ ̵̱̻̣͙̌͊́ͯ́̐̀́͂̕ ̵̱̻̣͙̌͊́ͯ́̐̀́͂̕f̨̪͈̲̬̟̝͎̀̔̋ͨ̀͡ͅf̨̪͈̲̬̟̝͎̀̔̋ͨ̀͡ͅi̵̫̱̝͒̇̏̃ͭ̂̄̄̊̐͠i̵̫̱̝͒̇̏̃ͭ̂̄̄̊̐͠n̵̷͖̻̦͍̻̑̈́̎̓͑ͪͫ͟n̵̷͖̻̦͍̻̑̈́̎̓͑ͪͫ͟a̗̗̠̫̦̻̹͇ͯͦͫ͗̽ͥ̚a̗̗̠̫̦̻̹͇ͯͦͫ͗̽ͥ̚l̵̨̨͙̜͔̘̗̯͌̋͂̑̄͢l̵̨̨͙̜͔̘̗̯͌̋͂̑̄͢ ̵̷̛̺̳͙̲̥̬̊̌̽̓̇͝ ̵̷̛̺̳͙̲̥̬̊̌̽̓̇͝"̸̨̫̟̤̥͉̮̥̀ͤ̐͊̇̑"̸̨̫̟̤̥͉̮̥̀ͤ̐͊̇̑c̵̭̘͍͇̜ͨ̔̍̆͛͌͂͌͛c̵̭̘͍͇̜ͨ̔̍̆͛͌͂͌͛l̸̷̢̡̫̩̃́͐̆͒ͨ̔̄͟l̸̷̢̡̫̩̃́͐̆͒ͨ̔̄͟è̙͎̝̞̠̹͍́́̅͐͌͘ͅè̙͎̝̞̠̹͍́́̅͐͌͘ͅą̶̢̡̘̦͖̰̌̎͌ͩ́̓ͮą̶̢̡̘̦͖̰̌̎͌ͩ́̓ͮȑ̹̻̦͙̠͂̽͆ͫ͛͟͏̺͡ȑ̹̻̦͙̠͂̽͆ͫ͛͟͏̺͡"̣̞͕̇ͯ̓̽ͪ͑́̍̚͘͘ͅ"̣̞͕̇ͯ̓̽ͪ͑́̍̚͘͘ͅ ̛̩̘̗̜̗̥̂͊ͥ̀ͨͫ̾ͧ ̛̩̘̗̜̗̥̂͊ͥ̀ͨͫ̾ͧe̵̡͚͉̤̯̮͛̎̓ͪ̾̉̆͟e̵̡͚͉̤̯̮͛̎̓ͪ̾̉̆͟l̢̩͓̙͓͍̈̊ͫͣ́̅ͧ͛͞l̢̩͓̙͓͍̈̊ͫͣ́̅ͧ͛͞e̠̱͉̯͔̙͔̓ͩ̃͋͌ͬͭ͋e̠̱͉̯͔̙͔̓ͩ̃͋͌ͬͭ͋m̧̡̛̤͕̻̳̽͛̓̈́ͣ̊̚͟m̧̡̛̤͕̻̳̽͛̓̈́ͣ̊̚͟e̯͎̺͔̼̾͂͐̄ͮ͌̍͑̕ͅe̯͎̺͔̼̾͂͐̄ͮ͌̍͑̕ͅṉ̸͔̩̙̙̹͚̭ͩ͛͗̀̾ͅṉ̸͔̩̙̙̹͚̭ͩ͛͗̀̾ͅt̸̢̧̝͉̺͉̓́̇ͯ̕͠͠͝t̸̢̧̝͉̺͉̓́̇ͯ̕͠͠͝ ̳͔̜̹̘̪̅̋̎̉͆̑ͤ͡͞ ̳͔̜̹̘̪̅̋̎̉͆̑ͤ͡͞a̷̳͓̞̱͈̓́̒̔̆ͩͯ̊͠a̷̳͓̞̱͈̓́̒̔̆ͩͯ̊͠s̷̛͙̘̳͙͓̠̞̫̆̓̚̚͡s̷̛͙̘̳͙͓̠̞̫̆̓̚̚͡ ̷̢̦̘̮̰͕̏̀ͩ̓͂ͦ͟ͅ ̷̢̦̘̮̰͕̏̀ͩ̓͂ͦ͟ͅw̮͔͇͕̝͖ͪ͒̏̇ͪ̇̍ͦ͠w̮͔͇͕̝͖ͪ͒̏̇ͪ̇̍ͦ͠ẹ̗͚̮̭̓̍̐̃̍́̐͘̕͘ẹ̗͚̮̭̓̍̐̃̍́̐͘̕͘l̴̡̩̱̞͛ͤ͑͑͂̀ͣ̉̌̕l̴̡̩̱̞͛ͤ͑͑͂̀ͣ̉̌̕l̷̨̘̤̣̯͇̲̝ͨ̍ͦ͝͝ͅl̷̨̘̤̣̯͇̲̝ͨ̍ͦ͝͝ͅ,̷̻̙̖͔͚͋̇̂̑͗̕̕͢ͅ,̷̻̙̖͔͚͋̇̂̑͗̕̕͢ͅ ̴̶̮̣̪̣̟͚̅͊ͧͭ̂͂̓ ̴̶̮̣̪̣̟͚̅͊ͧͭ̂͂̓h̡͈̗̙͚̳͕̤͍̿̄͑ͥ͊̉h̡͈̗̙͚̳͕̤͍̿̄͑ͥ͊̉e̡̞͔̘͐̋͌̅̓̈͆̇̿͟͠e̡̞͔̘͐̋͌̅̓̈͆̇̿͟͠n̤͕͉̣̐͊́̽̍́͂͏̙̾ͅn̤͕͉̣̐͊́̽̍́͂͏̙̾ͅc̨̧̛̜͈͍͓̣̹ͮͧ̊͟͡ͅc̨̧̛̜͈͍͓̣̹ͮͧ̊͟͡ͅȩ̢̛͚̣͓͙ͣͮͩ̌̌́͟͠ȩ̢̛͚̣͓͙ͣͮͩ̌̌́͟͠ ̷̜̀ͪ̅͐͝͏̤̮̄͑̾ͬ͝ ̷̜̀ͪ̅͐͝͏̤̮̄͑̾ͬ͝t̙͔̳͕̝̝͔͐̊̓ͩ̈́ͪ̒͠t̙͔̳͕̝̝͔͐̊̓ͩ̈́ͪ̒͠h̩̖̪̠͂ͪ́͐̿͊ͨ̈̋̃͟h̩̖̪̠͂ͪ́͐̿͊ͨ̈̋̃͟e̋͆ͬ͞͏̵̘̹̙̂ͥ̐̀́ͅe̋͆ͬ͞͏̵̘̹̙̂ͥ̐̀́ͅ ̧̱͌̆̏͘͏̬̠̹̏ͦ̓͋̕ ̧̱͌̆̏͘͏̬̠̹̏ͦ̓͋̕f̸̧͎̣̪̯̻̗̟̎̎͂ͫ̕͝f̸̧͎̣̪̯̻̗̟̎̎͂ͫ̕͝a̟̓̃͒͏̸̧̣̱̎̽̏̐̓ͤa̟̓̃͒͏̸̧̣̱̎̽̏̐̓ͤć̯͚͕͡͏̴̛̜̺̣͙͉̀̍ć̯͚͕͡͏̴̛̜̺̣͙͉̀̍ț̢̹͇͈ͩ̋́̔̇̉ͤ̽͢ͅț̢̹͇͈ͩ̋́̔̇̉ͤ̽͢ͅo͈̥͊ͦ̔̃ͬ̎͛ͧ͌͌͟͟͝o͈̥͊ͦ̔̃ͬ̎͛ͧ͌͌͟͟͝r̸̨͕̙̟̟͈̼͔ͥͮ͋ͫ͒͝r̸̨͕̙̟̟͈̼͔ͥͮ͋ͫ͒͝ ̧̗̝̘͇͉̩̗̅̊̓͂̐͊͡ ̧̗̝̘͇͉̩̗̅̊̓͂̐͊͡o̬̻̪̹̘̥̳̲̫̳̫̳ͬͩ͊o̬̻̪̹̘̥̳̲̫̳̫̳ͬͩ͊f̡̣̜̗͚̙͇̓͑̍ͬ̅̚͞͞f̡̣̜̗͚̙͇̓͑̍ͬ̅̚͞͞ ̵̸͓̱̻̱̌̃ͫ̑̿͐͂ͨ̇ ̵̸͓̱̻̱̌̃ͫ̑̿͐͂ͨ̇2̡̝̥̯͚͔̖̫ͫ̽̔͑̅̚͞2̡̝̥̯͚͔̖̫ͫ̽̔͑̅̚͞ ̨̠̰̯̤͕̗̗̳͆̑͐͌̕͟ ̨̠̰̯̤͕̗̗̳͆̑͐͌̕͟i͇̫̲̲͓̖͙͖̱ͤ͊̎̃ͧ͢i͇̫̲̲͓̖͙͖̱ͤ͊̎̃ͧ͢n̵̢̻̦̭̏̓̂́͏̲̪̙̏͌n̵̢̻̦̭̏̓̂́͏̲̪̙̏͌ ̸̵̛̹͇͚̓̋̎̏̽̚̚͢͜ ̸̵̛̹͇͚̓̋̎̏̽̚̚͢͜t͈͔̤̲̬ͧͩ̔̀̂́͑̂ͭ͘t͈͔̤̲̬ͧͩ̔̀̂́͑̂ͭ͘h̡̻̙̖̪̱̍͗̍́͗́́̅͠h̡̻̙̖̪̱̍͗̍́͗́́̅͠e̶̖̩̜͐ͥͨͪͣ̆͋̋̉͢͡e̶̖̩̜͐ͥͨͪͣ̆͋̋̉͢͡ ̧͈̙̤͉͌ͩ̓͐̌̄ͦ͌ͥ͝ ̧͈̙̤͉͌ͩ̓͐̌̄ͦ͌ͥ͝c̨̨̹̗̬͕̩̈̑̉̃̑́̆͞c̨̨̹̗̬͕̩̈̑̉̃̑́̆͞h̭͚̦̻̘͈͆ͪ̿̌́̏̐͊͠h̭͚̦̻̘͈͆ͪ̿̌́̏̐͊͠ę̙͍͚ͮͦ́ͭͥ̈͑ͧ̕̕͢ę̙͍͚ͮͦ́ͭͥ̈͑ͧ̕̕͢c̝̭͓̹̙̠̄̍ͦ̌̏̉̇͛ͥc̝̭͓̹̙̠̄̍ͦ̌̏̉̇͛ͥk̶̡͚̦̰̣͖̔͌́̋͋̔ͥ̕k̶̡͚̦̰̣͖̔͌́̋͋̔ͥ̕.̛̱̖͓̼͚̲ͪ̆̈́̃̚͜͞ͅ.̛̱̖͓̼͚̲ͪ̆̈́̃̚͜͞ͅ ̶̴̺̹̜̺͇ͮ̉ͯ͋͗͝͝ͅ ̶̴̺̹̜̺͇ͮ̉ͯ͋͗͝͝ͅA͉̺̰̲̟̺͚͙̽́̀̌ͬͩ͠A͉̺̰̲̟̺͚͙̽́̀̌ͬͩ͠l̜͈̟͖̾͑̈́́̇ͮ̐ͦͮ͋̋l̜͈̟͖̾͑̈́́̇ͮ̐ͦͮ͋̋s̡̛̯̜̩̪̤̹̅͛̋̓͂̊͡s̡̛̯̜̩̪̤̹̅͛̋̓͂̊͡o̶͉̱͉̠̫̻ͤͣ̓ͭ͊͑͆ͅo̶͉̱͉̠̫̻ͤͣ̓ͭ͊͑͆ͅ ̢̗͉̝̞͗͛̊́ͦ͛̍̚͜͡ ̢̗͉̝̞͗͛̊́ͦ͛̍̚͜͡i̵̧͍̝̦̬̭̽̎̈̃̓̚͟͝i̵̧͍̝̦̬̭̽̎̈̃̓̚͟͝f̴̛̻̞̬̟͖̙̦̑́̀̍̇͝f̴̛̻̞̬̟͖̙̦̑́̀̍̇͝ ͓͚͇̺ͪ͛ͦͥ̓̎ͨ͒͊̚͟ ͓͚͇̺ͪ͛ͦͥ̓̎ͨ͒͊̚͟t̶̯͙̹̟͖̂̇͒̾ͭ͒͐͛̌t̶̯͙̹̟͖̂̇͒̾ͭ͒͐͛̌ḧ̠̬́͋̕͏̧̙̼͑͂̀̌̉̈ḧ̠̬́͋̕͏̧̙̼͑͂̀̌̉̈e̫̺͔̗̳͋̒͌ͤͬ̔͗̕̕͡e̫̺͔̗̳͋̒͌ͤͬ̔͗̕̕͡r̴̛̭͙͑́ͤ̓̒͊̈ͥ̑ͮ͞r̴̛̭͙͑́ͤ̓̒͊̈ͥ̑ͮ͞ȇ̷̛̝͓̜̮̩͙ͨ̎̎͛̌̽ȇ̷̛̝͓̜̮̩͙ͨ̎̎͛̌̽ ̸̩̤̘̖̳̻̋̈͛͑̈́̌̒͝ ̸̩̤̘̖̳̻̋̈͛͑̈́̌̒͝i̧͖͕̞̩̱̭̎̽̀͏̮ͬͨ͡i̧͖͕̞̩̱̭̎̽̀͏̮ͬͨ͡ś̶͍̞̉ͯ̊ͦͧ̀̆̌͂͞͝ś̶͍̞̉ͯ̊ͦͧ̀̆̌͂͞͝ ̡̛͙͈̜̗̻̮̞͍̝̝̓̅͋ ̡̛͙͈̜̗̻̮̞͍̝̝̓̅͋ȃ̢͍̟̬̳ͣ͏̝̔͑̎ͣ̕ͅȃ̢͍̟̬̳ͣ͏̝̔͑̎ͣ̕ͅ ̖̼͙̯͎̒͒ͩͣ̀̆͗̇͝ͅ ̖̼͙̯͎̒͒ͩͣ̀̆͗̇͝ͅ}|
        {100:    }{101:ļ̩̪͕͓͖͌̐͛̊ͫͪ̿͊ͤļ̩̪͕͓͖͌̐͛̊ͫͪ̿͊ͤȍ͍̜͓̣͈͍̳̄͊̄ͣ͢͝ͅȍ͍̜͓̣͈͍̳̄͊̄ͣ͢͝ͅt̵̷̛̤͎͕̖̹ͦ̇͒́͂ͮ͘t̵̷̛̤͎͕̖̹ͦ̇͒́͂ͮ͘ ̶̨̳̺̖̼̯͎̯̬̀̈̐ͦ͜ ̶̨̳̺̖̼̯͎̯̬̀̈̐ͦ͜o̷̡̯̼͇̻͙̙̙͒͂ͩ̉̀ͮo̷̡̯̼͇̻͙̙̙͒͂ͩ̉̀ͮf̻̺̖͉̊ͤ̇ͤ̍͌͛̐͟͟͝f̻̺̖͉̊ͤ̇ͤ̍͌͛̐͟͟͝ ̷̴̟̠̹͎̱̈́̈́̆̈́̇̃͘͞ ̷̴̟̠̹͎̱̈́̈́̆̈́̇̃͘͞p̸̸̧̨̦̪͕̮̀͒͒͌͌̑͢p̸̸̧̨̦̪͕̮̀͒͒͌͌̑͢a̯͙ͥͬ̊̓ͯ͌ͮ͊̎̂͒͢͠a̯͙ͥͬ̊̓ͯ͌ͮ͊̎̂͒͢͠ċ̛̫̱̥͕̳̲ͫͤ̎͐̄̕͠ċ̛̫̱̥͕̳̲ͫͤ̎͐̄̕͠k̵̡̹̲͓̾̉͏̶̻́ͩ̎̓̊k̵̡̹̲͓̾̉͏̶̻́ͩ̎̓̊e̵̡̢̢̪͍̲̣͒̒̍ͯͤ͊͑e̵̡̢̢̪͍̲̣͒̒̍ͯͤ͊͑ḑ̘͖̼̌ͭ̐͑ͯ͋ͬ̈́͆̓̚ḑ̘͖̼̌ͭ̐͑ͯ͋ͬ̈́͆̓̚ ̶̶̧̺̰͖̯͇̏̄̿ͤ̔͘͡ ̶̶̧̺̰͖̯͇̏̄̿ͤ̔͘͡c̩̝̯͂͆ͤ̈́͆̏̑̄ͦ͗̿͞c̩̝̯͂͆ͤ̈́͆̏̑̄ͦ͗̿͞e̥̫͌̓̚͏̵̧͙̣̻̞́̌̉e̥̫͌̓̚͏̵̧͙̣̻̞́̌̉l̵̪̣̪̘̙̟̘͊͒̿̏͗͝͠l̵̪̣̪̘̙̟̘͊͒̿̏͗͝͠l̨̮͚̟̺̹ͮ͊ͭ̊ͮ̿͘͘͟l̨̮͚̟̺̹ͮ͊ͭ̊ͮ̿͘͘͟s̩̯͌̊͒͝͏̥͑̈̎ͥ̀͟͞s̩̯͌̊͒͝͏̥͑̈̎ͥ̀͟͞,̡̙̻̭̪̭̖̀̇̒ͮ̈̒̇̕,̡̙̻̭̪̭̖̀̇̒ͮ̈̒̇̕ ̷̢̫̳̺͉̯̳̂̓ͨ͋͛͂͞ ̷̢̫̳̺͉̯̳̂̓ͨ͋͛͂͞p̟̰͖̹̦̲͙̉̑͐͑͗̀͟͢p̟̰͖̹̦̲͙̉̑͐͑͗̀͟͢á̳̜̈̓͛͠͏̢̄ͧ̉ͧͨͅá̳̜̈̓͛͠͏̢̄ͧ̉ͧͨͅs̵̢̥̖̰̥̼̯ͪͭ̿͗͌ͪ͊s̵̢̥̖̰̥̼̯ͪͭ̿͗͌ͪ͊s̘̱͕̓̅͛̆̅ͮ́̅ͧͣ̈̚s̘̱͕̓̅͛̆̅ͮ́̅ͧͣ̈̚ ̡̮̹̹̗͎̝ͣ̇́ͤ̾ͤ͞ͅ ̡̮̹̹̗͎̝ͣ̇́ͤ̾ͤ͞ͅt̡̛̝̯̱̖̠̱ͤ͒͆̍̚͘ͅt̡̛̝̯̱̖̠̱ͤ͒͆̍̚͘ͅh̛̯̻̩̘̤̳̿̓͂̐͐͡͡͠h̛̯̻̩̘̤̳̿̓͂̐͐͡͡͠ë́͏̩̣̗̺̟ͭ̃̾͂̀͘͢͝ë́͏̩̣̗̺̟ͭ̃̾͂̀͘͢͝m̸̜͔̮̥̹̲̖̲̄̈̊̎ͬ͠m̸̜͔̮̥̹̲̖̲̄̈̊̎ͬ͠ ̝͇̏͏̡̟̥͍ͦ̋̎ͤ̕͢͜ ̝͇̏͏̡̟̥͍ͦ̋̎ͤ̕͢͜ó͖̬̩̤̜̫͔͑̌ͫ̈́͡͏̇ó͖̬̩̤̜̫͔͑̌ͫ̈́͡͏̇f̴̶̴̥͔̗͖̬̆͒ͨ͊ͬͅͅf̴̶̴̥͔̗͖̬̆͒ͨ͊ͬͅͅ ̮̦̲̼̞̑ͨ͒̌ͤ̿́̌͐͒ ̮̦̲̼̞̑ͨ͒̌ͤ̿́̌͐͒t̛̮̰͔̋͂͗̓͋̇̅̆̒͊ͅt̛̮̰͔̋͂͗̓͋̇̅̆̒͊ͅo̴̢͚̮̳͙̟̳̐ͬ͑̃̓͡ͅo̴̢͚̮̳͙̟̳̐ͬ͑̃̓͡ͅ ̝̺͖͒ͯͭ̑ͤ͊ͮ͂̚̚͡ͅ ̝̺͖͒ͯͭ̑ͤ͊ͮ͂̚̚͡ͅt̶̨̲̦͉͖̲̤̗ͬ̋̋ͦ̂ͯt̶̨̲̦͉͖̲̤̗ͬ̋̋ͦ̂ͯh̶̨̤͍̜̯͈͇͛ͩ́̀͊ͮͩh̶̨̤͍̜̯͈͇͛ͩ́̀͊ͮͩe̡̢̤̰̰͙̭͈̓ͯͬ͑ͨ̃͢e̡̢̤̰̰͙̭͈̓ͯͬ͑ͨ̃͢ ̶̛̺̦̯̹̞͎̟̥̞̝̐̕ͅ ̶̛̺̦̯̹̞͎̟̥̞̝̐̕ͅU͙̓ͤ͏̟̗̟͕̥̰́͑̆ͧ͟U͙̓ͤ͏̟̗̟͕̥̰́͑̆ͧ͟I̪̩͕͕̼̎͐ͭͦ͐̋̐̅ͯ͢I̪̩͕͕̼̎͐ͭͦ͐̋̐̅ͯ͢ ̢̦͍̰̬͚ͯ́ͫ̿ͨ͘͘͝͡ ̢̦͍̰̬͚ͯ́ͫ̿ͨ͘͘͝͡t̴̨͉̟̥͛͐̇̀̎̒̄ͧ̀̇t̴̨͉̟̥͛͐̇̀̎̒̄ͧ̀̇o̸͇͏̷͍͉̠ͤ̈̾ͤ͛ͥ̚͝o̸͇͏̷͍͉̠ͤ̈̾ͤ͛ͥ̚͝ ̸̣͕̝̯̼́ͨͯ̏̋̂̑́͘ ̸̣͕̝̯̼́ͨͯ̏̋̂̑́͘l̵̨̛̬̱̣̬͚̺̫̳ͨ̽ͫͯl̵̨̛̬̱̣̬͚̺̫̳ͨ̽ͫͯḙ̳͕̝̥̄̆ͮ̄ͦͨ̓̚͏͢ḙ̳͕̝̥̄̆ͮ̄ͦͨ̓̚͏͢t̵̼̲̣̉̀̌ͭ̒̓͌̀͒̀ͅt̵̼̲̣̉̀̌ͭ̒̓͌̀͒̀ͅ ̺͈͙̘̣̩̙͕̋̇͆̀͊̇̂ ̺͈͙̘̣̩̙͕̋̇͆̀͊̇̂i̧̭̲̼̗̥̪̣̭ͬ͑ͨ̓̈̐i̧̭̲̼̗̥̪̣̭ͬ͑ͨ̓̈̐ṱ̴̯̲̱̫̲̥ͭ̀͋̂̚̕͟ṱ̴̯̲̱̫̲̥ͭ̀͋̂̚̕͟ ̖̗͔̦̲̻̌̀̏͆͊ͤ̑ͨ̉ ̖̗͔̦̲̻̌̀̏͆͊ͤ̑ͨ̉s̨̢͔̄̃ͯ̈́̑͐̚̕͢͝͝s̨̢͔̄̃ͯ̈́̑͐̚̕͢͝͝t͚͉͇̳̘̅̇̋͌̀̋́ͨ́͏t͚͉͇̳̘̅̇̋͌̀̋́ͨ́͏â͇͎̺ͨͭ͡͏͚̯ͤͩ͋ͧ͘â͇͎̺ͨͭ͡͏͚̯ͤͩ͋ͧ͘r̷̘̜͍̫̯ͭͪ͑͗̓͆͟͝ͅr̷̘̜͍̫̯ͭͪ͑͗̓͆͟͝ͅṭ̷̢̱͕̎ͬ̿́́͊̐͟͢͜ṭ̷̢̱͕̎ͬ̿́́͊̐͟͢͜ ̟̥̖͕̭͍̲ͩ̆̒̑͊̕̚͡ ̟̥̖͕̭͍̲ͩ̆̒̑͊̕̚͡p̶̴̨̧̜̰̜̘̉̽ͤ́͘͢͜p̶̴̨̧̜̰̜̘̉̽ͤ́͘͢͜ṙ̠̘̟̭͖̜̜͍ͦ̾͒͜͠͞ṙ̠̘̟̭͖̜̜͍ͦ̾͒͜͠͞ǒ̷̶̫͈̫̹ͬ͂ͧ̿̿̑͂͂ǒ̷̶̫͈̫̹ͬ͂ͧ̿̿̑͂͂c̵̸̮͕̮͈̞̼͎͉͍̑́ͭ̀c̵̸̮͕̮͈̞̼͎͉͍̑́ͭ̀e̸̡͔͓̳̞̟̘ͩ̍̽ͪͤͣ͢e̸̡͔͓̳̞̟̘ͩ̍̽ͪͤͣ͢s̨͍̬̖͂̏ͦ̀ͫ̐ͨͪ͡ͅͅs̨͍̬̖͂̏ͦ̀ͫ̐ͨͪ͡ͅͅs̡̻̹̜͍̺͌͛ͮ̓̀̈́̉̿͑s̡̻̹̜͍̺͌͛ͮ̓̀̈́̉̿͑í̤̜ͭ͢͏̶̶̹͍ͤͪ̉ͫ̋í̤̜ͭ͢͏̶̶̹͍ͤͪ̉ͫ̋ņ̵̼̪̭̲̤͙͖̒̑ͨ͂̌͘ņ̵̼̪̭̲̤͙͖̒̑ͨ͂̌͘g̷̢̢̭̘̙̥̖̲̭͛ͣ́̀̍g̷̢̢̭̘̙̥̖̲̭͛ͣ́̀̍ ̷̺̱̙̹͔̗͆ͪ̾ͬͬ̓͘͠ ̷̺̱̙̹͔̗͆ͪ̾ͬͬ̓͘͠t̛̤̻̲̓̓ͩ̇ͫͫ͆ͦ͛̕͞t̛̤̻̲̓̓ͩ̇ͫͫ͆ͦ͛̕͞ḣ̜̘̮̫͍̗͇ͥͯ̽̕͢͢͝ḣ̜̘̮̫͍̗͇ͥͯ̽̕͢͢͝ẹ̵̱̹͚͖̜̞̆̾͌ͬ̀̿͞ẹ̵̱̹͚͖̜̞̆̾͌ͬ̀̿͞m̶̢̧͉̝̜̝̍͑͂̽̌̉̈́ͩm̶̢̧͉̝̜̝̍͑͂̽̌̉̈́ͩ                                                                                                }|
        {100:~                                                                                                                                                                                                                                                         }|*6
                                                                                                                                                                                                                                                                  |
      ]],
    })
  end)

  it('works with arabic input and arabicshape', function()
    command('set arabic')

    command('set noarabicshape')
    feed('isghl!<esc>')
    screen:expect {
      grid = [[
                                                             ^!مالس|
      {1:                                                           ~}|*4
                                                                  |
    ]],
    }

    command('set arabicshape')
    screen:expect {
      grid = [[
                                                              ^!ﻡﻼﺳ|
      {1:                                                           ~}|*4
                                                                  |
    ]],
    }
  end)

  it('works with arabic input and arabicshape and norightleft', function()
    command('set arabic norightleft')

    command('set noarabicshape')
    feed('isghl!<esc>')
    screen:expect {
      grid = [[
      سلام^!                                                       |
      {1:~                                                           }|*4
                                                                  |
    ]],
    }

    command('set arabicshape')
    screen:expect {
      grid = [[
      ﺱﻼﻣ^!                                                        |
      {1:~                                                           }|*4
                                                                  |
    ]],
    }
  end)

  it('works with arabicshape and multiple composing chars', function()
    -- this tests an important edge case: arabicshape might increase the byte size of the base
    -- character in a way so that the last composing char no longer fits. use "g8" on the text
    -- to observe what is happening (the final E1 80 B7 gets deleted with 'arabicshape')
    -- If we would increase the schar_t size, say from 32 to 64 bytes, we need to extend the
    -- test text with even more zalgo energy to still touch this edge case.

    api.nvim_buf_set_lines(0, 0, -1, true, { 'سلام့̀́̂̃̄̅̆̇̈̉̊̋̌' })
    command('set noarabicshape')

    screen:expect {
      grid = [[
      ^سلام့̀́̂̃̄̅̆̇̈̉̊̋̌                                                        |
      {1:~                                                           }|*4
                                                                  |
    ]],
    }

    command('set arabicshape')
    screen:expect {
      grid = [[
      ^ﺱﻼﻣ̀́̂̃̄̅̆̇̈̉̊̋̌                                                         |
      {1:~                                                           }|*4
                                                                  |
    ]],
    }
  end)

  it('supports emoji with variant selectors and ZWJ', function()
    command('set ruler')
    insert('🏳️‍⚧️')
    screen:expect([[
      ^🏳️‍⚧️                                                          |
      {1:~                                                           }|*4
                                                1,1           All |
    ]])

    feed('a word<esc>')
    screen:expect([[
      🏳️‍⚧️ wor^d                                                     |
      {1:~                                                           }|*4
                                                1,21-7        All |
    ]])

    feed('0')
    screen:expect([[
      ^🏳️‍⚧️ word                                                     |
      {1:~                                                           }|*4
                                                1,1           All |
    ]])

    feed('l')
    screen:expect([[
        🏳️‍⚧️^ word                                                     |
        {1:~                                                           }|*4
                                                  1,17-3        All |
    ]])

    feed('h')
    screen:expect([[
      ^🏳️‍⚧️ word                                                     |
      {1:~                                                           }|*4
                                                1,1           All |
    ]])

    feed('o❤️ variant selected<esc>')
    screen:expect([[
      🏳️‍⚧️ word                                                     |
      ❤️ variant selecte^d                                         |
      {1:~                                                           }|*3
                                                2,23-19       All |
    ]])

    feed('0')
    screen:expect([[
      🏳️‍⚧️ word                                                     |
      ^❤️ variant selected                                         |
      {1:~                                                           }|*3
                                                2,1           All |
    ]])

    feed('l')
    screen:expect([[
      🏳️‍⚧️ word                                                     |
      ❤️^ variant selected                                         |
      {1:~                                                           }|*3
                                                2,7-3         All |
    ]])

    feed('h')
    screen:expect([[
      🏳️‍⚧️ word                                                     |
      ^❤️ variant selected                                         |
      {1:~                                                           }|*3
                                                2,1           All |
    ]])

    -- without selector: single width (note column 18 and not 19)
    feed('o❤ variant selected<esc>')
    screen:expect([[
      🏳️‍⚧️ word                                                     |
      ❤️ variant selected                                         |
      ❤ variant selecte^d                                          |
      {1:~                                                           }|*2
                                                3,20-18       All |
    ]])
  end)
end)

describe('multibyte rendering: statusline', function()
  local screen

  before_each(function()
    clear()
    screen = Screen.new(40, 4)
    screen:attach()
    command('set laststatus=2')
  end)

  it('last char shows (multibyte)', function()
    command('set statusline=你好')
    screen:expect([[
    ^                                        |
    {1:~                                       }|
    {3:你好                                    }|
                                            |
    ]])
  end)
  it('last char shows (single byte)', function()
    command('set statusline=abc')
    screen:expect([[
    ^                                        |
    {1:~                                       }|
    {3:abc                                     }|
                                            |
    ]])
  end)
  it('unicode control points', function()
    command('set statusline=')
    screen:expect([[
    ^                                        |
    {1:~                                       }|
    {3:<9f>                                    }|
                                            |
    ]])
  end)
  it('MAX_MCO (6) unicode combination points', function()
    command('set statusline=o̸⃯ᷰ⃐⃧⃝')
    -- o + U+1DF0 + U+20EF + U+0338 + U+20D0 + U+20E7 + U+20DD
    screen:expect([[
    ^                                        |
    {1:~                                       }|
    {3:o̸⃯ᷰ⃐⃧⃝                                       }|
                                            |
    ]])
  end)
  it('non-printable followed by MAX_MCO unicode combination points', function()
    command('set statusline=̸⃯ᷰ⃐⃧⃝')
    -- U+9F + U+1DF0 + U+20EF + U+0338 + U+20D0 + U+20E7 + U+20DD
    -- TODO: not ideal, better with plain ">" and then space+combining
    screen:expect([[
      ^                                        |
      {1:~                                       }|
      {3:<9f≯⃯ᷰ⃐⃧⃝                                    }|
                                              |
    ]])
  end)

  it('hidden group %( %) does not cause invalid unicode', function()
    command("let &statusline = '%#StatColorHi2#%(✓%#StatColorHi2#%) Q≡'")
    screen:expect {
      grid = [[
      ^                                        |
      {1:~                                       }|
      {3: Q≡                                     }|
                                              |
    ]],
    }
  end)

  it('emoji with ZWJ in filename with default stl', function()
    command('file 🧑‍💻')
    screen:expect {
      grid = [[
      ^                                        |
      {1:~                                       }|
      {3:🧑‍💻                                      }|
                                              |
    ]],
    }
  end)

  it('unprintable chars in filename with default stl', function()
    command('file 🧑​💻')
    screen:expect {
      grid = [[
      ^                                        |
      {1:~                                       }|
      {3:🧑�💻                                   }|
                                              |
    ]],
    }
  end)

  it('emoji with ZWJ in filename with custom stl', function()
    screen:add_extra_attr_ids {
      [100] = {
        bold = true,
        reverse = true,
        foreground = Screen.colors.Gray100,
        background = Screen.colors.Red,
      },
    }
    command('set statusline=xx%#ErrorMsg#%f%##yy')
    command('file 🧑‍💻')
    screen:expect {
      grid = [[
      ^                                        |
      {1:~                                       }|
      {3:xx}{100:🧑‍💻}{3:yy                                  }|
                                              |
    ]],
    }
  end)

  it('unprintable chars in filename with custom stl', function()
    screen:add_extra_attr_ids {
      [100] = {
        bold = true,
        reverse = true,
        foreground = Screen.colors.Gray100,
        background = Screen.colors.Red,
      },
    }
    command('set statusline=xx%#ErrorMsg#%f%##yy')
    command('file 🧑​💻')
    screen:expect {
      grid = [[
      ^                                        |
      {1:~                                       }|
      {3:xx}{100:🧑<200b>💻}{3:yy                          }|
                                              |
    ]],
    }
  end)
end)
