/*
 * Copyright (c) Forge Development LLC and contributors
 * SPDX-License-Identifier: LGPL-2.1-only
 */

package net.neoforged.neoforge.oldtest.fluid;

import com.mojang.blaze3d.shaders.FogShape;
import com.mojang.blaze3d.vertex.VertexConsumer;
import java.util.stream.Stream;
import net.minecraft.client.Camera;
import net.minecraft.client.Minecraft;
import net.minecraft.client.multiplayer.ClientLevel;
import net.minecraft.client.renderer.FogParameters;
import net.minecraft.client.renderer.FogRenderer;
import net.minecraft.client.renderer.ItemBlockRenderTypes;
import net.minecraft.client.renderer.RenderType;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Holder;
import net.minecraft.core.particles.ParticleTypes;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.sounds.SoundEvents;
import net.minecraft.world.item.BucketItem;
import net.minecraft.world.item.CreativeModeTabs;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.Items;
import net.minecraft.world.level.BlockAndTintGetter;
import net.minecraft.world.level.block.Blocks;
import net.minecraft.world.level.block.LiquidBlock;
import net.minecraft.world.level.block.state.BlockBehaviour;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.material.FlowingFluid;
import net.minecraft.world.level.material.Fluid;
import net.minecraft.world.level.material.FluidState;
import net.neoforged.bus.api.IEventBus;
import net.neoforged.fml.common.Mod;
import net.neoforged.fml.event.lifecycle.FMLClientSetupEvent;
import net.neoforged.fml.event.lifecycle.FMLCommonSetupEvent;
import net.neoforged.fml.loading.FMLEnvironment;
import net.neoforged.neoforge.client.event.RegisterColorHandlersEvent;
import net.neoforged.neoforge.client.extensions.common.IClientFluidTypeExtensions;
import net.neoforged.neoforge.client.extensions.common.RegisterClientExtensionsEvent;
import net.neoforged.neoforge.client.model.pipeline.VertexConsumerWrapper;
import net.neoforged.neoforge.common.NeoForgeMod;
import net.neoforged.neoforge.event.BuildCreativeModeTabContentsEvent;
import net.neoforged.neoforge.fluids.BaseFlowingFluid;
import net.neoforged.neoforge.fluids.FluidInteractionRegistry;
import net.neoforged.neoforge.fluids.FluidType;
import net.neoforged.neoforge.registries.DeferredBlock;
import net.neoforged.neoforge.registries.DeferredHolder;
import net.neoforged.neoforge.registries.DeferredItem;
import net.neoforged.neoforge.registries.DeferredRegister;
import net.neoforged.neoforge.registries.NeoForgeRegistries;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.joml.Vector4f;

/**
 * A test case used to define and test fluid type integration into fluids.
 *
 * <ul>
 * <li>Checks that each fluid has a fluid type</li>
 * <li>Adds a new fluid with its type, source/flowing, block, and bucket</li>
 * <li>Sets properties to test out fluid logic</li>
 * <li>Overrides fluid rendering methods</li>
 * <li>Adds block color and render layer</li>
 * <li>Adds fluid interaction definitions</li>
 * <li>Adds the ability for the fluid to drip from Pointed Dripstone stalactites into a cauldron below</li>
 * </ul>
 */
@Mod(FluidTypeTest.ID)
public class FluidTypeTest {
    private static final boolean ENABLE = true;

    protected static final String ID = "fluid_type_test";
    private static Logger logger;

    private static final DeferredRegister<FluidType> FLUID_TYPES = DeferredRegister.create(NeoForgeRegistries.Keys.FLUID_TYPES, ID);
    private static final DeferredRegister<Fluid> FLUIDS = DeferredRegister.create(BuiltInRegistries.FLUID, ID);
    private static final DeferredRegister.Blocks BLOCKS = DeferredRegister.createBlocks(ID);
    private static final DeferredRegister.Items ITEMS = DeferredRegister.createItems(ID);

    private static BaseFlowingFluid.Properties fluidProperties() {
        return new BaseFlowingFluid.Properties(TEST_FLUID_TYPE::value, TEST_FLUID, TEST_FLUID_FLOWING)
                .block(TEST_FLUID_BLOCK)
                .bucket(TEST_FLUID_BUCKET);
    }

    private static final Holder<FluidType> TEST_FLUID_TYPE = FLUID_TYPES.register("test_fluid", () -> new FluidType(FluidType.Properties.create()
            .supportsBoating(true)
            .canHydrate(true)
            .addDripstoneDripping(0.25F, ParticleTypes.SCULK_SOUL, Blocks.POWDER_SNOW_CAULDRON, SoundEvents.END_PORTAL_SPAWN)));
    private static final DeferredHolder<Fluid, FlowingFluid> TEST_FLUID = FLUIDS.register("test_fluid", () -> new BaseFlowingFluid.Source(fluidProperties()));
    private static final DeferredHolder<Fluid, Fluid> TEST_FLUID_FLOWING = FLUIDS.register("test_fluid_flowing", () -> new BaseFlowingFluid.Flowing(fluidProperties()));
    private static final DeferredBlock<LiquidBlock> TEST_FLUID_BLOCK = BLOCKS.registerBlock("test_fluid_block", props -> new LiquidBlock(TEST_FLUID.get(), props), BlockBehaviour.Properties.of().noCollission().strength(100.0F).noLootTable());
    private static final DeferredItem<Item> TEST_FLUID_BUCKET = ITEMS.registerItem("test_fluid_bucket", props -> new BucketItem(TEST_FLUID.get(), props.craftRemainder(Items.BUCKET).stacksTo(1)));

    public FluidTypeTest(IEventBus modEventBus) {
        if (ENABLE) {
            logger = LogManager.getLogger();
            NeoForgeMod.enableMilkFluid();

            FLUID_TYPES.register(modEventBus);
            FLUIDS.register(modEventBus);
            BLOCKS.register(modEventBus);
            ITEMS.register(modEventBus);

            modEventBus.addListener(this::commonSetup);
            modEventBus.addListener(this::addCreative);

            if (FMLEnvironment.dist.isClient()) {
                new FluidTypeTestClient(modEventBus);
            }
        }
    }

    private void addCreative(BuildCreativeModeTabContentsEvent event) {
        if (event.getTabKey() == CreativeModeTabs.INGREDIENTS)
            event.accept(TEST_FLUID_BUCKET);
    }

    private void commonSetup(FMLCommonSetupEvent event) {
        // Add Interactions for sources
        FluidInteractionRegistry.addInteraction(TEST_FLUID_TYPE.value(), new FluidInteractionRegistry.InteractionInformation(NeoForgeMod.LAVA_TYPE.value(), Blocks.GOLD_BLOCK.defaultBlockState()));
        FluidInteractionRegistry.addInteraction(NeoForgeMod.WATER_TYPE.value(), new FluidInteractionRegistry.InteractionInformation(TEST_FLUID_TYPE.value(), state -> state.isSource() ? Blocks.DIAMOND_BLOCK.defaultBlockState() : Blocks.IRON_BLOCK.defaultBlockState()));

        // Log Fluid Types for all fluids
        event.enqueueWork(() -> BuiltInRegistries.FLUID.forEach(fluid -> logger.info("Fluid {} has FluidType {}", BuiltInRegistries.FLUID.getKey(fluid), NeoForgeRegistries.FLUID_TYPES.getKey(fluid.getFluidType()))));
    }

    private static class FluidTypeTestClient {
        private FluidTypeTestClient(IEventBus modEventBus) {
            modEventBus.addListener(this::clientSetup);
            modEventBus.addListener(this::registerBlockColors);
            modEventBus.addListener(this::registerClientExtensions);
        }

        private void clientSetup(FMLClientSetupEvent event) {
            Stream.of(TEST_FLUID, TEST_FLUID_FLOWING).map(DeferredHolder::get)
                    .forEach(fluid -> ItemBlockRenderTypes.setRenderLayer(fluid, RenderType.translucent()));
        }

        private void registerBlockColors(RegisterColorHandlersEvent.Block event) {
            event.register((state, getter, pos, index) -> {
                if (getter != null && pos != null) {
                    FluidState fluidState = getter.getFluidState(pos);
                    return IClientFluidTypeExtensions.of(fluidState).getTintColor(fluidState, getter, pos);
                } else return 0xAF7FFFD4;
            }, TEST_FLUID_BLOCK.get());
        }

        private void registerClientExtensions(RegisterClientExtensionsEvent event) {
            event.registerFluidType(new IClientFluidTypeExtensions() {
                private static final ResourceLocation STILL = ResourceLocation.withDefaultNamespace("block/water_still"),
                        FLOW = ResourceLocation.withDefaultNamespace("block/water_flow"),
                        OVERLAY = ResourceLocation.withDefaultNamespace("block/obsidian"),
                        VIEW_OVERLAY = ResourceLocation.withDefaultNamespace("textures/block/obsidian.png");

                @Override
                public ResourceLocation getStillTexture() {
                    return STILL;
                }

                @Override
                public ResourceLocation getFlowingTexture() {
                    return FLOW;
                }

                @Override
                public ResourceLocation getOverlayTexture() {
                    return OVERLAY;
                }

                @Override
                public ResourceLocation getRenderOverlayTexture(Minecraft mc) {
                    return VIEW_OVERLAY;
                }

                @Override
                public int getTintColor() {
                    return 0xAF7FFFD4;
                }

                @Override
                public Vector4f modifyFogColor(Camera camera, float partialTick, ClientLevel level, int renderDistance, float darkenWorldAmount, Vector4f fluidFogColor) {
                    int color = this.getTintColor();
                    return new Vector4f((color >> 16 & 0xFF) / 255F, (color >> 8 & 0xFF) / 255F, (color & 0xFF) / 255F, fluidFogColor.w);
                }

                @Override
                public FogParameters modifyFogRender(Camera camera, FogRenderer.FogMode mode, float renderDistance, float partialTick, FogParameters parameters) {
                    float nearDistance = -8F;
                    float farDistance = 24F;
                    FogShape shape = parameters.shape();

                    if (farDistance > renderDistance) {
                        farDistance = renderDistance;
                        shape = FogShape.CYLINDER;
                    }

                    return new FogParameters(nearDistance, farDistance, shape, parameters.red(), parameters.green(), parameters.blue(), parameters.alpha());
                }

                @Override
                public boolean renderFluid(FluidState fluidState, BlockAndTintGetter getter, BlockPos pos, VertexConsumer vertexConsumer, BlockState blockState) {
                    // Flip RGB to BGR *only* for fluid blocks rendered at Y 100
                    if (pos.getY() == 100) {
                        vertexConsumer = new VertexConsumerWrapper(vertexConsumer) {
                            @Override
                            public VertexConsumer setColor(int r, int g, int b, int a) {
                                return super.setColor(b, g, r, a);
                            }
                        };
                    }
                    // Replace vanilla fluid rendering
                    Minecraft.getInstance().getBlockRenderer().getLiquidBlockRenderer().tesselate(getter, pos, vertexConsumer, blockState, fluidState);
                    return true;
                }
            }, TEST_FLUID_TYPE.value());
        }
    }
}
