import nltk


class NLTKWordTokenizer():
    """
    A NLTKWordTokenizer splits strings into word tokens with NLTK.
    """
    def tokenize(self, sentence):
        """
        Given a string, tokenize it into words (with the conventional notion
        of word).

        Parameters
        ----------
        sentence: str
            The string to tokenize.

        Returns
        -------
        tokenized_sentence: List[str]
            The tokenized representation of the string, as a list of tokens.
        """
        return nltk.word_tokenize(sentence.lower())

    def get_words_for_indexer(self, text):
        """
        Given a string, get the tokens in it that should be run
        through a DataIndexer. In this class, this involves tokenizing
        it into words (with the conventional notion of word), and then
        returning a namespace dictionary suitable for the DataIndexer.

        Parameters
        ----------
        text: str
            The string to tokenize.

        Returns
        -------
        namespace_dict: Dict of {str:List[str]}
            Return a dictionary, where the key is the name of the namespace
            ("words") for this tokenizer, and the value is the tokenized
            representation of the string, as a list of tokens.
        """
        words = self.tokenize(text)
        characters = [char for word in words for char in word]
        return {"words": words, "characters": characters}

    def index_text(self, tokenized_sentence, data_indexer):
        """
        Given a namespace dictionary with "words" and "characters" keys, as
        well as a DataIndexer, assign indices to each of the "words" and
        "characters" tokens based on the DataIndexer.

        Parameters
        ----------
        tokenized_setence: Dict {"words": List[str], "characters": List[List[str]]}
            A dictionary representing the string as tokenized at the word and
            character level, as generated by ``self.get_words_for_indexer``.

        data_indexer: DataIndexer
            A DataIndexer to use in deciding what index to assign each word.
            This should be a pre-fit DataIndexer.

        Returns
        -------
        indexed_text: (List[int], List[List[int]])
            A tuple with two parts. The first element is a list of int, where
            each index of the list represents a word-level token of
            the original string that was then assigned an index by the
            DataIndexer. The second element is a list of list of ints, where
            each inner list is a word tokenized into characters, and the
            contents of each inner list are the indices assigned to each
            character.
        """
        word_indexed_text = [data_indexer.get_word_index(word, namespace="words")
                             for word in tokenized_sentence["words"]]
        character_indexed_text = []
        for word in tokenized_sentence["characters"]:
            character_indexed_word = []
            for character in word:
                character_index = data_indexer.get_word_index(character,
                                                              namespace="characters")
                character_indexed_word.append(character_index)
            character_indexed_text.append(character_indexed_word)
        return (word_indexed_text, character_indexed_text)
