<?php

/**
 * @package Nekofar\Nobitex
 *
 * @author Milad Nekofar <milad@nekofar.com>
 */

declare(strict_types=1);

namespace Nekofar\Nobitex\Auth;

use Http\Client\HttpClient;
use Http\Discovery\HttpClientDiscovery;
use Http\Discovery\MessageFactoryDiscovery;
use Http\Discovery\StreamFactoryDiscovery;
use Http\Message\Authentication;
use Http\Message\RequestFactory;
use Http\Message\StreamFactory;
use Nekofar\Nobitex\Config;
use Psr\Http\Message\RequestInterface;

/**
 * Class Basic
 */
class Basic implements Authentication
{
    /**
     * @var string
     */
    private $apiUrl;

    /**
     * @var string
     */
    private $username;

    /**
     * @var string
     */
    private $password;

    /**
     * @var boolean
     */
    private $remember;

    /**
     * @var string|null
     */
    private $accessToken;

    /**
     * @var \Http\Client\HttpClient
     */
    private $httpClient;

    /**
     * @var \Http\Message\RequestFactory
     */
    private $requestFactory;

    /**
     * @var \Http\Message\StreamFactory
     */
    private $streamFactory;

    /**
     * @var integer|null
     */
    private $totpToken;

    /**
     * Basic constructor.
     *
     * @param string $username Username for authentication.
     * @param string $password Password for authentication.
     * @param boolean $remember Long term token generation.
     * @param integer|null $totpToken TOTP token generated by Authenticator apps.
     * @param \Http\Client\HttpClient|null $httpClient Required HTTP Client for retrieve token.
     */
    public function __construct(
        string $username,
        string $password,
        bool $remember = true,
        ?int $totpToken = null,
        ?HttpClient $httpClient = null,
        ?RequestFactory $requestFactory = null,
        ?StreamFactory $streamFactory = null
    ) {
        $this->apiUrl = Config::DEFAULT_API_URL;

        $this->username = $username;
        $this->password = $password;
        $this->remember = $remember;

        $this->totpToken = $totpToken;

        $this->httpClient = $httpClient ? $httpClient : HttpClientDiscovery::find(); // phpcs:ignore
        $this->requestFactory = $requestFactory ? $requestFactory : MessageFactoryDiscovery::find(); // phpcs:ignore
        $this->streamFactory = $streamFactory ? $streamFactory : StreamFactoryDiscovery::find(); // phpcs:ignore
    }

    /**
     * Refresh authentication access token.
     *
     * @throws \Http\Client\Exception
     */
    public function refreshToken(): ?string
    {
        $this->accessToken = $this->retrieveAuthToken();

        return $this->accessToken;
    }

    /**
     * Authenticates a request.
     */
    public function authenticate(RequestInterface $request): RequestInterface
    {
        return null !== $this->accessToken ? $request->withHeader(
            'Authorization',
            sprintf('Token %s', $this->accessToken),
        ) : $request;
    }

    /**
     * Authenticates a request.
     *
     * @throws \Psr\Http\Client\ClientExceptionInterface
     * @throws \JsonException
     */
    private function retrieveAuthToken(): ?string
    {
        $response = $this->httpClient
            ->sendRequest(
                $this->createAuthRequest(),
            );

        if (200 === $response->getStatusCode()) {
            return json_decode((string) $response->getBody())->key;
        }

        return null;
    }

    /**
     * @throws \JsonException
     */
    private function createAuthRequest(): RequestInterface
    {
        return $this->requestFactory
            ->createRequest(
                'POST',
                $this->apiUrl . '/auth/login/',
                [
                    'Content-Type' => 'application/json',
                    'X-TOTP' => $this->totpToken,
                ],
                json_encode([
                    'username' => $this->username,
                    'password' => $this->password,
                    'remember' => $this->remember === true ? 'yes' : 'no', // phpcs:ignore
                    'captcha' => 'api',
                ], JSON_THROW_ON_ERROR),
            );
    }
}
