# Builder generator

[![License](https://poser.pugx.org/natitech/builder-generator/license)](https://packagist.org/packages/natitech/builder-generator)

PHP standalone library to generate a [builder pattern](https://en.wikipedia.org/wiki/Builder_pattern) from a class.

## Installation

By using composer on your project or globally

```shell script
composer require natitech/builder-generator
composer global require natitech/builder-generator
```

## Usage

You can use the binary to generate a builder near a class :  

```shell script
/path/to/vendor/bin/generate-builder /path/to/class
```

OR you can use it inside another PHP script :

```php
\Nati\BuilderGenerator\FileBuilderGenerator::create()->generateFrom('/path/to/class');
```

## What will be generated 

This will generate a builder class aside the built class.  

Example :
```php
//From /path/to/MyClass.php file = the built class
class MyClass {
    public int $id;
    
    public string $label;
}

//This new file /path/to/MyClassBuilder.php will be generated = the builder class
class MyClassBuilder {
    private int $id;
    
    private string $label;
    
    public function __construct(Faker $faker)
    {
        $this->id = $faker->number();
        $this->label = $faker->word;
    }
    
    public function build(): MyClass
    {
        $myClass = new MyClass();
        $myClass->id = $this->id;
        $myClass->label = $this->label;
        
        return $myClass;
    }
    
    //this will have to be generated by your IDE
    public function withLabel(string $label): self
    {
        $this->label = $label;
        
        return $this;
    }
}

//The ultimate goal is to use this in tests
/** @test */
public function canTestSomething()
{
    $this->assertEquals(
        'test', 
        $this->service->something($this->myClass()->withLabel('used in test')->build())
    );
}

private function myClass(): MyClassBuilder
{
    return new MyClassBuilder(Faker\Factory::create());
}
```

The builder class may need to receive updates on codestyle, faker usages, infered types, namespace, etc. Also, to avoid producing unused code, there are no setters for builder properties. Your IDE should be able to easily fix all of that.

The generator will try to detect property types (php types, phpdoc types, doctrine orm attributes or annotations) and will try to detect the appropriate faker method based on the type and the name of the properties.

## Faker

You can use the --faker option to try to set the most relevant values. In that case, [Faker](https://fakerphp.github.io) will be used as a dependency of the builder class. 

## Build strategies

There are many strategies to build a class : public properties, setter (fluent or not), constructor. A brief explanation on each strategies is given below, but you might prefer read unit tests to fully understand them.

By default, the generator will try to detect the most used strategy inside the built class and will use it for the entire builder class. If you are using setters on all properties but one, the generator will use setters on properties that support it and ignore the property that does not. 

But you can also explictly pass a strategy using the '--strategy' option. 

### Public strategy

When properties are public. See example above.

### Constructor strategy

When properties are protected/private and set inside the __construct method.

```php
//Built class
class MyClass {
    private int $id;
    
    public function __construct(int $id, private string $label) 
    {
        $this->id = $id;
    }
}

//Builder class
class MyClassBuilder {
    private int $id;
    
    private string $label;
    
    public function __construct(Faker $faker)
    {
        $this->id = $faker->number();
        $this->label = $faker->word;
    }
    
    public function build(): MyClass
    {
        return new MyClass(
            $this->id,
            $this->label
        );
    }
}
```

### Setter strategy

When properties are protected/private but can be set using public setters. The setters can be fluent or not.

```php
//Built class
class MyClass {
    protected int $id;
    
    protected string $label;
    
    public function setId(int $id) 
    {
        $this->id = $id;
        
        return $this;
    }
    
    public function setLabel(string $label) 
    {
        $this->label = $label;
        
        return $this;
    }
}

//Builder class
class MyClassBuilder {
    private int $id;
    
    private string $label;
    
    public function __construct(Faker $faker)
    {
        $this->id = $faker->number();
        $this->label = $faker->word;
    }
    
    public function build(): MyClass
    {
        return (new MyClass())
            ->setId($this->id)
            ->setLabel($this->label);
    }
}
```

### Build method strategy

This is a less conventionnal way to build a class. In that case, properties are protected/private and set by public domain methods. So, there is no easy way to set the built class in a certain state. 
If, for test purposes, you want to be able to quickly set the state of that object properties by properties, a solution is to add a static build() method in the built class.

```php
//Built class
class MyClass {
    private int $id;
    
    private ?string $label = null;
    
    public static function create(int $id): self 
    {
        $myClass = new self();
        $myClass->id = $id;
        
        return $myClass;
    }
    
    public function updateLabel(string $label): self 
    {
        $this->label = $label;
        
        return $this;
    }
    
    //This method will have to be generated by you IDE from the builder class 
    //It allows you to hack the state of this object without relying on its public interface
    public static function build(int $id, string $label): self
    {
        $myClass = new self();
        $myClass->id = $this->id;
        $myClass->label = $this->label;
        
        return $myClass;
    }
}

//Builder class
class MyClassBuilder {
    private int $id;
    
    private string $label;
    
    public function __construct(Faker $faker)
    {
        $this->id = $faker->number();
        $this->label = $faker->word;
    }
    
    public function build(): MyClass
    {
        return MyClass::build(
            $this->id,
            $this->label
        );
    }
}
```

### Custom strategy / contributions

To create a custom strategy :
* you need to implement ```Nati\BuilderGenerator\Property\PropertyBuildStrategy``` 
and add it to ```Nati\BuilderGenerator\Property\PropertyBuildStrategyCollection```. 
This will decide HOW properties are built. 
* you also need to complete ```\Nati\BuilderGenerator\Analyzer\BuildableClassAnalyzer::getWriteStrategies()```. 
This will decide WHEN properties could be built with this strategy. 

## IDE / PHPStorm

You can use this tool as an external tool in your IDE. 

For PHPStorm user, see https://www.jetbrains.com/help/phpstorm/configuring-third-party-tools.html. Example configuration :
* Name : Generate builder
* Description : Generate a builder class from a PHP class
* Program : /path/to/your/home/.composer/vendor/bin/generate-builder 
* Arguments : $FilePath$
* Working directory : $FileDir$ 
