import { useRouter } from 'vue-router';
import { createPinia, setActivePinia } from 'pinia';
import type { PushMessage, PushPayload } from '@n8n/api-types';

import { usePushConnection } from '@/composables/usePushConnection';
import { usePushConnectionStore } from '@/stores/pushConnection.store';
import { useOrchestrationStore } from '@/stores/orchestration.store';

vi.mock('vue-router', () => {
	return {
		RouterLink: vi.fn(),
		useRouter: () => ({
			push: vi.fn(),
		}),
		useRoute: () => ({}),
	};
});

vi.useFakeTimers();

describe('usePushConnection()', () => {
	let router: ReturnType<typeof useRouter>;
	let pushStore: ReturnType<typeof usePushConnectionStore>;
	let orchestrationStore: ReturnType<typeof useOrchestrationStore>;
	let pushConnection: ReturnType<typeof usePushConnection>;

	beforeEach(() => {
		setActivePinia(createPinia());

		router = vi.mocked(useRouter)();
		pushStore = usePushConnectionStore();
		orchestrationStore = useOrchestrationStore();
		pushConnection = usePushConnection({ router });
	});

	describe('initialize()', () => {
		it('should add event listener to the pushStore', () => {
			const spy = vi.spyOn(pushStore, 'addEventListener').mockImplementation(() => () => {});

			pushConnection.initialize();

			expect(spy).toHaveBeenCalled();
		});
	});

	describe('terminate()', () => {
		it('should remove event listener from the pushStore', () => {
			const returnFn = vi.fn();
			vi.spyOn(pushStore, 'addEventListener').mockImplementation(() => returnFn);

			pushConnection.initialize();
			pushConnection.terminate();

			expect(returnFn).toHaveBeenCalled();
		});
	});

	describe('queuePushMessage()', () => {
		it('should add message to the queue and sets timeout if not already set', () => {
			const event: PushMessage = {
				type: 'sendWorkerStatusMessage',
				data: {
					workerId: '1',
					status: {} as PushPayload<'sendWorkerStatusMessage'>['status'],
				},
			};

			pushConnection.queuePushMessage(event, 5);

			expect(pushConnection.pushMessageQueue.value).toHaveLength(1);
			expect(pushConnection.pushMessageQueue.value[0]).toEqual({ message: event, retriesLeft: 5 });
			expect(pushConnection.retryTimeout.value).not.toBeNull();
		});
	});

	describe('processWaitingPushMessages()', () => {
		it('should clear the queue and reset the timeout', async () => {
			const event: PushMessage = { type: 'executionRecovered', data: { executionId: '1' } };

			pushConnection.queuePushMessage(event, 0);
			expect(pushConnection.pushMessageQueue.value).toHaveLength(1);
			expect(pushConnection.retryTimeout.value).toBeDefined();

			await pushConnection.processWaitingPushMessages();

			expect(pushConnection.pushMessageQueue.value).toHaveLength(0);
			expect(pushConnection.retryTimeout.value).toBeNull();
		});
	});

	describe('pushMessageReceived()', () => {
		describe('sendWorkerStatusMessage', () => {
			it('should handle event type correctly', async () => {
				const spy = vi.spyOn(orchestrationStore, 'updateWorkerStatus').mockImplementation(() => {});
				const event: PushMessage = {
					type: 'sendWorkerStatusMessage',
					data: {
						workerId: '1',
						status: {} as PushPayload<'sendWorkerStatusMessage'>['status'],
					},
				};

				const result = await pushConnection.pushMessageReceived(event);

				expect(spy).toHaveBeenCalledWith(event.data.status);
				expect(result).toBeTruthy();
			});
		});
	});
});
