# Validator Types
==================

PHP does not have generic type, hence every instance of Collection and Map gets object which describes type. 
This object is called TypeValidator and it is a contract for the collection. Object TypeValidator type defines what kind of type elements will be held in Collection.
The TypeValidator class is exactly a validator for elements passed to the collection. It checks if element is valid - checks if element has got proper type.

Every TypeValidator must implement `CollectionType\TypeValidator\TypeValidatorInterface`. `TypeValidatorInterface` has one method: `isValid($value) : bool`

For example, for IntegerTypeValidator implementation of `isValid` looks like this:

```php
public function isValid($value)
{
    return is_integer($value);
}
```

## Predefined type validators
-----------------------------

CollectionType library provides predefined types:

### Scalar types
---------------

#### IntegerTypeValidator
-------------------------

It checks if passed element is an integer. Value of element must be between `-PHP_INT_MAX-1` and `PHP_INT_MAX` values.
 
```php
use CollectionType\TypeValidator\IntegerTypeValidator;

$integerTypeValidator = new IntegerTypeValidator();
```

#### FloatTypeValidator
-----------------------

It checks if passed element is a float. Integer values like: 1 are unacceptable. Correct value is for example: 1.0
 
```php
use CollectionType\TypeValidator\FloatTypeValidator;

$floatTypeValidator = new FloatTypeValidator();
```

#### StringTypeValidator
------------------------

It checks if passed element is a string. Empty string '' is correct value for this validator.
 
```php
use CollectionType\TypeValidator\StringTypeValidator;

$stringTypeValidator = new StringTypeValidator();
```

#### BoolTypeValidator
------------------------

It checks if passed element is a boolean. Be careful when using BoolTypeValidator as a key in Map. In Map it is possible add only two elements as boolean type: `true` and `false`. 
Every new element with boolean as a key will be replaced with stored value.
 
```php
use CollectionType\TypeValidator\BoolTypeValidator;

$boolTypeValidator = new BoolTypeValidator();
```

### Compound types
-----------------

#### ArrayTypeValidator
-----------------------

It checks if passed element is an array.
 
```php
use CollectionType\TypeValidator\ArrayTypeValidator;

$arrayTypeValidator = new ArrayTypeValidator();
```

#### ObjectTypeValidator
------------------------

It checks if passed element is an object.
 
```php
use CollectionType\TypeValidator\ObjectTypeValidator;

$objectTypeValidator = new ObjectTypeValidator();
```

### Special type
----------------

#### NullTypeValidator
----------------------

It checks if passed element is a null. This validator accepts only `null` value. Be careful when using NullTypeValidator as a key in Map. In Map it is possible add only one element `null`. 
Every new element with `null` as a key will be replaced with stored value.
 
```php
use CollectionType\TypeValidator\NullTypeValidator;

$nullTypeValidator = new NullTypeValidator();
```

## How to create own type
-------------------------

In custom cases, you will need special types defined for your purposes. For example, you need collection of some kind attributes.
Your attribute object will look like this:

```php
namespace my_own_namespace;

class Attribute
{
    private $name;
    
    private $value;
    
    public function __construct($name, $value) 
    {
        $this->name = $name;
        $this->value = $value;
    }
    
    public function getName()
    {
        return $this->name;
    }
    
    public function getValue()
    {
        return $value;
    }
}
```

Predefined `CollectionType\TypeValidator\ObjectTypeValidator` is not good enough for collection to hold attribute objects, because
`CollectionType\TypeValidator\ObjectTypeValidator` validates (`isValid($object)` returns `true`) every object, not only `Attribute` instance.
 
You can create your own TypeValidator:

```php
namespace my_own_namespace\CollectionType\TypeValidator;

class AttributeTypeValidator implements TypeValidatorInterface
{

    public function isValid($value)
    {
        return 'my_own_namespace\Attribute' === get_class($value);
    }
}
```

Better practise is to define interface for `Attribute` class and check them in `isValid` method:

```php
namespace my_own_namespace;

interface AttributeInterface
{}

class Attribute implements AttributeInterface
{
    // ...
}

//...

namespace my_own_namespace\CollectionType\TypeValidator;

class AttributeTypeValidator implements TypeValidatorInterface
{

    public function isValid($value)
    {
        return $value instanceof TestInterface;
    }
}
```

### Navigate
------------

* next to [API](/docs/5.API.md)
* back to [Usage](/docs/3.Usage.md)
* go to [index](/docs/README.md)

