//! \file       ArcALL.cs
//! \date       2022 May 07
//! \brief      Giga resource archive.
//
// Copyright (C) 2022 by morkt
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
//

using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.IO;
using GameRes.Utility;

namespace GameRes.Formats.Giga
{
    [Export(typeof(ArchiveFormat))]
    public class AllOpener : ArchiveFormat
    {
        public override string         Tag { get { return "ALL/GIGA"; } }
        public override string Description { get { return "Giga resource archive"; } }
        public override uint     Signature { get { return 0; } }
        public override bool  IsHierarchic { get { return false; } }
        public override bool      CanWrite { get { return false; } }

        public AllOpener ()
        {
            Extensions = new[] { "273" };
            ContainedFormats = new[] { "BMP", "DAT/GENERIC" };
        }

        public override ArcFile TryOpen (ArcView file)
        {
            var base_name = Path.GetFileName (file.Name);
            List<Entry> dir;
            if (!FileMap273.TryGetValue (base_name, out dir) || 0 == dir.Count)
                return null;
            return new ArcFile (file, this, dir);
        }

        public override Stream OpenEntry (ArcFile arc, Entry entry)
        {
            var pent = entry as PackedEntry;
            if (null == pent)
                return base.OpenEntry (arc, entry);
            var data = new byte[pent.UnpackedSize];
            using (var input = arc.File.CreateStream (pent.Offset, pent.Size))
                LzssUnpack (input, data);
            return new BinMemoryStream (data);
        }

        internal static void LzssUnpack (IBinaryStream input, byte[] output)
        {
            int dst = 0;
            int bitcount = input.ReadInt32();
            if (0 == bitcount)
            {
                input.Read (output, 0, output.Length);
                return;
            }
            int bytecount = bitcount >> 3;
            int bits = 8;
            while (dst < output.Length)
            {
                do
                {
                    int ctl = input.ReadUInt8();
                    --bytecount;
                    for (int i = 0; i < bits; ++i)
                    {
                        if ((ctl & 1) != 0)
                        {
                            output[dst++] = input.ReadUInt8();
                        }
                        else
                        {
                            int count = input.ReadUInt16();
                            int offset = dst - (count & 0xFFF) - 1;
                            if ((count & 0xF000) != 0)
                            {
                                count = (count >> 12) + 2;
                            }
                            else
                            {
                                count = input.ReadUInt8() + 0x12;
                            }
                            Binary.CopyOverlapped (output, offset, dst, count);
                            dst += count;
                        }
                        ctl >>= 1;
                    }
                }
                while (bytecount > 0);
                if (bitcount <= 0)
                    break;
                bits = bitcount & 7;
                bitcount = 0;
            }
        }

        static readonly Dictionary<string, List<Entry>>  FileMap273 = new Dictionary<string, List<Entry>> (StringComparer.OrdinalIgnoreCase)
        {
            {
                "ALLCHP.273", new List<Entry>() {
                    new PackedEntry { Name = "ankei_a.chp", Type = "image", Offset = 0x0, UnpackedSize = 0x0C436, Size = 0x13C5, IsPacked = true },
                    new PackedEntry { Name = "ankei_b.chp", Type = "image", Offset = 0x13C5, UnpackedSize = 0x8436, Size = 0x108C, IsPacked = true },
                    new PackedEntry { Name = "ankei_c.chp", Type = "image", Offset = 0x2451, UnpackedSize = 0x0C436, Size = 0x136C, IsPacked = true },
                    new PackedEntry { Name = "beem_a.chp", Type = "image", Offset = 0x37BD, UnpackedSize = 0x8436, Size = 0x0ED0, IsPacked = true },
                    new PackedEntry { Name = "beem_b.chp", Type = "image", Offset = 0x468D, UnpackedSize = 0x8436, Size = 0x0CE6, IsPacked = true },
                    new PackedEntry { Name = "beem_c.chp", Type = "image", Offset = 0x5373, UnpackedSize = 0x8436, Size = 0x0ED5, IsPacked = true },
                    new PackedEntry { Name = "b_abel_a.chp", Type = "image", Offset = 0x6248, UnpackedSize = 0x78436, Size = 0x1E2BF, IsPacked = true },
                    new PackedEntry { Name = "b_ann_a.chp", Type = "image", Offset = 0x24507, UnpackedSize = 0x0A8436, Size = 0x2B22D, IsPacked = true },
                    new PackedEntry { Name = "b_ann_b.chp", Type = "image", Offset = 0x4F734, UnpackedSize = 0x0AC436, Size = 0x2B620, IsPacked = true },
                    new PackedEntry { Name = "b_ann_c.chp", Type = "image", Offset = 0x7AD54, UnpackedSize = 0x0A4436, Size = 0x27C72, IsPacked = true },
                    new PackedEntry { Name = "b_bal1_a.chp", Type = "image", Offset = 0x0A29C6, UnpackedSize = 0x3C436, Size = 0x104BB, IsPacked = true },
                    new PackedEntry { Name = "b_bat_a.chp", Type = "image", Offset = 0x0B2E81, UnpackedSize = 0x34436, Size = 0x0A164, IsPacked = true },
                    new PackedEntry { Name = "b_bat_b.chp", Type = "image", Offset = 0x0BCFE5, UnpackedSize = 0x34436, Size = 0x0A21F, IsPacked = true },
                    new PackedEntry { Name = "b_bee1_a.chp", Type = "image", Offset = 0x0C7204, UnpackedSize = 0x64436, Size = 0x15DC1, IsPacked = true },
                    new PackedEntry { Name = "b_bobe_a.chp", Type = "image", Offset = 0x0DCFC5, UnpackedSize = 0x38436, Size = 0x0D98F, IsPacked = true },
                    new PackedEntry { Name = "b_bosc_a.chp", Type = "image", Offset = 0x0EA954, UnpackedSize = 0x20436, Size = 0x7562, IsPacked = true },
                    new PackedEntry { Name = "b_cain_a.chp", Type = "image", Offset = 0x0F1EB6, UnpackedSize = 0x0A4436, Size = 0x28B09, IsPacked = true },
                    new PackedEntry { Name = "b_cain_b.chp", Type = "image", Offset = 0x11A9BF, UnpackedSize = 0x0AC436, Size = 0x297BB, IsPacked = true },
                    new PackedEntry { Name = "b_cain_c.chp", Type = "image", Offset = 0x14417A, UnpackedSize = 0x9C436, Size = 0x2557D, IsPacked = true },
                    new PackedEntry { Name = "b_chia_a.chp", Type = "image", Offset = 0x1696F7, UnpackedSize = 0x64436, Size = 0x185EA, IsPacked = true },
                    new PackedEntry { Name = "b_chib_a.chp", Type = "image", Offset = 0x181CE1, UnpackedSize = 0x64436, Size = 0x189B8, IsPacked = true },
                    new PackedEntry { Name = "b_chic_a.chp", Type = "image", Offset = 0x19A699, UnpackedSize = 0x2C436, Size = 0x0BFD9, IsPacked = true },
                    new PackedEntry { Name = "b_clim_a.chp", Type = "image", Offset = 0x1A6672, UnpackedSize = 0x3C436, Size = 0x0F216, IsPacked = true },
                    new PackedEntry { Name = "b_clim_b.chp", Type = "image", Offset = 0x1B5888, UnpackedSize = 0x44436, Size = 0x0F9E8, IsPacked = true },
                    new PackedEntry { Name = "b_clim_c.chp", Type = "image", Offset = 0x1C5270, UnpackedSize = 0x34436, Size = 0x0BD94, IsPacked = true },
                    new PackedEntry { Name = "b_cur1_a.chp", Type = "image", Offset = 0x1D1004, UnpackedSize = 0x14436, Size = 0x55D8, IsPacked = true },
                    new PackedEntry { Name = "b_cur2_a.chp", Type = "image", Offset = 0x1D65DC, UnpackedSize = 0x68436, Size = 0x14FB9, IsPacked = true },
                    new PackedEntry { Name = "b_dem1_a.chp", Type = "image", Offset = 0x1EB595, UnpackedSize = 0x54436, Size = 0x14028, IsPacked = true },
                    new PackedEntry { Name = "b_dem1_b.chp", Type = "image", Offset = 0x1FF5BD, UnpackedSize = 0x54436, Size = 0x1378C, IsPacked = true },
                    new PackedEntry { Name = "b_dog_b.chp", Type = "image", Offset = 0x212D49, UnpackedSize = 0x58436, Size = 0x15D6C, IsPacked = true },
                    new PackedEntry { Name = "b_ga1a_b.chp", Type = "image", Offset = 0x228AB5, UnpackedSize = 0x2C436, Size = 0x0AF19, IsPacked = true },
                    new PackedEntry { Name = "b_ga1b_b.chp", Type = "image", Offset = 0x2339CE, UnpackedSize = 0x68436, Size = 0x1625A, IsPacked = true },
                    new PackedEntry { Name = "b_ga2a_a.chp", Type = "image", Offset = 0x249C28, UnpackedSize = 0x10436, Size = 0x340E, IsPacked = true },
                    new PackedEntry { Name = "b_ga2b_a.chp", Type = "image", Offset = 0x24D036, UnpackedSize = 0x48436, Size = 0x0F5F0, IsPacked = true },
                    new PackedEntry { Name = "b_hang_a.chp", Type = "image", Offset = 0x25C626, UnpackedSize = 0x78436, Size = 0x1B8CD, IsPacked = true },
                    new PackedEntry { Name = "b_harp_a.chp", Type = "image", Offset = 0x277EF3, UnpackedSize = 0x5C436, Size = 0x1AF96, IsPacked = true },
                    new PackedEntry { Name = "b_harp_b.chp", Type = "image", Offset = 0x292E89, UnpackedSize = 0x58436, Size = 0x1A35F, IsPacked = true },
                    new PackedEntry { Name = "b_hebi_a.chp", Type = "image", Offset = 0x2AD1E8, UnpackedSize = 0x40436, Size = 0x114DB, IsPacked = true },
                    new PackedEntry { Name = "b_hei1_a.chp", Type = "image", Offset = 0x2BE6C3, UnpackedSize = 0x5C436, Size = 0x15985, IsPacked = true },
                    new PackedEntry { Name = "b_icon.chp", Type = "image", Offset = 0x2D4048, UnpackedSize = 0x20436, Size = 0x69B8, IsPacked = true },
                    new PackedEntry { Name = "b_iconb.chp", Type = "image", Offset = 0x2DAA00, UnpackedSize = 0x8436, Size = 0x1247, IsPacked = true },
                    new PackedEntry { Name = "b_jian_b.chp", Type = "image", Offset = 0x2DBC47, UnpackedSize = 0x98436, Size = 0x23179, IsPacked = true },
                    new PackedEntry { Name = "b_kasi_a.chp", Type = "image", Offset = 0x2FEDC0, UnpackedSize = 0x6C436, Size = 0x1AFAF, IsPacked = true },
                    new PackedEntry { Name = "b_kime_b.chp", Type = "image", Offset = 0x319D6F, UnpackedSize = 0x8C436, Size = 0x28E6F, IsPacked = true },
                    new PackedEntry { Name = "b_kin1_a.chp", Type = "image", Offset = 0x342BDE, UnpackedSize = 0x64436, Size = 0x17298, IsPacked = true },
                    new PackedEntry { Name = "b_kin2_a.chp", Type = "image", Offset = 0x359E76, UnpackedSize = 0x20436, Size = 0x6BE9, IsPacked = true },
                    new PackedEntry { Name = "b_kin3_a.chp", Type = "image", Offset = 0x360A5F, UnpackedSize = 0x48436, Size = 0x0E5D6, IsPacked = true },
                    new PackedEntry { Name = "b_kobo_a.chp", Type = "image", Offset = 0x36F035, UnpackedSize = 0x30436, Size = 0x0B881, IsPacked = true },
                    new PackedEntry { Name = "b_kobo_b.chp", Type = "image", Offset = 0x37A8B6, UnpackedSize = 0x2C436, Size = 0x0A1FC, IsPacked = true },
                    new PackedEntry { Name = "b_kobo_c.chp", Type = "image", Offset = 0x384AB2, UnpackedSize = 0x28436, Size = 0x8DEA, IsPacked = true },
                    new PackedEntry { Name = "b_mac1_b.chp", Type = "image", Offset = 0x38D89C, UnpackedSize = 0x10436, Size = 0x3160, IsPacked = true },
                    new PackedEntry { Name = "b_mac2_b.chp", Type = "image", Offset = 0x3909FC, UnpackedSize = 0x10436, Size = 0x30D4, IsPacked = true },
                    new PackedEntry { Name = "b_mac3_b.chp", Type = "image", Offset = 0x393AD0, UnpackedSize = 0x50436, Size = 0x111D5, IsPacked = true },
                    new PackedEntry { Name = "b_magic0.chp", Type = "image", Offset = 0x3A4CA5, UnpackedSize = 0x14436, Size = 0x3E5C, IsPacked = true },
                    new PackedEntry { Name = "b_magic1.chp", Type = "image", Offset = 0x3A8B01, UnpackedSize = 0x14436, Size = 0x3A5F, IsPacked = true },
                    new PackedEntry { Name = "b_magic2.chp", Type = "image", Offset = 0x3AC560, UnpackedSize = 0x4436, Size = 0x558, IsPacked = true },
                    new PackedEntry { Name = "b_magic3.chp", Type = "image", Offset = 0x3ACAB8, UnpackedSize = 0x4436, Size = 0x3AF, IsPacked = true },
                    new PackedEntry { Name = "b_magic4.chp", Type = "image", Offset = 0x3ACE67, UnpackedSize = 0x10436, Size = 0x3FB6, IsPacked = true },
                    new PackedEntry { Name = "b_magic5.chp", Type = "image", Offset = 0x3B0E1D, UnpackedSize = 0x18436, Size = 0x5BE3, IsPacked = true },
                    new PackedEntry { Name = "b_magic6.chp", Type = "image", Offset = 0x3B6A00, UnpackedSize = 0x18436, Size = 0x5D43, IsPacked = true },
                    new PackedEntry { Name = "b_magic7.chp", Type = "image", Offset = 0x3BC743, UnpackedSize = 0x20436, Size = 0x5913, IsPacked = true },
                    new PackedEntry { Name = "b_magic8.chp", Type = "image", Offset = 0x3C2056, UnpackedSize = 0x0C436, Size = 0x230C, IsPacked = true },
                    new PackedEntry { Name = "b_maj1_a.chp", Type = "image", Offset = 0x3C4362, UnpackedSize = 0x30436, Size = 0x0B252, IsPacked = true },
                    new PackedEntry { Name = "b_pon_a.chp", Type = "image", Offset = 0x3CF5B4, UnpackedSize = 0x70436, Size = 0x1D13E, IsPacked = true },
                    new PackedEntry { Name = "b_pon_b.chp", Type = "image", Offset = 0x3EC6F2, UnpackedSize = 0x6C436, Size = 0x1C2BB, IsPacked = true },
                    new PackedEntry { Name = "b_pon_c.chp", Type = "image", Offset = 0x4089AD, UnpackedSize = 0x74436, Size = 0x1C839, IsPacked = true },
                    new PackedEntry { Name = "b_riq_a.chp", Type = "image", Offset = 0x4251E6, UnpackedSize = 0x5C436, Size = 0x18D82, IsPacked = true },
                    new PackedEntry { Name = "b_riq_b.chp", Type = "image", Offset = 0x43DF68, UnpackedSize = 0x5C436, Size = 0x17535, IsPacked = true },
                    new PackedEntry { Name = "b_riq_c.chp", Type = "image", Offset = 0x45549D, UnpackedSize = 0x58436, Size = 0x14B4C, IsPacked = true },
                    new PackedEntry { Name = "b_rob1_a.chp", Type = "image", Offset = 0x469FE9, UnpackedSize = 0x48436, Size = 0x12554, IsPacked = true },
                    new PackedEntry { Name = "b_russ_a.chp", Type = "image", Offset = 0x47C53D, UnpackedSize = 0x6C436, Size = 0x19E1F, IsPacked = true },
                    new PackedEntry { Name = "b_russ_b.chp", Type = "image", Offset = 0x49635C, UnpackedSize = 0x6C436, Size = 0x1A2F6, IsPacked = true },
                    new PackedEntry { Name = "b_russ_c.chp", Type = "image", Offset = 0x4B0652, UnpackedSize = 0x60436, Size = 0x178CE, IsPacked = true },
                    new PackedEntry { Name = "b_sab1_a.chp", Type = "image", Offset = 0x4C7F20, UnpackedSize = 0x90436, Size = 0x23C3D, IsPacked = true },
                    new PackedEntry { Name = "b_semu_b.chp", Type = "image", Offset = 0x4EBB5D, UnpackedSize = 0x74436, Size = 0x1A958, IsPacked = true },
                    new PackedEntry { Name = "b_skal_a.chp", Type = "image", Offset = 0x5064B5, UnpackedSize = 0x5C436, Size = 0x1596C, IsPacked = true },
                    new PackedEntry { Name = "b_skal_b.chp", Type = "image", Offset = 0x51BE21, UnpackedSize = 0x5C436, Size = 0x15C42, IsPacked = true },
                    new PackedEntry { Name = "b_skal_c.chp", Type = "image", Offset = 0x531A63, UnpackedSize = 0x58436, Size = 0x147BA, IsPacked = true },
                    new PackedEntry { Name = "b_sura_a.chp", Type = "image", Offset = 0x54621D, UnpackedSize = 0x28436, Size = 0x6655, IsPacked = true },
                    new PackedEntry { Name = "b_sura_b.chp", Type = "image", Offset = 0x54C872, UnpackedSize = 0x24436, Size = 0x624F, IsPacked = true },
                    new PackedEntry { Name = "b_sura_c.chp", Type = "image", Offset = 0x552AC1, UnpackedSize = 0x24436, Size = 0x5EDE, IsPacked = true },
                    new PackedEntry { Name = "b_sys.chp", Type = "image", Offset = 0x55899F, UnpackedSize = 0x18436, Size = 0x34EC, IsPacked = true },
                    new PackedEntry { Name = "b_sys2.chp", Type = "image", Offset = 0x55BE8B, UnpackedSize = 0x18436, Size = 0x3201, IsPacked = true },
                    new PackedEntry { Name = "b_thi1_a.chp", Type = "image", Offset = 0x55F08C, UnpackedSize = 0x44436, Size = 0x11DFF, IsPacked = true },
                    new PackedEntry { Name = "b_tira_a.chp", Type = "image", Offset = 0x570E8B, UnpackedSize = 0x70436, Size = 0x1A351, IsPacked = true },
                    new PackedEntry { Name = "b_tora_a.chp", Type = "image", Offset = 0x58B1DC, UnpackedSize = 0x80436, Size = 0x24E08, IsPacked = true },
                    new PackedEntry { Name = "b_tora_b.chp", Type = "image", Offset = 0x5AFFE4, UnpackedSize = 0x88436, Size = 0x25294, IsPacked = true },
                    new PackedEntry { Name = "b_tora_c.chp", Type = "image", Offset = 0x5D5278, UnpackedSize = 0x7C436, Size = 0x20C2C, IsPacked = true },
                    new PackedEntry { Name = "b_tree_a.chp", Type = "image", Offset = 0x5F5EA4, UnpackedSize = 0x5C436, Size = 0x13587, IsPacked = true },
                    new PackedEntry { Name = "b_uma.chp", Type = "image", Offset = 0x60942B, UnpackedSize = 0x4436, Size = 0x10C, IsPacked = true },
                    new PackedEntry { Name = "b_uma1_b.chp", Type = "image", Offset = 0x609537, UnpackedSize = 0x8C436, Size = 0x2193B, IsPacked = true },
                    new PackedEntry { Name = "b_wasi_a.chp", Type = "image", Offset = 0x62AE72, UnpackedSize = 0x70436, Size = 0x1C948, IsPacked = true },
                    new PackedEntry { Name = "b_wasi_b.chp", Type = "image", Offset = 0x6477BA, UnpackedSize = 0x70436, Size = 0x1C856, IsPacked = true },
                    new PackedEntry { Name = "b_wiz1_a.chp", Type = "image", Offset = 0x664010, UnpackedSize = 0x64436, Size = 0x173C1, IsPacked = true },
                    new PackedEntry { Name = "b_yaji_c.chp", Type = "image", Offset = 0x67B3D1, UnpackedSize = 0x80436, Size = 0x20154, IsPacked = true },
                    new PackedEntry { Name = "b_zom1_b.chp", Type = "image", Offset = 0x69B525, UnpackedSize = 0x40436, Size = 0x0E7FA, IsPacked = true },
                    new PackedEntry { Name = "con_chr.chp", Type = "image", Offset = 0x6A9D1F, UnpackedSize = 0x10436, Size = 0x12C4, IsPacked = true },
                    new PackedEntry { Name = "f_abel.chp", Type = "image", Offset = 0x6AAFE3, UnpackedSize = 0x20436, Size = 0x6F04, IsPacked = true },
                    new PackedEntry { Name = "f_annin.chp", Type = "image", Offset = 0x6B1EE7, UnpackedSize = 0x24436, Size = 0x911F, IsPacked = true },
                    new PackedEntry { Name = "f_baba.chp", Type = "image", Offset = 0x6BB006, UnpackedSize = 0x10436, Size = 0x361A, IsPacked = true },
                    new PackedEntry { Name = "f_bajil.chp", Type = "image", Offset = 0x6BE620, UnpackedSize = 0x14436, Size = 0x492E, IsPacked = true },
                    new PackedEntry { Name = "f_baten.chp", Type = "image", Offset = 0x6C2F4E, UnpackedSize = 0x10436, Size = 0x331B, IsPacked = true },
                    new PackedEntry { Name = "f_bobes.chp", Type = "image", Offset = 0x6C6269, UnpackedSize = 0x34436, Size = 0x0D637, IsPacked = true },
                    new PackedEntry { Name = "f_bosco.chp", Type = "image", Offset = 0x6D38A0, UnpackedSize = 0x28436, Size = 0x0A5D0, IsPacked = true },
                    new PackedEntry { Name = "f_boy1.chp", Type = "image", Offset = 0x6DDE70, UnpackedSize = 0x18436, Size = 0x54C6, IsPacked = true },
                    new PackedEntry { Name = "f_buki.chp", Type = "image", Offset = 0x6E3336, UnpackedSize = 0x10436, Size = 0x4599, IsPacked = true },
                    new PackedEntry { Name = "f_cain.chp", Type = "image", Offset = 0x6E78CF, UnpackedSize = 0x48436, Size = 0x1312E, IsPacked = true },
                    new PackedEntry { Name = "f_chibi.chp", Type = "image", Offset = 0x6FA9FD, UnpackedSize = 0x10436, Size = 0x3346, IsPacked = true },
                    new PackedEntry { Name = "f_clim1.chp", Type = "image", Offset = 0x6FDD43, UnpackedSize = 0x10436, Size = 0x372A, IsPacked = true },
                    new PackedEntry { Name = "f_clim2.chp", Type = "image", Offset = 0x70146D, UnpackedSize = 0x18436, Size = 0x5C51, IsPacked = true },
                    new PackedEntry { Name = "f_coin.chp", Type = "image", Offset = 0x7070BE, UnpackedSize = 0x10436, Size = 0x4830, IsPacked = true },
                    new PackedEntry { Name = "f_cura.chp", Type = "image", Offset = 0x70B8EE, UnpackedSize = 0x14436, Size = 0x44C6, IsPacked = true },
                    new PackedEntry { Name = "f_door1.chp", Type = "image", Offset = 0x70FDB4, UnpackedSize = 0x8436, Size = 0x0E4B, IsPacked = true },
                    new PackedEntry { Name = "f_door10.chp", Type = "image", Offset = 0x710BFF, UnpackedSize = 0x8436, Size = 0x802, IsPacked = true },
                    new PackedEntry { Name = "f_door11.chp", Type = "image", Offset = 0x711401, UnpackedSize = 0x0C436, Size = 0x0CBF, IsPacked = true },
                    new PackedEntry { Name = "f_door12.chp", Type = "image", Offset = 0x7120C0, UnpackedSize = 0x0C436, Size = 0x2F24, IsPacked = true },
                    new PackedEntry { Name = "f_door2.chp", Type = "image", Offset = 0x714FE4, UnpackedSize = 0x0C436, Size = 0x1633, IsPacked = true },
                    new PackedEntry { Name = "f_door3.chp", Type = "image", Offset = 0x716617, UnpackedSize = 0x0C436, Size = 0x159C, IsPacked = true },
                    new PackedEntry { Name = "f_door4.chp", Type = "image", Offset = 0x717BB3, UnpackedSize = 0x8436, Size = 0x0EA9, IsPacked = true },
                    new PackedEntry { Name = "f_door5.chp", Type = "image", Offset = 0x718A5C, UnpackedSize = 0x8436, Size = 0x0C35, IsPacked = true },
                    new PackedEntry { Name = "f_door6.chp", Type = "image", Offset = 0x719691, UnpackedSize = 0x8436, Size = 0x0D1A, IsPacked = true },
                    new PackedEntry { Name = "f_door7.chp", Type = "image", Offset = 0x71A3AB, UnpackedSize = 0x4436, Size = 0x74C, IsPacked = true },
                    new PackedEntry { Name = "f_door8.chp", Type = "image", Offset = 0x71AAF7, UnpackedSize = 0x8436, Size = 0x0AAF, IsPacked = true },
                    new PackedEntry { Name = "f_door9.chp", Type = "image", Offset = 0x71B5A6, UnpackedSize = 0x0C436, Size = 0x0CC2, IsPacked = true },
                    new PackedEntry { Name = "f_dummy.chp", Type = "image", Offset = 0x71C268, UnpackedSize = 0x4436, Size = 0x10C, IsPacked = true },
                    new PackedEntry { Name = "f_funsui.chp", Type = "image", Offset = 0x71C374, UnpackedSize = 0x20436, Size = 0x74D2, IsPacked = true },
                    new PackedEntry { Name = "f_futa.chp", Type = "image", Offset = 0x723846, UnpackedSize = 0x4436, Size = 0x0A73, IsPacked = true },
                    new PackedEntry { Name = "f_gaki.chp", Type = "image", Offset = 0x7242B9, UnpackedSize = 0x0C436, Size = 0x2564, IsPacked = true },
                    new PackedEntry { Name = "f_gal1.chp", Type = "image", Offset = 0x72681D, UnpackedSize = 0x18436, Size = 0x5128, IsPacked = true },
                    new PackedEntry { Name = "f_gal2.chp", Type = "image", Offset = 0x72B945, UnpackedSize = 0x18436, Size = 0x5160, IsPacked = true },
                    new PackedEntry { Name = "f_gana.chp", Type = "image", Offset = 0x730AA5, UnpackedSize = 0x14436, Size = 0x4C76, IsPacked = true },
                    new PackedEntry { Name = "f_hasi.chp", Type = "image", Offset = 0x73571B, UnpackedSize = 0x4436, Size = 0x646, IsPacked = true },
                    new PackedEntry { Name = "f_hei.chp", Type = "image", Offset = 0x735D61, UnpackedSize = 0x28436, Size = 0x8F85, IsPacked = true },
                    new PackedEntry { Name = "f_icon.chp", Type = "image", Offset = 0x73ECE6, UnpackedSize = 0x24436, Size = 0x962B, IsPacked = true },
                    new PackedEntry { Name = "f_iconb.chp", Type = "image", Offset = 0x748311, UnpackedSize = 0x14436, Size = 0x31C5, IsPacked = true },
                    new PackedEntry { Name = "f_jiji1.chp", Type = "image", Offset = 0x74B4D6, UnpackedSize = 0x18436, Size = 0x448C, IsPacked = true },
                    new PackedEntry { Name = "f_jiji2.chp", Type = "image", Offset = 0x74F962, UnpackedSize = 0x18436, Size = 0x4C6D, IsPacked = true },
                    new PackedEntry { Name = "f_kasira.chp", Type = "image", Offset = 0x7545CF, UnpackedSize = 0x1C436, Size = 0x4276, IsPacked = true },
                    new PackedEntry { Name = "f_king.chp", Type = "image", Offset = 0x758845, UnpackedSize = 0x18436, Size = 0x630B, IsPacked = true },
                    new PackedEntry { Name = "f_kuro1.chp", Type = "image", Offset = 0x75EB50, UnpackedSize = 0x10436, Size = 0x2F64, IsPacked = true },
                    new PackedEntry { Name = "f_kuro2.chp", Type = "image", Offset = 0x761AB4, UnpackedSize = 0x8436, Size = 0x158A, IsPacked = true },
                    new PackedEntry { Name = "f_kuro3.chp", Type = "image", Offset = 0x76303E, UnpackedSize = 0x8436, Size = 0x1557, IsPacked = true },
                    new PackedEntry { Name = "f_mado.chp", Type = "image", Offset = 0x764595, UnpackedSize = 0x1C436, Size = 0x62E6, IsPacked = true },
                    new PackedEntry { Name = "f_mafin1.chp", Type = "image", Offset = 0x76A87B, UnpackedSize = 0x8436, Size = 0x1695, IsPacked = true },
                    new PackedEntry { Name = "f_mafin2.chp", Type = "image", Offset = 0x76BF10, UnpackedSize = 0x10436, Size = 0x3AA7, IsPacked = true },
                    new PackedEntry { Name = "f_mafin3.chp", Type = "image", Offset = 0x76F9B7, UnpackedSize = 0x8436, Size = 0x1A9E, IsPacked = true },
                    new PackedEntry { Name = "f_mafin4.chp", Type = "image", Offset = 0x771455, UnpackedSize = 0x1C436, Size = 0x56DE, IsPacked = true },
                    new PackedEntry { Name = "f_mam.chp", Type = "image", Offset = 0x776B33, UnpackedSize = 0x18436, Size = 0x4C4B, IsPacked = true },
                    new PackedEntry { Name = "f_maypl.chp", Type = "image", Offset = 0x77B77E, UnpackedSize = 0x0C436, Size = 0x2E89, IsPacked = true },
                    new PackedEntry { Name = "f_meido.chp", Type = "image", Offset = 0x77E607, UnpackedSize = 0x18436, Size = 0x4C5E, IsPacked = true },
                    new PackedEntry { Name = "f_milf1.chp", Type = "image", Offset = 0x783265, UnpackedSize = 0x2C436, Size = 0x9B21, IsPacked = true },
                    new PackedEntry { Name = "f_milf2.chp", Type = "image", Offset = 0x78CD86, UnpackedSize = 0x20436, Size = 0x7AB0, IsPacked = true },
                    new PackedEntry { Name = "f_milf3.chp", Type = "image", Offset = 0x794836, UnpackedSize = 0x8436, Size = 0x1D55, IsPacked = true },
                    new PackedEntry { Name = "f_mitugo.chp", Type = "image", Offset = 0x79658B, UnpackedSize = 0x10436, Size = 0x2D45, IsPacked = true },
                    new PackedEntry { Name = "f_oba.chp", Type = "image", Offset = 0x7992D0, UnpackedSize = 0x18436, Size = 0x4FFE, IsPacked = true },
                    new PackedEntry { Name = "f_panna.chp", Type = "image", Offset = 0x79E2CE, UnpackedSize = 0x14436, Size = 0x4E5F, IsPacked = true },
                    new PackedEntry { Name = "f_ponti.chp", Type = "image", Offset = 0x7A312D, UnpackedSize = 0x24436, Size = 0x8D7F, IsPacked = true },
                    new PackedEntry { Name = "f_queen.chp", Type = "image", Offset = 0x7ABEAC, UnpackedSize = 0x14436, Size = 0x58C3, IsPacked = true },
                    new PackedEntry { Name = "f_riqur1.chp", Type = "image", Offset = 0x7B176F, UnpackedSize = 0x1C436, Size = 0x688D, IsPacked = true },
                    new PackedEntry { Name = "f_riqur2.chp", Type = "image", Offset = 0x7B7FFC, UnpackedSize = 0x8436, Size = 0x1ABC, IsPacked = true },
                    new PackedEntry { Name = "f_russ1.chp", Type = "image", Offset = 0x7B9AB8, UnpackedSize = 0x20436, Size = 0x6C37, IsPacked = true },
                    new PackedEntry { Name = "f_russ2.chp", Type = "image", Offset = 0x7C06EF, UnpackedSize = 0x14436, Size = 0x500B, IsPacked = true },
                    new PackedEntry { Name = "f_sable.chp", Type = "image", Offset = 0x7C56FA, UnpackedSize = 0x18436, Size = 0x59B6, IsPacked = true },
                    new PackedEntry { Name = "f_sable2.chp", Type = "image", Offset = 0x7CB0B0, UnpackedSize = 0x30436, Size = 0x948E, IsPacked = true },
                    new PackedEntry { Name = "f_selis.chp", Type = "image", Offset = 0x7D453E, UnpackedSize = 0x14436, Size = 0x50EB, IsPacked = true },
                    new PackedEntry { Name = "f_selis1.chp", Type = "image", Offset = 0x7D9629, UnpackedSize = 0x0C436, Size = 0x2C74, IsPacked = true },
                    new PackedEntry { Name = "f_selis2.chp", Type = "image", Offset = 0x7DC29D, UnpackedSize = 0x1C436, Size = 0x7405, IsPacked = true },
                    new PackedEntry { Name = "f_selis3.chp", Type = "image", Offset = 0x7E36A2, UnpackedSize = 0x10436, Size = 0x3F4E, IsPacked = true },
                    new PackedEntry { Name = "f_shake.chp", Type = "image", Offset = 0x7E75F0, UnpackedSize = 0x18436, Size = 0x6E76, IsPacked = true },
                    new PackedEntry { Name = "f_shop1.chp", Type = "image", Offset = 0x7EE466, UnpackedSize = 0x1C436, Size = 0x5BEE, IsPacked = true },
                    new PackedEntry { Name = "f_shop3.chp", Type = "image", Offset = 0x7F4054, UnpackedSize = 0x1C436, Size = 0x5978, IsPacked = true },
                    new PackedEntry { Name = "f_sipul1.chp", Type = "image", Offset = 0x7F99CC, UnpackedSize = 0x10436, Size = 0x32F1, IsPacked = true },
                    new PackedEntry { Name = "f_sipul2.chp", Type = "image", Offset = 0x7FCCBD, UnpackedSize = 0x8436, Size = 0x16C7, IsPacked = true },
                    new PackedEntry { Name = "f_sipul3.chp", Type = "image", Offset = 0x7FE384, UnpackedSize = 0x2C436, Size = 0x9F3A, IsPacked = true },
                    new PackedEntry { Name = "f_syoko.chp", Type = "image", Offset = 0x8082BE, UnpackedSize = 0x10436, Size = 0x4199, IsPacked = true },
                    new PackedEntry { Name = "f_syoko2.chp", Type = "image", Offset = 0x80C457, UnpackedSize = 0x28436, Size = 0x6CD1, IsPacked = true },
                    new PackedEntry { Name = "f_sys.chp", Type = "image", Offset = 0x813128, UnpackedSize = 0x8436, Size = 0x0A01, IsPacked = true },
                    new PackedEntry { Name = "f_sys2.chp", Type = "image", Offset = 0x813B29, UnpackedSize = 0x8436, Size = 0x72C, IsPacked = true },
                    new PackedEntry { Name = "f_syuku.chp", Type = "image", Offset = 0x814255, UnpackedSize = 0x10436, Size = 0x3520, IsPacked = true },
                    new PackedEntry { Name = "f_taka.chp", Type = "image", Offset = 0x817775, UnpackedSize = 0x4436, Size = 0x0B3D, IsPacked = true },
                    new PackedEntry { Name = "f_tira.chp", Type = "image", Offset = 0x8182B2, UnpackedSize = 0x10436, Size = 0x247E, IsPacked = true },
                    new PackedEntry { Name = "f_tolte.chp", Type = "image", Offset = 0x81A730, UnpackedSize = 0x20436, Size = 0x6E12, IsPacked = true },
                    new PackedEntry { Name = "f_tolte1.chp", Type = "image", Offset = 0x821542, UnpackedSize = 0x4436, Size = 0x10C, IsPacked = true },
                    new PackedEntry { Name = "f_tryu.chp", Type = "image", Offset = 0x82164E, UnpackedSize = 0x18436, Size = 0x49BC, IsPacked = true },
                    new PackedEntry { Name = "f_uma.chp", Type = "image", Offset = 0x82600A, UnpackedSize = 0x2C436, Size = 0x8DD6, IsPacked = true },
                    new PackedEntry { Name = "f_zaha.chp", Type = "image", Offset = 0x82EDE0, UnpackedSize = 0x18436, Size = 0x54D1, IsPacked = true },
                    new PackedEntry { Name = "g_cain.chp", Type = "image", Offset = 0x8342B1, UnpackedSize = 0x20436, Size = 0x8E99, IsPacked = true },
                    new PackedEntry { Name = "ha.chp", Type = "image", Offset = 0x83D14A, UnpackedSize = 0x14436, Size = 0x2CC6, IsPacked = true },
                    new PackedEntry { Name = "kidan_a.chp", Type = "image", Offset = 0x83FE10, UnpackedSize = 0x1C436, Size = 0x5019, IsPacked = true },
                    new PackedEntry { Name = "kidan_b.chp", Type = "image", Offset = 0x844E29, UnpackedSize = 0x14436, Size = 0x35A3, IsPacked = true },
                    new PackedEntry { Name = "kidan_c.chp", Type = "image", Offset = 0x8483CC, UnpackedSize = 0x1C436, Size = 0x4FA3, IsPacked = true },
                    new PackedEntry { Name = "kin2_a.chp", Type = "image", Offset = 0x84D36F, UnpackedSize = 0x4436, Size = 0x10C, IsPacked = true },
                    new PackedEntry { Name = "obj.chp", Type = "image", Offset = 0x84D47B, UnpackedSize = 0x4436, Size = 0x3AF, IsPacked = true },
                    new PackedEntry { Name = "start.chp", Type = "image", Offset = 0x84D82A, UnpackedSize = 0x0C436, Size = 0x21D1, IsPacked = true },
                    new PackedEntry { Name = "s_cain.chp", Type = "image", Offset = 0x84F9FB, UnpackedSize = 0x14436, Size = 0x3716, IsPacked = true },
                    new PackedEntry { Name = "winc.chp", Type = "image", Offset = 0x853111, UnpackedSize = 0x4436, Size = 0x867, IsPacked = true },
                    new PackedEntry { Name = "winc2.chp", Type = "image", Offset = 0x853978, UnpackedSize = 0x4436, Size = 0x85E, IsPacked = true },
                    new PackedEntry { Name = "winct.chp", Type = "image", Offset = 0x8541D6, UnpackedSize = 0x4436, Size = 0x85C, IsPacked = true },
                    new PackedEntry { Name = "x_sys.chp", Type = "image", Offset = 0x854A32, UnpackedSize = 0x4436, Size = 0x115, IsPacked = true },
                }
            },
            {
                "ALLGRP.273", new List<Entry>() {
                    new PackedEntry { Name = "Bgm.gem", Type = "image", Offset = 0x0, UnpackedSize = 0x3EC36, Size = 0x19A9F, IsPacked = true },
                    new PackedEntry { Name = "Black.gps", Type = "image", Offset = 0x19A9F, UnpackedSize = 0x3EC36, Size = 0x0BEF, IsPacked = true },
                    new PackedEntry { Name = "Cgm_anni.gem", Type = "image", Offset = 0x1A68E, UnpackedSize = 0x3EC36, Size = 0x25D4, IsPacked = true },
                    new PackedEntry { Name = "Cgm_bg.gps", Type = "image", Offset = 0x1CC62, UnpackedSize = 0x3EC36, Size = 0x0E53E, IsPacked = true },
                    new PackedEntry { Name = "Cgm_crim.gem", Type = "image", Offset = 0x2B1A0, UnpackedSize = 0x3EC36, Size = 0x24D5, IsPacked = true },
                    new PackedEntry { Name = "Cgm_gana.gem", Type = "image", Offset = 0x2D675, UnpackedSize = 0x3EC36, Size = 0x2859, IsPacked = true },
                    new PackedEntry { Name = "Cgm_king.gem", Type = "image", Offset = 0x2FECE, UnpackedSize = 0x3EC36, Size = 0x17D1, IsPacked = true },
                    new PackedEntry { Name = "Cgm_mafi.gem", Type = "image", Offset = 0x3169F, UnpackedSize = 0x3EC36, Size = 0x2CEA, IsPacked = true },
                    new PackedEntry { Name = "Cgm_milf.gem", Type = "image", Offset = 0x34389, UnpackedSize = 0x3EC36, Size = 0x3054, IsPacked = true },
                    new PackedEntry { Name = "Cgm_pann.gem", Type = "image", Offset = 0x373DD, UnpackedSize = 0x3EC36, Size = 0x286E, IsPacked = true },
                    new PackedEntry { Name = "Cgm_quee.gem", Type = "image", Offset = 0x39C4B, UnpackedSize = 0x3EC36, Size = 0x2FDC, IsPacked = true },
                    new PackedEntry { Name = "Cgm_riqu.gem", Type = "image", Offset = 0x3CC27, UnpackedSize = 0x3EC36, Size = 0x2974, IsPacked = true },
                    new PackedEntry { Name = "Cgm_ruse.gem", Type = "image", Offset = 0x3F59B, UnpackedSize = 0x3EC36, Size = 0x2AE2, IsPacked = true },
                    new PackedEntry { Name = "Cgm_seli.gem", Type = "image", Offset = 0x4207D, UnpackedSize = 0x3EC36, Size = 0x2A67, IsPacked = true },
                    new PackedEntry { Name = "Cgm_shak.gem", Type = "image", Offset = 0x44AE4, UnpackedSize = 0x3EC36, Size = 0x2947, IsPacked = true },
                    new PackedEntry { Name = "Cgm_shok.gem", Type = "image", Offset = 0x4742B, UnpackedSize = 0x3EC36, Size = 0x2943, IsPacked = true },
                    new PackedEntry { Name = "Cgm_shuk.gem", Type = "image", Offset = 0x49D6E, UnpackedSize = 0x3EC36, Size = 0x254A, IsPacked = true },
                    new PackedEntry { Name = "Cgm_tira.gem", Type = "image", Offset = 0x4C2B8, UnpackedSize = 0x3EC36, Size = 0x2873, IsPacked = true },
                    new PackedEntry { Name = "Cgm_tolt.gem", Type = "image", Offset = 0x4EB2B, UnpackedSize = 0x3EC36, Size = 0x26B6, IsPacked = true },
                    new PackedEntry { Name = "Clr_bg.gps", Type = "image", Offset = 0x511E1, UnpackedSize = 0x3EC36, Size = 0x0CD03, IsPacked = true },
                    new PackedEntry { Name = "Config.gps", Type = "image", Offset = 0x5DEE4, UnpackedSize = 0x3EC36, Size = 0x3D93, IsPacked = true },
                    new PackedEntry { Name = "Equ.gem", Type = "image", Offset = 0x61C77, UnpackedSize = 0x3EC36, Size = 0x3214, IsPacked = true },
                    new PackedEntry { Name = "Fol.gps", Type = "image", Offset = 0x64E8B, UnpackedSize = 0x3EC36, Size = 0x870E, IsPacked = true },
                    new PackedEntry { Name = "Fon.gps", Type = "image", Offset = 0x6D599, UnpackedSize = 0x3EC36, Size = 0x6445, IsPacked = true },
                    new PackedEntry { Name = "Giga_ff.gem", Type = "image", Offset = 0x739DE, UnpackedSize = 0x3EC36, Size = 0x1447, IsPacked = true },
                    new PackedEntry { Name = "G_over.gps", Type = "image", Offset = 0x74E25, UnpackedSize = 0x3EC36, Size = 0x0F997, IsPacked = true },
                    new PackedEntry { Name = "Hb02_5b.gps", Type = "image", Offset = 0x847BC, UnpackedSize = 0x3EC36, Size = 0x167E8, IsPacked = true },
                    new PackedEntry { Name = "Hb03_01.gem", Type = "image", Offset = 0x9AFA4, UnpackedSize = 0x3EC36, Size = 0x179F0, IsPacked = true },
                    new PackedEntry { Name = "Hb03_02a.gps", Type = "image", Offset = 0x0B2994, UnpackedSize = 0x27036, Size = 0x9FBE, IsPacked = true },
                    new PackedEntry { Name = "Hb03_02b.gem", Type = "image", Offset = 0x0BC952, UnpackedSize = 0x28436, Size = 0x3536, IsPacked = true },
                    new PackedEntry { Name = "Hb03_03.gem", Type = "image", Offset = 0x0BFE88, UnpackedSize = 0x3EC36, Size = 0x2E6A8, IsPacked = true },
                    new PackedEntry { Name = "Hb03_04.gem", Type = "image", Offset = 0x0EE530, UnpackedSize = 0x3EC36, Size = 0x35499, IsPacked = true },
                    new PackedEntry { Name = "Hb03_05.gem", Type = "image", Offset = 0x1239C9, UnpackedSize = 0x3EC36, Size = 0x225E8, IsPacked = true },
                    new PackedEntry { Name = "Hb03_06.gps", Type = "image", Offset = 0x145FB1, UnpackedSize = 0x3EC36, Size = 0x28BBF, IsPacked = true },
                    new PackedEntry { Name = "Hb03_07.gps", Type = "image", Offset = 0x16EB70, UnpackedSize = 0x3EC36, Size = 0x264BB, IsPacked = true },
                    new PackedEntry { Name = "Hb03_08a.gps", Type = "image", Offset = 0x19502B, UnpackedSize = 0x3EC36, Size = 0x194EE, IsPacked = true },
                    new PackedEntry { Name = "Hb03_08b.gem", Type = "image", Offset = 0x1AE519, UnpackedSize = 0x3EC36, Size = 0x84B3, IsPacked = true },
                    new PackedEntry { Name = "Hb03_08c.gem", Type = "image", Offset = 0x1B69CC, UnpackedSize = 0x3EC36, Size = 0x243F, IsPacked = true },
                    new PackedEntry { Name = "Hb03_08d.gem", Type = "image", Offset = 0x1B8E0B, UnpackedSize = 0x3EC36, Size = 0x1B7B, IsPacked = true },
                    new PackedEntry { Name = "Hb03_09.gps", Type = "image", Offset = 0x1BA986, UnpackedSize = 0x3EC36, Size = 0x16640, IsPacked = true },
                    new PackedEntry { Name = "Hb03_10.gps", Type = "image", Offset = 0x1D0FC6, UnpackedSize = 0x3EC36, Size = 0x223F0, IsPacked = true },
                    new PackedEntry { Name = "Hb03_e01.gps", Type = "image", Offset = 0x1F33B6, UnpackedSize = 0x3EC36, Size = 0x28491, IsPacked = true },
                    new PackedEntry { Name = "Hb03_e02.gps", Type = "image", Offset = 0x21B847, UnpackedSize = 0x3EC36, Size = 0x2533D, IsPacked = true },
                    new PackedEntry { Name = "Hb06_01.gem", Type = "image", Offset = 0x240B84, UnpackedSize = 0x3EC36, Size = 0x14617, IsPacked = true },
                    new PackedEntry { Name = "Hb06_02.gem", Type = "image", Offset = 0x25519B, UnpackedSize = 0x3EC36, Size = 0x1474A, IsPacked = true },
                    new PackedEntry { Name = "Hb06_03.gps", Type = "image", Offset = 0x2698E5, UnpackedSize = 0x28436, Size = 0x19A56, IsPacked = true },
                    new PackedEntry { Name = "Hb06_04a.gps", Type = "image", Offset = 0x28333B, UnpackedSize = 0x3EC36, Size = 0x18B90, IsPacked = true },
                    new PackedEntry { Name = "Hb06_04b.gem", Type = "image", Offset = 0x29BECB, UnpackedSize = 0x3EC36, Size = 0x16A7E, IsPacked = true },
                    new PackedEntry { Name = "Hb06_05.gps", Type = "image", Offset = 0x2B2949, UnpackedSize = 0x3EC36, Size = 0x2506D, IsPacked = true },
                    new PackedEntry { Name = "Hb06_06.gps", Type = "image", Offset = 0x2D79B6, UnpackedSize = 0x3EC36, Size = 0x14756, IsPacked = true },
                    new PackedEntry { Name = "Hb06_07.gps", Type = "image", Offset = 0x2EC10C, UnpackedSize = 0x3EC36, Size = 0x2168A, IsPacked = true },
                    new PackedEntry { Name = "Hb06_08a.gps", Type = "image", Offset = 0x30D796, UnpackedSize = 0x3EC36, Size = 0x1CE0D, IsPacked = true },
                    new PackedEntry { Name = "Hb06_08b.gem", Type = "image", Offset = 0x32A5A3, UnpackedSize = 0x3EC36, Size = 0x25AF, IsPacked = true },
                    new PackedEntry { Name = "Hb06_e01.gps", Type = "image", Offset = 0x32CB52, UnpackedSize = 0x3EC36, Size = 0x1CCCC, IsPacked = true },
                    new PackedEntry { Name = "Hb06_e02.gps", Type = "image", Offset = 0x34981E, UnpackedSize = 0x3EC36, Size = 0x210A1, IsPacked = true },
                    new PackedEntry { Name = "Hb07_01.gps", Type = "image", Offset = 0x36A8BF, UnpackedSize = 0x28436, Size = 0x123D6, IsPacked = true },
                    new PackedEntry { Name = "Hb07_02.gps", Type = "image", Offset = 0x37CC95, UnpackedSize = 0x28436, Size = 0x135C7, IsPacked = true },
                    new PackedEntry { Name = "Hb07_03.gps", Type = "image", Offset = 0x39025C, UnpackedSize = 0x28436, Size = 0x1A933, IsPacked = true },
                    new PackedEntry { Name = "Hb07_04.gps", Type = "image", Offset = 0x3AAB8F, UnpackedSize = 0x28436, Size = 0x1C620, IsPacked = true },
                    new PackedEntry { Name = "Hb07_05.gps", Type = "image", Offset = 0x3C71AF, UnpackedSize = 0x28436, Size = 0x14D8C, IsPacked = true },
                    new PackedEntry { Name = "Hb07_06a.gps", Type = "image", Offset = 0x3DBF3B, UnpackedSize = 0x3EC36, Size = 0x19715, IsPacked = true },
                    new PackedEntry { Name = "Hb07_06b.gem", Type = "image", Offset = 0x3F5650, UnpackedSize = 0x3EC36, Size = 0x0A56E, IsPacked = true },
                    new PackedEntry { Name = "Hb07_07.gps", Type = "image", Offset = 0x3FFBBE, UnpackedSize = 0x3EC36, Size = 0x16B47, IsPacked = true },
                    new PackedEntry { Name = "Hb07_08.gps", Type = "image", Offset = 0x416705, UnpackedSize = 0x3EC36, Size = 0x17E53, IsPacked = true },
                    new PackedEntry { Name = "Hb07_08a.gps", Type = "image", Offset = 0x42E558, UnpackedSize = 0x3EC36, Size = 0x17894, IsPacked = true },
                    new PackedEntry { Name = "Hb07_08b.gem", Type = "image", Offset = 0x445DEC, UnpackedSize = 0x3EC36, Size = 0x2581, IsPacked = true },
                    new PackedEntry { Name = "Hb07_e01.gps", Type = "image", Offset = 0x44836D, UnpackedSize = 0x3EC36, Size = 0x27388, IsPacked = true },
                    new PackedEntry { Name = "Hb07_e02.gps", Type = "image", Offset = 0x46F6F5, UnpackedSize = 0x3EC36, Size = 0x216FD, IsPacked = true },
                    new PackedEntry { Name = "Hb11_01.gps", Type = "image", Offset = 0x490DF2, UnpackedSize = 0x28436, Size = 0x189BD, IsPacked = true },
                    new PackedEntry { Name = "Hb11_02a.gps", Type = "image", Offset = 0x4A97AF, UnpackedSize = 0x28436, Size = 0x10592, IsPacked = true },
                    new PackedEntry { Name = "Hb11_02b.gem", Type = "image", Offset = 0x4B9D41, UnpackedSize = 0x28436, Size = 0x2356, IsPacked = true },
                    new PackedEntry { Name = "Hb11_03.gem", Type = "image", Offset = 0x4BC097, UnpackedSize = 0x3EC36, Size = 0x1AB65, IsPacked = true },
                    new PackedEntry { Name = "Hb11_04.gps", Type = "image", Offset = 0x4D6BFC, UnpackedSize = 0x28436, Size = 0x1C34B, IsPacked = true },
                    new PackedEntry { Name = "Hb11_05a.gps", Type = "image", Offset = 0x4F2F47, UnpackedSize = 0x3EC36, Size = 0x1B837, IsPacked = true },
                    new PackedEntry { Name = "Hb11_05b.gem", Type = "image", Offset = 0x50E77E, UnpackedSize = 0x3EC36, Size = 0x5940, IsPacked = true },
                    new PackedEntry { Name = "Hb11_05c.gem", Type = "image", Offset = 0x5140BE, UnpackedSize = 0x3EC36, Size = 0x0A85B, IsPacked = true },
                    new PackedEntry { Name = "Hb11_05d.gem", Type = "image", Offset = 0x51E919, UnpackedSize = 0x3EC36, Size = 0x9848, IsPacked = true },
                    new PackedEntry { Name = "Hb11_06.gps", Type = "image", Offset = 0x528161, UnpackedSize = 0x3EC36, Size = 0x1A922, IsPacked = true },
                    new PackedEntry { Name = "Hb11_07.gps", Type = "image", Offset = 0x542A83, UnpackedSize = 0x3EC36, Size = 0x194FE, IsPacked = true },
                    new PackedEntry { Name = "Hb11_08a.gps", Type = "image", Offset = 0x55BF81, UnpackedSize = 0x3EC36, Size = 0x1B41D, IsPacked = true },
                    new PackedEntry { Name = "Hb11_08b.gem", Type = "image", Offset = 0x57739E, UnpackedSize = 0x3EC36, Size = 0x2CC6, IsPacked = true },
                    new PackedEntry { Name = "Hb11_e01.gps", Type = "image", Offset = 0x57A064, UnpackedSize = 0x3EC36, Size = 0x20D2D, IsPacked = true },
                    new PackedEntry { Name = "Hb11_e02.gps", Type = "image", Offset = 0x59AD91, UnpackedSize = 0x3EC36, Size = 0x1A64A, IsPacked = true },
                    new PackedEntry { Name = "Hb12_01.gem", Type = "image", Offset = 0x5B53DB, UnpackedSize = 0x3EC36, Size = 0x16C61, IsPacked = true },
                    new PackedEntry { Name = "Hb12_02.gps", Type = "image", Offset = 0x5CC03C, UnpackedSize = 0x28436, Size = 0x1A315, IsPacked = true },
                    new PackedEntry { Name = "Hb12_03.gps", Type = "image", Offset = 0x5E6351, UnpackedSize = 0x28436, Size = 0x14ADB, IsPacked = true },
                    new PackedEntry { Name = "Hb12_04.gps", Type = "image", Offset = 0x5FAE2C, UnpackedSize = 0x28436, Size = 0x0FC3C, IsPacked = true },
                    new PackedEntry { Name = "Hb12_05a.gps", Type = "image", Offset = 0x60AA68, UnpackedSize = 0x3EC36, Size = 0x17FE8, IsPacked = true },
                    new PackedEntry { Name = "Hb12_05b.gem", Type = "image", Offset = 0x622A50, UnpackedSize = 0x3EC36, Size = 0x6BE9, IsPacked = true },
                    new PackedEntry { Name = "Hb12_05c.gem", Type = "image", Offset = 0x629639, UnpackedSize = 0x3EC36, Size = 0x4C40, IsPacked = true },
                    new PackedEntry { Name = "Hb12_05d.gem", Type = "image", Offset = 0x62E279, UnpackedSize = 0x3EC36, Size = 0x4022, IsPacked = true },
                    new PackedEntry { Name = "Hb12_06.gps", Type = "image", Offset = 0x63229B, UnpackedSize = 0x3EC36, Size = 0x17E19, IsPacked = true },
                    new PackedEntry { Name = "Hb12_07.gps", Type = "image", Offset = 0x64A0B4, UnpackedSize = 0x3EC36, Size = 0x22683, IsPacked = true },
                    new PackedEntry { Name = "Hb12_08a.gps", Type = "image", Offset = 0x66C737, UnpackedSize = 0x3EC36, Size = 0x16963, IsPacked = true },
                    new PackedEntry { Name = "Hb12_08b.gem", Type = "image", Offset = 0x68309A, UnpackedSize = 0x3EC36, Size = 0x2DDE, IsPacked = true },
                    new PackedEntry { Name = "Hb12_e01.gps", Type = "image", Offset = 0x685E78, UnpackedSize = 0x3EC36, Size = 0x22D4F, IsPacked = true },
                    new PackedEntry { Name = "Hb12_e02.gps", Type = "image", Offset = 0x6A8BC7, UnpackedSize = 0x3EC36, Size = 0x1C1C9, IsPacked = true },
                    new PackedEntry { Name = "Hb13_01.gem", Type = "image", Offset = 0x6C4D90, UnpackedSize = 0x3EC36, Size = 0x195E7, IsPacked = true },
                    new PackedEntry { Name = "Hb13_02.gps", Type = "image", Offset = 0x6DE377, UnpackedSize = 0x28436, Size = 0x15C11, IsPacked = true },
                    new PackedEntry { Name = "Hb13_03.gps", Type = "image", Offset = 0x6F3F88, UnpackedSize = 0x28436, Size = 0x18F28, IsPacked = true },
                    new PackedEntry { Name = "Hb13_04.gem", Type = "image", Offset = 0x70CEB0, UnpackedSize = 0x3EC36, Size = 0x17D9D, IsPacked = true },
                    new PackedEntry { Name = "Hb13_05.gps", Type = "image", Offset = 0x724C4D, UnpackedSize = 0x28436, Size = 0x17931, IsPacked = true },
                    new PackedEntry { Name = "Hb13_06a.gps", Type = "image", Offset = 0x73C57E, UnpackedSize = 0x3EC36, Size = 0x20A47, IsPacked = true },
                    new PackedEntry { Name = "Hb13_06b.gem", Type = "image", Offset = 0x75CFC5, UnpackedSize = 0x3EC36, Size = 0x0B1C1, IsPacked = true },
                    new PackedEntry { Name = "Hb13_06c.gem", Type = "image", Offset = 0x768186, UnpackedSize = 0x3EC36, Size = 0x0B256, IsPacked = true },
                    new PackedEntry { Name = "Hb13_06d.gem", Type = "image", Offset = 0x7733DC, UnpackedSize = 0x3EC36, Size = 0x1E8A, IsPacked = true },
                    new PackedEntry { Name = "Hb13_07.gps", Type = "image", Offset = 0x775266, UnpackedSize = 0x3EC36, Size = 0x178B6, IsPacked = true },
                    new PackedEntry { Name = "Hb13_08a.gps", Type = "image", Offset = 0x78CB1C, UnpackedSize = 0x3EC36, Size = 0x1276D, IsPacked = true },
                    new PackedEntry { Name = "Hb13_08b.gem", Type = "image", Offset = 0x79F289, UnpackedSize = 0x3EC36, Size = 0x37AA, IsPacked = true },
                    new PackedEntry { Name = "Hb13_e01.gps", Type = "image", Offset = 0x7A2A33, UnpackedSize = 0x3EC36, Size = 0x22397, IsPacked = true },
                    new PackedEntry { Name = "Hb13_e02.gps", Type = "image", Offset = 0x7C4DCA, UnpackedSize = 0x3EC36, Size = 0x1F07F, IsPacked = true },
                    new PackedEntry { Name = "Hb14_01.gps", Type = "image", Offset = 0x7E3E49, UnpackedSize = 0x28436, Size = 0x1B66E, IsPacked = true },
                    new PackedEntry { Name = "Hb14_02a.gps", Type = "image", Offset = 0x7FF4B7, UnpackedSize = 0x28436, Size = 0x18825, IsPacked = true },
                    new PackedEntry { Name = "Hb14_02b.gem", Type = "image", Offset = 0x817CDC, UnpackedSize = 0x28436, Size = 0x3028, IsPacked = true },
                    new PackedEntry { Name = "Hb14_03.gps", Type = "image", Offset = 0x81AD04, UnpackedSize = 0x28436, Size = 0x0F841, IsPacked = true },
                    new PackedEntry { Name = "Hb14_04a.gps", Type = "image", Offset = 0x82A545, UnpackedSize = 0x28436, Size = 0x12E49, IsPacked = true },
                    new PackedEntry { Name = "Hb14_04b.gem", Type = "image", Offset = 0x83D38E, UnpackedSize = 0x28436, Size = 0x2C51, IsPacked = true },
                    new PackedEntry { Name = "Hb14_05.gps", Type = "image", Offset = 0x83FFDF, UnpackedSize = 0x28436, Size = 0x19464, IsPacked = true },
                    new PackedEntry { Name = "Hb14_06a.gps", Type = "image", Offset = 0x859443, UnpackedSize = 0x3EC36, Size = 0x159D3, IsPacked = true },
                    new PackedEntry { Name = "Hb14_06b.gem", Type = "image", Offset = 0x86EE16, UnpackedSize = 0x3EC36, Size = 0x3C0F, IsPacked = true },
                    new PackedEntry { Name = "Hb14_06c.gem", Type = "image", Offset = 0x872A25, UnpackedSize = 0x3EC36, Size = 0x7DF5, IsPacked = true },
                    new PackedEntry { Name = "Hb14_06d.gem", Type = "image", Offset = 0x87A81A, UnpackedSize = 0x3EC36, Size = 0x1B8A, IsPacked = true },
                    new PackedEntry { Name = "Hb14_07.gps", Type = "image", Offset = 0x87C3A4, UnpackedSize = 0x3EC36, Size = 0x168D0, IsPacked = true },
                    new PackedEntry { Name = "Hb14_08a.gps", Type = "image", Offset = 0x892C74, UnpackedSize = 0x3EC36, Size = 0x1A952, IsPacked = true },
                    new PackedEntry { Name = "Hb14_08b.gem", Type = "image", Offset = 0x8AD5C6, UnpackedSize = 0x3EC36, Size = 0x2362, IsPacked = true },
                    new PackedEntry { Name = "Hb14_e01.gps", Type = "image", Offset = 0x8AF928, UnpackedSize = 0x3EC36, Size = 0x1EF08, IsPacked = true },
                    new PackedEntry { Name = "Hb14_e02.gps", Type = "image", Offset = 0x8CE830, UnpackedSize = 0x3EC36, Size = 0x230F5, IsPacked = true },
                    new PackedEntry { Name = "Hb15_01.gps", Type = "image", Offset = 0x8F1925, UnpackedSize = 0x28436, Size = 0x15781, IsPacked = true },
                    new PackedEntry { Name = "Hb15_02.gps", Type = "image", Offset = 0x9070A6, UnpackedSize = 0x28436, Size = 0x19206, IsPacked = true },
                    new PackedEntry { Name = "Hb15_03.gps", Type = "image", Offset = 0x9202AC, UnpackedSize = 0x28836, Size = 0x188E5, IsPacked = true },
                    new PackedEntry { Name = "Hb15_04.gps", Type = "image", Offset = 0x938B91, UnpackedSize = 0x28436, Size = 0x1630C, IsPacked = true },
                    new PackedEntry { Name = "Hb15_05.gem", Type = "image", Offset = 0x94EE9D, UnpackedSize = 0x3EC36, Size = 0x1144F, IsPacked = true },
                    new PackedEntry { Name = "Hb15_06a.gps", Type = "image", Offset = 0x9602EC, UnpackedSize = 0x3EC36, Size = 0x19313, IsPacked = true },
                    new PackedEntry { Name = "Hb15_06b.gem", Type = "image", Offset = 0x9795FF, UnpackedSize = 0x3EC36, Size = 0x0C102, IsPacked = true },
                    new PackedEntry { Name = "Hb15_06c.gem", Type = "image", Offset = 0x985701, UnpackedSize = 0x3EC36, Size = 0x0AF19, IsPacked = true },
                    new PackedEntry { Name = "Hb15_06d.gem", Type = "image", Offset = 0x99061A, UnpackedSize = 0x3EC36, Size = 0x1FD5, IsPacked = true },
                    new PackedEntry { Name = "Hb15_07.gps", Type = "image", Offset = 0x9925EF, UnpackedSize = 0x3EC36, Size = 0x1BE25, IsPacked = true },
                    new PackedEntry { Name = "Hb15_08a.gps", Type = "image", Offset = 0x9AE414, UnpackedSize = 0x3EC36, Size = 0x1B2C3, IsPacked = true },
                    new PackedEntry { Name = "Hb15_08b.gem", Type = "image", Offset = 0x9C96D7, UnpackedSize = 0x3EC36, Size = 0x412C, IsPacked = true },
                    new PackedEntry { Name = "Hb15_e01.gps", Type = "image", Offset = 0x9CD803, UnpackedSize = 0x3EC36, Size = 0x1E116, IsPacked = true },
                    new PackedEntry { Name = "Hb15_e02.gps", Type = "image", Offset = 0x9EB919, UnpackedSize = 0x3EC36, Size = 0x1A31C, IsPacked = true },
                    new PackedEntry { Name = "Hb16_01.gps", Type = "image", Offset = 0x0A05C35, UnpackedSize = 0x28436, Size = 0x12066, IsPacked = true },
                    new PackedEntry { Name = "Hb16_02.gps", Type = "image", Offset = 0x0A17C9B, UnpackedSize = 0x28436, Size = 0x16EE0, IsPacked = true },
                    new PackedEntry { Name = "Hb16_03.gem", Type = "image", Offset = 0x0A2EB7B, UnpackedSize = 0x3EC36, Size = 0x12E21, IsPacked = true },
                    new PackedEntry { Name = "Hb16_04.gps", Type = "image", Offset = 0x0A4199C, UnpackedSize = 0x28436, Size = 0x1554E, IsPacked = true },
                    new PackedEntry { Name = "Hb16_05a.gps", Type = "image", Offset = 0x0A56EEA, UnpackedSize = 0x3EC36, Size = 0x14F0D, IsPacked = true },
                    new PackedEntry { Name = "Hb16_05b.gem", Type = "image", Offset = 0x0A6BDF7, UnpackedSize = 0x3EC36, Size = 0x113A8, IsPacked = true },
                    new PackedEntry { Name = "Hb16_05c.gem", Type = "image", Offset = 0x0A7D19F, UnpackedSize = 0x3EC36, Size = 0x6925, IsPacked = true },
                    new PackedEntry { Name = "Hb16_05d.gem", Type = "image", Offset = 0x0A83AC4, UnpackedSize = 0x3EC36, Size = 0x2853, IsPacked = true },
                    new PackedEntry { Name = "Hb16_06.gps", Type = "image", Offset = 0x0A86317, UnpackedSize = 0x3EC36, Size = 0x16A4A, IsPacked = true },
                    new PackedEntry { Name = "Hb16_07.gps", Type = "image", Offset = 0x0A9CD61, UnpackedSize = 0x3EC36, Size = 0x1456F, IsPacked = true },
                    new PackedEntry { Name = "Hb16_08a.gps", Type = "image", Offset = 0x0AB12D0, UnpackedSize = 0x3EC36, Size = 0x17724, IsPacked = true },
                    new PackedEntry { Name = "Hb16_08b.gem", Type = "image", Offset = 0x0AC89F4, UnpackedSize = 0x3EC36, Size = 0x2444, IsPacked = true },
                    new PackedEntry { Name = "Hb16_e01.gps", Type = "image", Offset = 0x0ACAE38, UnpackedSize = 0x3EC36, Size = 0x2A0A6, IsPacked = true },
                    new PackedEntry { Name = "Hb16_e02.gps", Type = "image", Offset = 0x0AF4EDE, UnpackedSize = 0x3EC36, Size = 0x238DD, IsPacked = true },
                    new PackedEntry { Name = "Hb17_01.gps", Type = "image", Offset = 0x0B187BB, UnpackedSize = 0x28436, Size = 0x1801C, IsPacked = true },
                    new PackedEntry { Name = "Hb17_02.gps", Type = "image", Offset = 0x0B307D7, UnpackedSize = 0x28436, Size = 0x13D41, IsPacked = true },
                    new PackedEntry { Name = "Hb17_03a.gps", Type = "image", Offset = 0x0B44518, UnpackedSize = 0x3EC36, Size = 0x13B51, IsPacked = true },
                    new PackedEntry { Name = "Hb17_03b.gem", Type = "image", Offset = 0x0B58069, UnpackedSize = 0x3EC36, Size = 0x0D070, IsPacked = true },
                    new PackedEntry { Name = "Hb17_03c.gem", Type = "image", Offset = 0x0B650D9, UnpackedSize = 0x3EC36, Size = 0x7809, IsPacked = true },
                    new PackedEntry { Name = "Hb17_04a.gps", Type = "image", Offset = 0x0B6C8E2, UnpackedSize = 0x3EC36, Size = 0x1AD4F, IsPacked = true },
                    new PackedEntry { Name = "Hb17_04b.gem", Type = "image", Offset = 0x0B87631, UnpackedSize = 0x3EC36, Size = 0x2359, IsPacked = true },
                    new PackedEntry { Name = "Hb17_05.gps", Type = "image", Offset = 0x0B8998A, UnpackedSize = 0x3EC36, Size = 0x1D87E, IsPacked = true },
                    new PackedEntry { Name = "Hb17_e01.gps", Type = "image", Offset = 0x0BA7208, UnpackedSize = 0x3EC36, Size = 0x2290B, IsPacked = true },
                    new PackedEntry { Name = "Hb17_e02.gps", Type = "image", Offset = 0x0BC9B13, UnpackedSize = 0x3EC36, Size = 0x1F73F, IsPacked = true },
                    new PackedEntry { Name = "Hb18_01.gem", Type = "image", Offset = 0x0BE9252, UnpackedSize = 0x3EC36, Size = 0x1895B, IsPacked = true },
                    new PackedEntry { Name = "Hb18_02a.gps", Type = "image", Offset = 0x0C01BAD, UnpackedSize = 0x3EC36, Size = 0x1D715, IsPacked = true },
                    new PackedEntry { Name = "Hb18_02b.gem", Type = "image", Offset = 0x0C1F2C2, UnpackedSize = 0x3EC36, Size = 0x1DE3, IsPacked = true },
                    new PackedEntry { Name = "Hb18_03.gps", Type = "image", Offset = 0x0C210A5, UnpackedSize = 0x3EC36, Size = 0x15C41, IsPacked = true },
                    new PackedEntry { Name = "Hb18_04a.gps", Type = "image", Offset = 0x0C36CE6, UnpackedSize = 0x3EC36, Size = 0x16CE3, IsPacked = true },
                    new PackedEntry { Name = "Hb18_04b.gem", Type = "image", Offset = 0x0C4D9C9, UnpackedSize = 0x3EC36, Size = 0x29B5, IsPacked = true },
                    new PackedEntry { Name = "Hb18_05.gps", Type = "image", Offset = 0x0C5037E, UnpackedSize = 0x28436, Size = 0x19472, IsPacked = true },
                    new PackedEntry { Name = "Hb18_06.gps", Type = "image", Offset = 0x0C697F0, UnpackedSize = 0x28436, Size = 0x17046, IsPacked = true },
                    new PackedEntry { Name = "Hb18_e01.gps", Type = "image", Offset = 0x0C80836, UnpackedSize = 0x3EC36, Size = 0x1D572, IsPacked = true },
                    new PackedEntry { Name = "Hb18_e02.gps", Type = "image", Offset = 0x0C9DDA8, UnpackedSize = 0x3EC36, Size = 0x1B241, IsPacked = true },
                    new PackedEntry { Name = "Hb19_01.gem", Type = "image", Offset = 0x0CB8FE9, UnpackedSize = 0x3EC36, Size = 0x1A34D, IsPacked = true },
                    new PackedEntry { Name = "Hb19_02.gps", Type = "image", Offset = 0x0CD3336, UnpackedSize = 0x3EC36, Size = 0x18208, IsPacked = true },
                    new PackedEntry { Name = "Hb19_03.gps", Type = "image", Offset = 0x0CEB53E, UnpackedSize = 0x28436, Size = 0x1855A, IsPacked = true },
                    new PackedEntry { Name = "Hb19_04.gps", Type = "image", Offset = 0x0D03A98, UnpackedSize = 0x28436, Size = 0x1411A, IsPacked = true },
                    new PackedEntry { Name = "Hb19_05a.gps", Type = "image", Offset = 0x0D17BB2, UnpackedSize = 0x3EC36, Size = 0x22611, IsPacked = true },
                    new PackedEntry { Name = "Hb19_05b.gem", Type = "image", Offset = 0x0D3A1C3, UnpackedSize = 0x3EC36, Size = 0x98D0, IsPacked = true },
                    new PackedEntry { Name = "Hb19_05c.gem", Type = "image", Offset = 0x0D43A93, UnpackedSize = 0x3EC36, Size = 0x7B0E, IsPacked = true },
                    new PackedEntry { Name = "Hb19_05d.gem", Type = "image", Offset = 0x0D4B5A1, UnpackedSize = 0x3EC36, Size = 0x2603, IsPacked = true },
                    new PackedEntry { Name = "Hb19_06.gps", Type = "image", Offset = 0x0D4DBA4, UnpackedSize = 0x3EC36, Size = 0x1A2AB, IsPacked = true },
                    new PackedEntry { Name = "Hb19_07.gps", Type = "image", Offset = 0x0D67E4F, UnpackedSize = 0x3EC36, Size = 0x190E7, IsPacked = true },
                    new PackedEntry { Name = "Hb19_08a.gps", Type = "image", Offset = 0x0D80F36, UnpackedSize = 0x3EC36, Size = 0x175F8, IsPacked = true },
                    new PackedEntry { Name = "Hb19_08b.gem", Type = "image", Offset = 0x0D9852E, UnpackedSize = 0x3EC36, Size = 0x33B4, IsPacked = true },
                    new PackedEntry { Name = "Hb19_e01.gps", Type = "image", Offset = 0x0D9B8E2, UnpackedSize = 0x3EC36, Size = 0x262C3, IsPacked = true },
                    new PackedEntry { Name = "Hb19_e02.gps", Type = "image", Offset = 0x0DC1BA5, UnpackedSize = 0x3EC36, Size = 0x20CA0, IsPacked = true },
                    new PackedEntry { Name = "Hb19_uta.gps", Type = "image", Offset = 0x0DE2845, UnpackedSize = 0x28836, Size = 0x4339, IsPacked = true },
                    new PackedEntry { Name = "Hb20_01.gps", Type = "image", Offset = 0x0DE6B7E, UnpackedSize = 0x28436, Size = 0x15778, IsPacked = true },
                    new PackedEntry { Name = "Hb20_02.gps", Type = "image", Offset = 0x0DFC2F6, UnpackedSize = 0x28436, Size = 0x147D2, IsPacked = true },
                    new PackedEntry { Name = "Hb20_03.gps", Type = "image", Offset = 0x0E10AC8, UnpackedSize = 0x28436, Size = 0x161D5, IsPacked = true },
                    new PackedEntry { Name = "Hb20_04a.gps", Type = "image", Offset = 0x0E26C9D, UnpackedSize = 0x3EC36, Size = 0x20333, IsPacked = true },
                    new PackedEntry { Name = "Hb20_04b.gem", Type = "image", Offset = 0x0E46FD0, UnpackedSize = 0x3EC36, Size = 0x9BE9, IsPacked = true },
                    new PackedEntry { Name = "Hb20_04c.gem", Type = "image", Offset = 0x0E50BB9, UnpackedSize = 0x3EC36, Size = 0x2426, IsPacked = true },
                    new PackedEntry { Name = "Hb20_05.gps", Type = "image", Offset = 0x0E52FDF, UnpackedSize = 0x3EC36, Size = 0x271A7, IsPacked = true },
                    new PackedEntry { Name = "Hb20_06.gps", Type = "image", Offset = 0x0E7A186, UnpackedSize = 0x3EC36, Size = 0x190EA, IsPacked = true },
                    new PackedEntry { Name = "Hb20_07a.gps", Type = "image", Offset = 0x0E93270, UnpackedSize = 0x3EC36, Size = 0x16FCC, IsPacked = true },
                    new PackedEntry { Name = "Hb20_07b.gem", Type = "image", Offset = 0x0EAA23C, UnpackedSize = 0x3EC36, Size = 0x2220, IsPacked = true },
                    new PackedEntry { Name = "Hb20_e01.gps", Type = "image", Offset = 0x0EAC45C, UnpackedSize = 0x3EC36, Size = 0x1FAF6, IsPacked = true },
                    new PackedEntry { Name = "Hb20_e02.gps", Type = "image", Offset = 0x0ECBF52, UnpackedSize = 0x3EC36, Size = 0x28199, IsPacked = true },
                    new PackedEntry { Name = "Hb21_01.gps", Type = "image", Offset = 0x0EF40EB, UnpackedSize = 0x28436, Size = 0x11248, IsPacked = true },
                    new PackedEntry { Name = "Hb21_02.gps", Type = "image", Offset = 0x0F05333, UnpackedSize = 0x28436, Size = 0x118B4, IsPacked = true },
                    new PackedEntry { Name = "Hb21_03.gps", Type = "image", Offset = 0x0F16BE7, UnpackedSize = 0x28436, Size = 0x0DE2D, IsPacked = true },
                    new PackedEntry { Name = "Hb21_04.gps", Type = "image", Offset = 0x0F24A14, UnpackedSize = 0x28436, Size = 0x0FFD8, IsPacked = true },
                    new PackedEntry { Name = "Hb21_05.gps", Type = "image", Offset = 0x0F349EC, UnpackedSize = 0x28436, Size = 0x16D32, IsPacked = true },
                    new PackedEntry { Name = "Hb21_06.gps", Type = "image", Offset = 0x0F4B71E, UnpackedSize = 0x28436, Size = 0x1406C, IsPacked = true },
                    new PackedEntry { Name = "Hb21_07.gps", Type = "image", Offset = 0x0F5F78A, UnpackedSize = 0x3EC36, Size = 0x22B2A, IsPacked = true },
                    new PackedEntry { Name = "Hb21_08a.gps", Type = "image", Offset = 0x0F822B4, UnpackedSize = 0x3EC36, Size = 0x191DF, IsPacked = true },
                    new PackedEntry { Name = "Hb21_08b.gem", Type = "image", Offset = 0x0F9B493, UnpackedSize = 0x3EC36, Size = 0x1FCA, IsPacked = true },
                    new PackedEntry { Name = "Hb21_08c.gem", Type = "image", Offset = 0x0F9D45D, UnpackedSize = 0x3EC36, Size = 0x10B2, IsPacked = true },
                    new PackedEntry { Name = "Hb21_08d.gem", Type = "image", Offset = 0x0F9E50F, UnpackedSize = 0x3EC36, Size = 0x1075, IsPacked = true },
                    new PackedEntry { Name = "Hb21_08e.gem", Type = "image", Offset = 0x0F9F584, UnpackedSize = 0x3EC36, Size = 0x1092, IsPacked = true },
                    new PackedEntry { Name = "Hb21_08f.gem", Type = "image", Offset = 0x0FA0616, UnpackedSize = 0x3EC36, Size = 0x1101, IsPacked = true },
                    new PackedEntry { Name = "Hb21_08g.gem", Type = "image", Offset = 0x0FA1717, UnpackedSize = 0x3EC36, Size = 0x1113, IsPacked = true },
                    new PackedEntry { Name = "Hb21_e01.gps", Type = "image", Offset = 0x0FA282A, UnpackedSize = 0x3EC36, Size = 0x2155D, IsPacked = true },
                    new PackedEntry { Name = "Hb21_e02.gps", Type = "image", Offset = 0x0FC3D87, UnpackedSize = 0x3EC36, Size = 0x1E3DF, IsPacked = true },
                    new PackedEntry { Name = "Hb26_01.gem", Type = "image", Offset = 0x0FE2166, UnpackedSize = 0x3EC36, Size = 0x12C17, IsPacked = true },
                    new PackedEntry { Name = "Hb26_02.gps", Type = "image", Offset = 0x0FF4D7D, UnpackedSize = 0x28436, Size = 0x18804, IsPacked = true },
                    new PackedEntry { Name = "Hb26_03.gem", Type = "image", Offset = 0x100D581, UnpackedSize = 0x3EC36, Size = 0x16091, IsPacked = true },
                    new PackedEntry { Name = "Hb26_04a.gps", Type = "image", Offset = 0x1023612, UnpackedSize = 0x3EC36, Size = 0x17ECF, IsPacked = true },
                    new PackedEntry { Name = "Hb26_04b.gem", Type = "image", Offset = 0x103B4E1, UnpackedSize = 0x3EC36, Size = 0x0CACC, IsPacked = true },
                    new PackedEntry { Name = "Hb26_04c.gem", Type = "image", Offset = 0x1047FAD, UnpackedSize = 0x3EC36, Size = 0x9BEB, IsPacked = true },
                    new PackedEntry { Name = "Hb26_05.gps", Type = "image", Offset = 0x1051B98, UnpackedSize = 0x3EC36, Size = 0x1D990, IsPacked = true },
                    new PackedEntry { Name = "Hb26_06.gps", Type = "image", Offset = 0x106F528, UnpackedSize = 0x3EC36, Size = 0x22899, IsPacked = true },
                    new PackedEntry { Name = "Hb26_e01.gps", Type = "image", Offset = 0x1091DC1, UnpackedSize = 0x3EC36, Size = 0x1E7D1, IsPacked = true },
                    new PackedEntry { Name = "Hb26_e02.gps", Type = "image", Offset = 0x10B0592, UnpackedSize = 0x3EC36, Size = 0x24B98, IsPacked = true },
                    new PackedEntry { Name = "Hbev_01.gps", Type = "image", Offset = 0x10D512A, UnpackedSize = 0x28436, Size = 0x16501, IsPacked = true },
                    new PackedEntry { Name = "Hbev_02.gps", Type = "image", Offset = 0x10EB62B, UnpackedSize = 0x28436, Size = 0x1E057, IsPacked = true },
                    new PackedEntry { Name = "Hbev_03.gps", Type = "image", Offset = 0x1109682, UnpackedSize = 0x28436, Size = 0x185EE, IsPacked = true },
                    new PackedEntry { Name = "Hbev_04.gps", Type = "image", Offset = 0x1121C70, UnpackedSize = 0x3EC36, Size = 0x2208A, IsPacked = true },
                    new PackedEntry { Name = "Hbop_02.gem", Type = "image", Offset = 0x1143CFA, UnpackedSize = 0x3EC36, Size = 0x194EB, IsPacked = true },
                    new PackedEntry { Name = "Hbop_03.gem", Type = "image", Offset = 0x115D1E5, UnpackedSize = 0x3EC36, Size = 0x21063, IsPacked = true },
                    new PackedEntry { Name = "Hbop_04.gps", Type = "image", Offset = 0x117E248, UnpackedSize = 0x3EC36, Size = 0x1D54D, IsPacked = true },
                    new PackedEntry { Name = "Hbop_05.gem", Type = "image", Offset = 0x119B795, UnpackedSize = 0x3EC36, Size = 0x32A86, IsPacked = true },
                    new PackedEntry { Name = "Hb_moji.gps", Type = "image", Offset = 0x11CE21B, UnpackedSize = 0x1B96, Size = 0x38F, IsPacked = true },
                    new PackedEntry { Name = "Hb_roll0.gem", Type = "image", Offset = 0x11CE5AA, UnpackedSize = 0x3EC36, Size = 0x1B59, IsPacked = true },
                    new PackedEntry { Name = "Hb_roll1.gem", Type = "image", Offset = 0x11D0103, UnpackedSize = 0x3EC36, Size = 0x2106, IsPacked = true },
                    new PackedEntry { Name = "Hb_roll2.gem", Type = "image", Offset = 0x11D2209, UnpackedSize = 0x3EC36, Size = 0x2287, IsPacked = true },
                    new PackedEntry { Name = "Hb_roll3.gem", Type = "image", Offset = 0x11D4490, UnpackedSize = 0x3EC36, Size = 0x261A, IsPacked = true },
                    new PackedEntry { Name = "Hb_roll4.gem", Type = "image", Offset = 0x11D6AAA, UnpackedSize = 0x3EC36, Size = 0x1594, IsPacked = true },
                    new PackedEntry { Name = "Kana.gem", Type = "image", Offset = 0x11D803E, UnpackedSize = 0x3EC36, Size = 0x1A76, IsPacked = true },
                    new PackedEntry { Name = "K_abel.gem", Type = "image", Offset = 0x11D9AB4, UnpackedSize = 0x19436, Size = 0x0A5CB, IsPacked = true },
                    new PackedEntry { Name = "K_annin.gem", Type = "image", Offset = 0x11E407F, UnpackedSize = 0x17B36, Size = 0x87CA, IsPacked = true },
                    new PackedEntry { Name = "K_bajil.gem", Type = "image", Offset = 0x11EC849, UnpackedSize = 0x187B6, Size = 0x8E93, IsPacked = true },
                    new PackedEntry { Name = "K_bobes.gem", Type = "image", Offset = 0x11F56DC, UnpackedSize = 0x24FB6, Size = 0x0FD0F, IsPacked = true },
                    new PackedEntry { Name = "K_bosco.gem", Type = "image", Offset = 0x12053EB, UnpackedSize = 0x19436, Size = 0x0B246, IsPacked = true },
                    new PackedEntry { Name = "K_clim1.gem", Type = "image", Offset = 0x1210631, UnpackedSize = 0x187B6, Size = 0x8232, IsPacked = true },
                    new PackedEntry { Name = "K_clim2.gem", Type = "image", Offset = 0x1218863, UnpackedSize = 0x349B6, Size = 0x10D9B, IsPacked = true },
                    new PackedEntry { Name = "K_cura.gem", Type = "image", Offset = 0x12295FE, UnpackedSize = 0x19436, Size = 0x0AD4B, IsPacked = true },
                    new PackedEntry { Name = "K_gana1.gem", Type = "image", Offset = 0x1234349, UnpackedSize = 0x17B36, Size = 0x96BE, IsPacked = true },
                    new PackedEntry { Name = "K_gana2.gem", Type = "image", Offset = 0x123DA07, UnpackedSize = 0x17B36, Size = 0x72D0, IsPacked = true },
                    new PackedEntry { Name = "K_king.gem", Type = "image", Offset = 0x1244CD7, UnpackedSize = 0x17B36, Size = 0x8D33, IsPacked = true },
                    new PackedEntry { Name = "K_mafi1.gem", Type = "image", Offset = 0x124DA0A, UnpackedSize = 0x13CB6, Size = 0x60F9, IsPacked = true },
                    new PackedEntry { Name = "K_mafi2.gem", Type = "image", Offset = 0x1253B03, UnpackedSize = 0x17B36, Size = 0x58DB, IsPacked = true },
                    new PackedEntry { Name = "K_mafin1.gem", Type = "image", Offset = 0x12593DE, UnpackedSize = 0x13CB6, Size = 0x60F9, IsPacked = true },
                    new PackedEntry { Name = "K_mafin2.gem", Type = "image", Offset = 0x125F4D7, UnpackedSize = 0x17B36, Size = 0x58DB, IsPacked = true },
                    new PackedEntry { Name = "K_mcsr.gps", Type = "image", Offset = 0x1264DB2, UnpackedSize = 0x5436, Size = 0x3A8, IsPacked = true },
                    new PackedEntry { Name = "K_milf1.gem", Type = "image", Offset = 0x126515A, UnpackedSize = 0x236B6, Size = 0x0C081, IsPacked = true },
                    new PackedEntry { Name = "K_milf2.gem", Type = "image", Offset = 0x12711DB, UnpackedSize = 0x1DF36, Size = 0x8839, IsPacked = true },
                    new PackedEntry { Name = "K_panna.gem", Type = "image", Offset = 0x1279A14, UnpackedSize = 0x1EBB6, Size = 0x0B8F8, IsPacked = true },
                    new PackedEntry { Name = "K_ponti.gem", Type = "image", Offset = 0x128530C, UnpackedSize = 0x268B6, Size = 0x0FD6A, IsPacked = true },
                    new PackedEntry { Name = "K_queen.gem", Type = "image", Offset = 0x1295076, UnpackedSize = 0x16236, Size = 0x0AA57, IsPacked = true },
                    new PackedEntry { Name = "K_riqu1.gem", Type = "image", Offset = 0x129FACD, UnpackedSize = 0x1A0B6, Size = 0x7AB9, IsPacked = true },
                    new PackedEntry { Name = "K_riqu2.gem", Type = "image", Offset = 0x12A7586, UnpackedSize = 0x1B9B6, Size = 0x5818, IsPacked = true },
                    new PackedEntry { Name = "K_riqur1.gem", Type = "image", Offset = 0x12ACD9E, UnpackedSize = 0x1A0B6, Size = 0x88A7, IsPacked = true },
                    new PackedEntry { Name = "K_riqur2.gem", Type = "image", Offset = 0x12B5645, UnpackedSize = 0x1B9B6, Size = 0x5818, IsPacked = true },
                    new PackedEntry { Name = "K_russ1.gem", Type = "image", Offset = 0x12BAE5D, UnpackedSize = 0x1A0B6, Size = 0x9633, IsPacked = true },
                    new PackedEntry { Name = "K_russ2.gem", Type = "image", Offset = 0x12C4490, UnpackedSize = 0x236B6, Size = 0x0B5D3, IsPacked = true },
                    new PackedEntry { Name = "K_sable.gem", Type = "image", Offset = 0x12CFA63, UnpackedSize = 0x19436, Size = 0x95C2, IsPacked = true },
                    new PackedEntry { Name = "K_seli1.gem", Type = "image", Offset = 0x12D9025, UnpackedSize = 0x1DF36, Size = 0x7BE8, IsPacked = true },
                    new PackedEntry { Name = "K_seli2.gem", Type = "image", Offset = 0x12E0C0D, UnpackedSize = 0x19436, Size = 0x7254, IsPacked = true },
                    new PackedEntry { Name = "K_selis1.gem", Type = "image", Offset = 0x12E7E61, UnpackedSize = 0x1DF36, Size = 0x7BE8, IsPacked = true },
                    new PackedEntry { Name = "K_selis2.gem", Type = "image", Offset = 0x12EFA49, UnpackedSize = 0x19436, Size = 0x7254, IsPacked = true },
                    new PackedEntry { Name = "K_shake.gem", Type = "image", Offset = 0x12F6C9D, UnpackedSize = 0x1A0B6, Size = 0x77C4, IsPacked = true },
                    new PackedEntry { Name = "K_syoko.gem", Type = "image", Offset = 0x12FE461, UnpackedSize = 0x12436, Size = 0x5E67, IsPacked = true },
                    new PackedEntry { Name = "K_syuku.gem", Type = "image", Offset = 0x13042C8, UnpackedSize = 0x187B6, Size = 0x76C9, IsPacked = true },
                    new PackedEntry { Name = "K_tira.gem", Type = "image", Offset = 0x130B991, UnpackedSize = 0x24336, Size = 0x928C, IsPacked = true },
                    new PackedEntry { Name = "K_tolte.gem", Type = "image", Offset = 0x1314C1D, UnpackedSize = 0x1B9B6, Size = 0x8670, IsPacked = true },
                    new PackedEntry { Name = "K_tryu.gem", Type = "image", Offset = 0x131D28D, UnpackedSize = 0x16236, Size = 0x8A06, IsPacked = true },
                    new PackedEntry { Name = "Load.gem", Type = "image", Offset = 0x1325C93, UnpackedSize = 0x3EC36, Size = 0x4937, IsPacked = true },
                    new PackedEntry { Name = "Man_bg.gps", Type = "image", Offset = 0x132A5CA, UnpackedSize = 0x3EC36, Size = 0x0D6A4, IsPacked = true },
                    new PackedEntry { Name = "Mapwin.gem", Type = "image", Offset = 0x1337C6E, UnpackedSize = 0x3EC36, Size = 0x2236, IsPacked = true },
                    new PackedEntry { Name = "Mcsr.gps", Type = "image", Offset = 0x1339EA4, UnpackedSize = 0x5436, Size = 0x3B9, IsPacked = true },
                    new PackedEntry { Name = "Oki.gem", Type = "image", Offset = 0x133A25D, UnpackedSize = 0x3EC36, Size = 0x45DB, IsPacked = true },
                    new PackedEntry { Name = "Opt_bg.gps", Type = "image", Offset = 0x133E838, UnpackedSize = 0x3EC36, Size = 0x1E5A, IsPacked = true },
                    new PackedEntry { Name = "Opt_bgm.gem", Type = "image", Offset = 0x1340692, UnpackedSize = 0x3EC36, Size = 0x0EE0, IsPacked = true },
                    new PackedEntry { Name = "Opt_cgm.gem", Type = "image", Offset = 0x1341572, UnpackedSize = 0x3EC36, Size = 0x0EAA, IsPacked = true },
                    new PackedEntry { Name = "Opt_manu.gem", Type = "image", Offset = 0x134241C, UnpackedSize = 0x3EC36, Size = 0x0EB8, IsPacked = true },
                    new PackedEntry { Name = "Op_05x.gem", Type = "image", Offset = 0x13432D4, UnpackedSize = 0x3EC36, Size = 0x10ED3, IsPacked = true },
                    new PackedEntry { Name = "Op_06a.gem", Type = "image", Offset = 0x13541A7, UnpackedSize = 0x3EC36, Size = 0x7843, IsPacked = true },
                    new PackedEntry { Name = "Op_06b.gem", Type = "image", Offset = 0x135B9EA, UnpackedSize = 0x3EC36, Size = 0x7F21, IsPacked = true },
                    new PackedEntry { Name = "Op_06c.gem", Type = "image", Offset = 0x136390B, UnpackedSize = 0x3EC36, Size = 0x8F06, IsPacked = true },
                    new PackedEntry { Name = "Op_06d.gem", Type = "image", Offset = 0x136C811, UnpackedSize = 0x3EC36, Size = 0x0A93A, IsPacked = true },
                    new PackedEntry { Name = "Op_07a.gem", Type = "image", Offset = 0x137714B, UnpackedSize = 0x1D236, Size = 0x3451, IsPacked = true },
                    new PackedEntry { Name = "Op_07b.gem", Type = "image", Offset = 0x137A59C, UnpackedSize = 0x3EC36, Size = 0x3B9E, IsPacked = true },
                    new PackedEntry { Name = "Op_07m.gem", Type = "image", Offset = 0x137E13A, UnpackedSize = 0x1D236, Size = 0x0AC9, IsPacked = true },
                    new PackedEntry { Name = "Op_08.gem", Type = "image", Offset = 0x137EC03, UnpackedSize = 0x3EC36, Size = 0x6AE6, IsPacked = true },
                    new PackedEntry { Name = "Op_18x.gem", Type = "image", Offset = 0x13856E9, UnpackedSize = 0x3EC36, Size = 0x317F, IsPacked = true },
                    new PackedEntry { Name = "Op_26x.gem", Type = "image", Offset = 0x1388868, UnpackedSize = 0x3EC36, Size = 0x4FCF, IsPacked = true },
                    new PackedEntry { Name = "Op_29x.gem", Type = "image", Offset = 0x138D837, UnpackedSize = 0x3EC36, Size = 0x0C8FE, IsPacked = true },
                    new PackedEntry { Name = "Op_32x.gem", Type = "image", Offset = 0x139A135, UnpackedSize = 0x3EC36, Size = 0x5BCB, IsPacked = true },
                    new PackedEntry { Name = "Op_34x.gem", Type = "image", Offset = 0x139FD00, UnpackedSize = 0x3EC36, Size = 0x680A, IsPacked = true },
                    new PackedEntry { Name = "Op_35x.gem", Type = "image", Offset = 0x13A650A, UnpackedSize = 0x3EC36, Size = 0x8D76, IsPacked = true },
                    new PackedEntry { Name = "Op_36x.gem", Type = "image", Offset = 0x13AF280, UnpackedSize = 0x3EC36, Size = 0x9C34, IsPacked = true },
                    new PackedEntry { Name = "Op_37x.gem", Type = "image", Offset = 0x13B8EB4, UnpackedSize = 0x3EC36, Size = 0x90B8, IsPacked = true },
                    new PackedEntry { Name = "Op_tate.gem", Type = "image", Offset = 0x13C1F6C, UnpackedSize = 0x3EC36, Size = 0x75C7, IsPacked = true },
                    new PackedEntry { Name = "Save.gem", Type = "image", Offset = 0x13C9533, UnpackedSize = 0x3EC36, Size = 0x4987, IsPacked = true },
                    new PackedEntry { Name = "Shop.gem", Type = "image", Offset = 0x13CDEBA, UnpackedSize = 0x3EC36, Size = 0x2FE0, IsPacked = true },
                    new PackedEntry { Name = "Start_t.gps", Type = "image", Offset = 0x13D0E9A, UnpackedSize = 0x3EC36, Size = 0x6EB2, IsPacked = true },
                    new PackedEntry { Name = "Stat_a.gem", Type = "image", Offset = 0x13D7D4C, UnpackedSize = 0x3EC36, Size = 0x465A, IsPacked = true },
                    new PackedEntry { Name = "Stat_b.gem", Type = "image", Offset = 0x13DC3A6, UnpackedSize = 0x3EC36, Size = 0x262B, IsPacked = true },
                    new PackedEntry { Name = "St_ann.gem", Type = "image", Offset = 0x13DE9D1, UnpackedSize = 0x3EC36, Size = 0x278A, IsPacked = true },
                    new PackedEntry { Name = "St_cain.gem", Type = "image", Offset = 0x13E115B, UnpackedSize = 0x3EC36, Size = 0x27F8, IsPacked = true },
                    new PackedEntry { Name = "St_clim.gem", Type = "image", Offset = 0x13E3953, UnpackedSize = 0x3EC36, Size = 0x2558, IsPacked = true },
                    new PackedEntry { Name = "St_pon.gem", Type = "image", Offset = 0x13E5EAB, UnpackedSize = 0x3EC36, Size = 0x3A07, IsPacked = true },
                    new PackedEntry { Name = "St_riq.gem", Type = "image", Offset = 0x13E98B2, UnpackedSize = 0x3EC36, Size = 0x25CA, IsPacked = true },
                    new PackedEntry { Name = "St_russ.gem", Type = "image", Offset = 0x13EBE7C, UnpackedSize = 0x3EC36, Size = 0x2885, IsPacked = true },
                    new PackedEntry { Name = "Tai.gps", Type = "image", Offset = 0x13EE701, UnpackedSize = 0x3EC36, Size = 0x3D79, IsPacked = true },
                    new PackedEntry { Name = "Tit.gps", Type = "image", Offset = 0x13F247A, UnpackedSize = 0x3EC36, Size = 0x5EDE, IsPacked = true },
                    new PackedEntry { Name = "Title_de.gem", Type = "image", Offset = 0x13F8358, UnpackedSize = 0x3EC36, Size = 0x3D6C, IsPacked = true },
                    new PackedEntry { Name = "T_mcsr.gps", Type = "image", Offset = 0x13FC0C4, UnpackedSize = 0x5436, Size = 0x244, IsPacked = true },
                    new PackedEntry { Name = "Wipe.gem", Type = "image", Offset = 0x13FC308, UnpackedSize = 0x3EC36, Size = 0x25BE, IsPacked = true },
                    new PackedEntry { Name = "Wiz.gps", Type = "image", Offset = 0x13FE8C6, UnpackedSize = 0x3EC36, Size = 0x675A, IsPacked = true },
                    new PackedEntry { Name = "Bat.bmp", Type = "image", Offset = 0x1405020, UnpackedSize = 0x259E, Size = 0x6E9, IsPacked = true },
                    new PackedEntry { Name = "Cgco.bmp", Type = "image", Offset = 0x1405709, UnpackedSize = 0x3EBEE, Size = 0x193FD, IsPacked = true },
                    new PackedEntry { Name = "Hb_roll5.bmp", Type = "image", Offset = 0x141EB06, UnpackedSize = 0x3EC38, Size = 0x1499, IsPacked = true },
                    new PackedEntry { Name = "Hi.bmp", Type = "image", Offset = 0x141FF9F, UnpackedSize = 0x213E, Size = 0x747, IsPacked = true },
                    new PackedEntry { Name = "Off.bmp", Type = "image", Offset = 0x14206E6, UnpackedSize = 0x0D4E, Size = 0x51B, IsPacked = true },
                    new PackedEntry { Name = "Off2.bmp", Type = "image", Offset = 0x1420C01, UnpackedSize = 0x0D4E, Size = 0x51B, IsPacked = true },
                    new PackedEntry { Name = "On.bmp", Type = "image", Offset = 0x142111C, UnpackedSize = 0x0AF6, Size = 0x51A, IsPacked = true },
                    new PackedEntry { Name = "On2.bmp", Type = "image", Offset = 0x1421636, UnpackedSize = 0x0AF6, Size = 0x51A, IsPacked = true },
                    new PackedEntry { Name = "Qui.bmp", Type = "image", Offset = 0x1421B50, UnpackedSize = 0x1686, Size = 0x5E6, IsPacked = true },
                    new PackedEntry { Name = "Qui2.bmp", Type = "image", Offset = 0x1422136, UnpackedSize = 0x1686, Size = 0x5E6, IsPacked = true },
                    new PackedEntry { Name = "Re.bmp", Type = "image", Offset = 0x142271C, UnpackedSize = 0x1096, Size = 0x593, IsPacked = true },
                    new PackedEntry { Name = "Re2.bmp", Type = "image", Offset = 0x1422CAF, UnpackedSize = 0x1096, Size = 0x593, IsPacked = true },
                    new PackedEntry { Name = "Sou.bmp", Type = "image", Offset = 0x1423242, UnpackedSize = 0x13AE, Size = 0x611, IsPacked = true },
                    new PackedEntry { Name = "Vo.bmp", Type = "image", Offset = 0x1423853, UnpackedSize = 0x117E, Size = 0x5B5, IsPacked = true },
                }
            },
            {
                "ALLXXX.273", new List<Entry>() {
                    new PackedEntry { Name = "b_abel.spt", Type = "image", Offset = 0x0, UnpackedSize = 0x610, Size = 0x113, IsPacked = true },
                    new PackedEntry { Name = "b_ann.spt", Type = "image", Offset = 0x113, UnpackedSize = 0x7B0, Size = 0x2E0, IsPacked = true },
                    new PackedEntry { Name = "b_bal1.spt", Type = "image", Offset = 0x3F3, UnpackedSize = 0x0D0, Size = 0x8D, IsPacked = true },
                    new PackedEntry { Name = "b_bal2.spt", Type = "image", Offset = 0x480, UnpackedSize = 0x0D0, Size = 0x8D, IsPacked = true },
                    new PackedEntry { Name = "b_bat.spt", Type = "image", Offset = 0x50D, UnpackedSize = 0x0D0, Size = 0x62, IsPacked = true },
                    new PackedEntry { Name = "b_bat2.spt", Type = "image", Offset = 0x56F, UnpackedSize = 0x0D0, Size = 0x5F, IsPacked = true },
                    new PackedEntry { Name = "b_bee1.spt", Type = "image", Offset = 0x5CE, UnpackedSize = 0x0C0, Size = 0x61, IsPacked = true },
                    new PackedEntry { Name = "b_bobe.spt", Type = "image", Offset = 0x62F, UnpackedSize = 0x0F0, Size = 0x66, IsPacked = true },
                    new PackedEntry { Name = "b_bosc.spt", Type = "image", Offset = 0x695, UnpackedSize = 0x0E0, Size = 0x6C, IsPacked = true },
                    new PackedEntry { Name = "b_cain.spt", Type = "image", Offset = 0x701, UnpackedSize = 0x8A0, Size = 0x2AC, IsPacked = true },
                    new PackedEntry { Name = "b_chia.spt", Type = "image", Offset = 0x9AD, UnpackedSize = 0x150, Size = 0x0EB, IsPacked = true },
                    new PackedEntry { Name = "b_chib.spt", Type = "image", Offset = 0x0A98, UnpackedSize = 0x150, Size = 0x0EB, IsPacked = true },
                    new PackedEntry { Name = "b_chic.spt", Type = "image", Offset = 0x0B83, UnpackedSize = 0x130, Size = 0x84, IsPacked = true },
                    new PackedEntry { Name = "b_clim.spt", Type = "image", Offset = 0x0C07, UnpackedSize = 0x450, Size = 0x1BF, IsPacked = true },
                    new PackedEntry { Name = "b_cur1.spt", Type = "image", Offset = 0x0DC6, UnpackedSize = 0x50, Size = 0x1C, IsPacked = true },
                    new PackedEntry { Name = "b_cur2.spt", Type = "image", Offset = 0x0DE2, UnpackedSize = 0x90, Size = 0x5F, IsPacked = true },
                    new PackedEntry { Name = "b_dem1.spt", Type = "image", Offset = 0x0E41, UnpackedSize = 0x0C0, Size = 0x57, IsPacked = true },
                    new PackedEntry { Name = "b_dem2.spt", Type = "image", Offset = 0x0E98, UnpackedSize = 0x0C0, Size = 0x57, IsPacked = true },
                    new PackedEntry { Name = "b_dog.spt", Type = "image", Offset = 0x0EEF, UnpackedSize = 0x100, Size = 0x6C, IsPacked = true },
                    new PackedEntry { Name = "b_dog2.spt", Type = "image", Offset = 0x0F5B, UnpackedSize = 0x140, Size = 0x0B6, IsPacked = true },
                    new PackedEntry { Name = "b_dog3.spt", Type = "image", Offset = 0x1011, UnpackedSize = 0x160, Size = 0x0B0, IsPacked = true },
                    new PackedEntry { Name = "b_ga1a.spt", Type = "image", Offset = 0x10C1, UnpackedSize = 0x80, Size = 0x4D, IsPacked = true },
                    new PackedEntry { Name = "b_ga1b.spt", Type = "image", Offset = 0x110E, UnpackedSize = 0x0D0, Size = 0x88, IsPacked = true },
                    new PackedEntry { Name = "b_ga2a.spt", Type = "image", Offset = 0x1196, UnpackedSize = 0x80, Size = 0x4E, IsPacked = true },
                    new PackedEntry { Name = "b_ga2b.spt", Type = "image", Offset = 0x11E4, UnpackedSize = 0x0E0, Size = 0x92, IsPacked = true },
                    new PackedEntry { Name = "b_han2.spt", Type = "image", Offset = 0x1276, UnpackedSize = 0x160, Size = 0x92, IsPacked = true },
                    new PackedEntry { Name = "b_hang.spt", Type = "image", Offset = 0x1308, UnpackedSize = 0x160, Size = 0x92, IsPacked = true },
                    new PackedEntry { Name = "b_har2.spt", Type = "image", Offset = 0x139A, UnpackedSize = 0x0E0, Size = 0x0A3, IsPacked = true },
                    new PackedEntry { Name = "b_harp.spt", Type = "image", Offset = 0x143D, UnpackedSize = 0x0E0, Size = 0x0A3, IsPacked = true },
                    new PackedEntry { Name = "b_hebi.spt", Type = "image", Offset = 0x14E0, UnpackedSize = 0x150, Size = 0x64, IsPacked = true },
                    new PackedEntry { Name = "b_hei1.spt", Type = "image", Offset = 0x1544, UnpackedSize = 0x0C0, Size = 0x4E, IsPacked = true },
                    new PackedEntry { Name = "b_hei2.spt", Type = "image", Offset = 0x1592, UnpackedSize = 0x0D0, Size = 0x6B, IsPacked = true },
                    new PackedEntry { Name = "b_hei3.spt", Type = "image", Offset = 0x15FD, UnpackedSize = 0x0E0, Size = 0x76, IsPacked = true },
                    new PackedEntry { Name = "b_jia2.spt", Type = "image", Offset = 0x1673, UnpackedSize = 0x0B0, Size = 0x68, IsPacked = true },
                    new PackedEntry { Name = "b_jia3.spt", Type = "image", Offset = 0x16DB, UnpackedSize = 0x0B0, Size = 0x6B, IsPacked = true },
                    new PackedEntry { Name = "b_jia4.spt", Type = "image", Offset = 0x1746, UnpackedSize = 0x0B0, Size = 0x69, IsPacked = true },
                    new PackedEntry { Name = "b_jian.spt", Type = "image", Offset = 0x17AF, UnpackedSize = 0x0E0, Size = 0x0A5, IsPacked = true },
                    new PackedEntry { Name = "b_kasi.spt", Type = "image", Offset = 0x1854, UnpackedSize = 0x100, Size = 0x89, IsPacked = true },
                    new PackedEntry { Name = "b_kim2.spt", Type = "image", Offset = 0x18DD, UnpackedSize = 0x0B0, Size = 0x7D, IsPacked = true },
                    new PackedEntry { Name = "b_kime.spt", Type = "image", Offset = 0x195A, UnpackedSize = 0x0D0, Size = 0x89, IsPacked = true },
                    new PackedEntry { Name = "b_kin1.spt", Type = "image", Offset = 0x19E3, UnpackedSize = 0x250, Size = 0x99, IsPacked = true },
                    new PackedEntry { Name = "b_kin2.spt", Type = "image", Offset = 0x1A7C, UnpackedSize = 0x0B0, Size = 0x44, IsPacked = true },
                    new PackedEntry { Name = "b_kin3.spt", Type = "image", Offset = 0x1AC0, UnpackedSize = 0x50, Size = 0x2D, IsPacked = true },
                    new PackedEntry { Name = "b_kob2.spt", Type = "image", Offset = 0x1AED, UnpackedSize = 0x0B0, Size = 0x68, IsPacked = true },
                    new PackedEntry { Name = "b_kobo.spt", Type = "image", Offset = 0x1B55, UnpackedSize = 0x0B0, Size = 0x67, IsPacked = true },
                    new PackedEntry { Name = "b_mac1.spt", Type = "image", Offset = 0x1BBC, UnpackedSize = 0x80, Size = 0x54, IsPacked = true },
                    new PackedEntry { Name = "b_mac2.spt", Type = "image", Offset = 0x1C10, UnpackedSize = 0x0A0, Size = 0x64, IsPacked = true },
                    new PackedEntry { Name = "b_mac3.spt", Type = "image", Offset = 0x1C74, UnpackedSize = 0x0B0, Size = 0x7A, IsPacked = true },
                    new PackedEntry { Name = "b_maj1.spt", Type = "image", Offset = 0x1CEE, UnpackedSize = 0x90, Size = 0x47, IsPacked = true },
                    new PackedEntry { Name = "b_maj2.spt", Type = "image", Offset = 0x1D35, UnpackedSize = 0x90, Size = 0x47, IsPacked = true },
                    new PackedEntry { Name = "b_maj3.spt", Type = "image", Offset = 0x1D7C, UnpackedSize = 0x90, Size = 0x48, IsPacked = true },
                    new PackedEntry { Name = "b_pon.spt", Type = "image", Offset = 0x1DC4, UnpackedSize = 0x430, Size = 0x278, IsPacked = true },
                    new PackedEntry { Name = "b_riq.spt", Type = "image", Offset = 0x203C, UnpackedSize = 0x440, Size = 0x264, IsPacked = true },
                    new PackedEntry { Name = "b_rob1.spt", Type = "image", Offset = 0x22A0, UnpackedSize = 0x1B0, Size = 0x0B8, IsPacked = true },
                    new PackedEntry { Name = "b_rob2.spt", Type = "image", Offset = 0x2358, UnpackedSize = 0x1B0, Size = 0x0B7, IsPacked = true },
                    new PackedEntry { Name = "b_russ.spt", Type = "image", Offset = 0x240F, UnpackedSize = 0x790, Size = 0x263, IsPacked = true },
                    new PackedEntry { Name = "b_sab1.spt", Type = "image", Offset = 0x2672, UnpackedSize = 0x1A0, Size = 0x0E4, IsPacked = true },
                    new PackedEntry { Name = "b_sab2.spt", Type = "image", Offset = 0x2756, UnpackedSize = 0x130, Size = 0x95, IsPacked = true },
                    new PackedEntry { Name = "b_sem2.spt", Type = "image", Offset = 0x27EB, UnpackedSize = 0x0D0, Size = 0x8F, IsPacked = true },
                    new PackedEntry { Name = "b_semu.spt", Type = "image", Offset = 0x287A, UnpackedSize = 0x0D0, Size = 0x7B, IsPacked = true },
                    new PackedEntry { Name = "b_ska2.spt", Type = "image", Offset = 0x28F5, UnpackedSize = 0x160, Size = 0x0A4, IsPacked = true },
                    new PackedEntry { Name = "b_ska3.spt", Type = "image", Offset = 0x2999, UnpackedSize = 0x160, Size = 0x0A4, IsPacked = true },
                    new PackedEntry { Name = "b_skal.spt", Type = "image", Offset = 0x2A3D, UnpackedSize = 0x0C0, Size = 0x72, IsPacked = true },
                    new PackedEntry { Name = "b_sur2.spt", Type = "image", Offset = 0x2AAF, UnpackedSize = 0x100, Size = 0x7B, IsPacked = true },
                    new PackedEntry { Name = "b_sur3.spt", Type = "image", Offset = 0x2B2A, UnpackedSize = 0x100, Size = 0x7B, IsPacked = true },
                    new PackedEntry { Name = "b_sura.spt", Type = "image", Offset = 0x2BA5, UnpackedSize = 0x100, Size = 0x77, IsPacked = true },
                    new PackedEntry { Name = "b_sys.spt", Type = "image", Offset = 0x2C1C, UnpackedSize = 0x0F0, Size = 0x59, IsPacked = true },
                    new PackedEntry { Name = "b_thi1.spt", Type = "image", Offset = 0x2C75, UnpackedSize = 0x0C0, Size = 0x4E, IsPacked = true },
                    new PackedEntry { Name = "b_thi2.spt", Type = "image", Offset = 0x2CC3, UnpackedSize = 0x0C0, Size = 0x70, IsPacked = true },
                    new PackedEntry { Name = "b_tira.spt", Type = "image", Offset = 0x2D33, UnpackedSize = 0x1B0, Size = 0x0EB, IsPacked = true },
                    new PackedEntry { Name = "b_tora.spt", Type = "image", Offset = 0x2E1E, UnpackedSize = 0x130, Size = 0x0A7, IsPacked = true },
                    new PackedEntry { Name = "b_tre2.spt", Type = "image", Offset = 0x2EC5, UnpackedSize = 0x100, Size = 0x88, IsPacked = true },
                    new PackedEntry { Name = "b_tree.spt", Type = "image", Offset = 0x2F4D, UnpackedSize = 0x100, Size = 0x87, IsPacked = true },
                    new PackedEntry { Name = "b_uma1.spt", Type = "image", Offset = 0x2FD4, UnpackedSize = 0x0E0, Size = 0x81, IsPacked = true },
                    new PackedEntry { Name = "b_uma2.spt", Type = "image", Offset = 0x3055, UnpackedSize = 0x0E0, Size = 0x81, IsPacked = true },
                    new PackedEntry { Name = "b_was2.spt", Type = "image", Offset = 0x30D6, UnpackedSize = 0x120, Size = 0x0B9, IsPacked = true },
                    new PackedEntry { Name = "b_wasi.spt", Type = "image", Offset = 0x318F, UnpackedSize = 0x120, Size = 0x0B6, IsPacked = true },
                    new PackedEntry { Name = "b_wiz1.spt", Type = "image", Offset = 0x3245, UnpackedSize = 0x120, Size = 0x5C, IsPacked = true },
                    new PackedEntry { Name = "b_wiz2.spt", Type = "image", Offset = 0x32A1, UnpackedSize = 0x0C0, Size = 0x4F, IsPacked = true },
                    new PackedEntry { Name = "b_yaj2.spt", Type = "image", Offset = 0x32F0, UnpackedSize = 0x0C0, Size = 0x72, IsPacked = true },
                    new PackedEntry { Name = "b_yaj3.spt", Type = "image", Offset = 0x3362, UnpackedSize = 0x0D0, Size = 0x98, IsPacked = true },
                    new PackedEntry { Name = "b_yaji.spt", Type = "image", Offset = 0x33FA, UnpackedSize = 0x0D0, Size = 0x98, IsPacked = true },
                    new PackedEntry { Name = "b_zom1.spt", Type = "image", Offset = 0x3492, UnpackedSize = 0x0E0, Size = 0x8E, IsPacked = true },
                    new PackedEntry { Name = "b_zom2.spt", Type = "image", Offset = 0x3520, UnpackedSize = 0x0E0, Size = 0x8E, IsPacked = true },
                    new PackedEntry { Name = "allmap.cod", Type = "image", Offset = 0x35AE, UnpackedSize = 0x5D58, Size = 0x2C61, IsPacked = true },
                    new PackedEntry { Name = "badend.cod", Type = "image", Offset = 0x620F, UnpackedSize = 0x0C06, Size = 0x6F7, IsPacked = true },
                    new PackedEntry { Name = "cgm_anni.cod", Type = "image", Offset = 0x6906, UnpackedSize = 0x5B4, Size = 0x26C, IsPacked = true },
                    new PackedEntry { Name = "cgm_crim.cod", Type = "image", Offset = 0x6B72, UnpackedSize = 0x58E, Size = 0x26E, IsPacked = true },
                    new PackedEntry { Name = "cgm_gana.cod", Type = "image", Offset = 0x6DE0, UnpackedSize = 0x4D4, Size = 0x231, IsPacked = true },
                    new PackedEntry { Name = "cgm_king.cod", Type = "image", Offset = 0x7011, UnpackedSize = 0x30A, Size = 0x1AE, IsPacked = true },
                    new PackedEntry { Name = "cgm_mafi.cod", Type = "image", Offset = 0x71BF, UnpackedSize = 0x63C, Size = 0x26C, IsPacked = true },
                    new PackedEntry { Name = "cgm_milf.cod", Type = "image", Offset = 0x742B, UnpackedSize = 0x488, Size = 0x226, IsPacked = true },
                    new PackedEntry { Name = "cgm_pann.cod", Type = "image", Offset = 0x7651, UnpackedSize = 0x3FC, Size = 0x205, IsPacked = true },
                    new PackedEntry { Name = "cgm_quee.cod", Type = "image", Offset = 0x7856, UnpackedSize = 0x468, Size = 0x242, IsPacked = true },
                    new PackedEntry { Name = "cgm_riqu.cod", Type = "image", Offset = 0x7A98, UnpackedSize = 0x5A0, Size = 0x264, IsPacked = true },
                    new PackedEntry { Name = "cgm_ruse.cod", Type = "image", Offset = 0x7CFC, UnpackedSize = 0x58E, Size = 0x247, IsPacked = true },
                    new PackedEntry { Name = "cgm_seli.cod", Type = "image", Offset = 0x7F43, UnpackedSize = 0x582, Size = 0x24A, IsPacked = true },
                    new PackedEntry { Name = "cgm_shak.cod", Type = "image", Offset = 0x818D, UnpackedSize = 0x42A, Size = 0x244, IsPacked = true },
                    new PackedEntry { Name = "cgm_shok.cod", Type = "image", Offset = 0x83D1, UnpackedSize = 0x5D0, Size = 0x288, IsPacked = true },
                    new PackedEntry { Name = "cgm_shuk.cod", Type = "image", Offset = 0x8659, UnpackedSize = 0x40E, Size = 0x1EA, IsPacked = true },
                    new PackedEntry { Name = "cgm_tira.cod", Type = "image", Offset = 0x8843, UnpackedSize = 0x3D0, Size = 0x1E2, IsPacked = true },
                    new PackedEntry { Name = "cgm_tolt.cod", Type = "image", Offset = 0x8A25, UnpackedSize = 0x58E, Size = 0x269, IsPacked = true },
                    new PackedEntry { Name = "cron.cod", Type = "image", Offset = 0x8C8E, UnpackedSize = 0x309E, Size = 0x1807, IsPacked = true },
                    new PackedEntry { Name = "cudu.cod", Type = "image", Offset = 0x0A495, UnpackedSize = 0x0FD0, Size = 0x6C0, IsPacked = true },
                    new PackedEntry { Name = "dragon.cod", Type = "image", Offset = 0x0AB55, UnpackedSize = 0x55C8, Size = 0x29AA, IsPacked = true },
                    new PackedEntry { Name = "ending.cod", Type = "image", Offset = 0x0D4FF, UnpackedSize = 0x0DE88, Size = 0x69F2, IsPacked = true },
                    new PackedEntry { Name = "endsyoko.cod", Type = "image", Offset = 0x13EF1, UnpackedSize = 0x535C, Size = 0x2872, IsPacked = true },
                    new PackedEntry { Name = "flap.cod", Type = "image", Offset = 0x16763, UnpackedSize = 0x0C6F0, Size = 0x595D, IsPacked = true },
                    new PackedEntry { Name = "fond.cod", Type = "image", Offset = 0x1C0C0, UnpackedSize = 0x0BB0, Size = 0x4F3, IsPacked = true },
                    new PackedEntry { Name = "garu.cod", Type = "image", Offset = 0x1C5B3, UnpackedSize = 0x0E06, Size = 0x5E2, IsPacked = true },
                    new PackedEntry { Name = "ikki.cod", Type = "image", Offset = 0x1CB95, UnpackedSize = 0x56F8, Size = 0x2541, IsPacked = true },
                    new PackedEntry { Name = "intro.cod", Type = "image", Offset = 0x1F0D6, UnpackedSize = 0x607C, Size = 0x2DE1, IsPacked = true },
                    new PackedEntry { Name = "iseki.cod", Type = "image", Offset = 0x21EB7, UnpackedSize = 0x470E, Size = 0x20FF, IsPacked = true },
                    new PackedEntry { Name = "joka.cod", Type = "image", Offset = 0x23FB6, UnpackedSize = 0x8BE8, Size = 0x424E, IsPacked = true },
                    new PackedEntry { Name = "jokaann.cod", Type = "image", Offset = 0x28204, UnpackedSize = 0x2884, Size = 0x145A, IsPacked = true },
                    new PackedEntry { Name = "jokamaf.cod", Type = "image", Offset = 0x2965E, UnpackedSize = 0x0AF66, Size = 0x5001, IsPacked = true },
                    new PackedEntry { Name = "kaiz.cod", Type = "image", Offset = 0x2E65F, UnpackedSize = 0x1B40, Size = 0x0C7F, IsPacked = true },
                    new PackedEntry { Name = "kata.cod", Type = "image", Offset = 0x2F2DE, UnpackedSize = 0x0BF30, Size = 0x5935, IsPacked = true },
                    new PackedEntry { Name = "koro.cod", Type = "image", Offset = 0x34C13, UnpackedSize = 0x80E4, Size = 0x3D6F, IsPacked = true },
                    new PackedEntry { Name = "kuro.cod", Type = "image", Offset = 0x38982, UnpackedSize = 0x4294, Size = 0x1F42, IsPacked = true },
                    new PackedEntry { Name = "last.cod", Type = "image", Offset = 0x3A8C4, UnpackedSize = 0x4DAC, Size = 0x23F9, IsPacked = true },
                    new PackedEntry { Name = "matuclim.cod", Type = "image", Offset = 0x3CCBD, UnpackedSize = 0x49E0, Size = 0x2187, IsPacked = true },
                    new PackedEntry { Name = "mura.cod", Type = "image", Offset = 0x3EE44, UnpackedSize = 0x6426, Size = 0x2F6F, IsPacked = true },
                    new PackedEntry { Name = "ojou.cod", Type = "image", Offset = 0x41DB3, UnpackedSize = 0x0E032, Size = 0x6248, IsPacked = true },
                    new PackedEntry { Name = "oldw.cod", Type = "image", Offset = 0x47FFB, UnpackedSize = 0x4E7E, Size = 0x265A, IsPacked = true },
                    new PackedEntry { Name = "pare.cod", Type = "image", Offset = 0x4A655, UnpackedSize = 0x0E4FE, Size = 0x6984, IsPacked = true },
                    new PackedEntry { Name = "quick.cod", Type = "image", Offset = 0x50FD9, UnpackedSize = 0x10, Size = 0x14, IsPacked = true },
                    new PackedEntry { Name = "retry.cod", Type = "image", Offset = 0x50FED, UnpackedSize = 0x4E, Size = 0x52, IsPacked = true },
                    new PackedEntry { Name = "saba.cod", Type = "image", Offset = 0x5103F, UnpackedSize = 0x9CA, Size = 0x326, IsPacked = true },
                    new PackedEntry { Name = "selijoka.cod", Type = "image", Offset = 0x51365, UnpackedSize = 0x2D64, Size = 0x162F, IsPacked = true },
                    new PackedEntry { Name = "sepu.cod", Type = "image", Offset = 0x52994, UnpackedSize = 0x0A468, Size = 0x481B, IsPacked = true },
                    new PackedEntry { Name = "shikei.cod", Type = "image", Offset = 0x571AF, UnpackedSize = 0x2FA8, Size = 0x185F, IsPacked = true },
                    new PackedEntry { Name = "sima.cod", Type = "image", Offset = 0x58A0E, UnpackedSize = 0x958, Size = 0x390, IsPacked = true },
                    new PackedEntry { Name = "start.cod", Type = "image", Offset = 0x58D9E, UnpackedSize = 0x10, Size = 0x14, IsPacked = true },
                    new PackedEntry { Name = "syud.cod", Type = "image", Offset = 0x58DB2, UnpackedSize = 0x6F4A, Size = 0x3353, IsPacked = true },
                    new PackedEntry { Name = "test.cod", Type = "image", Offset = 0x5C105, UnpackedSize = 0x26, Size = 0x22, IsPacked = true },
                    new PackedEntry { Name = "tidu.cod", Type = "image", Offset = 0x5C127, UnpackedSize = 0x5FDA, Size = 0x2CA3, IsPacked = true },
                    new PackedEntry { Name = "tita.cod", Type = "image", Offset = 0x5EDCA, UnpackedSize = 0x940, Size = 0x37E, IsPacked = true },
                    new PackedEntry { Name = "tori.cod", Type = "image", Offset = 0x5F148, UnpackedSize = 0x0E3C2, Size = 0x6455, IsPacked = true },
                    new PackedEntry { Name = "umaya.cod", Type = "image", Offset = 0x6559D, UnpackedSize = 0x52A8, Size = 0x2832, IsPacked = true },
                    new PackedEntry { Name = "yama.cod", Type = "image", Offset = 0x67DCF, UnpackedSize = 0x1282, Size = 0x757, IsPacked = true },
                    new PackedEntry { Name = "yuru.cod", Type = "image", Offset = 0x68526, UnpackedSize = 0x0F48C, Size = 0x7274, IsPacked = true },
                    new PackedEntry { Name = "ankei_a.sp2", Type = "image", Offset = 0x6F79A, UnpackedSize = 0x143, Size = 0x147, IsPacked = true },
                    new PackedEntry { Name = "ankei_b.sp2", Type = "image", Offset = 0x6F8E1, UnpackedSize = 0x10D, Size = 0x111, IsPacked = true },
                    new PackedEntry { Name = "ankei_c.sp2", Type = "image", Offset = 0x6F9F2, UnpackedSize = 0x143, Size = 0x147, IsPacked = true },
                    new PackedEntry { Name = "beem_a.sp2", Type = "image", Offset = 0x6FB39, UnpackedSize = 0x413, Size = 0x356, IsPacked = true },
                    new PackedEntry { Name = "beem_b.sp2", Type = "image", Offset = 0x6FE8F, UnpackedSize = 0x38F, Size = 0x2EC, IsPacked = true },
                    new PackedEntry { Name = "beem_c.sp2", Type = "image", Offset = 0x7017B, UnpackedSize = 0x446, Size = 0x382, IsPacked = true },
                    new PackedEntry { Name = "b_abel_a.sp2", Type = "image", Offset = 0x704FD, UnpackedSize = 0x1136, Size = 0x0DF1, IsPacked = true },
                    new PackedEntry { Name = "b_ann_a.sp2", Type = "image", Offset = 0x712EE, UnpackedSize = 0x25D9, Size = 0x1DA6, IsPacked = true },
                    new PackedEntry { Name = "b_ann_b.sp2", Type = "image", Offset = 0x73094, UnpackedSize = 0x251F, Size = 0x1BFD, IsPacked = true },
                    new PackedEntry { Name = "b_ann_c.sp2", Type = "image", Offset = 0x74C91, UnpackedSize = 0x2543, Size = 0x1C14, IsPacked = true },
                    new PackedEntry { Name = "b_bal1_a.sp2", Type = "image", Offset = 0x768A5, UnpackedSize = 0x1AF0, Size = 0x0AD5, IsPacked = true },
                    new PackedEntry { Name = "b_bat_a.sp2", Type = "image", Offset = 0x7737A, UnpackedSize = 0x8D8, Size = 0x5CF, IsPacked = true },
                    new PackedEntry { Name = "b_bat_b.sp2", Type = "image", Offset = 0x77949, UnpackedSize = 0x8D8, Size = 0x5DC, IsPacked = true },
                    new PackedEntry { Name = "b_bee1_a.sp2", Type = "image", Offset = 0x77F25, UnpackedSize = 0x0D04, Size = 0x0ADF, IsPacked = true },
                    new PackedEntry { Name = "b_bobe_a.sp2", Type = "image", Offset = 0x78A04, UnpackedSize = 0x6D1, Size = 0x5C6, IsPacked = true },
                    new PackedEntry { Name = "b_bosc_a.sp2", Type = "image", Offset = 0x78FCA, UnpackedSize = 0x48E, Size = 0x3F8, IsPacked = true },
                    new PackedEntry { Name = "b_cain_a.sp2", Type = "image", Offset = 0x793C2, UnpackedSize = 0x1C4F, Size = 0x16E7, IsPacked = true },
                    new PackedEntry { Name = "b_cain_b.sp2", Type = "image", Offset = 0x7AAA9, UnpackedSize = 0x1D00, Size = 0x17AE, IsPacked = true },
                    new PackedEntry { Name = "b_cain_c.sp2", Type = "image", Offset = 0x7C257, UnpackedSize = 0x1AFC, Size = 0x1611, IsPacked = true },
                    new PackedEntry { Name = "b_chia_a.sp2", Type = "image", Offset = 0x7D868, UnpackedSize = 0x17CC, Size = 0x0F99, IsPacked = true },
                    new PackedEntry { Name = "b_chib_a.sp2", Type = "image", Offset = 0x7E801, UnpackedSize = 0x17AB, Size = 0x0FDD, IsPacked = true },
                    new PackedEntry { Name = "b_chic_a.sp2", Type = "image", Offset = 0x7F7DE, UnpackedSize = 0x1358, Size = 0x64A, IsPacked = true },
                    new PackedEntry { Name = "b_clim_a.sp2", Type = "image", Offset = 0x7FE28, UnpackedSize = 0x0C59, Size = 0x729, IsPacked = true },
                    new PackedEntry { Name = "b_clim_b.sp2", Type = "image", Offset = 0x80551, UnpackedSize = 0x0CE3, Size = 0x7F8, IsPacked = true },
                    new PackedEntry { Name = "b_clim_c.sp2", Type = "image", Offset = 0x80D49, UnpackedSize = 0x0A52, Size = 0x6CE, IsPacked = true },
                    new PackedEntry { Name = "b_cur1_a.sp2", Type = "image", Offset = 0x81417, UnpackedSize = 0x85D, Size = 0x306, IsPacked = true },
                    new PackedEntry { Name = "b_cur2_a.sp2", Type = "image", Offset = 0x8171D, UnpackedSize = 0x212F, Size = 0x0CCE, IsPacked = true },
                    new PackedEntry { Name = "b_dem1_a.sp2", Type = "image", Offset = 0x823EB, UnpackedSize = 0x8AE, Size = 0x8B2, IsPacked = true },
                    new PackedEntry { Name = "b_dem1_b.sp2", Type = "image", Offset = 0x82C9D, UnpackedSize = 0x8FC, Size = 0x900, IsPacked = true },
                    new PackedEntry { Name = "b_dog_b.sp2", Type = "image", Offset = 0x8359D, UnpackedSize = 0x1589, Size = 0x0FDE, IsPacked = true },
                    new PackedEntry { Name = "b_ga1a_b.sp2", Type = "image", Offset = 0x8457B, UnpackedSize = 0x156B, Size = 0x529, IsPacked = true },
                    new PackedEntry { Name = "b_ga1b_b.sp2", Type = "image", Offset = 0x84AA4, UnpackedSize = 0x25F4, Size = 0x0D3D, IsPacked = true },
                    new PackedEntry { Name = "b_ga2a_a.sp2", Type = "image", Offset = 0x857E1, UnpackedSize = 0x0C7D, Size = 0x381, IsPacked = true },
                    new PackedEntry { Name = "b_ga2b_a.sp2", Type = "image", Offset = 0x85B62, UnpackedSize = 0x38F3, Size = 0x0E90, IsPacked = true },
                    new PackedEntry { Name = "b_hang_a.sp2", Type = "image", Offset = 0x869F2, UnpackedSize = 0x1790, Size = 0x1341, IsPacked = true },
                    new PackedEntry { Name = "b_harp_a.sp2", Type = "image", Offset = 0x87D33, UnpackedSize = 0x1BAA, Size = 0x0B67, IsPacked = true },
                    new PackedEntry { Name = "b_harp_b.sp2", Type = "image", Offset = 0x8889A, UnpackedSize = 0x1B80, Size = 0x0B09, IsPacked = true },
                    new PackedEntry { Name = "b_hebi_a.sp2", Type = "image", Offset = 0x893A3, UnpackedSize = 0x0F98, Size = 0x0DF4, IsPacked = true },
                    new PackedEntry { Name = "b_hei1_a.sp2", Type = "image", Offset = 0x8A197, UnpackedSize = 0x1667, Size = 0x0D77, IsPacked = true },
                    new PackedEntry { Name = "b_icon.sp2", Type = "image", Offset = 0x8AF0E, UnpackedSize = 0x3E9, Size = 0x34E, IsPacked = true },
                    new PackedEntry { Name = "b_iconb.sp2", Type = "image", Offset = 0x8B25C, UnpackedSize = 0x31D, Size = 0x14C, IsPacked = true },
                    new PackedEntry { Name = "b_jian_b.sp2", Type = "image", Offset = 0x8B3A8, UnpackedSize = 0x386C, Size = 0x1B31, IsPacked = true },
                    new PackedEntry { Name = "b_kasi_a.sp2", Type = "image", Offset = 0x8CED9, UnpackedSize = 0x1604, Size = 0x0C18, IsPacked = true },
                    new PackedEntry { Name = "b_kime_b.sp2", Type = "image", Offset = 0x8DAF1, UnpackedSize = 0x1E65, Size = 0x0FAF, IsPacked = true },
                    new PackedEntry { Name = "b_kin1_a.sp2", Type = "image", Offset = 0x8EAA0, UnpackedSize = 0x5855, Size = 0x14C3, IsPacked = true },
                    new PackedEntry { Name = "b_kin2_a.sp2", Type = "image", Offset = 0x8FF63, UnpackedSize = 0x0EED, Size = 0x0C2B, IsPacked = true },
                    new PackedEntry { Name = "b_kin3_a.sp2", Type = "image", Offset = 0x90B8E, UnpackedSize = 0x20D2, Size = 0x1A9C, IsPacked = true },
                    new PackedEntry { Name = "b_kobo_a.sp2", Type = "image", Offset = 0x9262A, UnpackedSize = 0x848, Size = 0x53A, IsPacked = true },
                    new PackedEntry { Name = "b_kobo_b.sp2", Type = "image", Offset = 0x92B64, UnpackedSize = 0x7A6, Size = 0x4C5, IsPacked = true },
                    new PackedEntry { Name = "b_kobo_c.sp2", Type = "image", Offset = 0x93029, UnpackedSize = 0x6FE, Size = 0x452, IsPacked = true },
                    new PackedEntry { Name = "b_mac1_b.sp2", Type = "image", Offset = 0x9347B, UnpackedSize = 0x6C8, Size = 0x1B6, IsPacked = true },
                    new PackedEntry { Name = "b_mac2_b.sp2", Type = "image", Offset = 0x93631, UnpackedSize = 0x0AC7, Size = 0x2EB, IsPacked = true },
                    new PackedEntry { Name = "b_mac3_b.sp2", Type = "image", Offset = 0x9391C, UnpackedSize = 0x6D70, Size = 0x13CC, IsPacked = true },
                    new PackedEntry { Name = "b_magic0.sp2", Type = "image", Offset = 0x94CE8, UnpackedSize = 0x1EB, Size = 0x1EF, IsPacked = true },
                    new PackedEntry { Name = "b_magic1.sp2", Type = "image", Offset = 0x94ED7, UnpackedSize = 0x3CB, Size = 0x3CF, IsPacked = true },
                    new PackedEntry { Name = "b_magic2.sp2", Type = "image", Offset = 0x952A6, UnpackedSize = 0x0C5, Size = 0x94, IsPacked = true },
                    new PackedEntry { Name = "b_magic3.sp2", Type = "image", Offset = 0x9533A, UnpackedSize = 0x56, Size = 0x5A, IsPacked = true },
                    new PackedEntry { Name = "b_magic4.sp2", Type = "image", Offset = 0x95394, UnpackedSize = 0x1CA, Size = 0x1CE, IsPacked = true },
                    new PackedEntry { Name = "b_magic5.sp2", Type = "image", Offset = 0x95562, UnpackedSize = 0x8C0, Size = 0x8C4, IsPacked = true },
                    new PackedEntry { Name = "b_magic6.sp2", Type = "image", Offset = 0x95E26, UnpackedSize = 0x8C0, Size = 0x8C4, IsPacked = true },
                    new PackedEntry { Name = "b_magic7.sp2", Type = "image", Offset = 0x966EA, UnpackedSize = 0x54E, Size = 0x552, IsPacked = true },
                    new PackedEntry { Name = "b_magic8.sp2", Type = "image", Offset = 0x96C3C, UnpackedSize = 0x428, Size = 0x42C, IsPacked = true },
                    new PackedEntry { Name = "b_maj1_a.sp2", Type = "image", Offset = 0x97068, UnpackedSize = 0x740, Size = 0x516, IsPacked = true },
                    new PackedEntry { Name = "b_pon_a.sp2", Type = "image", Offset = 0x9757E, UnpackedSize = 0x10FA, Size = 0x0D3D, IsPacked = true },
                    new PackedEntry { Name = "b_pon_b.sp2", Type = "image", Offset = 0x982BB, UnpackedSize = 0x0FE6, Size = 0x0BD1, IsPacked = true },
                    new PackedEntry { Name = "b_pon_c.sp2", Type = "image", Offset = 0x98E8C, UnpackedSize = 0x10F7, Size = 0x0CE1, IsPacked = true },
                    new PackedEntry { Name = "b_riq_a.sp2", Type = "image", Offset = 0x99B6D, UnpackedSize = 0x102B, Size = 0x0A2A, IsPacked = true },
                    new PackedEntry { Name = "b_riq_b.sp2", Type = "image", Offset = 0x9A597, UnpackedSize = 0x104C, Size = 0x0A1D, IsPacked = true },
                    new PackedEntry { Name = "b_riq_c.sp2", Type = "image", Offset = 0x9AFB4, UnpackedSize = 0x0F9E, Size = 0x981, IsPacked = true },
                    new PackedEntry { Name = "b_rob1_a.sp2", Type = "image", Offset = 0x9B935, UnpackedSize = 0x6F26, Size = 0x218F, IsPacked = true },
                    new PackedEntry { Name = "b_russ_a.sp2", Type = "image", Offset = 0x9DAC4, UnpackedSize = 0x17F6, Size = 0x0FEB, IsPacked = true },
                    new PackedEntry { Name = "b_russ_b.sp2", Type = "image", Offset = 0x9EAAF, UnpackedSize = 0x1838, Size = 0x0F20, IsPacked = true },
                    new PackedEntry { Name = "b_russ_c.sp2", Type = "image", Offset = 0x9F9CF, UnpackedSize = 0x16C4, Size = 0x0E23, IsPacked = true },
                    new PackedEntry { Name = "b_sab1_a.sp2", Type = "image", Offset = 0x0A07F2, UnpackedSize = 0x407F, Size = 0x1ABE, IsPacked = true },
                    new PackedEntry { Name = "b_semu_b.sp2", Type = "image", Offset = 0x0A22B0, UnpackedSize = 0x1B08, Size = 0x1108, IsPacked = true },
                    new PackedEntry { Name = "b_skal_a.sp2", Type = "image", Offset = 0x0A33B8, UnpackedSize = 0x1073, Size = 0x0BE5, IsPacked = true },
                    new PackedEntry { Name = "b_skal_b.sp2", Type = "image", Offset = 0x0A3F9D, UnpackedSize = 0x101F, Size = 0x0BC7, IsPacked = true },
                    new PackedEntry { Name = "b_skal_c.sp2", Type = "image", Offset = 0x0A4B64, UnpackedSize = 0x1013, Size = 0x0B59, IsPacked = true },
                    new PackedEntry { Name = "b_sura_a.sp2", Type = "image", Offset = 0x0A56BD, UnpackedSize = 0x596, Size = 0x442, IsPacked = true },
                    new PackedEntry { Name = "b_sura_b.sp2", Type = "image", Offset = 0x0A5AFF, UnpackedSize = 0x51E, Size = 0x417, IsPacked = true },
                    new PackedEntry { Name = "b_sura_c.sp2", Type = "image", Offset = 0x0A5F16, UnpackedSize = 0x590, Size = 0x41C, IsPacked = true },
                    new PackedEntry { Name = "b_sys.sp2", Type = "image", Offset = 0x0A6332, UnpackedSize = 0x28A, Size = 0x28E, IsPacked = true },
                    new PackedEntry { Name = "b_sys2.sp2", Type = "image", Offset = 0x0A65C0, UnpackedSize = 0x266, Size = 0x26A, IsPacked = true },
                    new PackedEntry { Name = "b_thi1_a.sp2", Type = "image", Offset = 0x0A682A, UnpackedSize = 0x0DFD, Size = 0x81E, IsPacked = true },
                    new PackedEntry { Name = "b_tira_a.sp2", Type = "image", Offset = 0x0A7048, UnpackedSize = 0x1BAD, Size = 0x1040, IsPacked = true },
                    new PackedEntry { Name = "b_tora_a.sp2", Type = "image", Offset = 0x0A8088, UnpackedSize = 0x19E2, Size = 0x11CC, IsPacked = true },
                    new PackedEntry { Name = "b_tora_b.sp2", Type = "image", Offset = 0x0A9254, UnpackedSize = 0x1AC9, Size = 0x1078, IsPacked = true },
                    new PackedEntry { Name = "b_tora_c.sp2", Type = "image", Offset = 0x0AA2CC, UnpackedSize = 0x1A54, Size = 0x10FE, IsPacked = true },
                    new PackedEntry { Name = "b_tree_a.sp2", Type = "image", Offset = 0x0AB3CA, UnpackedSize = 0x2756, Size = 0x0E97, IsPacked = true },
                    new PackedEntry { Name = "b_uma.sp2", Type = "image", Offset = 0x0AC261, UnpackedSize = 0x0D0D, Size = 0x0BF0, IsPacked = true },
                    new PackedEntry { Name = "b_uma1_b.sp2", Type = "image", Offset = 0x0ACE51, UnpackedSize = 0x194F, Size = 0x10D7, IsPacked = true },
                    new PackedEntry { Name = "b_wasi_a.sp2", Type = "image", Offset = 0x0ADF28, UnpackedSize = 0x1B29, Size = 0x10D8, IsPacked = true },
                    new PackedEntry { Name = "b_wasi_b.sp2", Type = "image", Offset = 0x0AF000, UnpackedSize = 0x1B5F, Size = 0x1084, IsPacked = true },
                    new PackedEntry { Name = "b_wiz1_a.sp2", Type = "image", Offset = 0x0B0084, UnpackedSize = 0x0CD1, Size = 0x0AD9, IsPacked = true },
                    new PackedEntry { Name = "b_yaji_c.sp2", Type = "image", Offset = 0x0B0B5D, UnpackedSize = 0x1D81, Size = 0x1261, IsPacked = true },
                    new PackedEntry { Name = "b_zom1_b.sp2", Type = "image", Offset = 0x0B1DBE, UnpackedSize = 0x0A01, Size = 0x6F3, IsPacked = true },
                    new PackedEntry { Name = "con_chr.sp2", Type = "image", Offset = 0x0B24B1, UnpackedSize = 0x1E5, Size = 0x1E9, IsPacked = true },
                    new PackedEntry { Name = "f_abel.sp2", Type = "image", Offset = 0x0B269A, UnpackedSize = 0x68C, Size = 0x4FC, IsPacked = true },
                    new PackedEntry { Name = "f_annin.sp2", Type = "image", Offset = 0x0B2B96, UnpackedSize = 0x716, Size = 0x5B7, IsPacked = true },
                    new PackedEntry { Name = "f_baba.sp2", Type = "image", Offset = 0x0B314D, UnpackedSize = 0x686, Size = 0x26B, IsPacked = true },
                    new PackedEntry { Name = "f_bajil.sp2", Type = "image", Offset = 0x0B33B8, UnpackedSize = 0x6EC, Size = 0x323, IsPacked = true },
                    new PackedEntry { Name = "f_baten.sp2", Type = "image", Offset = 0x0B36DB, UnpackedSize = 0x560, Size = 0x248, IsPacked = true },
                    new PackedEntry { Name = "f_bobes.sp2", Type = "image", Offset = 0x0B3923, UnpackedSize = 0x869, Size = 0x6ED, IsPacked = true },
                    new PackedEntry { Name = "f_bosco.sp2", Type = "image", Offset = 0x0B4010, UnpackedSize = 0x71F, Size = 0x63E, IsPacked = true },
                    new PackedEntry { Name = "f_boy1.sp2", Type = "image", Offset = 0x0B464E, UnpackedSize = 0x5D8, Size = 0x418, IsPacked = true },
                    new PackedEntry { Name = "f_buki.sp2", Type = "image", Offset = 0x0B4A66, UnpackedSize = 0x61A, Size = 0x28A, IsPacked = true },
                    new PackedEntry { Name = "f_cain.sp2", Type = "image", Offset = 0x0B4CF0, UnpackedSize = 0x1DF6, Size = 0x0CBB, IsPacked = true },
                    new PackedEntry { Name = "f_chibi.sp2", Type = "image", Offset = 0x0B59AB, UnpackedSize = 0x42E, Size = 0x271, IsPacked = true },
                    new PackedEntry { Name = "f_clim1.sp2", Type = "image", Offset = 0x0B5C1C, UnpackedSize = 0x66E, Size = 0x206, IsPacked = true },
                    new PackedEntry { Name = "f_clim2.sp2", Type = "image", Offset = 0x0B5E22, UnpackedSize = 0x0B36, Size = 0x368, IsPacked = true },
                    new PackedEntry { Name = "f_coin.sp2", Type = "image", Offset = 0x0B618A, UnpackedSize = 0x1C7, Size = 0x1CB, IsPacked = true },
                    new PackedEntry { Name = "f_cura.sp2", Type = "image", Offset = 0x0B6355, UnpackedSize = 0x5D8, Size = 0x28B, IsPacked = true },
                    new PackedEntry { Name = "f_door1.sp2", Type = "image", Offset = 0x0B65E0, UnpackedSize = 0x0F5, Size = 0x0F9, IsPacked = true },
                    new PackedEntry { Name = "f_door10.sp2", Type = "image", Offset = 0x0B66D9, UnpackedSize = 0x95, Size = 0x99, IsPacked = true },
                    new PackedEntry { Name = "f_door11.sp2", Type = "image", Offset = 0x0B6772, UnpackedSize = 0x128, Size = 0x12C, IsPacked = true },
                    new PackedEntry { Name = "f_door12.sp2", Type = "image", Offset = 0x0B689E, UnpackedSize = 0x0BAE, Size = 0x157, IsPacked = true },
                    new PackedEntry { Name = "f_door2.sp2", Type = "image", Offset = 0x0B69F5, UnpackedSize = 0x12B, Size = 0x12F, IsPacked = true },
                    new PackedEntry { Name = "f_door3.sp2", Type = "image", Offset = 0x0B6B24, UnpackedSize = 0x146, Size = 0x14A, IsPacked = true },
                    new PackedEntry { Name = "f_door4.sp2", Type = "image", Offset = 0x0B6C6E, UnpackedSize = 0x0FB, Size = 0x0FF, IsPacked = true },
                    new PackedEntry { Name = "f_door5.sp2", Type = "image", Offset = 0x0B6D6D, UnpackedSize = 0x8C, Size = 0x90, IsPacked = true },
                    new PackedEntry { Name = "f_door6.sp2", Type = "image", Offset = 0x0B6DFD, UnpackedSize = 0x95, Size = 0x99, IsPacked = true },
                    new PackedEntry { Name = "f_door7.sp2", Type = "image", Offset = 0x0B6E96, UnpackedSize = 0x7D, Size = 0x81, IsPacked = true },
                    new PackedEntry { Name = "f_door8.sp2", Type = "image", Offset = 0x0B6F17, UnpackedSize = 0x95, Size = 0x99, IsPacked = true },
                    new PackedEntry { Name = "f_door9.sp2", Type = "image", Offset = 0x0B6FB0, UnpackedSize = 0x125, Size = 0x129, IsPacked = true },
                    new PackedEntry { Name = "f_dummy.sp2", Type = "image", Offset = 0x0B70D9, UnpackedSize = 0x16AF, Size = 0x1153, IsPacked = true },
                    new PackedEntry { Name = "f_funsui.sp2", Type = "image", Offset = 0x0B822C, UnpackedSize = 0x2D5, Size = 0x2D9, IsPacked = true },
                    new PackedEntry { Name = "f_futa.sp2", Type = "image", Offset = 0x0B8505, UnpackedSize = 0x4D, Size = 0x51, IsPacked = true },
                    new PackedEntry { Name = "f_gaki.sp2", Type = "image", Offset = 0x0B8556, UnpackedSize = 0x386, Size = 0x1FD, IsPacked = true },
                    new PackedEntry { Name = "f_gal1.sp2", Type = "image", Offset = 0x0B8753, UnpackedSize = 0x5A2, Size = 0x3FE, IsPacked = true },
                    new PackedEntry { Name = "f_gal2.sp2", Type = "image", Offset = 0x0B8B51, UnpackedSize = 0x5A2, Size = 0x407, IsPacked = true },
                    new PackedEntry { Name = "f_gana.sp2", Type = "image", Offset = 0x0B8F58, UnpackedSize = 0x746, Size = 0x330, IsPacked = true },
                    new PackedEntry { Name = "f_hasi.sp2", Type = "image", Offset = 0x0B9288, UnpackedSize = 0x23, Size = 0x27, IsPacked = true },
                    new PackedEntry { Name = "f_hei.sp2", Type = "image", Offset = 0x0B92AF, UnpackedSize = 0x674, Size = 0x55D, IsPacked = true },
                    new PackedEntry { Name = "f_icon.sp2", Type = "image", Offset = 0x0B980C, UnpackedSize = 0x503, Size = 0x459, IsPacked = true },
                    new PackedEntry { Name = "f_iconb.sp2", Type = "image", Offset = 0x0B9C65, UnpackedSize = 0x3A7, Size = 0x2AE, IsPacked = true },
                    new PackedEntry { Name = "f_jiji1.sp2", Type = "image", Offset = 0x0B9F13, UnpackedSize = 0x596, Size = 0x41E, IsPacked = true },
                    new PackedEntry { Name = "f_jiji2.sp2", Type = "image", Offset = 0x0BA331, UnpackedSize = 0x596, Size = 0x44D, IsPacked = true },
                    new PackedEntry { Name = "f_kasira.sp2", Type = "image", Offset = 0x0BA77E, UnpackedSize = 0x1202, Size = 0x367, IsPacked = true },
                    new PackedEntry { Name = "f_king.sp2", Type = "image", Offset = 0x0BAAE5, UnpackedSize = 0x61A, Size = 0x37F, IsPacked = true },
                    new PackedEntry { Name = "f_kuro1.sp2", Type = "image", Offset = 0x0BAE64, UnpackedSize = 0x3EC, Size = 0x2E9, IsPacked = true },
                    new PackedEntry { Name = "f_kuro2.sp2", Type = "image", Offset = 0x0BB14D, UnpackedSize = 0x0A4, Size = 0x0A8, IsPacked = true },
                    new PackedEntry { Name = "f_kuro3.sp2", Type = "image", Offset = 0x0BB1F5, UnpackedSize = 0x0A4, Size = 0x0A8, IsPacked = true },
                    new PackedEntry { Name = "f_mado.sp2", Type = "image", Offset = 0x0BB29D, UnpackedSize = 0x644, Size = 0x457, IsPacked = true },
                    new PackedEntry { Name = "f_mafin1.sp2", Type = "image", Offset = 0x0BB6F4, UnpackedSize = 0x1A6, Size = 0x8A, IsPacked = true },
                    new PackedEntry { Name = "f_mafin2.sp2", Type = "image", Offset = 0x0BB77E, UnpackedSize = 0x73A, Size = 0x241, IsPacked = true },
                    new PackedEntry { Name = "f_mafin3.sp2", Type = "image", Offset = 0x0BB9BF, UnpackedSize = 0x632, Size = 0x0E1, IsPacked = true },
                    new PackedEntry { Name = "f_mafin4.sp2", Type = "image", Offset = 0x0BBAA0, UnpackedSize = 0x64A, Size = 0x42F, IsPacked = true },
                    new PackedEntry { Name = "f_mam.sp2", Type = "image", Offset = 0x0BBECF, UnpackedSize = 0x512, Size = 0x3B0, IsPacked = true },
                    new PackedEntry { Name = "f_maypl.sp2", Type = "image", Offset = 0x0BC27F, UnpackedSize = 0x842, Size = 0x168, IsPacked = true },
                    new PackedEntry { Name = "f_meido.sp2", Type = "image", Offset = 0x0BC3E7, UnpackedSize = 0x5FC, Size = 0x3FE, IsPacked = true },
                    new PackedEntry { Name = "f_milf1.sp2", Type = "image", Offset = 0x0BC7E5, UnpackedSize = 0x872, Size = 0x60D, IsPacked = true },
                    new PackedEntry { Name = "f_milf2.sp2", Type = "image", Offset = 0x0BCDF2, UnpackedSize = 0x650, Size = 0x4C5, IsPacked = true },
                    new PackedEntry { Name = "f_milf3.sp2", Type = "image", Offset = 0x0BD2B7, UnpackedSize = 0x8D2, Size = 0x0E0, IsPacked = true },
                    new PackedEntry { Name = "f_mitugo.sp2", Type = "image", Offset = 0x0BD397, UnpackedSize = 0x536, Size = 0x27D, IsPacked = true },
                    new PackedEntry { Name = "f_oba.sp2", Type = "image", Offset = 0x0BD614, UnpackedSize = 0x5BA, Size = 0x3EC, IsPacked = true },
                    new PackedEntry { Name = "f_panna.sp2", Type = "image", Offset = 0x0BDA00, UnpackedSize = 0x93E, Size = 0x2D4, IsPacked = true },
                    new PackedEntry { Name = "f_ponti.sp2", Type = "image", Offset = 0x0BDCD4, UnpackedSize = 0x78E, Size = 0x56D, IsPacked = true },
                    new PackedEntry { Name = "f_queen.sp2", Type = "image", Offset = 0x0BE241, UnpackedSize = 0x902, Size = 0x2C3, IsPacked = true },
                    new PackedEntry { Name = "f_riqur1.sp2", Type = "image", Offset = 0x0BE504, UnpackedSize = 0x6BC, Size = 0x4C7, IsPacked = true },
                    new PackedEntry { Name = "f_riqur2.sp2", Type = "image", Offset = 0x0BE9CB, UnpackedSize = 0x572, Size = 0x0D4, IsPacked = true },
                    new PackedEntry { Name = "f_russ1.sp2", Type = "image", Offset = 0x0BEA9F, UnpackedSize = 0x668, Size = 0x637, IsPacked = true },
                    new PackedEntry { Name = "f_russ2.sp2", Type = "image", Offset = 0x0BF0D6, UnpackedSize = 0x6FE, Size = 0x310, IsPacked = true },
                    new PackedEntry { Name = "f_sable.sp2", Type = "image", Offset = 0x0BF3E6, UnpackedSize = 0x761, Size = 0x33B, IsPacked = true },
                    new PackedEntry { Name = "f_sable2.sp2", Type = "image", Offset = 0x0BF721, UnpackedSize = 0x4F7, Size = 0x4FB, IsPacked = true },
                    new PackedEntry { Name = "f_selis.sp2", Type = "image", Offset = 0x0BFC1C, UnpackedSize = 0x74C, Size = 0x317, IsPacked = true },
                    new PackedEntry { Name = "f_selis1.sp2", Type = "image", Offset = 0x0BFF33, UnpackedSize = 0x74C, Size = 0x2A7, IsPacked = true },
                    new PackedEntry { Name = "f_selis2.sp2", Type = "image", Offset = 0x0C01DA, UnpackedSize = 0x74C, Size = 0x341, IsPacked = true },
                    new PackedEntry { Name = "f_selis3.sp2", Type = "image", Offset = 0x0C051B, UnpackedSize = 0x5AB, Size = 0x1A5, IsPacked = true },
                    new PackedEntry { Name = "f_shake.sp2", Type = "image", Offset = 0x0C06C0, UnpackedSize = 0x866, Size = 0x3CD, IsPacked = true },
                    new PackedEntry { Name = "f_shop1.sp2", Type = "image", Offset = 0x0C0A8D, UnpackedSize = 0x620, Size = 0x460, IsPacked = true },
                    new PackedEntry { Name = "f_shop3.sp2", Type = "image", Offset = 0x0C0EED, UnpackedSize = 0x620, Size = 0x460, IsPacked = true },
                    new PackedEntry { Name = "f_sipul1.sp2", Type = "image", Offset = 0x0C134D, UnpackedSize = 0x3EC, Size = 0x2F0, IsPacked = true },
                    new PackedEntry { Name = "f_sipul2.sp2", Type = "image", Offset = 0x0C163D, UnpackedSize = 0x6C2, Size = 0x0B5, IsPacked = true },
                    new PackedEntry { Name = "f_sipul3.sp2", Type = "image", Offset = 0x0C16F2, UnpackedSize = 0x626, Size = 0x4D3, IsPacked = true },
                    new PackedEntry { Name = "f_syoko.sp2", Type = "image", Offset = 0x0C1BC5, UnpackedSize = 0x5D8, Size = 0x2B8, IsPacked = true },
                    new PackedEntry { Name = "f_syoko2.sp2", Type = "image", Offset = 0x0C1E7D, UnpackedSize = 0x914, Size = 0x71B, IsPacked = true },
                    new PackedEntry { Name = "f_sys.sp2", Type = "image", Offset = 0x0C2598, UnpackedSize = 0x0D7, Size = 0x0DB, IsPacked = true },
                    new PackedEntry { Name = "f_sys2.sp2", Type = "image", Offset = 0x0C2673, UnpackedSize = 0x0B3, Size = 0x0B7, IsPacked = true },
                    new PackedEntry { Name = "f_syuku.sp2", Type = "image", Offset = 0x0C272A, UnpackedSize = 0x7B2, Size = 0x263, IsPacked = true },
                    new PackedEntry { Name = "f_taka.sp2", Type = "image", Offset = 0x0C298D, UnpackedSize = 0x6B, Size = 0x6F, IsPacked = true },
                    new PackedEntry { Name = "f_tira.sp2", Type = "image", Offset = 0x0C29FC, UnpackedSize = 0x79A, Size = 0x1C7, IsPacked = true },
                    new PackedEntry { Name = "f_tolte.sp2", Type = "image", Offset = 0x0C2BC3, UnpackedSize = 0x710, Size = 0x4C9, IsPacked = true },
                    new PackedEntry { Name = "f_tolte1.sp2", Type = "image", Offset = 0x0C308C, UnpackedSize = 0x6B, Size = 0x6F, IsPacked = true },
                    new PackedEntry { Name = "f_tryu.sp2", Type = "image", Offset = 0x0C30FB, UnpackedSize = 0x716, Size = 0x338, IsPacked = true },
                    new PackedEntry { Name = "f_uma.sp2", Type = "image", Offset = 0x0C3433, UnpackedSize = 0x7E2, Size = 0x670, IsPacked = true },
                    new PackedEntry { Name = "f_zaha.sp2", Type = "image", Offset = 0x0C3AA3, UnpackedSize = 0x5FC, Size = 0x44A, IsPacked = true },
                    new PackedEntry { Name = "g_cain.sp2", Type = "image", Offset = 0x0C3EED, UnpackedSize = 0x20AE, Size = 0x66E, IsPacked = true },
                    new PackedEntry { Name = "ha.sp2", Type = "image", Offset = 0x0C455B, UnpackedSize = 0x230, Size = 0x234, IsPacked = true },
                    new PackedEntry { Name = "kidan_a.sp2", Type = "image", Offset = 0x0C478F, UnpackedSize = 0x275, Size = 0x279, IsPacked = true },
                    new PackedEntry { Name = "kidan_b.sp2", Type = "image", Offset = 0x0C4A08, UnpackedSize = 0x1BB, Size = 0x1BF, IsPacked = true },
                    new PackedEntry { Name = "kidan_c.sp2", Type = "image", Offset = 0x0C4BC7, UnpackedSize = 0x266, Size = 0x26A, IsPacked = true },
                    new PackedEntry { Name = "kin2_a.sp2", Type = "image", Offset = 0x0C4E31, UnpackedSize = 0x156B, Size = 0x1162, IsPacked = true },
                    new PackedEntry { Name = "obj.sp2", Type = "image", Offset = 0x0C5F93, UnpackedSize = 0x56, Size = 0x5A, IsPacked = true },
                    new PackedEntry { Name = "start.sp2", Type = "image", Offset = 0x0C5FED, UnpackedSize = 0x29C, Size = 0x2A0, IsPacked = true },
                    new PackedEntry { Name = "s_cain.sp2", Type = "image", Offset = 0x0C628D, UnpackedSize = 0x33E, Size = 0x2CA, IsPacked = true },
                    new PackedEntry { Name = "x_sys.sp2", Type = "image", Offset = 0x0C6557, UnpackedSize = 0x68, Size = 0x33, IsPacked = true },
                    new PackedEntry { Name = "k_annin.kao", Type = "image", Offset = 0x0C658A, UnpackedSize = 0x0CC36, Size = 0x21D2, IsPacked = true },
                    new PackedEntry { Name = "k_clim1.kao", Type = "image", Offset = 0x0C875C, UnpackedSize = 0x0CC36, Size = 0x1DD5, IsPacked = true },
                    new PackedEntry { Name = "k_clim2.kao", Type = "image", Offset = 0x0CA531, UnpackedSize = 0x0CC36, Size = 0x16AC, IsPacked = true },
                    new PackedEntry { Name = "k_gana1.kao", Type = "image", Offset = 0x0CBBDD, UnpackedSize = 0x0CC36, Size = 0x2443, IsPacked = true },
                    new PackedEntry { Name = "k_gana2.kao", Type = "image", Offset = 0x0CE020, UnpackedSize = 0x0CC36, Size = 0x2573, IsPacked = true },
                    new PackedEntry { Name = "k_mafi1.kao", Type = "image", Offset = 0x0D0593, UnpackedSize = 0x0CC36, Size = 0x21F9, IsPacked = true },
                    new PackedEntry { Name = "k_mafi2.kao", Type = "image", Offset = 0x0D278C, UnpackedSize = 0x0CC36, Size = 0x234C, IsPacked = true },
                    new PackedEntry { Name = "k_mafin1.kao", Type = "image", Offset = 0x0D4AD8, UnpackedSize = 0x0CC36, Size = 0x21F9, IsPacked = true },
                    new PackedEntry { Name = "k_mafin2.kao", Type = "image", Offset = 0x0D6CD1, UnpackedSize = 0x0CC36, Size = 0x234C, IsPacked = true },
                    new PackedEntry { Name = "k_milf1.kao", Type = "image", Offset = 0x0D901D, UnpackedSize = 0x0CC36, Size = 0x3349, IsPacked = true },
                    new PackedEntry { Name = "k_milf2.kao", Type = "image", Offset = 0x0DC366, UnpackedSize = 0x0CC36, Size = 0x214A, IsPacked = true },
                    new PackedEntry { Name = "k_panna.kao", Type = "image", Offset = 0x0DE4B0, UnpackedSize = 0x0CC36, Size = 0x2CEB, IsPacked = true },
                    new PackedEntry { Name = "k_queen.kao", Type = "image", Offset = 0x0E119B, UnpackedSize = 0x0CC36, Size = 0x234A, IsPacked = true },
                    new PackedEntry { Name = "k_riqu1.kao", Type = "image", Offset = 0x0E34E5, UnpackedSize = 0x0CC36, Size = 0x22ED, IsPacked = true },
                    new PackedEntry { Name = "k_riqu2.kao", Type = "image", Offset = 0x0E57D2, UnpackedSize = 0x0CC36, Size = 0x1F3D, IsPacked = true },
                    new PackedEntry { Name = "k_riqur1.kao", Type = "image", Offset = 0x0E770F, UnpackedSize = 0x0CC36, Size = 0x2950, IsPacked = true },
                    new PackedEntry { Name = "k_riqur2.kao", Type = "image", Offset = 0x0EA05F, UnpackedSize = 0x0CC36, Size = 0x1EAF, IsPacked = true },
                    new PackedEntry { Name = "k_russ1.kao", Type = "image", Offset = 0x0EBF0E, UnpackedSize = 0x0CC36, Size = 0x2420, IsPacked = true },
                    new PackedEntry { Name = "k_russ2.kao", Type = "image", Offset = 0x0EE32E, UnpackedSize = 0x0CC36, Size = 0x2CE0, IsPacked = true },
                    new PackedEntry { Name = "k_seli1.kao", Type = "image", Offset = 0x0F100E, UnpackedSize = 0x0CC36, Size = 0x26FD, IsPacked = true },
                    new PackedEntry { Name = "k_seli2.kao", Type = "image", Offset = 0x0F370B, UnpackedSize = 0x0CC36, Size = 0x17E8, IsPacked = true },
                    new PackedEntry { Name = "k_selis1.kao", Type = "image", Offset = 0x0F4EF3, UnpackedSize = 0x0CC36, Size = 0x26FD, IsPacked = true },
                    new PackedEntry { Name = "k_selis2.kao", Type = "image", Offset = 0x0F75F0, UnpackedSize = 0x0CC36, Size = 0x17E8, IsPacked = true },
                    new PackedEntry { Name = "k_shake.kao", Type = "image", Offset = 0x0F8DD8, UnpackedSize = 0x0CC36, Size = 0x1755, IsPacked = true },
                    new PackedEntry { Name = "k_syoko.kao", Type = "image", Offset = 0x0FA52D, UnpackedSize = 0x0CC36, Size = 0x1FF6, IsPacked = true },
                    new PackedEntry { Name = "k_syuku.kao", Type = "image", Offset = 0x0FC523, UnpackedSize = 0x0CC36, Size = 0x2881, IsPacked = true },
                    new PackedEntry { Name = "k_tira.kao", Type = "image", Offset = 0x0FEDA4, UnpackedSize = 0x0CC36, Size = 0x1BF2, IsPacked = true },
                    new PackedEntry { Name = "k_tolte.kao", Type = "image", Offset = 0x100996, UnpackedSize = 0x0CC36, Size = 0x329B, IsPacked = true },
                }
            },
            {
                "ALLMAP.273", new List<Entry>() {
                    new PackedEntry { Name = "allmap.ma2", Offset = 0x0, UnpackedSize = 0x5DE0, Size = 0x1E4D, IsPacked = true },
                    new PackedEntry { Name = "bukiya.ma2", Offset = 0x1E4D, UnpackedSize = 0x0DCC, Size = 0x3B8, IsPacked = true },
                    new PackedEntry { Name = "cron00.ma2", Offset = 0x2205, UnpackedSize = 0x1240, Size = 0x2F5, IsPacked = true },
                    new PackedEntry { Name = "cudu00.ma2", Offset = 0x24FA, UnpackedSize = 0x6998, Size = 0x96F, IsPacked = true },
                    new PackedEntry { Name = "cudu01.ma2", Offset = 0x2E69, UnpackedSize = 0x4670, Size = 0x706, IsPacked = true },
                    new PackedEntry { Name = "cudu02.ma2", Offset = 0x356F, UnpackedSize = 0x4670, Size = 0x78F, IsPacked = true },
                    new PackedEntry { Name = "cudu03.ma2", Offset = 0x3CFE, UnpackedSize = 0x3AB8, Size = 0x392, IsPacked = true },
                    new PackedEntry { Name = "cudu04.ma2", Offset = 0x4090, UnpackedSize = 0x3AB8, Size = 0x2B2, IsPacked = true },
                    new PackedEntry { Name = "cudu05.ma2", Offset = 0x4342, UnpackedSize = 0x41C0, Size = 0x674, IsPacked = true },
                    new PackedEntry { Name = "cudu06.ma2", Offset = 0x49B6, UnpackedSize = 0x3AB8, Size = 0x58E, IsPacked = true },
                    new PackedEntry { Name = "cudu07.ma2", Offset = 0x4F44, UnpackedSize = 0x0FC0, Size = 0x3C4, IsPacked = true },
                    new PackedEntry { Name = "cudu08.ma2", Offset = 0x5308, UnpackedSize = 0x3AB8, Size = 0x51F, IsPacked = true },
                    new PackedEntry { Name = "cudu09.ma2", Offset = 0x5827, UnpackedSize = 0x11B4, Size = 0x27D, IsPacked = true },
                    new PackedEntry { Name = "cudu10.ma2", Offset = 0x5AA4, UnpackedSize = 0x0FC0, Size = 0x210, IsPacked = true },
                    new PackedEntry { Name = "doguya.ma2", Offset = 0x5CB4, UnpackedSize = 0x0DCC, Size = 0x393, IsPacked = true },
                    new PackedEntry { Name = "dokutu01.ma2", Offset = 0x6047, UnpackedSize = 0x1790, Size = 0x598, IsPacked = true },
                    new PackedEntry { Name = "dokutu02.ma2", Offset = 0x65DF, UnpackedSize = 0x1790, Size = 0x56A, IsPacked = true },
                    new PackedEntry { Name = "dokutu03.ma2", Offset = 0x6B49, UnpackedSize = 0x65B0, Size = 0x0B89, IsPacked = true },
                    new PackedEntry { Name = "dokutu04.ma2", Offset = 0x76D2, UnpackedSize = 0x44E0, Size = 0x0CD0, IsPacked = true },
                    new PackedEntry { Name = "dokutu05.ma2", Offset = 0x83A2, UnpackedSize = 0x5DE0, Size = 0x0C06, IsPacked = true },
                    new PackedEntry { Name = "dokutu06.ma2", Offset = 0x8FA8, UnpackedSize = 0x5DE0, Size = 0x0BD9, IsPacked = true },
                    new PackedEntry { Name = "dokutu07.ma2", Offset = 0x9B81, UnpackedSize = 0x6BF0, Size = 0x0FA2, IsPacked = true },
                    new PackedEntry { Name = "dokutu08.ma2", Offset = 0x0AB23, UnpackedSize = 0x1C40, Size = 0x5D9, IsPacked = true },
                    new PackedEntry { Name = "dokutu09.ma2", Offset = 0x0B0FC, UnpackedSize = 0x2668, Size = 0x73C, IsPacked = true },
                    new PackedEntry { Name = "flap00.ma2", Offset = 0x0B838, UnpackedSize = 0x2BE0, Size = 0x1134, IsPacked = true },
                    new PackedEntry { Name = "flap01.ma2", Offset = 0x0C96C, UnpackedSize = 0x82B4, Size = 0x1293, IsPacked = true },
                    new PackedEntry { Name = "flap02.ma2", Offset = 0x0DBFF, UnpackedSize = 0x75F0, Size = 0x0A08, IsPacked = true },
                    new PackedEntry { Name = "flap03.ma2", Offset = 0x0E607, UnpackedSize = 0x5DE0, Size = 0x836, IsPacked = true },
                    new PackedEntry { Name = "flap04.ma2", Offset = 0x0EE3D, UnpackedSize = 0x2668, Size = 0x66F, IsPacked = true },
                    new PackedEntry { Name = "flap05.ma2", Offset = 0x0F4AC, UnpackedSize = 0x6A60, Size = 0x20E0, IsPacked = true },
                    new PackedEntry { Name = "flap06.ma2", Offset = 0x1158C, UnpackedSize = 0x30F4, Size = 0x922, IsPacked = true },
                    new PackedEntry { Name = "flap07.ma2", Offset = 0x11EAE, UnpackedSize = 0x7EB0, Size = 0x0DC7, IsPacked = true },
                    new PackedEntry { Name = "flap08.ma2", Offset = 0x12C75, UnpackedSize = 0x36D0, Size = 0x0DE9, IsPacked = true },
                    new PackedEntry { Name = "flap09.ma2", Offset = 0x13A5E, UnpackedSize = 0x4670, Size = 0x16D7, IsPacked = true },
                    new PackedEntry { Name = "flap10.ma2", Offset = 0x15135, UnpackedSize = 0x2348, Size = 0x5AA, IsPacked = true },
                    new PackedEntry { Name = "flap11.ma2", Offset = 0x156DF, UnpackedSize = 0x1D6C, Size = 0x49F, IsPacked = true },
                    new PackedEntry { Name = "flap12.ma2", Offset = 0x15B7E, UnpackedSize = 0x1B78, Size = 0x638, IsPacked = true },
                    new PackedEntry { Name = "flap13.ma2", Offset = 0x161B6, UnpackedSize = 0x0FC0, Size = 0x444, IsPacked = true },
                    new PackedEntry { Name = "flap14.ma2", Offset = 0x165FA, UnpackedSize = 0x0FC0, Size = 0x353, IsPacked = true },
                    new PackedEntry { Name = "flap15.ma2", Offset = 0x1694D, UnpackedSize = 0x0FC0, Size = 0x34F, IsPacked = true },
                    new PackedEntry { Name = "flap16.ma2", Offset = 0x16C9C, UnpackedSize = 0x0FC0, Size = 0x36E, IsPacked = true },
                    new PackedEntry { Name = "flap17.ma2", Offset = 0x1700A, UnpackedSize = 0x0FC0, Size = 0x3C0, IsPacked = true },
                    new PackedEntry { Name = "flap18.ma2", Offset = 0x173CA, UnpackedSize = 0x1344, Size = 0x597, IsPacked = true },
                    new PackedEntry { Name = "fond00.ma2", Offset = 0x17961, UnpackedSize = 0x1F60, Size = 0x822, IsPacked = true },
                    new PackedEntry { Name = "fond01.ma2", Offset = 0x18183, UnpackedSize = 0x2730, Size = 0x93E, IsPacked = true },
                    new PackedEntry { Name = "fond02.ma2", Offset = 0x18AC1, UnpackedSize = 0x2F00, Size = 0x0B19, IsPacked = true },
                    new PackedEntry { Name = "fond03.ma2", Offset = 0x195DA, UnpackedSize = 0x1B78, Size = 0x79B, IsPacked = true },
                    new PackedEntry { Name = "fond04.ma2", Offset = 0x19D75, UnpackedSize = 0x0EF8, Size = 0x528, IsPacked = true },
                    new PackedEntry { Name = "fond05.ma2", Offset = 0x1A29D, UnpackedSize = 0x29BA, Size = 0x929, IsPacked = true },
                    new PackedEntry { Name = "fond06.ma2", Offset = 0x1ABC6, UnpackedSize = 0x2F00, Size = 0x0B76, IsPacked = true },
                    new PackedEntry { Name = "fond07.ma2", Offset = 0x1B73C, UnpackedSize = 0x1AD8, Size = 0x6E4, IsPacked = true },
                    new PackedEntry { Name = "fond08.ma2", Offset = 0x1BE20, UnpackedSize = 0x1984, Size = 0x8AF, IsPacked = true },
                    new PackedEntry { Name = "fond09.ma2", Offset = 0x1C6CF, UnpackedSize = 0x0B10, Size = 0x325, IsPacked = true },
                    new PackedEntry { Name = "garu01.ma2", Offset = 0x1C9F4, UnpackedSize = 0x0FC0, Size = 0x2E8, IsPacked = true },
                    new PackedEntry { Name = "garu02.ma2", Offset = 0x1CCDC, UnpackedSize = 0x2F00, Size = 0x7CF, IsPacked = true },
                    new PackedEntry { Name = "garu03.ma2", Offset = 0x1D4AB, UnpackedSize = 0x1F60, Size = 0x405, IsPacked = true },
                    new PackedEntry { Name = "garu04.ma2", Offset = 0x1D8B0, UnpackedSize = 0x2620, Size = 0x520, IsPacked = true },
                    new PackedEntry { Name = "garu05.ma2", Offset = 0x1DDD0, UnpackedSize = 0x30E0, Size = 0x70A, IsPacked = true },
                    new PackedEntry { Name = "garu06.ma2", Offset = 0x1E4DA, UnpackedSize = 0x3EA0, Size = 0x0A74, IsPacked = true },
                    new PackedEntry { Name = "garu07.ma2", Offset = 0x1EF4E, UnpackedSize = 0x4F20, Size = 0x0AD7, IsPacked = true },
                    new PackedEntry { Name = "garu08.ma2", Offset = 0x1FA25, UnpackedSize = 0x1600, Size = 0x392, IsPacked = true },
                    new PackedEntry { Name = "garu09.ma2", Offset = 0x1FDB7, UnpackedSize = 0x2620, Size = 0x5D9, IsPacked = true },
                    new PackedEntry { Name = "garu10.ma2", Offset = 0x20390, UnpackedSize = 0x2620, Size = 0x663, IsPacked = true },
                    new PackedEntry { Name = "hbmura01.ma2", Offset = 0x209F3, UnpackedSize = 0x5FD4, Size = 0x1F88, IsPacked = true },
                    new PackedEntry { Name = "isek00.ma2", Offset = 0x2297B, UnpackedSize = 0x3C20, Size = 0x0D05, IsPacked = true },
                    new PackedEntry { Name = "isek01.ma2", Offset = 0x23680, UnpackedSize = 0x500C, Size = 0x8AD, IsPacked = true },
                    new PackedEntry { Name = "isek02.ma2", Offset = 0x23F2D, UnpackedSize = 0x8130, Size = 0x0D57, IsPacked = true },
                    new PackedEntry { Name = "isek03.ma2", Offset = 0x24C84, UnpackedSize = 0x3470, Size = 0x697, IsPacked = true },
                    new PackedEntry { Name = "isek04.ma2", Offset = 0x2531B, UnpackedSize = 0x7870, Size = 0x0B6F, IsPacked = true },
                    new PackedEntry { Name = "isek05.ma2", Offset = 0x25E8A, UnpackedSize = 0x99E0, Size = 0x0D69, IsPacked = true },
                    new PackedEntry { Name = "isek06.ma2", Offset = 0x26BF3, UnpackedSize = 0x5E58, Size = 0x8E9, IsPacked = true },
                    new PackedEntry { Name = "jobin.ma2", Offset = 0x274DC, UnpackedSize = 0x1470, Size = 0x3A1, IsPacked = true },
                    new PackedEntry { Name = "jobuki.ma2", Offset = 0x2787D, UnpackedSize = 0x1E34, Size = 0x3E9, IsPacked = true },
                    new PackedEntry { Name = "jodog.ma2", Offset = 0x27C66, UnpackedSize = 0x159C, Size = 0x2E9, IsPacked = true },
                    new PackedEntry { Name = "jofuka1.ma2", Offset = 0x27F4F, UnpackedSize = 0x0CA0, Size = 0x320, IsPacked = true },
                    new PackedEntry { Name = "jofuka2.ma2", Offset = 0x2826F, UnpackedSize = 0x0CA0, Size = 0x2E8, IsPacked = true },
                    new PackedEntry { Name = "jofuka3.ma2", Offset = 0x28557, UnpackedSize = 0x1920, Size = 0x270, IsPacked = true },
                    new PackedEntry { Name = "joido.ma2", Offset = 0x287C7, UnpackedSize = 0x0FC0, Size = 0x1CC, IsPacked = true },
                    new PackedEntry { Name = "joisya.ma2", Offset = 0x28993, UnpackedSize = 0x2460, Size = 0x6AF, IsPacked = true },
                    new PackedEntry { Name = "jomin1.ma2", Offset = 0x29042, UnpackedSize = 0x159C, Size = 0x41E, IsPacked = true },
                    new PackedEntry { Name = "jomin2.ma2", Offset = 0x29460, UnpackedSize = 0x159C, Size = 0x3D3, IsPacked = true },
                    new PackedEntry { Name = "jouka.ma2", Offset = 0x29833, UnpackedSize = 0x99E0, Size = 0x1C02, IsPacked = true },
                    new PackedEntry { Name = "jouma.ma2", Offset = 0x2B435, UnpackedSize = 0x1880, Size = 0x4B3, IsPacked = true },
                    new PackedEntry { Name = "joyado1.ma2", Offset = 0x2B8E8, UnpackedSize = 0x25A0, Size = 0x5E4, IsPacked = true },
                    new PackedEntry { Name = "joyado2.ma2", Offset = 0x2BECC, UnpackedSize = 0x25A0, Size = 0x4CA, IsPacked = true },
                    new PackedEntry { Name = "kain1.ma2", Offset = 0x2C396, UnpackedSize = 0x1FC4, Size = 0x7E1, IsPacked = true },
                    new PackedEntry { Name = "kain2.ma2", Offset = 0x2CB77, UnpackedSize = 0x1FC4, Size = 0x676, IsPacked = true },
                    new PackedEntry { Name = "kaiz00.ma2", Offset = 0x2D1ED, UnpackedSize = 0x88D8, Size = 0x7EE, IsPacked = true },
                    new PackedEntry { Name = "kaiz01.ma2", Offset = 0x2D9DB, UnpackedSize = 0x1D6C, Size = 0x261, IsPacked = true },
                    new PackedEntry { Name = "kaiz02.ma2", Offset = 0x2DC3C, UnpackedSize = 0x1D6C, Size = 0x23C, IsPacked = true },
                    new PackedEntry { Name = "kaiz03.ma2", Offset = 0x2DE78, UnpackedSize = 0x893C, Size = 0x922, IsPacked = true },
                    new PackedEntry { Name = "kaiz04.ma2", Offset = 0x2E79A, UnpackedSize = 0x9EB8, Size = 0x0CCC, IsPacked = true },
                    new PackedEntry { Name = "kaiz05.ma2", Offset = 0x2F466, UnpackedSize = 0x6D80, Size = 0x9F6, IsPacked = true },
                    new PackedEntry { Name = "kaiz06.ma2", Offset = 0x2FE5C, UnpackedSize = 0x6D80, Size = 0x9F2, IsPacked = true },
                    new PackedEntry { Name = "kaiz07.ma2", Offset = 0x3084E, UnpackedSize = 0x2348, Size = 0x905, IsPacked = true },
                    new PackedEntry { Name = "katabuki.ma2", Offset = 0x31153, UnpackedSize = 0x1088, Size = 0x3DD, IsPacked = true },
                    new PackedEntry { Name = "katadogu.ma2", Offset = 0x31530, UnpackedSize = 0x0DCC, Size = 0x33F, IsPacked = true },
                    new PackedEntry { Name = "katamin1.ma2", Offset = 0x3186F, UnpackedSize = 0x1600, Size = 0x4BB, IsPacked = true },
                    new PackedEntry { Name = "katamin2.ma2", Offset = 0x31D2A, UnpackedSize = 0x1678, Size = 0x555, IsPacked = true },
                    new PackedEntry { Name = "katamina.ma2", Offset = 0x3227F, UnpackedSize = 0x1560, Size = 0x71C, IsPacked = true },
                    new PackedEntry { Name = "katania.ma2", Offset = 0x3299B, UnpackedSize = 0x7DE8, Size = 0x219D, IsPacked = true },
                    new PackedEntry { Name = "kataryo1.ma2", Offset = 0x34B38, UnpackedSize = 0x1678, Size = 0x56D, IsPacked = true },
                    new PackedEntry { Name = "kataryo2.ma2", Offset = 0x350A5, UnpackedSize = 0x1470, Size = 0x5EF, IsPacked = true },
                    new PackedEntry { Name = "katasou1.ma2", Offset = 0x35694, UnpackedSize = 0x0FC0, Size = 0x2EA, IsPacked = true },
                    new PackedEntry { Name = "katasou2.ma2", Offset = 0x3597E, UnpackedSize = 0x0FC0, Size = 0x31B, IsPacked = true },
                    new PackedEntry { Name = "katasyo.ma2", Offset = 0x35C99, UnpackedSize = 0x0E30, Size = 0x435, IsPacked = true },
                    new PackedEntry { Name = "katayad1.ma2", Offset = 0x360CE, UnpackedSize = 0x0DCC, Size = 0x3A9, IsPacked = true },
                    new PackedEntry { Name = "katayad2.ma2", Offset = 0x36477, UnpackedSize = 0x15A8, Size = 0x477, IsPacked = true },
                    new PackedEntry { Name = "koro00.ma2", Offset = 0x368EE, UnpackedSize = 0x14A0, Size = 0x51F, IsPacked = true },
                    new PackedEntry { Name = "koro01.ma2", Offset = 0x36E0D, UnpackedSize = 0x13A8, Size = 0x3FC, IsPacked = true },
                    new PackedEntry { Name = "koro02.ma2", Offset = 0x37209, UnpackedSize = 0x1920, Size = 0x2B7, IsPacked = true },
                    new PackedEntry { Name = "koro03.ma2", Offset = 0x374C0, UnpackedSize = 0x36D0, Size = 0x840, IsPacked = true },
                    new PackedEntry { Name = "kuro00.ma2", Offset = 0x37D00, UnpackedSize = 0x0FC0, Size = 0x548, IsPacked = true },
                    new PackedEntry { Name = "kuro01.ma2", Offset = 0x38248, UnpackedSize = 0x3C48, Size = 0x813, IsPacked = true },
                    new PackedEntry { Name = "kuro02.ma2", Offset = 0x38A5B, UnpackedSize = 0x3C48, Size = 0x8F2, IsPacked = true },
                    new PackedEntry { Name = "kuro03.ma2", Offset = 0x3934D, UnpackedSize = 0x5DE0, Size = 0x0E6E, IsPacked = true },
                    new PackedEntry { Name = "last00.ma2", Offset = 0x3A1BB, UnpackedSize = 0x0FC0, Size = 0x609, IsPacked = true },
                    new PackedEntry { Name = "last01.ma2", Offset = 0x3A7C4, UnpackedSize = 0x12E0, Size = 0x223, IsPacked = true },
                    new PackedEntry { Name = "last02.ma2", Offset = 0x3A9E7, UnpackedSize = 0x2348, Size = 0x2EC, IsPacked = true },
                    new PackedEntry { Name = "last03.ma2", Offset = 0x3ACD3, UnpackedSize = 0x19E8, Size = 0x203, IsPacked = true },
                    new PackedEntry { Name = "last04.ma2", Offset = 0x3AED6, UnpackedSize = 0x2348, Size = 0x2D0, IsPacked = true },
                    new PackedEntry { Name = "last05.ma2", Offset = 0x3B1A6, UnpackedSize = 0x32C0, Size = 0x341, IsPacked = true },
                    new PackedEntry { Name = "last06.ma2", Offset = 0x3B4E7, UnpackedSize = 0x52F0, Size = 0x384, IsPacked = true },
                    new PackedEntry { Name = "last07.ma2", Offset = 0x3B86B, UnpackedSize = 0x1BA0, Size = 0x226, IsPacked = true },
                    new PackedEntry { Name = "last08.ma2", Offset = 0x3BA91, UnpackedSize = 0x19E8, Size = 0x216, IsPacked = true },
                    new PackedEntry { Name = "last09.ma2", Offset = 0x3BCA7, UnpackedSize = 0x19E8, Size = 0x216, IsPacked = true },
                    new PackedEntry { Name = "last10.ma2", Offset = 0x3BEBD, UnpackedSize = 0x19E8, Size = 0x1FD, IsPacked = true },
                    new PackedEntry { Name = "last11.ma2", Offset = 0x3C0BA, UnpackedSize = 0x2B18, Size = 0x898, IsPacked = true },
                    new PackedEntry { Name = "last12.ma2", Offset = 0x3C952, UnpackedSize = 0x13A8, Size = 0x444, IsPacked = true },
                    new PackedEntry { Name = "last13.ma2", Offset = 0x3CD96, UnpackedSize = 0x4E40, Size = 0x691, IsPacked = true },
                    new PackedEntry { Name = "last14.ma2", Offset = 0x3D427, UnpackedSize = 0x15EC, Size = 0x556, IsPacked = true },
                    new PackedEntry { Name = "last15.ma2", Offset = 0x3D97D, UnpackedSize = 0x5DE0, Size = 0x922, IsPacked = true },
                    new PackedEntry { Name = "last16.ma2", Offset = 0x3E29F, UnpackedSize = 0x1FC4, Size = 0x65E, IsPacked = true },
                    new PackedEntry { Name = "last17.ma2", Offset = 0x3E8FD, UnpackedSize = 0x13A8, Size = 0x3F8, IsPacked = true },
                    new PackedEntry { Name = "last18.ma2", Offset = 0x3ECF5, UnpackedSize = 0x1FC4, Size = 0x62D, IsPacked = true },
                    new PackedEntry { Name = "last19.ma2", Offset = 0x3F322, UnpackedSize = 0x2730, Size = 0x6C8, IsPacked = true },
                    new PackedEntry { Name = "last20.ma2", Offset = 0x3F9EA, UnpackedSize = 0x2BE0, Size = 0x50C, IsPacked = true },
                    new PackedEntry { Name = "last21.ma2", Offset = 0x3FEF6, UnpackedSize = 0x0FC0, Size = 0x554, IsPacked = true },
                    new PackedEntry { Name = "last22.ma2", Offset = 0x4044A, UnpackedSize = 0x0FC0, Size = 0x562, IsPacked = true },
                    new PackedEntry { Name = "last23.ma2", Offset = 0x409AC, UnpackedSize = 0x0FC0, Size = 0x26A, IsPacked = true },
                    new PackedEntry { Name = "last24.ma2", Offset = 0x40C16, UnpackedSize = 0x0FC0, Size = 0x468, IsPacked = true },
                    new PackedEntry { Name = "matu00.ma2", Offset = 0x4107E, UnpackedSize = 0x99E0, Size = 0x1BE6, IsPacked = true },
                    new PackedEntry { Name = "ojou01.ma2", Offset = 0x42C64, UnpackedSize = 0x63A8, Size = 0x1146, IsPacked = true },
                    new PackedEntry { Name = "ojou02.ma2", Offset = 0x43DAA, UnpackedSize = 0x4A80, Size = 0x6F0, IsPacked = true },
                    new PackedEntry { Name = "ojou03.ma2", Offset = 0x4449A, UnpackedSize = 0x3F04, Size = 0x0AE9, IsPacked = true },
                    new PackedEntry { Name = "ojou04.ma2", Offset = 0x44F83, UnpackedSize = 0x3C20, Size = 0x0C20, IsPacked = true },
                    new PackedEntry { Name = "ojou05.ma2", Offset = 0x45BA3, UnpackedSize = 0x1F60, Size = 0x5B2, IsPacked = true },
                    new PackedEntry { Name = "ojou06.ma2", Offset = 0x46155, UnpackedSize = 0x2D20, Size = 0x471, IsPacked = true },
                    new PackedEntry { Name = "ojou07.ma2", Offset = 0x465C6, UnpackedSize = 0x5610, Size = 0x74A, IsPacked = true },
                    new PackedEntry { Name = "ojou08.ma2", Offset = 0x46D10, UnpackedSize = 0x5610, Size = 0x810, IsPacked = true },
                    new PackedEntry { Name = "ojou09.ma2", Offset = 0x47520, UnpackedSize = 0x5610, Size = 0x99E, IsPacked = true },
                    new PackedEntry { Name = "ojou10.ma2", Offset = 0x47EBE, UnpackedSize = 0x5610, Size = 0x9A0, IsPacked = true },
                    new PackedEntry { Name = "ojou11.ma2", Offset = 0x4885E, UnpackedSize = 0x5610, Size = 0x859, IsPacked = true },
                    new PackedEntry { Name = "ojou12.ma2", Offset = 0x490B7, UnpackedSize = 0x5610, Size = 0x8B3, IsPacked = true },
                    new PackedEntry { Name = "ojou13.ma2", Offset = 0x4996A, UnpackedSize = 0x5610, Size = 0x941, IsPacked = true },
                    new PackedEntry { Name = "ojou14.ma2", Offset = 0x4A2AB, UnpackedSize = 0x1920, Size = 0x229, IsPacked = true },
                    new PackedEntry { Name = "ojou15.ma2", Offset = 0x4A4D4, UnpackedSize = 0x5610, Size = 0x722, IsPacked = true },
                    new PackedEntry { Name = "ojou16.ma2", Offset = 0x4ABF6, UnpackedSize = 0x5610, Size = 0x7D7, IsPacked = true },
                    new PackedEntry { Name = "ojou17.ma2", Offset = 0x4B3CD, UnpackedSize = 0x5610, Size = 0x7CF, IsPacked = true },
                    new PackedEntry { Name = "ojou18.ma2", Offset = 0x4BB9C, UnpackedSize = 0x5610, Size = 0x73A, IsPacked = true },
                    new PackedEntry { Name = "ojou19.ma2", Offset = 0x4C2D6, UnpackedSize = 0x5610, Size = 0x707, IsPacked = true },
                    new PackedEntry { Name = "ojou20.ma2", Offset = 0x4C9DD, UnpackedSize = 0x5610, Size = 0x5F8, IsPacked = true },
                    new PackedEntry { Name = "ojou21.ma2", Offset = 0x4CFD5, UnpackedSize = 0x2E9C, Size = 0x34C, IsPacked = true },
                    new PackedEntry { Name = "ojou22.ma2", Offset = 0x4D321, UnpackedSize = 0x5610, Size = 0x869, IsPacked = true },
                    new PackedEntry { Name = "oldw00.ma2", Offset = 0x4DB8A, UnpackedSize = 0x0B10, Size = 0x25D, IsPacked = true },
                    new PackedEntry { Name = "oldw01.ma2", Offset = 0x4DDE7, UnpackedSize = 0x4670, Size = 0x122D, IsPacked = true },
                    new PackedEntry { Name = "oldw02.ma2", Offset = 0x4F014, UnpackedSize = 0x12E0, Size = 0x347, IsPacked = true },
                    new PackedEntry { Name = "oldw03.ma2", Offset = 0x4F35B, UnpackedSize = 0x0A70, Size = 0x25C, IsPacked = true },
                    new PackedEntry { Name = "oldw04.ma2", Offset = 0x4F5B7, UnpackedSize = 0x0A70, Size = 0x269, IsPacked = true },
                    new PackedEntry { Name = "oldw05.ma2", Offset = 0x4F820, UnpackedSize = 0x1088, Size = 0x27C, IsPacked = true },
                    new PackedEntry { Name = "oldw06.ma2", Offset = 0x4FA9C, UnpackedSize = 0x48A0, Size = 0x0E99, IsPacked = true },
                    new PackedEntry { Name = "oldw07.ma2", Offset = 0x50935, UnpackedSize = 0x0AC0, Size = 0x275, IsPacked = true },
                    new PackedEntry { Name = "oldw08.ma2", Offset = 0x50BAA, UnpackedSize = 0x0AC0, Size = 0x2CF, IsPacked = true },
                    new PackedEntry { Name = "oldw09.ma2", Offset = 0x50E79, UnpackedSize = 0x0AC0, Size = 0x286, IsPacked = true },
                    new PackedEntry { Name = "oldw10.ma2", Offset = 0x510FF, UnpackedSize = 0x19E8, Size = 0x41A, IsPacked = true },
                    new PackedEntry { Name = "oldw11.ma2", Offset = 0x51519, UnpackedSize = 0x2348, Size = 0x4B9, IsPacked = true },
                    new PackedEntry { Name = "oldw12.ma2", Offset = 0x519D2, UnpackedSize = 0x7F0, Size = 0x4DF, IsPacked = true },
                    new PackedEntry { Name = "omake.ma2", Offset = 0x51EB1, UnpackedSize = 0x7F0, Size = 0x2C6, IsPacked = true },
                    new PackedEntry { Name = "omake2.ma2", Offset = 0x52177, UnpackedSize = 0x7F0, Size = 0x2B9, IsPacked = true },
                    new PackedEntry { Name = "paremina.ma2", Offset = 0x52430, UnpackedSize = 0x1F60, Size = 0x7D1, IsPacked = true },
                    new PackedEntry { Name = "parerumo.ma2", Offset = 0x52C01, UnpackedSize = 0x8DBA, Size = 0x20B6, IsPacked = true },
                    new PackedEntry { Name = "pargil.ma2", Offset = 0x54CB7, UnpackedSize = 0x2280, Size = 0x3F6, IsPacked = true },
                    new PackedEntry { Name = "parsaka.ma2", Offset = 0x550AD, UnpackedSize = 0x2280, Size = 0x40C, IsPacked = true },
                    new PackedEntry { Name = "paruma.ma2", Offset = 0x554B9, UnpackedSize = 0x1880, Size = 0x4B3, IsPacked = true },
                    new PackedEntry { Name = "paryad1.ma2", Offset = 0x5596C, UnpackedSize = 0x1B32, Size = 0x552, IsPacked = true },
                    new PackedEntry { Name = "paryad2.ma2", Offset = 0x55EBE, UnpackedSize = 0x1B32, Size = 0x3F9, IsPacked = true },
                    new PackedEntry { Name = "saba00.ma2", Offset = 0x562B7, UnpackedSize = 0x3AB8, Size = 0x826, IsPacked = true },
                    new PackedEntry { Name = "saba01.ma2", Offset = 0x56ADD, UnpackedSize = 0x2348, Size = 0x53D, IsPacked = true },
                    new PackedEntry { Name = "saba02.ma2", Offset = 0x5701A, UnpackedSize = 0x4670, Size = 0x7AD, IsPacked = true },
                    new PackedEntry { Name = "saba03.ma2", Offset = 0x577C7, UnpackedSize = 0x3EA0, Size = 0x5A1, IsPacked = true },
                    new PackedEntry { Name = "saba04.ma2", Offset = 0x57D68, UnpackedSize = 0x1C40, Size = 0x4B8, IsPacked = true },
                    new PackedEntry { Name = "sakaba.ma2", Offset = 0x58220, UnpackedSize = 0x0B60, Size = 0x42B, IsPacked = true },
                    new PackedEntry { Name = "sepu00.ma2", Offset = 0x5864B, UnpackedSize = 0x4F3A, Size = 0x1378, IsPacked = true },
                    new PackedEntry { Name = "sepu01.ma2", Offset = 0x599C3, UnpackedSize = 0x980, Size = 0x252, IsPacked = true },
                    new PackedEntry { Name = "sepu02.ma2", Offset = 0x59C15, UnpackedSize = 0x8E0, Size = 0x24D, IsPacked = true },
                    new PackedEntry { Name = "sepu03.ma2", Offset = 0x59E62, UnpackedSize = 0x980, Size = 0x26E, IsPacked = true },
                    new PackedEntry { Name = "sepu04.ma2", Offset = 0x5A0D0, UnpackedSize = 0x980, Size = 0x234, IsPacked = true },
                    new PackedEntry { Name = "sepu05.ma2", Offset = 0x5A304, UnpackedSize = 0x1AB0, Size = 0x4A5, IsPacked = true },
                    new PackedEntry { Name = "sepu06.ma2", Offset = 0x5A7A9, UnpackedSize = 0x16A0, Size = 0x3AF, IsPacked = true },
                    new PackedEntry { Name = "sepu07.ma2", Offset = 0x5AB58, UnpackedSize = 0x6290, Size = 0x718, IsPacked = true },
                    new PackedEntry { Name = "sepu08.ma2", Offset = 0x5B270, UnpackedSize = 0x3200, Size = 0x3C5, IsPacked = true },
                    new PackedEntry { Name = "sepu09.ma2", Offset = 0x5B635, UnpackedSize = 0x25A0, Size = 0x443, IsPacked = true },
                    new PackedEntry { Name = "sepu10.ma2", Offset = 0x5BA78, UnpackedSize = 0x5740, Size = 0x674, IsPacked = true },
                    new PackedEntry { Name = "sepu11.ma2", Offset = 0x5C0EC, UnpackedSize = 0x3B80, Size = 0x457, IsPacked = true },
                    new PackedEntry { Name = "sepu12.ma2", Offset = 0x5C543, UnpackedSize = 0x32C0, Size = 0x5CF, IsPacked = true },
                    new PackedEntry { Name = "sepu13.ma2", Offset = 0x5CB12, UnpackedSize = 0x3AB8, Size = 0x387, IsPacked = true },
                    new PackedEntry { Name = "sepu14.ma2", Offset = 0x5CE99, UnpackedSize = 0x2BE0, Size = 0x69E, IsPacked = true },
                    new PackedEntry { Name = "sima00.ma2", Offset = 0x5D537, UnpackedSize = 0x0FC0, Size = 0x548, IsPacked = true },
                    new PackedEntry { Name = "sima01.ma2", Offset = 0x5DA7F, UnpackedSize = 0x5DE0, Size = 0x77A, IsPacked = true },
                    new PackedEntry { Name = "sima02.ma2", Offset = 0x5E1F9, UnpackedSize = 0x4E40, Size = 0x70F, IsPacked = true },
                    new PackedEntry { Name = "sima03.ma2", Offset = 0x5E908, UnpackedSize = 0x4E40, Size = 0x634, IsPacked = true },
                    new PackedEntry { Name = "sima04.ma2", Offset = 0x5EF3C, UnpackedSize = 0x5DE0, Size = 0x7E5, IsPacked = true },
                    new PackedEntry { Name = "sima05.ma2", Offset = 0x5F721, UnpackedSize = 0x5DE0, Size = 0x86D, IsPacked = true },
                    new PackedEntry { Name = "sima06.ma2", Offset = 0x5FF8E, UnpackedSize = 0x1F60, Size = 0x486, IsPacked = true },
                    new PackedEntry { Name = "sima07.ma2", Offset = 0x60414, UnpackedSize = 0x0CA0, Size = 0x1DD, IsPacked = true },
                    new PackedEntry { Name = "sougen.ma2", Offset = 0x605F1, UnpackedSize = 0x0FC0, Size = 0x604, IsPacked = true },
                    new PackedEntry { Name = "syoko01.ma2", Offset = 0x60BF5, UnpackedSize = 0x3DA6, Size = 0x0B2E, IsPacked = true },
                    new PackedEntry { Name = "syoko02.ma2", Offset = 0x61723, UnpackedSize = 0x1CD6, Size = 0x7F8, IsPacked = true },
                    new PackedEntry { Name = "syud00.ma2", Offset = 0x61F1B, UnpackedSize = 0x0B60, Size = 0x4F1, IsPacked = true },
                    new PackedEntry { Name = "syud01.ma2", Offset = 0x6240C, UnpackedSize = 0x5480, Size = 0x0E12, IsPacked = true },
                    new PackedEntry { Name = "tidu00.ma2", Offset = 0x6321E, UnpackedSize = 0x5FD4, Size = 0x103A, IsPacked = true },
                    new PackedEntry { Name = "tidu01.ma2", Offset = 0x64258, UnpackedSize = 0x8B30, Size = 0x0BB0, IsPacked = true },
                    new PackedEntry { Name = "tidu02.ma2", Offset = 0x64E08, UnpackedSize = 0x8298, Size = 0x1008, IsPacked = true },
                    new PackedEntry { Name = "tidu03.ma2", Offset = 0x65E10, UnpackedSize = 0x70A0, Size = 0x0B70, IsPacked = true },
                    new PackedEntry { Name = "tidu04.ma2", Offset = 0x66980, UnpackedSize = 0x89A0, Size = 0x0E08, IsPacked = true },
                    new PackedEntry { Name = "tidu05.ma2", Offset = 0x67788, UnpackedSize = 0x8EA0, Size = 0x107A, IsPacked = true },
                    new PackedEntry { Name = "tita00.ma2", Offset = 0x68802, UnpackedSize = 0x4670, Size = 0x1055, IsPacked = true },
                    new PackedEntry { Name = "tita01.ma2", Offset = 0x69857, UnpackedSize = 0x7820, Size = 0x984, IsPacked = true },
                    new PackedEntry { Name = "tita02.ma2", Offset = 0x6A1DB, UnpackedSize = 0x7820, Size = 0x0A07, IsPacked = true },
                    new PackedEntry { Name = "tita03.ma2", Offset = 0x6ABE2, UnpackedSize = 0x7820, Size = 0x967, IsPacked = true },
                    new PackedEntry { Name = "tita04.ma2", Offset = 0x6B549, UnpackedSize = 0x7820, Size = 0x9D2, IsPacked = true },
                    new PackedEntry { Name = "tita05.ma2", Offset = 0x6BF1B, UnpackedSize = 0x7820, Size = 0x917, IsPacked = true },
                    new PackedEntry { Name = "tita06.ma2", Offset = 0x6C832, UnpackedSize = 0x7820, Size = 0x9EE, IsPacked = true },
                    new PackedEntry { Name = "tita07.ma2", Offset = 0x6D220, UnpackedSize = 0x2F00, Size = 0x361, IsPacked = true },
                    new PackedEntry { Name = "tori00.ma2", Offset = 0x6D581, UnpackedSize = 0x5FD4, Size = 0x1B21, IsPacked = true },
                    new PackedEntry { Name = "tori01.ma2", Offset = 0x6F0A2, UnpackedSize = 0x0DCC, Size = 0x3A8, IsPacked = true },
                    new PackedEntry { Name = "tori02.ma2", Offset = 0x6F44A, UnpackedSize = 0x0DCC, Size = 0x3F3, IsPacked = true },
                    new PackedEntry { Name = "tori03.ma2", Offset = 0x6F83D, UnpackedSize = 0x0DCC, Size = 0x38E, IsPacked = true },
                    new PackedEntry { Name = "tori04.ma2", Offset = 0x6FBCB, UnpackedSize = 0x0B10, Size = 0x1E2, IsPacked = true },
                    new PackedEntry { Name = "tori05.ma2", Offset = 0x6FDAD, UnpackedSize = 0x1C40, Size = 0x4C6, IsPacked = true },
                    new PackedEntry { Name = "tori06.ma2", Offset = 0x70273, UnpackedSize = 0x20F0, Size = 0x6CF, IsPacked = true },
                    new PackedEntry { Name = "tori07.ma2", Offset = 0x70942, UnpackedSize = 0x0D40, Size = 0x2B5, IsPacked = true },
                    new PackedEntry { Name = "tori08.ma2", Offset = 0x70BF7, UnpackedSize = 0x0B10, Size = 0x2B3, IsPacked = true },
                    new PackedEntry { Name = "yama01.ma2", Offset = 0x70EAA, UnpackedSize = 0x1F60, Size = 0x65B, IsPacked = true },
                    new PackedEntry { Name = "yama02.ma2", Offset = 0x71505, UnpackedSize = 0x9DF0, Size = 0x1C46, IsPacked = true },
                    new PackedEntry { Name = "yama03.ma2", Offset = 0x7314B, UnpackedSize = 0x4FD0, Size = 0x8E0, IsPacked = true },
                    new PackedEntry { Name = "yama04.ma2", Offset = 0x73A2B, UnpackedSize = 0x5DE0, Size = 0x0AF6, IsPacked = true },
                    new PackedEntry { Name = "yama05.ma2", Offset = 0x74521, UnpackedSize = 0x6290, Size = 0x0F38, IsPacked = true },
                    new PackedEntry { Name = "yama06.ma2", Offset = 0x75459, UnpackedSize = 0x2730, Size = 0x0A47, IsPacked = true },
                    new PackedEntry { Name = "yujin.ma2", Offset = 0x75EA0, UnpackedSize = 0x20F0, Size = 0x701, IsPacked = true },
                    new PackedEntry { Name = "yuru00.ma2", Offset = 0x765A1, UnpackedSize = 0x7D20, Size = 0x1DCD, IsPacked = true },
                    new PackedEntry { Name = "yuru01.ma2", Offset = 0x7836E, UnpackedSize = 0x159C, Size = 0x51D, IsPacked = true },
                    new PackedEntry { Name = "yuru02.ma2", Offset = 0x7888B, UnpackedSize = 0x1376, Size = 0x4B3, IsPacked = true },
                    new PackedEntry { Name = "yuru03.ma2", Offset = 0x78D3E, UnpackedSize = 0x1880, Size = 0x4B3, IsPacked = true },
                    new PackedEntry { Name = "yuru04.ma2", Offset = 0x791F1, UnpackedSize = 0x159C, Size = 0x4BC, IsPacked = true },
                    new PackedEntry { Name = "yuru05.ma2", Offset = 0x796AD, UnpackedSize = 0x2280, Size = 0x7F6, IsPacked = true },
                    new PackedEntry { Name = "yuru06.ma2", Offset = 0x79EA3, UnpackedSize = 0x159C, Size = 0x4EA, IsPacked = true },
                    new PackedEntry { Name = "yuru07.ma2", Offset = 0x7A38D, UnpackedSize = 0x2B18, Size = 0x85E, IsPacked = true },
                    new PackedEntry { Name = "yuru08.ma2", Offset = 0x7ABEB, UnpackedSize = 0x3860, Size = 0x7E5, IsPacked = true },
                    new PackedEntry { Name = "yuru09.ma2", Offset = 0x7B3D0, UnpackedSize = 0x2348, Size = 0x547, IsPacked = true },
                    new PackedEntry { Name = "allmap.evt", Offset = 0x7B917, UnpackedSize = 0x11A, Size = 0x0FE, IsPacked = true },
                    new PackedEntry { Name = "allmap2.evt", Offset = 0x7BA15, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "bukiya.evt", Offset = 0x7BA1B, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "cron00.evt", Offset = 0x7BA2B, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "cudu00.evt", Offset = 0x7BA3B, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "cudu01.evt", Offset = 0x7BA73, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "cudu02.evt", Offset = 0x7BA8D, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "cudu03.evt", Offset = 0x7BAA7, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "cudu04.evt", Offset = 0x7BADF, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "cudu05.evt", Offset = 0x7BB17, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "cudu06.evt", Offset = 0x7BB3B, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "cudu07.evt", Offset = 0x7BB73, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "cudu08.evt", Offset = 0x7BBAB, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "cudu09.evt", Offset = 0x7BBE3, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "cudu10.evt", Offset = 0x7BBF3, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "doguya.evt", Offset = 0x7BC0D, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "dokutu01.evt", Offset = 0x7BC1D, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "dokutu02.evt", Offset = 0x7BC37, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "dokutu03.evt", Offset = 0x7BC47, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "dokutu04.evt", Offset = 0x7BC75, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "dokutu05.evt", Offset = 0x7BC8F, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "dokutu06.evt", Offset = 0x7BCA9, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "dokutu07.evt", Offset = 0x7BCCD, UnpackedSize = 0x34, Size = 0x31, IsPacked = true },
                    new PackedEntry { Name = "dokutu08.evt", Offset = 0x7BCFE, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "dokutu09.evt", Offset = 0x7BD0E, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "flap00.evt", Offset = 0x7BD3C, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "flap01.evt", Offset = 0x7BD6A, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "flap02.evt", Offset = 0x7BDA2, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "flap03.evt", Offset = 0x7BDDA, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "flap04.evt", Offset = 0x7BE08, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "flap05.evt", Offset = 0x7BE2C, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "flap06.evt", Offset = 0x7BE64, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "flap07.evt", Offset = 0x7BE7E, UnpackedSize = 0x48, Size = 0x4C, IsPacked = true },
                    new PackedEntry { Name = "flap08.evt", Offset = 0x7BECA, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "flap09.evt", Offset = 0x7BEE4, UnpackedSize = 0x48, Size = 0x4C, IsPacked = true },
                    new PackedEntry { Name = "flap10.evt", Offset = 0x7BF30, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "flap11.evt", Offset = 0x7BF4A, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "flap12.evt", Offset = 0x7BF5A, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "flap13.evt", Offset = 0x7BF74, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "flap14.evt", Offset = 0x7BF8E, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "flap15.evt", Offset = 0x7BFA8, UnpackedSize = 0x34, Size = 0x30, IsPacked = true },
                    new PackedEntry { Name = "flap16.evt", Offset = 0x7BFD8, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "flap17.evt", Offset = 0x7BFE8, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "flap18.evt", Offset = 0x7BFF8, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "fond00.evt", Offset = 0x7C012, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "fond01.evt", Offset = 0x7C040, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "fond02.evt", Offset = 0x7C064, UnpackedSize = 0x3E, Size = 0x42, IsPacked = true },
                    new PackedEntry { Name = "fond03.evt", Offset = 0x7C0A6, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "fond04.evt", Offset = 0x7C0D4, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "fond05.evt", Offset = 0x7C0F8, UnpackedSize = 0x3E, Size = 0x42, IsPacked = true },
                    new PackedEntry { Name = "fond06.evt", Offset = 0x7C13A, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "fond07.evt", Offset = 0x7C15E, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "fond08.evt", Offset = 0x7C178, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "fond09.evt", Offset = 0x7C19C, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "garu01.evt", Offset = 0x7C1AC, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "garu02.evt", Offset = 0x7C1D0, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "garu03.evt", Offset = 0x7C1F4, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "garu04.evt", Offset = 0x7C218, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "garu05.evt", Offset = 0x7C23C, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "garu06.evt", Offset = 0x7C26A, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "garu07.evt", Offset = 0x7C298, UnpackedSize = 0x3E, Size = 0x42, IsPacked = true },
                    new PackedEntry { Name = "garu08.evt", Offset = 0x7C2DA, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "garu09.evt", Offset = 0x7C2FE, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "garu10.evt", Offset = 0x7C32C, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "hbmura01.evt", Offset = 0x7C35A, UnpackedSize = 0x48, Size = 0x4C, IsPacked = true },
                    new PackedEntry { Name = "isek00.evt", Offset = 0x7C3A6, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "isek01.evt", Offset = 0x7C3D4, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "isek02.evt", Offset = 0x7C402, UnpackedSize = 0x3E, Size = 0x42, IsPacked = true },
                    new PackedEntry { Name = "isek03.evt", Offset = 0x7C444, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "isek04.evt", Offset = 0x7C47C, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "isek05.evt", Offset = 0x7C4B4, UnpackedSize = 0x3E, Size = 0x42, IsPacked = true },
                    new PackedEntry { Name = "isek06.evt", Offset = 0x7C4F6, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "jobin.evt", Offset = 0x7C52E, UnpackedSize = 0x70, Size = 0x6F, IsPacked = true },
                    new PackedEntry { Name = "jobuki.evt", Offset = 0x7C59D, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "jodog.evt", Offset = 0x7C5B7, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "jofuka1.evt", Offset = 0x7C5C7, UnpackedSize = 0x8E, Size = 0x8B, IsPacked = true },
                    new PackedEntry { Name = "jofuka2.evt", Offset = 0x7C652, UnpackedSize = 0x8E, Size = 0x8B, IsPacked = true },
                    new PackedEntry { Name = "jofuka3.evt", Offset = 0x7C6DD, UnpackedSize = 0x66, Size = 0x6A, IsPacked = true },
                    new PackedEntry { Name = "joido.evt", Offset = 0x7C747, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "joisya.evt", Offset = 0x7C761, UnpackedSize = 0x70, Size = 0x6F, IsPacked = true },
                    new PackedEntry { Name = "jomin1.evt", Offset = 0x7C7D0, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "jomin2.evt", Offset = 0x7C7E0, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "jouka.evt", Offset = 0x7C7F0, UnpackedSize = 0x0AC, Size = 0x0A8, IsPacked = true },
                    new PackedEntry { Name = "jouma.evt", Offset = 0x7C898, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "joyado1.evt", Offset = 0x7C8B2, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "joyado2.evt", Offset = 0x7C8D6, UnpackedSize = 0x2A, Size = 0x29, IsPacked = true },
                    new PackedEntry { Name = "kain1.evt", Offset = 0x7C8FF, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "kain2.evt", Offset = 0x7C919, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "kaiz00.evt", Offset = 0x7C933, UnpackedSize = 0x5C, Size = 0x60, IsPacked = true },
                    new PackedEntry { Name = "kaiz01.evt", Offset = 0x7C993, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "kaiz02.evt", Offset = 0x7C9A3, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "kaiz03.evt", Offset = 0x7C9BD, UnpackedSize = 0x48, Size = 0x4C, IsPacked = true },
                    new PackedEntry { Name = "kaiz04.evt", Offset = 0x7CA09, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "kaiz05.evt", Offset = 0x7CA41, UnpackedSize = 0x48, Size = 0x4C, IsPacked = true },
                    new PackedEntry { Name = "kaiz06.evt", Offset = 0x7CA8D, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "kaiz07.evt", Offset = 0x7CAC5, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "katabuki.evt", Offset = 0x7CAE9, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "katadogu.evt", Offset = 0x7CB03, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "katamin1.evt", Offset = 0x7CB13, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "katamin2.evt", Offset = 0x7CB23, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "katamina.evt", Offset = 0x7CB33, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "katania.evt", Offset = 0x7CB61, UnpackedSize = 0x5C, Size = 0x60, IsPacked = true },
                    new PackedEntry { Name = "kataryo1.evt", Offset = 0x7CBC1, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "kataryo2.evt", Offset = 0x7CBE5, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "katasou1.evt", Offset = 0x7CBFF, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "katasou2.evt", Offset = 0x7CC0F, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "katasyo.evt", Offset = 0x7CC33, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "katayad1.evt", Offset = 0x7CC43, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "katayad2.evt", Offset = 0x7CC5D, UnpackedSize = 0x3E, Size = 0x37, IsPacked = true },
                    new PackedEntry { Name = "koro00.evt", Offset = 0x7CC94, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "koro01.evt", Offset = 0x7CCAE, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "koro02.evt", Offset = 0x7CCC8, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "koro03.evt", Offset = 0x7CCE2, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "kuro00.evt", Offset = 0x7CCFC, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "kuro01.evt", Offset = 0x7CD16, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "kuro02.evt", Offset = 0x7CD30, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "kuro03.evt", Offset = 0x7CD4A, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "last00.evt", Offset = 0x7CD64, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "last01.evt", Offset = 0x7CD7E, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "last02.evt", Offset = 0x7CDAC, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "last03.evt", Offset = 0x7CDD0, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "last04.evt", Offset = 0x7CDF4, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "last05.evt", Offset = 0x7CE18, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "last06.evt", Offset = 0x7CE46, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "last07.evt", Offset = 0x7CE6A, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "last08.evt", Offset = 0x7CE84, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "last09.evt", Offset = 0x7CEA8, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "last10.evt", Offset = 0x7CEC2, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "last11.evt", Offset = 0x7CEE6, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "last12.evt", Offset = 0x7CF0A, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "last13.evt", Offset = 0x7CF24, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "last14.evt", Offset = 0x7CF5C, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "last15.evt", Offset = 0x7CF80, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "last16.evt", Offset = 0x7CFB8, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "last17.evt", Offset = 0x7CFF0, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "last18.evt", Offset = 0x7D000, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "last19.evt", Offset = 0x7D024, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "last20.evt", Offset = 0x7D052, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "last21.evt", Offset = 0x7D06C, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "last22.evt", Offset = 0x7D086, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "last23.evt", Offset = 0x7D0A0, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "last24.evt", Offset = 0x7D0BA, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "matu00.evt", Offset = 0x7D0CA, UnpackedSize = 0x0AC, Size = 0x0A8, IsPacked = true },
                    new PackedEntry { Name = "ojou01.evt", Offset = 0x7D172, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "ojou02.evt", Offset = 0x7D1AA, UnpackedSize = 0x48, Size = 0x4C, IsPacked = true },
                    new PackedEntry { Name = "ojou03.evt", Offset = 0x7D1F6, UnpackedSize = 0x52, Size = 0x4A, IsPacked = true },
                    new PackedEntry { Name = "ojou04.evt", Offset = 0x7D240, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "ojou05.evt", Offset = 0x7D264, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "ojou06.evt", Offset = 0x7D292, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "ojou07.evt", Offset = 0x7D2C0, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "ojou08.evt", Offset = 0x7D2DA, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "ojou09.evt", Offset = 0x7D308, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "ojou10.evt", Offset = 0x7D322, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "ojou11.evt", Offset = 0x7D33C, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "ojou12.evt", Offset = 0x7D36A, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "ojou13.evt", Offset = 0x7D38E, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "ojou14.evt", Offset = 0x7D3BC, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "ojou15.evt", Offset = 0x7D3D6, UnpackedSize = 0x48, Size = 0x4C, IsPacked = true },
                    new PackedEntry { Name = "ojou16.evt", Offset = 0x7D422, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "ojou17.evt", Offset = 0x7D43C, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "ojou18.evt", Offset = 0x7D456, UnpackedSize = 0x3E, Size = 0x39, IsPacked = true },
                    new PackedEntry { Name = "ojou19.evt", Offset = 0x7D48F, UnpackedSize = 0x3E, Size = 0x42, IsPacked = true },
                    new PackedEntry { Name = "ojou20.evt", Offset = 0x7D4D1, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "ojou21.evt", Offset = 0x7D4EB, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "ojou22.evt", Offset = 0x7D50F, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "oldw00.evt", Offset = 0x7D529, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "oldw01.evt", Offset = 0x7D539, UnpackedSize = 0x3E, Size = 0x42, IsPacked = true },
                    new PackedEntry { Name = "oldw02.evt", Offset = 0x7D57B, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "oldw03.evt", Offset = 0x7D58B, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "oldw04.evt", Offset = 0x7D59B, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "oldw05.evt", Offset = 0x7D5AB, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "oldw06.evt", Offset = 0x7D5BB, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "oldw07.evt", Offset = 0x7D5F3, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "oldw08.evt", Offset = 0x7D603, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "oldw09.evt", Offset = 0x7D613, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "oldw10.evt", Offset = 0x7D623, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "oldw11.evt", Offset = 0x7D63D, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "oldw12.evt", Offset = 0x7D657, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "omake.evt", Offset = 0x7D671, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "omake2.evt", Offset = 0x7D677, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "paremina.evt", Offset = 0x7D67D, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "parerumo.evt", Offset = 0x7D697, UnpackedSize = 0x5C, Size = 0x58, IsPacked = true },
                    new PackedEntry { Name = "pargil.evt", Offset = 0x7D6EF, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "parsaka.evt", Offset = 0x7D6FF, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "paruma.evt", Offset = 0x7D719, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "paryad1.evt", Offset = 0x7D733, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "paryad2.evt", Offset = 0x7D757, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "saba00.evt", Offset = 0x7D785, UnpackedSize = 0x48, Size = 0x4C, IsPacked = true },
                    new PackedEntry { Name = "saba01.evt", Offset = 0x7D7D1, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "saba02.evt", Offset = 0x7D7FF, UnpackedSize = 0x48, Size = 0x4C, IsPacked = true },
                    new PackedEntry { Name = "saba03.evt", Offset = 0x7D84B, UnpackedSize = 0x5C, Size = 0x60, IsPacked = true },
                    new PackedEntry { Name = "saba04.evt", Offset = 0x7D8AB, UnpackedSize = 0x3E, Size = 0x42, IsPacked = true },
                    new PackedEntry { Name = "sakaba.evt", Offset = 0x7D8ED, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "sepu.evt", Offset = 0x7D8FD, UnpackedSize = 0x48, Size = 0x47, IsPacked = true },
                    new PackedEntry { Name = "sepu00.evt", Offset = 0x7D944, UnpackedSize = 0x48, Size = 0x47, IsPacked = true },
                    new PackedEntry { Name = "sepu01.evt", Offset = 0x7D98B, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "sepu02.evt", Offset = 0x7D9A5, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "sepu03.evt", Offset = 0x7D9C9, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "sepu04.evt", Offset = 0x7D9D9, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "sepu05.evt", Offset = 0x7DA11, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "sepu06.evt", Offset = 0x7DA21, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "sepu07.evt", Offset = 0x7DA45, UnpackedSize = 0x48, Size = 0x4C, IsPacked = true },
                    new PackedEntry { Name = "sepu08.evt", Offset = 0x7DA91, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "sepu09.evt", Offset = 0x7DAC9, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "sepu10.evt", Offset = 0x7DAED, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "sepu11.evt", Offset = 0x7DB1B, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "sepu12.evt", Offset = 0x7DB49, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "sepu13.evt", Offset = 0x7DB77, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "sepu14.evt", Offset = 0x7DBA5, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "sima00.evt", Offset = 0x7DBD3, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "sima01.evt", Offset = 0x7DBED, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "sima02.evt", Offset = 0x7DC1B, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "sima03.evt", Offset = 0x7DC49, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "sima04.evt", Offset = 0x7DC77, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "sima05.evt", Offset = 0x7DCA5, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "sima06.evt", Offset = 0x7DCDD, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "sima07.evt", Offset = 0x7DD01, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "sougen.evt", Offset = 0x7DD11, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "syoko01.evt", Offset = 0x7DD2B, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "syoko02.evt", Offset = 0x7DD59, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "syud00.evt", Offset = 0x7DD7D, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "syud01.evt", Offset = 0x7DD97, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "tidu00.evt", Offset = 0x7DDB1, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "tidu01.evt", Offset = 0x7DDD5, UnpackedSize = 0x3E, Size = 0x42, IsPacked = true },
                    new PackedEntry { Name = "tidu02.evt", Offset = 0x7DE17, UnpackedSize = 0x3E, Size = 0x42, IsPacked = true },
                    new PackedEntry { Name = "tidu03.evt", Offset = 0x7DE59, UnpackedSize = 0x3E, Size = 0x42, IsPacked = true },
                    new PackedEntry { Name = "tidu04.evt", Offset = 0x7DE9B, UnpackedSize = 0x3E, Size = 0x42, IsPacked = true },
                    new PackedEntry { Name = "tidu05.evt", Offset = 0x7DEDD, UnpackedSize = 0x3E, Size = 0x42, IsPacked = true },
                    new PackedEntry { Name = "tita00.evt", Offset = 0x7DF1F, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "tita01.evt", Offset = 0x7DF39, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "tita02.evt", Offset = 0x7DF5D, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "tita03.evt", Offset = 0x7DF81, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "tita04.evt", Offset = 0x7DFAF, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "tita05.evt", Offset = 0x7DFDD, UnpackedSize = 0x34, Size = 0x38, IsPacked = true },
                    new PackedEntry { Name = "tita06.evt", Offset = 0x7E015, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "tita07.evt", Offset = 0x7E043, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "tori00.evt", Offset = 0x7E05D, UnpackedSize = 0x5C, Size = 0x60, IsPacked = true },
                    new PackedEntry { Name = "tori01.evt", Offset = 0x7E0BD, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "tori02.evt", Offset = 0x7E0CD, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "tori03.evt", Offset = 0x7E0DD, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "tori04.evt", Offset = 0x7E0ED, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "tori05.evt", Offset = 0x7E107, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "tori06.evt", Offset = 0x7E117, UnpackedSize = 0x34, Size = 0x33, IsPacked = true },
                    new PackedEntry { Name = "tori07.evt", Offset = 0x7E14A, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "tori08.evt", Offset = 0x7E15A, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "yama01.evt", Offset = 0x7E16A, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "yama02.evt", Offset = 0x7E184, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "yama03.evt", Offset = 0x7E1A8, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "yama04.evt", Offset = 0x7E1CC, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "yama05.evt", Offset = 0x7E1FA, UnpackedSize = 0x2A, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "yama06.evt", Offset = 0x7E228, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "yujin.evt", Offset = 0x7E242, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "yuru00.evt", Offset = 0x7E252, UnpackedSize = 0x70, Size = 0x6A, IsPacked = true },
                    new PackedEntry { Name = "yuru01.evt", Offset = 0x7E2BC, UnpackedSize = 0x20, Size = 0x24, IsPacked = true },
                    new PackedEntry { Name = "yuru02.evt", Offset = 0x7E2E0, UnpackedSize = 0x70, Size = 0x74, IsPacked = true },
                    new PackedEntry { Name = "yuru03.evt", Offset = 0x7E354, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "yuru04.evt", Offset = 0x7E36E, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "yuru05.evt", Offset = 0x7E37E, UnpackedSize = 0x0C, Size = 0x10, IsPacked = true },
                    new PackedEntry { Name = "yuru06.evt", Offset = 0x7E38E, UnpackedSize = 0x8E, Size = 0x8B, IsPacked = true },
                    new PackedEntry { Name = "yuru07.evt", Offset = 0x7E419, UnpackedSize = 0x5C, Size = 0x60, IsPacked = true },
                    new PackedEntry { Name = "yuru08.evt", Offset = 0x7E479, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "yuru09.evt", Offset = 0x7E493, UnpackedSize = 0x16, Size = 0x1A, IsPacked = true },
                    new PackedEntry { Name = "allmap.ts2", Offset = 0x7E4AD, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "allmap2.ts2", Offset = 0x7E4C5, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "bukiya.ts2", Offset = 0x7E4CB, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "cron00.ts2", Offset = 0x7E4F5, UnpackedSize = 0x122, Size = 0x0C1, IsPacked = true },
                    new PackedEntry { Name = "cudu00.ts2", Offset = 0x7E5B6, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "cudu01.ts2", Offset = 0x7E5BC, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "cudu02.ts2", Offset = 0x7E5C2, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "cudu03.ts2", Offset = 0x7E5C8, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "cudu04.ts2", Offset = 0x7E5E0, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "cudu05.ts2", Offset = 0x7E5F8, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "cudu06.ts2", Offset = 0x7E610, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "cudu07.ts2", Offset = 0x7E616, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "cudu08.ts2", Offset = 0x7E62E, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "cudu09.ts2", Offset = 0x7E634, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "cudu10.ts2", Offset = 0x7E63A, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "doguya.ts2", Offset = 0x7E640, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "dokutu01.ts2", Offset = 0x7E658, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "dokutu02.ts2", Offset = 0x7E65E, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "dokutu03.ts2", Offset = 0x7E676, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "dokutu04.ts2", Offset = 0x7E67C, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "dokutu05.ts2", Offset = 0x7E682, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "dokutu06.ts2", Offset = 0x7E688, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "dokutu07.ts2", Offset = 0x7E68E, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "dokutu08.ts2", Offset = 0x7E6A6, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "dokutu09.ts2", Offset = 0x7E6BE, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "flap00.ts2", Offset = 0x7E6C4, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "flap01.ts2", Offset = 0x7E6CA, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "flap02.ts2", Offset = 0x7E6E2, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "flap03.ts2", Offset = 0x7E6E8, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "flap04.ts2", Offset = 0x7E6EE, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "flap05.ts2", Offset = 0x7E6F4, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "flap06.ts2", Offset = 0x7E70C, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "flap07.ts2", Offset = 0x7E724, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "flap08.ts2", Offset = 0x7E72A, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "flap09.ts2", Offset = 0x7E730, UnpackedSize = 0x0A4, Size = 0x91, IsPacked = true },
                    new PackedEntry { Name = "flap10.ts2", Offset = 0x7E7C1, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "flap11.ts2", Offset = 0x7E7EB, UnpackedSize = 0x4A, Size = 0x46, IsPacked = true },
                    new PackedEntry { Name = "flap12.ts2", Offset = 0x7E831, UnpackedSize = 0x4A, Size = 0x4E, IsPacked = true },
                    new PackedEntry { Name = "flap13.ts2", Offset = 0x7E87F, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "flap14.ts2", Offset = 0x7E885, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "flap15.ts2", Offset = 0x7E8AF, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "flap16.ts2", Offset = 0x7E8B5, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "flap17.ts2", Offset = 0x7E8DF, UnpackedSize = 0x38, Size = 0x3C, IsPacked = true },
                    new PackedEntry { Name = "flap18.ts2", Offset = 0x7E91B, UnpackedSize = 0x6E, Size = 0x66, IsPacked = true },
                    new PackedEntry { Name = "fond00.ts2", Offset = 0x7E981, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "fond01.ts2", Offset = 0x7E987, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "fond02.ts2", Offset = 0x7E98D, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "fond03.ts2", Offset = 0x7E993, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "fond04.ts2", Offset = 0x7E999, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "fond05.ts2", Offset = 0x7E9B1, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "fond06.ts2", Offset = 0x7E9B7, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "fond07.ts2", Offset = 0x7E9CF, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "fond08.ts2", Offset = 0x7E9E7, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "fond09.ts2", Offset = 0x7E9FF, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "garu01.ts2", Offset = 0x7EA29, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "garu02.ts2", Offset = 0x7EA41, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "garu03.ts2", Offset = 0x7EA47, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "garu04.ts2", Offset = 0x7EA4D, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "garu05.ts2", Offset = 0x7EA65, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "garu06.ts2", Offset = 0x7EA7D, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "garu07.ts2", Offset = 0x7EA83, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "garu08.ts2", Offset = 0x7EA89, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "garu09.ts2", Offset = 0x7EAA1, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "garu10.ts2", Offset = 0x7EAA7, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "hbmura01.ts2", Offset = 0x7EABF, UnpackedSize = 0x80, Size = 0x74, IsPacked = true },
                    new PackedEntry { Name = "isek00.ts2", Offset = 0x7EB33, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "isek01.ts2", Offset = 0x7EB4B, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "isek02.ts2", Offset = 0x7EB75, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "isek03.ts2", Offset = 0x7EB7B, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "isek04.ts2", Offset = 0x7EB81, UnpackedSize = 0x38, Size = 0x3C, IsPacked = true },
                    new PackedEntry { Name = "isek05.ts2", Offset = 0x7EBBD, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "isek06.ts2", Offset = 0x7EBC3, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "jobin.ts2", Offset = 0x7EBC9, UnpackedSize = 0x0B6, Size = 0x0A3, IsPacked = true },
                    new PackedEntry { Name = "jobuki.ts2", Offset = 0x7EC6C, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "jodog.ts2", Offset = 0x7EC96, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "jofuka1.ts2", Offset = 0x7ECAE, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "jofuka2.ts2", Offset = 0x7ECC6, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "jofuka3.ts2", Offset = 0x7ECDE, UnpackedSize = 0x4A, Size = 0x45, IsPacked = true },
                    new PackedEntry { Name = "joido.ts2", Offset = 0x7ED23, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "joisya.ts2", Offset = 0x7ED29, UnpackedSize = 0x5C, Size = 0x5B, IsPacked = true },
                    new PackedEntry { Name = "jomin1.ts2", Offset = 0x7ED84, UnpackedSize = 0x38, Size = 0x3C, IsPacked = true },
                    new PackedEntry { Name = "jomin2.ts2", Offset = 0x7EDC0, UnpackedSize = 0x5C, Size = 0x57, IsPacked = true },
                    new PackedEntry { Name = "jouka.ts2", Offset = 0x7EE17, UnpackedSize = 0x1E8, Size = 0x186, IsPacked = true },
                    new PackedEntry { Name = "jouma.ts2", Offset = 0x7EF9D, UnpackedSize = 0x38, Size = 0x31, IsPacked = true },
                    new PackedEntry { Name = "joyado1.ts2", Offset = 0x7EFCE, UnpackedSize = 0x6E, Size = 0x62, IsPacked = true },
                    new PackedEntry { Name = "joyado2.ts2", Offset = 0x7F030, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "kain1.ts2", Offset = 0x7F036, UnpackedSize = 0x4A, Size = 0x4E, IsPacked = true },
                    new PackedEntry { Name = "kain2.ts2", Offset = 0x7F084, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "kaiz00.ts2", Offset = 0x7F09C, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "kaiz01.ts2", Offset = 0x7F0A2, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "kaiz02.ts2", Offset = 0x7F0A8, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "kaiz03.ts2", Offset = 0x7F0AE, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "kaiz04.ts2", Offset = 0x7F0D8, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "kaiz05.ts2", Offset = 0x7F102, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "kaiz06.ts2", Offset = 0x7F108, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "kaiz07.ts2", Offset = 0x7F120, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "katabuki.ts2", Offset = 0x7F126, UnpackedSize = 0x38, Size = 0x3C, IsPacked = true },
                    new PackedEntry { Name = "katadogu.ts2", Offset = 0x7F162, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "katamin1.ts2", Offset = 0x7F17A, UnpackedSize = 0x6E, Size = 0x6A, IsPacked = true },
                    new PackedEntry { Name = "katamin2.ts2", Offset = 0x7F1E4, UnpackedSize = 0x5C, Size = 0x59, IsPacked = true },
                    new PackedEntry { Name = "katamina.ts2", Offset = 0x7F23D, UnpackedSize = 0x0B6, Size = 0x0AA, IsPacked = true },
                    new PackedEntry { Name = "katania.ts2", Offset = 0x7F2E7, UnpackedSize = 0x0B6, Size = 0x0AA, IsPacked = true },
                    new PackedEntry { Name = "kataryo1.ts2", Offset = 0x7F391, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "kataryo2.ts2", Offset = 0x7F3BB, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "katasou1.ts2", Offset = 0x7F3D3, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "katasou2.ts2", Offset = 0x7F3D9, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "katasyo.ts2", Offset = 0x7F3DF, UnpackedSize = 0x38, Size = 0x36, IsPacked = true },
                    new PackedEntry { Name = "katayad1.ts2", Offset = 0x7F415, UnpackedSize = 0x38, Size = 0x3C, IsPacked = true },
                    new PackedEntry { Name = "katayad2.ts2", Offset = 0x7F451, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "koro00.ts2", Offset = 0x7F469, UnpackedSize = 0x38, Size = 0x34, IsPacked = true },
                    new PackedEntry { Name = "koro01.ts2", Offset = 0x7F49D, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "koro02.ts2", Offset = 0x7F4C7, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "koro03.ts2", Offset = 0x7F4CD, UnpackedSize = 0x6E, Size = 0x62, IsPacked = true },
                    new PackedEntry { Name = "kuro00.ts2", Offset = 0x7F52F, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "kuro01.ts2", Offset = 0x7F535, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "kuro02.ts2", Offset = 0x7F53B, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "kuro03.ts2", Offset = 0x7F541, UnpackedSize = 0x0DA, Size = 0x99, IsPacked = true },
                    new PackedEntry { Name = "last00.ts2", Offset = 0x7F5DA, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "last01.ts2", Offset = 0x7F5E0, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "last02.ts2", Offset = 0x7F5E6, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "last03.ts2", Offset = 0x7F5EC, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "last04.ts2", Offset = 0x7F5F2, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "last05.ts2", Offset = 0x7F5F8, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "last06.ts2", Offset = 0x7F622, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "last07.ts2", Offset = 0x7F63A, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "last08.ts2", Offset = 0x7F652, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "last09.ts2", Offset = 0x7F658, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "last10.ts2", Offset = 0x7F65E, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "last11.ts2", Offset = 0x7F664, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "last12.ts2", Offset = 0x7F66A, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "last13.ts2", Offset = 0x7F670, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "last14.ts2", Offset = 0x7F676, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "last15.ts2", Offset = 0x7F68E, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "last16.ts2", Offset = 0x7F694, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "last17.ts2", Offset = 0x7F69A, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "last18.ts2", Offset = 0x7F6B2, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "last19.ts2", Offset = 0x7F6B8, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "last20.ts2", Offset = 0x7F6BE, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "last21.ts2", Offset = 0x7F6C4, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "last22.ts2", Offset = 0x7F6CA, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "last23.ts2", Offset = 0x7F6D0, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "last24.ts2", Offset = 0x7F6D6, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "matu00.ts2", Offset = 0x7F6DC, UnpackedSize = 0x1E8, Size = 0x186, IsPacked = true },
                    new PackedEntry { Name = "ojou01.ts2", Offset = 0x7F862, UnpackedSize = 0x242, Size = 0x16B, IsPacked = true },
                    new PackedEntry { Name = "ojou02.ts2", Offset = 0x7F9CD, UnpackedSize = 0x80, Size = 0x70, IsPacked = true },
                    new PackedEntry { Name = "ojou03.ts2", Offset = 0x7FA3D, UnpackedSize = 0x38, Size = 0x3C, IsPacked = true },
                    new PackedEntry { Name = "ojou04.ts2", Offset = 0x7FA79, UnpackedSize = 0x38, Size = 0x36, IsPacked = true },
                    new PackedEntry { Name = "ojou05.ts2", Offset = 0x7FAAF, UnpackedSize = 0x80, Size = 0x5B, IsPacked = true },
                    new PackedEntry { Name = "ojou06.ts2", Offset = 0x7FB0A, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "ojou07.ts2", Offset = 0x7FB22, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "ojou08.ts2", Offset = 0x7FB3A, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "ojou09.ts2", Offset = 0x7FB40, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "ojou10.ts2", Offset = 0x7FB46, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "ojou11.ts2", Offset = 0x7FB4C, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "ojou12.ts2", Offset = 0x7FB64, UnpackedSize = 0x26, Size = 0x25, IsPacked = true },
                    new PackedEntry { Name = "ojou13.ts2", Offset = 0x7FB89, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "ojou14.ts2", Offset = 0x7FBA1, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "ojou15.ts2", Offset = 0x7FBA7, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "ojou16.ts2", Offset = 0x7FBAD, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "ojou17.ts2", Offset = 0x7FBB3, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "ojou18.ts2", Offset = 0x7FBB9, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "ojou19.ts2", Offset = 0x7FBBF, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "ojou20.ts2", Offset = 0x7FBC5, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "ojou21.ts2", Offset = 0x7FBDD, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "ojou22.ts2", Offset = 0x7FBF5, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "oldw00.ts2", Offset = 0x7FBFB, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "oldw01.ts2", Offset = 0x7FC13, UnpackedSize = 0x6E, Size = 0x64, IsPacked = true },
                    new PackedEntry { Name = "oldw02.ts2", Offset = 0x7FC77, UnpackedSize = 0x38, Size = 0x2E, IsPacked = true },
                    new PackedEntry { Name = "oldw03.ts2", Offset = 0x7FCA5, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "oldw04.ts2", Offset = 0x7FCBD, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "oldw05.ts2", Offset = 0x7FCD5, UnpackedSize = 0x5C, Size = 0x4D, IsPacked = true },
                    new PackedEntry { Name = "oldw06.ts2", Offset = 0x7FD22, UnpackedSize = 0x6E, Size = 0x5E, IsPacked = true },
                    new PackedEntry { Name = "oldw07.ts2", Offset = 0x7FD80, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "oldw08.ts2", Offset = 0x7FD98, UnpackedSize = 0x38, Size = 0x3C, IsPacked = true },
                    new PackedEntry { Name = "oldw09.ts2", Offset = 0x7FDD4, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "oldw10.ts2", Offset = 0x7FDEC, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "oldw11.ts2", Offset = 0x7FE04, UnpackedSize = 0x92, Size = 0x50, IsPacked = true },
                    new PackedEntry { Name = "oldw12.ts2", Offset = 0x7FE54, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "omake.ts2", Offset = 0x7FE5A, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "omake2.ts2", Offset = 0x7FE60, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "paremina.ts2", Offset = 0x7FE66, UnpackedSize = 0x38, Size = 0x3C, IsPacked = true },
                    new PackedEntry { Name = "parerumo.ts2", Offset = 0x7FEA2, UnpackedSize = 0x1C4, Size = 0x185, IsPacked = true },
                    new PackedEntry { Name = "pargil.ts2", Offset = 0x80027, UnpackedSize = 0x5C, Size = 0x55, IsPacked = true },
                    new PackedEntry { Name = "parsaka.ts2", Offset = 0x8007C, UnpackedSize = 0x5C, Size = 0x55, IsPacked = true },
                    new PackedEntry { Name = "paruma.ts2", Offset = 0x800D1, UnpackedSize = 0x38, Size = 0x2B, IsPacked = true },
                    new PackedEntry { Name = "paryad1.ts2", Offset = 0x800FC, UnpackedSize = 0x38, Size = 0x3C, IsPacked = true },
                    new PackedEntry { Name = "paryad2.ts2", Offset = 0x80138, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "saba00.ts2", Offset = 0x80150, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "saba01.ts2", Offset = 0x80156, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "saba02.ts2", Offset = 0x8015C, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "saba03.ts2", Offset = 0x80162, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "saba04.ts2", Offset = 0x8018C, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "sakaba.ts2", Offset = 0x801A4, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "sepu.ts2", Offset = 0x801CE, UnpackedSize = 0x92, Size = 0x7B, IsPacked = true },
                    new PackedEntry { Name = "sepu00.ts2", Offset = 0x80249, UnpackedSize = 0x92, Size = 0x7B, IsPacked = true },
                    new PackedEntry { Name = "sepu01.ts2", Offset = 0x802C4, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "sepu02.ts2", Offset = 0x802EE, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "sepu03.ts2", Offset = 0x80318, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "sepu04.ts2", Offset = 0x80342, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "sepu05.ts2", Offset = 0x8036C, UnpackedSize = 0x0A4, Size = 0x76, IsPacked = true },
                    new PackedEntry { Name = "sepu06.ts2", Offset = 0x803E2, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "sepu07.ts2", Offset = 0x803E8, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "sepu08.ts2", Offset = 0x80412, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "sepu09.ts2", Offset = 0x80418, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "sepu10.ts2", Offset = 0x8041E, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "sepu11.ts2", Offset = 0x80424, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "sepu12.ts2", Offset = 0x8042A, UnpackedSize = 0x38, Size = 0x31, IsPacked = true },
                    new PackedEntry { Name = "sepu13.ts2", Offset = 0x8045B, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "sepu14.ts2", Offset = 0x80461, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "sima00.ts2", Offset = 0x8048B, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "sima01.ts2", Offset = 0x80491, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "sima02.ts2", Offset = 0x80497, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "sima03.ts2", Offset = 0x804C1, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "sima04.ts2", Offset = 0x804D9, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "sima05.ts2", Offset = 0x804DF, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "sima06.ts2", Offset = 0x804F7, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "sima07.ts2", Offset = 0x804FD, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "sougen.ts2", Offset = 0x80515, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "syoko01.ts2", Offset = 0x8051B, UnpackedSize = 0x0EC, Size = 0x0BC, IsPacked = true },
                    new PackedEntry { Name = "syoko02.ts2", Offset = 0x805D7, UnpackedSize = 0x146, Size = 0x10D, IsPacked = true },
                    new PackedEntry { Name = "syud00.ts2", Offset = 0x806E4, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "syud01.ts2", Offset = 0x806EA, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "tidu00.ts2", Offset = 0x80702, UnpackedSize = 0x38, Size = 0x3C, IsPacked = true },
                    new PackedEntry { Name = "tidu01.ts2", Offset = 0x8073E, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "tidu02.ts2", Offset = 0x80744, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "tidu03.ts2", Offset = 0x8074A, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "tidu04.ts2", Offset = 0x80762, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "tidu05.ts2", Offset = 0x8077A, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "tita00.ts2", Offset = 0x807A4, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "tita01.ts2", Offset = 0x807AA, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "tita02.ts2", Offset = 0x807C2, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "tita03.ts2", Offset = 0x807C8, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "tita04.ts2", Offset = 0x807CE, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "tita05.ts2", Offset = 0x807F8, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "tita06.ts2", Offset = 0x807FE, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "tita07.ts2", Offset = 0x80828, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "tori00.ts2", Offset = 0x80840, UnpackedSize = 0x1A0, Size = 0x131, IsPacked = true },
                    new PackedEntry { Name = "tori01.ts2", Offset = 0x80971, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "tori02.ts2", Offset = 0x80989, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "tori03.ts2", Offset = 0x809B3, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "tori04.ts2", Offset = 0x809CB, UnpackedSize = 0x80, Size = 0x5A, IsPacked = true },
                    new PackedEntry { Name = "tori05.ts2", Offset = 0x80A25, UnpackedSize = 0x4A, Size = 0x4E, IsPacked = true },
                    new PackedEntry { Name = "tori06.ts2", Offset = 0x80A73, UnpackedSize = 0x38, Size = 0x3C, IsPacked = true },
                    new PackedEntry { Name = "tori07.ts2", Offset = 0x80AAF, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "tori08.ts2", Offset = 0x80AD9, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "yama01.ts2", Offset = 0x80B03, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "yama02.ts2", Offset = 0x80B1B, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "yama03.ts2", Offset = 0x80B21, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "yama04.ts2", Offset = 0x80B39, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "yama05.ts2", Offset = 0x80B51, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "yama06.ts2", Offset = 0x80B69, UnpackedSize = 0x2, Size = 0x6, IsPacked = true },
                    new PackedEntry { Name = "yujin.ts2", Offset = 0x80B6F, UnpackedSize = 0x6E, Size = 0x69, IsPacked = true },
                    new PackedEntry { Name = "yuru00.ts2", Offset = 0x80BD8, UnpackedSize = 0x146, Size = 0x120, IsPacked = true },
                    new PackedEntry { Name = "yuru01.ts2", Offset = 0x80CF8, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                    new PackedEntry { Name = "yuru02.ts2", Offset = 0x80D22, UnpackedSize = 0x5C, Size = 0x4F, IsPacked = true },
                    new PackedEntry { Name = "yuru03.ts2", Offset = 0x80D71, UnpackedSize = 0x38, Size = 0x30, IsPacked = true },
                    new PackedEntry { Name = "yuru04.ts2", Offset = 0x80DA1, UnpackedSize = 0x38, Size = 0x3C, IsPacked = true },
                    new PackedEntry { Name = "yuru05.ts2", Offset = 0x80DDD, UnpackedSize = 0x5C, Size = 0x60, IsPacked = true },
                    new PackedEntry { Name = "yuru06.ts2", Offset = 0x80E3D, UnpackedSize = 0x14, Size = 0x18, IsPacked = true },
                    new PackedEntry { Name = "yuru07.ts2", Offset = 0x80E55, UnpackedSize = 0x122, Size = 0x0BE, IsPacked = true },
                    new PackedEntry { Name = "yuru08.ts2", Offset = 0x80F13, UnpackedSize = 0x0B6, Size = 0x80, IsPacked = true },
                    new PackedEntry { Name = "yuru09.ts2", Offset = 0x80F93, UnpackedSize = 0x26, Size = 0x2A, IsPacked = true },
                }
            },
            {
                "ALLMCP.273", new List<Entry>() {
                    new PackedEntry { Name = "Allmap.mcp", Offset = 0x0, UnpackedSize = 0x5F436, Size = 0x1C61A, IsPacked = true },
                    new PackedEntry { Name = "Bukiya.mcp", Offset = 0x1C61A, UnpackedSize = 0x23436, Size = 0x7157, IsPacked = true },
                    new PackedEntry { Name = "Cron00.mcp", Offset = 0x23771, UnpackedSize = 0x19436, Size = 0x4652, IsPacked = true },
                    new PackedEntry { Name = "Cudu00.mcp", Offset = 0x27DC3, UnpackedSize = 0x14436, Size = 0x2893, IsPacked = true },
                    new PackedEntry { Name = "Cudu01.mcp", Offset = 0x2A656, UnpackedSize = 0x14436, Size = 0x2522, IsPacked = true },
                    new PackedEntry { Name = "Cudu02.mcp", Offset = 0x2CB78, UnpackedSize = 0x14436, Size = 0x2522, IsPacked = true },
                    new PackedEntry { Name = "Cudu03.mcp", Offset = 0x2F09A, UnpackedSize = 0x14436, Size = 0x2F8E, IsPacked = true },
                    new PackedEntry { Name = "Cudu04.mcp", Offset = 0x32028, UnpackedSize = 0x0F436, Size = 0x1C4F, IsPacked = true },
                    new PackedEntry { Name = "Cudu05.mcp", Offset = 0x33C77, UnpackedSize = 0x1E436, Size = 0x3EBF, IsPacked = true },
                    new PackedEntry { Name = "Cudu06.mcp", Offset = 0x37B36, UnpackedSize = 0x14436, Size = 0x29DA, IsPacked = true },
                    new PackedEntry { Name = "Cudu07.mcp", Offset = 0x3A510, UnpackedSize = 0x28436, Size = 0x45EF, IsPacked = true },
                    new PackedEntry { Name = "Cudu08.mcp", Offset = 0x3EAFF, UnpackedSize = 0x0F436, Size = 0x1677, IsPacked = true },
                    new PackedEntry { Name = "Cudu09.mcp", Offset = 0x40176, UnpackedSize = 0x14436, Size = 0x27F0, IsPacked = true },
                    new PackedEntry { Name = "Cudu10.mcp", Offset = 0x42966, UnpackedSize = 0x14436, Size = 0x249C, IsPacked = true },
                    new PackedEntry { Name = "Doguya.mcp", Offset = 0x44E02, UnpackedSize = 0x23436, Size = 0x68FA, IsPacked = true },
                    new PackedEntry { Name = "Dokutu01.mcp", Offset = 0x4B6FC, UnpackedSize = 0x32436, Size = 0x0C7C5, IsPacked = true },
                    new PackedEntry { Name = "Dokutu02.mcp", Offset = 0x57EC1, UnpackedSize = 0x2D436, Size = 0x0BD41, IsPacked = true },
                    new PackedEntry { Name = "Dokutu03.mcp", Offset = 0x63C02, UnpackedSize = 0x1E436, Size = 0x78BF, IsPacked = true },
                    new PackedEntry { Name = "Dokutu04.mcp", Offset = 0x6B4C1, UnpackedSize = 0x37436, Size = 0x0EB23, IsPacked = true },
                    new PackedEntry { Name = "Dokutu05.mcp", Offset = 0x79FE4, UnpackedSize = 0x1E436, Size = 0x7C91, IsPacked = true },
                    new PackedEntry { Name = "Dokutu06.mcp", Offset = 0x81C75, UnpackedSize = 0x23436, Size = 0x81BA, IsPacked = true },
                    new PackedEntry { Name = "Dokutu07.mcp", Offset = 0x89E2F, UnpackedSize = 0x32436, Size = 0x0AAE1, IsPacked = true },
                    new PackedEntry { Name = "Dokutu08.mcp", Offset = 0x94910, UnpackedSize = 0x32436, Size = 0x0CA7E, IsPacked = true },
                    new PackedEntry { Name = "Dokutu09.mcp", Offset = 0x0A138E, UnpackedSize = 0x32436, Size = 0x0C3CA, IsPacked = true },
                    new PackedEntry { Name = "Flap00.mcp", Offset = 0x0AD758, UnpackedSize = 0x3C436, Size = 0x0F905, IsPacked = true },
                    new PackedEntry { Name = "Flap01.mcp", Offset = 0x0BD05D, UnpackedSize = 0x1E436, Size = 0x6D8D, IsPacked = true },
                    new PackedEntry { Name = "Flap02.mcp", Offset = 0x0C3DEA, UnpackedSize = 0x1E436, Size = 0x6F0E, IsPacked = true },
                    new PackedEntry { Name = "Flap03.mcp", Offset = 0x0CACF8, UnpackedSize = 0x1E436, Size = 0x72C2, IsPacked = true },
                    new PackedEntry { Name = "Flap04.mcp", Offset = 0x0D1FBA, UnpackedSize = 0x2D436, Size = 0x0ACA5, IsPacked = true },
                    new PackedEntry { Name = "Flap05.mcp", Offset = 0x0DCC5F, UnpackedSize = 0x37436, Size = 0x0E7E9, IsPacked = true },
                    new PackedEntry { Name = "Flap06.mcp", Offset = 0x0EB448, UnpackedSize = 0x28436, Size = 0x0A7A2, IsPacked = true },
                    new PackedEntry { Name = "Flap07.mcp", Offset = 0x0F5BEA, UnpackedSize = 0x19436, Size = 0x6541, IsPacked = true },
                    new PackedEntry { Name = "Flap08.mcp", Offset = 0x0FC12B, UnpackedSize = 0x32436, Size = 0x0CF11, IsPacked = true },
                    new PackedEntry { Name = "Flap09.mcp", Offset = 0x10903C, UnpackedSize = 0x46436, Size = 0x0FE0D, IsPacked = true },
                    new PackedEntry { Name = "Flap10.mcp", Offset = 0x118E49, UnpackedSize = 0x23436, Size = 0x5E95, IsPacked = true },
                    new PackedEntry { Name = "Flap11.mcp", Offset = 0x11ECDE, UnpackedSize = 0x23436, Size = 0x5B5F, IsPacked = true },
                    new PackedEntry { Name = "Flap12.mcp", Offset = 0x12483D, UnpackedSize = 0x32436, Size = 0x8E8F, IsPacked = true },
                    new PackedEntry { Name = "Flap13.mcp", Offset = 0x12D6CC, UnpackedSize = 0x23436, Size = 0x6599, IsPacked = true },
                    new PackedEntry { Name = "Flap14.mcp", Offset = 0x133C65, UnpackedSize = 0x23436, Size = 0x567D, IsPacked = true },
                    new PackedEntry { Name = "Flap15.mcp", Offset = 0x1392E2, UnpackedSize = 0x1E436, Size = 0x4F19, IsPacked = true },
                    new PackedEntry { Name = "Flap16.mcp", Offset = 0x13E1FB, UnpackedSize = 0x23436, Size = 0x5DC6, IsPacked = true },
                    new PackedEntry { Name = "Flap17.mcp", Offset = 0x143FC1, UnpackedSize = 0x23436, Size = 0x619D, IsPacked = true },
                    new PackedEntry { Name = "Flap18.mcp", Offset = 0x14A15E, UnpackedSize = 0x1E436, Size = 0x69F8, IsPacked = true },
                    new PackedEntry { Name = "Fond00.mcp", Offset = 0x150B56, UnpackedSize = 0x14436, Size = 0x5074, IsPacked = true },
                    new PackedEntry { Name = "Fond01.mcp", Offset = 0x155BCA, UnpackedSize = 0x0F436, Size = 0x42A3, IsPacked = true },
                    new PackedEntry { Name = "Fond02.mcp", Offset = 0x159E6D, UnpackedSize = 0x0F436, Size = 0x4A95, IsPacked = true },
                    new PackedEntry { Name = "Fond03.mcp", Offset = 0x15E902, UnpackedSize = 0x14436, Size = 0x5405, IsPacked = true },
                    new PackedEntry { Name = "Fond04.mcp", Offset = 0x163D07, UnpackedSize = 0x14436, Size = 0x5FA0, IsPacked = true },
                    new PackedEntry { Name = "Fond05.mcp", Offset = 0x169CA7, UnpackedSize = 0x0F436, Size = 0x42A4, IsPacked = true },
                    new PackedEntry { Name = "Fond06.mcp", Offset = 0x16DF4B, UnpackedSize = 0x14436, Size = 0x5A9F, IsPacked = true },
                    new PackedEntry { Name = "Fond07.mcp", Offset = 0x1739EA, UnpackedSize = 0x0F436, Size = 0x44ED, IsPacked = true },
                    new PackedEntry { Name = "Fond08.mcp", Offset = 0x177ED7, UnpackedSize = 0x1E436, Size = 0x7F4E, IsPacked = true },
                    new PackedEntry { Name = "Fond09.mcp", Offset = 0x17FE25, UnpackedSize = 0x23436, Size = 0x6073, IsPacked = true },
                    new PackedEntry { Name = "Garu01.mcp", Offset = 0x185E98, UnpackedSize = 0x0F436, Size = 0x4740, IsPacked = true },
                    new PackedEntry { Name = "Garu02.mcp", Offset = 0x18A5D8, UnpackedSize = 0x0F436, Size = 0x43E3, IsPacked = true },
                    new PackedEntry { Name = "Garu03.mcp", Offset = 0x18E9BB, UnpackedSize = 0x0F436, Size = 0x3665, IsPacked = true },
                    new PackedEntry { Name = "Garu04.mcp", Offset = 0x192020, UnpackedSize = 0x0A436, Size = 0x33FE, IsPacked = true },
                    new PackedEntry { Name = "Garu05.mcp", Offset = 0x19541E, UnpackedSize = 0x0F436, Size = 0x4617, IsPacked = true },
                    new PackedEntry { Name = "Garu06.mcp", Offset = 0x199A35, UnpackedSize = 0x14436, Size = 0x5E12, IsPacked = true },
                    new PackedEntry { Name = "Garu07.mcp", Offset = 0x19F847, UnpackedSize = 0x0F436, Size = 0x38A8, IsPacked = true },
                    new PackedEntry { Name = "Garu08.mcp", Offset = 0x1A30EF, UnpackedSize = 0x0F436, Size = 0x3861, IsPacked = true },
                    new PackedEntry { Name = "Garu09.mcp", Offset = 0x1A6950, UnpackedSize = 0x0F436, Size = 0x4252, IsPacked = true },
                    new PackedEntry { Name = "Garu10.mcp", Offset = 0x1AABA2, UnpackedSize = 0x0F436, Size = 0x3D95, IsPacked = true },
                    new PackedEntry { Name = "Hbmura01.mcp", Offset = 0x1AE937, UnpackedSize = 0x46436, Size = 0x1228F, IsPacked = true },
                    new PackedEntry { Name = "Isek00.mcp", Offset = 0x1C0BC6, UnpackedSize = 0x2D436, Size = 0x0ADCE, IsPacked = true },
                    new PackedEntry { Name = "Isek01.mcp", Offset = 0x1CB994, UnpackedSize = 0x14436, Size = 0x2564, IsPacked = true },
                    new PackedEntry { Name = "Isek02.mcp", Offset = 0x1CDEF8, UnpackedSize = 0x1E436, Size = 0x3563, IsPacked = true },
                    new PackedEntry { Name = "Isek03.mcp", Offset = 0x1D145B, UnpackedSize = 0x2D436, Size = 0x7CC5, IsPacked = true },
                    new PackedEntry { Name = "Isek04.mcp", Offset = 0x1D9120, UnpackedSize = 0x19436, Size = 0x29AB, IsPacked = true },
                    new PackedEntry { Name = "Isek05.mcp", Offset = 0x1DBACB, UnpackedSize = 0x14436, Size = 0x23D0, IsPacked = true },
                    new PackedEntry { Name = "Isek06.mcp", Offset = 0x1DDE9B, UnpackedSize = 0x2D436, Size = 0x6241, IsPacked = true },
                    new PackedEntry { Name = "Jobin.mcp", Offset = 0x1E40DC, UnpackedSize = 0x19436, Size = 0x4180, IsPacked = true },
                    new PackedEntry { Name = "Jobuki.mcp", Offset = 0x1E825C, UnpackedSize = 0x23436, Size = 0x6634, IsPacked = true },
                    new PackedEntry { Name = "Jodog.mcp", Offset = 0x1EE890, UnpackedSize = 0x19436, Size = 0x481E, IsPacked = true },
                    new PackedEntry { Name = "Jofuka1.mcp", Offset = 0x1F30AE, UnpackedSize = 0x1E436, Size = 0x42F2, IsPacked = true },
                    new PackedEntry { Name = "Jofuka2.mcp", Offset = 0x1F73A0, UnpackedSize = 0x19436, Size = 0x4160, IsPacked = true },
                    new PackedEntry { Name = "Jofuka3.mcp", Offset = 0x1FB500, UnpackedSize = 0x0F436, Size = 0x2531, IsPacked = true },
                    new PackedEntry { Name = "Joido.mcp", Offset = 0x1FDA31, UnpackedSize = 0x0F436, Size = 0x3649, IsPacked = true },
                    new PackedEntry { Name = "Joisya.mcp", Offset = 0x20107A, UnpackedSize = 0x2D436, Size = 0x7BF9, IsPacked = true },
                    new PackedEntry { Name = "Jomin1.mcp", Offset = 0x208C73, UnpackedSize = 0x23436, Size = 0x5ADD, IsPacked = true },
                    new PackedEntry { Name = "Jomin2.mcp", Offset = 0x20E750, UnpackedSize = 0x1E436, Size = 0x50C4, IsPacked = true },
                    new PackedEntry { Name = "Jouka.mcp", Offset = 0x213814, UnpackedSize = 0x41436, Size = 0x11817, IsPacked = true },
                    new PackedEntry { Name = "Jouma.mcp", Offset = 0x22502B, UnpackedSize = 0x2D436, Size = 0x8097, IsPacked = true },
                    new PackedEntry { Name = "Joyado1.mcp", Offset = 0x22D0C2, UnpackedSize = 0x28436, Size = 0x5DC8, IsPacked = true },
                    new PackedEntry { Name = "Joyado2.mcp", Offset = 0x232E8A, UnpackedSize = 0x1E436, Size = 0x4AD1, IsPacked = true },
                    new PackedEntry { Name = "Kain1.mcp", Offset = 0x23795B, UnpackedSize = 0x37436, Size = 0x9E8A, IsPacked = true },
                    new PackedEntry { Name = "Kain2.mcp", Offset = 0x2417E5, UnpackedSize = 0x32436, Size = 0x8B6E, IsPacked = true },
                    new PackedEntry { Name = "Kaiz00.mcp", Offset = 0x24A353, UnpackedSize = 0x19436, Size = 0x4048, IsPacked = true },
                    new PackedEntry { Name = "Kaiz01.mcp", Offset = 0x24E39B, UnpackedSize = 0x14436, Size = 0x3260, IsPacked = true },
                    new PackedEntry { Name = "Kaiz02.mcp", Offset = 0x2515FB, UnpackedSize = 0x0F436, Size = 0x26AA, IsPacked = true },
                    new PackedEntry { Name = "Kaiz03.mcp", Offset = 0x253CA5, UnpackedSize = 0x19436, Size = 0x4153, IsPacked = true },
                    new PackedEntry { Name = "Kaiz04.mcp", Offset = 0x257DF8, UnpackedSize = 0x19436, Size = 0x3510, IsPacked = true },
                    new PackedEntry { Name = "Kaiz05.mcp", Offset = 0x25B308, UnpackedSize = 0x19436, Size = 0x3601, IsPacked = true },
                    new PackedEntry { Name = "Kaiz06.mcp", Offset = 0x25E909, UnpackedSize = 0x0F436, Size = 0x27CE, IsPacked = true },
                    new PackedEntry { Name = "Kaiz07.mcp", Offset = 0x2610D7, UnpackedSize = 0x28436, Size = 0x9E51, IsPacked = true },
                    new PackedEntry { Name = "Katabuki.mcp", Offset = 0x26AF28, UnpackedSize = 0x28436, Size = 0x6A79, IsPacked = true },
                    new PackedEntry { Name = "Katadogu.mcp", Offset = 0x2719A1, UnpackedSize = 0x23436, Size = 0x59A9, IsPacked = true },
                    new PackedEntry { Name = "Katamin1.mcp", Offset = 0x27734A, UnpackedSize = 0x23436, Size = 0x5F75, IsPacked = true },
                    new PackedEntry { Name = "Katamin2.mcp", Offset = 0x27D2BF, UnpackedSize = 0x2D436, Size = 0x81D8, IsPacked = true },
                    new PackedEntry { Name = "Katamina.mcp", Offset = 0x285497, UnpackedSize = 0x41436, Size = 0x0C628, IsPacked = true },
                    new PackedEntry { Name = "Katania.mcp", Offset = 0x291ABF, UnpackedSize = 0x46436, Size = 0x12EE8, IsPacked = true },
                    new PackedEntry { Name = "Kataryo1.mcp", Offset = 0x2A49A7, UnpackedSize = 0x28436, Size = 0x6C0A, IsPacked = true },
                    new PackedEntry { Name = "Kataryo2.mcp", Offset = 0x2AB5B1, UnpackedSize = 0x32436, Size = 0x89D1, IsPacked = true },
                    new PackedEntry { Name = "Katasou1.mcp", Offset = 0x2B3F82, UnpackedSize = 0x14436, Size = 0x2A01, IsPacked = true },
                    new PackedEntry { Name = "Katasou2.mcp", Offset = 0x2B6983, UnpackedSize = 0x14436, Size = 0x3122, IsPacked = true },
                    new PackedEntry { Name = "Katasyo.mcp", Offset = 0x2B9AA5, UnpackedSize = 0x28436, Size = 0x7EBC, IsPacked = true },
                    new PackedEntry { Name = "Katayad1.mcp", Offset = 0x2C1961, UnpackedSize = 0x23436, Size = 0x6050, IsPacked = true },
                    new PackedEntry { Name = "Katayad2.mcp", Offset = 0x2C79B1, UnpackedSize = 0x1E436, Size = 0x4F1D, IsPacked = true },
                    new PackedEntry { Name = "Koro00.mcp", Offset = 0x2CC8CE, UnpackedSize = 0x23436, Size = 0x6B2E, IsPacked = true },
                    new PackedEntry { Name = "Koro01.mcp", Offset = 0x2D33FC, UnpackedSize = 0x19436, Size = 0x5926, IsPacked = true },
                    new PackedEntry { Name = "Koro02.mcp", Offset = 0x2D8D22, UnpackedSize = 0x14436, Size = 0x3B9D, IsPacked = true },
                    new PackedEntry { Name = "Koro03.mcp", Offset = 0x2DC8BF, UnpackedSize = 0x28436, Size = 0x7C95, IsPacked = true },
                    new PackedEntry { Name = "Kuro00.mcp", Offset = 0x2E4554, UnpackedSize = 0x23436, Size = 0x9D0A, IsPacked = true },
                    new PackedEntry { Name = "Kuro01.mcp", Offset = 0x2EE25E, UnpackedSize = 0x32436, Size = 0x0CCE7, IsPacked = true },
                    new PackedEntry { Name = "Kuro02.mcp", Offset = 0x2FAF45, UnpackedSize = 0x32436, Size = 0x0CE36, IsPacked = true },
                    new PackedEntry { Name = "Kuro03.mcp", Offset = 0x307D7B, UnpackedSize = 0x41436, Size = 0x0F30A, IsPacked = true },
                    new PackedEntry { Name = "Last00.mcp", Offset = 0x317085, UnpackedSize = 0x28436, Size = 0x0A99C, IsPacked = true },
                    new PackedEntry { Name = "Last01.mcp", Offset = 0x321A21, UnpackedSize = 0x0F436, Size = 0x32D3, IsPacked = true },
                    new PackedEntry { Name = "Last02.mcp", Offset = 0x324CF4, UnpackedSize = 0x0F436, Size = 0x2EAA, IsPacked = true },
                    new PackedEntry { Name = "Last03.mcp", Offset = 0x327B9E, UnpackedSize = 0x0F436, Size = 0x2E77, IsPacked = true },
                    new PackedEntry { Name = "Last04.mcp", Offset = 0x32AA15, UnpackedSize = 0x0F436, Size = 0x2EAA, IsPacked = true },
                    new PackedEntry { Name = "Last05.mcp", Offset = 0x32D8BF, UnpackedSize = 0x0F436, Size = 0x33A3, IsPacked = true },
                    new PackedEntry { Name = "Last06.mcp", Offset = 0x330C62, UnpackedSize = 0x0A436, Size = 0x2738, IsPacked = true },
                    new PackedEntry { Name = "Last07.mcp", Offset = 0x33339A, UnpackedSize = 0x0A436, Size = 0x2644, IsPacked = true },
                    new PackedEntry { Name = "Last08.mcp", Offset = 0x3359DE, UnpackedSize = 0x0F436, Size = 0x29FA, IsPacked = true },
                    new PackedEntry { Name = "Last09.mcp", Offset = 0x3383D8, UnpackedSize = 0x0F436, Size = 0x29FA, IsPacked = true },
                    new PackedEntry { Name = "Last10.mcp", Offset = 0x33ADD2, UnpackedSize = 0x0F436, Size = 0x2CA3, IsPacked = true },
                    new PackedEntry { Name = "Last11.mcp", Offset = 0x33DA75, UnpackedSize = 0x28436, Size = 0x8A4A, IsPacked = true },
                    new PackedEntry { Name = "Last12.mcp", Offset = 0x3464BF, UnpackedSize = 0x1E436, Size = 0x6B7E, IsPacked = true },
                    new PackedEntry { Name = "Last13.mcp", Offset = 0x34D03D, UnpackedSize = 0x1E436, Size = 0x6CB9, IsPacked = true },
                    new PackedEntry { Name = "Last14.mcp", Offset = 0x353CF6, UnpackedSize = 0x2D436, Size = 0x0C33E, IsPacked = true },
                    new PackedEntry { Name = "Last15.mcp", Offset = 0x360034, UnpackedSize = 0x1E436, Size = 0x71E3, IsPacked = true },
                    new PackedEntry { Name = "Last16.mcp", Offset = 0x367217, UnpackedSize = 0x32436, Size = 0x0CE2D, IsPacked = true },
                    new PackedEntry { Name = "Last17.mcp", Offset = 0x374044, UnpackedSize = 0x19436, Size = 0x57E0, IsPacked = true },
                    new PackedEntry { Name = "Last18.mcp", Offset = 0x379824, UnpackedSize = 0x2D436, Size = 0x0BB0F, IsPacked = true },
                    new PackedEntry { Name = "Last19.mcp", Offset = 0x385333, UnpackedSize = 0x1E436, Size = 0x6C5E, IsPacked = true },
                    new PackedEntry { Name = "Last20.mcp", Offset = 0x38BF91, UnpackedSize = 0x23436, Size = 0x9708, IsPacked = true },
                    new PackedEntry { Name = "Last21.mcp", Offset = 0x395699, UnpackedSize = 0x32436, Size = 0x9AEE, IsPacked = true },
                    new PackedEntry { Name = "Last22.mcp", Offset = 0x39F187, UnpackedSize = 0x37436, Size = 0x0AA8D, IsPacked = true },
                    new PackedEntry { Name = "Last23.mcp", Offset = 0x3A9C14, UnpackedSize = 0x0F436, Size = 0x302C, IsPacked = true },
                    new PackedEntry { Name = "Last24.mcp", Offset = 0x3ACC40, UnpackedSize = 0x37436, Size = 0x0A078, IsPacked = true },
                    new PackedEntry { Name = "Matu00.mcp", Offset = 0x3B6CB8, UnpackedSize = 0x41436, Size = 0x11DBE, IsPacked = true },
                    new PackedEntry { Name = "Ojou01.mcp", Offset = 0x3C8A76, UnpackedSize = 0x3C436, Size = 0x0CF45, IsPacked = true },
                    new PackedEntry { Name = "Ojou02.mcp", Offset = 0x3D59BB, UnpackedSize = 0x28436, Size = 0x7CC1, IsPacked = true },
                    new PackedEntry { Name = "Ojou03.mcp", Offset = 0x3DD67C, UnpackedSize = 0x46436, Size = 0x0FBCF, IsPacked = true },
                    new PackedEntry { Name = "Ojou04.mcp", Offset = 0x3ED24B, UnpackedSize = 0x46436, Size = 0x0DEEC, IsPacked = true },
                    new PackedEntry { Name = "Ojou05.mcp", Offset = 0x3FB137, UnpackedSize = 0x23436, Size = 0x662B, IsPacked = true },
                    new PackedEntry { Name = "Ojou06.mcp", Offset = 0x401762, UnpackedSize = 0x1E436, Size = 0x5782, IsPacked = true },
                    new PackedEntry { Name = "Ojou07.mcp", Offset = 0x406EE4, UnpackedSize = 0x1E436, Size = 0x5D2F, IsPacked = true },
                    new PackedEntry { Name = "Ojou08.mcp", Offset = 0x40CC13, UnpackedSize = 0x19436, Size = 0x584C, IsPacked = true },
                    new PackedEntry { Name = "Ojou09.mcp", Offset = 0x41245F, UnpackedSize = 0x23436, Size = 0x71DA, IsPacked = true },
                    new PackedEntry { Name = "Ojou10.mcp", Offset = 0x419639, UnpackedSize = 0x23436, Size = 0x71DA, IsPacked = true },
                    new PackedEntry { Name = "Ojou11.mcp", Offset = 0x420813, UnpackedSize = 0x23436, Size = 0x7261, IsPacked = true },
                    new PackedEntry { Name = "Ojou12.mcp", Offset = 0x427A74, UnpackedSize = 0x14436, Size = 0x416B, IsPacked = true },
                    new PackedEntry { Name = "Ojou13.mcp", Offset = 0x42BBDF, UnpackedSize = 0x14436, Size = 0x4A1F, IsPacked = true },
                    new PackedEntry { Name = "Ojou14.mcp", Offset = 0x4305FE, UnpackedSize = 0x14436, Size = 0x27E4, IsPacked = true },
                    new PackedEntry { Name = "Ojou15.mcp", Offset = 0x432DE2, UnpackedSize = 0x14436, Size = 0x47C5, IsPacked = true },
                    new PackedEntry { Name = "Ojou16.mcp", Offset = 0x4375A7, UnpackedSize = 0x1E436, Size = 0x5C02, IsPacked = true },
                    new PackedEntry { Name = "Ojou17.mcp", Offset = 0x43D1A9, UnpackedSize = 0x19436, Size = 0x5819, IsPacked = true },
                    new PackedEntry { Name = "Ojou18.mcp", Offset = 0x4429C2, UnpackedSize = 0x1E436, Size = 0x5DE8, IsPacked = true },
                    new PackedEntry { Name = "Ojou19.mcp", Offset = 0x4487AA, UnpackedSize = 0x1E436, Size = 0x619C, IsPacked = true },
                    new PackedEntry { Name = "Ojou20.mcp", Offset = 0x44E946, UnpackedSize = 0x0F436, Size = 0x3818, IsPacked = true },
                    new PackedEntry { Name = "Ojou21.mcp", Offset = 0x45215E, UnpackedSize = 0x1E436, Size = 0x5932, IsPacked = true },
                    new PackedEntry { Name = "Ojou22.mcp", Offset = 0x457A90, UnpackedSize = 0x1E436, Size = 0x5C02, IsPacked = true },
                    new PackedEntry { Name = "Oldw00.mcp", Offset = 0x45D692, UnpackedSize = 0x19436, Size = 0x3B5C, IsPacked = true },
                    new PackedEntry { Name = "Oldw01.mcp", Offset = 0x4611EE, UnpackedSize = 0x32436, Size = 0x0C387, IsPacked = true },
                    new PackedEntry { Name = "Oldw02.mcp", Offset = 0x46D575, UnpackedSize = 0x19436, Size = 0x4935, IsPacked = true },
                    new PackedEntry { Name = "Oldw03.mcp", Offset = 0x471EAA, UnpackedSize = 0x14436, Size = 0x34E0, IsPacked = true },
                    new PackedEntry { Name = "Oldw04.mcp", Offset = 0x47538A, UnpackedSize = 0x14436, Size = 0x393D, IsPacked = true },
                    new PackedEntry { Name = "Oldw05.mcp", Offset = 0x478CC7, UnpackedSize = 0x14436, Size = 0x2EA2, IsPacked = true },
                    new PackedEntry { Name = "Oldw06.mcp", Offset = 0x47BB69, UnpackedSize = 0x2D436, Size = 0x0A85A, IsPacked = true },
                    new PackedEntry { Name = "Oldw07.mcp", Offset = 0x4863C3, UnpackedSize = 0x19436, Size = 0x3994, IsPacked = true },
                    new PackedEntry { Name = "Oldw08.mcp", Offset = 0x489D57, UnpackedSize = 0x19436, Size = 0x4663, IsPacked = true },
                    new PackedEntry { Name = "Oldw09.mcp", Offset = 0x48E3BA, UnpackedSize = 0x19436, Size = 0x3EA6, IsPacked = true },
                    new PackedEntry { Name = "Oldw10.mcp", Offset = 0x492260, UnpackedSize = 0x1E436, Size = 0x3C7F, IsPacked = true },
                    new PackedEntry { Name = "Oldw11.mcp", Offset = 0x495EDF, UnpackedSize = 0x19436, Size = 0x264F, IsPacked = true },
                    new PackedEntry { Name = "Oldw12.mcp", Offset = 0x49852E, UnpackedSize = 0x28436, Size = 0x0BC6C, IsPacked = true },
                    new PackedEntry { Name = "Omake.mcp", Offset = 0x4A419A, UnpackedSize = 0x23436, Size = 0x638B, IsPacked = true },
                    new PackedEntry { Name = "Omake2.mcp", Offset = 0x4AA525, UnpackedSize = 0x19436, Size = 0x4775, IsPacked = true },
                    new PackedEntry { Name = "Paremina.mcp", Offset = 0x4AEC9A, UnpackedSize = 0x2D436, Size = 0x7E22, IsPacked = true },
                    new PackedEntry { Name = "Parerumo.mcp", Offset = 0x4B6ABC, UnpackedSize = 0x46436, Size = 0x1178A, IsPacked = true },
                    new PackedEntry { Name = "Pargil.mcp", Offset = 0x4C8246, UnpackedSize = 0x1E436, Size = 0x5710, IsPacked = true },
                    new PackedEntry { Name = "Parsaka.mcp", Offset = 0x4CD956, UnpackedSize = 0x1E436, Size = 0x5054, IsPacked = true },
                    new PackedEntry { Name = "Paruma.mcp", Offset = 0x4D29AA, UnpackedSize = 0x2D436, Size = 0x8097, IsPacked = true },
                    new PackedEntry { Name = "Paryad1.mcp", Offset = 0x4DAA41, UnpackedSize = 0x23436, Size = 0x55C8, IsPacked = true },
                    new PackedEntry { Name = "Paryad2.mcp", Offset = 0x4E0009, UnpackedSize = 0x1E436, Size = 0x4B6B, IsPacked = true },
                    new PackedEntry { Name = "Saba00.mcp", Offset = 0x4E4B74, UnpackedSize = 0x19436, Size = 0x59AE, IsPacked = true },
                    new PackedEntry { Name = "Saba01.mcp", Offset = 0x4EA522, UnpackedSize = 0x14436, Size = 0x4715, IsPacked = true },
                    new PackedEntry { Name = "Saba02.mcp", Offset = 0x4EEC37, UnpackedSize = 0x19436, Size = 0x5C8A, IsPacked = true },
                    new PackedEntry { Name = "Saba03.mcp", Offset = 0x4F48C1, UnpackedSize = 0x19436, Size = 0x4EAD, IsPacked = true },
                    new PackedEntry { Name = "Saba04.mcp", Offset = 0x4F976E, UnpackedSize = 0x14436, Size = 0x430B, IsPacked = true },
                    new PackedEntry { Name = "Sakaba.mcp", Offset = 0x4FDA79, UnpackedSize = 0x2D436, Size = 0x8231, IsPacked = true },
                    new PackedEntry { Name = "Sepu00.mcp", Offset = 0x505CAA, UnpackedSize = 0x46436, Size = 0x0FB50, IsPacked = true },
                    new PackedEntry { Name = "Sepu01.mcp", Offset = 0x5157FA, UnpackedSize = 0x14436, Size = 0x3950, IsPacked = true },
                    new PackedEntry { Name = "Sepu02.mcp", Offset = 0x51914A, UnpackedSize = 0x14436, Size = 0x3B7D, IsPacked = true },
                    new PackedEntry { Name = "Sepu03.mcp", Offset = 0x51CCC7, UnpackedSize = 0x19436, Size = 0x3E88, IsPacked = true },
                    new PackedEntry { Name = "Sepu04.mcp", Offset = 0x520B4F, UnpackedSize = 0x19436, Size = 0x3B13, IsPacked = true },
                    new PackedEntry { Name = "Sepu05.mcp", Offset = 0x524662, UnpackedSize = 0x23436, Size = 0x6BED, IsPacked = true },
                    new PackedEntry { Name = "Sepu06.mcp", Offset = 0x52B24F, UnpackedSize = 0x19436, Size = 0x3745, IsPacked = true },
                    new PackedEntry { Name = "Sepu07.mcp", Offset = 0x52E994, UnpackedSize = 0x1E436, Size = 0x42AA, IsPacked = true },
                    new PackedEntry { Name = "Sepu08.mcp", Offset = 0x532C3E, UnpackedSize = 0x19436, Size = 0x3E45, IsPacked = true },
                    new PackedEntry { Name = "Sepu09.mcp", Offset = 0x536A83, UnpackedSize = 0x1E436, Size = 0x56CA, IsPacked = true },
                    new PackedEntry { Name = "Sepu10.mcp", Offset = 0x53C14D, UnpackedSize = 0x19436, Size = 0x4B66, IsPacked = true },
                    new PackedEntry { Name = "Sepu11.mcp", Offset = 0x540CB3, UnpackedSize = 0x14436, Size = 0x4163, IsPacked = true },
                    new PackedEntry { Name = "Sepu12.mcp", Offset = 0x544E16, UnpackedSize = 0x23436, Size = 0x690B, IsPacked = true },
                    new PackedEntry { Name = "Sepu13.mcp", Offset = 0x54B721, UnpackedSize = 0x14436, Size = 0x415F, IsPacked = true },
                    new PackedEntry { Name = "Sepu14.mcp", Offset = 0x54F880, UnpackedSize = 0x1E436, Size = 0x5A52, IsPacked = true },
                    new PackedEntry { Name = "Sima00.mcp", Offset = 0x5552D2, UnpackedSize = 0x23436, Size = 0x9D0A, IsPacked = true },
                    new PackedEntry { Name = "Sima01.mcp", Offset = 0x55EFDC, UnpackedSize = 0x19436, Size = 0x3E60, IsPacked = true },
                    new PackedEntry { Name = "Sima02.mcp", Offset = 0x562E3C, UnpackedSize = 0x19436, Size = 0x3BDD, IsPacked = true },
                    new PackedEntry { Name = "Sima03.mcp", Offset = 0x566A19, UnpackedSize = 0x1E436, Size = 0x4B47, IsPacked = true },
                    new PackedEntry { Name = "Sima04.mcp", Offset = 0x56B560, UnpackedSize = 0x1E436, Size = 0x4283, IsPacked = true },
                    new PackedEntry { Name = "Sima05.mcp", Offset = 0x56F7E3, UnpackedSize = 0x1E436, Size = 0x489C, IsPacked = true },
                    new PackedEntry { Name = "Sima06.mcp", Offset = 0x57407F, UnpackedSize = 0x19436, Size = 0x3BFD, IsPacked = true },
                    new PackedEntry { Name = "Sima07.mcp", Offset = 0x577C7C, UnpackedSize = 0x0A436, Size = 0x1C88, IsPacked = true },
                    new PackedEntry { Name = "Sougen.mcp", Offset = 0x579904, UnpackedSize = 0x14436, Size = 0x61A0, IsPacked = true },
                    new PackedEntry { Name = "Syoko01.mcp", Offset = 0x57FAA4, UnpackedSize = 0x41436, Size = 0x0B6AD, IsPacked = true },
                    new PackedEntry { Name = "Syoko02.mcp", Offset = 0x58B151, UnpackedSize = 0x3C436, Size = 0x0A967, IsPacked = true },
                    new PackedEntry { Name = "Syud00.mcp", Offset = 0x595AB8, UnpackedSize = 0x1E436, Size = 0x72AE, IsPacked = true },
                    new PackedEntry { Name = "Syud01.mcp", Offset = 0x59CD66, UnpackedSize = 0x37436, Size = 0x9EF4, IsPacked = true },
                    new PackedEntry { Name = "Tidu00.mcp", Offset = 0x5A6C5A, UnpackedSize = 0x23436, Size = 0x6DA2, IsPacked = true },
                    new PackedEntry { Name = "Tidu01.mcp", Offset = 0x5AD9FC, UnpackedSize = 0x1E436, Size = 0x6F94, IsPacked = true },
                    new PackedEntry { Name = "Tidu02.mcp", Offset = 0x5B4990, UnpackedSize = 0x1E436, Size = 0x78B8, IsPacked = true },
                    new PackedEntry { Name = "Tidu03.mcp", Offset = 0x5BC248, UnpackedSize = 0x1E436, Size = 0x742F, IsPacked = true },
                    new PackedEntry { Name = "Tidu04.mcp", Offset = 0x5C3677, UnpackedSize = 0x1E436, Size = 0x750F, IsPacked = true },
                    new PackedEntry { Name = "Tidu05.mcp", Offset = 0x5CAB86, UnpackedSize = 0x1E436, Size = 0x723E, IsPacked = true },
                    new PackedEntry { Name = "Tita00.mcp", Offset = 0x5D1DC4, UnpackedSize = 0x1E436, Size = 0x688B, IsPacked = true },
                    new PackedEntry { Name = "Tita01.mcp", Offset = 0x5D864F, UnpackedSize = 0x0F436, Size = 0x3162, IsPacked = true },
                    new PackedEntry { Name = "Tita02.mcp", Offset = 0x5DB7B1, UnpackedSize = 0x0F436, Size = 0x3128, IsPacked = true },
                    new PackedEntry { Name = "Tita03.mcp", Offset = 0x5DE8D9, UnpackedSize = 0x0F436, Size = 0x3128, IsPacked = true },
                    new PackedEntry { Name = "Tita04.mcp", Offset = 0x5E1A01, UnpackedSize = 0x0F436, Size = 0x31A0, IsPacked = true },
                    new PackedEntry { Name = "Tita05.mcp", Offset = 0x5E4BA1, UnpackedSize = 0x0F436, Size = 0x31A0, IsPacked = true },
                    new PackedEntry { Name = "Tita06.mcp", Offset = 0x5E7D41, UnpackedSize = 0x0F436, Size = 0x31B3, IsPacked = true },
                    new PackedEntry { Name = "Tita07.mcp", Offset = 0x5EAEF4, UnpackedSize = 0x19436, Size = 0x3809, IsPacked = true },
                    new PackedEntry { Name = "Tori00.mcp", Offset = 0x5EE6FD, UnpackedSize = 0x37436, Size = 0x0F315, IsPacked = true },
                    new PackedEntry { Name = "Tori01.mcp", Offset = 0x5FDA12, UnpackedSize = 0x23436, Size = 0x6D20, IsPacked = true },
                    new PackedEntry { Name = "Tori02.mcp", Offset = 0x604732, UnpackedSize = 0x28436, Size = 0x73D0, IsPacked = true },
                    new PackedEntry { Name = "Tori03.mcp", Offset = 0x60BB02, UnpackedSize = 0x23436, Size = 0x68FA, IsPacked = true },
                    new PackedEntry { Name = "Tori04.mcp", Offset = 0x6123FC, UnpackedSize = 0x14436, Size = 0x2882, IsPacked = true },
                    new PackedEntry { Name = "Tori05.mcp", Offset = 0x614C7E, UnpackedSize = 0x28436, Size = 0x70F3, IsPacked = true },
                    new PackedEntry { Name = "Tori06.mcp", Offset = 0x61BD71, UnpackedSize = 0x37436, Size = 0x947D, IsPacked = true },
                    new PackedEntry { Name = "Tori07.mcp", Offset = 0x6251EE, UnpackedSize = 0x19436, Size = 0x4478, IsPacked = true },
                    new PackedEntry { Name = "Tori08.mcp", Offset = 0x629666, UnpackedSize = 0x1E436, Size = 0x4CD1, IsPacked = true },
                    new PackedEntry { Name = "Yama01.mcp", Offset = 0x62E337, UnpackedSize = 0x23436, Size = 0x9483, IsPacked = true },
                    new PackedEntry { Name = "Yama02.mcp", Offset = 0x6377BA, UnpackedSize = 0x2D436, Size = 0x0BE5E, IsPacked = true },
                    new PackedEntry { Name = "Yama03.mcp", Offset = 0x643618, UnpackedSize = 0x28436, Size = 0x92D5, IsPacked = true },
                    new PackedEntry { Name = "Yama04.mcp", Offset = 0x64C8ED, UnpackedSize = 0x23436, Size = 0x85AF, IsPacked = true },
                    new PackedEntry { Name = "Yama05.mcp", Offset = 0x654E9C, UnpackedSize = 0x23436, Size = 0x8927, IsPacked = true },
                    new PackedEntry { Name = "Yama06.mcp", Offset = 0x65D7C3, UnpackedSize = 0x3C436, Size = 0x0D40C, IsPacked = true },
                    new PackedEntry { Name = "Yujin.mcp", Offset = 0x66ABCF, UnpackedSize = 0x32436, Size = 0x9342, IsPacked = true },
                    new PackedEntry { Name = "Yuru00.mcp", Offset = 0x673F11, UnpackedSize = 0x46436, Size = 0x11CB7, IsPacked = true },
                    new PackedEntry { Name = "Yuru01.mcp", Offset = 0x685BC8, UnpackedSize = 0x28436, Size = 0x63B9, IsPacked = true },
                    new PackedEntry { Name = "Yuru02.mcp", Offset = 0x68BF81, UnpackedSize = 0x23436, Size = 0x700A, IsPacked = true },
                    new PackedEntry { Name = "Yuru03.mcp", Offset = 0x692F8B, UnpackedSize = 0x28436, Size = 0x7E6D, IsPacked = true },
                    new PackedEntry { Name = "Yuru04.mcp", Offset = 0x69ADF8, UnpackedSize = 0x23436, Size = 0x6365, IsPacked = true },
                    new PackedEntry { Name = "Yuru05.mcp", Offset = 0x6A115D, UnpackedSize = 0x37436, Size = 0x9BCF, IsPacked = true },
                    new PackedEntry { Name = "Yuru06.mcp", Offset = 0x6AAD2C, UnpackedSize = 0x28436, Size = 0x7C33, IsPacked = true },
                    new PackedEntry { Name = "Yuru07.mcp", Offset = 0x6B295F, UnpackedSize = 0x28436, Size = 0x8999, IsPacked = true },
                    new PackedEntry { Name = "Yuru08.mcp", Offset = 0x6BB2F8, UnpackedSize = 0x28436, Size = 0x6847, IsPacked = true },
                    new PackedEntry { Name = "Yuru09.mcp", Offset = 0x6C1B3F, UnpackedSize = 0x23436, Size = 0x6286, IsPacked = true },
                }
            },
        };
    }
}
