//! \file       ArcPKK.cs
//! \date       2018 Oct 30
//! \brief      Electriciteit resource archive.
//
// Copyright (C) 2018 by morkt
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
//

using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.IO;
using GameRes.Utility;

// [010330][e-Erekiteru] Suiso ~1/2 no Kiseki~

namespace GameRes.Formats.Electriciteit
{
    [Export(typeof(ArchiveFormat))]
    public class PkkOpener : ArchiveFormat
    {
        public override string         Tag { get { return "PKK"; } }
        public override string Description { get { return "Electriciteit resource archive"; } }
        public override uint     Signature { get { return 0; } }
        public override bool  IsHierarchic { get { return false; } }
        public override bool      CanWrite { get { return false; } }

        public PkkOpener ()
        {
            Signatures = new uint[] { 0x695ECD6F, 0x695ECD6E };
            Extensions = new[] { "pkk", "skn" };
        }

        public override ArcFile TryOpen (ArcView file)
        {
            var header = file.View.ReadBytes (0, 0x14);
            Decrypt (header, DefaultKey);

            uint signature = header.ToUInt32 (0);
            if (signature != 0 && signature != 1)
                return null;
            int count = header.ToInt32 (0x10);
            if (!IsSaneCount (count))
                return null;
            uint index_offset = header.ToUInt32 (0xC);
            if (index_offset < 0x14 || index_offset >= file.MaxOffset)
                return null;
            var index = file.View.ReadBytes (index_offset, (uint)count * 0x28);
            Decrypt (index, DefaultKey);

            long data_offset = index_offset + index.Length;
            int pos = 0;
            var dir = new List<Entry> (count);
            for (int i = 0; i < count; ++i)
            {
                var name = Binary.GetCString (index, pos+8, 0x20);
                if (string.IsNullOrWhiteSpace (name))
                    return null;
                var entry = Create<Entry> (name);
                entry.Size   = index.ToUInt32 (pos);
                entry.Offset = index.ToUInt32 (pos+4) + data_offset;
                if (!entry.CheckPlacement (file.MaxOffset))
                    return null;
                dir.Add (entry);
                pos += 0x28;
            }
            return new ArcFile (file, this, dir);
        }

        public override Stream OpenEntry (ArcFile arc, Entry entry)
        {
            var input = arc.File.CreateStream (entry.Offset, entry.Size);
            return new ByteStringEncryptedStream (input, DefaultKey);
        }

        void Decrypt (byte[] data, byte[] key)
        {
            for (int i = 0; i < data.Length; ++i)
            {
                data[i] ^= key[i & 0x1FF];
            }
        }

        static readonly byte[] DefaultKey = {
            0x6E, 0xCD, 0x5E, 0x69, 0xC7, 0x57, 0x5F, 0xB7, 0x50, 0x56, 0xA8, 0x50, 0x52, 0x9F, 0x50, 0x52,
            0x9D, 0x4F, 0x51, 0x9C, 0x4E, 0x4F, 0x9B, 0x4E, 0x4F, 0x9C, 0x50, 0x52, 0x9C, 0x53, 0x54, 0x9D,
            0x57, 0x56, 0xA1, 0x5E, 0x5D, 0xAD, 0x66, 0x6A, 0xC2, 0x6E, 0x73, 0xD3, 0x72, 0x77, 0xDA, 0x75,
            0x78, 0xDC, 0x78, 0x7C, 0xDD, 0x7B, 0x7D, 0xDD, 0x7D, 0x80, 0xDD, 0x80, 0x81, 0xDC, 0x85, 0x85,
            0xDC, 0x87, 0x88, 0xDC, 0x8A, 0x8A, 0xDB, 0x8C, 0x8C, 0xDA, 0x8D, 0x8E, 0xD9, 0x8F, 0x8E, 0xD2,
            0x8F, 0x8B, 0xC7, 0x90, 0x88, 0xBA, 0x91, 0x86, 0xB0, 0x93, 0x86, 0xAE, 0x92, 0x85, 0xAD, 0x90,
            0x84, 0xAC, 0x8D, 0x82, 0xA9, 0x8A, 0x7F, 0xA7, 0x87, 0x7A, 0xA5, 0x87, 0x7A, 0xA1, 0xA0, 0x8E,
            0x9E, 0xC6, 0xAE, 0x9E, 0xD8, 0xBC, 0x9E, 0xD4, 0xB5, 0x93, 0xC8, 0xA4, 0x83, 0xC2, 0x9C, 0x7A,
            0xC2, 0x98, 0x77, 0xC0, 0x95, 0x77, 0xC0, 0x91, 0x76, 0xC2, 0x8D, 0x76, 0xC6, 0x8B, 0x76, 0xC5,
            0x86, 0x73, 0xC4, 0x82, 0x71, 0xC2, 0x7E, 0x71, 0xC1, 0x7D, 0x72, 0xBF, 0x7B, 0x71, 0xBD, 0x79,
            0x70, 0xBB, 0x78, 0x70, 0xBB, 0x77, 0x70, 0xBA, 0x75, 0x71, 0xB8, 0x75, 0x71, 0xB8, 0x75, 0x71,
            0xB7, 0x75, 0x70, 0xB6, 0x73, 0x6F, 0xB3, 0x71, 0x6F, 0xB2, 0x70, 0x6F, 0xB1, 0x70, 0x70, 0xB1,
            0x6F, 0x71, 0xAF, 0x6E, 0x72, 0xAE, 0x6E, 0x72, 0xAF, 0x6D, 0x74, 0xB0, 0x6D, 0x74, 0xB0, 0x6C,
            0x74, 0xB0, 0x6C, 0x74, 0xB1, 0x6C, 0x76, 0xB2, 0x6D, 0x77, 0xB5, 0x6E, 0x7A, 0xB3, 0x6D, 0x7A,
            0xB2, 0x6B, 0x79, 0xB3, 0x6C, 0x7A, 0x55, 0x52, 0x95, 0x58, 0x54, 0x95, 0x5A, 0x55, 0x97, 0x5B,
            0x57, 0x97, 0x5C, 0x57, 0x96, 0x5F, 0x57, 0x97, 0x60, 0x59, 0x98, 0x62, 0x5A, 0x98, 0x63, 0x5A,
            0x98, 0x63, 0x5A, 0x98, 0x65, 0x5B, 0x97, 0x65, 0x5B, 0x97, 0x66, 0x5B, 0x96, 0x66, 0x5A, 0x95,
            0x66, 0x59, 0x94, 0x67, 0x59, 0x94, 0x67, 0x59, 0x94, 0x67, 0x58, 0x93, 0x66, 0x57, 0x91, 0x64,
            0x57, 0x8F, 0x63, 0x56, 0x8F, 0x63, 0x54, 0x8E, 0x61, 0x51, 0x8E, 0x5E, 0x4E, 0x8C, 0x5B, 0x4C,
            0x8B, 0x57, 0x48, 0x89, 0x51, 0x44, 0x87, 0x4A, 0x3F, 0x83, 0x45, 0x3A, 0x80, 0x3D, 0x32, 0x7D,
            0x37, 0x2D, 0x7B, 0x31, 0x28, 0x77, 0x2B, 0x23, 0x75, 0x23, 0x1C, 0x6F, 0x1B, 0x18, 0x6A, 0x16,
            0x14, 0x66, 0x14, 0x13, 0x64, 0x11, 0x12, 0x60, 0x11, 0x11, 0x60, 0x11, 0x12, 0x5F, 0x11, 0x11,
            0x5D, 0x0F, 0x11, 0x5D, 0x11, 0x11, 0x5E, 0x11, 0x11, 0x5F, 0x14, 0x12, 0x60, 0x14, 0x13, 0x61,
            0x17, 0x14, 0x62, 0x17, 0x15, 0x60, 0x15, 0x13, 0x5E, 0x10, 0x10, 0x55, 0x0F, 0x0E, 0x52, 0x0F,
            0x0E, 0x54, 0x0F, 0x0E, 0x56, 0x11, 0x0E, 0x53, 0x0F, 0x0D, 0x4B, 0x10, 0x10, 0x4A, 0x15, 0x13,
            0x54, 0x19, 0x17, 0x61, 0x17, 0x14, 0x5F, 0x15, 0x10, 0x5C, 0x15, 0x0F, 0x5B, 0x15, 0x0F, 0x5B,
            0x12, 0x0F, 0x5B, 0x11, 0x0F, 0x5C, 0x11, 0x0F, 0x5C, 0x11, 0x0F, 0x5B, 0x10, 0x0F, 0x5C, 0x10,
            0x0F, 0x5D, 0x0F, 0x0F, 0x5D, 0x0F, 0x0F, 0x5D, 0x0F, 0x0F, 0x5D, 0x0F, 0x0F, 0x5D, 0x0F, 0x0F,
            0x5C, 0x0F, 0x0F, 0x5B, 0x0F, 0x0F, 0x5B, 0x0F, 0x0F, 0x5A, 0x0F, 0x0F, 0x59, 0x0F, 0x0F, 0x59,
            0x0F, 0x0F, 0x5C, 0x0F, 0x0F, 0x5C, 0x0F, 0x0F, 0x5D, 0x0F, 0x0F, 0x5D, 0x0F, 0x0F, 0x5D, 0x0F,
            0x0F, 0x5F, 0x0F, 0x10, 0x60, 0x0F, 0x10, 0x60, 0x0F, 0x10, 0x61, 0x0F, 0x10, 0x64, 0x0F, 0x10,
            0x66, 0x11, 0x11, 0x67, 0x12, 0x11, 0x68, 0x13, 0x11, 0x69, 0x14, 0x11, 0x6A, 0x15, 0x11, 0x6A,
        };
    }
}
