//! \file       Encryption.cs
//! \date       Sat Oct 01 22:25:09 2016
//! \brief      Encryption used in Primel the Adventure System.
//
// Copyright (C) 2016 by morkt
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
//

using System;
using System.Security.Cryptography;
using System.Linq;
using GameRes.Utility;

namespace GameRes.Formats.Primel
{
    internal abstract class PrimelEncyptionBase : ICryptoTransform
    {
        internal const int BlockSize = 16;

        public bool CanTransformMultipleBlocks { get { return true; } }
        public bool          CanReuseTransform { get { return false; } }
        public int              InputBlockSize { get { return BlockSize; } }
        public int             OutputBlockSize { get { return BlockSize; } }

        #region Static tables
        internal static readonly uint[] DefaultKey = {
            0x243F6A89, 0xB7E15163, 0x9E3779B9, 0x6F2DEC55, 
            0x6A09E668, 0xBB67AE86, 0xB504F334, 0x93CD3A2D, 
        };

        internal static readonly byte[,] CodeTable = new byte[4,256] {
            {
                0x29, 0x4e, 0xf8, 0x6c, 0x4d, 0xd1, 0x52, 0x5b, 0x2f, 0x4b, 0xe5, 0x0a, 0xa4, 0xbb, 0xf7, 0xdf, 
                0x9f, 0x8c, 0x27, 0xb9, 0x25, 0xfd, 0x07, 0x4a, 0x7c, 0x81, 0x10, 0x26, 0x46, 0x11, 0xa3, 0xdb, 
                0x39, 0x08, 0x55, 0x36, 0x96, 0xc4, 0x5c, 0xfe, 0x2e, 0x66, 0xa6, 0x74, 0x45, 0x60, 0xf3, 0x83, 
                0x72, 0x3f, 0xc5, 0x99, 0x0d, 0x16, 0x3d, 0x6d, 0x5d, 0x57, 0xae, 0x35, 0xde, 0x8e, 0xa0, 0x41, 
                0x8b, 0x34, 0xd7, 0xe3, 0xbe, 0x33, 0xb0, 0x44, 0xeb, 0xec, 0xff, 0x91, 0x6e, 0x85, 0x9d, 0x12, 
                0xaf, 0x8d, 0xe9, 0xba, 0x1c, 0x05, 0xee, 0x3a, 0x03, 0xfa, 0x7b, 0x23, 0xd4, 0x8a, 0xd9, 0x67, 
                0xca, 0x97, 0x02, 0x70, 0xd8, 0xfc, 0xe4, 0x78, 0x73, 0x19, 0x5e, 0x77, 0x59, 0x69, 0xe0, 0xd6, 
                0x14, 0x37, 0x00, 0x6b, 0x13, 0x17, 0x0f, 0x49, 0xaa, 0x22, 0x89, 0xe1, 0x7a, 0xad, 0x6a, 0x30, 
                0x2b, 0xb4, 0xfb, 0xb6, 0x88, 0xf0, 0x7e, 0xa1, 0x18, 0xe7, 0xd2, 0x3c, 0x9a, 0x84, 0xa7, 0x54, 
                0x40, 0x64, 0x90, 0x9b, 0x3e, 0x6f, 0xcf, 0x80, 0xf4, 0x93, 0x75, 0x1b, 0x51, 0xd3, 0x79, 0xe2, 
                0x42, 0x15, 0xe8, 0xb3, 0x0b, 0x9e, 0x31, 0xda, 0xac, 0x50, 0x7d, 0x87, 0xea, 0xdd, 0x04, 0x56, 
                0xc9, 0x28, 0x1e, 0x94, 0x63, 0x47, 0x8f, 0xc6, 0x1a, 0xc7, 0xf1, 0xf2, 0x71, 0xce, 0x1d, 0x53, 
                0x06, 0x0c, 0xdc, 0x82, 0x48, 0xbc, 0xcb, 0xa2, 0xab, 0x4f, 0x98, 0xb2, 0xcd, 0x1f, 0xa8, 0x3b, 
                0xef, 0xf5, 0x58, 0x20, 0xb8, 0xc0, 0xc1, 0x21, 0x2c, 0x86, 0x32, 0xa9, 0x2d, 0xc3, 0xbd, 0x62, 
                0xb7, 0x76, 0x95, 0xd5, 0x9c, 0x68, 0xe6, 0x92, 0x2a, 0xc8, 0x43, 0x4c, 0x0e, 0x7f, 0x01, 0x5f, 
                0xd0, 0x38, 0x65, 0x61, 0x24, 0xb1, 0xf6, 0xcc, 0xbf, 0xc2, 0x5a, 0x09, 0xb5, 0xed, 0xa5, 0xf9, 
            },
            {
                0xc5, 0xe6, 0xf6, 0x06, 0x9b, 0x77, 0xb8, 0x61, 0x95, 0x14, 0x6c, 0x01, 0xe5, 0xc8, 0xd3, 0x20, 
                0xfe, 0x73, 0xec, 0x3b, 0xcb, 0x07, 0xa3, 0x28, 0xde, 0x99, 0x2a, 0xc0, 0x90, 0x42, 0x10, 0x0a, 
                0x19, 0x16, 0xcc, 0xc3, 0x92, 0x86, 0x4b, 0x80, 0xc1, 0xae, 0xd1, 0x9f, 0x2f, 0x41, 0xe9, 0x63, 
                0x6e, 0x22, 0x55, 0x9c, 0x5d, 0x81, 0xa2, 0xb4, 0x0d, 0xa0, 0x4a, 0xdd, 0xd8, 0x54, 0x39, 0xfd, 
                0x8f, 0x49, 0xdc, 0x6d, 0x1e, 0x36, 0x02, 0xb3, 0x52, 0x85, 0x21, 0xe1, 0xdb, 0xe3, 0x23, 0x33, 
                0x00, 0x12, 0x04, 0x7f, 0x98, 0x79, 0xc6, 0x60, 0x8e, 0x03, 0x3c, 0x1c, 0xfb, 0x09, 0x6f, 0xad, 
                0x6b, 0xbb, 0x18, 0x3a, 0xbc, 0xf8, 0x3d, 0x9e, 0x9d, 0x26, 0xda, 0xff, 0xaf, 0xa5, 0xaa, 0x3f, 
                0x5b, 0x71, 0xa1, 0x70, 0xee, 0x4f, 0x68, 0xb0, 0x13, 0xd5, 0x87, 0x4d, 0x35, 0xf5, 0xa8, 0x17, 
                0xd4, 0xf1, 0x40, 0x57, 0x59, 0x1a, 0x93, 0x47, 0xbf, 0x05, 0x67, 0x8c, 0x1b, 0x69, 0x25, 0xc9, 
                0xbe, 0x5c, 0xea, 0x74, 0xc4, 0x2d, 0x38, 0xce, 0x5a, 0xe4, 0x94, 0x30, 0xc7, 0x65, 0x5f, 0x64, 
                0x7e, 0xe7, 0xd7, 0x8a, 0x0c, 0x2e, 0x9a, 0x82, 0xcf, 0x2b, 0x62, 0xf3, 0xb7, 0xcd, 0x3e, 0xf4, 
                0xbd, 0x83, 0xac, 0x0b, 0x7b, 0x45, 0xf9, 0x75, 0x32, 0xa4, 0x50, 0x58, 0xca, 0x78, 0xf7, 0xb6, 
                0x84, 0x51, 0x31, 0xab, 0x56, 0x11, 0x72, 0x76, 0xd9, 0x29, 0x43, 0x15, 0x4c, 0xe2, 0xe0, 0x4e, 
                0x89, 0x7a, 0xfa, 0x44, 0x6a, 0x91, 0xba, 0xb9, 0x7d, 0x2c, 0x46, 0x8b, 0xb1, 0x66, 0xdf, 0x08, 
                0x8d, 0xd6, 0xef, 0xd0, 0xb5, 0xa9, 0xeb, 0xfc, 0xed, 0x88, 0x7c, 0xf0, 0x27, 0x53, 0xe8, 0x96, 
                0x24, 0x5e, 0xd2, 0x1d, 0x0e, 0x48, 0x34, 0xa7, 0x1f, 0xc2, 0x0f, 0x37, 0xf2, 0xa6, 0xb2, 0x97, 
            },
            {
                0xe7, 0xf2, 0x6f, 0xcd, 0x4b, 0x52, 0xbc, 0x8b, 0xd1, 0x14, 0x68, 0xac, 0xd2, 0x92, 0x9f, 0x78, 
                0x4f, 0x59, 0x47, 0x15, 0x02, 0x79, 0x53, 0x93, 0x8d, 0x94, 0x03, 0x9e, 0x3e, 0x72, 0x19, 0x48, 
                0x89, 0x0f, 0x27, 0xd7, 0xe6, 0xaf, 0x91, 0x39, 0x9b, 0x44, 0xa8, 0xfe, 0x18, 0xa5, 0x8c, 0x42, 
                0x7c, 0x74, 0xea, 0x9c, 0x97, 0xe9, 0xdf, 0x65, 0x2e, 0xfb, 0xd6, 0xda, 0x5d, 0x49, 0x9d, 0x04, 
                0x81, 0xb2, 0xe4, 0x6c, 0x8f, 0xf7, 0x4d, 0x85, 0xa9, 0xb5, 0xce, 0xbf, 0xbe, 0xe3, 0x50, 0x21, 
                0x54, 0x46, 0xb6, 0x16, 0xef, 0x1e, 0x08, 0xcc, 0x6b, 0x35, 0xab, 0x7b, 0xc7, 0xa1, 0xc5, 0x90, 
                0x20, 0x63, 0x55, 0x1c, 0xc3, 0xb8, 0x3c, 0x3a, 0x29, 0xd5, 0x0b, 0x41, 0x5c, 0xc0, 0x57, 0x05, 
                0x0c, 0x0a, 0x4e, 0xa2, 0xfd, 0x28, 0x9a, 0x70, 0xdc, 0x51, 0x2a, 0xb4, 0x8e, 0x7e, 0x34, 0x10, 
                0xc1, 0x25, 0x4c, 0xe2, 0x6e, 0x84, 0x1d, 0x61, 0x17, 0xb3, 0x6d, 0xc2, 0x43, 0xeb, 0x24, 0x83, 
                0xff, 0x1b, 0x76, 0xae, 0x67, 0x87, 0x5a, 0x69, 0x58, 0xcb, 0x32, 0xdd, 0x26, 0x3f, 0x96, 0x5f, 
                0x82, 0x5e, 0x01, 0x7a, 0x2d, 0xf9, 0xf5, 0xf0, 0xad, 0x37, 0x45, 0xf8, 0x31, 0x38, 0x88, 0x75, 
                0x36, 0x7f, 0x64, 0x5b, 0x1a, 0xbd, 0xe0, 0xfa, 0x11, 0xb9, 0x33, 0xa3, 0x6a, 0x23, 0xf6, 0xdb, 
                0x66, 0x71, 0xf3, 0x77, 0x1f, 0xcf, 0x22, 0x3b, 0xe8, 0xec, 0x95, 0x98, 0x99, 0xd3, 0xca, 0x40, 
                0xb1, 0x30, 0x12, 0x3d, 0xa7, 0x06, 0x0d, 0xbb, 0xed, 0x07, 0xd4, 0xc4, 0x7d, 0xf1, 0xe1, 0xa4, 
                0x13, 0xa0, 0xde, 0x80, 0xc6, 0x2c, 0xba, 0x0e, 0xc8, 0xd9, 0xd0, 0x00, 0xee, 0xb0, 0x86, 0xaa, 
                0xfc, 0x73, 0xd8, 0xa6, 0x56, 0x2f, 0x2b, 0xb7, 0x09, 0x4a, 0xf4, 0xe5, 0x62, 0x8a, 0x60, 0xc9, 
            },
            {
                0x7a, 0x26, 0x6e, 0x12, 0x08, 0x7e, 0xe6, 0x15, 0xe8, 0x0e, 0x3d, 0x6a, 0x29, 0x5a, 0x45, 0xe4, 
                0x2f, 0x86, 0x5f, 0x76, 0xdf, 0xea, 0x09, 0xc5, 0x83, 0x16, 0xeb, 0xa0, 0x23, 0x9f, 0xd7, 0x30, 
                0x02, 0x79, 0x81, 0xaf, 0x97, 0x58, 0xd4, 0xba, 0xd5, 0xb1, 0x73, 0x6c, 0x2a, 0xab, 0x2e, 0x7b, 
                0x34, 0xd9, 0xf8, 0xb2, 0x67, 0xb6, 0x7d, 0xf3, 0x1e, 0xc1, 0x5c, 0xb7, 0x88, 0xf2, 0x0d, 0x1a, 
                0x8a, 0x5b, 0x55, 0xa8, 0xee, 0x78, 0x70, 0xed, 0x7c, 0xf9, 0xf6, 0xd3, 0x50, 0xbd, 0x56, 0xdc, 
                0x38, 0x21, 0xbc, 0x9b, 0x03, 0xf0, 0xe1, 0x04, 0x8f, 0x6d, 0x31, 0xc9, 0x8d, 0xfb, 0xbe, 0x14, 
                0x54, 0x3f, 0x91, 0x35, 0x72, 0xa7, 0xe9, 0x19, 0xfa, 0x17, 0xb8, 0x9d, 0xde, 0x24, 0x25, 0xa1, 
                0xdb, 0xcb, 0xcc, 0x39, 0xa5, 0xe0, 0xef, 0x57, 0x85, 0xbf, 0x49, 0x1b, 0x7f, 0x3b, 0x65, 0xc3, 
                0x77, 0x00, 0x10, 0x59, 0xb3, 0xc6, 0x95, 0x1c, 0x60, 0x5d, 0x41, 0xaa, 0x51, 0x71, 0xd6, 0x99, 
                0x1f, 0x52, 0x82, 0xcd, 0x2b, 0xd1, 0x61, 0xb9, 0x43, 0x63, 0x69, 0x46, 0xa2, 0x47, 0x8e, 0x36, 
                0x42, 0x87, 0xd8, 0xb4, 0xd2, 0x53, 0xec, 0x06, 0x6b, 0xc8, 0xe5, 0x33, 0x6f, 0x4b, 0x8b, 0x66, 
                0xe3, 0x74, 0xfe, 0x4d, 0xad, 0x28, 0x48, 0xe2, 0x90, 0xf5, 0x84, 0xb0, 0xda, 0xd0, 0x20, 0xac, 
                0xdd, 0x75, 0x92, 0xca, 0xf7, 0x2d, 0x3a, 0x9c, 0x0a, 0xb5, 0xc4, 0x07, 0x5e, 0x13, 0x05, 0xbb, 
                0x0c, 0xff, 0xa4, 0xf1, 0x27, 0x4e, 0xa3, 0x98, 0x11, 0xa9, 0x96, 0x93, 0x68, 0x4f, 0xfc, 0x37, 
                0x80, 0x4a, 0xcf, 0x01, 0x9a, 0xc0, 0x1d, 0x3c, 0x8c, 0x4c, 0x3e, 0x0b, 0xc7, 0xa6, 0xae, 0x0f, 
                0x2c, 0x62, 0xf4, 0x94, 0xce, 0x89, 0x40, 0x22, 0xfd, 0x18, 0x32, 0xc2, 0x9e, 0x64, 0x44, 0xe7, 
            }
        };
        #endregion

        public abstract int TransformBlock (byte[] inBuffer, int offset, int count, byte[] outBuffer, int outOffset);

        public byte[] TransformFinalBlock (byte[] inBuffer, int offset, int count)
        {
            if (count < BlockSize)
                return new ArraySegment<byte> (inBuffer, offset, count).ToArray();
            var output = new byte[count];
            int tail = count / BlockSize * BlockSize;
            count -= TransformBlock (inBuffer, offset, count, output, 0);
            if (count > 0)
                Buffer.BlockCopy (inBuffer, offset+tail, output, tail, count);
            return output;
        }

        static protected uint MutateKey (uint k)
        {
            return (uint)CodeTable[0, k & 0xFF]
                 | (uint)CodeTable[1, (k >> 8) & 0xFF] << 8
                 | (uint)CodeTable[2, (k >> 16) & 0xFF] << 16
                 | (uint)CodeTable[3, k >> 24] << 24;
        }

        #region IDisposable implementation
        bool _disposed = false;
        public void Dispose ()
        {
            Dispose (true);
            GC.SuppressFinalize (this);
        }

        protected virtual void Dispose (bool disposing)
        {
            if (!_disposed)
            {
                _disposed = true;
            }
        }
        #endregion
    }

    internal class Primel1Encyption : PrimelEncyptionBase
    {
        uint[] m_key = new uint[4];
        byte[] m_iv;

        public Primel1Encyption (byte[] key, byte[] iv)
        {
            m_iv = iv.Clone() as byte[];
            Buffer.BlockCopy (key, 0, m_key, 0, 0x10);
            uint k = 0;
            for (int i = 0; i < 0x10; ++i)
            {
                k = m_key[i & 3] ^ (k + DefaultKey[i & 7]);
                m_key[i & 3] = MutateKey (k);
            }
        }

        public override int TransformBlock (byte[] inBuffer, int offset, int count, byte[] outBuffer, int outOffset)
        {
            int out_count = count / BlockSize;
            for (int i = 0; i < out_count; ++i)
            {
                Transform (inBuffer, offset, outBuffer, outOffset);
                for (int j = 0; j < BlockSize; ++j)
                {
                    outBuffer[outOffset++] ^= m_iv[j];
                }
                Buffer.BlockCopy (inBuffer, offset, m_iv, 0, BlockSize);
                offset += BlockSize;
            }
            return out_count * BlockSize;
        }

        private void Transform (byte[] input, int src, byte[] output, int dst)
        {
            for (int k = 0; k < BlockSize / 4; ++k)
            {
                uint v = LittleEndian.ToUInt32 (input, src);
                v ^= m_key[(k - 1) & 3];
                v -= m_key[(k - 2) & 3];
                v ^= m_key[(k - 3) & 3];
                v -= m_key[(k - 4) & 3];
                LittleEndian.Pack (v, output, dst);
                src += 4;
                dst += 4;
            }
        }
    }

    internal class Primel2Encyption : PrimelEncyptionBase
    {
        int[]   m_shifts = new int[8];
        uint[]  m_key = new uint[4];
        byte[]  m_iv;

        public Primel2Encyption (byte[] key, byte[] iv)
        {
            m_iv = iv.Clone() as byte[];
            Buffer.BlockCopy (key, 0, m_key, 0, 0x10);
            uint k = 0;
            for (int i = 0; i < 0x10; ++i)
            {
                k = m_key[i & 3] ^ (k + DefaultKey[(i + 3) & 7]);
                m_key[i & 3] = MutateKey (k);
            }
            for (int i = 0; i < 4; ++i)
            {
                uint x = m_key[i];
                x = (x & 0x55555555) + ((x >> 1) & 0x55555555);
                x = (x & 0x33333333) + ((x >> 2) & 0x33333333);
                x = (x & 0x0F0F0F0F) + ((x >> 4) & 0x0F0F0F0F);
                x = (x & 0x00FF00FF) + ((x >> 8) & 0x00FF00FF);
                int s = (int)(x + (x >> 16));
                m_shifts[i + 4] = (s ^ ((s + i) >> 1)) & 0xF;
                m_shifts[i] = (s + i) & 0x1F;
            }
        }

        public override int TransformBlock (byte[] inBuffer, int offset, int count, byte[] outBuffer, int outOffset)
        {
            int out_count = count / BlockSize;
            for (int i = 0; i < out_count; ++i)
            {
                Transform (inBuffer, offset, outBuffer, outOffset);
                for (int j = 0; j < BlockSize; ++j)
                {
                    outBuffer[outOffset++] ^= m_iv[j];
                }
                Buffer.BlockCopy (inBuffer, offset, m_iv, 0, BlockSize);
                offset += BlockSize;
            }
            return out_count * BlockSize;
        }

        private void Transform (byte[] input, int src, byte[] output, int dst)
        {
            for (int k = 0; k < BlockSize / 4; ++k)
            {
                int i = (k - 1) & 3;
                int j = (k - 2) & 3;
                int m = (k - 3) & 3;
                int n = (k - 4) & 3;
                uint v = LittleEndian.ToUInt32 (input, src);
                v = Binary.RotR (v, m_shifts[i]) + m_key[i];
                v = Binary.RotR (v ^ m_key[j], m_shifts[j]);
                v = Binary.RotR (v, m_shifts[m]) - m_key[m];
                v = Binary.RotR (v ^ m_key[n], m_shifts[n]);
                LittleEndian.Pack (v, output, dst);
                src += 4;
                dst += 4;
            }
        }
    }

    internal class Primel3Encyption : PrimelEncyptionBase
    {
        int[]   m_shifts = new int[8];
        int[]   m_offsets = new int[4];
        uint[]  m_key = new uint[8];
        byte[]  m_iv;

        public Primel3Encyption (byte[] key, byte[] iv)
        {
            m_iv = iv.Clone() as byte[];
            Buffer.BlockCopy (key, 0, m_key, 0, 0x10);
            uint k = 0;
            for (int i = 0; i < 0x20; ++i)
            {
                k = m_key[i & 7] ^ (k + DefaultKey[(i - 3) & 7]);
                m_key[i & 7] = MutateKey (k);
                m_offsets[i & 3] = (int)(k >> 7) & 0xF;
            }
            for (int i = 0; i < 8; ++i)
            {
                uint x = m_key[i];
                x = (x & 0x55555555) + ((x >> 1) & 0x55555555);
                x = (x & 0x33333333) + ((x >> 2) & 0x33333333);
                x = (x & 0x0F0F0F0F) + ((x >> 4) & 0x0F0F0F0F);
                x = (x & 0x00FF00FF) + ((x >> 8) & 0x00FF00FF);
                int s = (int)(x + (x >> 16));
                m_shifts[i] = (s + i) & 0x1F;
            }
        }

        public override int TransformBlock (byte[] inBuffer, int offset, int count, byte[] outBuffer, int outOffset)
        {
            int out_count = count / BlockSize;
            for (int i = 0; i < out_count; ++i)
            {
                Buffer.BlockCopy (inBuffer, offset, outBuffer, outOffset, BlockSize);
                for (int j = 7; j >= 0; --j)
                {
                    Transform (outBuffer, outOffset, j);
                }
                for (int j = 0; j < BlockSize; ++j)
                {
                    outBuffer[outOffset++] ^= m_iv[j];
                }
                Buffer.BlockCopy (inBuffer, offset, m_iv, 0, BlockSize);
                offset += BlockSize;
            }
            return out_count * BlockSize;
        }

        private void Transform (byte[] buf, int src, int i)
        {
            i &= 3;
            for (int k = BlockSize / 4 - 2; k >= 0; --k)
            {
                int pos = src + k * 4 + 2;
                uint v = LittleEndian.ToUInt32 (buf, pos);
                v = Binary.RotL (v + m_key[i + 4], m_shifts[i + 4]);
                LittleEndian.Pack (v ^ m_key[i], buf, pos);
            }
            int s = 8 * m_offsets[i] + 4;
            for (int k = BlockSize / 4 - 1; k >= 0; --k)
            {
                int pos = src + k * 4;
                uint v = LittleEndian.ToUInt32 (buf, pos);
                v = Binary.RotR (m_key[i + 4] ^ v, m_shifts[i]);
                uint u = ByteMap[Offsets[s], v & 0xFF];
                u |= (uint)ByteMap[Offsets[s+1], (v >> 8)  & 0xFF] << 8;
                u |= (uint)ByteMap[Offsets[s+2], (v >> 16) & 0xFF] << 16;
                u |= (uint)ByteMap[Offsets[s+3], v >> 24] << 24;
                u -= m_key[i];
                LittleEndian.Pack (u, buf, pos);
            }
        }

        readonly static byte[] Offsets = {
            3, 0, 6, 5, 2, 1, 7, 4, 2, 4, 0, 6, 3, 5, 1, 7,
            3, 1, 7, 5, 2, 0, 6, 4, 0, 3, 6, 5, 1, 2, 7, 4,
            3, 1, 4, 7, 2, 0, 5, 6, 6, 3, 4, 0, 7, 2, 5, 1,
            4, 6, 2, 1, 5, 7, 3, 0, 6, 5, 2, 1, 7, 4, 3, 0,
            6, 2, 4, 0, 7, 3, 5, 1, 7, 4, 2, 1, 6, 5, 3, 0,
            5, 1, 3, 6, 4, 0, 2, 7, 1, 4, 6, 2, 0, 5, 7, 3,
            6, 3, 1, 4, 7, 2, 0, 5, 7, 3, 5, 1, 6, 2, 4, 0,
            3, 4, 6, 1, 2, 5, 7, 0, 5, 6, 1, 3, 4, 7, 0, 2,
        };
        readonly static byte[,] ByteMap = new byte[8,256] {
            {
            0x57, 0x38, 0x08, 0xE3, 0xCA, 0xDC, 0x8C, 0x62, 0x3C, 0x36, 0x0C, 0x5C, 0xC3, 0x63, 0x04, 0x77,
            0x12, 0x51, 0x5F, 0x73, 0x94, 0x76, 0x0A, 0xB5, 0x1B, 0xD1, 0x32, 0xA6, 0xF5, 0xFF, 0x44, 0x4D,
            0x02, 0x24, 0xA5, 0xBA, 0x59, 0xF2, 0x3A, 0x28, 0x5E, 0xEE, 0xE2, 0x98, 0xD4, 0x35, 0x90, 0xAE,
            0x4A, 0x6A, 0x29, 0x78, 0xE6, 0xDD, 0x14, 0x43, 0xE0, 0x5D, 0x8E, 0x33, 0xBF, 0x5B, 0xED, 0xC1,
            0x8B, 0x7E, 0x8A, 0x13, 0x1C, 0xC5, 0x79, 0x49, 0x2D, 0x9F, 0xA9, 0x6B, 0xC6, 0x58, 0x39, 0x68,
            0xB3, 0xE1, 0x61, 0x9A, 0xAF, 0xC9, 0x74, 0x48, 0xE7, 0x1D, 0x82, 0x52, 0x56, 0xDA, 0x2F, 0xFE,
            0xFD, 0xE9, 0x67, 0xBC, 0xD2, 0x2E, 0xA4, 0x10, 0x9D, 0xC8, 0x11, 0xCD, 0x86, 0x03, 0x64, 0x17,
            0xF4, 0x15, 0x81, 0xA0, 0xD5, 0x1E, 0x89, 0x71, 0xCC, 0xD3, 0x47, 0x72, 0xFA, 0xD9, 0x93, 0x6F,
            0x6D, 0x3E, 0x91, 0x00, 0x05, 0xB6, 0xE4, 0xC0, 0xB1, 0xC7, 0x7D, 0xBE, 0xBB, 0x01, 0x97, 0x70,
            0x07, 0xD6, 0x0F, 0x9C, 0x87, 0xF1, 0xEF, 0x8F, 0xA8, 0x8D, 0x46, 0x42, 0xB0, 0x21, 0x6C, 0xAC,
            0xF3, 0xAA, 0xF9, 0x66, 0x9B, 0xB4, 0xA1, 0x4B, 0xDF, 0x0E, 0xF8, 0x2A, 0x45, 0x41, 0xF6, 0x40,
            0xB8, 0xE5, 0x96, 0x23, 0x53, 0x0D, 0x2C, 0x54, 0xAD, 0x60, 0x34, 0xDB, 0x7F, 0xD0, 0x85, 0xD7,
            0xCE, 0x7C, 0x92, 0xBD, 0xCB, 0x4F, 0xC2, 0xAB, 0x3B, 0xA7, 0xEA, 0x84, 0x5A, 0xB9, 0x26, 0x1A,
            0x18, 0xB2, 0xDE, 0x27, 0x7B, 0x3F, 0xB7, 0x19, 0x0B, 0x50, 0xD8, 0xF0, 0x37, 0x06, 0xFC, 0xA3,
            0x9E, 0x3D, 0x2B, 0x22, 0x65, 0xEC, 0x95, 0xCF, 0x30, 0x6E, 0xA2, 0xC4, 0xEB, 0xFB, 0xF7, 0x1F,
            0x99, 0x4E, 0x20, 0xE8, 0x75, 0x09, 0x4C, 0x88, 0x69, 0x80, 0x16, 0x31, 0x7A, 0x55, 0x25, 0x83,
            },
            {
            0x83, 0x8D, 0x20, 0x6D, 0x0E, 0x84, 0xDD, 0x90, 0x02, 0xF5, 0x16, 0xD8, 0x0A, 0xB5, 0xA9, 0x92,
            0x67, 0x6A, 0x10, 0x43, 0x36, 0x71, 0xFA, 0x6F, 0xD0, 0xD7, 0xCF, 0x18, 0x44, 0x59, 0x75, 0xEF,
            0xF2, 0x9D, 0xE3, 0xB3, 0x21, 0xFE, 0xCE, 0xD3, 0x27, 0x32, 0xAB, 0xE2, 0xB6, 0x48, 0x65, 0x5E,
            0xE8, 0xFB, 0x1A, 0x3B, 0xBA, 0x2D, 0x09, 0xDC, 0x01, 0x4E, 0x26, 0xC8, 0x08, 0xE1, 0x81, 0xD5,
            0xAF, 0xAD, 0x9B, 0x37, 0x1E, 0xAC, 0x9A, 0x7A, 0x57, 0x47, 0x30, 0xA7, 0xF6, 0x1F, 0xF1, 0xC5,
            0xD9, 0x11, 0x5B, 0xB4, 0xB7, 0xFD, 0x5C, 0x00, 0x4D, 0x24, 0xCC, 0x3D, 0x0B, 0x39, 0x28, 0x12,
            0xB9, 0x52, 0x07, 0x0D, 0x6E, 0xE4, 0xA3, 0x62, 0x4F, 0xF8, 0x31, 0x4B, 0x9E, 0x80, 0xE9, 0x7F,
            0x8F, 0x77, 0x7B, 0x13, 0x56, 0xF4, 0x15, 0x0F, 0x33, 0x46, 0xFC, 0xD4, 0xC1, 0x8A, 0x41, 0xBC,
            0xF9, 0x72, 0x5A, 0xFF, 0xCB, 0xBE, 0x6C, 0x94, 0xF7, 0x76, 0x42, 0x40, 0x06, 0x99, 0x3A, 0x97,
            0x2E, 0x82, 0xC2, 0x7E, 0x14, 0xE6, 0xB2, 0x8E, 0x2B, 0xF0, 0x53, 0xA4, 0x93, 0x68, 0xE0, 0x49,
            0x73, 0xA6, 0xEA, 0xDF, 0x66, 0x22, 0x1B, 0xC9, 0x98, 0x4A, 0xA1, 0xC7, 0x9F, 0xB8, 0x2F, 0x54,
            0x9C, 0x88, 0xD1, 0x50, 0xA5, 0x17, 0x85, 0xD6, 0xB0, 0xCD, 0x23, 0x8C, 0x63, 0xC3, 0x8B, 0x3C,
            0x87, 0x3F, 0xC6, 0x0C, 0xEB, 0x45, 0x4C, 0x89, 0x69, 0x55, 0x04, 0xC4, 0x78, 0x6B, 0xC0, 0xE7,
            0xBD, 0x19, 0x64, 0x79, 0x2C, 0x74, 0x91, 0xBF, 0xDA, 0x7D, 0x5D, 0xBB, 0x05, 0x35, 0xD2, 0xA8,
            0x38, 0x51, 0x2A, 0x03, 0x86, 0xB1, 0x34, 0x58, 0xF3, 0x61, 0xCA, 0xEC, 0xE5, 0x3E, 0x29, 0x96,
            0xDB, 0x95, 0x25, 0xA0, 0x70, 0x1C, 0xAE, 0xEE, 0xAA, 0xA2, 0x7C, 0xED, 0xDE, 0x60, 0x5F, 0x1D,
            },
            {
            0xCB, 0x33, 0x9C, 0x03, 0xFB, 0x49, 0x07, 0xB5, 0xDF, 0x95, 0x0B, 0x43, 0x45, 0xDE, 0x5F, 0xFD,
            0x0C, 0x8A, 0x1B, 0xFC, 0x7F, 0x60, 0xB9, 0xD8, 0x10, 0xBA, 0x6B, 0x39, 0x5E, 0xD4, 0xEF, 0x84,
            0xE1, 0xB0, 0xA5, 0x16, 0x27, 0x55, 0x35, 0xA2, 0x0A, 0xA7, 0xB7, 0x5D, 0xC2, 0x2C, 0x61, 0xBB,
            0x82, 0x02, 0x24, 0x71, 0x67, 0x1D, 0x2D, 0x4B, 0xFE, 0x56, 0xAE, 0x65, 0x13, 0xA4, 0x7E, 0xF9,
            0xD7, 0xE5, 0xB2, 0xBD, 0x5B, 0x08, 0x1E, 0x74, 0x4C, 0x54, 0x48, 0x22, 0x29, 0x26, 0x3F, 0xEC,
            0x40, 0x3E, 0x09, 0xB1, 0x34, 0x01, 0xD6, 0xFF, 0x4D, 0x25, 0x50, 0xDA, 0xC7, 0x23, 0xE6, 0xEE,
            0xD9, 0xAC, 0x5A, 0xC9, 0x7C, 0x37, 0x98, 0xB3, 0x9D, 0x97, 0xE9, 0xE0, 0x2E, 0xE2, 0x8C, 0x04,
            0x62, 0xCF, 0xE7, 0x2A, 0xED, 0x19, 0x32, 0xAB, 0xCE, 0x42, 0x9E, 0x52, 0x51, 0xF8, 0x85, 0x80,
            0xD3, 0x72, 0xF2, 0x1C, 0xBE, 0x92, 0xAA, 0xA9, 0xD1, 0x57, 0xEB, 0x93, 0xF3, 0xC6, 0x14, 0x47,
            0x99, 0x6A, 0x7B, 0x1A, 0x12, 0x7D, 0x89, 0x15, 0xC0, 0xDC, 0x9B, 0xBC, 0x9A, 0x86, 0x91, 0x79,
            0xD5, 0xF5, 0x20, 0x38, 0x46, 0xBF, 0x75, 0xAD, 0x06, 0x1F, 0x18, 0x44, 0x36, 0xA3, 0xAF, 0x4E,
            0x28, 0x0E, 0xF7, 0xD0, 0x0D, 0x3C, 0xF4, 0xE8, 0xA6, 0x2F, 0x4F, 0x70, 0x31, 0x2B, 0xDD, 0x88,
            0xFA, 0x8F, 0x4A, 0x21, 0x64, 0x96, 0xF1, 0x8D, 0x8B, 0x6D, 0xCD, 0x05, 0x87, 0xCA, 0x3B, 0x90,
            0x41, 0x66, 0x3D, 0xDB, 0x30, 0x6E, 0x69, 0xC8, 0x81, 0xE3, 0xB4, 0x58, 0x6C, 0xB8, 0xA1, 0xCC,
            0xC3, 0xA8, 0x76, 0x94, 0x77, 0x3A, 0xC4, 0xC1, 0x83, 0x59, 0xA0, 0x5C, 0xF0, 0x73, 0x78, 0x68,
            0x7A, 0x8E, 0x11, 0x9F, 0xB6, 0x00, 0x6F, 0xE4, 0x0F, 0xC5, 0xD2, 0x53, 0x63, 0x17, 0xEA, 0xF6,
            },
            {
            0xF5, 0x55, 0x31, 0x03, 0x6F, 0xCB, 0xA8, 0x06, 0x45, 0x52, 0x28, 0x0A, 0x10, 0xB4, 0xB1, 0xF8,
            0x18, 0xF2, 0x94, 0x3C, 0x8E, 0x97, 0x23, 0xFD, 0xAA, 0x75, 0x93, 0x12, 0x83, 0x35, 0x46, 0xA9,
            0xA2, 0xC3, 0x4B, 0x5D, 0x32, 0x59, 0x4D, 0x24, 0xB0, 0x4C, 0x73, 0xBD, 0x2D, 0x36, 0x6C, 0xB9,
            0xD4, 0xBC, 0x76, 0x01, 0x54, 0x26, 0xAC, 0x65, 0xA3, 0x1B, 0xE5, 0xCE, 0xB5, 0xD2, 0x51, 0x4E,
            0x50, 0xD0, 0x79, 0x0B, 0xAB, 0x0C, 0xA4, 0x8F, 0x4A, 0x05, 0xC2, 0x37, 0x48, 0x58, 0xAF, 0xBA,
            0x5A, 0x7C, 0x7B, 0xFB, 0x49, 0x25, 0x39, 0x89, 0xDB, 0xE9, 0x62, 0x44, 0xEB, 0x2B, 0x1C, 0x0E,
            0x15, 0x2E, 0x70, 0xFC, 0xC4, 0x3B, 0xD1, 0x34, 0xEF, 0xD6, 0x91, 0x1A, 0xDC, 0xC9, 0xD5, 0xF6,
            0xBB, 0x33, 0x81, 0xED, 0x47, 0xA6, 0xE2, 0xE4, 0xEE, 0x9F, 0xF0, 0x92, 0x64, 0x95, 0x3E, 0x14,
            0x7F, 0xD8, 0x30, 0xE8, 0x1F, 0x7E, 0x9D, 0xCC, 0xBF, 0x96, 0x11, 0xC8, 0x6E, 0xC7, 0xF1, 0xC1,
            0xCF, 0x9E, 0x85, 0x8B, 0xE3, 0x09, 0xC5, 0x69, 0x66, 0x90, 0x9C, 0x9A, 0x02, 0x68, 0x7A, 0xF3,
            0xEA, 0xDE, 0x27, 0xAD, 0x3D, 0x22, 0xB8, 0x29, 0xE1, 0x87, 0x86, 0x77, 0x61, 0xA7, 0x3A, 0xAE,
            0x21, 0x53, 0x42, 0x67, 0xDA, 0x07, 0xF4, 0x2A, 0xDD, 0x16, 0x19, 0x2F, 0x9B, 0x43, 0x84, 0xA5,
            0x98, 0xE7, 0x2C, 0xE0, 0xE6, 0xF9, 0x8D, 0x5C, 0xD7, 0x63, 0xCD, 0x00, 0xDF, 0xCA, 0x78, 0x71,
            0xB3, 0x88, 0xFA, 0x80, 0x1D, 0xA0, 0x56, 0x40, 0x17, 0x60, 0x5B, 0xD3, 0x99, 0xBE, 0x0D, 0x08,
            0x6B, 0x20, 0x6D, 0xD9, 0xF7, 0x41, 0x5E, 0x72, 0xB7, 0x6A, 0xFE, 0x8A, 0x4F, 0x74, 0x5F, 0x1E,
            0xEC, 0xC6, 0x82, 0x8C, 0xB6, 0xA1, 0xFF, 0xB2, 0x7D, 0x3F, 0xC0, 0x04, 0x13, 0x0F, 0x38, 0x57,
            },
            {
            0x61, 0xA3, 0xDC, 0x11, 0x55, 0x8C, 0x67, 0x44, 0xA5, 0x26, 0x32, 0x6A, 0x43, 0xBC, 0x0B, 0xB6,
            0x01, 0x0E, 0xE7, 0xC9, 0x5D, 0xEF, 0x2A, 0xE2, 0xD7, 0xC7, 0xFF, 0xF6, 0xDE, 0xCA, 0x57, 0x24,
            0xB4, 0x5E, 0x1B, 0x75, 0xD9, 0x8D, 0x12, 0xA2, 0x53, 0x85, 0xD8, 0x59, 0xBF, 0xE1, 0x70, 0xC3,
            0x72, 0xC5, 0x28, 0xF9, 0x76, 0x90, 0x83, 0x62, 0x6E, 0x8B, 0xAF, 0x48, 0xBB, 0x8A, 0xC8, 0xBE,
            0x79, 0x0F, 0x8E, 0x91, 0xF2, 0x1C, 0x97, 0xF4, 0xC2, 0xE5, 0x10, 0x29, 0x51, 0xD3, 0xDD, 0x2E,
            0xCD, 0xC4, 0x06, 0x3E, 0x41, 0x7D, 0x99, 0xA7, 0xAB, 0x3D, 0x30, 0x00, 0xB5, 0x9E, 0x42, 0x09,
            0x1E, 0x08, 0xB1, 0x33, 0x1D, 0xB3, 0xF3, 0x04, 0x86, 0x0C, 0x80, 0xFA, 0xF8, 0xB0, 0x2B, 0xB8,
            0x07, 0xD5, 0xB7, 0x22, 0x73, 0x17, 0x5A, 0x9C, 0x3B, 0x98, 0x74, 0xB9, 0xEE, 0x8F, 0x64, 0x52,
            0x92, 0x5C, 0xB2, 0xE8, 0x18, 0x02, 0x89, 0xCB, 0x5B, 0x4B, 0xF1, 0xAE, 0xEC, 0x4D, 0x6F, 0x82,
            0x63, 0xDF, 0x56, 0xA8, 0x35, 0xF5, 0x66, 0x78, 0x1F, 0xA4, 0x13, 0xA6, 0xD6, 0xCC, 0x84, 0x05,
            0x15, 0x1A, 0x6D, 0x4A, 0xA0, 0xE6, 0xFB, 0x96, 0x16, 0xAA, 0x7E, 0xA9, 0xD1, 0xBD, 0xFD, 0x46,
            0x2C, 0x77, 0xED, 0x7B, 0xC0, 0xCE, 0x7A, 0xE9, 0x65, 0x37, 0x45, 0x88, 0x7C, 0xD0, 0xBA, 0xAD,
            0xEA, 0x03, 0x50, 0x40, 0xDA, 0x47, 0x71, 0xD2, 0x94, 0x2D, 0xF7, 0x19, 0x39, 0x25, 0xFC, 0x69,
            0x21, 0x4F, 0x9D, 0x4C, 0xE4, 0x9B, 0x3C, 0x87, 0xFE, 0x93, 0x95, 0xD4, 0x23, 0xE0, 0xCF, 0x9F,
            0x54, 0x6C, 0xEB, 0xAC, 0x2F, 0x81, 0xDB, 0x27, 0x4E, 0x49, 0x34, 0x0A, 0x68, 0x5F, 0x7F, 0xF0,
            0xC1, 0x20, 0x58, 0xA1, 0xE3, 0x60, 0x3A, 0x6B, 0x9A, 0x0D, 0x38, 0x14, 0x36, 0x31, 0xC6, 0x3F,
            },
            {
            0x5B, 0x10, 0x85, 0xC1, 0x67, 0x9F, 0x52, 0x70, 0x61, 0x5F, 0xEB, 0x0E, 0x69, 0xF9, 0x11, 0x41,
            0x4A, 0x03, 0x26, 0x9A, 0xFB, 0xA0, 0xA8, 0x75, 0x84, 0xCB, 0xA1, 0x22, 0x45, 0x64, 0x60, 0x98,
            0xF1, 0xD0, 0x73, 0xDC, 0x1F, 0xCD, 0x09, 0xE7, 0x32, 0x4B, 0x16, 0x6E, 0xB0, 0xC9, 0x4F, 0xE4,
            0x5A, 0xFD, 0x0A, 0x63, 0xEA, 0x94, 0xFC, 0xB9, 0xFA, 0xCC, 0xF6, 0x78, 0xD6, 0x59, 0x53, 0xFF,
            0xC3, 0x54, 0x5E, 0x0C, 0x07, 0xBA, 0xAF, 0xC5, 0x3B, 0xE9, 0xA3, 0x89, 0xD3, 0x8D, 0xE8, 0xD1,
            0xC2, 0x4C, 0x7F, 0x28, 0xE0, 0x04, 0x92, 0x1E, 0xF2, 0x2B, 0x76, 0x88, 0x81, 0x14, 0x21, 0xED,
            0xF5, 0x00, 0x37, 0x90, 0x7E, 0xB8, 0x96, 0x06, 0xEC, 0xCF, 0x0B, 0xF7, 0xE1, 0xA2, 0x38, 0x8E,
            0x2E, 0xC6, 0x30, 0x74, 0x7A, 0x23, 0x34, 0xB1, 0x97, 0x40, 0xB6, 0xB3, 0xBC, 0x55, 0xAA, 0xEE,
            0x6A, 0xE5, 0x8F, 0x36, 0x9E, 0x29, 0x68, 0xD7, 0xBB, 0x86, 0x3D, 0x39, 0x05, 0x25, 0x42, 0x7D,
            0x35, 0x43, 0x80, 0xD9, 0xC8, 0xDA, 0xA7, 0x46, 0x79, 0x56, 0xF8, 0xD5, 0x77, 0xD2, 0x5D, 0xDF,
            0xA4, 0xF3, 0x27, 0x01, 0x99, 0x08, 0x9B, 0x57, 0x93, 0xAB, 0xA9, 0x58, 0xE3, 0xBF, 0x8B, 0x3A,
            0x6D, 0x62, 0x82, 0x65, 0x20, 0x5C, 0x0F, 0x72, 0x6F, 0x7B, 0xBE, 0x3C, 0x0D, 0xAD, 0x3F, 0x2C,
            0xB4, 0xF0, 0x48, 0x2F, 0x51, 0x31, 0xFE, 0x19, 0x3E, 0x13, 0x1D, 0x87, 0x9D, 0x50, 0xB5, 0xDE,
            0xBD, 0xAC, 0xC7, 0x4D, 0xDB, 0x71, 0x9C, 0x18, 0x2A, 0x24, 0xC4, 0xE6, 0x02, 0x4E, 0x1C, 0x91,
            0xDD, 0x2D, 0x17, 0xF4, 0xD4, 0x49, 0xA5, 0x12, 0x83, 0xB7, 0xC0, 0xE2, 0x8C, 0xB2, 0x7C, 0x15,
            0xEF, 0x8A, 0x44, 0x66, 0x47, 0x95, 0x1B, 0xCA, 0x6C, 0x33, 0x6B, 0xA6, 0xCE, 0xAE, 0xD8, 0x1A,
            },
            {
            0x1B, 0x5C, 0x4D, 0x4C, 0xF3, 0x7B, 0x93, 0x44, 0x13, 0x90, 0x02, 0xD3, 0x4A, 0xB3, 0x69, 0x92,
            0x30, 0xCA, 0x62, 0x89, 0xD4, 0x1D, 0xD6, 0x25, 0xE2, 0xAD, 0xED, 0x33, 0x72, 0x71, 0x61, 0xFB,
            0xC1, 0xC3, 0x37, 0x1E, 0x8A, 0xE6, 0xC6, 0x95, 0xAE, 0x27, 0x00, 0x91, 0xF4, 0x43, 0x23, 0x64,
            0x7D, 0x55, 0xB1, 0xC0, 0x3A, 0x78, 0x0F, 0x03, 0x0D, 0xA1, 0x32, 0x4B, 0x20, 0xD5, 0x04, 0x6C,
            0xAC, 0xEE, 0x0C, 0x5F, 0xA5, 0xE0, 0xD2, 0xB5, 0xEB, 0xE9, 0x8E, 0x24, 0x4F, 0x17, 0x10, 0xAF,
            0x75, 0x15, 0x2C, 0x05, 0x8F, 0x63, 0x3F, 0xC2, 0x6E, 0x50, 0x66, 0x09, 0x85, 0x51, 0xBB, 0x47,
            0x57, 0x3B, 0xFE, 0xAA, 0x40, 0x8B, 0xBF, 0x56, 0xFC, 0x97, 0x98, 0xDE, 0x41, 0x94, 0x68, 0x84,
            0xC5, 0x5E, 0xBC, 0xF0, 0x70, 0x6B, 0xA7, 0x1C, 0x21, 0xCD, 0xBD, 0xEF, 0xD1, 0x87, 0x5D, 0xE1,
            0x74, 0x7A, 0x0A, 0x6D, 0x26, 0xA9, 0x39, 0x9A, 0x28, 0x9F, 0xE3, 0x16, 0x5A, 0x1F, 0x79, 0x42,
            0xF7, 0x58, 0xE8, 0x53, 0x49, 0x59, 0xB0, 0x29, 0x07, 0x88, 0x60, 0x80, 0x3E, 0x5B, 0x86, 0xF1,
            0xB6, 0x76, 0xFD, 0x46, 0x0B, 0x7E, 0x7F, 0x4E, 0x67, 0xDD, 0xA3, 0x8C, 0xBA, 0xB8, 0x35, 0xCC,
            0x99, 0xA4, 0xC8, 0x82, 0xD7, 0xF9, 0x01, 0x22, 0x3C, 0x81, 0xC7, 0x9C, 0x45, 0x54, 0x6A, 0xCF,
            0x0E, 0xDA, 0x65, 0x2D, 0xA6, 0x77, 0xCE, 0x52, 0xF5, 0xD0, 0xD8, 0x08, 0xA8, 0xEC, 0x2A, 0x18,
            0x06, 0xFF, 0x9E, 0x73, 0xE7, 0xBE, 0xCB, 0xDB, 0x83, 0xDF, 0xF6, 0xB9, 0x11, 0xB2, 0xFA, 0xDC,
            0x14, 0x3D, 0xB7, 0x34, 0xAB, 0x48, 0xC4, 0xA2, 0x96, 0x8D, 0xB4, 0x9B, 0x31, 0x2F, 0x12, 0xE5,
            0x36, 0x19, 0xA0, 0x2B, 0x2E, 0x7C, 0x9D, 0xF8, 0xC9, 0xE4, 0x1A, 0x38, 0x6F, 0xEA, 0xF2, 0xD9,
            },
            {
            0x2A, 0xB6, 0x0A, 0x37, 0x3E, 0x53, 0xD0, 0x98, 0xCB, 0x5B, 0x82, 0xA4, 0x42, 0x38, 0xC0, 0x36,
            0x4E, 0xDC, 0xEE, 0x08, 0xE0, 0x51, 0x8B, 0x4D, 0xCF, 0xF1, 0xFA, 0x00, 0x77, 0x15, 0x23, 0x8D,
            0x3C, 0x78, 0xB7, 0x2E, 0x4B, 0x17, 0x84, 0x29, 0x88, 0x97, 0xCE, 0xF3, 0x52, 0xC3, 0xF4, 0xED,
            0x10, 0xEC, 0x3A, 0x1B, 0xE3, 0xAE, 0xF0, 0x22, 0xFB, 0x86, 0x34, 0x61, 0xB8, 0xE1, 0x9C, 0x56,
            0x64, 0x6C, 0x8F, 0x2D, 0x07, 0xBC, 0xA3, 0x5F, 0xE5, 0x94, 0x0C, 0x3B, 0x03, 0x02, 0xA7, 0x4C,
            0x59, 0x5D, 0xC7, 0x93, 0xBD, 0x31, 0x67, 0x60, 0x91, 0x95, 0x8C, 0x9D, 0x01, 0x7E, 0x71, 0x43,
            0x9A, 0x1E, 0x12, 0x55, 0x2F, 0xC2, 0x5A, 0xA8, 0x6E, 0x0E, 0xBE, 0x75, 0x3F, 0x83, 0x58, 0xFC,
            0x74, 0x1D, 0x1C, 0xD3, 0x80, 0x50, 0xA1, 0xC5, 0x35, 0x8E, 0x81, 0x05, 0xF5, 0x30, 0xA5, 0xA6,
            0x9B, 0xB9, 0xB3, 0xD8, 0x6F, 0x5C, 0x9E, 0x7D, 0x99, 0x13, 0x24, 0x65, 0xAB, 0xE9, 0x4A, 0x54,
            0x09, 0x2B, 0x0F, 0x06, 0x6D, 0x27, 0xE8, 0x69, 0x6A, 0xB0, 0x87, 0xEB, 0xBB, 0xF6, 0xD2, 0x89,
            0xF2, 0x39, 0xE7, 0xAA, 0xB1, 0x44, 0xC4, 0x76, 0xCC, 0x85, 0x63, 0xE4, 0x40, 0x19, 0x28, 0x4F,
            0x96, 0x32, 0xDD, 0x0D, 0xEA, 0x47, 0xA0, 0xE2, 0xAD, 0xDB, 0xAC, 0x5E, 0x72, 0x7A, 0xD5, 0x66,
            0x33, 0x20, 0x57, 0x21, 0xE6, 0x70, 0x26, 0xBA, 0xB2, 0xF8, 0x11, 0xD6, 0xAF, 0x79, 0xC6, 0xBF,
            0xC9, 0x7C, 0x46, 0x0B, 0x14, 0x3D, 0x16, 0xB4, 0xCA, 0xFF, 0xC1, 0xD7, 0xDF, 0xA9, 0x6B, 0xD9,
            0x45, 0x7F, 0x18, 0x8A, 0xF9, 0xEF, 0x25, 0xD4, 0x92, 0x49, 0xFD, 0x48, 0xCD, 0x1A, 0x41, 0x7B,
            0x73, 0x9F, 0xFE, 0x04, 0x2C, 0xC8, 0xDA, 0x90, 0xF7, 0xB5, 0xDE, 0x1F, 0x68, 0xA2, 0x62, 0xD1,
            }
        };
    }
}
