//! \file       AudioKOE.cs
//! \date       2018 Nov 19
//! \brief      Mebius engine encrypted WAVE file.
//
// Copyright (C) 2018 by morkt
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
//

using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.IO;
using GameRes.Utility;

namespace GameRes.Formats.Mebius
{
    [Export(typeof(AudioFormat))]
    [ExportMetadata("Priority", 1)]
    public class KoeAudio : AudioFormat
    {
        public override string         Tag { get { return "KOE/MEBIUS"; } }
        public override string Description { get { return "Mebius engine encrypted WAVE file"; } }
        public override uint     Signature { get { return 0x46464952; } } // 'RIFF'
        public override bool      CanWrite { get { return false; } }

        static readonly string[] RequiredExtensions = { "koe", "mse", "bgm" };

        public KoeAudio ()
        {
            Extensions = RequiredExtensions;
        }

        public override SoundInput TryOpen (IBinaryStream file)
        {
            if (!file.Name.HasAnyOfExtensions (RequiredExtensions))
                return null;
            var ext = Path.GetExtension (file.Name).TrimStart ('.').ToLowerInvariant();
            byte[] key;
            if (!DefaultKeys.TryGetValue (ext, out key))
                return null;
            var header = file.ReadHeader (0x14);
            int fmt_length = header.ToInt32 (0x10);
            int data_pos = 0x14 + fmt_length;
            var wav_header = file.ReadHeader (data_pos + 8).ToArray();
            Stream data = new StreamRegion (file.AsStream, file.Position);
            data = new ByteStringEncryptedStream (data, key);
            data = new PrefixStream (wav_header, data);
            return new WaveInput (data);
        }

        Dictionary<string, byte[]> DefaultKeys => SchemeMap["Mebinya!"];

        static readonly Dictionary<string, Dictionary<string, byte[]>> SchemeMap =
            new Dictionary<string, Dictionary<string, byte[]>> {
            { "Mebinya!", new Dictionary<string, byte[]> {
                { "koe", new byte[] {
                0xCE, 0xC5, 0x94, 0xE8, 0xD5, 0x7F, 0xEB, 0xF4, 0x96, 0xCA, 0xAA, 0x80, 0xAC, 0x45, 0x60, 0x58,
                0x71, 0x50, 0xDD, 0x72, 0x20, 0x39, 0x08, 0x73, 0xFE, 0x46, 0x07, 0xC5, 0x78, 0x77, 0xC0, 0x23,
                0x49, 0x9F, 0xFC, 0xD1, 0x9A, 0x0F, 0x99, 0x7F, 0x3E, 0x7B, 0xAE, 0xF4, 0x66, 0xEE, 0x14, 0x94,
                0x75, 0xD0, 0x0E, 0xD8, 0x64, 0x60, 0xB4, 0x3B, 0x40, 0x33, 0xC3, 0x4E, 0x40, 0x0E, 0xE4, 0x6C,
                0x8D, 0x26, 0xBA, 0xB0, 0x17, 0xA5, 0x40, 0xB7, 0x27, 0x80, 0x79, 0x58, 0x92, 0xF8, 0x79, 0x3E,
                0x2A, 0xDA, 0xC8, 0x29, 0xD3, 0x43, 0x66, 0xC0, 0xE5, 0x16, 0xAB, 0x25, 0x35, 0x68, 0x60, 0xC1,
                0x77, 0x6E, 0x2B, 0x0E, 0x50, 0x58, 0xDC, 0xAE, 0xC5, 0x97, 0xE9, 0x27, 0xE1, 0xF3, 0x03, 0xA2,
                0x43, 0x77, 0x13, 0xF0, 0xEC, 0x8C, 0x40, 0xB4, 0x7F, 0x62, 0x8B, 0x84, 0x40, 0x68, 0xAF, 0xD2,
                0x10, 0xF2, 0xFE, 0x79, 0x3D, 0x63, 0x3D, 0xB4, 0x43, 0x65, 0xB8, 0x5F, 0x77, 0x13, 0x32, 0x56,
                0xA4, 0x93, 0xC9, 0x3D, 0x9F, 0x89, 0xFE, 0x0B, 0xD0, 0x6C, 0x81, 0x2D, 0x3F, 0x94, 0xDD, 0x16,
                0x1A, 0x12, 0x3A, 0x83, 0xC7, 0x26, 0xC3, 0xE0, 0xFE, 0xF1, 0xEC, 0x82, 0x6C, 0xAF, 0xA0, 0x30,
                0xEB, 0xFD, 0x1A, 0xA1, 0xD0, 0xA9, 0xEC, 0x7A, 0x52, 0x6D, 0x83, 0xE4, 0x84, 0x97, 0x8F, 0x44,
                0x89, 0x0E, 0xB7, 0xC1, 0x4F, 0xA1, 0x89, 0x8C, 0x09, 0xA6, 0xE5, 0x98, 0x4C, 0xC3, 0x7A, 0xCA,
                0xE6, 0x6D, 0x06, 0xB7, 0x5B, 0x82, 0x6C, 0x02, 0x2E, 0x03, 0x57, 0xF3, 0xD6, 0x3D, 0x79, 0x5B,
                0x87, 0x0E, 0xA2, 0x4E, 0xA6, 0xFE, 0xB8, 0x56, 0xA6, 0x55, 0xD3, 0x2B, 0x17, 0x6F, 0x7F, 0x84,
                0x16, 0xF7, 0xE6, 0x99, 0x8A, 0x4E, 0x73, 0xDE, 0x45, 0x2E, 0x1A, 0xA6, 0xEF, 0x78, 0x67, 0x1A,
                } },
                { "mse", new byte[] {
                0x40, 0xBA, 0x96, 0x7E, 0x07, 0xE1, 0x92, 0x95, 0x7E, 0x95, 0x17, 0x47, 0x3D, 0x1C, 0x08, 0x94,
                0x02, 0xA5, 0x39, 0x7D, 0x95, 0xCB, 0xD8, 0x57, 0x09, 0x52, 0x67, 0xFD, 0x86, 0x57, 0xFD, 0x81,
                0x04, 0xB9, 0x70, 0x54, 0x14, 0xC7, 0x8E, 0xA5, 0xA0, 0x11, 0xF5, 0xE2, 0xC5, 0x6E, 0xDB, 0x01,
                0xA8, 0x8C, 0xA9, 0x25, 0xEB, 0x98, 0xD6, 0xBA, 0xAD, 0xD9, 0x62, 0x00, 0xAE, 0x50, 0xCA, 0x3E,
                0x04, 0xAA, 0xF7, 0x98, 0xF9, 0x2C, 0xAE, 0xA4, 0x11, 0xCE, 0xF8, 0xCC, 0xAD, 0xB8, 0x07, 0xA5,
                0xE8, 0xDF, 0x28, 0x2A, 0xA1, 0xE4, 0x81, 0x1F, 0x35, 0x7B, 0x4C, 0x7F, 0xFA, 0x04, 0x75, 0x31,
                0x77, 0x0D, 0xD1, 0x79, 0xD3, 0x68, 0x2C, 0xDB, 0x16, 0x27, 0xBB, 0xD5, 0x2A, 0xFB, 0x2C, 0xBC,
                0xB1, 0x70, 0xE2, 0x1C, 0xA8, 0xF6, 0x1E, 0x53, 0xDA, 0xA0, 0x89, 0xED, 0xB9, 0x25, 0x0A, 0x55,
                0x08, 0x01, 0x37, 0xE7, 0x6B, 0xB4, 0xDB, 0x18, 0xE2, 0x13, 0x6B, 0x8E, 0x25, 0x98, 0x40, 0x05,
                0xE7, 0x32, 0x1F, 0x4B, 0xA9, 0x7C, 0xC8, 0x24, 0x51, 0x54, 0x16, 0xFD, 0x6F, 0xC8, 0x67, 0x2B,
                0xD2, 0xCD, 0x78, 0x18, 0xC2, 0xB0, 0xB6, 0xAA, 0x25, 0xB2, 0x4E, 0xCD, 0x3A, 0xD7, 0x0D, 0x43,
                0x64, 0xBD, 0x35, 0x52, 0xFC, 0x07, 0x70, 0x67, 0xBE, 0x48, 0xFB, 0xA9, 0xD2, 0x67, 0xC3, 0xB8,
                0x6A, 0xDC, 0x76, 0x04, 0x0E, 0xDD, 0xD3, 0xEB, 0x7A, 0x49, 0x39, 0xAC, 0xBD, 0xE5, 0x31, 0xBB,
                0x71, 0xCC, 0x91, 0x8A, 0xB1, 0x09, 0x57, 0xF3, 0x39, 0xD2, 0x5E, 0xAB, 0x4F, 0x5F, 0x24, 0x86,
                0xD5, 0x3D, 0xA8, 0xE7, 0x36, 0x23, 0x21, 0x32, 0x76, 0x3C, 0x98, 0x0A, 0x34, 0x51, 0x1E, 0xB8,
                0x51, 0x40, 0x34, 0x93, 0x0B, 0x5C, 0x94, 0x24, 0x50, 0x6A, 0x72, 0x85, 0x04, 0xF1, 0xE5, 0x20,
                } },
                { "bgm", new byte[] {
                0x16, 0x83, 0x0A, 0x4D, 0x6E, 0x39, 0xBF, 0xD8, 0x9C, 0x2B, 0x9E, 0x9F, 0xAE, 0x13, 0x8C, 0x63,
                0xBE, 0x53, 0x95, 0x2E, 0x61, 0xB3, 0xFC, 0x26, 0x1C, 0xA5, 0xBF, 0x99, 0x69, 0x29, 0x3C, 0x99,
                0xD7, 0x1E, 0x8B, 0xFD, 0xBD, 0x98, 0xC9, 0x12, 0x0E, 0x93, 0x5F, 0x59, 0x4E, 0x89, 0x7B, 0x26,
                0xA7, 0x53, 0x50, 0xF1, 0xB6, 0x52, 0x5A, 0xA6, 0x6D, 0xCD, 0x20, 0xD9, 0xC3, 0x82, 0xCB, 0x21,
                0xFD, 0x4D, 0x8B, 0xFA, 0x49, 0xEA, 0xC3, 0x7C, 0x81, 0x42, 0xEE, 0x38, 0xC3, 0xAB, 0xE0, 0x1A,
                0xBD, 0x9F, 0xB4, 0x98, 0x4F, 0x59, 0x60, 0x8D, 0xEE, 0x41, 0x92, 0x87, 0xEB, 0x30, 0x2A, 0x66,
                0xF4, 0x69, 0xA2, 0xA4, 0x0F, 0x53, 0xB6, 0x04, 0x4E, 0x4A, 0xB8, 0x9E, 0x8B, 0x23, 0xE0, 0xF8,
                0xE6, 0xA2, 0x1F, 0xA4, 0x46, 0x9B, 0x34, 0x09, 0x33, 0xE3, 0x0B, 0x66, 0xB7, 0xCC, 0x1F, 0xA9,
                0x1F, 0xEE, 0xF6, 0x1D, 0x42, 0x55, 0xE6, 0x19, 0x44, 0x61, 0xBA, 0xAE, 0x57, 0xFC, 0x6D, 0x08,
                0xFE, 0x6B, 0x84, 0x5C, 0x69, 0x50, 0xD0, 0xCC, 0xC3, 0xBC, 0x92, 0x7C, 0x33, 0x59, 0x4D, 0x2D,
                0x50, 0x00, 0x47, 0xCE, 0x4C, 0xDB, 0x7A, 0xB0, 0x25, 0x61, 0x07, 0x55, 0x8A, 0xAD, 0x50, 0x0B,
                0xD3, 0x2D, 0x6C, 0xC9, 0x39, 0x94, 0x82, 0x0F, 0x9B, 0xF9, 0x45, 0x95, 0x1C, 0xBA, 0xA5, 0xB9,
                0xD2, 0x60, 0xE3, 0xE3, 0xC7, 0x34, 0xAA, 0x43, 0x27, 0xC7, 0xC2, 0x3D, 0xBD, 0x8A, 0xA6, 0x4B,
                0xA9, 0x3F, 0xEF, 0xBB, 0x6B, 0xE4, 0x6B, 0x89, 0x2A, 0xE9, 0xD1, 0xC0, 0xE5, 0x3A, 0xED, 0x1A,
                0x61, 0xF9, 0xB3, 0xCC, 0x03, 0x0F, 0x82, 0xCD, 0x74, 0x36, 0x2A, 0xD8, 0x3E, 0x4E, 0xE0, 0x17,
                0x37, 0x1B, 0x41, 0xC2, 0xE8, 0xA7, 0x81, 0x7C, 0xD3, 0x02, 0xFD, 0x51, 0xB4, 0x02, 0x43, 0x9E,
                } },
            } },
            { "Tomodachi Ijou Koibito Miman", new Dictionary<string, byte[]> {
                { "koe", new byte[] {
                0x15, 0xEE, 0x1F, 0x83, 0x32, 0x20, 0xF8, 0x17, 0x53, 0xE3, 0x7B, 0xC0, 0x6A, 0x75, 0x93, 0xA5,
                0x79, 0x32, 0x36, 0x7A, 0x76, 0xC5, 0xF4, 0x06, 0xC5, 0x08, 0xF5, 0x1E, 0xE4, 0xD5, 0xED, 0x72,
                0x0B, 0xEC, 0x2A, 0x52, 0x6D, 0x87, 0xC3, 0x55, 0xD9, 0xC0, 0x07, 0x7A, 0x5E, 0x84, 0x35, 0x38,
                0xB7, 0x0C, 0x17, 0x8A, 0x22, 0xB4, 0x17, 0xFB, 0xEE, 0xA1, 0x57, 0xAE, 0x51, 0x09, 0xF3, 0xE9,
                0x65, 0x0F, 0x66, 0x3B, 0xD1, 0x91, 0x51, 0x0F, 0x08, 0x58, 0xC3, 0x75, 0x0D, 0x69, 0x3C, 0x65,
                0xC4, 0x92, 0x1E, 0x27, 0x32, 0x69, 0x93, 0xD3, 0x19, 0xBA, 0xAF, 0x00, 0x87, 0x38, 0x79, 0xFB,
                0x24, 0xEA, 0xAE, 0x4E, 0x4C, 0x1C, 0x06, 0xCF, 0xD9, 0xD7, 0x4E, 0x80, 0x2C, 0x27, 0xBF, 0x07,
                0x38, 0xA6, 0x48, 0xF9, 0x43, 0x2E, 0x32, 0xD4, 0x13, 0x09, 0x7B, 0xBB, 0xAC, 0x92, 0x99, 0xF8,
                0x70, 0xAC, 0xA1, 0xD0, 0x2A, 0x59, 0x8F, 0x17, 0xEF, 0xFE, 0x85, 0x9B, 0x53, 0x15, 0xDA, 0xE9,
                0xC7, 0xBD, 0xD4, 0x64, 0x55, 0x9C, 0x42, 0x38, 0x4E, 0x55, 0x7D, 0x3D, 0xCB, 0x96, 0xF0, 0xA8,
                0x14, 0x92, 0x21, 0x3E, 0xA1, 0xCC, 0xF0, 0xD9, 0x0F, 0xA1, 0x0B, 0x00, 0xFD, 0x5C, 0xAE, 0x4E,
                0x53, 0x61, 0xC6, 0xF6, 0xCE, 0xA5, 0x91, 0x2C, 0x62, 0x01, 0x3A, 0x17, 0x53, 0x1A, 0xA1, 0x47,
                0xFE, 0xF1, 0xD1, 0x42, 0x48, 0xD3, 0xBB, 0x7F, 0x1D, 0xA8, 0xC7, 0x96, 0x8E, 0xFC, 0x5E, 0xEA,
                0x5A, 0xAD, 0xE8, 0xFB, 0x78, 0x8B, 0x76, 0xD2, 0x86, 0x7B, 0x79, 0x0B, 0x96, 0xC4, 0x51, 0x04,
                0x43, 0x30, 0x20, 0x3F, 0x19, 0x19, 0x88, 0xE3, 0x27, 0x10, 0x65, 0xFE, 0xC8, 0x4A, 0x11, 0x67,
                0x01, 0x55, 0x46, 0xEE, 0x80, 0x68, 0xC9, 0xC1, 0x1B, 0x4C, 0x49, 0x14, 0xC9, 0x95, 0xA9, 0x7F,
                } },
                { "mse", new byte[] {
                0x06, 0xDE, 0xEF, 0x76, 0xD2, 0xDA, 0xE7, 0x95, 0x7A, 0x87, 0x6D, 0x7C, 0xF6, 0x17, 0x44, 0x9F,
                0x08, 0xD2, 0xC5, 0x89, 0xDC, 0xDE, 0xA1, 0x0F, 0x2D, 0xCB, 0xCA, 0xB8, 0x6E, 0xBB, 0x7F, 0x8A,
                0x9E, 0x63, 0x70, 0x58, 0xCC, 0xA8, 0x61, 0x34, 0x68, 0x98, 0xD8, 0xB3, 0x74, 0x18, 0x2C, 0x9B,
                0x1F, 0x64, 0xFD, 0x28, 0xF4, 0x56, 0x8E, 0x80, 0x98, 0x06, 0x5C, 0x13, 0x38, 0x31, 0xFE, 0x62,
                0xBD, 0x11, 0xBC, 0x05, 0x58, 0xF6, 0x38, 0xDD, 0x6A, 0x5A, 0x01, 0x4C, 0x03, 0xC1, 0x0A, 0xE0,
                0x58, 0xA2, 0x88, 0x4E, 0xF6, 0x96, 0x1A, 0x2A, 0xA0, 0xA1, 0xA5, 0xAF, 0x04, 0x4E, 0x08, 0x99,
                0xC8, 0xDA, 0x9B, 0x4C, 0xA4, 0xD2, 0x82, 0x51, 0xE2, 0xBB, 0x33, 0xFC, 0x23, 0xAF, 0xAD, 0x22,
                0xB7, 0x98, 0x5D, 0x36, 0xD3, 0xE7, 0x8C, 0x54, 0x0C, 0x6A, 0xE6, 0x6D, 0x53, 0x28, 0xFA, 0xAE,
                0xE7, 0x66, 0x36, 0x50, 0xE6, 0x40, 0xBA, 0xDF, 0xFA, 0xE2, 0xA3, 0xCC, 0xDB, 0x70, 0x89, 0x27,
                0x88, 0x0A, 0x59, 0x6D, 0x81, 0x06, 0xD1, 0x5A, 0x65, 0x5E, 0xC3, 0x7F, 0x2F, 0xC7, 0x5D, 0xB2,
                0x86, 0x91, 0x19, 0x0A, 0xD7, 0x33, 0xA8, 0xF0, 0x21, 0xA7, 0xE7, 0x19, 0xB5, 0x07, 0xB4, 0xCA,
                0x59, 0xEA, 0xB8, 0xD4, 0xFB, 0x21, 0xF8, 0xAC, 0x7C, 0x2F, 0x45, 0xEA, 0x22, 0x2B, 0x58, 0x4A,
                0x55, 0xEA, 0xB6, 0x45, 0x34, 0x96, 0xAE, 0xFD, 0x86, 0x97, 0xF9, 0x93, 0xBE, 0x6A, 0x6A, 0xFB,
                0x7B, 0x65, 0x21, 0x24, 0x42, 0x5C, 0x37, 0x4F, 0x64, 0x45, 0x58, 0x0C, 0xBC, 0xC1, 0xB7, 0xAD,
                0xC7, 0xB6, 0xE3, 0x21, 0xBB, 0xC8, 0xD2, 0x15, 0x1F, 0xF1, 0x39, 0x3F, 0x87, 0x86, 0x88, 0xBE,
                0x84, 0xD7, 0x1A, 0x63, 0xD5, 0x51, 0x63, 0xDB, 0x74, 0x39, 0x4C, 0x12, 0x12, 0xF1, 0x6E, 0x2C,
                } },
                { "bgm", new byte[] {
                0xB0, 0x6F, 0xA4, 0xD7, 0x8B, 0x81, 0xBD, 0xF3, 0x82, 0xAF, 0x95, 0x6B, 0x9D, 0x3E, 0x88, 0x73,
                0xB8, 0xF9, 0xD8, 0x09, 0x31, 0xF3, 0x84, 0xDA, 0xCC, 0xAF, 0x54, 0x60, 0xFD, 0x97, 0x04, 0xA6,
                0x05, 0x65, 0x20, 0x9A, 0xA7, 0x62, 0xD9, 0xD7, 0x5C, 0x98, 0x6F, 0x2D, 0x3A, 0x6E, 0x07, 0xF8,
                0x86, 0x34, 0xF9, 0x05, 0xAB, 0x25, 0xF5, 0x70, 0x79, 0x64, 0x03, 0x7C, 0xF2, 0xF6, 0xBF, 0x9B,
                0x91, 0xBB, 0x6B, 0x2A, 0xB3, 0xEB, 0xF2, 0x42, 0x39, 0x27, 0xD2, 0xF0, 0xEA, 0x00, 0x7A, 0x57,
                0xAF, 0xB2, 0xCE, 0xEE, 0xBE, 0xCE, 0x1B, 0x87, 0x4E, 0x1F, 0xA4, 0xB0, 0xD1, 0x8E, 0x79, 0x9D,
                0x6E, 0xC4, 0x26, 0xBF, 0x26, 0xDD, 0x39, 0x2D, 0x54, 0x49, 0x0B, 0xF6, 0x19, 0xDF, 0x3E, 0x19,
                0x2F, 0xA2, 0x6A, 0x2A, 0x66, 0xAC, 0x68, 0x60, 0xA6, 0xEB, 0xB7, 0x24, 0xC0, 0x85, 0x5E, 0x40,
                0xF8, 0x8B, 0xD8, 0x68, 0x78, 0xE8, 0x66, 0xA1, 0xAA, 0xA7, 0xC7, 0x4F, 0x2C, 0x6D, 0xD1, 0x5F,
                0xC4, 0xE6, 0x46, 0x6F, 0x1B, 0x69, 0xF7, 0xD1, 0x23, 0x89, 0x1C, 0x53, 0x6E, 0x75, 0x41, 0xAF,
                0x52, 0xCB, 0x6D, 0x03, 0x28, 0x38, 0xBE, 0x41, 0xFE, 0x99, 0x9D, 0xDF, 0x96, 0x7C, 0xD8, 0xDE,
                0xF3, 0x95, 0x3E, 0x47, 0x5E, 0xA7, 0x05, 0x43, 0xA8, 0x6B, 0x96, 0x8C, 0x89, 0x6D, 0x16, 0x29,
                0x60, 0x74, 0x31, 0x47, 0x36, 0xE2, 0x91, 0x3D, 0x57, 0xAD, 0x81, 0x63, 0xC8, 0xD5, 0x9A, 0x5F,
                0x03, 0x7A, 0x14, 0x10, 0x32, 0x7B, 0xF1, 0x33, 0xDE, 0xBA, 0x52, 0x74, 0xC7, 0x6E, 0xF8, 0x7E,
                0x4C, 0x2C, 0x58, 0x3B, 0xA9, 0x7A, 0x51, 0x5C, 0xFD, 0xA5, 0xCF, 0x67, 0xB8, 0x34, 0x85, 0x3D,
                0x7D, 0x93, 0xE9, 0x7E, 0x9E, 0x6E, 0xC3, 0xB2, 0xB1, 0xD0, 0x5C, 0x83, 0x61, 0x6F, 0x27, 0x18,
                } },
            } },
        };
    }
}
