//! \file       AudioPCM.cs
//! \date       Mon Jun 15 16:30:24 2015
//! \brief      Circus PCM audio format.
//
// Copyright (C) 2015-2016 by morkt
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
//

using System;
using System.ComponentModel.Composition;
using System.IO;
using GameRes.Compression;
using GameRes.Utility;

namespace GameRes.Formats.Circus
{
    [Export(typeof(AudioFormat))]
    public class PcmAudio : AudioFormat
    {
        public override string         Tag { get { return "PCM"; } }
        public override string Description { get { return "Circus PCM audio"; } }
        public override uint     Signature { get { return 0x4d435058; } } // 'XPCM'

        public override SoundInput TryOpen (IBinaryStream file)
        {
            file.Position = 4;
            int src_size = file.ReadInt32();
            if (src_size <= 0)
                throw new InvalidFormatException();
            int mode = file.ReadInt32();
            int extra = (mode >> 8) & 0xff;
            mode &= 0xff;
            if (5 == mode)
            {
                uint ogg_size = file.ReadUInt32();
                var ogg = new StreamRegion (file.AsStream, 0x10, ogg_size);
                return new OggInput (ogg);
            }
            var format = new WaveFormat();
            format.FormatTag                = file.ReadUInt16();
            format.Channels                 = file.ReadUInt16();
            format.SamplesPerSecond         = file.ReadUInt32();
            format.AverageBytesPerSecond    = file.ReadUInt32();
            format.BlockAlign               = file.ReadUInt16();
            format.BitsPerSample            = file.ReadUInt16();
            Stream pcm;
            if (0 == mode)
            {
                pcm = new StreamRegion (file.AsStream, file.Position, src_size);
            }
            else if (1 == mode || 3 == mode)
            {
                var decoder = new PcmDecoder (file, src_size, extra, (XpcmCompression)mode);
                pcm = new MemoryStream (decoder.Unpack(), 0, src_size);
                file.Dispose();
            }
            else
                throw new NotSupportedException ("Not supported Circus PCM audio compression");

            return new RawPcmInput (pcm, format);
        }
    }

    public enum XpcmCompression
    {
        None = 0,
        Lzss = 1,
        Zlib = 3,
    }

    internal class PcmDecoder
    {
        byte[]      m_pcm_data;
        byte[]      m_encoded;
        int         m_pcm_size;
        int         m_extra;

        public byte[] Data { get { return m_pcm_data; } }

        public PcmDecoder (IBinaryStream input, int pcm_size, int extra, XpcmCompression mode)
        {
            if (extra < 0 || extra > 3)
                throw new InvalidFormatException();
            int packed_size = input.ReadInt32();
            m_extra = extra;
            m_pcm_size = pcm_size;
            m_pcm_data = new byte[pcm_size + 8192];
            m_encoded = new byte[(pcm_size / 0xFE0 << 12) + 16386];
            if (XpcmCompression.Lzss == mode)
            {
                if (packed_size != input.Read (m_pcm_data, 0, packed_size))
                    throw new InvalidFormatException ("Unexpected end of file");
                UnpackV1 (m_pcm_data, packed_size, m_encoded);
            }
            else if (XpcmCompression.Zlib == mode)
            {
                using (var z = new ZLibStream (input.AsStream, CompressionMode.Decompress, true))
                    z.Read (m_encoded, 0, m_encoded.Length);
            }
            else
                throw new InvalidFormatException ("Unknown PCM compression mode");
        }

        public byte[] Unpack ()
        {
            Buffer.BlockCopy (unk_43A254, m_extra*0x40, dword_43A214, 0, 0x40);
            DecodeV1 (m_pcm_data, m_encoded, m_pcm_size);
            return m_pcm_data;
        }

        void UnpackV1 (byte[] input, int packed_size, byte[] output) // sub_412070
        {
            int flag = 0;
            int dst = 0;
            int src = 0;
            while (src < packed_size)
            {
                flag >>= 1;
                if (0 == (flag & 0x100))
                {
                    flag = input[src++] | 0xff00;
                }
                if (0 != (flag & 1))
                {
                    output[dst++] = input[src++];
                }
                else
                {
                    if (src >= packed_size)
                        break;
                    int offset, count;
                    int ctl = input[src++];
                    if (ctl >= 0xc0)
                    {
                        offset = input[src++] | ((ctl & 3) << 8);
                        count = 4 + ((ctl >> 2) & 0xF);
                    }
                    else if (0 != (ctl & 0x80))
                    {
                        offset = ctl & 0x1F;
                        count = 2 + ((ctl >> 5) & 3);
                        if (0 == offset)
                        {
                            offset = input[src++];
                        }
                    }
                    else if (0x7f == ctl)
                    {
                        count = 2 + LittleEndian.ToUInt16 (input, src);
                        offset = LittleEndian.ToUInt16 (input, src+2);
                        src += 4;
                    }
                    else
                    {
                        offset = LittleEndian.ToUInt16 (input, src);
                        src += 2;
                        count = ctl + 4;
                    }
                    offset = dst - offset;
                    Binary.CopyOverlapped (output, offset, dst, count);
                    dst += count;
                }
            }
        }

        int[] dword_43A214 = new int[0x10];
        byte[] unk_6A16C8 = new byte[0x2000];
        int[] dword_6996C8 = new int[0x1000];
        int[] dword_69D6C8 = new int[0x1000];

        void DecodeV1 (byte[] output, byte[] input, int output_size) // sub_4122B0
        {
            int v14 = output_size / 2;
            int v5 = 0; // within input
            int decoded = 0;
            for (int dst_sizea = 0; dst_sizea < v14; dst_sizea += 4064)
            {
                sub_4121C0 (input, v5);
                v5 += 8192;
                sub_411AB0 (12, dword_6996C8, dword_69D6C8);
                int v6 = decoded; // within output
                int v7 = 0;
                int v8 = 0; // within dword_6996C8;
                for (int v9 = 32; v9 > -4064; --v9)
                {
                    if (v7 + dst_sizea < v14)
                    {
                        int v11;
                        if (v9 > 0 && 0 != dst_sizea)
                        {
                            long v10 = (v7 * (uint)dword_6996C8[v8]) + (long)(v9 * LittleEndian.ToInt16 (output, v6));
                            v11 = ((int)((v10 >> 32) & 0x1F) + (int)v10) >> 5;
                        }
                        else
                        {
                            v11 = dword_6996C8[v8];
                        }
                        if (v11 > 32767)
                            v11 = 32767;
                        else if (v11 < -32768)
                            v11 = -32768;
                        LittleEndian.Pack ((short)v11, output, v6); // *(int16_t*)v6 = v11;
                    }
                    ++v7;
                    v6 += 2;
                    ++v8;
                }
                decoded += 8128;
            }
        }

        void sub_4121C0 (byte[] input, int a1)
        {
            int v1 = a1;
            int v5 = 1; // within unk_6A16C8;
            for (int v3 = 0; v3 < 0x1000; ++v3)
            {
                unk_6A16C8[v5] = input[v1++];
                v5 += 2;
            }
            v5 = 0; // within unk_6A16C8;
            for (int v6 = 0; v6 < 0x800; ++v6)
            {
                byte v7 = input[v1+0x800];
                byte v8 = input[v1];
                unk_6A16C8[v5]   = (byte)((v7 >> 4) | (v8 & 0xF0));
                unk_6A16C8[v5+2] = (byte)((v8 << 4) | (v7 & 0x0F));
                v5 += 4;
                v1++;
            }
            int v9 = 0;
            v5 = 0; // unk_6A16C8;
            for (int v11 = 0; v11 < 32768; v11 += 16)
            {
                int result = dword_43A214[v11 / 0x1000];
                int i1 = LittleEndian.ToUInt16 (unk_6A16C8, v5);
                int i2 = LittleEndian.ToUInt16 (unk_6A16C8, v5+2); 
                dword_6996C8[v9]   = result * word_6A56C8[i1];
                dword_69D6C8[v9++] = result * word_6A56C8[i2];
                v5 += 4;
            }
            for (int i = v9; i < 4096; ++i)
            {
                dword_6996C8[i] = 0;
                dword_69D6C8[i] = 0;
            }
        }

        void sub_411AB0 (int a1, int[] a2, int[] a3)
        {
            int v4 = 1 << a1;
            int v5 = 1;
            int v68 = 1;
            int v6 = 1 << a1;
            int v79 = 1 << a1;
            if (a1 >= 3)
            {
                int v7 = v4 >> 1;
                int v59 = a1 - 2;
                for (;;)
                {
                    int v63 = v4;
                    int v82 = dword_43A358[2 * v5];
                    int v66 = v7;
                    int v8 = v7 >> 1;
                    int v62 = v7 >> 1;
                    int v80 = dword_43A358[1 + 2 * v5];
                    int v77 = 0;
                    if (v6 > 0)
                    {
                        int v73 = 0; // within a2;
                        int v69 = v8; // within a2
                        int v75 = v7; // within a2;
                        int v9 = v8 + 1; // within a3
                        int v10 = v7 + v8;
                        int v64 = v9; // within a3
                        int v11 = 1; // within a3;
                        int v12 = v10 + 1; // within a3
                        int v71 = v10; // within a2
                        int v13 = v7 + 1; // within a3
                        do
                        {
                            int v14 = a2[v75];
                            int v15 = a2[v73];
                            int v16 = a3[v11 - 1] - a3[v13 - 1];
                            a2[v73] += a2[v75];
                            a3[v11 - 1] += a3[v13 - 1];
                            a2[v75] = v15 - v14;
                            a3[v13 - 1] = v16;
                            int v17 = a2[v13];
                            int v18 = a2[v11] - v17;
                            int v19 = a3[v11] - a3[v13];
                            a2[v11] += v17;
                            a3[v11] += a3[v13];
                            a2[v13] = (int)(((ulong)(v18 * (long)v82) >> 12)
                                            + ((ulong)(v19 * (long)v80) >> 12));
                            a3[v13] = (int)(((ulong)((int)v19 * (long)v82) >> 12)
                                            - ((ulong)((int)v18 * (long)v80) >> 12));
                            int v20 = a2[v69] - a2[v71];
                            int v21 = v64; // within a3
                            int v22 = a3[v64 - 1] - a3[v12 - 1];
                            a2[v69] += a2[v71];
                            a3[v21 - 1] += a3[v12 - 1];
                            a2[v71] = v22;
                            a3[v12 - 1] = -v20;
                            int v23 = a2[v12];
                            int v24 = a2[v64] - v23;
                            int v25 = a3[v64] - a3[v12];
                            a2[v21] += v23;
                            a3[v21] += a3[v12];
                            a2[v12] = (int)(((ulong)(v25 * (long)v82) >> 12)
                                            - ((ulong)(v24 * (long)v80) >> 12));

                            a3[v12] = (int)-(long)(((ulong)(v24 * (long)v82) >> 12)
                                                   + ((ulong)(v25 * (long)v80) >> 12));
                            v13 += v63;
                            v75 += v63;
                            v11 += v63;
                            v73 += v63;
                            v12 += v63;
                            v71 += v63;
                            v77 += v63;
                            v69 += v63;
                            v64 += v63;
                        }
                        while (v77 < v79);
                        v8 = v62;
                        v5 = v68;
                        v7 = v66;
                        v6 = 1 << a1;
                    }
                    if (v8 > 2)
                    {
                        int v70 = 2; // within a3
                        int v72 = v7 + 2; // within a3
                        int v74 = v8 + 2; // within a3
                        int v27 = 1 + 4 * v5; // within dword_43A358
                        int v60 = v8 - 2;
                        int v76 = v8 + 2 + v7;
                        do
                        {
                            int v83 = dword_43A358[v27 - 1];
                            int v81 = dword_43A358[v27];
                            int v78 = 0;
                            if (v6 > 0)
                            {
                                int v28 = v70; // within a3
                                int v29 = v72; // within a3
                                int v65 = v74; // within a3
                                int v85 = v76; // within a3
                                do
                                {
                                    int v31 = a2[v29];
                                    int v32 = a2[v28] - v31;
                                    int v33 = a3[v28] - a3[v29];
                                    a2[v28] += v31;
                                    a3[v28] += a3[v29];
                                    a2[v29] = (int)(((ulong)(v32 * (long)v83) >> 12)
                                                    + ((ulong)(v33 * (long)v81) >> 12));
                                    a3[v29] = (int)(((ulong)(v33 * (long)v83) >> 12)
                                                    - ((ulong)(v32 * (long)v81) >> 12));
                                    int v34 = a2[v65] - a2[v85];
                                    int v35 = a3[v65] - a3[v85];
                                    a2[v65] += a2[v85];
                                    a3[v65] += a3[v85];
                                    a2[v85] = (int)(((ulong)(v35 * (long)v83) >> 12)
                                                    - ((ulong)(v34 * (long)v81) >> 12));
                                    a3[v85] = (int)-(long)(((ulong)(v34 * (long)v83) >> 12)
                                                           + ((ulong)(v35 * (long)v81) >> 12));
                                    v29 += v63;
                                    v85 += v63;
                                    v28 += v63;
                                    v65 += v63;
                                    v78 += v63;
                                }
                                while (v78 < v79);
                                v5 = v68;
                                v6 = 1 << a1;
                            }
                            v27 += 2 * v5;
                            ++v70;
                            ++v72;
                            ++v74;
                            ++v76;
                        }
                        while (v60-- != 1);
                    }
                    v68 = 2 * v5;
                    --v59;
                    if (0 == v59)
                        break;
                    v7 = v62;
                    v5 *= 2;
                    v4 = v66;
                }
            }
            if (!(a1 < 2 || v6 <= 0))
            {
                int v37 = 1; // within a2
                int v38 = 3; // within a3
                uint v88 = (uint)(v6 + 3) >> 2;
                do
                {
                    int v39 = a2[v37 - 1];
                    int v40 = a2[v37 + 1];
                    int v41 = a3[v38 - 3] - a3[v38 - 1];
                    a2[v37 - 1] = v40 + v39;
                    v37 += 4;
                    a3[v38 - 3] += a3[v38 - 1];
                    a2[v37 - 3] = v39 - v40;
                    a3[v38 - 1] = v41;
                    int v42 = a2[v38];
                    int v43 = a3[v38 - 2] - a3[v38];
                    v38 += 4;
                    int v44 = a2[v37 - 4] - v42;
                    a2[v37 - 4] += v42;
                    a3[v38 - 6] += a3[v38 - 4];
                    a2[v38 - 4] = v43;
                    a3[v38 - 4] = -v44;
                    --v88;
                }
                while (0 != v88);
                v6 = v79;
            }
            int v45 = 0; // within a2;
            if (v6 > 0)
            {
                int v47 = 1; // within a3
                uint v89 = (uint)(v79 + 1) >> 1;
                do
                {
                    int v48 = a2[v47];
                    int v49 = a3[v47 - 1] - a3[v47];
                    int v50 = a2[v45] - v48;
                    v47 += 2;
                    a2[v45] += v48;
                    v45 += 2;
                    a3[v47 - 3] += a3[v47 - 2];
                    a2[v47 - 2] = v50;
                    a3[v47 - 2] = v49;
                    --v89;
                }
                while (0 != v89);
                v45 = 0; // within a2;
                v6 = v79;
            }
            int v51 = 0;
            int result = v6 / 2;
            int v67 = v6 / 2;
            int v90 = 1;
            if ( v6 - 1 > 1 )
            {
                int v54 = 1; // within a3
                for (;;)
                {
                    for ( ; result <= v51; result /= 2)
                        v51 -= result;
                    v51 += result;
                    if (v90 < v51)
                    {
                        int v55 = a2[v45 + v51];
                        a2[v45 + v51] = a2[v54];
                        a2[v54] = v55;
                        int v56 = a3[v51];
                        a3[v51] = a3[v54];
                        a3[v54] = v56;
                    }
                    ++v54;
                    result = v79 - 1;
                    ++v90;
                    if (v90 >= v79 - 1)
                        break;
                    result = v67;
                }
                v6 = v79;
            }
            while (v6 > 0)
            {
                int eax = a2[v45] << 4;
                int edx = (int)(((long)eax >> 32) & 0x3FFF);
                eax += edx;
                a2[v45++] = eax >> 14;
                --v6;
            }
        }

        static readonly short[] word_6A56C8 = InitTable();

        static short[] InitTable ()
        {
            var table = new short[0x10000];
            short cx = 0;
            short dx = 0;
            for (int i = 0; i < table.Length; )
            {
                table[i++] = dx--;
                table[i++] = ++cx;
            }
            return table;
        }

        static readonly byte[] unk_43A254 = new byte[]
        {
            0x00, 0x08, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00,
            0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,

            0x00, 0x08, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00,
            0x00, 0x10, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,

            0x00, 0x08, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00,
            0x00, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,

            0x00, 0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00,
            0x00, 0x08, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        };

        static readonly int[] dword_43A358 = new int[]
        {
            0x1000, 0x0000, 0x0FFF, 0x0006, 0x0FFF, 0x000C, 0x0FFF, 0x0012,
            0x0FFF, 0x0019, 0x0FFF, 0x001F, 0x0FFF, 0x0025, 0x0FFF, 0x002B,
            0x0FFF, 0x0032, 0x0FFF, 0x0038, 0x0FFF, 0x003E, 0x0FFF, 0x0045,
            0x0FFF, 0x004B, 0x0FFF, 0x0051, 0x0FFF, 0x0057, 0x0FFE, 0x005E,
            0x0FFE, 0x0064, 0x0FFE, 0x006A, 0x0FFE, 0x0071, 0x0FFE, 0x0077,
            0x0FFE, 0x007D, 0x0FFD, 0x0083, 0x0FFD, 0x008A, 0x0FFD, 0x0090,
            0x0FFD, 0x0096, 0x0FFC, 0x009D, 0x0FFC, 0x00A3, 0x0FFC, 0x00A9,
            0x0FFC, 0x00AF, 0x0FFB, 0x00B6, 0x0FFB, 0x00BC, 0x0FFB, 0x00C2,
            0x0FFB, 0x00C8, 0x0FFA, 0x00CF, 0x0FFA, 0x00D5, 0x0FFA, 0x00DB,
            0x0FF9, 0x00E2, 0x0FF9, 0x00E8, 0x0FF9, 0x00EE, 0x0FF8, 0x00F4,
            0x0FF8, 0x00FB, 0x0FF7, 0x0101, 0x0FF7, 0x0107, 0x0FF7, 0x010D,
            0x0FF6, 0x0114, 0x0FF6, 0x011A, 0x0FF5, 0x0120, 0x0FF5, 0x0127,
            0x0FF4, 0x012D, 0x0FF4, 0x0133, 0x0FF3, 0x0139, 0x0FF3, 0x0140,
            0x0FF2, 0x0146, 0x0FF2, 0x014C, 0x0FF1, 0x0152, 0x0FF1, 0x0159,
            0x0FF0, 0x015F, 0x0FF0, 0x0165, 0x0FEF, 0x016B, 0x0FEF, 0x0172,
            0x0FEE, 0x0178, 0x0FEE, 0x017E, 0x0FED, 0x0184, 0x0FEC, 0x018B,
            0x0FEC, 0x0191, 0x0FEB, 0x0197, 0x0FEB, 0x019D, 0x0FEA, 0x01A4,
            0x0FE9, 0x01AA, 0x0FE9, 0x01B0, 0x0FE8, 0x01B6, 0x0FE7, 0x01BD,
            0x0FE7, 0x01C3, 0x0FE6, 0x01C9, 0x0FE5, 0x01CF, 0x0FE4, 0x01D6,
            0x0FE4, 0x01DC, 0x0FE3, 0x01E2, 0x0FE2, 0x01E8, 0x0FE1, 0x01EF,
            0x0FE1, 0x01F5, 0x0FE0, 0x01FB, 0x0FDF, 0x0201, 0x0FDE, 0x0208,
            0x0FDE, 0x020E, 0x0FDD, 0x0214, 0x0FDC, 0x021A, 0x0FDB, 0x0221,
            0x0FDA, 0x0227, 0x0FD9, 0x022D, 0x0FD9, 0x0233, 0x0FD8, 0x0239,
            0x0FD7, 0x0240, 0x0FD6, 0x0246, 0x0FD5, 0x024C, 0x0FD4, 0x0252,
            0x0FD3, 0x0259, 0x0FD2, 0x025F, 0x0FD1, 0x0265, 0x0FD0, 0x026B,
            0x0FCF, 0x0271, 0x0FCE, 0x0278, 0x0FCD, 0x027E, 0x0FCC, 0x0284,
            0x0FCB, 0x028A, 0x0FCA, 0x0290, 0x0FC9, 0x0297, 0x0FC8, 0x029D,
            0x0FC7, 0x02A3, 0x0FC6, 0x02A9, 0x0FC5, 0x02AF, 0x0FC4, 0x02B6,
            0x0FC3, 0x02BC, 0x0FC2, 0x02C2, 0x0FC1, 0x02C8, 0x0FC0, 0x02CE,
            0x0FBF, 0x02D5, 0x0FBE, 0x02DB, 0x0FBD, 0x02E1, 0x0FBB, 0x02E7,
            0x0FBA, 0x02ED, 0x0FB9, 0x02F3, 0x0FB8, 0x02FA, 0x0FB7, 0x0300,
            0x0FB6, 0x0306, 0x0FB4, 0x030C, 0x0FB3, 0x0312, 0x0FB2, 0x0318,
            0x0FB1, 0x031F, 0x0FB0, 0x0325, 0x0FAE, 0x032B, 0x0FAD, 0x0331,
            0x0FAC, 0x0337, 0x0FAB, 0x033D, 0x0FA9, 0x0344, 0x0FA8, 0x034A,
            0x0FA7, 0x0350, 0x0FA5, 0x0356, 0x0FA4, 0x035C, 0x0FA3, 0x0362,
            0x0FA1, 0x0368, 0x0FA0, 0x036F, 0x0F9F, 0x0375, 0x0F9D, 0x037B,
            0x0F9C, 0x0381, 0x0F9B, 0x0387, 0x0F99, 0x038D, 0x0F98, 0x0393,
            0x0F96, 0x0399, 0x0F95, 0x03A0, 0x0F94, 0x03A6, 0x0F92, 0x03AC,
            0x0F91, 0x03B2, 0x0F8F, 0x03B8, 0x0F8E, 0x03BE, 0x0F8C, 0x03C4,
            0x0F8B, 0x03CA, 0x0F89, 0x03D0, 0x0F88, 0x03D7, 0x0F86, 0x03DD,
            0x0F85, 0x03E3, 0x0F83, 0x03E9, 0x0F82, 0x03EF, 0x0F80, 0x03F5,
            0x0F7F, 0x03FB, 0x0F7D, 0x0401, 0x0F7B, 0x0407, 0x0F7A, 0x040D,
            0x0F78, 0x0413, 0x0F77, 0x041A, 0x0F75, 0x0420, 0x0F73, 0x0426,
            0x0F72, 0x042C, 0x0F70, 0x0432, 0x0F6E, 0x0438, 0x0F6D, 0x043E,
            0x0F6B, 0x0444, 0x0F69, 0x044A, 0x0F68, 0x0450, 0x0F66, 0x0456,
            0x0F64, 0x045C, 0x0F63, 0x0462, 0x0F61, 0x0468, 0x0F5F, 0x046E,
            0x0F5D, 0x0474, 0x0F5C, 0x047A, 0x0F5A, 0x0480, 0x0F58, 0x0486,
            0x0F56, 0x048C, 0x0F55, 0x0492, 0x0F53, 0x0498, 0x0F51, 0x049E,
            0x0F4F, 0x04A5, 0x0F4D, 0x04AB, 0x0F4B, 0x04B1, 0x0F4A, 0x04B7,
            0x0F48, 0x04BD, 0x0F46, 0x04C3, 0x0F44, 0x04C9, 0x0F42, 0x04CF,
            0x0F40, 0x04D5, 0x0F3E, 0x04DB, 0x0F3C, 0x04E0, 0x0F3B, 0x04E6,
            0x0F39, 0x04EC, 0x0F37, 0x04F2, 0x0F35, 0x04F8, 0x0F33, 0x04FE,
            0x0F31, 0x0504, 0x0F2F, 0x050A, 0x0F2D, 0x0510, 0x0F2B, 0x0516,
            0x0F29, 0x051C, 0x0F27, 0x0522, 0x0F25, 0x0528, 0x0F23, 0x052E,
            0x0F21, 0x0534, 0x0F1F, 0x053A, 0x0F1D, 0x0540, 0x0F1B, 0x0546,
            0x0F18, 0x054C, 0x0F16, 0x0552, 0x0F14, 0x0558, 0x0F12, 0x055D,
            0x0F10, 0x0563, 0x0F0E, 0x0569, 0x0F0C, 0x056F, 0x0F0A, 0x0575,
            0x0F08, 0x057B, 0x0F05, 0x0581, 0x0F03, 0x0587, 0x0F01, 0x058D,
            0x0EFF, 0x0593, 0x0EFD, 0x0599, 0x0EFA, 0x059E, 0x0EF8, 0x05A4,
            0x0EF6, 0x05AA, 0x0EF4, 0x05B0, 0x0EF2, 0x05B6, 0x0EEF, 0x05BC,
            0x0EED, 0x05C2, 0x0EEB, 0x05C7, 0x0EE8, 0x05CD, 0x0EE6, 0x05D3,
            0x0EE4, 0x05D9, 0x0EE2, 0x05DF, 0x0EDF, 0x05E5, 0x0EDD, 0x05EB,
            0x0EDB, 0x05F0, 0x0ED8, 0x05F6, 0x0ED6, 0x05FC, 0x0ED4, 0x0602,
            0x0ED1, 0x0608, 0x0ECF, 0x060E, 0x0ECD, 0x0613, 0x0ECA, 0x0619,
            0x0EC8, 0x061F, 0x0EC5, 0x0625, 0x0EC3, 0x062B, 0x0EC0, 0x0630,
            0x0EBE, 0x0636, 0x0EBC, 0x063C, 0x0EB9, 0x0642, 0x0EB7, 0x0648,
            0x0EB4, 0x064D, 0x0EB2, 0x0653, 0x0EAF, 0x0659, 0x0EAD, 0x065F,
            0x0EAA, 0x0664, 0x0EA8, 0x066A, 0x0EA5, 0x0670, 0x0EA3, 0x0676,
            0x0EA0, 0x067B, 0x0E9E, 0x0681, 0x0E9B, 0x0687, 0x0E98, 0x068D,
            0x0E96, 0x0692, 0x0E93, 0x0698, 0x0E91, 0x069E, 0x0E8E, 0x06A3,
            0x0E8B, 0x06A9, 0x0E89, 0x06AF, 0x0E86, 0x06B5, 0x0E84, 0x06BA,
            0x0E81, 0x06C0, 0x0E7E, 0x06C6, 0x0E7C, 0x06CB, 0x0E79, 0x06D1,
            0x0E76, 0x06D7, 0x0E74, 0x06DC, 0x0E71, 0x06E2, 0x0E6E, 0x06E8,
            0x0E6B, 0x06ED, 0x0E69, 0x06F3, 0x0E66, 0x06F9, 0x0E63, 0x06FE,
            0x0E60, 0x0704, 0x0E5E, 0x070A, 0x0E5B, 0x070F, 0x0E58, 0x0715,
            0x0E55, 0x071B, 0x0E53, 0x0720, 0x0E50, 0x0726, 0x0E4D, 0x072B,
            0x0E4A, 0x0731, 0x0E47, 0x0737, 0x0E44, 0x073C, 0x0E42, 0x0742,
            0x0E3F, 0x0748, 0x0E3C, 0x074D, 0x0E39, 0x0753, 0x0E36, 0x0758,
            0x0E33, 0x075E, 0x0E30, 0x0763, 0x0E2D, 0x0769, 0x0E2B, 0x076F,
            0x0E28, 0x0774, 0x0E25, 0x077A, 0x0E22, 0x077F, 0x0E1F, 0x0785,
            0x0E1C, 0x078A, 0x0E19, 0x0790, 0x0E16, 0x0795, 0x0E13, 0x079B,
            0x0E10, 0x07A0, 0x0E0D, 0x07A6, 0x0E0A, 0x07AC, 0x0E07, 0x07B1,
            0x0E04, 0x07B7, 0x0E01, 0x07BC, 0x0DFE, 0x07C2, 0x0DFB, 0x07C7,
            0x0DF8, 0x07CD, 0x0DF5, 0x07D2, 0x0DF2, 0x07D7, 0x0DEE, 0x07DD,
            0x0DEB, 0x07E2, 0x0DE8, 0x07E8, 0x0DE5, 0x07ED, 0x0DE2, 0x07F3,
            0x0DDF, 0x07F8, 0x0DDC, 0x07FE, 0x0DD9, 0x0803, 0x0DD5, 0x0809,
            0x0DD2, 0x080E, 0x0DCF, 0x0813, 0x0DCC, 0x0819, 0x0DC9, 0x081E,
            0x0DC6, 0x0824, 0x0DC2, 0x0829, 0x0DBF, 0x082E, 0x0DBC, 0x0834,
            0x0DB9, 0x0839, 0x0DB6, 0x083F, 0x0DB2, 0x0844, 0x0DAF, 0x0849,
            0x0DAC, 0x084F, 0x0DA9, 0x0854, 0x0DA5, 0x085A, 0x0DA2, 0x085F,
            0x0D9F, 0x0864, 0x0D9B, 0x086A, 0x0D98, 0x086F, 0x0D95, 0x0874,
            0x0D91, 0x087A, 0x0D8E, 0x087F, 0x0D8B, 0x0884, 0x0D87, 0x088A,
            0x0D84, 0x088F, 0x0D81, 0x0894, 0x0D7D, 0x0899, 0x0D7A, 0x089F,
            0x0D77, 0x08A4, 0x0D73, 0x08A9, 0x0D70, 0x08AF, 0x0D6C, 0x08B4,
            0x0D69, 0x08B9, 0x0D65, 0x08BE, 0x0D62, 0x08C4, 0x0D5F, 0x08C9,
            0x0D5B, 0x08CE, 0x0D58, 0x08D3, 0x0D54, 0x08D9, 0x0D51, 0x08DE,
            0x0D4D, 0x08E3, 0x0D4A, 0x08E8, 0x0D46, 0x08EE, 0x0D43, 0x08F3,
            0x0D3F, 0x08F8, 0x0D3C, 0x08FD, 0x0D38, 0x0902, 0x0D35, 0x0908,
            0x0D31, 0x090D, 0x0D2D, 0x0912, 0x0D2A, 0x0917, 0x0D26, 0x091C,
            0x0D23, 0x0921, 0x0D1F, 0x0927, 0x0D1C, 0x092C, 0x0D18, 0x0931,
            0x0D14, 0x0936, 0x0D11, 0x093B, 0x0D0D, 0x0940, 0x0D09, 0x0945,
            0x0D06, 0x094B, 0x0D02, 0x0950, 0x0CFE, 0x0955, 0x0CFB, 0x095A,
            0x0CF7, 0x095F, 0x0CF3, 0x0964, 0x0CF0, 0x0969, 0x0CEC, 0x096E,
            0x0CE8, 0x0973, 0x0CE5, 0x0978, 0x0CE1, 0x097D, 0x0CDD, 0x0982,
            0x0CD9, 0x0987, 0x0CD6, 0x098D, 0x0CD2, 0x0992, 0x0CCE, 0x0997,
            0x0CCA, 0x099C, 0x0CC7, 0x09A1, 0x0CC3, 0x09A6, 0x0CBF, 0x09AB,
            0x0CBB, 0x09B0, 0x0CB7, 0x09B5, 0x0CB4, 0x09BA, 0x0CB0, 0x09BF,
            0x0CAC, 0x09C4, 0x0CA8, 0x09C9, 0x0CA4, 0x09CE, 0x0CA0, 0x09D3,
            0x0C9D, 0x09D7, 0x0C99, 0x09DC, 0x0C95, 0x09E1, 0x0C91, 0x09E6,
            0x0C8D, 0x09EB, 0x0C89, 0x09F0, 0x0C85, 0x09F5, 0x0C81, 0x09FA,
            0x0C7D, 0x09FF, 0x0C79, 0x0A04, 0x0C76, 0x0A09, 0x0C72, 0x0A0E,
            0x0C6E, 0x0A12, 0x0C6A, 0x0A17, 0x0C66, 0x0A1C, 0x0C62, 0x0A21,
            0x0C5E, 0x0A26, 0x0C5A, 0x0A2B, 0x0C56, 0x0A30, 0x0C52, 0x0A35,
            0x0C4E, 0x0A39, 0x0C4A, 0x0A3E, 0x0C46, 0x0A43, 0x0C42, 0x0A48,
            0x0C3E, 0x0A4D, 0x0C3A, 0x0A51, 0x0C36, 0x0A56, 0x0C31, 0x0A5B,
            0x0C2D, 0x0A60, 0x0C29, 0x0A65, 0x0C25, 0x0A69, 0x0C21, 0x0A6E,
            0x0C1D, 0x0A73, 0x0C19, 0x0A78, 0x0C15, 0x0A7C, 0x0C11, 0x0A81,
            0x0C0D, 0x0A86, 0x0C08, 0x0A8B, 0x0C04, 0x0A8F, 0x0C00, 0x0A94,
            0x0BFC, 0x0A99, 0x0BF8, 0x0A9D, 0x0BF4, 0x0AA2, 0x0BEF, 0x0AA7,
            0x0BEB, 0x0AAC, 0x0BE7, 0x0AB0, 0x0BE3, 0x0AB5, 0x0BDF, 0x0ABA,
            0x0BDA, 0x0ABE, 0x0BD6, 0x0AC3, 0x0BD2, 0x0AC8, 0x0BCE, 0x0ACC,
            0x0BCA, 0x0AD1, 0x0BC5, 0x0AD5, 0x0BC1, 0x0ADA, 0x0BBD, 0x0ADF,
            0x0BB8, 0x0AE3, 0x0BB4, 0x0AE8, 0x0BB0, 0x0AEC, 0x0BAC, 0x0AF1,
            0x0BA7, 0x0AF6, 0x0BA3, 0x0AFA, 0x0B9F, 0x0AFF, 0x0B9A, 0x0B03,
            0x0B96, 0x0B08, 0x0B92, 0x0B0C, 0x0B8D, 0x0B11, 0x0B89, 0x0B15,
            0x0B85, 0x0B1A, 0x0B80, 0x0B1F, 0x0B7C, 0x0B23, 0x0B78, 0x0B28,
            0x0B73, 0x0B2C, 0x0B6F, 0x0B31, 0x0B6A, 0x0B35, 0x0B66, 0x0B3A,
            0x0B62, 0x0B3E, 0x0B5D, 0x0B42, 0x0B59, 0x0B47, 0x0B54, 0x0B4B,
            0x0B50, 0x0B50, 0x0B4B, 0x0B54, 0x0B47, 0x0B59, 0x0B42, 0x0B5D,
            0x0B3E, 0x0B62, 0x0B3A, 0x0B66, 0x0B35, 0x0B6A, 0x0B31, 0x0B6F,
            0x0B2C, 0x0B73, 0x0B28, 0x0B78, 0x0B23, 0x0B7C, 0x0B1F, 0x0B80,
            0x0B1A, 0x0B85, 0x0B15, 0x0B89, 0x0B11, 0x0B8D, 0x0B0C, 0x0B92,
            0x0B08, 0x0B96, 0x0B03, 0x0B9A, 0x0AFF, 0x0B9F, 0x0AFA, 0x0BA3,
            0x0AF6, 0x0BA7, 0x0AF1, 0x0BAC, 0x0AEC, 0x0BB0, 0x0AE8, 0x0BB4,
            0x0AE3, 0x0BB8, 0x0ADF, 0x0BBD, 0x0ADA, 0x0BC1, 0x0AD5, 0x0BC5,
            0x0AD1, 0x0BCA, 0x0ACC, 0x0BCE, 0x0AC8, 0x0BD2, 0x0AC3, 0x0BD6,
            0x0ABE, 0x0BDA, 0x0ABA, 0x0BDF, 0x0AB5, 0x0BE3, 0x0AB0, 0x0BE7,
            0x0AAC, 0x0BEB, 0x0AA7, 0x0BEF, 0x0AA2, 0x0BF4, 0x0A9D, 0x0BF8,
            0x0A99, 0x0BFC, 0x0A94, 0x0C00, 0x0A8F, 0x0C04, 0x0A8B, 0x0C08,
            0x0A86, 0x0C0D, 0x0A81, 0x0C11, 0x0A7C, 0x0C15, 0x0A78, 0x0C19,
            0x0A73, 0x0C1D, 0x0A6E, 0x0C21, 0x0A69, 0x0C25, 0x0A65, 0x0C29,
            0x0A60, 0x0C2D, 0x0A5B, 0x0C31, 0x0A56, 0x0C36, 0x0A51, 0x0C3A,
            0x0A4D, 0x0C3E, 0x0A48, 0x0C42, 0x0A43, 0x0C46, 0x0A3E, 0x0C4A,
            0x0A39, 0x0C4E, 0x0A35, 0x0C52, 0x0A30, 0x0C56, 0x0A2B, 0x0C5A,
            0x0A26, 0x0C5E, 0x0A21, 0x0C62, 0x0A1C, 0x0C66, 0x0A17, 0x0C6A,
            0x0A12, 0x0C6E, 0x0A0E, 0x0C72, 0x0A09, 0x0C76, 0x0A04, 0x0C79,
            0x09FF, 0x0C7D, 0x09FA, 0x0C81, 0x09F5, 0x0C85, 0x09F0, 0x0C89,
            0x09EB, 0x0C8D, 0x09E6, 0x0C91, 0x09E1, 0x0C95, 0x09DC, 0x0C99,
            0x09D7, 0x0C9D, 0x09D3, 0x0CA0, 0x09CE, 0x0CA4, 0x09C9, 0x0CA8,
            0x09C4, 0x0CAC, 0x09BF, 0x0CB0, 0x09BA, 0x0CB4, 0x09B5, 0x0CB7,
            0x09B0, 0x0CBB, 0x09AB, 0x0CBF, 0x09A6, 0x0CC3, 0x09A1, 0x0CC7,
            0x099C, 0x0CCA, 0x0997, 0x0CCE, 0x0992, 0x0CD2, 0x098D, 0x0CD6,
            0x0987, 0x0CD9, 0x0982, 0x0CDD, 0x097D, 0x0CE1, 0x0978, 0x0CE5,
            0x0973, 0x0CE8, 0x096E, 0x0CEC, 0x0969, 0x0CF0, 0x0964, 0x0CF3,
            0x095F, 0x0CF7, 0x095A, 0x0CFB, 0x0955, 0x0CFE, 0x0950, 0x0D02,
            0x094B, 0x0D06, 0x0945, 0x0D09, 0x0940, 0x0D0D, 0x093B, 0x0D11,
            0x0936, 0x0D14, 0x0931, 0x0D18, 0x092C, 0x0D1C, 0x0927, 0x0D1F,
            0x0921, 0x0D23, 0x091C, 0x0D26, 0x0917, 0x0D2A, 0x0912, 0x0D2D,
            0x090D, 0x0D31, 0x0908, 0x0D35, 0x0902, 0x0D38, 0x08FD, 0x0D3C,
            0x08F8, 0x0D3F, 0x08F3, 0x0D43, 0x08EE, 0x0D46, 0x08E8, 0x0D4A,
            0x08E3, 0x0D4D, 0x08DE, 0x0D51, 0x08D9, 0x0D54, 0x08D3, 0x0D58,
            0x08CE, 0x0D5B, 0x08C9, 0x0D5F, 0x08C4, 0x0D62, 0x08BE, 0x0D65,
            0x08B9, 0x0D69, 0x08B4, 0x0D6C, 0x08AF, 0x0D70, 0x08A9, 0x0D73,
            0x08A4, 0x0D77, 0x089F, 0x0D7A, 0x0899, 0x0D7D, 0x0894, 0x0D81,
            0x088F, 0x0D84, 0x088A, 0x0D87, 0x0884, 0x0D8B, 0x087F, 0x0D8E,
            0x087A, 0x0D91, 0x0874, 0x0D95, 0x086F, 0x0D98, 0x086A, 0x0D9B,
            0x0864, 0x0D9F, 0x085F, 0x0DA2, 0x085A, 0x0DA5, 0x0854, 0x0DA9,
            0x084F, 0x0DAC, 0x0849, 0x0DAF, 0x0844, 0x0DB2, 0x083F, 0x0DB6,
            0x0839, 0x0DB9, 0x0834, 0x0DBC, 0x082E, 0x0DBF, 0x0829, 0x0DC2,
            0x0824, 0x0DC6, 0x081E, 0x0DC9, 0x0819, 0x0DCC, 0x0813, 0x0DCF,
            0x080E, 0x0DD2, 0x0809, 0x0DD5, 0x0803, 0x0DD9, 0x07FE, 0x0DDC,
            0x07F8, 0x0DDF, 0x07F3, 0x0DE2, 0x07ED, 0x0DE5, 0x07E8, 0x0DE8,
            0x07E2, 0x0DEB, 0x07DD, 0x0DEE, 0x07D7, 0x0DF2, 0x07D2, 0x0DF5,
            0x07CD, 0x0DF8, 0x07C7, 0x0DFB, 0x07C2, 0x0DFE, 0x07BC, 0x0E01,
            0x07B7, 0x0E04, 0x07B1, 0x0E07, 0x07AC, 0x0E0A, 0x07A6, 0x0E0D,
            0x07A0, 0x0E10, 0x079B, 0x0E13, 0x0795, 0x0E16, 0x0790, 0x0E19,
            0x078A, 0x0E1C, 0x0785, 0x0E1F, 0x077F, 0x0E22, 0x077A, 0x0E25,
            0x0774, 0x0E28, 0x076F, 0x0E2B, 0x0769, 0x0E2D, 0x0763, 0x0E30,
            0x075E, 0x0E33, 0x0758, 0x0E36, 0x0753, 0x0E39, 0x074D, 0x0E3C,
            0x0748, 0x0E3F, 0x0742, 0x0E42, 0x073C, 0x0E44, 0x0737, 0x0E47,
            0x0731, 0x0E4A, 0x072B, 0x0E4D, 0x0726, 0x0E50, 0x0720, 0x0E53,
            0x071B, 0x0E55, 0x0715, 0x0E58, 0x070F, 0x0E5B, 0x070A, 0x0E5E,
            0x0704, 0x0E60, 0x06FE, 0x0E63, 0x06F9, 0x0E66, 0x06F3, 0x0E69,
            0x06ED, 0x0E6B, 0x06E8, 0x0E6E, 0x06E2, 0x0E71, 0x06DC, 0x0E74,
            0x06D7, 0x0E76, 0x06D1, 0x0E79, 0x06CB, 0x0E7C, 0x06C6, 0x0E7E,
            0x06C0, 0x0E81, 0x06BA, 0x0E84, 0x06B5, 0x0E86, 0x06AF, 0x0E89,
            0x06A9, 0x0E8B, 0x06A3, 0x0E8E, 0x069E, 0x0E91, 0x0698, 0x0E93,
            0x0692, 0x0E96, 0x068D, 0x0E98, 0x0687, 0x0E9B, 0x0681, 0x0E9E,
            0x067B, 0x0EA0, 0x0676, 0x0EA3, 0x0670, 0x0EA5, 0x066A, 0x0EA8,
            0x0664, 0x0EAA, 0x065F, 0x0EAD, 0x0659, 0x0EAF, 0x0653, 0x0EB2,
            0x064D, 0x0EB4, 0x0648, 0x0EB7, 0x0642, 0x0EB9, 0x063C, 0x0EBC,
            0x0636, 0x0EBE, 0x0630, 0x0EC0, 0x062B, 0x0EC3, 0x0625, 0x0EC5,
            0x061F, 0x0EC8, 0x0619, 0x0ECA, 0x0613, 0x0ECD, 0x060E, 0x0ECF,
            0x0608, 0x0ED1, 0x0602, 0x0ED4, 0x05FC, 0x0ED6, 0x05F6, 0x0ED8,
            0x05F0, 0x0EDB, 0x05EB, 0x0EDD, 0x05E5, 0x0EDF, 0x05DF, 0x0EE2,
            0x05D9, 0x0EE4, 0x05D3, 0x0EE6, 0x05CD, 0x0EE8, 0x05C7, 0x0EEB,
            0x05C2, 0x0EED, 0x05BC, 0x0EEF, 0x05B6, 0x0EF2, 0x05B0, 0x0EF4,
            0x05AA, 0x0EF6, 0x05A4, 0x0EF8, 0x059E, 0x0EFA, 0x0599, 0x0EFD,
            0x0593, 0x0EFF, 0x058D, 0x0F01, 0x0587, 0x0F03, 0x0581, 0x0F05,
            0x057B, 0x0F08, 0x0575, 0x0F0A, 0x056F, 0x0F0C, 0x0569, 0x0F0E,
            0x0563, 0x0F10, 0x055D, 0x0F12, 0x0558, 0x0F14, 0x0552, 0x0F16,
            0x054C, 0x0F18, 0x0546, 0x0F1B, 0x0540, 0x0F1D, 0x053A, 0x0F1F,
            0x0534, 0x0F21, 0x052E, 0x0F23, 0x0528, 0x0F25, 0x0522, 0x0F27,
            0x051C, 0x0F29, 0x0516, 0x0F2B, 0x0510, 0x0F2D, 0x050A, 0x0F2F,
            0x0504, 0x0F31, 0x04FE, 0x0F33, 0x04F8, 0x0F35, 0x04F2, 0x0F37,
            0x04EC, 0x0F39, 0x04E6, 0x0F3B, 0x04E0, 0x0F3C, 0x04DB, 0x0F3E,
            0x04D5, 0x0F40, 0x04CF, 0x0F42, 0x04C9, 0x0F44, 0x04C3, 0x0F46,
            0x04BD, 0x0F48, 0x04B7, 0x0F4A, 0x04B1, 0x0F4B, 0x04AB, 0x0F4D,
            0x04A5, 0x0F4F, 0x049E, 0x0F51, 0x0498, 0x0F53, 0x0492, 0x0F55,
            0x048C, 0x0F56, 0x0486, 0x0F58, 0x0480, 0x0F5A, 0x047A, 0x0F5C,
            0x0474, 0x0F5D, 0x046E, 0x0F5F, 0x0468, 0x0F61, 0x0462, 0x0F63,
            0x045C, 0x0F64, 0x0456, 0x0F66, 0x0450, 0x0F68, 0x044A, 0x0F69,
            0x0444, 0x0F6B, 0x043E, 0x0F6D, 0x0438, 0x0F6E, 0x0432, 0x0F70,
            0x042C, 0x0F72, 0x0426, 0x0F73, 0x0420, 0x0F75, 0x041A, 0x0F77,
            0x0413, 0x0F78, 0x040D, 0x0F7A, 0x0407, 0x0F7B, 0x0401, 0x0F7D,
            0x03FB, 0x0F7F, 0x03F5, 0x0F80, 0x03EF, 0x0F82, 0x03E9, 0x0F83,
            0x03E3, 0x0F85, 0x03DD, 0x0F86, 0x03D7, 0x0F88, 0x03D0, 0x0F89,
            0x03CA, 0x0F8B, 0x03C4, 0x0F8C, 0x03BE, 0x0F8E, 0x03B8, 0x0F8F,
            0x03B2, 0x0F91, 0x03AC, 0x0F92, 0x03A6, 0x0F94, 0x03A0, 0x0F95,
            0x0399, 0x0F96, 0x0393, 0x0F98, 0x038D, 0x0F99, 0x0387, 0x0F9B,
            0x0381, 0x0F9C, 0x037B, 0x0F9D, 0x0375, 0x0F9F, 0x036F, 0x0FA0,
            0x0368, 0x0FA1, 0x0362, 0x0FA3, 0x035C, 0x0FA4, 0x0356, 0x0FA5,
            0x0350, 0x0FA7, 0x034A, 0x0FA8, 0x0344, 0x0FA9, 0x033D, 0x0FAB,
            0x0337, 0x0FAC, 0x0331, 0x0FAD, 0x032B, 0x0FAE, 0x0325, 0x0FB0,
            0x031F, 0x0FB1, 0x0318, 0x0FB2, 0x0312, 0x0FB3, 0x030C, 0x0FB4,
            0x0306, 0x0FB6, 0x0300, 0x0FB7, 0x02FA, 0x0FB8, 0x02F3, 0x0FB9,
            0x02ED, 0x0FBA, 0x02E7, 0x0FBB, 0x02E1, 0x0FBD, 0x02DB, 0x0FBE,
            0x02D5, 0x0FBF, 0x02CE, 0x0FC0, 0x02C8, 0x0FC1, 0x02C2, 0x0FC2,
            0x02BC, 0x0FC3, 0x02B6, 0x0FC4, 0x02AF, 0x0FC5, 0x02A9, 0x0FC6,
            0x02A3, 0x0FC7, 0x029D, 0x0FC8, 0x0297, 0x0FC9, 0x0290, 0x0FCA,
            0x028A, 0x0FCB, 0x0284, 0x0FCC, 0x027E, 0x0FCD, 0x0278, 0x0FCE,
            0x0271, 0x0FCF, 0x026B, 0x0FD0, 0x0265, 0x0FD1, 0x025F, 0x0FD2,
            0x0259, 0x0FD3, 0x0252, 0x0FD4, 0x024C, 0x0FD5, 0x0246, 0x0FD6,
            0x0240, 0x0FD7, 0x0239, 0x0FD8, 0x0233, 0x0FD9, 0x022D, 0x0FD9,
            0x0227, 0x0FDA, 0x0221, 0x0FDB, 0x021A, 0x0FDC, 0x0214, 0x0FDD,
            0x020E, 0x0FDE, 0x0208, 0x0FDE, 0x0201, 0x0FDF, 0x01FB, 0x0FE0,
            0x01F5, 0x0FE1, 0x01EF, 0x0FE1, 0x01E8, 0x0FE2, 0x01E2, 0x0FE3,
            0x01DC, 0x0FE4, 0x01D6, 0x0FE4, 0x01CF, 0x0FE5, 0x01C9, 0x0FE6,
            0x01C3, 0x0FE7, 0x01BD, 0x0FE7, 0x01B6, 0x0FE8, 0x01B0, 0x0FE9,
            0x01AA, 0x0FE9, 0x01A4, 0x0FEA, 0x019D, 0x0FEB, 0x0197, 0x0FEB,
            0x0191, 0x0FEC, 0x018B, 0x0FEC, 0x0184, 0x0FED, 0x017E, 0x0FEE,
            0x0178, 0x0FEE, 0x0172, 0x0FEF, 0x016B, 0x0FEF, 0x0165, 0x0FF0,
            0x015F, 0x0FF0, 0x0159, 0x0FF1, 0x0152, 0x0FF1, 0x014C, 0x0FF2,
            0x0146, 0x0FF2, 0x0140, 0x0FF3, 0x0139, 0x0FF3, 0x0133, 0x0FF4,
            0x012D, 0x0FF4, 0x0127, 0x0FF5, 0x0120, 0x0FF5, 0x011A, 0x0FF6,
            0x0114, 0x0FF6, 0x010D, 0x0FF7, 0x0107, 0x0FF7, 0x0101, 0x0FF7,
            0x00FB, 0x0FF8, 0x00F4, 0x0FF8, 0x00EE, 0x0FF9, 0x00E8, 0x0FF9,
            0x00E2, 0x0FF9, 0x00DB, 0x0FFA, 0x00D5, 0x0FFA, 0x00CF, 0x0FFA,
            0x00C8, 0x0FFB, 0x00C2, 0x0FFB, 0x00BC, 0x0FFB, 0x00B6, 0x0FFB,
            0x00AF, 0x0FFC, 0x00A9, 0x0FFC, 0x00A3, 0x0FFC, 0x009D, 0x0FFC,
            0x0096, 0x0FFD, 0x0090, 0x0FFD, 0x008A, 0x0FFD, 0x0083, 0x0FFD,
            0x007D, 0x0FFE, 0x0077, 0x0FFE, 0x0071, 0x0FFE, 0x006A, 0x0FFE,
            0x0064, 0x0FFE, 0x005E, 0x0FFE, 0x0057, 0x0FFF, 0x0051, 0x0FFF,
            0x004B, 0x0FFF, 0x0045, 0x0FFF, 0x003E, 0x0FFF, 0x0038, 0x0FFF,
            0x0032, 0x0FFF, 0x002B, 0x0FFF, 0x0025, 0x0FFF, 0x001F, 0x0FFF,
            0x0019, 0x0FFF, 0x0012, 0x0FFF, 0x000C, 0x0FFF, 0x0006, 0x0FFF,
        };
    }
}
