﻿using System;
using System.Collections.Generic;
using System.Linq;
using CppSharp;
using CppSharp.AST;
using CppSharp.Generators;
using CppSharp.Passes;
using Embeddinator.Passes;

namespace Embeddinator.Generators
{
    public class CGenerator : Generator
    {
        public CGenerator(BindingContext context) : base(context)
        {
        }

        public override List<CodeGenerator> Generate(IEnumerable<TranslationUnit> units)
        {
            var unit = units.First();
            var headers = new CHeaders(Context, unit);
            var sources = new CSources(Context, unit);

            return new List<CodeGenerator> { headers, sources };
        }

        public static string GenId(string id)
        {
            return "__" + id.Replace('-', '_');
        }

        public static Options Options;

        public static string QualifiedName(Declaration decl)
        {
            if (Options.GeneratorKind == GeneratorKind.CPlusPlus)
                return decl.Name;

            var isSwiftTarget = Options.GeneratorKinds.Contains(GeneratorKind.Swift);

            return (isSwiftTarget && !decl.IsImplicit) ? $"_{decl.QualifiedName}" : decl.QualifiedName;
        }

        public static string ObjectInstanceId => GenId("object");

        public static string AssemblyId(TranslationUnit unit)
        {
            return GenId(unit.FileName).Replace('.', '_').Replace('-', '_');
        }

        private static CppTypePrintFlavorKind GetTypePrinterFlavorKind(GeneratorKind kind)
        {
            switch (kind)
            {
                case GeneratorKind.C:
                    return CppTypePrintFlavorKind.C;
                case GeneratorKind.CPlusPlus:
                    return CppTypePrintFlavorKind.Cpp;
                case GeneratorKind.ObjectiveC:
                    return CppTypePrintFlavorKind.ObjC;
            }

            throw new NotImplementedException();
        }

        public static CManagedToNativeTypePrinter GetCTypePrinter(GeneratorKind kind)
        {
            var typePrinter = new CManagedToNativeTypePrinter
            {
                PrintScopeKind = TypePrintScopeKind.Qualified,
                PrintFlavorKind = GetTypePrinterFlavorKind(kind),
                PrintVariableArrayAsPointers = true
            };

            return typePrinter;
        }

        public virtual CManagedToNativeTypePrinter TypePrinter =>
            GetCTypePrinter(GeneratorKind.C);

        public override bool SetupPasses()
        {
            SetupPasses(Context.TranslationUnitPasses);
            return true;
        }

        public static void SetupPasses(PassBuilder<TranslationUnitPass> passes)
        {
            passes.AddPass(new FixMethodParametersPass());
        }

        public static void RunPasses(BindingContext context,
            PassBuilder<TranslationUnitPass> passes)
        {
            // Since the native declarations are targetting the generated C bindings,
            // we need to modify the AST to match the one generated by the C target.

            var generatorKind = context.Options.GeneratorKind;

            context.Options.GeneratorKind = GeneratorKind.C;

            passes.RunPasses(pass =>
            {
                pass.Context = context;
                pass.VisitASTContext(context.ASTContext);
            });

            context.Options.GeneratorKind = generatorKind;
        }

        protected override string TypePrinterDelegate(CppSharp.AST.Type type)
        {
            return type.Visit(TypePrinter);
        }
    }

    public abstract class CCodeGenerator : CodeGenerator
    {
        public TranslationUnit Unit;

        Options EmbedOptions => Context.Options as Options; 

        public CCodeGenerator(BindingContext context,
            TranslationUnit unit) : base(context, unit)
        {
            Unit = unit;
            VisitOptions.VisitPropertyAccessors = true;
        }

        public override string GeneratedIdentifier(string id)
        {
            return CGenerator.GenId(id);
        }

        public CManagedToNativeTypePrinter CTypePrinter =>
                CGenerator.GetCTypePrinter(Options.GeneratorKind);

        public virtual void WriteHeaders() { }

        public void WriteInclude(string include)
        {
            if (EmbedOptions.GenerateSupportFiles)
                WriteLine("#include \"{0}\"", include);
            else
                WriteLine("#include <{0}>", include);
        }

        public static Dictionary<string, string> GeneratedMethodNames =
            new Dictionary<string, string>();

        public static string GetMethodIdentifier(Method method)
        {
            var methodName = (method.IsConstructor || method.IsDestructor) ?
                method.Name : method.OriginalName;

            var @class = method.Namespace as Class;
            var name = $"{@class.QualifiedName}_{methodName}";

            var associated = method.AssociatedDeclaration ?? method;

            if (associated.DefinitionOrder != 0)
                name += $"_{associated.DefinitionOrder}";

            // Store the name so it can be re-used later by other generators.
            GeneratedMethodNames[method.ManagedQualifiedName()] = name;

            return name;
        }

        public override bool VisitDeclaration(Declaration decl)
        {
            return decl.IsGenerated && !AlreadyVisited(decl);
        }

        public override void GenerateMethodSpecifier(Method method, Class @class)
        {
            var retType = method.ReturnType.Visit(CTypePrinter);

            Write($"{retType} {GetMethodIdentifier(method)}(");

            Write(CTypePrinter.VisitParameters(method.Parameters));

            Write(")");
        }

        public virtual string GenerateClassObjectAlloc(Declaration decl)
        {
            var typeName = decl.Visit(CTypePrinter);
            return $"({typeName}*) calloc(1, sizeof({typeName}))";
        }

        public override bool VisitTypedefDecl(TypedefDecl typedef)
        {
            if (!VisitDeclaration(typedef))
                return false;

            PushBlock();

            var typeName = typedef.Type.Visit(CTypePrinter);
            WriteLine($"typedef {typeName} {typedef};");

            var newlineKind = NewLineKind.BeforeNextBlock;

            var declarations = typedef.Namespace.Declarations;
            var newIndex = declarations.FindIndex(d => d == typedef) + 1;
            if (newIndex < declarations.Count)
            {
                if (declarations[newIndex] is TypedefDecl)
                    newlineKind = NewLineKind.Never;
            }

            PopBlock(newlineKind);

            return true;
        }

        public override bool VisitFieldDecl(Field field)
        {
            return true;
        }
    }
}
