# -*- coding: utf-8 -*-
"""E2E: Single Sample Demo of Multilingual Speech to Speech Translation.ipynb

Automatically generated by Colaboratory.

Original file is located at
    https://colab.research.google.com/drive/1u58_FoERagMP3sHTKI8CW_GqhOyGhBVA

Authors,

*   Md. Nazmuddhoha Ansary (Ansary)
*   Syed Mobassir (Shabab)

# install required packages
"""

# Commented out IPython magic to ensure Python compatibility.
# %%capture
# 
# !pip install git+https://github.com/openai/whisper.git 
# !pip install pytube==12.1.0
# !pip install pydub==0.25.1
# 
# !pip install -q transformers==4.24.0
# 
# !pip install sentencepiece==0.1.97
# !pip install git+https://github.com/csebuetnlp/normalizer
# 
# # !pip install ffmpeg==1.4
# !pip install ffmpeg-python==0.2.0
# 
# #! pip install -U pip
# ! pip install TTS==0.8.0
# ! pip install bnnumerizer==0.0.2
# ! pip install bangla==0.0.2
# ! pip install gdown==4.5.1
# ! pip install bnunicodenormalizer==0.1.1
# ! pip install PyArabic==0.6.15
# #! pip install pydub
# ! pip install Unidecode==1.3.6
# #! pip install numpy
# !pip install pydload==1.0.9
#

"""# Cool Imports"""

import os
import pytube
from pydub import AudioSegment
from pydub.silence import split_on_silence
import re
import whisper
import librosa
import pandas as pd
from tqdm import tqdm
import soundfile as sf
import shutil
import ffmpeg
from IPython.display import Audio 

import torch
import bangla
from transformers import AutoModelForSeq2SeqLM, AutoTokenizer
from normalizer import normalize # pip install git+https://github.com/csebuetnlp/normalizer


model = whisper.load_model("large")

"""# Download single youtube video and convert that into .wav file"""

#https://towardsdatascience.com/speech-to-text-with-openais-whisper-53d5cea9005e

# Reading from Youtube video link
video = 'https://www.youtube.com/watch?v=GmQo2kTbZc0'
data = pytube.YouTube(video)
# Converting and downloading as 'MP4' file
audio = data.streams.get_audio_only()
audio.download()

!ffmpeg -i '/content/The Dangers of Showing Off  Shaykh Dr Yasir Qadhi.mp4' -ac 2 -f wav './en_ar_lecture.wav'

# #https://stackoverflow.com/questions/47420304/download-video-in-mp3-format-using-pytube

# from pytube.cli import on_progress

# PATH_SAVE = "./"


# yt = pytube.YouTube('https://www.youtube.com/watch?v=g8YHuKn5NpY', on_progress_callback=on_progress)
# #Download mp3
# audio_file = yt.streams.filter(only_audio=True).first().download(PATH_SAVE)
# base, ext = os.path.splitext(audio_file)
# new_file = base + '.wav'
# os.rename(audio_file, new_file)

# Audio('/content/abc.wav', autoplay=False)

"""# STEP-1 : Multilingual Speech to Text Conversion"""

def collapse_whitespace(text):
    # Regular expression matching whitespace:
    _whitespace_re = re.compile(r"\s+")
    return re.sub(_whitespace_re, " ", text)

def mlt_speech_to_text_convertor(path = "seerah_of_the_prophet.wav",silence_based_conversion = False):
    '''
        #modified from https://www.geeksforgeeks.org/python-speech-recognition-on-large-audio-files/

        -> a function that splits the audio file into chunks and applies multilingual speech recognition.
        -> supports both silence_based_conversion and fixed length chunking
    '''
    shutil.rmtree('audio_chunks', ignore_errors=True)
    # create a directory to store the audio chunks.
    try:
        os.mkdir('audio_chunks')
    except(FileExistsError):
        pass
  
    # move into the directory to
    # store the audio files.
    os.chdir('audio_chunks')
    

    if(not silence_based_conversion):
        full_audio, fs = librosa.load(path)
        TEXTS=[]
        MAX_AUDIO_LEN=30*fs
        for idx in range(0,full_audio.shape[0],MAX_AUDIO_LEN):
          audio=full_audio[idx:idx+MAX_AUDIO_LEN]
          sf.write(f"{idx}.wav",audio,fs)
          audio = whisper.load_audio(f"{idx}.wav")
          audio = whisper.pad_or_trim(audio)
          # make log-Mel spectrogram and move to the same device as the model
          mel = whisper.log_mel_spectrogram(audio).to(model.device)

          # decode the audio
          options = whisper.DecodingOptions()
          result = whisper.decode(model, mel, options)
      
          #take only arabic and english
          result=re.sub('[^\u0600-\u06FF a-zA-Z0-9,?!.\']',' ',result.text)
          result = collapse_whitespace(result)
          
          # print the recognized text
          TEXTS.append(result)
        os.chdir('..')
        shutil.rmtree('audio_chunks', ignore_errors=True)
        return TEXTS

    # open the audio file stored in
    # the local system as a wav file.
    lecture = AudioSegment.from_wav(path)

    # split track where silence is 1 seconds 
    # or more and get chunks
    chunks = split_on_silence(lecture,
        # must be silent for at least half seconds
        # or 1000 ms. adjust this value based on user
        # requirement. if the speaker stays silent for 
        # longer, increase this value. else, decrease it.
        min_silence_len = 500,
  
        # consider it silent if quieter than -16 dBFS
        # adjust this per requirement
        silence_thresh = -16
    )
  
  
    TEXTS=[]
    i = 0
    # process each chunk
    for chunk in chunks:
              
        # duration specified in milliseconds (default duration: 1000ms, default frame_rate: 11025).
        chunk_silent = AudioSegment.silent(duration = 500, frame_rate=11025)
  
        # add 0.5 sec silence to beginning and 
        # end of audio chunk. This is done so that
        # it doesn't seem abruptly sliced.
        audio_chunk = chunk_silent + chunk + chunk_silent
  
        # export audio chunk and save it in 
        # the current directory.

        # specify the bitrate to be 192 k
        audio_chunk.export("./chunk{0}.wav".format(i), bitrate ='192k', format ="wav")
  
        # the name of the newly created chunk
        filename = 'chunk'+str(i)+'.wav'


        # recognize the chunk
     
        audio = whisper.load_audio(filename)
        audio = whisper.pad_or_trim(audio)
        # make log-Mel spectrogram and move to the same device as the model
        mel = whisper.log_mel_spectrogram(audio).to(model.device)

        # decode the audio
        options = whisper.DecodingOptions()
        result = whisper.decode(model, mel, options)
        #take only arabic and english
        result=re.sub('[^\u0600-\u06FF a-zA-Z0-9,?!.\']',' ',result.text)
        result = collapse_whitespace(result)
        TEXTS.append(result)
        i+=1
    os.chdir('..')
    shutil.rmtree('audio_chunks', ignore_errors=True)
    return TEXTS

"""# With silence_based_conversion"""

# Commented out IPython magic to ensure Python compatibility.
# %%time
# 
# TEXTS = mlt_speech_to_text_convertor(path = "/content/en_ar_lecture.wav",silence_based_conversion = True)

TEXTS

"""# Without silence_based_conversion (fixed length chunking)"""

# Commented out IPython magic to ensure Python compatibility.
# %%time
# 
# TEXTS = mlt_speech_to_text_convertor(path = "/content/en_ar_lecture.wav",silence_based_conversion = False)
# 
# TEXTS

"""# STEP-2 : Multilingual text Translator (Translate only english and leave arabic as it is)"""

def tag_arabic_text(text,ar_pattern=u'[\u0600-\u06FF]+',english_only = False):
    # remove multiple spaces
    data=re.sub(' +', ' ',text)
    texts=[]
    if "।" in data:punct="।"
    elif "." in data:punct="."
    else:punct="\n"
    for text in data.split(punct):    
        # create start and end
        text="start"+text+"end"
        # tag text
        parts=re.split(ar_pattern, text)
        parts=[p for p in parts if p.strip()]
        parts=set(parts)
        for m in parts:
            if len(m.strip())>1:text=text.replace(m,f"</ar>{m}<ar>")
        # clean-up invalid combos
        text=text.replace("</ar>start",'')
        text=text.replace("end<ar>",'')
        texts.append(text)
    text=f"{punct}".join(texts)
    if(english_only):
        #https://stackoverflow.com/questions/55656429/replace-or-remove-html-tag-content-python-regex
        return re.sub(r'(?s)<ar>.*?</ar>', '', text)
    return text

mlt_text = ' '.join(TEXTS)
tag_texts = tag_arabic_text(mlt_text,ar_pattern=u'[\u0600-\u06FF]+',english_only = False)
tag_texts

torch_device = 'cuda' if torch.cuda.is_available() else 'cpu'
print(torch_device)
model = AutoModelForSeq2SeqLM.from_pretrained("csebuetnlp/banglat5_nmt_en_bn").to(torch_device)
tokenizer = AutoTokenizer.from_pretrained("csebuetnlp/banglat5_nmt_en_bn",use_fast=True)

def translate_en_bn(input_sentence):
    # thanks to banglanmt : https://huggingface.co/csebuetnlp/banglat5_nmt_en_bn/discussions
    input_ids = tokenizer(normalize(input_sentence), return_tensors="pt").input_ids
    input_ids = input_ids.to(torch_device)
    generated_tokens = model.generate(input_ids)
    decoded_tokens = tokenizer.batch_decode(generated_tokens)[0]
    decoded_tokens=decoded_tokens.replace("<pad>","").replace("</s>","")
    # sen=decoded_tokens.split()
    # words=[w for w in sen if w.strip()]
    # sen=" ".join(words)
    return decoded_tokens

def EN_AR_to_BN_AR_Translator(en_text,tag_text = False):
    '''
    translates multilingual english-arabic code mixed text into 
    multilingual bengali-arabic code mixed text
    ''' 
    if(tag_text):
        en_text = tag_arabic_text(en_text,english_only=False)
    
    sentenceEnders = re.compile('[.,!?]')
    sentences = sentenceEnders.split(en_text)
    main_list = []
    for i in range(len(sentences)):
        
        list_str = sentences[i].split('<ar>')
        if(len(list_str) == 1):
            main_list.append(list_str[0])
        else:
            for j in range(len(list_str)):
                if('</ar>' in list_str[j]):
                    list_str1 = list_str[j].split('</ar>')
                    main_list.append("<ar>"+list_str1[0]+"</ar>")
                    main_list.append(list_str1[1])
                else:
                    main_list.append(list_str[j])

    while(" " in main_list):
        main_list.remove(" ")
    for idx in range(len(main_list)):
        if('<ar>' not in main_list[idx] or '</ar>' not in main_list[idx]):
            
            output_sentence = []
            for word in main_list[idx].split():
                output_sentence.append(word)
     
            main_list[idx] = ' '.join(output_sentence)
            #numerizer
            main_list[idx] = bangla.convert_english_digit_to_bangla_digit(main_list[idx])
            # multilingual english-arabic to multilingual bengali-arabic
            try:
                if len(main_list[idx])>1:
                    main_list[idx]=translate_en_bn(main_list[idx])
                            
            except:
                print("failed -> ",main_list[idx])
    
    bn_mlt = " ".join(main_list)
    bn_mlt = re.sub(' ্ ','',bn_mlt)
    bn_mlt = re.sub("\\'","",bn_mlt)#replace \'
    bn_mlt = re.sub('<unk>','',bn_mlt)
    return bn_mlt
        
text=translate_en_bn("alhamdulillah for everything.")
print(text)

# Commented out IPython magic to ensure Python compatibility.
# %%time
# 
# bn_mlt = EN_AR_to_BN_AR_Translator(tag_texts,tag_text = False)
# bn_mlt

bn_mlt=bn_mlt.replace("</ar>",' ')
bn_mlt=bn_mlt.replace("<ar>",' ')
print(bn_mlt)

"""Step 3 : Multilingual Text to Speech Pipeline

# Arabic tacotron2 TTS
"""

!git clone https://github.com/nipponjo/tts-arabic-tacotron2

import sys
sys.path.append('/content/tts-arabic-tacotron2')
import gdown
import pandas as pd
from model.networks import Tacotron2Wave
from IPython.display import Audio
import soundfile as sf
import shutil
import bangla

url = "https://drive.google.com/drive/folders/196xZbqqxzsBQdKr1UKdh_wL_9qtJkt6u?usp=sharing"
gdown.download_folder(url=url, quiet=False, use_cookies=False)

shutil.copy('/content/arabic_tts/UNIVERSAL_V1/config.json', './tts-arabic-tacotron2/pretrained/hifigan-universal-v1')

shutil.copy('/content/arabic_tts/UNIVERSAL_V1/do_02500000', './tts-arabic-tacotron2/pretrained/hifigan-universal-v1')

shutil.copy('/content/arabic_tts/UNIVERSAL_V1/g_02500000', './tts-arabic-tacotron2/pretrained/hifigan-universal-v1')

shutil.copy('/content/arabic_tts/tacotron2_ar.pth', './tts-arabic-tacotron2/pretrained')

model_sd_path = './tts-arabic-tacotron2/pretrained/tacotron2_ar.pth'
# vocoder
vocoder_state_path = './tts-arabic-tacotron2/pretrained/hifigan-universal-v1/g_02500000'

vocoder_config_path = './tts-arabic-tacotron2/pretrained/hifigan-universal-v1/config.json'

# Commented out IPython magic to ensure Python compatibility.
# %%writefile /content/tts-arabic-tacotron2/utils/__init__.py
# import sys
# import yaml
# try:
#     from yaml import CLoader as Loader
# except ImportError:
#     from yaml import Loader
# 
# 
# class DictConfig(object):
#     """Creates a Config object from a dict 
#        such that object attributes correspond to dict keys.    
#     """
# 
#     def __init__(self, config_dict):
#         self.__dict__.update(config_dict)
# 
#     def __str__(self):
#         return '\n'.join(f"{key}: {val}" for key, val in self.__dict__.items())
# 
#     def __repr__(self):
#         return self.__str__()
# 
# 
# def get_custom_config(fname):
#     with open(fname, 'r') as stream:
#         config_dict = yaml.load(stream, Loader)
#     config = DictConfig(config_dict)
#     return config
# 
# 
# def get_basic_config():
#     return get_custom_config('./tts-arabic-tacotron2/configs/basic.yaml')
# 
# 
# def get_config(fname):
#     config = get_basic_config()
#     custom_config = get_custom_config(fname)
# 
#     config.__dict__.update(custom_config.__dict__)
#     return config
# 
# 
# def read_lines_from_file(path, encoding='utf-8'):
#     lines = []
#     with open(path, 'r', encoding=encoding) as f:
#         for line in f:
#             lines.append(line.strip())
#     return lines
# 
# 
# def progbar(iterable, length=30, symbol='='):
#     """Wrapper generator function for an iterable. 
#        Prints a progressbar when yielding an item. \\
#        Args:
#           iterable: an object supporting iteration
#           length: length of the progressbar
#     """
#     n = len(iterable)
#     for i, item in enumerate(iterable):
#         steps = length*(i+1) // n
#         sys.stdout.write('\r')
#         sys.stdout.write(f"[{symbol*steps:{length}}] {(100/n*(i+1)):.1f}%")
#         if i == (n-1):
#             sys.stdout.write('\n')
#         sys.stdout.flush()
#         yield item

ar_model = Tacotron2Wave(model_sd_path = model_sd_path,vocoder_sd = vocoder_state_path, vocoder_config = vocoder_config_path)
#ar_model = ar_model.cuda()
ar_model = ar_model.to(torch_device)

"""# Bangla TTS"""

try:
  from TTS.utils.synthesizer import Synthesizer
except:
  print("coundn't import TTS synthesizer,trying again!")
#from TTS.utils.synthesizer import Synthesizer

import TTS

TTS.__version__

"""Removing annoying print statement during inference"""

# from https://github.com/coqui-ai/TTS/blob/dev/TTS/utils/synthesizer.py
import time
from typing import List

import numpy as np
import pysbd
import torch

from TTS.config import load_config
from TTS.tts.models import setup_model as setup_tts_model

# pylint: disable=unused-wildcard-import
# pylint: disable=wildcard-import
from TTS.tts.utils.synthesis import synthesis, transfer_voice, trim_silence
from TTS.utils.audio import AudioProcessor
from TTS.vocoder.models import setup_model as setup_vocoder_model
from TTS.vocoder.utils.generic_utils import interpolate_vocoder_input


class Synthesizer(object):
    def __init__(
        self,
        tts_checkpoint: str,
        tts_config_path: str,
        tts_speakers_file: str = "",
        tts_languages_file: str = "",
        vocoder_checkpoint: str = "",
        vocoder_config: str = "",
        encoder_checkpoint: str = "",
        encoder_config: str = "",
        use_cuda: bool = False,
    ) -> None:
        """General 🐸 TTS interface for inference. It takes a tts and a vocoder
        model and synthesize speech from the provided text.
        The text is divided into a list of sentences using `pysbd` and synthesize
        speech on each sentence separately.
        If you have certain special characters in your text, you need to handle
        them before providing the text to Synthesizer.
        TODO: set the segmenter based on the source language
        Args:
            tts_checkpoint (str): path to the tts model file.
            tts_config_path (str): path to the tts config file.
            vocoder_checkpoint (str, optional): path to the vocoder model file. Defaults to None.
            vocoder_config (str, optional): path to the vocoder config file. Defaults to None.
            encoder_checkpoint (str, optional): path to the speaker encoder model file. Defaults to `""`,
            encoder_config (str, optional): path to the speaker encoder config file. Defaults to `""`,
            use_cuda (bool, optional): enable/disable cuda. Defaults to False.
        """
        self.tts_checkpoint = tts_checkpoint
        self.tts_config_path = tts_config_path
        self.tts_speakers_file = tts_speakers_file
        self.tts_languages_file = tts_languages_file
        self.vocoder_checkpoint = vocoder_checkpoint
        self.vocoder_config = vocoder_config
        self.encoder_checkpoint = encoder_checkpoint
        self.encoder_config = encoder_config
        self.use_cuda = use_cuda

        self.tts_model = None
        self.vocoder_model = None
        self.speaker_manager = None
        self.num_speakers = 0
        self.tts_speakers = {}
        self.language_manager = None
        self.num_languages = 0
        self.tts_languages = {}
        self.d_vector_dim = 0
        self.seg = self._get_segmenter("en")
        self.use_cuda = use_cuda

        if self.use_cuda:
            assert torch.cuda.is_available(), "CUDA is not availabe on this machine."
        self._load_tts(tts_checkpoint, tts_config_path, use_cuda)
        self.output_sample_rate = self.tts_config.audio["sample_rate"]
        if vocoder_checkpoint:
            self._load_vocoder(vocoder_checkpoint, vocoder_config, use_cuda)
            self.output_sample_rate = self.vocoder_config.audio["sample_rate"]

    @staticmethod
    def _get_segmenter(lang: str):
        """get the sentence segmenter for the given language.
        Args:
            lang (str): target language code.
        Returns:
            [type]: [description]
        """
        return pysbd.Segmenter(language=lang, clean=True)

    def _load_tts(self, tts_checkpoint: str, tts_config_path: str, use_cuda: bool) -> None:
        """Load the TTS model.
        1. Load the model config.
        2. Init the model from the config.
        3. Load the model weights.
        4. Move the model to the GPU if CUDA is enabled.
        5. Init the speaker manager in the model.
        Args:
            tts_checkpoint (str): path to the model checkpoint.
            tts_config_path (str): path to the model config file.
            use_cuda (bool): enable/disable CUDA use.
        """
        # pylint: disable=global-statement
        self.tts_config = load_config(tts_config_path)
        if self.tts_config["use_phonemes"] and self.tts_config["phonemizer"] is None:
            raise ValueError("Phonemizer is not defined in the TTS config.")

        self.tts_model = setup_tts_model(config=self.tts_config)

        if not self.encoder_checkpoint:
            self._set_speaker_encoder_paths_from_tts_config()

        self.tts_model.load_checkpoint(self.tts_config, tts_checkpoint, eval=True)
        if use_cuda:
            self.tts_model.cuda()

        if self.encoder_checkpoint and hasattr(self.tts_model, "speaker_manager"):
            self.tts_model.speaker_manager.init_encoder(self.encoder_checkpoint, self.encoder_config, use_cuda)

    def _set_speaker_encoder_paths_from_tts_config(self):
        """Set the encoder paths from the tts model config for models with speaker encoders."""
        if hasattr(self.tts_config, "model_args") and hasattr(
            self.tts_config.model_args, "speaker_encoder_config_path"
        ):
            self.encoder_checkpoint = self.tts_config.model_args.speaker_encoder_model_path
            self.encoder_config = self.tts_config.model_args.speaker_encoder_config_path

    def _load_vocoder(self, model_file: str, model_config: str, use_cuda: bool) -> None:
        """Load the vocoder model.
        1. Load the vocoder config.
        2. Init the AudioProcessor for the vocoder.
        3. Init the vocoder model from the config.
        4. Move the model to the GPU if CUDA is enabled.
        Args:
            model_file (str): path to the model checkpoint.
            model_config (str): path to the model config file.
            use_cuda (bool): enable/disable CUDA use.
        """
        self.vocoder_config = load_config(model_config)
        self.vocoder_ap = AudioProcessor(verbose=False, **self.vocoder_config.audio)
        self.vocoder_model = setup_vocoder_model(self.vocoder_config)
        self.vocoder_model.load_checkpoint(self.vocoder_config, model_file, eval=True)
        if use_cuda:
            self.vocoder_model.cuda()

    def split_into_sentences(self, text) -> List[str]:
        """Split give text into sentences.
        Args:
            text (str): input text in string format.
        Returns:
            List[str]: list of sentences.
        """
        return self.seg.segment(text)

    def save_wav(self, wav: List[int], path: str) -> None:
        """Save the waveform as a file.
        Args:
            wav (List[int]): waveform as a list of values.
            path (str): output path to save the waveform.
        """
        wav = np.array(wav)
        self.tts_model.ap.save_wav(wav, path, self.output_sample_rate)

    def tts(
        self,
        text: str = "",
        speaker_name: str = "",
        language_name: str = "",
        speaker_wav=None,
        style_wav=None,
        style_text=None,
        reference_wav=None,
        reference_speaker_name=None,
    ) -> List[int]:
        """🐸 TTS magic. Run all the models and generate speech.
        Args:
            text (str): input text.
            speaker_name (str, optional): spekaer id for multi-speaker models. Defaults to "".
            language_name (str, optional): language id for multi-language models. Defaults to "".
            speaker_wav (Union[str, List[str]], optional): path to the speaker wav. Defaults to None.
            style_wav ([type], optional): style waveform for GST. Defaults to None.
            style_text ([type], optional): transcription of style_wav for Capacitron. Defaults to None.
            reference_wav ([type], optional): reference waveform for voice conversion. Defaults to None.
            reference_speaker_name ([type], optional): spekaer id of reference waveform. Defaults to None.
        Returns:
            List[int]: [description]
        """
        start_time = time.time()
        wavs = []

        if not text and not reference_wav:
            raise ValueError(
                "You need to define either `text` (for sythesis) or a `reference_wav` (for voice conversion) to use the Coqui TTS API."
            )

        if text:
            sens = self.split_into_sentences(text)
            # print(" > Text splitted to sentences.")
            # print(sens)

        # handle multi-speaker
        speaker_embedding = None
        speaker_id = None
        if self.tts_speakers_file or hasattr(self.tts_model.speaker_manager, "name_to_id"):
            if speaker_name and isinstance(speaker_name, str):
                if self.tts_config.use_d_vector_file:
                    # get the average speaker embedding from the saved d_vectors.
                    speaker_embedding = self.tts_model.speaker_manager.get_mean_embedding(
                        speaker_name, num_samples=None, randomize=False
                    )
                    speaker_embedding = np.array(speaker_embedding)[None, :]  # [1 x embedding_dim]
                else:
                    # get speaker idx from the speaker name
                    speaker_id = self.tts_model.speaker_manager.name_to_id[speaker_name]

            elif not speaker_name and not speaker_wav:
                raise ValueError(
                    " [!] Look like you use a multi-speaker model. "
                    "You need to define either a `speaker_name` or a `speaker_wav` to use a multi-speaker model."
                )
            else:
                speaker_embedding = None
        else:
            if speaker_name:
                raise ValueError(
                    f" [!] Missing speakers.json file path for selecting speaker {speaker_name}."
                    "Define path for speaker.json if it is a multi-speaker model or remove defined speaker idx. "
                )

        # handle multi-lingaul
        language_id = None
        if self.tts_languages_file or (
            hasattr(self.tts_model, "language_manager") and self.tts_model.language_manager is not None
        ):
            if language_name and isinstance(language_name, str):
                language_id = self.tts_model.language_manager.name_to_id[language_name]

            elif not language_name:
                raise ValueError(
                    " [!] Look like you use a multi-lingual model. "
                    "You need to define either a `language_name` or a `style_wav` to use a multi-lingual model."
                )

            else:
                raise ValueError(
                    f" [!] Missing language_ids.json file path for selecting language {language_name}."
                    "Define path for language_ids.json if it is a multi-lingual model or remove defined language idx. "
                )

        # compute a new d_vector from the given clip.
        if speaker_wav is not None:
            speaker_embedding = self.tts_model.speaker_manager.compute_embedding_from_clip(speaker_wav)

        use_gl = self.vocoder_model is None

        if not reference_wav:
            for sen in sens:
                # synthesize voice
                outputs = synthesis(
                    model=self.tts_model,
                    text=sen,
                    CONFIG=self.tts_config,
                    use_cuda=self.use_cuda,
                    speaker_id=speaker_id,
                    style_wav=style_wav,
                    style_text=style_text,
                    use_griffin_lim=use_gl,
                    d_vector=speaker_embedding,
                    language_id=language_id,
                )
                waveform = outputs["wav"]
                mel_postnet_spec = outputs["outputs"]["model_outputs"][0].detach().cpu().numpy()
                if not use_gl:
                    # denormalize tts output based on tts audio config
                    mel_postnet_spec = self.tts_model.ap.denormalize(mel_postnet_spec.T).T
                    device_type = "cuda" if self.use_cuda else "cpu"
                    # renormalize spectrogram based on vocoder config
                    vocoder_input = self.vocoder_ap.normalize(mel_postnet_spec.T)
                    # compute scale factor for possible sample rate mismatch
                    scale_factor = [
                        1,
                        self.vocoder_config["audio"]["sample_rate"] / self.tts_model.ap.sample_rate,
                    ]
                    if scale_factor[1] != 1:
                        print(" > interpolating tts model output.")
                        vocoder_input = interpolate_vocoder_input(scale_factor, vocoder_input)
                    else:
                        vocoder_input = torch.tensor(vocoder_input).unsqueeze(0)  # pylint: disable=not-callable
                    # run vocoder model
                    # [1, T, C]
                    waveform = self.vocoder_model.inference(vocoder_input.to(device_type))
                if self.use_cuda and not use_gl:
                    waveform = waveform.cpu()
                if not use_gl:
                    waveform = waveform.numpy()
                waveform = waveform.squeeze()

                # trim silence
                if "do_trim_silence" in self.tts_config.audio and self.tts_config.audio["do_trim_silence"]:
                    waveform = trim_silence(waveform, self.tts_model.ap)

                wavs += list(waveform)
                wavs += [0] * 10000
        else:
            # get the speaker embedding or speaker id for the reference wav file
            reference_speaker_embedding = None
            reference_speaker_id = None
            if self.tts_speakers_file or hasattr(self.tts_model.speaker_manager, "name_to_id"):
                if reference_speaker_name and isinstance(reference_speaker_name, str):
                    if self.tts_config.use_d_vector_file:
                        # get the speaker embedding from the saved d_vectors.
                        reference_speaker_embedding = self.tts_model.speaker_manager.get_embeddings_by_name(
                            reference_speaker_name
                        )[0]
                        reference_speaker_embedding = np.array(reference_speaker_embedding)[
                            None, :
                        ]  # [1 x embedding_dim]
                    else:
                        # get speaker idx from the speaker name
                        reference_speaker_id = self.tts_model.speaker_manager.name_to_id[reference_speaker_name]
                else:
                    reference_speaker_embedding = self.tts_model.speaker_manager.compute_embedding_from_clip(
                        reference_wav
                    )
            outputs = transfer_voice(
                model=self.tts_model,
                CONFIG=self.tts_config,
                use_cuda=self.use_cuda,
                reference_wav=reference_wav,
                speaker_id=speaker_id,
                d_vector=speaker_embedding,
                use_griffin_lim=use_gl,
                reference_speaker_id=reference_speaker_id,
                reference_d_vector=reference_speaker_embedding,
            )
            waveform = outputs
            if not use_gl:
                mel_postnet_spec = outputs[0].detach().cpu().numpy()
                # denormalize tts output based on tts audio config
                mel_postnet_spec = self.tts_model.ap.denormalize(mel_postnet_spec.T).T
                device_type = "cuda" if self.use_cuda else "cpu"
                # renormalize spectrogram based on vocoder config
                vocoder_input = self.vocoder_ap.normalize(mel_postnet_spec.T)
                # compute scale factor for possible sample rate mismatch
                scale_factor = [
                    1,
                    self.vocoder_config["audio"]["sample_rate"] / self.tts_model.ap.sample_rate,
                ]
                if scale_factor[1] != 1:
                    print(" > interpolating tts model output.")
                    vocoder_input = interpolate_vocoder_input(scale_factor, vocoder_input)
                else:
                    vocoder_input = torch.tensor(vocoder_input).unsqueeze(0)  # pylint: disable=not-callable
                # run vocoder model
                # [1, T, C]
                waveform = self.vocoder_model.inference(vocoder_input.to(device_type))
            if self.use_cuda:
                waveform = waveform.cpu()
            if not use_gl:
                waveform = waveform.numpy()
            wavs = waveform.squeeze()

        # compute stats
        process_time = time.time() - start_time
        audio_time = len(wavs) / self.tts_config.audio["sample_rate"]
        # print(f" > Processing time: {process_time}")
        # print(f" > Real-time factor: {process_time / audio_time}")
        return wavs



# from https://github.com/coqui-ai/TTS/blob/dev/TTS/tts/utils/text/tokenizer.py

from typing import Callable, Dict, List, Union

from TTS.tts.utils.text import cleaners
from TTS.tts.utils.text.characters import Graphemes, IPAPhonemes
from TTS.tts.utils.text.phonemizers import DEF_LANG_TO_PHONEMIZER, get_phonemizer_by_name
from TTS.utils.generic_utils import get_import_path, import_class


class TTSTokenizer:
    """🐸TTS tokenizer to convert input characters to token IDs and back.
    Token IDs for OOV chars are discarded but those are stored in `self.not_found_characters` for later.
    Args:
        use_phonemes (bool):
            Whether to use phonemes instead of characters. Defaults to False.
        characters (Characters):
            A Characters object to use for character-to-ID and ID-to-character mappings.
        text_cleaner (callable):
            A function to pre-process the text before tokenization and phonemization. Defaults to None.
        phonemizer (Phonemizer):
            A phonemizer object or a dict that maps language codes to phonemizer objects. Defaults to None.
    Example:
        >>> from TTS.tts.utils.text.tokenizer import TTSTokenizer
        >>> tokenizer = TTSTokenizer(use_phonemes=False, characters=Graphemes())
        >>> text = "Hello world!"
        >>> ids = tokenizer.text_to_ids(text)
        >>> text_hat = tokenizer.ids_to_text(ids)
        >>> assert text == text_hat
    """

    def __init__(
        self,
        use_phonemes=False,
        text_cleaner: Callable = None,
        characters: "BaseCharacters" = None,
        phonemizer: Union["Phonemizer", Dict] = None,
        add_blank: bool = False,
        use_eos_bos=False,
    ):
        self.text_cleaner = text_cleaner
        self.use_phonemes = use_phonemes
        self.add_blank = add_blank
        self.use_eos_bos = use_eos_bos
        self.characters = characters
        self.not_found_characters = []
        self.phonemizer = phonemizer

    @property
    def characters(self):
        return self._characters

    @characters.setter
    def characters(self, new_characters):
        self._characters = new_characters
        self.pad_id = self.characters.char_to_id(self.characters.pad) if self.characters.pad else None
        self.blank_id = self.characters.char_to_id(self.characters.blank) if self.characters.blank else None

    def encode(self, text: str) -> List[int]:
        """Encodes a string of text as a sequence of IDs."""
        token_ids = []
        for char in text:
            try:
                idx = self.characters.char_to_id(char)
                token_ids.append(idx)
            except KeyError:
                # discard but store not found characters
                if char not in self.not_found_characters:
                    self.not_found_characters.append(char)
                    # print(text)
                    # print(f" [!] Character {repr(char)} not found in the vocabulary. Discarding it.")
        return token_ids

    def decode(self, token_ids: List[int]) -> str:
        """Decodes a sequence of IDs to a string of text."""
        text = ""
        for token_id in token_ids:
            text += self.characters.id_to_char(token_id)
        return text

    def text_to_ids(self, text: str, language: str = None) -> List[int]:  # pylint: disable=unused-argument
        """Converts a string of text to a sequence of token IDs.
        Args:
            text(str):
                The text to convert to token IDs.
            language(str):
                The language code of the text. Defaults to None.
        TODO:
            - Add support for language-specific processing.
        1. Text normalizatin
        2. Phonemization (if use_phonemes is True)
        3. Add blank char between characters
        4. Add BOS and EOS characters
        5. Text to token IDs
        """
        # TODO: text cleaner should pick the right routine based on the language
        if self.text_cleaner is not None:
            text = self.text_cleaner(text)
        if self.use_phonemes:
            text = self.phonemizer.phonemize(text, separator="")
        if self.add_blank:
            text = self.intersperse_blank_char(text, True)
        if self.use_eos_bos:
            text = self.pad_with_bos_eos(text)
        return self.encode(text)

    def ids_to_text(self, id_sequence: List[int]) -> str:
        """Converts a sequence of token IDs to a string of text."""
        return self.decode(id_sequence)

    def pad_with_bos_eos(self, char_sequence: List[str]):
        """Pads a sequence with the special BOS and EOS characters."""
        return [self.characters.bos] + list(char_sequence) + [self.characters.eos]

    def intersperse_blank_char(self, char_sequence: List[str], use_blank_char: bool = False):
        """Intersperses the blank character between characters in a sequence.
        Use the ```blank``` character if defined else use the ```pad``` character.
        """
        char_to_use = self.characters.blank if use_blank_char else self.characters.pad
        result = [char_to_use] * (len(char_sequence) * 2 + 1)
        result[1::2] = char_sequence
        return result

    def print_logs(self, level: int = 0):
        indent = "\t" * level
        print(f"{indent}| > add_blank: {self.add_blank}")
        print(f"{indent}| > use_eos_bos: {self.use_eos_bos}")
        print(f"{indent}| > use_phonemes: {self.use_phonemes}")
        if self.use_phonemes:
            print(f"{indent}| > phonemizer:")
            self.phonemizer.print_logs(level + 1)
        if len(self.not_found_characters) > 0:
            print(f"{indent}| > {len(self.not_found_characters)} not found characters:")
            for char in self.not_found_characters:
                print(f"{indent}| > {char}")

    @staticmethod
    def init_from_config(config: "Coqpit", characters: "BaseCharacters" = None):
        """Init Tokenizer object from config
        Args:
            config (Coqpit): Coqpit model config.
            characters (BaseCharacters): Defines the model character set. If not set, use the default options based on
                the config values. Defaults to None.
        """
        # init cleaners
        text_cleaner = None
        if isinstance(config.text_cleaner, (str, list)):
            text_cleaner = getattr(cleaners, config.text_cleaner)

        # init characters
        if characters is None:
            # set characters based on defined characters class
            if config.characters and config.characters.characters_class:
                CharactersClass = import_class(config.characters.characters_class)
                characters, new_config = CharactersClass.init_from_config(config)
            # set characters based on config
            else:
                if config.use_phonemes:
                    # init phoneme set
                    characters, new_config = IPAPhonemes().init_from_config(config)
                else:
                    # init character set
                    characters, new_config = Graphemes().init_from_config(config)

        else:
            characters, new_config = characters.init_from_config(config)

        # set characters class
        new_config.characters.characters_class = get_import_path(characters)

        # init phonemizer
        phonemizer = None
        if config.use_phonemes:
            phonemizer_kwargs = {"language": config.phoneme_language}

            if "phonemizer" in config and config.phonemizer:
                phonemizer = get_phonemizer_by_name(config.phonemizer, **phonemizer_kwargs)
            else:
                try:
                    phonemizer = get_phonemizer_by_name(
                        DEF_LANG_TO_PHONEMIZER[config.phoneme_language], **phonemizer_kwargs
                    )
                    new_config.phonemizer = phonemizer.name()
                except KeyError as e:
                    raise ValueError(
                        f"""No phonemizer found for language {config.phoneme_language}.
                        You may need to install a third party library for this language."""
                    ) from e

        return (
            TTSTokenizer(
                config.use_phonemes, text_cleaner, characters, phonemizer, config.add_blank, config.enable_eos_bos_chars
            ),
            new_config,
        )

"""# model selection"""

# link -> hhttps://drive.google.com/drive/folders/1IMCiQpyYBqu98dlRMSINjFNc34fI6zhs?usp=sharing
url = "https://drive.google.com/drive/folders/1IMCiQpyYBqu98dlRMSINjFNc34fI6zhs?usp=sharing"
gdown.download_folder(url=url, quiet=True, use_cookies=False)   

male = True
if(male):
  # test_ckpt = '/content/bangla_tts/bn_glow_tts/male/checkpoint_328000.pth'
  # test_config = '/content/bangla_tts/bn_glow_tts/male/config.json'

  test_ckpt = '/content/bangla_tts/bn_vits/male/best_model_2360.pth'
  test_config = '/content/bangla_tts/bn_vits/male/config.json'

else:
  test_ckpt = '/content/bangla_tts/bn_glow_tts/female/checkpoint_180000.pth'
  test_config = '/content/bangla_tts/bn_glow_tts/female/config.json'

bn_model=Synthesizer(test_ckpt,test_config)

"""# Mlt large text to audio"""

import re
import torchaudio.functional as F
import torchaudio
from bnnumerizer import numerize
import gc
from bnunicodenormalizer import Normalizer 
from pydub import AudioSegment
from pyarabic.araby import strip_diacritics

# initialize
bnorm=Normalizer()

# Create empty audio file of half second duration (purpose -> post processing)

audio = AudioSegment.silent(duration=500)
sound = audio.set_frame_rate(audio.frame_rate*2)
sound.export("./empty.wav", format="wav")

#loading empty audio file of  1 second to append before and after each arabic chunk for increasing mlt reading rhythm.

empty_audio, rate_of_sample = torchaudio.load('/content/empty.wav')
empty_audio = empty_audio.flatten()

def normalize(sen):
    _words = [bnorm(word)['normalized']  for word in sen.split()]
    return " ".join([word for word in _words if word is not None]) 

class BigTextToAudio(object):
    
    def __init__(self,
                 ar_model,
                 bn_model,
                 ar_sample_rate=22050,
                 bn_sample_rate=22050,
                 out_sample_rate=22050,
        
                 attribution_dict={"সাঃ":"সাল্লাল্লাহু আলাইহি ওয়া সাল্লাম",    
                                   "স.":"সাল্লাল্লাহু আলাইহি ওয়া সাল্লাম",                
                                  "আঃ":"আলাইহিস সালাম",
                                  "রাঃ":"রাদিআল্লাহু আনহু",
                                  "রহঃ":"রহমাতুল্লাহি আলাইহি",
                                  "রহিঃ":"রহিমাহুল্লাহ",
                                  "হাফিঃ":"হাফিযাহুল্লাহ",
                                  # "বায়ান":"বাইআন",
                                  "দাঃবাঃ":"দামাত বারাকাতুহুম,দামাত বারাকাতুল্লাহ",
                                  #"আয়াত" : "আইআত",#আইআত
                                  # "ওয়া" : "ওআ",
                                  # "ওয়াসাল্লাম"  : "ওআসাল্লাম",
                                  # "কেন"  : "কেনো",
                                  # "কোন" : "কোনো",
                                  # "বল"   : "বলো",
                                  # "চল"   : "চলো",
                                  # "কর"   : "করো",
                                  # "রাখ"   : "রাখো",
                                  "’"     :  "",
                                  "‘"     : "",
                                  # "য়"     : "অ",
                                  # "সম্প্রদায়" : "সম্প্রদাই",
                                  # "রয়েছে"   : "রইছে",
                                  # "রয়েছ"    : "রইছ",
                                   "/"   : " বাই ",
                                  },
                resample_params={"lowpass_filter_width": 64,
                                "rolloff": 0.9475937167399596,
                                "resampling_method": "kaiser_window",
                                "beta": 14.769656459379492}
                ):
        '''
            Instantiates a Big Text to Audio conversion object for bangla and arabic
            args:
                ar_model : arabic tts model
                bn_model : bangla tts model
                ar_sample_rate : arabic audio sample rate [optional] default: 22050
                bn_sample_rate : bangla audio sample rate [optional] default: 22050
                out_sample_rate : audio sample rate [optional] default: 22050
                attribution_dict : a dict for attribution expansion [optional]
                resample_params : audio resampling parameters [optional]
            resources:
                # Main class: modified from https://github.com/snakers4/silero-models/pull/174
                # Audio converter:https://www.kaggle.com/code/shahruk10/inference-notebook-wav2vec2
        '''
        self.ar_model = ar_model
        self.bn_model = bn_model

        self.attribution_dict=attribution_dict

        self.ar_sample_rate=ar_sample_rate
        self.bn_sample_rate=bn_sample_rate
        self.sample_rate=out_sample_rate  
        self.resample_params=resample_params
        
    # public
    def ar_tts(self,text):
        '''
            args: 
                text: arabic text (string)
            returns:
                audio as torch tensor
        '''
        text = strip_diacritics(text)
        try:
          audio = self.ar_model.tts(text)
          audio = torch.cat([empty_audio,audio], axis=0) #start empty
          audio = torch.cat([audio,empty_audio], axis=0) #end empty
          audio = audio 
        except:
            print("failed ar =>",text,"end")
            audio = empty_audio 
                  
        
        return audio
    # public
    def bn_tts(self,text):
        '''
            args: 
                text   : bangla text (string)
            returns:
                audio as torch tensor
        '''


        return torch.as_tensor(self.bn_model.tts(text))
    
    # public
    def expand_full_attribution(self,text):
        for word in self.attribution_dict:
            if word in text:
                text = text.replace(word, normalize(self.attribution_dict[word]))
        return text
    

    def collapse_whitespace(self,text):
        # Regular expression matching whitespace:
        _whitespace_re = re.compile(r"\s+")
        return re.sub(_whitespace_re, " ", text)

    # public
    def tag_text(self,text):
        '''
            * tags arabic text with <ar>text</ar>
            * tags bangla text with <bn>text</bn>
        '''
        # remove multiple spaces
        text=re.sub(' +', ' ',text)
        # create start and end
        text="start"+text+"end"
        # tag text
        parts=re.split(u'[\u0600-\u06FF]+', text)
        # remove non chars
        parts=[p for p in parts if p.strip()]
        # unique parts
        parts=set(parts)
        # tag the text
        for m in parts:
            if len(m.strip())>1:text=text.replace(m,f"</ar><SPLIT><bn>{m}</bn><SPLIT><ar>")
        # clean-tags
        text=text.replace("</ar><SPLIT><bn>start",'<bn>')
        text=text.replace("end</bn><SPLIT><ar>",'</bn>')
        return text

    def process_text(self,text):
        '''
        process multilingual text for suitable MLT TTS format
            * expand attributions
            * numerize text
            * tag sections of the text
            * sequentially list text blocks
            * Split based on sentence ending Characters

        '''
        
        # english numbers to bangla conversion
        res = re.search('[0-9]', text)
        if res is not None:
          text = bangla.convert_english_digit_to_bangla_digit(text)
        
        #replace ':' in between two bangla numbers with ' এর '
        pattern=r"[০, ১, ২, ৩, ৪, ৫, ৬, ৭, ৮, ৯]:[০, ১, ২, ৩, ৪, ৫, ৬, ৭, ৮, ৯]"
        matches=re.findall(pattern,text)
        for m in matches:
            r=m.replace(":"," এর ")
            text=text.replace(m,r)

        # numerize text
        try:
          text=numerize(text)
        except:
          print("couldn't numerize bengali.")
        # tag sections
        # text=self.tag_text(text)

        # text="।".join([self.tag_text(line) for line in text.split("।")])

        if "।" in text:punct="।"
        elif "." in text:punct="."
        else:punct="\n"

        text=punct.join([self.tag_text(line) for line in text.split(punct)])

        # text blocks
        blocks=text.split("<SPLIT>")
        blocks=[b for b in blocks if b.strip()]
        # create tuple of (lang,text)
        data=[]
        for block in blocks:
            lang=None
            if "<bn>" in block:
                block=block.replace("<bn>",'').replace("</bn>",'')
                lang="bn"
            elif "<ar>" in block:
                block=block.replace("<ar>",'').replace("</ar>",'')
                lang="ar"
            
            # Split based on sentence ending Characters

            if lang == "bn":
              bn_text = block.strip()

              sentenceEnders = re.compile('[।,!?]')
              sentences = sentenceEnders.split(str(bn_text))

              for i in range(len(sentences)):
                  res = re.sub('\n','',sentences[i])
                  res = normalize(res)
                  # expand attributes
                  res=self.expand_full_attribution(res)

                  res = self.collapse_whitespace(res)
                  res += '।'
                  data.append((lang,res))

            elif lang == "ar":
                ar_text = block.strip()
                ar_text = re.sub("؟", "?", ar_text) # replace any ؟ with ?

                sentenceEnders = re.compile('[.,!?]')
                sentences = sentenceEnders.split(str(ar_text))

                for i in range(len(sentences)):
                    res = re.sub('\n','',sentences[i])
                    res = self.collapse_whitespace(res)
                    data.append((lang,res))
                    
        return data
    
    def resample_audio(self,audio,sr):
        '''
            resample audio with sample rate
            args:
                audio : torch.tensor audio
                sr: audi sample rate
        '''
        if sr==self.sample_rate:
            return audio
        else:
            return F.resample(audio,sr,self.sample_rate,**self.resample_params)
        
    
    def get_audio(self,data):
        '''
            creates audio from given data 
                * data=List[Tuples(lang,text)]
        '''
        audio_list = []
        for block in data:
            lang,text=block
            if lang=="bn":
                audio=self.bn_tts(text)
                sr=self.bn_sample_rate
            else:
                audio=self.ar_tts(text)
                sr=self.ar_sample_rate
            
            if self.resample_audio_to_out_sample_rate:
                audio=self.resample_audio(audio,sr)
                
            audio_list.append(audio)
  
        audio = torch.cat([k for k in audio_list])
        return audio
    
    # call
    def __call__(self,text,resample_audio_to_out_sample_rate=True):
        '''
            args: 
                text   : bangla text (string)
                resample_audio_to_out_sample_rate: for different sample rate in different models, resample the output audio 
                                                   in uniform sample rate 
                                                   * default:True
            returns:
                audio as numpy data
        '''
        self.resample_audio_to_out_sample_rate=resample_audio_to_out_sample_rate
        data=self.process_text(text)
        audio=self.get_audio(data)
        return audio.detach().cpu().numpy()

mytext = bn_mlt

MLT_TTS=BigTextToAudio(ar_model=ar_model,
                   bn_model=bn_model)

data=MLT_TTS.process_text(mytext)
data

"""# Cross Lingual Text to Speech"""

# Commented out IPython magic to ensure Python compatibility.
# %%time
# 
# audio=MLT_TTS(mytext,resample_audio_to_out_sample_rate=False)
# sf.write('e2e_mlt_stt.wav', audio, 22050)

Audio(audio, rate=MLT_TTS.sample_rate)

