// +build ignore

package main

import (
	"encoding/base64"
	"fmt"
	"io/ioutil"

	chrome "github.com/mkenney/go-chrome/tot"
	"github.com/mkenney/go-chrome/tot/emulation"
	"github.com/mkenney/go-chrome/tot/page"
	"github.com/mkenney/go-chrome/tot/socket"
)

func main() {
	var err error

	// Create a chrome instance
	browser := chrome.New(
		&chrome.Flags{
			"addr":                     "0.0.0.0",
			"remote-debugging-address": "0.0.0.0",
			"remote-debugging-port":    9222,
		}, "", "", "", "",
	)

	// Open a tab and navigate to the URL you want to screenshot.
	tab, err := browser.NewTab("https://www.google.com")
	if nil != err {
		panic(err)
	}

	// Enable Page events for this tab.
	if enableResult := <-tab.Page().Enable(); nil != enableResult.Err {
		panic(enableResult.Err)
	}

	// Create a channel to receive the screenshot data generated by the
	// event handler.
	results := make(chan *page.CaptureScreenshotResult)

	// Create an event handler that executes when the page load event is
	// received.
	loadEventHandler := socket.NewEventHandler(
		"Page.loadEventFired",

		// This function will generate a screenshot and write the data
		// to the results channel.
		func(response *socket.Response) {

			// Set the device emulation parameters.
			overrideResult := <-tab.Emulation().SetDeviceMetricsOverride(
				&emulation.SetDeviceMetricsOverrideParams{
					Width:  1440,
					Height: 1440,
					ScreenOrientation: &emulation.ScreenOrientation{
						Type:  emulation.OrientationType.PortraitPrimary,
						Angle: 90,
					},
				},
			)
			if nil != overrideResult.Err {
				panic(overrideResult.Err)
			}

			// Capture a screenshot of the current state of the current
			// page.
			screenshotResult := <-tab.Page().CaptureScreenshot(
				&page.CaptureScreenshotParams{
					Format:  page.Format.Jpeg,
					Quality: 50,
				},
			)
			if nil != screenshotResult.Err {
				panic(screenshotResult.Err)
			}

			results <- screenshotResult
		},
	)

	// Add the handler to the stack for this tab. Multiple handlers can
	// listen to the same event.
	tab.AddEventHandler(loadEventHandler)

	// Wait for the handler to fire
	result := <-results

	// Decode the base64 encoded image data
	data, err := base64.StdEncoding.DecodeString(result.Data)
	if nil != err {
		panic(err)
	}

	// Write the generated image to a file
	err = ioutil.WriteFile("/tmp/test/example.jpg", data, 0644)
	if nil != err {
		panic(err)
	}

	fmt.Println("Finished rendering example.jpg")
}
