# fuzzbunny

`fuzzbunny` is a small (1k), fast & memory efficient fuzzy string searching/matching/highlighting library.
It works equally well in a browser environment or Node.js.

![Build Status](https://github.com/mixpanel/fuzzbunny/actions/workflows/tests.yml/badge.svg)
[![npm](https://img.shields.io/npm/v/fuzzbunny.svg)](https://www.npmjs.com/package/fuzzbunny)
[![npm](https://img.shields.io/npm/dm/fuzzbunny.svg)](https://www.npmjs.com/package/fuzzbunny)
[![npm](https://img.shields.io/npm/l/fuzzbunny.svg)](https://github.com/mixpanel/fuzzbunny/blob/master/LICENCE)

## Why fuzzbunny?

- **Human friendly** - fuzzbunny scoring and algorithm is more tuned to "human" searching patterns. It surfaces what you're looking for with minimal keystrokes.
- **Lightweight** - ~3KB minified and has zero dependencies.
- **Ultra fast** - ~million lines/second on a 2.4Ghz virtual core.

Other similar libraries are [fuzzymatch](https://www.npmjs.com/package/fuzzymatch), [fuzzy](https://www.npmjs.com/package/fuzzy), [fuzzy-search](fuzzy-search), [fuzzyjs](fuzzyjs).

`fuzzbunny` aims to be nimble and fast. It has a simple api that can easily be integrated with any frontend library to build great search UI. We use it at [mixpanel.com](https://mixpanel.com) to power our UI dropdowns and tables.

## Installation

`npm install --save fuzzbunny` or `yarn add fuzzbunny`

## Demo

[Fuzzbunny Gutenberg Catalog Demo →](https://mixpanel.github.io/fuzzbunny)

![Fuzzbunny demo](https://user-images.githubusercontent.com/1018196/77124047-0fbf6580-69ff-11ea-8d44-f8006b7770fd.gif)

## Usage

```js
const {fuzzyFilter, fuzzyMatch} = require(`fuzzbunny`);
// or import {fuzzyFilter, fuzzyMatch} from 'fuzzbunny';

const heroes = [
  {
    name: `Claire Bennet`,
    ability: `Rapid cellular regeneration`,
  },
  {
    name: `Micah Sanders`,
    ability: `Technopathy`,
  },
  {
    name: `Hiro Nakamura`,
    ability: `Space-time manipulation`,
  },
  {
    name: `Peter Petrelli`,
    ability: `Tactile power mimicry`,
  },
];

// Use fuzzyFilter to filter an array of items on specific fields and get filtered + score-sorted results with highlights.
const results = fuzzyFilter(heroes, `stm`, {fields: [`name`, `ability`]});
/*
results = [
  {
    item: {
      name: 'Peter Petrelli',
      ability: 'Tactile power mimicry',
    },
    score: 1786,
    highlights: {
      ability: ['', 'T', 'actile power ', 'm', 'imicry'],
    },
  },
  {
    item: {
      name: 'Hiro Nakamura',
      ability: 'Space-time manipulation',
    },
    score: 983,
    highlights: {
      ability: ['Space-', 't', 'ime ', 'm', 'anipulation'],
    },
  },
];
*/

// Use fuzzyMatch to match a single string to get score + highlights. Returns null if no match found.
const match = fuzzyMatch(heroes[0].name, `ben`);
/*
match = {
  score: 2893,
  highlights: ['Claire ', 'Ben', 'net'],
};
*/
```

## Scoring and Sort order

`fuzzbunny` uses a scoring algorithm that prioritizes following signals. See `_getMatchScore` function.

Example 1:

- Start of string - `{Mayfl}ower` ranks above `The {Mayfl}ower`
- Closer to start - `The {Mayfl}ower` ranks above `Story of the {Mayfl}ower`
- Contiguous length - `The {Mayfl}ower` ranks above `{May} {fl}ower`
- Alphabetically - `The {May} {fl}ower` ranks above `This {May} {fl}ower`

![image](https://user-images.githubusercontent.com/1018196/77127584-58305080-6a0a-11ea-9fee-d8eaf28744b8.png)

Example 2:

```js
const f = require(`fuzzbunny`);
f.fuzzyMatch(`Gobbling pupusas`, `usa`);
// {score: 2700, highlights: ['Gobbling pup', 'usa', 's']}
f.fuzzyMatch(`United Sheets of Antarctica`, `usa`);
// {score: 2276, highlights: ['', 'U', 'nited ', 'S', 'heets of ', 'A', 'ntarctica']}
```

`Gobbling pup{usa}s` wins because 3 letter contiguous sequence yields a higher score.

**NOTE**: `fuzzbunny` optmizes for meaningful results. It only does substring/prefix/acronym-matching, not greedy matching.

This is because humans brains are great at prefix recall.
e.g words that start with "ca" are much easier to recall than words that contain the letters "c" and "a" somewhere.
It's easy to remember that `{usa}` stands for `{U}nited {S}tates of {A}merica`, not `F{u}ll Java{s}cript Fr{a}mework`

## Performance

`fuzzbunny` matches ~ million lines/second on modern hardware. Tested on 2018 MacBook Pro with 2.4Ghz CPU.
See [tests/performance.js](tests/performance.js)

## Types

`fuzzbunny` comes with autogenerated TypeScript types. See [index.d.ts](index.d.ts)
