using System;
using System.Collections.Generic;
using System.Globalization;
using System.Numerics;
using System.Security.Cryptography;
using System.Text;

namespace UChainDB.BingChain.Engine.Cryptography.EC {
	internal static class Helper {
		private static int BitLen(int w) {
			return (w < 1 << 15
				? (w < 1 << 7
					? (w < 1 << 3
						? (w < 1 << 1
							? (w < 1 << 0 ? (w < 0 ? 32 : 0) : 1)
							: (w < 1 << 2 ? 2 : 3))
						: (w < 1 << 5
							? (w < 1 << 4 ? 4 : 5)
							: (w < 1 << 6 ? 6 : 7)))
					: (w < 1 << 11
						? (w < 1 << 9 ? (w < 1 << 8 ? 8 : 9) : (w < 1 << 10 ? 10 : 11))
						: (w < 1 << 13 ? (w < 1 << 12 ? 12 : 13) : (w < 1 << 14 ? 14 : 15))))
				: (w < 1 << 23
					? (w < 1 << 19
						? (w < 1 << 17 ? (w < 1 << 16 ? 16 : 17) : (w < 1 << 18 ? 18 : 19))
						: (w < 1 << 21 ? (w < 1 << 20 ? 20 : 21) : (w < 1 << 22 ? 22 : 23)))
					: (w < 1 << 27
						? (w < 1 << 25 ? (w < 1 << 24 ? 24 : 25) : (w < 1 << 26 ? 26 : 27))
						: (w < 1 << 29 ? (w < 1 << 28 ? 28 : 29) : (w < 1 << 30 ? 30 : 31)))));
		}

		internal static int GetBitLength(this BigInteger i) {
			byte[] b = i.ToByteArray();
			return (b.Length - 1) * 8 + BitLen(i.Sign > 0 ? b[b.Length - 1] : 255 - b[b.Length - 1]);
		}

		internal static int GetLowestSetBit(this BigInteger i) {
			if (i.Sign == 0)
				return -1;
			byte[] b = i.ToByteArray();
			int w = 0;
			while (b[w] == 0)
				w++;
			for (int x = 0; x < 8; x++)
				if ((b[w] & 1 << x) > 0)
					return x + w * 8;
			throw new Exception();
		}

		public static byte[] HexToBytes(this string value) {
			if (value == null || value.Length == 0)
				return new byte[0];
			if (value.Length % 2 == 1)
				throw new FormatException();
			byte[] result = new byte[value.Length / 2];
			for (int i = 0; i < result.Length; i++)
				result[i] = byte.Parse(value.Substring(i * 2, 2), NumberStyles.AllowHexSpecifier);
			return result;
		}

		internal static BigInteger Mod(this BigInteger x, BigInteger y) {
			x %= y;
			if (x.Sign < 0)
				x += y;
			return x;
		}

		internal static BigInteger ModInverse(this BigInteger a, BigInteger n) {
			BigInteger i = n, v = 0, d = 1;
			while (a > 0) {
				BigInteger t = i / a, x = a;
				a = i % x;
				i = x;
				x = d;
				d = v - t * x;
				v = x;
			}

			v %= n;
			if (v < 0) v = (v + n) % n;
			return v;
		}

		internal static BigInteger NextBigInteger(this Random rand, int sizeInBits) {
			if (sizeInBits < 0)
				throw new ArgumentException("sizeInBits must be non-negative");
			if (sizeInBits == 0)
				return 0;
			byte[] b = new byte[sizeInBits / 8 + 1];
			rand.NextBytes(b);
			if (sizeInBits % 8 == 0)
				b[b.Length - 1] = 0;
			else
				b[b.Length - 1] &= (byte) ((1 << sizeInBits % 8) - 1);
			return new BigInteger(b);
		}

		internal static BigInteger NextBigInteger(this RandomNumberGenerator rng, int sizeInBits) {
			if (sizeInBits < 0)
				throw new ArgumentException("sizeInBits must be non-negative");
			if (sizeInBits == 0)
				return 0;
			byte[] b = new byte[sizeInBits / 8 + 1];
			rng.GetBytes(b);
			if (sizeInBits % 8 == 0)
				b[b.Length - 1] = 0;
			else
				b[b.Length - 1] &= (byte) ((1 << sizeInBits % 8) - 1);
			return new BigInteger(b);
		}

		internal static bool TestBit(this BigInteger i, int index) {
			return (i & (BigInteger.One << index)) > BigInteger.Zero;
		}

		public static string ToHexString(this IEnumerable<byte> value) {
			StringBuilder sb = new StringBuilder();
			foreach (byte b in value)
				sb.AppendFormat("{0:x2}", b);
			return sb.ToString();
		}
	}
}
