import React from "react";
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  IconButton,
  TextField,
  useTheme,
  Theme,
} from "@mui/material";
import { makeStyles } from "@mui/styles";
import { MdClose } from "react-icons/md";
import { useForm } from "react-hook-form";
import { PortableText } from "@portabletext/react";
import { I18n } from "@aws-amplify/core";
import { ActiveExperience, MongoExperience } from "../types/LearnTypes";

/**
 * This component is for a Coach to approve the work of the student, and to leave feedback.
 * @TODO Issue #64
 */

const useStyles = (theme: Theme) => makeStyles(() => ({
  root: {
    "& .MuiTextField-root": {
      width: "100%",
    },
    "& .MuiInputBase-input": {
      fontSize: 16,
      color: "#000",
    },
    "& .MuiButtonBase-root": {
      marginTop: theme?.spacing(1) || 8,
      marginRight: 6,
      fontSize: 10,
      padding: "6px 32px",
      textTransform: "none",
    },
    "& .MuiDialogActions-root": {
      padding: 0,
      display: "inline-flex",
      alignItems: "end",
    },
  },
}));

export interface ArenaProofModalProps {
  show: boolean;
  activeExperience: ActiveExperience;
  mongoExperience: MongoExperience;
  closeModal: () => void;
  markRequestRevisions: (activeExperience: ActiveExperience, mongoExperience: MongoExperience, coachNotes: string) => void;
  markComplete: (activeExperience: ActiveExperience, mongoExperience: MongoExperience, coachNotes: string) => void;
}

const ConfirmModal = ({
  show,
  activeExperience,
  mongoExperience,
  closeModal,
  markRequestRevisions,
  markComplete,
}: ArenaProofModalProps) => {
  const { register, handleSubmit } = useForm<{ coachNotes: string }>();
  const theme = useTheme();
  const classes = useStyles(theme)();

  const handleMarkRequestRevisions = handleSubmit((data) => {
    markRequestRevisions(activeExperience, mongoExperience, data.coachNotes);
  });

  const handleMarkComplete = handleSubmit((data) => {
    markComplete(activeExperience, mongoExperience, data.coachNotes);
  })

  return (
    <Dialog
      fullWidth
      open={show}
      onClose={closeModal}
      keepMounted
      hideBackdrop={false}
    >
      <DialogTitle>
        {I18n.get("reviewFor")} {activeExperience.title}
        <IconButton
          aria-label="close"
          onClick={closeModal}
          sx={{
            position: "absolute",
            right: 8,
            top: 8,
          }}
        >
          <MdClose />
        </IconButton>
      </DialogTitle>
      <DialogContent className={classes.root}> { // skipcq: JS-0394
      }
        <h3>{I18n.get("description")}</h3>
        <PortableText value={activeExperience.overview} />
        <h3>{I18n.get("notesForCoach")}</h3>
        <p>{mongoExperience[activeExperience.priority].athleteNotes}</p>
        <h3>{I18n.get("submitLink")}</h3>
        <a
          href={mongoExperience[activeExperience.priority].github}
          target="_blank"
          rel="noopener noreferrer"
        >
          {I18n.get("openLink")}
        </a>
        <br />
        {mongoExperience[activeExperience.priority].coachNotes.length > 0 ? (
          <>
            <h3>{I18n.get("coachesNotes")}</h3>
            <p>{mongoExperience[activeExperience.priority].coachNotes}</p>
            <Button
              size="large"
              variant="contained"
              color="primary"
              onClick={closeModal}
            >
              {I18n.get("close")}
            </Button>
          </>
        ) : (
          <form>
            <h3>{I18n.get("coachesNotes")}</h3>
            <TextField
              id="coachNotes"
              variant="filled"
              size="medium"
              {...register("coachNotes")}
            />
            <DialogActions>
              <Button
                size="large"
                variant="contained"
                color="primary"
                onClick={closeModal}
              >
                {I18n.get("close")}
              </Button>
              {mongoExperience[activeExperience.priority].approved ===
                true ? null : (
                <>
                  {mongoExperience[activeExperience.priority]
                    .revisionsNeeded === false ? (
                    <Button
                      size="medium"
                      variant="contained"
                      color="primary"
                      onClick={handleMarkRequestRevisions}
                    >
                      {I18n.get("requestRevisions")}
                    </Button>
                  ) : null}
                  <Button
                    size="large"
                    variant="contained"
                    color="primary"
                    onClick={handleMarkComplete}
                  >
                    {I18n.get("confirmAchievement")}
                  </Button>
                </>
              )}
            </DialogActions>
          </form>
        )}
      </DialogContent>
    </Dialog>
  );
};
export default ConfirmModal;
