#!/usr/bin/env python3
# Copyright 2016 The Fontbakery Authors
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
import enum

# =====================================
# GLOBAL CONSTANTS DEFINITIONS

# These variable font naming rules will soon change.
# For more detail, see:
# https://github.com/googlefonts/fontbakery/issues/2396#issuecomment-473250089
VARFONT_SUFFIXES = ["VF", "Italic-VF", "Roman-VF"]

STATIC_STYLE_NAMES = [
    "Thin",
    "ExtraLight",
    "Light",
    "Regular",
    "Medium",
    "SemiBold",
    "Bold",
    "ExtraBold",
    "Black",
    "Thin Italic",
    "ExtraLight Italic",
    "Light Italic",
    "Italic",
    "Medium Italic",
    "SemiBold Italic",
    "Bold Italic",
    "ExtraBold Italic",
    "Black Italic",
]

RIBBI_STYLE_NAMES = [
    "Regular",
    "Italic",
    "Bold",
    "BoldItalic",
    "Bold Italic",
]  # <-- Do we really need this one?
# some name ids use spaces other dont

REGISTERED_AXIS_TAGS = ["ital", "opsz", "slnt", "wdth", "wght"]

PLACEHOLDER_LICENSING_TEXT = {
    "UFL.txt": "Licensed under the Ubuntu Font Licence 1.0.",
    "OFL.txt": "This Font Software is licensed under the SIL Open Font "
    "License, Version 1.1. This license is available with a FAQ "
    "at: https://scripts.sil.org/OFL",
    "LICENSE.txt": "Licensed under the Apache License, Version 2.0",
}

SHOW_GF_DOCS_MSG = (
    "Further info can be found in our spec "
    "https://github.com/googlefonts/gf-docs/tree/main/Spec"
)


# ANSI color codes for the helper logging class:
def color(bg, fg, bold=False):
    return (f"\033[{int(bold)};{bg};{fg + 10}m" "{}\033[0m").format


def no_color(s):
    return s


BLACK = 30
RED = 31
GREEN = 32
YELLOW = 33
BLUE = 34
MAGENTA = 35
CYAN = 36
WHITE = 37
BRIGHT_BLACK = 90
BRIGHT_RED = 91
BRIGHT_GREEN = 92
BRIGHT_YELLOW = 93
BRIGHT_BLUE = 94
BRIGHT_MAGENTA = 95
BRIGHT_CYAN = 96
BRIGHT_WHITE = 97

NO_COLORS_THEME = {
    "header": no_color,
    "url": no_color,
    "check-id": no_color,
    "description": no_color,
    "rationale-title": no_color,
    "rationale-text": no_color,
    "INFO": no_color,
    "WARN": no_color,
    "ERROR": no_color,
    "SKIP": no_color,
    "PASS": no_color,
    "FAIL": no_color,
    "cupcake": no_color,
    "spinner": no_color,
    "list-checks: section": no_color,
    "list-checks: check-id": no_color,
    "list-checks: description": no_color,
}

DARK_THEME = {
    # Foreground, Background
    "header": color(WHITE, BLACK, bold=True),
    "url": color(CYAN, BLACK, bold=True),
    "check-id": color(CYAN, BLACK, bold=True),
    "description": color(MAGENTA, BLACK),
    "rationale-title": color(BRIGHT_CYAN, BRIGHT_BLACK, bold=True),
    "rationale-text": color(WHITE, BLACK),
    "INFO": color(CYAN, BLACK),
    "WARN": color(YELLOW, BLACK),
    "ERROR": color(BRIGHT_WHITE, RED),
    "SKIP": color(BLUE, BLACK),
    "PASS": color(GREEN, BLACK),
    "FAIL": color(RED, BLACK),
    "cupcake": color(MAGENTA, BLACK),
    "spinner": color(GREEN, BLACK),
    "list-checks: section": color(WHITE, BLACK),
    "list-checks: check-id": color(CYAN, BLACK),
    "list-checks: description": color(BLUE, BLACK),
}

LIGHT_THEME = {
    # Foreground, Background
    "header": color(BLACK, BRIGHT_WHITE, bold=True),
    "url": color(CYAN, BRIGHT_WHITE, bold=True),
    "check-id": color(MAGENTA, BRIGHT_WHITE, bold=True),
    "description": color(CYAN, BRIGHT_WHITE),
    "rationale-title": color(MAGENTA, BRIGHT_WHITE, bold=True),
    "rationale-text": color(BLACK, BRIGHT_WHITE),
    "INFO": color(CYAN, BRIGHT_WHITE),
    "WARN": color(BLACK, BRIGHT_YELLOW, bold=True),
    "ERROR": color(BRIGHT_WHITE, BRIGHT_RED, bold=True),
    "SKIP": color(BLUE, BRIGHT_WHITE),
    "PASS": color(GREEN, BRIGHT_WHITE),
    "FAIL": color(BRIGHT_RED, BRIGHT_WHITE, bold=True),
    "cupcake": color(MAGENTA, BRIGHT_WHITE),
    "spinner": color(GREEN, BRIGHT_WHITE),
    "list-checks: section": color(WHITE, BRIGHT_WHITE, bold=True),
    "list-checks: check-id": color(CYAN, BRIGHT_WHITE, bold=True),
    "list-checks: description": color(BLUE, BRIGHT_WHITE),
}


class NameID(enum.IntEnum):
    """nameID definitions for the name table"""

    COPYRIGHT_NOTICE = 0
    FONT_FAMILY_NAME = 1
    FONT_SUBFAMILY_NAME = 2
    UNIQUE_FONT_IDENTIFIER = 3
    FULL_FONT_NAME = 4
    VERSION_STRING = 5
    POSTSCRIPT_NAME = 6
    TRADEMARK = 7
    MANUFACTURER_NAME = 8
    DESIGNER = 9
    DESCRIPTION = 10
    VENDOR_URL = 11
    DESIGNER_URL = 12
    LICENSE_DESCRIPTION = 13
    LICENSE_INFO_URL = 14
    # Name ID 15 is RESERVED
    TYPOGRAPHIC_FAMILY_NAME = 16
    TYPOGRAPHIC_SUBFAMILY_NAME = 17
    COMPATIBLE_FULL_MACONLY = 18
    SAMPLE_TEXT = 19
    POSTSCRIPT_CID_NAME = 20
    WWS_FAMILY_NAME = 21
    WWS_SUBFAMILY_NAME = 22
    LIGHT_BACKGROUND_PALETTE = 23
    DARK_BACKGROUD_PALETTE = 24


class GlyphClass(enum.IntEnum):
    BASE = 1
    LIGATURE = 2
    MARK = 3
    COMPONENT = 4


class FsSelection(enum.IntEnum):
    ITALIC = 1 << 0
    UNDERSCORE = 1 << 1
    NEGATIVE = 1 << 2
    OUTLINED = 1 << 3
    STRIKEOUT = 1 << 4
    BOLD = 1 << 5
    REGULAR = 1 << 6
    USETYPOMETRICS = 1 << 7
    WWS = 1 << 8
    OBLIQUE = 1 << 9


class MacStyle(enum.IntEnum):
    BOLD = 1 << 0
    ITALIC = 1 << 1


class PANOSE_Family_Type(enum.IntEnum):
    ANY = 0
    NO_FIT = 1
    LATIN_TEXT = 2
    LATIN_HAND_WRITTEN = 3
    LATIN_DECORATIVE = 4
    LATIN_SYMBOL = 5  # aka LATIN_PICTURE


class PANOSE_Proportion(enum.IntEnum):
    ANY = 0
    NO_FIT = 1
    OLD_STYLE = 2
    MODERN = 3
    EVEN_WIDTH = 4
    EXTENDED = 5
    CONDENSED = 6
    VERY_EXTENDED = 7
    VERY_CONDENSED = 8
    MONOSPACED = 9


class PANOSE_Spacing(enum.IntEnum):
    ANY = 0
    NO_FIT = 1
    PROPORTIONAL = 2
    MONOSPACED = 3


class IsFixedWidth(enum.IntEnum):
    """'post' table / isFixedWidth definitions"""

    NOT_MONOSPACED = 0
    # Do NOT use `MONOSPACED = 1` because *any* non-zero value means monospaced.
    # I've commented it out because we were incorrectly testing against it. - CJC


class PlatformID(enum.IntEnum):
    UNICODE = 0
    MACINTOSH = 1
    ISO = 2
    WINDOWS = 3
    CUSTOM = 4


class UnicodeEncodingID(enum.IntEnum):
    """Unicode platform-specific encoding IDs
    (when platID == 0)
    """

    UNICODE_1_0 = 0
    UNICODE_1_1 = 1
    ISO_IEC_10646 = 2
    UNICODE_2_0_BMP_ONLY = 3  # Basic Multilingual Plane
    UNICODE_2_0_FULL = 4
    UNICODE_VARIATION_SEQUENCES = 5
    UNICODE_FULL = 6


class MacintoshEncodingID(enum.IntEnum):
    """Encoding IDs defined for use
    with the Macintosh platform
    (when platID = 1)
    """

    ROMAN = 0
    JAPANESE = 1
    CHINESE_TRADITIONAL = 2
    KOREAN = 3
    ARABIC = 4
    HEBREW = 5
    GREEK = 6
    RUSSIAN = 7
    RSYMBOL = 8
    DEVANAGARI = 9
    GURMUKHI = 10
    GUJARATI = 11
    ORIYA = 12
    BENGALI = 13
    TAMIL = 14
    TELUGU = 15
    KANNADA = 16
    MALAYALAM = 17
    SINHALESE = 18
    BURMESE = 19
    KHMER = 20
    THAI = 21
    LAOTIAN = 22
    GEORGIAN = 23
    ARMENIAN = 24
    CHINESE_SIMPLIFIED = 25
    TIBETAN = 26
    MONGOLIAN = 27
    GEEZ = 28
    SLAVIC = 29
    VIETNAMESE = 30
    SINDHI = 31
    UNINTERPRETED = 32


class WindowsEncodingID(enum.IntEnum):
    """Windows platform-specific encoding IDs
    (when platID == 3)
    """

    SYMBOL = 0
    UNICODE_BMP = 1  # Basic Multilingual Plane
    SHIFTJIS = 2
    PRC = 3
    BIG5 = 4
    WANSUNG = 5
    JOHAB = 6
    # IDs 7, 8 and 9 are reserved.
    UNICODE_FULL_REPERTOIRE = 10


class MacintoshLanguageID(enum.IntEnum):
    """Platform-specific Language IDs
    assigned by Apple
    """

    ENGLISH = 0


class WindowsLanguageID(enum.IntEnum):
    """Platform-specific Language IDs
    assigned by Microsoft
    """

    ENGLISH_USA = 0x0409


# https://docs.microsoft.com/en-us/typography/opentype/spec/os2
UNICODERANGE_DATA = [
    [(0, "Basic Latin", 0x00000, 0x0007F)],
    [(1, "Latin-1 Supplement", 0x00080, 0x000FF)],
    [(2, "Latin Extended-A", 0x00100, 0x0017F)],
    [(3, "Latin Extended-B", 0x00180, 0x0024F)],
    [
        (4, "IPA Extensions", 0x00250, 0x002AF),
        (4, "Phonetic Extensions", 0x01D00, 0x01D7F),
        (4, "Phonetic Extensions Supplement", 0x01D80, 0x01DBF),
    ],
    [
        (5, "Spacing Modifier Letters", 0x002B0, 0x002FF),
        (5, "Modifier Tone Letters", 0x0A700, 0x0A71F),
    ],
    [
        (6, "Combining Diacritical Marks", 0x00300, 0x0036F),
        (6, "Combining Diacritical Marks Supplement", 0x01DC0, 0x01DFF),
    ],
    [(7, "Greek and Coptic", 0x00370, 0x003FF)],
    [(8, "Coptic", 0x02C80, 0x02CFF)],
    [
        (9, "Cyrillic", 0x00400, 0x004FF),
        (9, "Cyrillic Supplement", 0x00500, 0x0052F),
        (9, "Cyrillic Extended-A", 0x02DE0, 0x02DFF),
        (9, "Cyrillic Extended-B", 0x0A640, 0x0A69F),
    ],
    [(10, "Armenian", 0x00530, 0x0058F)],
    [(11, "Hebrew", 0x00590, 0x005FF)],
    [(12, "Vai", 0x0A500, 0x0A63F)],
    [(13, "Arabic", 0x00600, 0x006FF), (13, "Arabic Supplement", 0x00750, 0x0077F)],
    [(14, "NKo", 0x007C0, 0x007FF)],
    [(15, "Devanagari", 0x00900, 0x0097F)],
    [(16, "Bengali", 0x00980, 0x009FF)],
    [(17, "Gurmukhi", 0x00A00, 0x00A7F)],
    [(18, "Gujarati", 0x00A80, 0x00AFF)],
    [(19, "Oriya", 0x00B00, 0x00B7F)],
    [(20, "Tamil", 0x00B80, 0x00BFF)],
    [(21, "Telugu", 0x00C00, 0x00C7F)],
    [(22, "Kannada", 0x00C80, 0x00CFF)],
    [(23, "Malayalam", 0x00D00, 0x00D7F)],
    [(24, "Thai", 0x00E00, 0x00E7F)],
    [(25, "Lao", 0x00E80, 0x00EFF)],
    [(26, "Georgian", 0x010A0, 0x010FF), (26, "Georgian Supplement", 0x02D00, 0x02D2F)],
    [(27, "Balinese", 0x01B00, 0x01B7F)],
    [(28, "Hangul Jamo", 0x01100, 0x011FF)],
    [
        (29, "Latin Extended Additional", 0x01E00, 0x01EFF),
        (29, "Latin Extended-C", 0x02C60, 0x02C7F),
        (29, "Latin Extended-D", 0x0A720, 0x0A7FF),
    ],
    [(30, "Greek Extended", 0x01F00, 0x01FFF)],
    [
        (31, "General Punctuation", 0x02000, 0x0206F),
        (31, "Supplemental Punctuation", 0x02E00, 0x02E7F),
    ],
    [(32, "Superscripts And Subscripts", 0x02070, 0x0209F)],
    [(33, "Currency Symbols", 0x020A0, 0x020CF)],
    [(34, "Combining Diacritical Marks For Symbols", 0x020D0, 0x020FF)],
    [(35, "Letterlike Symbols", 0x02100, 0x0214F)],
    [(36, "Number Forms", 0x02150, 0x0218F)],
    [
        (37, "Arrows", 0x02190, 0x021FF),
        (37, "Supplemental Arrows-A", 0x027F0, 0x027FF),
        (37, "Supplemental Arrows-B", 0x02900, 0x0297F),
        (37, "Miscellaneous Symbols and Arrows", 0x02B00, 0x02BFF),
    ],
    [
        (38, "Mathematical Operators", 0x02200, 0x022FF),
        (38, "Supplemental Mathematical Operators", 0x02A00, 0x02AFF),
        (38, "Miscellaneous Mathematical Symbols-A", 0x027C0, 0x027EF),
        (38, "Miscellaneous Mathematical Symbols-B", 0x02980, 0x029FF),
    ],
    [(39, "Miscellaneous Technical", 0x02300, 0x023FF)],
    [(40, "Control Pictures", 0x02400, 0x0243F)],
    [(41, "Optical Character Recognition", 0x02440, 0x0245F)],
    [(42, "Enclosed Alphanumerics", 0x02460, 0x024FF)],
    [(43, "Box Drawing", 0x02500, 0x0257F)],
    [(44, "Block Elements", 0x02580, 0x0259F)],
    [(45, "Geometric Shapes", 0x025A0, 0x025FF)],
    [(46, "Miscellaneous Symbols", 0x02600, 0x026FF)],
    [(47, "Dingbats", 0x02700, 0x027BF)],
    [(48, "CJK Symbols And Punctuation", 0x03000, 0x0303F)],
    [(49, "Hiragana", 0x03040, 0x0309F)],
    [
        (50, "Katakana", 0x030A0, 0x030FF),
        (50, "Katakana Phonetic Extensions", 0x031F0, 0x031FF),
    ],
    [(51, "Bopomofo", 0x03100, 0x0312F), (51, "Bopomofo Extended", 0x031A0, 0x031BF)],
    [(52, "Hangul Compatibility Jamo", 0x03130, 0x0318F)],
    [(53, "Phags-pa", 0x0A840, 0x0A87F)],
    [(54, "Enclosed CJK Letters And Months", 0x03200, 0x032FF)],
    [(55, "CJK Compatibility", 0x03300, 0x033FF)],
    [(56, "Hangul Syllables", 0x0AC00, 0x0D7AF)],
    [(57, "Non-Plane 0 *", 0x10000, 0x10FFFF)],
    [(58, "Phoenician", 0x10900, 0x1091F)],
    [
        (59, "CJK Unified Ideographs", 0x04E00, 0x09FFF),
        (59, "CJK Radicals Supplement", 0x02E80, 0x02EFF),
        (59, "Kangxi Radicals", 0x02F00, 0x02FDF),
        (59, "Ideographic Description Characters", 0x02FF0, 0x02FFF),
        (59, "CJK Unified Ideographs Extension A", 0x03400, 0x04DBF),
        (59, "CJK Unified Ideographs Extension B", 0x20000, 0x2A6DF),
        (59, "Kanbun", 0x03190, 0x0319F),
    ],
    [(60, "Private Use Area (plane 0)", 0x0E000, 0x0F8FF)],
    [
        (61, "CJK Strokes", 0x031C0, 0x031EF),
        (61, "CJK Compatibility Ideographs", 0x0F900, 0x0FAFF),
        (61, "CJK Compatibility Ideographs Supplement", 0x2F800, 0x2FA1F),
    ],
    [(62, "Alphabetic Presentation Forms", 0x0FB00, 0x0FB4F)],
    [(63, "Arabic Presentation Forms-A", 0x0FB50, 0x0FDFF)],
    [(64, "Combining Half Marks", 0x0FE20, 0x0FE2F)],
    [
        (65, "Vertical Forms", 0x0FE10, 0x0FE1F),
        (65, "CJK Compatibility Forms", 0x0FE30, 0x0FE4F),
    ],
    [(66, "Small Form Variants", 0x0FE50, 0x0FE6F)],
    [(67, "Arabic Presentation Forms-B", 0x0FE70, 0x0FEFF)],
    [(68, "Halfwidth And Fullwidth Forms", 0x0FF00, 0x0FFEF)],
    [(69, "Specials", 0x0FFF0, 0x0FFFF)],
    [(70, "Tibetan", 0x00F00, 0x00FFF)],
    [(71, "Syriac", 0x00700, 0x0074F)],
    [(72, "Thaana", 0x00780, 0x007BF)],
    [(73, "Sinhala", 0x00D80, 0x00DFF)],
    [(74, "Myanmar", 0x01000, 0x0109F)],
    [
        (75, "Ethiopic", 0x01200, 0x0137F),
        (75, "Ethiopic Supplement", 0x01380, 0x0139F),
        (75, "Ethiopic Extended", 0x02D80, 0x02DDF),
    ],
    [(76, "Cherokee", 0x013A0, 0x013FF)],
    [(77, "Unified Canadian Aboriginal Syllabics", 0x01400, 0x0167F)],
    [(78, "Ogham", 0x01680, 0x0169F)],
    [(79, "Runic", 0x016A0, 0x016FF)],
    [(80, "Khmer", 0x01780, 0x017FF), (80, "Khmer Symbols", 0x019E0, 0x019FF)],
    [(81, "Mongolian", 0x01800, 0x018AF)],
    [(82, "Braille Patterns", 0x02800, 0x028FF)],
    [(83, "Yi Syllables", 0x0A000, 0x0A48F), (83, "Yi Radicals", 0x0A490, 0x0A4CF)],
    [
        (84, "Tagalog", 0x01700, 0x0171F),
        (84, "Hanunoo", 0x01720, 0x0173F),
        (84, "Buhid", 0x01740, 0x0175F),
        (84, "Tagbanwa", 0x01760, 0x0177F),
    ],
    [(85, "Old Italic", 0x10300, 0x1032F)],
    [(86, "Gothic", 0x10330, 0x1034F)],
    [(87, "Deseret", 0x10400, 0x1044F)],
    [
        (88, "Byzantine Musical Symbols", 0x1D000, 0x1D0FF),
        (88, "Musical Symbols", 0x1D100, 0x1D1FF),
        (88, "Ancient Greek Musical Notation", 0x1D200, 0x1D24F),
    ],
    [(89, "Mathematical Alphanumeric Symbols", 0x1D400, 0x1D7FF)],
    [
        (90, "Private Use (plane 15)", 0xFF000, 0xFFFFD),
        (90, "Private Use (plane 16)", 0x100000, 0x10FFFD),
    ],
    [
        (91, "Variation Selectors", 0x0FE00, 0x0FE0F),
        (91, "Variation Selectors Supplement", 0xE0100, 0xE01EF),
    ],
    [(92, "Tags", 0xE0000, 0xE007F)],
    [(93, "Limbu", 0x01900, 0x0194F)],
    [(94, "Tai Le", 0x01950, 0x0197F)],
    [(95, "New Tai Lue", 0x01980, 0x019DF)],
    [(96, "Buginese", 0x01A00, 0x01A1F)],
    [(97, "Glagolitic", 0x02C00, 0x02C5F)],
    [(98, "Tifinagh", 0x02D30, 0x02D7F)],
    [(99, "Yijing Hexagram Symbols", 0x04DC0, 0x04DFF)],
    [(100, "Syloti Nagri", 0x0A800, 0x0A82F)],
    [
        (101, "Linear B Syllabary", 0x10000, 0x1007F),
        (101, "Linear B Ideograms", 0x10080, 0x100FF),
        (101, "Aegean Numbers", 0x10100, 0x1013F),
    ],
    [(102, "Ancient Greek Numbers", 0x10140, 0x1018F)],
    [(103, "Ugaritic", 0x10380, 0x1039F)],
    [(104, "Old Persian", 0x103A0, 0x103DF)],
    [(105, "Shavian", 0x10450, 0x1047F)],
    [(106, "Osmanya", 0x10480, 0x104AF)],
    [(107, "Cypriot Syllabary", 0x10800, 0x1083F)],
    [(108, "Kharoshthi", 0x10A00, 0x10A5F)],
    [(109, "Tai Xuan Jing Symbols", 0x1D300, 0x1D35F)],
    [
        (110, "Cuneiform", 0x12000, 0x123FF),
        (110, "Cuneiform Numbers and Punctuation", 0x12400, 0x1247F),
    ],
    [(111, "Counting Rod Numerals", 0x1D360, 0x1D37F)],
    [(112, "Sundanese", 0x01B80, 0x01BBF)],
    [(113, "Lepcha", 0x01C00, 0x01C4F)],
    [(114, "Ol Chiki", 0x01C50, 0x01C7F)],
    [(115, "Saurashtra", 0x0A880, 0x0A8DF)],
    [(116, "Kayah Li", 0x0A900, 0x0A92F)],
    [(117, "Rejang", 0x0A930, 0x0A95F)],
    [(118, "Cham", 0x0AA00, 0x0AA5F)],
    [(119, "Ancient Symbols", 0x10190, 0x101CF)],
    [(120, "Phaistos Disc", 0x101D0, 0x101FF)],
    [
        (121, "Carian", 0x102A0, 0x102DF),
        (121, "Lycian", 0x10280, 0x1029F),
        (121, "Lydian", 0x10920, 0x1093F),
    ],
    [(122, "Domino Tiles", 0x1F030, 0x1F09F), (122, "Mahjong Tiles", 0x1F000, 0x1F02F)],
]


CJK_CODEPAGE_BITS = {
    "JIS/Japan": 17,
    "Chinese: Simplified chars—PRC and Singapore": 18,
    "Korean Wansung": 19,
    "Chinese: Traditional chars—Taiwan and Hong Kong": 20,
    "Korean Johab": 21,
}


# FIXME: This is a bit redundant with UNICODERANGE_DATA:
CJK_UNICODE_RANGE_BITS = {
    "Hangul Jamo": 28,
    #  'CJK Symbols And Punctuation': 48,
    "Hiragana": 49,
    "Katakana": 50,
    "Bopomofo": 51,
    "Hangul Compatibility Jamo": 52,
    "Enclosed CJK Letters And Months": 54,
    "CJK Compatibility": 55,
    "Hangul Syllables": 56,
    "CJK Unified Ideographs": 59,
    "CJK Strokes": 61,
}


# FIXME: This is a bit redundant with UNICODERANGE_DATA:
CJK_UNICODE_RANGES = [
    [0x1100, 0x11FF],  # Hangul Jamo
    # [0x3000, 0x303F],  # CJK Symbols and Punctuation
    [0x3040, 0x309F],  # Hiragana
    [0x30A0, 0x30FF],  # Katakana
    [0x31F0, 0x31FF],  # Katakana Phonetic Extensions
    [0x3100, 0x312F],  # Bopomofo
    [0x31A0, 0x31BF],  # Bopomofo Extended (Bopomofo)
    [0x3130, 0x318F],  # Hangul Compatibility Jamo
    [0x3200, 0x32FF],  # Enclosed CJK Letters and Months
    [0x3300, 0x33FF],  # CJK Compatibility
    [0xAC00, 0xD7AF],  # Hangul Syllables
    [0x4E00, 0x9FFF],  # CJK Unified Ideographs
    [0x2E80, 0x2EFF],  # CJK Radicals Supplement (CJK Unified Ideographs)
    [0x2F00, 0x2FDF],  # Kangxi Radicals (CJK Unified Ideographs)
    [0x2FF0, 0x2FFF],  # Ideographic Description Characters (CJK Unified Ideographs)
    [0x3400, 0x4DBF],  # CJK Unified Ideographs Extension A (CJK Unified Ideographs)
    [0x20000, 0x2A6DF],  # CJK Unified Ideographs Extension B (CJK Unified Ideographs)
    [0x3190, 0x319F],  # Kanbun (CJK Unified Ideographs)
    [0x31C0, 0x31EF],  # CJK Strokes
    [0xF900, 0xFAFF],  # CJK Compatibility Ideographs (CJK Strokes)
    [0x2F800, 0x2FA1F],  # CJK Compatibility Ideographs Supplement (CJK Strokes)
]

OFL_BODY_TEXT = """

This Font Software is licensed under the SIL Open Font License, Version 1.1.
This license is copied below, and is also available with a FAQ at:
https://scripts.sil.org/OFL


-----------------------------------------------------------
SIL OPEN FONT LICENSE Version 1.1 - 26 February 2007
-----------------------------------------------------------

PREAMBLE
The goals of the Open Font License (OFL) are to stimulate worldwide
development of collaborative font projects, to support the font creation
efforts of academic and linguistic communities, and to provide a free and
open framework in which fonts may be shared and improved in partnership
with others.

The OFL allows the licensed fonts to be used, studied, modified and
redistributed freely as long as they are not sold by themselves. The
fonts, including any derivative works, can be bundled, embedded,
redistributed and/or sold with any software provided that any reserved
names are not used by derivative works. The fonts and derivatives,
however, cannot be released under any other type of license. The
requirement for fonts to remain under this license does not apply
to any document created using the fonts or their derivatives.

DEFINITIONS
"Font Software" refers to the set of files released by the Copyright
Holder(s) under this license and clearly marked as such. This may
include source files, build scripts and documentation.

"Reserved Font Name" refers to any names specified as such after the
copyright statement(s).

"Original Version" refers to the collection of Font Software components as
distributed by the Copyright Holder(s).

"Modified Version" refers to any derivative made by adding to, deleting,
or substituting -- in part or in whole -- any of the components of the
Original Version, by changing formats or by porting the Font Software to a
new environment.

"Author" refers to any designer, engineer, programmer, technical
writer or other person who contributed to the Font Software.

PERMISSION & CONDITIONS
Permission is hereby granted, free of charge, to any person obtaining
a copy of the Font Software, to use, study, copy, merge, embed, modify,
redistribute, and sell modified and unmodified copies of the Font
Software, subject to the following conditions:

1) Neither the Font Software nor any of its individual components,
in Original or Modified Versions, may be sold by itself.

2) Original or Modified Versions of the Font Software may be bundled,
redistributed and/or sold with any software, provided that each copy
contains the above copyright notice and this license. These can be
included either as stand-alone text files, human-readable headers or
in the appropriate machine-readable metadata fields within text or
binary files as long as those fields can be easily viewed by the user.

3) No Modified Version of the Font Software may use the Reserved Font
Name(s) unless explicit written permission is granted by the corresponding
Copyright Holder. This restriction only applies to the primary font name as
presented to the users.

4) The name(s) of the Copyright Holder(s) or the Author(s) of the Font
Software shall not be used to promote, endorse or advertise any
Modified Version, except to acknowledge the contribution(s) of the
Copyright Holder(s) and the Author(s) or with their explicit written
permission.

5) The Font Software, modified or unmodified, in part or in whole,
must be distributed entirely under this license, and must not be
distributed under any other license. The requirement for fonts to
remain under this license does not apply to any document created
using the Font Software.

TERMINATION
This license becomes null and void if any of the above conditions are
not met.

DISCLAIMER
THE FONT SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO ANY WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
OF COPYRIGHT, PATENT, TRADEMARK, OR OTHER RIGHT. IN NO EVENT SHALL THE
COPYRIGHT HOLDER BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
INCLUDING ANY GENERAL, SPECIAL, INDIRECT, INCIDENTAL, OR CONSEQUENTIAL
DAMAGES, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF THE USE OR INABILITY TO USE THE FONT SOFTWARE OR FROM
OTHER DEALINGS IN THE FONT SOFTWARE.
"""

LATEST_TTFAUTOHINT_VERSION = "1.8.4"


# Set of 2,350 modern Korean hangul syllables from KS X 1001 encoding.
# The file linked below provides the mapping between KS X 1001 and Unicode.
# http://unicode.org/Public/MAPPINGS/OBSOLETE/EASTASIA/KSC/KSX1001.TXT
MODERN_HANGUL_SYLLABLES_CODEPOINTS = {
    0xAC00,
    0xAC01,
    0xAC04,
    0xAC07,
    0xAC08,
    0xAC09,
    0xAC0A,
    0xAC10,
    0xAC11,
    0xAC12,
    0xAC13,
    0xAC14,
    0xAC15,
    0xAC16,
    0xAC17,
    0xAC19,
    0xAC1A,
    0xAC1B,
    0xAC1C,
    0xAC1D,
    0xAC20,
    0xAC24,
    0xAC2C,
    0xAC2D,
    0xAC2F,
    0xAC30,
    0xAC31,
    0xAC38,
    0xAC39,
    0xAC3C,
    0xAC40,
    0xAC4B,
    0xAC4D,
    0xAC54,
    0xAC58,
    0xAC5C,
    0xAC70,
    0xAC71,
    0xAC74,
    0xAC77,
    0xAC78,
    0xAC7A,
    0xAC80,
    0xAC81,
    0xAC83,
    0xAC84,
    0xAC85,
    0xAC86,
    0xAC89,
    0xAC8A,
    0xAC8B,
    0xAC8C,
    0xAC90,
    0xAC94,
    0xAC9C,
    0xAC9D,
    0xAC9F,
    0xACA0,
    0xACA1,
    0xACA8,
    0xACA9,
    0xACAA,
    0xACAC,
    0xACAF,
    0xACB0,
    0xACB8,
    0xACB9,
    0xACBB,
    0xACBC,
    0xACBD,
    0xACC1,
    0xACC4,
    0xACC8,
    0xACCC,
    0xACD5,
    0xACD7,
    0xACE0,
    0xACE1,
    0xACE4,
    0xACE7,
    0xACE8,
    0xACEA,
    0xACEC,
    0xACEF,
    0xACF0,
    0xACF1,
    0xACF3,
    0xACF5,
    0xACF6,
    0xACFC,
    0xACFD,
    0xAD00,
    0xAD04,
    0xAD06,
    0xAD0C,
    0xAD0D,
    0xAD0F,
    0xAD11,
    0xAD18,
    0xAD1C,
    0xAD20,
    0xAD29,
    0xAD2C,
    0xAD2D,
    0xAD34,
    0xAD35,
    0xAD38,
    0xAD3C,
    0xAD44,
    0xAD45,
    0xAD47,
    0xAD49,
    0xAD50,
    0xAD54,
    0xAD58,
    0xAD61,
    0xAD63,
    0xAD6C,
    0xAD6D,
    0xAD70,
    0xAD73,
    0xAD74,
    0xAD75,
    0xAD76,
    0xAD7B,
    0xAD7C,
    0xAD7D,
    0xAD7F,
    0xAD81,
    0xAD82,
    0xAD88,
    0xAD89,
    0xAD8C,
    0xAD90,
    0xAD9C,
    0xAD9D,
    0xADA4,
    0xADB7,
    0xADC0,
    0xADC1,
    0xADC4,
    0xADC8,
    0xADD0,
    0xADD1,
    0xADD3,
    0xADDC,
    0xADE0,
    0xADE4,
    0xADF8,
    0xADF9,
    0xADFC,
    0xADFF,
    0xAE00,
    0xAE01,
    0xAE08,
    0xAE09,
    0xAE0B,
    0xAE0D,
    0xAE14,
    0xAE30,
    0xAE31,
    0xAE34,
    0xAE37,
    0xAE38,
    0xAE3A,
    0xAE40,
    0xAE41,
    0xAE43,
    0xAE45,
    0xAE46,
    0xAE4A,
    0xAE4C,
    0xAE4D,
    0xAE4E,
    0xAE50,
    0xAE54,
    0xAE56,
    0xAE5C,
    0xAE5D,
    0xAE5F,
    0xAE60,
    0xAE61,
    0xAE65,
    0xAE68,
    0xAE69,
    0xAE6C,
    0xAE70,
    0xAE78,
    0xAE79,
    0xAE7B,
    0xAE7C,
    0xAE7D,
    0xAE84,
    0xAE85,
    0xAE8C,
    0xAEBC,
    0xAEBD,
    0xAEBE,
    0xAEC0,
    0xAEC4,
    0xAECC,
    0xAECD,
    0xAECF,
    0xAED0,
    0xAED1,
    0xAED8,
    0xAED9,
    0xAEDC,
    0xAEE8,
    0xAEEB,
    0xAEED,
    0xAEF4,
    0xAEF8,
    0xAEFC,
    0xAF07,
    0xAF08,
    0xAF0D,
    0xAF10,
    0xAF2C,
    0xAF2D,
    0xAF30,
    0xAF32,
    0xAF34,
    0xAF3C,
    0xAF3D,
    0xAF3F,
    0xAF41,
    0xAF42,
    0xAF43,
    0xAF48,
    0xAF49,
    0xAF50,
    0xAF5C,
    0xAF5D,
    0xAF64,
    0xAF65,
    0xAF79,
    0xAF80,
    0xAF84,
    0xAF88,
    0xAF90,
    0xAF91,
    0xAF95,
    0xAF9C,
    0xAFB8,
    0xAFB9,
    0xAFBC,
    0xAFC0,
    0xAFC7,
    0xAFC8,
    0xAFC9,
    0xAFCB,
    0xAFCD,
    0xAFCE,
    0xAFD4,
    0xAFDC,
    0xAFE8,
    0xAFE9,
    0xAFF0,
    0xAFF1,
    0xAFF4,
    0xAFF8,
    0xB000,
    0xB001,
    0xB004,
    0xB00C,
    0xB010,
    0xB014,
    0xB01C,
    0xB01D,
    0xB028,
    0xB044,
    0xB045,
    0xB048,
    0xB04A,
    0xB04C,
    0xB04E,
    0xB053,
    0xB054,
    0xB055,
    0xB057,
    0xB059,
    0xB05D,
    0xB07C,
    0xB07D,
    0xB080,
    0xB084,
    0xB08C,
    0xB08D,
    0xB08F,
    0xB091,
    0xB098,
    0xB099,
    0xB09A,
    0xB09C,
    0xB09F,
    0xB0A0,
    0xB0A1,
    0xB0A2,
    0xB0A8,
    0xB0A9,
    0xB0AB,
    0xB0AC,
    0xB0AD,
    0xB0AE,
    0xB0AF,
    0xB0B1,
    0xB0B3,
    0xB0B4,
    0xB0B5,
    0xB0B8,
    0xB0BC,
    0xB0C4,
    0xB0C5,
    0xB0C7,
    0xB0C8,
    0xB0C9,
    0xB0D0,
    0xB0D1,
    0xB0D4,
    0xB0D8,
    0xB0E0,
    0xB0E5,
    0xB108,
    0xB109,
    0xB10B,
    0xB10C,
    0xB110,
    0xB112,
    0xB113,
    0xB118,
    0xB119,
    0xB11B,
    0xB11C,
    0xB11D,
    0xB123,
    0xB124,
    0xB125,
    0xB128,
    0xB12C,
    0xB134,
    0xB135,
    0xB137,
    0xB138,
    0xB139,
    0xB140,
    0xB141,
    0xB144,
    0xB148,
    0xB150,
    0xB151,
    0xB154,
    0xB155,
    0xB158,
    0xB15C,
    0xB160,
    0xB178,
    0xB179,
    0xB17C,
    0xB180,
    0xB182,
    0xB188,
    0xB189,
    0xB18B,
    0xB18D,
    0xB192,
    0xB193,
    0xB194,
    0xB198,
    0xB19C,
    0xB1A8,
    0xB1CC,
    0xB1D0,
    0xB1D4,
    0xB1DC,
    0xB1DD,
    0xB1DF,
    0xB1E8,
    0xB1E9,
    0xB1EC,
    0xB1F0,
    0xB1F9,
    0xB1FB,
    0xB1FD,
    0xB204,
    0xB205,
    0xB208,
    0xB20B,
    0xB20C,
    0xB214,
    0xB215,
    0xB217,
    0xB219,
    0xB220,
    0xB234,
    0xB23C,
    0xB258,
    0xB25C,
    0xB260,
    0xB268,
    0xB269,
    0xB274,
    0xB275,
    0xB27C,
    0xB284,
    0xB285,
    0xB289,
    0xB290,
    0xB291,
    0xB294,
    0xB298,
    0xB299,
    0xB29A,
    0xB2A0,
    0xB2A1,
    0xB2A3,
    0xB2A5,
    0xB2A6,
    0xB2AA,
    0xB2AC,
    0xB2B0,
    0xB2B4,
    0xB2C8,
    0xB2C9,
    0xB2CC,
    0xB2D0,
    0xB2D2,
    0xB2D8,
    0xB2D9,
    0xB2DB,
    0xB2DD,
    0xB2E2,
    0xB2E4,
    0xB2E5,
    0xB2E6,
    0xB2E8,
    0xB2EB,
    0xB2EC,
    0xB2ED,
    0xB2EE,
    0xB2EF,
    0xB2F3,
    0xB2F4,
    0xB2F5,
    0xB2F7,
    0xB2F8,
    0xB2F9,
    0xB2FA,
    0xB2FB,
    0xB2FF,
    0xB300,
    0xB301,
    0xB304,
    0xB308,
    0xB310,
    0xB311,
    0xB313,
    0xB314,
    0xB315,
    0xB31C,
    0xB354,
    0xB355,
    0xB356,
    0xB358,
    0xB35B,
    0xB35C,
    0xB35E,
    0xB35F,
    0xB364,
    0xB365,
    0xB367,
    0xB369,
    0xB36B,
    0xB36E,
    0xB370,
    0xB371,
    0xB374,
    0xB378,
    0xB380,
    0xB381,
    0xB383,
    0xB384,
    0xB385,
    0xB38C,
    0xB390,
    0xB394,
    0xB3A0,
    0xB3A1,
    0xB3A8,
    0xB3AC,
    0xB3C4,
    0xB3C5,
    0xB3C8,
    0xB3CB,
    0xB3CC,
    0xB3CE,
    0xB3D0,
    0xB3D4,
    0xB3D5,
    0xB3D7,
    0xB3D9,
    0xB3DB,
    0xB3DD,
    0xB3E0,
    0xB3E4,
    0xB3E8,
    0xB3FC,
    0xB410,
    0xB418,
    0xB41C,
    0xB420,
    0xB428,
    0xB429,
    0xB42B,
    0xB434,
    0xB450,
    0xB451,
    0xB454,
    0xB458,
    0xB460,
    0xB461,
    0xB463,
    0xB465,
    0xB46C,
    0xB480,
    0xB488,
    0xB49D,
    0xB4A4,
    0xB4A8,
    0xB4AC,
    0xB4B5,
    0xB4B7,
    0xB4B9,
    0xB4C0,
    0xB4C4,
    0xB4C8,
    0xB4D0,
    0xB4D5,
    0xB4DC,
    0xB4DD,
    0xB4E0,
    0xB4E3,
    0xB4E4,
    0xB4E6,
    0xB4EC,
    0xB4ED,
    0xB4EF,
    0xB4F1,
    0xB4F8,
    0xB514,
    0xB515,
    0xB518,
    0xB51B,
    0xB51C,
    0xB524,
    0xB525,
    0xB527,
    0xB528,
    0xB529,
    0xB52A,
    0xB530,
    0xB531,
    0xB534,
    0xB538,
    0xB540,
    0xB541,
    0xB543,
    0xB544,
    0xB545,
    0xB54B,
    0xB54C,
    0xB54D,
    0xB550,
    0xB554,
    0xB55C,
    0xB55D,
    0xB55F,
    0xB560,
    0xB561,
    0xB5A0,
    0xB5A1,
    0xB5A4,
    0xB5A8,
    0xB5AA,
    0xB5AB,
    0xB5B0,
    0xB5B1,
    0xB5B3,
    0xB5B4,
    0xB5B5,
    0xB5BB,
    0xB5BC,
    0xB5BD,
    0xB5C0,
    0xB5C4,
    0xB5CC,
    0xB5CD,
    0xB5CF,
    0xB5D0,
    0xB5D1,
    0xB5D8,
    0xB5EC,
    0xB610,
    0xB611,
    0xB614,
    0xB618,
    0xB625,
    0xB62C,
    0xB634,
    0xB648,
    0xB664,
    0xB668,
    0xB69C,
    0xB69D,
    0xB6A0,
    0xB6A4,
    0xB6AB,
    0xB6AC,
    0xB6B1,
    0xB6D4,
    0xB6F0,
    0xB6F4,
    0xB6F8,
    0xB700,
    0xB701,
    0xB705,
    0xB728,
    0xB729,
    0xB72C,
    0xB72F,
    0xB730,
    0xB738,
    0xB739,
    0xB73B,
    0xB744,
    0xB748,
    0xB74C,
    0xB754,
    0xB755,
    0xB760,
    0xB764,
    0xB768,
    0xB770,
    0xB771,
    0xB773,
    0xB775,
    0xB77C,
    0xB77D,
    0xB780,
    0xB784,
    0xB78C,
    0xB78D,
    0xB78F,
    0xB790,
    0xB791,
    0xB792,
    0xB796,
    0xB797,
    0xB798,
    0xB799,
    0xB79C,
    0xB7A0,
    0xB7A8,
    0xB7A9,
    0xB7AB,
    0xB7AC,
    0xB7AD,
    0xB7B4,
    0xB7B5,
    0xB7B8,
    0xB7C7,
    0xB7C9,
    0xB7EC,
    0xB7ED,
    0xB7F0,
    0xB7F4,
    0xB7FC,
    0xB7FD,
    0xB7FF,
    0xB800,
    0xB801,
    0xB807,
    0xB808,
    0xB809,
    0xB80C,
    0xB810,
    0xB818,
    0xB819,
    0xB81B,
    0xB81D,
    0xB824,
    0xB825,
    0xB828,
    0xB82C,
    0xB834,
    0xB835,
    0xB837,
    0xB838,
    0xB839,
    0xB840,
    0xB844,
    0xB851,
    0xB853,
    0xB85C,
    0xB85D,
    0xB860,
    0xB864,
    0xB86C,
    0xB86D,
    0xB86F,
    0xB871,
    0xB878,
    0xB87C,
    0xB88D,
    0xB8A8,
    0xB8B0,
    0xB8B4,
    0xB8B8,
    0xB8C0,
    0xB8C1,
    0xB8C3,
    0xB8C5,
    0xB8CC,
    0xB8D0,
    0xB8D4,
    0xB8DD,
    0xB8DF,
    0xB8E1,
    0xB8E8,
    0xB8E9,
    0xB8EC,
    0xB8F0,
    0xB8F8,
    0xB8F9,
    0xB8FB,
    0xB8FD,
    0xB904,
    0xB918,
    0xB920,
    0xB93C,
    0xB93D,
    0xB940,
    0xB944,
    0xB94C,
    0xB94F,
    0xB951,
    0xB958,
    0xB959,
    0xB95C,
    0xB960,
    0xB968,
    0xB969,
    0xB96B,
    0xB96D,
    0xB974,
    0xB975,
    0xB978,
    0xB97C,
    0xB984,
    0xB985,
    0xB987,
    0xB989,
    0xB98A,
    0xB98D,
    0xB98E,
    0xB9AC,
    0xB9AD,
    0xB9B0,
    0xB9B4,
    0xB9BC,
    0xB9BD,
    0xB9BF,
    0xB9C1,
    0xB9C8,
    0xB9C9,
    0xB9CC,
    0xB9CE,
    0xB9CF,
    0xB9D0,
    0xB9D1,
    0xB9D2,
    0xB9D8,
    0xB9D9,
    0xB9DB,
    0xB9DD,
    0xB9DE,
    0xB9E1,
    0xB9E3,
    0xB9E4,
    0xB9E5,
    0xB9E8,
    0xB9EC,
    0xB9F4,
    0xB9F5,
    0xB9F7,
    0xB9F8,
    0xB9F9,
    0xB9FA,
    0xBA00,
    0xBA01,
    0xBA08,
    0xBA15,
    0xBA38,
    0xBA39,
    0xBA3C,
    0xBA40,
    0xBA42,
    0xBA48,
    0xBA49,
    0xBA4B,
    0xBA4D,
    0xBA4E,
    0xBA53,
    0xBA54,
    0xBA55,
    0xBA58,
    0xBA5C,
    0xBA64,
    0xBA65,
    0xBA67,
    0xBA68,
    0xBA69,
    0xBA70,
    0xBA71,
    0xBA74,
    0xBA78,
    0xBA83,
    0xBA84,
    0xBA85,
    0xBA87,
    0xBA8C,
    0xBAA8,
    0xBAA9,
    0xBAAB,
    0xBAAC,
    0xBAB0,
    0xBAB2,
    0xBAB8,
    0xBAB9,
    0xBABB,
    0xBABD,
    0xBAC4,
    0xBAC8,
    0xBAD8,
    0xBAD9,
    0xBAFC,
    0xBB00,
    0xBB04,
    0xBB0D,
    0xBB0F,
    0xBB11,
    0xBB18,
    0xBB1C,
    0xBB20,
    0xBB29,
    0xBB2B,
    0xBB34,
    0xBB35,
    0xBB36,
    0xBB38,
    0xBB3B,
    0xBB3C,
    0xBB3D,
    0xBB3E,
    0xBB44,
    0xBB45,
    0xBB47,
    0xBB49,
    0xBB4D,
    0xBB4F,
    0xBB50,
    0xBB54,
    0xBB58,
    0xBB61,
    0xBB63,
    0xBB6C,
    0xBB88,
    0xBB8C,
    0xBB90,
    0xBBA4,
    0xBBA8,
    0xBBAC,
    0xBBB4,
    0xBBB7,
    0xBBC0,
    0xBBC4,
    0xBBC8,
    0xBBD0,
    0xBBD3,
    0xBBF8,
    0xBBF9,
    0xBBFC,
    0xBBFF,
    0xBC00,
    0xBC02,
    0xBC08,
    0xBC09,
    0xBC0B,
    0xBC0C,
    0xBC0D,
    0xBC0F,
    0xBC11,
    0xBC14,
    0xBC15,
    0xBC16,
    0xBC17,
    0xBC18,
    0xBC1B,
    0xBC1C,
    0xBC1D,
    0xBC1E,
    0xBC1F,
    0xBC24,
    0xBC25,
    0xBC27,
    0xBC29,
    0xBC2D,
    0xBC30,
    0xBC31,
    0xBC34,
    0xBC38,
    0xBC40,
    0xBC41,
    0xBC43,
    0xBC44,
    0xBC45,
    0xBC49,
    0xBC4C,
    0xBC4D,
    0xBC50,
    0xBC5D,
    0xBC84,
    0xBC85,
    0xBC88,
    0xBC8B,
    0xBC8C,
    0xBC8E,
    0xBC94,
    0xBC95,
    0xBC97,
    0xBC99,
    0xBC9A,
    0xBCA0,
    0xBCA1,
    0xBCA4,
    0xBCA7,
    0xBCA8,
    0xBCB0,
    0xBCB1,
    0xBCB3,
    0xBCB4,
    0xBCB5,
    0xBCBC,
    0xBCBD,
    0xBCC0,
    0xBCC4,
    0xBCCD,
    0xBCCF,
    0xBCD0,
    0xBCD1,
    0xBCD5,
    0xBCD8,
    0xBCDC,
    0xBCF4,
    0xBCF5,
    0xBCF6,
    0xBCF8,
    0xBCFC,
    0xBD04,
    0xBD05,
    0xBD07,
    0xBD09,
    0xBD10,
    0xBD14,
    0xBD24,
    0xBD2C,
    0xBD40,
    0xBD48,
    0xBD49,
    0xBD4C,
    0xBD50,
    0xBD58,
    0xBD59,
    0xBD64,
    0xBD68,
    0xBD80,
    0xBD81,
    0xBD84,
    0xBD87,
    0xBD88,
    0xBD89,
    0xBD8A,
    0xBD90,
    0xBD91,
    0xBD93,
    0xBD95,
    0xBD99,
    0xBD9A,
    0xBD9C,
    0xBDA4,
    0xBDB0,
    0xBDB8,
    0xBDD4,
    0xBDD5,
    0xBDD8,
    0xBDDC,
    0xBDE9,
    0xBDF0,
    0xBDF4,
    0xBDF8,
    0xBE00,
    0xBE03,
    0xBE05,
    0xBE0C,
    0xBE0D,
    0xBE10,
    0xBE14,
    0xBE1C,
    0xBE1D,
    0xBE1F,
    0xBE44,
    0xBE45,
    0xBE48,
    0xBE4C,
    0xBE4E,
    0xBE54,
    0xBE55,
    0xBE57,
    0xBE59,
    0xBE5A,
    0xBE5B,
    0xBE60,
    0xBE61,
    0xBE64,
    0xBE68,
    0xBE6A,
    0xBE70,
    0xBE71,
    0xBE73,
    0xBE74,
    0xBE75,
    0xBE7B,
    0xBE7C,
    0xBE7D,
    0xBE80,
    0xBE84,
    0xBE8C,
    0xBE8D,
    0xBE8F,
    0xBE90,
    0xBE91,
    0xBE98,
    0xBE99,
    0xBEA8,
    0xBED0,
    0xBED1,
    0xBED4,
    0xBED7,
    0xBED8,
    0xBEE0,
    0xBEE3,
    0xBEE4,
    0xBEE5,
    0xBEEC,
    0xBF01,
    0xBF08,
    0xBF09,
    0xBF18,
    0xBF19,
    0xBF1B,
    0xBF1C,
    0xBF1D,
    0xBF40,
    0xBF41,
    0xBF44,
    0xBF48,
    0xBF50,
    0xBF51,
    0xBF55,
    0xBF94,
    0xBFB0,
    0xBFC5,
    0xBFCC,
    0xBFCD,
    0xBFD0,
    0xBFD4,
    0xBFDC,
    0xBFDF,
    0xBFE1,
    0xC03C,
    0xC051,
    0xC058,
    0xC05C,
    0xC060,
    0xC068,
    0xC069,
    0xC090,
    0xC091,
    0xC094,
    0xC098,
    0xC0A0,
    0xC0A1,
    0xC0A3,
    0xC0A5,
    0xC0AC,
    0xC0AD,
    0xC0AF,
    0xC0B0,
    0xC0B3,
    0xC0B4,
    0xC0B5,
    0xC0B6,
    0xC0BC,
    0xC0BD,
    0xC0BF,
    0xC0C0,
    0xC0C1,
    0xC0C5,
    0xC0C8,
    0xC0C9,
    0xC0CC,
    0xC0D0,
    0xC0D8,
    0xC0D9,
    0xC0DB,
    0xC0DC,
    0xC0DD,
    0xC0E4,
    0xC0E5,
    0xC0E8,
    0xC0EC,
    0xC0F4,
    0xC0F5,
    0xC0F7,
    0xC0F9,
    0xC100,
    0xC104,
    0xC108,
    0xC110,
    0xC115,
    0xC11C,
    0xC11D,
    0xC11E,
    0xC11F,
    0xC120,
    0xC123,
    0xC124,
    0xC126,
    0xC127,
    0xC12C,
    0xC12D,
    0xC12F,
    0xC130,
    0xC131,
    0xC136,
    0xC138,
    0xC139,
    0xC13C,
    0xC140,
    0xC148,
    0xC149,
    0xC14B,
    0xC14C,
    0xC14D,
    0xC154,
    0xC155,
    0xC158,
    0xC15C,
    0xC164,
    0xC165,
    0xC167,
    0xC168,
    0xC169,
    0xC170,
    0xC174,
    0xC178,
    0xC185,
    0xC18C,
    0xC18D,
    0xC18E,
    0xC190,
    0xC194,
    0xC196,
    0xC19C,
    0xC19D,
    0xC19F,
    0xC1A1,
    0xC1A5,
    0xC1A8,
    0xC1A9,
    0xC1AC,
    0xC1B0,
    0xC1BD,
    0xC1C4,
    0xC1C8,
    0xC1CC,
    0xC1D4,
    0xC1D7,
    0xC1D8,
    0xC1E0,
    0xC1E4,
    0xC1E8,
    0xC1F0,
    0xC1F1,
    0xC1F3,
    0xC1FC,
    0xC1FD,
    0xC200,
    0xC204,
    0xC20C,
    0xC20D,
    0xC20F,
    0xC211,
    0xC218,
    0xC219,
    0xC21C,
    0xC21F,
    0xC220,
    0xC228,
    0xC229,
    0xC22B,
    0xC22D,
    0xC22F,
    0xC231,
    0xC232,
    0xC234,
    0xC248,
    0xC250,
    0xC251,
    0xC254,
    0xC258,
    0xC260,
    0xC265,
    0xC26C,
    0xC26D,
    0xC270,
    0xC274,
    0xC27C,
    0xC27D,
    0xC27F,
    0xC281,
    0xC288,
    0xC289,
    0xC290,
    0xC298,
    0xC29B,
    0xC29D,
    0xC2A4,
    0xC2A5,
    0xC2A8,
    0xC2AC,
    0xC2AD,
    0xC2B4,
    0xC2B5,
    0xC2B7,
    0xC2B9,
    0xC2DC,
    0xC2DD,
    0xC2E0,
    0xC2E3,
    0xC2E4,
    0xC2EB,
    0xC2EC,
    0xC2ED,
    0xC2EF,
    0xC2F1,
    0xC2F6,
    0xC2F8,
    0xC2F9,
    0xC2FB,
    0xC2FC,
    0xC300,
    0xC308,
    0xC309,
    0xC30C,
    0xC30D,
    0xC313,
    0xC314,
    0xC315,
    0xC318,
    0xC31C,
    0xC324,
    0xC325,
    0xC328,
    0xC329,
    0xC345,
    0xC368,
    0xC369,
    0xC36C,
    0xC370,
    0xC372,
    0xC378,
    0xC379,
    0xC37C,
    0xC37D,
    0xC384,
    0xC388,
    0xC38C,
    0xC3C0,
    0xC3D8,
    0xC3D9,
    0xC3DC,
    0xC3DF,
    0xC3E0,
    0xC3E2,
    0xC3E8,
    0xC3E9,
    0xC3ED,
    0xC3F4,
    0xC3F5,
    0xC3F8,
    0xC408,
    0xC410,
    0xC424,
    0xC42C,
    0xC430,
    0xC434,
    0xC43C,
    0xC43D,
    0xC448,
    0xC464,
    0xC465,
    0xC468,
    0xC46C,
    0xC474,
    0xC475,
    0xC479,
    0xC480,
    0xC494,
    0xC49C,
    0xC4B8,
    0xC4BC,
    0xC4E9,
    0xC4F0,
    0xC4F1,
    0xC4F4,
    0xC4F8,
    0xC4FA,
    0xC4FF,
    0xC500,
    0xC501,
    0xC50C,
    0xC510,
    0xC514,
    0xC51C,
    0xC528,
    0xC529,
    0xC52C,
    0xC530,
    0xC538,
    0xC539,
    0xC53B,
    0xC53D,
    0xC544,
    0xC545,
    0xC548,
    0xC549,
    0xC54A,
    0xC54C,
    0xC54D,
    0xC54E,
    0xC553,
    0xC554,
    0xC555,
    0xC557,
    0xC558,
    0xC559,
    0xC55D,
    0xC55E,
    0xC560,
    0xC561,
    0xC564,
    0xC568,
    0xC570,
    0xC571,
    0xC573,
    0xC574,
    0xC575,
    0xC57C,
    0xC57D,
    0xC580,
    0xC584,
    0xC587,
    0xC58C,
    0xC58D,
    0xC58F,
    0xC591,
    0xC595,
    0xC597,
    0xC598,
    0xC59C,
    0xC5A0,
    0xC5A9,
    0xC5B4,
    0xC5B5,
    0xC5B8,
    0xC5B9,
    0xC5BB,
    0xC5BC,
    0xC5BD,
    0xC5BE,
    0xC5C4,
    0xC5C5,
    0xC5C6,
    0xC5C7,
    0xC5C8,
    0xC5C9,
    0xC5CA,
    0xC5CC,
    0xC5CE,
    0xC5D0,
    0xC5D1,
    0xC5D4,
    0xC5D8,
    0xC5E0,
    0xC5E1,
    0xC5E3,
    0xC5E5,
    0xC5EC,
    0xC5ED,
    0xC5EE,
    0xC5F0,
    0xC5F4,
    0xC5F6,
    0xC5F7,
    0xC5FC,
    0xC5FD,
    0xC5FE,
    0xC5FF,
    0xC600,
    0xC601,
    0xC605,
    0xC606,
    0xC607,
    0xC608,
    0xC60C,
    0xC610,
    0xC618,
    0xC619,
    0xC61B,
    0xC61C,
    0xC624,
    0xC625,
    0xC628,
    0xC62C,
    0xC62D,
    0xC62E,
    0xC630,
    0xC633,
    0xC634,
    0xC635,
    0xC637,
    0xC639,
    0xC63B,
    0xC640,
    0xC641,
    0xC644,
    0xC648,
    0xC650,
    0xC651,
    0xC653,
    0xC654,
    0xC655,
    0xC65C,
    0xC65D,
    0xC660,
    0xC66C,
    0xC66F,
    0xC671,
    0xC678,
    0xC679,
    0xC67C,
    0xC680,
    0xC688,
    0xC689,
    0xC68B,
    0xC68D,
    0xC694,
    0xC695,
    0xC698,
    0xC69C,
    0xC6A4,
    0xC6A5,
    0xC6A7,
    0xC6A9,
    0xC6B0,
    0xC6B1,
    0xC6B4,
    0xC6B8,
    0xC6B9,
    0xC6BA,
    0xC6C0,
    0xC6C1,
    0xC6C3,
    0xC6C5,
    0xC6CC,
    0xC6CD,
    0xC6D0,
    0xC6D4,
    0xC6DC,
    0xC6DD,
    0xC6E0,
    0xC6E1,
    0xC6E8,
    0xC6E9,
    0xC6EC,
    0xC6F0,
    0xC6F8,
    0xC6F9,
    0xC6FD,
    0xC704,
    0xC705,
    0xC708,
    0xC70C,
    0xC714,
    0xC715,
    0xC717,
    0xC719,
    0xC720,
    0xC721,
    0xC724,
    0xC728,
    0xC730,
    0xC731,
    0xC733,
    0xC735,
    0xC737,
    0xC73C,
    0xC73D,
    0xC740,
    0xC744,
    0xC74A,
    0xC74C,
    0xC74D,
    0xC74F,
    0xC751,
    0xC752,
    0xC753,
    0xC754,
    0xC755,
    0xC756,
    0xC757,
    0xC758,
    0xC75C,
    0xC760,
    0xC768,
    0xC76B,
    0xC774,
    0xC775,
    0xC778,
    0xC77C,
    0xC77D,
    0xC77E,
    0xC783,
    0xC784,
    0xC785,
    0xC787,
    0xC788,
    0xC789,
    0xC78A,
    0xC78E,
    0xC790,
    0xC791,
    0xC794,
    0xC796,
    0xC797,
    0xC798,
    0xC79A,
    0xC7A0,
    0xC7A1,
    0xC7A3,
    0xC7A4,
    0xC7A5,
    0xC7A6,
    0xC7AC,
    0xC7AD,
    0xC7B0,
    0xC7B4,
    0xC7BC,
    0xC7BD,
    0xC7BF,
    0xC7C0,
    0xC7C1,
    0xC7C8,
    0xC7C9,
    0xC7CC,
    0xC7CE,
    0xC7D0,
    0xC7D8,
    0xC7DD,
    0xC7E4,
    0xC7E8,
    0xC7EC,
    0xC800,
    0xC801,
    0xC804,
    0xC808,
    0xC80A,
    0xC810,
    0xC811,
    0xC813,
    0xC815,
    0xC816,
    0xC81C,
    0xC81D,
    0xC820,
    0xC824,
    0xC82C,
    0xC82D,
    0xC82F,
    0xC831,
    0xC838,
    0xC83C,
    0xC840,
    0xC848,
    0xC849,
    0xC84C,
    0xC84D,
    0xC854,
    0xC870,
    0xC871,
    0xC874,
    0xC878,
    0xC87A,
    0xC880,
    0xC881,
    0xC883,
    0xC885,
    0xC886,
    0xC887,
    0xC88B,
    0xC88C,
    0xC88D,
    0xC894,
    0xC89D,
    0xC89F,
    0xC8A1,
    0xC8A8,
    0xC8BC,
    0xC8BD,
    0xC8C4,
    0xC8C8,
    0xC8CC,
    0xC8D4,
    0xC8D5,
    0xC8D7,
    0xC8D9,
    0xC8E0,
    0xC8E1,
    0xC8E4,
    0xC8F5,
    0xC8FC,
    0xC8FD,
    0xC900,
    0xC904,
    0xC905,
    0xC906,
    0xC90C,
    0xC90D,
    0xC90F,
    0xC911,
    0xC918,
    0xC92C,
    0xC934,
    0xC950,
    0xC951,
    0xC954,
    0xC958,
    0xC960,
    0xC961,
    0xC963,
    0xC96C,
    0xC970,
    0xC974,
    0xC97C,
    0xC988,
    0xC989,
    0xC98C,
    0xC990,
    0xC998,
    0xC999,
    0xC99B,
    0xC99D,
    0xC9C0,
    0xC9C1,
    0xC9C4,
    0xC9C7,
    0xC9C8,
    0xC9CA,
    0xC9D0,
    0xC9D1,
    0xC9D3,
    0xC9D5,
    0xC9D6,
    0xC9D9,
    0xC9DA,
    0xC9DC,
    0xC9DD,
    0xC9E0,
    0xC9E2,
    0xC9E4,
    0xC9E7,
    0xC9EC,
    0xC9ED,
    0xC9EF,
    0xC9F0,
    0xC9F1,
    0xC9F8,
    0xC9F9,
    0xC9FC,
    0xCA00,
    0xCA08,
    0xCA09,
    0xCA0B,
    0xCA0C,
    0xCA0D,
    0xCA14,
    0xCA18,
    0xCA29,
    0xCA4C,
    0xCA4D,
    0xCA50,
    0xCA54,
    0xCA5C,
    0xCA5D,
    0xCA5F,
    0xCA60,
    0xCA61,
    0xCA68,
    0xCA7D,
    0xCA84,
    0xCA98,
    0xCABC,
    0xCABD,
    0xCAC0,
    0xCAC4,
    0xCACC,
    0xCACD,
    0xCACF,
    0xCAD1,
    0xCAD3,
    0xCAD8,
    0xCAD9,
    0xCAE0,
    0xCAEC,
    0xCAF4,
    0xCB08,
    0xCB10,
    0xCB14,
    0xCB18,
    0xCB20,
    0xCB21,
    0xCB41,
    0xCB48,
    0xCB49,
    0xCB4C,
    0xCB50,
    0xCB58,
    0xCB59,
    0xCB5D,
    0xCB64,
    0xCB78,
    0xCB79,
    0xCB9C,
    0xCBB8,
    0xCBD4,
    0xCBE4,
    0xCBE7,
    0xCBE9,
    0xCC0C,
    0xCC0D,
    0xCC10,
    0xCC14,
    0xCC1C,
    0xCC1D,
    0xCC21,
    0xCC22,
    0xCC27,
    0xCC28,
    0xCC29,
    0xCC2C,
    0xCC2E,
    0xCC30,
    0xCC38,
    0xCC39,
    0xCC3B,
    0xCC3C,
    0xCC3D,
    0xCC3E,
    0xCC44,
    0xCC45,
    0xCC48,
    0xCC4C,
    0xCC54,
    0xCC55,
    0xCC57,
    0xCC58,
    0xCC59,
    0xCC60,
    0xCC64,
    0xCC66,
    0xCC68,
    0xCC70,
    0xCC75,
    0xCC98,
    0xCC99,
    0xCC9C,
    0xCCA0,
    0xCCA8,
    0xCCA9,
    0xCCAB,
    0xCCAC,
    0xCCAD,
    0xCCB4,
    0xCCB5,
    0xCCB8,
    0xCCBC,
    0xCCC4,
    0xCCC5,
    0xCCC7,
    0xCCC9,
    0xCCD0,
    0xCCD4,
    0xCCE4,
    0xCCEC,
    0xCCF0,
    0xCD01,
    0xCD08,
    0xCD09,
    0xCD0C,
    0xCD10,
    0xCD18,
    0xCD19,
    0xCD1B,
    0xCD1D,
    0xCD24,
    0xCD28,
    0xCD2C,
    0xCD39,
    0xCD5C,
    0xCD60,
    0xCD64,
    0xCD6C,
    0xCD6D,
    0xCD6F,
    0xCD71,
    0xCD78,
    0xCD88,
    0xCD94,
    0xCD95,
    0xCD98,
    0xCD9C,
    0xCDA4,
    0xCDA5,
    0xCDA7,
    0xCDA9,
    0xCDB0,
    0xCDC4,
    0xCDCC,
    0xCDD0,
    0xCDE8,
    0xCDEC,
    0xCDF0,
    0xCDF8,
    0xCDF9,
    0xCDFB,
    0xCDFD,
    0xCE04,
    0xCE08,
    0xCE0C,
    0xCE14,
    0xCE19,
    0xCE20,
    0xCE21,
    0xCE24,
    0xCE28,
    0xCE30,
    0xCE31,
    0xCE33,
    0xCE35,
    0xCE58,
    0xCE59,
    0xCE5C,
    0xCE5F,
    0xCE60,
    0xCE61,
    0xCE68,
    0xCE69,
    0xCE6B,
    0xCE6D,
    0xCE74,
    0xCE75,
    0xCE78,
    0xCE7C,
    0xCE84,
    0xCE85,
    0xCE87,
    0xCE89,
    0xCE90,
    0xCE91,
    0xCE94,
    0xCE98,
    0xCEA0,
    0xCEA1,
    0xCEA3,
    0xCEA4,
    0xCEA5,
    0xCEAC,
    0xCEAD,
    0xCEC1,
    0xCEE4,
    0xCEE5,
    0xCEE8,
    0xCEEB,
    0xCEEC,
    0xCEF4,
    0xCEF5,
    0xCEF7,
    0xCEF8,
    0xCEF9,
    0xCF00,
    0xCF01,
    0xCF04,
    0xCF08,
    0xCF10,
    0xCF11,
    0xCF13,
    0xCF15,
    0xCF1C,
    0xCF20,
    0xCF24,
    0xCF2C,
    0xCF2D,
    0xCF2F,
    0xCF30,
    0xCF31,
    0xCF38,
    0xCF54,
    0xCF55,
    0xCF58,
    0xCF5C,
    0xCF64,
    0xCF65,
    0xCF67,
    0xCF69,
    0xCF70,
    0xCF71,
    0xCF74,
    0xCF78,
    0xCF80,
    0xCF85,
    0xCF8C,
    0xCFA1,
    0xCFA8,
    0xCFB0,
    0xCFC4,
    0xCFE0,
    0xCFE1,
    0xCFE4,
    0xCFE8,
    0xCFF0,
    0xCFF1,
    0xCFF3,
    0xCFF5,
    0xCFFC,
    0xD000,
    0xD004,
    0xD011,
    0xD018,
    0xD02D,
    0xD034,
    0xD035,
    0xD038,
    0xD03C,
    0xD044,
    0xD045,
    0xD047,
    0xD049,
    0xD050,
    0xD054,
    0xD058,
    0xD060,
    0xD06C,
    0xD06D,
    0xD070,
    0xD074,
    0xD07C,
    0xD07D,
    0xD081,
    0xD0A4,
    0xD0A5,
    0xD0A8,
    0xD0AC,
    0xD0B4,
    0xD0B5,
    0xD0B7,
    0xD0B9,
    0xD0C0,
    0xD0C1,
    0xD0C4,
    0xD0C8,
    0xD0C9,
    0xD0D0,
    0xD0D1,
    0xD0D3,
    0xD0D4,
    0xD0D5,
    0xD0DC,
    0xD0DD,
    0xD0E0,
    0xD0E4,
    0xD0EC,
    0xD0ED,
    0xD0EF,
    0xD0F0,
    0xD0F1,
    0xD0F8,
    0xD10D,
    0xD130,
    0xD131,
    0xD134,
    0xD138,
    0xD13A,
    0xD140,
    0xD141,
    0xD143,
    0xD144,
    0xD145,
    0xD14C,
    0xD14D,
    0xD150,
    0xD154,
    0xD15C,
    0xD15D,
    0xD15F,
    0xD161,
    0xD168,
    0xD16C,
    0xD17C,
    0xD184,
    0xD188,
    0xD1A0,
    0xD1A1,
    0xD1A4,
    0xD1A8,
    0xD1B0,
    0xD1B1,
    0xD1B3,
    0xD1B5,
    0xD1BA,
    0xD1BC,
    0xD1C0,
    0xD1D8,
    0xD1F4,
    0xD1F8,
    0xD207,
    0xD209,
    0xD210,
    0xD22C,
    0xD22D,
    0xD230,
    0xD234,
    0xD23C,
    0xD23D,
    0xD23F,
    0xD241,
    0xD248,
    0xD25C,
    0xD264,
    0xD280,
    0xD281,
    0xD284,
    0xD288,
    0xD290,
    0xD291,
    0xD295,
    0xD29C,
    0xD2A0,
    0xD2A4,
    0xD2AC,
    0xD2B1,
    0xD2B8,
    0xD2B9,
    0xD2BC,
    0xD2BF,
    0xD2C0,
    0xD2C2,
    0xD2C8,
    0xD2C9,
    0xD2CB,
    0xD2D4,
    0xD2D8,
    0xD2DC,
    0xD2E4,
    0xD2E5,
    0xD2F0,
    0xD2F1,
    0xD2F4,
    0xD2F8,
    0xD300,
    0xD301,
    0xD303,
    0xD305,
    0xD30C,
    0xD30D,
    0xD30E,
    0xD310,
    0xD314,
    0xD316,
    0xD31C,
    0xD31D,
    0xD31F,
    0xD320,
    0xD321,
    0xD325,
    0xD328,
    0xD329,
    0xD32C,
    0xD330,
    0xD338,
    0xD339,
    0xD33B,
    0xD33C,
    0xD33D,
    0xD344,
    0xD345,
    0xD37C,
    0xD37D,
    0xD380,
    0xD384,
    0xD38C,
    0xD38D,
    0xD38F,
    0xD390,
    0xD391,
    0xD398,
    0xD399,
    0xD39C,
    0xD3A0,
    0xD3A8,
    0xD3A9,
    0xD3AB,
    0xD3AD,
    0xD3B4,
    0xD3B8,
    0xD3BC,
    0xD3C4,
    0xD3C5,
    0xD3C8,
    0xD3C9,
    0xD3D0,
    0xD3D8,
    0xD3E1,
    0xD3E3,
    0xD3EC,
    0xD3ED,
    0xD3F0,
    0xD3F4,
    0xD3FC,
    0xD3FD,
    0xD3FF,
    0xD401,
    0xD408,
    0xD41D,
    0xD440,
    0xD444,
    0xD45C,
    0xD460,
    0xD464,
    0xD46D,
    0xD46F,
    0xD478,
    0xD479,
    0xD47C,
    0xD47F,
    0xD480,
    0xD482,
    0xD488,
    0xD489,
    0xD48B,
    0xD48D,
    0xD494,
    0xD4A9,
    0xD4CC,
    0xD4D0,
    0xD4D4,
    0xD4DC,
    0xD4DF,
    0xD4E8,
    0xD4EC,
    0xD4F0,
    0xD4F8,
    0xD4FB,
    0xD4FD,
    0xD504,
    0xD508,
    0xD50C,
    0xD514,
    0xD515,
    0xD517,
    0xD53C,
    0xD53D,
    0xD540,
    0xD544,
    0xD54C,
    0xD54D,
    0xD54F,
    0xD551,
    0xD558,
    0xD559,
    0xD55C,
    0xD560,
    0xD565,
    0xD568,
    0xD569,
    0xD56B,
    0xD56D,
    0xD574,
    0xD575,
    0xD578,
    0xD57C,
    0xD584,
    0xD585,
    0xD587,
    0xD588,
    0xD589,
    0xD590,
    0xD5A5,
    0xD5C8,
    0xD5C9,
    0xD5CC,
    0xD5D0,
    0xD5D2,
    0xD5D8,
    0xD5D9,
    0xD5DB,
    0xD5DD,
    0xD5E4,
    0xD5E5,
    0xD5E8,
    0xD5EC,
    0xD5F4,
    0xD5F5,
    0xD5F7,
    0xD5F9,
    0xD600,
    0xD601,
    0xD604,
    0xD608,
    0xD610,
    0xD611,
    0xD613,
    0xD614,
    0xD615,
    0xD61C,
    0xD620,
    0xD624,
    0xD62D,
    0xD638,
    0xD639,
    0xD63C,
    0xD640,
    0xD645,
    0xD648,
    0xD649,
    0xD64B,
    0xD64D,
    0xD651,
    0xD654,
    0xD655,
    0xD658,
    0xD65C,
    0xD667,
    0xD669,
    0xD670,
    0xD671,
    0xD674,
    0xD683,
    0xD685,
    0xD68C,
    0xD68D,
    0xD690,
    0xD694,
    0xD69D,
    0xD69F,
    0xD6A1,
    0xD6A8,
    0xD6AC,
    0xD6B0,
    0xD6B9,
    0xD6BB,
    0xD6C4,
    0xD6C5,
    0xD6C8,
    0xD6CC,
    0xD6D1,
    0xD6D4,
    0xD6D7,
    0xD6D9,
    0xD6E0,
    0xD6E4,
    0xD6E8,
    0xD6F0,
    0xD6F5,
    0xD6FC,
    0xD6FD,
    0xD700,
    0xD704,
    0xD711,
    0xD718,
    0xD719,
    0xD71C,
    0xD720,
    0xD728,
    0xD729,
    0xD72B,
    0xD72D,
    0xD734,
    0xD735,
    0xD738,
    0xD73C,
    0xD744,
    0xD747,
    0xD749,
    0xD750,
    0xD751,
    0xD754,
    0xD756,
    0xD757,
    0xD758,
    0xD759,
    0xD760,
    0xD761,
    0xD763,
    0xD765,
    0xD769,
    0xD76C,
    0xD770,
    0xD774,
    0xD77C,
    0xD77D,
    0xD781,
    0xD788,
    0xD789,
    0xD78C,
    0xD790,
    0xD798,
    0xD799,
    0xD79B,
    0xD79D,
}

# Set based on the character set of the Hahmlet Fonts Project (Version 1.002)
# https://github.com/hyper-type/hahmlet#hahmlet-fonts-project
OTHER_COMMON_HANGUL_SYLLABLES_CODEPOINTS = {
    0xAC0B,
    0xAC23,
    0xAC65,
    0xAC82,
    0xAC8D,
    0xACB7,
    0xACBF,
    0xACD8,
    0xACF9,
    0xAD10,
    0xAD12,
    0xAD22,
    0xAD60,
    0xAD65,
    0xAD83,
    0xADB8,
    0xADD5,
    0xADEC,
    0xADED,
    0xAE02,
    0xAE07,
    0xAE0F,
    0xAE11,
    0xAE13,
    0xAE18,
    0xAE29,
    0xAE44,
    0xAE6F,
    0xAE86,
    0xAE8D,
    0xAEA0,
    0xAEA4,
    0xAED3,
    0xAED5,
    0xAF09,
    0xAF33,
    0xAF45,
    0xAF4C,
    0xAF68,
    0xAF78,
    0xAFCF,
    0xAFD8,
    0xB030,
    0xB03C,
    0xB060,
    0xB064,
    0xB090,
    0xB0BB,
    0xB0D7,
    0xB0E1,
    0xB0E3,
    0xB0EC,
    0xB10F,
    0xB111,
    0xB122,
    0xB12B,
    0xB13E,
    0xB147,
    0xB171,
    0xB181,
    0xB191,
    0xB1A5,
    0xB1B0,
    0xB1C3,
    0xB1C4,
    0xB1CD,
    0xB1E1,
    0xB1F8,
    0xB20D,
    0xB21D,
    0xB24F,
    0xB27B,
    0xB297,
    0xB2A7,
    0xB2AB,
    0xB2BC,
    0xB2BF,
    0xB2C1,
    0xB2CF,
    0xB2DE,
    0xB2E0,
    0xB2FD,
    0xB316,
    0xB320,
    0xB331,
    0xB36F,
    0xB40F,
    0xB42C,
    0xB42D,
    0xB457,
    0xB45A,
    0xB499,
    0xB4B8,
    0xB4E5,
    0xB4E7,
    0xB4FC,
    0xB500,
    0xB52B,
    0xB52E,
    0xB532,
    0xB537,
    0xB54E,
    0xB568,
    0xB584,
    0xB588,
    0xB594,
    0xB599,
    0xB620,
    0xB621,
    0xB623,
    0xB62D,
    0xB630,
    0xB647,
    0xB65C,
    0xB6A7,
    0xB6AF,
    0xB6B8,
    0xB704,
    0xB70C,
    0xB733,
    0xB73D,
    0xB73E,
    0xB743,
    0xB761,
    0xB783,
    0xB79F,
    0xB7B0,
    0xB7B2,
    0xB7C5,
    0xB7D0,
    0xB7D4,
    0xB7F2,
    0xB7F3,
    0xB81C,
    0xB821,
    0xB82B,
    0xB863,
    0xB879,
    0xB894,
    0xB908,
    0xB924,
    0xB98F,
    0xB990,
    0xB994,
    0xB9C0,
    0xB9C6,
    0xB9D4,
    0xB9DC,
    0xB9DF,
    0xB9E2,
    0xB9EB,
    0xB9FB,
    0xB9FD,
    0xBA04,
    0xBA10,
    0xBA4C,
    0xBA5B,
    0xBA6B,
    0xBA77,
    0xBA90,
    0xBAAF,
    0xBAB1,
    0xBAE0,
    0xBAF4,
    0xBB0C,
    0xBB64,
    0xBB65,
    0xBB7F,
    0xBB8A,
    0xBB99,
    0xBB9B,
    0xBBB9,
    0xBBC1,
    0xBBD1,
    0xBBD5,
    0xBBDC,
    0xBBE0,
    0xBBED,
    0xBBF1,
    0xBC23,
    0xBC28,
    0xBC2B,
    0xBC32,
    0xBC37,
    0xBC5C,
    0xBC5F,
    0xBC61,
    0xBC98,
    0xBC9D,
    0xBC9F,
    0xBCCC,
    0xBCD3,
    0xBCFB,
    0xBCFD,
    0xBCFE,
    0xBCFF,
    0xBD23,
    0xBD25,
    0xBD5C,
    0xBD74,
    0xBD94,
    0xBD9D,
    0xBDA0,
    0xBDAD,
    0xBDB4,
    0xBDC1,
    0xBDE5,
    0xBE01,
    0xBE15,
    0xBE21,
    0xBE28,
    0xBE29,
    0xBE30,
    0xBE3D,
    0xBE4B,
    0xBE58,
    0xBE67,
    0xBE69,
    0xBE7E,
    0xBE9C,
    0xBED9,
    0xBEF0,
    0xBF0C,
    0xBF53,
    0xBFB1,
    0xBFD5,
    0xBFDD,
    0xC020,
    0xC06D,
    0xC0C6,
    0xC0CF,
    0xC0FE,
    0xC101,
    0xC13F,
    0xC151,
    0xC171,
    0xC180,
    0xC181,
    0xC193,
    0xC1F4,
    0xC1F5,
    0xC216,
    0xC28C,
    0xC2AB,
    0xC2B2,
    0xC2C0,
    0xC2C1,
    0xC2E5,
    0xC2F0,
    0xC2F3,
    0xC2FF,
    0xC330,
    0xC331,
    0xC382,
    0xC385,
    0xC394,
    0xC3A0,
    0xC3A4,
    0xC3B5,
    0xC3BC,
    0xC3DA,
    0xC3FC,
    0xC40B,
    0xC440,
    0xC45D,
    0xC490,
    0xC4A0,
    0xC4AC,
    0xC4D4,
    0xC503,
    0xC52B,
    0xC53C,
    0xC53F,
    0xC54B,
    0xC54F,
    0xC55C,
    0xC590,
    0xC5AC,
    0xC5AD,
    0xC5BA,
    0xC5CF,
    0xC5E4,
    0xC60F,
    0xC61D,
    0xC626,
    0xC62B,
    0xC62F,
    0xC64E,
    0xC658,
    0xC66D,
    0xC670,
    0xC68C,
    0xC6B7,
    0xC6C2,
    0xC6C7,
    0xC6DF,
    0xC6FB,
    0xC718,
    0xC73E,
    0xC743,
    0xC747,
    0xC74E,
    0xC769,
    0xC76C,
    0xC76D,
    0xC77B,
    0xC78C,
    0xC78D,
    0xC7F5,
    0xC802,
    0xC807,
    0xC809,
    0xC80B,
    0xC814,
    0xC839,
    0xC842,
    0xC84B,
    0xC890,
    0xC8A6,
    0xC8AC,
    0xC8C5,
    0xC96D,
    0xC981,
    0xC992,
    0xC9A4,
    0xC9D4,
    0xC9D7,
    0xC9E3,
    0xC9F2,
    0xCA24,
    0xCA30,
    0xCA70,
    0xCAC3,
    0xCAD2,
    0xCADC,
    0xCB27,
    0xCB2C,
    0xCB2D,
    0xCB32,
    0xCB5B,
    0xCB80,
    0xCB93,
    0xCBCD,
    0xCBD5,
    0xCC13,
    0xCC1F,
    0xCC26,
    0xCC3F,
    0xCCCA,
    0xCCE1,
    0xCCE5,
    0xCD23,
    0xCD25,
    0xCD35,
    0xCD40,
    0xCDCD,
    0xCDD4,
    0xCE05,
    0xCE29,
    0xCE3C,
    0xCE62,
    0xCE6C,
    0xCE6E,
    0xCE70,
    0xCE7B,
    0xCEA8,
    0xCEB0,
    0xCEC4,
    0xCEFD,
    0xCF14,
    0xCF18,
    0xCF19,
    0xCF5B,
    0xCF83,
    0xCFC8,
    0xCFDC,
    0xCFFD,
    0xD00C,
    0xD01C,
    0xD020,
    0xD072,
    0xD084,
    0xD088,
    0xD0B8,
    0xD0C7,
    0xD114,
    0xD13B,
    0xD166,
    0xD170,
    0xD17B,
    0xD1A7,
    0xD1FB,
    0xD225,
    0xD236,
    0xD277,
    0xD2CD,
    0xD2D1,
    0xD311,
    0xD313,
    0xD324,
    0xD32F,
    0xD35D,
    0xD396,
    0xD3B5,
    0xD3D4,
    0xD405,
    0xD409,
    0xD519,
    0xD520,
    0xD535,
    0xD550,
    0xD564,
    0xD570,
    0xD571,
    0xD573,
    0xD57B,
    0xD58B,
    0xD58F,
    0xD594,
    0xD5A3,
    0xD5AC,
    0xD5C0,
    0xD5D7,
    0xD5E0,
    0xD5E1,
    0xD5E3,
    0xD5F8,
    0xD5FF,
    0xD665,
    0xD668,
    0xD6BD,
    0xD6CD,
    0xD6D0,
    0xD6D3,
    0xD6D5,
    0xD75D,
    0xD79C,
    0xD7A3,
}

# "Hangul Syllables" Unicode v2.0 block (Full range: U+AC00..U+D7AF)
# https://en.wikipedia.org/wiki/Hangul_Syllables
ALL_HANGUL_SYLLABLES_CODEPOINTS = set(range(0xAC00, 0xD7A3 + 1))

LESS_COMMON_HANGUL_SYLLABLES_CODEPOINTS = (
    ALL_HANGUL_SYLLABLES_CODEPOINTS
    - MODERN_HANGUL_SYLLABLES_CODEPOINTS
    - OTHER_COMMON_HANGUL_SYLLABLES_CODEPOINTS
)

assert len(ALL_HANGUL_SYLLABLES_CODEPOINTS) == 11172
assert len(MODERN_HANGUL_SYLLABLES_CODEPOINTS) == 2350
assert len(OTHER_COMMON_HANGUL_SYLLABLES_CODEPOINTS) == 438  # 438 + 2350 = 2788
assert len(LESS_COMMON_HANGUL_SYLLABLES_CODEPOINTS) == 8384
