#!/usr/bin/env python3

# Copyright (c) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License.

import os

import numpy as np
import rasterio
from rasterio.crs import CRS
from rasterio.transform import Affine

SIZE = 128

np.random.seed(0)

# from Chesapeak data.py
wkt = """
PROJCS["USA_Contiguous_Albers_Equal_Area_Conic_USGS_version",
    GEOGCS["NAD83",
        DATUM["North_American_Datum_1983",
            SPHEROID["GRS 1980",6378137,298.257222101004,
                AUTHORITY["EPSG","7019"]],
            AUTHORITY["EPSG","6269"]],
        PRIMEM["Greenwich",0],
        UNIT["degree",0.0174532925199433,
            AUTHORITY["EPSG","9122"]],
        AUTHORITY["EPSG","4269"]],
    PROJECTION["Albers_Conic_Equal_Area"],
    PARAMETER["latitude_of_center",23],
    PARAMETER["longitude_of_center",-96],
    PARAMETER["standard_parallel_1",29.5],
    PARAMETER["standard_parallel_2",45.5],
    PARAMETER["false_easting",0],
    PARAMETER["false_northing",0],
    UNIT["metre",1,
        AUTHORITY["EPSG","9001"]],
    AXIS["Easting",EAST],
    AXIS["Northing",NORTH]]
"""


def create_file(path: str, dtype: str, num_channels: int) -> None:
    profile = {}
    profile['driver'] = 'GTiff'
    profile['dtype'] = dtype
    profile['count'] = num_channels
    profile['crs'] = CRS.from_wkt(wkt)
    profile['transform'] = Affine(
        1.0, 0.0, 1303555.0000000005, 0.0, -1.0, 2535064.999999998
    )
    profile['height'] = SIZE
    profile['width'] = SIZE
    profile['compress'] = 'lzw'
    profile['predictor'] = 2

    if 'float' in profile['dtype']:
        Z = np.random.randn(SIZE, SIZE).astype(profile['dtype'])
    else:
        Z = np.random.randint(
            np.iinfo(profile['dtype']).max, size=(SIZE, SIZE), dtype=profile['dtype']
        )

    with rasterio.open(path, 'w', **profile) as src:
        for i in range(1, profile['count'] + 1):
            src.write(Z, i)


if __name__ == '__main__':
    filenames = ['m_3807511_ne_18_060_20181104.tif', 'm_3807511_ne_18_060_20190605.tif']

    for f in filenames:
        create_file(os.path.join(os.getcwd(), f), 'uint8', 4)
