// Copyright (c) Microsoft Corporation. All rights reserved. Licensed under the MIT license.
// See LICENSE in the project root for license information.

import {
  type ExecSyncOptionsWithBufferEncoding,
  type SpawnSyncOptionsWithBufferEncoding,
  execSync,
  spawnSync
} from 'child_process';
import * as process from 'process';
import * as fs from 'fs';

interface IEslintBulkConfigurationJson {
  /**
   * `@rushtack/eslint`-bulk should report an error if its package.json is older than this number
   */
  minCliVersion: string;
  /**
   * `@rushtack/eslint-bulk` will invoke this entry point
   */
  cliEntryPoint: string;
}

function findPatchPath(): string {
  const candidatePaths: string[] = [`${process.cwd()}/.eslintrc.js`, `${process.cwd()}/.eslintrc.cjs`];
  let eslintrcPath: string | undefined;
  for (const candidatePath of candidatePaths) {
    if (fs.existsSync(candidatePath)) {
      eslintrcPath = candidatePath;
      break;
    }
  }

  if (!eslintrcPath) {
    console.error(
      '@rushstack/eslint-bulk: Please run this command from the directory that contains .eslintrc.js or .eslintrc.cjs'
    );
    process.exit(1);
  }

  const env: NodeJS.ProcessEnv = { ...process.env, _RUSHSTACK_ESLINT_BULK_DETECT: 'true' };

  let eslintPackageJsonPath: string | undefined;
  try {
    eslintPackageJsonPath = require.resolve('eslint/package.json', { paths: [process.cwd()] });
  } catch (e) {
    if (e.code !== 'MODULE_NOT_FOUND') {
      throw e;
    }
  }

  let eslintBinPath: string | undefined;
  if (eslintPackageJsonPath) {
    eslintPackageJsonPath = eslintPackageJsonPath.replace(/\\/g, '/');
    const packagePath: string = eslintPackageJsonPath.substring(0, eslintPackageJsonPath.lastIndexOf('/'));
    const { bin: { eslint: relativeEslintBinPath } = {} }: { bin?: Record<string, string> } = require(
      eslintPackageJsonPath
    );
    if (relativeEslintBinPath) {
      eslintBinPath = `${packagePath}/${relativeEslintBinPath}`;
    } else {
      console.warn(
        `@rushstack/eslint-bulk: The eslint package resolved at "${packagePath}" does not contain an eslint bin path. ` +
          'Attempting to use a globally-installed eslint instead.'
      );
    }
  } else {
    console.log(
      '@rushstack/eslint-bulk: Unable to resolve the eslint package as a dependency of the current project. ' +
        'Attempting to use a globally-installed eslint instead.'
    );
  }

  const eslintArgs: string[] = ['--stdin', '--config'];
  const spawnOrExecOptions: SpawnSyncOptionsWithBufferEncoding & ExecSyncOptionsWithBufferEncoding = {
    env,
    input: '',
    stdio: 'pipe'
  };
  let runEslintFn: () => Buffer;
  if (eslintBinPath) {
    runEslintFn = () =>
      spawnSync(process.argv0, [eslintBinPath, ...eslintArgs, eslintrcPath], spawnOrExecOptions).stdout;
  } else {
    // Try to use a globally-installed eslint if a local package was not found
    runEslintFn = () => execSync(`eslint ${eslintArgs.join(' ')} "${eslintrcPath}"`, spawnOrExecOptions);
  }

  let stdout: Buffer;
  try {
    stdout = runEslintFn();
  } catch (e) {
    console.error('@rushstack/eslint-bulk: Error finding patch path: ' + e.message);
    process.exit(1);
  }

  const startDelimiter: string = 'RUSHSTACK_ESLINT_BULK_START';
  const endDelimiter: string = 'RUSHSTACK_ESLINT_BULK_END';

  const regex: RegExp = new RegExp(`${startDelimiter}(.*?)${endDelimiter}`);
  const match: RegExpMatchArray | null = stdout.toString().match(regex);

  if (match) {
    // The configuration data will look something like this:
    //
    // RUSHSTACK_ESLINT_BULK_START{"minCliVersion":"0.0.0","cliEntryPoint":"path/to/eslint-bulk.js"}RUSHSTACK_ESLINT_BULK_END
    const configurationJson: string = match[1].trim();
    let configuration: IEslintBulkConfigurationJson;
    try {
      configuration = JSON.parse(configurationJson);
      if (!configuration.minCliVersion || !configuration.cliEntryPoint) {
        throw new Error('Required field is missing');
      }
    } catch (e) {
      console.error('@rushstack/eslint-bulk: Error parsing patch configuration object:' + e.message);
      process.exit(1);
    }

    const myVersion: string = require('../package.json').version;
    const myVersionParts: number[] = myVersion.split('.').map((x) => parseInt(x, 10));
    const minVersion: string = configuration.minCliVersion;
    const minVersionParts: number[] = minVersion.split('.').map((x) => parseInt(x, 10));
    if (
      myVersionParts.length !== 3 ||
      minVersionParts.length !== 3 ||
      myVersionParts.some((x) => isNaN(x)) ||
      minVersionParts.some((x) => isNaN(x))
    ) {
      console.error(`@rushstack/eslint-bulk: Unable to compare versions "${myVersion}" and "${minVersion}"`);
      process.exit(1);
    }

    for (let i: number = 0; i < 3; ++i) {
      if (myVersionParts[i] > minVersionParts[i]) {
        break;
      }
      if (myVersionParts[i] < minVersionParts[i]) {
        console.error(
          `@rushstack/eslint-bulk: The @rushstack/eslint-bulk version ${myVersion} is too old;` +
            ` please upgrade to ${minVersion} or newer.`
        );
        process.exit(1);
      }
    }

    return configuration.cliEntryPoint;
  }

  console.error(
    '@rushstack/eslint-bulk: Error finding patch path. Are you sure the package you are in has @rushstack/eslint-patch as a direct or indirect dependency?'
  );
  process.exit(1);
}

const patchPath: string = findPatchPath();
try {
  require(patchPath);
} catch (e) {
  console.error(`@rushstack/eslint-bulk: Error running patch at ${patchPath}:\n` + e.message);
  process.exit(1);
}
