"use strict";

import * as del from "del";
import * as path from "path";
import * as tl from "azure-pipelines-task-lib/task";
import * as tr from "azure-pipelines-task-lib/toolrunner";
import * as yaml from "js-yaml";
import * as DockerComposeUtils from "./dockercomposeutils";

import ContainerConnection from "azure-pipelines-tasks-docker-common/containerconnection"
import AuthenticationToken from "azure-pipelines-tasks-docker-common/registryauthenticationprovider/registryauthenticationtoken"
import * as Utils from "./utils";

export default class DockerComposeConnection extends ContainerConnection {
    private dockerComposePath: string;
    private dockerComposeFile: string;
    private dockerComposeVersion: string;
    private additionalDockerComposeFiles: string[];
    private requireAdditionalDockerComposeFiles: boolean;
    private projectName: string;
    private finalComposeFile: string;

    constructor() {
        super();
        this.setDockerComposePath();
        this.dockerComposeFile = DockerComposeUtils.findDockerFile(tl.getInput("dockerComposeFile", true), tl.getInput("cwd"));
        if (!this.dockerComposeFile) {
            throw new Error("No Docker Compose file matching " + tl.getInput("dockerComposeFile") + " was found.");
        }
        this.dockerComposeVersion = "2";
        this.additionalDockerComposeFiles = tl.getDelimitedInput("additionalDockerComposeFiles", "\n");
        this.requireAdditionalDockerComposeFiles = tl.getBoolInput("requireAdditionalDockerComposeFiles");
        this.projectName = tl.getInput("projectName");
        this.validateProjectNameDockerComposeV2();
    }

    public open(hostEndpoint?: string, authenticationToken?: AuthenticationToken): any {
        super.open(hostEndpoint, authenticationToken);

        if (this.hostUrl) {
            process.env["DOCKER_HOST"] = this.hostUrl;
            process.env["DOCKER_TLS_VERIFY"] = "1";
            process.env["DOCKER_CERT_PATH"] = this.certsDir;
        }

        tl.getDelimitedInput("dockerComposeFileArgs", "\n").forEach(envVar => {
            var tokens = envVar.split("=");
            if (tokens.length < 2) {
                throw new Error("Environment variable '" + envVar + "' is invalid.");
            }
            process.env[tokens[0].trim()] = tokens.slice(1).join("=").trim();
        });

        return this.getImages(true).then(images => {
            var qualifyImageNames = tl.getBoolInput("qualifyImageNames");
            if (!qualifyImageNames) {
                return;
            }
            var agentDirectory = tl.getVariable("Agent.HomeDirectory");
            this.finalComposeFile = path.join(agentDirectory, Utils.getFinalComposeFileName());
            var services = {};
            if (qualifyImageNames) {
                for (var serviceName in images) {
                    images[serviceName] = this.getQualifiedImageNameIfRequired(images[serviceName]);
                }
            }
            for (var serviceName in images) {
                services[serviceName] = {
                    image: images[serviceName]
                };
            }
            Utils.writeFileSync(this.finalComposeFile, yaml.safeDump({
                version: this.dockerComposeVersion,
                services: services
            }, { lineWidth: -1 } as any));
        });
    }

    public async execCommandWithLogging(command, options?: tr.IExecOptions): Promise<string> {
        // setup variable to store the command output
        let output = "";
        command.on("stdout", data => {
            output += data;
        });

        command.on("stderr", data => {
            output += data;
        });
        await this.execCommand(command, options);
        return output || '\n';
    }

    public createComposeCommand(): tr.ToolRunner {
        var command = tl.tool(this.dockerComposePath);

        if (!tl.getInput('dockerComposePath') && !this.dockerComposePath.includes("docker-compose")) {
            command.arg("compose");
            process.env["COMPOSE_COMPATIBILITY"] = "true";
        }

        command.arg(["-f", this.dockerComposeFile]);
        var basePath = path.dirname(this.dockerComposeFile);
        this.additionalDockerComposeFiles.forEach(file => {
            file = this.resolveAdditionalDockerComposeFilePath(basePath, file);
            if (this.requireAdditionalDockerComposeFiles || tl.exist(file)) {
                command.arg(["-f", file]);
            }
        });
        if (this.finalComposeFile) {
            command.arg(["-f", this.finalComposeFile]);
        }

        if (this.projectName) {
            command.arg(["-p", this.projectName]);
        }
        return command;
    }

    public getCombinedConfig(imageDigestComposeFile?: string): any {
        var command = this.createComposeCommand();
        if (imageDigestComposeFile) {
            command.arg(["-f", imageDigestComposeFile]);
        }
        command.arg("config");
        var result = "";
        command.on("stdout", data => {
            result += data;
        });
        command.on("errline", line => {
            tl.error(line);
        });
        return command.exec({ silent: true } as any).then(() => result);
    }

    public getImages(builtOnly?: boolean): any {
        return this.getCombinedConfig().then(input => {
            var doc = yaml.safeLoad(input);
            if (doc.version) {
                this.dockerComposeVersion = doc.version;
            }
            var projectName = this.projectName;
            if (!projectName) {
                projectName = path.basename(path.dirname(this.dockerComposeFile));
            }
            var images: any = {};
            for (var serviceName in doc.services || {}) {
                var service = doc.services[serviceName];
                var image = service.image;
                if (!image) {
                    image = projectName.toLowerCase().replace(/[^0-9a-z]/g, "") + "_" + serviceName;
                }
                if (!builtOnly || service.build) {
                    images[serviceName] = image;
                }
            }
            return images;
        });
    }

    public getVersion(): string {
        return this.dockerComposeVersion;
    }

    public close(): void {
        if (this.finalComposeFile && tl.exist(this.finalComposeFile)) {
            del.sync(this.finalComposeFile, { force: true });
        }
        super.close();
    }

    private resolveAdditionalDockerComposeFilePath(dockerComposeFolderPath: string, additionalComposeFilePath: string): string {
        if (!path.isAbsolute(additionalComposeFilePath)) {
            additionalComposeFilePath = path.join(dockerComposeFolderPath, additionalComposeFilePath);
        }

        if (!tl.exist(additionalComposeFilePath)) {
            tl.warning(tl.loc('AdditionalDockerComposeFileDoesNotExists', additionalComposeFilePath));
        }

        return additionalComposeFilePath;
    }

    private setDockerComposePath(): void {
        //Priority to docker-compose path provided by user
        this.dockerComposePath = tl.getInput('dockerComposePath');
        if (!this.dockerComposePath) {
            this.dockerComposePath = tl.which("docker-compose");

            if (!this.dockerComposePath) {
                this.dockerComposePath = tl.which("docker");
            }

            if (!this.dockerComposePath) {
                throw new Error("Docker Compose was not found. You can provide the path to docker-compose via 'dockerComposePath' ");
            }
        } else {
            console.log("Using docker-compose from 'dockerComposePath' ");
        }
    }

    private validateProjectNameDockerComposeV2() {
        tl.debug(`Start validating project name ${this.projectName}`);

        if (this.dockerComposePath.includes("docker-compose")) {
            tl.warning(tl.loc("MigrateToDockerComposeV2"));
            return;
        }

        // The regular expression pattern is taken from compose-spec.json
        // https://github.com/compose-spec/compose-spec/blob/864b24c24f7f24a26aa2c2b8a89a82478ce03a32/schema/compose-spec.json#L16
        const regexpPattern = new RegExp("^[a-z0-9][a-z0-9_-]*$");

        if (!regexpPattern.test(this.projectName)) {
            tl.warning(tl.loc("InvalidProjectName", this.projectName));

            console.log(`##vso[telemetry.publish area=TaskHub;feature=DockerComposeV0]${JSON.stringify({
                "SYSTEM_JOBID": tl.getVariable("SYSTEM_JOBID"),
                "SYSTEM_PLANID": tl.getVariable("SYSTEM_PLANID"),
                "BUILD_BUILDID": tl.getVariable("BUILD_BUILDID"),
                "IS_PROJECT_NAME_VALID": false,
            })}`);
            return;
        }
        
        tl.debug("Project name is valid");
    }
}
