/* vim:set ts=2 sw=2 sts=2 et: */
/**
 * \author     Marcus Holland-Moritz (github@mhxnet.de)
 * \copyright  Copyright (c) Marcus Holland-Moritz
 *
 * This file is part of dwarfs.
 *
 * dwarfs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * dwarfs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with dwarfs.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <array>
#include <optional>
#include <sstream>
#include <string_view>

#include <gtest/gtest.h>

#include <filesystem>

#include <fmt/format.h>

#include <range/v3/view/enumerate.hpp>

#include <dwarfs/writer/internal/chmod_transformer.h>

using namespace dwarfs::writer;
namespace fs = std::filesystem;

namespace {

class octal_mode {
 public:
  friend std::ostream& operator<<(std::ostream& os, octal_mode const& mode);

  explicit octal_mode(std::optional<uint16_t> mode)
      : mode_{mode} {}

  bool operator==(octal_mode const& rhs) const { return mode_ == rhs.mode_; }

 private:
  std::optional<uint16_t> mode_;
};

std::ostream& operator<<(std::ostream& os, octal_mode const& mode) {
  if (mode.mode_.has_value()) {
    auto const m = mode.mode_.value();
    os << fmt::format("0{:o} ", m);
    os << (m & uint16_t(fs::perms::set_uid) ? 'U' : '-');
    os << (m & uint16_t(fs::perms::set_gid) ? 'G' : '-');
    os << (m & uint16_t(fs::perms::sticky_bit) ? 'S' : '-');
    os << (m & uint16_t(fs::perms::owner_read) ? 'r' : '-');
    os << (m & uint16_t(fs::perms::owner_write) ? 'w' : '-');
    os << (m & uint16_t(fs::perms::owner_exec) ? 'x' : '-');
    os << (m & uint16_t(fs::perms::group_read) ? 'r' : '-');
    os << (m & uint16_t(fs::perms::group_write) ? 'w' : '-');
    os << (m & uint16_t(fs::perms::group_exec) ? 'x' : '-');
    os << (m & uint16_t(fs::perms::others_read) ? 'r' : '-');
    os << (m & uint16_t(fs::perms::others_write) ? 'w' : '-');
    os << (m & uint16_t(fs::perms::others_exec) ? 'x' : '-');
  } else {
    os << "std::nullopt";
  }
  return os;
}

} // namespace

#define EXPECT_EQ_MODE(a, b) EXPECT_EQ(octal_mode{a}, octal_mode{b})

TEST(chmod_transformer, octal_mode) {
  std::ostringstream os;
  os << octal_mode{07777};
  EXPECT_EQ(os.str(), "07777 UGSrwxrwxrwx");

  os.str("");
  os << octal_mode{std::nullopt};
  EXPECT_EQ(os.str(), "std::nullopt");

  os.str("");
  os << octal_mode{0644};
  EXPECT_EQ(os.str(), "0644 ---rw-r--r--");
}

TEST(chmod_transformer, basic) {
  {
    internal::chmod_transformer ct{"u+x", 0022};
    EXPECT_EQ_MODE(ct.transform(0644, false), 0744);
    EXPECT_EQ_MODE(ct.transform(0755, false), 0755);
    EXPECT_EQ_MODE(ct.transform(0644, true), 0744);
    EXPECT_EQ_MODE(ct.transform(0755, true), 0755);
  }

  {
    internal::chmod_transformer ct{"Fu+x", 0022};
    EXPECT_EQ_MODE(ct.transform(0644, false), 0744);
    EXPECT_EQ_MODE(ct.transform(0755, false), 0755);
    EXPECT_EQ_MODE(ct.transform(0644, true), std::nullopt);
    EXPECT_EQ_MODE(ct.transform(0755, true), std::nullopt);
  }

  {
    internal::chmod_transformer ct{"Du+x", 0022};
    EXPECT_EQ_MODE(ct.transform(0644, false), std::nullopt);
    EXPECT_EQ_MODE(ct.transform(0755, false), std::nullopt);
    EXPECT_EQ_MODE(ct.transform(0644, true), 0744);
    EXPECT_EQ_MODE(ct.transform(0755, true), 0755);
  }
}

namespace {

using mode_type = internal::chmod_transformer::mode_type;

struct random_test {
  std::string_view spec;
  mode_type umask;
  mode_type mode;
  mode_type expected;
};

// clang-format off
constexpr std::array<random_test, 5000> const random_tests{{
    {"ga-X", 0703, 02114, 02004},
    {"ou=u", 0452, 01141, 0141},
    {"u=rws", 0272, 02350, 06650},
    {"a+g", 0337, 0424, 0626},
    {"a+wt", 0321, 00, 01222},
    {"a=r", 0430, 01374, 0444},
    {"o+w-wt", 0343, 074, 074},
    {"o+tx", 0623, 0107, 01107},
    {"ogu=g", 0674, 0601, 00},
    {"o=rXx", 0504, 0645, 0645},
    {"gau-Xrt", 0104, 0672, 0222},
    {"uo+u", 0033, 0300, 0303},
    {"u-s", 0634, 0677, 0677},
    {"4677", 0357, 0432, 04677},
    {"u-o", 0202, 010, 010},
    {"gu=xts", 0024, 0157, 06117},
    {"a-r", 0273, 02237, 02233},
    {"g=r", 0101, 06, 046},
    {"u+t", 0727, 0644, 0644},
    {"ua+t", 0000, 0313, 01313},
    {"ag=xX", 0577, 031, 0111},
    {"ug=tX", 0647, 0743, 0113},
    {"u=xw", 0522, 052, 0352},
    {"u=wx", 0253, 0735, 0335},
    {"oau-wX", 0511, 0446, 0444},
    {"aog+tXr", 0752, 0544, 01555},
    {"u+o", 0232, 0234, 0634},
    {"oa-xw", 0337, 077, 044},
    {"go-rx", 0525, 0261, 0220},
    {"og+r", 0653, 0401, 0445},
    {"ug-Xtw", 0350, 011, 01},
    {"a=rw", 0105, 0442, 0666},
    {"go-swt", 0013, 0777, 0755},
    {"ou-t", 0411, 01072, 072},
    {"ug+g", 0464, 04432, 04732},
    {"ou-Xwr", 0022, 0131, 030},
    {"g=Xs", 0267, 0265, 02215},
    {"a+r-ts", 0466, 0720, 0764},
    {"au-g", 0627, 0751, 0200},
    {"ua=g", 0774, 0344, 0444},
    {"u-wXr", 0137, 0734, 034},
    {"au-w", 0473, 0733, 0511},
    {"oa+x", 0727, 0723, 0733},
    {"og-Xr", 0615, 0742, 0702},
    {"ga+ts", 0437, 0200, 07200},
    {"305", 0431, 0470, 0305},
    {"700", 0562, 0652, 0700},
    {"g=xs", 0263, 0525, 02515},
    {"o-tr", 0105, 0470, 0470},
    {"o+wX", 0523, 02747, 02747},
    {"a=o", 0402, 056, 0666},
    {"uga-g", 0462, 0771, 00},
    {"ua=r", 0156, 0607, 0444},
    {"oa=x", 0664, 00, 0111},
    {"ou=xwX", 0203, 0444, 0343},
    {"716", 0601, 02433, 0716},
    {"o=wt", 0357, 0647, 01642},
    {"g+Xs", 0572, 0316, 02316},
    {"2045", 0662, 0612, 02045},
    {"u=rtw", 0544, 0756, 0656},
    {"ag+Xw", 0233, 0215, 0337},
    {"ou-rw", 0472, 0552, 0150},
    {"ag-t", 0424, 023, 023},
    {"uoa-x", 0762, 04351, 04240},
    {"a-X", 0066, 0660, 0660},
    {"gu-g", 0156, 0575, 05},
    {"g-rt", 0456, 0125, 0125},
    {"ou=tX", 0346, 0323, 01121},
    {"u-st", 0172, 0764, 0764},
    {"o-g", 0357, 0421, 0421},
    {"=066", 0606, 0647, 066},
    {"au-t", 0415, 072, 072},
    {"510", 0527, 0621, 0510},
    {"g-ws", 0601, 0407, 0407},
    {"a=X", 0476, 040, 00},
    {"a+trs", 0032, 0463, 07467},
    {"556", 0707, 0174, 0556},
    {"u-r", 0350, 0325, 0325},
    {"u+rw", 0533, 0461, 0661},
    {"aog-Xs", 0117, 0577, 0466},
    {"ug+xw", 0271, 0761, 0771},
    {"oa=w", 0044, 0737, 0222},
    {"g=wt", 0512, 0650, 0620},
    {"g-swx", 0230, 0710, 0700},
    {"ao+xXs", 0107, 0553, 06553},
    {"gu=rt", 0044, 0163, 0443},
    {"g-twx", 0377, 0353, 0343},
    {"g=rsw", 0234, 04171, 06161},
    {"aou=u", 0661, 0601, 0666},
    {"ag+u", 0140, 01650, 01676},
    {"g=t", 0644, 0513, 0503},
    {"g+t", 0030, 0613, 0613},
    {"u=xw", 0543, 00, 0300},
    {"ou=r+w", 0302, 0301, 0606},
    {"ou=t", 0064, 0623, 01020},
    {"g-s=sw", 0203, 0744, 02724},
    {"u=tXs", 0010, 0425, 04125},
    {"ag-wt", 0700, 04731, 04511},
    {"uo-rt", 0406, 044, 040},
    {"211", 0207, 0775, 0211},
    {"g-st", 0041, 06, 06},
    {"a+wst", 0705, 0247, 07267},
    {"gu+g", 0062, 0730, 0730},
    {"ug=stx", 0226, 075, 06115},
    {"ag-wtX", 0277, 0322, 00},
    {"gu-o", 0125, 0747, 07},
    {"oua-rx", 0647, 0522, 022},
    {"ag+Xr", 0336, 0744, 0755},
    {"g+sx", 0342, 0502, 02512},
    {"ga+r", 0622, 0155, 0555},
    {"u+t-X", 0606, 04542, 04442},
    {"a-s", 0276, 047, 047},
    {"g=rX", 0264, 0723, 0753},
    {"uga=Xrw", 0177, 0442, 0666},
    {"o+xrX", 0625, 0563, 0567},
    {"go+twr", 0036, 0131, 01177},
    {"+372", 0316, 0420, 0772},
    {"go+tx", 0632, 02025, 03035},
    {"ga=u", 0026, 053, 00},
    {"ug+r", 0205, 0743, 0743},
    {"u=X", 0060, 0174, 0174},
    {"a+o", 0325, 0674, 0674},
    {"guo+Xtr", 0221, 0677, 01777},
    {"ga=sw", 0444, 0273, 06222},
    {"o+Xr", 0010, 057, 057},
    {"u=u", 0302, 0141, 0141},
    {"g-sw", 0710, 0664, 0644},
    {"oga-rX", 0347, 0310, 0200},
    {"gu-X", 0732, 0310, 0200},
    {"uoa-u", 0711, 0125, 024},
    {"u-sx", 0307, 01633, 01633},
    {"ou-wx", 0130, 0572, 0470},
    {"ga+trs", 0405, 0762, 07766},
    {"ou-g", 0645, 01736, 01434},
    {"a-u", 0260, 0411, 011},
    {"a=twr", 0205, 0537, 01666},
    {"u-X+x", 0434, 033, 0133},
    {"go-wsX", 0300, 0111, 0100},
    {"o=stX", 0512, 0526, 01521},
    {"oa+sX", 0333, 0264, 06264},
    {"a=t+rw", 0037, 0410, 01666},
    {"o-o", 0426, 054, 050},
    {"gau+X", 0045, 01101, 01111},
    {"u+ws", 0413, 0100, 04300},
    {"og+X", 0250, 0402, 0402},
    {"go=o", 0351, 0736, 0766},
    {"630", 0223, 0776, 0630},
    {"ago-rw", 0555, 02454, 02010},
    {"o+rx", 0312, 024, 025},
    {"og-o", 0107, 0364, 0320},
    {"ou-u", 0005, 020, 020},
    {"ua+w", 0673, 0247, 0267},
    {"oa+u", 0555, 0640, 0666},
    {"oa=tw", 0766, 0150, 01222},
    {"o-xXt", 0372, 02134, 02134},
    {"+764", 0311, 0110, 0774},
    {"au-t", 0611, 0701, 0701},
    {"uo=X", 0254, 0540, 0141},
    {"ou+x=tr", 0047, 0655, 01454},
    {"uoa+sx", 0012, 0540, 06551},
    {"a-sxX", 0534, 0373, 0262},
    {"uga+stx", 0527, 0645, 07755},
    {"a=sr", 0666, 0730, 06444},
    {"g=xsw", 0005, 0440, 02430},
    {"g+rw", 0260, 02774, 02774},
    {"uo=tX", 0403, 0477, 01171},
    {"a=u", 0575, 0464, 0444},
    {"gu-x", 0654, 0130, 020},
    {"114", 0602, 0221, 0114},
    {"o=Xts", 0676, 0326, 01321},
    {"o-x=wX", 0146, 0600, 0602},
    {"oga+rX", 0333, 0513, 0557},
    {"oa+Xs", 0667, 0260, 06260},
    {"g-r", 0113, 0260, 0220},
    {"u=srw", 0602, 0602, 04602},
    {"o+X", 0032, 0636, 0637},
    {"uo-r=tX", 0175, 0201, 01101},
    {"u+Xxt", 0075, 0332, 0332},
    {"ua=Xx", 0243, 02353, 0111},
    {"o=t", 0151, 0275, 01270},
    {"ag=x", 0565, 0202, 0111},
    {"576", 0044, 0751, 0576},
    {"g-g", 0173, 01361, 01301},
    {"u+u", 0402, 0317, 0317},
    {"g-Xr", 0615, 0457, 0407},
    {"ag=o", 0160, 01146, 0666},
    {"au-xw", 0170, 0421, 0400},
    {"uo-rw", 0437, 0300, 0100},
    {"ao-s=t", 0264, 0105, 01000},
    {"ug-s", 0232, 020, 020},
    {"a=w", 0727, 0300, 0222},
    {"u=tr", 0106, 0223, 0423},
    {"oa-Xw", 0212, 0774, 0444},
    {"a-Xx", 0103, 0505, 0404},
    {"u+swX", 0076, 0504, 04704},
    {"o-xtX", 0344, 022, 022},
    {"oa-wx", 0467, 024, 04},
    {"uo-sw", 0646, 02703, 02501},
    {"goa=r", 0722, 02166, 0444},
    {"4136", 0424, 0171, 04136},
    {"u+rw", 0021, 04, 0604},
    {"u=ts", 0452, 02671, 06071},
    {"g-o", 0214, 0220, 0220},
    {"ag-u", 0374, 0260, 040},
    {"626", 0104, 0460, 0626},
    {"ua+wsX", 0415, 0625, 06737},
    {"oa=t", 0021, 0236, 01000},
    {"a=X+xs", 0763, 0540, 06111},
    {"oa-u", 0723, 0676, 010},
    {"a-wX", 0471, 0747, 0444},
    {"a=w", 0065, 0465, 0222},
    {"g+sw", 0123, 02622, 02622},
    {"674", 0701, 01050, 0674},
    {"ug-o", 0053, 0766, 0106},
    {"ua-Xxr", 0230, 01262, 01222},
    {"373", 0537, 0446, 0373},
    {"116", 0313, 0440, 0116},
    {"ug-rs", 0737, 0572, 0132},
    {"754", 0402, 0122, 0754},
    {"ug-o", 0062, 0477, 07},
    {"0454", 0470, 04050, 0454},
    {"a-wX", 0153, 0337, 04},
    {"u-u", 0224, 0746, 046},
    {"2340", 0052, 0766, 02340},
    {"uga-wXs", 0750, 02456, 0444},
    {"g=xw", 0421, 0630, 0630},
    {"065", 0452, 0720, 065},
    {"u-trs", 0126, 01100, 01100},
    {"ou+u", 0210, 0312, 0313},
    {"gu+g", 0536, 0617, 0717},
    {"au-t", 0100, 0636, 0636},
    {"ao+g", 0521, 0654, 0755},
    {"ou=u", 0526, 0773, 0777},
    {"u-tX", 0365, 02717, 02617},
    {"g-tX", 0700, 0665, 0665},
    {"go-g", 0551, 0206, 0206},
    {"ug-Xs", 0171, 0630, 0620},
    {"o+g", 0522, 047, 047},
    {"uo-g", 0773, 0370, 070},
    {"ug-rx", 0127, 0211, 0201},
    {"au=sx", 0457, 0632, 06111},
    {"og+w=xw", 0044, 076, 033},
    {"367", 0603, 0757, 0367},
    {"a+rXt", 0470, 02713, 03757},
    {"a+Xx", 0022, 0526, 0537},
    {"u+wX", 0765, 0540, 0740},
    {"ua=wx", 0546, 0617, 0333},
    {"355", 0723, 01407, 0355},
    {"a-w", 0526, 0561, 0541},
    {"ao-s", 0331, 0447, 0447},
    {"u=o", 0121, 0155, 0555},
    {"uag=w", 0363, 0315, 0222},
    {"u=wXr", 0336, 0753, 0753},
    {"ug+rs", 0035, 0645, 06645},
    {"g-ws", 0720, 0557, 0557},
    {"au-s", 0617, 0423, 0423},
    {"og-wrx", 0754, 0362, 0300},
    {"ou-x", 0237, 0346, 0246},
    {"agu+u", 0370, 02613, 02677},
    {"ua=o", 0053, 0407, 0777},
    {"u-rs", 0452, 0240, 0240},
    {"g+t", 0070, 0110, 0110},
    {"oa-rs", 0455, 0143, 0103},
    {"ua-g", 0417, 0241, 0201},
    {"g=xs", 0642, 0261, 02211},
    {"g+x", 0165, 0651, 0651},
    {"+461", 0746, 0606, 0667},
    {"uo-u", 0124, 0202, 00},
    {"a+g", 0132, 0744, 0744},
    {"a-s", 0151, 0676, 0676},
    {"o-xr", 0356, 0365, 0360},
    {"a-Xs", 0340, 04735, 0624},
    {"g=w", 0054, 0345, 0325},
    {"u=u", 0652, 0513, 0513},
    {"o-Xxs", 0670, 01251, 01250},
    {"4511", 0100, 0736, 04511},
    {"a=r", 0676, 0240, 0444},
    {"u-Xr", 0010, 0471, 071},
    {"o-tX", 0764, 0627, 0626},
    {"go=x", 0317, 0777, 0711},
    {"oa=sx", 0021, 0424, 06111},
    {"go-g", 0412, 072, 00},
    {"uo+w", 0707, 0406, 0606},
    {"264", 0261, 0716, 0264},
    {"ug+sxt", 0444, 0203, 06313},
    {"ga-X", 0620, 0460, 0460},
    {"g-xXt", 0560, 013, 03},
    {"uga=t", 0464, 0235, 01000},
    {"au-sr", 0211, 0332, 0332},
    {"go=stx", 0502, 0252, 03211},
    {"o-g", 0413, 047, 043},
    {"g-g", 0663, 0226, 0206},
    {"u+s", 0612, 072, 04072},
    {"ao+r", 0300, 0777, 0777},
    {"ou+ws", 0745, 0432, 04632},
    {"+567", 0163, 01273, 01777},
    {"ga=x-r+t", 0766, 0255, 01111},
    {"oug=r+X-Xs", 0176, 02623, 0444},
    {"0310", 0376, 0527, 0310},
    {"ua=o", 0566, 016, 0666},
    {"au=Xsw", 0644, 0324, 06333},
    {"au-r", 0105, 0777, 0333},
    {"oa+t-st", 0423, 043, 043},
    {"u+w", 0602, 04246, 04246},
    {"g+x", 0017, 0145, 0155},
    {"=504", 0133, 01771, 0504},
    {"u=wX", 0464, 0474, 0374},
    {"a+s", 0544, 04551, 06551},
    {"og+rx", 0456, 00, 055},
    {"go-o", 0204, 0514, 0510},
    {"g+wxt", 0600, 06, 036},
    {"o-rs", 0420, 0237, 0233},
    {"aou-Xsx", 0433, 01156, 01046},
    {"ua+u", 0515, 0112, 0113},
    {"go+t", 0321, 0634, 01634},
    {"o-tXr", 0765, 0626, 0622},
    {"ag-tX", 0614, 0526, 0426},
    {"ua=g", 0506, 0502, 00},
    {"o-o", 0160, 0121, 0120},
    {"u+wt", 0422, 0127, 0327},
    {"o-tr", 0424, 0571, 0571},
    {"gau=s", 0240, 0214, 06000},
    {"gua+tw", 0350, 0521, 01723},
    {"o+s", 0116, 0660, 0660},
    {"u-w", 0420, 01071, 01071},
    {"u+st", 0146, 061, 04061},
    {"a=u", 0325, 0117, 0111},
    {"uo=sX", 0664, 0260, 04060},
    {"gau=g", 0247, 04135, 0333},
    {"ao+g", 0214, 0652, 0757},
    {"o=rw", 0422, 0374, 0376},
    {"a=x", 0420, 01460, 0111},
    {"og-r+st", 0707, 00, 03000},
    {"g-X", 0505, 02617, 02607},
    {"g+g", 0342, 0146, 0146},
    {"g+srX", 0101, 0767, 02777},
    {"a-u", 0362, 0175, 064},
    {"=2722", 0153, 0613, 02722},
    {"aog+Xr", 0043, 0457, 0557},
    {"g=tw", 0470, 01046, 01026},
    {"a=o", 0625, 074, 0444},
    {"ug-rt", 0321, 0435, 035},
    {"a-sw", 0522, 0445, 0445},
    {"106", 0710, 0211, 0106},
    {"ag+ts", 0553, 0200, 07200},
    {"au-xr", 0573, 017, 02},
    {"oa=tw", 0604, 0235, 01222},
    {"g+st", 0500, 0455, 02455},
    {"ug+u", 0126, 04044, 04044},
    {"aog+Xr", 0601, 0525, 0575},
    {"og=Xw", 0470, 0677, 0633},
    {"ua+w", 0021, 04, 0226},
    {"aou=g", 0374, 0570, 0777},
    {"a+st", 0004, 0240, 07240},
    {"au-xs", 0607, 014, 04},
    {"a+o", 0657, 020, 020},
    {"o+ws", 0622, 0644, 0646},
    {"u-sX", 0210, 02055, 02055},
    {"o-Xx", 0005, 0575, 0574},
    {"ag-g", 0526, 0345, 0301},
    {"ag=t", 0570, 0202, 01000},
    {"g+t", 0772, 023, 023},
    {"uog=ws", 0274, 0665, 06222},
    {"g+rxt", 0065, 075, 075},
    {"a+X", 0465, 04604, 04604},
    {"a=t", 0261, 0223, 01000},
    {"u-g", 0011, 0560, 0160},
    {"ou-o", 0024, 033, 030},
    {"u-ws", 0412, 0302, 0102},
    {"ua-X", 0036, 0162, 062},
    {"ou+Xw", 0470, 0443, 0743},
    {"334", 0260, 0605, 0334},
    {"g+g", 0612, 00, 00},
    {"o+wr", 0332, 01472, 01476},
    {"agu-rs", 0721, 0573, 0133},
    {"a+w=sw", 0177, 0434, 06222},
    {"ug=t+x-r", 0554, 0576, 0116},
    {"214", 0553, 0472, 0214},
    {"ao=o", 0642, 0424, 0444},
    {"ago-tr", 0600, 0214, 0210},
    {"u=sx", 0416, 0372, 04172},
    {"a=r", 0177, 0161, 0444},
    {"g=Xxr", 0676, 0437, 0457},
    {"au+s", 0672, 0277, 06277},
    {"u=o", 0540, 0727, 0727},
    {"ao+u", 0456, 0545, 0555},
    {"au-wsr", 0051, 0637, 011},
    {"go=Xtw", 0704, 066, 01022},
    {"066", 0665, 0567, 066},
    {"a=xX", 0763, 0555, 0111},
    {"oua=r", 0057, 04726, 0444},
    {"uo+Xx", 0727, 0572, 0573},
    {"a+o", 0276, 0314, 0754},
    {"ao+g", 0731, 0513, 0513},
    {"oa+ts", 0775, 0761, 07761},
    {"gu=wX", 0720, 077, 0337},
    {"oug+t", 0003, 0600, 01600},
    {"a+g", 0025, 0461, 0667},
    {"u+tXs", 0675, 0460, 04460},
    {"oa+x-wx", 0324, 0470, 0440},
    {"u+u", 0025, 0604, 0604},
    {"a+X", 0625, 0555, 0555},
    {"a+r", 0452, 0364, 0764},
    {"og-t=tx", 0440, 0255, 01211},
    {"u=stw", 0266, 00, 04200},
    {"go-sx", 0164, 0742, 0742},
    {"oua+xXs", 0714, 0653, 06753},
    {"uoa-X", 0514, 0456, 0446},
    {"ug+xts", 0472, 04523, 06533},
    {"ou+x", 0012, 0630, 0731},
    {"g-s", 0463, 0574, 0574},
    {"-077", 0424, 0627, 0600},
    {"o+wsx", 0726, 0103, 0103},
    {"aou=w+s", 0535, 02373, 06222},
    {"ga-Xt", 0774, 011, 00},
    {"ag-ws", 0161, 0724, 0504},
    {"g-g", 0703, 060, 00},
    {"450", 0563, 040, 0450},
    {"u=sw", 0141, 0272, 04272},
    {"uo-x", 0717, 050, 050},
    {"a-st", 0323, 0123, 0123},
    {"gu+w", 0175, 0252, 0272},
    {"gu-g", 0251, 0365, 0105},
    {"o+o", 0537, 0200, 0200},
    {"o+s", 0056, 0104, 0104},
    {"uoa-o", 0236, 0506, 0100},
    {"gu+swX", 0434, 027, 06337},
    {"ao+o", 0616, 0101, 0111},
    {"uag-u", 0450, 04247, 04045},
    {"gu-xw", 0531, 04545, 04445},
    {"u=wx", 0062, 0213, 0313},
    {"g=x", 0045, 01137, 01117},
    {"g+o", 0471, 0612, 0632},
    {"au=w", 0770, 0557, 0222},
    {"o=t-r", 0430, 0510, 01510},
    {"uo+t", 0013, 042, 01042},
    {"ua-X", 0320, 0222, 0222},
    {"g+Xxr", 0625, 0523, 0573},
    {"gu+x", 0144, 0124, 0134},
    {"uo+u", 0030, 0146, 0147},
    {"u-rw", 0237, 06, 06},
    {"o-w", 0702, 0415, 0415},
    {"766", 0366, 0355, 0766},
    {"au=g", 0333, 0317, 0111},
    {"oau=Xx", 0137, 0305, 0111},
    {"464", 0042, 0237, 0464},
    {"ug-rw", 0731, 0532, 0112},
    {"o+g", 0331, 0162, 0166},
    {"uo+x", 0042, 0465, 0565},
    {"gau-xt", 0466, 02214, 02204},
    {"u-u", 0467, 0763, 063},
    {"ao=t", 0562, 00, 01000},
    {"a+t", 0116, 026, 01026},
    {"oga=wX", 0716, 0576, 0333},
    {"u=u", 0303, 0552, 0552},
    {"ua=u", 0600, 0547, 0555},
    {"ogu-xt", 0237, 066, 066},
    {"gu=twX", 0647, 0766, 0336},
    {"a=r", 0646, 0406, 0444},
    {"g=Xxt", 0231, 04253, 04213},
    {"a-Xr", 0154, 0640, 0200},
    {"ao-u", 0617, 0773, 00},
    {"=150", 0526, 0371, 0150},
    {"o+wrX", 0315, 0700, 0707},
    {"ou-rw", 0072, 0256, 050},
    {"u+tx", 0124, 0330, 0330},
    {"gu+u", 0004, 0224, 0224},
    {"gao+tr", 0367, 0470, 01474},
    {"u-u", 0735, 0463, 063},
    {"431", 0740, 04027, 0431},
    {"og-o", 0433, 02665, 02620},
    {"aug=u", 0507, 0231, 0222},
    {"=076", 0201, 04717, 076},
    {"au=g", 0225, 0666, 0666},
    {"a+u", 0105, 0210, 0232},
    {"u=o", 0230, 0702, 0202},
    {"004", 0704, 0256, 04},
    {"oga-g", 0262, 0715, 0604},
    {"oga+w", 0002, 0413, 0633},
    {"uo=X", 0056, 053, 0151},
    {"ua-o", 0736, 0244, 0200},
    {"au+x", 0712, 0127, 0137},
    {"go=xw", 0767, 0256, 0233},
    {"uog-rw", 0410, 0104, 0100},
    {"o+tx", 0057, 0576, 01577},
    {"a=u", 0504, 01207, 0222},
    {"u=w", 0346, 0751, 0251},
    {"guo+x", 0366, 0243, 0353},
    {"a=twX", 0763, 0231, 01333},
    {"uo-u", 0330, 0114, 014},
    {"aou=xX", 0775, 0576, 0111},
    {"ou=wrt", 0023, 0556, 01656},
    {"oa+r", 0630, 0517, 0557},
    {"oa=Xs", 0131, 0135, 06111},
    {"643", 0310, 0776, 0643},
    {"oa+w", 0573, 0350, 0372},
    {"o=tw", 0060, 0543, 01542},
    {"ao+tr", 0702, 020, 01464},
    {"ga-tx", 0412, 047, 046},
    {"g+g", 0361, 0372, 0372},
    {"o=rs", 0653, 0551, 0554},
    {"a+o", 0527, 0364, 0764},
    {"a-tws", 0737, 040, 040},
    {"a-X", 0474, 02062, 02062},
    {"o-xX", 0735, 0110, 0110},
    {"og-w", 0266, 04614, 04614},
    {"ao-r", 0431, 0645, 0201},
    {"aog+st", 0441, 0227, 07227},
    {"gu-wr", 0363, 04060, 04000},
    {"o+Xwr", 0471, 0470, 0477},
    {"ogu+xst", 0656, 0665, 07775},
    {"g+t", 0523, 0754, 0754},
    {"o-X", 0772, 0654, 0654},
    {"gu+w", 0214, 0166, 0366},
    {"a+u", 0260, 0433, 0477},
    {"a=Xs", 0460, 0306, 06111},
    {"ua+sxX", 0423, 0650, 06751},
    {"u=X", 0370, 0216, 0116},
    {"445", 0426, 0665, 0445},
    {"a=tX", 0727, 0562, 01111},
    {"g+ws", 0414, 0202, 02222},
    {"g=X", 0212, 0407, 0417},
    {"og=xsw", 0346, 0346, 02333},
    {"uog-r", 0661, 0773, 0333},
    {"g+w", 0705, 055, 075},
    {"uo=s", 0710, 0375, 04070},
    {"g+t", 0510, 0516, 0516},
    {"go+g", 0031, 0700, 0700},
    {"o-r", 0075, 0656, 0652},
    {"ug-st", 0470, 0517, 0517},
    {"605", 0275, 01511, 0605},
    {"u+xt", 0023, 0200, 0300},
    {"au-s", 0350, 0615, 0615},
    {"o=wxX", 0344, 00, 03},
    {"ao+wrx", 0665, 0673, 0777},
    {"u-rw", 0023, 0737, 0137},
    {"ug=rx", 0136, 0650, 0550},
    {"006", 0626, 0203, 06},
    {"u=rXs", 0733, 044, 04444},
    {"ug=tx", 0107, 0461, 0111},
    {"ugo+r", 0454, 037, 0477},
    {"u+g", 0043, 0777, 0777},
    {"ou=t", 0332, 0555, 01050},
    {"ua-u", 0747, 0622, 00},
    {"ag-x", 0405, 0420, 0420},
    {"ag-wt", 0423, 0660, 0440},
    {"uo-sX", 0551, 073, 072},
    {"uog-r", 0500, 0475, 031},
    {"ag+u", 0560, 073, 073},
    {"ou+xtr", 0675, 0240, 01745},
    {"a+Xx", 0756, 0441, 0551},
    {"ua+wr", 0315, 0553, 0777},
    {"a=s", 0177, 0655, 06000},
    {"ag+tx", 0514, 0210, 01311},
    {"au=t", 0605, 0722, 01000},
    {"1677", 0435, 01760, 01677},
    {"ga=ws", 0555, 0336, 06222},
    {"u=wxs", 0722, 0741, 04341},
    {"ga-rX", 0423, 0765, 0220},
    {"gu=xX", 0524, 04322, 0112},
    {"uo+tX", 0215, 0176, 01177},
    {"u=u", 0042, 074, 074},
    {"g+tw", 0543, 0555, 0575},
    {"+536", 0202, 0714, 0736},
    {"ag=s", 0255, 02647, 06000},
    {"u+wXs", 0351, 0262, 04262},
    {"g=Xt", 0161, 0324, 0314},
    {"uag-u", 0166, 02053, 02053},
    {"a-wrt", 0663, 0321, 0101},
    {"au+tx", 0444, 0555, 01555},
    {"oa+x", 0620, 0433, 0533},
    {"go=s", 0673, 0626, 02600},
    {"g-sr", 0277, 02032, 032},
    {"g=xX", 0271, 0165, 0115},
    {"uao=tXr", 0015, 0673, 01555},
    {"g=X", 0433, 04247, 04217},
    {"g-rs", 0653, 02276, 0236},
    {"ug=sw", 0537, 0265, 06225},
    {"g+rx", 0104, 0460, 0470},
    {"g=xst", 0452, 0626, 02616},
    {"501", 0644, 02473, 0501},
    {"a=xtw", 0372, 0222, 01333},
    {"ou+wrX", 0774, 0776, 0777},
    {"ga+r", 0734, 0513, 0557},
    {"ao=o", 0403, 0363, 0333},
    {"+105", 0042, 0601, 0705},
    {"uog+X", 0540, 0262, 0262},
    {"u+X", 0526, 0104, 0104},
    {"a+tx", 0054, 0637, 01737},
    {"a-r", 0513, 0335, 0331},
    {"a=u", 0412, 0760, 0777},
    {"au=xXs", 0442, 00, 06111},
    {"u-ts", 0424, 0501, 0501},
    {"ag+sw", 0173, 0204, 06226},
    {"ago-st", 0076, 0307, 0307},
    {"g-tx", 0620, 0301, 0301},
    {"g-t", 0217, 0426, 0426},
    {"go=w", 0771, 0106, 0122},
    {"au+g", 0137, 0554, 0555},
    {"ugo+w-xt", 0371, 0630, 0622},
    {"350", 0061, 0327, 0350},
    {"ao-xrw", 0066, 0343, 00},
    {"o+o", 0540, 0232, 0232},
    {"ug=rXt", 0377, 0711, 0551},
    {"ua+sX", 0461, 0251, 06351},
    {"guo=t", 0270, 0761, 01000},
    {"aou+s-ts", 0165, 0142, 0142},
    {"ag=trw", 0710, 02475, 01666},
    {"020", 0752, 0463, 020},
    {"u+g", 0462, 064, 0664},
    {"ga-xXt", 0074, 057, 046},
    {"a=s", 0134, 065, 06000},
    {"ug=u", 0505, 02261, 0221},
    {"ao=w", 0767, 0437, 0222},
    {"a-st", 0374, 01643, 0643},
    {"g+xw", 0707, 0345, 0375},
    {"oa=g", 0201, 0602, 00},
    {"ou+g", 0152, 0470, 0777},
    {"a+g", 0474, 00, 00},
    {"uo-Xsx", 0503, 0774, 0674},
    {"og+X", 0227, 0773, 0773},
    {"ou-r", 0134, 0243, 0243},
    {"a-xs", 0157, 0664, 0664},
    {"a-wxX", 0735, 01324, 01004},
    {"g+xt", 0241, 06, 016},
    {"a+xX", 0722, 0740, 0751},
    {"ua-sx", 0005, 0466, 0466},
    {"u-wt", 0102, 0204, 04},
    {"762", 0763, 0546, 0762},
    {"ou-u", 0107, 0620, 020},
    {"u-tr", 0367, 0342, 0342},
    {"a=trw", 0514, 0431, 01666},
    {"au-Xr", 0303, 0766, 0222},
    {"oa+xXt", 0425, 042, 01153},
    {"ga-x", 0404, 0247, 0246},
    {"au+xr", 0156, 0511, 0555},
    {"o+rXx", 0664, 0273, 0277},
    {"ao-o", 0404, 0762, 0540},
    {"ag-xr", 0111, 0614, 0200},
    {"auo+Xr", 0276, 0371, 0775},
    {"ua=g", 0447, 0347, 0444},
    {"ag-Xxt", 0100, 044, 044},
    {"u-xw", 0070, 00, 00},
    {"oa+Xsw", 0053, 06, 06226},
    {"gu+X", 0532, 055, 0155},
    {"557", 0173, 01271, 0557},
    {"o-g", 0114, 050, 050},
    {"a+X", 0766, 0472, 0573},
    {"ao+Xr", 0211, 0571, 0575},
    {"o-u", 0672, 0206, 0204},
    {"gao+rts", 0210, 04523, 07567},
    {"go+xs", 0754, 01207, 03217},
    {"ao-tsx", 0304, 0427, 0426},
    {"uoa+g", 0621, 0563, 0767},
    {"g+r", 0777, 033, 073},
    {"o-t", 0145, 0402, 0402},
    {"ua+u", 0242, 0600, 0666},
    {"u-w", 0216, 0125, 0125},
    {"u+X", 0745, 0700, 0700},
    {"uag=g", 0316, 01143, 0444},
    {"g-sw", 0677, 0371, 0351},
    {"uo=u", 0712, 0644, 0646},
    {"oga=t", 0056, 0670, 01000},
    {"u-g", 0251, 0300, 0300},
    {"075", 0605, 011, 075},
    {"gu=r+wX", 0343, 0546, 0666},
    {"au-sX", 0362, 0560, 0460},
    {"go+w", 0044, 0165, 0167},
    {"a-r", 0066, 0622, 0222},
    {"a-w", 0320, 03, 01},
    {"agu-s", 0272, 0100, 0100},
    {"g+sX", 0534, 0105, 02115},
    {"u-r", 0773, 0657, 0257},
    {"430", 0627, 032, 0430},
    {"g-xw", 0543, 0316, 0306},
    {"717", 0013, 020, 0717},
    {"oua-wrX", 0662, 0620, 00},
    {"ao-stx", 0201, 0322, 0222},
    {"ou-wX", 0132, 0400, 0400},
    {"0414", 0111, 0673, 0414},
    {"oa-tXx", 0501, 0262, 0262},
    {"goa=tsX", 0021, 046, 07000},
    {"u+rs", 0510, 0606, 04606},
    {"g-x", 0710, 0402, 0402},
    {"ag+tw", 0026, 01, 01223},
    {"g=tr", 0223, 0571, 0541},
    {"ugo=u", 0304, 0442, 0444},
    {"og=xs", 0101, 01306, 02311},
    {"ug=g", 0350, 0451, 0551},
    {"ug=Xxw", 0445, 0423, 0333},
    {"a+wx", 0273, 012, 0333},
    {"og-X=tr", 0246, 0774, 01744},
    {"ag+t", 0323, 0500, 01500},
    {"ugo=s", 0234, 0314, 06000},
    {"g+w", 0371, 0200, 0220},
    {"g-s", 0312, 0316, 0316},
    {"oa+s", 0571, 0302, 06302},
    {"o+xts", 0316, 04221, 05221},
    {"242", 0247, 01212, 0242},
    {"o=tXr", 0276, 02501, 03505},
    {"og-t", 0711, 02546, 02546},
    {"oga+rs", 0072, 0230, 06674},
    {"au+s", 0002, 02133, 06133},
    {"o+X", 0272, 0233, 0233},
    {"aou-rs", 0122, 076, 032},
    {"go-o", 0133, 0400, 0400},
    {"ag-ws", 0522, 0352, 0150},
    {"ua-u", 0003, 0577, 022},
    {"o+X=xX", 0674, 075, 071},
    {"g+Xw", 0007, 0620, 0620},
    {"a-X=sx", 0202, 0252, 06111},
    {"go-r", 0167, 0476, 0432},
    {"go=wt", 0563, 0464, 01422},
    {"goa-rst", 0352, 0420, 020},
    {"a-r", 0123, 0644, 0200},
    {"o=Xx", 0073, 01614, 0611},
    {"g-wx", 0413, 0224, 0204},
    {"a-sw", 0363, 0514, 0514},
    {"uog-tr", 0670, 01505, 0101},
    {"au=wt", 0345, 04631, 01222},
    {"agu=tXw", 0104, 0415, 01333},
    {"ou-xr", 0445, 0767, 0262},
    {"340", 0145, 02504, 0340},
    {"a-Xw", 0034, 0344, 044},
    {"a=wrX", 0147, 0571, 0777},
    {"uo-u", 0026, 0141, 040},
    {"ag=Xx", 0030, 01005, 0111},
    {"ao-wt", 0610, 066, 044},
    {"g=X", 0500, 0756, 0716},
    {"a-o", 0057, 0435, 020},
    {"o=wxX", 0127, 0440, 0443},
    {"u=u", 0763, 0514, 0514},
    {"og=t", 0402, 0666, 01600},
    {"o-x", 0025, 0632, 0632},
    {"2216", 0363, 01126, 02216},
    {"g+xst", 0765, 02076, 02076},
    {"uo+xX", 0640, 0663, 0763},
    {"ou+u", 0666, 0107, 0107},
    {"a+r", 0066, 0654, 0654},
    {"u+w", 0515, 01321, 01321},
    {"go=wX", 0375, 0703, 0733},
    {"g+rs", 0031, 0657, 02657},
    {"+2154", 0321, 0347, 02357},
    {"515", 0114, 0641, 0515},
    {"agu=g", 0256, 0721, 0222},
    {"o=t", 0271, 01043, 01040},
    {"347", 0065, 0713, 0347},
    {"ou+wr", 0174, 01460, 01666},
    {"g+o", 0314, 0270, 0270},
    {"o-Xw", 0401, 0770, 0770},
    {"0260", 0513, 0264, 0260},
    {"aou=X", 0544, 0326, 0111},
    {"u+rt", 0320, 0467, 0467},
    {"uga+t", 0473, 0326, 01326},
    {"a=t", 0566, 0463, 01000},
    {"g=tx", 0212, 0522, 0512},
    {"oga=Xst", 0174, 0777, 07111},
    {"ug=txs", 0213, 0203, 06113},
    {"ug-xs", 0556, 0471, 0461},
    {"ug=x", 0101, 0222, 0112},
    {"gau+s", 0741, 023, 06023},
    {"ag-sx", 0201, 0134, 024},
    {"agu-tX", 0206, 0561, 0460},
    {"gu+wr", 0472, 0706, 0766},
    {"ao+wr", 0644, 042, 0666},
    {"o-s=xt", 0614, 0626, 01621},
    {"ogu+w", 0457, 0773, 0773},
    {"oua+tsX", 0532, 0366, 07377},
    {"=413", 0603, 0677, 0413},
    {"u=Xw", 0276, 012, 0312},
    {"ua=sX", 0772, 0515, 06111},
    {"ug+t", 0700, 011, 011},
    {"og+g", 0160, 0100, 0100},
    {"u=s", 0111, 0243, 04043},
    {"gu+g", 0246, 04506, 04506},
    {"o+xw", 0041, 0133, 0133},
    {"ao=txw", 0223, 01046, 01333},
    {"o+w", 0127, 01425, 01427},
    {"o+xXr", 0661, 0101, 0105},
    {"go=u", 0455, 0524, 0555},
    {"uo-x", 0165, 04430, 04430},
    {"o+Xtr", 0735, 0714, 01715},
    {"g-sx", 0072, 0452, 0442},
    {"126", 0521, 01741, 0126},
    {"aog=sw", 0451, 0340, 06222},
    {"ogu=rx", 0777, 037, 0555},
    {"g-ts", 0174, 0515, 0515},
    {"ag-s", 0004, 0234, 0234},
    {"ag+u", 0215, 0623, 0667},
    {"og=w", 0674, 0555, 0522},
    {"a-xX", 0701, 044, 044},
    {"aou+wt", 0644, 0610, 01632},
    {"o=xr", 0313, 0245, 0245},
    {"o=u", 0257, 0665, 0666},
    {"uao-s", 0160, 0155, 0155},
    {"go-ts", 0425, 0721, 0721},
    {"og-t", 0737, 0710, 0710},
    {"o=rx", 0114, 01643, 0645},
    {"oa=x", 0275, 044, 0111},
    {"ag=x", 0711, 0377, 0111},
    {"g=wt", 0542, 0711, 0721},
    {"ag-x=sxX", 0474, 047, 06111},
    {"446", 0420, 01346, 0446},
    {"au+wx", 0272, 0510, 0733},
    {"uog-t", 0675, 0163, 0163},
    {"a+wX", 0510, 0317, 0337},
    {"gu-g", 0503, 0142, 0102},
    {"ou=tx", 0561, 04, 01101},
    {"ga+stw", 0265, 0157, 07377},
    {"ugo+Xr", 0474, 0465, 0575},
    {"-613", 0103, 064, 064},
    {"gua=X", 0563, 0365, 0111},
    {"ua+rt", 0604, 0177, 01577},
    {"go+o", 0022, 0627, 0677},
    {"g-o", 0503, 02245, 02205},
    {"u+t-Xt", 0100, 03, 03},
    {"go+r-w=ws", 0716, 011, 02022},
    {"oga+u", 0022, 0665, 0667},
    {"au=o", 0174, 0700, 00},
    {"a+x", 0633, 0326, 0337},
    {"g-o", 0302, 02766, 02706},
    {"a+u", 0624, 0611, 0677},
    {"o+rX", 0721, 05, 05},
    {"u+w", 0703, 0374, 0374},
    {"og+r", 0757, 0127, 0167},
    {"g-ws", 0126, 0341, 0341},
    {"oau-t+w=rws", 0316, 0111, 06666},
    {"g=tw", 0506, 0660, 0620},
    {"ag+w", 0445, 0664, 0666},
    {"og-twX", 0167, 0315, 0304},
    {"a=s", 0324, 033, 06000},
    {"u+s", 0575, 0776, 04776},
    {"oau-xs", 0371, 07, 06},
    {"0241", 0465, 02074, 0241},
    {"o+t", 0054, 0152, 01152},
    {"a=tXw", 0003, 0667, 01333},
    {"o-stw", 0522, 0747, 0745},
    {"ag+u", 0037, 0603, 0667},
    {"au+tX", 0357, 015, 01115},
    {"u+xs", 0672, 0167, 04167},
    {"g+u", 0341, 0407, 0447},
    {"=363", 0045, 0510, 0363},
    {"aou-Xr", 0276, 0421, 020},
    {"427", 0017, 0453, 0427},
    {"a+tx", 0455, 0663, 01773},
    {"u-wx", 0012, 0677, 0477},
    {"guo+g", 0646, 0107, 0107},
    {"o-r+Xrw", 0342, 01054, 01057},
    {"a+rt", 0660, 0216, 01656},
    {"go-rws", 0442, 0634, 0610},
    {"a+rt", 0177, 0600, 01644},
    {"g+rx", 0122, 0312, 0352},
    {"u-s", 0346, 070, 070},
    {"gu=o", 0703, 0771, 0111},
    {"og=s", 0023, 04745, 06700},
    {"u-g", 0330, 032, 032},
    {"au+Xwt", 0357, 0241, 01373},
    {"o-o", 0454, 0443, 0440},
    {"ga=tXs", 0700, 0350, 07111},
    {"u+u", 0200, 0757, 0757},
    {"g=X", 0730, 0571, 0511},
    {"go-tws", 0627, 063, 041},
    {"aug=wX", 0562, 02506, 0333},
    {"ug-xw", 0440, 0543, 0443},
    {"g-xXs", 0416, 0750, 0740},
    {"ou+g", 0351, 01274, 01777},
    {"u-sw", 0613, 0356, 0156},
    {"-0721", 0132, 04654, 04054},
    {"u-t", 0111, 0101, 0101},
    {"+445", 0035, 0405, 0445},
    {"554", 0102, 0430, 0554},
    {"o-x=x", 0331, 0474, 0471},
    {"ao+t", 0234, 04645, 05645},
    {"a-x+xts", 0252, 0130, 07131},
    {"257", 0706, 0325, 0257},
    {"go-twx", 0562, 0126, 0104},
    {"ao=xr", 0043, 0754, 0555},
    {"gu+s", 0734, 055, 06055},
    {"uo=t", 0552, 0204, 01000},
    {"ug+s", 0565, 0457, 06457},
    {"gu+xr", 0012, 02324, 02774},
    {"ao=Xxw", 0643, 0324, 0333},
    {"u-wr", 0566, 07, 07},
    {"ug-ws", 0402, 011, 011},
    {"oa=x", 0405, 0737, 0111},
    {"a+g", 0267, 020, 0222},
    {"uga+ts", 0612, 0265, 07265},
    {"g-t", 0357, 0342, 0342},
    {"go+X", 0476, 04404, 04404},
    {"ao+trw", 0735, 042, 01666},
    {"auo+rw", 0572, 0763, 0767},
    {"goa-g", 0152, 0654, 0200},
    {"ga=Xr", 0307, 0706, 0555},
    {"gu-rtX", 0507, 0463, 023},
    {"a-X", 0732, 03, 02},
    {"u=x", 0676, 0632, 0132},
    {"ao-u", 0200, 0664, 00},
    {"ao+rw", 0704, 067, 0667},
    {"1615", 0171, 0106, 01615},
    {"gu-u", 0221, 04552, 04002},
    {"au=s", 0317, 0440, 06000},
    {"ug+wx", 0317, 0677, 0777},
    {"ugo=g", 0747, 0211, 0111},
    {"go-xsw", 0426, 016, 04},
    {"ua=s", 0626, 0750, 06000},
    {"ao-tsX", 0537, 0365, 0264},
    {"oa+X", 0616, 016, 0117},
    {"gou-ts", 0652, 020, 020},
    {"a+sx", 0721, 00, 06111},
    {"ua-Xtr", 0204, 0705, 0200},
    {"o-x", 0607, 046, 046},
    {"025", 0542, 0414, 025},
    {"u+w", 0527, 0240, 0240},
    {"uo=sw", 0324, 040, 04242},
    {"ag+x", 0161, 0376, 0377},
    {"ou=sX", 0025, 0477, 04171},
    {"gou=X", 0666, 02570, 0111},
    {"ug-o", 0664, 0476, 016},
    {"agu-sr", 0414, 01253, 01213},
    {"uo=sx", 0611, 040, 04141},
    {"ago-r+r", 0333, 0642, 0646},
    {"-203", 0666, 0613, 0410},
    {"362", 0451, 0456, 0362},
    {"go-rxt", 0421, 0525, 0520},
    {"goa=x", 0350, 01257, 0111},
    {"og=st", 0121, 0100, 03100},
    {"u=stX", 0640, 01454, 05154},
    {"gou=x", 0770, 02670, 0111},
    {"gou+w", 0435, 0331, 0333},
    {"675", 0604, 0162, 0675},
    {"au-ws", 0402, 0742, 0540},
    {"a-r", 0733, 0651, 0211},
    {"u=w+s", 0313, 0570, 04270},
    {"=447", 0376, 0230, 0447},
    {"oa-sr", 0565, 0651, 0211},
    {"go=x", 0622, 0710, 0711},
    {"002", 0526, 0724, 02},
    {"1730", 0334, 0761, 01730},
    {"g+o", 0602, 024, 064},
    {"o=wrt", 0636, 0417, 01416},
    {"a+r", 0022, 0666, 0666},
    {"ga=t-r+X", 0231, 01447, 01000},
    {"g=wr", 0636, 0464, 0464},
    {"g-sr", 0321, 02714, 0714},
    {"u+tw", 0027, 04370, 04370},
    {"aou=o", 0036, 0441, 0111},
    {"g=u", 0653, 0510, 0550},
    {"g=r", 0047, 0611, 0641},
    {"ug=X", 0053, 01333, 01113},
    {"ago=X", 0122, 0540, 0111},
    {"g=o", 0645, 0770, 0700},
    {"u=x", 0406, 0712, 0112},
    {"gau=ts", 0273, 0763, 07000},
    {"ug=s", 0726, 0620, 06000},
    {"gao+txw", 0406, 01074, 01377},
    {"4454", 0626, 0613, 04454},
    {"u-xwX", 0236, 04376, 04076},
    {"a+trx", 0371, 0177, 01577},
    {"ga=s", 0034, 0565, 06000},
    {"uag-tw", 0652, 02653, 02451},
    {"oa+wx", 0570, 0206, 0337},
    {"732", 0777, 0564, 0732},
    {"ga=t", 0305, 0241, 01000},
    {"g-x", 0306, 0474, 0464},
    {"aug=Xxt", 0746, 0574, 01111},
    {"=707", 0456, 036, 0707},
    {"uag-g", 0542, 0564, 0100},
    {"ou=rxX", 0250, 01136, 0535},
    {"ag=o", 0404, 057, 0777},
    {"4673", 0764, 0117, 04673},
    {"ag+X", 0052, 0310, 0311},
    {"ga=t", 0153, 047, 01000},
    {"ga+rw", 0277, 0676, 0676},
    {"a+u", 0555, 0615, 0677},
    {"ao=xw", 0673, 04616, 0333},
    {"uo+t", 0000, 02050, 03050},
    {"a=s-r", 0773, 0725, 06000},
    {"ag+g", 0436, 0643, 0647},
    {"a-X", 0573, 01035, 01024},
    {"o+u", 0434, 0763, 0767},
    {"ga-X", 0411, 02451, 02440},
    {"ga=X", 0623, 01046, 00},
    {"oau=Xxt", 0714, 0677, 01111},
    {"og=w", 0634, 0232, 0222},
    {"545", 0647, 0710, 0545},
    {"g=o", 0427, 0270, 0200},
    {"go+txw", 0204, 02427, 03437},
    {"ao=o", 0277, 015, 0555},
    {"+267", 0346, 0254, 0277},
    {"u+g", 0755, 0707, 0707},
    {"u=swx", 0763, 04, 04304},
    {"o+Xst", 0065, 0345, 01345},
    {"uo+rxt", 0774, 0552, 01557},
    {"ou+sxw", 0336, 0232, 04333},
    {"oa-u", 0430, 0336, 04},
    {"g=X-Xwt", 0221, 0152, 0102},
    {"ou-w", 0775, 0653, 0451},
    {"ug+tx", 0033, 01744, 01754},
    {"a+tw", 0377, 0432, 01632},
    {"o+w", 0307, 0452, 0452},
    {"u-rx", 0235, 0323, 0223},
    {"u-s", 0063, 0174, 0174},
    {"oa+sxr", 0561, 00, 06555},
    {"ag+r", 0430, 0631, 0675},
    {"o-rX", 0477, 0144, 0140},
    {"ag-u", 0423, 0770, 00},
    {"u=rXw", 0737, 0345, 0745},
    {"gau-u", 0420, 0401, 01},
    {"ao+r", 0661, 02, 0446},
    {"ag+txr", 0651, 0627, 01777},
    {"ua=stw", 0106, 0307, 07222},
    {"ga+Xt", 0270, 04443, 05553},
    {"g=s", 0465, 0206, 02206},
    {"a-o", 0670, 0412, 0410},
    {"062", 0725, 0360, 062},
    {"goa=x", 0414, 04106, 0111},
    {"ag-wst", 0444, 0640, 0440},
    {"o+s", 0112, 0701, 0701},
    {"ug-r", 0522, 0764, 0324},
    {"g=r", 0614, 0662, 0642},
    {"au+r", 0116, 0372, 0776},
    {"u-g", 0315, 057, 057},
    {"gu=X", 0365, 036, 0116},
    {"o-o", 0477, 0213, 0210},
    {"ua-w", 0610, 0541, 0541},
    {"g+r", 0044, 0402, 0442},
    {"oua+s", 0611, 0210, 06210},
    {"ga-Xtr", 0064, 0542, 02},
    {"o=x", 0715, 0662, 0661},
    {"a+o", 0610, 0176, 0776},
    {"ag+s=rw", 0334, 0275, 0666},
    {"uo-sx", 0061, 0660, 0660},
    {"aou-sr", 0461, 04516, 0112},
    {"g-xX", 0766, 0764, 0764},
    {"523", 0045, 00, 0523},
    {"u=w", 0535, 067, 0267},
    {"og=rx", 0512, 0741, 0755},
    {"ga-ts", 0233, 0614, 0614},
    {"gu=tws", 0052, 0776, 06226},
    {"ag-o", 0474, 0676, 010},
    {"g+w", 0357, 0773, 0773},
    {"ao-X", 0763, 0570, 0460},
    {"o-xt", 0705, 0207, 0206},
    {"a+s", 0404, 042, 06042},
    {"uo-r", 0072, 0701, 0301},
    {"oau+r", 0310, 0617, 0657},
    {"o-o", 0471, 0546, 0540},
    {"gau=xs", 0365, 0546, 06111},
    {"og+wrt", 0462, 0714, 01776},
    {"=1674", 0217, 064, 01674},
    {"oau-sx", 0545, 0564, 0464},
    {"uog+xw", 0651, 014, 0337},
    {"ga=sr", 0545, 0764, 06444},
    {"g+g", 0156, 0776, 0776},
    {"go-tr", 0427, 026, 022},
    {"u+xtX", 0117, 0502, 0502},
    {"gu=x-tsX", 0053, 0600, 00},
    {"u+s", 0771, 0325, 04325},
    {"u=tw", 0623, 040, 0240},
    {"g+x-w", 0105, 0343, 0353},
    {"oa-u", 0736, 060, 060},
    {"ao=X", 0761, 0526, 0111},
    {"oug+X", 0211, 0647, 0757},
    {"gu+w", 0520, 0154, 0374},
    {"g-r", 0503, 0321, 0321},
    {"ou=tw", 0617, 0347, 01242},
    {"uog+Xs", 0117, 01760, 07771},
    {"og+t", 0472, 0431, 01431},
    {"ao+g", 0216, 02107, 02107},
    {"o=wt", 0310, 04026, 05022},
    {"a=Xw", 0156, 0222, 0222},
    {"uo+t", 0266, 0516, 01516},
    {"ua-t", 0076, 0623, 0623},
    {"u-w=rX", 0460, 0132, 0532},
    {"u=r", 0373, 0222, 0422},
    {"a=x", 0561, 0161, 0111},
    {"o=tX", 0410, 0514, 01511},
    {"ua-o", 0006, 0241, 0240},
    {"uo+wX", 0311, 0374, 0377},
    {"044", 0622, 02337, 044},
    {"u=t", 0333, 02627, 02027},
    {"415", 0163, 0355, 0415},
    {"u+Xsw", 0123, 0423, 04723},
    {"700", 0767, 01007, 0700},
    {"240", 0642, 0523, 0240},
    {"366", 0621, 0333, 0366},
    {"025", 0200, 0377, 025},
    {"a+Xr", 0536, 016, 0557},
    {"og-o", 0724, 0217, 0200},
    {"o=sx", 0131, 0313, 0311},
    {"go=w", 0213, 0116, 0122},
    {"o-o", 0421, 0441, 0440},
    {"+0255", 0162, 0664, 0675},
    {"og=u", 0207, 0243, 0222},
    {"g+w", 0050, 0651, 0671},
    {"170", 0275, 0376, 0170},
    {"go=st", 0630, 0401, 03400},
    {"uo+xr", 0004, 0150, 0555},
    {"go-w", 0131, 0471, 0451},
    {"ou=g", 0770, 0446, 0444},
    {"ogu-wrX", 0323, 04420, 04000},
    {"411", 0333, 0631, 0411},
    {"oa=Xsr", 0522, 0616, 06555},
    {"og=tX", 0167, 0263, 01211},
    {"o-xwX", 0556, 0462, 0460},
    {"a-swx", 0444, 0624, 0404},
    {"og-u", 0225, 0415, 0411},
    {"u-xr", 0503, 0266, 0266},
    {"ua+s", 0670, 04323, 06323},
    {"g=r", 0243, 0560, 0540},
    {"a=u", 0360, 05, 00},
    {"go=wsr", 0123, 02573, 02566},
    {"aug+rXs", 0570, 055, 06555},
    {"gu-wr", 0140, 0767, 0107},
    {"a+g", 0523, 0646, 0646},
    {"gao+x", 0717, 020, 0131},
    {"2502", 0567, 0504, 02502},
    {"a+g", 0646, 02725, 02727},
    {"u-s", 0400, 0532, 0532},
    {"uog=u", 0167, 0401, 0444},
    {"u=X-xXt", 0153, 01564, 01064},
    {"g+g", 0745, 0606, 0606},
    {"-0750", 0031, 0340, 00},
    {"ug+x-txw", 0752, 0327, 07},
    {"a-ws", 0537, 0200, 00},
    {"o-g", 0530, 0315, 0314},
    {"a=rt", 0261, 0400, 01444},
    {"oug+sw", 0573, 0225, 06227},
    {"g+x", 0725, 0705, 0715},
    {"o=Xxt", 0516, 0113, 01111},
    {"g=Xws", 0676, 0443, 02433},
    {"u-twX", 0334, 066, 066},
    {"u=rXw", 0535, 0744, 0744},
    {"ou+Xx", 0005, 0101, 0101},
    {"a=rxw", 0770, 00, 0777},
    {"oa+X", 0653, 0243, 0353},
    {"uao+w", 0701, 022, 0222},
    {"ogu+rwx", 0372, 0427, 0777},
    {"gu=sw", 0270, 0207, 06227},
    {"u=s-X", 0126, 0526, 04026},
    {"g=g", 0316, 01071, 01071},
    {"ogu-ws", 0752, 0771, 0551},
    {"oua+u", 0262, 02721, 02777},
    {"ga+X", 0717, 02, 02},
    {"ao-s=X", 0003, 0617, 0111},
    {"goa=rwx", 0014, 01772, 0777},
    {"ao-ws", 0130, 0214, 014},
    {"u=wt", 0555, 0200, 0200},
    {"g+Xr", 0066, 0200, 0240},
    {"ag+Xs", 0644, 0161, 06171},
    {"=623", 0623, 0532, 0623},
    {"oa=u", 0465, 0475, 0444},
    {"go=g", 0272, 0412, 0411},
    {"g+rx", 0227, 0652, 0652},
    {"au+xs", 0741, 013, 06113},
    {"ag+g", 0603, 01067, 01667},
    {"go-x", 0632, 0326, 0326},
    {"o-u", 0423, 0371, 0370},
    {"134", 0206, 02151, 0134},
    {"g-t", 0413, 043, 043},
    {"a+Xxt", 0534, 0224, 01335},
    {"u=r", 0306, 0576, 0476},
    {"a-rw", 0615, 021, 01},
    {"u=Xs", 0106, 0647, 04147},
    {"oa=x", 0327, 0532, 0111},
    {"a=s", 0217, 0573, 06000},
    {"ao=xXt", 0710, 0267, 01111},
    {"au=tr", 0263, 0462, 01444},
    {"oa+Xr", 0173, 0404, 0444},
    {"ou+sX", 0415, 011, 04111},
    {"au=X", 0321, 0135, 0111},
    {"u=ws", 0015, 0346, 04246},
    {"ag+r=s-r", 0265, 02325, 06000},
    {"ug-rs", 0456, 0606, 0206},
    {"o-o", 0413, 0433, 0430},
    {"ua-trX", 0607, 01575, 020},
    {"ug-t", 0545, 0541, 0541},
    {"a=Xt", 0644, 04204, 01000},
    {"ug-sr", 0736, 0166, 0126},
    {"u-sw", 0731, 04555, 0555},
    {"560", 0641, 0141, 0560},
    {"g-o", 0313, 0676, 0616},
    {"auo+x", 0643, 0452, 0553},
    {"uoa+Xt", 0377, 00, 01000},
    {"ug-u", 0631, 0466, 026},
    {"ug=s", 0740, 02570, 06000},
    {"a=X", 0412, 0103, 0111},
    {"ga+s-t", 0021, 0752, 06752},
    {"745", 0553, 0204, 0745},
    {"o+s", 0711, 0163, 0163},
    {"o=g", 0760, 0223, 0222},
    {"u+xt", 0553, 0504, 0504},
    {"ga-wx", 0207, 01364, 01044},
    {"oua+tr", 0757, 0742, 01746},
    {"g-Xxs", 0377, 04011, 04001},
    {"ua+xt", 0460, 0410, 01511},
    {"1215", 0166, 0413, 01215},
    {"ou-Xxr", 0174, 0400, 00},
    {"o=Xw", 0300, 0502, 0503},
    {"uo+sw", 0546, 0674, 04676},
    {"u-u", 0425, 02423, 02023},
    {"ua+g", 0555, 0454, 0555},
    {"oa-r", 0372, 0671, 0231},
    {"o+g", 0556, 0115, 0115},
    {"ga+s-x", 0035, 0143, 06042},
    {"a-x", 0200, 02342, 02242},
    {"o+x", 0722, 0662, 0663},
    {"u-x", 0470, 02357, 02257},
    {"g-xw", 0330, 01544, 01544},
    {"g+Xsw", 0153, 0440, 02460},
    {"154", 0604, 0131, 0154},
    {"au=ts", 0220, 0306, 07000},
    {"oa=r", 0751, 04445, 0444},
    {"ua=X", 0110, 0660, 00},
    {"ua+t", 0114, 015, 01015},
    {"au+tX", 0716, 0546, 01557},
    {"guo+rt", 0602, 0341, 01745},
    {"ao-Xtw", 0336, 062, 040},
    {"gu-r", 0510, 0162, 0122},
    {"u-tr", 0063, 0540, 0140},
    {"ao-Xx", 0056, 04432, 04422},
    {"au=Xxw", 0137, 0235, 0333},
    {"u+s", 0514, 0577, 04577},
    {"go=g", 0615, 0355, 0355},
    {"go-tX", 0301, 0575, 0564},
    {"oa=o", 0610, 0336, 0666},
    {"au+xX", 0343, 0343, 0353},
    {"ou+s", 0627, 01773, 05773},
    {"+731", 0513, 0414, 0735},
    {"u-xt", 0544, 0331, 0231},
    {"aou=t", 0723, 0756, 01000},
    {"o+s", 0576, 0612, 0612},
    {"a+rs", 0045, 0400, 06444},
    {"g+s", 0052, 0136, 02136},
    {"g+wr", 0042, 0275, 0275},
    {"562", 0741, 04714, 0562},
    {"ugo-sX", 0474, 0633, 0622},
    {"a=t", 0011, 0361, 01000},
    {"a+tw", 0051, 0706, 01726},
    {"-621", 0260, 0641, 040},
    {"go+sX", 0654, 0645, 02655},
    {"og+u", 0102, 0444, 0444},
    {"oga=Xx", 0376, 0403, 0111},
    {"uo=X-tx", 0660, 061, 060},
    {"gu=r", 0226, 0226, 0446},
    {"o+o", 0501, 02614, 02614},
    {"ua-g", 0500, 01103, 01103},
    {"u-w", 0424, 0675, 0475},
    {"u=sw", 0467, 0323, 04223},
    {"u+x", 0753, 02413, 02513},
    {"oa-u", 0623, 067, 067},
    {"gu=t", 0431, 0605, 05},
    {"a+Xs", 0506, 0770, 06771},
    {"oa-X", 0374, 0314, 0204},
    {"u+Xt", 0364, 04146, 04146},
    {"ug-tX", 0754, 0562, 0462},
    {"ou+st", 0026, 0720, 05720},
    {"=137", 0255, 0152, 0137},
    {"o=rw", 0724, 040, 046},
    {"gou+tr", 0430, 0115, 01555},
    {"2055", 0146, 0757, 02055},
    {"go-u", 0745, 022, 022},
    {"ag+o", 0502, 0346, 0766},
    {"a-Xsw", 0130, 0656, 0444},
    {"uga+tw", 0311, 0445, 01667},
    {"ga-r=xw", 0172, 0600, 0333},
    {"ua=tw", 0564, 0533, 01222},
    {"gau=srw", 0555, 0570, 06666},
    {"u+xw", 0151, 0673, 0773},
    {"ou-wXr", 0255, 0503, 00},
    {"207", 0110, 0272, 0207},
    {"a=X", 0115, 0441, 0111},
    {"g-twr", 0104, 01527, 01507},
    {"=133", 0023, 0767, 0133},
    {"a=o", 0263, 0626, 0666},
    {"o-sX", 0203, 0502, 0502},
    {"ou+wst", 0717, 0550, 05752},
    {"u-X", 0352, 0526, 0426},
    {"oa=srx", 0745, 0533, 06555},
    {"a-rX", 0054, 0326, 0222},
    {"goa+g", 0172, 0345, 0745},
    {"oug+xw", 0043, 01310, 01333},
    {"770", 0013, 013, 0770},
    {"au=o", 0533, 0646, 0666},
    {"g+o", 0365, 0130, 0130},
    {"a+t", 0303, 0404, 01404},
    {"og+g", 0762, 0741, 0745},
    {"au+t", 0773, 0142, 01142},
    {"oua-x", 0503, 0522, 0422},
    {"u-Xsr", 0046, 0277, 0277},
    {"a-x", 0366, 0442, 0442},
    {"o-xt", 0517, 0674, 0674},
    {"uao=tx", 0370, 0145, 01111},
    {"-244", 0631, 010, 010},
    {"uog=u", 0003, 064, 00},
    {"u=o", 0555, 0151, 0151},
    {"u-tXx", 0005, 0777, 0677},
    {"a=Xsw", 0643, 0301, 06333},
    {"o-Xs", 0235, 0274, 0274},
    {"ou=sX", 0754, 0403, 04101},
    {"o-xr", 0521, 0365, 0360},
    {"o+w", 0537, 0501, 0503},
    {"u+s", 0704, 0536, 04536},
    {"201", 0665, 04755, 0201},
    {"ag-o", 0065, 0704, 0300},
    {"g+r", 0764, 011, 051},
    {"au-o", 0644, 0757, 00},
    {"ao-twx", 0231, 0641, 0440},
    {"agu-u", 0656, 0634, 010},
    {"u-wt", 0212, 0672, 0472},
    {"u=rw", 0555, 020, 0620},
    {"u+wr", 0674, 0271, 0671},
    {"o=t+ts", 0154, 020, 01020},
    {"o+X", 0741, 0456, 0457},
    {"ou-o", 0474, 0622, 0420},
    {"u+xX", 0610, 042, 0142},
    {"oua=r", 0657, 0322, 0444},
    {"ao+Xs", 0721, 0571, 06571},
    {"go=g", 0511, 0542, 0544},
    {"a+twr", 0377, 0141, 01767},
    {"g=ts", 0313, 0357, 02307},
    {"u-u", 0232, 0626, 026},
    {"u=ws", 0707, 0402, 04202},
    {"ou+X", 0777, 0622, 0622},
    {"a=X", 0571, 0566, 0111},
    {"uga=x", 0457, 0226, 0111},
    {"u-w", 0767, 0703, 0503},
    {"auo=x", 0326, 0426, 0111},
    {"360", 0236, 013, 0360},
    {"oa-x", 0562, 0343, 0242},
    {"gu+sr", 0641, 0761, 06761},
    {"052", 0273, 0452, 052},
    {"ao-tw", 0070, 034, 014},
    {"gou-rs", 0264, 0703, 0303},
    {"ao=rw", 0623, 0730, 0666},
    {"uog=r", 0236, 0561, 0444},
    {"g=t+sr", 0056, 0224, 02244},
    {"u+w", 0116, 0250, 0250},
    {"ga+o", 0677, 00, 00},
    {"g+t", 0107, 0370, 0370},
    {"aog-tXx", 0265, 0637, 0626},
    {"ua+t", 0035, 0761, 01761},
    {"uo=o", 0547, 0652, 0252},
    {"og-rX", 0312, 0173, 0122},
    {"u=r", 0031, 0624, 0424},
    {"ua-Xx", 0376, 0677, 0666},
    {"ao=Xt", 0714, 0506, 01111},
    {"g+wX", 0270, 0330, 0330},
    {"u-s", 0534, 0143, 0143},
    {"111", 0436, 054, 0111},
    {"021", 0433, 033, 021},
    {"oa-tx", 0600, 0667, 0666},
    {"u+tsw", 0215, 0462, 04662},
    {"1476", 0504, 0465, 01476},
    {"711", 0207, 036, 0711},
    {"ao+xs", 0767, 0572, 06573},
    {"u=o", 0642, 0615, 0515},
    {"ug=xt", 0032, 0375, 0115},
    {"oa-r", 0137, 0125, 0121},
    {"uoa+twx", 0212, 0424, 01737},
    {"520", 0646, 020, 0520},
    {"a+tr", 0445, 0220, 01664},
    {"og+g", 0674, 0232, 0233},
    {"oga=r", 0172, 02717, 0444},
    {"ao+rXw", 0172, 0461, 0777},
    {"oag-tsX", 0575, 0527, 0426},
    {"a+trw", 0155, 06, 01666},
    {"ago-tX", 0677, 0532, 0422},
    {"ua+rs", 0101, 016, 06456},
    {"g-rsw", 0567, 0321, 0301},
    {"u=Xs", 0103, 0525, 04125},
    {"a-u", 0670, 0537, 022},
    {"o=o", 0037, 0522, 0522},
    {"au=ts", 0723, 01242, 07000},
    {"ao+rxw", 0534, 0655, 0777},
    {"oa-xtX", 0113, 056, 046},
    {"u=s", 0130, 0747, 04047},
    {"go+x-w", 0142, 0364, 0355},
    {"o=wr", 0355, 0420, 0426},
    {"oa-o", 0415, 0240, 0240},
    {"agu-o", 0426, 0247, 00},
    {"ug-wsx", 0753, 0647, 0447},
    {"gu-rwt", 0534, 0304, 0104},
    {"au=ws", 0400, 04034, 06222},
    {"o-s", 0114, 0405, 0405},
    {"og-t", 0442, 0364, 0364},
    {"a-u", 0505, 0527, 022},
    {"gu=t-sw", 0114, 01602, 01002},
    {"a=g", 0677, 0146, 0444},
    {"oug+xs", 0342, 0370, 06371},
    {"ao+xX", 0000, 0377, 0377},
    {"ou=Xx", 0176, 0206, 0101},
    {"u+s", 0525, 0657, 04657},
    {"go-g", 0337, 0165, 0101},
    {"og+g", 0411, 0266, 0266},
    {"ou=r+X-xw", 0322, 04477, 0474},
    {"u=o", 0050, 0125, 0525},
    {"a+t", 0327, 0355, 01355},
    {"o-rxw", 0064, 01630, 01630},
    {"ag=Xr", 0737, 0531, 0555},
    {"oa=rXw", 0234, 0327, 0777},
    {"622", 0562, 0757, 0622},
    {"au-x", 0473, 00, 00},
    {"ag-w", 0465, 0373, 0151},
    {"oa=xsX", 0337, 00, 06111},
    {"ug+tw", 0055, 040, 0260},
    {"o-o", 0320, 0761, 0760},
    {"o-s", 0163, 0147, 0147},
    {"ga+X", 0600, 0175, 0175},
    {"ug=s", 0061, 0601, 06001},
    {"u+sX", 0327, 01331, 05331},
    {"u=s", 0157, 0217, 04017},
    {"ua=g", 0545, 0773, 0777},
    {"u=Xx", 0653, 0727, 0127},
    {"oa-ws", 0506, 0763, 0541},
    {"a=Xs", 0710, 0524, 06111},
    {"ug=xX", 0341, 0525, 0115},
    {"ag+u", 0303, 04, 04},
    {"ug=u", 0304, 0267, 0227},
    {"154", 0234, 0642, 0154},
    {"gu-wrx", 0577, 0200, 00},
    {"go=xt", 0004, 0777, 01711},
    {"ogu+Xt", 0654, 0171, 01171},
    {"gau=x", 0435, 0140, 0111},
    {"g+u", 0624, 075, 075},
    {"go+t", 0551, 0607, 01607},
    {"ua+u", 0765, 012, 012},
    {"gau=w", 0273, 0471, 0222},
    {"ug+Xwt", 0164, 0672, 0772},
    {"og=xs", 0373, 0654, 02611},
    {"go-t+xw", 0016, 0534, 0537},
    {"uog+u", 0322, 0521, 0575},
    {"g-Xwx", 0361, 0610, 0600},
    {"a=s", 0213, 0647, 06000},
    {"gu=X", 0665, 0156, 0116},
    {"o-X+swt", 0416, 01076, 01076},
    {"a-r", 0357, 0574, 0130},
    {"oga-u", 0077, 02203, 02001},
    {"u-t", 0360, 030, 030},
    {"g+o", 0645, 0462, 0462},
    {"ao+sx", 0052, 0221, 06331},
    {"ao+x", 0536, 0753, 0753},
    {"353", 0751, 0431, 0353},
    {"ogu=x+sxX", 0613, 0167, 06111},
    {"oau=rsw", 0030, 0313, 06666},
    {"g-Xr", 0237, 0520, 0520},
    {"oua+X", 0401, 0515, 0515},
    {"au-u", 0711, 0512, 02},
    {"oug+o", 0035, 0753, 0773},
    {"ao-xX", 0670, 01776, 01666},
    {"uog-xw", 0660, 0626, 0404},
    {"u+s", 0054, 0252, 04252},
    {"oa+u", 0437, 0536, 0577},
    {"au=tw", 0776, 0403, 01222},
    {"ga-w", 0705, 0471, 0451},
    {"ag=g", 0404, 01226, 0222},
    {"ag+xtw", 0144, 023, 01333},
    {"ag+rt", 0451, 06, 01446},
    {"u=u", 0537, 040, 040},
    {"oa+u", 0643, 04267, 04267},
    {"og+Xxw", 0414, 04076, 04077},
    {"uag=wX", 0620, 0152, 0333},
    {"2256", 0470, 0533, 02256},
    {"o=r+w", 0756, 04177, 04176},
    {"o-tw", 0277, 0245, 0245},
    {"ug-o", 0052, 01207, 01007},
    {"ag=w-wx", 0674, 0704, 00},
    {"og=wX", 0777, 0161, 0133},
    {"o-xXt", 0517, 0772, 0772},
    {"u-w+Xr", 0115, 0467, 0567},
    {"u-g", 0373, 01370, 01070},
    {"ag=o", 0760, 0355, 0555},
    {"4620", 0066, 0404, 04620},
    {"103", 0170, 055, 0103},
    {"ua=t", 0067, 0334, 01000},
    {"ogu+rx", 0553, 0634, 0775},
    {"oa+g", 0676, 02216, 02317},
    {"au-t", 0222, 0434, 0434},
    {"ag=r", 0426, 0205, 0444},
    {"+220", 0605, 060, 0260},
    {"au-X", 0424, 04040, 04040},
    {"ao+r", 0050, 0661, 0665},
    {"uo-Xx", 0271, 0717, 0616},
    {"a-g", 0601, 024, 04},
    {"235", 0254, 01444, 0235},
    {"a+xtw", 0360, 0547, 01777},
    {"ga=x", 0310, 0262, 0111},
    {"a-st", 0744, 0767, 0767},
    {"ga-Xt", 0204, 0704, 0604},
    {"u=w", 0415, 0333, 0233},
    {"a-sw", 0642, 0361, 0141},
    {"gu+t", 0464, 0727, 0727},
    {"a+tsr", 0513, 040, 07444},
    {"uoa+o", 0512, 0500, 0500},
    {"au=u", 0367, 0667, 0666},
    {"gua-rX", 0235, 0402, 02},
    {"au+wr", 0330, 0566, 0766},
    {"uo=o", 0450, 0234, 0434},
    {"a+xtw", 0677, 0676, 01777},
    {"oua+o", 0204, 0532, 0732},
    {"og+x", 0535, 0264, 0275},
    {"a+s", 0065, 0204, 06204},
    {"g=w", 0115, 0370, 0320},
    {"ga-Xs", 0003, 0104, 04},
    {"ua=x", 0673, 0754, 0111},
    {"061", 0124, 026, 061},
    {"u+w", 0750, 0534, 0734},
    {"o=wrX", 0211, 067, 067},
    {"oa-xt", 0432, 0753, 0642},
    {"g-u", 0320, 0762, 0702},
    {"oug=rw", 0712, 040, 0666},
    {"351", 0032, 01575, 0351},
    {"u=tr", 0464, 0337, 0437},
    {"uog-u", 0530, 0425, 021},
    {"og=rXs", 0360, 0201, 02255},
    {"oug-g", 0574, 02314, 02204},
    {"o=Xt", 0303, 0404, 01400},
    {"o-u", 0104, 0305, 0304},
    {"135", 0653, 017, 0135},
    {"ga=st", 0137, 04625, 07000},
    {"go=x+wt", 0310, 0773, 01733},
    {"g=t", 0047, 014, 04},
    {"u+sX", 0261, 0421, 04521},
    {"753", 0250, 0406, 0753},
    {"771", 0130, 012, 0771},
    {"ug+rw", 0540, 02333, 02773},
    {"ou+st", 0300, 0513, 05513},
    {"g-wX", 0150, 0514, 0504},
    {"g+s", 0546, 0223, 02223},
    {"oa-g", 0733, 015, 04},
    {"o+s", 0220, 01453, 01453},
    {"g+xt", 0520, 0346, 0356},
    {"go-t", 0743, 04127, 04127},
    {"a=wt", 0407, 0663, 01222},
    {"o+o", 0613, 021, 021},
    {"ga-xr", 0552, 0205, 0200},
    {"go+rxs", 0445, 0276, 02277},
    {"2073", 0175, 0444, 02073},
    {"a+g", 0565, 0627, 0627},
    {"oa+rw", 0053, 02667, 02667},
    {"go=tw", 0770, 0366, 01322},
    {"g-sx", 0762, 04, 04},
    {"og=u", 0457, 0271, 0222},
    {"ao-sx", 0444, 0746, 0646},
    {"ou-s", 0203, 0743, 0743},
    {"aog+xr", 0067, 0256, 0757},
    {"oa-ws", 0606, 0214, 014},
    {"oug=x", 0125, 0640, 0111},
    {"o-xs", 0724, 04664, 04664},
    {"o+o", 0025, 042, 042},
    {"og-Xw", 0117, 0327, 0304},
    {"ao+wsr", 0173, 0344, 06766},
    {"g-ws", 0002, 0224, 0204},
    {"gu=X", 0114, 0255, 0115},
    {"ga+r", 0151, 0220, 0664},
    {"au+X", 0360, 0721, 0731},
    {"u-t+r", 0472, 0721, 0721},
    {"aog-wX", 0360, 0736, 0404},
    {"o-rX", 0316, 0364, 0360},
    {"aog+s", 0753, 0315, 06315},
    {"a+swt", 0671, 0276, 07276},
    {"u=Xs", 0617, 0114, 04114},
    {"u=u", 0453, 07, 07},
    {"a=sX", 0316, 0122, 06111},
    {"g-Xw", 0531, 0210, 0200},
    {"oa+Xwt", 0500, 0153, 01373},
    {"og-g", 0605, 0665, 0601},
    {"o=wX", 0655, 0470, 0473},
    {"+063", 0526, 0673, 0673},
    {"a-s", 0431, 0761, 0761},
    {"g+rw", 0773, 04754, 04774},
    {"u=s", 0304, 0262, 04062},
    {"go=t", 0407, 0307, 01300},
    {"ao-sr", 0705, 0766, 0322},
    {"oa-wX", 0006, 0615, 0404},
    {"aog=o", 0127, 0702, 0222},
    {"oa=sxw", 0225, 046, 06333},
    {"gu+s", 0636, 04412, 06412},
    {"ua=u", 0553, 0734, 0777},
    {"543", 0653, 0745, 0543},
    {"og+X", 0251, 0547, 0557},
    {"o+u", 0772, 0413, 0417},
    {"uog=Xs", 0063, 0173, 06111},
    {"gu+xX", 0424, 0617, 0717},
    {"o=g", 0344, 071, 077},
    {"a+Xw", 0311, 0127, 0337},
    {"ag-w", 0320, 0406, 0404},
    {"u+o", 0130, 0657, 0757},
    {"gu-X", 0750, 01675, 01665},
    {"uo-o", 0741, 02342, 02140},
    {"o-wtX", 0016, 0323, 0320},
    {"a+s", 0522, 0456, 06456},
    {"ou=X", 0254, 0442, 040},
    {"a-Xs", 0274, 0306, 0206},
    {"ao-g", 0031, 01700, 01700},
    {"u-g", 0023, 0672, 072},
    {"o-x", 0451, 060, 060},
    {"uo-r=Xt", 0107, 0730, 01131},
    {"g=Xx", 0413, 0510, 0510},
    {"a+r", 0200, 064, 0464},
    {"uo+u", 0306, 0474, 0474},
    {"ao=u", 0103, 012, 00},
    {"uo-tx", 0466, 0527, 0426},
    {"ua=wt", 0637, 0775, 01222},
    {"gou-t", 0527, 0131, 0131},
    {"=344", 0331, 035, 0344},
    {"u-twX", 0213, 0745, 0445},
    {"o+X", 0307, 0405, 0405},
    {"u+t", 0351, 0175, 0175},
    {"4274", 0305, 0331, 04274},
    {"ga=Xw", 0676, 0165, 0333},
    {"ug=Xr", 0523, 0626, 0446},
    {"g=o", 0144, 0510, 0500},
    {"ga=r", 0076, 0561, 0444},
    {"u-wx", 0743, 0713, 0413},
    {"u+ts", 0671, 02, 04002},
    {"au=st", 0550, 0364, 07000},
    {"a=w", 0113, 0767, 0222},
    {"u+r", 0346, 011, 0411},
    {"ua-swr", 0527, 0424, 00},
    {"u=x", 0204, 0201, 0101},
    {"712", 0511, 01203, 0712},
    {"uag+o", 0070, 0760, 0760},
    {"au-wX", 0304, 0301, 00},
    {"0375", 0413, 02155, 0375},
    {"ag=o", 0270, 01124, 0444},
    {"ag-X", 0161, 0641, 0640},
    {"ua=xX", 0416, 0573, 0111},
    {"aug=w", 0341, 02361, 0222},
    {"u=Xx", 0740, 0174, 0174},
    {"ou+ts", 0212, 074, 05074},
    {"uo+g", 0543, 0200, 0200},
    {"go-tsr", 0223, 02132, 0132},
    {"au=w", 0130, 0777, 0222},
    {"uo-Xx", 0333, 0107, 06},
    {"a+Xw", 0110, 0171, 0373},
    {"g=ts", 0031, 0514, 02504},
    {"u=s", 0617, 0121, 04021},
    {"a+r", 0645, 0677, 0677},
    {"u=Xsw", 0047, 0707, 04307},
    {"a-o", 0322, 04607, 04000},
    {"547", 0602, 0325, 0547},
    {"gu-w", 0215, 0546, 0546},
    {"+253", 0630, 0342, 0353},
    {"g+rs", 0777, 0776, 02776},
    {"u=rw", 0757, 0676, 0676},
    {"ou-s", 0062, 0223, 0223},
    {"u+w", 0065, 01241, 01241},
    {"u-r", 0573, 02156, 02156},
    {"a-u", 0635, 04751, 04000},
    {"go=rtw", 0333, 0361, 01366},
    {"uo=g", 0106, 0550, 0555},
    {"a=g", 0501, 0271, 0777},
    {"gu=u", 0316, 036, 06},
    {"go=o", 0135, 0677, 0677},
    {"gu+t", 0502, 077, 077},
    {"oa=Xw", 0661, 0102, 0333},
    {"-266", 0156, 072, 010},
    {"g+o", 0033, 0460, 0460},
    {"ug-u", 0766, 01651, 01011},
    {"au=Xt", 0721, 0161, 01111},
    {"ag+g", 0516, 0146, 0546},
    {"o-Xwx", 0523, 0356, 0354},
    {"ga=X", 0302, 0550, 0111},
    {"a=wx", 0705, 0724, 0333},
    {"a-o", 0331, 01171, 01060},
    {"aog+Xx", 0200, 0333, 0333},
    {"ua-tw", 0465, 0332, 0110},
    {"oag=w", 0614, 074, 0222},
    {"4373", 0462, 0153, 04373},
    {"uo-xr", 0567, 0210, 0210},
    {"a-o", 0441, 0160, 0160},
    {"ga=x", 0703, 0637, 0111},
    {"ou=sXr", 0211, 0501, 04505},
    {"gu+X=stx", 0451, 0327, 06117},
    {"g+o", 0055, 075, 075},
    {"o+Xt", 0660, 0537, 01537},
    {"uo-sx", 0564, 0462, 0462},
    {"oug-rw", 0354, 0632, 010},
    {"ua+X", 0376, 0677, 0777},
    {"oua=ts", 0724, 00, 07000},
    {"a=sr", 0005, 00, 06444},
    {"g-w", 0026, 0777, 0757},
    {"o=u", 0675, 0353, 0353},
    {"761", 0346, 0330, 0761},
    {"ao=t+rsx", 0776, 0723, 07555},
    {"460", 0205, 04060, 0460},
    {"uoa-w", 0460, 0204, 04},
    {"g-xX", 0350, 0453, 0443},
    {"go=sXw", 0114, 0620, 02622},
    {"au-t", 0362, 024, 024},
    {"u-X", 0155, 0111, 011},
    {"g=xrw", 0060, 0774, 0774},
    {"uo-rx", 0446, 0327, 0222},
    {"ga-X", 0132, 0420, 0420},
    {"g+t", 0151, 0621, 0621},
    {"432", 0502, 0404, 0432},
    {"u+o", 0032, 00, 00},
    {"og-X", 0440, 04377, 04366},
    {"ou=Xx", 0063, 0220, 0121},
    {"ua-g", 0042, 0136, 04},
    {"ag-ws", 0362, 0640, 0440},
    {"uoa=xr", 0575, 0246, 0555},
    {"155", 0435, 0353, 0155},
    {"2354", 0670, 0722, 02354},
    {"og+X", 0721, 0357, 0357},
    {"ou+u", 0407, 0516, 0517},
    {"o-rx", 0710, 0163, 0162},
    {"u+t", 0443, 0203, 0203},
    {"go=o", 0243, 0406, 0466},
    {"ug-u", 0670, 0473, 033},
    {"uo=rw", 0565, 02245, 02646},
    {"oau=o", 0430, 00, 00},
    {"uao+u", 0622, 0345, 0377},
    {"ga=rw", 0335, 0664, 0666},
    {"uo-ws", 0662, 0577, 0575},
    {"u-rsx", 0257, 0364, 0264},
    {"og=trX", 0747, 0672, 01655},
    {"675", 0740, 0277, 0675},
    {"171", 0271, 01277, 0171},
    {"uao-tXx", 0150, 0677, 0666},
    {"a-s+sw", 0136, 0170, 06372},
    {"ug+x", 0447, 00, 0110},
    {"uo+u", 0006, 0254, 0256},
    {"ag+g", 0661, 06, 06},
    {"a=str", 0670, 032, 07444},
    {"+112", 0774, 01151, 01153},
    {"g+Xsr", 0112, 0653, 02653},
    {"o+swX", 0314, 01527, 01527},
    {"ao=sXx", 0222, 0451, 06111},
    {"u+wt", 0520, 0774, 0774},
    {"ao-t", 0560, 0104, 0104},
    {"oa-t+X=x", 0124, 0567, 0111},
    {"ua-t", 0042, 0507, 0507},
    {"gu=o", 0067, 0410, 00},
    {"og=x", 0301, 0605, 0611},
    {"g=tX", 0425, 01, 011},
    {"ga=u", 0611, 0126, 0111},
    {"g=Xx", 0503, 0446, 0416},
    {"guo=u", 0034, 0636, 0666},
    {"uga=sXt", 0146, 0140, 07111},
    {"gu-xtr", 0032, 0370, 0220},
    {"u-t", 0476, 0574, 0574},
    {"a=w", 0642, 0724, 0222},
    {"og-t", 0672, 033, 033},
    {"o=Xr", 0422, 0421, 0425},
    {"u+Xtw", 0735, 0125, 0325},
    {"gu=wxt", 0546, 0546, 0336},
    {"ga=X", 0262, 0372, 0111},
    {"o+u", 0573, 0510, 0515},
    {"g-w", 0650, 0172, 0152},
    {"ou=xX", 0171, 0437, 0131},
    {"g+X", 0471, 0422, 0422},
    {"au+xw", 0541, 0551, 0773},
    {"757", 0415, 0575, 0757},
    {"og=sw", 0566, 0422, 02422},
    {"go+u", 0264, 02275, 02277},
    {"a-X=rx", 0271, 0446, 0555},
    {"ao=xX", 0704, 0711, 0111},
    {"uog-X", 0027, 041, 040},
    {"705", 0062, 0433, 0705},
    {"g+g", 0614, 0707, 0707},
    {"o=g", 0431, 010, 011},
    {"u+xsr", 0125, 0701, 04701},
    {"oa-r", 0500, 042, 02},
    {"ua=s", 0346, 02625, 06000},
    {"uo=s", 0070, 0347, 04040},
    {"og=u", 0317, 0616, 0666},
    {"ou=tX", 0565, 0663, 01161},
    {"uag+xXr", 0433, 0600, 0755},
    {"ua-rX", 0166, 0252, 0202},
    {"og-s", 0265, 04645, 04645},
    {"uga=twr", 0607, 0573, 01666},
    {"ug-rX", 0320, 0647, 0207},
    {"ou+w=x", 0617, 0307, 0101},
    {"go=r", 0672, 044, 044},
    {"a+s", 0003, 0764, 06764},
    {"o-sr", 0731, 01722, 01722},
    {"543", 0415, 0565, 0543},
    {"gu-tw", 0470, 0621, 0401},
    {"g=srt", 0314, 0402, 02442},
    {"ag=x", 0175, 0555, 0111},
    {"ou+x-s", 0554, 04470, 0571},
    {"ga=r", 0553, 0777, 0444},
    {"ugo=r-x", 0236, 065, 0444},
    {"a+t", 0142, 0143, 01143},
    {"gu-t+rXx", 0505, 0535, 0575},
    {"uoa+r", 0373, 0533, 0577},
    {"g-tr", 0751, 037, 037},
    {"g+w", 0463, 0162, 0162},
    {"go+o", 0270, 0557, 0577},
    {"ua-u", 0277, 0266, 044},
    {"u-r", 0612, 0570, 0170},
    {"ug+x", 0200, 0225, 0335},
    {"g-r", 0246, 0401, 0401},
    {"uao+o", 0430, 0463, 0773},
    {"uog=o", 0027, 0337, 0777},
    {"u-rws", 0132, 0702, 0102},
    {"g=Xw", 0066, 044, 024},
    {"og-s", 0721, 0323, 0323},
    {"2053", 0111, 056, 02053},
    {"au=t", 0004, 0244, 01000},
    {"ao=ts", 0631, 0535, 07000},
    {"o-g", 0574, 01037, 01034},
    {"ga+Xx", 0240, 030, 0131},
    {"ao+x=t", 0450, 041, 01000},
    {"a=sx", 0611, 0133, 06111},
    {"uoa=w", 0221, 01276, 0222},
    {"ug+Xs", 0126, 0411, 06511},
    {"o=t", 0677, 0147, 01140},
    {"001", 0677, 0565, 01},
    {"gau=g", 0400, 0776, 0777},
    {"ou+w", 0106, 0260, 0262},
    {"ua+xw", 0640, 0340, 0373},
    {"g=xwt", 0071, 0756, 0736},
    {"o+o", 0552, 0763, 0763},
    {"400", 0071, 023, 0400},
    {"ug+g", 0112, 0253, 0753},
    {"+535", 0320, 0674, 0775},
    {"uo+X-Xs", 0502, 0406, 0406},
    {"uag+t", 0205, 0576, 01576},
    {"361", 0232, 0477, 0361},
    {"1346", 0672, 0232, 01346},
    {"ua-o", 0366, 0620, 0620},
    {"aou-x", 0757, 050, 040},
    {"uo-g", 0013, 0102, 0102},
    {"o=u", 0656, 0416, 0414},
    {"g+u", 0504, 0777, 0777},
    {"ag+g", 0221, 0210, 0311},
    {"u+trw", 0222, 04, 0604},
    {"u=o", 0222, 0600, 00},
    {"ag-X", 0450, 0536, 0426},
    {"ou-w", 0033, 0631, 0431},
    {"ao-x", 0347, 0355, 0244},
    {"oa-s", 0352, 0567, 0567},
    {"431", 0332, 0577, 0431},
    {"g-g", 0336, 0115, 0105},
    {"oag-w+sx", 0271, 0334, 06115},
    {"oga=rx", 0424, 0567, 0555},
    {"a=Xs", 0604, 0403, 06111},
    {"gu-t", 0045, 0664, 0664},
    {"ga=Xr", 0614, 04221, 0555},
    {"a+Xt", 0417, 0331, 01331},
    {"o+rX", 0030, 0767, 0767},
    {"a-o", 0366, 0522, 0500},
    {"ua+r", 0412, 0124, 0564},
    {"ua=u", 0552, 0107, 0111},
    {"a=t", 0222, 0553, 01000},
    {"go+t", 0631, 0530, 01530},
    {"uo=rtw", 0521, 0647, 01646},
    {"oga-t", 0772, 0432, 0432},
    {"gu=tw", 0444, 0347, 0227},
    {"auo=tws", 0400, 0667, 07222},
    {"o=tx", 0356, 0645, 01641},
    {"go-w", 0433, 0130, 0110},
    {"a-sxw", 0260, 01, 00},
    {"u+X", 0715, 0467, 0567},
    {"o+Xs", 0361, 0231, 0231},
    {"413", 0757, 0400, 0413},
    {"756", 0112, 0457, 0756},
    {"oga=sx", 0134, 02356, 06111},
    {"a=o", 0315, 02243, 0333},
    {"ag+wx", 0500, 0661, 0773},
    {"u=r", 0350, 053, 0453},
    {"ao+wX", 0777, 00, 0222},
    {"au=u", 0665, 0476, 0444},
    {"oug-w+ts", 0105, 0233, 07011},
    {"g=Xxr", 0730, 044, 054},
    {"a-Xw", 0115, 02663, 02440},
    {"730", 0361, 0174, 0730},
    {"oa+sw", 0400, 0620, 06622},
    {"oa+wX", 0320, 0123, 0333},
    {"743", 0640, 024, 0743},
    {"117", 0766, 0457, 0117},
    {"og=w+wt", 0730, 040, 01022},
    {"ug-Xw", 0130, 0750, 0440},
    {"gu+x-wrs", 0727, 040, 0110},
    {"ua=g", 0670, 01532, 0333},
    {"uag+sr", 0061, 031, 06475},
    {"u=r", 0607, 0405, 0405},
    {"ga+xr", 0154, 0444, 0555},
    {"u+s", 0114, 0500, 04500},
    {"go-r=X", 0655, 0636, 0611},
    {"u+u", 0047, 0233, 0233},
    {"uo=rX", 0511, 0770, 0575},
    {"a+xXw", 0042, 0236, 0337},
    {"u=Xwt", 0577, 01235, 01335},
    {"uo-xt", 0403, 054, 054},
    {"u=o", 0316, 00, 00},
    {"ou-srX", 0032, 0607, 0202},
    {"oa-s", 0155, 0710, 0710},
    {"gu=w", 0470, 056, 0226},
    {"ga+sX", 0302, 00, 06000},
    {"ag=tx", 0316, 0314, 01111},
    {"ga+xwX", 0134, 00, 0333},
    {"745", 0514, 0337, 0745},
    {"a=xrw", 0425, 0141, 0777},
    {"oag-rw", 0747, 0135, 0111},
    {"416", 0502, 01016, 0416},
    {"uo-u", 0531, 0451, 051},
    {"og-g", 0057, 037, 04},
    {"176", 0627, 0516, 0176},
    {"206", 0571, 0355, 0206},
    {"g+t", 0715, 05, 05},
    {"g=sXx", 0134, 0715, 02715},
    {"gu=Xx", 0476, 04362, 0112},
    {"ua+xr", 0052, 0215, 0755},
    {"ago-wts", 0047, 0662, 0440},
    {"u=xr", 0261, 012, 0512},
    {"ao+sX", 0533, 04501, 06511},
    {"o-stw", 0662, 0373, 0371},
    {"o-w", 0720, 0600, 0600},
    {"guo-wX", 0326, 0143, 040},
    {"ua=Xrw", 0162, 0600, 0666},
    {"g+rxs", 0012, 0643, 02653},
    {"go+rXx", 0763, 0455, 0455},
    {"ua+ts", 0333, 01237, 07237},
    {"g+rxw", 0536, 0513, 0573},
    {"og+Xr", 0053, 0401, 0455},
    {"gou+g", 0176, 0171, 0777},
    {"ag=u", 0300, 0452, 0444},
    {"u-Xt", 0314, 076, 076},
    {"g=x", 0542, 0267, 0217},
    {"oa+rw", 0457, 0710, 0776},
    {"oga-stw", 0356, 04453, 0451},
    {"ao-rts", 0655, 0434, 030},
    {"o-o", 0017, 01455, 01450},
    {"guo+t", 0302, 0765, 01765},
    {"ou=tX", 0163, 04664, 01060},
    {"273", 0657, 0436, 0273},
    {"ao+Xtr", 0731, 04046, 05446},
    {"ga=sw", 0104, 0547, 06222},
    {"741", 0005, 0713, 0741},
    {"4617", 0621, 0400, 04617},
    {"g+wrX", 0761, 057, 077},
    {"g=u", 0367, 0257, 0227},
    {"gu=xr", 0212, 02764, 0554},
    {"642", 0075, 0413, 0642},
    {"g+s=sw", 0652, 0630, 02620},
    {"u=x", 0446, 04657, 0157},
    {"o+u", 0444, 0601, 0607},
    {"o+Xr", 0401, 0526, 0527},
    {"g+t", 0547, 0371, 0371},
    {"aou-tr", 0064, 04722, 04322},
    {"o=x", 0625, 01703, 0701},
    {"gao-rXt", 0727, 0206, 0202},
    {"g+wx", 0575, 0321, 0331},
    {"uo-xwt", 0031, 0601, 0400},
    {"go+rw", 0750, 0255, 0277},
    {"o+wt", 0053, 015, 01017},
    {"oa=X", 0470, 04326, 0111},
    {"go=xs", 0256, 0432, 02411},
    {"o+X", 0631, 0303, 0303},
    {"a+r-xts", 0531, 0610, 0644},
    {"g+w", 0467, 071, 071},
    {"aog-g", 0627, 0675, 00},
    {"o=xtw", 0346, 0411, 01413},
    {"g+Xw", 0255, 0754, 0774},
    {"g=o", 0437, 021, 011},
    {"ou-rwt", 0005, 0672, 070},
    {"ou=wXx", 0365, 0502, 0303},
    {"+4103", 0714, 01132, 05133},
    {"og+ts", 0061, 0761, 03761},
    {"og+xXw", 0635, 0441, 0473},
    {"gu+o", 0427, 0675, 0775},
    {"oa-Xsr", 0000, 0447, 02},
    {"gau=u", 0134, 0306, 0333},
    {"o-t", 0301, 0777, 0777},
    {"uga+rs", 0652, 0667, 06667},
    {"uo=g", 0610, 0326, 0222},
    {"u=rst", 0433, 02006, 06406},
    {"ao+txX", 0636, 0103, 01113},
    {"g-o", 0703, 0660, 0660},
    {"gau+X", 0073, 0314, 0315},
    {"aug=x", 0204, 0666, 0111},
    {"ou-rs", 0531, 0727, 0323},
    {"oua-ts", 0021, 02070, 070},
    {"g-g", 0455, 0333, 0303},
    {"ug=xr", 0547, 0166, 0556},
    {"uao=t", 0622, 0312, 01000},
    {"go=wts", 0017, 0750, 03722},
    {"o+rx", 0442, 024, 025},
    {"oau=r", 0167, 0373, 0444},
    {"g+rs", 0710, 0260, 02260},
    {"u=rw", 0642, 0156, 0656},
    {"716", 0201, 01540, 0716},
    {"o=tx", 0362, 0404, 01401},
    {"uo+s", 0035, 0546, 04546},
    {"ogu=Xx", 0531, 02501, 0111},
    {"u-x", 0541, 0412, 0412},
    {"gua-r", 0202, 0423, 023},
    {"o=sx", 0132, 045, 041},
    {"a+str", 0136, 0637, 07677},
    {"ag=tr", 0300, 0720, 01444},
    {"ugo=tX", 0244, 0633, 01111},
    {"g+X", 0766, 0655, 0655},
    {"og-o", 0047, 026, 00},
    {"gu-sX", 0771, 0224, 0224},
    {"uao+sw", 0373, 053, 06273},
    {"ga-x", 0434, 0301, 0200},
    {"o+Xs", 0213, 047, 047},
    {"a=sX", 0776, 0152, 06111},
    {"a=Xs", 0565, 0412, 06111},
    {"uog+rxX", 0055, 0677, 0777},
    {"a+tr", 0467, 04731, 05775},
    {"aug=x", 0172, 02626, 0111},
    {"ao+w", 0771, 0244, 0266},
    {"oa=rt", 0055, 0357, 01444},
    {"ua=X", 0232, 0744, 0111},
    {"ua+ts", 0314, 0400, 07400},
    {"o+g", 0726, 0111, 0111},
    {"ao-w", 0600, 02055, 02055},
    {"g=w", 0146, 0445, 0425},
    {"uo+x", 0122, 01543, 01543},
    {"oga-x=Xsw", 0253, 0736, 06222},
    {"ou-sX", 0465, 0754, 0654},
    {"a=o", 0370, 0142, 0222},
    {"au+s=sr", 0273, 0730, 06444},
    {"g-x", 0422, 0255, 0245},
    {"737", 0335, 02375, 0737},
    {"ou=o", 0104, 01625, 0525},
    {"guo=X", 0577, 0475, 0111},
    {"oa-txw", 0133, 0617, 0404},
    {"g+x", 0652, 01043, 01053},
    {"g=o", 0610, 01110, 01100},
    {"g=rt", 0375, 0654, 0644},
    {"a+xt", 0557, 0430, 01531},
    {"o+Xx", 0473, 0764, 0765},
    {"o-g", 0125, 0507, 0507},
    {"og=u", 0444, 0443, 0444},
    {"g+r", 0052, 0745, 0745},
    {"ou+wsr", 0122, 040, 04646},
    {"o+rX", 0730, 0127, 0127},
    {"g+o", 0456, 0420, 0420},
    {"a-x", 0101, 0404, 0404},
    {"go=sx", 0337, 0440, 02411},
    {"o=x+s-t", 0372, 0175, 0171},
    {"o+o", 0536, 0655, 0655},
    {"og-g", 0013, 035, 04},
    {"og-st", 0301, 0410, 0410},
    {"ug=s", 0301, 0661, 06001},
    {"u-u", 0447, 0456, 056},
    {"o+s", 0021, 0650, 0650},
    {"go=X", 0734, 021, 011},
    {"uo+sxw", 0221, 052, 04353},
    {"ug=xsr", 0332, 0157, 06557},
    {"u-g", 0551, 0447, 047},
    {"oa-xt", 0514, 0111, 00},
    {"au-r+xs", 0573, 0376, 06333},
    {"u-g", 0251, 0577, 077},
    {"o+t", 0341, 01077, 01077},
    {"o+txw", 0451, 0172, 01173},
    {"ua-tw", 0407, 0225, 05},
    {"ag=st", 0241, 01553, 07000},
    {"gao-tX", 0021, 0275, 0264},
    {"ag-rsw", 0573, 0334, 0110},
    {"ao=wrX", 0522, 02104, 0777},
    {"og+X", 0606, 071, 071},
    {"o+g", 0135, 0142, 0146},
    {"gu+g", 0032, 0614, 0714},
    {"u-ws", 0606, 0345, 0145},
    {"go=r", 0325, 0502, 0544},
    {"ou+X-rs", 0017, 0511, 0111},
    {"aug+s=tX", 0132, 0205, 01111},
    {"u=o", 0672, 0672, 0272},
    {"0134", 0146, 0632, 0134},
    {"ou=twX", 0761, 0124, 01323},
    {"o+X", 0534, 0777, 0777},
    {"oau+x", 0106, 0753, 0753},
    {"g-w=srx", 0761, 0536, 02556},
    {"777", 0007, 01766, 0777},
    {"a=Xxr", 0213, 02257, 0555},
    {"+304", 0723, 01507, 01707},
    {"uo=tw", 0007, 0106, 01202},
    {"ao=u", 0367, 054, 00},
    {"ao-Xxt", 0656, 00, 00},
    {"go+s", 0353, 04177, 06177},
    {"u-X", 0606, 0771, 0671},
    {"ago+sXr", 0345, 0602, 06646},
    {"a+w", 0651, 0647, 0667},
    {"ao=wt", 0020, 0704, 01222},
    {"a-rt", 0534, 0711, 0311},
    {"ua=tw", 0232, 0613, 01222},
    {"001", 0676, 01412, 01},
    {"g-u", 0245, 0740, 0700},
    {"g-rsw", 0765, 0603, 0603},
    {"a=rtx", 0647, 0765, 01555},
    {"270", 0500, 04333, 0270},
    {"g-wX", 0364, 0755, 0745},
    {"o-o", 0672, 070, 070},
    {"g-ws", 0336, 02465, 0445},
    {"a=X", 0103, 02140, 0111},
    {"ao-r", 0723, 0373, 0333},
    {"o+X", 0667, 0155, 0155},
    {"uo=wxt", 0111, 0237, 01333},
    {"ag=g", 0617, 02042, 0444},
    {"=262", 0565, 01615, 0262},
    {"go+r", 0031, 0777, 0777},
    {"goa-xw", 0073, 04276, 04044},
    {"a=X-tX", 0077, 04322, 00},
    {"go-rtx", 0726, 01772, 0722},
    {"ga-wr", 0630, 011, 011},
    {"a+u", 0713, 0760, 0777},
    {"g=wrt", 0010, 051, 061},
    {"gu-u", 0763, 0573, 023},
    {"uo-tXw", 0426, 023, 020},
    {"ou=g", 0210, 0236, 0333},
    {"g=wx", 0327, 0320, 0330},
    {"ou-Xs", 0021, 0666, 0666},
    {"uog+wt", 0204, 0527, 01727},
    {"ug=g", 0632, 04, 04},
    {"u-o", 0073, 0522, 0522},
    {"o-o", 0232, 0753, 0750},
    {"g=xw", 0166, 0266, 0236},
    {"0313", 0774, 0427, 0313},
    {"g=ws", 0375, 0671, 02621},
    {"au+X", 0022, 0751, 0751},
    {"g+rst", 0413, 0700, 02740},
    {"ou+X=xrX", 0377, 0173, 0575},
    {"gu-ts", 0017, 0313, 0313},
    {"g+X", 0317, 0125, 0135},
    {"g=X", 0554, 0542, 0512},
    {"+722", 0216, 0455, 0777},
    {"g=Xsx", 0162, 02151, 02111},
    {"o-o", 0723, 047, 040},
    {"oug-w", 0626, 0106, 0104},
    {"oug+r", 0307, 0200, 0644},
    {"uao-u", 0305, 0102, 02},
    {"ogu=xs", 0663, 0667, 06111},
    {"oa=s-s", 0170, 01320, 00},
    {"u-rw", 0342, 0500, 0100},
    {"og+o", 0316, 0252, 0272},
    {"goa+sw", 0157, 0475, 06677},
    {"u=X", 0516, 04335, 0135},
    {"ug=Xw", 0473, 074, 0334},
    {"o-w", 0674, 0730, 0730},
    {"a-X=tw", 0643, 00, 01222},
    {"ug-x", 0336, 0477, 0467},
    {"+610", 0644, 0672, 0672},
    {"o=s", 0660, 031, 030},
    {"uoa+wXr", 0230, 075, 0777},
    {"uo-X+s", 0446, 021, 04020},
    {"u=u", 0601, 044, 044},
    {"ago+rXw", 0325, 0765, 0777},
    {"oa=xXr", 0450, 077, 0555},
    {"g-rt", 0501, 04436, 04436},
    {"a+t", 0407, 0634, 01634},
    {"au+rX", 0657, 0644, 0644},
    {"oa-st", 0477, 0452, 0452},
    {"ga+xw", 0126, 0400, 0733},
    {"512", 0347, 0630, 0512},
    {"2457", 0464, 04374, 02457},
    {"go-Xwr", 0277, 0721, 0700},
    {"ao=tx", 0036, 0644, 01111},
    {"uoa=w", 0634, 0167, 0222},
    {"u-t", 0146, 01762, 01762},
    {"u+s", 0645, 02577, 06577},
    {"ua-w", 0445, 0730, 0510},
    {"ag=g", 0125, 0656, 0555},
    {"go-tX", 0377, 0400, 0400},
    {"og+Xs", 0434, 0676, 02677},
    {"og-sr", 0634, 0515, 0511},
    {"o+rt", 0010, 0162, 01166},
    {"a+st", 0112, 0774, 07774},
    {"ga-xr", 0612, 0674, 0220},
    {"u+u", 0472, 0761, 0761},
    {"g+X", 0240, 02012, 02012},
    {"a+wX", 0370, 0470, 0773},
    {"a-xX", 0011, 0222, 0222},
    {"ou=tx", 0165, 0241, 01141},
    {"gau-X", 0623, 0667, 0666},
    {"g=Xt", 0554, 0457, 0417},
    {"ug=tX", 0030, 0202, 02},
    {"go-ts", 0224, 0743, 0743},
    {"gau+rx", 0613, 0676, 0777},
    {"au=sr", 0462, 0144, 06444},
    {"oag-wt", 0431, 01302, 0100},
    {"og+tw", 0656, 0534, 01536},
    {"g+w=xwt", 0244, 0102, 0132},
    {"=611", 0766, 01675, 0611},
    {"u=Xs", 0315, 0233, 04133},
    {"uo+u", 0326, 0451, 0455},
    {"g=ts", 0173, 0202, 02202},
    {"uo-tr", 0373, 0334, 0330},
    {"uo-Xt", 0333, 0501, 0400},
    {"o-sXr", 0305, 0441, 0440},
    {"ou-rX", 0006, 0231, 0230},
    {"ou=sw", 0506, 0622, 04222},
    {"uo-xtr", 0337, 0765, 0260},
    {"oa-o", 0056, 0665, 0220},
    {"o+Xx", 0423, 0767, 0767},
    {"632", 0717, 0107, 0632},
    {"og+sXx", 0100, 01777, 03777},
    {"ogu-t", 0471, 01060, 060},
    {"ua+tr", 0704, 0433, 01477},
    {"gu+sxr", 0601, 0663, 06773},
    {"ga-Xw", 0563, 0576, 0444},
    {"u=wX", 0332, 044, 0244},
    {"og+xr", 0414, 0171, 0175},
    {"ou-s", 0031, 0711, 0711},
    {"o=rt", 0064, 0676, 01674},
    {"ag+X", 0346, 0360, 0371},
    {"ug+x", 0772, 0217, 0317},
    {"gu-g", 0442, 04, 04},
    {"262", 0110, 0467, 0262},
    {"aug+X", 0625, 0307, 0317},
    {"ao+w", 0123, 0346, 0366},
    {"ao-x", 0166, 0443, 0442},
    {"a=u", 0525, 04715, 0777},
    {"g-xwr", 0677, 0317, 0307},
    {"a-s", 0445, 0754, 0754},
    {"gu-s", 0771, 0630, 0630},
    {"ag+u", 0625, 02277, 02277},
    {"a+xw", 0165, 0161, 0373},
    {"u-tw", 0334, 0444, 0444},
    {"uoa+ws", 0610, 0200, 06222},
    {"go+s", 0756, 0666, 02666},
    {"a+o", 0452, 04203, 04333},
    {"u-w", 0575, 02474, 02474},
    {"g=sX", 0220, 0417, 02417},
    {"605", 0234, 01430, 0605},
    {"go=r", 0261, 0320, 0344},
    {"g=sw", 0032, 0210, 02220},
    {"-210", 0230, 0445, 0445},
    {"ug+w=xr", 0766, 0470, 0550},
    {"gou=w+rX", 0163, 0146, 0666},
    {"go+o", 0232, 0544, 0544},
    {"o-xt", 0752, 0643, 0642},
    {"ao=r+tXs", 0565, 0760, 07444},
    {"a-o", 0001, 0676, 010},
    {"g=wt", 0246, 01605, 01625},
    {"u+txw", 0631, 074, 0374},
    {"og+Xw", 0072, 0450, 0473},
    {"a-xwt", 0511, 02700, 02400},
    {"u+s", 0373, 02467, 06467},
    {"oa+o", 0462, 051, 0151},
    {"o-t", 0774, 0401, 0401},
    {"ou+rx", 0506, 0142, 0547},
    {"go=r-ws", 0003, 04005, 04044},
    {"gua=o", 0500, 02040, 00},
    {"ua+g", 0535, 0725, 0727},
    {"ag+t", 0415, 0765, 01765},
    {"ga+wt", 0020, 0423, 01623},
    {"au-t", 0777, 02013, 02013},
    {"u-ts", 0667, 0537, 0537},
    {"ou-tsw", 0150, 0726, 0524},
    {"ga-o", 0742, 0146, 0100},
    {"go+xr", 0752, 0663, 0677},
    {"ou=o", 0637, 0613, 0313},
    {"ou=w+w", 0141, 0124, 0222},
    {"g+xw", 0015, 02453, 02473},
    {"oa-sr", 0531, 044, 00},
    {"534", 0424, 0337, 0534},
    {"oa=o", 0343, 02373, 0333},
    {"uoa+tX", 0244, 0316, 01317},
    {"626", 0467, 0147, 0626},
    {"a-ts", 0760, 0626, 0626},
    {"o-x", 0141, 050, 050},
    {"o-rXx", 0360, 0726, 0722},
    {"a-o", 0663, 0444, 00},
    {"u+u", 0231, 0244, 0244},
    {"u-X=X", 0274, 01004, 01004},
    {"ao+wX", 0445, 0243, 0373},
    {"o-Xtx", 0077, 0537, 0536},
    {"167", 0260, 01020, 0167},
    {"u=o", 0617, 0162, 0262},
    {"ag=rs", 0176, 0400, 06444},
    {"g-rxw", 0362, 0143, 0103},
    {"ua=r", 0462, 01237, 0444},
    {"oa-Xr", 0304, 0577, 022},
    {"og+w", 0277, 0164, 0166},
    {"au+Xxw", 0345, 01265, 01377},
    {"ou-sr", 0715, 0726, 0322},
    {"333", 0545, 0764, 0333},
    {"g-Xw", 0404, 0201, 0201},
    {"uo+r=x-swX", 0400, 0322, 020},
    {"ag+t", 0424, 0121, 01121},
    {"u=w", 0273, 04, 0204},
    {"ao=rxX", 0451, 0337, 0555},
    {"g+t", 0075, 063, 063},
    {"ugo-twr", 0670, 0770, 0110},
    {"ga+tw", 0621, 0126, 01326},
    {"gua-t+rx", 0630, 0305, 0755},
    {"o+o", 0064, 0732, 0732},
    {"u=x", 0412, 0607, 0107},
    {"ug=trs", 0011, 0643, 06443},
    {"uo-g", 0744, 0261, 061},
    {"o+rx", 0425, 04244, 04245},
    {"g+xr", 0002, 0676, 0676},
    {"605", 0060, 01634, 0605},
    {"u+w", 0257, 0776, 0776},
    {"uog=r", 0525, 0675, 0444},
    {"oug-Xts", 0645, 0707, 0606},
    {"0524", 0647, 0101, 0524},
    {"u=sx", 0235, 0324, 04124},
    {"g-rw", 0446, 0214, 0214},
    {"ug+rt", 0254, 020, 0460},
    {"g=wXs", 0720, 02527, 02537},
    {"ua-xtr", 0233, 0256, 0202},
    {"uao=X", 0004, 062, 00},
    {"g-o", 0663, 04510, 04510},
    {"ogu-xts", 0623, 022, 022},
    {"o=srw", 0043, 0744, 0746},
    {"g=X", 0617, 040, 00},
    {"a=xX", 0446, 01352, 0111},
    {"o-X=wtr", 0754, 024, 01026},
    {"uoa+rXs", 0355, 040, 06444},
    {"ou-wX", 0163, 0611, 0410},
    {"ug=Xs", 0347, 072, 06112},
    {"oga-sr", 0034, 0754, 0310},
    {"ao+Xs", 0776, 02226, 06226},
    {"ao=wtr", 0667, 0430, 01666},
    {"uag=Xrw", 0521, 0473, 0777},
    {"ou+w", 0342, 0277, 0277},
    {"ao=o", 0446, 0306, 0666},
    {"oa+w", 0375, 0537, 0737},
    {"+520", 0035, 017, 0537},
    {"o-sr", 0511, 0240, 0240},
    {"go-ws", 0215, 0227, 0205},
    {"o-wx", 0107, 0551, 0550},
    {"oag+rst", 0611, 0446, 07446},
    {"u=w", 0575, 053, 0253},
    {"aog-g", 0120, 0701, 0701},
    {"ao+o", 0515, 02041, 02151},
    {"gu=trw", 0367, 01416, 01666},
    {"o=rst", 0120, 04, 01004},
    {"u+wsr", 0105, 0102, 04702},
    {"gau=r", 0221, 0446, 0444},
    {"032", 0652, 024, 032},
    {"ag=sr", 0440, 0670, 06444},
    {"ao-x+w", 0534, 01064, 01266},
    {"o=Xwx", 0201, 04507, 04503},
    {"ua+t", 0744, 020, 01020},
    {"ao+g", 0715, 0170, 0777},
    {"gu=s", 0750, 04, 06004},
    {"ao+Xr", 0626, 01055, 01555},
    {"g+xXt", 0611, 02244, 02254},
    {"oa-s", 0526, 0547, 0547},
    {"o+w", 0153, 025, 027},
    {"au-X", 0643, 0402, 0402},
    {"o=tsr", 0177, 0663, 01664},
    {"o=xsw", 0042, 0252, 0253},
    {"o=wxs", 0543, 0162, 0163},
    {"au+g", 0503, 0424, 0626},
    {"o=o", 0072, 0400, 0400},
    {"a+Xt", 0737, 04231, 05331},
    {"au+t", 0401, 0153, 01153},
    {"o+r=t", 0124, 02, 01000},
    {"o+o", 0634, 0566, 0566},
    {"705", 0761, 04641, 0705},
    {"g=xX", 0662, 0222, 0212},
    {"o+g", 0174, 0530, 0533},
    {"o+xt", 0033, 0173, 01173},
    {"u-r", 0640, 072, 072},
    {"a+xs", 0577, 0433, 06533},
    {"ag=X", 0704, 0430, 0111},
    {"g-trs", 0531, 0317, 0317},
    {"oag-g", 0755, 0204, 0204},
    {"a=t", 0576, 0457, 01000},
    {"a=ws", 0344, 0763, 06222},
    {"ou=t", 0527, 0116, 01010},
    {"ua=st", 0321, 04, 07000},
    {"o=t+xXt", 0456, 0576, 01571},
    {"ug-g", 0210, 0630, 0400},
    {"uo=xts", 0064, 0250, 05151},
    {"goa+st", 0044, 0254, 07254},
    {"a+r", 0474, 0652, 0656},
    {"g=rXx", 0123, 012, 052},
    {"o-Xr", 0544, 063, 062},
    {"aug+rt", 0300, 0607, 01647},
    {"a-Xt", 0644, 04330, 04220},
    {"u+s", 0024, 0501, 04501},
    {"g-tw", 0666, 0664, 0644},
    {"o-t", 0170, 0620, 0620},
    {"og=u", 0035, 0217, 0222},
    {"a-u", 0355, 0573, 022},
    {"ou+wr", 0661, 0612, 0616},
    {"agu+t", 0501, 0670, 01670},
    {"455", 0750, 020, 0455},
    {"ou+X", 0106, 0320, 0321},
    {"o+t", 0733, 0173, 01173},
    {"a+g", 0072, 0654, 0755},
    {"u-t", 0637, 04250, 04250},
    {"u-xw", 0731, 0633, 0433},
    {"au-s", 0564, 0377, 0377},
    {"gu+st", 0274, 014, 06014},
    {"u+x", 0411, 0573, 0573},
    {"ua=Xx", 0301, 0131, 0111},
    {"og=ws", 0711, 04711, 06722},
    {"uo+swr", 0177, 0756, 04756},
    {"421", 0521, 0644, 0421},
    {"ag=X", 0032, 0670, 0111},
    {"auo=u", 0057, 0231, 0222},
    {"og=g", 0004, 0660, 0666},
    {"o-sX", 0207, 0150, 0150},
    {"go+wr", 0626, 0564, 0566},
    {"a+X=tX", 0743, 0460, 01000},
    {"ga-rs", 0037, 0575, 0131},
    {"u=wst", 0271, 04755, 04255},
    {"ou+srt", 0416, 0245, 05645},
    {"ag-Xx", 0322, 0116, 06},
    {"a=xts", 0500, 0671, 07111},
    {"g=swX", 0201, 0714, 02734},
    {"u=r", 0701, 0340, 0440},
    {"uo-g", 0417, 0444, 040},
    {"uo=rtX", 0307, 0205, 01505},
    {"u=t", 0571, 0311, 011},
    {"g+u", 0243, 0606, 0666},
    {"u=X-xrt", 0605, 0430, 030},
    {"uo-w", 0574, 0240, 040},
    {"u=X+t", 0556, 0640, 040},
    {"g+w", 0447, 0457, 0477},
    {"ao+tXr", 0123, 02051, 03555},
    {"o=x", 0272, 013, 011},
    {"go=wx", 0704, 044, 033},
    {"uo=rxs", 0575, 00, 04505},
    {"g+wsx", 0631, 0567, 02577},
    {"g+wx", 0740, 0357, 0377},
    {"ug=rxs", 0237, 0235, 06555},
    {"a=o", 0263, 0703, 0333},
    {"u-xt", 0645, 00, 00},
    {"-505", 0763, 0427, 022},
    {"o=Xtx", 0024, 0361, 01361},
    {"ua+r", 0676, 00, 0444},
    {"171", 0371, 0204, 0171},
    {"ug=s", 0740, 0424, 06004},
    {"a=w-x", 0607, 04715, 0222},
    {"u-sr", 0565, 0435, 035},
    {"a+u", 0367, 0272, 0272},
    {"ua+st", 0734, 0327, 07327},
    {"o=xw", 0012, 0172, 0173},
    {"a=o", 0711, 01737, 0777},
    {"+527", 0165, 0673, 0777},
    {"g+o", 0362, 0404, 0444},
    {"g-u", 0724, 02, 02},
    {"uag-r", 0206, 0553, 0113},
    {"g-t", 0564, 0502, 0502},
    {"ou-ws", 0277, 0717, 0515},
    {"706", 0357, 0512, 0706},
    {"au=rs", 0717, 04717, 06444},
    {"oag+xt", 0773, 0331, 01331},
    {"og=rt", 0770, 0223, 01244},
    {"uoa-Xwr", 0464, 031, 00},
    {"o+xX", 0236, 0135, 0135},
    {"au-r", 0054, 01344, 01300},
    {"o+X", 0125, 0236, 0237},
    {"oga+tr", 0471, 0401, 01445},
    {"uo=o", 0403, 0133, 0333},
    {"oau=w-s+w", 0240, 0435, 0222},
    {"g-x", 0371, 0321, 0321},
    {"g-rXx", 0775, 053, 03},
    {"oua-wx", 0004, 0300, 00},
    {"u-g", 0530, 0737, 0437},
    {"a-xw", 0442, 0723, 0400},
    {"o+trx", 0567, 0306, 01307},
    {"uo-s+sr", 0072, 073, 04477},
    {"u+ts", 0560, 0247, 04247},
    {"ag+wt", 0655, 0125, 01327},
    {"oa+x", 0217, 0205, 0315},
    {"ou=x", 0720, 0213, 0111},
    {"gu-Xr", 0054, 0143, 03},
    {"g=u", 0516, 0701, 0771},
    {"o+xX", 0725, 00, 01},
    {"u=Xtr", 0547, 04151, 0551},
    {"ua+xs", 0005, 0254, 06355},
    {"o=tx", 0726, 017, 01011},
    {"ug+x", 0044, 0104, 0114},
    {"a-xtX", 0562, 0202, 0202},
    {"ao-u", 0147, 01634, 01010},
    {"oa+g", 0614, 0463, 0667},
    {"g=w", 0372, 0353, 0323},
    {"ua+X", 0021, 0454, 0555},
    {"042", 0364, 070, 042},
    {"ga=wx", 0372, 0550, 0333},
    {"og-xwt", 0275, 0355, 0344},
    {"ogu-tw", 0355, 01220, 00},
    {"ug-tX", 0456, 0545, 0445},
    {"2612", 0225, 0106, 02612},
    {"ga=wr", 0112, 0536, 0666},
    {"227", 0447, 020, 0227},
    {"g-sX", 0007, 060, 060},
    {"ga-X", 0555, 0756, 0646},
    {"u+s", 0167, 0256, 04256},
    {"g+o", 0623, 0111, 0111},
    {"ao+Xsw", 0607, 0304, 06337},
    {"o=X", 0454, 0202, 0200},
    {"o-txX", 0236, 0570, 0570},
    {"og=tr", 0413, 0133, 01144},
    {"a+w", 0512, 0660, 0662},
    {"ga=rt", 0315, 0713, 01444},
    {"ug-t", 0646, 0111, 0111},
    {"g+rxX", 0255, 071, 071},
    {"ug+o", 0457, 02034, 02474},
    {"ga+sx", 0417, 0446, 06557},
    {"g-g", 0053, 024, 04},
    {"og-wtr", 0573, 02220, 02200},
    {"gu=rt", 0320, 0766, 0446},
    {"ou+sw", 0572, 02231, 06233},
    {"ugo-o", 0107, 0667, 00},
    {"ga-tx", 0271, 0757, 0646},
    {"gau=xXs", 0540, 0327, 06111},
    {"561", 0355, 0175, 0561},
    {"au+rt", 0167, 0643, 01647},
    {"aug+g", 0175, 0240, 0644},
    {"og=Xs", 0677, 044, 02000},
    {"auo+Xs", 0544, 0132, 06133},
    {"o+tx", 0120, 0137, 01137},
    {"a+x", 0724, 0626, 0737},
    {"u=w", 0756, 0317, 0217},
    {"au+tr", 0254, 0533, 01577},
    {"ou-r", 0543, 0220, 0220},
    {"g=u", 0442, 0623, 0663},
    {"ua=xwX", 0522, 0642, 0333},
    {"uo+s", 0727, 022, 04022},
    {"ao+wsX", 0045, 0717, 06737},
    {"gu+Xr", 0526, 0640, 0640},
    {"235", 0377, 0400, 0235},
    {"g+u", 0402, 0737, 0777},
    {"u-stx", 0147, 0224, 0224},
    {"gu+rs", 0310, 026, 06466},
    {"og+r", 0451, 01056, 01056},
    {"ua-s", 0662, 054, 054},
    {"ug-r+r", 0126, 040, 0440},
    {"314", 0727, 0551, 0314},
    {"au-r+xw", 0042, 0775, 0333},
    {"267", 0752, 02534, 0267},
    {"ua=g", 0200, 074, 0777},
    {"u=tsr", 0171, 02776, 06476},
    {"302", 0306, 0724, 0302},
    {"g=rxX", 0116, 0605, 0655},
    {"g-wx", 0301, 0232, 0202},
    {"0604", 0007, 0674, 0604},
    {"oa=trX", 0767, 0536, 01555},
    {"o+o", 0122, 0141, 0141},
    {"o+Xtx", 0517, 0511, 01511},
    {"g=xs", 0603, 0162, 02112},
    {"uo=X", 0305, 0122, 0121},
    {"u=Xsw", 0054, 0350, 04350},
    {"u=rwt", 0713, 0530, 0630},
    {"ug-sw", 0035, 0752, 0552},
    {"ug-tX", 0447, 04537, 04427},
    {"ga=txw", 0711, 02670, 01333},
    {"u=g", 0460, 0165, 0665},
    {"gu=xXt", 0407, 0377, 0117},
    {"g=u", 0640, 0623, 0663},
    {"o-rst", 0031, 0576, 0572},
    {"o=sr", 0554, 0532, 0534},
    {"uga-rw", 0744, 04, 00},
    {"uo=ts", 0504, 0750, 05050},
    {"og=xs", 0436, 0716, 02711},
    {"ga=o", 0312, 0441, 0111},
    {"ao+wtX", 0177, 0523, 01733},
    {"ua+o", 0302, 0217, 0777},
    {"a+g", 0402, 0360, 0766},
    {"ug+Xwt", 0620, 0167, 0377},
    {"+4776", 0337, 0510, 04776},
    {"o-X+X=rw", 0373, 0317, 0316},
    {"oa-X", 0741, 0163, 062},
    {"ogu+g", 0623, 0616, 0717},
    {"g-rxw", 0247, 0112, 0102},
    {"o-s", 0030, 0723, 0723},
    {"o-rX", 0260, 04623, 04622},
    {"oa-sx", 0743, 0246, 0246},
    {"ug-xw", 0060, 02527, 02407},
    {"u+xt", 0132, 0757, 0757},
    {"o=t-x", 0413, 0415, 01410},
    {"au=u", 0267, 0400, 0444},
    {"ugo=w", 0071, 0167, 0222},
    {"u-x", 0401, 0371, 0271},
    {"433", 0355, 0530, 0433},
    {"a+wts", 0110, 0201, 07223},
    {"o=trx", 0371, 0102, 01105},
    {"ag-x", 0345, 0633, 0622},
    {"ga+x", 0653, 01763, 01773},
    {"o=wx", 0346, 0453, 0453},
    {"ua+rst", 0741, 0210, 07654},
    {"g=s-xw", 0540, 022, 02002},
    {"ao-s", 0526, 0110, 0110},
    {"uag+r=twX", 0247, 0166, 01333},
    {"u=sr", 0206, 00, 04400},
    {"ag+wrt", 0706, 0417, 01677},
    {"u-g", 0162, 0200, 0200},
    {"o+t", 0605, 01163, 01163},
    {"a=s-tX", 0434, 0752, 06000},
    {"ua+xs", 0521, 0277, 06377},
    {"gu+t", 0460, 04420, 04420},
    {"go+u", 0251, 0733, 0777},
    {"au+wt", 0067, 035, 01237},
    {"307", 0616, 0476, 0307},
    {"aug+tx", 0361, 0367, 01377},
    {"ga-w", 0074, 0643, 0441},
    {"717", 0013, 0316, 0717},
    {"a-Xx", 0163, 0320, 0220},
    {"u-rs", 0073, 02007, 02007},
    {"au-tx", 0452, 020, 020},
    {"+4346", 0057, 0311, 04357},
    {"u+X", 0103, 060, 060},
    {"ua=g", 0106, 01540, 0444},
    {"oa+u", 0530, 0246, 0266},
    {"040", 0543, 0520, 040},
    {"ou-t", 0360, 0124, 0124},
    {"ug-X", 0356, 0125, 025},
    {"go=wx", 0276, 0642, 0633},
    {"117", 0250, 0514, 0117},
    {"ugo+Xtw", 0506, 02134, 03337},
    {"ag+wxr", 0241, 02455, 02777},
    {"u+g", 0250, 0641, 0641},
    {"agu=rw", 0676, 0525, 0666},
    {"235", 0520, 0436, 0235},
    {"u+Xr", 0024, 0442, 0442},
    {"gu=tr", 0760, 0507, 0447},
    {"g=sXr", 0671, 0717, 02757},
    {"au+xs", 0424, 02561, 06571},
    {"ga+o", 0720, 0167, 0777},
    {"ag-tw", 0673, 04, 04},
    {"g+wr", 0604, 0654, 0674},
    {"oug=o", 0221, 0576, 0666},
    {"ogu+sX", 0560, 0510, 06511},
    {"025", 0446, 0463, 025},
    {"ou-tx", 0443, 0663, 0662},
    {"230", 0671, 0142, 0230},
    {"ao=X", 0236, 0367, 0111},
    {"gu=ws", 0601, 0611, 06221},
    {"ug+o", 0344, 0454, 0454},
    {"g=o", 0321, 0444, 0444},
    {"o+o", 0653, 0275, 0275},
    {"ou+rx", 0253, 0367, 0767},
    {"agu=Xxr", 0344, 01066, 0555},
    {"ag=X", 0264, 01613, 0111},
    {"u-s", 0727, 0765, 0765},
    {"au-X", 0704, 026, 026},
    {"ua+rx", 0073, 0533, 0577},
    {"uo+xs", 0434, 034, 04135},
    {"u=g", 0376, 01426, 01226},
    {"go+sr", 0061, 013, 02057},
    {"2434", 0373, 0706, 02434},
    {"a-x", 0716, 01273, 01262},
    {"ua=wt", 0747, 0430, 01222},
    {"au+o", 0254, 0572, 0772},
    {"o+g", 0360, 02412, 02413},
    {"ao+X", 0060, 0131, 0131},
    {"o=xsw", 0575, 0337, 0333},
    {"u=xtw", 0510, 046, 0346},
    {"u-Xw", 0223, 050, 050},
    {"u=wt", 0536, 0241, 0241},
    {"u+sx", 0134, 0560, 04560},
    {"ga=g", 0363, 00, 00},
    {"o+s", 0713, 066, 066},
    {"ua+rtx", 0027, 0767, 01777},
    {"u-o", 0015, 0350, 0350},
    {"-214", 0137, 04775, 04561},
    {"g=o", 0756, 04532, 04522},
    {"+321", 0701, 0564, 0765},
    {"g+s", 0360, 0401, 02401},
    {"u-rw", 0260, 0247, 047},
    {"ua+wXx", 0625, 02063, 02373},
    {"ug+ts", 0624, 0472, 06472},
    {"uo=w", 0027, 0126, 0222},
    {"a+xwX", 0077, 0243, 0373},
    {"uog-s+wr", 0600, 0722, 0766},
    {"au-u", 0232, 0554, 00},
    {"u-w", 0365, 064, 064},
    {"a+t", 0535, 0325, 01325},
    {"o=tXw", 0753, 0517, 01513},
    {"uo=wXs", 0147, 0702, 04303},
    {"ag=tX", 0615, 0372, 01111},
    {"426", 0753, 0127, 0426},
    {"o-xt", 0437, 0203, 0202},
    {"o+swX", 0560, 0416, 0417},
    {"g=xt", 0366, 0116, 0116},
    {"g-r", 0100, 051, 011},
    {"u+r", 0521, 0656, 0656},
    {"ag-sX", 0023, 0673, 0662},
    {"ago=r", 0061, 0533, 0444},
    {"501", 0624, 04404, 0501},
    {"gu+wr", 0341, 0556, 0776},
    {"365", 0422, 036, 0365},
    {"o=w", 0320, 0753, 0752},
    {"og-x=wr", 0516, 0605, 0666},
    {"ga=xw", 0452, 0664, 0333},
    {"1413", 0417, 0254, 01413},
    {"ao=sx", 0405, 0304, 06111},
    {"g+tx", 0323, 0776, 0776},
    {"uo=t", 0614, 0731, 01030},
    {"o-t", 0477, 01735, 0735},
    {"g-w", 0603, 022, 02},
    {"732", 0106, 0440, 0732},
    {"ga-r", 0407, 0331, 0331},
    {"ag-Xr", 0015, 01636, 01222},
    {"ou+o", 0543, 0505, 0505},
    {"uao=xw", 0274, 021, 0333},
    {"og-s", 0324, 0743, 0743},
    {"o+rx", 0660, 077, 077},
    {"ug=u", 0104, 0635, 0665},
    {"o=o", 0476, 0421, 0421},
    {"ua+ws", 0404, 0452, 06672},
    {"4016", 0017, 0634, 04016},
    {"u-st", 0635, 0533, 0533},
    {"g=srx", 0063, 0736, 02756},
    {"go-xr", 0543, 0755, 0700},
    {"250", 0457, 02547, 0250},
    {"a+s", 0267, 0147, 06147},
    {"og=s", 0352, 0201, 02200},
    {"o-tXs", 0132, 047, 046},
    {"uog+xw", 0361, 0317, 0337},
    {"ua-s", 0136, 0161, 0161},
    {"ugo=r", 0322, 0357, 0444},
    {"gou-r", 0774, 0103, 0103},
    {"g=tXw", 0564, 033, 033},
    {"oa-x", 0263, 012, 02},
    {"a=w", 0730, 0322, 0222},
    {"og=u", 0272, 02515, 0555},
    {"a-s", 0617, 0777, 0777},
    {"ag-sr", 0101, 02640, 0200},
    {"o+ws", 0730, 0117, 0117},
    {"gua=r", 0722, 0357, 0444},
    {"ag-rxt", 0670, 0366, 0222},
    {"a+trx", 0751, 0501, 01555},
    {"ag-wst", 0623, 053, 051},
    {"u+sxr", 0036, 0775, 04775},
    {"ag+Xx", 0446, 0427, 0537},
    {"ua=xXt", 0454, 0270, 01111},
    {"uag=x", 0570, 0516, 0111},
    {"g+u", 0573, 02676, 02676},
    {"u-ts", 0100, 0314, 0314},
    {"g-w", 0625, 066, 046},
    {"ug=st", 0506, 0550, 06000},
    {"og+Xtr", 0041, 0157, 01157},
    {"g=tx", 0231, 02750, 0710},
    {"gu=rs", 0544, 01325, 07445},
    {"a+w", 0004, 0150, 0372},
    {"u=u", 0717, 02442, 02442},
    {"gao-wt", 0142, 0371, 0151},
    {"uog=o", 0330, 0604, 0444},
    {"ga+w", 0207, 0436, 0636},
    {"ga=xws", 0007, 0460, 06333},
    {"ua+X", 0507, 0260, 0260},
    {"2017", 0535, 04127, 02017},
    {"gu=g", 0446, 0676, 0776},
    {"g-Xt", 0031, 0137, 0127},
    {"ua+Xx", 0000, 01766, 01777},
    {"162", 0046, 021, 0162},
    {"g=s", 0355, 0616, 02606},
    {"ou+o", 0234, 02174, 02574},
    {"1150", 0254, 0746, 01150},
    {"a-x", 0154, 0152, 042},
    {"u=X", 0512, 0350, 0150},
    {"a=g", 0505, 0625, 0222},
    {"uo=Xwx", 0713, 04711, 0313},
    {"og=tr", 0655, 0110, 01144},
    {"guo-s", 0650, 04035, 035},
    {"ug+o", 0766, 04332, 04332},
    {"ug=t", 0513, 04342, 02},
    {"u+Xw", 0762, 0571, 0771},
    {"ug+g", 0500, 0226, 0226},
    {"u-tr", 0342, 0424, 024},
    {"oa-X", 0535, 025, 024},
    {"ou+wt", 0312, 0660, 01662},
    {"au+rXw", 0040, 011, 0777},
    {"ao=tXx", 0200, 0272, 01111},
    {"a-tx", 0115, 04163, 04062},
    {"uo=Xw", 0571, 0657, 0353},
    {"u-w+wr", 0772, 0331, 0731},
    {"o-xt", 0170, 0672, 0672},
    {"u+g", 0216, 0537, 0737},
    {"ag=t", 0561, 0647, 01000},
    {"aug+wt", 0605, 0160, 01362},
    {"a+xt", 0764, 00, 01111},
    {"uo+u", 0150, 0777, 0777},
    {"720", 0756, 0427, 0720},
    {"g+wr", 0227, 0236, 0276},
    {"u-t", 0406, 02072, 02072},
    {"ua+tX", 0307, 02673, 03773},
    {"o-Xs", 0341, 0451, 0450},
    {"ga-Xs", 0113, 04453, 0442},
    {"g+xw", 0213, 04543, 04573},
    {"og=t", 0510, 0745, 01700},
    {"oa=g", 0373, 0643, 0444},
    {"ag+w", 0414, 0344, 0366},
    {"au-Xx", 0507, 0656, 0646},
    {"u-sX", 0312, 060, 060},
    {"uo=sw", 0221, 0775, 04272},
    {"ag+t", 0315, 062, 01062},
    {"oa-X", 0063, 0732, 0622},
    {"ao+x", 0441, 0222, 0333},
    {"=233", 0515, 0322, 0233},
    {"u-x", 0301, 04057, 04057},
    {"ag-w=t+w", 0313, 0555, 01222},
    {"g-r", 0756, 0514, 0514},
    {"u-Xtw", 0432, 0566, 0466},
    {"ga+g", 0670, 06, 06},
    {"oa-u", 0426, 0226, 04},
    {"o+u", 0310, 0351, 0353},
    {"g=X+s", 0725, 0634, 02614},
    {"oa=rx", 0525, 0622, 0555},
    {"a=txX", 0302, 0467, 01111},
    {"u-s", 0741, 0526, 0526},
    {"ag-tr", 0773, 0313, 0313},
    {"u-x", 0142, 0547, 0447},
    {"a-rt", 0070, 0161, 0121},
    {"ga+rs", 0173, 014, 06454},
    {"a-xt", 0423, 0776, 0666},
    {"g+u", 0271, 0206, 0226},
    {"uao-xw", 0002, 0430, 0400},
    {"go=r", 0522, 0726, 0744},
    {"oa-x", 0734, 0774, 0664},
    {"ou=t", 0103, 0627, 01020},
    {"uao-X", 0202, 0622, 0622},
    {"ou+o", 0430, 0763, 0763},
    {"ua-w", 0610, 0665, 0445},
    {"4654", 0747, 0677, 04654},
    {"oa+wXs", 0637, 02100, 06333},
    {"o+sx", 0772, 00, 01},
    {"au=o", 0625, 0701, 0111},
    {"ua-t", 0715, 0542, 0542},
    {"u-xr", 0536, 0775, 0275},
    {"uag+twx", 0277, 04525, 05737},
    {"ag=rsx", 0175, 012, 06555},
    {"o-wr", 0003, 017, 011},
    {"u-w=wt", 0020, 01756, 01256},
    {"a-Xw", 0136, 02175, 02044},
    {"a-w", 0727, 0475, 0455},
    {"g+g", 0776, 0214, 0214},
    {"aou-rs", 0427, 0626, 0222},
    {"354", 0457, 0677, 0354},
    {"ag=u", 0462, 0204, 0222},
    {"go=rXt", 0524, 0471, 01455},
    {"u-g", 0475, 04112, 04012},
    {"ag+x", 0043, 0630, 0731},
    {"534", 0431, 0617, 0534},
    {"ao=xt", 0022, 0117, 01111},
    {"uo+u", 0165, 0222, 0222},
    {"g=u", 0615, 0634, 0664},
    {"ou-g", 0324, 0462, 060},
    {"au=X", 0174, 0234, 0111},
    {"u=sX", 0272, 0676, 04176},
    {"go=xts", 0364, 0244, 03211},
    {"u+xt", 0520, 0562, 0562},
    {"g-u", 0016, 0644, 0604},
    {"go-sx", 0023, 0404, 0404},
    {"oga+xX", 0435, 0470, 0571},
    {"o-t", 0333, 0456, 0456},
    {"og+r", 0264, 0366, 0366},
    {"ou+u", 0455, 0416, 0416},
    {"ag-wX", 0561, 0204, 04},
    {"agu-xs", 0357, 015, 04},
    {"go+t-X=wXr", 0412, 0217, 0266},
    {"og-X", 0464, 0756, 0746},
    {"uo-xs", 0262, 0532, 0432},
    {"og-u", 0514, 0274, 0254},
    {"u-tx", 0437, 0705, 0605},
    {"agu=s", 0763, 0276, 06000},
    {"u-tx", 0407, 0657, 0657},
    {"ua-xs", 0466, 04056, 046},
    {"o=w", 0057, 046, 042},
    {"a-t", 0211, 0523, 0523},
    {"uo+xX", 0712, 0517, 0517},
    {"ua=wr", 0647, 0472, 0666},
    {"au-t", 0567, 01757, 0757},
    {"gau-x", 0562, 0200, 0200},
    {"ag+r", 0145, 0756, 0756},
    {"og=xst", 0577, 0402, 03411},
    {"302", 0230, 0447, 0302},
    {"656", 0627, 0564, 0656},
    {"256", 0155, 0657, 0256},
    {"u=sw", 0534, 042, 04242},
    {"o=t", 0331, 04130, 05130},
    {"ug+r", 0372, 0431, 0471},
    {"u=sx", 0116, 0741, 04141},
    {"g+u", 0560, 0270, 0270},
    {"g+r", 0211, 0142, 0142},
    {"u=X", 0516, 0231, 0131},
    {"1310", 0557, 0310, 01310},
    {"au+o", 0454, 0632, 0632},
    {"uag=Xr", 0472, 0340, 0555},
    {"o=s", 0554, 04203, 04200},
    {"ua-rx", 0605, 0671, 0220},
    {"g=g", 0143, 0713, 0713},
    {"404", 0372, 0266, 0404},
    {"u=xr", 0037, 0277, 0577},
    {"g=ts", 0245, 01072, 03002},
    {"ag-o", 0512, 0555, 00},
    {"au+tX", 0631, 0764, 01775},
    {"oa+rw", 0236, 050, 0676},
    {"a-t", 0741, 04657, 04657},
    {"ag=X", 0401, 0654, 0111},
    {"og-w", 0065, 0445, 0445},
    {"014", 0371, 0266, 014},
    {"ao=tX", 0064, 0155, 01111},
    {"ug+rXt", 0270, 0376, 0776},
    {"o-o", 0304, 0365, 0360},
    {"g+Xt", 0251, 044, 044},
    {"ua=xt", 0174, 0413, 01111},
    {"gu=g", 0120, 0656, 0556},
    {"ou=u", 0675, 013, 010},
    {"og=trs", 0152, 0675, 03644},
    {"aou-sX", 0441, 0636, 0626},
    {"u+g", 0105, 0712, 0712},
    {"au-xwt", 0141, 0640, 0440},
    {"ug=s-s+Xrx", 0030, 0543, 0553},
    {"ga-X", 0326, 04432, 04422},
    {"g+rXw", 0250, 0342, 0372},
    {"a=Xs", 0433, 0355, 06111},
    {"uo+rxX", 0537, 0156, 0557},
    {"u-ts", 0144, 0611, 0611},
    {"u+tx", 0252, 0162, 0162},
    {"ao+Xx", 0037, 0704, 0715},
    {"ou=o", 0606, 01513, 0313},
    {"o-txw", 0303, 04241, 04240},
    {"a=rs", 0355, 0453, 06444},
    {"g=sx", 0171, 0160, 02110},
    {"2442", 0623, 0254, 02442},
    {"g=s", 0771, 072, 02002},
    {"uog+xX", 0055, 0162, 0173},
    {"a+wrs", 0563, 0223, 06667},
    {"ag+xw", 0434, 0706, 0737},
    {"u+o", 0234, 0434, 0434},
    {"og-ws", 0001, 0162, 0140},
    {"ugo=s-X", 0075, 0707, 06000},
    {"uag-X", 0172, 0264, 0264},
    {"o+u", 0773, 0235, 0237},
    {"au+sw", 0130, 0162, 06362},
    {"g+g", 0271, 0704, 0704},
    {"ag-X", 0170, 0212, 0202},
    {"u=sX", 0404, 0427, 04127},
    {"374", 0533, 0324, 0374},
    {"goa-rtw", 0674, 04154, 04110},
    {"ua=rxt", 0031, 0461, 01555},
    {"ao+r", 0276, 0205, 0645},
    {"gu+wrX", 0523, 0240, 0660},
    {"a+g", 0454, 0325, 0327},
    {"u=u", 0345, 0420, 0420},
    {"u=ts", 0040, 0357, 04057},
    {"g+x", 0243, 0104, 0114},
    {"g+Xt", 0066, 0515, 0515},
    {"go+sx", 0174, 0757, 02757},
    {"a-t=sXx", 0576, 0341, 06111},
    {"a+Xx", 0716, 020, 0131},
    {"u=Xtr", 0452, 0162, 0562},
    {"auo-o", 0324, 0617, 00},
    {"ug+w-tr", 0467, 0434, 0234},
    {"oau=g", 0025, 01070, 0777},
    {"416", 0544, 0502, 0416},
    {"o-s", 0350, 0560, 0560},
    {"go-X=x", 0275, 0672, 0611},
    {"au-w+w", 0250, 0630, 0632},
    {"g+wsr", 0230, 0531, 02571},
    {"ua+t", 0401, 0645, 01645},
    {"uo+tr", 0714, 02110, 03514},
    {"aou=xX", 0720, 01025, 0111},
    {"a-tXx", 0176, 04733, 04622},
    {"au=Xr", 0515, 0135, 0555},
    {"o+g", 0234, 0617, 0617},
    {"a+X", 0123, 0245, 0355},
    {"u-tr", 0636, 0450, 050},
    {"ao-wx", 0717, 04162, 04040},
    {"ao-g", 0164, 0642, 0202},
    {"u=r", 0524, 0772, 0472},
    {"gu=x", 0375, 0443, 0113},
    {"uo+Xt", 0522, 0765, 01765},
    {"gu-wXs", 0713, 0134, 04},
    {"gao-rx", 0335, 0344, 0200},
    {"au=sr", 0360, 02756, 06444},
    {"635", 0763, 055, 0635},
    {"au=g", 0532, 0102, 00},
    {"a+x", 0512, 0577, 0577},
    {"ug=wr", 0676, 025, 0665},
    {"ag+sxX", 0121, 04036, 06137},
    {"o=tsw", 0073, 0660, 01662},
    {"ag=u", 0002, 0257, 0222},
    {"aou-x", 0643, 0553, 0442},
    {"ou=s", 0051, 046, 04040},
    {"gua-w", 0537, 0410, 0410},
    {"auo+X", 0552, 0576, 0577},
    {"oau+xt", 0537, 0242, 01353},
    {"go+t", 0234, 04752, 05752},
    {"oa=Xs", 0660, 0754, 06111},
    {"177", 0261, 0516, 0177},
    {"a+xr", 0076, 0757, 0757},
    {"g=xs", 0416, 0765, 02715},
    {"ug=o", 0010, 02055, 0555},
    {"oga+o", 0224, 0176, 0776},
    {"g+t", 0565, 0707, 0707},
    {"o-st", 0555, 0207, 0207},
    {"ao-s", 0430, 02067, 067},
    {"113", 0457, 0523, 0113},
    {"ao-Xrt", 0303, 0577, 022},
    {"a+t", 0112, 0450, 01450},
    {"u-rx", 0775, 0461, 061},
    {"ua+ts", 0635, 02, 07002},
    {"g+w=w-sr", 0360, 0516, 0526},
    {"o-g", 0263, 0102, 0102},
    {"u+rx", 0261, 0747, 0747},
    {"o=wX", 0462, 071, 073},
    {"g=o", 0637, 0132, 0122},
    {"ag=u", 0114, 0516, 0555},
    {"o=X", 0457, 0463, 0461},
    {"oug-ts", 0417, 05, 05},
    {"au+Xwt", 0631, 0577, 01777},
    {"1117", 0240, 0770, 01117},
    {"ao-r", 0433, 0264, 0220},
    {"go=t+xw", 0321, 0656, 01633},
    {"a=srx", 0434, 0617, 06555},
    {"uo-o", 0427, 0265, 0260},
    {"auo+g", 0534, 01456, 01557},
    {"o-Xw", 0206, 0221, 0220},
    {"og+o", 0747, 020, 020},
    {"051", 0664, 011, 051},
    {"0467", 0235, 0273, 0467},
    {"gou-s", 0523, 0754, 0754},
    {"=151", 0147, 0776, 0151},
    {"1120", 0376, 0255, 01120},
    {"gua=xt", 0261, 0430, 01111},
    {"oa-X+s", 0734, 042, 06042},
    {"og-r", 0421, 0427, 0423},
    {"ou=xt", 0241, 02310, 03111},
    {"o+g", 0270, 043, 047},
    {"u=tr", 0124, 04545, 0445},
    {"ag=ws", 0337, 04333, 06222},
    {"g+s=tx", 0043, 0627, 0617},
    {"au-s", 0661, 0315, 0315},
    {"o+sX", 0626, 0577, 0577},
    {"ago=o", 0641, 0361, 0111},
    {"g+s-tx", 0542, 0173, 02163},
    {"u-s", 0641, 01175, 01175},
    {"o+t", 0264, 0706, 01706},
    {"u-X", 0303, 074, 074},
    {"u-g", 0034, 01746, 01346},
    {"ou-sw", 0037, 0747, 0545},
    {"247", 0116, 0775, 0247},
    {"a=u", 0532, 04051, 00},
    {"ua-tsx", 0117, 0464, 0464},
    {"4475", 0463, 0217, 04475},
    {"og=u", 0736, 0306, 0333},
    {"424", 0077, 01152, 0424},
    {"u=s", 0130, 0514, 04014},
    {"ug-r+Xw", 0260, 0511, 0331},
    {"u+r", 0616, 0710, 0710},
    {"o=ws", 0400, 0326, 0322},
    {"u+wr", 0327, 0367, 0767},
    {"u=s", 0233, 0600, 04000},
    {"gu=sx", 0122, 0170, 06110},
    {"014", 0610, 0563, 014},
    {"o=o", 0532, 0306, 0306},
    {"ugo+wrX", 0760, 0227, 0777},
    {"a-g", 0652, 0557, 02},
    {"au=s", 0215, 0517, 06000},
    {"226", 0276, 00, 0226},
    {"uo=xw", 0115, 0112, 0313},
    {"ag-wt", 0462, 0672, 0450},
    {"ao=r", 0154, 0740, 0444},
    {"ou+Xxr", 0247, 0420, 0525},
    {"g=wr", 0257, 0415, 0465},
    {"ou=x", 0650, 0325, 0121},
    {"ga-Xw", 0124, 0760, 0440},
    {"043", 0575, 0510, 043},
    {"o-sr", 0150, 0566, 0562},
    {"og+Xx", 0114, 0166, 0177},
    {"ou-X", 0077, 0435, 0434},
    {"a-sr", 0512, 0302, 0302},
    {"u+rs", 0067, 0401, 04401},
    {"ua-tw", 0534, 0202, 00},
    {"g=t", 0061, 01154, 01104},
    {"o=xt", 0065, 0301, 01301},
    {"ou=o", 0054, 0427, 0727},
    {"017", 0265, 0206, 017},
    {"au=X", 0162, 0730, 0111},
    {"oa+u", 0722, 066, 066},
    {"ga+o", 0716, 073, 0373},
    {"u+s", 0716, 060, 04060},
    {"au+t", 0764, 0143, 01143},
    {"ua=t", 0550, 0343, 01000},
    {"ag=xr", 0073, 0570, 0555},
    {"u-X", 0667, 0654, 0654},
    {"u-o", 0602, 02624, 02224},
    {"o-o", 0421, 0421, 0420},
    {"uo=sw", 0563, 0250, 04252},
    {"g-rx", 0642, 0274, 0224},
    {"=054", 0332, 0254, 054},
    {"o+r", 0512, 0164, 0164},
    {"o=sw", 0403, 0665, 0662},
    {"ao=r", 0774, 0167, 0444},
    {"oa-sx", 0533, 0364, 0264},
    {"gu+Xw", 0270, 0611, 0731},
    {"au=s", 0735, 0745, 06000},
    {"ug=rw", 0251, 0616, 0666},
    {"g=x", 0417, 0151, 0111},
    {"020", 0501, 0523, 020},
    {"-671", 0365, 0765, 0104},
    {"aou+X", 0151, 0575, 0575},
    {"og+u", 0645, 055, 055},
    {"a+w-rxs", 0211, 0422, 0222},
    {"gu=tX", 0471, 0411, 0111},
    {"o-tw", 0154, 0225, 0225},
    {"a+rsw", 0274, 0706, 06766},
    {"1047", 0406, 0774, 01047},
    {"703", 0142, 01757, 0703},
    {"u-t+t=rt", 0214, 0730, 0430},
    {"u=xtX", 0730, 0467, 0167},
    {"guo-s", 0776, 0325, 0325},
    {"a=xt", 0215, 067, 01111},
    {"gou=u", 0563, 0735, 0777},
    {"oa-tX", 0453, 0432, 0422},
    {"og=u", 0372, 0415, 0444},
    {"124", 0725, 00, 0124},
    {"o=t+tws", 0344, 02126, 03122},
    {"244", 0007, 0334, 0244},
    {"g+x", 0114, 0771, 0771},
    {"+151", 0336, 0365, 0375},
    {"uo+tX", 0625, 01133, 01133},
    {"u=t", 0076, 02562, 02062},
    {"og-s", 0016, 0666, 0666},
    {"uga=xsr", 0600, 0421, 06555},
    {"u=X", 0027, 0265, 0165},
    {"g+x=tXw", 0773, 0523, 0533},
    {"gu-g", 0376, 0576, 06},
    {"o=x", 0407, 0205, 0201},
    {"uag=o", 0161, 025, 0555},
    {"og=x+X", 0546, 0424, 0411},
    {"og+g", 0141, 052, 057},
    {"gu+s", 0650, 0640, 06640},
    {"oua+u", 0722, 01421, 01465},
    {"o=txs", 0332, 0130, 01131},
    {"goa=tX", 0211, 00, 01000},
    {"uag+r", 0211, 0320, 0764},
    {"ag-xr", 0515, 0211, 0200},
    {"o=wt", 0172, 0365, 01362},
    {"661", 0137, 066, 0661},
    {"go-sw", 0766, 0712, 0710},
    {"ua-Xtw", 0246, 01730, 0400},
    {"aou+wt", 0614, 0304, 01326},
    {"1555", 0146, 0260, 01555},
    {"ug-xX", 0414, 0544, 0444},
    {"aug+sx", 0245, 0661, 06771},
    {"o-o", 0167, 04315, 04310},
    {"-106", 0772, 0651, 0651},
    {"u-s", 0612, 0764, 0764},
    {"a-trx", 0116, 0376, 0222},
    {"u+tsx", 0060, 0721, 04721},
    {"gao+ts", 0211, 02007, 07007},
    {"oga=xr", 0430, 05, 0555},
    {"u-X", 0050, 0362, 0262},
    {"o+r", 0607, 0677, 0677},
    {"g+ts", 0435, 061, 02061},
    {"ao-rt", 0063, 0455, 011},
    {"au=u", 0040, 024, 00},
    {"u-wsX", 0045, 04717, 0417},
    {"ag-X", 0361, 02326, 02226},
    {"ag-trx", 0671, 0730, 0220},
    {"364", 0017, 0721, 0364},
    {"u=s", 0074, 0444, 04044},
    {"oua=X", 0470, 0145, 0111},
    {"oua=g", 0710, 0770, 0777},
    {"a=x", 0302, 0463, 0111},
    {"go+tw", 0347, 0125, 01127},
    {"o=r", 0332, 04251, 04254},
    {"u-t", 0603, 02374, 02374},
    {"gu=o", 0126, 0511, 0111},
    {"go+o", 0076, 0236, 0276},
    {"ou+xt", 0021, 0267, 01367},
    {"o-w", 0513, 0360, 0360},
    {"uag=tx", 0703, 0273, 01111},
    {"gu+u", 0011, 0323, 0333},
    {"ga-wr", 0760, 02720, 02100},
    {"ugo+X", 0323, 071, 0171},
    {"g-o", 0302, 0564, 0524},
    {"g-rs", 0464, 0437, 0437},
    {"ao-wtx", 0122, 0611, 0400},
    {"og+o", 0564, 054, 054},
    {"go=u", 0322, 0411, 0444},
    {"a-Xx", 0241, 01773, 01662},
    {"au+Xwt", 0363, 042, 01262},
    {"ou=r", 0607, 0462, 0464},
    {"au-rtx", 0177, 0314, 0200},
    {"=1303", 0404, 02042, 01303},
    {"au-u", 0204, 04676, 04010},
    {"g+rXx", 0653, 0525, 0575},
    {"uoa-x+r", 0757, 063, 0466},
    {"o+tw", 0177, 062, 01062},
    {"uo=rsw", 0522, 0420, 04626},
    {"uo+o", 0167, 0253, 0353},
    {"g=sX", 0443, 01243, 03213},
    {"go-sxX", 0241, 0675, 0664},
    {"u+w", 0453, 0400, 0600},
    {"g+ts", 0743, 077, 02077},
    {"g-sw", 0530, 0321, 0301},
    {"a-Xs", 0650, 00, 00},
    {"ag+wX", 0301, 0164, 0377},
    {"gou+s", 0632, 041, 06041},
    {"ua=Xwt", 0264, 0771, 01333},
    {"g+u", 0014, 0232, 0232},
    {"ga-wt", 0276, 0433, 0411},
    {"a+t", 0252, 02315, 03315},
    {"g=u", 0403, 0266, 0226},
    {"u-X", 0320, 0134, 034},
    {"ou+wr", 0746, 0266, 0666},
    {"ou-w", 0724, 0211, 011},
    {"g=ts", 0326, 072, 02002},
    {"u+s", 0603, 00, 04000},
    {"aou+X", 0033, 01675, 01775},
    {"ao-u", 0271, 0446, 02},
    {"go=wst", 0024, 0201, 03222},
    {"a-Xx", 0654, 0541, 0440},
    {"gau-Xx", 0661, 0444, 0444},
    {"au+xsw", 0150, 04242, 06373},
    {"o=ts", 0425, 0456, 01450},
    {"ga=g", 0102, 04612, 0111},
    {"173", 0751, 0607, 0173},
    {"ag-x=sX", 0771, 0301, 06000},
    {"uao=tX", 0071, 0115, 01111},
    {"o=rw", 0016, 0631, 0636},
    {"o+xw", 0305, 0744, 0747},
    {"gu-X+rsx", 0400, 0633, 06773},
    {"gou+ws", 0636, 0437, 06637},
    {"o+u", 0463, 0577, 0577},
    {"u=xsr", 0700, 0274, 04574},
    {"oug+rw", 0770, 0265, 0667},
    {"ao-t", 0370, 0427, 0427},
    {"g=w", 0603, 0456, 0426},
    {"o=u", 0371, 0774, 0777},
    {"o+r", 0654, 02117, 02117},
    {"a+xtr", 0113, 0304, 01755},
    {"+246", 0553, 0264, 0266},
    {"a-ws", 0472, 0272, 050},
    {"o-s+sXx", 0116, 0233, 0233},
    {"a-t", 0313, 04403, 04403},
    {"u+wX", 0671, 03, 0303},
    {"uga-o", 0072, 0140, 0140},
    {"ga+Xw", 0742, 0272, 0373},
    {"o-s", 0355, 0776, 0776},
    {"a-xt", 0334, 0244, 0244},
    {"ga+x", 0260, 0565, 0575},
    {"g=Xwr", 0523, 0524, 0574},
    {"ou-rx", 0266, 02136, 02032},
    {"a=sXw", 0445, 0232, 06333},
    {"og-s", 0331, 0145, 0145},
    {"au-x", 0102, 0365, 0264},
    {"a-rxX", 0442, 037, 022},
    {"u+s", 0653, 0663, 04663},
    {"gu+ws", 0242, 013, 06233},
    {"ou=o", 0302, 0223, 0323},
    {"a+tX", 0311, 04173, 05173},
    {"ou-x", 0055, 0127, 026},
    {"a=s", 0750, 0613, 06000},
    {"o+wx", 0246, 0400, 0403},
    {"gu=g", 0657, 0704, 04},
    {"o=tX", 0211, 062, 01060},
    {"ua=g", 0447, 0175, 0777},
    {"ou=xs", 0244, 0562, 04161},
    {"au+s", 0555, 0756, 06756},
    {"745", 0367, 0347, 0745},
    {"g=r+xt", 0627, 036, 056},
    {"o+sXx", 0634, 02664, 02665},
    {"=224", 0751, 0331, 0224},
    {"ou-Xxw", 0355, 0662, 0460},
    {"uoa-x", 0216, 0311, 0200},
    {"0566", 0150, 0222, 0566},
    {"gu+tX", 0741, 0777, 0777},
    {"g=u", 0667, 0364, 0334},
    {"ao+x", 0553, 0572, 0573},
    {"gu-xs", 0612, 0567, 0467},
    {"a=st", 0536, 0335, 07000},
    {"gua+t", 0032, 0776, 01776},
    {"gu=xX", 0205, 047, 0117},
    {"u+Xt", 0247, 0773, 0773},
    {"ua-sw", 0212, 012, 010},
    {"au+o", 0106, 0710, 0710},
    {"a+st", 0430, 00, 07000},
    {"1451", 0771, 0551, 01451},
    {"a-s", 0503, 022, 022},
    {"a=sr", 0505, 0372, 06444},
    {"ga=r", 0344, 02707, 0444},
    {"ga-tx", 0426, 0123, 022},
    {"u-xsw", 0435, 0736, 0436},
    {"ug-r", 0100, 0202, 0202},
    {"o+xX", 0573, 0427, 0427},
    {"og+xX", 0227, 02325, 02335},
    {"g-u", 0760, 0775, 0705},
    {"ao=xr", 0426, 0163, 0555},
    {"ao=s", 0477, 0312, 06000},
    {"a+u", 0630, 0742, 0777},
    {"ua+srw", 0515, 077, 06677},
    {"4463", 0776, 04733, 04463},
    {"g+r", 0617, 0571, 0571},
    {"o=Xt", 0146, 01142, 01141},
    {"u+sw", 0443, 02672, 06672},
    {"uga-s+Xx", 0723, 0310, 0311},
    {"guo+wt", 0130, 0177, 01377},
    {"ga+xXt", 0613, 0574, 01575},
    {"-4326", 0246, 0123, 01},
    {"ou-xX", 0054, 0523, 0422},
    {"oa+tX", 0145, 04100, 05111},
    {"ago=X", 0317, 04772, 0111},
    {"o-xX", 0154, 0164, 0164},
    {"g=xw", 0776, 0365, 0335},
    {"-070", 0577, 0260, 0200},
    {"go=s", 0770, 0636, 02600},
    {"a=ws", 0467, 0157, 06222},
    {"au=x", 0170, 0706, 0111},
    {"554", 0767, 0536, 0554},
    {"266", 0764, 02162, 0266},
    {"a=X-rw", 0106, 0401, 0111},
    {"a+wx", 0666, 00, 0333},
    {"oga-x", 0270, 0146, 046},
    {"aug-w", 0174, 0450, 0450},
    {"ao+w", 0431, 0276, 0276},
    {"au-t", 0124, 0472, 0472},
    {"u+o", 0625, 0644, 0644},
    {"a+wr", 0477, 0127, 0767},
    {"g-r", 0446, 0636, 0636},
    {"o+g", 0217, 0771, 0777},
    {"-352", 0070, 0607, 0405},
    {"aou=ts", 0113, 0236, 07000},
    {"o+ws", 0060, 0626, 0626},
    {"266", 0555, 067, 0266},
    {"g+tw", 0536, 0444, 0464},
    {"a=sX", 0512, 0512, 06111},
    {"ou=u", 0246, 04760, 0767},
    {"go=x", 0736, 0654, 0611},
    {"401", 0600, 0715, 0401},
    {"gu=twr", 0667, 0117, 0667},
    {"a=tsr", 0271, 0315, 07444},
    {"a=Xst", 0576, 0501, 07111},
    {"o=r", 0273, 034, 034},
    {"ug+Xw", 0741, 0300, 0330},
    {"301", 0660, 0711, 0301},
    {"o+tsX", 0072, 0561, 01561},
    {"uao-x=rts", 0142, 0147, 07444},
    {"g-o", 0004, 0411, 0401},
    {"uao=x", 0421, 0242, 0111},
    {"u-s", 0564, 0244, 0244},
    {"gu+xsw", 0510, 0477, 06777},
    {"au+rxX", 0543, 0450, 0555},
    {"ao+s", 0311, 0144, 06144},
    {"ug-x", 0510, 0171, 061},
    {"au+w", 0610, 055, 0277},
    {"a+txs", 0233, 040, 07151},
    {"ag=s", 0315, 0640, 06000},
    {"o-o", 0662, 045, 040},
    {"u=u", 0341, 0646, 0646},
    {"g=w", 0406, 0264, 0224},
    {"a-g", 0442, 0362, 0100},
    {"a=xw", 0740, 0176, 0333},
    {"o+xs", 0540, 0442, 0443},
    {"au+u", 0354, 02133, 02133},
    {"u-X", 0530, 0327, 0227},
    {"go+tx", 0475, 0247, 01257},
    {"ga+r", 0367, 0641, 0645},
    {"005", 0176, 04746, 05},
    {"uo=u", 0125, 0461, 0464},
    {"a=xsX", 0046, 0232, 06111},
    {"ua=sXw", 0040, 0241, 06333},
    {"ao+s", 0113, 0477, 06477},
    {"a-Xrs", 0564, 0304, 0200},
    {"go-g", 0543, 0471, 0400},
    {"uag=r", 0211, 0612, 0444},
    {"ug+rt", 0303, 0116, 0556},
    {"u+sr", 0535, 02646, 06646},
    {"oa=tw", 0515, 0370, 01222},
    {"ao=rX", 0007, 04422, 0444},
    {"ou-X", 0177, 0433, 0432},
    {"og-u", 0144, 02221, 02201},
    {"o=tsx", 0014, 0366, 01361},
    {"oa=x", 0376, 0256, 0111},
    {"oua+r", 0377, 0100, 0544},
    {"oug-x+t", 0313, 0566, 01466},
    {"aog=wXt", 0201, 0521, 01333},
    {"ga-r", 0130, 0732, 0332},
    {"a=wrt", 0121, 02, 01666},
    {"g+rs", 0123, 042, 02042},
    {"goa=xw", 0314, 0106, 0333},
    {"uao+ws", 0305, 0310, 06332},
    {"o+tXx", 0262, 0526, 01527},
    {"oau=tx", 0016, 037, 01111},
    {"ug-X", 0141, 0150, 040},
    {"go+x=s", 0766, 02273, 02200},
    {"uog-sr", 0741, 0736, 0332},
    {"g+r", 0327, 01746, 01746},
    {"gu+wXs", 0554, 0702, 06732},
    {"u-u", 0227, 0350, 050},
    {"+667", 0610, 0244, 0667},
    {"o+s", 0176, 0400, 0400},
    {"647", 0470, 0267, 0647},
    {"a+r", 0217, 0542, 0546},
    {"u=t", 0060, 0777, 077},
    {"o=r", 0017, 0400, 0404},
    {"u-st", 0346, 0513, 0513},
    {"o=st", 0352, 0764, 01760},
    {"ou=s", 0154, 04525, 04020},
    {"uo+sr", 0376, 047, 04447},
    {"ao=tr", 0232, 0420, 01444},
    {"ua+Xts", 0310, 0212, 07313},
    {"o+s", 0151, 0643, 0643},
    {"gua=w", 0422, 073, 0222},
    {"au+w", 0127, 0662, 0662},
    {"gau=w", 0023, 0700, 0222},
    {"ug=sx", 0250, 0176, 06116},
    {"a=g", 0104, 0352, 0555},
    {"g+g", 0770, 0420, 0420},
    {"o+x", 0267, 0661, 0661},
    {"agu+g", 0613, 057, 0557},
    {"ag+x=X", 0352, 0150, 0111},
    {"a+w=wx", 0453, 04524, 0333},
    {"a=Xw", 0312, 0247, 0333},
    {"614", 0430, 0310, 0614},
    {"o=xs", 0444, 0221, 0221},
    {"u-Xtw", 0037, 0310, 010},
    {"541", 0452, 01757, 0541},
    {"au-o", 0615, 051, 040},
    {"ogu-xw", 0145, 02, 00},
    {"go-x", 0263, 0536, 0526},
    {"uag=Xt", 0365, 0712, 01111},
    {"ug=g", 0232, 0172, 0772},
    {"ua+ws", 0017, 0671, 06673},
    {"ug+xt", 0175, 023, 0133},
    {"u-u", 0513, 0454, 054},
    {"uao=w", 0310, 0243, 0222},
    {"u=xst", 0155, 0453, 04153},
    {"o-sX", 0402, 01106, 01106},
    {"ua=o", 0207, 0271, 0111},
    {"ua-ws", 0441, 0140, 0140},
    {"g=wtr", 0763, 0123, 0163},
    {"au=x", 0415, 0646, 0111},
    {"a-g", 0311, 0407, 0407},
    {"uo=tx", 0703, 0466, 01161},
    {"uo+rxs", 0673, 0412, 04517},
    {"-651", 0071, 02740, 02100},
    {"aog-t", 0014, 0777, 0777},
    {"a-rw", 0634, 0731, 0111},
    {"ug-o", 0611, 0761, 0661},
    {"4774", 0357, 0645, 04774},
    {"gu+wt", 0655, 0753, 0773},
    {"g+xt", 0464, 0702, 0712},
    {"uo+Xws", 0075, 0524, 04727},
    {"agu=trx", 0253, 0467, 01555},
    {"gu=Xxw", 0743, 0576, 0336},
    {"gau+tXr", 0757, 0625, 01775},
    {"o-r", 0270, 045, 041},
    {"o-xr", 0477, 0737, 0732},
    {"g=wx", 0507, 0564, 0534},
    {"uao+Xt", 0164, 0651, 01751},
    {"ag-r=tx", 0463, 04270, 01111},
    {"oag-wrs", 0367, 0320, 0100},
    {"ou+xs", 0414, 0476, 04577},
    {"o-sxX", 0660, 0550, 0550},
    {"ag=ws", 0632, 0523, 06222},
    {"g-s", 0715, 070, 070},
    {"uo-x", 0417, 0166, 066},
    {"oa=r", 0140, 0234, 0444},
    {"a=wX", 0033, 0446, 0222},
    {"oa+t", 0750, 0601, 01601},
    {"gu-sw", 0441, 0340, 0140},
    {"ugo+t", 0315, 0171, 01171},
    {"au+xt", 0173, 0636, 01737},
    {"ug=tXr", 0566, 0603, 0553},
    {"g=wX", 0517, 04731, 04731},
    {"ug+xs", 0330, 0442, 06552},
    {"ago+t", 0670, 0671, 01671},
    {"guo=u", 0436, 016, 00},
    {"og-o", 0142, 0362, 0340},
    {"a=rws", 0543, 04164, 06666},
    {"ga=g", 0133, 0611, 0111},
    {"ao=t", 0465, 0173, 01000},
    {"og-u", 0222, 0471, 0431},
    {"gu+rs", 0763, 0307, 06747},
    {"og+u", 0540, 0306, 0337},
    {"024", 0621, 0730, 024},
    {"au=X-xws", 0742, 0140, 00},
    {"632", 0200, 04067, 0632},
    {"ao-rX", 0472, 0653, 0202},
    {"og=s", 0464, 0371, 02300},
    {"aog+Xs", 0362, 0227, 06337},
    {"oa-xs", 0357, 0212, 0202},
    {"o=sw", 0245, 0377, 0372},
    {"g=r", 0210, 0217, 0247},
    {"ugo+rX", 0145, 0757, 0757},
    {"ao-twx", 0442, 0605, 0404},
    {"ag-s", 0306, 0431, 0431},
    {"ou-u", 0310, 0620, 020},
    {"ug+r", 0471, 0213, 0653},
    {"og-r+rxw", 0436, 0277, 0277},
    {"ua-r", 0774, 0353, 0313},
    {"ug-tx", 0007, 0734, 0624},
    {"g-wt", 0213, 027, 07},
    {"uag-xr", 0444, 02, 02},
    {"a+w=w-tw", 0711, 0117, 00},
    {"a-r", 0050, 0766, 0322},
    {"go-o", 0641, 023, 00},
    {"o+sw", 0553, 023, 023},
    {"gou=txr", 0404, 0772, 01555},
    {"g=wst", 0545, 03, 02023},
    {"aug-Xtr", 0222, 0422, 022},
    {"g-X=w", 0035, 0501, 0521},
    {"=264", 0742, 02204, 0264},
    {"557", 0164, 0424, 0557},
    {"a=o", 0235, 0450, 00},
    {"200", 0473, 0301, 0200},
    {"oag-sw", 0333, 02307, 0105},
    {"656", 0443, 03, 0656},
    {"og=x", 0222, 0565, 0511},
    {"go+wX", 0010, 0264, 0266},
    {"uo+Xrw", 0556, 0265, 0767},
    {"go=o", 0233, 01534, 0544},
    {"g-Xxw", 0636, 0171, 0141},
    {"a-r", 0176, 02210, 02210},
    {"a+u", 0227, 0503, 0557},
    {"uo=xw", 0244, 0404, 0303},
    {"a-g", 0433, 0344, 0300},
    {"oa=ws", 0261, 0476, 06222},
    {"oa+g", 0112, 0740, 0744},
    {"ag-X", 0126, 0440, 0440},
    {"ag-w", 0460, 0326, 0104},
    {"ao=u", 0200, 054, 00},
    {"u+trs", 0161, 0353, 04753},
    {"og=o", 0367, 0661, 0611},
    {"ag+u", 0725, 0161, 0171},
    {"oa-t", 0046, 01403, 0403},
    {"go=Xx", 0324, 0772, 0711},
    {"a=txX", 0565, 02220, 01111},
    {"g=xs", 0442, 021, 02011},
    {"2045", 0340, 0765, 02045},
    {"-726", 0120, 0733, 011},
    {"u=X", 0273, 04162, 0162},
    {"gou+t", 0174, 0622, 01622},
    {"047", 0713, 0474, 047},
    {"345", 0534, 0201, 0345},
    {"oag-sx", 0364, 0517, 0406},
    {"oga+g", 0164, 0770, 0777},
    {"a+g", 0433, 0407, 0407},
    {"+144", 0035, 0226, 0366},
    {"u=u", 0434, 0646, 0646},
    {"a=x-xw", 0623, 04355, 00},
    {"uoa+o", 0300, 0116, 0776},
    {"a-s", 0106, 0242, 0242},
    {"go+g", 0506, 0765, 0767},
    {"ag-Xwx", 0073, 0475, 0444},
    {"g=g", 0532, 0244, 0244},
    {"a-w", 0162, 0104, 0104},
    {"o=w", 0016, 0756, 0752},
    {"a-g", 0214, 0477, 00},
    {"u=xX", 0201, 0326, 0126},
    {"o+g", 0015, 0327, 0327},
    {"ua+s", 0534, 0225, 06225},
    {"ga-tr", 0004, 0420, 020},
    {"g=X", 0271, 0472, 0412},
    {"gu=s+r", 0313, 04765, 06445},
    {"ua+g", 0351, 023, 0223},
    {"g-wX", 0774, 0712, 0702},
    {"g=g", 0466, 0713, 0713},
    {"637", 0250, 02034, 0637},
    {"ag-st", 0404, 0705, 0705},
    {"ga+r", 0162, 0103, 0547},
    {"a=wt", 0246, 0366, 01222},
    {"o-g", 0030, 0624, 0624},
    {"gu+x", 0776, 0552, 0552},
    {"go=X-xt", 0066, 0653, 0600},
    {"a=srt", 0014, 0545, 07444},
    {"u=w", 0134, 0342, 0242},
    {"ao+t", 0537, 0546, 01546},
    {"u=w", 0527, 074, 0274},
    {"174", 0227, 0437, 0174},
    {"o=t", 0644, 0632, 01630},
    {"ou=w", 0042, 0546, 0242},
    {"ao-Xt", 0637, 0665, 0664},
    {"u+o", 0511, 0527, 0727},
    {"776", 0772, 0210, 0776},
    {"oag+x=stx", 0723, 0604, 07111},
    {"au-Xs", 0054, 0772, 0662},
    {"ugo=t+x", 0260, 0717, 01111},
    {"uo+tx", 0525, 0576, 01577},
    {"g-twx", 0430, 0420, 0400},
    {"u-x", 0504, 0532, 0432},
    {"a+xts", 0144, 0210, 07311},
    {"a+x", 0601, 0516, 0517},
    {"621", 0110, 0420, 0621},
    {"oa-w", 0064, 0672, 0450},
    {"oga+r-s", 0234, 0245, 0645},
    {"go-Xr", 0547, 051, 00},
    {"ou-x", 0516, 0200, 0200},
    {"ua+X-t", 0610, 0101, 0111},
    {"a-X", 0474, 033, 022},
    {"gu+Xwr", 0503, 04572, 04772},
    {"ga=sx", 0727, 0637, 06111},
    {"u-sXr", 0750, 02063, 02063},
    {"og=Xt", 0317, 00, 01000},
    {"o=r", 0567, 00, 04},
    {"ao=g", 0052, 0510, 0111},
    {"guo-wX", 0043, 043, 040},
    {"ga=ws", 0310, 020, 06222},
    {"a-X", 0530, 0412, 0402},
    {"og=sx", 0663, 0253, 02211},
    {"gu=r+tX", 0525, 0152, 0442},
    {"oa-o", 0540, 0361, 0260},
    {"u-X", 0055, 0733, 0633},
    {"au-wtx", 0410, 0666, 0444},
    {"u=g", 0653, 0475, 0775},
    {"ou-r+t", 0223, 0774, 01370},
    {"o-st", 0226, 0167, 0167},
    {"au=s", 0723, 026, 06000},
    {"u+o", 0051, 0506, 0706},
    {"u+s=srX", 0007, 0173, 04573},
    {"oa-g", 0412, 0646, 0202},
    {"uo-g", 0037, 0305, 0305},
    {"ag+g", 0521, 062, 0666},
    {"g+g", 0526, 0416, 0416},
    {"gu-Xt", 0027, 0621, 0621},
    {"-277", 0446, 0265, 00},
    {"ou+o", 0721, 041, 0141},
    {"g=X", 0665, 036, 016},
    {"ao-Xx", 0157, 0554, 0444},
    {"go+r", 0022, 02736, 02776},
    {"u=g", 0114, 0350, 0550},
    {"ao+Xwr", 0632, 0156, 0777},
    {"u-twx", 0423, 0101, 01},
    {"ou+t", 0025, 0731, 01731},
    {"g=tx", 0515, 0662, 0612},
    {"u+xrs", 0322, 021, 04521},
    {"ag=rX", 0761, 0147, 0555},
    {"ao=s", 0054, 037, 06000},
    {"g-Xrs", 0255, 0273, 0223},
    {"goa+xs", 0314, 0213, 06313},
    {"gu-x", 0214, 0400, 0400},
    {"au=t", 0641, 02641, 01000},
    {"a-r", 0112, 0146, 0102},
    {"ag+xrw", 0777, 0710, 0777},
    {"au-rx", 0033, 0744, 0200},
    {"o=xt", 0413, 0451, 01451},
    {"au+rX", 0246, 04511, 04555},
    {"323", 0151, 01202, 0323},
    {"u-wr", 0562, 01000, 01000},
    {"2577", 0447, 0441, 02577},
    {"oau=r", 0240, 0434, 0444},
    {"o=u", 0413, 0676, 0676},
    {"a-t", 0157, 0743, 0743},
    {"auo-X+Xwx", 0143, 0100, 0333},
    {"o-o", 0611, 067, 060},
    {"535", 0013, 0450, 0535},
    {"ou=u", 0303, 0333, 0333},
    {"u+sxw", 0412, 0127, 04327},
    {"1304", 0440, 04672, 01304},
    {"570", 0350, 0121, 0570},
    {"141", 0531, 0703, 0141},
    {"oau-Xt", 0433, 0461, 0460},
    {"ua+u", 0673, 0251, 0273},
    {"og+w-xrt", 0442, 0126, 0122},
    {"141", 0255, 0703, 0141},
    {"g=sr", 0742, 01575, 03545},
    {"ou+g", 0516, 0775, 0777},
    {"guo+x", 0250, 0642, 0753},
    {"a=g", 0503, 035, 0333},
    {"ug-sx", 0346, 02533, 0423},
    {"u+g", 0116, 0600, 0600},
    {"ua+tX", 0353, 054, 01155},
    {"aog=g", 0544, 0222, 0222},
    {"557", 0017, 0257, 0557},
    {"oau+g", 0505, 0100, 0100},
    {"ua-xs", 0741, 010, 00},
    {"uo+x-t", 0154, 0240, 0341},
    {"oa-Xwt", 0160, 0302, 00},
    {"a-rXw", 0620, 02451, 02000},
    {"og=w", 0253, 0772, 0722},
    {"uao-ts", 0257, 0714, 0714},
    {"o+w", 0744, 0116, 0116},
    {"au+u", 0506, 0401, 0445},
    {"u+u", 0705, 04266, 04266},
    {"oug+tr", 0366, 0452, 01456},
    {"ua-wX", 0526, 0534, 0404},
    {"554", 0413, 0364, 0554},
    {"g-g", 0621, 0547, 0507},
    {"u=o", 0120, 0770, 070},
    {"a-tr", 0672, 0132, 0132},
    {"a-u", 0052, 025, 025},
    {"a=w", 0377, 0634, 0222},
    {"uo+t", 0557, 066, 01066},
    {"ug+rsw", 0444, 0254, 06674},
    {"oga+xr", 0731, 0743, 0757},
    {"uao+Xxt", 0471, 0766, 01777},
    {"-1343", 0336, 0370, 030},
    {"o=r", 0270, 0453, 0454},
    {"u-rsw", 0773, 0175, 0175},
    {"uo=sxr", 0672, 0414, 04515},
    {"og=rsx", 0660, 066, 02055},
    {"goa-Xsw", 0114, 0626, 0404},
    {"gao+r-t", 0613, 0716, 0756},
    {"=323", 0213, 075, 0323},
    {"ao-rx", 0115, 0200, 0200},
    {"gu+u", 0621, 0546, 0556},
    {"go+rt", 0466, 00, 01044},
    {"241", 0470, 0541, 0241},
    {"go=xt", 0405, 01464, 01411},
    {"ogu=sw", 0233, 0532, 06222},
    {"g=wXs", 0131, 0604, 02624},
    {"a+rws", 0270, 04, 06666},
    {"og=Xw", 0236, 0647, 0633},
    {"oua=x", 0576, 0724, 0111},
    {"og+w", 0403, 015, 037},
    {"ug=s-xt", 0655, 071, 06001},
    {"u=t", 0027, 0203, 03},
    {"aou-r", 0301, 0235, 0231},
    {"a-g", 0167, 0227, 05},
    {"126", 0201, 020, 0126},
    {"ago+sx", 0232, 0776, 06777},
    {"ugo+Xxw", 0356, 01207, 01337},
    {"ao-s", 0560, 076, 076},
    {"567", 0010, 0505, 0567},
    {"gao=o", 0407, 0170, 00},
    {"a-st", 0653, 0521, 0521},
    {"a=tXx", 0366, 0674, 01111},
    {"g+Xt", 0045, 0376, 0376},
    {"gao-rXt", 0520, 0241, 0200},
    {"ou-t", 0761, 0735, 0735},
    {"uo=tr", 0327, 0722, 01424},
    {"g+r", 0343, 02612, 02652},
    {"ou=Xr", 0632, 040, 0444},
    {"oa=x+rs", 0147, 0610, 06555},
    {"gu=t", 0365, 0367, 07},
    {"g-t", 0330, 043, 043},
    {"oa+o", 0310, 0101, 0111},
    {"uo-rX", 0206, 062, 062},
    {"ao=rX", 0153, 0613, 0555},
    {"u=xXs", 0244, 04210, 04110},
    {"guo=x", 0011, 02617, 0111},
    {"o-st", 0134, 0141, 0141},
    {"o+g", 0326, 0353, 0357},
    {"u-s", 0227, 0730, 0730},
    {"oa=tX", 0254, 0634, 01111},
    {"g-u", 0545, 02, 02},
    {"g-s", 0440, 0462, 0462},
    {"g-Xr", 0533, 0747, 0707},
    {"4116", 0452, 0213, 04116},
    {"u+o", 0501, 0675, 0775},
    {"o+x", 0733, 0705, 0705},
    {"uo+ws", 0232, 040, 04242},
    {"ug-u", 0525, 0113, 03},
    {"u-sr", 0222, 0447, 047},
    {"o=rs", 0410, 01722, 0724},
    {"g+X", 0256, 0142, 0152},
    {"o-g", 0434, 0654, 0650},
    {"uga+rs", 0713, 066, 06466},
    {"ou+st", 0163, 0517, 05517},
    {"oau-g", 0513, 02301, 02301},
    {"o+r", 0321, 0317, 0317},
    {"au+X", 0213, 0720, 0731},
    {"ou=w", 0537, 0256, 0252},
    {"a-o", 0325, 0244, 0200},
    {"a=g", 0326, 0115, 0111},
    {"a-xs", 0445, 0314, 0204},
    {"o=Xrx", 0747, 060, 065},
    {"ou-xw", 0233, 0143, 040},
    {"ug=Xt", 0777, 0553, 0113},
    {"g-ts", 0517, 0252, 0252},
    {"uag-xr", 0100, 0644, 0200},
    {"go=w+w", 0744, 0644, 0622},
    {"g+X", 0647, 0610, 0610},
    {"o+txs", 0044, 0312, 01313},
    {"oag+x", 0754, 0133, 0133},
    {"ug-wtr", 0676, 0415, 015},
    {"g-xXw", 0726, 0144, 0144},
    {"ao-w=xs", 0732, 01742, 06111},
    {"a=wXx", 0313, 02577, 0333},
    {"u=sw", 0762, 0621, 04221},
    {"a-wX", 0600, 0300, 00},
    {"og+s", 0501, 02674, 02674},
    {"g=w", 0575, 0260, 0220},
    {"g=twX", 0056, 0552, 0532},
    {"465", 0032, 01600, 0465},
    {"a-rXw", 0437, 0563, 00},
    {"uo+w", 0530, 0313, 0313},
    {"a+u", 0062, 0200, 0222},
    {"a=s", 0452, 0636, 06000},
    {"gu-xrX", 0435, 0354, 0204},
    {"g=s", 0072, 0573, 02503},
    {"ga-Xw", 0543, 0265, 044},
    {"ao=u", 0624, 0434, 0444},
    {"ua+rt", 0256, 016, 01456},
    {"g+s", 0010, 0473, 02473},
    {"go+s", 0332, 0227, 02227},
    {"u+t", 0425, 0202, 0202},
    {"g+x", 0321, 0253, 0253},
    {"ao+rw", 0201, 0674, 0676},
    {"g-sx", 0310, 0340, 0340},
    {"g+r", 0437, 02250, 02250},
    {"a-w", 0115, 0223, 01},
    {"a=rs", 0776, 0556, 06444},
    {"a=o", 0113, 0171, 0111},
    {"u=u", 0722, 0276, 0276},
    {"1626", 0422, 0700, 01626},
    {"au=wsr", 0726, 00, 06666},
    {"o=Xx", 0223, 0533, 0531},
    {"+712", 0166, 0460, 0772},
    {"gu=sw", 0120, 0676, 06226},
    {"o-tr", 0221, 0446, 0442},
    {"ao=rx", 0437, 0441, 0555},
    {"a-x", 0027, 0461, 0460},
    {"a-tXw", 0520, 077, 044},
    {"u+o", 0206, 0325, 0725},
    {"ga+s", 0706, 0361, 06361},
    {"ao=sx", 0062, 0604, 06111},
    {"a+rXt", 0370, 071, 01575},
    {"uo-s", 0572, 0663, 0663},
    {"oua-r", 0131, 0406, 02},
    {"g-r", 0665, 02133, 02133},
    {"uo=t", 0360, 0643, 01040},
    {"uo+w", 0552, 0766, 0766},
    {"oa=t", 0670, 01332, 01000},
    {"oa+s", 0204, 0634, 06634},
    {"gou+o", 0424, 0145, 0555},
    {"-347", 0701, 0113, 010},
    {"au-wxt", 0130, 0336, 04},
    {"o-g", 0241, 0766, 0760},
    {"g=o", 0672, 0617, 0677},
    {"110", 0023, 0640, 0110},
    {"a-r", 0531, 02060, 02020},
    {"g-u", 0100, 0403, 0403},
    {"u-o", 0157, 0352, 0152},
    {"uo=r", 0111, 0343, 0444},
    {"ua=s", 0735, 073, 06000},
    {"a-tr", 0764, 0332, 0332},
    {"g+s", 0616, 0751, 02751},
    {"=657", 0157, 0114, 0657},
    {"og-sw", 0101, 04677, 04655},
    {"uo+rw", 0541, 0133, 0737},
    {"a=tX", 0017, 022, 01000},
    {"oa+o", 0073, 04, 0444},
    {"g+txX", 0645, 0617, 0617},
    {"go+tsr", 0441, 037, 03077},
    {"410", 0466, 04150, 0410},
    {"476", 0753, 0763, 0476},
    {"g+w", 0121, 0421, 0421},
    {"u+sw", 0436, 0574, 04774},
    {"ua-rXw", 0153, 0450, 00},
    {"ua=g", 0652, 0444, 0444},
    {"og-tx", 0302, 01104, 0104},
    {"a=g", 0143, 0462, 0666},
    {"ao+srx", 0636, 0446, 06557},
    {"uog-wXx", 0106, 0660, 0440},
    {"u-tx", 0303, 01477, 01477},
    {"u-Xx", 0263, 0203, 0203},
    {"u-wsX", 0352, 0254, 054},
    {"oa=X", 0541, 0735, 0111},
    {"ou+g", 0535, 0233, 0333},
    {"-4072", 0775, 0350, 0300},
    {"ua+o", 0541, 04156, 04776},
    {"uo-s", 0452, 0514, 0514},
    {"216", 0057, 0634, 0216},
    {"g-Xs", 0176, 07, 07},
    {"o-u", 0113, 0415, 0411},
    {"o-ts", 0107, 0566, 0566},
    {"u+x", 0003, 0366, 0366},
    {"ao-o", 0625, 0610, 0610},
    {"ugo+rsX", 0403, 0571, 06575},
    {"ag+xw", 0014, 0262, 0373},
    {"ag+x", 0127, 0267, 0377},
    {"a+sX", 0307, 0747, 06757},
    {"g-rX", 0426, 0377, 0327},
    {"4007", 0612, 027, 04007},
    {"o-xr", 0606, 0152, 0152},
    {"aug-xX", 0017, 0253, 0242},
    {"aug=rwt", 0166, 0734, 01666},
    {"ou-u", 0517, 0156, 056},
    {"oug=sXx", 0412, 0224, 06111},
    {"o-sx", 0432, 0164, 0164},
    {"u+s", 0541, 0571, 04571},
    {"uo-g", 0532, 0576, 070},
    {"o=g", 0055, 0354, 0355},
    {"u+rw", 0651, 0150, 0750},
    {"au=u", 0045, 04726, 0777},
    {"o+s", 0277, 0155, 0155},
    {"uo+x", 0201, 0700, 0701},
    {"o-stX", 0740, 0625, 0624},
    {"u=wr", 0665, 01653, 01653},
    {"og=X-Xwx", 0172, 0340, 0300},
    {"go+sX", 0200, 0527, 02537},
    {"635", 0666, 0616, 0635},
    {"441", 0015, 04136, 0441},
    {"o+Xs", 0654, 00, 00},
    {"a-o", 0304, 02037, 02000},
    {"a-wx", 0620, 0520, 0400},
    {"o-xt", 0217, 040, 040},
    {"g=x", 0421, 0532, 0512},
    {"a=rtw", 0570, 0221, 01666},
    {"go=s", 0736, 0410, 02400},
    {"713", 0130, 0441, 0713},
    {"ua=o", 0321, 046, 0666},
    {"oa-g", 0666, 02270, 02000},
    {"u=u", 0275, 0565, 0565},
    {"o+sX", 0330, 0657, 0657},
    {"uo=o", 0030, 02403, 02303},
    {"ao=tX", 0535, 0240, 01000},
    {"u+rXt", 0647, 0272, 0772},
    {"o-s", 0706, 0124, 0124},
    {"ug-Xwt", 0064, 0505, 0405},
    {"a+s", 0415, 0566, 06566},
    {"a-t", 0575, 0373, 0373},
    {"oa+Xr", 0250, 0343, 0757},
    {"ga-s", 0170, 0602, 0602},
    {"a+rsX", 0357, 0616, 06757},
    {"ug-x+tx", 0523, 0323, 0333},
    {"u+Xr", 0257, 0301, 0701},
    {"ag-o", 0616, 020, 020},
    {"u+xwr", 0342, 0372, 0772},
    {"ag+sx", 0427, 0113, 06113},
    {"ogu=xr", 0463, 0364, 0555},
    {"ogu=Xt", 0474, 04712, 01111},
    {"a+ts", 0147, 0635, 07635},
    {"ago=xw", 0513, 0746, 0333},
    {"a=rxw", 0073, 00, 0777},
    {"ga=wst", 0345, 073, 07222},
    {"oa-g", 0617, 060, 00},
    {"a+u", 0030, 0725, 0777},
    {"go=tX", 0653, 0600, 01600},
    {"a=t", 0436, 0471, 01000},
    {"a+tX", 0463, 0144, 01155},
    {"g=xt", 0303, 0532, 0512},
    {"317", 0353, 0202, 0317},
    {"go=u", 0470, 04005, 04000},
    {"731", 0765, 0567, 0731},
    {"aog-wX", 0367, 0741, 0440},
    {"ao-wx", 0560, 066, 044},
    {"o=ts", 0236, 0734, 01730},
    {"o-g", 0214, 0217, 0216},
    {"oug+t", 0172, 0536, 01536},
    {"o+sr", 0015, 0525, 0525},
    {"o=r", 0173, 02627, 02624},
    {"ag-X", 0323, 0770, 0660},
    {"g=sx", 0753, 04375, 06315},
    {"u=rwX", 0631, 0111, 0711},
    {"u+o", 0273, 0704, 0704},
    {"a=ws", 0077, 0651, 06222},
    {"oa-w", 0376, 0147, 0145},
    {"ao+u", 0742, 020, 020},
    {"o=wX", 0601, 0446, 0442},
    {"o+g", 0171, 0501, 0501},
    {"o-wx", 0173, 0360, 0360},
    {"go+rs", 0301, 0646, 02646},
    {"g+x", 0631, 01267, 01277},
    {"g-sxt", 0215, 0472, 0462},
    {"gu+sx", 0706, 01303, 07313},
    {"o+x", 0061, 0305, 0305},
    {"oa=st", 0710, 0147, 07000},
    {"gou+x=swx", 0755, 0576, 06333},
    {"a-st", 0666, 02564, 0564},
    {"ug+wt", 0454, 0650, 0670},
    {"oa=xX", 0642, 0602, 0111},
    {"aou+u", 0623, 0236, 0236},
    {"ua+r", 0146, 053, 0457},
    {"ua=x", 0067, 0702, 0111},
    {"ua=x-sX", 0274, 0124, 00},
    {"ag=X", 0507, 0352, 0111},
    {"334", 0143, 0163, 0334},
    {"ag-xw", 0571, 01626, 01404},
    {"go-rxs", 0646, 0314, 0300},
    {"uo-u", 0375, 0706, 00},
    {"a+o", 0351, 0634, 0674},
    {"g-s", 0137, 0323, 0323},
    {"a+g", 0551, 0464, 0666},
    {"346", 0325, 0667, 0346},
    {"ag-xtw", 0507, 0234, 04},
    {"au=sw", 0576, 02467, 06222},
    {"o=s-w", 0216, 0776, 0770},
    {"aug=w-r", 0076, 00, 0222},
    {"721", 0424, 0426, 0721},
    {"o-ts", 0772, 0612, 0612},
    {"g+g", 0373, 01542, 01542},
    {"o=s", 0716, 0125, 0120},
    {"ga-Xrw", 0210, 0404, 00},
    {"g+u", 0641, 0177, 0177},
    {"oug+o", 0611, 0340, 0340},
    {"o-g", 0030, 0306, 0306},
    {"0746", 0737, 0745, 0746},
    {"go=sw", 0626, 0735, 02722},
    {"gau+o", 0166, 0321, 0331},
    {"o-sx", 0650, 0455, 0454},
    {"ou+tr", 0360, 0344, 01744},
    {"a-u", 0742, 0217, 015},
    {"ao+xw", 0506, 0132, 0333},
    {"o=st", 0631, 0414, 01410},
    {"au=s", 0527, 01672, 06000},
    {"ga-trw", 0007, 047, 01},
    {"a=wrx", 0574, 01370, 0777},
    {"gao=twx", 0411, 0556, 01333},
    {"g+wt", 0406, 036, 036},
    {"gu+r", 0465, 01401, 01441},
    {"ga=rt", 0352, 02642, 01444},
    {"u+wsX", 0776, 0643, 04743},
    {"go=x+X-X", 0377, 04606, 04600},
    {"uo-ts", 0774, 0260, 0260},
    {"gu=txr", 0326, 04323, 0553},
    {"gu=t", 0343, 0257, 07},
    {"0506", 0662, 0542, 0506},
    {"uoa-g", 0066, 0625, 0405},
    {"a=r", 0663, 04, 0444},
    {"a=w", 0702, 0125, 0222},
    {"g+xst", 0035, 0132, 02132},
    {"o-x", 0405, 0765, 0764},
    {"a=X", 0733, 052, 0111},
    {"a+s", 0441, 02656, 06656},
    {"a+X", 0053, 0203, 0313},
    {"1052", 0536, 02006, 01052},
    {"u+rts", 0524, 0464, 04464},
    {"g-r", 0414, 04523, 04523},
    {"ou=u", 0573, 0477, 0474},
    {"g=rXw", 0040, 0641, 0671},
    {"a+xr", 0320, 0763, 0777},
    {"go+g", 0005, 0205, 0205},
    {"ou-x", 0032, 0644, 0644},
    {"a=sr", 0366, 01076, 06444},
    {"a+wts", 0150, 0777, 07777},
    {"go=Xx", 0637, 0541, 0511},
    {"o=r", 0021, 036, 034},
    {"auo+o", 0655, 02770, 02770},
    {"ogu+s", 0721, 0476, 06476},
    {"auo-x", 0365, 0667, 0666},
    {"au=Xrt", 0660, 0250, 01555},
    {"u+s", 0371, 0441, 04441},
    {"ua+X", 0372, 0473, 0573},
    {"ao=w", 0725, 0161, 0222},
    {"au-xw", 0172, 0355, 044},
    {"g-xwX", 0234, 0431, 0401},
    {"g+xwX", 0652, 0323, 0333},
    {"ou-Xxw", 0644, 0243, 040},
    {"2725", 0070, 0716, 02725},
    {"u-g", 0017, 01431, 01431},
    {"ua+x", 0001, 0604, 0715},
    {"ga-rw", 0073, 00, 00},
    {"og-sr", 0065, 0400, 0400},
    {"a=o", 0532, 0242, 0222},
    {"ua=x", 0611, 07, 0111},
    {"a=Xwr", 0736, 00, 0666},
    {"625", 0354, 0756, 0625},
    {"=043", 0676, 0733, 043},
    {"oug-t", 0252, 01557, 0557},
    {"g=rxw", 0562, 0467, 0477},
    {"g=u", 0613, 0411, 0441},
    {"uga-t=x", 0437, 0667, 0111},
    {"oa+r", 0002, 0305, 0745},
    {"au+o", 0534, 0772, 0772},
    {"g+w", 0561, 0676, 0676},
    {"ou-xwt", 0461, 0245, 044},
    {"ag-g", 0301, 0223, 01},
    {"g-x", 0655, 0146, 0146},
    {"au+xrX", 0556, 0575, 0575},
    {"u+s-rX", 0035, 06, 04006},
    {"2662", 0517, 0206, 02662},
    {"oa=r", 0263, 0712, 0444},
    {"og-txw", 0070, 077, 044},
    {"ou=xrt", 0071, 0561, 01565},
    {"ug=w", 0531, 0226, 0226},
    {"ug=sXr", 0760, 0156, 06556},
    {"u-t", 0530, 0755, 0755},
    {"600", 0361, 04534, 0600},
    {"ga-X", 0353, 0524, 0424},
    {"g=r", 0260, 0431, 0441},
    {"oug=ws", 0257, 01461, 06222},
    {"gu+o", 0046, 0567, 0777},
    {"u+sX", 0675, 02, 04002},
    {"o=g", 0567, 0105, 0100},
    {"ou-sx", 0524, 0203, 0202},
    {"go+Xwr", 0041, 0114, 0177},
    {"gu+u", 0372, 0510, 0550},
    {"au=t", 0653, 030, 01000},
    {"og+st", 0112, 0402, 03402},
    {"ua=tr", 0002, 0107, 01444},
    {"a+r", 0712, 0475, 0475},
    {"=331", 0312, 0213, 0331},
    {"677", 0402, 01754, 0677},
    {"au+tX", 0113, 0667, 01777},
    {"g+wr", 0603, 0556, 0576},
    {"374", 0662, 0607, 0374},
    {"u=Xw", 0317, 0177, 0377},
    {"ug-t", 0132, 0406, 0406},
    {"ou=u", 0677, 0451, 0454},
    {"o+u", 0001, 0352, 0353},
    {"g=t", 0062, 0321, 0301},
    {"ua+u", 0754, 0266, 0266},
    {"uoa+o", 0017, 0133, 0333},
    {"436", 0771, 0511, 0436},
    {"uo+x", 0174, 01434, 01535},
    {"a+sxw", 0210, 0527, 06737},
    {"g+x", 0241, 0566, 0576},
    {"og-s", 0431, 0130, 0130},
    {"a-o", 0704, 0762, 0540},
    {"o-g", 0166, 04373, 04370},
    {"ao=X", 0247, 0661, 0111},
    {"ogu-t+sr", 0245, 0660, 06664},
    {"ou=sr", 0311, 0462, 04464},
    {"a+wx", 0125, 0471, 0773},
    {"go+rxs", 0112, 0220, 02275},
    {"uag-sw", 0424, 0240, 040},
    {"oa=Xr", 0425, 021, 0555},
    {"4130", 0655, 0131, 04130},
    {"a=t", 0226, 0230, 01000},
    {"g-ws", 0044, 0422, 0402},
    {"ua+rs", 0342, 0703, 06747},
    {"uo+tr", 0004, 0405, 01405},
    {"163", 0424, 0213, 0163},
    {"u-wxr", 0670, 0301, 01},
    {"u-Xs", 0416, 0353, 0253},
    {"u-rx", 0677, 0472, 072},
    {"u=rts", 0150, 0200, 04400},
    {"og+Xx", 0515, 0265, 0275},
    {"a+sw", 0177, 0700, 06722},
    {"ug-g", 0145, 0605, 0605},
    {"u=rt", 0643, 0434, 0434},
    {"guo=xsX", 0251, 0520, 06111},
    {"gu-s", 0054, 06, 06},
    {"ga-o", 0316, 0403, 0400},
    {"oug+X", 0603, 0400, 0400},
    {"gu+g", 0766, 0525, 0725},
    {"ug-rwx", 0715, 0150, 00},
    {"ug=srx", 0665, 00, 06550},
    {"ua=txr", 0404, 0232, 01555},
    {"ag+twx", 0347, 02435, 03737},
    {"oa-ws", 0031, 0666, 0444},
    {"g=tX", 0067, 0546, 0516},
    {"o-g", 0456, 0556, 0552},
    {"u+wx", 0356, 0400, 0700},
    {"ou-x=wtX", 0561, 0537, 01333},
    {"ou-u", 0430, 02575, 02070},
    {"ua=Xs", 0014, 0631, 06111},
    {"uga=Xr", 0653, 0305, 0555},
    {"u+g", 0642, 0766, 0766},
    {"oga+r", 0124, 0641, 0645},
    {"uo=wtX", 0504, 01362, 01363},
    {"a+X", 0314, 0611, 0711},
    {"go+st", 0112, 0460, 03460},
    {"oa+w", 0421, 0374, 0376},
    {"oua=tr", 0126, 0653, 01444},
    {"ug+w", 0650, 0245, 0265},
    {"o=X", 0272, 04743, 04741},
    {"uga+o", 0232, 01615, 01755},
    {"u+r", 0603, 04376, 04776},
    {"ua+o", 0205, 0720, 0720},
    {"og+tx", 0750, 0635, 01635},
    {"g=t+rs", 0436, 0354, 02344},
    {"g+w", 0365, 0331, 0331},
    {"oa-sx", 0434, 0242, 0242},
    {"au-Xr", 0444, 0622, 0222},
    {"=0700", 0767, 0171, 0700},
    {"uoa=u", 0652, 0246, 0222},
    {"go-o", 0610, 062, 040},
    {"a=tsw", 0530, 0622, 07222},
    {"g+sx", 0550, 01, 02011},
    {"ugo-Xsr", 0757, 0635, 0220},
    {"a-w", 0463, 0755, 0555},
    {"ua=xw", 0403, 025, 0333},
    {"og+Xs", 0423, 0425, 02435},
    {"og+wXt", 0554, 0222, 01222},
    {"ao+rX", 0635, 0604, 0644},
    {"o-ws", 0434, 0276, 0274},
    {"aug-u", 0732, 0234, 014},
    {"gu-t", 0370, 0375, 0375},
    {"uog-t", 0464, 0502, 0502},
    {"1233", 0700, 0621, 01233},
    {"oua-u", 0245, 0402, 02},
    {"ugo+s", 0536, 0541, 06541},
    {"u+u", 0777, 0564, 0564},
    {"gu+u", 0275, 04756, 04776},
    {"g-wx", 0762, 02507, 02507},
    {"o-x", 0066, 0735, 0734},
    {"o+wX", 0154, 0623, 0623},
    {"o=o", 0223, 04135, 04135},
    {"a+s", 0031, 062, 06062},
    {"003", 0517, 0537, 03},
    {"gua+rxt", 0223, 0217, 01757},
    {"ug-w", 0715, 04755, 04555},
    {"u-ws", 0036, 0562, 0562},
    {"og-xw", 0143, 072, 040},
    {"ug+xt", 0753, 0750, 0750},
    {"ag+s", 0711, 0667, 06667},
    {"g+s", 0724, 035, 02035},
    {"uo+rs", 0270, 02, 04406},
    {"g+X", 0045, 0402, 0402},
    {"730", 0536, 0526, 0730},
    {"g+w", 0377, 0332, 0332},
    {"o-s", 0536, 04735, 04735},
    {"ao+x", 0033, 04, 0115},
    {"ga+r", 0425, 0645, 0645},
    {"ug-o", 0742, 00, 00},
    {"o=tx", 0061, 0724, 01721},
    {"ag-s", 0224, 0236, 0236},
    {"ou=sxw", 0004, 0231, 04333},
    {"o-t", 0704, 0216, 0216},
    {"0742", 0772, 0435, 0742},
    {"a+s=xr", 0224, 0713, 0555},
    {"oua-u", 0544, 044, 044},
    {"a+g", 0512, 0667, 0667},
    {"a-x+srt", 0302, 0206, 07646},
    {"gu-xX", 0270, 0506, 0406},
    {"002", 0474, 0354, 02},
    {"ga+r", 0347, 04756, 04756},
    {"g-rx", 0017, 0277, 0227},
    {"go=tw", 0217, 0426, 01422},
    {"og=rtw", 0446, 0575, 01566},
    {"u-rX", 0273, 0415, 015},
    {"361", 0623, 0265, 0361},
    {"ou-xw", 0207, 02457, 02454},
    {"g+s", 0026, 0324, 02324},
    {"uog-g", 0222, 0673, 00},
    {"u+w-r", 0720, 0711, 0311},
    {"u=ts", 0075, 02570, 06070},
    {"ug=stX", 0727, 02703, 06113},
    {"663", 0531, 044, 0663},
    {"go=x", 0343, 00, 011},
    {"617", 0665, 0465, 0617},
    {"gu+tXs", 0711, 0175, 06175},
    {"665", 0606, 0407, 0665},
    {"u+rx", 0460, 0465, 0565},
    {"go=rXx", 0433, 04, 055},
    {"u-w", 0337, 024, 024},
    {"a+sx", 0312, 024, 06135},
    {"ao-xs", 0065, 0400, 0400},
    {"u=g", 0464, 04607, 07},
    {"a=u", 0017, 0723, 0777},
    {"u=x", 0732, 0724, 0124},
    {"guo=X", 0650, 0522, 0111},
    {"au-g", 0400, 052, 02},
    {"361", 0443, 0774, 0361},
    {"og+rsx", 0144, 0433, 02477},
    {"o=tr", 0376, 0475, 01474},
    {"au=ts", 0067, 0234, 07000},
    {"ua=x", 0117, 053, 0111},
    {"go+u", 0077, 0174, 0175},
    {"ao+sx", 0165, 0600, 06711},
    {"a-x", 0373, 03, 02},
    {"oa-o", 0722, 0223, 00},
    {"a=X+sX", 0726, 0577, 06111},
    {"a=o", 0557, 0375, 0555},
    {"+001", 0344, 0204, 0205},
    {"g-Xx", 0144, 0627, 0627},
    {"o-t", 0223, 062, 062},
    {"oa=s", 0356, 0343, 06000},
    {"663", 0701, 04313, 0663},
    {"u=trw", 0631, 0451, 0651},
    {"g+w", 0110, 0516, 0536},
    {"ou+wsx", 0730, 0544, 04747},
    {"a+s", 0325, 0146, 06146},
    {"uo=ts", 0406, 0477, 05070},
    {"uo+w", 0403, 0713, 0713},
    {"g=w", 0775, 01331, 01321},
    {"go=xt", 0046, 01643, 01611},
    {"a=sX", 0742, 0220, 06000},
    {"uga-s", 0054, 0361, 0361},
    {"a=t", 0037, 0523, 01000},
    {"g=st", 0524, 04347, 06307},
    {"o=g", 0056, 0311, 0311},
    {"ao+st", 0017, 0642, 07642},
    {"a+o", 0224, 0231, 0331},
    {"ga=g", 0621, 0644, 0444},
    {"u-xt", 0207, 0345, 0245},
    {"o-X+Xs", 0501, 0633, 0633},
    {"u+t", 0342, 0456, 0456},
    {"uo=sxw", 0123, 053, 04353},
    {"a=rxX", 0107, 0775, 0555},
    {"gou+wt", 0102, 02114, 03336},
    {"oag=x", 0571, 0337, 0111},
    {"g+sw", 0351, 0307, 02327},
    {"u=tX", 0764, 057, 0157},
    {"ga+X", 0126, 0760, 0771},
    {"ug-w=tw", 0170, 0410, 0220},
    {"ou=tw", 0357, 0466, 01262},
    {"ou=wXt", 0406, 0351, 01353},
    {"424", 0606, 040, 0424},
    {"1512", 0215, 02264, 01512},
    {"au-wrx", 0350, 02, 00},
    {"uo=Xr", 0540, 0336, 0535},
    {"ug-w", 0030, 04723, 04503},
    {"ou=xsw", 0012, 0330, 04333},
    {"u-u", 0603, 0151, 051},
    {"uo=g", 0562, 00, 00},
    {"g=sr", 0325, 0436, 02446},
    {"uo-g", 0511, 0142, 0142},
    {"o+w", 0606, 0575, 0577},
    {"ogu+s", 0373, 02466, 06466},
    {"a=tr", 0672, 0615, 01444},
    {"uo-X", 0424, 02672, 02672},
    {"200", 0751, 0310, 0200},
    {"762", 0023, 0673, 0762},
    {"751", 0252, 0527, 0751},
    {"u=r", 0667, 0761, 0461},
    {"u-o", 0603, 0426, 026},
    {"gou=g", 0565, 0571, 0777},
    {"-240", 0066, 0157, 0117},
    {"a-X", 0134, 0144, 044},
    {"aou+u", 0737, 0605, 0667},
    {"oa=sr", 0432, 0631, 06444},
    {"uo=xX", 0723, 0124, 0121},
    {"uoa-r", 0620, 02111, 02111},
    {"g-X", 0533, 0226, 0226},
    {"722", 0610, 0200, 0722},
    {"au-g", 0050, 0334, 04},
    {"ua+X", 0112, 0155, 0155},
    {"a+o", 0141, 0515, 0555},
    {"g=rx", 0627, 0365, 0355},
    {"og=tw", 0557, 041, 01022},
    {"ag+rsx", 0254, 02, 06557},
    {"723", 0325, 0107, 0723},
    {"uo+X", 0103, 0535, 0535},
    {"ago=g", 0024, 020, 0222},
    {"u-w", 0327, 04074, 04074},
    {"g=Xx", 0651, 0743, 0713},
    {"uga+o", 0356, 0332, 0332},
    {"u-rs", 0736, 0451, 051},
    {"-712", 0610, 02104, 02004},
    {"o+tx", 0437, 0602, 01603},
    {"agu=st", 0217, 0626, 07000},
    {"ug=X", 0261, 0131, 0111},
    {"g+wXs", 0255, 0272, 02272},
    {"ga-o", 0637, 02521, 02420},
    {"g+sr", 0257, 070, 02070},
    {"uag+swx", 0725, 0236, 06337},
    {"o=X-stX", 0532, 01636, 0630},
    {"guo=tr", 0003, 0465, 01444},
    {"a-X", 0431, 0674, 0664},
    {"au-o", 0566, 0254, 0210},
    {"ao-u", 0227, 0373, 040},
    {"ou-g", 0424, 0320, 0120},
    {"o+rX", 0357, 05, 05},
    {"a-rwt", 0052, 07, 01},
    {"u+Xxs", 0436, 0211, 04311},
    {"aog+x", 0136, 0533, 0533},
    {"=571", 0250, 0661, 0571},
    {"ao=wxs", 0777, 04430, 06333},
    {"u-Xwt", 0145, 0130, 030},
    {"og=s", 0450, 0160, 02100},
    {"oug+xXr", 0747, 0575, 0575},
    {"goa+g", 0625, 0656, 0757},
    {"au+Xw", 0011, 01735, 01737},
    {"ag=g", 0110, 0327, 0222},
    {"a+o", 0607, 0661, 0771},
    {"u+sX", 0555, 020, 04020},
    {"g=s", 0725, 0674, 02604},
    {"+176", 0220, 0771, 0777},
    {"o+u", 0153, 0765, 0767},
    {"gau-xsr", 0536, 0543, 02},
    {"auo-tr", 0446, 0451, 011},
    {"g=srx", 0432, 0754, 02754},
    {"aou+o", 0232, 0352, 0372},
    {"773", 0660, 02231, 0773},
    {"ogu=u", 0007, 0265, 0222},
    {"oau+rs", 0247, 0470, 06474},
    {"236", 0227, 0645, 0236},
    {"ao+xw", 0766, 04734, 04737},
    {"ou=t", 0207, 0141, 01040},
    {"o=ws", 0435, 04707, 04702},
    {"og=rX", 0646, 0215, 0255},
    {"guo-Xs", 0534, 04051, 040},
    {"go+w", 0016, 0722, 0722},
    {"ao+wX", 0271, 0451, 0773},
    {"og=rts", 0612, 07, 03044},
    {"g-o", 0432, 0673, 0643},
    {"a-Xr", 0767, 0633, 0222},
    {"og-t", 0260, 0373, 0373},
    {"o=g", 0635, 0401, 0400},
    {"=540", 0060, 0466, 0540},
    {"aug-rX", 0502, 0227, 0222},
    {"1601", 0710, 0467, 01601},
    {"oa-wXt", 0673, 042, 040},
    {"o=ts", 0141, 0640, 01640},
    {"og+X", 0146, 050, 051},
    {"go+o", 0362, 040, 040},
    {"uo=o", 0211, 0117, 0717},
    {"au-Xxt", 0127, 04320, 04220},
    {"oa-Xw", 0542, 04141, 04040},
    {"ga-xr", 0541, 0224, 0220},
    {"a=wx", 0363, 04633, 0333},
    {"go+o", 0127, 0530, 0530},
    {"ga-Xx", 0435, 0766, 0666},
    {"oa=ws", 0772, 04, 06222},
    {"ao+xw", 0001, 02740, 02773},
    {"a-sw", 0124, 0637, 0415},
    {"g+u", 0104, 01274, 01274},
    {"gau+u", 0331, 0474, 0474},
    {"g-sx", 0224, 0416, 0406},
    {"ga=o", 0762, 075, 0555},
    {"o=s", 0213, 0110, 0110},
    {"uoa-x", 0043, 0126, 026},
    {"a-t+xt", 0504, 052, 01153},
    {"ug=sw", 0626, 0345, 06225},
    {"a-o", 0753, 067, 00},
    {"go=x", 0076, 016, 011},
    {"a=r", 0260, 0467, 0444},
    {"ua=rX", 0323, 0340, 0555},
    {"ag+rt", 0560, 0642, 01646},
    {"g=wxs", 0070, 0310, 02330},
    {"og+tx", 0376, 0646, 01657},
    {"oa=sr", 0665, 052, 06444},
    {"ga+rt", 0672, 04261, 05665},
    {"ug+u", 0114, 01746, 01776},
    {"ga-u", 0307, 01424, 01020},
    {"gao-wts", 0212, 0132, 0110},
    {"agu=g", 0160, 0755, 0555},
    {"u=sx", 0425, 0747, 04147},
    {"uo-t", 0345, 0576, 0576},
    {"u=Xr", 0524, 0722, 0522},
    {"o=g", 0550, 067, 066},
    {"ou+X=wr", 0142, 0700, 0606},
    {"o-xst", 0715, 0412, 0412},
    {"a=w", 0410, 0275, 0222},
    {"-731", 0745, 0113, 02},
    {"ao+g", 0144, 02347, 02747},
    {"ag-u", 0024, 0143, 042},
    {"u=s", 0624, 0536, 04036},
    {"uo+Xs", 0027, 0521, 04521},
    {"u+o", 0736, 0377, 0777},
    {"557", 0221, 0715, 0557},
    {"oa=w", 0422, 0114, 0222},
    {"g=x", 0656, 0564, 0514},
    {"a-wx", 0216, 0674, 0444},
    {"u-Xts", 0506, 0410, 0410},
    {"u=xs", 0427, 0533, 04133},
    {"oua=rs", 0025, 0742, 06444},
    {"=260", 0054, 0435, 0260},
    {"a-r", 0523, 0100, 0100},
    {"u-s", 0363, 0400, 0400},
    {"ua+s", 0115, 0644, 06644},
    {"u+xX", 0317, 0737, 0737},
    {"u+tw", 0455, 04262, 04262},
    {"ag+rw", 0110, 02442, 02666},
    {"a+tw", 0372, 0352, 01372},
    {"ao+tr", 0333, 0117, 01557},
    {"ou=wr", 0046, 02745, 02646},
    {"g-trx", 0550, 0447, 0407},
    {"777", 0507, 01602, 0777},
    {"ga+xt", 0442, 026, 01137},
    {"1506", 0147, 037, 01506},
    {"aou-tXw", 0466, 0750, 0440},
    {"a-X", 0713, 0567, 0466},
    {"gu=x+wX", 0155, 0741, 0331},
    {"au-s", 0475, 0724, 0724},
    {"uog-w", 0432, 033, 011},
    {"gu=rsw", 0502, 0264, 06664},
    {"au=u", 0654, 04673, 0666},
    {"ga-o", 0674, 0350, 0350},
    {"ga=ws", 0010, 0267, 06222},
    {"a-g", 0077, 060, 00},
    {"gu-x", 0243, 04475, 04465},
    {"o=tX", 0651, 0565, 01561},
    {"o-s", 0126, 017, 017},
    {"o-X", 0517, 0670, 0670},
    {"a=tX", 0054, 0313, 01111},
    {"o+rst", 0021, 0335, 01335},
    {"uo+xt", 0533, 04666, 05767},
    {"o-tx", 0515, 0335, 0334},
    {"uo-st", 0172, 0363, 0363},
    {"1037", 0447, 017, 01037},
    {"oga-rws", 0361, 0404, 00},
    {"ua=xX", 0065, 0615, 0111},
    {"oa+tr", 0220, 0676, 01676},
    {"u=xrX", 0651, 0250, 0550},
    {"u=ts", 0221, 01420, 05020},
    {"g+x", 0561, 02, 012},
    {"g-xst", 0567, 04, 04},
    {"o+X-t=t", 0247, 0143, 01140},
    {"g=w+r-sr", 0533, 0406, 0426},
    {"635", 0773, 0176, 0635},
    {"o=w", 0410, 01016, 012},
    {"744", 0456, 0331, 0744},
    {"au-tx", 0120, 0270, 0260},
    {"uag=t", 0467, 0331, 01000},
    {"g-xt", 0072, 01032, 01022},
    {"ogu-sx", 0352, 0116, 06},
    {"0552", 0743, 0422, 0552},
    {"u-x", 0547, 0747, 0647},
    {"uo-Xtw", 0432, 0542, 0440},
    {"a-g", 0225, 0334, 04},
    {"o=x", 0051, 0372, 0371},
    {"ga-o", 0374, 0616, 010},
    {"g-w", 0507, 0334, 0314},
    {"au=x-Xr", 0306, 0736, 00},
    {"gua+o", 0313, 0764, 0764},
    {"o-X", 0630, 042, 042},
    {"a=s", 0772, 0116, 06000},
    {"ua=rXx", 0260, 0115, 0555},
    {"u=rw", 0602, 0676, 0676},
    {"o=Xr", 0472, 00, 04},
    {"o-Xs", 0713, 04036, 04036},
    {"aog=s", 0444, 00, 06000},
    {"2507", 0206, 01676, 02507},
    {"go+rX", 0454, 0202, 0246},
    {"o-sx", 0636, 0540, 0540},
    {"og+wtX", 0521, 0715, 01737},
    {"=722", 0764, 0634, 0722},
    {"g=rt", 0337, 0615, 0645},
    {"101", 0007, 0735, 0101},
    {"0462", 0431, 0232, 0462},
    {"o+X", 0005, 0451, 0451},
    {"ga+sX", 0764, 0524, 06535},
    {"a+xX", 0451, 0363, 0373},
    {"go=X", 0354, 020, 00},
    {"ag=g", 0333, 07, 00},
    {"ou-s", 0063, 0501, 0501},
    {"a+w", 0705, 04711, 04733},
    {"auo+tr", 0715, 0236, 01676},
    {"ou+t", 0431, 0722, 01722},
    {"ao+g", 0341, 0275, 0777},
    {"ga+X", 0123, 04350, 04351},
    {"a=rxX", 0424, 0527, 0555},
    {"a+wts", 0172, 0667, 07667},
    {"gu-Xts", 0366, 04574, 0464},
    {"u+Xt", 0564, 0532, 0532},
    {"gu+X", 0225, 0666, 0666},
    {"g=txw", 0225, 04252, 04232},
    {"uo+X", 0575, 0241, 0341},
    {"ua-X", 0062, 0531, 0420},
    {"u+rs", 0350, 0245, 04645},
    {"o=Xst", 0503, 0610, 01611},
    {"146", 0463, 0523, 0146},
    {"ao+Xs", 0114, 0743, 06753},
    {"gu+sx", 0045, 04340, 06350},
    {"agu+wXs", 0706, 02321, 06333},
    {"ga+tws", 0423, 0331, 07333},
    {"guo+s", 0303, 0661, 06661},
    {"go+xs", 0171, 0726, 02737},
    {"715", 0755, 0110, 0715},
    {"u=u", 0416, 0600, 0600},
    {"272", 0410, 0626, 0272},
    {"o-rt", 0557, 0456, 0452},
    {"g+xs", 0473, 0235, 02235},
    {"uoa=Xrx", 0020, 0657, 0555},
    {"=372", 0106, 0764, 0372},
    {"og=r-swx", 0174, 0201, 0244},
    {"522", 0405, 0645, 0522},
    {"a-x", 0437, 0740, 0640},
    {"a=o", 0513, 0770, 00},
    {"oug=str", 0600, 0557, 07444},
    {"u-w", 0674, 0642, 0442},
    {"gu-t", 0117, 0151, 0151},
    {"uag-xs", 0721, 0423, 0422},
    {"u+x", 0405, 02605, 02705},
    {"u-r", 0667, 04021, 04021},
    {"ou=u", 0736, 0275, 0272},
    {"gu=w", 0223, 0443, 0223},
    {"ag-stw", 0634, 0476, 0454},
    {"u+sX", 0572, 0130, 04130},
    {"ag-s+s", 0731, 0104, 06104},
    {"ag=tr", 0216, 02072, 01444},
    {"a+w", 0616, 0222, 0222},
    {"ag-u", 0554, 0514, 00},
    {"ug-Xs", 0334, 0354, 0244},
    {"o=t+s-r", 0532, 06, 01000},
    {"ou-o", 0206, 0522, 0520},
    {"og=r", 0720, 0464, 0444},
    {"au=sw", 0525, 0355, 06222},
    {"a+sw", 0620, 02344, 06366},
    {"og=sX", 0667, 0724, 02711},
    {"o+wX", 0334, 0777, 0777},
    {"u+u", 0436, 01732, 01732},
    {"a-g", 0565, 0152, 02},
    {"g+t", 0733, 01571, 01571},
    {"g-u", 0433, 0207, 0207},
    {"ga-o", 0375, 0702, 0500},
    {"o=X", 0320, 0767, 0761},
    {"o+X", 0343, 0100, 0101},
    {"ou-s", 0041, 0772, 0772},
    {"og=rt", 0221, 04545, 05544},
    {"g+g", 0570, 0476, 0476},
    {"u+x", 0533, 0204, 0304},
    {"o-r", 0526, 0142, 0142},
    {"o-Xw", 0307, 0122, 0120},
    {"o=t", 0055, 0620, 01620},
    {"ag=t", 0352, 0252, 01000},
    {"oa=st", 0550, 024, 07000},
    {"ug-w+x", 0443, 0407, 0517},
    {"oa=sw", 0445, 0742, 06222},
    {"ao-s", 0056, 0363, 0363},
    {"g+wrt", 0100, 0157, 0177},
    {"o-o", 0203, 04654, 04650},
    {"ag+g", 0751, 06, 06},
    {"ua=X", 0514, 0412, 0111},
    {"g-sr", 0145, 0242, 0202},
    {"ou=Xt", 0407, 0774, 01171},
    {"ua=xr", 0717, 060, 0555},
    {"o=t", 0447, 0755, 01750},
    {"016", 0511, 0376, 016},
    {"1535", 0115, 01236, 01535},
    {"o-wtx", 0452, 0175, 0174},
    {"g+xt", 0055, 013, 013},
    {"g=u", 0662, 00, 00},
    {"1134", 0445, 0405, 01134},
    {"ug-X", 0374, 0451, 0441},
    {"ag+xsX", 0142, 0304, 06315},
    {"a-t", 0406, 0201, 0201},
    {"o+rx", 0442, 0415, 0415},
    {"o=g", 0643, 0244, 0244},
    {"gua=rxs", 0437, 0505, 06555},
    {"ga=t-r", 0466, 0720, 01000},
    {"og-wxt", 0236, 0375, 0344},
    {"-565", 0612, 0770, 0210},
    {"236", 0607, 0346, 0236},
    {"a+Xxr", 0527, 0715, 0755},
    {"o-x", 0201, 0522, 0522},
    {"ga-Xr", 0164, 0400, 00},
    {"u+xr", 0644, 012, 0512},
    {"g-g", 0576, 0644, 0604},
    {"a=xt", 0702, 0217, 01111},
    {"uo-w", 0250, 0455, 0455},
    {"u-X", 0515, 0667, 0667},
    {"o+r=w", 0215, 0311, 0312},
    {"goa=u", 0112, 0767, 0777},
    {"a=X", 0671, 0645, 0111},
    {"4412", 0453, 0443, 04412},
    {"g+sw", 0501, 0430, 02430},
    {"oa-X", 0655, 036, 026},
    {"ua+xr", 0551, 00, 0555},
    {"601", 0630, 0703, 0601},
    {"go-r", 0225, 01663, 01623},
    {"uo-tx", 0100, 06, 06},
    {"g-s+ws", 0322, 0200, 02220},
    {"o+Xw", 0457, 0260, 0262},
    {"a-txs", 0655, 0322, 0222},
    {"og+tx", 0020, 04100, 05111},
    {"o=twX", 0316, 0222, 01222},
    {"a+o", 0644, 0375, 0775},
    {"oa=trs", 0173, 0136, 07444},
    {"og=ts", 0012, 0374, 03300},
    {"ua+wX", 0472, 04472, 04773},
    {"go=s-X", 0057, 0107, 02100},
    {"ug+u", 0227, 0610, 0670},
    {"u+X", 0440, 0610, 0710},
    {"gau+rw", 0746, 020, 0666},
    {"oa-Xw", 0550, 02473, 02440},
    {"ua=x", 0662, 0442, 0111},
    {"u=wx", 0375, 02646, 02346},
    {"ou-st", 0314, 02, 02},
    {"g=st", 0306, 0456, 02406},
    {"g-sr", 0423, 0330, 0330},
    {"oga+X", 0221, 0104, 0115},
    {"ou+u", 0762, 0510, 0515},
    {"-126", 0221, 0567, 0441},
    {"ug+s", 0470, 0756, 06756},
    {"o+sr", 0347, 0400, 0404},
    {"ou+r", 0203, 0460, 0464},
    {"go+tr", 0625, 04606, 05646},
    {"au=rx", 0106, 0121, 0555},
    {"313", 0311, 0613, 0313},
    {"ag+Xs", 0373, 0774, 06775},
    {"ou-t", 0660, 0645, 0645},
    {"ou=w", 0266, 02023, 02222},
    {"o-rx", 0241, 04234, 04230},
    {"a+rX", 0721, 06, 0446},
    {"ga=rx", 0540, 0363, 0555},
    {"uo=w", 0204, 0717, 0212},
    {"gu-w", 0303, 0410, 0410},
    {"og-g", 0357, 01776, 01700},
    {"2257", 0235, 040, 02257},
    {"oua-wx", 0066, 0371, 040},
    {"a+g", 0026, 0540, 0544},
    {"a=xs", 0424, 040, 06111},
    {"ao-w", 0541, 0745, 0545},
    {"ua=t", 0670, 0436, 01000},
    {"1310", 0342, 0602, 01310},
    {"163", 0247, 04625, 0163},
    {"ogu-o", 0423, 0124, 0120},
    {"o+X", 0004, 0673, 0673},
    {"ago+rs", 0574, 0200, 06644},
    {"g-sr", 0703, 0705, 0705},
    {"gu=X-X", 0326, 0170, 00},
    {"g+Xw", 0015, 0514, 0534},
    {"o-o", 0275, 02416, 02410},
    {"o+s", 0323, 04064, 04064},
    {"o+g", 0757, 0517, 0517},
    {"ua-tX", 0163, 0666, 0666},
    {"uo+st", 0642, 0423, 05423},
    {"oa-rX", 0541, 04176, 04022},
    {"o=o", 0307, 0453, 0453},
    {"u=tx", 0770, 02322, 02122},
    {"auo=x-wsx", 0557, 0605, 00},
    {"u+w", 0123, 0371, 0371},
    {"a=g", 0507, 0135, 0333},
    {"511", 0527, 02356, 0511},
    {"aog=u", 0242, 0703, 0777},
    {"o+r-Xwr", 0627, 067, 060},
    {"ua=r", 0711, 0225, 0444},
    {"gu=Xrw", 0710, 0476, 0776},
    {"uo-w", 0367, 0645, 0445},
    {"ag+x=rsw", 0627, 02366, 06666},
    {"u=trx", 0234, 0144, 0544},
    {"a-rt", 0555, 0377, 0333},
    {"352", 0435, 02, 0352},
    {"u=sx", 0156, 04502, 04102},
    {"uog=Xt", 0717, 0326, 01111},
    {"aug-x", 0156, 0536, 0426},
    {"uog=o", 0161, 026, 0666},
    {"uao+g", 0367, 0516, 0517},
    {"ag-t", 0465, 0664, 0664},
    {"o-X", 0141, 0300, 0300},
    {"ag-u", 0347, 0742, 00},
    {"au+s-tx", 0536, 0451, 06440},
    {"632", 0270, 030, 0632},
    {"go+s", 0656, 055, 02055},
    {"uo=rt", 0646, 0532, 01434},
    {"ga=o", 0221, 0137, 0777},
    {"gu=w", 0052, 02350, 0220},
    {"1372", 0371, 02742, 01372},
    {"og-xw", 0122, 0674, 0644},
    {"a+twx", 0415, 0101, 01333},
    {"au-s=x", 0326, 0464, 0111},
    {"go-tw", 0523, 02265, 02245},
    {"ou+xtw", 0745, 0250, 01353},
    {"o=r", 0517, 0435, 0434},
    {"go=Xw", 0105, 033, 033},
    {"ag-g", 0047, 0100, 0100},
    {"og+g", 0355, 0664, 0666},
    {"uo+r", 0730, 0532, 0536},
    {"g-tw", 0535, 02612, 02612},
    {"o+wx", 0775, 0525, 0527},
    {"o+rxt", 0344, 0577, 01577},
    {"635", 0437, 0422, 0635},
    {"g+Xtx", 0425, 0426, 0436},
    {"o+wX", 0505, 0562, 0563},
    {"u-rxs", 0466, 0213, 0213},
    {"a+X", 0625, 0357, 0357},
    {"ag-tx", 0114, 0222, 0222},
    {"o-xrX", 0342, 0317, 0312},
    {"ou=wtx", 0703, 036, 01333},
    {"g=g", 0016, 01747, 01747},
    {"o+o", 0514, 0345, 0345},
    {"4423", 0073, 0474, 04423},
    {"g+rxX", 0511, 0437, 0477},
    {"o-wrs", 0136, 0240, 0240},
    {"ua-o", 0073, 057, 00},
    {"ua=wX", 0037, 0320, 0333},
    {"au=t", 0241, 04663, 01000},
    {"auo-x", 0101, 034, 024},
    {"g=wr", 0413, 0422, 0462},
    {"g=rXs", 0766, 0330, 02350},
    {"gua=t", 0710, 04054, 01000},
    {"og=xr", 0536, 0470, 0455},
    {"gu+xs", 0053, 0126, 06136},
    {"uo-Xw", 0602, 04322, 04020},
    {"ag+sXw", 0727, 0644, 06666},
    {"au=xw", 0560, 027, 0333},
    {"uao=s+t", 0034, 0402, 07000},
    {"uo+X", 0012, 0664, 0664},
    {"oau-w", 0161, 04154, 04154},
    {"og+w", 0664, 077, 077},
    {"gu-g", 0463, 0540, 0100},
    {"a=s", 0315, 0572, 06000},
    {"o-X", 0666, 00, 00},
    {"aou=sw", 0653, 0240, 06222},
    {"a-o", 0061, 0176, 0110},
    {"g-s", 0625, 0566, 0566},
    {"ug=rtX", 0326, 02547, 0557},
    {"og-wt", 0260, 040, 040},
    {"u-ws", 0623, 0474, 0474},
    {"u+ws", 0263, 0420, 04620},
    {"gao=rs", 0746, 0733, 06444},
    {"230", 0731, 04474, 0230},
    {"ou+X", 0746, 0431, 0531},
    {"g=xw", 0721, 0706, 0736},
    {"a-x", 0610, 0260, 0260},
    {"g-Xtx", 0756, 0650, 0640},
    {"u+xr", 0545, 0145, 0545},
    {"ao=Xr", 0064, 0106, 0555},
    {"u=w-w", 0605, 022, 022},
    {"g=Xwx", 0120, 01775, 01735},
    {"271", 0030, 0512, 0271},
    {"a-Xr", 0403, 036, 022},
    {"go+xX", 0470, 0501, 0511},
    {"uo+rt", 0225, 05, 01405},
    {"a+tr", 0022, 0657, 01657},
    {"o+x", 0627, 01253, 01253},
    {"o=o", 0104, 076, 076},
    {"gua+sr", 0345, 0420, 06464},
    {"agu=xrw", 0006, 0720, 0777},
    {"go-rx", 0106, 012, 02},
    {"ug=s", 0611, 076, 06006},
    {"a=s", 0324, 0604, 06000},
    {"go+Xr", 0152, 0313, 0357},
    {"go=t", 0335, 02, 01000},
    {"o-Xt", 0313, 0510, 0510},
    {"ag+xrw", 0713, 04152, 04777},
    {"a-txX", 0063, 017, 06},
    {"u-tx", 0422, 0412, 0412},
    {"u=ts", 0747, 044, 04044},
    {"ogu=xX", 0215, 0263, 0111},
    {"612", 0514, 0677, 0612},
    {"ag=sr", 0701, 0323, 06444},
    {"o+g", 0425, 0603, 0603},
    {"o+u", 0252, 0400, 0404},
    {"ou=sw", 0053, 0265, 04262},
    {"u-s", 0724, 063, 063},
    {"gu+w", 0001, 04727, 04727},
    {"ug-g", 0264, 01, 01},
    {"g+ws", 0467, 0625, 02625},
    {"u+sX", 0562, 04235, 04335},
    {"a-trx", 0353, 0263, 0222},
    {"guo=rX", 0314, 0242, 0444},
    {"gu-o", 0212, 0216, 016},
    {"oug+o", 0005, 0210, 0210},
    {"og-rw", 0646, 0350, 0310},
    {"ou+rXs", 0252, 0540, 04545},
    {"a=sr", 0725, 04675, 06444},
    {"o-x", 0273, 0124, 0124},
    {"ou-o", 0477, 0140, 0140},
    {"g=X", 0155, 0420, 0400},
    {"g=ws", 0776, 0742, 02722},
    {"g-x", 0276, 01045, 01045},
    {"og=sr", 0032, 0201, 02244},
    {"g+xX", 0656, 021, 031},
    {"0552", 0435, 035, 0552},
    {"a-w+x", 0073, 0567, 0555},
    {"a+r", 0071, 02317, 02757},
    {"ag=o", 0553, 02000, 00},
    {"u=g", 0563, 0473, 0773},
    {"agu-xX", 0052, 0605, 0604},
    {"goa-xt", 0720, 0756, 0646},
    {"a-u", 0754, 04226, 04004},
    {"1571", 0502, 0447, 01571},
    {"ao-swr", 0714, 01161, 01101},
    {"aou-xw", 0167, 0735, 0404},
    {"770", 0504, 0367, 0770},
    {"gu=wXt", 0154, 0370, 0330},
    {"og+X=x", 0403, 0563, 0511},
    {"u=t", 0016, 00, 00},
    {"og=tsw", 0414, 02111, 03122},
    {"uo=r", 0351, 0476, 0474},
    {"ga=sXx", 0157, 010, 06111},
    {"a+w=wx", 0475, 0212, 0333},
    {"o+t-rX", 0431, 015, 01010},
    {"oa+tXr", 0652, 0546, 01557},
    {"u+xw", 0071, 04746, 04746},
    {"og-u", 0237, 0771, 0700},
    {"au+wx", 0307, 0102, 0333},
    {"ua=x-tr", 0222, 0655, 0111},
    {"=245", 0260, 0315, 0245},
    {"ga=w", 0741, 0264, 0222},
    {"g=g", 0761, 0223, 0223},
    {"g+srw", 0336, 0343, 02363},
    {"u=sX", 0702, 040, 04040},
    {"o=xs", 0776, 0556, 0551},
    {"g=tsr", 0043, 041, 02041},
    {"ug-r", 0210, 046, 06},
    {"+171", 0777, 0772, 0773},
    {"u=xtw", 0140, 01624, 01324},
    {"aug=u", 0225, 0333, 0333},
    {"140", 0441, 04004, 0140},
    {"go-o", 0402, 00, 00},
    {"g+s", 0564, 037, 02037},
    {"a-srX", 0327, 04520, 020},
    {"513", 0636, 0553, 0513},
    {"uo-X=x", 0274, 0162, 0161},
    {"g-Xs", 0641, 0653, 0643},
    {"u=x-X", 0474, 0771, 071},
    {"o+X", 0374, 0417, 0417},
    {"a+wr", 0243, 0620, 0666},
    {"gu+x=x", 0376, 0737, 0117},
    {"a-rt", 0441, 0565, 0121},
    {"u+sr", 0720, 0545, 04545},
    {"o-xr", 0561, 04737, 04732},
    {"uo=s", 0171, 010, 04010},
    {"og+rtX", 0432, 0365, 01375},
    {"g+w", 0756, 0265, 0265},
    {"ua+xsr", 0124, 0400, 06555},
    {"u-r=sr", 0060, 04143, 04443},
    {"g-o", 0653, 046, 06},
    {"uo-Xxw", 0776, 0203, 00},
    {"u+rws", 0537, 01156, 05756},
    {"u-X", 0337, 0553, 0453},
    {"g+Xtw", 0167, 061, 071},
    {"573", 0720, 0271, 0573},
    {"ag+o", 0341, 0773, 0773},
    {"uoa=u", 0262, 047, 00},
    {"ug-Xs", 0002, 0555, 0445},
    {"g+tw", 0171, 01446, 01466},
    {"gu=g", 0315, 0401, 01},
    {"u=xt", 0263, 0662, 0162},
    {"g+X", 0574, 0376, 0376},
    {"uoa-r", 0464, 0163, 0123},
    {"u+X", 0160, 01450, 01550},
    {"uoa=sx", 0430, 0151, 06111},
    {"a-o", 0222, 0321, 0220},
    {"ug-t", 0014, 01166, 01166},
    {"571", 0611, 0444, 0571},
    {"474", 0301, 0153, 0474},
    {"o=x-X", 0164, 0576, 0570},
    {"g+w", 0047, 01324, 01324},
    {"644", 0207, 0502, 0644},
    {"g+w=x", 0275, 0747, 0717},
    {"ua+g", 0772, 0511, 0511},
    {"ua=w", 0677, 0702, 0222},
    {"o-wxt", 0743, 0763, 0760},
    {"uao-rx", 0644, 0332, 0222},
    {"o-Xw", 0144, 02100, 02100},
    {"ao-s=rw", 0215, 0422, 0666},
    {"ou=g", 0101, 04606, 00},
    {"ga=rsw", 0000, 0277, 06666},
    {"a=X-s", 0154, 0563, 0111},
    {"a-r", 0700, 0164, 0120},
    {"a-r", 0543, 020, 020},
    {"go+g", 0247, 01222, 01222},
    {"agu+Xs", 0272, 0776, 06777},
    {"ou+o", 0212, 0367, 0767},
    {"g+srx", 0001, 0727, 02777},
    {"og=tr", 0672, 02, 01044},
    {"ao=Xt", 0632, 0415, 01111},
    {"g-sx", 0446, 0724, 0724},
    {"uo+Xxw", 0706, 0763, 0763},
    {"g+wXt", 0115, 0323, 0333},
    {"oag+x", 0061, 02637, 02737},
    {"uog=Xrt", 0347, 0703, 01555},
    {"g=rw", 0620, 01174, 01164},
    {"go+srw", 0203, 0656, 02676},
    {"g=Xr", 0765, 0737, 0757},
    {"uo-rw", 0101, 0420, 020},
    {"ugo-xr", 0225, 0666, 0222},
    {"ago-st", 0700, 0520, 0520},
    {"2727", 0716, 04346, 02727},
    {"2412", 0004, 0554, 02412},
    {"o=s", 0215, 017, 010},
    {"og=t", 0370, 01776, 01700},
    {"u+xs", 0045, 02727, 06727},
    {"au=wx", 0103, 0267, 0333},
    {"g+Xx", 0106, 0172, 0172},
    {"o-rs", 0143, 0764, 0760},
    {"au-u", 0555, 0153, 042},
    {"uao=xt", 0136, 0564, 01111},
    {"u+rxw", 0126, 0645, 0745},
    {"oa=xw", 0471, 0450, 0333},
    {"og=xwt", 0253, 0126, 01133},
    {"ao=X", 0654, 0425, 0111},
    {"a+w=Xxw", 0571, 01132, 0333},
    {"-601", 0115, 010, 010},
    {"go=x", 0760, 042, 011},
    {"a+tw", 0173, 0146, 01366},
    {"gu-rw", 0252, 0724, 0104},
    {"og=ts", 0221, 012, 03000},
    {"uo-g", 0117, 0120, 0120},
    {"goa=u", 0532, 064, 00},
    {"uo+g", 0673, 04031, 04333},
    {"au-u", 0163, 0536, 022},
    {"ag=t", 0655, 0152, 01000},
    {"ua-s", 0757, 02725, 0725},
    {"o=Xx", 0122, 053, 051},
    {"ou=xs", 0607, 0220, 04121},
    {"=0752", 0427, 0566, 0752},
    {"uag+o", 0033, 0740, 0740},
    {"gou-X=tx", 0355, 0367, 01111},
    {"u+tr", 0350, 053, 0453},
    {"oa+r-x", 0026, 02201, 02644},
    {"u-Xt", 0007, 0673, 0673},
    {"ou+o", 0433, 0226, 0626},
    {"oug+t", 0434, 0363, 01363},
    {"uog=Xr", 0765, 0210, 0555},
    {"ou+u", 0117, 0256, 0256},
    {"ag+xrX", 0067, 00, 0555},
    {"ga+g", 0651, 0433, 0733},
    {"oa+w", 0150, 030, 0232},
    {"uoa+u", 0201, 0567, 0577},
    {"g=Xt", 0424, 0212, 0212},
    {"g+X", 0166, 0606, 0606},
    {"gu+r", 0700, 0406, 0446},
    {"g-wt", 0726, 04631, 04611},
    {"ag-t", 0106, 0262, 0262},
    {"g+w", 0346, 0606, 0626},
    {"oa=sr", 0751, 0576, 06444},
    {"057", 0410, 0376, 057},
    {"g-sw", 0266, 0545, 0545},
    {"og-u", 0264, 0314, 0304},
    {"736", 0743, 0665, 0736},
    {"260", 0426, 02557, 0260},
    {"ag-xs", 0473, 0101, 00},
    {"517", 0111, 0762, 0517},
    {"a-rs", 0626, 0560, 0120},
    {"ugo=x+Xwt", 0451, 00, 01333},
    {"a=X", 0237, 0533, 0111},
    {"gu+x", 0775, 032, 0132},
    {"137", 0544, 0233, 0137},
    {"uog+s-w", 0723, 0642, 06440},
    {"ua-r", 0017, 0163, 0123},
    {"go-rt", 0414, 0425, 0421},
    {"aou-g", 0153, 0604, 0604},
    {"g=Xt", 0526, 056, 016},
    {"o-wXx", 0734, 0506, 0504},
    {"a=X", 0774, 0502, 0111},
    {"uo=ws", 0131, 0374, 04272},
    {"oa=o", 0044, 0736, 0666},
    {"g-Xt", 0751, 0704, 0704},
    {"a=X+t", 0607, 04560, 01111},
    {"oa+xrX", 0112, 0155, 0555},
    {"ua-twr", 0276, 01362, 0100},
    {"uag-u", 0145, 0152, 042},
    {"uo+Xs", 0071, 047, 04147},
    {"g-wt", 0255, 0423, 0403},
    {"og-rw", 0246, 0270, 0210},
    {"ua-u", 0730, 0142, 042},
    {"a=rw", 0130, 0600, 0666},
    {"o-rx", 0631, 0377, 0372},
    {"au+r", 0414, 042, 0446},
    {"o+rx", 0341, 052, 057},
    {"g+u", 0321, 0777, 0777},
    {"u+srX", 0647, 01715, 05715},
    {"g+rwt", 0607, 0206, 0266},
    {"217", 0124, 0207, 0217},
    {"414", 0737, 0455, 0414},
    {"oa+o", 0741, 043, 0373},
    {"g-t", 0230, 02716, 02716},
    {"gu+x", 0360, 0247, 0357},
    {"uo-rt", 0772, 0544, 0140},
}};
// clang-format on

} // namespace

TEST(chmod_transformer, random) {
  for (auto const& [i, t] : ranges::views::enumerate(random_tests)) {
    internal::chmod_transformer ct{t.spec, t.umask};
    EXPECT_EQ_MODE(ct.transform(t.mode, false), t.expected)
        << "test " << i << ": " << t.spec << ", " << t.umask << ", " << t.mode
        << ", " << t.expected;
  }
}
