/* vim:set ts=2 sw=2 sts=2 et: */
/**
 * \author     Marcus Holland-Moritz (github@mhxnet.de)
 * \copyright  Copyright (c) Marcus Holland-Moritz
 *
 * This file is part of ricepp.
 *
 * ricepp is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * ricepp is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ricepp.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <array>
#include <cstdint>
#include <deque>
#include <iterator>
#include <list>
#include <random>
#include <stdexcept>

#include <gmock/gmock.h>
#include <gtest/gtest.h>

#include <ricepp/bitstream_reader.h>
#include <ricepp/bitstream_writer.h>

TEST(ricepp, bitstream_basic_test) {
  std::vector<uint8_t> data;
  {
    ricepp::bitstream_writer writer(std::back_inserter(data));
    writer.write_bits(0x1234567U, 28);
    writer.write_bits(UINT64_C(0x123456789abcdef), 60);
    writer.write_bit(false, 13);
    writer.write_bit(true);
    writer.write_bit(true, 200);
    writer.write_bit(false, 200);
    writer.write_bit(true);
    writer.write_bit(false, 136);
    writer.write_bit(true);
    writer.flush();
  }

  static_assert(std::contiguous_iterator<decltype(data)::iterator>);

  {
    ricepp::bitstream_reader reader(data.begin(), data.end());
    EXPECT_EQ(0x1234567U, reader.read_bits<uint32_t>(28));
    EXPECT_EQ(UINT64_C(0x123456789abcdef), reader.read_bits<uint64_t>(60));
    EXPECT_EQ(13, reader.find_first_set());
    EXPECT_EQ(UINT64_C(0x000000FFFFFFFFFF), reader.read_bits<uint64_t>(40));
    EXPECT_EQ(UINT64_C(0x000000FFFFFFFFFF), reader.read_bits<uint64_t>(40));
    EXPECT_EQ(UINT64_C(0x000000FFFFFFFFFF), reader.read_bits<uint64_t>(40));
    EXPECT_EQ(UINT64_C(0x000000FFFFFFFFFF), reader.read_bits<uint64_t>(40));
    EXPECT_EQ(UINT64_C(0x000000FFFFFFFFFF), reader.read_bits<uint64_t>(40));
    EXPECT_EQ(200, reader.find_first_set());
    EXPECT_EQ(136, reader.find_first_set());

    EXPECT_THAT([&] { auto v = reader.read_bits<uint64_t>(10); },
                testing::Throws<std::out_of_range>());
  }

  std::list<uint8_t> data2{data.begin(), data.end()};
  static_assert(!std::contiguous_iterator<decltype(data2)::iterator>);

  {
    ricepp::bitstream_reader reader(data2.begin(), data2.end());
    EXPECT_EQ(0x1234567U, reader.read_bits<uint32_t>(28));
    EXPECT_EQ(UINT64_C(0x123456789abcdef), reader.read_bits<uint64_t>(60));
    EXPECT_EQ(13, reader.find_first_set());
    EXPECT_EQ(UINT64_C(0x000000FFFFFFFFFF), reader.read_bits<uint64_t>(40));
    EXPECT_EQ(UINT64_C(0x000000FFFFFFFFFF), reader.read_bits<uint64_t>(40));
    EXPECT_EQ(UINT64_C(0x000000FFFFFFFFFF), reader.read_bits<uint64_t>(40));
    EXPECT_EQ(UINT64_C(0x000000FFFFFFFFFF), reader.read_bits<uint64_t>(40));
    EXPECT_EQ(UINT64_C(0x000000FFFFFFFFFF), reader.read_bits<uint64_t>(40));
    EXPECT_EQ(200, reader.find_first_set());
    EXPECT_EQ(136, reader.find_first_set());

    EXPECT_THAT([&] { auto v = reader.read_bits<uint64_t>(10); },
                testing::Throws<std::out_of_range>());
  }
}

namespace {

enum class oper {
  single,
  sequence,
  multi,
};

struct opdata {
  oper op;
  size_t bits;
  uint64_t value;
};

constexpr std::array<opdata, 1000> const testdata{{
    {oper::sequence, 35, 0},
    {oper::multi, 49, UINT64_C(0x191cdb21376eb)},
    {oper::multi, 60, UINT64_C(0x4acb8211fbbfae7)},
    {oper::multi, 12, UINT64_C(0x52c)},
    {oper::multi, 24, UINT64_C(0xddc087)},
    {oper::multi, 63, UINT64_C(0x980f711499f16e1)},
    {oper::multi, 24, UINT64_C(0x2f96fc)},
    {oper::single, 0, 1},
    {oper::multi, 46, UINT64_C(0x1e4f1b150d6f)},
    {oper::multi, 33, UINT64_C(0x3b1411e4)},
    {oper::multi, 14, UINT64_C(0x396)},
    {oper::multi, 10, UINT64_C(0x394)},
    {oper::single, 0, 0},
    {oper::multi, 29, UINT64_C(0x4154151)},
    {oper::multi, 60, UINT64_C(0xb5fd3ebbb52080a)},
    {oper::multi, 39, UINT64_C(0x6a7c6189ab)},
    {oper::multi, 60, UINT64_C(0x6e0fdb65f6556c2)},
    {oper::multi, 15, UINT64_C(0x10c1)},
    {oper::sequence, 48, 0},
    {oper::multi, 30, UINT64_C(0x110cee74)},
    {oper::multi, 1, UINT64_C(0x0)},
    {oper::single, 0, 1},
    {oper::multi, 32, UINT64_C(0x6324e42)},
    {oper::multi, 10, UINT64_C(0x332)},
    {oper::sequence, 9, 0},
    {oper::single, 0, 1},
    {oper::single, 0, 1},
    {oper::single, 0, 1},
    {oper::multi, 60, UINT64_C(0xa3282b479c6d269)},
    {oper::multi, 9, UINT64_C(0x11a)},
    {oper::sequence, 32, 0},
    {oper::multi, 59, UINT64_C(0x66d43ce33679888)},
    {oper::multi, 58, UINT64_C(0xb97a2044eae797)},
    {oper::sequence, 94, 0},
    {oper::multi, 32, UINT64_C(0xbf216187)},
    {oper::sequence, 25, 0},
    {oper::multi, 56, UINT64_C(0x841cc9b00f4b29)},
    {oper::multi, 21, UINT64_C(0x59ae0)},
    {oper::multi, 35, UINT64_C(0x5a74161c9)},
    {oper::sequence, 62, 0},
    {oper::multi, 23, UINT64_C(0x22e195)},
    {oper::multi, 12, UINT64_C(0xd6b)},
    {oper::multi, 49, UINT64_C(0x1398c67b8b58b)},
    {oper::sequence, 25, 0},
    {oper::multi, 48, UINT64_C(0xbb9592f60dd3)},
    {oper::single, 0, 1},
    {oper::multi, 56, UINT64_C(0x2c3f5cefc6725e)},
    {oper::single, 0, 1},
    {oper::multi, 56, UINT64_C(0x349b52d18e5c62)},
    {oper::sequence, 39, 0},
    {oper::single, 0, 1},
    {oper::sequence, 82, 0},
    {oper::sequence, 8, 0},
    {oper::multi, 47, UINT64_C(0x8d44c830c7f)},
    {oper::multi, 60, UINT64_C(0xad10bc379138423)},
    {oper::multi, 1, UINT64_C(0x1)},
    {oper::multi, 11, UINT64_C(0x1d0)},
    {oper::multi, 21, UINT64_C(0x15ea8b)},
    {oper::single, 0, 1},
    {oper::single, 0, 1},
    {oper::multi, 41, UINT64_C(0x996e592876)},
    {oper::multi, 54, UINT64_C(0x23b192ee83a037)},
    {oper::sequence, 70, 0},
    {oper::single, 0, 1},
    {oper::multi, 44, UINT64_C(0x17c63f449ab)},
    {oper::multi, 47, UINT64_C(0x41ee16d67e02)},
    {oper::multi, 54, UINT64_C(0x33b7be69fa8efd)},
    {oper::multi, 7, UINT64_C(0x3a)},
    {oper::multi, 57, UINT64_C(0x4eef858776e69a)},
    {oper::multi, 49, UINT64_C(0x1dcc8ca3a1147)},
    {oper::single, 0, 1},
    {oper::multi, 53, UINT64_C(0x13934c81051bb0)},
    {oper::multi, 51, UINT64_C(0x46b533890619a)},
    {oper::multi, 32, UINT64_C(0x43679cc9)},
    {oper::single, 0, 0},
    {oper::single, 0, 1},
    {oper::sequence, 49, 0},
    {oper::multi, 4, UINT64_C(0xa)},
    {oper::single, 0, 0},
    {oper::multi, 7, UINT64_C(0x40)},
    {oper::sequence, 44, 0},
    {oper::sequence, 39, 0},
    {oper::multi, 64, UINT64_C(0x7cd8900ab162f946)},
    {oper::single, 0, 0},
    {oper::single, 0, 0},
    {oper::multi, 37, UINT64_C(0xabe190150)},
    {oper::single, 0, 0},
    {oper::sequence, 48, 0},
    {oper::single, 0, 1},
    {oper::multi, 2, UINT64_C(0x3)},
    {oper::multi, 62, UINT64_C(0x12045a0af31e2423)},
    {oper::multi, 47, UINT64_C(0x93ace80ef1d)},
    {oper::multi, 29, UINT64_C(0xca9f68d)},
    {oper::multi, 10, UINT64_C(0x212)},
    {oper::multi, 9, UINT64_C(0x7e)},
    {oper::multi, 7, UINT64_C(0x59)},
    {oper::single, 0, 0},
    {oper::multi, 12, UINT64_C(0x650)},
    {oper::multi, 9, UINT64_C(0x114)},
    {oper::sequence, 98, 0},
    {oper::multi, 24, UINT64_C(0x53dde8)},
    {oper::multi, 45, UINT64_C(0x1764527e0df)},
    {oper::multi, 31, UINT64_C(0xba3843a)},
    {oper::sequence, 9, 0},
    {oper::single, 0, 0},
    {oper::multi, 11, UINT64_C(0x23e)},
    {oper::single, 0, 0},
    {oper::multi, 55, UINT64_C(0x6d673a69861803)},
    {oper::multi, 15, UINT64_C(0x6fd9)},
    {oper::multi, 32, UINT64_C(0xa825d0c9)},
    {oper::sequence, 8, 0},
    {oper::multi, 26, UINT64_C(0x2b01f64)},
    {oper::multi, 33, UINT64_C(0x1ff1bedb3)},
    {oper::multi, 45, UINT64_C(0x11ca982755a6)},
    {oper::multi, 38, UINT64_C(0x308f813291)},
    {oper::multi, 64, UINT64_C(0x30dca1960533e446)},
    {oper::multi, 26, UINT64_C(0x32832a8)},
    {oper::single, 0, 0},
    {oper::multi, 43, UINT64_C(0xdf6deb312d)},
    {oper::multi, 27, UINT64_C(0x1070d01)},
    {oper::multi, 3, UINT64_C(0x3)},
    {oper::multi, 7, UINT64_C(0x7e)},
    {oper::sequence, 23, 0},
    {oper::sequence, 9, 0},
    {oper::single, 0, 1},
    {oper::multi, 43, UINT64_C(0xaaaf45eb64)},
    {oper::single, 0, 0},
    {oper::sequence, 99, 0},
    {oper::multi, 58, UINT64_C(0x3393b88b289956e)},
    {oper::sequence, 97, 0},
    {oper::sequence, 2, 0},
    {oper::single, 0, 1},
    {oper::multi, 31, UINT64_C(0x24007fe)},
    {oper::multi, 17, UINT64_C(0x1528f)},
    {oper::multi, 20, UINT64_C(0x84e8f)},
    {oper::sequence, 69, 0},
    {oper::multi, 22, UINT64_C(0x299d)},
    {oper::multi, 28, UINT64_C(0xb4f3fe7)},
    {oper::multi, 40, UINT64_C(0x778bf93fc9)},
    {oper::multi, 10, UINT64_C(0x13b)},
    {oper::sequence, 77, 0},
    {oper::multi, 1, UINT64_C(0x1)},
    {oper::single, 0, 1},
    {oper::multi, 48, UINT64_C(0x178e69a42609)},
    {oper::multi, 43, UINT64_C(0xd73011c4fb)},
    {oper::multi, 44, UINT64_C(0xf2c7f15eefe)},
    {oper::multi, 29, UINT64_C(0xb0e98af)},
    {oper::sequence, 69, 0},
    {oper::multi, 53, UINT64_C(0x1c6cd200bf6093)},
    {oper::sequence, 12, 0},
    {oper::sequence, 34, 0},
    {oper::sequence, 23, 0},
    {oper::single, 0, 0},
    {oper::multi, 37, UINT64_C(0x1f28a9da93)},
    {oper::sequence, 35, 0},
    {oper::sequence, 41, 0},
    {oper::single, 0, 0},
    {oper::multi, 29, UINT64_C(0x9c59310)},
    {oper::multi, 4, UINT64_C(0x0)},
    {oper::multi, 54, UINT64_C(0x2c6cf00f05e10a)},
    {oper::single, 0, 1},
    {oper::sequence, 93, 0},
    {oper::multi, 58, UINT64_C(0x3a3dc81d5aa59)},
    {oper::sequence, 63, 0},
    {oper::multi, 16, UINT64_C(0x40e6)},
    {oper::sequence, 77, 0},
    {oper::multi, 5, UINT64_C(0x1e)},
    {oper::multi, 44, UINT64_C(0xfad71ed12bf)},
    {oper::multi, 23, UINT64_C(0x52b083)},
    {oper::multi, 10, UINT64_C(0x64)},
    {oper::multi, 9, UINT64_C(0x1a0)},
    {oper::single, 0, 1},
    {oper::multi, 21, UINT64_C(0xd3af1)},
    {oper::multi, 1, UINT64_C(0x0)},
    {oper::multi, 48, UINT64_C(0x4d7a2559ec5d)},
    {oper::multi, 56, UINT64_C(0xc1bc767aa8df5)},
    {oper::multi, 6, UINT64_C(0x25)},
    {oper::multi, 45, UINT64_C(0xd3d038b31ea)},
    {oper::sequence, 66, 0},
    {oper::multi, 24, UINT64_C(0xfeb51a)},
    {oper::multi, 32, UINT64_C(0x79cfb338)},
    {oper::multi, 23, UINT64_C(0x493337)},
    {oper::sequence, 84, 0},
    {oper::multi, 20, UINT64_C(0xcd153)},
    {oper::multi, 20, UINT64_C(0xb46e0)},
    {oper::multi, 18, UINT64_C(0x16d1c)},
    {oper::sequence, 76, 0},
    {oper::multi, 50, UINT64_C(0x97806628d511)},
    {oper::multi, 33, UINT64_C(0x1488926cd)},
    {oper::multi, 23, UINT64_C(0x93411)},
    {oper::multi, 33, UINT64_C(0xb9789b8b)},
    {oper::multi, 51, UINT64_C(0x690f639e62fb8)},
    {oper::sequence, 48, 0},
    {oper::multi, 5, UINT64_C(0x16)},
    {oper::single, 0, 1},
    {oper::sequence, 87, 0},
    {oper::multi, 38, UINT64_C(0x1c50c8baf8)},
    {oper::multi, 50, UINT64_C(0x282239aef706b)},
    {oper::sequence, 17, 0},
    {oper::sequence, 37, 0},
    {oper::single, 0, 1},
    {oper::single, 0, 1},
    {oper::multi, 15, UINT64_C(0x16aa)},
    {oper::multi, 21, UINT64_C(0x118dd2)},
    {oper::sequence, 86, 0},
    {oper::sequence, 79, 0},
    {oper::multi, 58, UINT64_C(0x2d8cdec0a68b1c6)},
    {oper::sequence, 73, 0},
    {oper::multi, 28, UINT64_C(0xe008ae)},
    {oper::single, 0, 0},
    {oper::multi, 48, UINT64_C(0x4dd47248aa1a)},
    {oper::single, 0, 1},
    {oper::multi, 50, UINT64_C(0x467d1098d766)},
    {oper::multi, 50, UINT64_C(0x18f5504e92d4f)},
    {oper::multi, 51, UINT64_C(0xf557e804c667)},
    {oper::sequence, 44, 0},
    {oper::sequence, 37, 0},
    {oper::multi, 21, UINT64_C(0x1ff9fa)},
    {oper::multi, 19, UINT64_C(0x2fa97)},
    {oper::sequence, 43, 0},
    {oper::multi, 57, UINT64_C(0x742cfe19db0e66)},
    {oper::sequence, 14, 0},
    {oper::sequence, 1, 0},
    {oper::multi, 56, UINT64_C(0x20160c73bf2827)},
    {oper::multi, 59, UINT64_C(0x34c86d5ccf39f56)},
    {oper::multi, 10, UINT64_C(0x3d7)},
    {oper::multi, 32, UINT64_C(0x6541165e)},
    {oper::multi, 25, UINT64_C(0x5200e6)},
    {oper::sequence, 71, 0},
    {oper::multi, 2, UINT64_C(0x2)},
    {oper::multi, 30, UINT64_C(0x6a48066)},
    {oper::multi, 34, UINT64_C(0x668a1253)},
    {oper::multi, 15, UINT64_C(0x1575)},
    {oper::sequence, 1, 0},
    {oper::sequence, 72, 0},
    {oper::multi, 55, UINT64_C(0x189c6c91bcff2d)},
    {oper::multi, 18, UINT64_C(0x3f36f)},
    {oper::multi, 9, UINT64_C(0x1af)},
    {oper::sequence, 33, 0},
    {oper::multi, 30, UINT64_C(0x3f3ed3f9)},
    {oper::single, 0, 1},
    {oper::sequence, 76, 0},
    {oper::multi, 5, UINT64_C(0xa)},
    {oper::sequence, 3, 0},
    {oper::sequence, 53, 0},
    {oper::sequence, 15, 0},
    {oper::sequence, 65, 0},
    {oper::multi, 7, UINT64_C(0x71)},
    {oper::multi, 44, UINT64_C(0xfe4a9bb262a)},
    {oper::multi, 23, UINT64_C(0x472d49)},
    {oper::multi, 31, UINT64_C(0x737735ae)},
    {oper::sequence, 47, 0},
    {oper::multi, 21, UINT64_C(0x173eb1)},
    {oper::sequence, 12, 0},
    {oper::multi, 15, UINT64_C(0x135b)},
    {oper::multi, 59, UINT64_C(0x2daa03bcce314ba)},
    {oper::multi, 17, UINT64_C(0x1b36a)},
    {oper::multi, 23, UINT64_C(0x71f1b9)},
    {oper::multi, 38, UINT64_C(0x14a1ac91bc)},
    {oper::sequence, 96, 0},
    {oper::multi, 46, UINT64_C(0xa63cd4b8e3b)},
    {oper::multi, 61, UINT64_C(0x1b94dead1925b347)},
    {oper::multi, 56, UINT64_C(0x245fc8cb7535ae)},
    {oper::multi, 12, UINT64_C(0xb9c)},
    {oper::multi, 43, UINT64_C(0x4347fe58e0c)},
    {oper::multi, 35, UINT64_C(0xb4776a3)},
    {oper::multi, 53, UINT64_C(0xdb831abf4f8e7)},
    {oper::multi, 19, UINT64_C(0x7441f)},
    {oper::sequence, 79, 0},
    {oper::sequence, 75, 0},
    {oper::multi, 31, UINT64_C(0x691fded8)},
    {oper::multi, 33, UINT64_C(0x1e1b5782c)},
    {oper::single, 0, 1},
    {oper::multi, 17, UINT64_C(0x10951)},
    {oper::sequence, 4, 0},
    {oper::multi, 32, UINT64_C(0xf827d9b9)},
    {oper::multi, 35, UINT64_C(0x3647ee9e0)},
    {oper::multi, 26, UINT64_C(0x2266eb0)},
    {oper::sequence, 10, 0},
    {oper::multi, 32, UINT64_C(0x4cc44078)},
    {oper::multi, 47, UINT64_C(0x5a43b0caf139)},
    {oper::multi, 16, UINT64_C(0xf89e)},
    {oper::multi, 7, UINT64_C(0x56)},
    {oper::multi, 24, UINT64_C(0x7eb6a)},
    {oper::multi, 61, UINT64_C(0xc2b7102a0ea42c0)},
    {oper::multi, 33, UINT64_C(0x1fdcbd259)},
    {oper::sequence, 2, 0},
    {oper::multi, 24, UINT64_C(0xbca9f5)},
    {oper::multi, 61, UINT64_C(0xa29e09c1e8fbf0c)},
    {oper::multi, 53, UINT64_C(0xe777153067389)},
    {oper::sequence, 42, 0},
    {oper::sequence, 81, 0},
    {oper::multi, 44, UINT64_C(0xc8bbae0f1c8)},
    {oper::single, 0, 1},
    {oper::single, 0, 1},
    {oper::sequence, 52, 0},
    {oper::multi, 30, UINT64_C(0x183bf843)},
    {oper::multi, 7, UINT64_C(0x34)},
    {oper::multi, 32, UINT64_C(0x5976f4ca)},
    {oper::multi, 32, UINT64_C(0x77ea9e53)},
    {oper::multi, 57, UINT64_C(0x14488ec95374469)},
    {oper::multi, 36, UINT64_C(0xae5fdd9c2)},
    {oper::multi, 38, UINT64_C(0x2a493c790)},
    {oper::multi, 11, UINT64_C(0x224)},
    {oper::sequence, 91, 0},
    {oper::sequence, 45, 0},
    {oper::multi, 55, UINT64_C(0x2fa1b5c5278900)},
    {oper::single, 0, 1},
    {oper::multi, 24, UINT64_C(0x8cdf7b)},
    {oper::sequence, 23, 0},
    {oper::sequence, 45, 0},
    {oper::multi, 8, UINT64_C(0xf1)},
    {oper::sequence, 91, 0},
    {oper::multi, 36, UINT64_C(0xa73a42b13)},
    {oper::multi, 34, UINT64_C(0x2fe56eec7)},
    {oper::multi, 50, UINT64_C(0x10c1093beb659)},
    {oper::multi, 5, UINT64_C(0x3)},
    {oper::multi, 33, UINT64_C(0xf263af98)},
    {oper::multi, 46, UINT64_C(0x12f23a50087e)},
    {oper::sequence, 1, 0},
    {oper::sequence, 32, 0},
    {oper::multi, 27, UINT64_C(0x340b00)},
    {oper::multi, 45, UINT64_C(0x16cab4e40293)},
    {oper::multi, 52, UINT64_C(0xf0380a85eb9c5)},
    {oper::multi, 20, UINT64_C(0xff1e2)},
    {oper::single, 0, 1},
    {oper::multi, 40, UINT64_C(0x2e885c9029)},
    {oper::multi, 9, UINT64_C(0x196)},
    {oper::multi, 37, UINT64_C(0x11318c0267)},
    {oper::sequence, 58, 0},
    {oper::multi, 12, UINT64_C(0xf7c)},
    {oper::multi, 29, UINT64_C(0xbbb78dc)},
    {oper::sequence, 19, 0},
    {oper::single, 0, 1},
    {oper::sequence, 0, 0},
    {oper::multi, 39, UINT64_C(0x70f0117d66)},
    {oper::sequence, 54, 0},
    {oper::sequence, 2, 0},
    {oper::single, 0, 0},
    {oper::sequence, 60, 0},
    {oper::multi, 64, UINT64_C(0x81bac8063acee28c)},
    {oper::sequence, 0, 0},
    {oper::multi, 59, UINT64_C(0xa3361315821594)},
    {oper::multi, 20, UINT64_C(0x79c3d)},
    {oper::multi, 50, UINT64_C(0x5831a2c37b04)},
    {oper::single, 0, 1},
    {oper::multi, 49, UINT64_C(0xc088e7cd9a53)},
    {oper::multi, 53, UINT64_C(0xefa40ac98f9db)},
    {oper::multi, 38, UINT64_C(0x2ab113745c)},
    {oper::multi, 40, UINT64_C(0x644bbf4417)},
    {oper::multi, 34, UINT64_C(0x24269dd4)},
    {oper::multi, 48, UINT64_C(0xccf84199d6a2)},
    {oper::multi, 62, UINT64_C(0x19a372e9d6b92559)},
    {oper::multi, 62, UINT64_C(0xa9a048c0f617dfb)},
    {oper::multi, 15, UINT64_C(0x2230)},
    {oper::multi, 45, UINT64_C(0x13c9dcd0c98e)},
    {oper::multi, 57, UINT64_C(0x7a15d2f7a735d2)},
    {oper::single, 0, 1},
    {oper::sequence, 53, 0},
    {oper::multi, 38, UINT64_C(0x1e633e5cd2)},
    {oper::sequence, 41, 0},
    {oper::multi, 50, UINT64_C(0x1736e606b09e6)},
    {oper::multi, 33, UINT64_C(0xc8b84e31)},
    {oper::multi, 37, UINT64_C(0x1522abcf06)},
    {oper::multi, 10, UINT64_C(0x1f9)},
    {oper::multi, 1, UINT64_C(0x0)},
    {oper::multi, 4, UINT64_C(0x8)},
    {oper::multi, 5, UINT64_C(0x18)},
    {oper::multi, 58, UINT64_C(0x289907fc1942c82)},
    {oper::multi, 47, UINT64_C(0x62746758a2c5)},
    {oper::multi, 43, UINT64_C(0x66cc4431c92)},
    {oper::sequence, 53, 0},
    {oper::multi, 24, UINT64_C(0xff3e28)},
    {oper::multi, 1, UINT64_C(0x1)},
    {oper::sequence, 9, 0},
    {oper::sequence, 99, 0},
    {oper::multi, 15, UINT64_C(0x16bf)},
    {oper::multi, 51, UINT64_C(0x714559f69449a)},
    {oper::multi, 36, UINT64_C(0xce4fe17e6)},
    {oper::multi, 21, UINT64_C(0x169a7b)},
    {oper::multi, 30, UINT64_C(0x24b33ae9)},
    {oper::sequence, 57, 0},
    {oper::sequence, 73, 0},
    {oper::sequence, 29, 0},
    {oper::single, 0, 1},
    {oper::single, 0, 1},
    {oper::sequence, 3, 0},
    {oper::multi, 41, UINT64_C(0x1666fda01c2)},
    {oper::single, 0, 0},
    {oper::multi, 19, UINT64_C(0x1e67c)},
    {oper::multi, 58, UINT64_C(0x2c4c39e9ac75127)},
    {oper::sequence, 100, 0},
    {oper::sequence, 74, 0},
    {oper::sequence, 47, 0},
    {oper::sequence, 5, 0},
    {oper::multi, 43, UINT64_C(0x72e46324f49)},
    {oper::single, 0, 0},
    {oper::sequence, 51, 0},
    {oper::multi, 20, UINT64_C(0x4f8f0)},
    {oper::multi, 30, UINT64_C(0x4a33d7f)},
    {oper::sequence, 53, 0},
    {oper::multi, 58, UINT64_C(0x2298cb585458976)},
    {oper::multi, 27, UINT64_C(0x5fa9a7c)},
    {oper::single, 0, 0},
    {oper::multi, 13, UINT64_C(0x9b4)},
    {oper::single, 0, 0},
    {oper::multi, 21, UINT64_C(0xd0da7)},
    {oper::multi, 4, UINT64_C(0xf)},
    {oper::multi, 32, UINT64_C(0x686f861d)},
    {oper::multi, 20, UINT64_C(0x614d6)},
    {oper::multi, 7, UINT64_C(0x5b)},
    {oper::sequence, 45, 0},
    {oper::multi, 22, UINT64_C(0x296d4d)},
    {oper::multi, 55, UINT64_C(0x39d11a6231bb54)},
    {oper::multi, 59, UINT64_C(0x381d4a9c32512d5)},
    {oper::multi, 61, UINT64_C(0x6392bb6c7dc70c3)},
    {oper::sequence, 25, 0},
    {oper::multi, 21, UINT64_C(0x18180e)},
    {oper::multi, 19, UINT64_C(0x3b5ed)},
    {oper::multi, 46, UINT64_C(0x25ad0f5388a7)},
    {oper::multi, 16, UINT64_C(0xec00)},
    {oper::multi, 3, UINT64_C(0x5)},
    {oper::multi, 47, UINT64_C(0x249b2f7064c8)},
    {oper::multi, 55, UINT64_C(0x1209271636d9c)},
    {oper::sequence, 64, 0},
    {oper::sequence, 25, 0},
    {oper::multi, 51, UINT64_C(0x14a021360804)},
    {oper::multi, 59, UINT64_C(0x67e35acf565f63)},
    {oper::sequence, 96, 0},
    {oper::multi, 15, UINT64_C(0x47fa)},
    {oper::multi, 34, UINT64_C(0x17ac1b4fc)},
    {oper::multi, 2, UINT64_C(0x3)},
    {oper::multi, 2, UINT64_C(0x0)},
    {oper::multi, 8, UINT64_C(0x54)},
    {oper::multi, 42, UINT64_C(0x8742e95127)},
    {oper::single, 0, 0},
    {oper::multi, 32, UINT64_C(0xa4797f57)},
    {oper::multi, 29, UINT64_C(0x6c8929c)},
    {oper::multi, 7, UINT64_C(0x29)},
    {oper::single, 0, 0},
    {oper::single, 0, 0},
    {oper::multi, 8, UINT64_C(0x76)},
    {oper::multi, 47, UINT64_C(0x55f75b9cbb4e)},
    {oper::multi, 6, UINT64_C(0x3b)},
    {oper::multi, 46, UINT64_C(0x26e33220e448)},
    {oper::multi, 58, UINT64_C(0x22bafaa4400976e)},
    {oper::multi, 41, UINT64_C(0x170559371ad)},
    {oper::multi, 14, UINT64_C(0x1361)},
    {oper::multi, 26, UINT64_C(0xef6457)},
    {oper::sequence, 91, 0},
    {oper::multi, 58, UINT64_C(0xc36ed6254fa2af)},
    {oper::multi, 55, UINT64_C(0x73295fe4c31f6d)},
    {oper::multi, 57, UINT64_C(0xb7b1c1636efa6e)},
    {oper::sequence, 47, 0},
    {oper::multi, 5, UINT64_C(0xc)},
    {oper::multi, 19, UINT64_C(0x124c6)},
    {oper::multi, 5, UINT64_C(0x14)},
    {oper::sequence, 47, 0},
    {oper::sequence, 5, 0},
    {oper::sequence, 64, 0},
    {oper::multi, 44, UINT64_C(0x612540607d7)},
    {oper::multi, 53, UINT64_C(0x9ceafb03b1a4)},
    {oper::sequence, 54, 0},
    {oper::multi, 24, UINT64_C(0xd8705c)},
    {oper::single, 0, 0},
    {oper::single, 0, 1},
    {oper::multi, 39, UINT64_C(0x5b8fbf270f)},
    {oper::multi, 5, UINT64_C(0x1f)},
    {oper::sequence, 78, 0},
    {oper::single, 0, 0},
    {oper::multi, 40, UINT64_C(0x32dda23e1b)},
    {oper::sequence, 27, 0},
    {oper::sequence, 28, 0},
    {oper::multi, 13, UINT64_C(0x10b4)},
    {oper::multi, 62, UINT64_C(0x253317219c04df0d)},
    {oper::multi, 40, UINT64_C(0x131af9df0a)},
    {oper::multi, 11, UINT64_C(0x72c)},
    {oper::sequence, 5, 0},
    {oper::multi, 42, UINT64_C(0x1fdae853ec2)},
    {oper::sequence, 5, 0},
    {oper::multi, 23, UINT64_C(0x4eba51)},
    {oper::multi, 32, UINT64_C(0x94029a41)},
    {oper::multi, 22, UINT64_C(0x7c1f0)},
    {oper::multi, 29, UINT64_C(0x15c36b14)},
    {oper::multi, 10, UINT64_C(0x9d)},
    {oper::multi, 53, UINT64_C(0x1eb606e5789fa1)},
    {oper::multi, 54, UINT64_C(0x1e834fc7409d37)},
    {oper::multi, 5, UINT64_C(0x1f)},
    {oper::single, 0, 0},
    {oper::single, 0, 1},
    {oper::multi, 33, UINT64_C(0x1092fbf43)},
    {oper::multi, 26, UINT64_C(0x530381)},
    {oper::multi, 16, UINT64_C(0x6082)},
    {oper::multi, 1, UINT64_C(0x0)},
    {oper::sequence, 78, 0},
    {oper::multi, 7, UINT64_C(0x6b)},
    {oper::multi, 19, UINT64_C(0x16e89)},
    {oper::multi, 53, UINT64_C(0x43bd4d41b74d7)},
    {oper::sequence, 51, 0},
    {oper::multi, 62, UINT64_C(0x43b873a0416d5cd)},
    {oper::multi, 9, UINT64_C(0x120)},
    {oper::multi, 39, UINT64_C(0xd730c0164)},
    {oper::multi, 15, UINT64_C(0x27a8)},
    {oper::multi, 58, UINT64_C(0x35f0441068d65d6)},
    {oper::single, 0, 1},
    {oper::single, 0, 0},
    {oper::sequence, 58, 0},
    {oper::multi, 45, UINT64_C(0x1770a5936446)},
    {oper::sequence, 56, 0},
    {oper::multi, 20, UINT64_C(0xf58a5)},
    {oper::sequence, 84, 0},
    {oper::single, 0, 0},
    {oper::sequence, 75, 0},
    {oper::multi, 60, UINT64_C(0x1f83d44c9a79384)},
    {oper::sequence, 71, 0},
    {oper::multi, 1, UINT64_C(0x1)},
    {oper::multi, 30, UINT64_C(0x22fde129)},
    {oper::multi, 13, UINT64_C(0x30c)},
    {oper::sequence, 25, 0},
    {oper::sequence, 81, 0},
    {oper::sequence, 31, 0},
    {oper::multi, 50, UINT64_C(0xf6efedaace42)},
    {oper::multi, 44, UINT64_C(0xa9269b056c2)},
    {oper::multi, 39, UINT64_C(0xd31291893)},
    {oper::multi, 41, UINT64_C(0x2dd867ae76)},
    {oper::multi, 23, UINT64_C(0x21f95d)},
    {oper::multi, 23, UINT64_C(0x245d58)},
    {oper::multi, 42, UINT64_C(0x201bc1138ea)},
    {oper::sequence, 14, 0},
    {oper::sequence, 78, 0},
    {oper::single, 0, 0},
    {oper::multi, 33, UINT64_C(0x1e7b18528)},
    {oper::multi, 44, UINT64_C(0x17ea22a34b2)},
    {oper::multi, 30, UINT64_C(0x8387d7)},
    {oper::multi, 63, UINT64_C(0x608a51408786aedc)},
    {oper::sequence, 41, 0},
    {oper::sequence, 6, 0},
    {oper::sequence, 70, 0},
    {oper::sequence, 100, 0},
    {oper::sequence, 78, 0},
    {oper::multi, 19, UINT64_C(0x5c5a3)},
    {oper::multi, 19, UINT64_C(0x353c8)},
    {oper::multi, 11, UINT64_C(0x319)},
    {oper::sequence, 35, 0},
    {oper::multi, 25, UINT64_C(0x1462a59)},
    {oper::sequence, 66, 0},
    {oper::multi, 9, UINT64_C(0x16b)},
    {oper::multi, 26, UINT64_C(0x2146e04)},
    {oper::multi, 16, UINT64_C(0x66ef)},
    {oper::sequence, 94, 0},
    {oper::multi, 30, UINT64_C(0x325e5828)},
    {oper::multi, 51, UINT64_C(0x2e606413379d)},
    {oper::multi, 26, UINT64_C(0xa5ecc8)},
    {oper::multi, 30, UINT64_C(0xbd1e1db)},
    {oper::multi, 58, UINT64_C(0x305154c56bb7e67)},
    {oper::single, 0, 0},
    {oper::multi, 38, UINT64_C(0x2e33fbeb55)},
    {oper::multi, 34, UINT64_C(0x3bdf8ad2b)},
    {oper::multi, 59, UINT64_C(0x3994c070a0f3c59)},
    {oper::multi, 15, UINT64_C(0x4d07)},
    {oper::sequence, 28, 0},
    {oper::multi, 44, UINT64_C(0x9ad1a135d6c)},
    {oper::multi, 19, UINT64_C(0x34080)},
    {oper::multi, 46, UINT64_C(0x2d1264f31fdc)},
    {oper::multi, 22, UINT64_C(0x34e85a)},
    {oper::single, 0, 0},
    {oper::sequence, 97, 0},
    {oper::sequence, 27, 0},
    {oper::multi, 63, UINT64_C(0xb70c06d6f43d4a5)},
    {oper::single, 0, 1},
    {oper::multi, 8, UINT64_C(0xf0)},
    {oper::single, 0, 0},
    {oper::single, 0, 1},
    {oper::multi, 17, UINT64_C(0xc1bc)},
    {oper::multi, 23, UINT64_C(0x402d47)},
    {oper::multi, 56, UINT64_C(0xb26a5bf1722502)},
    {oper::multi, 26, UINT64_C(0x3a16935)},
    {oper::single, 0, 1},
    {oper::sequence, 1, 0},
    {oper::multi, 32, UINT64_C(0x35b2f2fc)},
    {oper::multi, 13, UINT64_C(0xaf8)},
    {oper::sequence, 98, 0},
    {oper::multi, 8, UINT64_C(0x2)},
    {oper::single, 0, 0},
    {oper::single, 0, 0},
    {oper::single, 0, 1},
    {oper::multi, 53, UINT64_C(0xbc4f9faa1d3bc)},
    {oper::multi, 44, UINT64_C(0x400a6b64e7d)},
    {oper::multi, 33, UINT64_C(0xcdff4f78)},
    {oper::multi, 50, UINT64_C(0x27065c6cb2cd)},
    {oper::multi, 3, UINT64_C(0x6)},
    {oper::multi, 23, UINT64_C(0x149bf0)},
    {oper::multi, 38, UINT64_C(0x250031e56f)},
    {oper::single, 0, 1},
    {oper::multi, 2, UINT64_C(0x2)},
    {oper::sequence, 11, 0},
    {oper::multi, 37, UINT64_C(0x1bb0c44a15)},
    {oper::multi, 37, UINT64_C(0x169268ea35)},
    {oper::sequence, 62, 0},
    {oper::sequence, 16, 0},
    {oper::multi, 26, UINT64_C(0xa2226e)},
    {oper::sequence, 46, 0},
    {oper::multi, 15, UINT64_C(0x4fd3)},
    {oper::multi, 21, UINT64_C(0x1741e0)},
    {oper::single, 0, 1},
    {oper::single, 0, 1},
    {oper::multi, 53, UINT64_C(0x122cd0ef2741df)},
    {oper::multi, 35, UINT64_C(0x7e74e00d0)},
    {oper::multi, 64, UINT64_C(0xf88f062eb31ad106)},
    {oper::multi, 21, UINT64_C(0x1983cd)},
    {oper::single, 0, 1},
    {oper::sequence, 97, 0},
    {oper::multi, 31, UINT64_C(0x16eeab09)},
    {oper::sequence, 32, 0},
    {oper::multi, 20, UINT64_C(0xc3860)},
    {oper::sequence, 17, 0},
    {oper::multi, 25, UINT64_C(0xa32b91)},
    {oper::multi, 36, UINT64_C(0xd7efe67e0)},
    {oper::multi, 7, UINT64_C(0x5f)},
    {oper::multi, 60, UINT64_C(0xdee8a7cec010813)},
    {oper::sequence, 88, 0},
    {oper::multi, 6, UINT64_C(0x24)},
    {oper::multi, 62, UINT64_C(0x10014bcee0f961c1)},
    {oper::sequence, 72, 0},
    {oper::sequence, 87, 0},
    {oper::multi, 56, UINT64_C(0x467ce768045112)},
    {oper::sequence, 87, 0},
    {oper::multi, 61, UINT64_C(0x10203c202d75dd6f)},
    {oper::multi, 25, UINT64_C(0x1839472)},
    {oper::multi, 31, UINT64_C(0x50749e58)},
    {oper::single, 0, 1},
    {oper::multi, 62, UINT64_C(0x23c2f2e98975f0bb)},
    {oper::multi, 49, UINT64_C(0x2138878f302f)},
    {oper::sequence, 90, 0},
    {oper::multi, 58, UINT64_C(0x34c445b95f6d4e0)},
    {oper::multi, 28, UINT64_C(0xb8ea069)},
    {oper::multi, 4, UINT64_C(0x8)},
    {oper::multi, 46, UINT64_C(0x8893ce9cdd5)},
    {oper::single, 0, 1},
    {oper::multi, 61, UINT64_C(0xb2d768d0fd9232d)},
    {oper::multi, 47, UINT64_C(0x5ff077473869)},
    {oper::single, 0, 1},
    {oper::multi, 56, UINT64_C(0xccd42149dd851)},
    {oper::multi, 11, UINT64_C(0xc7)},
    {oper::sequence, 76, 0},
    {oper::multi, 41, UINT64_C(0xdfccacf7a3)},
    {oper::multi, 2, UINT64_C(0x1)},
    {oper::multi, 4, UINT64_C(0x5)},
    {oper::sequence, 63, 0},
    {oper::sequence, 69, 0},
    {oper::multi, 8, UINT64_C(0x87)},
    {oper::multi, 49, UINT64_C(0xb26e8a269e0f)},
    {oper::sequence, 9, 0},
    {oper::multi, 21, UINT64_C(0xf472b)},
    {oper::multi, 39, UINT64_C(0x19e53e8f41)},
    {oper::multi, 49, UINT64_C(0x35dca7969fc1)},
    {oper::multi, 34, UINT64_C(0x318c868fe)},
    {oper::single, 0, 1},
    {oper::multi, 30, UINT64_C(0x1f864d30)},
    {oper::multi, 16, UINT64_C(0x52bc)},
    {oper::sequence, 78, 0},
    {oper::multi, 16, UINT64_C(0xac08)},
    {oper::multi, 49, UINT64_C(0x112339394f1ed)},
    {oper::multi, 35, UINT64_C(0x30db872d)},
    {oper::multi, 3, UINT64_C(0x6)},
    {oper::multi, 40, UINT64_C(0xb69266052f)},
    {oper::sequence, 25, 0},
    {oper::sequence, 33, 0},
    {oper::single, 0, 0},
    {oper::multi, 29, UINT64_C(0x1069ce7)},
    {oper::multi, 24, UINT64_C(0x8f97e3)},
    {oper::sequence, 40, 0},
    {oper::sequence, 40, 0},
    {oper::single, 0, 0},
    {oper::multi, 53, UINT64_C(0x43dcc77b5d515)},
    {oper::multi, 16, UINT64_C(0xebc4)},
    {oper::sequence, 19, 0},
    {oper::multi, 13, UINT64_C(0x1d0)},
    {oper::multi, 60, UINT64_C(0x8b64f391c940a7d)},
    {oper::sequence, 41, 0},
    {oper::multi, 6, UINT64_C(0x3a)},
    {oper::multi, 16, UINT64_C(0x7a61)},
    {oper::multi, 53, UINT64_C(0x6f7c515913ab5)},
    {oper::multi, 9, UINT64_C(0x184)},
    {oper::multi, 52, UINT64_C(0x433e21b71351e)},
    {oper::multi, 63, UINT64_C(0x5b80143f8d7c3f17)},
    {oper::sequence, 43, 0},
    {oper::multi, 39, UINT64_C(0x3b6c625389)},
    {oper::multi, 7, UINT64_C(0x53)},
    {oper::sequence, 60, 0},
    {oper::sequence, 11, 0},
    {oper::sequence, 40, 0},
    {oper::multi, 51, UINT64_C(0x7006837f0e7bb)},
    {oper::multi, 4, UINT64_C(0x5)},
    {oper::multi, 44, UINT64_C(0x1532e7e8973)},
    {oper::single, 0, 1},
    {oper::multi, 56, UINT64_C(0x18c64e3b8e01fa)},
    {oper::sequence, 59, 0},
    {oper::sequence, 81, 0},
    {oper::single, 0, 0},
    {oper::multi, 6, UINT64_C(0xb)},
    {oper::sequence, 62, 0},
    {oper::single, 0, 0},
    {oper::multi, 46, UINT64_C(0xb0004e00a4)},
    {oper::multi, 3, UINT64_C(0x7)},
    {oper::multi, 46, UINT64_C(0x16016baf7694)},
    {oper::multi, 7, UINT64_C(0x3a)},
    {oper::multi, 29, UINT64_C(0xe9e2ee1)},
    {oper::single, 0, 0},
    {oper::multi, 25, UINT64_C(0x61c40d)},
    {oper::sequence, 81, 0},
    {oper::multi, 40, UINT64_C(0x2e52a5edf8)},
    {oper::sequence, 39, 0},
    {oper::multi, 31, UINT64_C(0xaedd8ff)},
    {oper::sequence, 81, 0},
    {oper::sequence, 67, 0},
    {oper::multi, 26, UINT64_C(0x1a34a73)},
    {oper::multi, 13, UINT64_C(0x1cb0)},
    {oper::multi, 20, UINT64_C(0xa0531)},
    {oper::sequence, 13, 0},
    {oper::sequence, 56, 0},
    {oper::multi, 25, UINT64_C(0x1e268be)},
    {oper::sequence, 7, 0},
    {oper::multi, 17, UINT64_C(0xa2f8)},
    {oper::multi, 51, UINT64_C(0x427c53882c260)},
    {oper::multi, 51, UINT64_C(0x14bc36aaf0226)},
    {oper::sequence, 11, 0},
    {oper::sequence, 96, 0},
    {oper::multi, 15, UINT64_C(0x7e07)},
    {oper::multi, 24, UINT64_C(0x4dff7d)},
    {oper::sequence, 36, 0},
    {oper::sequence, 79, 0},
    {oper::multi, 10, UINT64_C(0x2a2)},
    {oper::multi, 13, UINT64_C(0x799)},
    {oper::multi, 43, UINT64_C(0x1d57c21bcd0)},
    {oper::multi, 54, UINT64_C(0x2c3e69ded4dae0)},
    {oper::multi, 39, UINT64_C(0x750cc111f3)},
    {oper::sequence, 31, 0},
    {oper::multi, 31, UINT64_C(0x197d5ce2)},
    {oper::sequence, 58, 0},
    {oper::multi, 9, UINT64_C(0x61)},
    {oper::multi, 61, UINT64_C(0x144fa70cac90cf93)},
    {oper::sequence, 2, 0},
    {oper::multi, 20, UINT64_C(0xa3471)},
    {oper::multi, 11, UINT64_C(0x5a6)},
    {oper::sequence, 38, 0},
    {oper::sequence, 70, 0},
    {oper::multi, 48, UINT64_C(0x9166e5c53e89)},
    {oper::multi, 54, UINT64_C(0x196059c0bd069c)},
    {oper::sequence, 83, 0},
    {oper::sequence, 49, 0},
    {oper::sequence, 58, 0},
    {oper::sequence, 34, 0},
    {oper::sequence, 21, 0},
    {oper::sequence, 61, 0},
    {oper::multi, 53, UINT64_C(0x4afb81e236064)},
    {oper::sequence, 90, 0},
    {oper::sequence, 82, 0},
    {oper::multi, 51, UINT64_C(0x1b58258d2cbbf)},
    {oper::sequence, 44, 0},
    {oper::multi, 35, UINT64_C(0x256835a03)},
    {oper::multi, 49, UINT64_C(0x121be5be49f31)},
    {oper::multi, 51, UINT64_C(0x368a62cc826d9)},
    {oper::multi, 9, UINT64_C(0xf8)},
    {oper::multi, 64, UINT64_C(0x69a2f807beb682c0)},
    {oper::sequence, 97, 0},
    {oper::multi, 23, UINT64_C(0x6916d5)},
    {oper::multi, 9, UINT64_C(0x62)},
    {oper::multi, 22, UINT64_C(0x3e699e)},
    {oper::multi, 64, UINT64_C(0x4eee41f1ec375c7d)},
    {oper::sequence, 19, 0},
    {oper::sequence, 56, 0},
    {oper::sequence, 87, 0},
    {oper::sequence, 52, 0},
    {oper::multi, 1, UINT64_C(0x0)},
    {oper::multi, 40, UINT64_C(0x94849e3975)},
    {oper::multi, 41, UINT64_C(0x167f1563eb8)},
    {oper::sequence, 29, 0},
    {oper::multi, 27, UINT64_C(0x2325c52)},
    {oper::multi, 35, UINT64_C(0x3f5807c9f)},
    {oper::multi, 8, UINT64_C(0x2f)},
    {oper::single, 0, 1},
    {oper::multi, 38, UINT64_C(0x11f689a9ec)},
    {oper::sequence, 52, 0},
    {oper::sequence, 91, 0},
    {oper::multi, 64, UINT64_C(0xe8bf6e8ad6c72d55)},
    {oper::multi, 61, UINT64_C(0x1996b0c58a80665b)},
    {oper::sequence, 42, 0},
    {oper::sequence, 41, 0},
    {oper::multi, 27, UINT64_C(0x26db9c6)},
    {oper::sequence, 99, 0},
    {oper::multi, 59, UINT64_C(0x2b2c0647b60bf0a)},
    {oper::sequence, 1, 0},
    {oper::multi, 2, UINT64_C(0x3)},
    {oper::single, 0, 1},
    {oper::multi, 15, UINT64_C(0x7968)},
    {oper::sequence, 67, 0},
    {oper::multi, 19, UINT64_C(0x3f770)},
    {oper::multi, 52, UINT64_C(0x3df381d86e766)},
    {oper::multi, 64, UINT64_C(0x3e01f22b3efe0a27)},
    {oper::sequence, 78, 0},
    {oper::multi, 7, UINT64_C(0x71)},
    {oper::single, 0, 0},
    {oper::multi, 35, UINT64_C(0x4972e5f4e)},
    {oper::multi, 3, UINT64_C(0x2)},
    {oper::multi, 4, UINT64_C(0xc)},
    {oper::sequence, 86, 0},
    {oper::sequence, 86, 0},
    {oper::sequence, 18, 0},
    {oper::single, 0, 1},
    {oper::multi, 7, UINT64_C(0x5d)},
    {oper::multi, 54, UINT64_C(0x3ecfae07ec364b)},
    {oper::sequence, 71, 0},
    {oper::multi, 18, UINT64_C(0x25c8e)},
    {oper::multi, 54, UINT64_C(0x1624ba0683b28d)},
    {oper::multi, 8, UINT64_C(0x71)},
    {oper::multi, 9, UINT64_C(0x141)},
    {oper::sequence, 68, 0},
    {oper::sequence, 59, 0},
    {oper::multi, 42, UINT64_C(0x1b28eeff58a)},
    {oper::multi, 49, UINT64_C(0x3674eb015526)},
    {oper::multi, 49, UINT64_C(0x1339fa64d93a8)},
    {oper::multi, 42, UINT64_C(0x39641ec1c1)},
    {oper::multi, 41, UINT64_C(0x13c411058e4)},
    {oper::multi, 33, UINT64_C(0x18e10aac6)},
    {oper::single, 0, 1},
    {oper::sequence, 13, 0},
    {oper::multi, 37, UINT64_C(0x17fc99ed1)},
    {oper::multi, 50, UINT64_C(0x36f26af709301)},
    {oper::sequence, 67, 0},
    {oper::sequence, 13, 0},
    {oper::single, 0, 1},
    {oper::single, 0, 0},
    {oper::multi, 45, UINT64_C(0x70b10f8083c)},
    {oper::multi, 49, UINT64_C(0x16719fc3acbec)},
    {oper::sequence, 17, 0},
    {oper::multi, 18, UINT64_C(0x1e2e8)},
    {oper::multi, 21, UINT64_C(0x15ddcd)},
    {oper::multi, 10, UINT64_C(0x3b)},
    {oper::sequence, 24, 0},
    {oper::multi, 5, UINT64_C(0x8)},
    {oper::multi, 48, UINT64_C(0x3a88c112e383)},
    {oper::sequence, 52, 0},
    {oper::multi, 54, UINT64_C(0x4ed94b5e92596)},
    {oper::sequence, 45, 0},
    {oper::multi, 17, UINT64_C(0x1ff10)},
    {oper::sequence, 59, 0},
    {oper::multi, 40, UINT64_C(0xc24b5e943)},
    {oper::sequence, 41, 0},
    {oper::sequence, 75, 0},
    {oper::multi, 16, UINT64_C(0x85b)},
    {oper::single, 0, 1},
    {oper::sequence, 73, 0},
    {oper::sequence, 88, 0},
    {oper::multi, 41, UINT64_C(0x1b8e83e24f2)},
    {oper::sequence, 86, 0},
    {oper::multi, 8, UINT64_C(0x18)},
    {oper::sequence, 58, 0},
    {oper::single, 0, 1},
    {oper::sequence, 39, 0},
    {oper::multi, 15, UINT64_C(0x4783)},
    {oper::multi, 40, UINT64_C(0x646373a2b0)},
    {oper::multi, 20, UINT64_C(0xfd518)},
    {oper::single, 0, 1},
    {oper::multi, 49, UINT64_C(0x11c37a3b2d06a)},
    {oper::sequence, 0, 0},
    {oper::multi, 34, UINT64_C(0x1a8c5efef)},
    {oper::multi, 31, UINT64_C(0x141920e3)},
    {oper::sequence, 11, 0},
    {oper::multi, 16, UINT64_C(0x7db9)},
    {oper::sequence, 75, 0},
    {oper::sequence, 20, 0},
    {oper::multi, 4, UINT64_C(0x0)},
    {oper::single, 0, 1},
    {oper::single, 0, 0},
    {oper::multi, 17, UINT64_C(0x7f03)},
    {oper::sequence, 72, 0},
    {oper::sequence, 18, 0},
    {oper::sequence, 30, 0},
    {oper::multi, 8, UINT64_C(0x3b)},
    {oper::single, 0, 0},
    {oper::multi, 31, UINT64_C(0x364022f3)},
    {oper::sequence, 43, 0},
    {oper::sequence, 74, 0},
    {oper::multi, 26, UINT64_C(0x2803d1c)},
    {oper::sequence, 50, 0},
    {oper::multi, 3, UINT64_C(0x4)},
    {oper::multi, 54, UINT64_C(0x1211fd8bcbc6cb)},
    {oper::multi, 33, UINT64_C(0x2fff822)},
    {oper::sequence, 9, 0},
    {oper::multi, 45, UINT64_C(0x3fe5d84f6d2)},
    {oper::multi, 5, UINT64_C(0xd)},
    {oper::sequence, 41, 0},
    {oper::multi, 14, UINT64_C(0x17ab)},
    {oper::multi, 5, UINT64_C(0xb)},
    {oper::sequence, 55, 0},
    {oper::sequence, 6, 0},
    {oper::multi, 45, UINT64_C(0x12041bc1fa4b)},
    {oper::multi, 53, UINT64_C(0x33ed00fbfb754)},
    {oper::sequence, 62, 0},
    {oper::single, 0, 1},
    {oper::sequence, 39, 0},
    {oper::sequence, 48, 0},
    {oper::multi, 37, UINT64_C(0x14393caecb)},
    {oper::sequence, 2, 0},
    {oper::multi, 12, UINT64_C(0xd)},
    {oper::multi, 13, UINT64_C(0x1ef6)},
    {oper::multi, 49, UINT64_C(0x240e412bb03)},
    {oper::multi, 54, UINT64_C(0x3064c5f90fc0e8)},
    {oper::multi, 31, UINT64_C(0x4db288ee)},
    {oper::sequence, 100, 0},
    {oper::multi, 24, UINT64_C(0x4c7ec5)},
    {oper::single, 0, 0},
    {oper::multi, 56, UINT64_C(0x367e1c4087033b)},
    {oper::sequence, 60, 0},
    {oper::sequence, 29, 0},
    {oper::multi, 16, UINT64_C(0x844)},
    {oper::single, 0, 1},
    {oper::multi, 46, UINT64_C(0x30b319b7d1d1)},
    {oper::single, 0, 1},
    {oper::sequence, 57, 0},
    {oper::multi, 37, UINT64_C(0x1f7432ee01)},
    {oper::multi, 7, UINT64_C(0x3f)},
    {oper::sequence, 20, 0},
    {oper::multi, 45, UINT64_C(0x15f2959951e6)},
    {oper::multi, 27, UINT64_C(0x16170fc)},
    {oper::multi, 14, UINT64_C(0x2d72)},
    {oper::multi, 37, UINT64_C(0x124b2f5695)},
    {oper::multi, 19, UINT64_C(0x325fa)},
    {oper::multi, 7, UINT64_C(0xd)},
    {oper::single, 0, 0},
    {oper::multi, 6, UINT64_C(0xd)},
    {oper::sequence, 38, 0},
    {oper::sequence, 74, 0},
    {oper::multi, 29, UINT64_C(0x7b3c0a5)},
    {oper::multi, 38, UINT64_C(0x3e6cfcb0c5)},
    {oper::multi, 48, UINT64_C(0x39c9eca918cc)},
    {oper::multi, 60, UINT64_C(0xe41734e82096a4d)},
    {oper::sequence, 60, 0},
    {oper::single, 0, 1},
    {oper::sequence, 29, 0},
    {oper::multi, 43, UINT64_C(0x63d53597661)},
    {oper::multi, 13, UINT64_C(0xf9f)},
    {oper::multi, 35, UINT64_C(0x359b06c21)},
    {oper::multi, 6, UINT64_C(0xb)},
    {oper::multi, 1, UINT64_C(0x0)},
    {oper::sequence, 61, 0},
    {oper::sequence, 81, 0},
    {oper::multi, 40, UINT64_C(0xdc4d1d7882)},
    {oper::multi, 27, UINT64_C(0x47a3a0c)},
    {oper::multi, 16, UINT64_C(0x7142)},
    {oper::sequence, 96, 0},
    {oper::multi, 13, UINT64_C(0x1306)},
    {oper::multi, 46, UINT64_C(0x37ecfb45b451)},
    {oper::multi, 25, UINT64_C(0x90e162)},
    {oper::multi, 37, UINT64_C(0x14f9288b38)},
    {oper::multi, 15, UINT64_C(0x1f82)},
    {oper::multi, 55, UINT64_C(0x1b0aec288c97a5)},
    {oper::sequence, 80, 0},
    {oper::multi, 20, UINT64_C(0x37c04)},
    {oper::multi, 17, UINT64_C(0x10b1f)},
    {oper::multi, 38, UINT64_C(0x275f12b37)},
    {oper::multi, 60, UINT64_C(0x27cec155b3a8b24)},
    {oper::multi, 39, UINT64_C(0x52aa8ebf1)},
    {oper::sequence, 58, 0},
    {oper::single, 0, 0},
    {oper::multi, 64, UINT64_C(0x2d47b3656f530b3f)},
    {oper::sequence, 98, 0},
    {oper::single, 0, 0},
    {oper::multi, 3, UINT64_C(0x2)},
    {oper::sequence, 1, 0},
    {oper::sequence, 18, 0},
    {oper::sequence, 19, 0},
    {oper::multi, 20, UINT64_C(0xb5485)},
    {oper::multi, 51, UINT64_C(0xe341218da9f5)},
    {oper::multi, 1, UINT64_C(0x0)},
    {oper::multi, 11, UINT64_C(0x12)},
    {oper::multi, 58, UINT64_C(0x2582da46d0daf58)},
    {oper::multi, 35, UINT64_C(0x1120929eb)},
    {oper::multi, 50, UINT64_C(0x1f6a03a695af7)},
    {oper::multi, 28, UINT64_C(0xc33c823)},
    {oper::multi, 56, UINT64_C(0x6a8d0d40bb7ad1)},
    {oper::sequence, 72, 0},
    {oper::multi, 56, UINT64_C(0xa00b808bf4bf38)},
    {oper::multi, 28, UINT64_C(0x9c15b5d)},
    {oper::sequence, 45, 0},
    {oper::multi, 2, UINT64_C(0x1)},
    {oper::multi, 28, UINT64_C(0xd22be85)},
    {oper::multi, 63, UINT64_C(0x30ba7237f707c3b1)},
    {oper::multi, 6, UINT64_C(0x8)},
    {oper::multi, 16, UINT64_C(0xb241)},
    {oper::multi, 37, UINT64_C(0x1c1a2df9f0)},
    {oper::sequence, 99, 0},
    {oper::multi, 31, UINT64_C(0x54713449)},
    {oper::sequence, 22, 0},
    {oper::multi, 24, UINT64_C(0xef4afd)},
    {oper::single, 0, 0},
    {oper::multi, 48, UINT64_C(0x439e8bf93a88)},
    {oper::single, 0, 0},
    {oper::multi, 43, UINT64_C(0x2dbdfdd4433)},
}};

constexpr std::array<uint8_t, 4186> const expected_bitstream{{
    0x00, 0x00, 0x00, 0x00, 0xb8, 0x6e, 0x37, 0x21, 0xdb, 0x1c, 0xf9, 0x5c,
    0x7f, 0xf7, 0x23, 0x04, 0x97, 0x95, 0x58, 0xea, 0x10, 0xb8, 0x3b, 0xdc,
    0xe2, 0x33, 0x29, 0xe2, 0x1e, 0x30, 0xc1, 0x6f, 0xf9, 0xf2, 0xad, 0xa1,
    0x62, 0xe3, 0xc9, 0x23, 0x8f, 0xa0, 0xd8, 0x61, 0x39, 0x50, 0x2e, 0x2a,
    0xa8, 0x82, 0x28, 0x20, 0x48, 0xed, 0xae, 0x4f, 0x7f, 0xed, 0x6a, 0x62,
    0x18, 0x9f, 0x5a, 0xd8, 0xaa, 0xec, 0xcb, 0xb6, 0x1f, 0xdc, 0x82, 0x21,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe9, 0xdc, 0x19, 0x22, 0x85, 0x9c,
    0x64, 0x0c, 0x64, 0x06, 0xf0, 0x69, 0xd2, 0xc6, 0x79, 0xb4, 0x82, 0x32,
    0xaa, 0x11, 0x00, 0x00, 0x00, 0x20, 0x22, 0xe6, 0xd9, 0x8c, 0xf3, 0x50,
    0x9b, 0x2f, 0xcf, 0xd5, 0x89, 0x40, 0xf4, 0x72, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x86, 0x85, 0xfc,
    0x02, 0x00, 0x00, 0x98, 0xb2, 0xf4, 0x00, 0x9b, 0xcc, 0x41, 0x08, 0xae,
    0x59, 0x92, 0xc3, 0x82, 0x4e, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xac, 0x0c, 0x17, 0xad, 0xf5, 0x62, 0x2d, 0xee, 0x19, 0x63, 0x4e,
    0x00, 0x00, 0x00, 0xa7, 0x1b, 0xec, 0x25, 0x2b, 0x77, 0x7b, 0xc9, 0x19,
    0xbf, 0x73, 0xfd, 0xb0, 0x14, 0xe3, 0x72, 0x8c, 0x96, 0xda, 0xa4, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x40, 0x80, 0x7f, 0x0c, 0x83, 0x4c, 0xd4, 0x88, 0x11, 0xc2,
    0x89, 0xbc, 0xe1, 0x85, 0x68, 0x0d, 0x9d, 0x45, 0xf5, 0xba, 0x1d, 0x4a,
    0x96, 0x5b, 0xa6, 0x1b, 0xd0, 0x41, 0x77, 0xc9, 0xd8, 0x11, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x78, 0x35, 0x89, 0x7e, 0x8c, 0x2f,
    0x04, 0xfc, 0xac, 0x2d, 0xdc, 0x83, 0xfd, 0x8e, 0xfa, 0x69, 0xbe, 0xb7,
    0xb3, 0x4e, 0xd3, 0xdc, 0xee, 0xb0, 0xf0, 0xdd, 0xc9, 0x51, 0x84, 0x8e,
    0x32, 0x32, 0xf7, 0xb0, 0x1b, 0x05, 0x81, 0x4c, 0x93, 0x53, 0x33, 0x0c,
    0x12, 0x67, 0x6a, 0x8d, 0xc9, 0x9c, 0x67, 0x43, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0xa8, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00,
    0x00, 0x00, 0xd0, 0x28, 0x5f, 0x2c, 0x56, 0x01, 0x12, 0x9b, 0x0f, 0xa8,
    0x80, 0x0c, 0x5f, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe0, 0x47, 0x48,
    0x3c, 0xe6, 0x15, 0xb4, 0x08, 0xa4, 0x8e, 0x77, 0x40, 0x67, 0x9d, 0x44,
    0xa3, 0x7d, 0x2a, 0x93, 0xd0, 0x4f, 0x16, 0x94, 0x51, 0x04, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x7a, 0xf7,
    0xd4, 0x37, 0xf8, 0x49, 0x91, 0x5d, 0xd0, 0x21, 0x1c, 0x5d, 0x00, 0xc8,
    0x47, 0x06, 0x30, 0x0c, 0xd3, 0x74, 0xce, 0xda, 0xd9, 0xef, 0x64, 0xe8,
    0x12, 0x54, 0x80, 0x64, 0x1f, 0xb0, 0xce, 0xb6, 0x6f, 0xfc, 0x37, 0xad,
    0x3a, 0xc1, 0x54, 0x8e, 0x91, 0x32, 0x81, 0x8f, 0xb0, 0x11, 0xf9, 0x4c,
    0x81, 0x65, 0x28, 0x37, 0x0c, 0xaa, 0x0c, 0xca, 0x5a, 0x62, 0xd6, 0xdb,
    0xbe, 0x11, 0xd0, 0x70, 0x90, 0xf9, 0x01, 0x00, 0x00, 0x01, 0x4c, 0xb6,
    0x5e, 0xf4, 0xaa, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xe8, 0x56, 0x99, 0x28, 0x8b, 0xb8, 0x93, 0x33,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
    0xec, 0x7f, 0x00, 0x24, 0x78, 0x94, 0xfa, 0xe8, 0x84, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x67, 0x0a, 0x70, 0xfe, 0xf3, 0xb4,
    0xc9, 0x3f, 0xf9, 0x8b, 0x77, 0x3b, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x80, 0x27, 0x98, 0x90, 0xa6, 0x39, 0x5e, 0xec, 0x13,
    0x47, 0xc0, 0x5c, 0xc3, 0xdf, 0xbd, 0xe2, 0x8f, 0xe5, 0x5f, 0x31, 0x1d,
    0x16, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x09, 0xf6,
    0x0b, 0x20, 0xcd, 0xc6, 0x01, 0x20, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
    0x00, 0x4d, 0x6a, 0xa7, 0xa2, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00,
    0x00, 0x00, 0x00, 0x10, 0xc4, 0x64, 0x71, 0x02, 0x85, 0xf0, 0x82, 0x07,
    0x78, 0x36, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x98, 0xa5, 0x5a, 0x1d, 0xc8, 0x3d, 0x3a, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x39, 0x10, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xe8, 0x7f, 0x25, 0xda, 0xe3, 0x5a, 0x7f,
    0x10, 0x56, 0x4a, 0x06, 0xe8, 0xf1, 0x3a, 0x4d, 0x17, 0x7b, 0x56, 0x89,
    0x5e, 0x53, 0x7d, 0xa3, 0xea, 0xd9, 0xf1, 0x06, 0x43, 0xa9, 0x1e, 0xb3,
    0x38, 0xd0, 0xd3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa8,
    0x51, 0xeb, 0x8f, 0x33, 0xfb, 0x9c, 0x77, 0x33, 0x93, 0x04, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x53, 0xd1, 0x0c, 0x6e,
    0xb4, 0x1c, 0x6d, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xc0, 0x88, 0x6a, 0x14, 0x33, 0xc0, 0x4b, 0x9a, 0x4d, 0x12, 0x91, 0x46,
    0xd0, 0x24, 0x16, 0x37, 0xf1, 0x72, 0xe1, 0xbe, 0x98, 0xe7, 0xd8, 0x43,
    0x1a, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x0d, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0xaf, 0x8b, 0x0c, 0xc5, 0xad,
    0xc1, 0xbd, 0x6b, 0x8e, 0x08, 0x0a, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00,
    0xb8, 0xaa, 0x45, 0xba, 0x31, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x8d, 0x63, 0xd1, 0x14, 0xd8, 0x9b, 0xb1, 0x05, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd0, 0x15, 0x01, 0x1c, 0x68, 0xa8,
    0x22, 0xc9, 0x51, 0x37, 0x35, 0xbb, 0xc6, 0x84, 0xe8, 0x33, 0xe2, 0xa9,
    0x25, 0x9d, 0xa0, 0xea, 0xb1, 0x33, 0x63, 0x02, 0xf4, 0xab, 0x7a, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x50, 0x3f, 0xff,
    0x5f, 0xea, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0xcd, 0x1c, 0xb6, 0x33,
    0xfc, 0x59, 0xe8, 0x00, 0x00, 0x3d, 0x41, 0xf9, 0x9d, 0x63, 0xb0, 0x00,
    0xb1, 0xfa, 0x9c, 0x67, 0xae, 0x36, 0x64, 0xda, 0xf5, 0x5e, 0x16, 0x41,
    0x65, 0xe6, 0x00, 0x52, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x35, 0x03, 0x24, 0x35, 0xa6, 0x24, 0x14, 0xcd, 0xa8, 0xab, 0x08,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xe5, 0x9f, 0x37,
    0x92, 0x8d, 0x13, 0xf3, 0x36, 0xff, 0x6b, 0x00, 0x00, 0x00, 0x00, 0xf3,
    0xa7, 0x7d, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x50, 0x21, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x08, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x5c, 0xc5, 0x64, 0x37, 0x95,
    0xfc, 0x93, 0x5a, 0x8e, 0xae, 0x35, 0x77, 0x73, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xc0, 0x58, 0x9f, 0x0b, 0x00, 0xb7, 0x26, 0xba, 0x14, 0xe3, 0xcc,
    0x3b, 0xa0, 0xda, 0x52, 0x9b, 0x9d, 0x1b, 0x1f, 0xe7, 0x8d, 0x64, 0x0d,
    0xa5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xee, 0x38, 0x2e, 0x35, 0x8f, 0x29, 0x47, 0xb3, 0x25, 0x19, 0xad,
    0xde, 0x94, 0xdb, 0xb5, 0xa6, 0x6e, 0x19, 0xf9, 0x8b, 0x84, 0x73, 0x19,
    0x1c, 0xcb, 0xff, 0x68, 0x38, 0x6a, 0x77, 0xb4, 0x80, 0x73, 0x7c, 0xfa,
    0xd5, 0x18, 0xdc, 0xf6, 0x41, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xc4, 0xf6, 0xfe, 0x48, 0xb3, 0xe0, 0xd5, 0x86, 0x1f, 0x95, 0x10,
    0xe6, 0x66, 0x9f, 0xe0, 0x83, 0xa7, 0xfb, 0x91, 0x0d, 0xd6, 0xcd, 0x44,
    0x00, 0xe2, 0x01, 0x11, 0x33, 0xe5, 0xc4, 0x2b, 0xc3, 0x0e, 0x69, 0x3d,
    0xf1, 0xad, 0x6a, 0xeb, 0x07, 0xc0, 0x42, 0xea, 0xa0, 0x02, 0x71, 0x2b,
    0x2c, 0x4b, 0x7a, 0xb9, 0x3f, 0xeb, 0x53, 0x79, 0x19, 0x7e, 0x1f, 0x3d,
    0x38, 0xc1, 0x53, 0x54, 0xe2, 0x9c, 0xc1, 0x54, 0xdc, 0x9d, 0x03, 0x00,
    0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x80, 0xc8, 0xf1, 0xe0, 0xba, 0x8b, 0x3c, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x1c, 0xc2, 0xdf, 0xc1, 0x68, 0xca, 0xf4, 0x76, 0x59, 0x53,
    0x9e, 0xea, 0x77, 0x69, 0x44, 0x37, 0x95, 0xec, 0x88, 0x44, 0x85, 0xb3,
    0xfb, 0xcb, 0x15, 0xf2, 0x78, 0x92, 0x54, 0x20, 0x11, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x00, 0x89, 0x27, 0xc5, 0xb5, 0xa1, 0xaf, 0x7b, 0xdf, 0x8c,
    0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x3c, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4e, 0xac, 0x90,
    0xce, 0xe9, 0xb1, 0xbb, 0x95, 0xbf, 0x59, 0xb6, 0xbe, 0x93, 0x10, 0x0c,
    0x0d, 0xcc, 0xd7, 0x31, 0x79, 0x7e, 0x08, 0x50, 0x3a, 0xf2, 0x92, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x16, 0x68, 0x30, 0x29, 0x40, 0x4e, 0xab, 0x6c,
    0x8b, 0x73, 0xbd, 0x50, 0x01, 0x07, 0x5e, 0x3c, 0xfe, 0xa7, 0x40, 0x72,
    0x21, 0xba, 0x58, 0x3e, 0x13, 0x60, 0x8c, 0x89, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xe4, 0x7b, 0x6e, 0xbc, 0xdd, 0x05, 0x00, 0x80, 0x9b,
    0xf5, 0x45, 0xc0, 0xc3, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x04,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0xc5, 0x9d, 0x75, 0x0c,
    0x90, 0x75, 0x03, 0x53, 0x56, 0x08, 0x56, 0x4c, 0xd8, 0x8c, 0xa2, 0x87,
    0xf3, 0x08, 0xf6, 0x86, 0x45, 0x63, 0xb0, 0x38, 0xa5, 0xd9, 0x7c, 0x8e,
    0x08, 0x6c, 0x3b, 0x1f, 0x93, 0x15, 0x48, 0xdf, 0x71, 0xd1, 0x4d, 0xc4,
    0xaa, 0x17, 0x44, 0xbf, 0x4b, 0x64, 0xd4, 0x9d, 0x26, 0x24, 0x88, 0x5a,
    0x67, 0x06, 0xe1, 0x33, 0x67, 0x95, 0xe4, 0x5a, 0xa7, 0xcb, 0x8d, 0x66,
    0xfb, 0x7d, 0x61, 0x0f, 0x8c, 0x04, 0x9a, 0x0a, 0x8c, 0xc8, 0x31, 0x19,
    0x9a, 0x3b, 0x79, 0x4a, 0xd7, 0x9c, 0xde, 0x4b, 0x57, 0xe8, 0x09, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x73, 0xf9, 0x8c, 0x79, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x9a, 0x27, 0xac, 0x81, 0xb9, 0xcd, 0x15, 0xe3, 0x84,
    0x8b, 0xcc, 0xe0, 0x79, 0x55, 0xa4, 0xe6, 0x07, 0xb1, 0x20, 0x0b, 0x65,
    0xf0, 0x1f, 0x64, 0xa2, 0xc5, 0xa2, 0x58, 0x67, 0x74, 0x62, 0x49, 0x8e,
    0x21, 0x62, 0x36, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x28, 0x3e,
    0xff, 0x01, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0xc0, 0x5f, 0x8b, 0x26, 0x51, 0xda, 0x67, 0x15, 0xc5, 0xcd,
    0x2f, 0xfc, 0xc9, 0x79, 0x4f, 0xd3, 0xa6, 0xeb, 0xcc, 0x92, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x8e, 0xc2, 0x01, 0xda, 0x6f, 0x66,
    0xf1, 0x99, 0xe7, 0x24, 0xea, 0x58, 0xd3, 0x73, 0x98, 0x58, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x40, 0x30, 0xe9, 0x49, 0xc6, 0xc8, 0xe5, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x10, 0x1e, 0x9f, 0xfe, 0x7a, 0x46, 0x09, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xd0, 0x2e, 0xb1, 0xa8, 0xb0, 0x96, 0x31, 0x45, 0x3e,
    0x4d, 0xfd, 0xa2, 0x4d, 0x4e, 0x1b, 0xda, 0x77, 0x18, 0xbe, 0xa1, 0x59,
    0x53, 0xd8, 0x16, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6c, 0x6a, 0x4b, 0xa9,
    0x76, 0x63, 0xc4, 0x34, 0xa2, 0x73, 0xd5, 0x12, 0x25, 0xc3, 0xa9, 0xd4,
    0x81, 0x1b, 0x86, 0xe3, 0x3e, 0xb6, 0x5d, 0xc9, 0x31, 0x00, 0x00, 0x00,
    0x3a, 0x60, 0xe0, 0xf6, 0xda, 0x9d, 0x22, 0x4e, 0x3d, 0xb4, 0x96, 0x00,
    0xec, 0x45, 0x26, 0x83, 0x7b, 0xd9, 0x24, 0x71, 0xb6, 0x8d, 0xc5, 0x49,
    0x82, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
    0x00, 0x48, 0x80, 0x60, 0x13, 0x02, 0x4a, 0x81, 0xb1, 0x2f, 0xab, 0x67,
    0xad, 0xf1, 0x33, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xd4, 0x3f, 0xf2, 0xd3, 0x06, 0xeb, 0x35, 0x54, 0x27,
    0x51, 0xe9, 0x42, 0x87, 0xb8, 0xfa, 0xcb, 0x23, 0xe5, 0x94, 0x44, 0x36,
    0x29, 0xec, 0x9c, 0x76, 0x39, 0xb7, 0xee, 0xab, 0x3b, 0x12, 0x39, 0x88,
    0xcc, 0xb8, 0xe9, 0x76, 0x09, 0x40, 0xa4, 0xfa, 0xba, 0x62, 0x6b, 0xdc,
    0x64, 0x15, 0xdc, 0xb0, 0xe9, 0x8a, 0xec, 0x1d, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0x15, 0x7d, 0x2a, 0xb1,
    0x76, 0x1b, 0xa6, 0xed, 0x63, 0x98, 0xfc, 0x2b, 0x65, 0xee, 0xa6, 0xef,
    0x36, 0x16, 0x1c, 0x7b, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x90, 0x19,
    0x93, 0x84, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x82, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xaf, 0x0f, 0x0c, 0xa8, 0x24, 0x8c, 0x34,
    0x76, 0x60, 0x5f, 0x9d, 0x13, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xb9, 0xe0, 0xb0, 0x7d, 0x38, 0xf9, 0x7d, 0xdc, 0x7e, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x0d, 0x1f, 0xd1, 0x6e, 0x19,
    0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x80, 0xb4, 0xb0, 0xe1, 0x9b, 0x80,
    0x33, 0xe4, 0x62, 0xa6, 0x54, 0xf8, 0xce, 0xd7, 0x98, 0x60, 0x39, 0x28,
    0xec, 0x53, 0xe8, 0xda, 0x1f, 0x18, 0xa5, 0xeb, 0x0c, 0xd2, 0x14, 0xa0,
    0x84, 0x0f, 0x3e, 0x28, 0xd6, 0x86, 0x6b, 0x27, 0xa1, 0x9f, 0x78, 0xe5,
    0x06, 0xb6, 0xfe, 0xa6, 0x13, 0xe8, 0xf8, 0x69, 0xd0, 0xfb, 0x0e, 0xfd,
    0xbe, 0x24, 0x0c, 0x1c, 0x98, 0x42, 0x10, 0x0c, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x70, 0x9d, 0xe8, 0x96, 0x6b, 0xba, 0x0d,
    0x6a, 0xea, 0x1d, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0xcd, 0xd5,
    0x16, 0x04, 0x3a, 0x87, 0x3b, 0x04, 0x48, 0xb2, 0x00, 0x86, 0xb9, 0x06,
    0xea, 0xc9, 0xba, 0xac, 0xd1, 0x20, 0x88, 0xe0, 0xeb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x68, 0x44, 0x36, 0x59, 0x0a, 0x77, 0x01, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x96, 0x62, 0x3d, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x80, 0x84, 0x93, 0xa7, 0xc9, 0x44, 0x3d, 0xf8, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x25, 0xbc, 0x5f,
    0x64, 0x18, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x28, 0xe4, 0xac, 0xda, 0xfe,
    0x6e, 0x8f, 0xb0, 0x15, 0x6c, 0x9a, 0xa4, 0x4e, 0x62, 0xa4, 0xc4, 0x34,
    0xec, 0x5c, 0xcf, 0xb0, 0x5b, 0x74, 0xe5, 0x87, 0xb0, 0xba, 0x48, 0xea,
    0x38, 0x11, 0xbc, 0x01, 0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x80, 0x50, 0x0a, 0x63, 0xcf, 0xcb, 0xd2, 0xa8, 0x88,
    0xfa, 0xc5, 0xf5, 0xe1, 0x20, 0xc0, 0xed, 0x6a, 0x78, 0x08, 0x14, 0xa5,
    0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x10, 0x08, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0xc0, 0xd1, 0xe2, 0x22, 0x4f, 0x2d, 0x63, 0x00, 0x00, 0x00,
    0x00, 0x98, 0xa5, 0x62, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x6b, 0x09, 0xdc, 0x28, 0x7c, 0x37, 0x03, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa2, 0x60, 0x79, 0xc9, 0x9d,
    0x37, 0x13, 0x64, 0x60, 0x2e, 0x40, 0x66, 0x2f, 0x65, 0x3b, 0x3c, 0x7a,
    0x39, 0xf3, 0xdb, 0xb5, 0x62, 0xaa, 0x28, 0x58, 0xd5, 0xfa, 0xfe, 0x8c,
    0xbb, 0xd2, 0x8a, 0xdf, 0x7b, 0x16, 0xcf, 0x83, 0xc2, 0x01, 0x53, 0xe6,
    0x0e, 0x9a, 0x00, 0x00, 0x00, 0x90, 0xad, 0x6b, 0x42, 0xa3, 0x35, 0x01,
    0x81, 0xc6, 0xfd, 0x31, 0x4f, 0x26, 0xd1, 0x6a, 0xa1, 0xd3, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00,
    0x00, 0xc0, 0x52, 0xea, 0xa1, 0xb7, 0x36, 0x60, 0xb8, 0x45, 0x78, 0x79,
    0x83, 0x1d, 0xb5, 0x00, 0x05, 0x4a, 0xe4, 0xe2, 0xb7, 0xd4, 0x64, 0x6b,
    0xd2, 0x42, 0x2f, 0xbf, 0xbc, 0x6c, 0x0d, 0xbe, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x00, 0x79, 0xa7,
    0x43, 0xf5, 0xf3, 0x89, 0x57, 0x9f, 0x93, 0xad, 0x29, 0x00, 0xe1, 0x3d,
    0xfd, 0x37, 0x6b, 0x96, 0x65, 0xe3, 0x32, 0x38, 0xc1, 0xf0, 0x9b, 0x94,
    0xb7, 0xf2, 0x18, 0x80, 0xb2, 0x00, 0x58, 0xa1, 0x44, 0x0c, 0xbb, 0x6b,
    0xd4, 0xd1, 0x24, 0x2d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
    0x00, 0xa0, 0x9b, 0x88, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0xe9,
    0x27, 0x78, 0xd0, 0xfd, 0x3b, 0xe8, 0xe4, 0x1d, 0x9a, 0x45, 0x42, 0x03,
    0x38, 0x9d, 0xdf, 0x20, 0x5a, 0x63, 0xd6, 0xc5, 0xe0, 0x11, 0xbf, 0x79,
    0x30, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x30, 0x61, 0xd5, 0xdd, 0x02, 0x00, 0x00, 0x00, 0x10, 0x0c, 0x87,
    0x01, 0x00, 0x8c, 0x5c, 0x19, 0x05, 0x7e, 0xe6, 0xef, 0xd7, 0xdf, 0x09,
    0x84, 0x00, 0x76, 0x3e, 0x45, 0xf7, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x06, 0x87, 0xe5, 0x83, 0x3b, 0x2f,
    0x05, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x25, 0xa2,
    0x08, 0xd0, 0xce, 0xf9, 0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xdf, 0xba, 0xeb, 0x5a, 0x40, 0x78, 0x40, 0xa0,
    0x1c, 0xe5, 0x60, 0x2c, 0x4f, 0x3a, 0xe8, 0x5d, 0xf8, 0xba, 0xc4, 0x74,
    0x79, 0xe1, 0xf1, 0x05, 0xe6, 0xf1, 0x10, 0x27, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc1, 0xa9, 0xed, 0x2b,
    0xb7, 0x88, 0x98, 0x4e, 0x03, 0x75, 0x5c, 0xac, 0x6e, 0x4e, 0xe7, 0x49,
    0x44, 0xb6, 0x8c, 0x64, 0x3f, 0x34, 0xda, 0xb5, 0xac, 0x34, 0x9c, 0xa3,
    0x3b, 0xf8, 0xef, 0x28, 0xec, 0x4e, 0x0a, 0xa1, 0x66, 0x86, 0x63, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0xd1, 0x7b, 0x56,
    0xe6, 0x6f, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x78, 0xf8, 0xe0, 0x69, 0xa2,
    0xe8, 0x26, 0x0b, 0xc0, 0x95, 0xa3, 0x17, 0xf4, 0xe8, 0x53, 0x9e, 0x09,
    0xfe, 0xb4, 0x3c, 0xe5, 0xae, 0xe1, 0x8f, 0x86, 0x8c, 0x71, 0x98, 0x26,
    0xc3, 0x8f, 0x57, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x88, 0xc0, 0xda, 0x1e, 0x4f, 0x39, 0x39, 0x23, 0xb1, 0xe5, 0x70,
    0x1b, 0x06, 0x7e, 0x29, 0x30, 0x93, 0xb4, 0x05, 0x00, 0x00, 0x10, 0x00,
    0x00, 0x00, 0x40, 0xe7, 0x9c, 0x06, 0x61, 0xfc, 0xf2, 0x11, 0x00, 0x00,
    0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x40, 0x15, 0xd5, 0xb5, 0x77,
    0xcc, 0x3d, 0x84, 0x78, 0x1d, 0x00, 0x00, 0xa1, 0x43, 0x9f, 0x02, 0x25,
    0x47, 0xce, 0x93, 0x2d, 0x02, 0x00, 0x00, 0x00, 0x00, 0xa8, 0x87, 0xe9,
    0xd5, 0xea, 0x44, 0x56, 0x14, 0xdf, 0x1b, 0xc2, 0x1e, 0x35, 0x71, 0x1b,
    0xe2, 0x33, 0x74, 0xf1, 0xc3, 0xd7, 0xf8, 0x43, 0x01, 0xb8, 0x05, 0x00,
    0x00, 0x00, 0x00, 0xc0, 0xc4, 0x29, 0x31, 0xb6, 0xdd, 0x14, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x20, 0x00, 0x00, 0x00, 0x00, 0xc0,
    0xdd, 0x73, 0xf8, 0x1b, 0x34, 0x80, 0xd7, 0x5c, 0xa2, 0x9f, 0xcb, 0x54,
    0xd4, 0x0f, 0x70, 0xdc, 0x71, 0x32, 0xc6, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x2d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0xa4, 0x00,
    0x4e, 0x00, 0xb0, 0xc0, 0x29, 0xed, 0x5e, 0xd7, 0x02, 0x2c, 0x5d, 0xb8,
    0x8b, 0xa7, 0xd3, 0x40, 0x1c, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x40, 0xfc, 0xf6, 0x52, 0x29, 0x17, 0x00, 0x00, 0x00,
    0x00, 0xc0, 0x7f, 0xec, 0x76, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x38, 0xa7, 0x34, 0x1a, 0x2c, 0x8f, 0x29, 0x50, 0x00, 0x10, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xa0, 0x2f, 0x9a, 0x78, 0x40, 0x7c, 0x51, 0x60,
    0xc2, 0x82, 0x38, 0xc5, 0x27, 0x34, 0x11, 0x78, 0x55, 0x1b, 0x5e, 0x0a,
    0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x3c, 0xf0, 0xf7, 0xfd, 0x37, 0x01, 0x00, 0x00, 0x00, 0x40, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x51, 0x33, 0x0f,
    0x34, 0x6f, 0x08, 0x5f, 0x75, 0xc0, 0xb5, 0xa9, 0xbd, 0xd3, 0x7c, 0xd8,
    0xf9, 0x88, 0x60, 0x86, 0x3a, 0x00, 0x00, 0x00, 0xa0, 0x38, 0x57, 0x5f,
    0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x61, 0x26, 0x9f, 0x21,
    0x59, 0x19, 0x4e, 0x9f, 0x28, 0xe3, 0x68, 0xd4, 0xb4, 0x00, 0x00, 0x00,
    0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xa2,
    0x4f, 0x71, 0xb9, 0x59, 0x24, 0xa7, 0x41, 0x2f, 0x70, 0x16, 0x58, 0x06,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
    0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x48, 0x06, 0x36, 0xe2, 0x81, 0xfb, 0x4a, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0xdf, 0x65, 0x69, 0x2c, 0xc1,
    0xda, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x01, 0xad, 0x41, 0x2b, 0xc5,
    0x7c, 0x92, 0x6f, 0xf9, 0x86, 0xcc, 0x36, 0x41, 0x66, 0x31, 0x45, 0x1b,
    0x3e, 0x60, 0x41, 0x5b, 0xdf, 0x03, 0x7c, 0xd1, 0x34, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xab, 0x2d, 0xd2,
    0x62, 0x3c, 0xd3, 0xfc, 0x3e, 0xae, 0x1b, 0xf6, 0xf8, 0x20, 0x77, 0x27,
    0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xd5, 0xe5, 0x78, 0x12, 0x52, 0xe2, 0xfa, 0x58, 0xc5,
    0x9f, 0x05, 0x00, 0x00, 0x00, 0xa5, 0xb8, 0x64, 0xf4, 0xc9, 0x07, 0x58,
    0xbf, 0x97, 0xec, 0xa9, 0x89, 0xf6, 0x11, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xc0, 0xaa, 0x96, 0x63, 0x6b, 0x45, 0xb7, 0x5f, 0xf4, 0x2d, 0x33, 0x40,
    0xc5, 0x62, 0x58, 0xcb, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x8d, 0x73, 0xdb, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x0a, 0xbf, 0x60, 0x7b, 0x64,
    0xc0, 0xb2, 0xf2, 0x68, 0x79, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x84, 0xbb, 0x9f, 0xd9, 0xb9, 0x61, 0x07, 0xce, 0xf7, 0x9c, 0x28,
    0xf8, 0xfb, 0xac, 0xc8, 0x07, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xe3, 0x9c, 0xbe, 0x5c, 0x2e, 0x29, 0x06, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0xb8, 0xbb,
    0x64, 0xc3, 0x7e, 0xe0, 0xfa, 0xec, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x3a, 0x72, 0xd9, 0x28, 0x3b, 0x68, 0xa0, 0x4b, 0x62,
    0xc5, 0x05, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa8, 0x58, 0xff, 0xee, 0x28, 0x9b,
    0x49, 0x55, 0xc0, 0x3a, 0x9d, 0x0d, 0xd4, 0xc9, 0x26, 0xd3, 0xcf, 0x99,
    0xc1, 0xc1, 0x1e, 0x64, 0x39, 0x90, 0x63, 0x41, 0x04, 0xf1, 0x34, 0x56,
    0x85, 0x70, 0x1c, 0x00, 0x8c, 0xf6, 0x4c, 0xfe, 0x0b, 0x01, 0x93, 0x70,
    0xaf, 0x26, 0x6f, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20,
    0x00, 0x18, 0x0f, 0x02, 0x3e, 0xc4, 0xc2, 0x61, 0x5f, 0xd6, 0xe1, 0xcf,
    0x38, 0x0b, 0x00, 0x20, 0xba, 0x78, 0xcd, 0xdd, 0x75, 0x07, 0x00, 0x00,
    0x80, 0x68, 0x70, 0x5c, 0x22, 0x18, 0x51, 0x07, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x5a, 0x96, 0xa4, 0xd7, 0x52, 0xb6, 0x13, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x20, 0xc4, 0x7f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x1c, 0x4a, 0xaf, 0x25, 0x61, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb7, 0x10, 0x02, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x9e, 0xc4, 0x07, 0x1d, 0x37,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x03,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x07, 0x8f, 0xb0, 0xa2, 0x73, 0x63, 0x64, 0x18, 0xd5, 0x5f, 0x0d, 0x5a,
    0x76, 0xf4, 0x86, 0xe3, 0xf7, 0xf7, 0x62, 0xd4, 0xc6, 0x41, 0x32, 0x28,
    0x00, 0x98, 0xdb, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x00, 0x00, 0x10, 0x1a, 0xf8, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x10, 0x00, 0x80, 0x00, 0x00, 0x00, 0xc0, 0x1d, 0xf3,
    0x22, 0x40, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x47, 0x0f, 0xa0, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xe4, 0xb2, 0xf1, 0xf2, 0x62, 0x7f, 0x84, 0x24, 0x82,
    0xff, 0x2f, 0x00, 0x40, 0x69, 0x7b, 0xc2, 0x2e, 0xff, 0xd1, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x5c, 0xbd, 0x16, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x70, 0x49, 0x3f, 0x78, 0x83, 0x40, 0x52, 0xdd, 0xfe, 0x3e, 0x40,
    0xfb, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00,
    0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0xcb, 0xae, 0x3c,
    0x39, 0x94, 0x0d, 0x60, 0xef, 0x07, 0x76, 0x25, 0xc8, 0x81, 0x04, 0xa0,
    0x03, 0x3f, 0xe4, 0x17, 0x93, 0xc1, 0xee, 0x88, 0xb2, 0x4d, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x58, 0xec,
    0xc7, 0x64, 0x67, 0xe0, 0x10, 0x88, 0xc3, 0xcf, 0x06, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x80, 0x44, 0x08, 0xa3, 0xa3,
    0x6f, 0x33, 0x66, 0xe1, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06,
    0xb8, 0xcb, 0xd0, 0xfd, 0x1f, 0x00, 0x00, 0x34, 0x8f, 0xca, 0xac, 0x94,
    0xaf, 0xfc, 0x70, 0x61, 0x91, 0x6b, 0x2b, 0xad, 0x5e, 0x96, 0xa4, 0x7e,
    0xc9, 0x1a, 0x1a, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x4b, 0x81, 0x67, 0x4f, 0x31, 0x2c, 0x3f,
    0x9b, 0xcf, 0x8c, 0x91, 0xca, 0x9e, 0x9c, 0xd3, 0xa4, 0x96, 0x20, 0xe8,
    0x34, 0x17, 0xe4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00,
    0x00, 0x00, 0x18, 0x66, 0x97, 0x35, 0xd5, 0xe3, 0xcf, 0x17, 0xc2, 0x06,
    0x9b, 0xb5, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x20, 0x5e, 0x47,
    0x13, 0x37, 0x83, 0x8e, 0x1e, 0x85, 0xe2, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1a, 0xcc, 0x28, 0xda, 0xa2,
    0x7d, 0xf6, 0x5b, 0x2c, 0x1c, 0x12, 0xce, 0x22, 0x4a, 0x3e, 0x15, 0xfc,
    0x94, 0x5e, 0x32, 0xa2, 0xb0, 0x2b, 0x6c, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x12, 0xf0, 0xcd, 0xc7, 0xc2, 0x9b, 0x95,
    0xf8, 0x3a, 0x81, 0x64, 0x51, 0x67, 0xab, 0x82, 0x9d, 0x4f, 0xe2, 0xd7,
    0x51, 0x55, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf4, 0xb3,
    0x30, 0xf5, 0x56, 0x36, 0x7b, 0xd4, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x12, 0x00, 0x80, 0x00, 0x00,
    0x58, 0x48, 0xb5, 0xf5, 0xa9, 0x8d, 0x21, 0x41, 0xe3, 0x20, 0x01, 0xac,
    0xd7, 0x86, 0x36, 0xd2, 0x16, 0x2c, 0xd7, 0x53, 0x12, 0x24, 0x72, 0xaf,
    0x95, 0xa6, 0x03, 0x6a, 0xdf, 0x08, 0xf2, 0x0c, 0x47, 0xeb, 0xed, 0x02,
    0x35, 0x34, 0xaa, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xc4, 0xf9, 0xa5, 0x5f, 0x04, 0x5c, 0x00, 0xed, 0xda, 0x0a, 0x4e, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xb0, 0x42, 0x5f, 0x91, 0x8e, 0x1d, 0x3e, 0xb8,
    0xbf, 0x91, 0xd3, 0x85, 0x21, 0x41, 0xb2, 0xf0, 0xf9, 0x2d, 0x1a, 0x1c,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x93, 0x68, 0xe2, 0xa8, 0x00, 0x00, 0xc0, 0x7e, 0xa5, 0x77, 0x88, 0x3a,
    0xf9, 0x8b, 0x9e, 0x43, 0x66, 0x88, 0xba, 0xbf, 0xb7, 0x05,
}};

} // namespace

TEST(ricepp, bitstream_random_test) {
  std::vector<uint8_t> data;
  {
    ricepp::bitstream_writer writer(std::back_inserter(data));
    for (const auto& td : testdata) {
      switch (td.op) {
      case oper::single:
        writer.write_bit(td.value);
        break;
      case oper::multi:
        writer.write_bits(td.value, td.bits);
        break;
      case oper::sequence:
        writer.write_bit(false, td.bits);
        writer.write_bit(true);
        break;
      }
    }
    writer.flush();
  }

  ASSERT_EQ(expected_bitstream.size(), data.size());
  for (size_t i = 0; i < expected_bitstream.size(); ++i) {
    EXPECT_EQ(expected_bitstream[i], data[i]) << "at index " << i;
  }

  static_assert(std::contiguous_iterator<decltype(data)::iterator>);

  {
    ricepp::bitstream_reader reader(data.begin(), data.end());
    for (const auto& td : testdata) {
      switch (td.op) {
      case oper::single:
        EXPECT_EQ(td.value, reader.read_bit());
        break;
      case oper::multi:
        EXPECT_EQ(td.value, reader.read_bits<uint64_t>(td.bits));
        break;
      case oper::sequence:
        EXPECT_EQ(td.bits, reader.find_first_set());
        break;
      }
    }
  }

  std::deque<uint8_t> data2{data.begin(), data.end()};
  static_assert(!std::contiguous_iterator<decltype(data2)::iterator>);

  {
    ricepp::bitstream_reader reader(data2.begin(), data2.end());
    for (const auto& td : testdata) {
      switch (td.op) {
      case oper::single:
        EXPECT_EQ(td.value, reader.read_bit());
        break;
      case oper::multi:
        EXPECT_EQ(td.value, reader.read_bits<uint64_t>(td.bits));
        break;
      case oper::sequence:
        EXPECT_EQ(td.bits, reader.find_first_set());
        break;
      }
    }
  }
}
