// @ts-nocheck
export type TPaths = {
  '/v2/objects': {
    get: {
      summary: 'List objects';
      description: 'Lists all system-defined and user-defined objects in your workspace.\n\nRequired scopes: `object_configuration:read`.';
      tags: ['Objects'];
      security: [
        {
          oauth2: ['object_configuration:read'];
        },
      ];
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    type: 'array';
                    items: {
                      $ref: '#/components/schemas/object';
                    };
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
      };
    };
    post: {
      summary: 'Create an object';
      description: 'Creates a new custom object in your workspace.\n\nRequired scopes: `object_configuration:read-write`.';
      tags: ['Objects'];
      security: [
        {
          oauth2: ['object_configuration:read-write'];
        },
      ];
      requestBody: {
        required: true;
        content: {
          'application/json': {
            schema: {
              type: 'object';
              properties: {
                data: {
                  type: 'object';
                  properties: {
                    api_slug: {
                      type: 'string';
                      description: 'A unique, human-readable slug to access the object through URLs and API calls. Should be formatted in snake case.';
                      example: 'people';
                    };
                    singular_noun: {
                      type: 'string';
                      minLength: 1;
                      description: "The singular form of the object's name.";
                      example: 'Person';
                    };
                    plural_noun: {
                      type: 'string';
                      minLength: 1;
                      description: "The plural form of the object's name.";
                      example: 'People';
                    };
                  };
                  required: ['api_slug', 'singular_noun', 'plural_noun'];
                };
              };
              required: ['data'];
            };
          };
        };
      };
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    $ref: '#/components/schemas/object';
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '409': {
          description: 'Conflict';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [409];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['slug_conflict'];
                  };
                  message: {
                    type: 'string';
                    example: 'Failed to create object. Please ensure your api_slug and names are unique.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Conflict';
              };
            };
          };
        };
      };
    };
  };
  '/v2/objects/{object}': {
    get: {
      summary: 'Get an object';
      description: 'Gets a single object by its `object_id` or slug.\n\nRequired scopes: `object_configuration:read`.';
      tags: ['Objects'];
      security: [
        {
          oauth2: ['object_configuration:read'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the object.';
            example: 'people';
          };
          required: true;
          name: 'object';
          in: 'path';
        },
      ];
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    $ref: '#/components/schemas/object';
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Object with slug/ID "people" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
      };
    };
    patch: {
      summary: 'Update an object';
      description: 'Updates a single object. The object to be updated is identified by its `object_id`.\n\nRequired scopes: `object_configuration:read-write`.';
      tags: ['Objects'];
      security: [
        {
          oauth2: ['object_configuration:read-write'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the object.';
            example: 'people';
          };
          required: true;
          name: 'object';
          in: 'path';
        },
      ];
      requestBody: {
        required: true;
        content: {
          'application/json': {
            schema: {
              type: 'object';
              properties: {
                data: {
                  type: 'object';
                  properties: {
                    api_slug: {
                      type: 'string';
                      description: 'A unique, human-readable slug to access the object through URLs and API calls. Should be formatted in snake case.';
                      example: 'people';
                    };
                    singular_noun: {
                      type: 'string';
                      minLength: 1;
                      description: "The singular form of the object's name.";
                      example: 'Person';
                    };
                    plural_noun: {
                      type: 'string';
                      minLength: 1;
                      description: "The plural form of the object's name.";
                      example: 'People';
                    };
                  };
                };
              };
              required: ['data'];
            };
          };
        };
      };
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    $ref: '#/components/schemas/object';
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '400': {
          description: 'Bad Request';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [400];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['validation_type'];
                  };
                  message: {
                    type: 'string';
                    example: 'You passed an empty payload. Please ensure you are updating at least one property in your request.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Bad Request';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Object with slug/ID "people" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
        '409': {
          description: 'Conflict';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [409];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['slug_conflict'];
                  };
                  message: {
                    type: 'string';
                    example: 'Failed to update object. Please ensure api_slug is unique.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Conflict';
              };
            };
          };
        };
      };
    };
  };
  '/v2/{target}/{identifier}/attributes': {
    get: {
      summary: 'List attributes';
      description: 'Lists all attributes defined on a specific object or list. Attributes are returned in the order that they are sorted by in the UI.\n\nRequired scopes: `object_configuration:read`.';
      tags: ['Attributes'];
      security: [
        {
          oauth2: ['object_configuration:read'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            enum: ['objects', 'lists'];
            description: 'Whether the attributes are on an object or a list.';
            example: 'lists';
          };
          required: true;
          name: 'target';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the object or list the attributes belong to.';
            example: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
          };
          required: true;
          name: 'identifier';
          in: 'path';
        },
        {
          schema: {
            type: 'integer';
            description: 'The maximum number of results to return. See the [full guide to pagination here](/docs/pagination).';
            example: 10;
          };
          required: false;
          name: 'limit';
          in: 'query';
        },
        {
          schema: {
            type: 'integer';
            description: 'The number of results to skip over before returning. See the [full guide to pagination here](/docs/pagination).';
            example: 5;
          };
          required: false;
          name: 'offset';
          in: 'query';
        },
        {
          schema: {
            type: 'boolean';
            description: 'Whether archived attributes should be included in the results. See the [full guide to archiving here](/docs/archiving-vs-deleting).';
            example: true;
          };
          required: false;
          name: 'show_archived';
          in: 'query';
        },
      ];
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    type: 'array';
                    items: {
                      $ref: '#/components/schemas/attribute';
                    };
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
      };
    };
    post: {
      summary: 'Create an attribute';
      description: 'Creates a new attribute on either an object or a list.\n\nTo create an attribute on an object, you must also have the `object_configuration:read-write` scope.\n\nTo create an attribute on a list, you must also have the `list_configuration:read-write` scope.';
      tags: ['Attributes'];
      security: [
        {
          oauth2: [];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            enum: ['objects', 'lists'];
            description: 'Whether the attribute is to be created on an object or a list.';
            example: 'lists';
          };
          required: true;
          name: 'target';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the object or list the attribute belongs to.';
            example: '97052eb9-e65e-443f-a297-f2d9a4a7f795';
          };
          required: true;
          name: 'identifier';
          in: 'path';
        },
      ];
      requestBody: {
        required: true;
        content: {
          'application/json': {
            schema: {
              type: 'object';
              properties: {
                data: {
                  type: 'object';
                  properties: {
                    title: {
                      type: 'string';
                      description: "The name of the attribute. The title will be visible across Attio's UI.";
                      example: 'Your Attribute';
                    };
                    description: {
                      type: ['string', 'null'];
                      description: 'A text description for the attribute.';
                      example: 'Lorem ipsum';
                    };
                    api_slug: {
                      type: 'string';
                      description: 'A unique, human-readable slug to access the attribute through URLs and API calls. Formatted in snake case.';
                      example: 'my-attribute';
                    };
                    type: {
                      type: 'string';
                      enum: [
                        'text',
                        'number',
                        'checkbox',
                        'currency',
                        'date',
                        'timestamp',
                        'rating',
                        'status',
                        'select',
                        'record-reference',
                        'actor-reference',
                        'location',
                        'domain',
                        'email-address',
                        'phone-number',
                      ];
                      description: 'The type of the attribute. This value affects the possible `config` values. Attributes of type "status" are not supported on objects.';
                      example: 'text';
                    };
                    is_required: {
                      type: 'boolean';
                      description: 'When `is_required` is `true`, new records/entries must have a value for this attribute. If `false`, values may be `null`. This value does not affect existing data and you do not need to backfill `null` values if changing `is_required` from `false` to `true`.';
                      example: true;
                    };
                    is_unique: {
                      type: 'boolean';
                      description: 'Whether or not new values for this attribute must be unique. Uniqueness restrictions are only applied to new data and do not apply retroactively to previously created data.';
                      example: true;
                    };
                    is_multiselect: {
                      type: 'boolean';
                      description: 'Whether or not this attribute can have multiple values. Multiselect is only available on some value types.';
                      example: true;
                    };
                    default_value: {
                      oneOf: [
                        {
                          type: 'object';
                          properties: {
                            type: {
                              type: 'string';
                              enum: ['dynamic'];
                              example: 'dynamic';
                            };
                            template: {
                              anyOf: [
                                {
                                  type: 'string';
                                  enum: ['current-user'];
                                  description: 'For actor reference attributes, you may pass a dynamic value of `"current-user"`. When creating new records or entries, this will cause the actor reference value to be populated with either the workspace member or API token that created the record/entry.';
                                  example: 'current-user';
                                },
                                {
                                  type: 'string';
                                  description: 'Timestamp attributes may use an ISO 8601 duration as a dynamic value. For example, `"P1M"` would set the value to the current time plus one month.';
                                  example: 'P1M';
                                },
                                {
                                  type: 'string';
                                  description: 'Date attributes may use an ISO 8601 duration as a dynamic value. For example, `"P1M"` would set the value to the current time plus one month.';
                                  example: 'P1M';
                                },
                              ];
                            };
                          };
                          required: ['type', 'template'];
                        },
                        {
                          type: 'object';
                          properties: {
                            type: {
                              type: 'string';
                              enum: ['static'];
                              example: 'static';
                            };
                            template: {
                              type: 'array';
                              items: {
                                $ref: '#/components/schemas/input-value';
                              };
                              example: [
                                {
                                  value: 5;
                                },
                              ];
                            };
                          };
                          required: ['type', 'template'];
                        },
                        {
                          type: 'null';
                        },
                      ];
                      description: 'The default value for this attribute. Static values are used to directly populate values using their contents. Dynamic values are used to lookup data at the point of creation. For example, you could use a dynamic value to insert a value for the currently logged in user. Which default values are available is dependent on the type of the attribute. Default values are not currently supported on people or company objects.';
                      example: {
                        type: 'static';
                        template: [
                          {
                            value: 5;
                          },
                        ];
                      };
                    };
                    config: {
                      type: 'object';
                      properties: {
                        currency: {
                          type: 'object';
                          properties: {
                            default_currency_code: {
                              type: 'string';
                              enum: [
                                'EUR',
                                'GBP',
                                'USD',
                                'CAD',
                                'AUD',
                                'NZD',
                                'AED',
                                'INR',
                                'CHF',
                                'JPY',
                                'CNY',
                                'PHP',
                                'ILS',
                                'SGD',
                                'HKD',
                                'MYR',
                                'NTD',
                                'NOK',
                                'SEK',
                                'ISK',
                                'BEL',
                                'PLN',
                                'KRW',
                                'DKK',
                                'BRL',
                              ];
                              description: 'The ISO4217 code representing the currency that values for this attribute should be stored in.';
                              example: 'USD';
                            };
                            display_type: {
                              type: 'string';
                              enum: ['code', 'name', 'narrowSymbol', 'symbol'];
                              description: 'How the currency should be displayed across the app. "code" will display the ISO currency code e.g. "USD", "name" will display the localized currency name e.g. "British pound", "narrowSymbol" will display "$1" instead of "US$1" and "symbol" will display a localized currency symbol such as "$".';
                              example: 'symbol';
                            };
                          };
                          required: ['default_currency_code', 'display_type'];
                          description: 'Configuration available for attributes of type "currency".';
                        };
                        record_reference: {
                          type: 'object';
                          properties: {
                            allowed_objects: {
                              type: 'array';
                              items: {
                                type: 'string';
                                example: 'people';
                              };
                              minItems: 1;
                              description: 'A list of slugs or UUIDs to indicate which objects records are allowed to belong to. Leave empty to to allow records from all object types.';
                            };
                          };
                          required: ['allowed_objects'];
                          description: 'Configuration available for attributes of type "record-reference".';
                        };
                      };
                    };
                  };
                  required: [
                    'title',
                    'description',
                    'api_slug',
                    'type',
                    'is_required',
                    'is_unique',
                    'is_multiselect',
                    'config',
                  ];
                  additionalProperties: false;
                };
              };
              required: ['data'];
            };
          };
        };
      };
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    $ref: '#/components/schemas/attribute';
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '400': {
          description: 'Bad Request';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [400];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['validation_type'];
                  };
                  message: {
                    type: 'string';
                    example: 'Could not find an object with slug "people" for default value.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Bad Request';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'List with slug/ID "33ebdbe9-e529-47c9-b894-0ba25e9c15c0" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
        '409': {
          description: 'Conflict';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [409];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['slug_conflict'];
                  };
                  message: {
                    type: 'string';
                    example: 'An attribute with the same API slug already exists on this list. Please choose a different API slug.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Conflict';
              };
            };
          };
        };
      };
    };
  };
  '/v2/{target}/{identifier}/attributes/{attribute}': {
    get: {
      summary: 'Get an attribute';
      description: 'Gets information about a single attribute on either an object or a list.\n\nRequired scopes: `object_configuration:read`.';
      tags: ['Attributes'];
      security: [
        {
          oauth2: ['object_configuration:read'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            enum: ['objects', 'lists'];
            description: 'Whether the attribute is on an object or a list.';
            example: 'lists';
          };
          required: true;
          name: 'target';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the object or list the attribute belongs to.';
            example: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
          };
          required: true;
          name: 'identifier';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the attribute.';
            example: '41252299-f8c7-4b5e-99c9-4ff8321d2f96';
          };
          required: true;
          name: 'attribute';
          in: 'path';
        },
      ];
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    $ref: '#/components/schemas/attribute';
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Attribute with slug/ID "my-attribute" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
      };
    };
    patch: {
      summary: 'Update an attribute';
      description: 'Updates a single attribute on a given object or list.\n\nRequired scopes: `object_configuration:read-write`.';
      tags: ['Attributes'];
      security: [
        {
          oauth2: ['object_configuration:read-write'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            enum: ['objects', 'lists'];
            description: 'Whether the attribute is on an object or a list.';
            example: 'lists';
          };
          required: true;
          name: 'target';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the object or list the attribute belongs to.';
            example: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
          };
          required: true;
          name: 'identifier';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the attribute.';
            example: '41252299-f8c7-4b5e-99c9-4ff8321d2f96';
          };
          required: true;
          name: 'attribute';
          in: 'path';
        },
      ];
      requestBody: {
        required: true;
        content: {
          'application/json': {
            schema: {
              type: 'object';
              properties: {
                data: {
                  type: 'object';
                  properties: {
                    title: {
                      type: 'string';
                      description: "The name of the attribute. The title will be visible across Attio's UI.";
                      example: 'Your Attribute';
                    };
                    description: {
                      type: ['string', 'null'];
                      description: 'A text description for the attribute.';
                      example: 'Lorem ipsum';
                    };
                    api_slug: {
                      type: 'string';
                      description: 'A unique, human-readable slug to access the attribute through URLs and API calls. Formatted in snake case.';
                      example: 'my-attribute';
                    };
                    is_required: {
                      type: 'boolean';
                      description: 'When `is_required` is `true`, new records/entries must have a value for this attribute. If `false`, values may be `null`. This value does not affect existing data and you do not need to backfill `null` values if changing `is_required` from `false` to `true`.';
                      example: true;
                    };
                    is_unique: {
                      type: 'boolean';
                      description: 'Whether or not new values for this attribute must be unique. Uniqueness restrictions are only applied to new data and do not apply retroactively to previously created data.';
                      example: true;
                    };
                    default_value: {
                      oneOf: [
                        {
                          type: 'object';
                          properties: {
                            type: {
                              type: 'string';
                              enum: ['dynamic'];
                              example: 'dynamic';
                            };
                            template: {
                              anyOf: [
                                {
                                  type: 'string';
                                  enum: ['current-user'];
                                  description: 'For actor reference attributes, you may pass a dynamic value of `"current-user"`. When creating new records or entries, this will cause the actor reference value to be populated with either the workspace member or API token that created the record/entry.';
                                  example: 'current-user';
                                },
                                {
                                  type: 'string';
                                  description: 'Timestamp attributes may use an ISO 8601 duration as a dynamic value. For example, `"P1M"` would set the value to the current time plus one month.';
                                  example: 'P1M';
                                },
                                {
                                  type: 'string';
                                  description: 'Date attributes may use an ISO 8601 duration as a dynamic value. For example, `"P1M"` would set the value to the current time plus one month.';
                                  example: 'P1M';
                                },
                              ];
                            };
                          };
                          required: ['type', 'template'];
                        },
                        {
                          type: 'object';
                          properties: {
                            type: {
                              type: 'string';
                              enum: ['static'];
                              example: 'static';
                            };
                            template: {
                              type: 'array';
                              items: {
                                $ref: '#/components/schemas/input-value';
                              };
                              example: [
                                {
                                  value: 5;
                                },
                              ];
                            };
                          };
                          required: ['type', 'template'];
                        },
                        {
                          type: 'null';
                        },
                      ];
                      description: 'The default value for this attribute. Static values are used to directly populate values using their contents. Dynamic values are used to lookup data at the point of creation. For example, you could use a dynamic value to insert a value for the currently logged in user. Which default values are available is dependent on the type of the attribute. Default values are not currently supported on people or company objects.';
                      example: {
                        type: 'static';
                        template: [
                          {
                            value: 5;
                          },
                        ];
                      };
                    };
                    config: {
                      type: 'object';
                      properties: {
                        currency: {
                          type: 'object';
                          properties: {
                            default_currency_code: {
                              type: 'string';
                              enum: [
                                'EUR',
                                'GBP',
                                'USD',
                                'CAD',
                                'AUD',
                                'NZD',
                                'AED',
                                'INR',
                                'CHF',
                                'JPY',
                                'CNY',
                                'PHP',
                                'ILS',
                                'SGD',
                                'HKD',
                                'MYR',
                                'NTD',
                                'NOK',
                                'SEK',
                                'ISK',
                                'BEL',
                                'PLN',
                                'KRW',
                                'DKK',
                                'BRL',
                              ];
                              description: 'The ISO4217 code representing the currency that values for this attribute should be stored in.';
                              example: 'USD';
                            };
                            display_type: {
                              type: 'string';
                              enum: ['code', 'name', 'narrowSymbol', 'symbol'];
                              description: 'How the currency should be displayed across the app. "code" will display the ISO currency code e.g. "USD", "name" will display the localized currency name e.g. "British pound", "narrowSymbol" will display "$1" instead of "US$1" and "symbol" will display a localized currency symbol such as "$".';
                              example: 'symbol';
                            };
                          };
                          required: ['default_currency_code', 'display_type'];
                          description: 'Configuration available for attributes of type "currency".';
                        };
                        record_reference: {
                          type: 'object';
                          properties: {
                            allowed_objects: {
                              type: 'array';
                              items: {
                                type: 'string';
                                example: 'people';
                              };
                              minItems: 1;
                              description: 'A list of slugs or UUIDs to indicate which objects records are allowed to belong to. Leave empty to to allow records from all object types.';
                            };
                          };
                          required: ['allowed_objects'];
                          description: 'Configuration available for attributes of type "record-reference".';
                        };
                      };
                      description: 'Additional, type-dependent configuration for the attribute.';
                    };
                    is_archived: {
                      type: 'boolean';
                      description: 'Whether the attribute has been archived or not. See our [archiving guide](/docs/archiving-vs-deleting) for more information on archiving.';
                      example: false;
                    };
                  };
                  additionalProperties: false;
                };
              };
              required: ['data'];
            };
          };
        };
      };
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    $ref: '#/components/schemas/attribute';
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '400': {
          description: 'Bad Request';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [400];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['system_edit_unauthorized'];
                  };
                  message: {
                    type: 'string';
                    example: 'Cannot update a System attribute.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Bad Request';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Attribute with slug/ID "my-attribute" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
      };
    };
  };
  '/v2/{target}/{identifier}/attributes/{attribute}/options': {
    get: {
      summary: 'List select options';
      description: 'Lists all select options for a particular attribute on either an object or a list.\n\nRequired scopes: `object_configuration:read`.';
      tags: ['Attributes'];
      security: [
        {
          oauth2: ['object_configuration:read'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            enum: ['objects', 'lists'];
            description: 'Whether the attribute is on an object or a list.';
            example: 'lists';
          };
          required: true;
          name: 'target';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the object or list the select attribute belongs to.';
            example: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
          };
          required: true;
          name: 'identifier';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the attribute you want to list select options on.';
            example: '41252299-f8c7-4b5e-99c9-4ff8321d2f96';
          };
          required: true;
          name: 'attribute';
          in: 'path';
        },
        {
          schema: {
            type: 'boolean';
            description: '`true` if you want the results to include archived select options.';
            example: true;
          };
          required: false;
          name: 'show_archived';
          in: 'query';
        },
      ];
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    type: 'array';
                    items: {
                      $ref: '#/components/schemas/select-option';
                    };
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Attribute with slug/ID "my-attribute" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
      };
    };
    post: {
      summary: 'Create a select option';
      description: 'Adds a select option to a select attribute on an object or a list.\n\nRequired scopes: `object_configuration:read-write`.';
      tags: ['Attributes'];
      security: [
        {
          oauth2: ['object_configuration:read-write'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            enum: ['objects', 'lists'];
            description: 'Whether the attribute is on an object or a list.';
            example: 'lists';
          };
          required: true;
          name: 'target';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the object or list the select attribute belongs to.';
            example: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
          };
          required: true;
          name: 'identifier';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the attribute to create a select option on.';
            example: '41252299-f8c7-4b5e-99c9-4ff8321d2f96';
          };
          required: true;
          name: 'attribute';
          in: 'path';
        },
      ];
      requestBody: {
        required: true;
        content: {
          'application/json': {
            schema: {
              type: 'object';
              properties: {
                data: {
                  type: 'object';
                  properties: {
                    title: {
                      type: 'string';
                      minLength: 1;
                      description: 'The Title of the select option';
                      example: 'Medium';
                    };
                  };
                  required: ['title'];
                };
              };
              required: ['data'];
            };
          };
        };
      };
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    $ref: '#/components/schemas/select-option';
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '400': {
          description: 'Bad Request';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [400];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['validation_type'];
                  };
                  message: {
                    type: 'string';
                    example: 'This attribute is not a select attribute.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Bad Request';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Attribute with slug/ID "my-attribute" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
        '409': {
          description: 'Conflict';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [409];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['slug_conflict'];
                  };
                  message: {
                    type: 'string';
                    example: 'There is already another select option with the title "Medium".';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Conflict';
              };
            };
          };
        };
      };
    };
  };
  '/v2/{target}/{identifier}/attributes/{attribute}/options/{option}': {
    patch: {
      summary: 'Update a select option';
      description: 'Updates a select option on an attribute on either an object or a list.\n\nRequired scopes: `object_configuration:read-write`.';
      tags: ['Attributes'];
      security: [
        {
          oauth2: ['object_configuration:read-write'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            enum: ['objects', 'lists'];
            description: 'Whether the attribute is on an object or a list.';
            example: 'lists';
          };
          required: true;
          name: 'target';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the object or list the select attribute belongs to.';
            example: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
          };
          required: true;
          name: 'identifier';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the select attribute.';
            example: '41252299-f8c7-4b5e-99c9-4ff8321d2f96';
          };
          required: true;
          name: 'attribute';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or select option title to identify the select option you would like to update.';
            example: 'Medium';
          };
          required: true;
          name: 'option';
          in: 'path';
        },
      ];
      requestBody: {
        required: true;
        content: {
          'application/json': {
            schema: {
              type: 'object';
              properties: {
                data: {
                  type: 'object';
                  properties: {
                    title: {
                      type: 'string';
                      minLength: 1;
                      description: 'The Title of the select option';
                      example: 'Medium';
                    };
                    is_archived: {
                      type: 'boolean';
                      description: 'Whether or not to archive the select option. See our [archiving guide](/docs/archiving-vs-deleting) for more information on archiving.';
                      example: false;
                    };
                  };
                };
              };
              required: ['data'];
            };
          };
        };
      };
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    $ref: '#/components/schemas/select-option';
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '400': {
          description: 'Bad Request';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [400];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['value_not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Cannot find select attribute with select option title "Medium".';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Bad Request';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Attribute with slug/ID "my-attribute" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
        '409': {
          description: 'Conflict';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [409];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['slug_conflict'];
                  };
                  message: {
                    type: 'string';
                    example: 'There is already another select option with the title "Medium".';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Conflict';
              };
            };
          };
        };
      };
    };
  };
  '/v2/{target}/{identifier}/attributes/{attribute}/statuses': {
    get: {
      summary: 'List statuses';
      description: 'Lists all statuses for a particular status attribute on either an object or a list.\n\nRequired scopes: `object_configuration:read`.';
      tags: ['Attributes'];
      security: [
        {
          oauth2: ['object_configuration:read'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            enum: ['lists', 'objects'];
            description: 'Whether the attribute is on an object or a list. Please note that the company and people objects do not support status attributes at this time.';
            example: 'lists';
          };
          required: true;
          name: 'target';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the object or list the status attribute belongs to.';
            example: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
          };
          required: true;
          name: 'identifier';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the attribute you want to list statuses on.';
            example: '41252299-f8c7-4b5e-99c9-4ff8321d2f96';
          };
          required: true;
          name: 'attribute';
          in: 'path';
        },
        {
          schema: {
            type: 'boolean';
            description: '`true` if you want the results to include archived statuses. See our [archiving guide](/docs/archiving-vs-deleting) for more information on archiving.';
            example: true;
            default: false;
          };
          required: false;
          name: 'show_archived';
          in: 'query';
        },
      ];
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    type: 'array';
                    items: {
                      $ref: '#/components/schemas/status';
                    };
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Attribute with slug/ID "my-attribute" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
      };
    };
    post: {
      summary: 'Create a status';
      description: 'Add a new status to a status attribute on either an object or a list.\n\nRequired scopes: `object_configuration:read-write`.';
      tags: ['Attributes'];
      security: [
        {
          oauth2: ['object_configuration:read-write'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            enum: ['lists', 'objects'];
            description: 'Whether the attribute is on an object or a list. Please note that company and person objects do not support status attributes at this time.';
            example: 'lists';
          };
          required: true;
          name: 'target';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the object or list the status attribute belongs to.';
            example: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
          };
          required: true;
          name: 'identifier';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the attribute the status will belong to.';
            example: '41252299-f8c7-4b5e-99c9-4ff8321d2f96';
          };
          required: true;
          name: 'attribute';
          in: 'path';
        },
      ];
      requestBody: {
        required: true;
        content: {
          'application/json': {
            schema: {
              type: 'object';
              properties: {
                data: {
                  type: 'object';
                  properties: {
                    title: {
                      type: 'string';
                      minLength: 1;
                      description: 'The Title of the status';
                      example: 'In Progress';
                    };
                    celebration_enabled: {
                      type: 'boolean';
                      default: false;
                      description: 'Whether arriving at this status triggers a celebration effect';
                      example: true;
                    };
                    target_time_in_status: {
                      type: ['string', 'null'];
                      pattern: 'P(?:(\\d+Y)?(\\d+M)?(\\d+W)?(\\d+D)?(?:T(\\d+(?:[\\.,]\\d+)?H)?(\\d+(?:[\\.,]\\d+)?M)?(\\d+(?:[\\.,]\\d+)?S)?)?)';
                      description: 'Target time for a record to spend in given status expressed as a ISO-8601 duration string';
                      example: 'P0Y0M1DT0H0M0S';
                    };
                  };
                  required: ['title'];
                };
              };
              required: ['data'];
            };
          };
        };
      };
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    $ref: '#/components/schemas/status';
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '400': {
          description: 'Bad Request';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [400];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['validation_type'];
                  };
                  message: {
                    type: 'string';
                    example: 'This attribute is not a status attribute.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Bad Request';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Attribute with slug/ID "my-attribute" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
        '409': {
          description: 'Conflict';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [409];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['slug_conflict'];
                  };
                  message: {
                    type: 'string';
                    example: 'There is already another status with the title "In Progress".';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Conflict';
              };
            };
          };
        };
      };
    };
  };
  '/v2/{target}/{identifier}/attributes/{attribute}/statuses/{status}': {
    patch: {
      summary: 'Update a status';
      description: 'Update a status on an status attribute on either an object or a list.\n\nRequired scopes: `object_configuration:read-write`.';
      tags: ['Attributes'];
      security: [
        {
          oauth2: ['object_configuration:read-write'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            enum: ['lists', 'objects'];
            description: 'Whether the attribute is on an object or a list. Please note that company and person objects do not support status attributes at this time.';
            example: 'lists';
          };
          required: true;
          name: 'target';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the object or list the status attribute belongs to.';
            example: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
          };
          required: true;
          name: 'identifier';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the attribute to update the status on.';
            example: '41252299-f8c7-4b5e-99c9-4ff8321d2f96';
          };
          required: true;
          name: 'attribute';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or status title to identify the status to update.';
            example: 'In Progress';
          };
          required: true;
          name: 'status';
          in: 'path';
        },
      ];
      requestBody: {
        required: true;
        content: {
          'application/json': {
            schema: {
              type: 'object';
              properties: {
                data: {
                  type: 'object';
                  properties: {
                    title: {
                      type: 'string';
                      minLength: 1;
                      description: 'The Title of the status';
                      example: 'In Progress';
                    };
                    celebration_enabled: {
                      type: 'boolean';
                      default: false;
                      description: 'Whether arriving at this status triggers a celebration effect';
                      example: true;
                    };
                    target_time_in_status: {
                      type: ['string', 'null'];
                      pattern: 'P(?:(\\d+Y)?(\\d+M)?(\\d+W)?(\\d+D)?(?:T(\\d+(?:[\\.,]\\d+)?H)?(\\d+(?:[\\.,]\\d+)?M)?(\\d+(?:[\\.,]\\d+)?S)?)?)';
                      description: 'Target time for a record to spend in given status expressed as a ISO-8601 duration string';
                      example: 'P0Y0M1DT0H0M0S';
                    };
                    is_archived: {
                      type: 'boolean';
                      description: 'Whether or not to archive the status. See our [archiving guide](/docs/archiving-vs-deleting) for more information on archiving.';
                      example: false;
                    };
                  };
                };
              };
              required: ['data'];
            };
          };
        };
      };
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    $ref: '#/components/schemas/status';
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '400': {
          description: 'Bad Request';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [400];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['value_not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Cannot find status attribute with status option title "Medium".';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Bad Request';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Attribute with slug/ID "my-attribute" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
        '409': {
          description: 'Conflict';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [409];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['slug_conflict'];
                  };
                  message: {
                    type: 'string';
                    example: 'There is already another status with the title "In Progress".';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Conflict';
              };
            };
          };
        };
      };
    };
  };
  '/v2/objects/{object}/records/query': {
    post: {
      summary: 'List records';
      description: 'Lists people, company or other records, with the option to filter and sort results.\n\nRequired scopes: `record_permission:read`, `object_configuration:read`.';
      tags: ['Records'];
      security: [
        {
          oauth2: ['record_permission:read', 'object_configuration:read'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the object to list records for.';
            example: 'people';
          };
          required: true;
          name: 'object';
          in: 'path';
        },
      ];
      requestBody: {
        required: true;
        content: {
          'application/json': {
            schema: {
              type: 'object';
              properties: {
                filter: {
                  type: 'object';
                  description: 'An object used to filter results to a subset of results. See the [full guide to filtering and sorting here](/docs/filtering-and-sorting).';
                  additionalProperties: true;
                  example: {
                    name: 'Ada Lovelace';
                  };
                };
                sorts: {
                  type: 'array';
                  items: {
                    anyOf: [
                      {
                        type: 'object';
                        properties: {
                          direction: {
                            type: 'string';
                            enum: ['asc', 'desc'];
                            description: 'The direction to sort the results by.';
                          };
                          attribute: {
                            type: 'string';
                            description: 'A slug or ID to identify the attribute to sort by.';
                          };
                          field: {
                            type: 'string';
                            description: 'Which field on the value to sort by e.g. "last_name" on a name value.';
                          };
                        };
                        required: ['direction', 'attribute'];
                        description: 'Sort by attribute';
                      },
                      {
                        type: 'object';
                        properties: {
                          direction: {
                            type: 'string';
                            enum: ['asc', 'desc'];
                            description: 'The direction to sort the results by.';
                          };
                          path: {
                            type: 'array';
                            items: {
                              type: 'array';
                              items: {
                                anyOf: [
                                  {
                                    type: 'string';
                                    description: 'The slug or ID of the object e.g. "people".';
                                  },
                                  {
                                    type: 'string';
                                    description: 'A slug or ID to identify the attribute to sort by.';
                                  },
                                ];
                              };
                              minItems: 2;
                              maxItems: 2;
                            };
                            description: "You may use the `path` property to traverse record reference attributes and parent records on list entries. `path` accepts an array of tuples where the first element of each tuple is the slug or ID of a list/object, and the second element is the slug or ID of an attribute on that list/object. The first element of the first tuple must correspond to the list or object that you are querying. For example, if you wanted to sort by the name of the parent record (a company) on a list with the slug \"sales\", you would pass the value `[['sales', 'parent_record'], ['companies', 'name']]`.";
                          };
                          field: {
                            type: 'string';
                            description: 'Which field on the value to sort by e.g. "last_name" on a name value.';
                          };
                        };
                        required: ['direction', 'path'];
                        description: 'Sort by path';
                      },
                    ];
                  };
                  description: 'An object used to sort results. See the [full guide to filtering and sorting here](/docs/filtering-and-sorting).';
                  example: [
                    {
                      direction: 'asc';
                      attribute: 'name';
                      field: 'last_name';
                    },
                  ];
                };
                limit: {
                  type: 'number';
                  description: 'The maximum number of results to return. Defaults to 500. See the [full guide to pagination here](/docs/pagination).';
                  example: 500;
                };
                offset: {
                  type: 'number';
                  description: 'The number of results to skip over before returning. Defaults to 0. See the [full guide to pagination here](/docs/pagination).';
                  example: 0;
                };
              };
            };
          };
        };
      };
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    type: 'array';
                    items: {
                      type: 'object';
                      properties: {
                        id: {
                          type: 'object';
                          properties: {
                            workspace_id: {
                              type: 'string';
                              format: 'uuid';
                              description: 'A UUID identifying the workspace this record belongs to.';
                              example: '14beef7a-99f7-4534-a87e-70b564330a4c';
                            };
                            object_id: {
                              type: 'string';
                              format: 'uuid';
                              description: 'A UUID identifying the object this record belongs to.';
                              example: '97052eb9-e65e-443f-a297-f2d9a4a7f795';
                            };
                            record_id: {
                              type: 'string';
                              format: 'uuid';
                              description: 'A UUID identifying this record.';
                              example: 'bf071e1f-6035-429d-b874-d83ea64ea13b';
                            };
                          };
                          required: ['workspace_id', 'object_id', 'record_id'];
                        };
                        created_at: {
                          type: 'string';
                          description: 'When this record was created.';
                          example: '2022-11-21T13:22:49.061281000Z';
                        };
                        values: {
                          type: 'object';
                          additionalProperties: {
                            type: 'array';
                            items: {
                              oneOf: [
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    referenced_actor_type: {
                                      type: 'string';
                                      enum: ['api-token', 'workspace-member', 'system'];
                                      description: 'The type of the referenced actor. [Read more information on actor types here](/docs/actors).';
                                      example: 'workspace-member';
                                    };
                                    referenced_actor_id: {
                                      type: ['string', 'null'];
                                      format: 'uuid';
                                      description: 'The ID of the referenced actor.';
                                      example: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['actor-reference'];
                                      description: 'The attribute type of the value.';
                                      example: 'actor-reference';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'referenced_actor_type',
                                    'referenced_actor_id',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    value: {
                                      type: 'boolean';
                                      description: "A boolean representing whether the checkbox is checked or not. The string values 'true' and 'false' are also accepted.";
                                      example: true;
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['checkbox'];
                                      description: 'The attribute type of the value.';
                                      example: 'checkbox';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'value',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    currency_value: {
                                      type: 'number';
                                      description: 'A numerical representation of the currency value. A decimal with a max of 4 decimal places.';
                                      example: 99;
                                    };
                                    currency_code: {
                                      type: ['string', 'null'];
                                      enum: [
                                        'EUR',
                                        'GBP',
                                        'USD',
                                        'CAD',
                                        'AUD',
                                        'NZD',
                                        'AED',
                                        'INR',
                                        'CHF',
                                        'JPY',
                                        'CNY',
                                        'PHP',
                                        'ILS',
                                        'SGD',
                                        'HKD',
                                        'MYR',
                                        'NTD',
                                        'NOK',
                                        'SEK',
                                        'ISK',
                                        'BEL',
                                        'PLN',
                                        'KRW',
                                        'DKK',
                                        'BRL',
                                      ];
                                      description: 'The ISO4217 currency code representing the currency that the value is stored in.';
                                      example: 'USD';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['currency'];
                                      description: 'The attribute type of the value.';
                                      example: 'currency';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'currency_value',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['date'];
                                      description: 'The attribute type of the value.';
                                      example: 'date';
                                    };
                                    value: {
                                      type: 'string';
                                      description: 'A date represents a single calendar year, month and day, independent of timezone. If hours, months, seconds or timezones are provided, they will be trimmed. For example, "2023" and "2023-01" will be coerced into "2023-01-01", and "2023-01-02", "2023-01-02T13:00", "2023-01-02T14:00:00", "2023-01-02T15:00:00.000000000", and "2023-01-02T15:00:00.000000000+02:00" will all be coerced to "2023-01-02". If a timezone is provided that would result in a different calendar date in UTC, the date will be coerced to UTC and then the timezone component will be trimmed. For example, the value "2023-01-02T23:00:00-10:00" will be returned as "2023-01-03". The maximum date is "9999-12-31".';
                                      example: '2023-01-01';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'attribute_type',
                                    'value',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    domain: {
                                      type: 'string';
                                      example: 'app.attio.com';
                                    };
                                    root_domain: {
                                      type: 'string';
                                      example: 'attio.com';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['domain'];
                                      description: 'The attribute type of the value.';
                                      example: 'domain';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'domain',
                                    'root_domain',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    original_email_address: {
                                      type: 'string';
                                      example: 'alice@app.attio.com';
                                    };
                                    email_address: {
                                      type: 'string';
                                      example: 'alice@app.attio.com';
                                    };
                                    email_domain: {
                                      type: 'string';
                                      example: 'app.attio.com';
                                    };
                                    email_root_domain: {
                                      type: 'string';
                                      example: 'attio.com';
                                    };
                                    email_local_specifier: {
                                      type: 'string';
                                      example: 'alice';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['email-address'];
                                      description: 'The attribute type of the value.';
                                      example: 'email-address';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'original_email_address',
                                    'email_address',
                                    'email_domain',
                                    'email_root_domain',
                                    'email_local_specifier',
                                    'attribute_type',
                                  ];
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    target_object: {
                                      type: 'string';
                                      description: 'A slug identifying the object that the referenced record belongs to.';
                                      example: 'people';
                                    };
                                    target_record_id: {
                                      type: 'string';
                                      format: 'uuid';
                                      description: 'A UUID to identify the referenced record.';
                                      example: '891dcbfc-9141-415d-9b2a-2238a6cc012d';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['record-reference'];
                                      description: 'The attribute type of the value.';
                                      example: 'record-reference';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'target_object',
                                    'target_record_id',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    interaction_type: {
                                      type: 'string';
                                      enum: ['calendar-event', 'call', 'chat-thread', 'email', 'in-person-meeting'];
                                      description: 'The type of interaction e.g. calendar or email.';
                                      example: 'email';
                                    };
                                    interacted_at: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'When the interaction occurred.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    owner_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['interaction'];
                                      description: 'The attribute type of the value.';
                                      example: 'interaction';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'interaction_type',
                                    'interacted_at',
                                    'owner_actor',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    line_1: {
                                      type: ['string', 'null'];
                                      description: 'The first line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                      example: '1 Infinite Loop';
                                    };
                                    line_2: {
                                      type: ['string', 'null'];
                                      description: 'The second line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                      example: 'Block 1';
                                    };
                                    line_3: {
                                      type: ['string', 'null'];
                                      description: 'The third line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                      example: 'Hilldrop Estate';
                                    };
                                    line_4: {
                                      type: ['string', 'null'];
                                      description: 'The fourth line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                      example: 'Westborough';
                                    };
                                    locality: {
                                      type: ['string', 'null'];
                                      description: 'The town, neighborhood or area the location is in.';
                                      example: 'Cupertino';
                                    };
                                    region: {
                                      type: ['string', 'null'];
                                      description: 'The state, county, province or region that the location is in.';
                                      example: 'CA';
                                    };
                                    postcode: {
                                      type: ['string', 'null'];
                                      description: 'The postcode or zip code for the location. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.}';
                                      example: '95014';
                                    };
                                    country_code: {
                                      type: ['string', 'null'];
                                      enum: [
                                        'AF',
                                        'AX',
                                        'AL',
                                        'DZ',
                                        'AS',
                                        'AD',
                                        'AO',
                                        'AI',
                                        'AQ',
                                        'AG',
                                        'AR',
                                        'AM',
                                        'AW',
                                        'AU',
                                        'AT',
                                        'AZ',
                                        'BS',
                                        'BH',
                                        'BD',
                                        'BB',
                                        'BY',
                                        'BE',
                                        'BZ',
                                        'BJ',
                                        'BM',
                                        'BT',
                                        'BO',
                                        'BA',
                                        'BW',
                                        'BV',
                                        'BR',
                                        'IO',
                                        'BN',
                                        'BG',
                                        'BF',
                                        'BI',
                                        'KH',
                                        'CM',
                                        'CA',
                                        'CV',
                                        'KY',
                                        'CF',
                                        'TD',
                                        'CL',
                                        'CN',
                                        'CX',
                                        'CC',
                                        'CO',
                                        'KM',
                                        'CG',
                                        'CD',
                                        'CK',
                                        'CR',
                                        'CI',
                                        'HR',
                                        'CU',
                                        'CW',
                                        'CY',
                                        'CZ',
                                        'DK',
                                        'DJ',
                                        'DM',
                                        'DO',
                                        'EC',
                                        'EG',
                                        'SV',
                                        'GQ',
                                        'ER',
                                        'EE',
                                        'ET',
                                        'FK',
                                        'FO',
                                        'FJ',
                                        'FI',
                                        'FR',
                                        'GF',
                                        'PF',
                                        'TF',
                                        'GA',
                                        'GM',
                                        'GE',
                                        'DE',
                                        'GH',
                                        'GI',
                                        'GR',
                                        'GL',
                                        'GD',
                                        'GP',
                                        'GU',
                                        'GT',
                                        'GG',
                                        'GN',
                                        'GW',
                                        'GY',
                                        'HT',
                                        'HM',
                                        'VA',
                                        'HN',
                                        'HK',
                                        'HU',
                                        'IS',
                                        'IN',
                                        'ID',
                                        'IR',
                                        'IQ',
                                        'IE',
                                        'IM',
                                        'IL',
                                        'IT',
                                        'JM',
                                        'JP',
                                        'JE',
                                        'JO',
                                        'KZ',
                                        'KE',
                                        'KI',
                                        'KR',
                                        'KW',
                                        'KG',
                                        'LA',
                                        'LV',
                                        'LB',
                                        'LS',
                                        'LR',
                                        'LY',
                                        'LI',
                                        'LT',
                                        'LU',
                                        'MO',
                                        'MK',
                                        'MG',
                                        'MW',
                                        'MY',
                                        'MV',
                                        'ML',
                                        'MT',
                                        'MH',
                                        'MQ',
                                        'MR',
                                        'MU',
                                        'YT',
                                        'MX',
                                        'FM',
                                        'MD',
                                        'MC',
                                        'MN',
                                        'ME',
                                        'MS',
                                        'MA',
                                        'MZ',
                                        'MM',
                                        'NA',
                                        'NR',
                                        'NP',
                                        'NL',
                                        'AN',
                                        'NC',
                                        'NZ',
                                        'NI',
                                        'NE',
                                        'NG',
                                        'NU',
                                        'NF',
                                        'MP',
                                        'NO',
                                        'OM',
                                        'PK',
                                        'PW',
                                        'PS',
                                        'PA',
                                        'PG',
                                        'PY',
                                        'PE',
                                        'PH',
                                        'PN',
                                        'PL',
                                        'PT',
                                        'PR',
                                        'QA',
                                        'RE',
                                        'RO',
                                        'RU',
                                        'RW',
                                        'BL',
                                        'SH',
                                        'KN',
                                        'LC',
                                        'MF',
                                        'PM',
                                        'VC',
                                        'WS',
                                        'SM',
                                        'ST',
                                        'SA',
                                        'SN',
                                        'RS',
                                        'SC',
                                        'SL',
                                        'SG',
                                        'SK',
                                        'SI',
                                        'SB',
                                        'SO',
                                        'ZA',
                                        'GS',
                                        'ES',
                                        'LK',
                                        'SD',
                                        'SR',
                                        'SJ',
                                        'SZ',
                                        'SE',
                                        'CH',
                                        'SY',
                                        'TW',
                                        'TJ',
                                        'TZ',
                                        'TH',
                                        'TL',
                                        'TG',
                                        'TK',
                                        'TO',
                                        'TT',
                                        'TN',
                                        'TR',
                                        'TM',
                                        'TC',
                                        'TV',
                                        'UG',
                                        'UA',
                                        'AE',
                                        'GB',
                                        'US',
                                        'UM',
                                        'UY',
                                        'UZ',
                                        'VU',
                                        'VE',
                                        'VN',
                                        'VG',
                                        'VI',
                                        'WF',
                                        'EH',
                                        'YE',
                                        'ZM',
                                        'ZW',
                                      ];
                                      description: 'The ISO 3166-1 alpha-2 country code for the country this location is in.';
                                      example: 'US';
                                    };
                                    latitude: {
                                      type: ['string', 'null'];
                                      pattern: '^[-+]?([1-8]?\\d(\\.\\d+)?|90(\\.0+)?)$';
                                      description: 'The latitude of the location. Validated by the regular expression `/^[-+]?([1-8]?\\d(\\.\\d+)?|90(\\.0+)?)$/`. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.}';
                                      example: '37.331741';
                                    };
                                    longitude: {
                                      type: ['string', 'null'];
                                      pattern: '^[-+]?(180(\\.0+)?|((1[0-7]\\d)|([1-9]?\\d))(\\.\\d+)?)$';
                                      description: 'The longitude of the location. Validated by the regular expression `/^[-+]?(180(\\.0+)?|((1[0-7]\\d)|([1-9]?\\d))(\\.\\d+)?)$/`';
                                      example: '-122.030333';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['location'];
                                      description: 'The attribute type of the value.';
                                      example: 'location';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'line_1',
                                    'line_2',
                                    'line_3',
                                    'line_4',
                                    'locality',
                                    'region',
                                    'postcode',
                                    'country_code',
                                    'latitude',
                                    'longitude',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    value: {
                                      type: 'number';
                                      description: 'Numbers are persisted as 64 bit floats.';
                                      example: 42;
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['number'];
                                      description: 'The attribute type of the value.';
                                      example: 'number';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'value',
                                    'attribute_type',
                                  ];
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    first_name: {
                                      type: 'string';
                                      description: 'The first name.';
                                      example: 'Ada';
                                    };
                                    last_name: {
                                      type: 'string';
                                      description: 'The last name.';
                                      example: 'Lovelace';
                                    };
                                    full_name: {
                                      type: 'string';
                                      description: 'The full name.';
                                      example: 'Ada Lovelace';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['personal-name'];
                                      description: 'The attribute type of the value.';
                                      example: 'personal-name';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'first_name',
                                    'last_name',
                                    'full_name',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    original_phone_number: {
                                      type: 'string';
                                      description: 'The raw, original phone number, as inputted.';
                                      example: '5558675309';
                                    };
                                    country_code: {
                                      type: 'string';
                                      description: 'The ISO 3166-1 alpha-2 country code representing the country that this phone number belongs to.';
                                      enum: [
                                        'AF',
                                        'AX',
                                        'AL',
                                        'DZ',
                                        'AS',
                                        'AD',
                                        'AO',
                                        'AI',
                                        'AQ',
                                        'AG',
                                        'AR',
                                        'AM',
                                        'AW',
                                        'AU',
                                        'AT',
                                        'AZ',
                                        'BS',
                                        'BH',
                                        'BD',
                                        'BB',
                                        'BY',
                                        'BE',
                                        'BZ',
                                        'BJ',
                                        'BM',
                                        'BT',
                                        'BO',
                                        'BA',
                                        'BW',
                                        'BV',
                                        'BR',
                                        'IO',
                                        'BN',
                                        'BG',
                                        'BF',
                                        'BI',
                                        'KH',
                                        'CM',
                                        'CA',
                                        'CV',
                                        'KY',
                                        'CF',
                                        'TD',
                                        'CL',
                                        'CN',
                                        'CX',
                                        'CC',
                                        'CO',
                                        'KM',
                                        'CG',
                                        'CD',
                                        'CK',
                                        'CR',
                                        'CI',
                                        'HR',
                                        'CU',
                                        'CW',
                                        'CY',
                                        'CZ',
                                        'DK',
                                        'DJ',
                                        'DM',
                                        'DO',
                                        'EC',
                                        'EG',
                                        'SV',
                                        'GQ',
                                        'ER',
                                        'EE',
                                        'ET',
                                        'FK',
                                        'FO',
                                        'FJ',
                                        'FI',
                                        'FR',
                                        'GF',
                                        'PF',
                                        'TF',
                                        'GA',
                                        'GM',
                                        'GE',
                                        'DE',
                                        'GH',
                                        'GI',
                                        'GR',
                                        'GL',
                                        'GD',
                                        'GP',
                                        'GU',
                                        'GT',
                                        'GG',
                                        'GN',
                                        'GW',
                                        'GY',
                                        'HT',
                                        'HM',
                                        'VA',
                                        'HN',
                                        'HK',
                                        'HU',
                                        'IS',
                                        'IN',
                                        'ID',
                                        'IR',
                                        'IQ',
                                        'IE',
                                        'IM',
                                        'IL',
                                        'IT',
                                        'JM',
                                        'JP',
                                        'JE',
                                        'JO',
                                        'KZ',
                                        'KE',
                                        'KI',
                                        'KR',
                                        'KW',
                                        'KG',
                                        'LA',
                                        'LV',
                                        'LB',
                                        'LS',
                                        'LR',
                                        'LY',
                                        'LI',
                                        'LT',
                                        'LU',
                                        'MO',
                                        'MK',
                                        'MG',
                                        'MW',
                                        'MY',
                                        'MV',
                                        'ML',
                                        'MT',
                                        'MH',
                                        'MQ',
                                        'MR',
                                        'MU',
                                        'YT',
                                        'MX',
                                        'FM',
                                        'MD',
                                        'MC',
                                        'MN',
                                        'ME',
                                        'MS',
                                        'MA',
                                        'MZ',
                                        'MM',
                                        'NA',
                                        'NR',
                                        'NP',
                                        'NL',
                                        'AN',
                                        'NC',
                                        'NZ',
                                        'NI',
                                        'NE',
                                        'NG',
                                        'NU',
                                        'NF',
                                        'MP',
                                        'NO',
                                        'OM',
                                        'PK',
                                        'PW',
                                        'PS',
                                        'PA',
                                        'PG',
                                        'PY',
                                        'PE',
                                        'PH',
                                        'PN',
                                        'PL',
                                        'PT',
                                        'PR',
                                        'QA',
                                        'RE',
                                        'RO',
                                        'RU',
                                        'RW',
                                        'BL',
                                        'SH',
                                        'KN',
                                        'LC',
                                        'MF',
                                        'PM',
                                        'VC',
                                        'WS',
                                        'SM',
                                        'ST',
                                        'SA',
                                        'SN',
                                        'RS',
                                        'SC',
                                        'SL',
                                        'SG',
                                        'SK',
                                        'SI',
                                        'SB',
                                        'SO',
                                        'ZA',
                                        'GS',
                                        'ES',
                                        'LK',
                                        'SD',
                                        'SR',
                                        'SJ',
                                        'SZ',
                                        'SE',
                                        'CH',
                                        'SY',
                                        'TW',
                                        'TJ',
                                        'TZ',
                                        'TH',
                                        'TL',
                                        'TG',
                                        'TK',
                                        'TO',
                                        'TT',
                                        'TN',
                                        'TR',
                                        'TM',
                                        'TC',
                                        'TV',
                                        'UG',
                                        'UA',
                                        'AE',
                                        'GB',
                                        'US',
                                        'UM',
                                        'UY',
                                        'UZ',
                                        'VU',
                                        'VE',
                                        'VN',
                                        'VG',
                                        'VI',
                                        'WF',
                                        'EH',
                                        'YE',
                                        'ZM',
                                        'ZW',
                                      ];
                                      example: 'US';
                                    };
                                    phone_number: {
                                      type: 'string';
                                      example: '+15558675309';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['phone-number'];
                                      description: 'The attribute type of the value.';
                                      example: 'phone-number';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'original_phone_number',
                                    'country_code',
                                    'phone_number',
                                    'attribute_type',
                                  ];
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    status: {
                                      $ref: '#/components/schemas/status';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['status'];
                                      description: 'The attribute type of the value.';
                                      example: 'status';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'status',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    value: {
                                      type: 'number';
                                      description: 'A number between 0 and 5 (inclusive) to represent a star rating.';
                                      example: 3;
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['rating'];
                                      description: 'The attribute type of the value.';
                                      example: 'rating';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'value',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    option: {
                                      $ref: '#/components/schemas/select-option';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['select'];
                                      description: 'The attribute type of the value.';
                                      example: 'select';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'option',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    value: {
                                      type: 'string';
                                      description: 'A raw text field. Values are limited to 10MB.';
                                      example: 'Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['text'];
                                      description: 'The attribute type of the value.';
                                      example: 'text';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'value',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['timestamp'];
                                      description: 'The attribute type of the value.';
                                      example: 'timestamp';
                                    };
                                    value: {
                                      type: 'string';
                                      description: 'A timestamp value represents a single, universal moment in time using an ISO 8601 formatted string. This means that a timestamp consists of a date, a time (with nanosecond precision), and a time zone. Attio will coerce timestamps which do not provide full nanosecond precision and UTC is assumed if no time zone is provided. For example, "2023", "2023-01", "2023-01-02", "2023-01-02T13:00", "2023-01-02T13:00:00", and "2023-01-02T13:00:00.000000000" will all be coerced to "2023-01-02T13:00:00.000000000Z". Timestamps are always returned in UTC. For example, writing a timestamp value using the string "2023-01-02T13:00:00.000000000+02:00" will result in the value "2023-01-02T11:00:00.000000000Z" being returned. The maximum date is "9999-12-31T23:59:59.999999999Z".';
                                      format: 'date';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'attribute_type',
                                    'value',
                                  ];
                                  additionalProperties: false;
                                },
                              ];
                            };
                          };
                          description: 'A record type with an attribute `api_slug` as the key, and an array of value objects as the values.';
                        };
                      };
                      required: ['id', 'created_at', 'values'];
                    };
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '400': {
          description: 'Bad Request';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [400];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['filter_error'];
                  };
                  message: {
                    type: 'string';
                    example: 'Error in filter.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Bad Request';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'The referenced object was not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
      };
    };
  };
  '/v2/objects/{object}/records': {
    post: {
      summary: 'Create a record';
      description: 'Creates a new person, company or other record. This endpoint will throw on conflicts of unique attributes. If you would prefer to update records on conflicts, please use the [Assert record endpoint](/reference/put_v2-objects-object-records) instead.\n\nRequired scopes: `record_permission:read-write`, `object_configuration:read`.';
      tags: ['Records'];
      security: [
        {
          oauth2: ['record_permission:read-write', 'object_configuration:read'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            description: 'The UUID or slug identifying the object the created record should belong to.';
            example: 'people';
          };
          required: true;
          name: 'object';
          in: 'path';
        },
      ];
      requestBody: {
        required: true;
        content: {
          'application/json': {
            schema: {
              type: 'object';
              properties: {
                data: {
                  type: 'object';
                  properties: {
                    values: {
                      type: 'object';
                      description: 'An object with an attribute `api_slug` or `attribute_id` as the key, and a single value (for single-select attributes), or an array of values (for single or multi-select attributes) as the values. For complete documentation on values for all attribute types, please see our [attribute type docs](/docs/attribute-types).';
                      additionalProperties: {
                        type: 'array';
                      };
                      example: {
                        '41252299-f8c7-4b5e-99c9-4ff8321d2f96': 'Text value';
                        multiselect_attribute: ['Select option 1', 'Select option 2'];
                      };
                    };
                  };
                  required: ['values'];
                };
              };
              required: ['data'];
            };
          };
        };
      };
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    type: 'object';
                    properties: {
                      id: {
                        type: 'object';
                        properties: {
                          workspace_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying the workspace this record belongs to.';
                            example: '14beef7a-99f7-4534-a87e-70b564330a4c';
                          };
                          object_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying the object this record belongs to.';
                            example: '97052eb9-e65e-443f-a297-f2d9a4a7f795';
                          };
                          record_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying this record.';
                            example: 'bf071e1f-6035-429d-b874-d83ea64ea13b';
                          };
                        };
                        required: ['workspace_id', 'object_id', 'record_id'];
                      };
                      created_at: {
                        type: 'string';
                        description: 'When this record was created.';
                        example: '2022-11-21T13:22:49.061281000Z';
                      };
                      values: {
                        type: 'object';
                        additionalProperties: {
                          type: 'array';
                          items: {
                            oneOf: [
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  referenced_actor_type: {
                                    type: 'string';
                                    enum: ['api-token', 'workspace-member', 'system'];
                                    description: 'The type of the referenced actor. [Read more information on actor types here](/docs/actors).';
                                    example: 'workspace-member';
                                  };
                                  referenced_actor_id: {
                                    type: ['string', 'null'];
                                    format: 'uuid';
                                    description: 'The ID of the referenced actor.';
                                    example: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['actor-reference'];
                                    description: 'The attribute type of the value.';
                                    example: 'actor-reference';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'referenced_actor_type',
                                  'referenced_actor_id',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'boolean';
                                    description: "A boolean representing whether the checkbox is checked or not. The string values 'true' and 'false' are also accepted.";
                                    example: true;
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['checkbox'];
                                    description: 'The attribute type of the value.';
                                    example: 'checkbox';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  currency_value: {
                                    type: 'number';
                                    description: 'A numerical representation of the currency value. A decimal with a max of 4 decimal places.';
                                    example: 99;
                                  };
                                  currency_code: {
                                    type: ['string', 'null'];
                                    enum: [
                                      'EUR',
                                      'GBP',
                                      'USD',
                                      'CAD',
                                      'AUD',
                                      'NZD',
                                      'AED',
                                      'INR',
                                      'CHF',
                                      'JPY',
                                      'CNY',
                                      'PHP',
                                      'ILS',
                                      'SGD',
                                      'HKD',
                                      'MYR',
                                      'NTD',
                                      'NOK',
                                      'SEK',
                                      'ISK',
                                      'BEL',
                                      'PLN',
                                      'KRW',
                                      'DKK',
                                      'BRL',
                                    ];
                                    description: 'The ISO4217 currency code representing the currency that the value is stored in.';
                                    example: 'USD';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['currency'];
                                    description: 'The attribute type of the value.';
                                    example: 'currency';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'currency_value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['date'];
                                    description: 'The attribute type of the value.';
                                    example: 'date';
                                  };
                                  value: {
                                    type: 'string';
                                    description: 'A date represents a single calendar year, month and day, independent of timezone. If hours, months, seconds or timezones are provided, they will be trimmed. For example, "2023" and "2023-01" will be coerced into "2023-01-01", and "2023-01-02", "2023-01-02T13:00", "2023-01-02T14:00:00", "2023-01-02T15:00:00.000000000", and "2023-01-02T15:00:00.000000000+02:00" will all be coerced to "2023-01-02". If a timezone is provided that would result in a different calendar date in UTC, the date will be coerced to UTC and then the timezone component will be trimmed. For example, the value "2023-01-02T23:00:00-10:00" will be returned as "2023-01-03". The maximum date is "9999-12-31".';
                                    example: '2023-01-01';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'attribute_type',
                                  'value',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  domain: {
                                    type: 'string';
                                    example: 'app.attio.com';
                                  };
                                  root_domain: {
                                    type: 'string';
                                    example: 'attio.com';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['domain'];
                                    description: 'The attribute type of the value.';
                                    example: 'domain';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'domain',
                                  'root_domain',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  original_email_address: {
                                    type: 'string';
                                    example: 'alice@app.attio.com';
                                  };
                                  email_address: {
                                    type: 'string';
                                    example: 'alice@app.attio.com';
                                  };
                                  email_domain: {
                                    type: 'string';
                                    example: 'app.attio.com';
                                  };
                                  email_root_domain: {
                                    type: 'string';
                                    example: 'attio.com';
                                  };
                                  email_local_specifier: {
                                    type: 'string';
                                    example: 'alice';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['email-address'];
                                    description: 'The attribute type of the value.';
                                    example: 'email-address';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'original_email_address',
                                  'email_address',
                                  'email_domain',
                                  'email_root_domain',
                                  'email_local_specifier',
                                  'attribute_type',
                                ];
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  target_object: {
                                    type: 'string';
                                    description: 'A slug identifying the object that the referenced record belongs to.';
                                    example: 'people';
                                  };
                                  target_record_id: {
                                    type: 'string';
                                    format: 'uuid';
                                    description: 'A UUID to identify the referenced record.';
                                    example: '891dcbfc-9141-415d-9b2a-2238a6cc012d';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['record-reference'];
                                    description: 'The attribute type of the value.';
                                    example: 'record-reference';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'target_object',
                                  'target_record_id',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  interaction_type: {
                                    type: 'string';
                                    enum: ['calendar-event', 'call', 'chat-thread', 'email', 'in-person-meeting'];
                                    description: 'The type of interaction e.g. calendar or email.';
                                    example: 'email';
                                  };
                                  interacted_at: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'When the interaction occurred.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  owner_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['interaction'];
                                    description: 'The attribute type of the value.';
                                    example: 'interaction';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'interaction_type',
                                  'interacted_at',
                                  'owner_actor',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  line_1: {
                                    type: ['string', 'null'];
                                    description: 'The first line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: '1 Infinite Loop';
                                  };
                                  line_2: {
                                    type: ['string', 'null'];
                                    description: 'The second line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: 'Block 1';
                                  };
                                  line_3: {
                                    type: ['string', 'null'];
                                    description: 'The third line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: 'Hilldrop Estate';
                                  };
                                  line_4: {
                                    type: ['string', 'null'];
                                    description: 'The fourth line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: 'Westborough';
                                  };
                                  locality: {
                                    type: ['string', 'null'];
                                    description: 'The town, neighborhood or area the location is in.';
                                    example: 'Cupertino';
                                  };
                                  region: {
                                    type: ['string', 'null'];
                                    description: 'The state, county, province or region that the location is in.';
                                    example: 'CA';
                                  };
                                  postcode: {
                                    type: ['string', 'null'];
                                    description: 'The postcode or zip code for the location. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.}';
                                    example: '95014';
                                  };
                                  country_code: {
                                    type: ['string', 'null'];
                                    enum: [
                                      'AF',
                                      'AX',
                                      'AL',
                                      'DZ',
                                      'AS',
                                      'AD',
                                      'AO',
                                      'AI',
                                      'AQ',
                                      'AG',
                                      'AR',
                                      'AM',
                                      'AW',
                                      'AU',
                                      'AT',
                                      'AZ',
                                      'BS',
                                      'BH',
                                      'BD',
                                      'BB',
                                      'BY',
                                      'BE',
                                      'BZ',
                                      'BJ',
                                      'BM',
                                      'BT',
                                      'BO',
                                      'BA',
                                      'BW',
                                      'BV',
                                      'BR',
                                      'IO',
                                      'BN',
                                      'BG',
                                      'BF',
                                      'BI',
                                      'KH',
                                      'CM',
                                      'CA',
                                      'CV',
                                      'KY',
                                      'CF',
                                      'TD',
                                      'CL',
                                      'CN',
                                      'CX',
                                      'CC',
                                      'CO',
                                      'KM',
                                      'CG',
                                      'CD',
                                      'CK',
                                      'CR',
                                      'CI',
                                      'HR',
                                      'CU',
                                      'CW',
                                      'CY',
                                      'CZ',
                                      'DK',
                                      'DJ',
                                      'DM',
                                      'DO',
                                      'EC',
                                      'EG',
                                      'SV',
                                      'GQ',
                                      'ER',
                                      'EE',
                                      'ET',
                                      'FK',
                                      'FO',
                                      'FJ',
                                      'FI',
                                      'FR',
                                      'GF',
                                      'PF',
                                      'TF',
                                      'GA',
                                      'GM',
                                      'GE',
                                      'DE',
                                      'GH',
                                      'GI',
                                      'GR',
                                      'GL',
                                      'GD',
                                      'GP',
                                      'GU',
                                      'GT',
                                      'GG',
                                      'GN',
                                      'GW',
                                      'GY',
                                      'HT',
                                      'HM',
                                      'VA',
                                      'HN',
                                      'HK',
                                      'HU',
                                      'IS',
                                      'IN',
                                      'ID',
                                      'IR',
                                      'IQ',
                                      'IE',
                                      'IM',
                                      'IL',
                                      'IT',
                                      'JM',
                                      'JP',
                                      'JE',
                                      'JO',
                                      'KZ',
                                      'KE',
                                      'KI',
                                      'KR',
                                      'KW',
                                      'KG',
                                      'LA',
                                      'LV',
                                      'LB',
                                      'LS',
                                      'LR',
                                      'LY',
                                      'LI',
                                      'LT',
                                      'LU',
                                      'MO',
                                      'MK',
                                      'MG',
                                      'MW',
                                      'MY',
                                      'MV',
                                      'ML',
                                      'MT',
                                      'MH',
                                      'MQ',
                                      'MR',
                                      'MU',
                                      'YT',
                                      'MX',
                                      'FM',
                                      'MD',
                                      'MC',
                                      'MN',
                                      'ME',
                                      'MS',
                                      'MA',
                                      'MZ',
                                      'MM',
                                      'NA',
                                      'NR',
                                      'NP',
                                      'NL',
                                      'AN',
                                      'NC',
                                      'NZ',
                                      'NI',
                                      'NE',
                                      'NG',
                                      'NU',
                                      'NF',
                                      'MP',
                                      'NO',
                                      'OM',
                                      'PK',
                                      'PW',
                                      'PS',
                                      'PA',
                                      'PG',
                                      'PY',
                                      'PE',
                                      'PH',
                                      'PN',
                                      'PL',
                                      'PT',
                                      'PR',
                                      'QA',
                                      'RE',
                                      'RO',
                                      'RU',
                                      'RW',
                                      'BL',
                                      'SH',
                                      'KN',
                                      'LC',
                                      'MF',
                                      'PM',
                                      'VC',
                                      'WS',
                                      'SM',
                                      'ST',
                                      'SA',
                                      'SN',
                                      'RS',
                                      'SC',
                                      'SL',
                                      'SG',
                                      'SK',
                                      'SI',
                                      'SB',
                                      'SO',
                                      'ZA',
                                      'GS',
                                      'ES',
                                      'LK',
                                      'SD',
                                      'SR',
                                      'SJ',
                                      'SZ',
                                      'SE',
                                      'CH',
                                      'SY',
                                      'TW',
                                      'TJ',
                                      'TZ',
                                      'TH',
                                      'TL',
                                      'TG',
                                      'TK',
                                      'TO',
                                      'TT',
                                      'TN',
                                      'TR',
                                      'TM',
                                      'TC',
                                      'TV',
                                      'UG',
                                      'UA',
                                      'AE',
                                      'GB',
                                      'US',
                                      'UM',
                                      'UY',
                                      'UZ',
                                      'VU',
                                      'VE',
                                      'VN',
                                      'VG',
                                      'VI',
                                      'WF',
                                      'EH',
                                      'YE',
                                      'ZM',
                                      'ZW',
                                    ];
                                    description: 'The ISO 3166-1 alpha-2 country code for the country this location is in.';
                                    example: 'US';
                                  };
                                  latitude: {
                                    type: ['string', 'null'];
                                    pattern: '^[-+]?([1-8]?\\d(\\.\\d+)?|90(\\.0+)?)$';
                                    description: 'The latitude of the location. Validated by the regular expression `/^[-+]?([1-8]?\\d(\\.\\d+)?|90(\\.0+)?)$/`. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.}';
                                    example: '37.331741';
                                  };
                                  longitude: {
                                    type: ['string', 'null'];
                                    pattern: '^[-+]?(180(\\.0+)?|((1[0-7]\\d)|([1-9]?\\d))(\\.\\d+)?)$';
                                    description: 'The longitude of the location. Validated by the regular expression `/^[-+]?(180(\\.0+)?|((1[0-7]\\d)|([1-9]?\\d))(\\.\\d+)?)$/`';
                                    example: '-122.030333';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['location'];
                                    description: 'The attribute type of the value.';
                                    example: 'location';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'line_1',
                                  'line_2',
                                  'line_3',
                                  'line_4',
                                  'locality',
                                  'region',
                                  'postcode',
                                  'country_code',
                                  'latitude',
                                  'longitude',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'number';
                                    description: 'Numbers are persisted as 64 bit floats.';
                                    example: 42;
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['number'];
                                    description: 'The attribute type of the value.';
                                    example: 'number';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  first_name: {
                                    type: 'string';
                                    description: 'The first name.';
                                    example: 'Ada';
                                  };
                                  last_name: {
                                    type: 'string';
                                    description: 'The last name.';
                                    example: 'Lovelace';
                                  };
                                  full_name: {
                                    type: 'string';
                                    description: 'The full name.';
                                    example: 'Ada Lovelace';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['personal-name'];
                                    description: 'The attribute type of the value.';
                                    example: 'personal-name';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'first_name',
                                  'last_name',
                                  'full_name',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  original_phone_number: {
                                    type: 'string';
                                    description: 'The raw, original phone number, as inputted.';
                                    example: '5558675309';
                                  };
                                  country_code: {
                                    type: 'string';
                                    description: 'The ISO 3166-1 alpha-2 country code representing the country that this phone number belongs to.';
                                    enum: [
                                      'AF',
                                      'AX',
                                      'AL',
                                      'DZ',
                                      'AS',
                                      'AD',
                                      'AO',
                                      'AI',
                                      'AQ',
                                      'AG',
                                      'AR',
                                      'AM',
                                      'AW',
                                      'AU',
                                      'AT',
                                      'AZ',
                                      'BS',
                                      'BH',
                                      'BD',
                                      'BB',
                                      'BY',
                                      'BE',
                                      'BZ',
                                      'BJ',
                                      'BM',
                                      'BT',
                                      'BO',
                                      'BA',
                                      'BW',
                                      'BV',
                                      'BR',
                                      'IO',
                                      'BN',
                                      'BG',
                                      'BF',
                                      'BI',
                                      'KH',
                                      'CM',
                                      'CA',
                                      'CV',
                                      'KY',
                                      'CF',
                                      'TD',
                                      'CL',
                                      'CN',
                                      'CX',
                                      'CC',
                                      'CO',
                                      'KM',
                                      'CG',
                                      'CD',
                                      'CK',
                                      'CR',
                                      'CI',
                                      'HR',
                                      'CU',
                                      'CW',
                                      'CY',
                                      'CZ',
                                      'DK',
                                      'DJ',
                                      'DM',
                                      'DO',
                                      'EC',
                                      'EG',
                                      'SV',
                                      'GQ',
                                      'ER',
                                      'EE',
                                      'ET',
                                      'FK',
                                      'FO',
                                      'FJ',
                                      'FI',
                                      'FR',
                                      'GF',
                                      'PF',
                                      'TF',
                                      'GA',
                                      'GM',
                                      'GE',
                                      'DE',
                                      'GH',
                                      'GI',
                                      'GR',
                                      'GL',
                                      'GD',
                                      'GP',
                                      'GU',
                                      'GT',
                                      'GG',
                                      'GN',
                                      'GW',
                                      'GY',
                                      'HT',
                                      'HM',
                                      'VA',
                                      'HN',
                                      'HK',
                                      'HU',
                                      'IS',
                                      'IN',
                                      'ID',
                                      'IR',
                                      'IQ',
                                      'IE',
                                      'IM',
                                      'IL',
                                      'IT',
                                      'JM',
                                      'JP',
                                      'JE',
                                      'JO',
                                      'KZ',
                                      'KE',
                                      'KI',
                                      'KR',
                                      'KW',
                                      'KG',
                                      'LA',
                                      'LV',
                                      'LB',
                                      'LS',
                                      'LR',
                                      'LY',
                                      'LI',
                                      'LT',
                                      'LU',
                                      'MO',
                                      'MK',
                                      'MG',
                                      'MW',
                                      'MY',
                                      'MV',
                                      'ML',
                                      'MT',
                                      'MH',
                                      'MQ',
                                      'MR',
                                      'MU',
                                      'YT',
                                      'MX',
                                      'FM',
                                      'MD',
                                      'MC',
                                      'MN',
                                      'ME',
                                      'MS',
                                      'MA',
                                      'MZ',
                                      'MM',
                                      'NA',
                                      'NR',
                                      'NP',
                                      'NL',
                                      'AN',
                                      'NC',
                                      'NZ',
                                      'NI',
                                      'NE',
                                      'NG',
                                      'NU',
                                      'NF',
                                      'MP',
                                      'NO',
                                      'OM',
                                      'PK',
                                      'PW',
                                      'PS',
                                      'PA',
                                      'PG',
                                      'PY',
                                      'PE',
                                      'PH',
                                      'PN',
                                      'PL',
                                      'PT',
                                      'PR',
                                      'QA',
                                      'RE',
                                      'RO',
                                      'RU',
                                      'RW',
                                      'BL',
                                      'SH',
                                      'KN',
                                      'LC',
                                      'MF',
                                      'PM',
                                      'VC',
                                      'WS',
                                      'SM',
                                      'ST',
                                      'SA',
                                      'SN',
                                      'RS',
                                      'SC',
                                      'SL',
                                      'SG',
                                      'SK',
                                      'SI',
                                      'SB',
                                      'SO',
                                      'ZA',
                                      'GS',
                                      'ES',
                                      'LK',
                                      'SD',
                                      'SR',
                                      'SJ',
                                      'SZ',
                                      'SE',
                                      'CH',
                                      'SY',
                                      'TW',
                                      'TJ',
                                      'TZ',
                                      'TH',
                                      'TL',
                                      'TG',
                                      'TK',
                                      'TO',
                                      'TT',
                                      'TN',
                                      'TR',
                                      'TM',
                                      'TC',
                                      'TV',
                                      'UG',
                                      'UA',
                                      'AE',
                                      'GB',
                                      'US',
                                      'UM',
                                      'UY',
                                      'UZ',
                                      'VU',
                                      'VE',
                                      'VN',
                                      'VG',
                                      'VI',
                                      'WF',
                                      'EH',
                                      'YE',
                                      'ZM',
                                      'ZW',
                                    ];
                                    example: 'US';
                                  };
                                  phone_number: {
                                    type: 'string';
                                    example: '+15558675309';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['phone-number'];
                                    description: 'The attribute type of the value.';
                                    example: 'phone-number';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'original_phone_number',
                                  'country_code',
                                  'phone_number',
                                  'attribute_type',
                                ];
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  status: {
                                    $ref: '#/components/schemas/status';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['status'];
                                    description: 'The attribute type of the value.';
                                    example: 'status';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'status',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'number';
                                    description: 'A number between 0 and 5 (inclusive) to represent a star rating.';
                                    example: 3;
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['rating'];
                                    description: 'The attribute type of the value.';
                                    example: 'rating';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  option: {
                                    $ref: '#/components/schemas/select-option';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['select'];
                                    description: 'The attribute type of the value.';
                                    example: 'select';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'option',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'string';
                                    description: 'A raw text field. Values are limited to 10MB.';
                                    example: 'Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['text'];
                                    description: 'The attribute type of the value.';
                                    example: 'text';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['timestamp'];
                                    description: 'The attribute type of the value.';
                                    example: 'timestamp';
                                  };
                                  value: {
                                    type: 'string';
                                    description: 'A timestamp value represents a single, universal moment in time using an ISO 8601 formatted string. This means that a timestamp consists of a date, a time (with nanosecond precision), and a time zone. Attio will coerce timestamps which do not provide full nanosecond precision and UTC is assumed if no time zone is provided. For example, "2023", "2023-01", "2023-01-02", "2023-01-02T13:00", "2023-01-02T13:00:00", and "2023-01-02T13:00:00.000000000" will all be coerced to "2023-01-02T13:00:00.000000000Z". Timestamps are always returned in UTC. For example, writing a timestamp value using the string "2023-01-02T13:00:00.000000000+02:00" will result in the value "2023-01-02T11:00:00.000000000Z" being returned. The maximum date is "9999-12-31T23:59:59.999999999Z".';
                                    format: 'date';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'attribute_type',
                                  'value',
                                ];
                                additionalProperties: false;
                              },
                            ];
                          };
                        };
                        description: 'A record type with an attribute `api_slug` as the key, and an array of value objects as the values.';
                      };
                    };
                    required: ['id', 'created_at', 'values'];
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '400': {
          description: 'Bad Request';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [400];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['value_not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Cannot find select attribute with select option title "In Progress".';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Bad Request';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Object with slug/ID "people" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
      };
    };
    put: {
      summary: 'Assert a record';
      description: 'Use this endpoint to create or update people, companies and other records. A matching attribute is used to search for existing records. If a record is found with the same value for the matching attribute, that record will be updated. If no record with the same value for the matching attribute is found, a new record will be created instead. If you would like to avoid matching, please use the [Create record endpoint](/reference/post_v2-objects-object-records).\n\nIf the matching attribute is a multiselect attribute, new values will be added and existing values will not be deleted. For any other multiselect attribute, all values will be either created or deleted as necessary to match the list of supplied values.\n\nRequired scopes: `record_permission:read-write`, `object_configuration:read`.';
      tags: ['Records'];
      security: [
        {
          oauth2: ['record_permission:read-write', 'object_configuration:read'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the object the record should belong to.';
            example: 'people';
          };
          required: true;
          name: 'object';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'The ID or slug of the attribute to use to check if a record already exists. The attribute must be unique.';
            example: '41252299-f8c7-4b5e-99c9-4ff8321d2f96';
          };
          required: true;
          name: 'matching_attribute';
          in: 'query';
        },
      ];
      requestBody: {
        required: true;
        content: {
          'application/json': {
            schema: {
              type: 'object';
              properties: {
                data: {
                  type: 'object';
                  properties: {
                    values: {
                      type: 'object';
                      description: 'An object with an attribute `api_slug` or `attribute_id` as the key, and a single value (for single-select attributes), or an array of values (for single or multi-select attributes) as the values. For complete documentation on values for all attribute types, please see our [attribute type docs](/docs/attribute-types).';
                      additionalProperties: {
                        type: 'array';
                      };
                      example: {
                        '41252299-f8c7-4b5e-99c9-4ff8321d2f96': 'Text value';
                        multiselect_attribute: ['Select option 1', 'Select option 2'];
                      };
                    };
                  };
                  required: ['values'];
                  additionalProperties: false;
                };
              };
              required: ['data'];
            };
          };
        };
      };
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    type: 'object';
                    properties: {
                      id: {
                        type: 'object';
                        properties: {
                          workspace_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying the workspace this record belongs to.';
                            example: '14beef7a-99f7-4534-a87e-70b564330a4c';
                          };
                          object_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying the object this record belongs to.';
                            example: '97052eb9-e65e-443f-a297-f2d9a4a7f795';
                          };
                          record_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying this record.';
                            example: 'bf071e1f-6035-429d-b874-d83ea64ea13b';
                          };
                        };
                        required: ['workspace_id', 'object_id', 'record_id'];
                      };
                      created_at: {
                        type: 'string';
                        description: 'When this record was created.';
                        example: '2022-11-21T13:22:49.061281000Z';
                      };
                      values: {
                        type: 'object';
                        additionalProperties: {
                          type: 'array';
                          items: {
                            oneOf: [
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  referenced_actor_type: {
                                    type: 'string';
                                    enum: ['api-token', 'workspace-member', 'system'];
                                    description: 'The type of the referenced actor. [Read more information on actor types here](/docs/actors).';
                                    example: 'workspace-member';
                                  };
                                  referenced_actor_id: {
                                    type: ['string', 'null'];
                                    format: 'uuid';
                                    description: 'The ID of the referenced actor.';
                                    example: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['actor-reference'];
                                    description: 'The attribute type of the value.';
                                    example: 'actor-reference';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'referenced_actor_type',
                                  'referenced_actor_id',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'boolean';
                                    description: "A boolean representing whether the checkbox is checked or not. The string values 'true' and 'false' are also accepted.";
                                    example: true;
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['checkbox'];
                                    description: 'The attribute type of the value.';
                                    example: 'checkbox';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  currency_value: {
                                    type: 'number';
                                    description: 'A numerical representation of the currency value. A decimal with a max of 4 decimal places.';
                                    example: 99;
                                  };
                                  currency_code: {
                                    type: ['string', 'null'];
                                    enum: [
                                      'EUR',
                                      'GBP',
                                      'USD',
                                      'CAD',
                                      'AUD',
                                      'NZD',
                                      'AED',
                                      'INR',
                                      'CHF',
                                      'JPY',
                                      'CNY',
                                      'PHP',
                                      'ILS',
                                      'SGD',
                                      'HKD',
                                      'MYR',
                                      'NTD',
                                      'NOK',
                                      'SEK',
                                      'ISK',
                                      'BEL',
                                      'PLN',
                                      'KRW',
                                      'DKK',
                                      'BRL',
                                    ];
                                    description: 'The ISO4217 currency code representing the currency that the value is stored in.';
                                    example: 'USD';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['currency'];
                                    description: 'The attribute type of the value.';
                                    example: 'currency';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'currency_value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['date'];
                                    description: 'The attribute type of the value.';
                                    example: 'date';
                                  };
                                  value: {
                                    type: 'string';
                                    description: 'A date represents a single calendar year, month and day, independent of timezone. If hours, months, seconds or timezones are provided, they will be trimmed. For example, "2023" and "2023-01" will be coerced into "2023-01-01", and "2023-01-02", "2023-01-02T13:00", "2023-01-02T14:00:00", "2023-01-02T15:00:00.000000000", and "2023-01-02T15:00:00.000000000+02:00" will all be coerced to "2023-01-02". If a timezone is provided that would result in a different calendar date in UTC, the date will be coerced to UTC and then the timezone component will be trimmed. For example, the value "2023-01-02T23:00:00-10:00" will be returned as "2023-01-03". The maximum date is "9999-12-31".';
                                    example: '2023-01-01';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'attribute_type',
                                  'value',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  domain: {
                                    type: 'string';
                                    example: 'app.attio.com';
                                  };
                                  root_domain: {
                                    type: 'string';
                                    example: 'attio.com';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['domain'];
                                    description: 'The attribute type of the value.';
                                    example: 'domain';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'domain',
                                  'root_domain',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  original_email_address: {
                                    type: 'string';
                                    example: 'alice@app.attio.com';
                                  };
                                  email_address: {
                                    type: 'string';
                                    example: 'alice@app.attio.com';
                                  };
                                  email_domain: {
                                    type: 'string';
                                    example: 'app.attio.com';
                                  };
                                  email_root_domain: {
                                    type: 'string';
                                    example: 'attio.com';
                                  };
                                  email_local_specifier: {
                                    type: 'string';
                                    example: 'alice';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['email-address'];
                                    description: 'The attribute type of the value.';
                                    example: 'email-address';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'original_email_address',
                                  'email_address',
                                  'email_domain',
                                  'email_root_domain',
                                  'email_local_specifier',
                                  'attribute_type',
                                ];
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  target_object: {
                                    type: 'string';
                                    description: 'A slug identifying the object that the referenced record belongs to.';
                                    example: 'people';
                                  };
                                  target_record_id: {
                                    type: 'string';
                                    format: 'uuid';
                                    description: 'A UUID to identify the referenced record.';
                                    example: '891dcbfc-9141-415d-9b2a-2238a6cc012d';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['record-reference'];
                                    description: 'The attribute type of the value.';
                                    example: 'record-reference';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'target_object',
                                  'target_record_id',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  interaction_type: {
                                    type: 'string';
                                    enum: ['calendar-event', 'call', 'chat-thread', 'email', 'in-person-meeting'];
                                    description: 'The type of interaction e.g. calendar or email.';
                                    example: 'email';
                                  };
                                  interacted_at: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'When the interaction occurred.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  owner_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['interaction'];
                                    description: 'The attribute type of the value.';
                                    example: 'interaction';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'interaction_type',
                                  'interacted_at',
                                  'owner_actor',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  line_1: {
                                    type: ['string', 'null'];
                                    description: 'The first line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: '1 Infinite Loop';
                                  };
                                  line_2: {
                                    type: ['string', 'null'];
                                    description: 'The second line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: 'Block 1';
                                  };
                                  line_3: {
                                    type: ['string', 'null'];
                                    description: 'The third line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: 'Hilldrop Estate';
                                  };
                                  line_4: {
                                    type: ['string', 'null'];
                                    description: 'The fourth line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: 'Westborough';
                                  };
                                  locality: {
                                    type: ['string', 'null'];
                                    description: 'The town, neighborhood or area the location is in.';
                                    example: 'Cupertino';
                                  };
                                  region: {
                                    type: ['string', 'null'];
                                    description: 'The state, county, province or region that the location is in.';
                                    example: 'CA';
                                  };
                                  postcode: {
                                    type: ['string', 'null'];
                                    description: 'The postcode or zip code for the location. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.}';
                                    example: '95014';
                                  };
                                  country_code: {
                                    type: ['string', 'null'];
                                    enum: [
                                      'AF',
                                      'AX',
                                      'AL',
                                      'DZ',
                                      'AS',
                                      'AD',
                                      'AO',
                                      'AI',
                                      'AQ',
                                      'AG',
                                      'AR',
                                      'AM',
                                      'AW',
                                      'AU',
                                      'AT',
                                      'AZ',
                                      'BS',
                                      'BH',
                                      'BD',
                                      'BB',
                                      'BY',
                                      'BE',
                                      'BZ',
                                      'BJ',
                                      'BM',
                                      'BT',
                                      'BO',
                                      'BA',
                                      'BW',
                                      'BV',
                                      'BR',
                                      'IO',
                                      'BN',
                                      'BG',
                                      'BF',
                                      'BI',
                                      'KH',
                                      'CM',
                                      'CA',
                                      'CV',
                                      'KY',
                                      'CF',
                                      'TD',
                                      'CL',
                                      'CN',
                                      'CX',
                                      'CC',
                                      'CO',
                                      'KM',
                                      'CG',
                                      'CD',
                                      'CK',
                                      'CR',
                                      'CI',
                                      'HR',
                                      'CU',
                                      'CW',
                                      'CY',
                                      'CZ',
                                      'DK',
                                      'DJ',
                                      'DM',
                                      'DO',
                                      'EC',
                                      'EG',
                                      'SV',
                                      'GQ',
                                      'ER',
                                      'EE',
                                      'ET',
                                      'FK',
                                      'FO',
                                      'FJ',
                                      'FI',
                                      'FR',
                                      'GF',
                                      'PF',
                                      'TF',
                                      'GA',
                                      'GM',
                                      'GE',
                                      'DE',
                                      'GH',
                                      'GI',
                                      'GR',
                                      'GL',
                                      'GD',
                                      'GP',
                                      'GU',
                                      'GT',
                                      'GG',
                                      'GN',
                                      'GW',
                                      'GY',
                                      'HT',
                                      'HM',
                                      'VA',
                                      'HN',
                                      'HK',
                                      'HU',
                                      'IS',
                                      'IN',
                                      'ID',
                                      'IR',
                                      'IQ',
                                      'IE',
                                      'IM',
                                      'IL',
                                      'IT',
                                      'JM',
                                      'JP',
                                      'JE',
                                      'JO',
                                      'KZ',
                                      'KE',
                                      'KI',
                                      'KR',
                                      'KW',
                                      'KG',
                                      'LA',
                                      'LV',
                                      'LB',
                                      'LS',
                                      'LR',
                                      'LY',
                                      'LI',
                                      'LT',
                                      'LU',
                                      'MO',
                                      'MK',
                                      'MG',
                                      'MW',
                                      'MY',
                                      'MV',
                                      'ML',
                                      'MT',
                                      'MH',
                                      'MQ',
                                      'MR',
                                      'MU',
                                      'YT',
                                      'MX',
                                      'FM',
                                      'MD',
                                      'MC',
                                      'MN',
                                      'ME',
                                      'MS',
                                      'MA',
                                      'MZ',
                                      'MM',
                                      'NA',
                                      'NR',
                                      'NP',
                                      'NL',
                                      'AN',
                                      'NC',
                                      'NZ',
                                      'NI',
                                      'NE',
                                      'NG',
                                      'NU',
                                      'NF',
                                      'MP',
                                      'NO',
                                      'OM',
                                      'PK',
                                      'PW',
                                      'PS',
                                      'PA',
                                      'PG',
                                      'PY',
                                      'PE',
                                      'PH',
                                      'PN',
                                      'PL',
                                      'PT',
                                      'PR',
                                      'QA',
                                      'RE',
                                      'RO',
                                      'RU',
                                      'RW',
                                      'BL',
                                      'SH',
                                      'KN',
                                      'LC',
                                      'MF',
                                      'PM',
                                      'VC',
                                      'WS',
                                      'SM',
                                      'ST',
                                      'SA',
                                      'SN',
                                      'RS',
                                      'SC',
                                      'SL',
                                      'SG',
                                      'SK',
                                      'SI',
                                      'SB',
                                      'SO',
                                      'ZA',
                                      'GS',
                                      'ES',
                                      'LK',
                                      'SD',
                                      'SR',
                                      'SJ',
                                      'SZ',
                                      'SE',
                                      'CH',
                                      'SY',
                                      'TW',
                                      'TJ',
                                      'TZ',
                                      'TH',
                                      'TL',
                                      'TG',
                                      'TK',
                                      'TO',
                                      'TT',
                                      'TN',
                                      'TR',
                                      'TM',
                                      'TC',
                                      'TV',
                                      'UG',
                                      'UA',
                                      'AE',
                                      'GB',
                                      'US',
                                      'UM',
                                      'UY',
                                      'UZ',
                                      'VU',
                                      'VE',
                                      'VN',
                                      'VG',
                                      'VI',
                                      'WF',
                                      'EH',
                                      'YE',
                                      'ZM',
                                      'ZW',
                                    ];
                                    description: 'The ISO 3166-1 alpha-2 country code for the country this location is in.';
                                    example: 'US';
                                  };
                                  latitude: {
                                    type: ['string', 'null'];
                                    pattern: '^[-+]?([1-8]?\\d(\\.\\d+)?|90(\\.0+)?)$';
                                    description: 'The latitude of the location. Validated by the regular expression `/^[-+]?([1-8]?\\d(\\.\\d+)?|90(\\.0+)?)$/`. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.}';
                                    example: '37.331741';
                                  };
                                  longitude: {
                                    type: ['string', 'null'];
                                    pattern: '^[-+]?(180(\\.0+)?|((1[0-7]\\d)|([1-9]?\\d))(\\.\\d+)?)$';
                                    description: 'The longitude of the location. Validated by the regular expression `/^[-+]?(180(\\.0+)?|((1[0-7]\\d)|([1-9]?\\d))(\\.\\d+)?)$/`';
                                    example: '-122.030333';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['location'];
                                    description: 'The attribute type of the value.';
                                    example: 'location';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'line_1',
                                  'line_2',
                                  'line_3',
                                  'line_4',
                                  'locality',
                                  'region',
                                  'postcode',
                                  'country_code',
                                  'latitude',
                                  'longitude',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'number';
                                    description: 'Numbers are persisted as 64 bit floats.';
                                    example: 42;
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['number'];
                                    description: 'The attribute type of the value.';
                                    example: 'number';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  first_name: {
                                    type: 'string';
                                    description: 'The first name.';
                                    example: 'Ada';
                                  };
                                  last_name: {
                                    type: 'string';
                                    description: 'The last name.';
                                    example: 'Lovelace';
                                  };
                                  full_name: {
                                    type: 'string';
                                    description: 'The full name.';
                                    example: 'Ada Lovelace';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['personal-name'];
                                    description: 'The attribute type of the value.';
                                    example: 'personal-name';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'first_name',
                                  'last_name',
                                  'full_name',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  original_phone_number: {
                                    type: 'string';
                                    description: 'The raw, original phone number, as inputted.';
                                    example: '5558675309';
                                  };
                                  country_code: {
                                    type: 'string';
                                    description: 'The ISO 3166-1 alpha-2 country code representing the country that this phone number belongs to.';
                                    enum: [
                                      'AF',
                                      'AX',
                                      'AL',
                                      'DZ',
                                      'AS',
                                      'AD',
                                      'AO',
                                      'AI',
                                      'AQ',
                                      'AG',
                                      'AR',
                                      'AM',
                                      'AW',
                                      'AU',
                                      'AT',
                                      'AZ',
                                      'BS',
                                      'BH',
                                      'BD',
                                      'BB',
                                      'BY',
                                      'BE',
                                      'BZ',
                                      'BJ',
                                      'BM',
                                      'BT',
                                      'BO',
                                      'BA',
                                      'BW',
                                      'BV',
                                      'BR',
                                      'IO',
                                      'BN',
                                      'BG',
                                      'BF',
                                      'BI',
                                      'KH',
                                      'CM',
                                      'CA',
                                      'CV',
                                      'KY',
                                      'CF',
                                      'TD',
                                      'CL',
                                      'CN',
                                      'CX',
                                      'CC',
                                      'CO',
                                      'KM',
                                      'CG',
                                      'CD',
                                      'CK',
                                      'CR',
                                      'CI',
                                      'HR',
                                      'CU',
                                      'CW',
                                      'CY',
                                      'CZ',
                                      'DK',
                                      'DJ',
                                      'DM',
                                      'DO',
                                      'EC',
                                      'EG',
                                      'SV',
                                      'GQ',
                                      'ER',
                                      'EE',
                                      'ET',
                                      'FK',
                                      'FO',
                                      'FJ',
                                      'FI',
                                      'FR',
                                      'GF',
                                      'PF',
                                      'TF',
                                      'GA',
                                      'GM',
                                      'GE',
                                      'DE',
                                      'GH',
                                      'GI',
                                      'GR',
                                      'GL',
                                      'GD',
                                      'GP',
                                      'GU',
                                      'GT',
                                      'GG',
                                      'GN',
                                      'GW',
                                      'GY',
                                      'HT',
                                      'HM',
                                      'VA',
                                      'HN',
                                      'HK',
                                      'HU',
                                      'IS',
                                      'IN',
                                      'ID',
                                      'IR',
                                      'IQ',
                                      'IE',
                                      'IM',
                                      'IL',
                                      'IT',
                                      'JM',
                                      'JP',
                                      'JE',
                                      'JO',
                                      'KZ',
                                      'KE',
                                      'KI',
                                      'KR',
                                      'KW',
                                      'KG',
                                      'LA',
                                      'LV',
                                      'LB',
                                      'LS',
                                      'LR',
                                      'LY',
                                      'LI',
                                      'LT',
                                      'LU',
                                      'MO',
                                      'MK',
                                      'MG',
                                      'MW',
                                      'MY',
                                      'MV',
                                      'ML',
                                      'MT',
                                      'MH',
                                      'MQ',
                                      'MR',
                                      'MU',
                                      'YT',
                                      'MX',
                                      'FM',
                                      'MD',
                                      'MC',
                                      'MN',
                                      'ME',
                                      'MS',
                                      'MA',
                                      'MZ',
                                      'MM',
                                      'NA',
                                      'NR',
                                      'NP',
                                      'NL',
                                      'AN',
                                      'NC',
                                      'NZ',
                                      'NI',
                                      'NE',
                                      'NG',
                                      'NU',
                                      'NF',
                                      'MP',
                                      'NO',
                                      'OM',
                                      'PK',
                                      'PW',
                                      'PS',
                                      'PA',
                                      'PG',
                                      'PY',
                                      'PE',
                                      'PH',
                                      'PN',
                                      'PL',
                                      'PT',
                                      'PR',
                                      'QA',
                                      'RE',
                                      'RO',
                                      'RU',
                                      'RW',
                                      'BL',
                                      'SH',
                                      'KN',
                                      'LC',
                                      'MF',
                                      'PM',
                                      'VC',
                                      'WS',
                                      'SM',
                                      'ST',
                                      'SA',
                                      'SN',
                                      'RS',
                                      'SC',
                                      'SL',
                                      'SG',
                                      'SK',
                                      'SI',
                                      'SB',
                                      'SO',
                                      'ZA',
                                      'GS',
                                      'ES',
                                      'LK',
                                      'SD',
                                      'SR',
                                      'SJ',
                                      'SZ',
                                      'SE',
                                      'CH',
                                      'SY',
                                      'TW',
                                      'TJ',
                                      'TZ',
                                      'TH',
                                      'TL',
                                      'TG',
                                      'TK',
                                      'TO',
                                      'TT',
                                      'TN',
                                      'TR',
                                      'TM',
                                      'TC',
                                      'TV',
                                      'UG',
                                      'UA',
                                      'AE',
                                      'GB',
                                      'US',
                                      'UM',
                                      'UY',
                                      'UZ',
                                      'VU',
                                      'VE',
                                      'VN',
                                      'VG',
                                      'VI',
                                      'WF',
                                      'EH',
                                      'YE',
                                      'ZM',
                                      'ZW',
                                    ];
                                    example: 'US';
                                  };
                                  phone_number: {
                                    type: 'string';
                                    example: '+15558675309';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['phone-number'];
                                    description: 'The attribute type of the value.';
                                    example: 'phone-number';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'original_phone_number',
                                  'country_code',
                                  'phone_number',
                                  'attribute_type',
                                ];
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  status: {
                                    $ref: '#/components/schemas/status';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['status'];
                                    description: 'The attribute type of the value.';
                                    example: 'status';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'status',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'number';
                                    description: 'A number between 0 and 5 (inclusive) to represent a star rating.';
                                    example: 3;
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['rating'];
                                    description: 'The attribute type of the value.';
                                    example: 'rating';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  option: {
                                    $ref: '#/components/schemas/select-option';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['select'];
                                    description: 'The attribute type of the value.';
                                    example: 'select';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'option',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'string';
                                    description: 'A raw text field. Values are limited to 10MB.';
                                    example: 'Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['text'];
                                    description: 'The attribute type of the value.';
                                    example: 'text';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['timestamp'];
                                    description: 'The attribute type of the value.';
                                    example: 'timestamp';
                                  };
                                  value: {
                                    type: 'string';
                                    description: 'A timestamp value represents a single, universal moment in time using an ISO 8601 formatted string. This means that a timestamp consists of a date, a time (with nanosecond precision), and a time zone. Attio will coerce timestamps which do not provide full nanosecond precision and UTC is assumed if no time zone is provided. For example, "2023", "2023-01", "2023-01-02", "2023-01-02T13:00", "2023-01-02T13:00:00", and "2023-01-02T13:00:00.000000000" will all be coerced to "2023-01-02T13:00:00.000000000Z". Timestamps are always returned in UTC. For example, writing a timestamp value using the string "2023-01-02T13:00:00.000000000+02:00" will result in the value "2023-01-02T11:00:00.000000000Z" being returned. The maximum date is "9999-12-31T23:59:59.999999999Z".';
                                    format: 'date';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'attribute_type',
                                  'value',
                                ];
                                additionalProperties: false;
                              },
                            ];
                          };
                        };
                        description: 'A record type with an attribute `api_slug` as the key, and an array of value objects as the values.';
                      };
                    };
                    required: ['id', 'created_at', 'values'];
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '400': {
          description: 'Bad Request';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [400];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['value_not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'No attribute was found for matching_attribute slug/ID "my-attribute".';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Bad Request';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Object with slug/ID "people" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
      };
    };
  };
  '/v2/objects/{object}/records/{record_id}': {
    get: {
      summary: 'Get a record';
      description: 'Gets a single person, company or other record by its `record_id`.\n\nRequired scopes: `record_permission:read`, `object_configuration:read`.';
      tags: ['Records'];
      security: [
        {
          oauth2: ['record_permission:read', 'object_configuration:read'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug identifying the object that the record belongs to.';
            example: 'people';
          };
          required: true;
          name: 'object';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            format: 'uuid';
            description: 'A UUID identifying the record.';
            example: '891dcbfc-9141-415d-9b2a-2238a6cc012d';
          };
          required: true;
          name: 'record_id';
          in: 'path';
        },
      ];
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    type: 'object';
                    properties: {
                      id: {
                        type: 'object';
                        properties: {
                          workspace_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying the workspace this record belongs to.';
                            example: '14beef7a-99f7-4534-a87e-70b564330a4c';
                          };
                          object_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying the object this record belongs to.';
                            example: '97052eb9-e65e-443f-a297-f2d9a4a7f795';
                          };
                          record_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying this record.';
                            example: 'bf071e1f-6035-429d-b874-d83ea64ea13b';
                          };
                        };
                        required: ['workspace_id', 'object_id', 'record_id'];
                      };
                      created_at: {
                        type: 'string';
                        description: 'When this record was created.';
                        example: '2022-11-21T13:22:49.061281000Z';
                      };
                      values: {
                        type: 'object';
                        additionalProperties: {
                          type: 'array';
                          items: {
                            oneOf: [
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  referenced_actor_type: {
                                    type: 'string';
                                    enum: ['api-token', 'workspace-member', 'system'];
                                    description: 'The type of the referenced actor. [Read more information on actor types here](/docs/actors).';
                                    example: 'workspace-member';
                                  };
                                  referenced_actor_id: {
                                    type: ['string', 'null'];
                                    format: 'uuid';
                                    description: 'The ID of the referenced actor.';
                                    example: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['actor-reference'];
                                    description: 'The attribute type of the value.';
                                    example: 'actor-reference';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'referenced_actor_type',
                                  'referenced_actor_id',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'boolean';
                                    description: "A boolean representing whether the checkbox is checked or not. The string values 'true' and 'false' are also accepted.";
                                    example: true;
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['checkbox'];
                                    description: 'The attribute type of the value.';
                                    example: 'checkbox';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  currency_value: {
                                    type: 'number';
                                    description: 'A numerical representation of the currency value. A decimal with a max of 4 decimal places.';
                                    example: 99;
                                  };
                                  currency_code: {
                                    type: ['string', 'null'];
                                    enum: [
                                      'EUR',
                                      'GBP',
                                      'USD',
                                      'CAD',
                                      'AUD',
                                      'NZD',
                                      'AED',
                                      'INR',
                                      'CHF',
                                      'JPY',
                                      'CNY',
                                      'PHP',
                                      'ILS',
                                      'SGD',
                                      'HKD',
                                      'MYR',
                                      'NTD',
                                      'NOK',
                                      'SEK',
                                      'ISK',
                                      'BEL',
                                      'PLN',
                                      'KRW',
                                      'DKK',
                                      'BRL',
                                    ];
                                    description: 'The ISO4217 currency code representing the currency that the value is stored in.';
                                    example: 'USD';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['currency'];
                                    description: 'The attribute type of the value.';
                                    example: 'currency';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'currency_value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['date'];
                                    description: 'The attribute type of the value.';
                                    example: 'date';
                                  };
                                  value: {
                                    type: 'string';
                                    description: 'A date represents a single calendar year, month and day, independent of timezone. If hours, months, seconds or timezones are provided, they will be trimmed. For example, "2023" and "2023-01" will be coerced into "2023-01-01", and "2023-01-02", "2023-01-02T13:00", "2023-01-02T14:00:00", "2023-01-02T15:00:00.000000000", and "2023-01-02T15:00:00.000000000+02:00" will all be coerced to "2023-01-02". If a timezone is provided that would result in a different calendar date in UTC, the date will be coerced to UTC and then the timezone component will be trimmed. For example, the value "2023-01-02T23:00:00-10:00" will be returned as "2023-01-03". The maximum date is "9999-12-31".';
                                    example: '2023-01-01';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'attribute_type',
                                  'value',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  domain: {
                                    type: 'string';
                                    example: 'app.attio.com';
                                  };
                                  root_domain: {
                                    type: 'string';
                                    example: 'attio.com';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['domain'];
                                    description: 'The attribute type of the value.';
                                    example: 'domain';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'domain',
                                  'root_domain',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  original_email_address: {
                                    type: 'string';
                                    example: 'alice@app.attio.com';
                                  };
                                  email_address: {
                                    type: 'string';
                                    example: 'alice@app.attio.com';
                                  };
                                  email_domain: {
                                    type: 'string';
                                    example: 'app.attio.com';
                                  };
                                  email_root_domain: {
                                    type: 'string';
                                    example: 'attio.com';
                                  };
                                  email_local_specifier: {
                                    type: 'string';
                                    example: 'alice';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['email-address'];
                                    description: 'The attribute type of the value.';
                                    example: 'email-address';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'original_email_address',
                                  'email_address',
                                  'email_domain',
                                  'email_root_domain',
                                  'email_local_specifier',
                                  'attribute_type',
                                ];
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  target_object: {
                                    type: 'string';
                                    description: 'A slug identifying the object that the referenced record belongs to.';
                                    example: 'people';
                                  };
                                  target_record_id: {
                                    type: 'string';
                                    format: 'uuid';
                                    description: 'A UUID to identify the referenced record.';
                                    example: '891dcbfc-9141-415d-9b2a-2238a6cc012d';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['record-reference'];
                                    description: 'The attribute type of the value.';
                                    example: 'record-reference';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'target_object',
                                  'target_record_id',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  interaction_type: {
                                    type: 'string';
                                    enum: ['calendar-event', 'call', 'chat-thread', 'email', 'in-person-meeting'];
                                    description: 'The type of interaction e.g. calendar or email.';
                                    example: 'email';
                                  };
                                  interacted_at: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'When the interaction occurred.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  owner_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['interaction'];
                                    description: 'The attribute type of the value.';
                                    example: 'interaction';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'interaction_type',
                                  'interacted_at',
                                  'owner_actor',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  line_1: {
                                    type: ['string', 'null'];
                                    description: 'The first line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: '1 Infinite Loop';
                                  };
                                  line_2: {
                                    type: ['string', 'null'];
                                    description: 'The second line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: 'Block 1';
                                  };
                                  line_3: {
                                    type: ['string', 'null'];
                                    description: 'The third line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: 'Hilldrop Estate';
                                  };
                                  line_4: {
                                    type: ['string', 'null'];
                                    description: 'The fourth line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: 'Westborough';
                                  };
                                  locality: {
                                    type: ['string', 'null'];
                                    description: 'The town, neighborhood or area the location is in.';
                                    example: 'Cupertino';
                                  };
                                  region: {
                                    type: ['string', 'null'];
                                    description: 'The state, county, province or region that the location is in.';
                                    example: 'CA';
                                  };
                                  postcode: {
                                    type: ['string', 'null'];
                                    description: 'The postcode or zip code for the location. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.}';
                                    example: '95014';
                                  };
                                  country_code: {
                                    type: ['string', 'null'];
                                    enum: [
                                      'AF',
                                      'AX',
                                      'AL',
                                      'DZ',
                                      'AS',
                                      'AD',
                                      'AO',
                                      'AI',
                                      'AQ',
                                      'AG',
                                      'AR',
                                      'AM',
                                      'AW',
                                      'AU',
                                      'AT',
                                      'AZ',
                                      'BS',
                                      'BH',
                                      'BD',
                                      'BB',
                                      'BY',
                                      'BE',
                                      'BZ',
                                      'BJ',
                                      'BM',
                                      'BT',
                                      'BO',
                                      'BA',
                                      'BW',
                                      'BV',
                                      'BR',
                                      'IO',
                                      'BN',
                                      'BG',
                                      'BF',
                                      'BI',
                                      'KH',
                                      'CM',
                                      'CA',
                                      'CV',
                                      'KY',
                                      'CF',
                                      'TD',
                                      'CL',
                                      'CN',
                                      'CX',
                                      'CC',
                                      'CO',
                                      'KM',
                                      'CG',
                                      'CD',
                                      'CK',
                                      'CR',
                                      'CI',
                                      'HR',
                                      'CU',
                                      'CW',
                                      'CY',
                                      'CZ',
                                      'DK',
                                      'DJ',
                                      'DM',
                                      'DO',
                                      'EC',
                                      'EG',
                                      'SV',
                                      'GQ',
                                      'ER',
                                      'EE',
                                      'ET',
                                      'FK',
                                      'FO',
                                      'FJ',
                                      'FI',
                                      'FR',
                                      'GF',
                                      'PF',
                                      'TF',
                                      'GA',
                                      'GM',
                                      'GE',
                                      'DE',
                                      'GH',
                                      'GI',
                                      'GR',
                                      'GL',
                                      'GD',
                                      'GP',
                                      'GU',
                                      'GT',
                                      'GG',
                                      'GN',
                                      'GW',
                                      'GY',
                                      'HT',
                                      'HM',
                                      'VA',
                                      'HN',
                                      'HK',
                                      'HU',
                                      'IS',
                                      'IN',
                                      'ID',
                                      'IR',
                                      'IQ',
                                      'IE',
                                      'IM',
                                      'IL',
                                      'IT',
                                      'JM',
                                      'JP',
                                      'JE',
                                      'JO',
                                      'KZ',
                                      'KE',
                                      'KI',
                                      'KR',
                                      'KW',
                                      'KG',
                                      'LA',
                                      'LV',
                                      'LB',
                                      'LS',
                                      'LR',
                                      'LY',
                                      'LI',
                                      'LT',
                                      'LU',
                                      'MO',
                                      'MK',
                                      'MG',
                                      'MW',
                                      'MY',
                                      'MV',
                                      'ML',
                                      'MT',
                                      'MH',
                                      'MQ',
                                      'MR',
                                      'MU',
                                      'YT',
                                      'MX',
                                      'FM',
                                      'MD',
                                      'MC',
                                      'MN',
                                      'ME',
                                      'MS',
                                      'MA',
                                      'MZ',
                                      'MM',
                                      'NA',
                                      'NR',
                                      'NP',
                                      'NL',
                                      'AN',
                                      'NC',
                                      'NZ',
                                      'NI',
                                      'NE',
                                      'NG',
                                      'NU',
                                      'NF',
                                      'MP',
                                      'NO',
                                      'OM',
                                      'PK',
                                      'PW',
                                      'PS',
                                      'PA',
                                      'PG',
                                      'PY',
                                      'PE',
                                      'PH',
                                      'PN',
                                      'PL',
                                      'PT',
                                      'PR',
                                      'QA',
                                      'RE',
                                      'RO',
                                      'RU',
                                      'RW',
                                      'BL',
                                      'SH',
                                      'KN',
                                      'LC',
                                      'MF',
                                      'PM',
                                      'VC',
                                      'WS',
                                      'SM',
                                      'ST',
                                      'SA',
                                      'SN',
                                      'RS',
                                      'SC',
                                      'SL',
                                      'SG',
                                      'SK',
                                      'SI',
                                      'SB',
                                      'SO',
                                      'ZA',
                                      'GS',
                                      'ES',
                                      'LK',
                                      'SD',
                                      'SR',
                                      'SJ',
                                      'SZ',
                                      'SE',
                                      'CH',
                                      'SY',
                                      'TW',
                                      'TJ',
                                      'TZ',
                                      'TH',
                                      'TL',
                                      'TG',
                                      'TK',
                                      'TO',
                                      'TT',
                                      'TN',
                                      'TR',
                                      'TM',
                                      'TC',
                                      'TV',
                                      'UG',
                                      'UA',
                                      'AE',
                                      'GB',
                                      'US',
                                      'UM',
                                      'UY',
                                      'UZ',
                                      'VU',
                                      'VE',
                                      'VN',
                                      'VG',
                                      'VI',
                                      'WF',
                                      'EH',
                                      'YE',
                                      'ZM',
                                      'ZW',
                                    ];
                                    description: 'The ISO 3166-1 alpha-2 country code for the country this location is in.';
                                    example: 'US';
                                  };
                                  latitude: {
                                    type: ['string', 'null'];
                                    pattern: '^[-+]?([1-8]?\\d(\\.\\d+)?|90(\\.0+)?)$';
                                    description: 'The latitude of the location. Validated by the regular expression `/^[-+]?([1-8]?\\d(\\.\\d+)?|90(\\.0+)?)$/`. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.}';
                                    example: '37.331741';
                                  };
                                  longitude: {
                                    type: ['string', 'null'];
                                    pattern: '^[-+]?(180(\\.0+)?|((1[0-7]\\d)|([1-9]?\\d))(\\.\\d+)?)$';
                                    description: 'The longitude of the location. Validated by the regular expression `/^[-+]?(180(\\.0+)?|((1[0-7]\\d)|([1-9]?\\d))(\\.\\d+)?)$/`';
                                    example: '-122.030333';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['location'];
                                    description: 'The attribute type of the value.';
                                    example: 'location';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'line_1',
                                  'line_2',
                                  'line_3',
                                  'line_4',
                                  'locality',
                                  'region',
                                  'postcode',
                                  'country_code',
                                  'latitude',
                                  'longitude',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'number';
                                    description: 'Numbers are persisted as 64 bit floats.';
                                    example: 42;
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['number'];
                                    description: 'The attribute type of the value.';
                                    example: 'number';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  first_name: {
                                    type: 'string';
                                    description: 'The first name.';
                                    example: 'Ada';
                                  };
                                  last_name: {
                                    type: 'string';
                                    description: 'The last name.';
                                    example: 'Lovelace';
                                  };
                                  full_name: {
                                    type: 'string';
                                    description: 'The full name.';
                                    example: 'Ada Lovelace';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['personal-name'];
                                    description: 'The attribute type of the value.';
                                    example: 'personal-name';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'first_name',
                                  'last_name',
                                  'full_name',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  original_phone_number: {
                                    type: 'string';
                                    description: 'The raw, original phone number, as inputted.';
                                    example: '5558675309';
                                  };
                                  country_code: {
                                    type: 'string';
                                    description: 'The ISO 3166-1 alpha-2 country code representing the country that this phone number belongs to.';
                                    enum: [
                                      'AF',
                                      'AX',
                                      'AL',
                                      'DZ',
                                      'AS',
                                      'AD',
                                      'AO',
                                      'AI',
                                      'AQ',
                                      'AG',
                                      'AR',
                                      'AM',
                                      'AW',
                                      'AU',
                                      'AT',
                                      'AZ',
                                      'BS',
                                      'BH',
                                      'BD',
                                      'BB',
                                      'BY',
                                      'BE',
                                      'BZ',
                                      'BJ',
                                      'BM',
                                      'BT',
                                      'BO',
                                      'BA',
                                      'BW',
                                      'BV',
                                      'BR',
                                      'IO',
                                      'BN',
                                      'BG',
                                      'BF',
                                      'BI',
                                      'KH',
                                      'CM',
                                      'CA',
                                      'CV',
                                      'KY',
                                      'CF',
                                      'TD',
                                      'CL',
                                      'CN',
                                      'CX',
                                      'CC',
                                      'CO',
                                      'KM',
                                      'CG',
                                      'CD',
                                      'CK',
                                      'CR',
                                      'CI',
                                      'HR',
                                      'CU',
                                      'CW',
                                      'CY',
                                      'CZ',
                                      'DK',
                                      'DJ',
                                      'DM',
                                      'DO',
                                      'EC',
                                      'EG',
                                      'SV',
                                      'GQ',
                                      'ER',
                                      'EE',
                                      'ET',
                                      'FK',
                                      'FO',
                                      'FJ',
                                      'FI',
                                      'FR',
                                      'GF',
                                      'PF',
                                      'TF',
                                      'GA',
                                      'GM',
                                      'GE',
                                      'DE',
                                      'GH',
                                      'GI',
                                      'GR',
                                      'GL',
                                      'GD',
                                      'GP',
                                      'GU',
                                      'GT',
                                      'GG',
                                      'GN',
                                      'GW',
                                      'GY',
                                      'HT',
                                      'HM',
                                      'VA',
                                      'HN',
                                      'HK',
                                      'HU',
                                      'IS',
                                      'IN',
                                      'ID',
                                      'IR',
                                      'IQ',
                                      'IE',
                                      'IM',
                                      'IL',
                                      'IT',
                                      'JM',
                                      'JP',
                                      'JE',
                                      'JO',
                                      'KZ',
                                      'KE',
                                      'KI',
                                      'KR',
                                      'KW',
                                      'KG',
                                      'LA',
                                      'LV',
                                      'LB',
                                      'LS',
                                      'LR',
                                      'LY',
                                      'LI',
                                      'LT',
                                      'LU',
                                      'MO',
                                      'MK',
                                      'MG',
                                      'MW',
                                      'MY',
                                      'MV',
                                      'ML',
                                      'MT',
                                      'MH',
                                      'MQ',
                                      'MR',
                                      'MU',
                                      'YT',
                                      'MX',
                                      'FM',
                                      'MD',
                                      'MC',
                                      'MN',
                                      'ME',
                                      'MS',
                                      'MA',
                                      'MZ',
                                      'MM',
                                      'NA',
                                      'NR',
                                      'NP',
                                      'NL',
                                      'AN',
                                      'NC',
                                      'NZ',
                                      'NI',
                                      'NE',
                                      'NG',
                                      'NU',
                                      'NF',
                                      'MP',
                                      'NO',
                                      'OM',
                                      'PK',
                                      'PW',
                                      'PS',
                                      'PA',
                                      'PG',
                                      'PY',
                                      'PE',
                                      'PH',
                                      'PN',
                                      'PL',
                                      'PT',
                                      'PR',
                                      'QA',
                                      'RE',
                                      'RO',
                                      'RU',
                                      'RW',
                                      'BL',
                                      'SH',
                                      'KN',
                                      'LC',
                                      'MF',
                                      'PM',
                                      'VC',
                                      'WS',
                                      'SM',
                                      'ST',
                                      'SA',
                                      'SN',
                                      'RS',
                                      'SC',
                                      'SL',
                                      'SG',
                                      'SK',
                                      'SI',
                                      'SB',
                                      'SO',
                                      'ZA',
                                      'GS',
                                      'ES',
                                      'LK',
                                      'SD',
                                      'SR',
                                      'SJ',
                                      'SZ',
                                      'SE',
                                      'CH',
                                      'SY',
                                      'TW',
                                      'TJ',
                                      'TZ',
                                      'TH',
                                      'TL',
                                      'TG',
                                      'TK',
                                      'TO',
                                      'TT',
                                      'TN',
                                      'TR',
                                      'TM',
                                      'TC',
                                      'TV',
                                      'UG',
                                      'UA',
                                      'AE',
                                      'GB',
                                      'US',
                                      'UM',
                                      'UY',
                                      'UZ',
                                      'VU',
                                      'VE',
                                      'VN',
                                      'VG',
                                      'VI',
                                      'WF',
                                      'EH',
                                      'YE',
                                      'ZM',
                                      'ZW',
                                    ];
                                    example: 'US';
                                  };
                                  phone_number: {
                                    type: 'string';
                                    example: '+15558675309';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['phone-number'];
                                    description: 'The attribute type of the value.';
                                    example: 'phone-number';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'original_phone_number',
                                  'country_code',
                                  'phone_number',
                                  'attribute_type',
                                ];
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  status: {
                                    $ref: '#/components/schemas/status';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['status'];
                                    description: 'The attribute type of the value.';
                                    example: 'status';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'status',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'number';
                                    description: 'A number between 0 and 5 (inclusive) to represent a star rating.';
                                    example: 3;
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['rating'];
                                    description: 'The attribute type of the value.';
                                    example: 'rating';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  option: {
                                    $ref: '#/components/schemas/select-option';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['select'];
                                    description: 'The attribute type of the value.';
                                    example: 'select';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'option',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'string';
                                    description: 'A raw text field. Values are limited to 10MB.';
                                    example: 'Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['text'];
                                    description: 'The attribute type of the value.';
                                    example: 'text';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['timestamp'];
                                    description: 'The attribute type of the value.';
                                    example: 'timestamp';
                                  };
                                  value: {
                                    type: 'string';
                                    description: 'A timestamp value represents a single, universal moment in time using an ISO 8601 formatted string. This means that a timestamp consists of a date, a time (with nanosecond precision), and a time zone. Attio will coerce timestamps which do not provide full nanosecond precision and UTC is assumed if no time zone is provided. For example, "2023", "2023-01", "2023-01-02", "2023-01-02T13:00", "2023-01-02T13:00:00", and "2023-01-02T13:00:00.000000000" will all be coerced to "2023-01-02T13:00:00.000000000Z". Timestamps are always returned in UTC. For example, writing a timestamp value using the string "2023-01-02T13:00:00.000000000+02:00" will result in the value "2023-01-02T11:00:00.000000000Z" being returned. The maximum date is "9999-12-31T23:59:59.999999999Z".';
                                    format: 'date';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'attribute_type',
                                  'value',
                                ];
                                additionalProperties: false;
                              },
                            ];
                          };
                        };
                        description: 'A record type with an attribute `api_slug` as the key, and an array of value objects as the values.';
                      };
                    };
                    required: ['id', 'created_at', 'values'];
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Record with ID "891dcbfc-9141-415d-9b2a-2238a6cc012d" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
      };
    };
    patch: {
      summary: 'Update a record';
      description: 'Use this endpoint to update people, companies and other records by `record_id`. If the update payload includes multiselect attributes, the values supplied will be created and prepended to the list of values that already exist (if any). Use the [Assert record endpoint](/reference/put_v2-objects-object-records) to overwrite or remove multiselect attribute values.\n\nRequired scopes: `record_permission:read-write`, `object_configuration:read`.';
      tags: ['Records'];
      security: [
        {
          oauth2: ['record_permission:read-write', 'object_configuration:read'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug of the object the record belongs to.';
            example: 'people';
          };
          required: true;
          name: 'object';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID of the record to update.';
            example: '891dcbfc-9141-415d-9b2a-2238a6cc012d';
          };
          required: true;
          name: 'record_id';
          in: 'path';
        },
      ];
      requestBody: {
        required: true;
        content: {
          'application/json': {
            schema: {
              type: 'object';
              properties: {
                data: {
                  type: 'object';
                  properties: {
                    values: {
                      type: 'object';
                      description: 'An object with an attribute `api_slug` or `attribute_id` as the key, and a single value (for single-select attributes), or an array of values (for single or multi-select attributes) as the values. For complete documentation on values for all attribute types, please see our [attribute type docs](/docs/attribute-types).';
                      additionalProperties: {
                        type: 'array';
                      };
                      example: {
                        '41252299-f8c7-4b5e-99c9-4ff8321d2f96': 'Text value';
                        multiselect_attribute: ['Select option 1', 'Select option 2'];
                      };
                    };
                  };
                  required: ['values'];
                  additionalProperties: false;
                };
              };
              required: ['data'];
            };
          };
        };
      };
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    type: 'object';
                    properties: {
                      id: {
                        type: 'object';
                        properties: {
                          workspace_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying the workspace this record belongs to.';
                            example: '14beef7a-99f7-4534-a87e-70b564330a4c';
                          };
                          object_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying the object this record belongs to.';
                            example: '97052eb9-e65e-443f-a297-f2d9a4a7f795';
                          };
                          record_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying this record.';
                            example: 'bf071e1f-6035-429d-b874-d83ea64ea13b';
                          };
                        };
                        required: ['workspace_id', 'object_id', 'record_id'];
                      };
                      created_at: {
                        type: 'string';
                        description: 'When this record was created.';
                        example: '2022-11-21T13:22:49.061281000Z';
                      };
                      values: {
                        type: 'object';
                        additionalProperties: {
                          type: 'array';
                          items: {
                            oneOf: [
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  referenced_actor_type: {
                                    type: 'string';
                                    enum: ['api-token', 'workspace-member', 'system'];
                                    description: 'The type of the referenced actor. [Read more information on actor types here](/docs/actors).';
                                    example: 'workspace-member';
                                  };
                                  referenced_actor_id: {
                                    type: ['string', 'null'];
                                    format: 'uuid';
                                    description: 'The ID of the referenced actor.';
                                    example: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['actor-reference'];
                                    description: 'The attribute type of the value.';
                                    example: 'actor-reference';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'referenced_actor_type',
                                  'referenced_actor_id',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'boolean';
                                    description: "A boolean representing whether the checkbox is checked or not. The string values 'true' and 'false' are also accepted.";
                                    example: true;
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['checkbox'];
                                    description: 'The attribute type of the value.';
                                    example: 'checkbox';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  currency_value: {
                                    type: 'number';
                                    description: 'A numerical representation of the currency value. A decimal with a max of 4 decimal places.';
                                    example: 99;
                                  };
                                  currency_code: {
                                    type: ['string', 'null'];
                                    enum: [
                                      'EUR',
                                      'GBP',
                                      'USD',
                                      'CAD',
                                      'AUD',
                                      'NZD',
                                      'AED',
                                      'INR',
                                      'CHF',
                                      'JPY',
                                      'CNY',
                                      'PHP',
                                      'ILS',
                                      'SGD',
                                      'HKD',
                                      'MYR',
                                      'NTD',
                                      'NOK',
                                      'SEK',
                                      'ISK',
                                      'BEL',
                                      'PLN',
                                      'KRW',
                                      'DKK',
                                      'BRL',
                                    ];
                                    description: 'The ISO4217 currency code representing the currency that the value is stored in.';
                                    example: 'USD';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['currency'];
                                    description: 'The attribute type of the value.';
                                    example: 'currency';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'currency_value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['date'];
                                    description: 'The attribute type of the value.';
                                    example: 'date';
                                  };
                                  value: {
                                    type: 'string';
                                    description: 'A date represents a single calendar year, month and day, independent of timezone. If hours, months, seconds or timezones are provided, they will be trimmed. For example, "2023" and "2023-01" will be coerced into "2023-01-01", and "2023-01-02", "2023-01-02T13:00", "2023-01-02T14:00:00", "2023-01-02T15:00:00.000000000", and "2023-01-02T15:00:00.000000000+02:00" will all be coerced to "2023-01-02". If a timezone is provided that would result in a different calendar date in UTC, the date will be coerced to UTC and then the timezone component will be trimmed. For example, the value "2023-01-02T23:00:00-10:00" will be returned as "2023-01-03". The maximum date is "9999-12-31".';
                                    example: '2023-01-01';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'attribute_type',
                                  'value',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  domain: {
                                    type: 'string';
                                    example: 'app.attio.com';
                                  };
                                  root_domain: {
                                    type: 'string';
                                    example: 'attio.com';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['domain'];
                                    description: 'The attribute type of the value.';
                                    example: 'domain';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'domain',
                                  'root_domain',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  original_email_address: {
                                    type: 'string';
                                    example: 'alice@app.attio.com';
                                  };
                                  email_address: {
                                    type: 'string';
                                    example: 'alice@app.attio.com';
                                  };
                                  email_domain: {
                                    type: 'string';
                                    example: 'app.attio.com';
                                  };
                                  email_root_domain: {
                                    type: 'string';
                                    example: 'attio.com';
                                  };
                                  email_local_specifier: {
                                    type: 'string';
                                    example: 'alice';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['email-address'];
                                    description: 'The attribute type of the value.';
                                    example: 'email-address';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'original_email_address',
                                  'email_address',
                                  'email_domain',
                                  'email_root_domain',
                                  'email_local_specifier',
                                  'attribute_type',
                                ];
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  target_object: {
                                    type: 'string';
                                    description: 'A slug identifying the object that the referenced record belongs to.';
                                    example: 'people';
                                  };
                                  target_record_id: {
                                    type: 'string';
                                    format: 'uuid';
                                    description: 'A UUID to identify the referenced record.';
                                    example: '891dcbfc-9141-415d-9b2a-2238a6cc012d';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['record-reference'];
                                    description: 'The attribute type of the value.';
                                    example: 'record-reference';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'target_object',
                                  'target_record_id',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  interaction_type: {
                                    type: 'string';
                                    enum: ['calendar-event', 'call', 'chat-thread', 'email', 'in-person-meeting'];
                                    description: 'The type of interaction e.g. calendar or email.';
                                    example: 'email';
                                  };
                                  interacted_at: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'When the interaction occurred.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  owner_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['interaction'];
                                    description: 'The attribute type of the value.';
                                    example: 'interaction';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'interaction_type',
                                  'interacted_at',
                                  'owner_actor',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  line_1: {
                                    type: ['string', 'null'];
                                    description: 'The first line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: '1 Infinite Loop';
                                  };
                                  line_2: {
                                    type: ['string', 'null'];
                                    description: 'The second line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: 'Block 1';
                                  };
                                  line_3: {
                                    type: ['string', 'null'];
                                    description: 'The third line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: 'Hilldrop Estate';
                                  };
                                  line_4: {
                                    type: ['string', 'null'];
                                    description: 'The fourth line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: 'Westborough';
                                  };
                                  locality: {
                                    type: ['string', 'null'];
                                    description: 'The town, neighborhood or area the location is in.';
                                    example: 'Cupertino';
                                  };
                                  region: {
                                    type: ['string', 'null'];
                                    description: 'The state, county, province or region that the location is in.';
                                    example: 'CA';
                                  };
                                  postcode: {
                                    type: ['string', 'null'];
                                    description: 'The postcode or zip code for the location. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.}';
                                    example: '95014';
                                  };
                                  country_code: {
                                    type: ['string', 'null'];
                                    enum: [
                                      'AF',
                                      'AX',
                                      'AL',
                                      'DZ',
                                      'AS',
                                      'AD',
                                      'AO',
                                      'AI',
                                      'AQ',
                                      'AG',
                                      'AR',
                                      'AM',
                                      'AW',
                                      'AU',
                                      'AT',
                                      'AZ',
                                      'BS',
                                      'BH',
                                      'BD',
                                      'BB',
                                      'BY',
                                      'BE',
                                      'BZ',
                                      'BJ',
                                      'BM',
                                      'BT',
                                      'BO',
                                      'BA',
                                      'BW',
                                      'BV',
                                      'BR',
                                      'IO',
                                      'BN',
                                      'BG',
                                      'BF',
                                      'BI',
                                      'KH',
                                      'CM',
                                      'CA',
                                      'CV',
                                      'KY',
                                      'CF',
                                      'TD',
                                      'CL',
                                      'CN',
                                      'CX',
                                      'CC',
                                      'CO',
                                      'KM',
                                      'CG',
                                      'CD',
                                      'CK',
                                      'CR',
                                      'CI',
                                      'HR',
                                      'CU',
                                      'CW',
                                      'CY',
                                      'CZ',
                                      'DK',
                                      'DJ',
                                      'DM',
                                      'DO',
                                      'EC',
                                      'EG',
                                      'SV',
                                      'GQ',
                                      'ER',
                                      'EE',
                                      'ET',
                                      'FK',
                                      'FO',
                                      'FJ',
                                      'FI',
                                      'FR',
                                      'GF',
                                      'PF',
                                      'TF',
                                      'GA',
                                      'GM',
                                      'GE',
                                      'DE',
                                      'GH',
                                      'GI',
                                      'GR',
                                      'GL',
                                      'GD',
                                      'GP',
                                      'GU',
                                      'GT',
                                      'GG',
                                      'GN',
                                      'GW',
                                      'GY',
                                      'HT',
                                      'HM',
                                      'VA',
                                      'HN',
                                      'HK',
                                      'HU',
                                      'IS',
                                      'IN',
                                      'ID',
                                      'IR',
                                      'IQ',
                                      'IE',
                                      'IM',
                                      'IL',
                                      'IT',
                                      'JM',
                                      'JP',
                                      'JE',
                                      'JO',
                                      'KZ',
                                      'KE',
                                      'KI',
                                      'KR',
                                      'KW',
                                      'KG',
                                      'LA',
                                      'LV',
                                      'LB',
                                      'LS',
                                      'LR',
                                      'LY',
                                      'LI',
                                      'LT',
                                      'LU',
                                      'MO',
                                      'MK',
                                      'MG',
                                      'MW',
                                      'MY',
                                      'MV',
                                      'ML',
                                      'MT',
                                      'MH',
                                      'MQ',
                                      'MR',
                                      'MU',
                                      'YT',
                                      'MX',
                                      'FM',
                                      'MD',
                                      'MC',
                                      'MN',
                                      'ME',
                                      'MS',
                                      'MA',
                                      'MZ',
                                      'MM',
                                      'NA',
                                      'NR',
                                      'NP',
                                      'NL',
                                      'AN',
                                      'NC',
                                      'NZ',
                                      'NI',
                                      'NE',
                                      'NG',
                                      'NU',
                                      'NF',
                                      'MP',
                                      'NO',
                                      'OM',
                                      'PK',
                                      'PW',
                                      'PS',
                                      'PA',
                                      'PG',
                                      'PY',
                                      'PE',
                                      'PH',
                                      'PN',
                                      'PL',
                                      'PT',
                                      'PR',
                                      'QA',
                                      'RE',
                                      'RO',
                                      'RU',
                                      'RW',
                                      'BL',
                                      'SH',
                                      'KN',
                                      'LC',
                                      'MF',
                                      'PM',
                                      'VC',
                                      'WS',
                                      'SM',
                                      'ST',
                                      'SA',
                                      'SN',
                                      'RS',
                                      'SC',
                                      'SL',
                                      'SG',
                                      'SK',
                                      'SI',
                                      'SB',
                                      'SO',
                                      'ZA',
                                      'GS',
                                      'ES',
                                      'LK',
                                      'SD',
                                      'SR',
                                      'SJ',
                                      'SZ',
                                      'SE',
                                      'CH',
                                      'SY',
                                      'TW',
                                      'TJ',
                                      'TZ',
                                      'TH',
                                      'TL',
                                      'TG',
                                      'TK',
                                      'TO',
                                      'TT',
                                      'TN',
                                      'TR',
                                      'TM',
                                      'TC',
                                      'TV',
                                      'UG',
                                      'UA',
                                      'AE',
                                      'GB',
                                      'US',
                                      'UM',
                                      'UY',
                                      'UZ',
                                      'VU',
                                      'VE',
                                      'VN',
                                      'VG',
                                      'VI',
                                      'WF',
                                      'EH',
                                      'YE',
                                      'ZM',
                                      'ZW',
                                    ];
                                    description: 'The ISO 3166-1 alpha-2 country code for the country this location is in.';
                                    example: 'US';
                                  };
                                  latitude: {
                                    type: ['string', 'null'];
                                    pattern: '^[-+]?([1-8]?\\d(\\.\\d+)?|90(\\.0+)?)$';
                                    description: 'The latitude of the location. Validated by the regular expression `/^[-+]?([1-8]?\\d(\\.\\d+)?|90(\\.0+)?)$/`. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.}';
                                    example: '37.331741';
                                  };
                                  longitude: {
                                    type: ['string', 'null'];
                                    pattern: '^[-+]?(180(\\.0+)?|((1[0-7]\\d)|([1-9]?\\d))(\\.\\d+)?)$';
                                    description: 'The longitude of the location. Validated by the regular expression `/^[-+]?(180(\\.0+)?|((1[0-7]\\d)|([1-9]?\\d))(\\.\\d+)?)$/`';
                                    example: '-122.030333';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['location'];
                                    description: 'The attribute type of the value.';
                                    example: 'location';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'line_1',
                                  'line_2',
                                  'line_3',
                                  'line_4',
                                  'locality',
                                  'region',
                                  'postcode',
                                  'country_code',
                                  'latitude',
                                  'longitude',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'number';
                                    description: 'Numbers are persisted as 64 bit floats.';
                                    example: 42;
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['number'];
                                    description: 'The attribute type of the value.';
                                    example: 'number';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  first_name: {
                                    type: 'string';
                                    description: 'The first name.';
                                    example: 'Ada';
                                  };
                                  last_name: {
                                    type: 'string';
                                    description: 'The last name.';
                                    example: 'Lovelace';
                                  };
                                  full_name: {
                                    type: 'string';
                                    description: 'The full name.';
                                    example: 'Ada Lovelace';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['personal-name'];
                                    description: 'The attribute type of the value.';
                                    example: 'personal-name';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'first_name',
                                  'last_name',
                                  'full_name',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  original_phone_number: {
                                    type: 'string';
                                    description: 'The raw, original phone number, as inputted.';
                                    example: '5558675309';
                                  };
                                  country_code: {
                                    type: 'string';
                                    description: 'The ISO 3166-1 alpha-2 country code representing the country that this phone number belongs to.';
                                    enum: [
                                      'AF',
                                      'AX',
                                      'AL',
                                      'DZ',
                                      'AS',
                                      'AD',
                                      'AO',
                                      'AI',
                                      'AQ',
                                      'AG',
                                      'AR',
                                      'AM',
                                      'AW',
                                      'AU',
                                      'AT',
                                      'AZ',
                                      'BS',
                                      'BH',
                                      'BD',
                                      'BB',
                                      'BY',
                                      'BE',
                                      'BZ',
                                      'BJ',
                                      'BM',
                                      'BT',
                                      'BO',
                                      'BA',
                                      'BW',
                                      'BV',
                                      'BR',
                                      'IO',
                                      'BN',
                                      'BG',
                                      'BF',
                                      'BI',
                                      'KH',
                                      'CM',
                                      'CA',
                                      'CV',
                                      'KY',
                                      'CF',
                                      'TD',
                                      'CL',
                                      'CN',
                                      'CX',
                                      'CC',
                                      'CO',
                                      'KM',
                                      'CG',
                                      'CD',
                                      'CK',
                                      'CR',
                                      'CI',
                                      'HR',
                                      'CU',
                                      'CW',
                                      'CY',
                                      'CZ',
                                      'DK',
                                      'DJ',
                                      'DM',
                                      'DO',
                                      'EC',
                                      'EG',
                                      'SV',
                                      'GQ',
                                      'ER',
                                      'EE',
                                      'ET',
                                      'FK',
                                      'FO',
                                      'FJ',
                                      'FI',
                                      'FR',
                                      'GF',
                                      'PF',
                                      'TF',
                                      'GA',
                                      'GM',
                                      'GE',
                                      'DE',
                                      'GH',
                                      'GI',
                                      'GR',
                                      'GL',
                                      'GD',
                                      'GP',
                                      'GU',
                                      'GT',
                                      'GG',
                                      'GN',
                                      'GW',
                                      'GY',
                                      'HT',
                                      'HM',
                                      'VA',
                                      'HN',
                                      'HK',
                                      'HU',
                                      'IS',
                                      'IN',
                                      'ID',
                                      'IR',
                                      'IQ',
                                      'IE',
                                      'IM',
                                      'IL',
                                      'IT',
                                      'JM',
                                      'JP',
                                      'JE',
                                      'JO',
                                      'KZ',
                                      'KE',
                                      'KI',
                                      'KR',
                                      'KW',
                                      'KG',
                                      'LA',
                                      'LV',
                                      'LB',
                                      'LS',
                                      'LR',
                                      'LY',
                                      'LI',
                                      'LT',
                                      'LU',
                                      'MO',
                                      'MK',
                                      'MG',
                                      'MW',
                                      'MY',
                                      'MV',
                                      'ML',
                                      'MT',
                                      'MH',
                                      'MQ',
                                      'MR',
                                      'MU',
                                      'YT',
                                      'MX',
                                      'FM',
                                      'MD',
                                      'MC',
                                      'MN',
                                      'ME',
                                      'MS',
                                      'MA',
                                      'MZ',
                                      'MM',
                                      'NA',
                                      'NR',
                                      'NP',
                                      'NL',
                                      'AN',
                                      'NC',
                                      'NZ',
                                      'NI',
                                      'NE',
                                      'NG',
                                      'NU',
                                      'NF',
                                      'MP',
                                      'NO',
                                      'OM',
                                      'PK',
                                      'PW',
                                      'PS',
                                      'PA',
                                      'PG',
                                      'PY',
                                      'PE',
                                      'PH',
                                      'PN',
                                      'PL',
                                      'PT',
                                      'PR',
                                      'QA',
                                      'RE',
                                      'RO',
                                      'RU',
                                      'RW',
                                      'BL',
                                      'SH',
                                      'KN',
                                      'LC',
                                      'MF',
                                      'PM',
                                      'VC',
                                      'WS',
                                      'SM',
                                      'ST',
                                      'SA',
                                      'SN',
                                      'RS',
                                      'SC',
                                      'SL',
                                      'SG',
                                      'SK',
                                      'SI',
                                      'SB',
                                      'SO',
                                      'ZA',
                                      'GS',
                                      'ES',
                                      'LK',
                                      'SD',
                                      'SR',
                                      'SJ',
                                      'SZ',
                                      'SE',
                                      'CH',
                                      'SY',
                                      'TW',
                                      'TJ',
                                      'TZ',
                                      'TH',
                                      'TL',
                                      'TG',
                                      'TK',
                                      'TO',
                                      'TT',
                                      'TN',
                                      'TR',
                                      'TM',
                                      'TC',
                                      'TV',
                                      'UG',
                                      'UA',
                                      'AE',
                                      'GB',
                                      'US',
                                      'UM',
                                      'UY',
                                      'UZ',
                                      'VU',
                                      'VE',
                                      'VN',
                                      'VG',
                                      'VI',
                                      'WF',
                                      'EH',
                                      'YE',
                                      'ZM',
                                      'ZW',
                                    ];
                                    example: 'US';
                                  };
                                  phone_number: {
                                    type: 'string';
                                    example: '+15558675309';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['phone-number'];
                                    description: 'The attribute type of the value.';
                                    example: 'phone-number';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'original_phone_number',
                                  'country_code',
                                  'phone_number',
                                  'attribute_type',
                                ];
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  status: {
                                    $ref: '#/components/schemas/status';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['status'];
                                    description: 'The attribute type of the value.';
                                    example: 'status';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'status',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'number';
                                    description: 'A number between 0 and 5 (inclusive) to represent a star rating.';
                                    example: 3;
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['rating'];
                                    description: 'The attribute type of the value.';
                                    example: 'rating';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  option: {
                                    $ref: '#/components/schemas/select-option';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['select'];
                                    description: 'The attribute type of the value.';
                                    example: 'select';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'option',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'string';
                                    description: 'A raw text field. Values are limited to 10MB.';
                                    example: 'Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['text'];
                                    description: 'The attribute type of the value.';
                                    example: 'text';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['timestamp'];
                                    description: 'The attribute type of the value.';
                                    example: 'timestamp';
                                  };
                                  value: {
                                    type: 'string';
                                    description: 'A timestamp value represents a single, universal moment in time using an ISO 8601 formatted string. This means that a timestamp consists of a date, a time (with nanosecond precision), and a time zone. Attio will coerce timestamps which do not provide full nanosecond precision and UTC is assumed if no time zone is provided. For example, "2023", "2023-01", "2023-01-02", "2023-01-02T13:00", "2023-01-02T13:00:00", and "2023-01-02T13:00:00.000000000" will all be coerced to "2023-01-02T13:00:00.000000000Z". Timestamps are always returned in UTC. For example, writing a timestamp value using the string "2023-01-02T13:00:00.000000000+02:00" will result in the value "2023-01-02T11:00:00.000000000Z" being returned. The maximum date is "9999-12-31T23:59:59.999999999Z".';
                                    format: 'date';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'attribute_type',
                                  'value',
                                ];
                                additionalProperties: false;
                              },
                            ];
                          };
                        };
                        description: 'A record type with an attribute `api_slug` as the key, and an array of value objects as the values.';
                      };
                    };
                    required: ['id', 'created_at', 'values'];
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '400': {
          description: 'Bad Request';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [400];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['missing_value'];
                  };
                  message: {
                    type: 'string';
                    example: 'Required value for attribute with ID "41252299-f8c7-4b5e-99c9-4ff8321d2f96" was not provided.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Bad Request';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Object with slug/ID "people" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
      };
    };
    delete: {
      summary: 'Delete a record';
      description: 'Deletes a single record (e.g. a company or person) by ID.\n\nRequired scopes: `object_configuration:read`, `record_permission:read-write`.';
      tags: ['Records'];
      security: [
        {
          oauth2: ['object_configuration:read', 'record_permission:read-write'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            description: 'The UUID or slug of the object the record belongs to.';
            example: 'people';
          };
          required: true;
          name: 'object';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'The UUID of the record to delete.';
            example: '891dcbfc-9141-415d-9b2a-2238a6cc012d';
          };
          required: true;
          name: 'record_id';
          in: 'path';
        },
      ];
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {};
                description: 'Success';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Record with ID "891dcbfc-9141-415d-9b2a-2238a6cc012d" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
      };
    };
  };
  '/v2/objects/{object}/records/{record_id}/attributes/{attribute}/values': {
    get: {
      summary: 'List record attribute values';
      description: 'Gets all values for a given attribute on a record. If the attribute is historic, this endpoint has the ability to return all historic values using the `show_historic` query param.\n\nRequired scopes: `record_permission:read`, `object_configuration:read`.';
      tags: ['Records'];
      security: [
        {
          oauth2: ['record_permission:read', 'object_configuration:read'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the object the record belongs to.';
            example: 'people';
          };
          required: true;
          name: 'object';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            format: 'uuid';
            description: 'A UUID to identify the record you want to query values on.';
            example: '891dcbfc-9141-415d-9b2a-2238a6cc012d';
          };
          required: true;
          name: 'record_id';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the attribute you want to query values on.';
            example: '41252299-f8c7-4b5e-99c9-4ff8321d2f96';
          };
          required: true;
          name: 'attribute';
          in: 'path';
        },
        {
          schema: {
            type: 'boolean';
            description: 'If `true`, the endpoint will return all historic values for the attribute. If `false`, the endpoint will only return the currently active value(s). Defaults to `false`. Can only be set to `true` for attributes which support historic data; the endpoint will throw if set to `true` for non-historic attributes.';
            example: true;
            default: false;
          };
          required: false;
          name: 'show_historic';
          in: 'query';
        },
        {
          schema: {
            type: 'integer';
            description: 'The maximum number of results to return. See the [full guide to pagination here](/docs/pagination).';
            example: 10;
          };
          required: false;
          name: 'limit';
          in: 'query';
        },
        {
          schema: {
            type: 'integer';
            description: 'The number of results to skip over before returning. See the [full guide to pagination here](/docs/pagination).';
            example: 5;
          };
          required: false;
          name: 'offset';
          in: 'query';
        },
      ];
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    type: 'array';
                    items: {
                      oneOf: [
                        {
                          type: 'object';
                          properties: {
                            active_from: {
                              type: 'string';
                              format: 'date-time';
                              description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            active_until: {
                              type: ['string', 'null'];
                              format: 'date-time';
                              description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            created_by_actor: {
                              type: 'object';
                              description: 'The actor that created this value.';
                              properties: {
                                id: {
                                  type: 'string';
                                  description: 'An ID to identify the actor.';
                                  nullable: true;
                                };
                                type: {
                                  type: 'string';
                                  enum: ['api-token', 'workspace-member', 'system'];
                                  nullable: true;
                                  description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                };
                              };
                              example: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                            };
                            referenced_actor_type: {
                              type: 'string';
                              enum: ['api-token', 'workspace-member', 'system'];
                              description: 'The type of the referenced actor. [Read more information on actor types here](/docs/actors).';
                              example: 'workspace-member';
                            };
                            referenced_actor_id: {
                              type: ['string', 'null'];
                              format: 'uuid';
                              description: 'The ID of the referenced actor.';
                              example: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                            };
                            attribute_type: {
                              type: 'string';
                              enum: ['actor-reference'];
                              description: 'The attribute type of the value.';
                              example: 'actor-reference';
                            };
                          };
                          required: [
                            'active_from',
                            'active_until',
                            'created_by_actor',
                            'referenced_actor_type',
                            'referenced_actor_id',
                            'attribute_type',
                          ];
                          additionalProperties: false;
                        },
                        {
                          type: 'object';
                          properties: {
                            active_from: {
                              type: 'string';
                              format: 'date-time';
                              description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            active_until: {
                              type: ['string', 'null'];
                              format: 'date-time';
                              description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            created_by_actor: {
                              type: 'object';
                              description: 'The actor that created this value.';
                              properties: {
                                id: {
                                  type: 'string';
                                  description: 'An ID to identify the actor.';
                                  nullable: true;
                                };
                                type: {
                                  type: 'string';
                                  enum: ['api-token', 'workspace-member', 'system'];
                                  nullable: true;
                                  description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                };
                              };
                              example: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                            };
                            value: {
                              type: 'boolean';
                              description: "A boolean representing whether the checkbox is checked or not. The string values 'true' and 'false' are also accepted.";
                              example: true;
                            };
                            attribute_type: {
                              type: 'string';
                              enum: ['checkbox'];
                              description: 'The attribute type of the value.';
                              example: 'checkbox';
                            };
                          };
                          required: ['active_from', 'active_until', 'created_by_actor', 'value', 'attribute_type'];
                          additionalProperties: false;
                        },
                        {
                          type: 'object';
                          properties: {
                            active_from: {
                              type: 'string';
                              format: 'date-time';
                              description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            active_until: {
                              type: ['string', 'null'];
                              format: 'date-time';
                              description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            created_by_actor: {
                              type: 'object';
                              description: 'The actor that created this value.';
                              properties: {
                                id: {
                                  type: 'string';
                                  description: 'An ID to identify the actor.';
                                  nullable: true;
                                };
                                type: {
                                  type: 'string';
                                  enum: ['api-token', 'workspace-member', 'system'];
                                  nullable: true;
                                  description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                };
                              };
                              example: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                            };
                            currency_value: {
                              type: 'number';
                              description: 'A numerical representation of the currency value. A decimal with a max of 4 decimal places.';
                              example: 99;
                            };
                            currency_code: {
                              type: ['string', 'null'];
                              enum: [
                                'EUR',
                                'GBP',
                                'USD',
                                'CAD',
                                'AUD',
                                'NZD',
                                'AED',
                                'INR',
                                'CHF',
                                'JPY',
                                'CNY',
                                'PHP',
                                'ILS',
                                'SGD',
                                'HKD',
                                'MYR',
                                'NTD',
                                'NOK',
                                'SEK',
                                'ISK',
                                'BEL',
                                'PLN',
                                'KRW',
                                'DKK',
                                'BRL',
                              ];
                              description: 'The ISO4217 currency code representing the currency that the value is stored in.';
                              example: 'USD';
                            };
                            attribute_type: {
                              type: 'string';
                              enum: ['currency'];
                              description: 'The attribute type of the value.';
                              example: 'currency';
                            };
                          };
                          required: [
                            'active_from',
                            'active_until',
                            'created_by_actor',
                            'currency_value',
                            'attribute_type',
                          ];
                          additionalProperties: false;
                        },
                        {
                          type: 'object';
                          properties: {
                            active_from: {
                              type: 'string';
                              format: 'date-time';
                              description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            active_until: {
                              type: ['string', 'null'];
                              format: 'date-time';
                              description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            created_by_actor: {
                              type: 'object';
                              description: 'The actor that created this value.';
                              properties: {
                                id: {
                                  type: 'string';
                                  description: 'An ID to identify the actor.';
                                  nullable: true;
                                };
                                type: {
                                  type: 'string';
                                  enum: ['api-token', 'workspace-member', 'system'];
                                  nullable: true;
                                  description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                };
                              };
                              example: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                            };
                            attribute_type: {
                              type: 'string';
                              enum: ['date'];
                              description: 'The attribute type of the value.';
                              example: 'date';
                            };
                            value: {
                              type: 'string';
                              description: 'A date represents a single calendar year, month and day, independent of timezone. If hours, months, seconds or timezones are provided, they will be trimmed. For example, "2023" and "2023-01" will be coerced into "2023-01-01", and "2023-01-02", "2023-01-02T13:00", "2023-01-02T14:00:00", "2023-01-02T15:00:00.000000000", and "2023-01-02T15:00:00.000000000+02:00" will all be coerced to "2023-01-02". If a timezone is provided that would result in a different calendar date in UTC, the date will be coerced to UTC and then the timezone component will be trimmed. For example, the value "2023-01-02T23:00:00-10:00" will be returned as "2023-01-03". The maximum date is "9999-12-31".';
                              example: '2023-01-01';
                            };
                          };
                          required: ['active_from', 'active_until', 'created_by_actor', 'attribute_type', 'value'];
                          additionalProperties: false;
                        },
                        {
                          type: 'object';
                          properties: {
                            active_from: {
                              type: 'string';
                              format: 'date-time';
                              description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            active_until: {
                              type: ['string', 'null'];
                              format: 'date-time';
                              description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            created_by_actor: {
                              type: 'object';
                              description: 'The actor that created this value.';
                              properties: {
                                id: {
                                  type: 'string';
                                  description: 'An ID to identify the actor.';
                                  nullable: true;
                                };
                                type: {
                                  type: 'string';
                                  enum: ['api-token', 'workspace-member', 'system'];
                                  nullable: true;
                                  description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                };
                              };
                              example: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                            };
                            domain: {
                              type: 'string';
                              example: 'app.attio.com';
                            };
                            root_domain: {
                              type: 'string';
                              example: 'attio.com';
                            };
                            attribute_type: {
                              type: 'string';
                              enum: ['domain'];
                              description: 'The attribute type of the value.';
                              example: 'domain';
                            };
                          };
                          required: [
                            'active_from',
                            'active_until',
                            'created_by_actor',
                            'domain',
                            'root_domain',
                            'attribute_type',
                          ];
                          additionalProperties: false;
                        },
                        {
                          type: 'object';
                          properties: {
                            active_from: {
                              type: 'string';
                              format: 'date-time';
                              description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            active_until: {
                              type: ['string', 'null'];
                              format: 'date-time';
                              description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            created_by_actor: {
                              type: 'object';
                              description: 'The actor that created this value.';
                              properties: {
                                id: {
                                  type: 'string';
                                  description: 'An ID to identify the actor.';
                                  nullable: true;
                                };
                                type: {
                                  type: 'string';
                                  enum: ['api-token', 'workspace-member', 'system'];
                                  nullable: true;
                                  description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                };
                              };
                              example: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                            };
                            original_email_address: {
                              type: 'string';
                              example: 'alice@app.attio.com';
                            };
                            email_address: {
                              type: 'string';
                              example: 'alice@app.attio.com';
                            };
                            email_domain: {
                              type: 'string';
                              example: 'app.attio.com';
                            };
                            email_root_domain: {
                              type: 'string';
                              example: 'attio.com';
                            };
                            email_local_specifier: {
                              type: 'string';
                              example: 'alice';
                            };
                            attribute_type: {
                              type: 'string';
                              enum: ['email-address'];
                              description: 'The attribute type of the value.';
                              example: 'email-address';
                            };
                          };
                          required: [
                            'active_from',
                            'active_until',
                            'created_by_actor',
                            'original_email_address',
                            'email_address',
                            'email_domain',
                            'email_root_domain',
                            'email_local_specifier',
                            'attribute_type',
                          ];
                        },
                        {
                          type: 'object';
                          properties: {
                            active_from: {
                              type: 'string';
                              format: 'date-time';
                              description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            active_until: {
                              type: ['string', 'null'];
                              format: 'date-time';
                              description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            created_by_actor: {
                              type: 'object';
                              description: 'The actor that created this value.';
                              properties: {
                                id: {
                                  type: 'string';
                                  description: 'An ID to identify the actor.';
                                  nullable: true;
                                };
                                type: {
                                  type: 'string';
                                  enum: ['api-token', 'workspace-member', 'system'];
                                  nullable: true;
                                  description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                };
                              };
                              example: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                            };
                            target_object: {
                              type: 'string';
                              description: 'A slug identifying the object that the referenced record belongs to.';
                              example: 'people';
                            };
                            target_record_id: {
                              type: 'string';
                              format: 'uuid';
                              description: 'A UUID to identify the referenced record.';
                              example: '891dcbfc-9141-415d-9b2a-2238a6cc012d';
                            };
                            attribute_type: {
                              type: 'string';
                              enum: ['record-reference'];
                              description: 'The attribute type of the value.';
                              example: 'record-reference';
                            };
                          };
                          required: [
                            'active_from',
                            'active_until',
                            'created_by_actor',
                            'target_object',
                            'target_record_id',
                            'attribute_type',
                          ];
                          additionalProperties: false;
                        },
                        {
                          type: 'object';
                          properties: {
                            active_from: {
                              type: 'string';
                              format: 'date-time';
                              description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            active_until: {
                              type: ['string', 'null'];
                              format: 'date-time';
                              description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            created_by_actor: {
                              type: 'object';
                              description: 'The actor that created this value.';
                              properties: {
                                id: {
                                  type: 'string';
                                  description: 'An ID to identify the actor.';
                                  nullable: true;
                                };
                                type: {
                                  type: 'string';
                                  enum: ['api-token', 'workspace-member', 'system'];
                                  nullable: true;
                                  description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                };
                              };
                              example: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                            };
                            interaction_type: {
                              type: 'string';
                              enum: ['calendar-event', 'call', 'chat-thread', 'email', 'in-person-meeting'];
                              description: 'The type of interaction e.g. calendar or email.';
                              example: 'email';
                            };
                            interacted_at: {
                              type: 'string';
                              format: 'date-time';
                              description: 'When the interaction occurred.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            owner_actor: {
                              type: 'object';
                              description: 'The actor that created this value.';
                              properties: {
                                id: {
                                  type: 'string';
                                  description: 'An ID to identify the actor.';
                                  nullable: true;
                                };
                                type: {
                                  type: 'string';
                                  enum: ['api-token', 'workspace-member', 'system'];
                                  nullable: true;
                                  description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                };
                              };
                              example: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                            };
                            attribute_type: {
                              type: 'string';
                              enum: ['interaction'];
                              description: 'The attribute type of the value.';
                              example: 'interaction';
                            };
                          };
                          required: [
                            'active_from',
                            'active_until',
                            'created_by_actor',
                            'interaction_type',
                            'interacted_at',
                            'owner_actor',
                            'attribute_type',
                          ];
                          additionalProperties: false;
                        },
                        {
                          type: 'object';
                          properties: {
                            active_from: {
                              type: 'string';
                              format: 'date-time';
                              description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            active_until: {
                              type: ['string', 'null'];
                              format: 'date-time';
                              description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            created_by_actor: {
                              type: 'object';
                              description: 'The actor that created this value.';
                              properties: {
                                id: {
                                  type: 'string';
                                  description: 'An ID to identify the actor.';
                                  nullable: true;
                                };
                                type: {
                                  type: 'string';
                                  enum: ['api-token', 'workspace-member', 'system'];
                                  nullable: true;
                                  description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                };
                              };
                              example: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                            };
                            line_1: {
                              type: ['string', 'null'];
                              description: 'The first line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                              example: '1 Infinite Loop';
                            };
                            line_2: {
                              type: ['string', 'null'];
                              description: 'The second line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                              example: 'Block 1';
                            };
                            line_3: {
                              type: ['string', 'null'];
                              description: 'The third line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                              example: 'Hilldrop Estate';
                            };
                            line_4: {
                              type: ['string', 'null'];
                              description: 'The fourth line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                              example: 'Westborough';
                            };
                            locality: {
                              type: ['string', 'null'];
                              description: 'The town, neighborhood or area the location is in.';
                              example: 'Cupertino';
                            };
                            region: {
                              type: ['string', 'null'];
                              description: 'The state, county, province or region that the location is in.';
                              example: 'CA';
                            };
                            postcode: {
                              type: ['string', 'null'];
                              description: 'The postcode or zip code for the location. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.}';
                              example: '95014';
                            };
                            country_code: {
                              type: ['string', 'null'];
                              enum: [
                                'AF',
                                'AX',
                                'AL',
                                'DZ',
                                'AS',
                                'AD',
                                'AO',
                                'AI',
                                'AQ',
                                'AG',
                                'AR',
                                'AM',
                                'AW',
                                'AU',
                                'AT',
                                'AZ',
                                'BS',
                                'BH',
                                'BD',
                                'BB',
                                'BY',
                                'BE',
                                'BZ',
                                'BJ',
                                'BM',
                                'BT',
                                'BO',
                                'BA',
                                'BW',
                                'BV',
                                'BR',
                                'IO',
                                'BN',
                                'BG',
                                'BF',
                                'BI',
                                'KH',
                                'CM',
                                'CA',
                                'CV',
                                'KY',
                                'CF',
                                'TD',
                                'CL',
                                'CN',
                                'CX',
                                'CC',
                                'CO',
                                'KM',
                                'CG',
                                'CD',
                                'CK',
                                'CR',
                                'CI',
                                'HR',
                                'CU',
                                'CW',
                                'CY',
                                'CZ',
                                'DK',
                                'DJ',
                                'DM',
                                'DO',
                                'EC',
                                'EG',
                                'SV',
                                'GQ',
                                'ER',
                                'EE',
                                'ET',
                                'FK',
                                'FO',
                                'FJ',
                                'FI',
                                'FR',
                                'GF',
                                'PF',
                                'TF',
                                'GA',
                                'GM',
                                'GE',
                                'DE',
                                'GH',
                                'GI',
                                'GR',
                                'GL',
                                'GD',
                                'GP',
                                'GU',
                                'GT',
                                'GG',
                                'GN',
                                'GW',
                                'GY',
                                'HT',
                                'HM',
                                'VA',
                                'HN',
                                'HK',
                                'HU',
                                'IS',
                                'IN',
                                'ID',
                                'IR',
                                'IQ',
                                'IE',
                                'IM',
                                'IL',
                                'IT',
                                'JM',
                                'JP',
                                'JE',
                                'JO',
                                'KZ',
                                'KE',
                                'KI',
                                'KR',
                                'KW',
                                'KG',
                                'LA',
                                'LV',
                                'LB',
                                'LS',
                                'LR',
                                'LY',
                                'LI',
                                'LT',
                                'LU',
                                'MO',
                                'MK',
                                'MG',
                                'MW',
                                'MY',
                                'MV',
                                'ML',
                                'MT',
                                'MH',
                                'MQ',
                                'MR',
                                'MU',
                                'YT',
                                'MX',
                                'FM',
                                'MD',
                                'MC',
                                'MN',
                                'ME',
                                'MS',
                                'MA',
                                'MZ',
                                'MM',
                                'NA',
                                'NR',
                                'NP',
                                'NL',
                                'AN',
                                'NC',
                                'NZ',
                                'NI',
                                'NE',
                                'NG',
                                'NU',
                                'NF',
                                'MP',
                                'NO',
                                'OM',
                                'PK',
                                'PW',
                                'PS',
                                'PA',
                                'PG',
                                'PY',
                                'PE',
                                'PH',
                                'PN',
                                'PL',
                                'PT',
                                'PR',
                                'QA',
                                'RE',
                                'RO',
                                'RU',
                                'RW',
                                'BL',
                                'SH',
                                'KN',
                                'LC',
                                'MF',
                                'PM',
                                'VC',
                                'WS',
                                'SM',
                                'ST',
                                'SA',
                                'SN',
                                'RS',
                                'SC',
                                'SL',
                                'SG',
                                'SK',
                                'SI',
                                'SB',
                                'SO',
                                'ZA',
                                'GS',
                                'ES',
                                'LK',
                                'SD',
                                'SR',
                                'SJ',
                                'SZ',
                                'SE',
                                'CH',
                                'SY',
                                'TW',
                                'TJ',
                                'TZ',
                                'TH',
                                'TL',
                                'TG',
                                'TK',
                                'TO',
                                'TT',
                                'TN',
                                'TR',
                                'TM',
                                'TC',
                                'TV',
                                'UG',
                                'UA',
                                'AE',
                                'GB',
                                'US',
                                'UM',
                                'UY',
                                'UZ',
                                'VU',
                                'VE',
                                'VN',
                                'VG',
                                'VI',
                                'WF',
                                'EH',
                                'YE',
                                'ZM',
                                'ZW',
                              ];
                              description: 'The ISO 3166-1 alpha-2 country code for the country this location is in.';
                              example: 'US';
                            };
                            latitude: {
                              type: ['string', 'null'];
                              pattern: '^[-+]?([1-8]?\\d(\\.\\d+)?|90(\\.0+)?)$';
                              description: 'The latitude of the location. Validated by the regular expression `/^[-+]?([1-8]?\\d(\\.\\d+)?|90(\\.0+)?)$/`. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.}';
                              example: '37.331741';
                            };
                            longitude: {
                              type: ['string', 'null'];
                              pattern: '^[-+]?(180(\\.0+)?|((1[0-7]\\d)|([1-9]?\\d))(\\.\\d+)?)$';
                              description: 'The longitude of the location. Validated by the regular expression `/^[-+]?(180(\\.0+)?|((1[0-7]\\d)|([1-9]?\\d))(\\.\\d+)?)$/`';
                              example: '-122.030333';
                            };
                            attribute_type: {
                              type: 'string';
                              enum: ['location'];
                              description: 'The attribute type of the value.';
                              example: 'location';
                            };
                          };
                          required: [
                            'active_from',
                            'active_until',
                            'created_by_actor',
                            'line_1',
                            'line_2',
                            'line_3',
                            'line_4',
                            'locality',
                            'region',
                            'postcode',
                            'country_code',
                            'latitude',
                            'longitude',
                            'attribute_type',
                          ];
                          additionalProperties: false;
                        },
                        {
                          type: 'object';
                          properties: {
                            active_from: {
                              type: 'string';
                              format: 'date-time';
                              description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            active_until: {
                              type: ['string', 'null'];
                              format: 'date-time';
                              description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            created_by_actor: {
                              type: 'object';
                              description: 'The actor that created this value.';
                              properties: {
                                id: {
                                  type: 'string';
                                  description: 'An ID to identify the actor.';
                                  nullable: true;
                                };
                                type: {
                                  type: 'string';
                                  enum: ['api-token', 'workspace-member', 'system'];
                                  nullable: true;
                                  description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                };
                              };
                              example: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                            };
                            value: {
                              type: 'number';
                              description: 'Numbers are persisted as 64 bit floats.';
                              example: 42;
                            };
                            attribute_type: {
                              type: 'string';
                              enum: ['number'];
                              description: 'The attribute type of the value.';
                              example: 'number';
                            };
                          };
                          required: ['active_from', 'active_until', 'created_by_actor', 'value', 'attribute_type'];
                        },
                        {
                          type: 'object';
                          properties: {
                            active_from: {
                              type: 'string';
                              format: 'date-time';
                              description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            active_until: {
                              type: ['string', 'null'];
                              format: 'date-time';
                              description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            created_by_actor: {
                              type: 'object';
                              description: 'The actor that created this value.';
                              properties: {
                                id: {
                                  type: 'string';
                                  description: 'An ID to identify the actor.';
                                  nullable: true;
                                };
                                type: {
                                  type: 'string';
                                  enum: ['api-token', 'workspace-member', 'system'];
                                  nullable: true;
                                  description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                };
                              };
                              example: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                            };
                            first_name: {
                              type: 'string';
                              description: 'The first name.';
                              example: 'Ada';
                            };
                            last_name: {
                              type: 'string';
                              description: 'The last name.';
                              example: 'Lovelace';
                            };
                            full_name: {
                              type: 'string';
                              description: 'The full name.';
                              example: 'Ada Lovelace';
                            };
                            attribute_type: {
                              type: 'string';
                              enum: ['personal-name'];
                              description: 'The attribute type of the value.';
                              example: 'personal-name';
                            };
                          };
                          required: [
                            'active_from',
                            'active_until',
                            'created_by_actor',
                            'first_name',
                            'last_name',
                            'full_name',
                            'attribute_type',
                          ];
                          additionalProperties: false;
                        },
                        {
                          type: 'object';
                          properties: {
                            active_from: {
                              type: 'string';
                              format: 'date-time';
                              description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            active_until: {
                              type: ['string', 'null'];
                              format: 'date-time';
                              description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            created_by_actor: {
                              type: 'object';
                              description: 'The actor that created this value.';
                              properties: {
                                id: {
                                  type: 'string';
                                  description: 'An ID to identify the actor.';
                                  nullable: true;
                                };
                                type: {
                                  type: 'string';
                                  enum: ['api-token', 'workspace-member', 'system'];
                                  nullable: true;
                                  description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                };
                              };
                              example: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                            };
                            original_phone_number: {
                              type: 'string';
                              description: 'The raw, original phone number, as inputted.';
                              example: '5558675309';
                            };
                            country_code: {
                              type: 'string';
                              description: 'The ISO 3166-1 alpha-2 country code representing the country that this phone number belongs to.';
                              enum: [
                                'AF',
                                'AX',
                                'AL',
                                'DZ',
                                'AS',
                                'AD',
                                'AO',
                                'AI',
                                'AQ',
                                'AG',
                                'AR',
                                'AM',
                                'AW',
                                'AU',
                                'AT',
                                'AZ',
                                'BS',
                                'BH',
                                'BD',
                                'BB',
                                'BY',
                                'BE',
                                'BZ',
                                'BJ',
                                'BM',
                                'BT',
                                'BO',
                                'BA',
                                'BW',
                                'BV',
                                'BR',
                                'IO',
                                'BN',
                                'BG',
                                'BF',
                                'BI',
                                'KH',
                                'CM',
                                'CA',
                                'CV',
                                'KY',
                                'CF',
                                'TD',
                                'CL',
                                'CN',
                                'CX',
                                'CC',
                                'CO',
                                'KM',
                                'CG',
                                'CD',
                                'CK',
                                'CR',
                                'CI',
                                'HR',
                                'CU',
                                'CW',
                                'CY',
                                'CZ',
                                'DK',
                                'DJ',
                                'DM',
                                'DO',
                                'EC',
                                'EG',
                                'SV',
                                'GQ',
                                'ER',
                                'EE',
                                'ET',
                                'FK',
                                'FO',
                                'FJ',
                                'FI',
                                'FR',
                                'GF',
                                'PF',
                                'TF',
                                'GA',
                                'GM',
                                'GE',
                                'DE',
                                'GH',
                                'GI',
                                'GR',
                                'GL',
                                'GD',
                                'GP',
                                'GU',
                                'GT',
                                'GG',
                                'GN',
                                'GW',
                                'GY',
                                'HT',
                                'HM',
                                'VA',
                                'HN',
                                'HK',
                                'HU',
                                'IS',
                                'IN',
                                'ID',
                                'IR',
                                'IQ',
                                'IE',
                                'IM',
                                'IL',
                                'IT',
                                'JM',
                                'JP',
                                'JE',
                                'JO',
                                'KZ',
                                'KE',
                                'KI',
                                'KR',
                                'KW',
                                'KG',
                                'LA',
                                'LV',
                                'LB',
                                'LS',
                                'LR',
                                'LY',
                                'LI',
                                'LT',
                                'LU',
                                'MO',
                                'MK',
                                'MG',
                                'MW',
                                'MY',
                                'MV',
                                'ML',
                                'MT',
                                'MH',
                                'MQ',
                                'MR',
                                'MU',
                                'YT',
                                'MX',
                                'FM',
                                'MD',
                                'MC',
                                'MN',
                                'ME',
                                'MS',
                                'MA',
                                'MZ',
                                'MM',
                                'NA',
                                'NR',
                                'NP',
                                'NL',
                                'AN',
                                'NC',
                                'NZ',
                                'NI',
                                'NE',
                                'NG',
                                'NU',
                                'NF',
                                'MP',
                                'NO',
                                'OM',
                                'PK',
                                'PW',
                                'PS',
                                'PA',
                                'PG',
                                'PY',
                                'PE',
                                'PH',
                                'PN',
                                'PL',
                                'PT',
                                'PR',
                                'QA',
                                'RE',
                                'RO',
                                'RU',
                                'RW',
                                'BL',
                                'SH',
                                'KN',
                                'LC',
                                'MF',
                                'PM',
                                'VC',
                                'WS',
                                'SM',
                                'ST',
                                'SA',
                                'SN',
                                'RS',
                                'SC',
                                'SL',
                                'SG',
                                'SK',
                                'SI',
                                'SB',
                                'SO',
                                'ZA',
                                'GS',
                                'ES',
                                'LK',
                                'SD',
                                'SR',
                                'SJ',
                                'SZ',
                                'SE',
                                'CH',
                                'SY',
                                'TW',
                                'TJ',
                                'TZ',
                                'TH',
                                'TL',
                                'TG',
                                'TK',
                                'TO',
                                'TT',
                                'TN',
                                'TR',
                                'TM',
                                'TC',
                                'TV',
                                'UG',
                                'UA',
                                'AE',
                                'GB',
                                'US',
                                'UM',
                                'UY',
                                'UZ',
                                'VU',
                                'VE',
                                'VN',
                                'VG',
                                'VI',
                                'WF',
                                'EH',
                                'YE',
                                'ZM',
                                'ZW',
                              ];
                              example: 'US';
                            };
                            phone_number: {
                              type: 'string';
                              example: '+15558675309';
                            };
                            attribute_type: {
                              type: 'string';
                              enum: ['phone-number'];
                              description: 'The attribute type of the value.';
                              example: 'phone-number';
                            };
                          };
                          required: [
                            'active_from',
                            'active_until',
                            'created_by_actor',
                            'original_phone_number',
                            'country_code',
                            'phone_number',
                            'attribute_type',
                          ];
                        },
                        {
                          type: 'object';
                          properties: {
                            active_from: {
                              type: 'string';
                              format: 'date-time';
                              description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            active_until: {
                              type: ['string', 'null'];
                              format: 'date-time';
                              description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            created_by_actor: {
                              type: 'object';
                              description: 'The actor that created this value.';
                              properties: {
                                id: {
                                  type: 'string';
                                  description: 'An ID to identify the actor.';
                                  nullable: true;
                                };
                                type: {
                                  type: 'string';
                                  enum: ['api-token', 'workspace-member', 'system'];
                                  nullable: true;
                                  description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                };
                              };
                              example: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                            };
                            status: {
                              $ref: '#/components/schemas/status';
                            };
                            attribute_type: {
                              type: 'string';
                              enum: ['status'];
                              description: 'The attribute type of the value.';
                              example: 'status';
                            };
                          };
                          required: ['active_from', 'active_until', 'created_by_actor', 'status', 'attribute_type'];
                          additionalProperties: false;
                        },
                        {
                          type: 'object';
                          properties: {
                            active_from: {
                              type: 'string';
                              format: 'date-time';
                              description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            active_until: {
                              type: ['string', 'null'];
                              format: 'date-time';
                              description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            created_by_actor: {
                              type: 'object';
                              description: 'The actor that created this value.';
                              properties: {
                                id: {
                                  type: 'string';
                                  description: 'An ID to identify the actor.';
                                  nullable: true;
                                };
                                type: {
                                  type: 'string';
                                  enum: ['api-token', 'workspace-member', 'system'];
                                  nullable: true;
                                  description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                };
                              };
                              example: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                            };
                            value: {
                              type: 'number';
                              description: 'A number between 0 and 5 (inclusive) to represent a star rating.';
                              example: 3;
                            };
                            attribute_type: {
                              type: 'string';
                              enum: ['rating'];
                              description: 'The attribute type of the value.';
                              example: 'rating';
                            };
                          };
                          required: ['active_from', 'active_until', 'created_by_actor', 'value', 'attribute_type'];
                          additionalProperties: false;
                        },
                        {
                          type: 'object';
                          properties: {
                            active_from: {
                              type: 'string';
                              format: 'date-time';
                              description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            active_until: {
                              type: ['string', 'null'];
                              format: 'date-time';
                              description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            created_by_actor: {
                              type: 'object';
                              description: 'The actor that created this value.';
                              properties: {
                                id: {
                                  type: 'string';
                                  description: 'An ID to identify the actor.';
                                  nullable: true;
                                };
                                type: {
                                  type: 'string';
                                  enum: ['api-token', 'workspace-member', 'system'];
                                  nullable: true;
                                  description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                };
                              };
                              example: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                            };
                            option: {
                              $ref: '#/components/schemas/select-option';
                            };
                            attribute_type: {
                              type: 'string';
                              enum: ['select'];
                              description: 'The attribute type of the value.';
                              example: 'select';
                            };
                          };
                          required: ['active_from', 'active_until', 'created_by_actor', 'option', 'attribute_type'];
                          additionalProperties: false;
                        },
                        {
                          type: 'object';
                          properties: {
                            active_from: {
                              type: 'string';
                              format: 'date-time';
                              description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            active_until: {
                              type: ['string', 'null'];
                              format: 'date-time';
                              description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            created_by_actor: {
                              type: 'object';
                              description: 'The actor that created this value.';
                              properties: {
                                id: {
                                  type: 'string';
                                  description: 'An ID to identify the actor.';
                                  nullable: true;
                                };
                                type: {
                                  type: 'string';
                                  enum: ['api-token', 'workspace-member', 'system'];
                                  nullable: true;
                                  description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                };
                              };
                              example: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                            };
                            value: {
                              type: 'string';
                              description: 'A raw text field. Values are limited to 10MB.';
                              example: 'Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.';
                            };
                            attribute_type: {
                              type: 'string';
                              enum: ['text'];
                              description: 'The attribute type of the value.';
                              example: 'text';
                            };
                          };
                          required: ['active_from', 'active_until', 'created_by_actor', 'value', 'attribute_type'];
                          additionalProperties: false;
                        },
                        {
                          type: 'object';
                          properties: {
                            active_from: {
                              type: 'string';
                              format: 'date-time';
                              description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            active_until: {
                              type: ['string', 'null'];
                              format: 'date-time';
                              description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                            created_by_actor: {
                              type: 'object';
                              description: 'The actor that created this value.';
                              properties: {
                                id: {
                                  type: 'string';
                                  description: 'An ID to identify the actor.';
                                  nullable: true;
                                };
                                type: {
                                  type: 'string';
                                  enum: ['api-token', 'workspace-member', 'system'];
                                  nullable: true;
                                  description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                };
                              };
                              example: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                            };
                            attribute_type: {
                              type: 'string';
                              enum: ['timestamp'];
                              description: 'The attribute type of the value.';
                              example: 'timestamp';
                            };
                            value: {
                              type: 'string';
                              description: 'A timestamp value represents a single, universal moment in time using an ISO 8601 formatted string. This means that a timestamp consists of a date, a time (with nanosecond precision), and a time zone. Attio will coerce timestamps which do not provide full nanosecond precision and UTC is assumed if no time zone is provided. For example, "2023", "2023-01", "2023-01-02", "2023-01-02T13:00", "2023-01-02T13:00:00", and "2023-01-02T13:00:00.000000000" will all be coerced to "2023-01-02T13:00:00.000000000Z". Timestamps are always returned in UTC. For example, writing a timestamp value using the string "2023-01-02T13:00:00.000000000+02:00" will result in the value "2023-01-02T11:00:00.000000000Z" being returned. The maximum date is "9999-12-31T23:59:59.999999999Z".';
                              format: 'date';
                              example: '2023-01-01T15:00:00.000000000Z';
                            };
                          };
                          required: ['active_from', 'active_until', 'created_by_actor', 'attribute_type', 'value'];
                          additionalProperties: false;
                        },
                      ];
                    };
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '400': {
          description: 'Bad Request';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [400];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['validation_type'];
                  };
                  message: {
                    type: 'string';
                    example: 'Cannot set show_historic param to true when querying a non-historic attribute.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Bad Request';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Attribute with slug/ID "my-attribute" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
      };
    };
  };
  '/v2/objects/{object}/records/{record_id}/entries': {
    get: {
      summary: 'List record entries';
      description: 'List all entries, across all lists, for which this record is the parent.\n\nRequired scopes: `record_permission:read`, `object_configuration:read`, `list_entry:read`.';
      tags: ['Records'];
      security: [
        {
          oauth2: ['record_permission:read', 'object_configuration:read', 'list_entry:read'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug identifying the object that the record belongs to.';
            example: 'people';
          };
          required: true;
          name: 'object';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            format: 'uuid';
            description: 'A UUID identifying the record.';
            example: '891dcbfc-9141-415d-9b2a-2238a6cc012d';
          };
          required: true;
          name: 'record_id';
          in: 'path';
        },
        {
          schema: {
            type: 'integer';
            description: 'The maximum number of results to return. The default is `100` and the maximum is `1000`. See the [full guide to pagination here](/docs/pagination).';
            example: 10;
          };
          required: false;
          name: 'limit';
          in: 'query';
        },
        {
          schema: {
            type: 'integer';
            description: 'The number of results to skip over before returning. The default is `0`. See the [full guide to pagination here](/docs/pagination).';
            example: 5;
          };
          required: false;
          name: 'offset';
          in: 'query';
        },
      ];
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    type: 'array';
                    items: {
                      type: 'object';
                      properties: {
                        list_id: {
                          type: 'string';
                          format: 'uuid';
                          description: 'A UUID identifying the list that this record is in.';
                          example: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
                        };
                        list_api_slug: {
                          type: 'string';
                          description: 'A human-readable slug for the list for use in URLs and responses.';
                          example: 'hiring-engineering';
                        };
                        entry_id: {
                          type: 'string';
                          format: 'uuid';
                          description: 'A UUID identifying this entry on the list.';
                          example: '2e6e29ea-c4e0-4f44-842d-78a891f8c156';
                        };
                        created_at: {
                          type: 'string';
                          description: 'When this entry was created.';
                          example: '2022-11-21T13:22:49.061281000Z';
                        };
                      };
                      required: ['list_id', 'list_api_slug', 'entry_id', 'created_at'];
                    };
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
      };
    };
  };
  '/v2/lists': {
    get: {
      summary: 'List all lists';
      description: 'List all lists that your access token has access to. lists are returned in the order that they are sorted in the sidebar.\n\nRequired scopes: `list_configuration:read`.';
      tags: ['Lists'];
      security: [
        {
          oauth2: ['list_configuration:read'];
        },
      ];
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    type: 'array';
                    items: {
                      $ref: '#/components/schemas/list';
                    };
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
      };
    };
    post: {
      summary: 'Create a list';
      description: 'Creates a new list.\n\nOnce you have your list, add attributes to it using the [Create attribute](/reference/post_v2-target-identifier-attributes) API, and add records to it using the [Add records to list](/reference/post_v2-lists-list-entries) API. \n\nNew lists must specify which records can be added with the `parent_object` parameter which accepts either an object slug or an object ID. Permissions for the list are controlled with the `workspace_access` and `workspace_member_access` parameters.\n\nPlease note that new lists must have either `workspace_access` set to `"full-access"` or one or more element of `workspace_member_access` with a `"full-access"` level. It is also possible to receive a `403` billing error if your workspace is not on a plan that supports either advanced workspace or workspace member-level access for lists.\n\nRequired scopes: `list_configuration:read-write`.';
      tags: ['Lists'];
      security: [
        {
          oauth2: ['list_configuration:read-write'];
        },
      ];
      requestBody: {
        required: true;
        content: {
          'application/json': {
            schema: {
              type: 'object';
              properties: {
                data: {
                  type: 'object';
                  properties: {
                    name: {
                      type: 'string';
                      description: 'The human-readable name of the list.';
                      example: 'Enterprise Sales';
                    };
                    api_slug: {
                      type: 'string';
                      description: 'A unique, human-readable slug to access the list through API calls. Should be formatted in snake case.';
                      example: 'enterprise_sales';
                    };
                    parent_object: {
                      type: 'string';
                      description: 'A UUID or slug to identify the allowed object type for records added to this list.';
                      example: 'people';
                    };
                    workspace_access: {
                      type: ['string', 'null'];
                      enum: ['full-access', 'read-and-write', 'read-only'];
                      description: 'The level of access granted to all members of the workspace for this list. Pass `null` to keep the list private and only grant access to specific workspace members.';
                      example: 'read-and-write';
                    };
                    workspace_member_access: {
                      type: 'array';
                      items: {
                        type: 'object';
                        properties: {
                          workspace_member_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID to identify the workspace member to grant access to.';
                            example: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                          };
                          level: {
                            type: 'string';
                            enum: ['full-access', 'read-and-write', 'read-only'];
                            description: 'The level of access to the list.';
                            example: 'read-and-write';
                          };
                        };
                        required: ['workspace_member_id', 'level'];
                      };
                      description: 'The level of access granted to specific workspace members for this list. Pass an empty array to grant access to no workspace members.';
                    };
                  };
                  required: ['name', 'api_slug', 'parent_object', 'workspace_access', 'workspace_member_access'];
                };
              };
              required: ['data'];
            };
          };
        };
      };
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    $ref: '#/components/schemas/list';
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '400': {
          description: 'Bad Request';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [400];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['value_not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Workspace member with ID "50cf242c-7fa3-4cad-87d0-75b1af71c57b" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Bad Request';
              };
            };
          };
        };
        '403': {
          description: 'Forbidden';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [403];
                  };
                  type: {
                    type: 'string';
                    enum: ['auth_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['billing_error'];
                  };
                  message: {
                    type: 'string';
                    example: 'You attempted to create a List using permissions features that are not currently supported on your plan. Please upgrade, contact sales, or try a different access configuration for your list.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Forbidden';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Parent object with slug/ID "people" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
      };
    };
  };
  '/v2/lists/{list}': {
    get: {
      summary: 'Get a list';
      description: 'Gets a single list in your workspace that your access token has access to.\n\nRequired scopes: `list_configuration:read`.';
      tags: ['Lists'];
      security: [
        {
          oauth2: ['list_configuration:read'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the list.';
            example: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
          };
          required: true;
          name: 'list';
          in: 'path';
        },
      ];
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    $ref: '#/components/schemas/list';
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'List with slug/ID "enterprise_sales" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
      };
    };
    patch: {
      summary: 'Update a list';
      description: 'Updates an existing list. Permissions for the list are controlled with the `workspace_access` and `workspace_member_access` parameters. Please note that lists must have either `workspace_access` set to `"full-access"` or one or more element of `workspace_member_access` with a `"full-access"` level. It is also possible to receive a `403` billing error if your workspace is not on a plan that supports either advanced workspace or workspace member level access for lists. Changing the parent object of a list is not possible through the API as it can have unintended side-effects that should be considered carefully. If you wish to carry out a parent object change you should do so through the UI.\n\nRequired scopes: `list_configuration:read-write`.';
      tags: ['Lists'];
      security: [
        {
          oauth2: ['list_configuration:read-write'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the list to update.';
            example: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
          };
          required: true;
          name: 'list';
          in: 'path';
        },
      ];
      requestBody: {
        required: true;
        content: {
          'application/json': {
            schema: {
              type: 'object';
              properties: {
                data: {
                  type: 'object';
                  properties: {
                    name: {
                      type: 'string';
                      description: 'The human-readable name of the list.';
                      example: 'Enterprise Sales';
                    };
                    api_slug: {
                      type: 'string';
                      description: 'A unique, human-readable slug to access the list through API calls. Should be formatted in snake case.';
                      example: 'enterprise_sales';
                    };
                    workspace_access: {
                      type: ['string', 'null'];
                      enum: ['full-access', 'read-and-write', 'read-only'];
                      description: 'The level of access granted to all members of the workspace for this list. Pass `null` to keep the list private and only grant access to specific workspace members.';
                      example: 'read-and-write';
                    };
                    workspace_member_access: {
                      type: 'array';
                      items: {
                        type: 'object';
                        properties: {
                          workspace_member_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID to identify the workspace member to grant access to.';
                            example: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                          };
                          level: {
                            type: 'string';
                            enum: ['full-access', 'read-and-write', 'read-only'];
                            description: 'The level of access to the list.';
                            example: 'read-and-write';
                          };
                        };
                        required: ['workspace_member_id', 'level'];
                      };
                      description: 'The level of access granted to specific workspace members for this list. Pass an empty array to grant access to no workspace members.';
                    };
                  };
                };
              };
              required: ['data'];
            };
          };
        };
      };
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    $ref: '#/components/schemas/list';
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '400': {
          description: 'Bad Request';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [400];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['value_not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Workspace member with ID "50cf242c-7fa3-4cad-87d0-75b1af71c57b" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Bad Request';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'List with slug/ID "enterprise_sales" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
      };
    };
  };
  '/v2/lists/{list}/entries/query': {
    post: {
      summary: 'List entries';
      description: 'Lists entries in a given list, with the option to filter and sort results.\n\nRequired scopes: `list_entry:read`, `list_configuration:read`.';
      tags: ['Entries'];
      security: [
        {
          oauth2: ['list_entry:read', 'list_configuration:read'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug to identify the list to retrieve entries from.';
            example: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
          };
          required: true;
          name: 'list';
          in: 'path';
        },
      ];
      requestBody: {
        required: true;
        content: {
          'application/json': {
            schema: {
              type: 'object';
              properties: {
                filter: {
                  type: 'object';
                  description: 'An object used to filter results to a subset of results. See the [full guide to filtering and sorting here](/docs/filtering-and-sorting).';
                  additionalProperties: true;
                  example: {
                    name: 'Ada Lovelace';
                  };
                };
                sorts: {
                  type: 'array';
                  items: {
                    anyOf: [
                      {
                        type: 'object';
                        properties: {
                          direction: {
                            type: 'string';
                            enum: ['asc', 'desc'];
                            description: 'The direction to sort the results by.';
                          };
                          attribute: {
                            type: 'string';
                            description: 'A slug or ID to identify the attribute to sort by.';
                          };
                          field: {
                            type: 'string';
                            description: 'Which field on the value to sort by e.g. "last_name" on a name value.';
                          };
                        };
                        required: ['direction', 'attribute'];
                        description: 'Sort by attribute';
                      },
                      {
                        type: 'object';
                        properties: {
                          direction: {
                            type: 'string';
                            enum: ['asc', 'desc'];
                            description: 'The direction to sort the results by.';
                          };
                          path: {
                            type: 'array';
                            items: {
                              type: 'array';
                              items: {
                                anyOf: [
                                  {
                                    type: 'string';
                                    description: 'The slug or ID of the object e.g. "people".';
                                  },
                                  {
                                    type: 'string';
                                    description: 'A slug or ID to identify the attribute to sort by.';
                                  },
                                ];
                              };
                              minItems: 2;
                              maxItems: 2;
                            };
                            description: "You may use the `path` property to traverse record reference attributes and parent records on list entries. `path` accepts an array of tuples where the first element of each tuple is the slug or ID of a list/object, and the second element is the slug or ID of an attribute on that list/object. The first element of the first tuple must correspond to the list or object that you are querying. For example, if you wanted to sort by the name of the parent record (a company) on a list with the slug \"sales\", you would pass the value `[['sales', 'parent_record'], ['companies', 'name']]`.";
                          };
                          field: {
                            type: 'string';
                            description: 'Which field on the value to sort by e.g. "last_name" on a name value.';
                          };
                        };
                        required: ['direction', 'path'];
                        description: 'Sort by path';
                      },
                    ];
                  };
                  description: 'An object used to sort results. See the [full guide to filtering and sorting here](/docs/filtering-and-sorting).';
                  example: [
                    {
                      direction: 'asc';
                      attribute: 'name';
                      field: 'last_name';
                    },
                  ];
                };
                limit: {
                  type: 'number';
                  description: 'The maximum number of results to return. Defaults to 500. See the [full guide to pagination here](/docs/pagination).';
                  example: 500;
                };
                offset: {
                  type: 'number';
                  description: 'The number of results to skip over before returning. Defaults to 0. See the [full guide to pagination here](/docs/pagination).';
                  example: 0;
                };
              };
            };
          };
        };
      };
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    type: 'array';
                    items: {
                      type: 'object';
                      properties: {
                        id: {
                          type: 'object';
                          properties: {
                            workspace_id: {
                              type: 'string';
                              format: 'uuid';
                              description: 'A UUID identifying the workspace this entry belongs to.';
                              example: '14beef7a-99f7-4534-a87e-70b564330a4c';
                            };
                            list_id: {
                              type: 'string';
                              format: 'uuid';
                              description: 'A UUID identifying the list this entry is in.';
                              example: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
                            };
                            entry_id: {
                              type: 'string';
                              format: 'uuid';
                              description: 'A UUID identifying this entry.';
                              example: '2e6e29ea-c4e0-4f44-842d-78a891f8c156';
                            };
                          };
                          required: ['workspace_id', 'list_id', 'entry_id'];
                        };
                        parent_record_id: {
                          type: 'string';
                          format: 'uuid';
                          description: 'A UUID identifying the record that is parent of the list entry.';
                          example: '891dcbfc-9141-415d-9b2a-2238a6cc012d';
                        };
                        parent_object: {
                          type: 'string';
                          description: 'A UUID or slug identifying the object that the parent record belongs to.';
                          example: 'people';
                        };
                        created_at: {
                          type: 'string';
                          description: 'When this entry was created.';
                          example: '2022-11-21T13:22:49.061281000Z';
                        };
                        entry_values: {
                          type: 'object';
                          additionalProperties: {
                            type: 'array';
                            items: {
                              oneOf: [
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    referenced_actor_type: {
                                      type: 'string';
                                      enum: ['api-token', 'workspace-member', 'system'];
                                      description: 'The type of the referenced actor. [Read more information on actor types here](/docs/actors).';
                                      example: 'workspace-member';
                                    };
                                    referenced_actor_id: {
                                      type: ['string', 'null'];
                                      format: 'uuid';
                                      description: 'The ID of the referenced actor.';
                                      example: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['actor-reference'];
                                      description: 'The attribute type of the value.';
                                      example: 'actor-reference';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'referenced_actor_type',
                                    'referenced_actor_id',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    value: {
                                      type: 'boolean';
                                      description: "A boolean representing whether the checkbox is checked or not. The string values 'true' and 'false' are also accepted.";
                                      example: true;
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['checkbox'];
                                      description: 'The attribute type of the value.';
                                      example: 'checkbox';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'value',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    currency_value: {
                                      type: 'number';
                                      description: 'A numerical representation of the currency value. A decimal with a max of 4 decimal places.';
                                      example: 99;
                                    };
                                    currency_code: {
                                      type: ['string', 'null'];
                                      enum: [
                                        'EUR',
                                        'GBP',
                                        'USD',
                                        'CAD',
                                        'AUD',
                                        'NZD',
                                        'AED',
                                        'INR',
                                        'CHF',
                                        'JPY',
                                        'CNY',
                                        'PHP',
                                        'ILS',
                                        'SGD',
                                        'HKD',
                                        'MYR',
                                        'NTD',
                                        'NOK',
                                        'SEK',
                                        'ISK',
                                        'BEL',
                                        'PLN',
                                        'KRW',
                                        'DKK',
                                        'BRL',
                                      ];
                                      description: 'The ISO4217 currency code representing the currency that the value is stored in.';
                                      example: 'USD';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['currency'];
                                      description: 'The attribute type of the value.';
                                      example: 'currency';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'currency_value',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['date'];
                                      description: 'The attribute type of the value.';
                                      example: 'date';
                                    };
                                    value: {
                                      type: 'string';
                                      description: 'A date represents a single calendar year, month and day, independent of timezone. If hours, months, seconds or timezones are provided, they will be trimmed. For example, "2023" and "2023-01" will be coerced into "2023-01-01", and "2023-01-02", "2023-01-02T13:00", "2023-01-02T14:00:00", "2023-01-02T15:00:00.000000000", and "2023-01-02T15:00:00.000000000+02:00" will all be coerced to "2023-01-02". If a timezone is provided that would result in a different calendar date in UTC, the date will be coerced to UTC and then the timezone component will be trimmed. For example, the value "2023-01-02T23:00:00-10:00" will be returned as "2023-01-03". The maximum date is "9999-12-31".';
                                      example: '2023-01-01';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'attribute_type',
                                    'value',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    domain: {
                                      type: 'string';
                                      example: 'app.attio.com';
                                    };
                                    root_domain: {
                                      type: 'string';
                                      example: 'attio.com';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['domain'];
                                      description: 'The attribute type of the value.';
                                      example: 'domain';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'domain',
                                    'root_domain',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    original_email_address: {
                                      type: 'string';
                                      example: 'alice@app.attio.com';
                                    };
                                    email_address: {
                                      type: 'string';
                                      example: 'alice@app.attio.com';
                                    };
                                    email_domain: {
                                      type: 'string';
                                      example: 'app.attio.com';
                                    };
                                    email_root_domain: {
                                      type: 'string';
                                      example: 'attio.com';
                                    };
                                    email_local_specifier: {
                                      type: 'string';
                                      example: 'alice';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['email-address'];
                                      description: 'The attribute type of the value.';
                                      example: 'email-address';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'original_email_address',
                                    'email_address',
                                    'email_domain',
                                    'email_root_domain',
                                    'email_local_specifier',
                                    'attribute_type',
                                  ];
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    target_object: {
                                      type: 'string';
                                      description: 'A slug identifying the object that the referenced record belongs to.';
                                      example: 'people';
                                    };
                                    target_record_id: {
                                      type: 'string';
                                      format: 'uuid';
                                      description: 'A UUID to identify the referenced record.';
                                      example: '891dcbfc-9141-415d-9b2a-2238a6cc012d';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['record-reference'];
                                      description: 'The attribute type of the value.';
                                      example: 'record-reference';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'target_object',
                                    'target_record_id',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    interaction_type: {
                                      type: 'string';
                                      enum: ['calendar-event', 'call', 'chat-thread', 'email', 'in-person-meeting'];
                                      description: 'The type of interaction e.g. calendar or email.';
                                      example: 'email';
                                    };
                                    interacted_at: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'When the interaction occurred.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    owner_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['interaction'];
                                      description: 'The attribute type of the value.';
                                      example: 'interaction';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'interaction_type',
                                    'interacted_at',
                                    'owner_actor',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    line_1: {
                                      type: ['string', 'null'];
                                      description: 'The first line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                      example: '1 Infinite Loop';
                                    };
                                    line_2: {
                                      type: ['string', 'null'];
                                      description: 'The second line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                      example: 'Block 1';
                                    };
                                    line_3: {
                                      type: ['string', 'null'];
                                      description: 'The third line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                      example: 'Hilldrop Estate';
                                    };
                                    line_4: {
                                      type: ['string', 'null'];
                                      description: 'The fourth line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                      example: 'Westborough';
                                    };
                                    locality: {
                                      type: ['string', 'null'];
                                      description: 'The town, neighborhood or area the location is in.';
                                      example: 'Cupertino';
                                    };
                                    region: {
                                      type: ['string', 'null'];
                                      description: 'The state, county, province or region that the location is in.';
                                      example: 'CA';
                                    };
                                    postcode: {
                                      type: ['string', 'null'];
                                      description: 'The postcode or zip code for the location. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.}';
                                      example: '95014';
                                    };
                                    country_code: {
                                      type: ['string', 'null'];
                                      enum: [
                                        'AF',
                                        'AX',
                                        'AL',
                                        'DZ',
                                        'AS',
                                        'AD',
                                        'AO',
                                        'AI',
                                        'AQ',
                                        'AG',
                                        'AR',
                                        'AM',
                                        'AW',
                                        'AU',
                                        'AT',
                                        'AZ',
                                        'BS',
                                        'BH',
                                        'BD',
                                        'BB',
                                        'BY',
                                        'BE',
                                        'BZ',
                                        'BJ',
                                        'BM',
                                        'BT',
                                        'BO',
                                        'BA',
                                        'BW',
                                        'BV',
                                        'BR',
                                        'IO',
                                        'BN',
                                        'BG',
                                        'BF',
                                        'BI',
                                        'KH',
                                        'CM',
                                        'CA',
                                        'CV',
                                        'KY',
                                        'CF',
                                        'TD',
                                        'CL',
                                        'CN',
                                        'CX',
                                        'CC',
                                        'CO',
                                        'KM',
                                        'CG',
                                        'CD',
                                        'CK',
                                        'CR',
                                        'CI',
                                        'HR',
                                        'CU',
                                        'CW',
                                        'CY',
                                        'CZ',
                                        'DK',
                                        'DJ',
                                        'DM',
                                        'DO',
                                        'EC',
                                        'EG',
                                        'SV',
                                        'GQ',
                                        'ER',
                                        'EE',
                                        'ET',
                                        'FK',
                                        'FO',
                                        'FJ',
                                        'FI',
                                        'FR',
                                        'GF',
                                        'PF',
                                        'TF',
                                        'GA',
                                        'GM',
                                        'GE',
                                        'DE',
                                        'GH',
                                        'GI',
                                        'GR',
                                        'GL',
                                        'GD',
                                        'GP',
                                        'GU',
                                        'GT',
                                        'GG',
                                        'GN',
                                        'GW',
                                        'GY',
                                        'HT',
                                        'HM',
                                        'VA',
                                        'HN',
                                        'HK',
                                        'HU',
                                        'IS',
                                        'IN',
                                        'ID',
                                        'IR',
                                        'IQ',
                                        'IE',
                                        'IM',
                                        'IL',
                                        'IT',
                                        'JM',
                                        'JP',
                                        'JE',
                                        'JO',
                                        'KZ',
                                        'KE',
                                        'KI',
                                        'KR',
                                        'KW',
                                        'KG',
                                        'LA',
                                        'LV',
                                        'LB',
                                        'LS',
                                        'LR',
                                        'LY',
                                        'LI',
                                        'LT',
                                        'LU',
                                        'MO',
                                        'MK',
                                        'MG',
                                        'MW',
                                        'MY',
                                        'MV',
                                        'ML',
                                        'MT',
                                        'MH',
                                        'MQ',
                                        'MR',
                                        'MU',
                                        'YT',
                                        'MX',
                                        'FM',
                                        'MD',
                                        'MC',
                                        'MN',
                                        'ME',
                                        'MS',
                                        'MA',
                                        'MZ',
                                        'MM',
                                        'NA',
                                        'NR',
                                        'NP',
                                        'NL',
                                        'AN',
                                        'NC',
                                        'NZ',
                                        'NI',
                                        'NE',
                                        'NG',
                                        'NU',
                                        'NF',
                                        'MP',
                                        'NO',
                                        'OM',
                                        'PK',
                                        'PW',
                                        'PS',
                                        'PA',
                                        'PG',
                                        'PY',
                                        'PE',
                                        'PH',
                                        'PN',
                                        'PL',
                                        'PT',
                                        'PR',
                                        'QA',
                                        'RE',
                                        'RO',
                                        'RU',
                                        'RW',
                                        'BL',
                                        'SH',
                                        'KN',
                                        'LC',
                                        'MF',
                                        'PM',
                                        'VC',
                                        'WS',
                                        'SM',
                                        'ST',
                                        'SA',
                                        'SN',
                                        'RS',
                                        'SC',
                                        'SL',
                                        'SG',
                                        'SK',
                                        'SI',
                                        'SB',
                                        'SO',
                                        'ZA',
                                        'GS',
                                        'ES',
                                        'LK',
                                        'SD',
                                        'SR',
                                        'SJ',
                                        'SZ',
                                        'SE',
                                        'CH',
                                        'SY',
                                        'TW',
                                        'TJ',
                                        'TZ',
                                        'TH',
                                        'TL',
                                        'TG',
                                        'TK',
                                        'TO',
                                        'TT',
                                        'TN',
                                        'TR',
                                        'TM',
                                        'TC',
                                        'TV',
                                        'UG',
                                        'UA',
                                        'AE',
                                        'GB',
                                        'US',
                                        'UM',
                                        'UY',
                                        'UZ',
                                        'VU',
                                        'VE',
                                        'VN',
                                        'VG',
                                        'VI',
                                        'WF',
                                        'EH',
                                        'YE',
                                        'ZM',
                                        'ZW',
                                      ];
                                      description: 'The ISO 3166-1 alpha-2 country code for the country this location is in.';
                                      example: 'US';
                                    };
                                    latitude: {
                                      type: ['string', 'null'];
                                      pattern: '^[-+]?([1-8]?\\d(\\.\\d+)?|90(\\.0+)?)$';
                                      description: 'The latitude of the location. Validated by the regular expression `/^[-+]?([1-8]?\\d(\\.\\d+)?|90(\\.0+)?)$/`. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.}';
                                      example: '37.331741';
                                    };
                                    longitude: {
                                      type: ['string', 'null'];
                                      pattern: '^[-+]?(180(\\.0+)?|((1[0-7]\\d)|([1-9]?\\d))(\\.\\d+)?)$';
                                      description: 'The longitude of the location. Validated by the regular expression `/^[-+]?(180(\\.0+)?|((1[0-7]\\d)|([1-9]?\\d))(\\.\\d+)?)$/`';
                                      example: '-122.030333';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['location'];
                                      description: 'The attribute type of the value.';
                                      example: 'location';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'line_1',
                                    'line_2',
                                    'line_3',
                                    'line_4',
                                    'locality',
                                    'region',
                                    'postcode',
                                    'country_code',
                                    'latitude',
                                    'longitude',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    value: {
                                      type: 'number';
                                      description: 'Numbers are persisted as 64 bit floats.';
                                      example: 42;
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['number'];
                                      description: 'The attribute type of the value.';
                                      example: 'number';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'value',
                                    'attribute_type',
                                  ];
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    first_name: {
                                      type: 'string';
                                      description: 'The first name.';
                                      example: 'Ada';
                                    };
                                    last_name: {
                                      type: 'string';
                                      description: 'The last name.';
                                      example: 'Lovelace';
                                    };
                                    full_name: {
                                      type: 'string';
                                      description: 'The full name.';
                                      example: 'Ada Lovelace';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['personal-name'];
                                      description: 'The attribute type of the value.';
                                      example: 'personal-name';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'first_name',
                                    'last_name',
                                    'full_name',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    original_phone_number: {
                                      type: 'string';
                                      description: 'The raw, original phone number, as inputted.';
                                      example: '5558675309';
                                    };
                                    country_code: {
                                      type: 'string';
                                      description: 'The ISO 3166-1 alpha-2 country code representing the country that this phone number belongs to.';
                                      enum: [
                                        'AF',
                                        'AX',
                                        'AL',
                                        'DZ',
                                        'AS',
                                        'AD',
                                        'AO',
                                        'AI',
                                        'AQ',
                                        'AG',
                                        'AR',
                                        'AM',
                                        'AW',
                                        'AU',
                                        'AT',
                                        'AZ',
                                        'BS',
                                        'BH',
                                        'BD',
                                        'BB',
                                        'BY',
                                        'BE',
                                        'BZ',
                                        'BJ',
                                        'BM',
                                        'BT',
                                        'BO',
                                        'BA',
                                        'BW',
                                        'BV',
                                        'BR',
                                        'IO',
                                        'BN',
                                        'BG',
                                        'BF',
                                        'BI',
                                        'KH',
                                        'CM',
                                        'CA',
                                        'CV',
                                        'KY',
                                        'CF',
                                        'TD',
                                        'CL',
                                        'CN',
                                        'CX',
                                        'CC',
                                        'CO',
                                        'KM',
                                        'CG',
                                        'CD',
                                        'CK',
                                        'CR',
                                        'CI',
                                        'HR',
                                        'CU',
                                        'CW',
                                        'CY',
                                        'CZ',
                                        'DK',
                                        'DJ',
                                        'DM',
                                        'DO',
                                        'EC',
                                        'EG',
                                        'SV',
                                        'GQ',
                                        'ER',
                                        'EE',
                                        'ET',
                                        'FK',
                                        'FO',
                                        'FJ',
                                        'FI',
                                        'FR',
                                        'GF',
                                        'PF',
                                        'TF',
                                        'GA',
                                        'GM',
                                        'GE',
                                        'DE',
                                        'GH',
                                        'GI',
                                        'GR',
                                        'GL',
                                        'GD',
                                        'GP',
                                        'GU',
                                        'GT',
                                        'GG',
                                        'GN',
                                        'GW',
                                        'GY',
                                        'HT',
                                        'HM',
                                        'VA',
                                        'HN',
                                        'HK',
                                        'HU',
                                        'IS',
                                        'IN',
                                        'ID',
                                        'IR',
                                        'IQ',
                                        'IE',
                                        'IM',
                                        'IL',
                                        'IT',
                                        'JM',
                                        'JP',
                                        'JE',
                                        'JO',
                                        'KZ',
                                        'KE',
                                        'KI',
                                        'KR',
                                        'KW',
                                        'KG',
                                        'LA',
                                        'LV',
                                        'LB',
                                        'LS',
                                        'LR',
                                        'LY',
                                        'LI',
                                        'LT',
                                        'LU',
                                        'MO',
                                        'MK',
                                        'MG',
                                        'MW',
                                        'MY',
                                        'MV',
                                        'ML',
                                        'MT',
                                        'MH',
                                        'MQ',
                                        'MR',
                                        'MU',
                                        'YT',
                                        'MX',
                                        'FM',
                                        'MD',
                                        'MC',
                                        'MN',
                                        'ME',
                                        'MS',
                                        'MA',
                                        'MZ',
                                        'MM',
                                        'NA',
                                        'NR',
                                        'NP',
                                        'NL',
                                        'AN',
                                        'NC',
                                        'NZ',
                                        'NI',
                                        'NE',
                                        'NG',
                                        'NU',
                                        'NF',
                                        'MP',
                                        'NO',
                                        'OM',
                                        'PK',
                                        'PW',
                                        'PS',
                                        'PA',
                                        'PG',
                                        'PY',
                                        'PE',
                                        'PH',
                                        'PN',
                                        'PL',
                                        'PT',
                                        'PR',
                                        'QA',
                                        'RE',
                                        'RO',
                                        'RU',
                                        'RW',
                                        'BL',
                                        'SH',
                                        'KN',
                                        'LC',
                                        'MF',
                                        'PM',
                                        'VC',
                                        'WS',
                                        'SM',
                                        'ST',
                                        'SA',
                                        'SN',
                                        'RS',
                                        'SC',
                                        'SL',
                                        'SG',
                                        'SK',
                                        'SI',
                                        'SB',
                                        'SO',
                                        'ZA',
                                        'GS',
                                        'ES',
                                        'LK',
                                        'SD',
                                        'SR',
                                        'SJ',
                                        'SZ',
                                        'SE',
                                        'CH',
                                        'SY',
                                        'TW',
                                        'TJ',
                                        'TZ',
                                        'TH',
                                        'TL',
                                        'TG',
                                        'TK',
                                        'TO',
                                        'TT',
                                        'TN',
                                        'TR',
                                        'TM',
                                        'TC',
                                        'TV',
                                        'UG',
                                        'UA',
                                        'AE',
                                        'GB',
                                        'US',
                                        'UM',
                                        'UY',
                                        'UZ',
                                        'VU',
                                        'VE',
                                        'VN',
                                        'VG',
                                        'VI',
                                        'WF',
                                        'EH',
                                        'YE',
                                        'ZM',
                                        'ZW',
                                      ];
                                      example: 'US';
                                    };
                                    phone_number: {
                                      type: 'string';
                                      example: '+15558675309';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['phone-number'];
                                      description: 'The attribute type of the value.';
                                      example: 'phone-number';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'original_phone_number',
                                    'country_code',
                                    'phone_number',
                                    'attribute_type',
                                  ];
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    status: {
                                      $ref: '#/components/schemas/status';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['status'];
                                      description: 'The attribute type of the value.';
                                      example: 'status';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'status',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    value: {
                                      type: 'number';
                                      description: 'A number between 0 and 5 (inclusive) to represent a star rating.';
                                      example: 3;
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['rating'];
                                      description: 'The attribute type of the value.';
                                      example: 'rating';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'value',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    option: {
                                      $ref: '#/components/schemas/select-option';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['select'];
                                      description: 'The attribute type of the value.';
                                      example: 'select';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'option',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    value: {
                                      type: 'string';
                                      description: 'A raw text field. Values are limited to 10MB.';
                                      example: 'Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.';
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['text'];
                                      description: 'The attribute type of the value.';
                                      example: 'text';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'value',
                                    'attribute_type',
                                  ];
                                  additionalProperties: false;
                                },
                                {
                                  type: 'object';
                                  properties: {
                                    active_from: {
                                      type: 'string';
                                      format: 'date-time';
                                      description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    active_until: {
                                      type: ['string', 'null'];
                                      format: 'date-time';
                                      description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                    created_by_actor: {
                                      type: 'object';
                                      description: 'The actor that created this value.';
                                      properties: {
                                        id: {
                                          type: 'string';
                                          description: 'An ID to identify the actor.';
                                          nullable: true;
                                        };
                                        type: {
                                          type: 'string';
                                          enum: ['api-token', 'workspace-member', 'system'];
                                          nullable: true;
                                          description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                        };
                                      };
                                      example: {
                                        type: 'workspace-member';
                                        id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                      };
                                    };
                                    attribute_type: {
                                      type: 'string';
                                      enum: ['timestamp'];
                                      description: 'The attribute type of the value.';
                                      example: 'timestamp';
                                    };
                                    value: {
                                      type: 'string';
                                      description: 'A timestamp value represents a single, universal moment in time using an ISO 8601 formatted string. This means that a timestamp consists of a date, a time (with nanosecond precision), and a time zone. Attio will coerce timestamps which do not provide full nanosecond precision and UTC is assumed if no time zone is provided. For example, "2023", "2023-01", "2023-01-02", "2023-01-02T13:00", "2023-01-02T13:00:00", and "2023-01-02T13:00:00.000000000" will all be coerced to "2023-01-02T13:00:00.000000000Z". Timestamps are always returned in UTC. For example, writing a timestamp value using the string "2023-01-02T13:00:00.000000000+02:00" will result in the value "2023-01-02T11:00:00.000000000Z" being returned. The maximum date is "9999-12-31T23:59:59.999999999Z".';
                                      format: 'date';
                                      example: '2023-01-01T15:00:00.000000000Z';
                                    };
                                  };
                                  required: [
                                    'active_from',
                                    'active_until',
                                    'created_by_actor',
                                    'attribute_type',
                                    'value',
                                  ];
                                  additionalProperties: false;
                                },
                              ];
                            };
                          };
                          description: 'A list of attribute values for the list entry (not attribute values for its parent record).';
                          example: {
                            status: [
                              {
                                active_from: '2023-01-01T15:00:00.000000000Z';
                                active_until: null;
                                created_by_actor: {
                                  type: 'workspace-member';
                                  id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                };
                                status: {
                                  id: {
                                    workspace_id: '14beef7a-99f7-4534-a87e-70b564330a4c';
                                    object_id: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
                                    attribute_id: 'e350362f-4b55-4c0f-93f4-379ae8ff2e5b';
                                    status_id: '527def35-7994-4ef7-9584-80ef8de352a8';
                                  };
                                  title: 'In Progress';
                                  is_archived: false;
                                  target_time_in_status: null;
                                  celebration_enabled: false;
                                };
                                attribute_type: 'status';
                              },
                            ];
                            created_at: [
                              {
                                active_from: '2023-01-01T15:00:00.000000000Z';
                                active_until: null;
                                created_by_actor: {
                                  type: 'workspace-member';
                                  id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                };
                                value: '2023-01-01T15:00:00.000000000Z';
                                attribute_type: 'timestamp';
                              },
                            ];
                            created_by: [
                              {
                                active_from: '2023-01-01T15:00:00.000000000Z';
                                active_until: null;
                                created_by_actor: {
                                  type: 'workspace-member';
                                  id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                };
                                referenced_actor_id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                referenced_actor_type: 'workspace-member';
                                attribute_type: 'actor-reference';
                              },
                            ];
                          };
                        };
                      };
                      required: ['id', 'parent_record_id', 'parent_object', 'created_at', 'entry_values'];
                    };
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'List with slug/ID "enterprise_sales" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
      };
    };
  };
  '/v2/lists/{list}/entries': {
    post: {
      summary: 'Create an entry (add record to list)';
      description: 'Adds a record to a list as a new list entry. This endpoint will throw on conflicts of unique attributes. Multiple list entries are allowed for the same parent record\n\nRequired scopes: `list_entry:read-write`, `list_configuration:read`.';
      tags: ['Entries'];
      security: [
        {
          oauth2: ['list_entry:read-write', 'list_configuration:read'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            description: 'The UUID or slug identifying the list that the created list entry should belong to.';
            example: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
          };
          required: true;
          name: 'list';
          in: 'path';
        },
      ];
      requestBody: {
        required: true;
        content: {
          'application/json': {
            schema: {
              type: 'object';
              properties: {
                data: {
                  type: 'object';
                  properties: {
                    parent_record_id: {
                      type: 'string';
                      format: 'uuid';
                      description: "A UUID identifying the record you want to add to the list. The record will become the 'parent' of the created list entry.";
                      example: '891dcbfc-9141-415d-9b2a-2238a6cc012d';
                    };
                    parent_object: {
                      type: 'string';
                      description: 'A UUID or slug identifying the object that the added parent record belongs to.';
                      example: 'people';
                    };
                    entry_values: {
                      type: 'object';
                      description: 'An object with an attribute `api_slug` or `attribute_id` as the key, and a single value (for single-select attributes), or an array of values (for single or multi-select attributes) as the values. For complete documentation on values for all attribute types, please see our [attribute type docs](/docs/attribute-types).';
                      additionalProperties: {
                        type: 'array';
                      };
                      example: {
                        '41252299-f8c7-4b5e-99c9-4ff8321d2f96': 'Text value';
                        multiselect_attribute: ['Select option 1', 'Select option 2'];
                      };
                    };
                  };
                  required: ['parent_record_id', 'parent_object', 'entry_values'];
                };
              };
              required: ['data'];
            };
          };
        };
      };
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    type: 'object';
                    properties: {
                      id: {
                        type: 'object';
                        properties: {
                          workspace_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying the workspace this entry belongs to.';
                            example: '14beef7a-99f7-4534-a87e-70b564330a4c';
                          };
                          list_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying the list this entry is in.';
                            example: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
                          };
                          entry_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying this entry.';
                            example: '2e6e29ea-c4e0-4f44-842d-78a891f8c156';
                          };
                        };
                        required: ['workspace_id', 'list_id', 'entry_id'];
                      };
                      parent_record_id: {
                        type: 'string';
                        format: 'uuid';
                        description: 'A UUID identifying the record that is parent of the list entry.';
                        example: '891dcbfc-9141-415d-9b2a-2238a6cc012d';
                      };
                      parent_object: {
                        type: 'string';
                        description: 'A UUID or slug identifying the object that the parent record belongs to.';
                        example: 'people';
                      };
                      created_at: {
                        type: 'string';
                        description: 'When this entry was created.';
                        example: '2022-11-21T13:22:49.061281000Z';
                      };
                      entry_values: {
                        type: 'object';
                        additionalProperties: {
                          type: 'array';
                          items: {
                            oneOf: [
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  referenced_actor_type: {
                                    type: 'string';
                                    enum: ['api-token', 'workspace-member', 'system'];
                                    description: 'The type of the referenced actor. [Read more information on actor types here](/docs/actors).';
                                    example: 'workspace-member';
                                  };
                                  referenced_actor_id: {
                                    type: ['string', 'null'];
                                    format: 'uuid';
                                    description: 'The ID of the referenced actor.';
                                    example: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['actor-reference'];
                                    description: 'The attribute type of the value.';
                                    example: 'actor-reference';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'referenced_actor_type',
                                  'referenced_actor_id',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'boolean';
                                    description: "A boolean representing whether the checkbox is checked or not. The string values 'true' and 'false' are also accepted.";
                                    example: true;
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['checkbox'];
                                    description: 'The attribute type of the value.';
                                    example: 'checkbox';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  currency_value: {
                                    type: 'number';
                                    description: 'A numerical representation of the currency value. A decimal with a max of 4 decimal places.';
                                    example: 99;
                                  };
                                  currency_code: {
                                    type: ['string', 'null'];
                                    enum: [
                                      'EUR',
                                      'GBP',
                                      'USD',
                                      'CAD',
                                      'AUD',
                                      'NZD',
                                      'AED',
                                      'INR',
                                      'CHF',
                                      'JPY',
                                      'CNY',
                                      'PHP',
                                      'ILS',
                                      'SGD',
                                      'HKD',
                                      'MYR',
                                      'NTD',
                                      'NOK',
                                      'SEK',
                                      'ISK',
                                      'BEL',
                                      'PLN',
                                      'KRW',
                                      'DKK',
                                      'BRL',
                                    ];
                                    description: 'The ISO4217 currency code representing the currency that the value is stored in.';
                                    example: 'USD';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['currency'];
                                    description: 'The attribute type of the value.';
                                    example: 'currency';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'currency_value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['date'];
                                    description: 'The attribute type of the value.';
                                    example: 'date';
                                  };
                                  value: {
                                    type: 'string';
                                    description: 'A date represents a single calendar year, month and day, independent of timezone. If hours, months, seconds or timezones are provided, they will be trimmed. For example, "2023" and "2023-01" will be coerced into "2023-01-01", and "2023-01-02", "2023-01-02T13:00", "2023-01-02T14:00:00", "2023-01-02T15:00:00.000000000", and "2023-01-02T15:00:00.000000000+02:00" will all be coerced to "2023-01-02". If a timezone is provided that would result in a different calendar date in UTC, the date will be coerced to UTC and then the timezone component will be trimmed. For example, the value "2023-01-02T23:00:00-10:00" will be returned as "2023-01-03". The maximum date is "9999-12-31".';
                                    example: '2023-01-01';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'attribute_type',
                                  'value',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  domain: {
                                    type: 'string';
                                    example: 'app.attio.com';
                                  };
                                  root_domain: {
                                    type: 'string';
                                    example: 'attio.com';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['domain'];
                                    description: 'The attribute type of the value.';
                                    example: 'domain';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'domain',
                                  'root_domain',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  original_email_address: {
                                    type: 'string';
                                    example: 'alice@app.attio.com';
                                  };
                                  email_address: {
                                    type: 'string';
                                    example: 'alice@app.attio.com';
                                  };
                                  email_domain: {
                                    type: 'string';
                                    example: 'app.attio.com';
                                  };
                                  email_root_domain: {
                                    type: 'string';
                                    example: 'attio.com';
                                  };
                                  email_local_specifier: {
                                    type: 'string';
                                    example: 'alice';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['email-address'];
                                    description: 'The attribute type of the value.';
                                    example: 'email-address';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'original_email_address',
                                  'email_address',
                                  'email_domain',
                                  'email_root_domain',
                                  'email_local_specifier',
                                  'attribute_type',
                                ];
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  target_object: {
                                    type: 'string';
                                    description: 'A slug identifying the object that the referenced record belongs to.';
                                    example: 'people';
                                  };
                                  target_record_id: {
                                    type: 'string';
                                    format: 'uuid';
                                    description: 'A UUID to identify the referenced record.';
                                    example: '891dcbfc-9141-415d-9b2a-2238a6cc012d';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['record-reference'];
                                    description: 'The attribute type of the value.';
                                    example: 'record-reference';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'target_object',
                                  'target_record_id',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  interaction_type: {
                                    type: 'string';
                                    enum: ['calendar-event', 'call', 'chat-thread', 'email', 'in-person-meeting'];
                                    description: 'The type of interaction e.g. calendar or email.';
                                    example: 'email';
                                  };
                                  interacted_at: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'When the interaction occurred.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  owner_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['interaction'];
                                    description: 'The attribute type of the value.';
                                    example: 'interaction';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'interaction_type',
                                  'interacted_at',
                                  'owner_actor',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  line_1: {
                                    type: ['string', 'null'];
                                    description: 'The first line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: '1 Infinite Loop';
                                  };
                                  line_2: {
                                    type: ['string', 'null'];
                                    description: 'The second line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: 'Block 1';
                                  };
                                  line_3: {
                                    type: ['string', 'null'];
                                    description: 'The third line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: 'Hilldrop Estate';
                                  };
                                  line_4: {
                                    type: ['string', 'null'];
                                    description: 'The fourth line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: 'Westborough';
                                  };
                                  locality: {
                                    type: ['string', 'null'];
                                    description: 'The town, neighborhood or area the location is in.';
                                    example: 'Cupertino';
                                  };
                                  region: {
                                    type: ['string', 'null'];
                                    description: 'The state, county, province or region that the location is in.';
                                    example: 'CA';
                                  };
                                  postcode: {
                                    type: ['string', 'null'];
                                    description: 'The postcode or zip code for the location. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.}';
                                    example: '95014';
                                  };
                                  country_code: {
                                    type: ['string', 'null'];
                                    enum: [
                                      'AF',
                                      'AX',
                                      'AL',
                                      'DZ',
                                      'AS',
                                      'AD',
                                      'AO',
                                      'AI',
                                      'AQ',
                                      'AG',
                                      'AR',
                                      'AM',
                                      'AW',
                                      'AU',
                                      'AT',
                                      'AZ',
                                      'BS',
                                      'BH',
                                      'BD',
                                      'BB',
                                      'BY',
                                      'BE',
                                      'BZ',
                                      'BJ',
                                      'BM',
                                      'BT',
                                      'BO',
                                      'BA',
                                      'BW',
                                      'BV',
                                      'BR',
                                      'IO',
                                      'BN',
                                      'BG',
                                      'BF',
                                      'BI',
                                      'KH',
                                      'CM',
                                      'CA',
                                      'CV',
                                      'KY',
                                      'CF',
                                      'TD',
                                      'CL',
                                      'CN',
                                      'CX',
                                      'CC',
                                      'CO',
                                      'KM',
                                      'CG',
                                      'CD',
                                      'CK',
                                      'CR',
                                      'CI',
                                      'HR',
                                      'CU',
                                      'CW',
                                      'CY',
                                      'CZ',
                                      'DK',
                                      'DJ',
                                      'DM',
                                      'DO',
                                      'EC',
                                      'EG',
                                      'SV',
                                      'GQ',
                                      'ER',
                                      'EE',
                                      'ET',
                                      'FK',
                                      'FO',
                                      'FJ',
                                      'FI',
                                      'FR',
                                      'GF',
                                      'PF',
                                      'TF',
                                      'GA',
                                      'GM',
                                      'GE',
                                      'DE',
                                      'GH',
                                      'GI',
                                      'GR',
                                      'GL',
                                      'GD',
                                      'GP',
                                      'GU',
                                      'GT',
                                      'GG',
                                      'GN',
                                      'GW',
                                      'GY',
                                      'HT',
                                      'HM',
                                      'VA',
                                      'HN',
                                      'HK',
                                      'HU',
                                      'IS',
                                      'IN',
                                      'ID',
                                      'IR',
                                      'IQ',
                                      'IE',
                                      'IM',
                                      'IL',
                                      'IT',
                                      'JM',
                                      'JP',
                                      'JE',
                                      'JO',
                                      'KZ',
                                      'KE',
                                      'KI',
                                      'KR',
                                      'KW',
                                      'KG',
                                      'LA',
                                      'LV',
                                      'LB',
                                      'LS',
                                      'LR',
                                      'LY',
                                      'LI',
                                      'LT',
                                      'LU',
                                      'MO',
                                      'MK',
                                      'MG',
                                      'MW',
                                      'MY',
                                      'MV',
                                      'ML',
                                      'MT',
                                      'MH',
                                      'MQ',
                                      'MR',
                                      'MU',
                                      'YT',
                                      'MX',
                                      'FM',
                                      'MD',
                                      'MC',
                                      'MN',
                                      'ME',
                                      'MS',
                                      'MA',
                                      'MZ',
                                      'MM',
                                      'NA',
                                      'NR',
                                      'NP',
                                      'NL',
                                      'AN',
                                      'NC',
                                      'NZ',
                                      'NI',
                                      'NE',
                                      'NG',
                                      'NU',
                                      'NF',
                                      'MP',
                                      'NO',
                                      'OM',
                                      'PK',
                                      'PW',
                                      'PS',
                                      'PA',
                                      'PG',
                                      'PY',
                                      'PE',
                                      'PH',
                                      'PN',
                                      'PL',
                                      'PT',
                                      'PR',
                                      'QA',
                                      'RE',
                                      'RO',
                                      'RU',
                                      'RW',
                                      'BL',
                                      'SH',
                                      'KN',
                                      'LC',
                                      'MF',
                                      'PM',
                                      'VC',
                                      'WS',
                                      'SM',
                                      'ST',
                                      'SA',
                                      'SN',
                                      'RS',
                                      'SC',
                                      'SL',
                                      'SG',
                                      'SK',
                                      'SI',
                                      'SB',
                                      'SO',
                                      'ZA',
                                      'GS',
                                      'ES',
                                      'LK',
                                      'SD',
                                      'SR',
                                      'SJ',
                                      'SZ',
                                      'SE',
                                      'CH',
                                      'SY',
                                      'TW',
                                      'TJ',
                                      'TZ',
                                      'TH',
                                      'TL',
                                      'TG',
                                      'TK',
                                      'TO',
                                      'TT',
                                      'TN',
                                      'TR',
                                      'TM',
                                      'TC',
                                      'TV',
                                      'UG',
                                      'UA',
                                      'AE',
                                      'GB',
                                      'US',
                                      'UM',
                                      'UY',
                                      'UZ',
                                      'VU',
                                      'VE',
                                      'VN',
                                      'VG',
                                      'VI',
                                      'WF',
                                      'EH',
                                      'YE',
                                      'ZM',
                                      'ZW',
                                    ];
                                    description: 'The ISO 3166-1 alpha-2 country code for the country this location is in.';
                                    example: 'US';
                                  };
                                  latitude: {
                                    type: ['string', 'null'];
                                    pattern: '^[-+]?([1-8]?\\d(\\.\\d+)?|90(\\.0+)?)$';
                                    description: 'The latitude of the location. Validated by the regular expression `/^[-+]?([1-8]?\\d(\\.\\d+)?|90(\\.0+)?)$/`. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.}';
                                    example: '37.331741';
                                  };
                                  longitude: {
                                    type: ['string', 'null'];
                                    pattern: '^[-+]?(180(\\.0+)?|((1[0-7]\\d)|([1-9]?\\d))(\\.\\d+)?)$';
                                    description: 'The longitude of the location. Validated by the regular expression `/^[-+]?(180(\\.0+)?|((1[0-7]\\d)|([1-9]?\\d))(\\.\\d+)?)$/`';
                                    example: '-122.030333';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['location'];
                                    description: 'The attribute type of the value.';
                                    example: 'location';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'line_1',
                                  'line_2',
                                  'line_3',
                                  'line_4',
                                  'locality',
                                  'region',
                                  'postcode',
                                  'country_code',
                                  'latitude',
                                  'longitude',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'number';
                                    description: 'Numbers are persisted as 64 bit floats.';
                                    example: 42;
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['number'];
                                    description: 'The attribute type of the value.';
                                    example: 'number';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  first_name: {
                                    type: 'string';
                                    description: 'The first name.';
                                    example: 'Ada';
                                  };
                                  last_name: {
                                    type: 'string';
                                    description: 'The last name.';
                                    example: 'Lovelace';
                                  };
                                  full_name: {
                                    type: 'string';
                                    description: 'The full name.';
                                    example: 'Ada Lovelace';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['personal-name'];
                                    description: 'The attribute type of the value.';
                                    example: 'personal-name';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'first_name',
                                  'last_name',
                                  'full_name',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  original_phone_number: {
                                    type: 'string';
                                    description: 'The raw, original phone number, as inputted.';
                                    example: '5558675309';
                                  };
                                  country_code: {
                                    type: 'string';
                                    description: 'The ISO 3166-1 alpha-2 country code representing the country that this phone number belongs to.';
                                    enum: [
                                      'AF',
                                      'AX',
                                      'AL',
                                      'DZ',
                                      'AS',
                                      'AD',
                                      'AO',
                                      'AI',
                                      'AQ',
                                      'AG',
                                      'AR',
                                      'AM',
                                      'AW',
                                      'AU',
                                      'AT',
                                      'AZ',
                                      'BS',
                                      'BH',
                                      'BD',
                                      'BB',
                                      'BY',
                                      'BE',
                                      'BZ',
                                      'BJ',
                                      'BM',
                                      'BT',
                                      'BO',
                                      'BA',
                                      'BW',
                                      'BV',
                                      'BR',
                                      'IO',
                                      'BN',
                                      'BG',
                                      'BF',
                                      'BI',
                                      'KH',
                                      'CM',
                                      'CA',
                                      'CV',
                                      'KY',
                                      'CF',
                                      'TD',
                                      'CL',
                                      'CN',
                                      'CX',
                                      'CC',
                                      'CO',
                                      'KM',
                                      'CG',
                                      'CD',
                                      'CK',
                                      'CR',
                                      'CI',
                                      'HR',
                                      'CU',
                                      'CW',
                                      'CY',
                                      'CZ',
                                      'DK',
                                      'DJ',
                                      'DM',
                                      'DO',
                                      'EC',
                                      'EG',
                                      'SV',
                                      'GQ',
                                      'ER',
                                      'EE',
                                      'ET',
                                      'FK',
                                      'FO',
                                      'FJ',
                                      'FI',
                                      'FR',
                                      'GF',
                                      'PF',
                                      'TF',
                                      'GA',
                                      'GM',
                                      'GE',
                                      'DE',
                                      'GH',
                                      'GI',
                                      'GR',
                                      'GL',
                                      'GD',
                                      'GP',
                                      'GU',
                                      'GT',
                                      'GG',
                                      'GN',
                                      'GW',
                                      'GY',
                                      'HT',
                                      'HM',
                                      'VA',
                                      'HN',
                                      'HK',
                                      'HU',
                                      'IS',
                                      'IN',
                                      'ID',
                                      'IR',
                                      'IQ',
                                      'IE',
                                      'IM',
                                      'IL',
                                      'IT',
                                      'JM',
                                      'JP',
                                      'JE',
                                      'JO',
                                      'KZ',
                                      'KE',
                                      'KI',
                                      'KR',
                                      'KW',
                                      'KG',
                                      'LA',
                                      'LV',
                                      'LB',
                                      'LS',
                                      'LR',
                                      'LY',
                                      'LI',
                                      'LT',
                                      'LU',
                                      'MO',
                                      'MK',
                                      'MG',
                                      'MW',
                                      'MY',
                                      'MV',
                                      'ML',
                                      'MT',
                                      'MH',
                                      'MQ',
                                      'MR',
                                      'MU',
                                      'YT',
                                      'MX',
                                      'FM',
                                      'MD',
                                      'MC',
                                      'MN',
                                      'ME',
                                      'MS',
                                      'MA',
                                      'MZ',
                                      'MM',
                                      'NA',
                                      'NR',
                                      'NP',
                                      'NL',
                                      'AN',
                                      'NC',
                                      'NZ',
                                      'NI',
                                      'NE',
                                      'NG',
                                      'NU',
                                      'NF',
                                      'MP',
                                      'NO',
                                      'OM',
                                      'PK',
                                      'PW',
                                      'PS',
                                      'PA',
                                      'PG',
                                      'PY',
                                      'PE',
                                      'PH',
                                      'PN',
                                      'PL',
                                      'PT',
                                      'PR',
                                      'QA',
                                      'RE',
                                      'RO',
                                      'RU',
                                      'RW',
                                      'BL',
                                      'SH',
                                      'KN',
                                      'LC',
                                      'MF',
                                      'PM',
                                      'VC',
                                      'WS',
                                      'SM',
                                      'ST',
                                      'SA',
                                      'SN',
                                      'RS',
                                      'SC',
                                      'SL',
                                      'SG',
                                      'SK',
                                      'SI',
                                      'SB',
                                      'SO',
                                      'ZA',
                                      'GS',
                                      'ES',
                                      'LK',
                                      'SD',
                                      'SR',
                                      'SJ',
                                      'SZ',
                                      'SE',
                                      'CH',
                                      'SY',
                                      'TW',
                                      'TJ',
                                      'TZ',
                                      'TH',
                                      'TL',
                                      'TG',
                                      'TK',
                                      'TO',
                                      'TT',
                                      'TN',
                                      'TR',
                                      'TM',
                                      'TC',
                                      'TV',
                                      'UG',
                                      'UA',
                                      'AE',
                                      'GB',
                                      'US',
                                      'UM',
                                      'UY',
                                      'UZ',
                                      'VU',
                                      'VE',
                                      'VN',
                                      'VG',
                                      'VI',
                                      'WF',
                                      'EH',
                                      'YE',
                                      'ZM',
                                      'ZW',
                                    ];
                                    example: 'US';
                                  };
                                  phone_number: {
                                    type: 'string';
                                    example: '+15558675309';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['phone-number'];
                                    description: 'The attribute type of the value.';
                                    example: 'phone-number';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'original_phone_number',
                                  'country_code',
                                  'phone_number',
                                  'attribute_type',
                                ];
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  status: {
                                    $ref: '#/components/schemas/status';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['status'];
                                    description: 'The attribute type of the value.';
                                    example: 'status';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'status',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'number';
                                    description: 'A number between 0 and 5 (inclusive) to represent a star rating.';
                                    example: 3;
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['rating'];
                                    description: 'The attribute type of the value.';
                                    example: 'rating';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  option: {
                                    $ref: '#/components/schemas/select-option';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['select'];
                                    description: 'The attribute type of the value.';
                                    example: 'select';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'option',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'string';
                                    description: 'A raw text field. Values are limited to 10MB.';
                                    example: 'Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['text'];
                                    description: 'The attribute type of the value.';
                                    example: 'text';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['timestamp'];
                                    description: 'The attribute type of the value.';
                                    example: 'timestamp';
                                  };
                                  value: {
                                    type: 'string';
                                    description: 'A timestamp value represents a single, universal moment in time using an ISO 8601 formatted string. This means that a timestamp consists of a date, a time (with nanosecond precision), and a time zone. Attio will coerce timestamps which do not provide full nanosecond precision and UTC is assumed if no time zone is provided. For example, "2023", "2023-01", "2023-01-02", "2023-01-02T13:00", "2023-01-02T13:00:00", and "2023-01-02T13:00:00.000000000" will all be coerced to "2023-01-02T13:00:00.000000000Z". Timestamps are always returned in UTC. For example, writing a timestamp value using the string "2023-01-02T13:00:00.000000000+02:00" will result in the value "2023-01-02T11:00:00.000000000Z" being returned. The maximum date is "9999-12-31T23:59:59.999999999Z".';
                                    format: 'date';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'attribute_type',
                                  'value',
                                ];
                                additionalProperties: false;
                              },
                            ];
                          };
                        };
                        description: 'A list of attribute values for the list entry (not attribute values for its parent record).';
                        example: {
                          status: [
                            {
                              active_from: '2023-01-01T15:00:00.000000000Z';
                              active_until: null;
                              created_by_actor: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                              status: {
                                id: {
                                  workspace_id: '14beef7a-99f7-4534-a87e-70b564330a4c';
                                  object_id: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
                                  attribute_id: 'e350362f-4b55-4c0f-93f4-379ae8ff2e5b';
                                  status_id: '527def35-7994-4ef7-9584-80ef8de352a8';
                                };
                                title: 'In Progress';
                                is_archived: false;
                                target_time_in_status: null;
                                celebration_enabled: false;
                              };
                              attribute_type: 'status';
                            },
                          ];
                          created_at: [
                            {
                              active_from: '2023-01-01T15:00:00.000000000Z';
                              active_until: null;
                              created_by_actor: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                              value: '2023-01-01T15:00:00.000000000Z';
                              attribute_type: 'timestamp';
                            },
                          ];
                          created_by: [
                            {
                              active_from: '2023-01-01T15:00:00.000000000Z';
                              active_until: null;
                              created_by_actor: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                              referenced_actor_id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              referenced_actor_type: 'workspace-member';
                              attribute_type: 'actor-reference';
                            },
                          ];
                        };
                      };
                    };
                    required: ['id', 'parent_record_id', 'parent_object', 'created_at', 'entry_values'];
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '400': {
          description: 'Bad Request';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [400];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['value_not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Parent record with slug/ID "891dcbfc-9141-415d-9b2a-2238a6cc012d" not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Bad Request';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'Could not find list with slug/ID "enterprise_sales".';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
      };
    };
    put: {
      summary: 'Assert a list entry by parent';
      description: 'Use this endpoint to create or update a list entry for a given parent record. If an entry with the specified parent record is found, that entry will be updated. If no such entry is found, a new entry will be created instead. If there are multiple entries with the same parent record, this endpoint with return the "MULTIPLE_MATCH_RESULTS" error. When writing to multi-select attributes, all values will be either created or deleted as necessary to match the list of values supplied in the request body.\n\nRequired scopes: `list_entry:read-write`, `list_configuration:read`.';
      tags: ['Entries'];
      security: [
        {
          oauth2: ['list_entry:read-write', 'list_configuration:read'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug of the list the list entry belongs to.';
            example: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
          };
          required: true;
          name: 'list';
          in: 'path';
        },
      ];
      requestBody: {
        required: true;
        content: {
          'application/json': {
            schema: {
              type: 'object';
              properties: {
                data: {
                  type: 'object';
                  properties: {
                    parent_record_id: {
                      type: 'string';
                      format: 'uuid';
                      description: "A UUID identifying the record you want to add to the list. The record will become the 'parent' of the created list entry.";
                      example: '891dcbfc-9141-415d-9b2a-2238a6cc012d';
                    };
                    parent_object: {
                      type: 'string';
                      description: 'A UUID or slug identifying the object that the added parent record belongs to.';
                      example: 'people';
                    };
                    entry_values: {
                      type: 'object';
                      description: 'An object with an attribute `api_slug` or `attribute_id` as the key, and a single value (for single-select attributes), or an array of values (for single or multi-select attributes) as the values. For complete documentation on values for all attribute types, please see our [attribute type docs](/docs/attribute-types).';
                      additionalProperties: {
                        type: 'array';
                      };
                      example: {
                        '41252299-f8c7-4b5e-99c9-4ff8321d2f96': 'Text value';
                        multiselect_attribute: ['Select option 1', 'Select option 2'];
                      };
                    };
                  };
                  required: ['parent_record_id', 'parent_object', 'entry_values'];
                };
              };
              required: ['data'];
            };
          };
        };
      };
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    type: 'object';
                    properties: {
                      id: {
                        type: 'object';
                        properties: {
                          workspace_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying the workspace this entry belongs to.';
                            example: '14beef7a-99f7-4534-a87e-70b564330a4c';
                          };
                          list_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying the list this entry is in.';
                            example: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
                          };
                          entry_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying this entry.';
                            example: '2e6e29ea-c4e0-4f44-842d-78a891f8c156';
                          };
                        };
                        required: ['workspace_id', 'list_id', 'entry_id'];
                      };
                      parent_record_id: {
                        type: 'string';
                        format: 'uuid';
                        description: 'A UUID identifying the record that is parent of the list entry.';
                        example: '891dcbfc-9141-415d-9b2a-2238a6cc012d';
                      };
                      parent_object: {
                        type: 'string';
                        description: 'A UUID or slug identifying the object that the parent record belongs to.';
                        example: 'people';
                      };
                      created_at: {
                        type: 'string';
                        description: 'When this entry was created.';
                        example: '2022-11-21T13:22:49.061281000Z';
                      };
                      entry_values: {
                        type: 'object';
                        additionalProperties: {
                          type: 'array';
                          items: {
                            oneOf: [
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  referenced_actor_type: {
                                    type: 'string';
                                    enum: ['api-token', 'workspace-member', 'system'];
                                    description: 'The type of the referenced actor. [Read more information on actor types here](/docs/actors).';
                                    example: 'workspace-member';
                                  };
                                  referenced_actor_id: {
                                    type: ['string', 'null'];
                                    format: 'uuid';
                                    description: 'The ID of the referenced actor.';
                                    example: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['actor-reference'];
                                    description: 'The attribute type of the value.';
                                    example: 'actor-reference';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'referenced_actor_type',
                                  'referenced_actor_id',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'boolean';
                                    description: "A boolean representing whether the checkbox is checked or not. The string values 'true' and 'false' are also accepted.";
                                    example: true;
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['checkbox'];
                                    description: 'The attribute type of the value.';
                                    example: 'checkbox';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  currency_value: {
                                    type: 'number';
                                    description: 'A numerical representation of the currency value. A decimal with a max of 4 decimal places.';
                                    example: 99;
                                  };
                                  currency_code: {
                                    type: ['string', 'null'];
                                    enum: [
                                      'EUR',
                                      'GBP',
                                      'USD',
                                      'CAD',
                                      'AUD',
                                      'NZD',
                                      'AED',
                                      'INR',
                                      'CHF',
                                      'JPY',
                                      'CNY',
                                      'PHP',
                                      'ILS',
                                      'SGD',
                                      'HKD',
                                      'MYR',
                                      'NTD',
                                      'NOK',
                                      'SEK',
                                      'ISK',
                                      'BEL',
                                      'PLN',
                                      'KRW',
                                      'DKK',
                                      'BRL',
                                    ];
                                    description: 'The ISO4217 currency code representing the currency that the value is stored in.';
                                    example: 'USD';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['currency'];
                                    description: 'The attribute type of the value.';
                                    example: 'currency';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'currency_value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['date'];
                                    description: 'The attribute type of the value.';
                                    example: 'date';
                                  };
                                  value: {
                                    type: 'string';
                                    description: 'A date represents a single calendar year, month and day, independent of timezone. If hours, months, seconds or timezones are provided, they will be trimmed. For example, "2023" and "2023-01" will be coerced into "2023-01-01", and "2023-01-02", "2023-01-02T13:00", "2023-01-02T14:00:00", "2023-01-02T15:00:00.000000000", and "2023-01-02T15:00:00.000000000+02:00" will all be coerced to "2023-01-02". If a timezone is provided that would result in a different calendar date in UTC, the date will be coerced to UTC and then the timezone component will be trimmed. For example, the value "2023-01-02T23:00:00-10:00" will be returned as "2023-01-03". The maximum date is "9999-12-31".';
                                    example: '2023-01-01';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'attribute_type',
                                  'value',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  domain: {
                                    type: 'string';
                                    example: 'app.attio.com';
                                  };
                                  root_domain: {
                                    type: 'string';
                                    example: 'attio.com';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['domain'];
                                    description: 'The attribute type of the value.';
                                    example: 'domain';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'domain',
                                  'root_domain',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  original_email_address: {
                                    type: 'string';
                                    example: 'alice@app.attio.com';
                                  };
                                  email_address: {
                                    type: 'string';
                                    example: 'alice@app.attio.com';
                                  };
                                  email_domain: {
                                    type: 'string';
                                    example: 'app.attio.com';
                                  };
                                  email_root_domain: {
                                    type: 'string';
                                    example: 'attio.com';
                                  };
                                  email_local_specifier: {
                                    type: 'string';
                                    example: 'alice';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['email-address'];
                                    description: 'The attribute type of the value.';
                                    example: 'email-address';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'original_email_address',
                                  'email_address',
                                  'email_domain',
                                  'email_root_domain',
                                  'email_local_specifier',
                                  'attribute_type',
                                ];
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  target_object: {
                                    type: 'string';
                                    description: 'A slug identifying the object that the referenced record belongs to.';
                                    example: 'people';
                                  };
                                  target_record_id: {
                                    type: 'string';
                                    format: 'uuid';
                                    description: 'A UUID to identify the referenced record.';
                                    example: '891dcbfc-9141-415d-9b2a-2238a6cc012d';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['record-reference'];
                                    description: 'The attribute type of the value.';
                                    example: 'record-reference';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'target_object',
                                  'target_record_id',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  interaction_type: {
                                    type: 'string';
                                    enum: ['calendar-event', 'call', 'chat-thread', 'email', 'in-person-meeting'];
                                    description: 'The type of interaction e.g. calendar or email.';
                                    example: 'email';
                                  };
                                  interacted_at: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'When the interaction occurred.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  owner_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['interaction'];
                                    description: 'The attribute type of the value.';
                                    example: 'interaction';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'interaction_type',
                                  'interacted_at',
                                  'owner_actor',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  line_1: {
                                    type: ['string', 'null'];
                                    description: 'The first line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: '1 Infinite Loop';
                                  };
                                  line_2: {
                                    type: ['string', 'null'];
                                    description: 'The second line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: 'Block 1';
                                  };
                                  line_3: {
                                    type: ['string', 'null'];
                                    description: 'The third line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: 'Hilldrop Estate';
                                  };
                                  line_4: {
                                    type: ['string', 'null'];
                                    description: 'The fourth line of the address. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.';
                                    example: 'Westborough';
                                  };
                                  locality: {
                                    type: ['string', 'null'];
                                    description: 'The town, neighborhood or area the location is in.';
                                    example: 'Cupertino';
                                  };
                                  region: {
                                    type: ['string', 'null'];
                                    description: 'The state, county, province or region that the location is in.';
                                    example: 'CA';
                                  };
                                  postcode: {
                                    type: ['string', 'null'];
                                    description: 'The postcode or zip code for the location. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.}';
                                    example: '95014';
                                  };
                                  country_code: {
                                    type: ['string', 'null'];
                                    enum: [
                                      'AF',
                                      'AX',
                                      'AL',
                                      'DZ',
                                      'AS',
                                      'AD',
                                      'AO',
                                      'AI',
                                      'AQ',
                                      'AG',
                                      'AR',
                                      'AM',
                                      'AW',
                                      'AU',
                                      'AT',
                                      'AZ',
                                      'BS',
                                      'BH',
                                      'BD',
                                      'BB',
                                      'BY',
                                      'BE',
                                      'BZ',
                                      'BJ',
                                      'BM',
                                      'BT',
                                      'BO',
                                      'BA',
                                      'BW',
                                      'BV',
                                      'BR',
                                      'IO',
                                      'BN',
                                      'BG',
                                      'BF',
                                      'BI',
                                      'KH',
                                      'CM',
                                      'CA',
                                      'CV',
                                      'KY',
                                      'CF',
                                      'TD',
                                      'CL',
                                      'CN',
                                      'CX',
                                      'CC',
                                      'CO',
                                      'KM',
                                      'CG',
                                      'CD',
                                      'CK',
                                      'CR',
                                      'CI',
                                      'HR',
                                      'CU',
                                      'CW',
                                      'CY',
                                      'CZ',
                                      'DK',
                                      'DJ',
                                      'DM',
                                      'DO',
                                      'EC',
                                      'EG',
                                      'SV',
                                      'GQ',
                                      'ER',
                                      'EE',
                                      'ET',
                                      'FK',
                                      'FO',
                                      'FJ',
                                      'FI',
                                      'FR',
                                      'GF',
                                      'PF',
                                      'TF',
                                      'GA',
                                      'GM',
                                      'GE',
                                      'DE',
                                      'GH',
                                      'GI',
                                      'GR',
                                      'GL',
                                      'GD',
                                      'GP',
                                      'GU',
                                      'GT',
                                      'GG',
                                      'GN',
                                      'GW',
                                      'GY',
                                      'HT',
                                      'HM',
                                      'VA',
                                      'HN',
                                      'HK',
                                      'HU',
                                      'IS',
                                      'IN',
                                      'ID',
                                      'IR',
                                      'IQ',
                                      'IE',
                                      'IM',
                                      'IL',
                                      'IT',
                                      'JM',
                                      'JP',
                                      'JE',
                                      'JO',
                                      'KZ',
                                      'KE',
                                      'KI',
                                      'KR',
                                      'KW',
                                      'KG',
                                      'LA',
                                      'LV',
                                      'LB',
                                      'LS',
                                      'LR',
                                      'LY',
                                      'LI',
                                      'LT',
                                      'LU',
                                      'MO',
                                      'MK',
                                      'MG',
                                      'MW',
                                      'MY',
                                      'MV',
                                      'ML',
                                      'MT',
                                      'MH',
                                      'MQ',
                                      'MR',
                                      'MU',
                                      'YT',
                                      'MX',
                                      'FM',
                                      'MD',
                                      'MC',
                                      'MN',
                                      'ME',
                                      'MS',
                                      'MA',
                                      'MZ',
                                      'MM',
                                      'NA',
                                      'NR',
                                      'NP',
                                      'NL',
                                      'AN',
                                      'NC',
                                      'NZ',
                                      'NI',
                                      'NE',
                                      'NG',
                                      'NU',
                                      'NF',
                                      'MP',
                                      'NO',
                                      'OM',
                                      'PK',
                                      'PW',
                                      'PS',
                                      'PA',
                                      'PG',
                                      'PY',
                                      'PE',
                                      'PH',
                                      'PN',
                                      'PL',
                                      'PT',
                                      'PR',
                                      'QA',
                                      'RE',
                                      'RO',
                                      'RU',
                                      'RW',
                                      'BL',
                                      'SH',
                                      'KN',
                                      'LC',
                                      'MF',
                                      'PM',
                                      'VC',
                                      'WS',
                                      'SM',
                                      'ST',
                                      'SA',
                                      'SN',
                                      'RS',
                                      'SC',
                                      'SL',
                                      'SG',
                                      'SK',
                                      'SI',
                                      'SB',
                                      'SO',
                                      'ZA',
                                      'GS',
                                      'ES',
                                      'LK',
                                      'SD',
                                      'SR',
                                      'SJ',
                                      'SZ',
                                      'SE',
                                      'CH',
                                      'SY',
                                      'TW',
                                      'TJ',
                                      'TZ',
                                      'TH',
                                      'TL',
                                      'TG',
                                      'TK',
                                      'TO',
                                      'TT',
                                      'TN',
                                      'TR',
                                      'TM',
                                      'TC',
                                      'TV',
                                      'UG',
                                      'UA',
                                      'AE',
                                      'GB',
                                      'US',
                                      'UM',
                                      'UY',
                                      'UZ',
                                      'VU',
                                      'VE',
                                      'VN',
                                      'VG',
                                      'VI',
                                      'WF',
                                      'EH',
                                      'YE',
                                      'ZM',
                                      'ZW',
                                    ];
                                    description: 'The ISO 3166-1 alpha-2 country code for the country this location is in.';
                                    example: 'US';
                                  };
                                  latitude: {
                                    type: ['string', 'null'];
                                    pattern: '^[-+]?([1-8]?\\d(\\.\\d+)?|90(\\.0+)?)$';
                                    description: 'The latitude of the location. Validated by the regular expression `/^[-+]?([1-8]?\\d(\\.\\d+)?|90(\\.0+)?)$/`. Note that this value is not currently represented in the UI but will be persisted and readable through API calls.}';
                                    example: '37.331741';
                                  };
                                  longitude: {
                                    type: ['string', 'null'];
                                    pattern: '^[-+]?(180(\\.0+)?|((1[0-7]\\d)|([1-9]?\\d))(\\.\\d+)?)$';
                                    description: 'The longitude of the location. Validated by the regular expression `/^[-+]?(180(\\.0+)?|((1[0-7]\\d)|([1-9]?\\d))(\\.\\d+)?)$/`';
                                    example: '-122.030333';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['location'];
                                    description: 'The attribute type of the value.';
                                    example: 'location';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'line_1',
                                  'line_2',
                                  'line_3',
                                  'line_4',
                                  'locality',
                                  'region',
                                  'postcode',
                                  'country_code',
                                  'latitude',
                                  'longitude',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'number';
                                    description: 'Numbers are persisted as 64 bit floats.';
                                    example: 42;
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['number'];
                                    description: 'The attribute type of the value.';
                                    example: 'number';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  first_name: {
                                    type: 'string';
                                    description: 'The first name.';
                                    example: 'Ada';
                                  };
                                  last_name: {
                                    type: 'string';
                                    description: 'The last name.';
                                    example: 'Lovelace';
                                  };
                                  full_name: {
                                    type: 'string';
                                    description: 'The full name.';
                                    example: 'Ada Lovelace';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['personal-name'];
                                    description: 'The attribute type of the value.';
                                    example: 'personal-name';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'first_name',
                                  'last_name',
                                  'full_name',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  original_phone_number: {
                                    type: 'string';
                                    description: 'The raw, original phone number, as inputted.';
                                    example: '5558675309';
                                  };
                                  country_code: {
                                    type: 'string';
                                    description: 'The ISO 3166-1 alpha-2 country code representing the country that this phone number belongs to.';
                                    enum: [
                                      'AF',
                                      'AX',
                                      'AL',
                                      'DZ',
                                      'AS',
                                      'AD',
                                      'AO',
                                      'AI',
                                      'AQ',
                                      'AG',
                                      'AR',
                                      'AM',
                                      'AW',
                                      'AU',
                                      'AT',
                                      'AZ',
                                      'BS',
                                      'BH',
                                      'BD',
                                      'BB',
                                      'BY',
                                      'BE',
                                      'BZ',
                                      'BJ',
                                      'BM',
                                      'BT',
                                      'BO',
                                      'BA',
                                      'BW',
                                      'BV',
                                      'BR',
                                      'IO',
                                      'BN',
                                      'BG',
                                      'BF',
                                      'BI',
                                      'KH',
                                      'CM',
                                      'CA',
                                      'CV',
                                      'KY',
                                      'CF',
                                      'TD',
                                      'CL',
                                      'CN',
                                      'CX',
                                      'CC',
                                      'CO',
                                      'KM',
                                      'CG',
                                      'CD',
                                      'CK',
                                      'CR',
                                      'CI',
                                      'HR',
                                      'CU',
                                      'CW',
                                      'CY',
                                      'CZ',
                                      'DK',
                                      'DJ',
                                      'DM',
                                      'DO',
                                      'EC',
                                      'EG',
                                      'SV',
                                      'GQ',
                                      'ER',
                                      'EE',
                                      'ET',
                                      'FK',
                                      'FO',
                                      'FJ',
                                      'FI',
                                      'FR',
                                      'GF',
                                      'PF',
                                      'TF',
                                      'GA',
                                      'GM',
                                      'GE',
                                      'DE',
                                      'GH',
                                      'GI',
                                      'GR',
                                      'GL',
                                      'GD',
                                      'GP',
                                      'GU',
                                      'GT',
                                      'GG',
                                      'GN',
                                      'GW',
                                      'GY',
                                      'HT',
                                      'HM',
                                      'VA',
                                      'HN',
                                      'HK',
                                      'HU',
                                      'IS',
                                      'IN',
                                      'ID',
                                      'IR',
                                      'IQ',
                                      'IE',
                                      'IM',
                                      'IL',
                                      'IT',
                                      'JM',
                                      'JP',
                                      'JE',
                                      'JO',
                                      'KZ',
                                      'KE',
                                      'KI',
                                      'KR',
                                      'KW',
                                      'KG',
                                      'LA',
                                      'LV',
                                      'LB',
                                      'LS',
                                      'LR',
                                      'LY',
                                      'LI',
                                      'LT',
                                      'LU',
                                      'MO',
                                      'MK',
                                      'MG',
                                      'MW',
                                      'MY',
                                      'MV',
                                      'ML',
                                      'MT',
                                      'MH',
                                      'MQ',
                                      'MR',
                                      'MU',
                                      'YT',
                                      'MX',
                                      'FM',
                                      'MD',
                                      'MC',
                                      'MN',
                                      'ME',
                                      'MS',
                                      'MA',
                                      'MZ',
                                      'MM',
                                      'NA',
                                      'NR',
                                      'NP',
                                      'NL',
                                      'AN',
                                      'NC',
                                      'NZ',
                                      'NI',
                                      'NE',
                                      'NG',
                                      'NU',
                                      'NF',
                                      'MP',
                                      'NO',
                                      'OM',
                                      'PK',
                                      'PW',
                                      'PS',
                                      'PA',
                                      'PG',
                                      'PY',
                                      'PE',
                                      'PH',
                                      'PN',
                                      'PL',
                                      'PT',
                                      'PR',
                                      'QA',
                                      'RE',
                                      'RO',
                                      'RU',
                                      'RW',
                                      'BL',
                                      'SH',
                                      'KN',
                                      'LC',
                                      'MF',
                                      'PM',
                                      'VC',
                                      'WS',
                                      'SM',
                                      'ST',
                                      'SA',
                                      'SN',
                                      'RS',
                                      'SC',
                                      'SL',
                                      'SG',
                                      'SK',
                                      'SI',
                                      'SB',
                                      'SO',
                                      'ZA',
                                      'GS',
                                      'ES',
                                      'LK',
                                      'SD',
                                      'SR',
                                      'SJ',
                                      'SZ',
                                      'SE',
                                      'CH',
                                      'SY',
                                      'TW',
                                      'TJ',
                                      'TZ',
                                      'TH',
                                      'TL',
                                      'TG',
                                      'TK',
                                      'TO',
                                      'TT',
                                      'TN',
                                      'TR',
                                      'TM',
                                      'TC',
                                      'TV',
                                      'UG',
                                      'UA',
                                      'AE',
                                      'GB',
                                      'US',
                                      'UM',
                                      'UY',
                                      'UZ',
                                      'VU',
                                      'VE',
                                      'VN',
                                      'VG',
                                      'VI',
                                      'WF',
                                      'EH',
                                      'YE',
                                      'ZM',
                                      'ZW',
                                    ];
                                    example: 'US';
                                  };
                                  phone_number: {
                                    type: 'string';
                                    example: '+15558675309';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['phone-number'];
                                    description: 'The attribute type of the value.';
                                    example: 'phone-number';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'original_phone_number',
                                  'country_code',
                                  'phone_number',
                                  'attribute_type',
                                ];
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  status: {
                                    $ref: '#/components/schemas/status';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['status'];
                                    description: 'The attribute type of the value.';
                                    example: 'status';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'status',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'number';
                                    description: 'A number between 0 and 5 (inclusive) to represent a star rating.';
                                    example: 3;
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['rating'];
                                    description: 'The attribute type of the value.';
                                    example: 'rating';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  option: {
                                    $ref: '#/components/schemas/select-option';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['select'];
                                    description: 'The attribute type of the value.';
                                    example: 'select';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'option',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'string';
                                    description: 'A raw text field. Values are limited to 10MB.';
                                    example: 'Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['text'];
                                    description: 'The attribute type of the value.';
                                    example: 'text';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['timestamp'];
                                    description: 'The attribute type of the value.';
                                    example: 'timestamp';
                                  };
                                  value: {
                                    type: 'string';
                                    description: 'A timestamp value represents a single, universal moment in time using an ISO 8601 formatted string. This means that a timestamp consists of a date, a time (with nanosecond precision), and a time zone. Attio will coerce timestamps which do not provide full nanosecond precision and UTC is assumed if no time zone is provided. For example, "2023", "2023-01", "2023-01-02", "2023-01-02T13:00", "2023-01-02T13:00:00", and "2023-01-02T13:00:00.000000000" will all be coerced to "2023-01-02T13:00:00.000000000Z". Timestamps are always returned in UTC. For example, writing a timestamp value using the string "2023-01-02T13:00:00.000000000+02:00" will result in the value "2023-01-02T11:00:00.000000000Z" being returned. The maximum date is "9999-12-31T23:59:59.999999999Z".';
                                    format: 'date';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'attribute_type',
                                  'value',
                                ];
                                additionalProperties: false;
                              },
                            ];
                          };
                        };
                        description: 'A list of attribute values for the list entry (not attribute values for its parent record).';
                        example: {
                          status: [
                            {
                              active_from: '2023-01-01T15:00:00.000000000Z';
                              active_until: null;
                              created_by_actor: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                              status: {
                                id: {
                                  workspace_id: '14beef7a-99f7-4534-a87e-70b564330a4c';
                                  object_id: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
                                  attribute_id: 'e350362f-4b55-4c0f-93f4-379ae8ff2e5b';
                                  status_id: '527def35-7994-4ef7-9584-80ef8de352a8';
                                };
                                title: 'In Progress';
                                is_archived: false;
                                target_time_in_status: null;
                                celebration_enabled: false;
                              };
                              attribute_type: 'status';
                            },
                          ];
                          created_at: [
                            {
                              active_from: '2023-01-01T15:00:00.000000000Z';
                              active_until: null;
                              created_by_actor: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                              value: '2023-01-01T15:00:00.000000000Z';
                              attribute_type: 'timestamp';
                            },
                          ];
                          created_by: [
                            {
                              active_from: '2023-01-01T15:00:00.000000000Z';
                              active_until: null;
                              created_by_actor: {
                                type: 'workspace-member';
                                id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              };
                              referenced_actor_id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                              referenced_actor_type: 'workspace-member';
                              attribute_type: 'actor-reference';
                            },
                          ];
                        };
                      };
                    };
                    required: ['id', 'parent_record_id', 'parent_object', 'created_at', 'entry_values'];
                  };
                };
                required: ['data'];
                description: 'Success';
              };
            };
          };
        };
        '400': {
          description: 'Bad Request';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [400];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['multiple_match_results'];
                  };
                  message: {
                    type: 'string';
                    example: 'Multiple entries were founds for the given parent record. Please remove duplicates or use the update or create entry endpoints instead.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Bad Request';
              };
            };
          };
        };
        '404': {
          description: 'Not Found';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  status_code: {
                    type: 'number';
                    enum: [404];
                  };
                  type: {
                    type: 'string';
                    enum: ['invalid_request_error'];
                  };
                  code: {
                    type: 'string';
                    enum: ['not_found'];
                  };
                  message: {
                    type: 'string';
                    example: 'List with slug/ID "33ebdbe9-e529-47c9-b894-0ba25e9c15c0" was not found.';
                  };
                };
                required: ['status_code', 'type', 'code', 'message'];
                description: 'Not Found';
              };
            };
          };
        };
      };
    };
  };
  '/v2/lists/{list}/entries/{entry_id}': {
    get: {
      summary: 'Get a list entry';
      description: 'Gets a single list entry by its `entry_id`.\n\nRequired scopes: `list_entry:read`, `list_configuration:read`.';
      tags: ['Entries'];
      security: [
        {
          oauth2: ['list_entry:read', 'list_configuration:read'];
        },
      ];
      parameters: [
        {
          schema: {
            type: 'string';
            description: 'A UUID or slug identifying the list the entry is in.';
            example: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
          };
          required: true;
          name: 'list';
          in: 'path';
        },
        {
          schema: {
            type: 'string';
            format: 'uuid';
            description: 'A UUID identifying the entry.';
            example: '2e6e29ea-c4e0-4f44-842d-78a891f8c156';
          };
          required: true;
          name: 'entry_id';
          in: 'path';
        },
      ];
      responses: {
        '200': {
          description: 'Success';
          content: {
            'application/json': {
              schema: {
                type: 'object';
                properties: {
                  data: {
                    type: 'object';
                    properties: {
                      id: {
                        type: 'object';
                        properties: {
                          workspace_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying the workspace this entry belongs to.';
                            example: '14beef7a-99f7-4534-a87e-70b564330a4c';
                          };
                          list_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying the list this entry is in.';
                            example: '33ebdbe9-e529-47c9-b894-0ba25e9c15c0';
                          };
                          entry_id: {
                            type: 'string';
                            format: 'uuid';
                            description: 'A UUID identifying this entry.';
                            example: '2e6e29ea-c4e0-4f44-842d-78a891f8c156';
                          };
                        };
                        required: ['workspace_id', 'list_id', 'entry_id'];
                      };
                      parent_record_id: {
                        type: 'string';
                        format: 'uuid';
                        description: 'A UUID identifying the record that is parent of the list entry.';
                        example: '891dcbfc-9141-415d-9b2a-2238a6cc012d';
                      };
                      parent_object: {
                        type: 'string';
                        description: 'A UUID or slug identifying the object that the parent record belongs to.';
                        example: 'people';
                      };
                      created_at: {
                        type: 'string';
                        description: 'When this entry was created.';
                        example: '2022-11-21T13:22:49.061281000Z';
                      };
                      entry_values: {
                        type: 'object';
                        additionalProperties: {
                          type: 'array';
                          items: {
                            oneOf: [
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  referenced_actor_type: {
                                    type: 'string';
                                    enum: ['api-token', 'workspace-member', 'system'];
                                    description: 'The type of the referenced actor. [Read more information on actor types here](/docs/actors).';
                                    example: 'workspace-member';
                                  };
                                  referenced_actor_id: {
                                    type: ['string', 'null'];
                                    format: 'uuid';
                                    description: 'The ID of the referenced actor.';
                                    example: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['actor-reference'];
                                    description: 'The attribute type of the value.';
                                    example: 'actor-reference';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'referenced_actor_type',
                                  'referenced_actor_id',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  value: {
                                    type: 'boolean';
                                    description: "A boolean representing whether the checkbox is checked or not. The string values 'true' and 'false' are also accepted.";
                                    example: true;
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['checkbox'];
                                    description: 'The attribute type of the value.';
                                    example: 'checkbox';
                                  };
                                };
                                required: [
                                  'active_from',
                                  'active_until',
                                  'created_by_actor',
                                  'value',
                                  'attribute_type',
                                ];
                                additionalProperties: false;
                              },
                              {
                                type: 'object';
                                properties: {
                                  active_from: {
                                    type: 'string';
                                    format: 'date-time';
                                    description: 'The point in time at which this value was made "active". `active_from` can be considered roughly analogous to `created_at`.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  active_until: {
                                    type: ['string', 'null'];
                                    format: 'date-time';
                                    description: 'The point in time at which this value was deactivated. If `null`, the value is active.';
                                    example: '2023-01-01T15:00:00.000000000Z';
                                  };
                                  created_by_actor: {
                                    type: 'object';
                                    description: 'The actor that created this value.';
                                    properties: {
                                      id: {
                                        type: 'string';
                                        description: 'An ID to identify the actor.';
                                        nullable: true;
                                      };
                                      type: {
                                        type: 'string';
                                        enum: ['api-token', 'workspace-member', 'system'];
                                        nullable: true;
                                        description: 'The type of actor. [Read more information on actor types here](/docs/actors).';
                                      };
                                    };
                                    example: {
                                      type: 'workspace-member';
                                      id: '50cf242c-7fa3-4cad-87d0-75b1af71c57b';
                                    };
                                  };
                                  currency_value: {
                                    type: 'number';
                                    description: 'A numerical representation of the currency value. A decimal with a max of 4 decimal places.';
                                    example: 99;
                                  };
                                  currency_code: {
                                    type: ['string', 'null'];
                                    enum: [
                                      'EUR',
                                      'GBP',
                                      'USD',
                                      'CAD',
                                      'AUD',
                                      'NZD',
                                      'AED',
                                      'INR',
                                      'CHF',
                                      'JPY',
                                      'CNY',
                                      'PHP',
                                      'ILS',
                                      'SGD',
                                      'HKD',
                                      'MYR',
                                      'NTD',
                                      'NOK',
                                      'SEK',
                                      'ISK',
                                      'BEL',
                                      'PLN',
                                      'KRW',
                                      'DKK',
                                      'BRL',
                                    ];
                                    description: 'The ISO4217 currency code representing the currency that the value is stored in.';
                                    example: 'USD';
                                  };
                                  attribute_type: {
                                    type: 'string';
                                    enum: ['currency'];
                                    description: 'The attribute type of the value.';
                  