#!/usr/bin/env python
# Jay Smith
# jay.smith@fireeye.com
# 
########################################################################
# Copyright 2015 FireEye
# Copyright 2012 Mandiant
#
# FireEye licenses this file to you under the Apache License, Version
# 2.0 (the "License"); you may not use this file except in compliance with the
# License. You may obtain a copy of the License at:
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied. See the License for the specific language governing
# permissions and limitations under the License.
########################################################################
#
# sogu strings decoder. Written for: 41d059f059795e91137fd411b5f4d74d
#
# Run this in 2 parts:
# 1) Call getArgs() to use argtracker to get all of the necessary arguments
#    and print them out as a list of tuples. Take this data and paste it
#    back into the script
# 2) Comment out getArgs() and call processStuff() to process each argument
#    tuple, decode the strings, and apply comments.
#
# I prefer doing these scripts in 2 passes, since recovering the arguments
# can be time consuming, and it allows me to verify that all arguments are
# correctly extracted before I start decoding.
########################################################################

import array

import idc
import idaapi
import idautils

import flare.jayutils as c_jayutils
import flare.argtracker as c_argtracker

#c_jayutils.configLogger('', logging.DEBUG)
c_jayutils.configLogger('', logging.INFO)
logger = c_jayutils.getLogger('')

def decodeString(strLoc, count, const1):
    buff = idc.GetManyBytes(strLoc, count)
    decString = decodeSoguBytes20100921_strings(buff, 0, count, const1)
    return decString

def decodeSoguBytes20100921_strings(inBytes, offset, count, const1):
    v9 = const1
    v7 = const1
    v8 = const1
    t1 = const1
    xBytes = array.array('B', inBytes[offset:offset+count])
    result = array.array('B', inBytes[offset:offset+count])

    for idx in xrange(len(xBytes)):
        v9 = 0xffffffff & (v9 + (-3 - 8*v9))
        t1 = 0xffffffff & (t1 + (-5 + -32 * t1))
        v7 = 0xffffffff & (129 * v7 + 7)
        v8 = 0xffffffff & (513 * v8 + 9)
        result[idx] = xBytes[idx] ^ (0xff & (v9 + t1 + v7 + v8))
    return result.tostring()


def getArgs():
    '''
    Phase 1:
    For each xref to the string decoder, use argracker to grab the interesting
    arguments. Print them out as a list of tuples that will be used in phase 2.
    '''


    decStringFunc = 0x10002F6C
    vw = c_jayutils.loadWorkspace(c_jayutils.getInputFilepath())
    logger.info('Loaded workspace')
    tracker = c_argtracker.ArgTracker(vw)
    interestingXrefs = idautils.CodeRefsTo(decStringFunc, 1)

    for xref in interestingXrefs:
        argsList = tracker.getPushArgs(xref, 2, ['eax', 'ecx', 'edi'])
        if len(argsList) == 0:
            logger.error('Unable to get push args at: 0x%08x', xref)
        for argDict in argsList:
            locVa, strLoc = argDict[1]
            lenVa, strLen = argDict['edi']
            keyVa, key = argDict['eax']
            print '(0x%08x, 0x%08x, 0x%08x, 0x%08x, 0x%08x, 0x%08x, 0x%08x),' % (
                xref, strLoc, locVa, strLen, lenVa, keyVa, key
            )

def processStuff(results):
    '''
    Phase 2:
    For each argument tuple, decode the string and apply 
    '''
    for cVa, strLoc, locVa, strLen, lenVa, constVa, const1 in results:
        #logger.info('Trying to process 0x%08x: 0x%08x (0x%08x) 0x%04x (0x%08x) 0x%08x (0x%08x)', cVa, strLoc, locVa, strLen, lenVa, const1, constVa)
        try:
            decString = decodeString(strLoc, strLen, const1)
            #logger.infoHex(decString, '0x%08x: %s', strLoc, decString)

            decStringOrig = decString
            if decString.find('\x00\x00') >= 0:
                decString = decString[:decString.find('\x00\x00')]
            if c_jayutils.isWideString(decString):
                decString = c_jayutils.extractBasicWideString(decString)
            if decString.find('\x00') >= 0:
                decString = decString[:decString.find('\x00')]
            idc.MakeUnkn(strLoc, idc.DOUNK_SIMPLE)
            print '0x%08x: %s' % (strLoc, decString)
            #logger.infoHex(decStringOrig, '0x%08x: %s', strLoc, decString)
            idc.MakeRptCmt(strLoc, decString)
            idc.MakeComm(locVa, decString)
        except Exception, err:
            logger.exception('Error processing entry: %s', str(err))

# stuff1 tuples are of the form: 
# callEa, strLoc, locVa, strLen, lenVa, const1, constVa
stuff1 =  [
(0x10001024, 0x10019060, 0x10001014, 0x0000000a, 0x10001019, 0x1000101b, 0x91735667),
(0x10001076, 0x1001906c, 0x10001066, 0x00000010, 0x1000106b, 0x1000106d, 0x4539aadc),
(0x100010c3, 0x10019080, 0x100010b3, 0x00000012, 0x100010b8, 0x100010ba, 0xa65979a0),
(0x10001110, 0x10019094, 0x10001100, 0x00000006, 0x10001105, 0x10001107, 0xfc65f1b4),
(0x10001162, 0x1001909c, 0x10001152, 0x0000000c, 0x10001157, 0x10001159, 0x1af06eb2),
(0x100011b4, 0x100190ac, 0x100011a4, 0x00000011, 0x100011a9, 0x100011ab, 0xe8d146ee),
(0x10001209, 0x100190c0, 0x100011f9, 0x0000000a, 0x100011fe, 0x10001200, 0xb62e2c25),
(0x1000125e, 0x100190cc, 0x1000124e, 0x00000017, 0x10001253, 0x10001255, 0xcdf6a00c),
(0x100012b9, 0x100190f4, 0x100012a9, 0x00000009, 0x100012ae, 0x100012b0, 0x0da8e1b7),
(0x1000130b, 0x10019100, 0x100012fb, 0x0000000d, 0x10001300, 0x10001302, 0x0a1dee36),
(0x10001358, 0x10019144, 0x10001348, 0x00000013, 0x1000134d, 0x1000134f, 0xc045921c),
(0x100013d3, 0x100190e4, 0x100013c3, 0x0000000c, 0x100013c8, 0x100013ca, 0xd97b42e3),
(0x100015cc, 0x1001b988, 0x100015bc, 0x0000000e, 0x100015c1, 0x100015c3, 0x494fd197),
(0x100016c5, 0x10019158, 0x100016b5, 0x00000022, 0x100016ba, 0x100016bc, 0xd07dcf68),
(0x100016ef, 0x1001917c, 0x100016df, 0x0000001e, 0x100016e4, 0x100016e6, 0xe9d2e335),
(0x100018e8, 0x10019110, 0x100018d8, 0x00000011, 0x100018dd, 0x100018df, 0x95230039),
(0x100019f4, 0x1001919c, 0x100019e7, 0x00000004, 0x100019d0, 0x100019ec, 0x028f2563),
(0x10001a58, 0x100191a4, 0x10001a4b, 0x00000004, 0x100019d0, 0x10001a50, 0x8b02b2f1),
(0x10001abc, 0x100191ac, 0x10001aaf, 0x00000004, 0x100019d0, 0x10001ab4, 0x4f1fa5bc),
(0x10001bab, 0x10019124, 0x10001b9a, 0x0000001e, 0x10001b9f, 0x10001ba1, 0x0f0e7132),
(0x10001dbf, 0x100191b4, 0x10001daf, 0x0000000f, 0x10001db4, 0x10001db6, 0xc02f937d),
(0x10001e0c, 0x100191d0, 0x10001dfc, 0x00000009, 0x10001e01, 0x10001e03, 0x297d95af),
(0x10001e74, 0x100191c4, 0x10001e64, 0x0000000a, 0x10001e69, 0x10001e6b, 0x0864eb05),
(0x10001f3c, 0x100191dc, 0x10001f2c, 0x0000000c, 0x10001f31, 0x10001f33, 0x5702fc13),
(0x10001fa8, 0x100191ec, 0x10001f87, 0x0000000a, 0x10001f96, 0x10001f9e, 0x68de42c7),
(0x10001fdf, 0x100191f8, 0x10001fcf, 0x0000000d, 0x10001fd4, 0x10001fd6, 0x0d549d06),
(0x10002024, 0x10019208, 0x1000200b, 0x0000000b, 0x10002016, 0x10002018, 0x6cf02295),
(0x10002069, 0x10019214, 0x10002053, 0x0000000d, 0x1000205e, 0x10002060, 0xe7e377d8),
(0x100020a9, 0x10019224, 0x1000209a, 0x0000000a, 0x10001f96, 0x1000209f, 0x6ce0aa96),
(0x100020ea, 0x10019230, 0x100020da, 0x0000000b, 0x100020df, 0x100020e1, 0x353b05bf),
(0x1000212e, 0x1001923c, 0x10002110, 0x0000000c, 0x10002120, 0x10002122, 0x7f01e135),
(0x1000216f, 0x1001924c, 0x10002162, 0x0000000c, 0x10002120, 0x10002167, 0xe08a9075),
(0x100021af, 0x1001925c, 0x100021a0, 0x0000000a, 0x10001f96, 0x100021a5, 0x7e2af460),
(0x100021f0, 0x10019268, 0x100021d5, 0x00000008, 0x100021e5, 0x100021e7, 0x0a399b67),
(0x10002231, 0x10019274, 0x10002216, 0x0000000d, 0x10002220, 0x10002228, 0xc50de9ef),
(0x10002272, 0x10019284, 0x10002257, 0x0000000c, 0x10002267, 0x10002269, 0x7d004d09),
(0x100022b3, 0x10019294, 0x10002298, 0x0000000b, 0x100022a2, 0x100022aa, 0x8e6a7a4b),
(0x100022e2, 0x100192a0, 0x100022d2, 0x0000000c, 0x100022d7, 0x100022d9, 0xe3a7c031),
(0x10002311, 0x100192b0, 0x10002301, 0x00000008, 0x10002306, 0x10002308, 0xf5156b67),
(0x10002340, 0x100192bc, 0x10002330, 0x0000000b, 0x10002335, 0x10002337, 0xc9134682),
(0x1000236e, 0x100192c8, 0x1000235f, 0x0000000a, 0x10001f96, 0x10002364, 0xf4043b31),
(0x100023a0, 0x100192d4, 0x1000238d, 0x0000000d, 0x10002392, 0x10002394, 0x984f1da7),
(0x100023d4, 0x100192e4, 0x100023c2, 0x0000000a, 0x10001f96, 0x100023c7, 0x665fad96),
(0x1000240c, 0x100192f0, 0x100023fc, 0x0000000a, 0x10001f96, 0x10002401, 0x9ca7daba),
(0x10002529, 0x100192fc, 0x10002519, 0x0000000f, 0x1000251e, 0x10002520, 0x68d3c682),
(0x10002583, 0x1001930c, 0x10002573, 0x0000000d, 0x10002578, 0x1000257a, 0xe453c3b7),
(0x100025df, 0x1001931c, 0x100025cf, 0x0000000c, 0x100025d4, 0x100025d6, 0xc9de612f),
(0x10002638, 0x1001932c, 0x10002628, 0x00000007, 0x1000262d, 0x1000262f, 0x98351f1e),
(0x10002691, 0x10019334, 0x10002681, 0x00000007, 0x10002686, 0x10002688, 0xe1732638),
(0x10002b27, 0x1001933c, 0x10002b17, 0x00000009, 0x10002b1c, 0x10002b1e, 0x3538064f),
(0x10002b79, 0x10019348, 0x10002b69, 0x00000009, 0x10002b6e, 0x10002b70, 0x1983fff7),
(0x10002bcb, 0x10019354, 0x10002bbb, 0x00000014, 0x10002bc0, 0x10002bc2, 0xb82e0aae),
(0x10002c2b, 0x1001936c, 0x10002c1b, 0x00000014, 0x10002c20, 0x10002c22, 0x38869473),
(0x10002c8f, 0x10019384, 0x10002c7f, 0x00000007, 0x10002c84, 0x10002c86, 0x6701971c),
(0x100030bf, 0x1001938c, 0x100030af, 0x0000001a, 0x100030b4, 0x100030b6, 0x6ab46a7a),
(0x1000312c, 0x100193a8, 0x1000311c, 0x00000016, 0x10003121, 0x10003123, 0xff2db7c6),
(0x100031b3, 0x100193c0, 0x1000319f, 0x0000000d, 0x100031a4, 0x100031a6, 0x03d12e80),
(0x100031f8, 0x100193d0, 0x100031e4, 0x00000022, 0x100031e9, 0x100031eb, 0xefd87fac),
(0x10003227, 0x100193f4, 0x10003213, 0x0000001e, 0x10003218, 0x1000321a, 0x7e1af7b7),
(0x1000351e, 0x1001982c, 0x1000350a, 0x00000014, 0x1000350f, 0x10003511, 0xd34526a7),
(0x1000360a, 0x10019874, 0x100035f6, 0x00000015, 0x100035fb, 0x100035fd, 0xaa0900a6),
(0x100036d2, 0x10019bfc, 0x100036be, 0x00000018, 0x100036c3, 0x100036c5, 0x0b958d36),
(0x1000371e, 0x10019c18, 0x1000370a, 0x00000004, 0x1000370f, 0x10003711, 0xd151cce0),
(0x10003754, 0x10019c20, 0x10003740, 0x00000020, 0x10003745, 0x10003747, 0xfa219161),
(0x100037a0, 0x10019c44, 0x1000378c, 0x00000004, 0x10003791, 0x10003793, 0x8d2d7cc8),
(0x100037d6, 0x10019c4c, 0x100037c2, 0x00000018, 0x100037c7, 0x100037c9, 0x1fd6f2ca),
(0x10003822, 0x10019c68, 0x1000380e, 0x00000004, 0x10003813, 0x10003815, 0xd5c72807),
(0x10003858, 0x10019c70, 0x10003844, 0x00000004, 0x10003813, 0x1000384b, 0x61023121),
(0x10003858, 0x10019c70, 0x10003844, 0x00000018, 0x100037c7, 0x1000384b, 0x61023121),
(0x100038a4, 0x10019c8c, 0x10003890, 0x00000004, 0x10003895, 0x10003897, 0x01ff218c),
(0x100038da, 0x10019c94, 0x100038c6, 0x0000001e, 0x100038cb, 0x100038cd, 0x67c6923b),
(0x10003926, 0x10019b90, 0x10003912, 0x00000004, 0x10003917, 0x10003919, 0xe39943cb),
(0x10003dc1, 0x10019414, 0x10003db1, 0x0000000c, 0x10003db6, 0x10003db8, 0x7c380348),
(0x10003e13, 0x10019424, 0x10003e03, 0x0000000d, 0x10003e08, 0x10003e0a, 0x77470a28),
(0x10003e65, 0x10019458, 0x10003e55, 0x00000014, 0x10003e5a, 0x10003e5c, 0x961825c7),
(0x10003ec0, 0x10019470, 0x10003eb0, 0x00000016, 0x10003eb5, 0x10003eb7, 0x37740d10),
(0x10003f25, 0x10019488, 0x10003f15, 0x00000011, 0x10003f1a, 0x10003f1c, 0x20111ec4),
(0x10003f79, 0x1001949c, 0x10003f69, 0x0000000c, 0x10003f6e, 0x10003f70, 0x7ce72d41),
(0x10003fdc, 0x100194ac, 0x10003fcc, 0x0000000c, 0x10003fd1, 0x10003fd3, 0x88fb0110),
(0x10004031, 0x100194bc, 0x10004021, 0x0000000c, 0x10004026, 0x10004028, 0xac64ca0b),
(0x1000408c, 0x100194cc, 0x1000407c, 0x0000000a, 0x10004081, 0x10004083, 0x9147791b),
(0x100040ea, 0x100194d8, 0x100040da, 0x00000009, 0x100040df, 0x100040e1, 0x6c16fc3f),
(0x10004148, 0x100194e4, 0x10004138, 0x0000000d, 0x1000413d, 0x1000413f, 0xdfd83f11),
(0x1000419a, 0x100194f4, 0x1000418a, 0x0000000c, 0x1000418f, 0x10004191, 0x31c7adcb),
(0x100041f5, 0x10019504, 0x100041e5, 0x0000000f, 0x100041ea, 0x100041ec, 0x5ab7f310),
(0x10004250, 0x10019514, 0x10004240, 0x0000000f, 0x10004245, 0x10004247, 0xc50de019),
(0x100042a5, 0x10019524, 0x10004295, 0x0000000e, 0x1000429a, 0x1000429c, 0xcd9821e1),
(0x100042fa, 0x10019534, 0x100042ea, 0x0000000a, 0x100042ef, 0x100042f1, 0x06b492d6),
(0x1000434c, 0x10019540, 0x1000433c, 0x00000011, 0x10004341, 0x10004343, 0xb70aabb5),
(0x1000439e, 0x10019554, 0x1000438e, 0x00000009, 0x10004393, 0x10004395, 0xcdaf5575),
(0x100043f5, 0x10019560, 0x100043e5, 0x0000000f, 0x100043ea, 0x100043ec, 0x730169b0),
(0x100044f2, 0x10019434, 0x100044e2, 0x00000010, 0x100044e7, 0x100044e9, 0xe934b7e1),
(0x10004546, 0x10019584, 0x1000453a, 0x00000022, 0x1000453b, 0x1000453d, 0x58bdac8e),
(0x1000455e, 0x10019584, 0x10004555, 0x00000022, 0x1000453b, 0x10004556, 0x58bdac8e),
(0x100045c8, 0x10019448, 0x100045b8, 0x0000000e, 0x100045bd, 0x100045bf, 0xd20e50e6),
(0x1000465a, 0x100195a4, 0x10004646, 0x00000006, 0x1000464b, 0x10004651, 0x1ed4cbe2),
(0x100046a0, 0x100195ac, 0x10004693, 0x00000006, 0x1000464b, 0x10004698, 0x0b052000),
(0x1000474f, 0x100195b4, 0x1000473f, 0x00000008, 0x10004744, 0x10004746, 0x00911cc1),
(0x10004eb1, 0x10019570, 0x10004ea4, 0x00000011, 0x10004e1b, 0x10004ea9, 0x2b23503d),
(0x10004f3f, 0x100195cc, 0x10004f2f, 0x00000008, 0x10004f34, 0x10004f36, 0x23c839b4),
(0x10005110, 0x10019624, 0x10005100, 0x0000000e, 0x10005105, 0x10005107, 0x82522304),
(0x1000523c, 0x10019600, 0x1000522c, 0x0000000e, 0x10005231, 0x10005233, 0x4c076756),
(0x100052c6, 0x10019610, 0x100052b6, 0x00000012, 0x100052bb, 0x100052bd, 0xb13919f5),
(0x1000535a, 0x10019638, 0x10005346, 0x00000006, 0x1000534b, 0x10005351, 0xb7d7bc05),
(0x1000539d, 0x10019640, 0x10005389, 0x00000006, 0x1000538e, 0x10005394, 0x566dc833),
(0x100053e4, 0x10019648, 0x100053d0, 0x00000006, 0x100053d5, 0x100053db, 0xf543ed02),
(0x1000542b, 0x10019650, 0x10005417, 0x00000006, 0x1000541c, 0x10005422, 0xe33c61b1),
(0x100054da, 0x10019658, 0x100054ca, 0x0000000e, 0x100054cf, 0x100054d1, 0x598a28b1),
(0x10005551, 0x10019700, 0x10005541, 0x0000000c, 0x10005546, 0x10005548, 0x8ded340a),
(0x100056f0, 0x1001975c, 0x100056e3, 0x00000004, 0x100056c9, 0x100056e8, 0xd5f81d63),
(0x1000572f, 0x10019764, 0x10005722, 0x00000004, 0x100056c9, 0x10005727, 0x33112221),
(0x10005759, 0x1001974c, 0x10005749, 0x0000000e, 0x1000574e, 0x10005750, 0x4c3e821b),
(0x10005783, 0x1001973c, 0x10005773, 0x0000000e, 0x10005778, 0x1000577a, 0xf14473d2),
(0x100057ad, 0x10019710, 0x1000579d, 0x00000028, 0x100057a2, 0x100057a4, 0xaee5a6d6),
(0x100058aa, 0x100196e0, 0x1000589a, 0x0000000c, 0x1000589f, 0x100058a1, 0xdbae0dc6),
(0x100059a0, 0x10019668, 0x10005990, 0x00000022, 0x10005995, 0x10005997, 0x334ef499),
(0x10005a49, 0x100196b0, 0x10005a39, 0x00000014, 0x10005a3e, 0x10005a40, 0x76d47df8),
(0x10005c22, 0x100197b8, 0x10005c15, 0x00000004, 0x10005bfb, 0x10005c1a, 0x0f320868),
(0x10005c61, 0x100197c0, 0x10005c54, 0x00000004, 0x10005bfb, 0x10005c59, 0x5dacf315),
(0x10005c8b, 0x100197a8, 0x10005c7b, 0x0000000e, 0x10005c80, 0x10005c82, 0x00e99fb3),
(0x10005cb5, 0x10019798, 0x10005ca5, 0x0000000e, 0x10005caa, 0x10005cac, 0x3bb2ed44),
(0x10005cdf, 0x1001976c, 0x10005ccf, 0x00000028, 0x10005cd4, 0x10005cd6, 0x5698361d),
(0x10005ddc, 0x100196f0, 0x10005dcc, 0x0000000c, 0x10005dd1, 0x10005dd3, 0x6f984145),
(0x10005eb2, 0x1001968c, 0x10005ea2, 0x00000022, 0x10005ea7, 0x10005ea9, 0x9ae207c9),
(0x10005f54, 0x100196c8, 0x10005f44, 0x00000014, 0x10005f49, 0x10005f4b, 0x7799cd31),
(0x10006224, 0x100197c8, 0x10006214, 0x0000000c, 0x10006219, 0x1000621b, 0x297d8f9c),
(0x10006279, 0x100197d8, 0x10006269, 0x0000000c, 0x1000626e, 0x10006270, 0x6e6931b1),
(0x100062cb, 0x100197e8, 0x100062bb, 0x0000000c, 0x100062c0, 0x100062c2, 0x871b4a0f),
(0x10006322, 0x100197f8, 0x10006312, 0x0000000f, 0x10006317, 0x10006319, 0x2826424f),
(0x10006378, 0x10019808, 0x10006368, 0x0000000d, 0x1000636d, 0x1000636f, 0x41eae9c6),
(0x100063d0, 0x10019818, 0x100063c0, 0x00000013, 0x100063c5, 0x100063c7, 0x5521b8fc),
(0x10006422, 0x10019844, 0x10006412, 0x00000014, 0x10006417, 0x10006419, 0xbc1e58fb),
(0x1000647f, 0x1001985c, 0x1000646f, 0x00000015, 0x10006474, 0x10006476, 0x9dda0b3a),
(0x100064e2, 0x1001988c, 0x100064d2, 0x0000000e, 0x100064d7, 0x100064d9, 0x2c0c97cb),
(0x10006534, 0x1001989c, 0x10006524, 0x0000000e, 0x10006529, 0x1000652b, 0x9fae6586),
(0x1000658a, 0x100198ac, 0x1000657a, 0x0000000f, 0x1000657f, 0x10006581, 0xb30d8922),
(0x100065e1, 0x100198bc, 0x100065d1, 0x0000000a, 0x100065d6, 0x100065d8, 0x8bffb26d),
(0x10006639, 0x100198c8, 0x10006629, 0x0000000a, 0x1000662e, 0x10006630, 0x2c0988ca),
(0x1000669a, 0x100198d4, 0x1000668a, 0x00000009, 0x1000668f, 0x10006691, 0x2c7b82fb),
(0x100066ec, 0x100198e0, 0x100066dc, 0x0000000d, 0x100066e1, 0x100066e3, 0x19d39ffa),
(0x1000673e, 0x100198f0, 0x1000672e, 0x0000000f, 0x10006733, 0x10006735, 0x426025a3),
(0x1000679a, 0x10019900, 0x1000678a, 0x0000000e, 0x1000678f, 0x10006791, 0xc33e5626),
(0x100067f2, 0x10019910, 0x100067e2, 0x00000013, 0x100067e7, 0x100067e9, 0x2a906e55),
(0x1000684d, 0x10019924, 0x1000683d, 0x00000015, 0x10006842, 0x10006844, 0xd34a1974),
(0x100068a8, 0x10019954, 0x10006898, 0x00000013, 0x1000689d, 0x1000689f, 0x6bb944ff),
(0x10006a7f, 0x100199f4, 0x10006a6f, 0x00000004, 0x10006a23, 0x10006a74, 0x06e338c3),
(0x10006aac, 0x100199fc, 0x10006a9f, 0x00000004, 0x10006a23, 0x10006aa4, 0x28c19d90),
(0x10006adc, 0x100199c8, 0x10006ac9, 0x0000001a, 0x10006ace, 0x10006ad0, 0x59c4b7ca),
(0x10006b0f, 0x100199e4, 0x10006afc, 0x0000000e, 0x10006b01, 0x10006b03, 0xf7111355),
(0x10006b3f, 0x1001999c, 0x10006b2f, 0x0000001a, 0x10006b34, 0x10006b36, 0xbfd64e9d),
(0x10006b6f, 0x100199b8, 0x10006b5c, 0x0000000e, 0x10006b61, 0x10006b63, 0x1db7b96c),
(0x10006b9f, 0x10019970, 0x10006b8c, 0x0000001a, 0x10006b91, 0x10006b93, 0x5906edb6),
(0x10006bd2, 0x1001998c, 0x10006bbf, 0x0000000e, 0x10006bc4, 0x10006bc6, 0x4b34ba4e),
(0x10006c2c, 0x10019a50, 0x10006c1c, 0x00000004, 0x10006bf7, 0x10006c21, 0x0e35406d),
(0x10006c5f, 0x10019a40, 0x10006c4c, 0x0000000e, 0x10006c51, 0x10006c53, 0x8d2c14e0),
(0x10006c8f, 0x10019a30, 0x10006c7c, 0x0000000e, 0x10006c81, 0x10006c83, 0xe8675e35),
(0x10006cbc, 0x10019a04, 0x10006cac, 0x00000028, 0x10006cb1, 0x10006cb3, 0x8a83b3e1),
(0x10006ce9, 0x10019a58, 0x10006cd9, 0x00000018, 0x10006cde, 0x10006ce0, 0x05a5b2e9),
(0x10006d2d, 0x10019a74, 0x10006d1a, 0x00000004, 0x10006d1f, 0x10006d21, 0x65ea6c34),
(0x10006d60, 0x10019a7c, 0x10006d4d, 0x00000020, 0x10006d52, 0x10006d54, 0xaf76393d),
(0x10006da4, 0x10019aa0, 0x10006d91, 0x00000004, 0x10006d96, 0x10006d98, 0xf0c32838),
(0x10006dd7, 0x10019aa8, 0x10006dc4, 0x00000018, 0x10006dc9, 0x10006dcb, 0xc6545e3f),
(0x10006e1b, 0x10019ac4, 0x10006e08, 0x00000004, 0x10006e0d, 0x10006e0f, 0x5ab86835),
(0x10006e4e, 0x10019acc, 0x10006e3b, 0x00000004, 0x10006e0d, 0x10006e42, 0x668c2794),
(0x10006e4e, 0x10019acc, 0x10006e3b, 0x00000018, 0x10006dc9, 0x10006e42, 0x668c2794),
(0x10006e92, 0x10019ae8, 0x10006e7f, 0x00000004, 0x10006e84, 0x10006e86, 0x307e845d),
(0x10006ec5, 0x10019af0, 0x10006eb2, 0x0000001e, 0x10006eb7, 0x10006eb9, 0x8d551891),
(0x10006f09, 0x10019b10, 0x10006ef6, 0x00000004, 0x10006efb, 0x10006efd, 0xdff2173a),
(0x100072c6, 0x1001993c, 0x100072b3, 0x00000015, 0x100072b8, 0x100072ba, 0xb2a14854),
(0x10007345, 0x10019ac4, 0x10007335, 0x00000004, 0x1000733a, 0x1000733c, 0x388853f5),
(0x10007374, 0x10019b18, 0x10007364, 0x0000000e, 0x10007369, 0x1000736b, 0x7429b73d),
(0x100073ca, 0x10019b48, 0x100073b7, 0x00000018, 0x100073bc, 0x100073be, 0x8df11f3c),
(0x10007411, 0x10019b64, 0x100073fe, 0x00000004, 0x10007403, 0x10007405, 0x6c429c5c),
(0x10007444, 0x10019b6c, 0x10007431, 0x00000020, 0x10007436, 0x10007438, 0x13369ce2),
(0x10007485, 0x10019b90, 0x10007475, 0x00000004, 0x1000747a, 0x1000747c, 0x8d42808b),
(0x100074b2, 0x10019b98, 0x100074a2, 0x00000018, 0x100074a7, 0x100074a9, 0x72e4f6e4),
(0x100074f0, 0x10019968, 0x100074e0, 0x00000004, 0x100074e5, 0x100074e7, 0xeaa87725),
(0x10007520, 0x10019bb4, 0x1000750d, 0x00000004, 0x100074e5, 0x10007514, 0xa9265fc3),
(0x10007520, 0x10019bb4, 0x1000750d, 0x00000018, 0x100074a7, 0x10007514, 0xa9265fc3),
(0x10007564, 0x10019b90, 0x10007551, 0x00000004, 0x10007556, 0x10007558, 0xf10aea4b),
(0x10007597, 0x10019bd0, 0x10007584, 0x0000001e, 0x10007589, 0x1000758b, 0x931a2bf6),
(0x100075db, 0x10019bf0, 0x100075c8, 0x00000004, 0x100075cd, 0x100075cf, 0xee366a39),
(0x1000762e, 0x10019b24, 0x1000761b, 0x0000001e, 0x10007620, 0x10007622, 0xc783745f),
(0x10007685, 0x10019b40, 0x10007672, 0x00000006, 0x10007677, 0x10007679, 0xfbd42250),
(0x10007c9e, 0x10019cb4, 0x10007c7a, 0x00000010, 0x10007c8a, 0x10007c8c, 0xcdb9b675),
(0x10007dfc, 0x10019cc8, 0x10007dec, 0x0000004c, 0x10007df1, 0x10007df3, 0x26411b90),
(0x10007e35, 0x10019d10, 0x10007e25, 0x0000001a, 0x10007e2a, 0x10007e2c, 0x3c2701cd),
(0x10007e69, 0x10019d2c, 0x10007e52, 0x00000016, 0x10007e57, 0x10007e59, 0x0f2ef796),
(0x10007edc, 0x10019d44, 0x10007ecc, 0x0000000e, 0x10007ed1, 0x10007ed3, 0xd4cac99a),
(0x10007f3d, 0x10019d54, 0x10007f2d, 0x00000010, 0x10007f32, 0x10007f34, 0xf22d3d04),
(0x10007fa0, 0x10019d68, 0x10007f90, 0x00000011, 0x10007f95, 0x10007f97, 0x7f4bb014),
(0x10007fff, 0x10019d7c, 0x10007fef, 0x0000000f, 0x10007ff4, 0x10007ff6, 0x4ddb2d14),
(0x1000805f, 0x10019d9c, 0x1000804f, 0x0000000c, 0x10008054, 0x10008056, 0xe23362e0),
(0x100080b1, 0x10019dac, 0x100080a1, 0x0000000b, 0x100080a6, 0x100080a8, 0x8ae2a68d),
(0x1000810b, 0x10019dd8, 0x100080fb, 0x0000000d, 0x10008100, 0x10008102, 0x9b4b0268),
(0x10008160, 0x10019de8, 0x10008150, 0x0000000f, 0x10008155, 0x10008157, 0xa068731b),
(0x100081b8, 0x10019df8, 0x100081a8, 0x0000000c, 0x100081ad, 0x100081af, 0x364cec96),
(0x10008363, 0x10019db8, 0x10008353, 0x0000000d, 0x10008358, 0x1000835a, 0x6bba9aa9),
(0x10008417, 0x10019d8c, 0x10008407, 0x0000000e, 0x1000840c, 0x1000840e, 0x2e1e1e4a),
(0x100087bb, 0x10019dc8, 0x100087ab, 0x0000000d, 0x100087b0, 0x100087b2, 0x6a89e6df),
(0x10008b35, 0x10019e08, 0x10008b25, 0x00000018, 0x10008b2a, 0x10008b2c, 0x36341c24),
(0x10008b82, 0x10019e38, 0x10008b72, 0x00000018, 0x10008b77, 0x10008b79, 0x6167730a),
(0x10008bd4, 0x10019e54, 0x10008bc4, 0x00000013, 0x10008bc9, 0x10008bcb, 0x1618cf21),
(0x10008c26, 0x10019e68, 0x10008c16, 0x00000011, 0x10008c1b, 0x10008c1d, 0x0aa0ff5a),
(0x10008c7c, 0x10019e7c, 0x10008c6c, 0x00000011, 0x10008c71, 0x10008c73, 0x91571bfd),
(0x10008cce, 0x10019e90, 0x10008cbe, 0x00000011, 0x10008cc3, 0x10008cc5, 0x8b1b6d8c),
(0x10008d20, 0x10019ea4, 0x10008d10, 0x0000000d, 0x10008d15, 0x10008d17, 0xce375d55),
(0x10008d75, 0x10019ee4, 0x10008d65, 0x00000013, 0x10008d6a, 0x10008d6c, 0xbe2d1e3f),
(0x10008dc2, 0x10019ef8, 0x10008db2, 0x0000000d, 0x10008db7, 0x10008db9, 0xde293adb),
(0x10008e20, 0x10019f08, 0x10008e10, 0x00000014, 0x10008e15, 0x10008e17, 0xedcb74b0),
(0x10008e75, 0x10019f20, 0x10008e65, 0x00000013, 0x10008e6a, 0x10008e6c, 0x9a9ae3b0),
(0x10008ec7, 0x10019f34, 0x10008eb7, 0x00000011, 0x10008ebc, 0x10008ebe, 0x645174b5),
(0x10008f25, 0x10019f48, 0x10008f15, 0x00000011, 0x10008f1a, 0x10008f1c, 0x6ade1b6b),
(0x10008f7f, 0x10019f5c, 0x10008f6f, 0x00000009, 0x10008f74, 0x10008f76, 0xb1131259),
(0x10008fcc, 0x10019f68, 0x10008fbc, 0x0000000e, 0x10008fc1, 0x10008fc3, 0x9fcfd4f7),
(0x10009023, 0x10019f78, 0x10009013, 0x00000007, 0x10009018, 0x1000901a, 0x7539f824),
(0x10009088, 0x10019f80, 0x10009078, 0x0000000d, 0x1000907d, 0x1000907f, 0x170304a0),
(0x1000910b, 0x10019f90, 0x100090ef, 0x00000010, 0x100090f4, 0x10009106, 0xcb4242cf),
(0x10009717, 0x10019fa4, 0x100096d0, 0x00000010, 0x100096d5, 0x100096de, 0x278b2c9c),
(0x1000993c, 0x10019fb8, 0x1000992c, 0x00000010, 0x10009931, 0x10009933, 0x1309e7c2),
(0x10009bef, 0x10019fcc, 0x10009bbd, 0x00000010, 0x10009bc2, 0x10009bdf, 0x51f7917e),
(0x10009f8e, 0x10019ed4, 0x10009f7e, 0x0000000d, 0x10009f83, 0x10009f85, 0x3a7c5629),
(0x10009ff7, 0x10019ec4, 0x10009fe7, 0x0000000c, 0x10009fec, 0x10009fee, 0x8304fbec),
(0x1000a073, 0x10019eb4, 0x1000a063, 0x0000000c, 0x1000a068, 0x1000a06a, 0x51bca90a),
(0x1000a0f8, 0x10019fe0, 0x1000a0da, 0x00000010, 0x1000a0e4, 0x1000a0ec, 0x3d331345),
(0x1000a11c, 0x10019e24, 0x1000a10c, 0x00000013, 0x1000a111, 0x1000a113, 0x28a23a17),
(0x1000a20a, 0x10019ff4, 0x1000a1fa, 0x00000009, 0x1000a1ff, 0x1000a201, 0x98d8f928),
(0x1000a257, 0x1001a000, 0x1000a247, 0x00000011, 0x1000a24c, 0x1000a24e, 0x5212f63d),
(0x1000a2ac, 0x1001a014, 0x1000a29c, 0x00000011, 0x1000a2a1, 0x1000a2a3, 0x2c2028f8),
(0x1000a316, 0x1001a028, 0x1000a306, 0x00000014, 0x1000a30b, 0x1000a30d, 0x8f3cc7e1),
(0x1000a370, 0x1001a040, 0x1000a360, 0x0000000d, 0x1000a365, 0x1000a367, 0xf97da6a2),
(0x1000a4d5, 0x1001a050, 0x1000a4ae, 0x00000022, 0x1000a4b3, 0x1000a4b5, 0xe0c29f7b),
(0x1000a578, 0x1001a06c, 0x1000a568, 0x00000022, 0x1000a4b3, 0x1000a56d, 0x7f7fc054),
(0x1000a65f, 0x1001a088, 0x1000a62c, 0x00000010, 0x1000a631, 0x1000a639, 0x5e79381f),
(0x1000ac26, 0x1001a09c, 0x1000ac16, 0x0000000f, 0x1000ac1b, 0x1000ac1d, 0x31c72d35),
(0x1000ac7e, 0x1001a0ac, 0x1000ac6e, 0x0000000e, 0x1000ac73, 0x1000ac75, 0xa0b94d21),
(0x1000acdb, 0x1001a0bc, 0x1000accb, 0x00000012, 0x1000acd0, 0x1000acd2, 0xba33448b),
(0x1000ad3c, 0x1001a0d0, 0x1000ad2c, 0x0000000e, 0x1000ad31, 0x1000ad33, 0x84aa4acc),
(0x1000ad8e, 0x1001a0e0, 0x1000ad7e, 0x0000000e, 0x1000ad83, 0x1000ad85, 0x5ef4d9a3),
(0x1000ade3, 0x1001a0f0, 0x1000add3, 0x0000000f, 0x1000add8, 0x1000adda, 0x360c8148),
(0x1000ae3a, 0x1001a110, 0x1000ae2a, 0x00000011, 0x1000ae2f, 0x1000ae31, 0x5573277f),
(0x1000ae8f, 0x1001a150, 0x1000ae7f, 0x0000000f, 0x1000ae84, 0x1000ae86, 0xc28a9fb0),
(0x1000b1d7, 0x1001a124, 0x1000b1c7, 0x00000011, 0x1000b1cc, 0x1000b1ce, 0x670db26f),
(0x1000b2a8, 0x1001a138, 0x1000b298, 0x00000009, 0x1000b29d, 0x1000b29f, 0x82bcaf2f),
(0x1000b309, 0x1001a144, 0x1000b2f9, 0x0000000b, 0x1000b2fe, 0x1000b300, 0xb4d3c6fd),
(0x1000b35f, 0x1001a160, 0x1000b34f, 0x0000000a, 0x1000b354, 0x1000b356, 0xf392297f),
(0x1000b410, 0x1001a100, 0x1000b400, 0x0000000f, 0x1000b405, 0x1000b407, 0x851738c4),
(0x1000b54f, 0x1001a190, 0x1000b53f, 0x0000000e, 0x1000b544, 0x1000b546, 0xe54fd61b),
(0x1000b5a3, 0x1001a1a0, 0x1000b593, 0x00000018, 0x1000b598, 0x1000b59a, 0x5be994a7),
(0x1000b61c, 0x1001a17c, 0x1000b60c, 0x00000010, 0x1000b611, 0x1000b613, 0xec0c7127),
(0x1000b6ec, 0x1001a1bc, 0x1000b6dc, 0x00000028, 0x1000b6e1, 0x1000b6e3, 0x09065e75),
(0x1000b757, 0x1001a1e8, 0x1000b747, 0x00000028, 0x1000b74c, 0x1000b74e, 0x3d5e5865),
(0x1000b7d6, 0x1001a214, 0x1000b7bd, 0x00000028, 0x1000b7c7, 0x1000b7cd, 0x540ee9d9),
(0x1000b8fd, 0x1001a16c, 0x1000b8ed, 0x0000000c, 0x1000b8f2, 0x1000b8f4, 0xc89ca3a6),
(0x1000b95f, 0x1001a2a4, 0x1000b94f, 0x0000000f, 0x1000b954, 0x1000b956, 0x863e3cfb),
(0x1000b9b4, 0x1001a2cc, 0x1000b9a4, 0x00000016, 0x1000b9a9, 0x1000b9ab, 0x8446831a),
(0x1000ba0b, 0x1001a330, 0x1000b9fb, 0x0000001b, 0x1000ba00, 0x1000ba02, 0x26d778ef),
(0x1000ba8b, 0x1001a2e4, 0x1000ba7b, 0x00000013, 0x1000ba80, 0x1000ba82, 0x52612b0b),
(0x1000bb12, 0x1001a280, 0x1000bb02, 0x0000000d, 0x1000bb07, 0x1000bb09, 0xb79b0aea),
(0x1000bb64, 0x1001a290, 0x1000bb54, 0x00000013, 0x1000bb59, 0x1000bb5b, 0xad445d9a),
(0x1000bbb7, 0x1001a2b4, 0x1000bba7, 0x00000016, 0x1000bbac, 0x1000bbae, 0xc9739b74),
(0x1000bbff, 0x1001a360, 0x1000bbef, 0x00000015, 0x1000bbf4, 0x1000bbf6, 0xb77e6056),
(0x1000bc4a, 0x1001a314, 0x1000bc3a, 0x0000001b, 0x1000bc3f, 0x1000bc41, 0x5ef7904f),
(0x1000bd62, 0x1001a34c, 0x1000bd52, 0x00000013, 0x1000bd57, 0x1000bd59, 0xbc71fd18),
(0x1000bfa3, 0x1001a388, 0x1000bf93, 0x00000008, 0x1000bf98, 0x1000bf9a, 0x1e70923a),
(0x1000bfd0, 0x1001a394, 0x1000bfc0, 0x00000004, 0x1000bfc5, 0x1000bfc7, 0x1d71ef99),
(0x1000bffd, 0x1001a39c, 0x1000bfed, 0x00000006, 0x1000bff2, 0x1000bff4, 0x134694bd),
(0x1000c033, 0x1001a25c, 0x1000c023, 0x0000000d, 0x1000c028, 0x1000c02a, 0x4a77ab06),
(0x1000c07f, 0x1001a26c, 0x1000c06f, 0x00000011, 0x1000c074, 0x1000c076, 0xb4f90b44),
(0x1000c0c6, 0x1001a240, 0x1000c0b6, 0x0000000b, 0x1000c0bb, 0x1000c0bd, 0x154d4eb8),
(0x1000c12b, 0x1001a24c, 0x1000c11b, 0x0000000d, 0x1000c120, 0x1000c122, 0xa34d3620),
(0x1000c2d3, 0x1001a378, 0x1000c2c3, 0x0000000c, 0x1000c2c8, 0x1000c2ca, 0xfcc0d4cb),
(0x1000c3a1, 0x1001a3a4, 0x1000c391, 0x0000000e, 0x1000c396, 0x1000c398, 0xc72ace2b),
(0x1000c406, 0x1001a2f8, 0x1000c3f6, 0x00000019, 0x1000c3fb, 0x1000c3fd, 0x5dc9220b),
(0x1000c4da, 0x1001a3a4, 0x1000c4ca, 0x0000000e, 0x1000c4cf, 0x1000c4d1, 0x8e97fb6b),
(0x1000c520, 0x1001a3b4, 0x1000c510, 0x00000010, 0x1000c515, 0x1000c517, 0x4d54b536),
(0x1000c5f6, 0x1001a3c8, 0x1000c5e6, 0x00000010, 0x1000c5eb, 0x1000c5ed, 0xce8e74a7),
(0x1000c658, 0x1001a3dc, 0x1000c648, 0x0000000f, 0x1000c64d, 0x1000c64f, 0xd096dd6a),
(0x1000c6b1, 0x1001a3ec, 0x1000c6a1, 0x0000000e, 0x1000c6a6, 0x1000c6a8, 0x894de3d4),
(0x1000c703, 0x1001a3fc, 0x1000c6f3, 0x00000011, 0x1000c6f8, 0x1000c6fa, 0xe6771139),
(0x1000c755, 0x1001a420, 0x1000c745, 0x00000009, 0x1000c74a, 0x1000c74c, 0xd4ae8c52),
(0x1000c7b3, 0x1001a42c, 0x1000c7a3, 0x0000000a, 0x1000c7a8, 0x1000c7aa, 0x785365bb),
(0x1000c808, 0x1001a448, 0x1000c7f8, 0x00000011, 0x1000c7fd, 0x1000c7ff, 0x63c79299),
(0x1000c85a, 0x1001a49c, 0x1000c84a, 0x00000021, 0x1000c84f, 0x1000c851, 0xab734be4),
(0x1000c97f, 0x1001a438, 0x1000c96f, 0x0000000d, 0x1000c974, 0x1000c976, 0xde94b36a),
(0x1000c9fc, 0x1001a4c0, 0x1000c9ec, 0x0000000e, 0x1000c9f1, 0x1000c9f3, 0x73850301),
(0x1000ca5e, 0x1001a47c, 0x1000ca4e, 0x0000001f, 0x1000ca53, 0x1000ca55, 0x9122b4c2),
(0x1000cadb, 0x1001a45c, 0x1000cacb, 0x0000001c, 0x1000cad0, 0x1000cad2, 0xa9947e91),
(0x1000cc04, 0x1001a410, 0x1000cbf4, 0x0000000f, 0x1000cbf9, 0x1000cbfb, 0x6faae226),
(0x1000cdb2, 0x1001a520, 0x1000cda2, 0x0000000d, 0x1000cda7, 0x1000cda9, 0x68bfa957),
(0x1000cdff, 0x1001a530, 0x1000cdef, 0x00000018, 0x1000cdf4, 0x1000cdf6, 0x6cdadb36),
(0x1000ce51, 0x1001a560, 0x1000ce41, 0x00000013, 0x1000ce46, 0x1000ce48, 0x8108bff2),
(0x1000cea3, 0x1001a574, 0x1000ce93, 0x0000001a, 0x1000ce98, 0x1000ce9a, 0xdb5e6059),
(0x1000cefb, 0x1001a590, 0x1000ceeb, 0x00000013, 0x1000cef0, 0x1000cef2, 0x1a0fd12b),
(0x1000cf53, 0x1001a5a4, 0x1000cf43, 0x00000011, 0x1000cf48, 0x1000cf4a, 0xc2e44d9d),
(0x1000cfab, 0x1001a5b8, 0x1000cf9b, 0x0000000d, 0x1000cfa0, 0x1000cfa2, 0xe4f860f2),
(0x1000cffd, 0x1001a5c8, 0x1000cfed, 0x00000014, 0x1000cff2, 0x1000cff4, 0x40234227),
(0x1000d05b, 0x1001a5f4, 0x1000d04b, 0x00000016, 0x1000d050, 0x1000d052, 0x2c12ca96),
(0x1000d0b2, 0x1001a60c, 0x1000d0a2, 0x00000016, 0x1000d0a7, 0x1000d0a9, 0x7c827edf),
(0x1000d10f, 0x1001a678, 0x1000d0ff, 0x00000014, 0x1000d104, 0x1000d106, 0x98081276),
(0x1000d169, 0x1001a690, 0x1000d159, 0x0000000f, 0x1000d15e, 0x1000d160, 0xef5e3406),
(0x1000d2c8, 0x1001a4f0, 0x1000d2b8, 0x00000015, 0x1000d2bd, 0x1000d2bf, 0x0fd782e6),
(0x1000d317, 0x1001a508, 0x1000d307, 0x00000014, 0x1000d30c, 0x1000d30e, 0x79f43f38),
(0x1000d3c8, 0x1001a54c, 0x1000d3b8, 0x00000011, 0x1000d3bd, 0x1000d3bf, 0x5c553cbb),
(0x1000d458, 0x1001a624, 0x1000d448, 0x0000000d, 0x1000d44d, 0x1000d44f, 0x24977875),
(0x1000d4de, 0x1001a6a0, 0x1000d4be, 0x0000000a, 0x1000d4cc, 0x1000d4ce, 0xcf34c848),
(0x1000d4fa, 0x1001a6b0, 0x1000d4ea, 0x00000066, 0x1000d4ef, 0x1000d4f1, 0x26d664c4),
(0x1000d562, 0x1001a710, 0x1000d54b, 0x0000000c, 0x1000d550, 0x1000d552, 0xd21546fd),
(0x1000d582, 0x1001a720, 0x1000d573, 0x0000002c, 0x1000d574, 0x1000d579, 0x530c6098),
(0x1000d5d8, 0x1001a74c, 0x1000d5c8, 0x0000000c, 0x1000d5cd, 0x1000d5cf, 0x608e65f1),
(0x1000d5f3, 0x1001a720, 0x1000d5e4, 0x0000002c, 0x1000d5e5, 0x1000d5ea, 0x530c6098),
(0x1000d660, 0x1001a4d0, 0x1000d650, 0x0000000e, 0x1000d655, 0x1000d657, 0xc0fe1820),
(0x1000d73d, 0x1001a760, 0x1000d72d, 0x00000052, 0x1000d732, 0x1000d734, 0x4df2d32a),
(0x1000d7b3, 0x1001a4e0, 0x1000d7a3, 0x0000000d, 0x1000d7a8, 0x1000d7aa, 0x14745433),
(0x1000d7f2, 0x1001a7b8, 0x1000d7e2, 0x00000048, 0x1000d7e7, 0x1000d7e9, 0x21315fe1),
(0x1000d862, 0x1001a804, 0x1000d852, 0x00000024, 0x1000d857, 0x1000d859, 0xd2d2792b),
(0x1000d99a, 0x1001a924, 0x1000d98a, 0x0000001c, 0x1000d994, 0x1000d98f, 0x85df6959),
(0x1000d99a, 0x1001a908, 0x1000d961, 0x0000001c, 0x1000d994, 0x1000d966, 0x264a81a8),
(0x1000d99a, 0x1001a8c0, 0x1000d929, 0x00000048, 0x1000d933, 0x1000d92e, 0x2028da6f),
(0x1000d99a, 0x1001a830, 0x1000d8d3, 0x00000048, 0x1000d933, 0x1000d8d8, 0x79a9e33e),
(0x1000d99a, 0x1001a878, 0x1000d8fc, 0x00000048, 0x1000d933, 0x1000d901, 0x2649688e),
(0x1000d9ca, 0x1001a940, 0x1000d9ba, 0x00000006, 0x1000d9bf, 0x1000d9c1, 0x1a53c98f),
(0x1000daa8, 0x1001a948, 0x1000da8d, 0x0000000e, 0x1000da97, 0x1000da9f, 0x5d8e035b),
(0x1000db1d, 0x1001a958, 0x1000daf3, 0x00000006, 0x1000daf8, 0x1000db18, 0x4d5ebe0a),
(0x1000dba4, 0x1001a960, 0x1000db97, 0x00000006, 0x1000daf8, 0x1000db9c, 0x17162985),
(0x1000dba4, 0x1001a960, 0x1000db97, 0x00000006, 0x1000daf8, 0x1000db9c, 0x17162985),
(0x1000dbda, 0x1001a968, 0x1000dbcd, 0x00000006, 0x1000daf8, 0x1000dbd2, 0xfe07e13d),
(0x1000dbda, 0x1001a968, 0x1000dbcd, 0x00000006, 0x1000daf8, 0x1000dbd2, 0xfe07e13d),
(0x1000dc10, 0x1001a970, 0x1000dc03, 0x00000006, 0x1000daf8, 0x1000dc08, 0xf7598804),
(0x1000dc10, 0x1001a970, 0x1000dc03, 0x00000006, 0x1000daf8, 0x1000dc08, 0xf7598804),
(0x1000dc46, 0x1001a978, 0x1000dc39, 0x00000006, 0x1000daf8, 0x1000dc3e, 0x9a70df62),
(0x1000dc46, 0x1001a978, 0x1000dc39, 0x00000006, 0x1000daf8, 0x1000dc3e, 0x9a70df62),
(0x1000dc81, 0x1001a980, 0x1000dc71, 0x0000000a, 0x1000dc76, 0x1000dc78, 0x3254ce2b),
(0x1000dcf9, 0x1001a988, 0x1000dce9, 0x00000019, 0x1000dcee, 0x1000dcf0, 0x7bac2cc4),
(0x1000dec1, 0x1001a9a4, 0x1000deb1, 0x0000000c, 0x1000deb6, 0x1000deb8, 0xc1343f89),
(0x1000e113, 0x1001a9b0, 0x1000e103, 0x0000000a, 0x1000e108, 0x1000e10a, 0x3260f40d),
(0x1000e199, 0x1001a9bc, 0x1000e189, 0x0000000a, 0x1000e18e, 0x1000e190, 0xba2e14dd),
(0x1000e263, 0x1001a9c8, 0x1000e253, 0x0000000e, 0x1000e258, 0x1000e25a, 0xf6927da0),
(0x1000e2ad, 0x1001a9d4, 0x1000e29d, 0x0000001e, 0x1000e2a2, 0x1000e2a4, 0xf93b5dec),
(0x1000e2f4, 0x1001a9f0, 0x1000e2e4, 0x00000006, 0x1000e2e9, 0x1000e2eb, 0x6231f34c),
(0x1000e367, 0x1001a5e0, 0x1000e357, 0x00000012, 0x1000e35c, 0x1000e35e, 0xd29186dd),
(0x1000e4fe, 0x1001a634, 0x1000e4ee, 0x00000017, 0x1000e4f3, 0x1000e4f5, 0xede69570),
(0x1000e5bd, 0x1001a64c, 0x1000e5ad, 0x0000000e, 0x1000e5b2, 0x1000e5b4, 0x2fe6b993),
(0x1000e656, 0x1001a65c, 0x1000e646, 0x00000018, 0x1000e64b, 0x1000e64d, 0x832bc2a1),
(0x1000e6dd, 0x1001a9f8, 0x1000e6cd, 0x00000036, 0x1000e6d2, 0x1000e6d4, 0x5ff3fa39),
(0x1000e728, 0x1001aa2c, 0x1000e718, 0x00000042, 0x1000e71d, 0x1000e71f, 0x8f929cf1),
(0x1000e898, 0x1001aa6c, 0x1000e888, 0x0000000f, 0x1000e88d, 0x1000e88f, 0x30599991),
(0x1000e8fe, 0x1001aa9c, 0x1000e8ee, 0x0000000c, 0x1000e8f3, 0x1000e8f5, 0x3beb44bd),
(0x1000e954, 0x1001aaac, 0x1000e944, 0x00000011, 0x1000e949, 0x1000e94b, 0x2226d162),
(0x1000ec0f, 0x1001aad8, 0x1000ebff, 0x00000068, 0x1000ec04, 0x1000ec06, 0x5a8ed21a),
(0x1000ec3c, 0x1001ab40, 0x1000ec2c, 0x00000048, 0x1000ec31, 0x1000ec33, 0x2cc9a6cf),
(0x1000ec69, 0x1001ab88, 0x1000ec59, 0x00000006, 0x1000ec5e, 0x1000ec60, 0x1969ff66),
(0x1000eca5, 0x1001ab90, 0x1000ec92, 0x0000001a, 0x1000ec97, 0x1000ec99, 0x1665fd77),
(0x1000ecd2, 0x1001abac, 0x1000ecc5, 0x0000001a, 0x1000ec97, 0x1000ecca, 0x966e858d),
(0x1000ed0b, 0x1001acd0, 0x1000ecf8, 0x00000072, 0x1000ecfd, 0x1000ed02, 0x1134a299),
(0x1000edc3, 0x1001ad3c, 0x1000edaf, 0x00000002, 0x1000edb4, 0x1000edba, 0x86c19794),
(0x1000edf7, 0x1001acd0, 0x1000ede7, 0x00000072, 0x1000edec, 0x1000edee, 0x1134a299),
(0x1000ee8e, 0x1001abc8, 0x1000ee7e, 0x00000016, 0x1000ee83, 0x1000ee85, 0x24687887),
(0x1000eee3, 0x1001abe0, 0x1000eed4, 0x00000010, 0x1000eed5, 0x1000eed7, 0x352ece17),
(0x1000eef7, 0x1001abe0, 0x1000eeee, 0x00000010, 0x1000eed5, 0x1000eeef, 0x352ece17),
(0x1000ef13, 0x1001abf4, 0x1000ef03, 0x00000018, 0x1000ef08, 0x1000ef0a, 0x1cd79a1c),
(0x1000efc2, 0x1001ac10, 0x1000efb2, 0x0000001a, 0x1000efb7, 0x1000efb9, 0xd0981a4e),
(0x1000effd, 0x1001acd0, 0x1000efea, 0x00000072, 0x1000efef, 0x1000eff4, 0x1134a299),
(0x1000f0be, 0x1001ad40, 0x1000f0ae, 0x00000002, 0x1000f0b3, 0x1000f0b5, 0x8efd66ef),
(0x1000f0f2, 0x1001acd0, 0x1000f0e2, 0x00000072, 0x1000f0e7, 0x1000f0e9, 0x1134a299),
(0x1000f188, 0x1001ac2c, 0x1000f177, 0x0000000a, 0x1000f17c, 0x1000f17e, 0x28600b1d),
(0x1000f1cc, 0x1001ac38, 0x1000f1b6, 0x00000006, 0x1000f1bb, 0x1000f1be, 0xba336bec),
(0x1000f238, 0x1001ac40, 0x1000f227, 0x0000001c, 0x1000f22c, 0x1000f22e, 0x11f374db),
(0x1000f272, 0x1001ac60, 0x1000f25f, 0x00000006, 0x1000f1bb, 0x1000f264, 0x769fbc85),
(0x1000f272, 0x1001ac60, 0x1000f25f, 0x00000006, 0x1000f1bb, 0x1000f264, 0x769fbc85),
(0x1000f2c0, 0x1001ac68, 0x1000f2af, 0x00000008, 0x1000f2b4, 0x1000f2b6, 0x2852ce6a),
(0x1000f2f8, 0x1001abe0, 0x1000f2e4, 0x00000010, 0x1000f2e9, 0x1000f2eb, 0x352ece17),
(0x1000f339, 0x1001ac70, 0x1000f324, 0x00000004, 0x1000f329, 0x1000f334, 0xddbb20d5),
(0x1000f36e, 0x1001919c, 0x1000f360, 0x00000004, 0x1000f329, 0x1000f365, 0x028f2563),
(0x1000f3a0, 0x1001ac78, 0x1000f392, 0x00000004, 0x1000f329, 0x1000f397, 0xf1e0f796),
(0x1000f3d7, 0x1001ac80, 0x1000f3c7, 0x00000006, 0x1000f1bb, 0x1000f3cc, 0x316cfccc),
(0x1000f3d7, 0x1001ac80, 0x1000f3c7, 0x00000006, 0x1000f1bb, 0x1000f3cc, 0x316cfccc),
(0x1000f418, 0x1001ac88, 0x1000f40a, 0x00000006, 0x1000f1bb, 0x1000f40f, 0xf71edbd4),
(0x1000f418, 0x1001ac88, 0x1000f40a, 0x00000006, 0x1000f1bb, 0x1000f40f, 0xf71edbd4),
(0x1000f541, 0x1001aa8c, 0x1000f531, 0x0000000c, 0x1000f536, 0x1000f538, 0xcfba766d),
(0x1000f5d7, 0x1001aac0, 0x1000f5c7, 0x00000013, 0x1000f5cc, 0x1000f5ce, 0xf589b5f5),
(0x1000f623, 0x1001aa7c, 0x1000f613, 0x0000000c, 0x1000f618, 0x1000f61a, 0xf336f52b),
(0x1000f8a1, 0x1001ac90, 0x1000f891, 0x00000015, 0x1000f896, 0x1000f898, 0x93cf9cec),
(0x1000f8fd, 0x1001aca8, 0x1000f8ed, 0x00000016, 0x1000f8f2, 0x1000f8f4, 0xc2e0cfe9),
(0x1000f956, 0x1001acc0, 0x1000f946, 0x0000000f, 0x1000f94b, 0x1000f94d, 0xaf494204),
(0x1000feed, 0x1001ad44, 0x1000fedd, 0x0000000f, 0x1000fee2, 0x1000fee4, 0xcc5f07b2),
(0x1000ff4d, 0x1001ad54, 0x1000ff3d, 0x0000000e, 0x1000ff42, 0x1000ff44, 0x7765e61a),
(0x1000ffa9, 0x1001ad64, 0x1000ff99, 0x00000013, 0x1000ff9e, 0x1000ffa0, 0xbd21202f),
(0x1001000b, 0x1001ad8c, 0x1000fffb, 0x00000019, 0x10010000, 0x10010002, 0xe2cbc149),
(0x10010060, 0x1001adb8, 0x10010050, 0x00000012, 0x10010055, 0x10010057, 0xfc7af4f8),
(0x100100b5, 0x1001ade8, 0x100100a5, 0x00000009, 0x100100aa, 0x100100ac, 0x4ad814eb),
(0x1001010a, 0x1001adf4, 0x100100fa, 0x00000008, 0x100100ff, 0x10010101, 0xd0507731),
(0x1001015c, 0x1001ae00, 0x1001014c, 0x00000010, 0x10010151, 0x10010153, 0x184bacff),
(0x100101ae, 0x1001ae14, 0x1001019e, 0x0000001c, 0x100101a3, 0x100101a5, 0x9ad7365d),
(0x1001020a, 0x1001ae44, 0x100101fa, 0x00000011, 0x100101ff, 0x10010201, 0xc599ddfb),
(0x100103ea, 0x1001adcc, 0x100103da, 0x00000019, 0x100103df, 0x100103e1, 0x7285d558),
(0x10010588, 0x1001ae68, 0x10010578, 0x0000000a, 0x1001057d, 0x1001057f, 0xa452da9b),
(0x100105bd, 0x1001ae74, 0x100105ad, 0x00000006, 0x100105b2, 0x100105b4, 0x6677cc2d),
(0x100105f5, 0x1001ae7c, 0x100105e8, 0x00000006, 0x100105b2, 0x100105ed, 0x940432e1),
(0x1001062e, 0x1001ae84, 0x1001061e, 0x00000008, 0x10010623, 0x10010625, 0xcbddfb45),
(0x1001065b, 0x1001abe0, 0x1001064b, 0x00000010, 0x10010650, 0x10010652, 0x352ece17),
(0x10010688, 0x1001ae8c, 0x10010678, 0x00000004, 0x1001067d, 0x1001067f, 0x3b105e62),
(0x100106b2, 0x1001919c, 0x100106a5, 0x00000004, 0x1001067d, 0x100106aa, 0x028f2563),
(0x10010784, 0x1001aebc, 0x10010774, 0x00000014, 0x10010779, 0x1001077b, 0x9cd5a9b2),
(0x100107b1, 0x1001aed0, 0x100107a1, 0x0000001c, 0x100107a6, 0x100107a8, 0x3aa39e12),
(0x10010894, 0x1001aef8, 0x10010884, 0x0000000d, 0x10010889, 0x1001088b, 0x17bbd056),
(0x100108ad, 0x1001af08, 0x100108a0, 0x0000000d, 0x10010889, 0x100108a5, 0xec3b15ca),
(0x100108f1, 0x1001af18, 0x100108e1, 0x0000000c, 0x100108e6, 0x100108e8, 0x0d2f6eee),
(0x1001090d, 0x1001af28, 0x100108fd, 0x0000000d, 0x10010902, 0x10010904, 0xb1bd7dd7),
(0x10010aa4, 0x1001af38, 0x10010a93, 0x00000006, 0x10010a98, 0x10010a9a, 0x3eb89c85),
(0x10010ad7, 0x1001af40, 0x10010ac3, 0x00000010, 0x10010ac8, 0x10010aca, 0xff57bfe1),
(0x10010b12, 0x1001ae34, 0x10010b01, 0x0000000f, 0x10010b06, 0x10010b08, 0x24adb4e1),
(0x10010bb7, 0x1001af54, 0x10010ba3, 0x0000001c, 0x10010ba8, 0x10010baa, 0x79a95eee),
(0x10010c1f, 0x1001a9f0, 0x10010c0e, 0x00000006, 0x10010c13, 0x10010c15, 0x665db54c),
(0x10010c52, 0x1001af74, 0x10010c3e, 0x00000010, 0x10010c43, 0x10010c45, 0x0c3def31),
(0x10010c85, 0x1001af88, 0x10010c74, 0x00000006, 0x10010c79, 0x10010c7b, 0x4032c7d6),
(0x10010cb8, 0x1001af90, 0x10010ca4, 0x0000001c, 0x10010ca9, 0x10010cab, 0x654ce9de),
(0x10010d04, 0x1001afb0, 0x10010cf3, 0x00000006, 0x10010cf8, 0x10010cfa, 0x6ccf894b),
(0x10010d34, 0x1001afb8, 0x10010d23, 0x00000010, 0x10010d2d, 0x10010d28, 0xd63dce2b),
(0x10010d64, 0x1001afcc, 0x10010d53, 0x00000006, 0x10010d58, 0x10010d5a, 0xaf6bcf92),
(0x10010d94, 0x1001afd4, 0x10010d83, 0x00000018, 0x10010d88, 0x10010d8a, 0x9494110b),
(0x10010ea6, 0x1001ae58, 0x10010e92, 0x0000000f, 0x10010e97, 0x10010e99, 0x1daea3ce),
(0x10010f28, 0x1001aff0, 0x10010f18, 0x0000001c, 0x10010f1d, 0x10010f1f, 0xb072b54f),
(0x10010f99, 0x1001b010, 0x10010f89, 0x0000000a, 0x10010f8e, 0x10010f90, 0x7867ea3e),
(0x10010ff1, 0x1001b01c, 0x10010fdb, 0x0000001c, 0x10010fe6, 0x10010fe8, 0x66937974),
(0x10011028, 0x1001b03c, 0x10011012, 0x00000006, 0x1001101d, 0x1001101f, 0x4d5fab10),
(0x1001106d, 0x1001afcc, 0x10011060, 0x00000006, 0x1001101d, 0x10011065, 0x8d3d0c92),
(0x100110ae, 0x1001b044, 0x1001109e, 0x00000008, 0x100110a3, 0x100110a5, 0x14a41869),
(0x100110e5, 0x1001abe0, 0x100110d2, 0x00000010, 0x100110d7, 0x100110dc, 0x352ece17),
(0x1001111f, 0x1001b04c, 0x1001110c, 0x00000004, 0x10011111, 0x10011116, 0x60174dc2),
(0x10011149, 0x1001919c, 0x1001113c, 0x00000004, 0x10011111, 0x10011141, 0x028f2563),
(0x10011234, 0x1001ad78, 0x10011224, 0x00000013, 0x10011229, 0x1001122b, 0xdcc1e553),
(0x100112c1, 0x1001b054, 0x100112b1, 0x0000000e, 0x100112b6, 0x100112b8, 0x7a25d689),
(0x100112e6, 0x1001ada8, 0x100112d6, 0x0000000c, 0x100112db, 0x100112dd, 0xb820a3e7),
(0x1001135b, 0x1001b084, 0x1001134b, 0x00000011, 0x10011350, 0x10011352, 0xbcdd0abb),
(0x100113a8, 0x1001b098, 0x10011398, 0x00000012, 0x1001139d, 0x1001139f, 0x190f13a5),
(0x100114f9, 0x1001b0c0, 0x100114e9, 0x00000011, 0x100114ee, 0x100114f0, 0xd34484e4),
(0x1001154b, 0x1001b0ec, 0x1001153b, 0x0000000e, 0x10011540, 0x10011542, 0xe75bbdc1),
(0x1001159d, 0x1001b0fc, 0x1001158d, 0x00000009, 0x10011592, 0x10011594, 0xc6c99979),
(0x100115f2, 0x1001b108, 0x100115e2, 0x0000000d, 0x100115e7, 0x100115e9, 0xd63535a6),
(0x1001164b, 0x1001b118, 0x1001163b, 0x0000000d, 0x10011640, 0x10011642, 0x0949b7f5),
(0x10011baa, 0x1001b128, 0x10011b99, 0x0000001b, 0x10011b9e, 0x10011ba0, 0xa692912a),
(0x10011e23, 0x1001b144, 0x10011e13, 0x00000040, 0x10011e18, 0x10011e1a, 0xdd1d6fbe),
(0x10011fc1, 0x1001b0ac, 0x10011fb1, 0x00000013, 0x10011fb6, 0x10011fb8, 0x81b41684),
(0x100120a1, 0x1001b0d4, 0x10012091, 0x00000015, 0x10012096, 0x10012098, 0xc610c76d),
(0x1001211e, 0x1001b180, 0x1001210a, 0x00000006, 0x1001210f, 0x10012115, 0x1ae144b2),
(0x10012158, 0x1001b188, 0x10012147, 0x00000006, 0x1001210f, 0x1001214c, 0xdf1afe23),
(0x1001219a, 0x1001b190, 0x10012186, 0x00000002, 0x1001218b, 0x1001218d, 0xea78ab10),
(0x100121b9, 0x1001b194, 0x100121a6, 0x00000006, 0x100121ab, 0x100121b0, 0x9ded798f),
(0x100121fc, 0x1001b19c, 0x100121eb, 0x00000006, 0x100121ab, 0x100121f0, 0xc9821991),
(0x10012237, 0x1001b1a4, 0x10012226, 0x00000006, 0x100121ab, 0x1001222b, 0xcdecc76e),
(0x10012275, 0x1001b1ac, 0x10012261, 0x00000002, 0x10012266, 0x10012268, 0x5935e92a),
(0x10012294, 0x1001b1b0, 0x10012281, 0x00000006, 0x10012286, 0x1001228b, 0x18c0f3a6),
(0x100124c4, 0x1001b1b8, 0x100124b4, 0x0000000a, 0x100124b9, 0x100124bb, 0x6b0939d3),
(0x100124f1, 0x1001b1c4, 0x100124e1, 0x0000001c, 0x100124e6, 0x100124e8, 0x3faa1064),
(0x10012521, 0x1001b1e4, 0x10012511, 0x00000006, 0x10012516, 0x10012518, 0x1ee9c4b9),
(0x10012553, 0x1001b1ec, 0x10012546, 0x00000006, 0x10012516, 0x1001254b, 0x43a89533),
(0x10012586, 0x1001b1f4, 0x10012576, 0x00000008, 0x10012580, 0x1001257b, 0x76e9da59),
(0x100125b0, 0x1001abe0, 0x100125a0, 0x00000010, 0x100125a5, 0x100125a7, 0x352ece17),
(0x100125da, 0x1001b1fc, 0x100125ca, 0x00000004, 0x100125cf, 0x100125d1, 0xff9d2e3b),
(0x10012601, 0x100191a4, 0x100125f4, 0x00000004, 0x100125cf, 0x100125f9, 0x8b02b2f1),
(0x100127f4, 0x1001b204, 0x100127e4, 0x00000016, 0x100127e9, 0x100127eb, 0x07a15f0d),
(0x10012846, 0x1001b23c, 0x10012836, 0x00000014, 0x1001283b, 0x1001283d, 0x73c2f80c),
(0x100128a8, 0x1001b254, 0x10012898, 0x00000012, 0x1001289d, 0x1001289f, 0xd41a7f44),
(0x1001292f, 0x1001b21c, 0x1001291f, 0x0000001f, 0x10012924, 0x10012926, 0x01cfc539),
(0x10012c6d, 0x1001b2a4, 0x10012c5d, 0x00000015, 0x10012c62, 0x10012c64, 0xead1db68),
(0x10012cc8, 0x1001b2bc, 0x10012cb8, 0x00000017, 0x10012cbd, 0x10012cbf, 0x979e34ad),
(0x10012d29, 0x1001b294, 0x10012d19, 0x0000000f, 0x10012d1e, 0x10012d20, 0x223033eb),
(0x10012efe, 0x1001b2d4, 0x10012eee, 0x00000011, 0x10012ef3, 0x10012ef5, 0x62230fcc),
(0x10012f44, 0x1001b2e8, 0x10012f37, 0x00000011, 0x10012ef3, 0x10012f3c, 0x49d77e33),
(0x10012f85, 0x1001b2fc, 0x10012f75, 0x00000013, 0x10012f7a, 0x10012f7c, 0x789d4e6d),
(0x10012fc0, 0x1001b310, 0x10012fb3, 0x00000013, 0x10012f7a, 0x10012fb8, 0xb54aa65a),
(0x10012fc0, 0x1001b310, 0x10012fb3, 0x00000013, 0x10012f7a, 0x10012fb8, 0xb54aa65a),
(0x10013142, 0x1001b268, 0x10013132, 0x00000012, 0x10013137, 0x10013139, 0x8dae88e6),
(0x100131b3, 0x1001b27c, 0x100131a3, 0x00000014, 0x100131a8, 0x100131aa, 0xe70f47d1),
(0x10013283, 0x1001b324, 0x10013273, 0x00000002, 0x10013278, 0x1001327a, 0x54d8b697),
(0x100132df, 0x1001b328, 0x100132cf, 0x0000000d, 0x100132d4, 0x100132d6, 0x3a6d0862),
(0x1001334c, 0x1001b338, 0x1001333c, 0x00000010, 0x10013341, 0x10013343, 0x29208d75),
(0x100133a3, 0x1001b374, 0x10013393, 0x0000001c, 0x10013398, 0x1001339a, 0x0e4989bc),
(0x100133f5, 0x1001b34c, 0x100133e5, 0x00000009, 0x100133ea, 0x100133ec, 0x43d8bc08),
(0x100134aa, 0x1001b368, 0x1001349a, 0x00000008, 0x1001349f, 0x100134a1, 0xa0ee8019),
(0x10013542, 0x1001b358, 0x10013531, 0x0000000c, 0x10013536, 0x10013538, 0x8ceac8ae),
(0x100135a4, 0x1001b394, 0x10013593, 0x00000016, 0x10013598, 0x1001359a, 0x2b661872),
(0x100135ed, 0x1001b3ac, 0x100135d8, 0x00000018, 0x100135dd, 0x100135e3, 0xc46e16e1),
(0x1001362c, 0x1001b3c8, 0x1001361e, 0x00000018, 0x100135dd, 0x10013623, 0xbeed92b9),
(0x1001366a, 0x1001b3e4, 0x10013659, 0x00000012, 0x1001365e, 0x10013660, 0xa5e4d73e),
(0x100136a5, 0x1001b3f8, 0x10013694, 0x00000042, 0x10013699, 0x1001369b, 0xa68a6425),
(0x1001370a, 0x1001b43c, 0x100136fa, 0x00000014, 0x100136ff, 0x10013701, 0xafb1f246),
(0x10013757, 0x1001b484, 0x10013747, 0x00000015, 0x1001374c, 0x1001374e, 0xda81edf3),
(0x100137ac, 0x1001b49c, 0x1001379c, 0x00000011, 0x100137a1, 0x100137a3, 0x00a00e60),
(0x1001380e, 0x1001b4b0, 0x100137fe, 0x00000015, 0x10013803, 0x10013805, 0xde7d88ad),
(0x10013877, 0x1001b454, 0x10013867, 0x00000015, 0x1001386c, 0x1001386e, 0x43c4dfec),
(0x100138cb, 0x1001b46c, 0x100138bb, 0x00000015, 0x100138c0, 0x100138c2, 0xb39ff5ee),
(0x10013f5b, 0x1001b4c8, 0x10013f4b, 0x0000001d, 0x10013f50, 0x10013f52, 0x6f7a2df9),
(0x1001400a, 0x1001b4e8, 0x10013ffa, 0x0000000a, 0x10013fff, 0x10014001, 0x1828a206),
(0x1001403f, 0x1001b4f4, 0x1001402f, 0x0000001c, 0x10014034, 0x10014036, 0xe3366bf2),
(0x1001406c, 0x1001b514, 0x1001405c, 0x00000006, 0x10014061, 0x10014063, 0xe9124061),
(0x100140a4, 0x1001b51c, 0x10014097, 0x00000006, 0x10014061, 0x1001409c, 0xd0324d65),
(0x100140dd, 0x1001b524, 0x100140cd, 0x00000008, 0x100140d2, 0x100140d4, 0xcd16344d),
(0x1001410a, 0x1001abe0, 0x100140fa, 0x00000010, 0x100140ff, 0x10014101, 0x352ece17),
(0x10014137, 0x1001b52c, 0x10014127, 0x00000004, 0x1001412c, 0x1001412e, 0xa24989a8),
(0x10014161, 0x100191ac, 0x10014154, 0x00000004, 0x1001412c, 0x10014159, 0x4f1fa5bc),
(0x100141f0, 0x1001b534, 0x100141e0, 0x0000000c, 0x100141e5, 0x100141e7, 0xa066ff1d),
(0x1001421c, 0x1001b544, 0x1001420c, 0x00000017, 0x10014211, 0x10014213, 0xa61c6433),
(0x1001424a, 0x1001b55c, 0x1001423a, 0x00000018, 0x1001423f, 0x10014241, 0x202fb71f),
(0x100143e9, 0x1001b144, 0x100143d9, 0x00000040, 0x100143de, 0x100143e0, 0xdd1d6fbe),
(0x10014512, 0x1001b578, 0x10014502, 0x0000000c, 0x10014507, 0x10014509, 0x195b2e5e),
(0x1001456d, 0x1001b588, 0x1001455d, 0x00000009, 0x10014562, 0x10014564, 0x0f2944a5),
(0x100145c2, 0x1001b594, 0x100145b2, 0x0000000a, 0x100145b7, 0x100145b9, 0x3f503d67),
(0x10014619, 0x1001b5a0, 0x10014609, 0x00000011, 0x1001460e, 0x10014610, 0x2c58a9b1),
(0x10014678, 0x1001b5b4, 0x10014668, 0x0000000f, 0x1001466d, 0x1001466f, 0x7a0cfb4a),
(0x10014707, 0x1001b5d4, 0x100146f2, 0x00000009, 0x100146f7, 0x100146fe, 0x09858dd9),
(0x100147e7, 0x1001b5e0, 0x100147c1, 0x0000000a, 0x100147dc, 0x100147de, 0xeb2932dd),
(0x1001481d, 0x1001b5ec, 0x10014806, 0x00000004, 0x10014812, 0x10014814, 0xdc56e95e),
(0x1001485c, 0x1001b5f4, 0x10014842, 0x00000009, 0x10014851, 0x10014853, 0x8288eb54),
(0x100148a3, 0x1001b600, 0x10014888, 0x0000000c, 0x10014892, 0x1001489a, 0x1187dac7),
(0x100148d0, 0x1001b610, 0x100148c1, 0x00000001, 0x10014739, 0x100148c6, 0x4b7f90ec),
(0x100148fb, 0x1001b614, 0x100148ee, 0x00000001, 0x10014739, 0x100148f3, 0x811973fa),
(0x1001494b, 0x1001b618, 0x1001493b, 0x00000010, 0x10014940, 0x10014942, 0x719dcb84),
(0x10014ab8, 0x1001b62c, 0x10014aa8, 0x0000003a, 0x10014aad, 0x10014aaf, 0x5a75dabb),
(0x10014b25, 0x1001b5c4, 0x10014b15, 0x0000000e, 0x10014b1a, 0x10014b1c, 0x1378a371),
(0x10014c4b, 0x1001b664, 0x10014c32, 0x00000006, 0x10014c3c, 0x10014c42, 0xbd57a780),
(0x10014c88, 0x1001b66c, 0x10014c78, 0x00000002, 0x10014c7d, 0x10014c7f, 0x79d06326),
(0x10014ca7, 0x1001b670, 0x10014c94, 0x00000012, 0x10014c99, 0x10014c9e, 0xb4b6f93e),
(0x10014d09, 0x1001b62c, 0x10014cf4, 0x0000003a, 0x10014cf9, 0x10014cfd, 0x5a75dabb),
(0x10014e54, 0x1001b684, 0x10014e44, 0x00000007, 0x10014e49, 0x10014e4b, 0xc4fb3677),
(0x10014eb1, 0x1001b68c, 0x10014ea1, 0x00000005, 0x10014ea6, 0x10014ea8, 0x1de72ebe),
(0x10014f07, 0x1001b69c, 0x10014ef7, 0x0000000b, 0x10014efc, 0x10014efe, 0xf2b86db7),
(0x10014f64, 0x1001b6a8, 0x10014f54, 0x00000009, 0x10014f59, 0x10014f5b, 0x33d89a62),
(0x10014fb8, 0x1001b6b4, 0x10014fa8, 0x0000000c, 0x10014fad, 0x10014faf, 0x84bd2758),
(0x10015014, 0x1001b6c4, 0x10015004, 0x0000000c, 0x10015009, 0x1001500b, 0x4842e1b0),
(0x1001506d, 0x1001b6dc, 0x1001505d, 0x00000006, 0x10015062, 0x10015064, 0x8ee6719f),
(0x100150bf, 0x1001b6f0, 0x100150af, 0x00000010, 0x100150b4, 0x100150b6, 0x6631cbdb),
(0x1001529f, 0x1001b694, 0x1001528e, 0x00000005, 0x10015293, 0x10015295, 0x2b2b5b15),
(0x1001539c, 0x1001b704, 0x1001538c, 0x00000002, 0x10015391, 0x10015393, 0xc39b4e8d),
(0x100153b6, 0x1001b708, 0x100153a8, 0x00000001, 0x100153b2, 0x100153ad, 0x8891178d),
(0x10015412, 0x1001b704, 0x10015402, 0x00000002, 0x10015407, 0x10015409, 0xaf4c608d),
(0x10015460, 0x1001b70c, 0x10015450, 0x00000002, 0x10015455, 0x10015457, 0x2869a395),
(0x100154fc, 0x1001b710, 0x100154ee, 0x00000001, 0x100154f8, 0x100154f3, 0x565e71c3),
(0x10015630, 0x1001b328, 0x10015618, 0x0000000d, 0x1001561d, 0x10015627, 0x3a6d0862),
(0x10015676, 0x1001b714, 0x10015666, 0x00000005, 0x1001566b, 0x1001566d, 0x65583849),
(0x100156aa, 0x1001b71c, 0x1001569a, 0x00000006, 0x1001569f, 0x100156a1, 0xd9d1f9ca),
(0x100156de, 0x1001b724, 0x100156ce, 0x00000009, 0x100156d3, 0x100156d5, 0xc650dea5),
(0x10015744, 0x1001b730, 0x10015734, 0x00000008, 0x10015739, 0x1001573b, 0xcb66dad9),
(0x10015799, 0x1001b328, 0x10015789, 0x0000000d, 0x1001578e, 0x10015790, 0x3a6d0862),
(0x10015875, 0x1001b6d4, 0x10015865, 0x00000006, 0x1001586a, 0x1001586c, 0x09940ebc),
(0x100158c3, 0x1001b6e4, 0x100158b3, 0x0000000a, 0x100158b8, 0x100158ba, 0xd910caac),
(0x10015cfd, 0x1001b73c, 0x10015ced, 0x0000000b, 0x10015cf2, 0x10015cf4, 0x94d537ec),
(0x10015d4f, 0x1001b748, 0x10015d3f, 0x00000018, 0x10015d44, 0x10015d46, 0xa5599d03),
(0x10015da3, 0x1001b784, 0x10015d93, 0x00000011, 0x10015d98, 0x10015d9a, 0x57d5c7e1),
(0x10015e02, 0x1001b7f8, 0x10015df2, 0x00000011, 0x10015df7, 0x10015df9, 0x9200df92),
(0x10015e5f, 0x1001b80c, 0x10015e4f, 0x00000013, 0x10015e54, 0x10015e56, 0xfb25d92c),
(0x10015eba, 0x1001b820, 0x10015eaa, 0x00000014, 0x10015eaf, 0x10015eb1, 0x8d117451),
(0x100164db, 0x1001b764, 0x100164ca, 0x0000000d, 0x100164cf, 0x100164d1, 0xe137088b),
(0x10016561, 0x1001b838, 0x10016551, 0x00000006, 0x10016556, 0x10016558, 0x17b60739),
(0x1001660f, 0x1001b840, 0x100165ff, 0x00000037, 0x10016604, 0x10016606, 0x67f4dd82),
(0x10016634, 0x1001b774, 0x10016624, 0x0000000e, 0x10016629, 0x1001662b, 0x0882171c),
(0x10016727, 0x1001b87c, 0x1001670b, 0x00000011, 0x10016710, 0x10016714, 0x06f3937f),
(0x1001676a, 0x1001b890, 0x10016757, 0x00000005, 0x1001675c, 0x1001675e, 0xcda2e17f),
(0x1001678f, 0x1001b7ac, 0x1001677f, 0x00000011, 0x10016784, 0x10016786, 0xc32a2246),
(0x1001683e, 0x1001b898, 0x1001682b, 0x0000000a, 0x10016830, 0x10016835, 0x0990cc69),
(0x1001685a, 0x1001b8a4, 0x1001684a, 0x00000007, 0x1001684f, 0x10016851, 0x231c9480),
(0x100168ad, 0x1001b8ac, 0x1001689a, 0x00000009, 0x1001689f, 0x100168a1, 0xa02c7ae0),
(0x100168cc, 0x1001b8b8, 0x100168b9, 0x00000007, 0x100168be, 0x100168c3, 0xfcae6629),
(0x10016919, 0x1001b8c0, 0x10016909, 0x00000007, 0x100168be, 0x1001690e, 0x5249c9d9),
(0x10016938, 0x1001b8c8, 0x10016928, 0x00000007, 0x100168be, 0x1001692d, 0xdde7b4fa),
(0x10016992, 0x1001b8d0, 0x10016982, 0x00000005, 0x10016987, 0x10016989, 0x1294dc33),
(0x100169b4, 0x1001b8d8, 0x1001699e, 0x00000007, 0x100169a3, 0x100169a8, 0xd8aae045),
(0x10016a6e, 0x1001b7c0, 0x10016a5e, 0x00000013, 0x10016a63, 0x10016a65, 0xd1acc255),
(0x10016ad2, 0x1001b798, 0x10016ac2, 0x00000012, 0x10016ac7, 0x10016ac9, 0x31187449),
(0x10016b2d, 0x1001b7d4, 0x10016b1d, 0x00000010, 0x10016b22, 0x10016b24, 0x11340aa2),
(0x10016b8a, 0x1001b8e0, 0x10016b7c, 0x00000001, 0x10016b86, 0x10016b81, 0x8d433f4e),
(0x10016c42, 0x1001b8e4, 0x10016c31, 0x00000001, 0x10016c3b, 0x10016c36, 0x2a80ecb6),
(0x10016c87, 0x1001b898, 0x10016c74, 0x0000000a, 0x10016c79, 0x10016c7b, 0x0990cc69),
(0x10016cd1, 0x1001b8ac, 0x10016cbe, 0x00000009, 0x10016cc3, 0x10016cc5, 0xa02c7ae0),
(0x10016d16, 0x1001b8c0, 0x10016d03, 0x00000007, 0x10016d08, 0x10016d0a, 0x5249c9d9),
(0x10016d5b, 0x1001b8d0, 0x10016d48, 0x00000005, 0x10016d4d, 0x10016d4f, 0x1294dc33),
(0x1001707b, 0x1001b8e8, 0x1001705d, 0x00000003, 0x10017062, 0x1001706b, 0x29dbf286),
(0x100170b2, 0x1001b7e8, 0x100170a2, 0x0000000f, 0x100170a7, 0x100170a9, 0x8fbd5af3),
(0x100174d0, 0x1001b8ec, 0x100174c0, 0x00000008, 0x100174c5, 0x100174c7, 0x47f9e687),
(0x10017527, 0x1001b908, 0x10017517, 0x0000000c, 0x1001751c, 0x1001751e, 0xd8364894),
(0x1001757f, 0x1001b924, 0x1001756f, 0x00000008, 0x10017574, 0x10017576, 0xb126b519),
(0x100175e3, 0x1001b930, 0x100175d3, 0x00000009, 0x100175d8, 0x100175da, 0xaf864a45),
(0x10017647, 0x1001b93c, 0x10017637, 0x00000017, 0x1001763c, 0x1001763e, 0xb01bc7d3),
(0x100178a5, 0x1001b954, 0x10017895, 0x0000000b, 0x1001789a, 0x1001789c, 0x087e91f1),
(0x1001790b, 0x1001b960, 0x100178fb, 0x0000000b, 0x10017900, 0x10017902, 0xe81e3cac),
(0x10017e25, 0x1001b8f8, 0x10017e15, 0x0000000c, 0x10017e1a, 0x10017e1c, 0xd4c65f35),
(0x10017ea5, 0x1001b918, 0x10017e95, 0x00000008, 0x10017e9a, 0x10017e9c, 0x770ac4db),
(0x10017fac, 0x1001b96c, 0x10017f9c, 0x00000009, 0x10017fa1, 0x10017fa3, 0x487a9e95),
(0x10018058, 0x1001b978, 0x10018048, 0x0000000c, 0x1001804d, 0x1001804f, 0xf3c9422e),
]
        
getArgs()
#processStuff(stuff1)
logger.info('Done')




