
package mage.abilities.keyword;

import mage.abilities.Ability;
import mage.abilities.common.EntersBattlefieldTriggeredAbility;
import mage.abilities.effects.OneShotEffect;
import mage.constants.Outcome;
import mage.filter.StaticFilters;
import mage.game.Game;
import mage.game.events.GameEvent;
import mage.game.permanent.Permanent;
import mage.players.Player;
import mage.target.Target;
import mage.target.common.TargetSacrifice;

/**
 * Exploit is the signature ability of the blue-black Silumgar clan. When a creature with exploit
 * enters the battlefield, you may sacrifice a creature you control.
 * <p>
 * But you're not just sacrificing your loyal minions for fun. Each creature with exploit has
 * another ability that gives you a benefit when it "exploits a creature." This means when you
 * sacrifice a creature because of its exploit ability. That ability doesn't trigger if you
 * sacrifice a creature for any other reason, including the exploit ability of a different creature.
 * <p>
 * You can sacrifice any creature you control when the exploit ability resolves, including the creature
 * with exploit itself. You don't have to sacrifice a creature if you don't want to. If you do, you choose
 * which one as the exploit ability resolves. To get the most out of your minions, look for creatures
 * with abilities that give you an added benefit when they die.
 *
 * @author LevelX2
 */
public class ExploitAbility extends EntersBattlefieldTriggeredAbility {

    public ExploitAbility() {
        super(new ExploitEffect(), true);
    }

    protected ExploitAbility(final ExploitAbility ability) {
        super(ability);
    }

    @Override
    public ExploitAbility copy() {
        return new ExploitAbility(this);
    }

    @Override
    public String getRule() {
        return "Exploit <i>(When this creature enters the battlefield, you may sacrifice a creature.)</i>";
    }

}

class ExploitEffect extends OneShotEffect {

    public ExploitEffect() {
        super(Outcome.Detriment);
        this.staticText = "you may sacrifice a creature";
    }

    protected ExploitEffect(final ExploitEffect effect) {
        super(effect);
    }

    @Override
    public ExploitEffect copy() {
        return new ExploitEffect(this);
    }

    @Override
    public boolean apply(Game game, Ability source) {
        Player controller = game.getPlayer(source.getControllerId());
        if (controller != null) {
            Target target = new TargetSacrifice(StaticFilters.FILTER_PERMANENT_A_CREATURE);
            target.withChooseHint("to exploit");
            if (target.canChoose(controller.getId(), source, game)) {
                controller.choose(Outcome.Sacrifice, target, source, game);
                Permanent permanent = game.getPermanent(target.getFirstTarget());
                if (permanent != null && (permanent.sacrifice(source, game))) {
                    game.fireEvent(GameEvent.getEvent(GameEvent.EventType.EXPLOITED_CREATURE, permanent.getId(), source, controller.getId()));
                }
            }
            return true;
        }
        return false;
    }
}
