package mage.sets;

import mage.cards.Card;
import mage.cards.ExpansionSet;
import mage.cards.repository.CardInfo;
import mage.constants.Rarity;
import mage.constants.SetType;
import mage.util.RandomUtil;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * https://mtg.gamepedia.com/Mystery_Booster
 * https://magic.wizards.com/en/articles/archive/feature/unraveling-mystery-booster-2019-11-14
 * Print sheets used for booster construction sourced from http://www.lethe.xyz/mtg/collation/mb1.html
 * <p>
 * This set has a very special booster layout: Each slot draws from its own distinct print sheet and
 * all cards have an equal probability. Therefore, this class implements booster construction by using
 * a list of cards for each slot in the booster.
 *
 * @author TheElk801, xenohedron
 */
public class MysteryBooster extends ExpansionSet {

    private static final MysteryBooster instance = new MysteryBooster();

    public static MysteryBooster getInstance() {
        return instance;
    }

    /**
     * This map defines which cards can go into which booster slot.
     * Will be populated when the first booster is requested.
     * <p>
     * Of note, this is booster collation for MB1, the 2019 Mystery Booster.
     * In the 2021 versions (not collated here), there are 1692 cards in common, and two changes:
     * - Goblin Trenches added instead of Selesnya Guildmage (See Scryfall note on https://scryfall.com/card/mb1/1695/goblin-trenches )
     * - Prophetic Bolt added instead of Nucklavee (See Scryfall note on https://scryfall.com/card/mb1/1696/prophetic-bolt )
     */
    protected final Map<Integer, List<CardInfo>> boosterMap = new HashMap<>();

    private MysteryBooster() {
        super("Mystery Booster", "MB1", ExpansionSet.buildDate(2019, 11, 7), SetType.REMIX);
        this.hasBoosters = true;
        this.hasBasicLands = false;
        // This set is implemented for the purpose of booster generation only.
        // Rather than include cards (scryfall moved them to PLST with thousands of other cards) with duplicate printings,
        // the collation just draws from the original printings in other sets.
        // However, this set would be excluded from view if it contained no cards, so these two cards are here as a workaround.
        cards.add(new SetCardInfo("Goblin Trenches", 203, Rarity.RARE, mage.cards.g.GoblinTrenches.class));
        cards.add(new SetCardInfo("Prophetic Bolt", 231, Rarity.RARE, mage.cards.p.PropheticBolt.class));
    }

    @Override
    public List<Card> createBooster() {
        synchronized (boosterMap) {
            // init the map only once, on first booster gen
            if (boosterMap.isEmpty()) {
                initBoosterMap();
            }
        }
        final List<Card> booster = new ArrayList<>(15);
        for (int slot = 1; slot < 16; ++slot) {
            final List<CardInfo> availableCards = boosterMap.get(slot);
            final int printSheetCardNumber = RandomUtil.nextInt(availableCards.size());
            final Card chosenCard = availableCards.get(printSheetCardNumber).createCard();
            booster.add(chosenCard);
        }
        return booster;
    }

    private void initBoosterMap() {
        for (int i = 1; i < 16; ++i) {
            this.boosterMap.put(i, new ArrayList<>(121));
        }
        addSlot1whiteA(boosterMap.get(1));
        addSlot2whiteB(boosterMap.get(2));
        addSlot3blueA(boosterMap.get(3));
        addSlot4blueB(boosterMap.get(4));
        addSlot5blackA(boosterMap.get(5));
        addSlot6blackB(boosterMap.get(6));
        addSlot7redA(boosterMap.get(7));
        addSlot8redB(boosterMap.get(8));
        addSlot9greenA(boosterMap.get(9));
        addSlot10greenB(boosterMap.get(10));
        addSlot11multicolored(boosterMap.get(11));
        addSlot12colorless(boosterMap.get(12));
        addSlot13oldFrame(boosterMap.get(13));
        addSlot14rare(boosterMap.get(14));
        addSlot15foil(boosterMap.get(15));
    }

    private void addSlot1whiteA(List<CardInfo> boosterList) {
        addCardInfoToList(boosterList, "Abzan Falconer", "KTK", "2");
        addCardInfoToList(boosterList, "Abzan Runemark", "FRF", "3");
        addCardInfoToList(boosterList, "Acrobatic Maneuver", "KLD", "1");
        addCardInfoToList(boosterList, "Affa Protector", "OGW", "14");
        addCardInfoToList(boosterList, "Ainok Bond-Kin", "KTK", "3");
        addCardInfoToList(boosterList, "Alley Evasion", "AER", "6");
        addCardInfoToList(boosterList, "Angelic Purge", "SOI", "3");
        addCardInfoToList(boosterList, "Angelsong", "DDC", "15");
        addCardInfoToList(boosterList, "Apostle's Blessing", "MM2", "8");
        addCardInfoToList(boosterList, "Arrester's Zeal", "RNA", "4");
        addCardInfoToList(boosterList, "Artful Maneuver", "DTK", "4");
        addCardInfoToList(boosterList, "Aura of Silence", "C15", "60");
        addCardInfoToList(boosterList, "Bartered Cow", "ELD", "6");
        addCardInfoToList(boosterList, "Bonds of Faith", "DDQ", "2");
        addCardInfoToList(boosterList, "Borrowed Grace", "EMN", "14");
        addCardInfoToList(boosterList, "Bulwark Giant", "WAR", "7");
        addCardInfoToList(boosterList, "Caravan Escort", "DDP", "3");
        addCardInfoToList(boosterList, "Caught in the Brights", "AER", "10");
        addCardInfoToList(boosterList, "Celestial Crusader", "C14", "68");
        addCardInfoToList(boosterList, "Celestial Flare", "ORI", "8");
        addCardInfoToList(boosterList, "Center Soul", "DTK", "8");
        addCardInfoToList(boosterList, "Cliffside Lookout", "BFZ", "20");
        addCardInfoToList(boosterList, "Conviction", "AER", "12");
        addCardInfoToList(boosterList, "Countless Gears Renegade", "AER", "13");
        addCardInfoToList(boosterList, "Court Street Denizen", "DDO", "5");
        addCardInfoToList(boosterList, "Crib Swap", "C18", "65");
        addCardInfoToList(boosterList, "Danitha Capashen, Paragon", "DOM", "12");
        addCardInfoToList(boosterList, "Daring Skyjek", "GK1", "79");
        addCardInfoToList(boosterList, "Decommission", "AER", "16");
        addCardInfoToList(boosterList, "Defiant Strike", "WAR", "9");
        addCardInfoToList(boosterList, "Desperate Sentry", "EMN", "21");
        addCardInfoToList(boosterList, "Devilthorn Fox", "SOI", "14");
        addCardInfoToList(boosterList, "Disposal Mummy", "HOU", "9");
        addCardInfoToList(boosterList, "Divine Favor", "M15", "10");
        addCardInfoToList(boosterList, "Dragon's Eye Sentry", "DTK", "11");
        addCardInfoToList(boosterList, "Dragon's Presence", "GS1", "16");
        addCardInfoToList(boosterList, "Eddytrail Hawk", "KLD", "12");
        addCardInfoToList(boosterList, "Enduring Victory", "DTK", "16");
        addCardInfoToList(boosterList, "Enlightened Ascetic", "ORI", "12");
        addCardInfoToList(boosterList, "Ephemeral Shields", "M15", "11");
        addCardInfoToList(boosterList, "Ephemerate", "MH1", "7");
        addCardInfoToList(boosterList, "Excoriate", "E01", "5");
        addCardInfoToList(boosterList, "Expose Evil", "SOI", "19");
        addCardInfoToList(boosterList, "Eyes in the Skies", "MM3", "5");
        addCardInfoToList(boosterList, "Faith's Fetters", "UMA", "16");
        addCardInfoToList(boosterList, "Feat of Resistance", "KTK", "10");
        addCardInfoToList(boosterList, "Felidar Umbra", "PCA", "6");
        addCardInfoToList(boosterList, "Firehoof Cavalry", "KTK", "11");
        addCardInfoToList(boosterList, "Ghostblade Eidolon", "C15", "70");
        addCardInfoToList(boosterList, "Gift of Estates", "C14", "73");
        addCardInfoToList(boosterList, "Glaring Aegis", "DTK", "18");
        addCardInfoToList(boosterList, "Glint-Sleeve Artisan", "KLD", "17");
        addCardInfoToList(boosterList, "God-Pharaoh's Faithful", "HOU", "14");
        addCardInfoToList(boosterList, "Grasp of the Hieromancer", "E01", "13");
        addCardInfoToList(boosterList, "Gust Walker", "AKH", "17");
        addCardInfoToList(boosterList, "Gustcloak Skirmisher", "DDO", "13");
        addCardInfoToList(boosterList, "Healing Hands", "ORI", "17");
        addCardInfoToList(boosterList, "Hyena Umbra", "UMA", "21");
        addCardInfoToList(boosterList, "Infantry Veteran", "DDN", "3");
        addCardInfoToList(boosterList, "Inquisitor's Ox", "SOI", "24");
        addCardInfoToList(boosterList, "Isolation Zone", "OGW", "22");
        addCardInfoToList(boosterList, "Knight of Old Benalia", "MH1", "17");
        addCardInfoToList(boosterList, "Knight of Sorrows", "RNA", "14");
        addCardInfoToList(boosterList, "Kor Skyfisher", "DDO", "16");
        addCardInfoToList(boosterList, "Leonin Relic-Warder", "C17", "65");
        addCardInfoToList(boosterList, "Lightform", "C18", "68");
        addCardInfoToList(boosterList, "Lone Missionary", "DDN", "49");
        addCardInfoToList(boosterList, "Lonesome Unicorn", "ELD", "21");
        addCardInfoToList(boosterList, "Lotus-Eye Mystics", "UMA", "23");
        addCardInfoToList(boosterList, "Loxodon Partisan", "DDO", "17");
        addCardInfoToList(boosterList, "Mardu Hordechief", "KTK", "17");
        addCardInfoToList(boosterList, "Marked by Honor", "M15", "17");
        addCardInfoToList(boosterList, "Meditation Puzzle", "M15", "19");
        addCardInfoToList(boosterList, "Mortal's Ardor", "DDO", "19");
        addCardInfoToList(boosterList, "Mother of Runes", "DDO", "20");
        addCardInfoToList(boosterList, "Ninth Bridge Patrol", "KLD", "22");
        addCardInfoToList(boosterList, "Ondu Greathorn", "BFZ", "40");
        addCardInfoToList(boosterList, "Ondu War Cleric", "OGW", "31");
        addCardInfoToList(boosterList, "Oreskos Swiftclaw", "M15", "22");
        addCardInfoToList(boosterList, "Oust", "DDP", "7");
        addCardInfoToList(boosterList, "Palace Jailer", "CN2", "18");
        addCardInfoToList(boosterList, "Path to Exile", "E02", "3");
        addCardInfoToList(boosterList, "Peace of Mind", "EMN", "36");
        addCardInfoToList(boosterList, "Prowling Caracal", "RNA", "17");
        addCardInfoToList(boosterList, "Resurrection", "UMA", "30");
        addCardInfoToList(boosterList, "Rhet-Crop Spearmaster", "AKH", "26");
        addCardInfoToList(boosterList, "Righteous Cause", "CMA", "21");
        addCardInfoToList(boosterList, "Savannah Lions", "A25", "33");
        addCardInfoToList(boosterList, "Searing Light", "OGW", "33");
        addCardInfoToList(boosterList, "Serra's Embrace", "DDC", "21");
        addCardInfoToList(boosterList, "Sheer Drop", "BFZ", "48");
        addCardInfoToList(boosterList, "Shining Aerosaur", "XLN", "36");
        addCardInfoToList(boosterList, "Shining Armor", "ELD", "29");
        addCardInfoToList(boosterList, "Siegecraft", "KTK", "23");
        addCardInfoToList(boosterList, "Skymarcher Aspirant", "RIX", "21");
        addCardInfoToList(boosterList, "Skyspear Cavalry", "UMA", "36");
        addCardInfoToList(boosterList, "Snubhorn Sentry", "RIX", "23");
        addCardInfoToList(boosterList, "Soul Parry", "DDO", "24");
        addCardInfoToList(boosterList, "Soul Summons", "FRF", "26");
        addCardInfoToList(boosterList, "Soul-Strike Technique", "MH1", "30");
        addCardInfoToList(boosterList, "Soulmender", "M15", "35");
        addCardInfoToList(boosterList, "Sparring Mummy", "AKH", "29");
        addCardInfoToList(boosterList, "Spectral Gateguards", "DDQ", "19");
        addCardInfoToList(boosterList, "Stave Off", "DDN", "61");
        addCardInfoToList(boosterList, "Steadfast Sentinel", "HOU", "24");
        addCardInfoToList(boosterList, "Stone Haven Medic", "BFZ", "51");
        addCardInfoToList(boosterList, "Suppression Bonds", "ORI", "34");
        addCardInfoToList(boosterList, "Survive the Night", "SOI", "41");
        addCardInfoToList(boosterList, "Territorial Hammerskull", "XLN", "41");
        addCardInfoToList(boosterList, "Thraben Inspector", "SOI", "44");
        addCardInfoToList(boosterList, "Thraben Standard Bearer", "EMN", "48");
        addCardInfoToList(boosterList, "Topan Freeblade", "ORI", "36");
        addCardInfoToList(boosterList, "Veteran Swordsmith", "DDO", "28");
        addCardInfoToList(boosterList, "Village Bell-Ringer", "DDQ", "22");
        addCardInfoToList(boosterList, "Voice of the Provinces", "DDQ", "23");
        addCardInfoToList(boosterList, "Wall of One Thousand Cuts", "MH1", "36");
        addCardInfoToList(boosterList, "Wandering Champion", "UMA", "42");
        addCardInfoToList(boosterList, "War Behemoth", "KTK", "29");
        addCardInfoToList(boosterList, "Windborne Charge", "CN2", "100");
        addCardInfoToList(boosterList, "Wing Shards", "IMA", "38");
        addCardInfoToList(boosterList, "Winged Shepherd", "AKH", "39");
    }

    private void addSlot2whiteB(List<CardInfo> boosterList) {
        addCardInfoToList(boosterList, "Adanto Vanguard", "XLN", "1");
        addCardInfoToList(boosterList, "Ajani's Pridemate", "WAR", "4");
        addCardInfoToList(boosterList, "Angel of Mercy", "IMA", "6");
        addCardInfoToList(boosterList, "Angel of Renewal", "BFZ", "18");
        addCardInfoToList(boosterList, "Angelic Gift", "BBD", "88");
        addCardInfoToList(boosterList, "Arrest", "MM2", "9");
        addCardInfoToList(boosterList, "Aven Battle Priest", "ORI", "6");
        addCardInfoToList(boosterList, "Aven Sentry", "DOM", "3");
        addCardInfoToList(boosterList, "Ballynock Cohort", "EMA", "3");
        addCardInfoToList(boosterList, "Battle Mastery", "BBD", "89");
        addCardInfoToList(boosterList, "Benevolent Ancestor", "IMA", "12");
        addCardInfoToList(boosterList, "Blade Instructor", "GRN", "1");
        addCardInfoToList(boosterList, "Blessed Spirits", "ORI", "7");
        addCardInfoToList(boosterList, "Built to Last", "KLD", "7");
        addCardInfoToList(boosterList, "Candlelight Vigil", "GRN", "3");
        addCardInfoToList(boosterList, "Cartouche of Solidarity", "AKH", "7");
        addCardInfoToList(boosterList, "Cast Out", "AKH", "8");
        addCardInfoToList(boosterList, "Cathar's Companion", "SOI", "9");
        addCardInfoToList(boosterList, "Champion of Arashin", "DTK", "9");
        addCardInfoToList(boosterList, "Charge", "DOM", "10");
        addCardInfoToList(boosterList, "Cloudshift", "A25", "7");
        addCardInfoToList(boosterList, "Coalition Honor Guard", "EMA", "6");
        addCardInfoToList(boosterList, "Collar the Culprit", "GRN", "5");
        addCardInfoToList(boosterList, "Congregate", "M15", "6");
        addCardInfoToList(boosterList, "Court Homunculus", "MM2", "13");
        addCardInfoToList(boosterList, "Darksteel Mutation", "CMA", "9");
        addCardInfoToList(boosterList, "Dauntless Cathar", "SOI", "11");
        addCardInfoToList(boosterList, "Dawnglare Invoker", "C15", "67");
        addCardInfoToList(boosterList, "Disenchant", "M20", "14");
        addCardInfoToList(boosterList, "Dismantling Blow", "MH1", "5");
        addCardInfoToList(boosterList, "Djeru's Renunciation", "HOU", "11");
        addCardInfoToList(boosterList, "Djeru's Resolve", "AKH", "11");
        addCardInfoToList(boosterList, "Doomed Traveler", "CN2", "83");
        addCardInfoToList(boosterList, "Dragon Bell Monk", "IMA", "17");
        addCardInfoToList(boosterList, "Emerge Unscathed", "IMA", "19");
        addCardInfoToList(boosterList, "Encampment Keeper", "XLN", "11");
        addCardInfoToList(boosterList, "Encircling Fissure", "BFZ", "23");
        addCardInfoToList(boosterList, "Excavation Elephant", "DOM", "17");
        addCardInfoToList(boosterList, "Expedition Raptor", "BBD", "92");
        addCardInfoToList(boosterList, "Exultant Skymarcher", "RIX", "7");
        addCardInfoToList(boosterList, "Faithbearer Paladin", "EMN", "25");
        addCardInfoToList(boosterList, "Felidar Guardian", "AER", "19");
        addCardInfoToList(boosterList, "Fencing Ace", "A25", "13");
        addCardInfoToList(boosterList, "Fiend Hunter", "CMA", "10");
        addCardInfoToList(boosterList, "Forsake the Worldly", "AKH", "13");
        addCardInfoToList(boosterList, "Fortify", "MM2", "17");
        addCardInfoToList(boosterList, "Fragmentize", "KLD", "14");
        addCardInfoToList(boosterList, "Geist of the Moors", "A25", "15");
        addCardInfoToList(boosterList, "Gideon's Lawkeeper", "MM3", "7");
        addCardInfoToList(boosterList, "Gleam of Resistance", "CN2", "87");
        addCardInfoToList(boosterList, "Gods Willing", "UMA", "18");
        addCardInfoToList(boosterList, "Great-Horn Krushok", "FRF", "13");
        addCardInfoToList(boosterList, "Guided Strike", "IMA", "23");
        addCardInfoToList(boosterList, "Healer's Hawk", "GRN", "14");
        addCardInfoToList(boosterList, "Healing Grace", "DOM", "20");
        addCardInfoToList(boosterList, "Heavy Infantry", "ORI", "18");
        addCardInfoToList(boosterList, "Humble", "EMA", "14");
        addCardInfoToList(boosterList, "Inspired Charge", "M19", "15");
        addCardInfoToList(boosterList, "Intrusive Packbeast", "GRN", "17");
        addCardInfoToList(boosterList, "Iona's Judgment", "IMA", "25");
        addCardInfoToList(boosterList, "Jubilant Mascot", "BBD", "28");
        addCardInfoToList(boosterList, "Knight of Cliffhaven", "DDP", "5");
        addCardInfoToList(boosterList, "Knight of the Skyward Eye", "A25", "19");
        addCardInfoToList(boosterList, "Knight of the Tusk", "M19", "18");
        addCardInfoToList(boosterList, "Kor Bladewhirl", "BFZ", "34");
        addCardInfoToList(boosterList, "Kor Firewalker", "A25", "21");
        addCardInfoToList(boosterList, "Kor Hookmaster", "EMA", "18");
        addCardInfoToList(boosterList, "Kor Sky Climber", "OGW", "24");
        addCardInfoToList(boosterList, "Lieutenants of the Guard", "CN2", "16");
        addCardInfoToList(boosterList, "Lightwalker", "BBD", "95");
        addCardInfoToList(boosterList, "Lingering Souls", "MM3", "12");
        addCardInfoToList(boosterList, "Looming Altisaur", "XLN", "23");
        addCardInfoToList(boosterList, "Loyal Sentry", "A25", "22");
        addCardInfoToList(boosterList, "Lunarch Mantle", "A25", "24");
        addCardInfoToList(boosterList, "Midnight Guard", "BBD", "99");
        addCardInfoToList(boosterList, "Momentary Blink", "MM3", "16");
        addCardInfoToList(boosterList, "Moonlit Strider", "MM2", "27");
        addCardInfoToList(boosterList, "Nyx-Fleece Ram", "A25", "26");
        addCardInfoToList(boosterList, "Pacifism", "M20", "32");
        addCardInfoToList(boosterList, "Palace Sentinels", "CN2", "19");
        addCardInfoToList(boosterList, "Paladin of the Bloodstained", "XLN", "25");
        addCardInfoToList(boosterList, "Path of Peace", "A25", "29");
        addCardInfoToList(boosterList, "Pegasus Courser", "M19", "32");
        addCardInfoToList(boosterList, "Pentarch Ward", "IMA", "27");
        addCardInfoToList(boosterList, "Pitfall Trap", "MM3", "18");
        addCardInfoToList(boosterList, "Pressure Point", "FRF", "21");
        addCardInfoToList(boosterList, "Promise of Bunrei", "A25", "30");
        addCardInfoToList(boosterList, "Rally the Peasants", "EMA", "25");
        addCardInfoToList(boosterList, "Raptor Companion", "RIX", "19");
        addCardInfoToList(boosterList, "Refurbish", "KLD", "25");
        addCardInfoToList(boosterList, "Renewed Faith", "A25", "31");
        addCardInfoToList(boosterList, "Retreat to Emeria", "BFZ", "44");
        addCardInfoToList(boosterList, "Reviving Dose", "CN2", "97");
        addCardInfoToList(boosterList, "Rootborn Defenses", "MM3", "21");
        addCardInfoToList(boosterList, "Sacred Cat", "AKH", "27");
        addCardInfoToList(boosterList, "Sanctum Gargoyle", "C16", "76");
        addCardInfoToList(boosterList, "Sandstorm Charger", "DTK", "34");
        addCardInfoToList(boosterList, "Seal of Cleansing", "EMA", "26");
        addCardInfoToList(boosterList, "Seeker of the Way", "KTK", "22");
        addCardInfoToList(boosterList, "Sensor Splicer", "MM3", "23");
        addCardInfoToList(boosterList, "Seraph of the Suns", "AKH", "28");
        addCardInfoToList(boosterList, "Serra Disciple", "DOM", "34");
        addCardInfoToList(boosterList, "Shoulder to Shoulder", "BBD", "105");
        addCardInfoToList(boosterList, "Silverchase Fox", "BBD", "106");
        addCardInfoToList(boosterList, "Skyhunter Skirmisher", "MM2", "32");
        addCardInfoToList(boosterList, "Slash of Talons", "XLN", "38");
        addCardInfoToList(boosterList, "Soul Warden", "MM3", "24");
        addCardInfoToList(boosterList, "Stalwart Aven", "IMA", "32");
        addCardInfoToList(boosterList, "Star-Crowned Stag", "M19", "38");
        addCardInfoToList(boosterList, "Sunlance", "MM2", "34");
        addCardInfoToList(boosterList, "Sunrise Seeker", "XLN", "40");
        addCardInfoToList(boosterList, "Swords to Plowshares", "C16", "78");
        addCardInfoToList(boosterList, "Take Vengeance", "M19", "40");
        addCardInfoToList(boosterList, "Tandem Tactics", "BBD", "112");
        addCardInfoToList(boosterList, "Terashi's Grasp", "MM2", "37");
        addCardInfoToList(boosterList, "Unwavering Initiate", "AKH", "36");
        addCardInfoToList(boosterList, "Wake the Reflections", "MM3", "28");
        addCardInfoToList(boosterList, "Wall of Omens", "EMA", "34");
        addCardInfoToList(boosterList, "Wild Griffin", "CN2", "99");
        addCardInfoToList(boosterList, "Youthful Knight", "MM3", "29");
        addCardInfoToList(boosterList, "Zealous Strike", "CN2", "101");
    }

    private void addSlot3blueA(List<CardInfo> boosterList) {
        addCardInfoToList(boosterList, "Amass the Components", "IMA", "41");
        addCardInfoToList(boosterList, "Anticipate", "M19", "44");
        addCardInfoToList(boosterList, "Artificer's Assistant", "DOM", "44");
        addCardInfoToList(boosterList, "Augury Owl", "PCA", "14");
        addCardInfoToList(boosterList, "Befuddle", "M19", "309");
        addCardInfoToList(boosterList, "Benthic Giant", "BBD", "113");
        addCardInfoToList(boosterList, "Calculated Dismissal", "ORI", "48");
        addCardInfoToList(boosterList, "Call to Heel", "BBD", "114");
        addCardInfoToList(boosterList, "Caller of Gales", "CN2", "103");
        addCardInfoToList(boosterList, "Cancel", "KTK", "33");
        addCardInfoToList(boosterList, "Capture Sphere", "GRN", "31");
        addCardInfoToList(boosterList, "Catalog", "SOI", "51");
        addCardInfoToList(boosterList, "Chart a Course", "XLN", "48");
        addCardInfoToList(boosterList, "Chillbringer", "RNA", "33");
        addCardInfoToList(boosterList, "Chronostutter", "M15", "48");
        addCardInfoToList(boosterList, "Circular Logic", "UMA", "47");
        addCardInfoToList(boosterList, "Clear the Mind", "RNA", "34");
        addCardInfoToList(boosterList, "Cloak of Mists", "GS1", "13");
        addCardInfoToList(boosterList, "Cloudkin Seer", "M20", "54");
        addCardInfoToList(boosterList, "Clutch of Currents", "BFZ", "72");
        addCardInfoToList(boosterList, "Compelling Argument", "AKH", "47");
        addCardInfoToList(boosterList, "Condescend", "IMA", "46");
        addCardInfoToList(boosterList, "Containment Membrane", "OGW", "52");
        addCardInfoToList(boosterList, "Contingency Plan", "EMN", "52");
        addCardInfoToList(boosterList, "Contradict", "DTK", "49");
        addCardInfoToList(boosterList, "Crashing Tide", "RIX", "34");
        addCardInfoToList(boosterList, "Crush Dissent", "WAR", "47");
        addCardInfoToList(boosterList, "Curio Vendor", "KLD", "42");
        addCardInfoToList(boosterList, "Daze", "DD2", "23");
        addCardInfoToList(boosterList, "Decision Paralysis", "AKH", "50");
        addCardInfoToList(boosterList, "Deep Freeze", "DOM", "50");
        addCardInfoToList(boosterList, "Dispel", "BFZ", "76");
        addCardInfoToList(boosterList, "Displace", "EMN", "55");
        addCardInfoToList(boosterList, "Drag Under", "W17", "9");
        addCardInfoToList(boosterList, "Dragon's Eye Savants", "KTK", "38");
        addCardInfoToList(boosterList, "Dreadwaters", "ORI", "56");
        addCardInfoToList(boosterList, "Embodiment of Spring", "KTK", "39");
        addCardInfoToList(boosterList, "Ensoul Artifact", "M15", "54");
        addCardInfoToList(boosterList, "Everdream", "MH1", "47");
        addCardInfoToList(boosterList, "Failed Inspection", "KLD", "47");
        addCardInfoToList(boosterList, "Flashfreeze", "MM2", "45");
        addCardInfoToList(boosterList, "Fledgling Mawcor", "DD2", "10");
        addCardInfoToList(boosterList, "Fleeting Distraction", "CN2", "110");
        addCardInfoToList(boosterList, "Fogwalker", "EMN", "60");
        addCardInfoToList(boosterList, "Foil", "UMA", "55");
        addCardInfoToList(boosterList, "Frantic Search", "UMA", "57");
        addCardInfoToList(boosterList, "Frilled Sea Serpent", "M19", "56");
        addCardInfoToList(boosterList, "Gaseous Form", "EMA", "51");
        addCardInfoToList(boosterList, "Glint", "DTK", "55");
        addCardInfoToList(boosterList, "Gone Missing", "SOI", "67");
        addCardInfoToList(boosterList, "Grasp of Phantoms", "MM3", "41");
        addCardInfoToList(boosterList, "Guard Gomazoa", "PCA", "17");
        addCardInfoToList(boosterList, "Gurmag Drowner", "DTK", "57");
        addCardInfoToList(boosterList, "Gush", "DD2", "27");
        addCardInfoToList(boosterList, "Hightide Hermit", "KLD", "51");
        addCardInfoToList(boosterList, "Hinterland Drake", "AER", "34");
        addCardInfoToList(boosterList, "Humongulus", "RNA", "41");
        addCardInfoToList(boosterList, "Inkfathom Divers", "DDT", "8");
        addCardInfoToList(boosterList, "Invisibility", "M15", "61");
        addCardInfoToList(boosterList, "Jeering Homunculus", "CN2", "33");
        addCardInfoToList(boosterList, "Jeskai Sage", "FRF", "38");
        addCardInfoToList(boosterList, "Kiora's Dambreaker", "WAR", "58");
        addCardInfoToList(boosterList, "Laboratory Brute", "EMN", "67");
        addCardInfoToList(boosterList, "Laboratory Maniac", "UMA", "61");
        addCardInfoToList(boosterList, "Labyrinth Guardian", "AKH", "60");
        addCardInfoToList(boosterList, "Messenger Jays", "CN2", "35");
        addCardInfoToList(boosterList, "Mind Sculpt", "M15", "70");
        addCardInfoToList(boosterList, "Mist Raven", "DDQ", "26");
        addCardInfoToList(boosterList, "Mnemonic Wall", "IMA", "67");
        addCardInfoToList(boosterList, "Monastery Loremaster", "DTK", "63");
        addCardInfoToList(boosterList, "Murder of Crows", "A25", "66");
        addCardInfoToList(boosterList, "Nagging Thoughts", "SOI", "74");
        addCardInfoToList(boosterList, "Niblis of Dusk", "SOI", "76");
        addCardInfoToList(boosterList, "Nine-Tail White Fox", "GS1", "8");
        addCardInfoToList(boosterList, "Ojutai's Breath", "DTK", "67");
        addCardInfoToList(boosterList, "Phyrexian Ingester", "EMA", "66");
        addCardInfoToList(boosterList, "Pondering Mage", "MH1", "63");
        addCardInfoToList(boosterList, "Predict", "C18", "98");
        addCardInfoToList(boosterList, "Purple-Crystal Crab", "GS1", "3");
        addCardInfoToList(boosterList, "Refocus", "FRF", "47");
        addCardInfoToList(boosterList, "Riftwing Cloudskate", "DD2", "15");
        addCardInfoToList(boosterList, "River Darter", "RIX", "47");
        addCardInfoToList(boosterList, "Sailor of Means", "RIX", "49");
        addCardInfoToList(boosterList, "Scroll Thief", "DDT", "17");
        addCardInfoToList(boosterList, "Send to Sleep", "ORI", "71");
        addCardInfoToList(boosterList, "Shipwreck Looter", "XLN", "76");
        addCardInfoToList(boosterList, "Silent Observer", "SOI", "86");
        addCardInfoToList(boosterList, "Silvergill Adept", "RIX", "53");
        addCardInfoToList(boosterList, "Singing Bell Strike", "KTK", "55");
        addCardInfoToList(boosterList, "Skaab Goliath", "ORI", "74");
        addCardInfoToList(boosterList, "Skitter Eel", "RNA", "53");
        addCardInfoToList(boosterList, "Sleep", "M19", "74");
        addCardInfoToList(boosterList, "Slipstream Eel", "CM2", "49");
        addCardInfoToList(boosterList, "Slither Blade", "AKH", "71");
        addCardInfoToList(boosterList, "Sphinx's Tutelage", "ORI", "76");
        addCardInfoToList(boosterList, "Stream of Thought", "MH1", "71");
        addCardInfoToList(boosterList, "Surrakar Banisher", "DDO", "43");
        addCardInfoToList(boosterList, "Syr Elenora, the Discerning", "ELD", "67");
        addCardInfoToList(boosterList, "Thought Collapse", "RNA", "57");
        addCardInfoToList(boosterList, "Thunder Drake", "WAR", "73");
        addCardInfoToList(boosterList, "Tidal Warrior", "DDT", "20");
        addCardInfoToList(boosterList, "Trail of Evidence", "SOI", "93");
        addCardInfoToList(boosterList, "Treasure Cruise", "KTK", "59");
        addCardInfoToList(boosterList, "Treasure Mage", "DDU", "40");
        addCardInfoToList(boosterList, "Trinket Mage", "DDU", "41");
        addCardInfoToList(boosterList, "Turn Aside", "EMN", "78");
        addCardInfoToList(boosterList, "Uncomfortable Chill", "M19", "82");
        addCardInfoToList(boosterList, "Wall of Frost", "MM3", "56");
        addCardInfoToList(boosterList, "Warden of Evos Isle", "EMA", "76");
        addCardInfoToList(boosterList, "Watercourser", "BBD", "137");
        addCardInfoToList(boosterList, "Weldfast Wingsmith", "KLD", "69");
        addCardInfoToList(boosterList, "Welkin Tern", "GS1", "5");
        addCardInfoToList(boosterList, "Wind Drake", "KLD", "70");
        addCardInfoToList(boosterList, "Wind Strider", "XLN", "88");
        addCardInfoToList(boosterList, "Wind-Kin Raiders", "AER", "50");
        addCardInfoToList(boosterList, "Windcaller Aven", "MH1", "77");
        addCardInfoToList(boosterList, "Wishcoin Crab", "GRN", "60");
        addCardInfoToList(boosterList, "Wishful Merfolk", "ELD", "73");
        addCardInfoToList(boosterList, "Wretched Gryff", "EMN", "12");
        addCardInfoToList(boosterList, "Write into Being", "FRF", "59");
        addCardInfoToList(boosterList, "Youthful Scholar", "DTK", "84");
    }

    private void addSlot4blueB(List<CardInfo> boosterList) {
        addCardInfoToList(boosterList, "Academy Journeymage", "DOM", "41");
        addCardInfoToList(boosterList, "Aether Tradewinds", "KLD", "38");
        addCardInfoToList(boosterList, "Aethersnipe", "MM2", "39");
        addCardInfoToList(boosterList, "Amphin Pathmage", "M15", "45");
        addCardInfoToList(boosterList, "Arcane Denial", "CMA", "30");
        addCardInfoToList(boosterList, "Archaeomancer", "C17", "81");
        addCardInfoToList(boosterList, "Archetype of Imagination", "C18", "81");
        addCardInfoToList(boosterList, "Augur of Bolas", "MM3", "30");
        addCardInfoToList(boosterList, "Bastion Inventor", "AER", "30");
        addCardInfoToList(boosterList, "Bewilder", "IMA", "43");
        addCardInfoToList(boosterList, "Blue Elemental Blast", "A25", "43");
        addCardInfoToList(boosterList, "Borrowing 100,000 Arrows", "A25", "45");
        addCardInfoToList(boosterList, "Brainstorm", "A25", "46");
        addCardInfoToList(boosterList, "Brilliant Spectrum", "BFZ", "70");
        addCardInfoToList(boosterList, "Brine Elemental", "A25", "47");
        addCardInfoToList(boosterList, "Cartouche of Knowledge", "AKH", "45");
        addCardInfoToList(boosterList, "Castaway's Despair", "XLN", "281");
        addCardInfoToList(boosterList, "Choking Tethers", "A25", "48");
        addCardInfoToList(boosterList, "Citywatch Sphinx", "GRN", "33");
        addCardInfoToList(boosterList, "Claustrophobia", "DDT", "3");
        addCardInfoToList(boosterList, "Cloud Elemental", "MM2", "42");
        addCardInfoToList(boosterList, "Cloudreader Sphinx", "DOM", "47");
        addCardInfoToList(boosterList, "Concentrate", "E02", "9");
        addCardInfoToList(boosterList, "Convolute", "EMN", "53");
        addCardInfoToList(boosterList, "Coral Trickster", "DDN", "44");
        addCardInfoToList(boosterList, "Coralhelm Guide", "BBD", "116");
        addCardInfoToList(boosterList, "Counterspell", "A25", "50");
        addCardInfoToList(boosterList, "Court Hussar", "A25", "51");
        addCardInfoToList(boosterList, "Curiosity", "A25", "52");
        addCardInfoToList(boosterList, "Dazzling Lights", "GRN", "34");
        addCardInfoToList(boosterList, "Deep Analysis", "EMA", "45");
        addCardInfoToList(boosterList, "Diminish", "IMA", "50");
        addCardInfoToList(boosterList, "Dirgur Nemesis", "DTK", "51");
        addCardInfoToList(boosterList, "Distortion Strike", "IMA", "52");
        addCardInfoToList(boosterList, "Divination", "M19", "51");
        addCardInfoToList(boosterList, "Doorkeeper", "IMA", "53");
        addCardInfoToList(boosterList, "Dream Cache", "C18", "88");
        addCardInfoToList(boosterList, "Dream Twist", "EMA", "47");
        addCardInfoToList(boosterList, "Eel Umbra", "C18", "89");
        addCardInfoToList(boosterList, "Enlightened Maniac", "EMN", "58");
        addCardInfoToList(boosterList, "Errant Ephemeron", "DD2", "20");
        addCardInfoToList(boosterList, "Essence Scatter", "M19", "54");
        addCardInfoToList(boosterList, "Exclude", "MH1", "48");
        addCardInfoToList(boosterList, "Fact or Fiction", "CM2", "42");
        addCardInfoToList(boosterList, "Faerie Invaders", "DDN", "57");
        addCardInfoToList(boosterList, "Faerie Mechanist", "DDU", "38");
        addCardInfoToList(boosterList, "Fascination", "FRF", "34");
        addCardInfoToList(boosterList, "Fathom Seer", "C14", "109");
        addCardInfoToList(boosterList, "Fog Bank", "BBD", "117");
        addCardInfoToList(boosterList, "Forbidden Alchemy", "MM3", "38");
        addCardInfoToList(boosterList, "Frost Lynx", "BBD", "118");
        addCardInfoToList(boosterList, "Ghost Ship", "A25", "60");
        addCardInfoToList(boosterList, "Glacial Crasher", "M15", "57");
        addCardInfoToList(boosterList, "Hieroglyphic Illumination", "AKH", "57");
        addCardInfoToList(boosterList, "Horseshoe Crab", "A25", "61");
        addCardInfoToList(boosterList, "Impulse", "BBD", "119");
        addCardInfoToList(boosterList, "Ior Ruin Expedition", "E01", "25");
        addCardInfoToList(boosterList, "Jace's Phantasm", "IMA", "60");
        addCardInfoToList(boosterList, "Jwar Isle Avenger", "OGW", "58");
        addCardInfoToList(boosterList, "Lay Claim", "AKH", "61");
        addCardInfoToList(boosterList, "Leapfrog", "GRN", "42");
        addCardInfoToList(boosterList, "Mahamoti Djinn", "IMA", "64");
        addCardInfoToList(boosterList, "Man-o'-War", "A25", "64");
        addCardInfoToList(boosterList, "Mana Leak", "DDN", "64");
        addCardInfoToList(boosterList, "Maximize Altitude", "GRN", "43");
        addCardInfoToList(boosterList, "Memory Lapse", "EMA", "60");
        addCardInfoToList(boosterList, "Merfolk Looter", "DDT", "10");
        addCardInfoToList(boosterList, "Metallic Rebuke", "AER", "39");
        addCardInfoToList(boosterList, "Mulldrifter", "CM2", "47");
        addCardInfoToList(boosterList, "Mystic of the Hidden Way", "A25", "67");
        addCardInfoToList(boosterList, "Mystical Teachings", "MM3", "44");
        addCardInfoToList(boosterList, "Negate", "RIX", "44");
        addCardInfoToList(boosterList, "Ninja of the Deep Hours", "C18", "95");
        addCardInfoToList(boosterList, "Ojutai Interceptor", "DTK", "66");
        addCardInfoToList(boosterList, "Omenspeaker", "BBD", "125");
        addCardInfoToList(boosterList, "Opportunity", "C17", "89");
        addCardInfoToList(boosterList, "Opt", "DOM", "60");
        addCardInfoToList(boosterList, "Peel from Reality", "DDO", "40");
        addCardInfoToList(boosterList, "Phantasmal Bear", "A25", "69");
        addCardInfoToList(boosterList, "Portent", "C18", "97");
        addCardInfoToList(boosterList, "Preordain", "C15", "101");
        addCardInfoToList(boosterList, "Prodigal Sorcerer", "EMA", "67");
        addCardInfoToList(boosterList, "Propaganda", "C16", "94");
        addCardInfoToList(boosterList, "Prosperous Pirates", "XLN", "69");
        addCardInfoToList(boosterList, "Repulse", "CN2", "119");
        addCardInfoToList(boosterList, "Retraction Helix", "A25", "71");
        addCardInfoToList(boosterList, "Ringwarden Owl", "ORI", "68");
        addCardInfoToList(boosterList, "River Serpent", "AKH", "66");
        addCardInfoToList(boosterList, "Riverwheel Aerialists", "IMA", "71");
        addCardInfoToList(boosterList, "Sage of Lat-Nam", "DOM", "64");
        addCardInfoToList(boosterList, "Sea Gate Oracle", "C17", "92");
        addCardInfoToList(boosterList, "Sealock Monster", "DDO", "42");
        addCardInfoToList(boosterList, "Secrets of the Golden City", "RIX", "52");
        addCardInfoToList(boosterList, "Shaper Parasite", "C14", "125");
        addCardInfoToList(boosterList, "Shimmerscale Drake", "AKH", "70");
        addCardInfoToList(boosterList, "Sigiled Starfish", "ORI", "73");
        addCardInfoToList(boosterList, "Skittering Crustacean", "CN2", "36");
        addCardInfoToList(boosterList, "Snap", "DDS", "10");
        addCardInfoToList(boosterList, "Snapping Drake", "M19", "75");
        addCardInfoToList(boosterList, "Somber Hoverguard", "MM2", "57");
        addCardInfoToList(boosterList, "Spire Monitor", "MM3", "52");
        addCardInfoToList(boosterList, "Steady Progress", "MM2", "58");
        addCardInfoToList(boosterList, "Stitched Drake", "DDQ", "49");
        addCardInfoToList(boosterList, "Storm Sculptor", "XLN", "85");
        addCardInfoToList(boosterList, "Strategic Planning", "HOU", "47");
        addCardInfoToList(boosterList, "Syncopate", "DOM", "67");
        addCardInfoToList(boosterList, "Tandem Lookout", "MM3", "53");
        addCardInfoToList(boosterList, "Temporal Fissure", "DDS", "12");
        addCardInfoToList(boosterList, "Thornwind Faeries", "CMA", "42");
        addCardInfoToList(boosterList, "Thought Scour", "IMA", "76");
        addCardInfoToList(boosterList, "Thoughtcast", "MM2", "64");
        addCardInfoToList(boosterList, "Thrummingbird", "CM2", "52");
        addCardInfoToList(boosterList, "Tidal Wave", "EMA", "75");
        addCardInfoToList(boosterList, "Totally Lost", "BBD", "135");
        addCardInfoToList(boosterList, "Treasure Hunt", "C18", "109");
        addCardInfoToList(boosterList, "Triton Tactics", "DDT", "23");
        addCardInfoToList(boosterList, "Vapor Snag", "MM2", "66");
        addCardInfoToList(boosterList, "Vigean Graftmage", "MM2", "68");
        addCardInfoToList(boosterList, "Wave-Wing Elemental", "BFZ", "88");
        addCardInfoToList(boosterList, "Whiplash Trap", "DDN", "70");
        addCardInfoToList(boosterList, "Windrider Eel", "E01", "30");
    }

    private void addSlot5blackA(List<CardInfo> boosterList) {
        addCardInfoToList(boosterList, "Aid the Fallen", "WAR", "76");
        addCardInfoToList(boosterList, "Alesha's Vanguard", "FRF", "60");
        addCardInfoToList(boosterList, "Alley Strangler", "AER", "52");
        addCardInfoToList(boosterList, "Ambitious Aetherborn", "KLD", "72");
        addCardInfoToList(boosterList, "Ancestral Vengeance", "FRF", "61");
        addCardInfoToList(boosterList, "Annihilate", "EMA", "79");
        addCardInfoToList(boosterList, "Bala Ged Scorpion", "IMA", "79");
        addCardInfoToList(boosterList, "Bitter Revelation", "KTK", "65");
        addCardInfoToList(boosterList, "Bladebrand", "M20", "86");
        addCardInfoToList(boosterList, "Blighted Bat", "AKH", "80");
        addCardInfoToList(boosterList, "Blistergrub", "DDR", "40");
        addCardInfoToList(boosterList, "Bone Splinters", "M20", "92");
        addCardInfoToList(boosterList, "Boon of Emrakul", "EMN", "81");
        addCardInfoToList(boosterList, "Breeding Pit", "DDC", "53");
        addCardInfoToList(boosterList, "Butcher's Glee", "IMA", "84");
        addCardInfoToList(boosterList, "Cabal Therapy", "EMA", "83");
        addCardInfoToList(boosterList, "Cackling Imp", "DDC", "41");
        addCardInfoToList(boosterList, "Cadaver Imp", "DDR", "41");
        addCardInfoToList(boosterList, "Catacomb Slug", "ORI", "86");
        addCardInfoToList(boosterList, "Certain Death", "EMN", "84");
        addCardInfoToList(boosterList, "Coat with Venom", "DTK", "91");
        addCardInfoToList(boosterList, "Corpsehatch", "DDP", "50");
        addCardInfoToList(boosterList, "Covenant of Blood", "M15", "91");
        addCardInfoToList(boosterList, "Crow of Dark Tidings", "SOI", "105");
        addCardInfoToList(boosterList, "Dark Dabbling", "ORI", "89");
        addCardInfoToList(boosterList, "Dark Withering", "C19", "110");
        addCardInfoToList(boosterList, "Darkblast", "GK1", "51");
        addCardInfoToList(boosterList, "Dead Reveler", "IMA", "86");
        addCardInfoToList(boosterList, "Deadeye Tormentor", "XLN", "98");
        addCardInfoToList(boosterList, "Defeat", "DTK", "97");
        addCardInfoToList(boosterList, "Demon's Grasp", "BFZ", "108");
        addCardInfoToList(boosterList, "Demonic Tutor", "DDC", "49");
        addCardInfoToList(boosterList, "Demonic Vigor", "DOM", "85");
        addCardInfoToList(boosterList, "Dismember", "MM2", "79");
        addCardInfoToList(boosterList, "Disowned Ancestor", "KTK", "70");
        addCardInfoToList(boosterList, "Doomed Dissenter", "AKH", "87");
        addCardInfoToList(boosterList, "Douse in Gloom", "FRF", "68");
        addCardInfoToList(boosterList, "Dread Return", "DDQ", "55");
        addCardInfoToList(boosterList, "Dregscape Zombie", "DDN", "5");
        addCardInfoToList(boosterList, "Dukhara Scavenger", "KLD", "77");
        addCardInfoToList(boosterList, "Dune Beetle", "AKH", "89");
        addCardInfoToList(boosterList, "Duress", "EMA", "86");
        addCardInfoToList(boosterList, "Farbog Revenant", "SOI", "110");
        addCardInfoToList(boosterList, "Fetid Imp", "ORI", "97");
        addCardInfoToList(boosterList, "First-Sphere Gargantua", "MH1", "91");
        addCardInfoToList(boosterList, "Flesh to Dust", "ORI", "280");
        addCardInfoToList(boosterList, "Fretwork Colony", "KLD", "83");
        addCardInfoToList(boosterList, "Genju of the Fens", "DDD", "47");
        addCardInfoToList(boosterList, "Ghoulcaller's Accomplice", "SOI", "112");
        addCardInfoToList(boosterList, "Grasping Scoundrel", "RIX", "74");
        addCardInfoToList(boosterList, "Gravepurge", "DTK", "104");
        addCardInfoToList(boosterList, "Grim Discovery", "DDR", "51");
        addCardInfoToList(boosterList, "Hideous End", "DDR", "52");
        addCardInfoToList(boosterList, "Induce Despair", "DDP", "53");
        addCardInfoToList(boosterList, "Infernal Scarring", "ORI", "102");
        addCardInfoToList(boosterList, "Infest", "CN2", "139");
        addCardInfoToList(boosterList, "Instill Infection", "MM2", "85");
        addCardInfoToList(boosterList, "Kalastria Nightwatch", "BFZ", "115");
        addCardInfoToList(boosterList, "Krumar Bond-Kin", "KTK", "77");
        addCardInfoToList(boosterList, "Lazotep Behemoth", "WAR", "95");
        addCardInfoToList(boosterList, "Macabre Waltz", "SOI", "121");
        addCardInfoToList(boosterList, "Marauding Boneslasher", "HOU", "70");
        addCardInfoToList(boosterList, "Mark of the Vampire", "UMA", "105");
        addCardInfoToList(boosterList, "Marsh Hulk", "DTK", "109");
        addCardInfoToList(boosterList, "Merciless Resolve", "SOI", "123");
        addCardInfoToList(boosterList, "Miasmic Mummy", "AKH", "100");
        addCardInfoToList(boosterList, "Mind Rake", "MH1", "96");
        addCardInfoToList(boosterList, "Mire's Malice", "BFZ", "117");
        addCardInfoToList(boosterList, "Murder", "M20", "109");
        addCardInfoToList(boosterList, "Murderous Compulsion", "SOI", "126");
        addCardInfoToList(boosterList, "Nantuko Husk", "ORI", "109");
        addCardInfoToList(boosterList, "Never Happened", "GRN", "80");
        addCardInfoToList(boosterList, "Nirkana Assassin", "BFZ", "118");
        addCardInfoToList(boosterList, "Plaguecrafter", "GRN", "82");
        addCardInfoToList(boosterList, "Prowling Pangolin", "EMA", "104");
        addCardInfoToList(boosterList, "Rakshasa's Secret", "KTK", "84");
        addCardInfoToList(boosterList, "Read the Bones", "DDP", "56");
        addCardInfoToList(boosterList, "Reaper of Night", "ELD", "102");
        addCardInfoToList(boosterList, "Reassembling Skeleton", "M19", "116");
        addCardInfoToList(boosterList, "Reckless Imp", "DTK", "115");
        addCardInfoToList(boosterList, "Reckless Spite", "E01", "37");
        addCardInfoToList(boosterList, "Returned Centaur", "ORI", "116");
        addCardInfoToList(boosterList, "Revenant", "ORI", "117");
        addCardInfoToList(boosterList, "Rite of the Serpent", "KTK", "86");
        addCardInfoToList(boosterList, "Ruin Rat", "HOU", "75");
        addCardInfoToList(boosterList, "Scrounger of Souls", "HOU", "76");
        addCardInfoToList(boosterList, "Sengir Vampire", "W17", "19");
        addCardInfoToList(boosterList, "Shambling Attendants", "KTK", "89");
        addCardInfoToList(boosterList, "Shambling Goblin", "DTK", "118");
        addCardInfoToList(boosterList, "Shriekmaw", "CMA", "68");
        addCardInfoToList(boosterList, "Silumgar Butcher", "DTK", "122");
        addCardInfoToList(boosterList, "Skeleton Archer", "M19", "118");
        addCardInfoToList(boosterList, "Stab Wound", "M15", "116");
        addCardInfoToList(boosterList, "Stallion of Ashmouth", "SOI", "136");
        addCardInfoToList(boosterList, "Stinkweed Imp", "GK1", "53");
        addCardInfoToList(boosterList, "Stromkirk Patrol", "CN2", "149");
        addCardInfoToList(boosterList, "Subtle Strike", "KLD", "100");
        addCardInfoToList(boosterList, "Sultai Runemark", "FRF", "86");
        addCardInfoToList(boosterList, "Tar Snare", "OGW", "90");
        addCardInfoToList(boosterList, "Thallid Omnivore", "DOM", "106");
        addCardInfoToList(boosterList, "The Eldest Reborn", "DOM", "90");
        addCardInfoToList(boosterList, "Thornbow Archer", "ORI", "121");
        addCardInfoToList(boosterList, "Thraben Foulbloods", "EMN", "108");
        addCardInfoToList(boosterList, "Torment of Venom", "HOU", "79");
        addCardInfoToList(boosterList, "Touch of Moonglove", "ORI", "123");
        addCardInfoToList(boosterList, "Twins of Maurer Estate", "SOI", "142");
        addCardInfoToList(boosterList, "Undercity's Embrace", "RNA", "89");
        addCardInfoToList(boosterList, "Untamed Hunger", "OGW", "91");
        addCardInfoToList(boosterList, "Unyielding Krumar", "KTK", "94");
        addCardInfoToList(boosterList, "Vampire Champion", "RIX", "198");
        addCardInfoToList(boosterList, "Vampire Envoy", "OGW", "92");
        addCardInfoToList(boosterList, "Vampire Nighthawk", "E02", "23");
        addCardInfoToList(boosterList, "Vessel of Malignity", "SOI", "144");
        addCardInfoToList(boosterList, "Voracious Null", "BFZ", "125");
        addCardInfoToList(boosterList, "Vraska's Finisher", "WAR", "112");
        addCardInfoToList(boosterList, "Walk the Plank", "XLN", "130");
        addCardInfoToList(boosterList, "Warteye Witch", "MH1", "115");
        addCardInfoToList(boosterList, "Weight of the Underworld", "ORI", "126");
        addCardInfoToList(boosterList, "Weirded Vampire", "EMN", "113");
        addCardInfoToList(boosterList, "Yargle, Glutton of Urborg", "DOM", "113");
        addCardInfoToList(boosterList, "Zulaport Chainmage", "OGW", "93");
    }

    private void addSlot6blackB(List<CardInfo> boosterList) {
        addCardInfoToList(boosterList, "Absorb Vis", "CN2", "126");
        addCardInfoToList(boosterList, "Accursed Spirit", "M15", "85");
        addCardInfoToList(boosterList, "Altar's Reap", "DDR", "37");
        addCardInfoToList(boosterList, "Animate Dead", "EMA", "78");
        addCardInfoToList(boosterList, "Baleful Ammit", "AKH", "79");
        addCardInfoToList(boosterList, "Balustrade Spy", "IMA", "80");
        addCardInfoToList(boosterList, "Bartizan Bats", "GRN", "62");
        addCardInfoToList(boosterList, "Black Cat", "M15", "86");
        addCardInfoToList(boosterList, "Blessing of Belzenlok", "DOM", "77");
        addCardInfoToList(boosterList, "Blightsoil Druid", "EMA", "80");
        addCardInfoToList(boosterList, "Blood Artist", "C17", "99");
        addCardInfoToList(boosterList, "Bloodrite Invoker", "DDP", "45");
        addCardInfoToList(boosterList, "Caligo Skin-Witch", "DOM", "80");
        addCardInfoToList(boosterList, "Carrion Feeder", "MH1", "81");
        addCardInfoToList(boosterList, "Carrion Imp", "RNA", "66");
        addCardInfoToList(boosterList, "Catacomb Crocodile", "RNA", "67");
        addCardInfoToList(boosterList, "Caustic Tar", "A25", "81");
        addCardInfoToList(boosterList, "Child of Night", "IMA", "85");
        addCardInfoToList(boosterList, "Costly Plunder", "XLN", "96");
        addCardInfoToList(boosterList, "Cower in Fear", "MM3", "62");
        addCardInfoToList(boosterList, "Crippling Blight", "M15", "92");
        addCardInfoToList(boosterList, "Cursed Minotaur", "AKH", "85");
        addCardInfoToList(boosterList, "Daring Demolition", "AER", "55");
        addCardInfoToList(boosterList, "Dark Ritual", "A25", "82");
        addCardInfoToList(boosterList, "Deadbridge Shaman", "EMA", "85");
        addCardInfoToList(boosterList, "Death Denied", "MM2", "76");
        addCardInfoToList(boosterList, "Desperate Castaways", "XLN", "101");
        addCardInfoToList(boosterList, "Diabolic Edict", "A25", "85");
        addCardInfoToList(boosterList, "Die Young", "KLD", "76");
        addCardInfoToList(boosterList, "Dinosaur Hunter", "RIX", "67");
        addCardInfoToList(boosterList, "Dirge of Dread", "A25", "86");
        addCardInfoToList(boosterList, "Dread Drone", "MM2", "80");
        addCardInfoToList(boosterList, "Dreadbringer Lampads", "C15", "122");
        addCardInfoToList(boosterList, "Driver of the Dead", "CN2", "133");
        addCardInfoToList(boosterList, "Drudge Sentinel", "DOM", "89");
        addCardInfoToList(boosterList, "Dusk Charger", "RIX", "69");
        addCardInfoToList(boosterList, "Dusk Legion Zealot", "RIX", "70");
        addCardInfoToList(boosterList, "Epicure of Blood", "M19", "95");
        addCardInfoToList(boosterList, "Erg Raiders", "A25", "90");
        addCardInfoToList(boosterList, "Eternal Thirst", "IMA", "89");
        addCardInfoToList(boosterList, "Evincar's Justice", "CMA", "58");
        addCardInfoToList(boosterList, "Executioner's Capsule", "C16", "109");
        addCardInfoToList(boosterList, "Eyeblight's Ending", "EMA", "88");
        addCardInfoToList(boosterList, "Fallen Angel", "A25", "91");
        addCardInfoToList(boosterList, "Fatal Push", "AER", "57");
        addCardInfoToList(boosterList, "Fen Hauler", "AER", "58");
        addCardInfoToList(boosterList, "Feral Abomination", "DOM", "92");
        addCardInfoToList(boosterList, "Festercreep", "CM2", "63");
        addCardInfoToList(boosterList, "Festering Newt", "IMA", "90");
        addCardInfoToList(boosterList, "Fill with Fright", "BBD", "144");
        addCardInfoToList(boosterList, "Fungal Infection", "DOM", "94");
        addCardInfoToList(boosterList, "Ghostly Changeling", "MM2", "83");
        addCardInfoToList(boosterList, "Gifted Aetherborn", "AER", "61");
        addCardInfoToList(boosterList, "Go for the Throat", "C17", "114");
        addCardInfoToList(boosterList, "Gravedigger", "CM2", "66");
        addCardInfoToList(boosterList, "Gray Merchant of Asphodel", "C14", "146");
        addCardInfoToList(boosterList, "Grim Affliction", "MM2", "84");
        addCardInfoToList(boosterList, "Grixis Slavedriver", "MM3", "74");
        addCardInfoToList(boosterList, "Grotesque Mutation", "BBD", "145");
        addCardInfoToList(boosterList, "Gruesome Fate", "RIX", "75");
        addCardInfoToList(boosterList, "Gurmag Angler", "UMA", "102");
        addCardInfoToList(boosterList, "Hired Blade", "M19", "100");
        addCardInfoToList(boosterList, "Hound of the Farbogs", "SOI", "117");
        addCardInfoToList(boosterList, "Innocent Blood", "EMA", "94");
        addCardInfoToList(boosterList, "Inquisition of Kozilek", "MM3", "75");
        addCardInfoToList(boosterList, "Lawless Broker", "KLD", "86");
        addCardInfoToList(boosterList, "Lethal Sting", "HOU", "67");
        addCardInfoToList(boosterList, "Lord of the Accursed", "AKH", "99");
        addCardInfoToList(boosterList, "March of the Drowned", "XLN", "112");
        addCardInfoToList(boosterList, "Mephitic Vapors", "GRN", "76");
        addCardInfoToList(boosterList, "Mind Rot", "W16", "7");
        addCardInfoToList(boosterList, "Moment of Craving", "RIX", "79");
        addCardInfoToList(boosterList, "Nameless Inversion", "MM2", "87");
        addCardInfoToList(boosterList, "Night's Whisper", "EMA", "100");
        addCardInfoToList(boosterList, "Noxious Dragon", "FRF", "77");
        addCardInfoToList(boosterList, "Okiba-Gang Shinobi", "PCA", "35");
        addCardInfoToList(boosterList, "Painful Lesson", "AKH", "102");
        addCardInfoToList(boosterList, "Phyrexian Rager", "CMA", "62");
        addCardInfoToList(boosterList, "Phyrexian Reclamation", "C15", "133");
        addCardInfoToList(boosterList, "Pit Keeper", "MM3", "81");
        addCardInfoToList(boosterList, "Plague Wight", "RNA", "82");
        addCardInfoToList(boosterList, "Plagued Rusalka", "MM2", "89");
        addCardInfoToList(boosterList, "Prakhata Club Security", "KLD", "98");
        addCardInfoToList(boosterList, "Queen's Agent", "XLN", "114");
        addCardInfoToList(boosterList, "Quest for the Gravelord", "BBD", "156");
        addCardInfoToList(boosterList, "Rabid Bloodsucker", "ORI", "113");
        addCardInfoToList(boosterList, "Rakdos Drake", "IMA", "103");
        addCardInfoToList(boosterList, "Ravenous Chupacabra", "RIX", "82");
        addCardInfoToList(boosterList, "Recover", "MM3", "82");
        addCardInfoToList(boosterList, "Renegade Demon", "DDR", "59");
        addCardInfoToList(boosterList, "Renegade's Getaway", "AER", "69");
        addCardInfoToList(boosterList, "Rotfeaster Maggot", "M15", "112");
        addCardInfoToList(boosterList, "Scarab Feast", "AKH", "106");
        addCardInfoToList(boosterList, "Scuttling Death", "MM2", "94");
        addCardInfoToList(boosterList, "Seal of Doom", "C15", "135");
        addCardInfoToList(boosterList, "Shadowcloak Vampire", "M15", "113");
        addCardInfoToList(boosterList, "Skeletal Scrying", "C14", "162");
        addCardInfoToList(boosterList, "Skulking Ghost", "EMA", "107");
        addCardInfoToList(boosterList, "Smiting Helix", "MH1", "109");
        addCardInfoToList(boosterList, "Spreading Rot", "XLN", "125");
        addCardInfoToList(boosterList, "Street Wraith", "A25", "108");
        addCardInfoToList(boosterList, "Tavern Swindler", "BBD", "162");
        addCardInfoToList(boosterList, "Tendrils of Corruption", "C14", "166");
        addCardInfoToList(boosterList, "Thorn of the Black Rose", "CN2", "48");
        addCardInfoToList(boosterList, "Tidy Conclusion", "KLD", "103");
        addCardInfoToList(boosterList, "Tragic Slip", "C14", "167");
        addCardInfoToList(boosterList, "Trespasser's Curse", "AKH", "112");
        addCardInfoToList(boosterList, "Trial of Ambition", "AKH", "113");
        addCardInfoToList(boosterList, "Typhoid Rats", "M15", "118");
        addCardInfoToList(boosterList, "Unburden", "AKH", "114");
        addCardInfoToList(boosterList, "Urborg Uprising", "EMA", "111");
        addCardInfoToList(boosterList, "Vampire Hexmage", "C14", "168");
        addCardInfoToList(boosterList, "Vampire Lacerator", "A25", "114");
        addCardInfoToList(boosterList, "Virulent Swipe", "IMA", "113");
        addCardInfoToList(boosterList, "Wake of Vultures", "EMA", "115");
        addCardInfoToList(boosterList, "Walking Corpse", "M19", "126");
        addCardInfoToList(boosterList, "Wander in Death", "AKH", "115");
        addCardInfoToList(boosterList, "Wight of Precinct Six", "C16", "118");
        addCardInfoToList(boosterList, "Will-o'-the-Wisp", "A25", "115");
        addCardInfoToList(boosterList, "Windgrace Acolyte", "DOM", "112");
        addCardInfoToList(boosterList, "Wrench Mind", "IMA", "115");
    }


    private void addSlot7redA(List<CardInfo> boosterList) {
        addCardInfoToList(boosterList, "Act on Impulse", "M15", "126");
        addCardInfoToList(boosterList, "Ainok Tracker", "KTK", "96");
        addCardInfoToList(boosterList, "Alchemist's Greeting", "EMN", "116");
        addCardInfoToList(boosterList, "Ancient Grudge", "MM3", "88");
        addCardInfoToList(boosterList, "Arc Trail", "PCA", "39");
        addCardInfoToList(boosterList, "Arrow Storm", "KTK", "98");
        addCardInfoToList(boosterList, "Azra Bladeseeker", "BBD", "55");
        addCardInfoToList(boosterList, "Balduvian Horde", "A25", "120");
        addCardInfoToList(boosterList, "Barrage of Boulders", "KTK", "100");
        addCardInfoToList(boosterList, "Beetleback Chief", "PCA", "40");
        addCardInfoToList(boosterList, "Bellows Lizard", "ORI", "132");
        addCardInfoToList(boosterList, "Blastfire Bolt", "M15", "130");
        addCardInfoToList(boosterList, "Blazing Volley", "AKH", "119");
        addCardInfoToList(boosterList, "Blindblast", "WAR", "114");
        addCardInfoToList(boosterList, "Blood Ogre", "E01", "41");
        addCardInfoToList(boosterList, "Bloodfire Expert", "KTK", "101");
        addCardInfoToList(boosterList, "Bloodlust Inciter", "AKH", "120");
        addCardInfoToList(boosterList, "Bloodstone Goblin", "DOM", "115");
        addCardInfoToList(boosterList, "Blow Your House Down", "ELD", "114");
        addCardInfoToList(boosterList, "Bombard", "GNT", "37");
        addCardInfoToList(boosterList, "Bomber Corps", "GK1", "80");
        addCardInfoToList(boosterList, "Borrowed Hostility", "EMN", "121");
        addCardInfoToList(boosterList, "Brazen Buccaneers", "XLN", "134");
        addCardInfoToList(boosterList, "Brazen Wolves", "EMN", "122");
        addCardInfoToList(boosterList, "Bring Low", "KTK", "103");
        addCardInfoToList(boosterList, "Brute Strength", "DDT", "35");
        addCardInfoToList(boosterList, "Built to Smash", "KLD", "108");
        addCardInfoToList(boosterList, "Burst Lightning", "MM2", "109");
        addCardInfoToList(boosterList, "Canyon Lurkers", "KTK", "105");
        addCardInfoToList(boosterList, "Chandra's Pyrohelix", "WAR", "120");
        addCardInfoToList(boosterList, "Charging Monstrosaur", "XLN", "138");
        addCardInfoToList(boosterList, "Cobblebrute", "ORI", "138");
        addCardInfoToList(boosterList, "Crowd's Favor", "M15", "138");
        addCardInfoToList(boosterList, "Crown-Hunter Hireling", "CN2", "50");
        addCardInfoToList(boosterList, "Curse of Opulence", "C17", "24");
        addCardInfoToList(boosterList, "Destructive Tampering", "AER", "78");
        addCardInfoToList(boosterList, "Direct Current", "GRN", "96");
        addCardInfoToList(boosterList, "Dragon Fodder", "GNT", "39");
        addCardInfoToList(boosterList, "Dynacharge", "MM3", "94");
        addCardInfoToList(boosterList, "Erratic Explosion", "PCA", "41");
        addCardInfoToList(boosterList, "Expedite", "BBD", "177");
        addCardInfoToList(boosterList, "Falkenrath Reaver", "W17", "21");
        addCardInfoToList(boosterList, "Fireball", "IMA", "128");
        addCardInfoToList(boosterList, "Flame Jab", "EMA", "131");
        addCardInfoToList(boosterList, "Forge Devil", "M15", "140");
        addCardInfoToList(boosterList, "Foundry Street Denizen", "M15", "141");
        addCardInfoToList(boosterList, "Frontline Rebel", "AER", "82");
        addCardInfoToList(boosterList, "Furnace Whelp", "IMA", "129");
        addCardInfoToList(boosterList, "Galvanic Blast", "DDU", "45");
        addCardInfoToList(boosterList, "Generator Servant", "M15", "143");
        addCardInfoToList(boosterList, "Geomancer's Gambit", "MH1", "125");
        addCardInfoToList(boosterList, "Ghitu Lavarunner", "DOM", "127");
        addCardInfoToList(boosterList, "Giant Spectacle", "KLD", "116");
        addCardInfoToList(boosterList, "Goblin Assault", "MM3", "95");
        addCardInfoToList(boosterList, "Goblin Bombardment", "DDN", "24");
        addCardInfoToList(boosterList, "Goblin Fireslinger", "MM2", "114");
        addCardInfoToList(boosterList, "Goblin Matron", "MH1", "129");
        addCardInfoToList(boosterList, "Goblin Roughrider", "M15", "146");
        addCardInfoToList(boosterList, "Goblin War Paint", "BFZ", "146");
        addCardInfoToList(boosterList, "Gore Swine", "FRF", "103");
        addCardInfoToList(boosterList, "Gorehorn Minotaurs", "E01", "49");
        addCardInfoToList(boosterList, "Granitic Titan", "HOU", "95");
        addCardInfoToList(boosterList, "Grapeshot", "DDS", "16");
        addCardInfoToList(boosterList, "Gravitic Punch", "GRN", "105");
        addCardInfoToList(boosterList, "Guttersnipe", "IMA", "131");
        addCardInfoToList(boosterList, "Hammerhand", "IMA", "132");
        addCardInfoToList(boosterList, "Hardened Berserker", "DTK", "139");
        addCardInfoToList(boosterList, "Hyena Pack", "AKH", "139");
        addCardInfoToList(boosterList, "Ill-Tempered Cyclops", "CN2", "166");
        addCardInfoToList(boosterList, "Impact Tremors", "DTK", "140");
        addCardInfoToList(boosterList, "Incorrigible Youths", "SOI", "166");
        addCardInfoToList(boosterList, "Inferno Fist", "M15", "150");
        addCardInfoToList(boosterList, "Inferno Jet", "HOU", "99");
        addCardInfoToList(boosterList, "Ingot Chewer", "CM2", "110");
        addCardInfoToList(boosterList, "Keldon Halberdier", "IMA", "135");
        addCardInfoToList(boosterList, "Kiln Fiend", "IMA", "137");
        addCardInfoToList(boosterList, "Krenko's Enforcer", "M15", "152");
        addCardInfoToList(boosterList, "Leaping Master", "KTK", "114");
        addCardInfoToList(boosterList, "Leopard-Spotted Jiao", "GS1", "23");
        addCardInfoToList(boosterList, "Madcap Skills", "MM3", "99");
        addCardInfoToList(boosterList, "Mardu Warshrieker", "KTK", "117");
        addCardInfoToList(boosterList, "Maximize Velocity", "GRN", "111");
        addCardInfoToList(boosterList, "Miner's Bane", "M15", "157");
        addCardInfoToList(boosterList, "Mogg Flunkies", "MM3", "102");
        addCardInfoToList(boosterList, "Molten Rain", "MM3", "103");
        addCardInfoToList(boosterList, "Monastery Swiftspear", "IMA", "140");
        addCardInfoToList(boosterList, "Ondu Champion", "BFZ", "149");
        addCardInfoToList(boosterList, "Outnumber", "BFZ", "150");
        addCardInfoToList(boosterList, "Price of Progress", "EMA", "141");
        addCardInfoToList(boosterList, "Pyrotechnics", "FRF", "111");
        addCardInfoToList(boosterList, "Quakefoot Cyclops", "MH1", "142");
        addCardInfoToList(boosterList, "Reckless Fireweaver", "KLD", "126");
        addCardInfoToList(boosterList, "Reckless Wurm", "UMA", "144");
        addCardInfoToList(boosterList, "Rivals' Duel", "PCA", "51");
        addCardInfoToList(boosterList, "Ruinous Gremlin", "KLD", "128");
        addCardInfoToList(boosterList, "Samut's Sprint", "WAR", "142");
        addCardInfoToList(boosterList, "Sarkhan's Rage", "DTK", "153");
        addCardInfoToList(boosterList, "Screamreach Brawler", "DTK", "155");
        addCardInfoToList(boosterList, "Seismic Shift", "DOM", "141");
        addCardInfoToList(boosterList, "Shattering Spree", "GK1", "34");
        addCardInfoToList(boosterList, "Shenanigans", "MH1", "146");
        addCardInfoToList(boosterList, "Smelt", "M19", "158");
        addCardInfoToList(boosterList, "Sparkmage Apprentice", "DDN", "48");
        addCardInfoToList(boosterList, "Sparkspitter", "UMA", "149");
        addCardInfoToList(boosterList, "Staggershock", "IMA", "147");
        addCardInfoToList(boosterList, "Stormblood Berserker", "E01", "58");
        addCardInfoToList(boosterList, "Swift Kick", "KTK", "122");
        addCardInfoToList(boosterList, "Tectonic Rift", "M19", "162");
        addCardInfoToList(boosterList, "Temur Battle Rage", "FRF", "116");
        addCardInfoToList(boosterList, "Thrill of Possibility", "ELD", "146");
        addCardInfoToList(boosterList, "Tibalt's Rager", "WAR", "147");
        addCardInfoToList(boosterList, "Torch Courier", "GRN", "119");
        addCardInfoToList(boosterList, "Valakut Invoker", "BFZ", "159");
        addCardInfoToList(boosterList, "Valakut Predator", "BFZ", "160");
        addCardInfoToList(boosterList, "Valley Dasher", "KTK", "125");
        addCardInfoToList(boosterList, "Vandalize", "DTK", "165");
        addCardInfoToList(boosterList, "Volcanic Dragon", "M19", "167");
        addCardInfoToList(boosterList, "Volcanic Rush", "DTK", "166");
        addCardInfoToList(boosterList, "Wall of Fire", "M15", "167");
        addCardInfoToList(boosterList, "Wayward Giant", "KLD", "139");
        addCardInfoToList(boosterList, "Wojek Bodyguard", "GRN", "120");
    }

    private void addSlot8redB(List<CardInfo> boosterList) {
        addCardInfoToList(boosterList, "Act of Treason", "RNA", "91");
        addCardInfoToList(boosterList, "Ahn-Crop Crasher", "AKH", "117");
        addCardInfoToList(boosterList, "Akroan Sergeant", "ORI", "130");
        addCardInfoToList(boosterList, "Anger", "UMA", "122");
        addCardInfoToList(boosterList, "Atarka Efreet", "DTK", "128");
        addCardInfoToList(boosterList, "Avarax", "EMA", "117");
        addCardInfoToList(boosterList, "Barging Sergeant", "GRN", "92");
        addCardInfoToList(boosterList, "Battle Rampart", "BBD", "165");
        addCardInfoToList(boosterList, "Battle-Rattle Shaman", "E01", "40");
        addCardInfoToList(boosterList, "Blades of Velis Vel", "MM2", "105");
        addCardInfoToList(boosterList, "Bloodmad Vampire", "SOI", "146");
        addCardInfoToList(boosterList, "Blur of Blades", "HOU", "84");
        addCardInfoToList(boosterList, "Boggart Brute", "ORI", "133");
        addCardInfoToList(boosterList, "Boiling Earth", "BFZ", "142");
        addCardInfoToList(boosterList, "Boulder Salvo", "OGW", "102");
        addCardInfoToList(boosterList, "Browbeat", "A25", "123");
        addCardInfoToList(boosterList, "Cartouche of Zeal", "AKH", "124");
        addCardInfoToList(boosterList, "Cathartic Reunion", "KLD", "109");
        addCardInfoToList(boosterList, "Chandra's Revolution", "AER", "77");
        addCardInfoToList(boosterList, "Chartooth Cougar", "A25", "125");
        addCardInfoToList(boosterList, "Cinder Hellion", "OGW", "105");
        addCardInfoToList(boosterList, "Cleansing Screech", "GS1", "37");
        addCardInfoToList(boosterList, "Cosmotronic Wave", "GRN", "95");
        addCardInfoToList(boosterList, "Crash Through", "M19", "133");
        addCardInfoToList(boosterList, "Curse of the Nightly Hunt", "CM2", "90");
        addCardInfoToList(boosterList, "Death by Dragons", "CMA", "80");
        addCardInfoToList(boosterList, "Defiant Ogre", "FRF", "96");
        addCardInfoToList(boosterList, "Demolish", "XLN", "139");
        addCardInfoToList(boosterList, "Desert Cerodon", "AKH", "128");
        addCardInfoToList(boosterList, "Desperate Ravings", "C15", "149");
        addCardInfoToList(boosterList, "Distemper of the Blood", "EMN", "126");
        addCardInfoToList(boosterList, "Dragon Breath", "BBD", "172");
        addCardInfoToList(boosterList, "Dragon Egg", "IMA", "124");
        addCardInfoToList(boosterList, "Dragon Whelp", "CMA", "81");
        addCardInfoToList(boosterList, "Dragonsoul Knight", "MM2", "112");
        addCardInfoToList(boosterList, "Dual Shot", "SOI", "153");
        addCardInfoToList(boosterList, "Earth Elemental", "BBD", "174");
        addCardInfoToList(boosterList, "Emrakul's Hatcher", "DDP", "59");
        addCardInfoToList(boosterList, "Enthralling Victor", "BBD", "176");
        addCardInfoToList(boosterList, "Faithless Looting", "CM2", "96");
        addCardInfoToList(boosterList, "Fall of the Hammer", "CM2", "97");
        addCardInfoToList(boosterList, "Fervent Strike", "DOM", "117");
        addCardInfoToList(boosterList, "Fierce Invocation", "FRF", "98");
        addCardInfoToList(boosterList, "Fiery Hellhound", "ORI", "284");
        addCardInfoToList(boosterList, "Fiery Temper", "SOI", "156");
        addCardInfoToList(boosterList, "Fire Elemental", "M19", "141");
        addCardInfoToList(boosterList, "Firebolt", "DDS", "37");
        addCardInfoToList(boosterList, "Firebrand Archer", "HOU", "92");
        addCardInfoToList(boosterList, "Flametongue Kavu", "E01", "48");
        addCardInfoToList(boosterList, "Flamewave Invoker", "BBD", "178");
        addCardInfoToList(boosterList, "Fling", "AKH", "132");
        addCardInfoToList(boosterList, "Frenzied Raptor", "XLN", "146");
        addCardInfoToList(boosterList, "Frilled Deathspitter", "RIX", "104");
        addCardInfoToList(boosterList, "Frontline Devastator", "HOU", "93");
        addCardInfoToList(boosterList, "Fury Charm", "IMA", "130");
        addCardInfoToList(boosterList, "Genju of the Spires", "A25", "132");
        addCardInfoToList(boosterList, "Goblin Balloon Brigade", "CN2", "159");
        addCardInfoToList(boosterList, "Goblin Locksmith", "GRN", "104");
        addCardInfoToList(boosterList, "Goblin Motivator", "M19", "143");
        addCardInfoToList(boosterList, "Goblin Oriflamme", "MH1", "130");
        addCardInfoToList(boosterList, "Goblin Warchief", "DOM", "130");
        addCardInfoToList(boosterList, "Gut Shot", "MM2", "117");
        addCardInfoToList(boosterList, "Hanweir Lancer", "MM3", "97");
        addCardInfoToList(boosterList, "Hijack", "XLN", "148");
        addCardInfoToList(boosterList, "Hulking Devil", "SOI", "165");
        addCardInfoToList(boosterList, "Insolent Neonate", "SOI", "168");
        addCardInfoToList(boosterList, "Jackal Pup", "A25", "139");
        addCardInfoToList(boosterList, "Keldon Overseer", "DOM", "134");
        addCardInfoToList(boosterList, "Khenra Scrapper", "HOU", "100");
        addCardInfoToList(boosterList, "Kird Ape", "EMA", "137");
        addCardInfoToList(boosterList, "Kolaghan Stormsinger", "DTK", "145");
        addCardInfoToList(boosterList, "Krenko's Command", "DDT", "53");
        addCardInfoToList(boosterList, "Lightning Bolt", "E01", "54");
        addCardInfoToList(boosterList, "Lightning Javelin", "ORI", "153");
        addCardInfoToList(boosterList, "Lightning Shrieker", "FRF", "106");
        addCardInfoToList(boosterList, "Lightning Talons", "BBD", "180");
        addCardInfoToList(boosterList, "Magma Spray", "AKH", "141");
        addCardInfoToList(boosterList, "Makindi Sliderunner", "BFZ", "148");
        addCardInfoToList(boosterList, "Mark of Mutiny", "PCA", "47");
        addCardInfoToList(boosterList, "Mogg Fanatic", "DD1", "44");
        addCardInfoToList(boosterList, "Mogg War Marshal", "EMA", "139");
        addCardInfoToList(boosterList, "Mutiny", "RIX", "106");
        addCardInfoToList(boosterList, "Nimble-Blade Khenra", "AKH", "145");
        addCardInfoToList(boosterList, "Orcish Cannonade", "DDN", "28");
        addCardInfoToList(boosterList, "Orcish Oriflamme", "EMA", "140");
        addCardInfoToList(boosterList, "Pillage", "A25", "144");
        addCardInfoToList(boosterList, "Prickleboar", "ORI", "158");
        addCardInfoToList(boosterList, "Prophetic Ravings", "EMN", "139");
        addCardInfoToList(boosterList, "Rampaging Cyclops", "DOM", "139");
        addCardInfoToList(boosterList, "Renegade Tactics", "KLD", "127");
        addCardInfoToList(boosterList, "Roast", "DTK", "151");
        addCardInfoToList(boosterList, "Rolling Thunder", "BFZ", "154");
        addCardInfoToList(boosterList, "Rubblebelt Maaka", "MM3", "109");
        addCardInfoToList(boosterList, "Rummaging Goblin", "XLN", "160");
        addCardInfoToList(boosterList, "Run Amok", "DOM", "140");
        addCardInfoToList(boosterList, "Rush of Adrenaline", "SOI", "177");
        addCardInfoToList(boosterList, "Salivating Gremlins", "KLD", "129");
        addCardInfoToList(boosterList, "Seismic Stomp", "EMA", "146");
        addCardInfoToList(boosterList, "Shatter", "RIX", "114");
        addCardInfoToList(boosterList, "Shock", "M19", "156");
        addCardInfoToList(boosterList, "Skirk Commando", "A25", "150");
        addCardInfoToList(boosterList, "Skirk Prospector", "DOM", "144");
        addCardInfoToList(boosterList, "Smash to Smithereens", "MM2", "124");
        addCardInfoToList(boosterList, "Sparktongue Dragon", "M19", "159");
        addCardInfoToList(boosterList, "Spikeshot Goblin", "A25", "152");
        addCardInfoToList(boosterList, "Sulfurous Blast", "CMA", "88");
        addCardInfoToList(boosterList, "Summit Prowler", "DTK", "160");
        addCardInfoToList(boosterList, "Sun-Crowned Hunters", "XLN", "164");
        addCardInfoToList(boosterList, "Swashbuckling", "XLN", "167");
        addCardInfoToList(boosterList, "Sweatworks Brawler", "AER", "100");
        addCardInfoToList(boosterList, "Tarfire", "DDT", "55");
        addCardInfoToList(boosterList, "Thresher Lizard", "AKH", "150");
        addCardInfoToList(boosterList, "Uncaged Fury", "A25", "155");
        addCardInfoToList(boosterList, "Undying Rage", "EMA", "152");
        addCardInfoToList(boosterList, "Vent Sentinel", "IMA", "153");
        addCardInfoToList(boosterList, "Vessel of Volatility", "SOI", "189");
        addCardInfoToList(boosterList, "Voldaren Duelist", "SOI", "191");
        addCardInfoToList(boosterList, "Wildfire Emissary", "EMA", "153");
        addCardInfoToList(boosterList, "Young Pyromancer", "DDS", "20");
        addCardInfoToList(boosterList, "Zada's Commando", "OGW", "120");
        addCardInfoToList(boosterList, "Zealot of the God-Pharaoh", "HOU", "207");
    }

    private void addSlot9greenA(List<CardInfo> boosterList) {
        addCardInfoToList(boosterList, "Affectionate Indrik", "GRN", "121");
        addCardInfoToList(boosterList, "Ancestral Mask", "EMA", "157");
        addCardInfoToList(boosterList, "Ancient Brontodon", "XLN", "175");
        addCardInfoToList(boosterList, "Arbor Armament", "DOM", "155");
        addCardInfoToList(boosterList, "Beastbreaker of Bala Ged", "DDP", "10");
        addCardInfoToList(boosterList, "Become Immense", "KTK", "130");
        addCardInfoToList(boosterList, "Blanchwood Armor", "M19", "169");
        addCardInfoToList(boosterList, "Blastoderm", "DDD", "7");
        addCardInfoToList(boosterList, "Borderland Explorer", "C18", "133");
        addCardInfoToList(boosterList, "Briarhorn", "DDR", "3");
        addCardInfoToList(boosterList, "Broodhunter Wurm", "BFZ", "171");
        addCardInfoToList(boosterList, "Byway Courier", "SOI", "196");
        addCardInfoToList(boosterList, "Centaur Courser", "M19", "171");
        addCardInfoToList(boosterList, "Creeping Mold", "KLD", "150");
        addCardInfoToList(boosterList, "Destructor Dragon", "FRF", "127");
        addCardInfoToList(boosterList, "Domesticated Hydra", "CN2", "63");
        addCardInfoToList(boosterList, "Dragon-Scarred Bear", "DTK", "183");
        addCardInfoToList(boosterList, "Elemental Uprising", "OGW", "130");
        addCardInfoToList(boosterList, "Elvish Fury", "MH1", "162");
        addCardInfoToList(boosterList, "Eternal Witness", "UMA", "163");
        addCardInfoToList(boosterList, "Feral Prowler", "HOU", "115");
        addCardInfoToList(boosterList, "Fierce Empath", "DDU", "10");
        addCardInfoToList(boosterList, "Frontier Mastodon", "FRF", "130");
        addCardInfoToList(boosterList, "Gaea's Blessing", "DOM", "161");
        addCardInfoToList(boosterList, "Gaea's Protector", "DOM", "162");
        addCardInfoToList(boosterList, "Gift of Growth", "DOM", "163");
        addCardInfoToList(boosterList, "Glade Watcher", "DTK", "188");
        addCardInfoToList(boosterList, "Grapple with the Past", "C18", "148");
        addCardInfoToList(boosterList, "Greater Basilisk", "IMA", "165");
        addCardInfoToList(boosterList, "Greater Sandwurm", "AKH", "168");
        addCardInfoToList(boosterList, "Hamlet Captain", "EMN", "161");
        addCardInfoToList(boosterList, "Hooded Brawler", "AKH", "173");
        addCardInfoToList(boosterList, "Hooting Mandrills", "KTK", "137");
        addCardInfoToList(boosterList, "Jungle Delver", "XLN", "195");
        addCardInfoToList(boosterList, "Jungle Wayfinder", "BBD", "72");
        addCardInfoToList(boosterList, "Kin-Tree Warden", "KTK", "139");
        addCardInfoToList(boosterList, "Kraul Foragers", "GRN", "135");
        addCardInfoToList(boosterList, "Krosan Druid", "DOM", "167");
        addCardInfoToList(boosterList, "Lead by Example", "BBD", "205");
        addCardInfoToList(boosterList, "Lead the Stampede", "DDU", "16");
        addCardInfoToList(boosterList, "Lifespring Druid", "BFZ", "177");
        addCardInfoToList(boosterList, "Lignify", "DDD", "16");
        addCardInfoToList(boosterList, "Llanowar Elves", "M19", "314");
        addCardInfoToList(boosterList, "Llanowar Empath", "DDU", "18");
        addCardInfoToList(boosterList, "Lure", "IMA", "175");
        addCardInfoToList(boosterList, "Mantle of Webs", "ORI", "187");
        addCardInfoToList(boosterList, "Map the Wastes", "FRF", "134");
        addCardInfoToList(boosterList, "Mulch", "CMA", "128");
        addCardInfoToList(boosterList, "Natural Connection", "DDR", "13");
        addCardInfoToList(boosterList, "Naturalize", "M19", "190");
        addCardInfoToList(boosterList, "Nature's Lore", "DDD", "17");
        addCardInfoToList(boosterList, "Nest Invader", "PCA", "69");
        addCardInfoToList(boosterList, "Nettle Sentinel", "A25", "182");
        addCardInfoToList(boosterList, "New Horizons", "XLN", "198");
        addCardInfoToList(boosterList, "Nimble Mongoose", "EMA", "179");
        addCardInfoToList(boosterList, "Ondu Giant", "PCA", "71");
        addCardInfoToList(boosterList, "Oran-Rief Invoker", "DDR", "17");
        addCardInfoToList(boosterList, "Overgrown Armasaur", "RIX", "141");
        addCardInfoToList(boosterList, "Pack's Favor", "GRN", "139");
        addCardInfoToList(boosterList, "Penumbra Spider", "MM3", "131");
        addCardInfoToList(boosterList, "Pierce the Sky", "DOM", "176");
        addCardInfoToList(boosterList, "Plummet", "RIX", "143");
        addCardInfoToList(boosterList, "Prey Upon", "GRN", "143");
        addCardInfoToList(boosterList, "Prey's Vengeance", "IMA", "182");
        addCardInfoToList(boosterList, "Pulse of Murasa", "OGW", "141");
        addCardInfoToList(boosterList, "Quiet Disrepair", "PCA", "75");
        addCardInfoToList(boosterList, "Rampant Growth", "DDS", "48");
        addCardInfoToList(boosterList, "Ranger's Guile", "M15", "193");
        addCardInfoToList(boosterList, "Ravenous Leucrocota", "CN2", "192");
        addCardInfoToList(boosterList, "Reclaim", "ORI", "195");
        addCardInfoToList(boosterList, "Revive", "MM3", "133");
        addCardInfoToList(boosterList, "Rhox Maulers", "ORI", "196");
        addCardInfoToList(boosterList, "Riparian Tiger", "KLD", "167");
        addCardInfoToList(boosterList, "Roar of the Wurm", "DDS", "49");
        addCardInfoToList(boosterList, "Root Out", "SOI", "224");
        addCardInfoToList(boosterList, "Rosethorn Halberd", "ELD", "175");
        addCardInfoToList(boosterList, "Runeclaw Bear", "M15", "197");
        addCardInfoToList(boosterList, "Sagu Archer", "KTK", "146");
        addCardInfoToList(boosterList, "Sakura-Tribe Elder", "C18", "160");
        addCardInfoToList(boosterList, "Saproling Migration", "DOM", "178");
        addCardInfoToList(boosterList, "Savage Punch", "KTK", "147");
        addCardInfoToList(boosterList, "Seal of Strength", "EMA", "184");
        addCardInfoToList(boosterList, "Search for Tomorrow", "IMA", "185");
        addCardInfoToList(boosterList, "Seek the Horizon", "DDR", "20");
        addCardInfoToList(boosterList, "Seek the Wilds", "BFZ", "189");
        addCardInfoToList(boosterList, "Shape the Sands", "DTK", "205");
        addCardInfoToList(boosterList, "Siege Wurm", "GRN", "144");
        addCardInfoToList(boosterList, "Silhana Ledgewalker", "PCA", "77");
        addCardInfoToList(boosterList, "Silkweaver Elite", "AER", "125");
        addCardInfoToList(boosterList, "Snake Umbra", "C18", "162");
        addCardInfoToList(boosterList, "Snapping Sailback", "XLN", "208");
        addCardInfoToList(boosterList, "Spider Spawning", "CMA", "149");
        addCardInfoToList(boosterList, "Stoic Builder", "SOI", "231");
        addCardInfoToList(boosterList, "Strength in Numbers", "MM3", "138");
        addCardInfoToList(boosterList, "Sylvan Bounty", "E01", "74");
        addCardInfoToList(boosterList, "Tajuru Pathwarden", "OGW", "145");
        addCardInfoToList(boosterList, "Take Down", "KLD", "170");
        addCardInfoToList(boosterList, "Talons of Wildwood", "M19", "202");
        addCardInfoToList(boosterList, "Territorial Baloth", "DDP", "24");
        addCardInfoToList(boosterList, "Thornhide Wolves", "M19", "204");
        addCardInfoToList(boosterList, "Thornweald Archer", "CMA", "154");
        addCardInfoToList(boosterList, "Thrive", "MM2", "166");
        addCardInfoToList(boosterList, "Timberwatch Elf", "EMA", "190");
        addCardInfoToList(boosterList, "Time to Feed", "DDO", "50");
        addCardInfoToList(boosterList, "Titanic Growth", "M19", "205");
        addCardInfoToList(boosterList, "Tukatongue Thallid", "PCA", "79");
        addCardInfoToList(boosterList, "Turntimber Basilisk", "E01", "76");
        addCardInfoToList(boosterList, "Vastwood Gorger", "ORI", "204");
        addCardInfoToList(boosterList, "Watcher in the Web", "SOI", "239");
        addCardInfoToList(boosterList, "Wellwisher", "CMA", "166");
        addCardInfoToList(boosterList, "Wild Growth", "C18", "165");
        addCardInfoToList(boosterList, "Wild Mongrel", "DDD", "5");
        addCardInfoToList(boosterList, "Wildsize", "IMA", "191");
        addCardInfoToList(boosterList, "Wolfkin Bond", "EMN", "178");
        addCardInfoToList(boosterList, "Woodborn Behemoth", "E01", "79");
        addCardInfoToList(boosterList, "Woolly Loxodon", "KTK", "158");
        addCardInfoToList(boosterList, "Wren's Run Vanquisher", "DD1", "19");
        addCardInfoToList(boosterList, "Yavimaya Elder", "C18", "166");
        addCardInfoToList(boosterList, "Yavimaya Sapherd", "DOM", "189");
        addCardInfoToList(boosterList, "Yeva's Forcemage", "ORI", "208");
        addCardInfoToList(boosterList, "Zendikar's Roil", "ORI", "209");
    }

    private void addSlot10greenB(List<CardInfo> boosterList) {
        addCardInfoToList(boosterList, "Abundant Growth", "EMA", "156");
        addCardInfoToList(boosterList, "Acidic Slime", "C18", "127");
        addCardInfoToList(boosterList, "Adventurous Impulse", "DOM", "153");
        addCardInfoToList(boosterList, "Aerie Bowmasters", "DTK", "170");
        addCardInfoToList(boosterList, "Aggressive Instinct", "GS1", "34");
        addCardInfoToList(boosterList, "Aggressive Urge", "RIX", "122");
        addCardInfoToList(boosterList, "Ainok Survivalist", "DTK", "172");
        addCardInfoToList(boosterList, "Alpine Grizzly", "KTK", "127");
        addCardInfoToList(boosterList, "Ambassador Oak", "A25", "158");
        addCardInfoToList(boosterList, "Ancient Stirrings", "A25", "159");
        addCardInfoToList(boosterList, "Arachnus Web", "MM3", "118");
        addCardInfoToList(boosterList, "Arbor Elf", "A25", "160");
        addCardInfoToList(boosterList, "Aura Gnarlid", "C18", "128");
        addCardInfoToList(boosterList, "Avacyn's Pilgrim", "MM3", "119");
        addCardInfoToList(boosterList, "Backwoods Survivalists", "EMN", "150");
        addCardInfoToList(boosterList, "Baloth Gorger", "DOM", "156");
        addCardInfoToList(boosterList, "Basking Rootwalla", "DDD", "2");
        addCardInfoToList(boosterList, "Beast Within", "BBD", "190");
        addCardInfoToList(boosterList, "Beneath the Sands", "HOU", "111");
        addCardInfoToList(boosterList, "Bestial Menace", "MM2", "141");
        addCardInfoToList(boosterList, "Bitterblade Warrior", "AKH", "157");
        addCardInfoToList(boosterList, "Bitterbow Sharpshooters", "HOU", "112");
        addCardInfoToList(boosterList, "Blossom Dryad", "XLN", "178");
        addCardInfoToList(boosterList, "Borderland Ranger", "E02", "31");
        addCardInfoToList(boosterList, "Bristling Boar", "M19", "170");
        addCardInfoToList(boosterList, "Broken Bond", "DOM", "157");
        addCardInfoToList(boosterList, "Canopy Spider", "BBD", "191");
        addCardInfoToList(boosterList, "Carnivorous Moss-Beast", "M15", "170");
        addCardInfoToList(boosterList, "Caustic Caterpillar", "ORI", "170");
        addCardInfoToList(boosterList, "Charging Rhino", "BBD", "192");
        addCardInfoToList(boosterList, "Citanul Woodreaders", "DDR", "4");
        addCardInfoToList(boosterList, "Clip Wings", "SOI", "197");
        addCardInfoToList(boosterList, "Colossal Dreadmaw", "XLN", "180");
        addCardInfoToList(boosterList, "Combo Attack", "BBD", "67");
        addCardInfoToList(boosterList, "Commune with Nature", "MM2", "142");
        addCardInfoToList(boosterList, "Commune with the Gods", "EMA", "162");
        addCardInfoToList(boosterList, "Conifer Strider", "DTK", "179");
        addCardInfoToList(boosterList, "Crop Rotation", "DDR", "7");
        addCardInfoToList(boosterList, "Crossroads Consecrator", "EMN", "154");
        addCardInfoToList(boosterList, "Crowned Ceratok", "IMA", "158");
        addCardInfoToList(boosterList, "Crushing Canopy", "GRN", "126");
        addCardInfoToList(boosterList, "Cultivate", "C18", "138");
        addCardInfoToList(boosterList, "Daggerback Basilisk", "M19", "174");
        addCardInfoToList(boosterList, "Dawn's Reflection", "C18", "139");
        addCardInfoToList(boosterList, "Death-Hood Cobra", "MM3", "123");
        addCardInfoToList(boosterList, "Desert Twister", "CMA", "100");
        addCardInfoToList(boosterList, "Dissenter's Deliverance", "AKH", "164");
        addCardInfoToList(boosterList, "Dragonscale Boon", "KTK", "131");
        addCardInfoToList(boosterList, "Durkwood Baloth", "IMA", "160");
        addCardInfoToList(boosterList, "Earthen Arms", "BFZ", "172");
        addCardInfoToList(boosterList, "Elephant Guide", "EMA", "163");
        addCardInfoToList(boosterList, "Elves of Deep Shadow", "GK1", "56");
        addCardInfoToList(boosterList, "Elvish Visionary", "BBD", "196");
        addCardInfoToList(boosterList, "Elvish Warrior", "DD1", "5");
        addCardInfoToList(boosterList, "Ember Weaver", "A25", "169");
        addCardInfoToList(boosterList, "Epic Confrontation", "DTK", "185");
        addCardInfoToList(boosterList, "Essence Warden", "CMA", "106");
        addCardInfoToList(boosterList, "Experiment One", "C15", "184");
        addCardInfoToList(boosterList, "Explore", "DDO", "45");
        addCardInfoToList(boosterList, "Explosive Vegetation", "C18", "144");
        addCardInfoToList(boosterList, "Ezuri's Archers", "DDU", "9");
        addCardInfoToList(boosterList, "Fade into Antiquity", "CN2", "181");
        addCardInfoToList(boosterList, "Farseek", "C17", "149");
        addCardInfoToList(boosterList, "Feed the Clan", "KTK", "132");
        addCardInfoToList(boosterList, "Feral Krushok", "FRF", "128");
        addCardInfoToList(boosterList, "Ferocious Zheng", "GS1", "28");
        addCardInfoToList(boosterList, "Fertile Ground", "C18", "147");
        addCardInfoToList(boosterList, "Fog", "EMA", "167");
        addCardInfoToList(boosterList, "Formless Nurturing", "FRF", "129");
        addCardInfoToList(boosterList, "Giant Growth", "BBD", "200");
        addCardInfoToList(boosterList, "Giant Spider", "AKH", "166");
        addCardInfoToList(boosterList, "Gift of Paradise", "AKH", "167");
        addCardInfoToList(boosterList, "Gnarlid Pack", "MM2", "144");
        addCardInfoToList(boosterList, "Grazing Gladehart", "DDP", "14");
        addCardInfoToList(boosterList, "Greenwood Sentinel", "M19", "187");
        addCardInfoToList(boosterList, "Groundswell", "DDP", "15");
        addCardInfoToList(boosterList, "Guardian Shield-Bearer", "DTK", "189");
        addCardInfoToList(boosterList, "Hardy Veteran", "RIX", "132");
        addCardInfoToList(boosterList, "Harmonize", "C17", "151");
        addCardInfoToList(boosterList, "Harrow", "C18", "150");
        addCardInfoToList(boosterList, "Hunt the Weak", "RIX", "133");
        addCardInfoToList(boosterList, "Hunter's Ambush", "M15", "180");
        addCardInfoToList(boosterList, "Imperious Perfect", "CMA", "118");
        addCardInfoToList(boosterList, "Invigorate", "A25", "173");
        addCardInfoToList(boosterList, "Ivy Lane Denizen", "DDU", "12");
        addCardInfoToList(boosterList, "Kavu Climber", "A25", "175");
        addCardInfoToList(boosterList, "Kavu Primarch", "MM2", "146");
        addCardInfoToList(boosterList, "Khalni Heart Expedition", "C18", "154");
        addCardInfoToList(boosterList, "Kozilek's Predator", "MM2", "147");
        addCardInfoToList(boosterList, "Kraul Warrior", "BBD", "204");
        addCardInfoToList(boosterList, "Krosan Tusker", "DDU", "14");
        addCardInfoToList(boosterList, "Larger Than Life", "KLD", "160");
        addCardInfoToList(boosterList, "Lay of the Land", "CN2", "185");
        addCardInfoToList(boosterList, "Longshot Squad", "KTK", "140");
        addCardInfoToList(boosterList, "Manglehorn", "AKH", "175");
        addCardInfoToList(boosterList, "Might of the Masses", "ORI", "188");
        addCardInfoToList(boosterList, "Nature's Claim", "IMA", "177");
        addCardInfoToList(boosterList, "Oakgnarl Warrior", "DDR", "15");
        addCardInfoToList(boosterList, "Overgrown Battlement", "IMA", "180");
        addCardInfoToList(boosterList, "Overrun", "CMA", "130");
        addCardInfoToList(boosterList, "Peema Outrider", "KLD", "166");
        addCardInfoToList(boosterList, "Pelakka Wurm", "MM2", "154");
        addCardInfoToList(boosterList, "Pinion Feast", "DTK", "195");
        addCardInfoToList(boosterList, "Pouncing Cheetah", "AKH", "179");
        addCardInfoToList(boosterList, "Priest of Titania", "C14", "210");
        addCardInfoToList(boosterList, "Rain of Thorns", "C17", "156");
        addCardInfoToList(boosterList, "Rancor", "A25", "186");
        addCardInfoToList(boosterList, "Reclaiming Vines", "BFZ", "185");
        addCardInfoToList(boosterList, "Regrowth", "A25", "187");
        addCardInfoToList(boosterList, "Relic Crush", "CM2", "142");
        addCardInfoToList(boosterList, "Return to the Earth", "BBD", "210");
        addCardInfoToList(boosterList, "Roots", "EMA", "183");
        addCardInfoToList(boosterList, "Scatter the Seeds", "GK1", "106");
        addCardInfoToList(boosterList, "Stalking Tiger", "W17", "28");
        addCardInfoToList(boosterList, "Sylvan Scrying", "BFZ", "192");
        addCardInfoToList(boosterList, "Tajuru Warcaller", "BFZ", "195");
        addCardInfoToList(boosterList, "Terrain Elemental", "KLD", "272");
        addCardInfoToList(boosterList, "The Crowd Goes Wild", "BBD", "68");
        addCardInfoToList(boosterList, "Thornscape Battlemage", "MM3", "142");
        addCardInfoToList(boosterList, "Thrashing Brontodon", "RIX", "148");
        addCardInfoToList(boosterList, "Venom Sliver", "M15", "205");
    }

    private void addSlot11multicolored(List<CardInfo> boosterList) {
        addCardInfoToList(boosterList, "Abzan Charm", "C16", "177");
        addCardInfoToList(boosterList, "Abzan Guide", "KTK", "162");
        addCardInfoToList(boosterList, "Agony Warp", "MM3", "150");
        addCardInfoToList(boosterList, "Akroan Hoplite", "CN2", "197");
        addCardInfoToList(boosterList, "Armadillo Cloak", "EMA", "195");
        addCardInfoToList(boosterList, "Armament Corps", "KTK", "165");
        addCardInfoToList(boosterList, "Azorius Charm", "IMA", "192");
        addCardInfoToList(boosterList, "Azra Oddsmaker", "BBD", "75");
        addCardInfoToList(boosterList, "Baleful Strix", "PCA", "82");
        addCardInfoToList(boosterList, "Baloth Null", "A25", "197");
        addCardInfoToList(boosterList, "Bear's Companion", "KTK", "167");
        addCardInfoToList(boosterList, "Belligerent Brontodon", "XLN", "218");
        addCardInfoToList(boosterList, "Bituminous Blast", "PCA", "83");
        addCardInfoToList(boosterList, "Bladewing the Risen", "IMA", "193");
        addCardInfoToList(boosterList, "Blightning", "A25", "198");
        addCardInfoToList(boosterList, "Bloodbraid Elf", "PCA", "84");
        addCardInfoToList(boosterList, "Boros Challenger", "GRN", "156");
        addCardInfoToList(boosterList, "Bounding Krasis", "ORI", "212");
        addCardInfoToList(boosterList, "Call of the Nightwing", "GK1", "8");
        addCardInfoToList(boosterList, "Campaign of Vengeance", "EMN", "182");
        addCardInfoToList(boosterList, "Cauldron Dance", "C17", "166");
        addCardInfoToList(boosterList, "Citadel Castellan", "ORI", "213");
        addCardInfoToList(boosterList, "Claim // Fame", "HOU", "150");
        addCardInfoToList(boosterList, "Coiling Oracle", "MM3", "157");
        addCardInfoToList(boosterList, "Contraband Kingpin", "KLD", "177");
        addCardInfoToList(boosterList, "Corpsejack Menace", "IMA", "197");
        addCardInfoToList(boosterList, "Crosis's Charm", "C17", "169");
        addCardInfoToList(boosterList, "Cunning Breezedancer", "DTK", "215");
        addCardInfoToList(boosterList, "Deathreap Ritual", "C18", "174");
        addCardInfoToList(boosterList, "Deny Reality", "PCA", "85");
        addCardInfoToList(boosterList, "Draconic Disciple", "M19", "215");
        addCardInfoToList(boosterList, "Drana's Emissary", "BFZ", "210");
        addCardInfoToList(boosterList, "Engineered Might", "KLD", "181");
        addCardInfoToList(boosterList, "Esper Charm", "C18", "179");
        addCardInfoToList(boosterList, "Ethercaste Knight", "MM2", "175");
        addCardInfoToList(boosterList, "Ethereal Ambush", "FRF", "152");
        addCardInfoToList(boosterList, "Extract from Darkness", "E01", "84");
        addCardInfoToList(boosterList, "Fire // Ice", "UMA", "225");
        addCardInfoToList(boosterList, "Fires of Yavimaya", "PCA", "92");
        addCardInfoToList(boosterList, "Flame-Kin Zealot", "EMA", "201");
        addCardInfoToList(boosterList, "Fusion Elemental", "PCA", "93");
        addCardInfoToList(boosterList, "Gelectrode", "GK1", "38");
        addCardInfoToList(boosterList, "Ghor-Clan Rampager", "MM3", "165");
        addCardInfoToList(boosterList, "Giantbaiting", "MM3", "208");
        addCardInfoToList(boosterList, "Gift of Orzhova", "MM3", "209");
        addCardInfoToList(boosterList, "Goblin Deathraiders", "DDN", "6");
        addCardInfoToList(boosterList, "Grim Contest", "FRF", "153");
        addCardInfoToList(boosterList, "Gwyllion Hedge-Mage", "CMA", "201");
        addCardInfoToList(boosterList, "Hammer Dropper", "GRN", "176");
        addCardInfoToList(boosterList, "Hidden Stockpile", "AER", "129");
        addCardInfoToList(boosterList, "Highspire Mantis", "KTK", "177");
        addCardInfoToList(boosterList, "Hypothesizzle", "GRN", "178");
        addCardInfoToList(boosterList, "Iroas's Champion", "ORI", "214");
        addCardInfoToList(boosterList, "Join Shields", "GRN", "181");
        addCardInfoToList(boosterList, "Jungle Barrier", "E02", "38");
        addCardInfoToList(boosterList, "Kathari Remnant", "PCA", "98");
        addCardInfoToList(boosterList, "Kin-Tree Invocation", "KTK", "183");
        addCardInfoToList(boosterList, "Kiora's Follower", "DDO", "52");
        addCardInfoToList(boosterList, "Kiss of the Amesha", "BBD", "225");
        addCardInfoToList(boosterList, "Lawmage's Binding", "RNA", "190");
        addCardInfoToList(boosterList, "Lightning Helix", "GK1", "90");
        addCardInfoToList(boosterList, "Mardu Roughrider", "KTK", "187");
        addCardInfoToList(boosterList, "Martial Glory", "GK1", "91");
        addCardInfoToList(boosterList, "Maverick Thopterist", "C18", "185");
        addCardInfoToList(boosterList, "Mercurial Geists", "EMN", "186");
        addCardInfoToList(boosterList, "Migratory Route", "CM2", "161");
        addCardInfoToList(boosterList, "Mistmeadow Witch", "CMA", "203");
        addCardInfoToList(boosterList, "Mortify", "C18", "186");
        addCardInfoToList(boosterList, "Naya Charm", "C16", "214");
        addCardInfoToList(boosterList, "Nucklavee", "DDS", "26");
        addCardInfoToList(boosterList, "Obelisk Spider", "HOU", "141");
        addCardInfoToList(boosterList, "Ochran Assassin", "GRN", "194");
        addCardInfoToList(boosterList, "Pillory of the Sleepless", "A25", "213");
        addCardInfoToList(boosterList, "Plaxcaster Frogling", "MM2", "184");
        addCardInfoToList(boosterList, "Pollenbright Wings", "GK1", "115");
        addCardInfoToList(boosterList, "Putrefy", "GK1", "68");
        addCardInfoToList(boosterList, "Qasali Pridemage", "C17", "189");
        addCardInfoToList(boosterList, "Raff Capashen, Ship's Mage", "DOM", "202");
        addCardInfoToList(boosterList, "Raging Swordtooth", "XLN", "226");
        addCardInfoToList(boosterList, "Reclusive Artificer", "DDU", "51");
        addCardInfoToList(boosterList, "Reflector Mage", "OGW", "157");
        addCardInfoToList(boosterList, "Rhox War Monk", "MM3", "180");
        addCardInfoToList(boosterList, "Riptide Crab", "BBD", "228");
        addCardInfoToList(boosterList, "River Hoopoe", "HOU", "143");
        addCardInfoToList(boosterList, "Rosemane Centaur", "GRN", "197");
        addCardInfoToList(boosterList, "Rosheen Meanderer", "IMA", "206");
        addCardInfoToList(boosterList, "Satyr Enchanter", "M19", "223");
        addCardInfoToList(boosterList, "Savage Twister", "C18", "190");
        addCardInfoToList(boosterList, "Sedraxis Specter", "MM3", "181");
        addCardInfoToList(boosterList, "Selesnya Guildmage", "GK1", "119");
        addCardInfoToList(boosterList, "Shambling Remains", "DDN", "12");
        addCardInfoToList(boosterList, "Shardless Agent", "PCA", "104");
        addCardInfoToList(boosterList, "Shipwreck Singer", "CN2", "206");
        addCardInfoToList(boosterList, "Shrewd Hatchling", "MM2", "198");
        addCardInfoToList(boosterList, "Skyward Eye Prophets", "CMA", "193");
        addCardInfoToList(boosterList, "Slave of Bolas", "E01", "86");
        addCardInfoToList(boosterList, "Soul Manipulation", "MM3", "185");
        addCardInfoToList(boosterList, "Sprouting Thrinax", "MM3", "189");
        addCardInfoToList(boosterList, "Stormchaser Chimera", "CN2", "207");
        addCardInfoToList(boosterList, "Sultai Charm", "KTK", "204");
        addCardInfoToList(boosterList, "Sultai Soothsayer", "KTK", "205");
        addCardInfoToList(boosterList, "Tatyova, Benthic Druid", "DOM", "206");
        addCardInfoToList(boosterList, "Terminate", "CMA", "195");
        addCardInfoToList(boosterList, "Thopter Foundry", "C16", "237");
        addCardInfoToList(boosterList, "Thought Erasure", "GRN", "206");
        addCardInfoToList(boosterList, "Tithe Drinker", "C17", "200");
        addCardInfoToList(boosterList, "Tower Gargoyle", "MM3", "196");
        addCardInfoToList(boosterList, "Treacherous Terrain", "C16", "47");
        addCardInfoToList(boosterList, "Underworld Coinsmith", "C15", "237");
        addCardInfoToList(boosterList, "Unflinching Courage", "C18", "192");
        addCardInfoToList(boosterList, "Unlicensed Disintegration", "KLD", "187");
        addCardInfoToList(boosterList, "Urban Evolution", "MM3", "198");
        addCardInfoToList(boosterList, "Vengeful Rebirth", "MM2", "188");
        addCardInfoToList(boosterList, "Warden of the Eye", "KTK", "212");
        addCardInfoToList(boosterList, "Wayfaring Temple", "MM3", "202");
        addCardInfoToList(boosterList, "Weapons Trainer", "OGW", "160");
        addCardInfoToList(boosterList, "Wee Dragonauts", "GRN", "214");
        addCardInfoToList(boosterList, "Winding Constrictor", "AER", "140");
        addCardInfoToList(boosterList, "Woolly Thoctar", "MM3", "203");
        addCardInfoToList(boosterList, "Zealous Persecution", "E02", "41");
        addCardInfoToList(boosterList, "Zhur-Taa Druid", "C16", "232");
    }

    private void addSlot12colorless(List<CardInfo> boosterList) {
        addCardInfoToList(boosterList, "Aether Hub", "KLD", "242");
        addCardInfoToList(boosterList, "Aether Spellbomb", "MMA", "196");
        addCardInfoToList(boosterList, "Akoum Refuge", "CMA", "238");
        addCardInfoToList(boosterList, "Alchemist's Vial", "ORI", "220");
        addCardInfoToList(boosterList, "Alloy Myr", "MM2", "201");
        addCardInfoToList(boosterList, "Arcane Sanctum", "C18", "232");
        addCardInfoToList(boosterList, "Armillary Sphere", "C17", "203");
        addCardInfoToList(boosterList, "Artisan of Kozilek", "CM2", "14");
        addCardInfoToList(boosterList, "Ash Barrens", "CM2", "235");
        addCardInfoToList(boosterList, "Ashnod's Altar", "EMA", "218");
        addCardInfoToList(boosterList, "Benthic Infiltrator", "BFZ", "55");
        addCardInfoToList(boosterList, "Blasted Landscape", "CM2", "238");
        addCardInfoToList(boosterList, "Blighted Fen", "BFZ", "230");
        addCardInfoToList(boosterList, "Blinding Souleater", "MM2", "202");
        addCardInfoToList(boosterList, "Blossoming Sands", "C18", "237");
        addCardInfoToList(boosterList, "Bojuka Bog", "C18", "238");
        addCardInfoToList(boosterList, "Bomat Bazaar Barge", "KLD", "198");
        addCardInfoToList(boosterList, "Bone Saw", "OGW", "161");
        addCardInfoToList(boosterList, "Bottle Gnomes", "CM2", "177");
        addCardInfoToList(boosterList, "Breaker of Armies", "BFZ", "3");
        addCardInfoToList(boosterList, "Burnished Hart", "C15", "248");
        addCardInfoToList(boosterList, "Call the Scions", "BFZ", "165");
        addCardInfoToList(boosterList, "Cathodion", "CM2", "179");
        addCardInfoToList(boosterList, "Coldsteel Heart", "CM2", "181");
        addCardInfoToList(boosterList, "Consulate Dreadnought", "AER", "146");
        addCardInfoToList(boosterList, "Copper Carapace", "MM2", "205");
        addCardInfoToList(boosterList, "Crumbling Necropolis", "C17", "244");
        addCardInfoToList(boosterList, "Crystal Ball", "C18", "201");
        addCardInfoToList(boosterList, "Crystal Chimes", "C15", "250");
        addCardInfoToList(boosterList, "Darksteel Citadel", "MM2", "238");
        addCardInfoToList(boosterList, "Diamond Mare", "M19", "231");
        addCardInfoToList(boosterList, "Dismal Backwater", "KTK", "232");
        addCardInfoToList(boosterList, "Dreadship Reef", "CM2", "247");
        addCardInfoToList(boosterList, "Eldrazi Devastator", "BFZ", "7");
        addCardInfoToList(boosterList, "Emmessi Tome", "EMA", "221");
        addCardInfoToList(boosterList, "Etched Oracle", "C16", "252");
        addCardInfoToList(boosterList, "Evolving Wilds", "C18", "245");
        addCardInfoToList(boosterList, "Faerie Conclave", "CMA", "248");
        addCardInfoToList(boosterList, "Farmstead Gleaner", "MH1", "222");
        addCardInfoToList(boosterList, "Field of Ruin", "XLN", "254");
        addCardInfoToList(boosterList, "Filigree Familiar", "GNT", "52");
        addCardInfoToList(boosterList, "Flayer Husk", "PCA", "110");
        addCardInfoToList(boosterList, "Forgotten Cave", "C18", "246");
        addCardInfoToList(boosterList, "Foundry Inspector", "KLD", "215");
        addCardInfoToList(boosterList, "Fountain of Renewal", "M19", "235");
        addCardInfoToList(boosterList, "Frogmite", "MM2", "215");
        addCardInfoToList(boosterList, "Frontier Bivouac", "C17", "251");
        addCardInfoToList(boosterList, "Gateway Plaza", "GRN", "247");
        addCardInfoToList(boosterList, "Ghost Quarter", "CM2", "253");
        addCardInfoToList(boosterList, "Goblin Burrows", "DD1", "58");
        addCardInfoToList(boosterList, "Graypelt Refuge", "C17", "253");
        addCardInfoToList(boosterList, "Great Furnace", "C18", "250");
        addCardInfoToList(boosterList, "Gruul Signet", "C16", "256");
        addCardInfoToList(boosterList, "Guardians of Meletis", "ORI", "228");
        addCardInfoToList(boosterList, "Heavy Arbalest", "A25", "225");
        addCardInfoToList(boosterList, "Herald's Horn", "C17", "53");
        addCardInfoToList(boosterList, "Hexplate Golem", "BBD", "237");
        addCardInfoToList(boosterList, "Hot Soup", "M15", "219");
        addCardInfoToList(boosterList, "Icy Manipulator", "DOM", "219");
        addCardInfoToList(boosterList, "Implement of Malice", "AER", "159");
        addCardInfoToList(boosterList, "Irontread Crusher", "AER", "161");
        addCardInfoToList(boosterList, "Juggernaut", "BBD", "238");
        addCardInfoToList(boosterList, "Jungle Hollow", "KTK", "235");
        addCardInfoToList(boosterList, "Jungle Shrine", "C17", "257");
        addCardInfoToList(boosterList, "Kazandu Refuge", "C18", "261");
        addCardInfoToList(boosterList, "Krosan Verge", "C18", "263");
        addCardInfoToList(boosterList, "Lightning Greaves", "C19", "217");
        addCardInfoToList(boosterList, "Loxodon Warhammer", "C17", "216");
        addCardInfoToList(boosterList, "Mask of Memory", "C14", "249");
        addCardInfoToList(boosterList, "Meteorite", "ORI", "233");
        addCardInfoToList(boosterList, "Millikin", "EMA", "226");
        addCardInfoToList(boosterList, "Millstone", "M19", "242");
        addCardInfoToList(boosterList, "Mind Stone", "C18", "210");
        addCardInfoToList(boosterList, "Mishra's Bauble", "IMA", "221");
        addCardInfoToList(boosterList, "Mishra's Factory", "A25", "242");
        addCardInfoToList(boosterList, "Moonglove Extract", "IMA", "222");
        addCardInfoToList(boosterList, "Mortarpod", "MM2", "222");
        addCardInfoToList(boosterList, "Myr Retriever", "CM2", "203");
        addCardInfoToList(boosterList, "Myr Sire", "CM2", "204");
        addCardInfoToList(boosterList, "New Benalia", "C18", "270");
        addCardInfoToList(boosterList, "Ornithopter", "M15", "223");
        addCardInfoToList(boosterList, "Orzhov Basilica", "C17", "268");
        addCardInfoToList(boosterList, "Palladium Myr", "CM2", "207");
        addCardInfoToList(boosterList, "Peace Strider", "BBD", "243");
        addCardInfoToList(boosterList, "Perilous Myr", "A25", "227");
        addCardInfoToList(boosterList, "Pilgrim's Eye", "GNT", "55");
        addCardInfoToList(boosterList, "Prophetic Prism", "A25", "229");
        addCardInfoToList(boosterList, "Reliquary Tower", "M19", "254");
        addCardInfoToList(boosterList, "Renegade Map", "AER", "173");
        addCardInfoToList(boosterList, "Rhonas's Monument", "AKH", "236");
        addCardInfoToList(boosterList, "Rogue's Passage", "C17", "272");
        addCardInfoToList(boosterList, "Sandsteppe Citadel", "CM2", "264");
        addCardInfoToList(boosterList, "Sandstone Oracle", "CM2", "213");
        addCardInfoToList(boosterList, "Scoured Barrens", "C18", "276");
        addCardInfoToList(boosterList, "Sejiri Refuge", "C18", "280");
        addCardInfoToList(boosterList, "Serrated Arrows", "DDD", "20");
        addCardInfoToList(boosterList, "Short Sword", "DOM", "229");
        addCardInfoToList(boosterList, "Sigil of Valor", "ORI", "239");
        addCardInfoToList(boosterList, "Simic Locket", "RNA", "240");
        addCardInfoToList(boosterList, "Skarrg, the Rage Pits", "PCA", "127");
        addCardInfoToList(boosterList, "Skullclamp", "C17", "222");
        addCardInfoToList(boosterList, "Skyscanner", "M19", "245");
        addCardInfoToList(boosterList, "Sol Ring", "C18", "222");
        addCardInfoToList(boosterList, "Sorcerer's Broom", "ELD", "232");
        //addCardInfoToList(boosterList, "Spy Kit", "CN2", "79"); // not yet implemented
        addCardInfoToList(boosterList, "Sunset Pyramid", "HOU", "166");
        addCardInfoToList(boosterList, "Suspicious Bookcase", "M19", "246");
        addCardInfoToList(boosterList, "Swiftwater Cliffs", "C18", "284");
        addCardInfoToList(boosterList, "Tectonic Edge", "C14", "313");
        addCardInfoToList(boosterList, "Temple of the False God", "C18", "285");
        addCardInfoToList(boosterList, "Thornwood Falls", "C18", "287");
        addCardInfoToList(boosterList, "Thought Vessel", "CM2", "224");
        addCardInfoToList(boosterList, "Thran Dynamo", "IMA", "230");
        addCardInfoToList(boosterList, "Thran Golem", "PCA", "114");
        addCardInfoToList(boosterList, "Tormod's Crypt", "C14", "278");
        addCardInfoToList(boosterList, "Trepanation Blade", "IMA", "231");
        addCardInfoToList(boosterList, "Unclaimed Territory", "XLN", "258");
        addCardInfoToList(boosterList, "Universal Automaton", "MH1", "235");
        addCardInfoToList(boosterList, "Universal Solvent", "AER", "178");
        addCardInfoToList(boosterList, "Whispersilk Cloak", "PCA", "115");
        addCardInfoToList(boosterList, "Wirewood Lodge", "DD1", "26");
    }

    private void addSlot13oldFrame(List<CardInfo> boosterList) {
        addCardInfoToList(boosterList, "Ana Sanctuary", "APC", "74");
        addCardInfoToList(boosterList, "Ancient Den", "MRD", "278");
        addCardInfoToList(boosterList, "Ancient Ziggurat", "CON", "141");
        addCardInfoToList(boosterList, "Angelic Destiny", "M12", "3");
        addCardInfoToList(boosterList, "Archangel", "C13", "5");
        addCardInfoToList(boosterList, "Asceticism", "SOM", "110");
        addCardInfoToList(boosterList, "Assemble the Legion", "GTC", "142");
        addCardInfoToList(boosterList, "Athreos, God of Passage", "JOU", "146");
        addCardInfoToList(boosterList, "Aura Shards", "CMD", "182");
        addCardInfoToList(boosterList, "Avalanche Riders", "ULG", "74");
        addCardInfoToList(boosterList, "Bear Cub", "P02", "123");
        addCardInfoToList(boosterList, "Belbe's Portal", "NEM", "127");
        addCardInfoToList(boosterList, "Black Knight", "M10", "85");
        addCardInfoToList(boosterList, "Bloom Tender", "EVE", "66");
        addCardInfoToList(boosterList, "Bonesplitter", "MRD", "146");
        addCardInfoToList(boosterList, "Bow of Nylea", "THS", "153");
        addCardInfoToList(boosterList, "Brimstone Dragon", "P02", "92");
        addCardInfoToList(boosterList, "Brimstone Mage", "ROE", "137");
        addCardInfoToList(boosterList, "Cairn Wanderer", "LRW", "105");
        addCardInfoToList(boosterList, "Carpet of Flowers", "USG", "240");
        addCardInfoToList(boosterList, "Centaur Glade", "ONS", "251");
        addCardInfoToList(boosterList, "Chancellor of the Annex", "NPH", "6");
        addCardInfoToList(boosterList, "Chatter of the Squirrel", "ODY", "233");
        addCardInfoToList(boosterList, "Chromatic Star", "TSP", "251");
        addCardInfoToList(boosterList, "Contagion Clasp", "SOM", "144");
        addCardInfoToList(boosterList, "Corrupted Conscience", "MBS", "22");
        addCardInfoToList(boosterList, "Cragganwick Cremator", "SHM", "87");
        addCardInfoToList(boosterList, "Crenellated Wall", "MMQ", "290");
        addCardInfoToList(boosterList, "Crystal Shard", "MRD", "159");
        addCardInfoToList(boosterList, "Darksteel Garrison", "FUT", "167");
        addCardInfoToList(boosterList, "Dauthi Mindripper", "TMP", "125");
        addCardInfoToList(boosterList, "Defense of the Heart", "ULG", "100");
        addCardInfoToList(boosterList, "Dictate of Erebos", "JOU", "65");
        addCardInfoToList(boosterList, "Dolmen Gate", "LRW", "256");
        addCardInfoToList(boosterList, "Dominus of Fealty", "CMD", "194");
        addCardInfoToList(boosterList, "Doomgape", "DDJ", "65");
        addCardInfoToList(boosterList, "Draco", "PLS", "131");
        addCardInfoToList(boosterList, "Dragon Broodmother", "ARB", "53");
        addCardInfoToList(boosterList, "Dragon Mask", "VIS", "144");
        addCardInfoToList(boosterList, "Dungrove Elder", "M12", "171");
        addCardInfoToList(boosterList, "Eater of Days", "DST", "120");
        addCardInfoToList(boosterList, "Elixir of Immortality", "M11", "206");
        addCardInfoToList(boosterList, "Empyrial Armor", "WTH", "13");
        addCardInfoToList(boosterList, "Enchanted Evening", "SHM", "140");
        addCardInfoToList(boosterList, "Energy Field", "USG", "73");
        addCardInfoToList(boosterList, "Exsanguinate", "SOM", "61");
        addCardInfoToList(boosterList, "Flameshot", "PCY", "90");
        addCardInfoToList(boosterList, "Floodgate", "MIR", "67");
        addCardInfoToList(boosterList, "Font of Mythos", "CON", "136");
        addCardInfoToList(boosterList, "Ghitu War Cry", "ULG", "78");
        addCardInfoToList(boosterList, "Gilt-Leaf Palace", "LRW", "268");
        addCardInfoToList(boosterList, "Goblin Game", "PLS", "61");
        addCardInfoToList(boosterList, "Greater Gargadon", "MMA", "117");
        addCardInfoToList(boosterList, "Guided Passage", "APC", "105");
        addCardInfoToList(boosterList, "Haakon, Stromgald Scourge", "CSP", "61");
        addCardInfoToList(boosterList, "Hedron Crab", "ZEN", "47");
        addCardInfoToList(boosterList, "Helm of Awakening", "VIS", "145");
        addCardInfoToList(boosterList, "Hunter of Eyeblights", "LRW", "119");
        addCardInfoToList(boosterList, "Hurricane", "10E", "270");
        addCardInfoToList(boosterList, "Hypnotic Specter", "M10", "100");
        addCardInfoToList(boosterList, "Impending Disaster", "ULG", "82");
        addCardInfoToList(boosterList, "Jushi Apprentice", "CHK", "70");
        addCardInfoToList(boosterList, "Kaervek's Torch", "MIR", "185");
        addCardInfoToList(boosterList, "Kargan Dragonlord", "ROE", "152");
        addCardInfoToList(boosterList, "Knight of Dawn", "TMP", "26");
        addCardInfoToList(boosterList, "Knollspine Dragon", "SHM", "98");
        addCardInfoToList(boosterList, "Kor Chant", "CNS", "73");
        addCardInfoToList(boosterList, "Kruphix, God of Horizons", "JOU", "152");
        addCardInfoToList(boosterList, "Lashknife Barrier", "PLS", "9");
        addCardInfoToList(boosterList, "Lotus Petal", "TMP", "294");
        addCardInfoToList(boosterList, "Maelstrom Archangel", "CON", "115");
        addCardInfoToList(boosterList, "Magus of the Moat", "FUT", "12");
        addCardInfoToList(boosterList, "Mana Tithe", "PLC", "25");
        addCardInfoToList(boosterList, "Manamorphose", "SHM", "211");
        addCardInfoToList(boosterList, "Martyr's Bond", "CMD", "19");
        addCardInfoToList(boosterList, "Martyr's Cause", "ULG", "13");
        addCardInfoToList(boosterList, "Master Transmuter", "CON", "31");
        addCardInfoToList(boosterList, "Meddling Mage", "ARB", "8");
        addCardInfoToList(boosterList, "Mistform Shrieker", "ONS", "96");
        addCardInfoToList(boosterList, "Nemesis of Reason", "ARB", "28");
        addCardInfoToList(boosterList, "Oracle of Nectars", "SHM", "233");
        addCardInfoToList(boosterList, "Pathrazer of Ulamog", "ROE", "9");
        addCardInfoToList(boosterList, "Perish", "TMP", "147");
        addCardInfoToList(boosterList, "Pestilence", "USG", "147");
        addCardInfoToList(boosterList, "Phantasmal Dragon", "DDM", "14");
        addCardInfoToList(boosterList, "Phantom Centaur", "JUD", "127");
        addCardInfoToList(boosterList, "Phyrexian Metamorph", "NPH", "42");
        addCardInfoToList(boosterList, "Phyrexian Soulgorger", "CSP", "141");
        addCardInfoToList(boosterList, "Purphoros, God of the Forge", "THS", "135");
        addCardInfoToList(boosterList, "Questing Phelddagrif", "PLS", "119");
        addCardInfoToList(boosterList, "Rage Reflection", "SHM", "104");
        addCardInfoToList(boosterList, "Recoup", "DDK", "63");
        addCardInfoToList(boosterList, "Release the Ants", "MOR", "98");
        addCardInfoToList(boosterList, "Rhys the Redeemed", "SHM", "237");
        addCardInfoToList(boosterList, "Rhystic Study", "PCY", "45");
        addCardInfoToList(boosterList, "Rishadan Footpad", "MMQ", "94");
        addCardInfoToList(boosterList, "Rith, the Awakener", "DDE", "48");
        addCardInfoToList(boosterList, "River Boa", "ZEN", "180");
        addCardInfoToList(boosterList, "Sadistic Hypnotist", "ODY", "159");
        addCardInfoToList(boosterList, "Sakashima the Impostor", "SOK", "53");
        addCardInfoToList(boosterList, "Sapphire Charm", "MIR", "89");
        addCardInfoToList(boosterList, "Shrouded Lore", "PLC", "91");
        addCardInfoToList(boosterList, "Soothsaying", "MMQ", "104");
        addCardInfoToList(boosterList, "Sorin Markov", "ZEN", "111");
        addCardInfoToList(boosterList, "Squirrel Wrangler", "PCY", "127");
        addCardInfoToList(boosterList, "Thieving Magpie", "UDS", "49");
        addCardInfoToList(boosterList, "Thrun, the Last Troll", "MBS", "92");
        addCardInfoToList(boosterList, "Time Sieve", "ARB", "31");
        addCardInfoToList(boosterList, "Timely Reinforcements", "M12", "40");
        addCardInfoToList(boosterList, "Tinker", "ULG", "45");
        addCardInfoToList(boosterList, "Tower of Eons", "MRD", "266");
        addCardInfoToList(boosterList, "Toxin Sliver", "LGN", "84");
        addCardInfoToList(boosterList, "Triumph of the Hordes", "NPH", "123");
        addCardInfoToList(boosterList, "Umbral Mantle", "SHM", "267");
        addCardInfoToList(boosterList, "Viashino Sandstalker", "VIS", "100");
        addCardInfoToList(boosterList, "Violent Ultimatum", "ALA", "206");
        addCardInfoToList(boosterList, "Volunteer Reserves", "WTH", "29");
        addCardInfoToList(boosterList, "Wargate", "ARB", "129");
        addCardInfoToList(boosterList, "Weathered Wayfarer", "ONS", "59");
        addCardInfoToList(boosterList, "Wild Nacatl", "ALA", "152");
        addCardInfoToList(boosterList, "Yavimaya's Embrace", "APC", "127");
    }

    private void addSlot14rare(List<CardInfo> boosterList) {
        addCardInfoToList(boosterList, "Adorned Pouncer", "HOU", "2");
        addCardInfoToList(boosterList, "Aetherflux Reservoir", "KLD", "192");
        addCardInfoToList(boosterList, "Akroan Horse", "C16", "241");
        addCardInfoToList(boosterList, "Alesha, Who Smiles at Death", "FRF", "90");
        addCardInfoToList(boosterList, "Alhammarret's Archive", "ORI", "221");
        addCardInfoToList(boosterList, "All Is Dust", "MM2", "1");
        addCardInfoToList(boosterList, "Aminatou's Augury", "C18", "6");
        addCardInfoToList(boosterList, "Angel of the Dire Hour", "C14", "1");
        addCardInfoToList(boosterList, "Anger of the Gods", "IMA", "116");
        addCardInfoToList(boosterList, "Animar, Soul of Elements", "A25", "196");
        addCardInfoToList(boosterList, "Approach of the Second Sun", "AKH", "4");
        addCardInfoToList(boosterList, "Arch of Orazca", "RIX", "185");
        addCardInfoToList(boosterList, "Basilisk Collar", "MM3", "216");
        addCardInfoToList(boosterList, "Beacon of Immortality", "E02", "1");
        addCardInfoToList(boosterList, "Beastmaster Ascension", "CMA", "92");
        addCardInfoToList(boosterList, "Birds of Paradise", "CN2", "176");
        addCardInfoToList(boosterList, "Black Market", "C17", "98");
        addCardInfoToList(boosterList, "Boompile", "C16", "52");
        addCardInfoToList(boosterList, "Boros Reckoner", "GK1", "85");
        addCardInfoToList(boosterList, "Caged Sun", "CM2", "178");
        addCardInfoToList(boosterList, "Cauldron of Souls", "CM2", "180");
        addCardInfoToList(boosterList, "Champion of the Parish", "DDQ", "4");
        addCardInfoToList(boosterList, "Chaos Warp", "C17", "131");
        addCardInfoToList(boosterList, "Chasm Skulker", "M15", "46");
        addCardInfoToList(boosterList, "Chromatic Lantern", "GRN", "233");
        addCardInfoToList(boosterList, "Coat of Arms", "DDS", "58");
        addCardInfoToList(boosterList, "Collective Brutality", "EMN", "85");
        addCardInfoToList(boosterList, "Commit // Memory", "AKH", "211");
        addCardInfoToList(boosterList, "Courser of Kruphix", "A25", "164");
        addCardInfoToList(boosterList, "Coveted Jewel", "C18", "54");
        addCardInfoToList(boosterList, "Daretti, Scrap Savant", "C14", "33");
        addCardInfoToList(boosterList, "Deadly Tempest", "C15", "19");
        addCardInfoToList(boosterList, "Debtors' Knell", "GK2", "39");
        addCardInfoToList(boosterList, "Decree of Justice", "DDO", "7");
        addCardInfoToList(boosterList, "Deepglow Skate", "CM2", "39");
        addCardInfoToList(boosterList, "Desolation Twin", "BFZ", "6");
        addCardInfoToList(boosterList, "Dictate of Heliod", "DDO", "8");
        addCardInfoToList(boosterList, "Djinn of Wishes", "C18", "87");
        addCardInfoToList(boosterList, "Dragonlord Ojutai", "DTK", "219");
        addCardInfoToList(boosterList, "Drana, Kalastria Bloodchief", "C17", "112");
        addCardInfoToList(boosterList, "Eldrazi Monument", "CMA", "216");
        addCardInfoToList(boosterList, "Eldritch Evolution", "EMN", "155");
        addCardInfoToList(boosterList, "Elesh Norn, Grand Cenobite", "IMA", "18");
        addCardInfoToList(boosterList, "Evra, Halcyon Witness", "DOM", "16");
        addCardInfoToList(boosterList, "Expropriate", "CN2", "30");
        addCardInfoToList(boosterList, "Fblthp, the Lost", "WAR", "50");
        addCardInfoToList(boosterList, "Felidar Sovereign", "BFZ", "26");
        addCardInfoToList(boosterList, "Gideon Jura", "E01", "10");
        addCardInfoToList(boosterList, "Goblin Charbelcher", "DDT", "57");
        addCardInfoToList(boosterList, "Goblin Piledriver", "ORI", "151");
        addCardInfoToList(boosterList, "Gonti, Lord of Luxury", "KLD", "84");
        addCardInfoToList(boosterList, "Grasp of Fate", "C15", "3");
        addCardInfoToList(boosterList, "Grave Titan", "C14", "145");
        addCardInfoToList(boosterList, "Gravecrawler", "DDQ", "59");
        addCardInfoToList(boosterList, "Greenbelt Rampager", "AER", "107");
        addCardInfoToList(boosterList, "Hornet Nest", "M15", "177");
        addCardInfoToList(boosterList, "Kiki-Jiki, Mirror Breaker", "MM2", "121");
        addCardInfoToList(boosterList, "Kolaghan's Command", "DTK", "224");
        addCardInfoToList(boosterList, "Krenko, Mob Boss", "DDT", "52");
        addCardInfoToList(boosterList, "Liliana, Death's Majesty", "AKH", "97");
        addCardInfoToList(boosterList, "Living Death", "A25", "96");
        addCardInfoToList(boosterList, "Mana Crypt", "EMA", "225");
        addCardInfoToList(boosterList, "Meandering Towershell", "KTK", "141");
        addCardInfoToList(boosterList, "Memory Erosion", "CM2", "45");
        addCardInfoToList(boosterList, "Meren of Clan Nel Toth", "C15", "49");
        addCardInfoToList(boosterList, "Mimic Vat", "C19", "219");
        addCardInfoToList(boosterList, "Mind Shatter", "MM3", "77");
        addCardInfoToList(boosterList, "Mind Spring", "DDT", "14");
        addCardInfoToList(boosterList, "Mirran Crusader", "MM2", "25");
        addCardInfoToList(boosterList, "Mirror Entity", "CMA", "16");
        addCardInfoToList(boosterList, "Misdirection", "DDT", "15");
        addCardInfoToList(boosterList, "Mizzix's Mastery", "C15", "29");
        addCardInfoToList(boosterList, "Mycoloth", "CMA", "129");
        addCardInfoToList(boosterList, "Mystic Confluence", "BBD", "122");
        addCardInfoToList(boosterList, "Nighthowler", "C15", "129");
        addCardInfoToList(boosterList, "Nin, the Pain Artist", "C17", "183");
        addCardInfoToList(boosterList, "Nissa, Voice of Zendikar", "OGW", "138");
        addCardInfoToList(boosterList, "Odric, Lunarch Marshal", "SOI", "31");
        addCardInfoToList(boosterList, "Phyrexian Arena", "CN2", "144");
        addCardInfoToList(boosterList, "Phyrexian Plaguelord", "CMA", "61");
        addCardInfoToList(boosterList, "Precursor Golem", "MM2", "225");
        addCardInfoToList(boosterList, "Preyseizer Dragon", "PCA", "50");
        addCardInfoToList(boosterList, "Queen Marchesa", "CN2", "78");
        addCardInfoToList(boosterList, "Reality Scramble", "C18", "25");
        addCardInfoToList(boosterList, "Recruiter of the Guard", "CN2", "22");
        addCardInfoToList(boosterList, "Release the Gremlins", "AER", "96");
        addCardInfoToList(boosterList, "Revel in Riches", "XLN", "117");
        addCardInfoToList(boosterList, "Rune-Scarred Demon", "IMA", "106");
        addCardInfoToList(boosterList, "Savage Knuckleblade", "KTK", "197");
        addCardInfoToList(boosterList, "Selvala, Heart of the Wilds", "CN2", "70");
        addCardInfoToList(boosterList, "Serendib Efreet", "EMA", "70");
        addCardInfoToList(boosterList, "Sewer Nemesis", "CM2", "75");
        addCardInfoToList(boosterList, "Shamanic Revelation", "FRF", "138");
        addCardInfoToList(boosterList, "Sliver Hivelord", "M15", "211");
        addCardInfoToList(boosterList, "Solemn Simulacrum", "DDU", "62");
        addCardInfoToList(boosterList, "Spawning Grounds", "C18", "163");
        addCardInfoToList(boosterList, "Star of Extinction", "XLN", "161");
        addCardInfoToList(boosterList, "Steamflogger Boss", "UST", "93");
        addCardInfoToList(boosterList, "Stunt Double", "CN2", "38");
        addCardInfoToList(boosterList, "Sudden Demise", "E01", "59");
        addCardInfoToList(boosterList, "Supreme Verdict", "IMA", "210");
        addCardInfoToList(boosterList, "Sword of the Animist", "E01", "89");
        addCardInfoToList(boosterList, "Talrand, Sky Summoner", "DDS", "11");
        addCardInfoToList(boosterList, "Taurean Mauler", "CM2", "122");
        addCardInfoToList(boosterList, "Teferi, Temporal Archmage", "C14", "19");
        addCardInfoToList(boosterList, "Teferi's Protection", "C17", "8");
        addCardInfoToList(boosterList, "Temporal Mastery", "MM3", "54");
        addCardInfoToList(boosterList, "Tempt with Discovery", "C16", "170");
        addCardInfoToList(boosterList, "Thalia's Lancers", "EMN", "47");
        addCardInfoToList(boosterList, "The Gitrog Monster", "SOI", "245");
        addCardInfoToList(boosterList, "The Mirari Conjecture", "DOM", "57");
        addCardInfoToList(boosterList, "Tireless Tracker", "SOI", "233");
        addCardInfoToList(boosterList, "Torment of Hailfire", "HOU", "77");
        addCardInfoToList(boosterList, "Trading Post", "CM2", "225");
        addCardInfoToList(boosterList, "Two-Headed Giant", "DOM", "147");
        addCardInfoToList(boosterList, "Urza's Rage", "C15", "169");
        addCardInfoToList(boosterList, "Vigor", "BBD", "215");
        addCardInfoToList(boosterList, "Wheel of Fate", "C16", "138");
        addCardInfoToList(boosterList, "Whelming Wave", "DDO", "44");
        addCardInfoToList(boosterList, "Whir of Invention", "AER", "49");
        addCardInfoToList(boosterList, "Yuriko, the Tiger's Shadow", "C18", "52");
    }

    private void addSlot15foil(List<CardInfo> boosterList) {
        addCardInfoToList(boosterList, "Alchemist's Refuge", "AVR", "225");
        addCardInfoToList(boosterList, "Allosaurus Rider", "CSP", "101");
        addCardInfoToList(boosterList, "Amulet of Vigor", "WWK", "121");
        addCardInfoToList(boosterList, "Archetype of Endurance", "BNG", "116");
        addCardInfoToList(boosterList, "Aurelia's Fury", "GTC", "144");
        addCardInfoToList(boosterList, "Balduvian Rage", "CSP", "76");
        addCardInfoToList(boosterList, "Balefire Liege", "EVE", "132");
        addCardInfoToList(boosterList, "Blasting Station", "5DN", "107");
        addCardInfoToList(boosterList, "Blighted Agent", "NPH", "29");
        addCardInfoToList(boosterList, "Boreal Druid", "CSP", "105");
        addCardInfoToList(boosterList, "Boundless Realms", "M13", "162");
        addCardInfoToList(boosterList, "Braid of Fire", "CSP", "78");
        addCardInfoToList(boosterList, "Bramblewood Paragon", "MOR", "115");
        addCardInfoToList(boosterList, "Bringer of the Black Dawn", "5DN", "43");
        addCardInfoToList(boosterList, "Burning Inquiry", "M10", "128");
        addCardInfoToList(boosterList, "Celestial Dawn", "TSB", "3");
        addCardInfoToList(boosterList, "Celestial Kirin", "SOK", "3");
        addCardInfoToList(boosterList, "Changeling Hero", "LRW", "9");
        addCardInfoToList(boosterList, "Chimney Imp", "MRD", "59");
        addCardInfoToList(boosterList, "Codex Shredder", "RTR", "228");
        addCardInfoToList(boosterList, "Conspiracy", "TSB", "39");
        addCardInfoToList(boosterList, "Council Guardian", "CNS", "15");
        addCardInfoToList(boosterList, "Delay", "FUT", "35");
        addCardInfoToList(boosterList, "Drogskol Captain", "DKA", "136");
        addCardInfoToList(boosterList, "Echoing Decay", "DST", "41");
        addCardInfoToList(boosterList, "Eidolon of Rhetoric", "JOU", "10");
        addCardInfoToList(boosterList, "Fatespinner", "MRD", "36");
        addCardInfoToList(boosterList, "Fiery Gambit", "MRD", "90");
        addCardInfoToList(boosterList, "Flamekin Harbinger", "LRW", "167");
        addCardInfoToList(boosterList, "Form of the Dragon", "9ED", "187");
        addCardInfoToList(boosterList, "Frozen Aether", "PLC", "54");
        addCardInfoToList(boosterList, "Funeral Charm", "TSB", "44");
        addCardInfoToList(boosterList, "Fungusaur", "8ED", "250");
        addCardInfoToList(boosterList, "Game-Trail Changeling", "MOR", "123");
        addCardInfoToList(boosterList, "Geth's Grimoire", "DST", "123");
        addCardInfoToList(boosterList, "Gilder Bairn", "EVE", "152");
        addCardInfoToList(boosterList, "Gleeful Sabotage", "SHM", "116");
        addCardInfoToList(boosterList, "Glittering Wish", "FUT", "156");
        addCardInfoToList(boosterList, "Goblin Bushwhacker", "ZEN", "125");
        addCardInfoToList(boosterList, "Grand Architect", "SOM", "33");
        addCardInfoToList(boosterList, "Greater Mossdog", "MMA", "146");
        addCardInfoToList(boosterList, "Guerrilla Tactics", "10E", "211");
        addCardInfoToList(boosterList, "Harmonic Sliver", "TSP", "240");
        addCardInfoToList(boosterList, "Helix Pinnacle", "EVE", "68");
        addCardInfoToList(boosterList, "Herald of Leshrac", "CSP", "62");
        addCardInfoToList(boosterList, "Hornet Sting", "M11", "181");
        addCardInfoToList(boosterList, "Intruder Alarm", "8ED", "86");
        addCardInfoToList(boosterList, "Iron Myr", "SOM", "168");
        addCardInfoToList(boosterList, "Isamaru, Hound of Konda", "CHK", "19");
        addCardInfoToList(boosterList, "Karrthus, Tyrant of Jund", "ARB", "117");
        addCardInfoToList(boosterList, "Knowledge Pool", "MBS", "111");
        addCardInfoToList(boosterList, "Kulrath Knight", "SHM", "190");
        addCardInfoToList(boosterList, "Lantern of Insight", "5DN", "135");
        addCardInfoToList(boosterList, "Lapse of Certainty", "CON", "9");
        addCardInfoToList(boosterList, "Leveler", "MRD", "195");
        addCardInfoToList(boosterList, "Lich's Mirror", "ALA", "210");
        addCardInfoToList(boosterList, "Lightning Storm", "CSP", "89");
        addCardInfoToList(boosterList, "Lumithread Field", "FUT", "25");
        addCardInfoToList(boosterList, "Maelstrom Nexus", "ARB", "130");
        addCardInfoToList(boosterList, "Magewright's Stone", "DIS", "162");
        addCardInfoToList(boosterList, "Manaweft Sliver", "M14", "184");
        addCardInfoToList(boosterList, "Maro", "8ED", "264");
        addCardInfoToList(boosterList, "Marrow-Gnawer", "CHK", "124");
        addCardInfoToList(boosterList, "Memnite", "SOM", "174");
        addCardInfoToList(boosterList, "Minamo, School at Water's Edge", "CHK", "279");
        addCardInfoToList(boosterList, "Mind Funeral", "MMA", "181");
        addCardInfoToList(boosterList, "Mindslaver", "MRD", "206");
        addCardInfoToList(boosterList, "Mirrodin's Core", "DST", "165");
        addCardInfoToList(boosterList, "Misthollow Griffin", "AVR", "68");
        addCardInfoToList(boosterList, "Myojin of Life's Web", "CHK", "229");
        addCardInfoToList(boosterList, "Nezumi Shortfang", "CHK", "131");
        addCardInfoToList(boosterList, "Noggle Bandit", "EVE", "106");
        addCardInfoToList(boosterList, "Norin the Wary", "TSP", "171");
        addCardInfoToList(boosterList, "Norn's Annex", "NPH", "17");
        addCardInfoToList(boosterList, "Not of This World", "ROE", "8");
        addCardInfoToList(boosterList, "Ogre Gatecrasher", "DIS", "67");
        addCardInfoToList(boosterList, "One with Nothing", "SOK", "84");
        addCardInfoToList(boosterList, "Panglacial Wurm", "CSP", "116");
        addCardInfoToList(boosterList, "Paradox Haze", "TSP", "71");
        addCardInfoToList(boosterList, "Patron of the Moon", "BOK", "45");
        addCardInfoToList(boosterList, "Pili-Pala", "SHM", "258");
        addCardInfoToList(boosterList, "Proclamation of Rebirth", "DIS", "15");
        addCardInfoToList(boosterList, "Puca's Mischief", "SHM", "47");
        addCardInfoToList(boosterList, "Pull from Eternity", "TSP", "35");
        addCardInfoToList(boosterList, "Pyretic Ritual", "M11", "153");
        addCardInfoToList(boosterList, "Ravenous Trap", "ZEN", "109");
        addCardInfoToList(boosterList, "Reaper King", "SHM", "260");
        addCardInfoToList(boosterList, "Reki, the History of Kamigawa", "SOK", "142");
        addCardInfoToList(boosterList, "Rescue from the Underworld", "THS", "102");
        addCardInfoToList(boosterList, "Rhox", "10E", "291");
        addCardInfoToList(boosterList, "Rune-Tail, Kitsune Ascendant", "SOK", "27");
        addCardInfoToList(boosterList, "Sakura-Tribe Scout", "SOK", "144");
        addCardInfoToList(boosterList, "Sarkhan the Mad", "ROE", "214");
        addCardInfoToList(boosterList, "Scourge of the Throne", "CNS", "35");
        addCardInfoToList(boosterList, "Scryb Ranger", "TSP", "215");
        addCardInfoToList(boosterList, "Sen Triplets", "ARB", "109");
        addCardInfoToList(boosterList, "Sheltering Ancient", "CSP", "121");
        addCardInfoToList(boosterList, "Shizo, Death's Storehouse", "CHK", "283");
        addCardInfoToList(boosterList, "Sinew Sliver", "PLC", "30");
        addCardInfoToList(boosterList, "Sosuke, Son of Seshiro", "CHK", "244");
        addCardInfoToList(boosterList, "Soul's Attendant", "ROE", "44");
        addCardInfoToList(boosterList, "Spelltithe Enforcer", "GPT", "18");
        addCardInfoToList(boosterList, "Spellweaver Volute", "FUT", "59");
        addCardInfoToList(boosterList, "Spike Feeder", "TSB", "84");
        addCardInfoToList(boosterList, "Springjack Shepherd", "EVE", "15");
        addCardInfoToList(boosterList, "Stalking Stones", "MRD", "284");
        addCardInfoToList(boosterList, "Stigma Lasher", "EVE", "62");
        addCardInfoToList(boosterList, "Storm Crow", "9ED", "100");
        addCardInfoToList(boosterList, "Sundial of the Infinite", "M12", "218");
        addCardInfoToList(boosterList, "Teferi's Puzzle Box", "8ED", "316");
        addCardInfoToList(boosterList, "Trailblazer's Boots", "ZEN", "208");
        addCardInfoToList(boosterList, "Treasonous Ogre", "CNS", "36");
        addCardInfoToList(boosterList, "Triskelion", "M11", "218");
        addCardInfoToList(boosterList, "Undead Warchief", "TSB", "52");
        addCardInfoToList(boosterList, "Viscera Seer", "M11", "120");
        addCardInfoToList(boosterList, "Wall of Shards", "CSP", "23");
        addCardInfoToList(boosterList, "Wear // Tear", "DGM", "135");
        addCardInfoToList(boosterList, "White Knight", "M10", "41");
        addCardInfoToList(boosterList, "Witchbane Orb", "ISD", "236");
        addCardInfoToList(boosterList, "Yore-Tiller Nephilim", "GPT", "140");
        addCardInfoToList(boosterList, "Zur's Weirding", "8ED", "116");
    }

}
