package mage.client.dialog;

import mage.cards.decks.importer.DeckImporter;
import mage.client.MageFrame;
import mage.client.SessionHandler;
import mage.client.components.MageComponents;
import mage.client.table.TablePlayerPanel;
import mage.client.util.Event;
import mage.client.util.IgnoreList;
import mage.client.util.Listener;
import mage.constants.MatchBufferTime;
import mage.constants.MatchTimeLimit;
import mage.constants.MultiplayerAttackOption;
import mage.constants.RangeOfInfluence;
import mage.constants.SkillLevel;
import mage.game.match.MatchOptions;
import mage.players.PlayerType;
import mage.view.GameTypeView;
import mage.view.TableView;
import org.apache.log4j.Logger;

import javax.swing.*;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * App GUI: create new GAME
 *
 * @author BetaSteward_at_googlemail.com, JayDi85
 */
public class NewTableDialog extends MageDialog {

    private static final Logger logger = Logger.getLogger(NewTableDialog.class);

    public static final int DEFAULT_COMPUTER_PLAYER_SKILL_LEVEL = 2;
    public static final String PLAYER_DATA_DELIMETER_OLD = ","; // need for compatibility with old version
    public static final String PLAYER_DATA_DELIMETER_NEW = "@@@";

    private final CustomOptionsDialog customOptions;
    private TableView table;
    private UUID playerId;
    private UUID roomId;
    private String lastSessionId;
    private final List<TablePlayerPanel> players = new ArrayList<>();

    // temp settings on loading players list
    private final List<PlayerType> prefPlayerTypes = new ArrayList<>();
    private final List<Integer> prefPlayerSkills = new ArrayList<>();
    private final List<String> prefPlayerDecks = new ArrayList<>();


    private static final String LIMITED = "Limited";

    public NewTableDialog() {
        lastSessionId = "";
        initComponents();
        this.customOptions = new CustomOptionsDialog(CustomOptionsDialog.SaveLoadKeys.TABLE, btnCustomOptions);
        MageFrame.getDesktop().add(customOptions, customOptions.isModal() ? JLayeredPane.MODAL_LAYER : JLayeredPane.PALETTE_LAYER);
        player1Panel.showLevel(false);
        this.spnNumWins.setModel(new SpinnerNumberModel(1, 1, 5, 1));
        this.spnQuitRatio.setModel(new SpinnerNumberModel(100, 0, 100, 5));
        this.spnMinimumRating.setModel(new SpinnerNumberModel(0, 0, 3000, 10));
        this.spnEdhPowerLevel.setModel(new SpinnerNumberModel(100, 0, 100, 5));
        MageFrame.getUI().addButton(MageComponents.NEW_TABLE_OK_BUTTON, btnOK);
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        popupSaveSettings = new javax.swing.JPopupMenu();
        menuSaveSettings1 = new javax.swing.JMenuItem();
        menuSaveSettings2 = new javax.swing.JMenuItem();
        popupLoadSettings = new javax.swing.JPopupMenu();
        menuLoadSettingsLast = new javax.swing.JMenuItem();
        separator1 = new javax.swing.JPopupMenu.Separator();
        menuLoadSettings1 = new javax.swing.JMenuItem();
        menuLoadSettings2 = new javax.swing.JMenuItem();
        separator2 = new javax.swing.JPopupMenu.Separator();
        menuLoadSettingsDefault = new javax.swing.JMenuItem();
        lblName = new javax.swing.JLabel();
        txtName = new javax.swing.JTextField();
        lblPassword = new javax.swing.JLabel();
        txtPassword = new javax.swing.JTextField();
        lbDeckType = new javax.swing.JLabel();
        cbDeckType = new javax.swing.JComboBox();
        lbTimeLimit = new javax.swing.JLabel();
        cbTimeLimit = new javax.swing.JComboBox();
        lbBufferTime = new javax.swing.JLabel();
        cbBufferTime = new javax.swing.JComboBox();
        lblGameType = new javax.swing.JLabel();
        cbGameType = new javax.swing.JComboBox();
        chkRollbackTurnsAllowed = new javax.swing.JCheckBox();
        chkSpectatorsAllowed = new javax.swing.JCheckBox();
        lblNumPlayers = new javax.swing.JLabel();
        spnNumPlayers = new javax.swing.JSpinner();
        lblSkillLevel = new javax.swing.JLabel();
        cbSkillLevel = new javax.swing.JComboBox();
        lblNumWins = new javax.swing.JLabel();
        spnNumWins = new javax.swing.JSpinner();
        jSeparator2 = new javax.swing.JSeparator();
        jLabel1 = new javax.swing.JLabel();
        player1Panel = new mage.client.table.NewPlayerPanel();
        jSeparator3 = new javax.swing.JSeparator();
        jLabel2 = new javax.swing.JLabel();
        pnlOtherPlayers = new javax.swing.JPanel();
        jSeparator1 = new javax.swing.JSeparator();
        btnOK = new javax.swing.JButton();
        btnCancel = new javax.swing.JButton();
        lblQuitRatio = new javax.swing.JLabel();
        spnQuitRatio = new javax.swing.JSpinner();
        lblEdhPowerLevel = new javax.swing.JLabel();
        spnEdhPowerLevel = new javax.swing.JSpinner();
        lblMinimumRating = new javax.swing.JLabel();
        spnMinimumRating = new javax.swing.JSpinner();
        chkRated = new javax.swing.JCheckBox();
        btnSettingsLoad = new javax.swing.JButton();
        btnSettingsSave = new javax.swing.JButton();
        lblSettings = new javax.swing.JLabel();
        btnCustomOptions = new javax.swing.JButton();
        lblRange = new javax.swing.JLabel();
        cbRange = new javax.swing.JComboBox();
        lblAttack = new javax.swing.JLabel();
        cbAttackOption = new javax.swing.JComboBox();

        menuSaveSettings1.setText("Save to config 1");
        menuSaveSettings1.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuSaveSettings1ActionPerformed(evt);
            }
        });
        popupSaveSettings.add(menuSaveSettings1);

        menuSaveSettings2.setText("Save to config 2");
        menuSaveSettings2.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuSaveSettings2ActionPerformed(evt);
            }
        });
        popupSaveSettings.add(menuSaveSettings2);

        menuLoadSettingsLast.setText("Load from last time");
        menuLoadSettingsLast.setToolTipText("");
        menuLoadSettingsLast.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuLoadSettingsLastActionPerformed(evt);
            }
        });
        popupLoadSettings.add(menuLoadSettingsLast);
        popupLoadSettings.add(separator1);

        menuLoadSettings1.setText("Load from config 1");
        menuLoadSettings1.setToolTipText("");
        menuLoadSettings1.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuLoadSettings1ActionPerformed(evt);
            }
        });
        popupLoadSettings.add(menuLoadSettings1);

        menuLoadSettings2.setText("Load from config 2");
        menuLoadSettings2.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuLoadSettings2ActionPerformed(evt);
            }
        });
        popupLoadSettings.add(menuLoadSettings2);
        popupLoadSettings.add(separator2);

        menuLoadSettingsDefault.setText("Load default settings");
        menuLoadSettingsDefault.setToolTipText("");
        menuLoadSettingsDefault.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuLoadSettingsDefaultActionPerformed(evt);
            }
        });
        popupLoadSettings.add(menuLoadSettingsDefault);

        setTitle("New Table");

        lblName.setLabelFor(txtName);
        lblName.setText("Name:");

        lblPassword.setLabelFor(txtPassword);
        lblPassword.setText("Password:");

        lbDeckType.setText("Deck Type:");

        lbTimeLimit.setText("Time Limit:");
        lbTimeLimit.setToolTipText("The active time a player may use to finish the match. If their time runs out, the player looses the current game.");

        lbBufferTime.setText("Buffer Time:");
        lbBufferTime.setToolTipText("The extra time a player gets whenever the timer starts before their normal time limit starts going down.");

        cbBufferTime.setToolTipText("The extra time a player gets whenever the timer starts before their normal time limit starts going down.");

        lblGameType.setText("Game Type:");

        cbGameType.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cbGameTypeActionPerformed(evt);
            }
        });

        chkRollbackTurnsAllowed.setText("Rollbacks");
        chkRollbackTurnsAllowed.setToolTipText("<HTML>Allow to rollback to the start of previous turns<br>\nif all players agree.\n");

        chkSpectatorsAllowed.setText("Spectators allowed");
        chkSpectatorsAllowed.setToolTipText("Allow spectators to view your game.");

        lblNumPlayers.setLabelFor(spnNumPlayers);
        lblNumPlayers.setText("Players:");

        spnNumPlayers.addChangeListener(new javax.swing.event.ChangeListener() {
            public void stateChanged(javax.swing.event.ChangeEvent evt) {
                numPlayersChanged(evt);
            }
        });

        lblSkillLevel.setText("Skill Level:");
        lblSkillLevel.setToolTipText("");

        cbSkillLevel.setToolTipText("<HTML>This option can be used to make it easier to find matches<br>\nwith opponents of the appropriate skill level.");

        lblNumWins.setLabelFor(spnNumWins);
        lblNumWins.setText("Wins:");
        lblNumWins.setToolTipText("How many games has a player to win to win the match.");

        jLabel1.setFont(new java.awt.Font("Tahoma", 1, 11)); // NOI18N
        jLabel1.setText("Player 1 (You)");

        jLabel2.setFont(new java.awt.Font("Tahoma", 1, 11)); // NOI18N
        jLabel2.setText("Other Players");

        pnlOtherPlayers.setLayout(new java.awt.GridLayout(0, 1));

        btnOK.setText("Create");
        btnOK.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                btnOKActionPerformed(evt);
            }
        });

        btnCancel.setText("Cancel");
        btnCancel.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                btnCancelActionPerformed(evt);
            }
        });

        lblQuitRatio.setText("Allowed quit %");

        lblEdhPowerLevel.setText("EDH power level:");

        lblMinimumRating.setLabelFor(spnMinimumRating);
        lblMinimumRating.setText("Minimum rating:");
        lblMinimumRating.setToolTipText("Players with rating less than this value can't join this table");

        chkRated.setText("Rated game");
        chkRated.setToolTipText("Indicates if match will be rated");

        btnSettingsLoad.setText("Load...");
        btnSettingsLoad.setToolTipText("Load settings");
        btnSettingsLoad.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                btnSettingsLoadMouseClicked(evt);
            }
        });

        btnSettingsSave.setText("Save...");
        btnSettingsSave.setToolTipText("Save settings");
        btnSettingsSave.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                btnSettingsSaveMouseClicked(evt);
            }
        });

        lblSettings.setText("Settings");

        btnCustomOptions.setText("Custom Options...");
        btnCustomOptions.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                btnCustomOptionsActionPerformed(evt);
            }
        });

        lblRange.setText("Range of Influence:");

        cbRange.setToolTipText("<HTML>An option for multiplayer games.\nA player's range of influence is the maximum distance from that player, measured in player seats,<br>\nthat the player can affect. Players within that many seats of the player are within that player's range<br>\nof influence. Objects controlled by players within a player's range of influence are also within that<br>\nplayer's range of influence. Range of influence covers spells, abilities, effects, damage dealing, attacking,<nr>\nmaking choices, and winning the game.");

        lblAttack.setText("Attack Option:");

        cbAttackOption.setToolTipText("<HTML>An option for multiplayer games that defines<br>\nwhich opponents can be attacked from a player.");

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                    .addGroup(layout.createSequentialGroup()
                        .addComponent(lblSettings)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(btnSettingsLoad)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(btnSettingsSave)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                        .addComponent(btnOK, javax.swing.GroupLayout.PREFERRED_SIZE, 100, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(btnCancel, javax.swing.GroupLayout.PREFERRED_SIZE, 100, javax.swing.GroupLayout.PREFERRED_SIZE))
                    .addComponent(jSeparator2)
                    .addComponent(player1Panel, javax.swing.GroupLayout.Alignment.LEADING, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(pnlOtherPlayers, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(jSeparator1, javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(javax.swing.GroupLayout.Alignment.LEADING, layout.createSequentialGroup()
                        .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(lblName)
                            .addComponent(lbDeckType)
                            .addComponent(lblGameType))
                        .addGap(6, 6, 6)
                        .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addGroup(layout.createSequentialGroup()
                                .addComponent(txtName, javax.swing.GroupLayout.PREFERRED_SIZE, 178, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(lbTimeLimit)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(cbTimeLimit, javax.swing.GroupLayout.PREFERRED_SIZE, 102, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(lbBufferTime)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(cbBufferTime, javax.swing.GroupLayout.PREFERRED_SIZE, 101, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(lblNumWins)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(spnNumWins, javax.swing.GroupLayout.PREFERRED_SIZE, 50, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addGap(8, 8, 8)
                                .addComponent(chkRollbackTurnsAllowed)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(chkSpectatorsAllowed))
                            .addGroup(layout.createSequentialGroup()
                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                                    .addComponent(cbDeckType, 0, 255, Short.MAX_VALUE)
                                    .addComponent(cbGameType, 0, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                    .addGroup(layout.createSequentialGroup()
                                        .addGap(0, 0, Short.MAX_VALUE)
                                        .addComponent(btnCustomOptions)
                                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                        .addComponent(lblSkillLevel)
                                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                        .addComponent(cbSkillLevel, javax.swing.GroupLayout.PREFERRED_SIZE, 102, javax.swing.GroupLayout.PREFERRED_SIZE)
                                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                        .addComponent(lblPassword)
                                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                        .addComponent(txtPassword, javax.swing.GroupLayout.PREFERRED_SIZE, 109, javax.swing.GroupLayout.PREFERRED_SIZE))
                                    .addGroup(layout.createSequentialGroup()
                                        .addComponent(chkRated)
                                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                        .addComponent(lblMinimumRating)
                                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                        .addComponent(spnMinimumRating, javax.swing.GroupLayout.PREFERRED_SIZE, 70, javax.swing.GroupLayout.PREFERRED_SIZE)
                                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                        .addComponent(lblQuitRatio)
                                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                        .addComponent(spnQuitRatio, javax.swing.GroupLayout.PREFERRED_SIZE, 60, javax.swing.GroupLayout.PREFERRED_SIZE)
                                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                        .addComponent(lblEdhPowerLevel)
                                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                        .addComponent(spnEdhPowerLevel, javax.swing.GroupLayout.PREFERRED_SIZE, 60, javax.swing.GroupLayout.PREFERRED_SIZE)
                                        .addGap(0, 0, Short.MAX_VALUE))))))
                    .addGroup(javax.swing.GroupLayout.Alignment.LEADING, layout.createSequentialGroup()
                        .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                            .addComponent(jLabel1, javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(jLabel2, javax.swing.GroupLayout.Alignment.LEADING)
                            .addGroup(javax.swing.GroupLayout.Alignment.LEADING, layout.createSequentialGroup()
                                .addComponent(lblNumPlayers)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(spnNumPlayers, javax.swing.GroupLayout.PREFERRED_SIZE, 57, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(lblRange)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(cbRange, javax.swing.GroupLayout.PREFERRED_SIZE, 60, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(lblAttack)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(cbAttackOption, javax.swing.GroupLayout.PREFERRED_SIZE, 150, javax.swing.GroupLayout.PREFERRED_SIZE)))
                        .addGap(0, 0, Short.MAX_VALUE)))
                .addContainerGap())
            .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                .addGroup(layout.createSequentialGroup()
                    .addContainerGap()
                    .addComponent(jSeparator3, javax.swing.GroupLayout.DEFAULT_SIZE, 933, Short.MAX_VALUE)
                    .addContainerGap()))
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addGap(2, 2, 2)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                        .addComponent(txtName)
                        .addComponent(lblName))
                    .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                        .addComponent(lbBufferTime)
                        .addComponent(cbBufferTime, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addComponent(lblNumWins)
                        .addComponent(spnNumWins, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addComponent(chkRollbackTurnsAllowed)
                        .addComponent(chkSpectatorsAllowed))
                    .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                        .addComponent(cbTimeLimit)
                        .addComponent(lbTimeLimit)))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(cbDeckType, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(lbDeckType)
                    .addComponent(lblQuitRatio)
                    .addComponent(lblEdhPowerLevel)
                    .addComponent(spnEdhPowerLevel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(spnQuitRatio, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(lblMinimumRating)
                    .addComponent(spnMinimumRating, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(chkRated))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                        .addComponent(lblPassword)
                        .addComponent(txtPassword)
                        .addComponent(cbSkillLevel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addComponent(lblSkillLevel)
                        .addComponent(btnCustomOptions))
                    .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                        .addComponent(cbGameType, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addComponent(lblGameType)))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                        .addComponent(cbAttackOption, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addComponent(lblAttack))
                    .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                        .addComponent(cbRange, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addComponent(lblRange))
                    .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                        .addComponent(lblNumPlayers)
                        .addComponent(spnNumPlayers, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(jSeparator2, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(jLabel1)
                .addGap(0, 0, 0)
                .addComponent(player1Panel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addGap(16, 16, 16)
                .addComponent(jLabel2)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(pnlOtherPlayers, javax.swing.GroupLayout.DEFAULT_SIZE, 131, Short.MAX_VALUE)
                .addGap(9, 9, 9)
                .addComponent(jSeparator1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(btnCancel, javax.swing.GroupLayout.PREFERRED_SIZE, 30, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(btnOK, javax.swing.GroupLayout.PREFERRED_SIZE, 30, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(btnSettingsLoad, javax.swing.GroupLayout.PREFERRED_SIZE, 30, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(btnSettingsSave, javax.swing.GroupLayout.PREFERRED_SIZE, 30, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(lblSettings))
                .addContainerGap())
            .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                .addGroup(layout.createSequentialGroup()
                    .addGap(201, 201, 201)
                    .addComponent(jSeparator3, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addContainerGap(180, Short.MAX_VALUE)))
        );

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void btnCancelActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_btnCancelActionPerformed
        this.table = null;
        this.playerId = null;
        doClose();
    }//GEN-LAST:event_btnCancelActionPerformed

    private void btnPreviousConfigurationActionPerformed(java.awt.event.ActionEvent evt, int i) {//GEN-FIRST:event_btnPreviousConfigurationActionPerformed
    }//GEN-LAST:event_btnPreviousConfigurationActionPerformed

    private void doClose() {
        if (this.customOptions.isVisible()) {
            this.customOptions.hideDialog();
        }
        this.hideDialog();
    }

    private void btnOKActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_btnOKActionPerformed

        MatchOptions options = getMatchOptions();
        if (!checkMatchOptions(options)) {
            return;
        }

        // save last used
        onSaveSettings(0, options, this.player1Panel.getDeckFile());

        // run
        table = SessionHandler.createTable(roomId, options);
        if (table == null) {
            JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Error creating table.", "Error", JOptionPane.ERROR_MESSAGE);
            return;
        }
        try {
            // join AI
            for (TablePlayerPanel player : players) {
                if (player.getPlayerType() != PlayerType.HUMAN) {
                    if (!player.joinTable(roomId, table.getTableId())) {
                        // error message must be sent by a server
                        SessionHandler.removeTable(roomId, table.getTableId());
                        table = null;
                        return;
                    }
                }
            }

            // join itself
            if (SessionHandler.joinTable(
                    roomId,
                    table.getTableId(),
                    this.player1Panel.getPlayerName(),
                    PlayerType.HUMAN, 1,
                    DeckImporter.importDeckFromFile(this.player1Panel.getDeckFile(), true),
                    this.txtPassword.getText())) {
                // all fine, can close create dialog (join dialog will be opened after feedback from server)
                doClose();
                return;
            }
        } catch (ClassNotFoundException | IOException ex) {
            handleError(ex);
        }
        // JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Error joining table.", "Error", JOptionPane.ERROR_MESSAGE);
        SessionHandler.removeTable(roomId, table.getTableId());
        table = null;
    }//GEN-LAST:event_btnOKActionPerformed

    private void cbGameTypeActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cbGameTypeActionPerformed
        setGameOptions();
    }//GEN-LAST:event_cbGameTypeActionPerformed

    private void numPlayersChanged(javax.swing.event.ChangeEvent evt) {//GEN-FIRST:event_numPlayersChanged
        int numPlayers = (Integer) this.spnNumPlayers.getValue() - 1;
        createPlayers(numPlayers);
    }//GEN-LAST:event_numPlayersChanged

    private void btnSettingsSaveMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_btnSettingsSaveMouseClicked
        popupSaveSettings.show(evt.getComponent(), evt.getX(), evt.getY());
    }//GEN-LAST:event_btnSettingsSaveMouseClicked

    private void btnSettingsLoadMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_btnSettingsLoadMouseClicked
        popupLoadSettings.show(evt.getComponent(), evt.getX(), evt.getY());
    }//GEN-LAST:event_btnSettingsLoadMouseClicked

    private void menuSaveSettings2ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuSaveSettings2ActionPerformed
        onSaveSettings(2);
    }//GEN-LAST:event_menuSaveSettings2ActionPerformed

    private void menuSaveSettings1ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuSaveSettings1ActionPerformed
        onSaveSettings(1);
    }//GEN-LAST:event_menuSaveSettings1ActionPerformed

    private void menuLoadSettingsLastActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuLoadSettingsLastActionPerformed
        onLoadSettings(0);
    }//GEN-LAST:event_menuLoadSettingsLastActionPerformed

    private void menuLoadSettings1ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuLoadSettings1ActionPerformed
        onLoadSettings(1);
    }//GEN-LAST:event_menuLoadSettings1ActionPerformed

    private void menuLoadSettings2ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuLoadSettings2ActionPerformed
        onLoadSettings(2);
    }//GEN-LAST:event_menuLoadSettings2ActionPerformed

    private void menuLoadSettingsDefaultActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuLoadSettingsDefaultActionPerformed
        onLoadSettings(-1);
    }//GEN-LAST:event_menuLoadSettingsDefaultActionPerformed

    private void btnCustomOptionsActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_btnCustomOptionsActionPerformed
        customOptions.showDialog();
    }//GEN-LAST:event_btnCustomOptionsActionPerformed

    private MatchOptions getMatchOptions() {
        // current settings
        GameTypeView gameType = (GameTypeView) cbGameType.getSelectedItem();
        MatchOptions options = new MatchOptions(this.txtName.getText(), gameType.getName(), false, 2);
        options.getPlayerTypes().add(PlayerType.HUMAN);
        for (TablePlayerPanel player : players) {
            options.getPlayerTypes().add(player.getPlayerType());
        }
        options.setDeckType((String) this.cbDeckType.getSelectedItem());
        options.setMatchTimeLimit((MatchTimeLimit) this.cbTimeLimit.getSelectedItem());
        options.setMatchBufferTime((MatchBufferTime) this.cbBufferTime.getSelectedItem());
        options.setAttackOption((MultiplayerAttackOption) this.cbAttackOption.getSelectedItem());
        options.setSkillLevel((SkillLevel) this.cbSkillLevel.getSelectedItem());
        options.setRange((RangeOfInfluence) this.cbRange.getSelectedItem());
        options.setWinsNeeded((Integer) this.spnNumWins.getValue());
        options.setRollbackTurnsAllowed(chkRollbackTurnsAllowed.isSelected());
        options.setSpectatorsAllowed(chkSpectatorsAllowed.isSelected());
        options.setRated(chkRated.isSelected());
        options.setPassword(this.txtPassword.getText());
        options.setQuitRatio((Integer) this.spnQuitRatio.getValue());
        options.setMinimumRating((Integer) this.spnMinimumRating.getValue());
        options.setEdhPowerLevel((Integer) this.spnEdhPowerLevel.getValue());
        String serverAddress = SessionHandler.getSession().getServerHost();
        options.setBannedUsers(IgnoreList.getIgnoredUsers(serverAddress));
        options.setLimited(options.getDeckType().startsWith("Limited"));
        if (options.getDeckType().startsWith("Variant Magic - Freeform Unlimited Commander")) {
            options.setLimited(true); // limited-style sideboarding with unlimited basics enabled for Freeform Unlimited Commander
        }

        customOptions.writeMatchOptionsTo(options);

        return options;
    }

    /**
     * Checks about not valid game option combinations and shows an error
     * message
     *
     * @return
     */
    private boolean checkMatchOptions(MatchOptions options) {

        // deck => game
        switch (options.getDeckType()) {
            case "Variant Magic - Commander":
            case "Variant Magic - Duel Commander":
            case "Variant Magic - MTGO 1v1 Commander":
            case "Variant Magic - Centurion Commander":
            case "Variant Magic - Penny Dreadful Commander":
                if (!options.getGameType().startsWith("Commander")) {
                    JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Deck type Commander needs also a Commander game type", "Error", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                break;
            case "Variant Magic - Freeform Commander":
                if (!options.getGameType().startsWith("Freeform Commander")) {
                    JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Deck type Freeform Commander needs also a Freeform Commander game type", "Error", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                break;
            case "Variant Magic - Freeform Unlimited Commander":
                if (!options.getGameType().startsWith("Freeform Unlimited Commander")) {
                    JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Deck type Freeform+ Commander needs also a Freeform Unlimited Commander game type", "Error", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                break;
            case "Variant Magic - Brawl":
            case "Variant Magic - Duel Brawl":
                if (!options.getGameType().startsWith("Brawl")) {
                    JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Deck type Brawl needs also a Brawl game type", "Error", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                break;
            case "Variant Magic - Tiny Leaders":
                if (!options.getGameType().startsWith("Tiny Leaders")) {
                    JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Deck type Tiny Leaders needs also a Tiny Leaders game type", "Error", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                break;
            case "Variant Magic - Momir Basic":
                if (!options.getGameType().startsWith("Momir Basic")) {
                    JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Deck type Momir Basic needs also a Momir Basic game type", "Error", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                break;
            case "Variant Magic - Oathbreaker":
                if (!options.getGameType().startsWith("Oathbreaker")) {
                    JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Deck type Oathbreaker needs also a Oathbreaker game type", "Error", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                break;
        }

        // game => deck
        switch (options.getGameType()) {
            case "Commander Two Player Duel":
            case "Commander Free For All":
                if (!options.getDeckType().equals("Variant Magic - Commander")
                        && !options.getDeckType().equals("Variant Magic - Duel Commander")
                        && !options.getDeckType().equals("Variant Magic - MTGO 1v1 Commander")
                        && !options.getDeckType().equals("Variant Magic - Centurion Commander")
                        && !options.getDeckType().equals("Variant Magic - Freeform Commander")
                        && !options.getDeckType().equals("Variant Magic - Penny Dreadful Commander")) {
                    JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Deck type Commander needs also a Commander game type", "Error", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                break;
            case "Freeform Commander Two Player Duel":
            case "Freeform Commander Free For All":
                if (!options.getDeckType().equals("Variant Magic - Freeform Commander")) {
                    JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Deck type Freeform Commander needs also a Freeform Commander game type", "Error", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                break;
            case "Freeform Unlimited Commander":
                if (!options.getDeckType().equals("Variant Magic - Freeform Unlimited Commander")) {
                    JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Deck type Freeform Unlimited Commander needs also a Freeform Unlimited Commander game type", "Error", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                break;
            case "Brawl Two Player Duel":
            case "Brawl Free For All":
                if (!options.getDeckType().equals("Variant Magic - Brawl")
                        && !options.getDeckType().equals("Variant Magic - Duel Brawl")) {
                    JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Deck type Brawl needs also a Brawl game type", "Error", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                break;
            case "Tiny Leaders Two Player Duel":
                if (!options.getDeckType().equals("Variant Magic - Tiny Leaders")) {
                    JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Deck type Tiny Leaders needs also a Tiny Leaders game type", "Error", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                break;
            case "Oathbreaker Two Player Duel":
            case "Oathbreaker Free For All":
                if (!options.getDeckType().equals("Variant Magic - Oathbreaker")) {
                    JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Deck type Oathbreaker needs also a Oathbreaker game type", "Error", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                break;
        }
        return true;
    }

    private void setGameOptions() {
        GameTypeView gameType = (GameTypeView) cbGameType.getSelectedItem();
        int oldValue = (Integer) this.spnNumPlayers.getValue();
        this.spnNumPlayers.setModel(new SpinnerNumberModel(gameType.getMinPlayers(), gameType.getMinPlayers(), gameType.getMaxPlayers(), 1));
        this.spnNumPlayers.setEnabled(gameType.getMinPlayers() != gameType.getMaxPlayers());
        if (oldValue >= gameType.getMinPlayers() && oldValue <= gameType.getMaxPlayers()) {
            this.spnNumPlayers.setValue(oldValue);
        }
        this.cbAttackOption.setEnabled(gameType.isUseAttackOption());
        this.cbRange.setEnabled(gameType.isUseRange());
        // hide multiplayer options row if none are editable, otherwise show it
        JComponent[] multiplayerOptions = {
                lblNumPlayers,
                spnNumPlayers,
                lblRange,
                cbRange,
                lblAttack,
                cbAttackOption,
        };
        boolean showMultiplayerOptions = Arrays.stream(multiplayerOptions).anyMatch(c -> !(c instanceof JLabel) && c.isEnabled());
        for (JComponent component : multiplayerOptions) {
            component.setVisible(showMultiplayerOptions);
        }
        createPlayers((Integer) spnNumPlayers.getValue() - 1);
    }

    private void createPlayers(int numPlayers) {
        // add miss panels
        if (numPlayers > players.size()) {
            while (players.size() != numPlayers) {
                TablePlayerPanel playerPanel = new TablePlayerPanel();
                players.add(playerPanel);
                playerPanel.addPlayerTypeEventListener(
                        (Listener<Event>) event -> drawPlayers()
                );
            }
        }

        // remove un-used panels
        if (numPlayers < players.size()) {
            while (players.size() != numPlayers) {
                players.remove(players.size() - 1);
            }
        }

        // load player data
        String prevGoodPlayerDeck = "";
        for (int i = 0; i < players.size(); i++) {
            TablePlayerPanel playerPanel = players.get(i);

            // find player type
            PlayerType playerType = PlayerType.HUMAN;
            if (i < prefPlayerTypes.size()) {
                playerType = prefPlayerTypes.get(i);
            }

            // find skill level
            int playerSkill = DEFAULT_COMPUTER_PLAYER_SKILL_LEVEL;
            if (i < prefPlayerSkills.size()) {
                playerSkill = prefPlayerSkills.get(i);
            }

            // find deck file
            String playerDeck = "";
            if (i < prefPlayerDecks.size()) {
                playerDeck = prefPlayerDecks.get(i);
                // use prev deck if loaded not found
                if (playerDeck.isEmpty() || !(new File(playerDeck).exists())) {
                    playerDeck = prevGoodPlayerDeck;
                } else {
                    prevGoodPlayerDeck = playerDeck;
                }
            }

            playerPanel.init(i + 2, playerType, playerSkill, playerDeck);
        }

        drawPlayers();
    }

    private void drawPlayers() {
        this.pnlOtherPlayers.removeAll();
        for (TablePlayerPanel panel : players) {
            this.pnlOtherPlayers.add(panel);
        }
        this.pack();
        this.revalidate();
        this.repaint();
    }

    private void handleError(Exception e) {
        logger.fatal("Can't join table due " + e, e);
        MageFrame.getInstance().showErrorDialog("CLIENT - error on join table", e);
    }

    public void showDialog(UUID roomId) {
        this.roomId = roomId;
        if (!lastSessionId.equals(SessionHandler.getSessionId())) {
            lastSessionId = SessionHandler.getSessionId();
            this.player1Panel.setPlayerName(SessionHandler.getUserName());
            cbGameType.setModel(new DefaultComboBoxModel(SessionHandler.getGameTypes().toArray()));
            cbDeckType.setModel(new DefaultComboBoxModel(SessionHandler.getDeckTypes()));
            selectLimitedByDefault();
            cbTimeLimit.setModel(new DefaultComboBoxModel(MatchTimeLimit.values()));
            cbBufferTime.setModel(new DefaultComboBoxModel(MatchBufferTime.values()));
            cbRange.setModel(new DefaultComboBoxModel(RangeOfInfluence.values()));
            cbAttackOption.setModel(new DefaultComboBoxModel(MultiplayerAttackOption.values()));
            cbSkillLevel.setModel(new DefaultComboBoxModel(SkillLevel.values()));
            this.setModal(true);
            setGameOptions();
            this.setLocation(150, 100);
        }

        // auto-load last settings
        onLoadSettings(0);

        this.setVisible(true);
    }

    public TableView getTable() {
        return table;
    }

    public UUID getPlayerId() {
        return playerId;
    }

    private void selectLimitedByDefault() {
        for (int i = 0; i < cbDeckType.getItemCount(); i++) {
            String name = (String) cbDeckType.getItemAt(i);
            if (name.equals(LIMITED)) {
                cbDeckType.setSelectedIndex(i);
                break;
            }
        }
    }

    private void onLoadSettings(int version) {

        String versionStr = "";
        switch (version) {
            case -1:
                versionStr = "-1"; // default (empty)
                break;
            case 1:
                versionStr = "1";
                break;
            case 2:
                versionStr = "2";
                break;
            default:
                versionStr = "";
                break;
        }

        txtName.setText(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_NAME + versionStr, "Game"));
        txtPassword.setText(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_PASSWORD + versionStr, ""));

        // load player data
        // player type
        String playerData = PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_PLAYER_TYPES + versionStr, "Human");
        prefPlayerTypes.clear();
        for (String playerTypeStr : playerData.split(PLAYER_DATA_DELIMETER_OLD)) {
            prefPlayerTypes.add(PlayerType.getByDescription(playerTypeStr));
        }
        // player skill
        playerData = PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_PLAYER_SKILLS + versionStr, String.valueOf(DEFAULT_COMPUTER_PLAYER_SKILL_LEVEL));
        prefPlayerSkills.clear();
        for (String playerSkillStr : playerData.split(PLAYER_DATA_DELIMETER_NEW)) {
            prefPlayerSkills.add(Integer.parseInt(playerSkillStr));
        }
        // player deck
        playerData = PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_PLAYER_DECKS + versionStr, "Human");
        prefPlayerDecks.clear();
        prefPlayerDecks.addAll(Arrays.asList(playerData.split(PLAYER_DATA_DELIMETER_NEW)));

        this.spnNumPlayers.setValue(Integer.parseInt(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_NUMBER_PLAYERS + versionStr, "2")));

        String gameTypeName = PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_GAME_TYPE + versionStr, "Two Player Duel");
        for (GameTypeView gtv : SessionHandler.getGameTypes()) {
            if (gtv.getName().equals(gameTypeName)) {
                cbGameType.setSelectedItem(gtv);
                break;
            }
        }
        int timeLimit = Integer.parseInt(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_TIME_LIMIT + versionStr, "1500"));
        for (MatchTimeLimit mtl : MatchTimeLimit.values()) {
            if (mtl.getPrioritySecs() == timeLimit) {
                this.cbTimeLimit.setSelectedItem(mtl);
                break;
            }
        }
        // TODO: Rethink defaults with buffer time?
        int bufferTime = Integer.parseInt(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_BUFFER_TIME + versionStr, "0"));
        for (MatchBufferTime mtl : MatchBufferTime.values()) {
            if (mtl.getBufferSecs() == bufferTime) {
                this.cbBufferTime.setSelectedItem(mtl);
                break;
            }
        }
        cbDeckType.setSelectedItem(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_DECK_TYPE + versionStr, "Limited"));
        String deckFile = PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_DECK_FILE + versionStr, null);
        if (deckFile != null) {
            this.player1Panel.setDeckFile(deckFile);
        }
        this.spnNumWins.setValue(Integer.parseInt(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_NUMBER_OF_WINS + versionStr, "2")));
        this.chkRollbackTurnsAllowed.setSelected(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_ROLLBACK_TURNS_ALLOWED + versionStr, "Yes").equals("Yes"));
        this.chkSpectatorsAllowed.setSelected(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_SPECTATORS_ALLOWED + versionStr, "Yes").equals("Yes"));
        this.chkRated.setSelected(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_RATED + versionStr, "No").equals("Yes"));

        int range = Integer.parseInt(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_RANGE + versionStr, "0"));
        for (RangeOfInfluence roi : RangeOfInfluence.values()) {
            if (roi.getRange() == range) {
                this.cbRange.setSelectedItem(roi);
                break;
            }
        }
        String attackOption = PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_ATTACK_OPTION + versionStr, "Attack Multiple Players");
        for (MultiplayerAttackOption mao : MultiplayerAttackOption.values()) {
            if (mao.toString().equals(attackOption)) {
                this.cbAttackOption.setSelectedItem(mao);
                break;
            }
        }
        String skillLevelDefault = PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_SKILL_LEVEL + versionStr, "Casual");
        for (SkillLevel skillLevel : SkillLevel.values()) {
            if (skillLevel.toString().equals(skillLevelDefault)) {
                this.cbSkillLevel.setSelectedItem(skillLevel);
                break;
            }
        }

        this.spnQuitRatio.setValue(Integer.parseInt(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_QUIT_RATIO + versionStr, "100")));
        this.spnMinimumRating.setValue(Integer.parseInt(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_MINIMUM_RATING + versionStr, "0")));
        this.spnEdhPowerLevel.setValue(Integer.parseInt(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_EDH_POWER_LEVEL + versionStr, "0")));

        this.customOptions.onLoadSettings(version);
    }

    private void onSaveSettings(int version) {
        MatchOptions options = getMatchOptions();
        onSaveSettings(version, options, this.player1Panel.getDeckFile());
    }

    private void onSaveSettings(int version, MatchOptions options, String deckFile) {

        String versionStr = "";
        switch (version) {
            case 1:
                versionStr = "1";
                break;
            case 2:
                versionStr = "2";
                break;
            default:
                versionStr = "";
                break;
        }

        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_NAME + versionStr, options.getName());
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_PASSWORD + versionStr, options.getPassword());
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_DECK_TYPE + versionStr, options.getDeckType());
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_TIME_LIMIT + versionStr, Integer.toString(options.getMatchTimeLimit().getPrioritySecs()));
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_BUFFER_TIME + versionStr, Integer.toString(options.getMatchBufferTime().getBufferSecs()));
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_GAME_TYPE + versionStr, options.getGameType());
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_NUMBER_OF_WINS + versionStr, Integer.toString(options.getWinsNeeded()));
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_ROLLBACK_TURNS_ALLOWED + versionStr, options.isRollbackTurnsAllowed() ? "Yes" : "No");
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_RATED + versionStr, options.isRated() ? "Yes" : "No");
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_DECK_FILE + versionStr, deckFile);
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_NUMBER_PLAYERS + versionStr, spnNumPlayers.getValue().toString());
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_SKILL_LEVEL + versionStr, options.getSkillLevel().toString());
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_SPECTATORS_ALLOWED + versionStr, options.isSpectatorsAllowed() ? "Yes" : "No");
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_QUIT_RATIO + versionStr, Integer.toString(options.getQuitRatio()));
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_MINIMUM_RATING + versionStr, Integer.toString(options.getMinimumRating()));
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_EDH_POWER_LEVEL + versionStr, Integer.toString(options.getEdhPowerLevel()));

        // save player data
        // player type
        String playerData = players.stream()
                .map(panel -> panel.getPlayerType().toString())
                .collect(Collectors.joining(PLAYER_DATA_DELIMETER_OLD));
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_PLAYER_TYPES + versionStr, playerData);
        // player skill
        playerData = players.stream()
                .map(panel -> String.valueOf(panel.getPlayerSkill()))
                .collect(Collectors.joining(PLAYER_DATA_DELIMETER_NEW));
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_PLAYER_SKILLS + versionStr, playerData);
        // player deck
        playerData = players.stream()
                .map(panel -> String.valueOf(panel.getPlayerDeck()))
                .collect(Collectors.joining(PLAYER_DATA_DELIMETER_NEW));
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_PLAYER_DECKS + versionStr, playerData);

        customOptions.onSaveSettings(version, options);
    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton btnCancel;
    private javax.swing.JButton btnCustomOptions;
    private javax.swing.JButton btnOK;
    private javax.swing.JButton btnSettingsLoad;
    private javax.swing.JButton btnSettingsSave;
    private javax.swing.JComboBox cbAttackOption;
    private javax.swing.JComboBox cbBufferTime;
    private javax.swing.JComboBox cbDeckType;
    private javax.swing.JComboBox cbGameType;
    private javax.swing.JComboBox cbRange;
    private javax.swing.JComboBox cbSkillLevel;
    private javax.swing.JComboBox cbTimeLimit;
    private javax.swing.JCheckBox chkRated;
    private javax.swing.JCheckBox chkRollbackTurnsAllowed;
    private javax.swing.JCheckBox chkSpectatorsAllowed;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JSeparator jSeparator1;
    private javax.swing.JSeparator jSeparator2;
    private javax.swing.JSeparator jSeparator3;
    private javax.swing.JLabel lbBufferTime;
    private javax.swing.JLabel lbDeckType;
    private javax.swing.JLabel lbTimeLimit;
    private javax.swing.JLabel lblAttack;
    private javax.swing.JLabel lblEdhPowerLevel;
    private javax.swing.JLabel lblGameType;
    private javax.swing.JLabel lblMinimumRating;
    private javax.swing.JLabel lblName;
    private javax.swing.JLabel lblNumPlayers;
    private javax.swing.JLabel lblNumWins;
    private javax.swing.JLabel lblPassword;
    private javax.swing.JLabel lblQuitRatio;
    private javax.swing.JLabel lblRange;
    private javax.swing.JLabel lblSettings;
    private javax.swing.JLabel lblSkillLevel;
    private javax.swing.JMenuItem menuLoadSettings1;
    private javax.swing.JMenuItem menuLoadSettings2;
    private javax.swing.JMenuItem menuLoadSettingsDefault;
    private javax.swing.JMenuItem menuLoadSettingsLast;
    private javax.swing.JMenuItem menuSaveSettings1;
    private javax.swing.JMenuItem menuSaveSettings2;
    private mage.client.table.NewPlayerPanel player1Panel;
    private javax.swing.JPanel pnlOtherPlayers;
    private javax.swing.JPopupMenu popupLoadSettings;
    private javax.swing.JPopupMenu popupSaveSettings;
    private javax.swing.JPopupMenu.Separator separator1;
    private javax.swing.JPopupMenu.Separator separator2;
    private javax.swing.JSpinner spnEdhPowerLevel;
    private javax.swing.JSpinner spnMinimumRating;
    private javax.swing.JSpinner spnNumPlayers;
    private javax.swing.JSpinner spnNumWins;
    private javax.swing.JSpinner spnQuitRatio;
    private javax.swing.JTextField txtName;
    private javax.swing.JTextField txtPassword;
    // End of variables declaration//GEN-END:variables
}
