import { NextResponse } from 'next/server';

import { voiceMap } from '@/utils/voices';

const { v4: uuidv4 } = require('uuid');
const axios = require('axios');

interface Voice {
  categories: {
    audioFileEndpointWithSas?: string;
    defaultStyle?: string;
    iconFileEndpointWithSas?: string;
    name: string;
  }[];
  description?: string;
  id: string;
  isHiddenFromAccPortal?: boolean;
  locale: string;
  masterpieces: {
    audioFileEndpointWithSas?: string;
    categories?: string[];
    description?: string;
    descriptionLocalizationId?: string;
    iconFileEndpointWithSas?: string;
    name?: string;
    ssmlFileEndpointWithSas?: string;
    title?: string;
    titleLocalizationId?: string;
    videoFileEndpointWithSas?: string;
  }[];
  name: string;
  properties: {
    AgeGroups?: string;
    DisplayName: string;
    DisplayVoiceName: string;
    ExpressAsRoleValues?: string;
    ExpressAsStyleValues?: string;
    FrontendVoiceType: string;
    Gender: string;
    IconFileEndpointWithSas?: string;
    IsTtsBreakEnabled?: string;
    LocalName: string;
    LocaleDescription: string;
    OrderInVoiceListApi: string;
    PreviewSentence: string;
    ReleaseScope: string;
    SampleRateHertz: string;
    SecondaryLocales: string;
    ShortName: string;
    VoiceModelKind: string;
    VoiceRoleNameDefinitions: string;
    VoiceRoleNames: string;
    VoiceStyleNameDefinitions: string;
    VoiceStyleNames: string;
    description?: string;
    locale: string;
    localeZH: string;
  };
  samples: {
    languageSamples: {
      audioFileEndpointWithSas?: string;
      languageName?: string;
      locale?: string;
    }[];
    roleSamples: {
      audioFileEndpointWithSas?: string;
      roleName: string;
    }[];
    styleSamples: {
      audioFileEndpointWithSas?: string;
      styleName: string;
    }[];
  };
  shortName: string;
  voiceType: string;
}

const cachedVoiceList: Voice[] = [
  {
    categories: [],
    id: 'abdda02b-9b62-411b-bc39-003fb1d9f2cc',
    locale: 'ko-KR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ko-KR, SoonBokNeural)',
    properties: {
      DisplayName: 'SoonBok',
      DisplayVoiceName: 'SoonBokNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: '순복',
      LocaleDescription: 'Korean',
      OrderInVoiceListApi: '313',
      PreviewSentence:
        '사용자들에게 자연스럽게 이야기하는 앱과 서비스를 만들며 접근성과 사용성을 개선시킵니다.',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ko-KR-SoonBokNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ko-KR',
      localeZH: '韩语(韩国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/78938d0c-5b13-4ea8-a74d-b6083591303b.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A43Z&se=2122-09-27T07%3A52%3A43Z&sr=b&sp=rl&sig=a4BfmVOXUuOTfwt01BB92behlLUwqCZTsFN1pnrAH7A%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ko-KR-SoonBokNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '520f8b71-e1cc-4e80-b9ea-006d2f816864',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, AIGenerate1Neural)',
    properties: {
      DisplayName: 'AIGenerate1',
      DisplayVoiceName: 'AIGenerate1Neural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'AIGenerate1',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '149',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-AIGenerate1Neural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2a97f7b9-d6cd-4a98-8bfa-b4c1001b24a1.wav?sv=2019-07-07&sr=b&sig=EQJbF1Zo1htV4uO5xtAyO%2BiJuI575pRTIge0tJtaVYg%3D&st=2022-08-01T10:34:21Z&se=2122-07-08T10:39:21Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-US-AIGenerate1Neural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '526184e4-10ee-403b-b9b7-01085f46c320',
    locale: 'de-DE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (de-DE, LouisaNeural)',
    properties: {
      DisplayName: 'Louisa',
      DisplayVoiceName: 'LouisaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Louisa',
      LocaleDescription: 'German (Germany)',
      OrderInVoiceListApi: '70',
      PreviewSentence:
        'Erstellen Sie Apps und Dienste, die auf natürliche Art mit Nutzern sprechen, und dadurch Zugänglichkeit und Benutzerfreundlichkeit verbessern.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'de-DE-LouisaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'de-DE',
      localeZH: '德语(德国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/3e822098-69a2-4b59-9d54-e604eca4884b.wav?sv=2019-07-07&sr=b&sig=F%2Fi5B5ZVpXmw0alfhqI0VJ9F9pdwar9jNutVyUW49gc%3D&st=2022-01-19T09:00:41Z&se=2121-12-26T09:05:41Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'de-DE-LouisaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '06526159-4cf8-4c95-91fa-01e83d6fe441',
    locale: 'pt-BR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pt-BR, LeilaNeural)',
    properties: {
      DisplayName: 'Leila',
      DisplayVoiceName: 'LeilaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Leila',
      LocaleDescription: 'Portuguese (Brazil)',
      OrderInVoiceListApi: '359',
      PreviewSentence:
        'Desenvolver aplicativos e serviços que conversem naturalmente com os usuários, melhorando a acessibilidade e usabilidade.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pt-BR-LeilaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'pt-BR',
      localeZH: '葡萄牙语(巴西)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b8e0a4a0-e1ba-4ded-823e-7d9f643ea6b4.wav?sv=2019-07-07&sr=b&sig=giWb3Ts5%2Bz7vBR9yN%2Fx8KfEQDzhvdNZzlF0JSxnzexQ%3D&st=2022-07-01T09:10:27Z&se=2122-06-07T09:15:27Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'pt-BR-LeilaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '06eaa924-21ab-4bd1-9d71-01ed40046865',
    locale: 'ko-KR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ko-KR, GookMinNeural)',
    properties: {
      DisplayName: 'GookMin',
      DisplayVoiceName: 'GookMinNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: '국민',
      LocaleDescription: 'Korean',
      OrderInVoiceListApi: '310',
      PreviewSentence:
        '사용자들에게 자연스럽게 이야기하는 앱과 서비스를 만들며 접근성과 사용성을 개선시킵니다.',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ko-KR-GookMinNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ko-KR',
      localeZH: '韩语(韩国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ab77ad3c-f167-45b7-a827-33033562319e.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A37Z&se=2122-09-27T07%3A52%3A37Z&sr=b&sp=rl&sig=JtoUfe%2FsQsH%2FlBx6rpAco44cJSFd9ft5NyV1WkGrbjY%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ko-KR-GookMinNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '090b3817-3fe2-4c37-8c5f-025db1335032',
    locale: 'vi-VN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (vi-VN, HoaiMyNeural)',
    properties: {
      DisplayName: 'HoaiMy',
      DisplayVoiceName: 'HoaiMyNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Hoài My',
      LocaleDescription: 'Vietnamese',
      OrderInVoiceListApi: '417',
      PreviewSentence:
        'Phát triển phần mềm và dịch vụ có thể giao tiếp tự nhiên với người dùng, nâng cao tính tiếp cận và tính khả dụng của sản phẩm',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'vi-VN-HoaiMyNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'vi-VN',
      localeZH: '越南语(越南)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/da4fea18-8e86-4f3b-8416-508f420b7644.wav?sv=2019-07-07&sr=b&sig=QXouIeBe1mX0tmfuajjG2YjqxbupzoLQ8izXr%2F31Va0%3D&st=2021-10-29T08:28:29Z&se=2121-10-05T08:33:29Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'vi-VN-HoaiMyNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'f078c87b-9723-4050-b57a-026f7efb38e4',
    locale: 'es-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-ES, NilNeural)',
    properties: {
      DisplayName: 'Nil',
      DisplayVoiceName: 'NilNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Nil',
      LocaleDescription: 'Spanish (Spain)',
      OrderInVoiceListApi: '178',
      PreviewSentence:
        'Crea aplicaciones y servicios que hablan de forma natural a los usuarios, y que mejoran la accesibilidad y la facilidad de uso.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-ES-NilNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-ES',
      localeZH: '西班牙语(西班牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/97080208-6fe9-44b6-9995-8c76460f0bdc.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A50Z&se=2122-09-27T07%3A51%3A50Z&sr=b&sp=rl&sig=fP3TMtVFKp66RiKyQ6D8tn0jdH68KjKjOYkI5RwKGaU%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-ES-NilNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'b35c7ab9-7933-4f6b-bf49-028a06114d0b',
    locale: 'ja-JP',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ja-JP, NanamiNeural)',
    properties: {
      DisplayName: 'Nanami',
      DisplayVoiceName: 'NanamiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: '七海',
      LocaleDescription: 'Japanese',
      OrderInVoiceListApi: '290',
      PreviewSentence:
        'アクセシビリティとユーザビリティを向上させながら、ユーザーに自然なことばを話すアプリとサービスを構築する',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ja-JP-NanamiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"cheerful":4,"Default":0}',
      VoiceStyleNames: 'chat,customerservice,cheerful,Default',
      locale: 'ja-JP',
      localeZH: '日语(日本)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6089ffb4-5260-4086-8f51-75e65b4678ed.wav?sv=2019-07-07&sr=b&sig=DW1VaFbJsmgEsezxszcxWaf%2FOK0Y52hddZJ2zSZqtfo%3D&st=2021-10-29T08:27:21Z&se=2121-10-05T08:32:21Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2f074470-e48d-419d-bfb7-53dff173d368.wav?sv=2019-07-07&sr=b&sig=sCPVdT3aBy1p5uM6YOwuSRUU36KaX2Kd0v704E%2F1zNY%3D&st=2021-12-02T05:00:50Z&se=2121-11-08T05:05:50Z&sp=rl',
          styleName: 'chat',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/1dd3d22d-c9ed-4229-b19c-67a85d869065.wav?sv=2019-07-07&sr=b&sig=nPNnWiDyF8W5zOTQ16xMB5tdotIlrRAoVYlunfrbMgk%3D&st=2021-12-02T05:00:53Z&se=2121-11-08T05:05:53Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e075449b-6aa1-4ed2-ae86-e17d82010386.wav?sv=2019-07-07&sr=b&sig=jpYOPxmYaAOf2layv3oJccB7v5DWVxkB3rxqXmabq6I%3D&st=2021-12-02T05:00:55Z&se=2121-11-08T05:05:55Z&sp=rl',
          styleName: 'customerservice',
        },
      ],
    },
    shortName: 'ja-JP-NanamiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'a53f44a0-b3fe-4f02-94d8-03829b8081be',
    locale: 'en-ZA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-ZA, LukeNeural)',
    properties: {
      DisplayName: 'Luke',
      DisplayVoiceName: 'LukeNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Luke',
      LocaleDescription: 'en-ZA',
      OrderInVoiceListApi: '152',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-ZA-LukeNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-ZA',
      localeZH: '英语(南非)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6566a6fe-1ba6-486c-8be2-28b595a24006.wav?sv=2019-07-07&sr=b&sig=%2FfNi4sz%2F2K6ivca%2BR4kEgHQqS72lGdocuRnH3w2pEKk%3D&st=2021-10-29T08:26:39Z&se=2121-10-05T08:31:39Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-ZA-LukeNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '6e185350-7363-4ee0-ab2c-038326c7ff41',
    locale: 'lo-LA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (lo-LA, ChanthavongNeural)',
    properties: {
      DisplayName: 'Chanthavong',
      DisplayVoiceName: 'ChanthavongNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'ຈັນທະວົງ',
      LocaleDescription: 'lo-LA',
      OrderInVoiceListApi: '315',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'lo-LA-ChanthavongNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'lo-LA',
      localeZH: '老挝语(老挝) ',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f039dc4f-87f2-44e0-be98-d9e4b0fb7415.wav?sv=2019-07-07&sr=b&sig=TGU1at1wfGYWJ7v0JXDNNafSbRwx7CeGWr019ll1Q1c%3D&st=2022-01-19T12:26:14Z&se=2121-12-26T12:31:14Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'lo-LA-ChanthavongNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'f4bf856d-6246-40be-a745-03c5e68f7d10',
    locale: 'es-MX',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-MX, CecilioNeural)',
    properties: {
      DisplayName: 'Cecilio',
      DisplayVoiceName: 'CecilioNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Cecilio',
      LocaleDescription: 'Spanish (Mexico)',
      OrderInVoiceListApi: '193',
      PreviewSentence:
        'Crea apps y servicios que hablen de forma natural con los usuarios, mejorando la accesibilidad y la usabilidad.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-MX-CecilioNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-MX',
      localeZH: '西班牙语(墨西哥)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e2dd970c-15e4-49e7-822a-70f4d88b4621.wav?sv=2019-07-07&sr=b&sig=WKcVz5GattaEHgnGPxZLTvved7tDfcyryFsDiyvyLsY%3D&st=2022-07-01T09:09:08Z&se=2122-06-07T09:14:08Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-MX-CecilioNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '25dc980d-ff1f-42e8-bef2-04cbb4bd16ce',
    locale: 'es-MX',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-MX, LucianoNeural)',
    properties: {
      DisplayName: 'Luciano',
      DisplayVoiceName: 'LucianoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Luciano',
      LocaleDescription: 'Spanish (Mexico)',
      OrderInVoiceListApi: '199',
      PreviewSentence:
        'Crea apps y servicios que hablen de forma natural con los usuarios, mejorando la accesibilidad y la usabilidad.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-MX-LucianoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-MX',
      localeZH: '西班牙语(墨西哥)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/10b259e0-a091-410a-9f08-4f1e3736d064.wav?sv=2019-07-07&sr=b&sig=AN%2F5yeSDwQwvL6PsPwe6FrQWIT1IyJhV0t1Zg32nujY%3D&st=2022-07-01T09:09:17Z&se=2122-06-07T09:14:17Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-MX-LucianoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '7777a5ee-439b-4780-88b8-060c527fbaf8',
    locale: 'ta-LK',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ta-LK, KumarNeural)',
    properties: {
      DisplayName: 'Kumar',
      DisplayVoiceName: 'KumarNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'குமார்',
      LocaleDescription: 'ta-LK',
      OrderInVoiceListApi: '396',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ta-LK-KumarNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ta-LK',
      localeZH: '泰米尔语(斯里兰卡)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/71bf3957-51c1-4f2c-929e-ec32be60b1fb.wav?sv=2019-07-07&sr=b&sig=78BDe%2FNgQe5sucPHXH63Krg0j%2BIPaOSEcvN87VMU45o%3D&st=2021-10-29T08:31:19Z&se=2121-10-05T08:36:19Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ta-LK-KumarNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '7ad56aad-0fda-4b70-bb36-06328dda42f5',
    locale: 'de-CH',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (de-CH, JanNeural)',
    properties: {
      DisplayName: 'Jan',
      DisplayVoiceName: 'JanNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Jan',
      LocaleDescription: 'German (Switzerland)',
      OrderInVoiceListApi: '58',
      PreviewSentence:
        'Erstellen Sie Apps und Dienste, die auf natürliche Art mit Nutzern sprechen, und dadurch Zugänglichkeit und Benutzerfreundlichkeit verbessern.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'de-CH-JanNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'de-CH',
      localeZH: '德语(瑞士)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/1b972b9f-df36-4fa2-b72d-4b31d0679cb3.wav?sv=2019-07-07&sr=b&sig=SbDGr%2BH1GbqLibdczO08MexYj1K2nPdzCPNWVx55ojI%3D&st=2021-10-29T08:26:12Z&se=2121-10-05T08:31:12Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'de-CH-JanNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'fe9d5f74-af06-4803-976d-0839093512b1',
    locale: 'gl-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (gl-ES, SabelaNeural)',
    properties: {
      DisplayName: 'Sabela',
      DisplayVoiceName: 'SabelaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Sabela',
      LocaleDescription: 'gl-ES',
      OrderInVoiceListApi: '258',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'gl-ES-SabelaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'gl-ES',
      localeZH: '加利西亚语(加利西亚语)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/dd13a9f3-0195-4342-ac86-430bc2729648.wav?sv=2019-07-07&sr=b&sig=Sldy9Ay1ErHes4xtm57LY0mFMjko3YIkRDj%2Fb0ELVcY%3D&st=2021-10-29T08:31:06Z&se=2121-10-05T08:36:06Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'gl-ES-SabelaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '9a376069-83e6-4a20-b954-093d4a7d1757',
    locale: 'ar-YE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-YE, MaryamNeural)',
    properties: {
      DisplayName: 'Maryam',
      DisplayVoiceName: 'MaryamNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'مريم',
      LocaleDescription: 'ar-YE',
      OrderInVoiceListApi: '35',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-YE-MaryamNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-YE',
      localeZH: '阿拉伯语(也门)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/90604848-6020-4a28-9a25-714a3c6d8270.wav?sv=2019-07-07&sr=b&sig=lW5mn4F%2FhuB18jt8MtWshBr3B862NTdOphqgoW1TYYc%3D&st=2021-10-29T08:30:19Z&se=2121-10-05T08:35:19Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-YE-MaryamNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '61284492-3628-4c8f-a33a-0972a4603ad4',
    locale: 'ja-JP',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ja-JP, DaichiNeural)',
    properties: {
      DisplayName: 'Daichi',
      DisplayVoiceName: 'DaichiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: '大智',
      LocaleDescription: 'Japanese',
      OrderInVoiceListApi: '292',
      PreviewSentence:
        'アクセシビリティとユーザビリティを向上させながら、ユーザーに自然なことばを話すアプリとサービスを構築する',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ja-JP-DaichiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ja-JP',
      localeZH: '日语(日本)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e041111e-38c8-414f-82bc-6dc2be783342.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A26Z&se=2122-09-27T07%3A52%3A26Z&sr=b&sp=rl&sig=F1bD%2BlxnYo8KhVqB%2BlDOZdqtKrQ5hwYUm%2Ba25q94zX0%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ja-JP-DaichiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'afed09aa-94df-4115-9600-09a007819371',
    locale: 'ta-SG',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ta-SG, VenbaNeural)',
    properties: {
      DisplayName: 'Venba',
      DisplayVoiceName: 'VenbaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'வெண்பா',
      LocaleDescription: 'ta-SG',
      OrderInVoiceListApi: '401',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ta-SG-VenbaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ta-SG',
      localeZH: '泰米尔语(新加坡)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4b6bdb06-f687-45b5-8079-6d2d93108e25.wav?sv=2019-07-07&sr=b&sig=IA9VNgHd9XTdTEPA2vS1K75El3srLSGUP6bCA3Dhr%2BM%3D&st=2021-10-29T08:31:22Z&se=2121-10-05T08:36:22Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ta-SG-VenbaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '83b52df6-a6e2-4bd0-9587-09bd9bfb73e4',
    locale: 'kk-KZ',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (kk-KZ, AigulNeural)',
    properties: {
      DisplayName: 'Aigul',
      DisplayVoiceName: 'AigulNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Айгүл',
      LocaleDescription: 'kk-KZ',
      OrderInVoiceListApi: '301',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'kk-KZ-AigulNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'kk-KZ',
      localeZH: '哈萨克语(哈萨克斯坦)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/1ca96ded-31cc-48a9-a30b-7259725a5620.wav?sv=2019-07-07&sr=b&sig=2AXO47wnhJqgS1xmlSR089POsw7R1TocibYBT1uN3Og%3D&st=2022-01-19T09:01:49Z&se=2121-12-26T09:06:49Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'kk-KZ-AigulNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'e9b470fe-c8eb-4aec-a993-09e806741339',
    locale: 'en-NG',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-NG, AbeoNeural)',
    properties: {
      DisplayName: 'Abeo',
      DisplayVoiceName: 'AbeoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Abeo',
      LocaleDescription: 'en-NG',
      OrderInVoiceListApi: '116',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-NG-AbeoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-NG',
      localeZH: '英语(尼日利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/79e59fdb-e530-4921-81dd-50ac3983e9fe.wav?sv=2019-07-07&sr=b&sig=N%2BJCCBPtv6YO4tUOeH0w1I%2B%2BdfK%2FX5tS%2FEvSY%2FqEu6k%3D&st=2021-10-29T08:30:25Z&se=2121-10-05T08:35:25Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-NG-AbeoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'f6d931ff-4b5a-417a-b3ce-0a846a4dda4c',
    locale: 'en-HK',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-HK, SamNeural)',
    properties: {
      DisplayName: 'Sam',
      DisplayVoiceName: 'SamNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Sam',
      LocaleDescription: 'en-HK',
      OrderInVoiceListApi: '108',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-HK-SamNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-HK',
      localeZH: '英语(香港特别行政区)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b8b41f71-8150-4461-b9a5-048bd597f500.wav?sv=2019-07-07&sr=b&sig=8BJDPaxL68pTI9hvgNN20WsVTepUCp8RE3ID0bBPFFI%3D&st=2021-10-29T08:26:26Z&se=2121-10-05T08:31:26Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-HK-SamNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '601ef07e-2f2a-4d7b-b59e-0b3b224948f0',
    locale: 'pt-PT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pt-PT, FernandaNeural)',
    properties: {
      DisplayName: 'Fernanda',
      DisplayVoiceName: 'FernandaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Fernanda',
      LocaleDescription: 'Portuguese (Portugal)',
      OrderInVoiceListApi: '366',
      PreviewSentence:
        'Constrói aplicações e serviços que falam naturalmente com os utilizadores, melhorando a acessibilidade e usabilidade.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pt-PT-FernandaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'pt-PT',
      localeZH: '葡萄牙语(葡萄牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/8dfe864f-9ad2-4e97-9bfd-2838301a7fbc.wav?sv=2019-07-07&sr=b&sig=0tP1MWoh%2FFBsN6c19dn3pDhMrhDJfhLormXYrr9rfFw%3D&st=2021-10-29T08:27:57Z&se=2121-10-05T08:32:57Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'pt-PT-FernandaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'e8e16409-b614-435f-8d10-0ba08088c3c0',
    locale: 'ta-MY',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ta-MY, KaniNeural)',
    properties: {
      DisplayName: 'Kani',
      DisplayVoiceName: 'KaniNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'கனி',
      LocaleDescription: 'ta-MY',
      OrderInVoiceListApi: '398',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ta-MY-KaniNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ta-MY',
      localeZH: '泰米尔语(马来西亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d63b91a7-091c-4103-90e1-86a0f345f7bb.wav?sv=2019-07-07&sr=b&sig=BhACX5wHoqV8fbTvp6ulX1Wfh9QN1eqwEFt2G5M8SR8%3D&st=2022-07-01T09:10:45Z&se=2122-06-07T09:15:45Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ta-MY-KaniNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '405d8cfc-479f-4b4c-80b7-0bf031084928',
    locale: 'es-SV',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-SV, RodrigoNeural)',
    properties: {
      DisplayName: 'Rodrigo',
      DisplayVoiceName: 'RodrigoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Rodrigo',
      LocaleDescription: 'es-SV',
      OrderInVoiceListApi: '216',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-SV-RodrigoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-SV',
      localeZH: '西班牙语(萨尔瓦多)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e48c5bf7-2cc4-46f8-a3bb-a21da247be7b.wav?sv=2019-07-07&sr=b&sig=Obn4dhSnZGk3hrTl3NkMxiq3LcqIiExMReKyUd3pakQ%3D&st=2021-10-29T08:30:57Z&se=2121-10-05T08:35:57Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-SV-RodrigoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'f1b81aff-514c-4ea8-b602-0c56180c7c63',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, ElizabethNeural)',
    properties: {
      DisplayName: 'Elizabeth',
      DisplayVoiceName: 'ElizabethNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Elizabeth',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '137',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-ElizabethNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/07aa3bb6-312f-4441-b8a1-e8da1e1fe1b3.wav?sv=2019-07-07&sr=b&sig=qwjhzO1FxVyVmnbzno6%2FL22EBjIyAjxmc20IQ7qOTHM%3D&st=2021-10-29T08:25:35Z&se=2121-10-05T08:30:35Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-US-ElizabethNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'ada99918-f740-4e47-86f7-0d3c8e95c027',
    locale: 'en-AU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-AU, WilliamNeural)',
    properties: {
      DisplayName: 'William',
      DisplayVoiceName: 'WilliamNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'William',
      LocaleDescription: 'English (Australia)',
      OrderInVoiceListApi: '90',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-AU-WilliamNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-AU',
      localeZH: '英语(澳大利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/56e80a84-57f2-4f25-9797-e692299001a8.wav?sv=2019-07-07&sr=b&sig=fDDKppahiltzRkh8z8%2B8haRW1yWoTLYRbB0ZHx%2B35qA%3D&st=2021-10-29T08:26:19Z&se=2121-10-05T08:31:19Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-AU-WilliamNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'e4fbab32-f3f3-4943-b4db-0d8a7469b383',
    locale: 'en-GB',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-GB, LibbyNeural)',
    properties: {
      DisplayName: 'Libby',
      DisplayVoiceName: 'LibbyNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Libby',
      LocaleDescription: 'English (UK)',
      OrderInVoiceListApi: '93',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-GB-LibbyNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-GB',
      localeZH: '英语(英国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/8a582811-2778-4133-8050-619759d43d98.wav?sv=2019-07-07&sr=b&sig=PeL5OIL%2B%2FOWz5t3YQfYCqoJSvob%2B6%2BfdOdZbdPYYr5I%3D&st=2021-10-29T08:26:22Z&se=2121-10-05T08:31:22Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-GB-LibbyNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '38715012-e919-4142-acbf-0eba9d45b70c',
    locale: 'en-AU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-AU, AnnetteNeural)',
    properties: {
      DisplayName: 'Annette',
      DisplayVoiceName: 'AnnetteNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Annette',
      LocaleDescription: 'English (Australia)',
      OrderInVoiceListApi: '77',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-AU-AnnetteNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-AU',
      localeZH: '英语(澳大利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/145d0b78-3bba-4579-9a6c-43fc46dc3aaa.wav?sv=2021-08-06&st=2022-10-21T07%3A45%3A58Z&se=2122-09-27T07%3A50%3A58Z&sr=b&sp=rl&sig=SEgnOd712ESUt1k3FmwN1DNTaIyUn5z7IfLC0RqtR0E%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-AU-AnnetteNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'a2dd132c-bfcc-4215-9077-0fcd2d57255e',
    locale: 'cy-GB',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (cy-GB, AledNeural)',
    properties: {
      DisplayName: 'Aled',
      DisplayVoiceName: 'AledNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Aled',
      LocaleDescription: 'Welsh (UK)',
      OrderInVoiceListApi: '52',
      PreviewSentence:
        'Mae’r feddalwedd creu cynnwys sain yn galluogi rheoli priodoleddau lleferydd yn weledol mewn amser real.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'cy-GB-AledNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'cy-GB',
      localeZH: '威尔士语(英国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5fd043f7-0f9d-42d9-92a5-e6b999aa3add.wav?sv=2019-07-07&sr=b&sig=4T6FY9gfX77DWDF5YEZq8EvD5zk0GOcXvQs4CVsKqtc%3D&st=2021-10-29T08:26:06Z&se=2121-10-05T08:31:06Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'cy-GB-AledNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '6077350a-dec0-40d0-8497-11976c4bb81f',
    locale: 'it-IT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (it-IT, ImeldaNeural)',
    properties: {
      DisplayName: 'Imelda',
      DisplayVoiceName: 'ImeldaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Imelda',
      LocaleDescription: 'Italian',
      OrderInVoiceListApi: '283',
      PreviewSentence:
        'Benvenuti nella piattaforma di generazione di contenuti audio di Microsoft.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'it-IT-ImeldaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'it-IT',
      localeZH: '意大利语(意大利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/524b11c6-9532-4a86-866a-45acd064f25c.wav?sv=2019-07-07&sr=b&sig=AJK5E1ntsuiqKtxmd7rZIcwz6fDkiVoVdJVokMg6Hkg%3D&st=2022-07-01T09:09:44Z&se=2122-06-07T09:14:44Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'it-IT-ImeldaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '16b6c635-eef5-43eb-87d3-11c780304891',
    locale: 'es-AR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-AR, ElenaNeural)',
    properties: {
      DisplayName: 'Elena',
      DisplayVoiceName: 'ElenaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Elena',
      LocaleDescription: 'es-AR',
      OrderInVoiceListApi: '153',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-AR-ElenaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-AR',
      localeZH: '西班牙语(阿根廷)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/bbf0e26e-6888-478c-ad80-3e90b9f0376b.wav?sv=2019-07-07&sr=b&sig=GZp8WvCjYJQAOEEovqXIJTWtF6IUIVtmFwlNFzfLioY%3D&st=2021-10-29T08:26:40Z&se=2121-10-05T08:31:40Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-AR-ElenaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '2367bbe4-0039-4222-a92a-12b37d66a362',
    isHiddenFromAccPortal: true,
    locale: 'en-GB',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-GB, MiaNeural)',
    properties: {
      DisplayName: 'Mia',
      DisplayVoiceName: 'MiaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Mia',
      LocaleDescription: 'English (UK)',
      OrderInVoiceListApi: '107',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicDeprecated',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-GB-MiaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-GB',
      localeZH: '英语(英国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b9488997-6d44-449f-a3ee-6dfafa95fb6f.wav?sv=2019-07-07&sr=b&sig=z%2BcV50uz327knPEC5jczY2bGQb6DP1FZ1WPi%2B1aerfY%3D&st=2021-10-29T08:26:23Z&se=2121-10-05T08:31:23Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-GB-MiaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '6f1346c4-e658-4395-9e98-132b0e4c1a62',
    locale: 'es-CO',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-CO, SalomeNeural)',
    properties: {
      DisplayName: 'Salome',
      DisplayVoiceName: 'SalomeNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Salome',
      LocaleDescription: 'es-CO',
      OrderInVoiceListApi: '160',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-CO-SalomeNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-CO',
      localeZH: '西班牙语(哥伦比亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/cde05b67-aba9-4edb-8473-5a11b8fcbebc.wav?sv=2019-07-07&sr=b&sig=Y4l00UeNRU1Xb04GlXbTddyA7Idgdnif5x5W%2FLS%2F%2FRk%3D&st=2021-10-29T08:26:42Z&se=2121-10-05T08:31:42Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-CO-SalomeNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '0c48f5ae-fe62-4be3-a239-134492206ac5',
    locale: 'fi-FI',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fi-FI, SelmaNeural)',
    properties: {
      DisplayName: 'Selma',
      DisplayVoiceName: 'SelmaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Selma',
      LocaleDescription: 'Finnish',
      OrderInVoiceListApi: '229',
      PreviewSentence:
        'Kehitä luonnolisesti puhuvia sovelluksia ja palveluja, jotka parantavat käytettävyyttä ja saavutettavuutta.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fi-FI-SelmaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fi-FI',
      localeZH: '芬兰语(芬兰)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/bc7d9122-ab4d-4cfe-8b47-afd69188ada0.wav?sv=2019-07-07&sr=b&sig=%2FmLYpAIv6iiioD0HKAhBE2Lym%2FZQxVaEbY6fe%2F6b0tA%3D&st=2021-10-29T08:26:53Z&se=2121-10-05T08:31:53Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fi-FI-SelmaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '17e97d15-e286-416b-bc1b-13f7b3cd67af',
    locale: 'ko-KR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ko-KR, SunHiNeural)',
    properties: {
      DisplayName: 'Sun-Hi',
      DisplayVoiceName: 'SunHiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: '선히',
      LocaleDescription: 'Korean',
      OrderInVoiceListApi: '307',
      PreviewSentence:
        '사용자들에게 자연스럽게 이야기하는 앱과 서비스를 만들며 접근성과 사용성을 개선시킵니다.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ko-KR-SunHiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ko-KR',
      localeZH: '韩语(韩国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f118c57b-9800-49ef-bb05-e3f57226bced.wav?sv=2019-07-07&sr=b&sig=KEFcz3Ot3EUbTqIbYIEDHj5dfryoATDldp7ySWCQgQc%3D&st=2021-10-29T08:27:23Z&se=2121-10-05T08:32:23Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ko-KR-SunHiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '865ed125-9b77-4022-bf44-142ca2522695',
    locale: 'en-GB',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-GB, RyanNeural)',
    properties: {
      DisplayName: 'Ryan',
      DisplayVoiceName: 'RyanNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Ryan',
      LocaleDescription: 'English (UK)',
      OrderInVoiceListApi: '104',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-GB-RyanNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"cheerful":4,"Default":0}',
      VoiceStyleNames: 'cheerful,chat,Default',
      locale: 'en-GB',
      localeZH: '英语(英国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e08d4f27-caef-4438-9935-25b866147878.wav?sv=2019-07-07&sr=b&sig=XUg1ZhsDfFZxLu06i2E%2F1OTBJO7XevlCUDhPVxHSpms%3D&st=2021-10-29T08:26:24Z&se=2121-10-05T08:31:24Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fe6f4fa7-e825-4524-9cfe-edc241e31337.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A23Z&se=2122-09-27T07%3A51%3A23Z&sr=b&sp=rl&sig=mMDG1iL%2BLrgaEbT%2Bfie%2BX4eeEaZ7dNTrQJz8qoIhanU%3D',
          styleName: 'chat',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d566b16e-4bd9-45f8-8096-97a42976df09.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A25Z&se=2122-09-27T07%3A51%3A25Z&sr=b&sp=rl&sig=PKGoanHoegkj1Q9C2Tkqv%2BScPS%2FHh%2FFW%2F%2B%2F87rsnm4E%3D',
          styleName: 'cheerful',
        },
      ],
    },
    shortName: 'en-GB-RyanNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'e99ee79d-06fc-4265-a27b-14c7fe231aca',
    locale: 'ja-JP',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ja-JP, ShioriNeural)',
    properties: {
      DisplayName: 'Shiori',
      DisplayVoiceName: 'ShioriNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: '志織',
      LocaleDescription: 'Japanese',
      OrderInVoiceListApi: '296',
      PreviewSentence:
        'アクセシビリティとユーザビリティを向上させながら、ユーザーに自然なことばを話すアプリとサービスを構築する',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ja-JP-ShioriNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ja-JP',
      localeZH: '日语(日本)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/69a73884-64a5-414e-a60e-fa7ab41c7e7f.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A32Z&se=2122-09-27T07%3A52%3A32Z&sr=b&sp=rl&sig=lxEqTOhKWbGl%2BZh3Z2IFjxYcTl4wywHLftbhRw8Fyiw%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ja-JP-ShioriNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '43572d52-a314-43ff-97d7-151585fb0f65',
    locale: 'ar-SA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-SA, HamedNeural)',
    properties: {
      DisplayName: 'Hamed',
      DisplayVoiceName: 'HamedNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'حامد',
      LocaleDescription: 'Arabic (Saudi Arabia)',
      OrderInVoiceListApi: '29',
      PreviewSentence:
        'إن التطبيقات التي تتحاور مع المستخدمين بصوره طبيعية،  تعمل على  تحسين امكانية الوصول اليها وسهولة الاستخدام',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-SA-HamedNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-SA',
      localeZH: '阿拉伯语(沙特阿拉伯)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e372a792-f10f-43fc-8f48-845cfca3b8f7.wav?sv=2019-07-07&sr=b&sig=hW8mIMoEDEZ%2FSKGYBL8AzTfmq2d1i0scvjTjX9Eog2k%3D&st=2021-10-29T08:25:57Z&se=2121-10-05T08:30:57Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-SA-HamedNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [
      {
        name: 'All',
      },
      {
        name: 'Audiobook',
      },
      {
        name: 'Chat',
      },
    ],
    description:
      'Confident and capable voice with rich market-play and emotions that is optimized for audio books.',
    id: '197d529b-8158-4bec-ab83-156c8485e81c',
    locale: 'zh-CN',
    masterpieces: [
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/31ca3221-7068-4b13-83ae-19ec4b0950ea?sv=2019-07-07&sr=b&sig=pJIcad7mIUUHo8OQNqltil%2Fr0v4EQPD%2BFpSdVo952%2Fw%3D&st=2021-10-29T08:24:50Z&se=2121-10-05T08:29:50Z&sp=rl',
        categories: ['Audiobook'],
        title: '晓萱有声书《迟桂花》',
        titleLocalizationId: 'zh-CN-XiaoxuanNeural_Masterpieces_Audiobook_ChiGuiHua_Title',
      },
    ],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, XiaoxuanNeural)',
    properties: {
      AgeGroups: 'YoungAdult',
      DisplayName: 'Xiaoxuan',
      DisplayVoiceName: 'XiaoxuanNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      IconFileEndpointWithSas:
        'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/3d066c1a-945b-4ba9-be68-7eba4c68c0d6?sv=2019-07-07&sr=b&sig=wIafT4580SFjW5tLBXIrp4zXx71rwuM0EuiEC1G2k%2BQ%3D&st=2021-10-29T08:24:27Z&se=2121-10-05T08:29:27Z&sp=rl',
      IsTtsBreakEnabled: 'True',
      LocalName: '晓萱',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '432',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-XiaoxuanNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions:
        '{"YoungAdultFemale":4,"YoungAdultMale":1,"OlderAdultFemale":5,"OlderAdultMale":2,"SeniorFemale":10,"SeniorMale":3,"Girl":8,"Boy":9,"Default":0}',
      VoiceRoleNames:
        'YoungAdultFemale,YoungAdultMale,OlderAdultFemale,OlderAdultMale,SeniorFemale,SeniorMale,Girl,Boy,Default',
      VoiceStyleNameDefinitions:
        '{"calm":2,"fearful":3,"cheerful":4,"disgruntled":5,"serious":6,"angry":7,"gentle":11,"depressed":10,"Default":0}',
      VoiceStyleNames: 'calm,fearful,cheerful,disgruntled,serious,angry,gentle,depressed,Default',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/528b1233-50c5-447e-acfa-bd79ca102c4b.wav?sv=2019-07-07&sr=b&sig=IMjiIvR9kV3eQdFOlmXkDhLS6Nx40gBLKeo7ps41kaE%3D&st=2021-10-29T08:31:39Z&se=2121-10-05T08:36:39Z&sp=rl',
          roleName: 'Default',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c7c9298c-d99d-4a85-896d-7a03d035070d.wav?sv=2019-07-07&sr=b&sig=HkCCkhJpTwVOdksUxHYUeOGCjPfwbLBuKKTaZFm0Oe4%3D&st=2021-10-29T08:31:38Z&se=2121-10-05T08:36:38Z&sp=rl',
          roleName: 'Boy',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/3e0ce86e-9418-4f30-b365-32e15b83d70c.wav?sv=2019-07-07&sr=b&sig=%2FQQbA5f7LMP%2B%2BuJIThvKZ4dl99JrwlTr5kb4Tgf51YY%3D&st=2021-10-29T08:31:40Z&se=2121-10-05T08:36:40Z&sp=rl',
          roleName: 'Girl',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fc462f38-f226-4f4b-934a-8e3e2770df0f.wav?sv=2019-07-07&sr=b&sig=C8W7ENarMQni8txKiJ7Dyy0UV3OIkSeeV6p%2B4Mgmg0E%3D&st=2021-10-29T08:31:41Z&se=2121-10-05T08:36:41Z&sp=rl',
          roleName: 'OlderAdultFemale',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/df1316f4-5d20-4a2a-aba8-36b85cf951fe.wav?sv=2019-07-07&sr=b&sig=SCM3HU2T8e%2FJeN4fCicmOKUfJU0CJSiKM1YLdTkwrNY%3D&st=2021-10-29T08:31:42Z&se=2121-10-05T08:36:42Z&sp=rl',
          roleName: 'OlderAdultMale',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4f5e875a-10ca-46af-856c-c759ef50da07.wav?sv=2019-07-07&sr=b&sig=FnfZj2FcUePBUN%2FC2QRXIkgWz16rhlS1TL%2BHz3wtm0k%3D&st=2021-10-29T08:31:42Z&se=2121-10-05T08:36:42Z&sp=rl',
          roleName: 'SeniorFemale',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/1ba9d228-f0aa-4551-83ff-78952b36c50c.wav?sv=2019-07-07&sr=b&sig=O7EvYbl8ZpInki6m3GPUVopeKSYRZPT%2F4GoEqnD3veI%3D&st=2021-10-29T08:31:43Z&se=2121-10-05T08:36:43Z&sp=rl',
          roleName: 'SeniorMale',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6e0af0e0-c30e-4c0d-b586-6b9ad24dc8d5.wav?sv=2019-07-07&sr=b&sig=sSBzpbYk7xnekhnTnxI7y785oArgucwPhMCTK4FDIUM%3D&st=2021-10-29T08:31:44Z&se=2121-10-05T08:36:44Z&sp=rl',
          roleName: 'YoungAdultFemale',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0c7837f6-8932-4d79-8727-bcc8f253f3f2.wav?sv=2019-07-07&sr=b&sig=uRuPt4D%2Fjc9pgSGGDN3fCjKUNRmDKR7kk%2BfnlcKJ1Ro%3D&st=2021-10-29T08:31:45Z&se=2121-10-05T08:36:45Z&sp=rl',
          roleName: 'YoungAdultMale',
        },
      ],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b698c45c-cfdc-4daf-b9f2-6824abddbe18.wav?sv=2019-07-07&sr=b&sig=YXzbdxaG8LaNyIr1pf1bUnMkwKRo6MZYRJbNXG9F81I%3D&st=2021-10-29T08:29:11Z&se=2121-10-05T08:34:11Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/1c50d80a-d118-421b-a7d7-1e84ea255a43.wav?sv=2019-07-07&sr=b&sig=ejc8H0CeS%2F5b8cv%2B%2FtLFal9gVmujjmwhGftg9aFTbBI%3D&st=2021-10-29T08:29:12Z&se=2121-10-05T08:34:12Z&sp=rl',
          styleName: 'calm',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/65cca866-93f8-42b3-890e-cfdfa5868c67.wav?sv=2019-07-07&sr=b&sig=JiTG0Lm88dIflX%2FlA%2FWSRzISppcaIvOSDOgB%2BuX2Sfs%3D&st=2021-10-29T08:29:13Z&se=2121-10-05T08:34:13Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5986a37c-68cc-4376-9403-182a9a5ea506.wav?sv=2019-07-07&sr=b&sig=EzOoQTk%2B3fiDqxu4FlyoPbd6Q8q9ZwZbJ5VevBOrOZQ%3D&st=2021-10-29T08:29:14Z&se=2121-10-05T08:34:14Z&sp=rl',
          styleName: 'depressed',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/255d323e-a477-41b6-a93a-655df5d2d8f9.wav?sv=2019-07-07&sr=b&sig=UWFMk5xrkyctMbwqa%2FCDjuhExtttDETua2aKKyEfHS8%3D&st=2021-10-29T08:29:15Z&se=2121-10-05T08:34:15Z&sp=rl',
          styleName: 'disgruntled',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/52f1e82c-ceb6-4012-bd98-1c33a9e0d698.wav?sv=2019-07-07&sr=b&sig=7%2FEPU%2BbiSk9esQIMw47Hn%2BmJfn7c2pKhy7S1eyKTuM8%3D&st=2021-10-29T08:29:16Z&se=2121-10-05T08:34:16Z&sp=rl',
          styleName: 'fearful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6c611f99-4298-4430-b92e-56effdae3a25.wav?sv=2019-07-07&sr=b&sig=Kz22JI%2BfE1hbBdI%2BjdnAg%2FG860CK4rO8k9QqfRCc4Os%3D&st=2021-10-29T08:29:17Z&se=2121-10-05T08:34:17Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b0ed5589-2b41-4658-8d08-73d7da6e2d00.wav?sv=2019-07-07&sr=b&sig=IXV9BKaFIRsFZWPyijyApoXYFBVEpq%2B76KvNPCZL3vU%3D&st=2021-10-29T08:29:18Z&se=2121-10-05T08:34:18Z&sp=rl',
          styleName: 'gentle',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ebacec70-4b66-4a92-bca0-d3341f975dc5.wav?sv=2019-07-07&sr=b&sig=5L9Hn2pFgD2pFw8D%2FMOxxctEPz9yeZ0buazD3FSyXQo%3D&st=2021-10-29T08:29:19Z&se=2121-10-05T08:34:19Z&sp=rl',
          styleName: 'serious',
        },
      ],
    },
    shortName: 'zh-CN-XiaoxuanNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '15b8867d-a217-4abb-9dde-158670c0615c',
    locale: 'en-NG',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-NG, EzinneNeural)',
    properties: {
      DisplayName: 'Ezinne',
      DisplayVoiceName: 'EzinneNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Ezinne',
      LocaleDescription: 'en-NG',
      OrderInVoiceListApi: '117',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-NG-EzinneNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-NG',
      localeZH: '英语(尼日利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5029885d-574a-462e-a839-63d55b0a3160.wav?sv=2019-07-07&sr=b&sig=e9IqlT1818lvddfMOWEvfyabE%2BSd3ZqaBi8%2BsGrb9ZA%3D&st=2021-10-29T08:30:26Z&se=2121-10-05T08:35:26Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-NG-EzinneNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [
      {
        name: 'All',
      },
      {
        name: 'Chat',
      },
    ],
    id: '16cf511c-1865-404e-b2da-160362b7dff6',
    locale: 'zh-CN',
    masterpieces: [
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/202ba518-e7fc-4254-a31d-5d8c23e3bbee.wav?sv=2019-07-07&sr=b&sig=PeEKEniX%2FXSEyLSw79EwmTYLLGAl6s60GwvyryWKDy4%3D&st=2021-10-29T08:31:50Z&se=2121-10-05T08:36:50Z&sp=rl',
        categories: ['Chat'],
        title: '晓辰客户对话展示',
        titleLocalizationId: 'zh-CN-XiaochenNeural_Masterpieces_Chat_CustomerConversation_Title',
      },
    ],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, XiaochenNeural)',
    properties: {
      AgeGroups: 'YoungAdult',
      DisplayName: 'Xiaochen',
      DisplayVoiceName: 'XiaochenNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      IconFileEndpointWithSas:
        'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5a7529db-8f27-45bc-838e-8c0a6e16a06a.png?sv=2019-07-07&sr=b&sig=sWkJpp9DbQSI6J0y2WQ50%2B1Q%2B2lYe8Ls7zCV8c7XBDE%3D&st=2021-10-29T08:24:32Z&se=2121-10-05T08:29:32Z&sp=rl',
      IsTtsBreakEnabled: 'True',
      LocalName: '晓辰',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '425',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-XiaochenNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/57214773-6bfe-45cd-9c0f-07a5a09e7b76.wav?sv=2019-07-07&sr=b&sig=0isjvhamcDGVJSM6WoyzY%2B7vy%2FejMw7y4E9zlF%2F%2B6NI%3D&st=2021-10-29T08:29:43Z&se=2121-10-05T08:34:43Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'zh-CN-XiaochenNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'ff915f3d-7c13-4914-a7ed-1615d17d4039',
    locale: 'he-IL',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (he-IL, AvriNeural)',
    properties: {
      DisplayName: 'Avri',
      DisplayVoiceName: 'AvriNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'אברי',
      LocaleDescription: 'Hebrew (Israel)',
      OrderInVoiceListApi: '261',
      PreviewSentence:
        'בנה יישומים ושירותים שמדברים בטבעיות למשתמשים, שמשפרים את  הנגישות והשימושיות.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'he-IL-AvriNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'he-IL',
      localeZH: '希伯来语(以色列)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e33123ca-e9ea-4d2b-ade5-1a1513c05ee9.wav?sv=2019-07-07&sr=b&sig=B04jlPJbXefzFE3Mz7IGHnfCycZPy0G9bg2xCrOww0Y%3D&st=2021-10-29T08:27:07Z&se=2121-10-05T08:32:07Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'he-IL-AvriNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '5189e67d-9cc7-4fd8-b894-165ab7c1d504',
    locale: 'uk-UA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (uk-UA, PolinaNeural)',
    properties: {
      DisplayName: 'Polina',
      DisplayVoiceName: 'PolinaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Поліна',
      LocaleDescription: 'Ukrainian (Ukraine)',
      OrderInVoiceListApi: '410',
      PreviewSentence:
        'Створення аудіовмісту дозволяє візуально контролювати мовні атрибути в реальному часі.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'uk-UA-PolinaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'uk-UA',
      localeZH: '乌克兰语(乌克兰)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/61f5e44e-c714-4e7c-b705-406728f0f680.wav?sv=2019-07-07&sr=b&sig=ARzDmD3Hrrc8NSzmapEoToA0RPxZ9GF6iJkogabBigM%3D&st=2021-10-29T08:28:26Z&se=2121-10-05T08:33:26Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'uk-UA-PolinaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [
      {
        name: 'All',
      },
      {
        defaultStyle: 'newscast',
        iconFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/52b4ac03-bf02-49d6-b9fd-482f4d2a1c77?sv=2019-07-07&sr=b&sig=6d3sdtxzsnEv%2BQ8XgUaGCcoy8l8n1dalU1iLMHRgZPA%3D&st=2021-10-29T08:24:26Z&se=2121-10-05T08:29:26Z&sp=rl',
        name: 'Newscast',
      },
      {
        defaultStyle: 'lyrical',
        iconFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4b216047-db86-4eba-8d40-eebcb4f3dfe2?sv=2019-07-07&sr=b&sig=kbWHWHE%2BRU9cPDrLnq7aP4Jdz3U6igfWnE9kjQD%2BkvA%3D&st=2021-10-29T08:24:25Z&se=2121-10-05T08:29:25Z&sp=rl',
        name: 'Audiobook',
      },
      {
        defaultStyle: 'customerservice',
        name: 'CustomerService',
      },
      {
        defaultStyle: 'chat',
        name: 'Chat',
      },
      {
        defaultStyle: 'assistant',
        iconFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4147367b-5ec9-474b-8fb5-f542e8767cb5?sv=2019-07-07&sr=b&sig=b3naEouPSWl6zT2kiWDon87YTbwT5AZNXx8AIz%2BrZCY%3D&st=2021-10-29T08:24:24Z&se=2121-10-05T08:29:24Z&sp=rl',
        name: 'Assistant',
      },
    ],
    description: 'Lively and warm voice with multiple scenario styles and emotions.',
    id: '5f55541d-c844-4e04-a7f8-1723ffbea4a9',
    locale: 'zh-CN',
    masterpieces: [
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ad560aa9-d061-409c-ac46-99210fdd97a6?sv=2019-07-07&sr=b&sig=m3%2Fv2WLVcO5tscAKesA6vJ0B1si9KJ4yhvOTy%2FU8MyE%3D&st=2021-10-29T08:24:39Z&se=2121-10-05T08:29:39Z&sp=rl',
        categories: ['Audiobook'],
        description: '晓晓多情感语音演绎黛玉细腻的情绪变化。',
        descriptionLocalizationId:
          'zh-CN-XiaoxiaoNeural_Masterpieces_Audiobook_DaiYuHuXiaoXinYi_Description',
        ssmlFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/676de7d8-2440-4281-9eec-ba8910b6cd7e?sv=2019-07-07&sr=b&sig=TBTgfM7Xw5jP1C12iaadWp8K1jW1LpRDsX0btcaqWoc%3D&st=2021-10-29T08:24:38Z&se=2121-10-05T08:29:38Z&sp=rl',
        title: '晓晓有声书《宝黛互晓心意》',
        titleLocalizationId: 'zh-CN-XiaoxiaoNeural_Masterpieces_Audiobook_DaiYuHuXiaoXinYi_Title',
      },
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c4a7ed19-de0b-49f5-b3b0-7b732324e2d3?sv=2019-07-07&sr=b&sig=Fz4UgeOZgC75M%2F3bOViddsFExfFFap%2F6DBAXx6stD6o%3D&st=2021-10-29T08:24:47Z&se=2121-10-05T08:29:47Z&sp=rl',
        categories: ['Newscast'],
        title: '晓晓新闻',
        titleLocalizationId: 'zh-CN-XiaoxiaoNeural_Masterpieces_Newscast_XiaoxiaoNews_Title',
      },
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/291ed98c-3ede-404f-8730-7b6b7870537a?sv=2019-07-07&sr=b&sig=Z8jQ8C2JUqzvV2D9WgpLjenbZAjDE%2BJbyKVhQgNHT1A%3D&st=2021-10-29T08:24:40Z&se=2121-10-05T08:29:40Z&sp=rl',
        categories: ['CustomerService'],
        description: '晓晓帮助加快客服服务流程。',
        title: '晓晓智能客服',
        titleLocalizationId:
          'zh-CN-XiaoxiaoNeural_Masterpieces_CustomerService_XiaoxiaoCustomerService_Title',
      },
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/dfdffd66-315f-41f9-aacd-e8a97f279302?sv=2019-07-07&sr=b&sig=iGJYW2KdqwN%2BxXwIhok%2FgeTK%2FKvi8dGyDZiJlaVa3Zk%3D&st=2021-10-29T08:24:41Z&se=2121-10-05T08:29:41Z&sp=rl',
        categories: [],
        description:
          '晓晓多情感语音升级后支持多达14种不同风格，能在丰富的情绪和场景中自由转换，比如多情感有声读物、新闻、客服、助理、聊天等。可以满足不同领域的多样化需求。',
        iconFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5465b8bb-902e-49e7-8355-27c936b612b1?sv=2019-07-07&sr=b&sig=cpA%2BlsiDFMRCMNCOEfqFY7cmO%2F5TN5ndcIh3d7JL5zs%3D&st=2021-10-29T08:24:42Z&se=2121-10-05T08:29:42Z&sp=rl',
        title: '晓晓多情感展示',
        titleLocalizationId: 'zh-CN-XiaoxiaoNeural_Masterpieces_None_MutipleEmotions_Title',
        videoFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b5a28f58-bd8d-4c52-a36c-a4e52482a98d?sv=2019-07-07&sr=b&sig=dOKk%2BfC4RbfT2Hm5eEq2tjyd4pARHR6hJeHXZL7A5eM%3D&st=2021-10-29T08:24:43Z&se=2121-10-05T08:29:43Z&sp=rl',
      },
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ff402971-7793-42a0-b13b-afeef88ef31c?sv=2019-07-07&sr=b&sig=arsayZJVIdZaYDBtnALU4wNYbzbJCqE1MzvBN63VeYc%3D&st=2021-10-29T08:24:45Z&se=2121-10-05T08:29:45Z&sp=rl',
        categories: ['Audiobook'],
        title: '晓晓有声书《小城三月》',
        titleLocalizationId: 'zh-CN-XiaoxiaoNeural_Masterpieces_AudioBook_XiaoChengSanYue_Title',
      },
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/aa55e174-db74-4d39-9548-7a2122a4720a?sv=2019-07-07&sr=b&sig=NmAMTUEixbGR2toeFgIAPHRTOlOxYaN5WYaGBK2iRi0%3D&st=2021-10-29T08:24:44Z&se=2121-10-05T08:29:44Z&sp=rl',
        categories: ['Audiobook'],
        title: '晓晓有声书《秋》',
        titleLocalizationId: 'zh-CN-XiaoxiaoNeural_Masterpieces_AudioBook_Qiu_Title',
      },
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0908c94f-11aa-4684-be13-2490cc58146b.wav?sv=2019-07-07&sr=b&sig=epq3c%2BEZ5%2FhqnxEnqQ44fzMr346WV08id5swxiBKqjA%3D&st=2021-10-29T08:24:45Z&se=2121-10-05T08:29:45Z&sp=rl',
        categories: ['Audiobook'],
        title: '晓晓有声书《野草》',
        titleLocalizationId: 'zh-CN-XiaoxiaoNeural_Masterpieces_AudioBook_YeCao_Title',
      },
    ],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, XiaoxiaoNeural)',
    properties: {
      AgeGroups: 'YoungAdult',
      DisplayName: 'Xiaoxiao',
      DisplayVoiceName: 'XiaoxiaoNeural',
      ExpressAsRoleValues: 'Default,CustomerService,Newscast,Lyrical,VoiceAssistant',
      ExpressAsStyleValues: 'Default,CustomerService,Newscast,Lyrical,VoiceAssistant',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      IconFileEndpointWithSas:
        'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/56a84a35-3feb-457e-a1c4-db39407e5a5f?sv=2019-07-07&sr=b&sig=VUfdT6ng%2BFDpQpleWcfDx8rTqMKYNN3h%2FRaz8V96Yo4%3D&st=2021-10-29T08:24:23Z&se=2121-10-05T08:29:23Z&sp=rl',
      IsTtsBreakEnabled: 'True',
      LocalName: '晓晓',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '423',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-XiaoxiaoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions:
        '{"affectionate":9,"angry":7,"calm":2,"cheerful":4,"disgruntled":5,"fearful":3,"gentle":11,"sad":8,"serious":6,"Default":0}',
      VoiceStyleNames:
        'assistant,chat,customerservice,newscast,affectionate,angry,calm,cheerful,disgruntled,fearful,gentle,lyrical,sad,serious,poetry-reading,friendly,Default',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c6181015-94ae-4df0-8c2d-ff44c01b8bba.wav?sv=2019-07-07&sr=b&sig=gXZ5rhZzXr6DNYpuTwMMXpRvuIZj%2Bm8YXQFGDh7pv2c%3D&st=2021-10-29T08:28:56Z&se=2121-10-05T08:33:56Z&sp=rl',
          styleName: 'affectionate',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6c39231e-1837-46cb-9685-44a380914bb6.wav?sv=2019-07-07&sr=b&sig=wZIN4BCTZCXXP1kYSo8WJg7jxNy1jRTNhSe7ohkIeVs%3D&st=2021-10-29T08:28:57Z&se=2121-10-05T08:33:57Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/1e0ae7ec-e791-499a-a38c-6e69c7204139.wav?sv=2019-07-07&sr=b&sig=%2FTar4XqjueANyna64xtIhMZpwg4iNUNICQMATJ%2FJxlg%3D&st=2021-10-29T08:28:58Z&se=2121-10-05T08:33:58Z&sp=rl',
          styleName: 'assistant',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d0c3856b-f831-44f7-b5c0-d16e63edfaa3.wav?sv=2019-07-07&sr=b&sig=aT1n5AoSK1hszjyAOEss%2B5BVy9I%2BjcyGf%2FbH3XIQAlw%3D&st=2021-10-29T08:28:59Z&se=2121-10-05T08:33:59Z&sp=rl',
          styleName: 'calm',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2f223455-7c94-4eac-a2b0-6f45f7e30ec0.wav?sv=2019-07-07&sr=b&sig=vRHMyl6IrPRorPHpdXxd%2BdtO96DAsSchbUn5uyHoy0c%3D&st=2021-10-29T08:29:00Z&se=2121-10-05T08:34:00Z&sp=rl',
          styleName: 'chat',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/7f293767-f7e8-4602-8b5e-728ef69e61f5.wav?sv=2019-07-07&sr=b&sig=Ev0jD63M0NLea4QNKnFVI9Vze09J5OCjS1QEjLZAbwE%3D&st=2021-10-29T08:29:01Z&se=2121-10-05T08:34:01Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/3c6199b4-7bf2-46ec-a5a8-32d38f5a73fd.wav?sv=2019-07-07&sr=b&sig=WKvs7eQvNc8mmptgZm0nAndShLP2sjcbAq4EDKrVr8c%3D&st=2021-10-29T08:29:02Z&se=2121-10-05T08:34:02Z&sp=rl',
          styleName: 'customerservice',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6c5b9606-7fac-457d-99ef-e08b843e8bbb.wav?sv=2019-07-07&sr=b&sig=X%2B%2B18WyUEa%2Bz0m38DxBT0VbsqGi9chzL9UVej9UFhZA%3D&st=2021-10-29T08:29:03Z&se=2121-10-05T08:34:03Z&sp=rl',
          styleName: 'disgruntled',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b1002a0d-b6ba-4d65-8f88-29750329b078.wav?sv=2019-07-07&sr=b&sig=FHuu7kY2%2FIgEVpPL3NWhpYkXTR%2Fqgf25iUcBesTTh%2FA%3D&st=2021-10-29T08:29:04Z&se=2121-10-05T08:34:04Z&sp=rl',
          styleName: 'fearful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2656000c-3fea-487b-aead-8e44bdb2eebb.wav?sv=2019-07-07&sr=b&sig=68DNTVvPVYBufWdkX6KRNKd668BCXrk%2FI4IP%2FDxS8BE%3D&st=2021-10-29T08:29:05Z&se=2121-10-05T08:34:05Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5d7c036e-6887-4f6a-8b76-ef9d4e3b1521.wav?sv=2019-07-07&sr=b&sig=iNmPkepyjZQYzHQWfa4XXBQ84duCLo8AHMP%2BG7JnD4g%3D&st=2021-10-29T08:29:06Z&se=2121-10-05T08:34:06Z&sp=rl',
          styleName: 'gentle',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/31d8ef07-96c3-4180-a62e-86fdb1fa2041.wav?sv=2019-07-07&sr=b&sig=nxUcp3ZzIezKCDBDS66JbMiGzyM0tXkvO8uUJeW4Mtc%3D&st=2021-10-29T08:29:07Z&se=2121-10-05T08:34:07Z&sp=rl',
          styleName: 'lyrical',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/001752cd-ffdf-40a1-be45-29cc6bb337ce.wav?sv=2019-07-07&sr=b&sig=lxLC3LL%2F67mOrZuajDjTaDonXaFU9KtcUUAOOBStlq4%3D&st=2021-10-29T08:29:08Z&se=2121-10-05T08:34:08Z&sp=rl',
          styleName: 'newscast',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/dc5ef7bc-ac6f-4108-bcfc-e18de09d3dad.wav?sv=2019-07-07&sr=b&sig=h2LqPe4%2BuliwEnhw2%2B2bnkPek%2FTKT3t66U4OMhcwxLo%3D&st=2021-10-29T08:29:09Z&se=2121-10-05T08:34:09Z&sp=rl',
          styleName: 'sad',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ec2e7db4-c93b-47a1-ba43-a07e414f68c3.wav?sv=2019-07-07&sr=b&sig=8LHPeCukknY213R2AN6kXTNTr%2BsGlspyxfznlI2NyB0%3D&st=2021-10-29T08:29:10Z&se=2121-10-05T08:34:10Z&sp=rl',
          styleName: 'serious',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/27547f85-06b6-44d2-87c9-f45383d29cae.wav?sv=2019-07-07&sr=b&sig=p2LYfWhP%2BP1ewOaSadSLs38yIN3VJ87ab8KUgsu3vrE%3D&st=2022-06-02T09:01:46Z&se=2122-05-09T09:06:46Z&sp=rl',
          styleName: 'poetry-reading',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/97a6c4bf-d269-462c-b5ca-f7644ca5df5b.wav?sv=2021-10-04&st=2023-03-03T05%3A58%3A57Z&se=2123-02-07T06%3A03%3A57Z&sr=b&sp=rl&sig=39RB1DN5h6SL1kwWavrqO1iQWF%2BQNNg%2FQnPc3zXends%3D',
          styleName: 'friendly',
        },
      ],
    },
    shortName: 'zh-CN-XiaoxiaoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '85e7f814-532a-4345-97e8-18a61717dd8e',
    locale: 'it-IT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (it-IT, DiegoNeural)',
    properties: {
      DisplayName: 'Diego',
      DisplayVoiceName: 'DiegoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Diego',
      LocaleDescription: 'Italian',
      OrderInVoiceListApi: '289',
      PreviewSentence:
        'Benvenuti nella piattaforma di generazione di contenuti audio di Microsoft.',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'it-IT-DiegoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'it-IT',
      localeZH: '意大利语(意大利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2aa84483-b966-4840-af86-a8a8b8871302.wav?sv=2019-07-07&sr=b&sig=dZKIPZQmVjlMCXIVUYQFH8bHN1TbOKly1GkmoOkjQJs%3D&st=2021-10-29T08:27:17Z&se=2121-10-05T08:32:17Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'it-IT-DiegoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'b7533283-392f-4969-b977-1a364b1adbae',
    locale: 'de-DE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (de-DE, KatjaNeural)',
    properties: {
      DisplayName: 'Katja',
      DisplayVoiceName: 'KatjaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Katja',
      LocaleDescription: 'German (Germany)',
      OrderInVoiceListApi: '60',
      PreviewSentence:
        'Erstellen Sie Apps und Dienste, die auf natürliche Art mit Nutzern sprechen, und dadurch Zugänglichkeit und Benutzerfreundlichkeit verbessern.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'de-DE-KatjaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'de-DE',
      localeZH: '德语(德国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2c864f0e-9dc5-4b72-b2f3-45e14f88387a.wav?sv=2019-07-07&sr=b&sig=S8y3Qk%2FVuUNITFEV3DOI5PdX4LF3gnJdkK2HF5aC1Ug%3D&st=2021-10-29T08:26:15Z&se=2121-10-05T08:31:15Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'de-DE-KatjaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '9314d042-c519-4087-ba99-1a49d78e2d9f',
    locale: 'es-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-ES, ElviraNeural)',
    properties: {
      DisplayName: 'Elvira',
      DisplayVoiceName: 'ElviraNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Elvira',
      LocaleDescription: 'Spanish (Spain)',
      OrderInVoiceListApi: '173',
      PreviewSentence:
        'Crea aplicaciones y servicios que hablan de forma natural a los usuarios, y que mejoran la accesibilidad y la facilidad de uso.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-ES-ElviraNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-ES',
      localeZH: '西班牙语(西班牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/abfdb173-2ef8-4433-8e01-769f1467b289.wav?sv=2019-07-07&sr=b&sig=k4MALkTBVqaQBMSpPQNTzKrRrSnrvwIvPkNEI6CbMbo%3D&st=2021-10-29T08:26:44Z&se=2121-10-05T08:31:44Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-ES-ElviraNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '92fde113-0078-4166-b94c-1a5277a9a8c0',
    locale: 'si-LK',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (si-LK, SameeraNeural)',
    properties: {
      DisplayName: 'Sameera',
      DisplayVoiceName: 'SameeraNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'සමීර',
      LocaleDescription: 'si-LK',
      OrderInVoiceListApi: '373',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'si-LK-SameeraNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'si-LK',
      localeZH: '僧伽罗语(斯里兰卡)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/14abea49-eb75-4ec0-9162-a189a76f1454.wav?sv=2019-07-07&sr=b&sig=FhWvJOQ8tyEJZ2fzAIv0ltSK0WcWjvzRQRoOQEWoxYM%3D&st=2022-01-19T09:02:16Z&se=2121-12-26T09:07:16Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'si-LK-SameeraNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [
      {
        name: 'All',
      },
      {
        name: 'Audiobook',
      },
      {
        name: 'Chat',
      },
    ],
    description:
      'Warm and sweet voice with rich emotions that can be used in many conversation scenarios.',
    id: '2a741315-7d02-4b45-b91f-1b620ac545bc',
    locale: 'zh-CN',
    masterpieces: [
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0ad14630-4206-43aa-ab40-a8aa01f98686.wav?sv=2019-07-07&sr=b&sig=ygJjV9XQxedAduoalfrukBLSGJGFvA8zMHAb4FW5fgw%3D&st=2021-10-29T08:31:46Z&se=2121-10-05T08:36:46Z&sp=rl',
        categories: ['Audiobook'],
        title: '金粉世家',
        titleLocalizationId: 'zh-CN-XiaohanNeural_Masterpieces_AudioBook_JinFenShiJian_Title',
      },
    ],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, XiaohanNeural)',
    properties: {
      AgeGroups: 'YoungAdult',
      DisplayName: 'Xiaohan',
      DisplayVoiceName: 'XiaohanNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      IconFileEndpointWithSas:
        'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e7b408f7-7406-4ce6-bc72-34894aa2e11b?sv=2019-07-07&sr=b&sig=1mPcj71Arx2TrCG6QoD%2FJx5s9v%2F%2BWyNQDts8eGuWrqU%3D&st=2021-10-29T08:24:20Z&se=2121-10-05T08:29:20Z&sp=rl',
      IsTtsBreakEnabled: 'True',
      LocalName: '晓涵',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '426',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-XiaohanNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions:
        '{"calm":2,"fearful":3,"cheerful":4,"disgruntled":5,"serious":6,"angry":7,"sad":8,"gentle":11,"affectionate":9,"embarrassed":1,"Default":0}',
      VoiceStyleNames:
        'calm,fearful,cheerful,disgruntled,serious,angry,sad,gentle,affectionate,embarrassed,Default',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c9ca5b9b-dec7-4639-803a-726431782557.wav?sv=2019-07-07&sr=b&sig=3Cp9lWmO6jcRZ1BY1j0ds3AO%2BKErs3eHNX%2F807WaIqY%3D&st=2021-10-29T08:28:31Z&se=2121-10-05T08:33:31Z&sp=rl',
          styleName: 'affectionate',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/61a53392-877e-4181-b637-4f8b4a603e13.wav?sv=2019-07-07&sr=b&sig=7YL8%2FEffLKdaq8ALbqoql2BqMAA1j%2FgWr1UTMY5O2w0%3D&st=2021-10-29T08:28:32Z&se=2121-10-05T08:33:32Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/eabe827e-b67f-417f-b5d5-2940284e6715.wav?sv=2019-07-07&sr=b&sig=C7ABvvBzPYXC3zMWHpfO3w0UdcSg593W2gyKf9DziTc%3D&st=2021-10-29T08:28:33Z&se=2121-10-05T08:33:33Z&sp=rl',
          styleName: 'calm',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e0168677-5b8b-406c-8cd0-68f6a1081430.wav?sv=2019-07-07&sr=b&sig=XPH5yoJWHr%2FxrllkyN9QuKW7NY7ZrgwoJ174rbHEbms%3D&st=2021-10-29T08:28:34Z&se=2121-10-05T08:33:34Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b4ca5849-cac2-4ec3-8297-3cb2ae332442.wav?sv=2019-07-07&sr=b&sig=vFqQdK3XkpX2lHWMQw%2F8ZRIFhwE4LmxhdgqsUNYrYrM%3D&st=2021-10-29T08:28:35Z&se=2121-10-05T08:33:35Z&sp=rl',
          styleName: 'disgruntled',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/186d233b-90da-4de9-b4e6-59506cebca8e.wav?sv=2019-07-07&sr=b&sig=iXjTDFaCaWCj7%2BGYhbDdCnxzS7HTL8hkqHg8i8zyU5E%3D&st=2021-10-29T08:28:36Z&se=2121-10-05T08:33:36Z&sp=rl',
          styleName: 'embarrassed',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/1351be95-2ea5-4b47-af5c-f80a6eeeb9f7.wav?sv=2019-07-07&sr=b&sig=iRNBY8KTrpmfM3EXa9QkkiZdjt4MIJ6hhW%2B0hXp7WBQ%3D&st=2021-10-29T08:28:37Z&se=2121-10-05T08:33:37Z&sp=rl',
          styleName: 'fearful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ffea4480-fe1e-4d00-88dc-f9002bfe571a.wav?sv=2019-07-07&sr=b&sig=JKCH93ueXx6Fao24TcNWFonOTKzorzXoHLJePycJq4A%3D&st=2021-10-29T08:28:38Z&se=2121-10-05T08:33:38Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ef761363-b49f-4562-87b0-309e29525a6c.wav?sv=2019-07-07&sr=b&sig=phbHjjoFMJJWf0FHUqLo4ubQ0fbV869FXMVsjNl7C1A%3D&st=2021-10-29T08:28:39Z&se=2121-10-05T08:33:39Z&sp=rl',
          styleName: 'gentle',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/3b7d1627-d6a3-42d0-a260-d384ec7276ef.wav?sv=2019-07-07&sr=b&sig=knt9vN9a6D7MVg4vKBScCsMJ4OE%2FwAc6XbgVa4Ep1bA%3D&st=2021-10-29T08:28:40Z&se=2121-10-05T08:33:40Z&sp=rl',
          styleName: 'sad',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/16cb1adc-6e1f-4880-8f47-2a1d73ebd0cc.wav?sv=2019-07-07&sr=b&sig=lOlU9iVqaSAE7ZS1yJA4S17TPUjYCghcsk9k2gZ8%2BkM%3D&st=2021-10-29T08:28:41Z&se=2121-10-05T08:33:41Z&sp=rl',
          styleName: 'serious',
        },
      ],
    },
    shortName: 'zh-CN-XiaohanNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '2b869701-13e4-4612-9183-1d67ed0e6ccc',
    locale: 'it-IT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (it-IT, PierinaNeural)',
    properties: {
      DisplayName: 'Pierina',
      DisplayVoiceName: 'PierinaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Pierina',
      LocaleDescription: 'Italian',
      OrderInVoiceListApi: '287',
      PreviewSentence:
        'Benvenuti nella piattaforma di generazione di contenuti audio di Microsoft.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'it-IT-PierinaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'it-IT',
      localeZH: '意大利语(意大利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b2c3ba83-6562-4c57-9dcb-b0ce77922994.wav?sv=2019-07-07&sr=b&sig=3D77CYeLNd5Y5G4Z6jRAUnKsIs%2FfD9pccg2w5qYbJ2E%3D&st=2022-07-01T09:09:53Z&se=2122-06-07T09:14:53Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'it-IT-PierinaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'cc82356f-b180-424d-ae74-1dbbdfdac93b',
    locale: 'es-PY',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-PY, TaniaNeural)',
    properties: {
      DisplayName: 'Tania',
      DisplayVoiceName: 'TaniaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Tania',
      LocaleDescription: 'es-PY',
      OrderInVoiceListApi: '214',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-PY-TaniaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-PY',
      localeZH: '西班牙语(巴拉圭)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/3129bdec-9e5c-4069-86e1-317ac64a140c.wav?sv=2019-07-07&sr=b&sig=uDXbQJqt3xNYD6p4lNTRttuP7ERdqzGsm%2BEFokqjp8E%3D&st=2021-10-29T08:30:55Z&se=2121-10-05T08:35:55Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-PY-TaniaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '1f13dc7d-e108-4432-94b6-1e25445cb533',
    locale: 'de-DE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (de-DE, MajaNeural)',
    properties: {
      DisplayName: 'Maja',
      DisplayVoiceName: 'MajaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Maja',
      LocaleDescription: 'German (Germany)',
      OrderInVoiceListApi: '71',
      PreviewSentence:
        'Erstellen Sie Apps und Dienste, die auf natürliche Art mit Nutzern sprechen, und dadurch Zugänglichkeit und Benutzerfreundlichkeit verbessern.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'de-DE-MajaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'de-DE',
      localeZH: '德语(德国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b8eea38b-240c-4cb5-a5a4-0c5e9126f8a6.wav?sv=2019-07-07&sr=b&sig=USXipwdcnXPw6rLGKT%2Byv8IQyH72ypBtm8NoJzTxH4Y%3D&st=2022-01-19T09:00:44Z&se=2121-12-26T09:05:44Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'de-DE-MajaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '8e2bd817-6415-4cde-9e56-1e45e2fb6973',
    locale: 'sw-TZ',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (sw-TZ, RehemaNeural)',
    properties: {
      DisplayName: 'Rehema',
      DisplayVoiceName: 'RehemaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Rehema',
      LocaleDescription: 'sw-TZ',
      OrderInVoiceListApi: '393',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'sw-TZ-RehemaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'sw-TZ',
      localeZH: '斯瓦希里语(坦桑尼亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4137ce9d-bfc1-447f-b079-c3a78c53ec56.wav?sv=2019-07-07&sr=b&sig=dPrqWVtNiNfpctCc8wx5RLBxAKRN65pwTv%2FTnUdzBRc%3D&st=2021-10-29T08:31:18Z&se=2121-10-05T08:36:18Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'sw-TZ-RehemaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'd91fc86e-c4ac-4528-aad1-1f17a869b835',
    locale: 'mn-MN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (mn-MN, YesuiNeural)',
    properties: {
      DisplayName: 'Yesui',
      DisplayVoiceName: 'YesuiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Есүй',
      LocaleDescription: 'mn-MN',
      OrderInVoiceListApi: '326',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'mn-MN-YesuiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'mn-MN',
      localeZH: '蒙古语(蒙古)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b380000c-72e2-4d84-9c91-919a4f807643.wav?sv=2019-07-07&sr=b&sig=TURGtPuY9rVlHfbnI9t%2BoARQxMEmVW5fi3JULFbowMk%3D&st=2022-07-01T09:10:04Z&se=2122-06-07T09:15:04Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'mn-MN-YesuiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'c87ed4d5-30e1-43c7-9d21-1f61797fc706',
    locale: 'de-DE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (de-DE, AmalaNeural)',
    properties: {
      DisplayName: 'Amala',
      DisplayVoiceName: 'AmalaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Amala',
      LocaleDescription: 'German (Germany)',
      OrderInVoiceListApi: '61',
      PreviewSentence:
        'Erstellen Sie Apps und Dienste, die auf natürliche Art mit Nutzern sprechen, und dadurch Zugänglichkeit und Benutzerfreundlichkeit verbessern.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'de-DE-AmalaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'de-DE',
      localeZH: '德语(德国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ebb55a3d-e393-4d56-ada0-86fdc8ce096b.wav?sv=2019-07-07&sr=b&sig=z0P4aXcCwNfRwT6LqDjQ2BhXB7kYVIOjZ41BUP4GVpY%3D&st=2022-01-19T09:00:20Z&se=2121-12-26T09:05:20Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'de-DE-AmalaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '36b7073c-1426-49ea-8393-206c4cd2ce99',
    locale: 'en-AU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-AU, TinaNeural)',
    properties: {
      DisplayName: 'Tina',
      DisplayVoiceName: 'TinaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Tina',
      LocaleDescription: 'English (Australia)',
      OrderInVoiceListApi: '89',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-AU-TinaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-AU',
      localeZH: '英语(澳大利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e2e65a36-b67f-4abf-890f-789c70ac878e.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A21Z&se=2122-09-27T07%3A51%3A21Z&sr=b&sp=rl&sig=YkeZYg8iq6j0d06AEWxscYukmI8Ynp492%2Fr4bHB%2FUbU%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-AU-TinaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '2e840c4d-fc29-40aa-bdb6-209cc07f26e6',
    locale: 'pt-BR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pt-BR, NicolauNeural)',
    properties: {
      DisplayName: 'Nicolau',
      DisplayVoiceName: 'NicolauNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Nicolau',
      LocaleDescription: 'Portuguese (Brazil)',
      OrderInVoiceListApi: '362',
      PreviewSentence:
        'Desenvolver aplicativos e serviços que conversem naturalmente com os usuários, melhorando a acessibilidade e usabilidade.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pt-BR-NicolauNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'pt-BR',
      localeZH: '葡萄牙语(巴西)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/74fc758b-2468-4872-a391-23e5e9bec1b5.wav?sv=2019-07-07&sr=b&sig=afYRgqQMn4yBckDnqkL1nDuyz5AZN05biJaGkIIxDPA%3D&st=2022-07-01T09:10:34Z&se=2122-06-07T09:15:34Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'pt-BR-NicolauNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'e59988b1-2234-4607-9f69-2177f5892e0d',
    locale: 'en-AU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-AU, KimNeural)',
    properties: {
      DisplayName: 'Kim',
      DisplayVoiceName: 'KimNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Kim',
      LocaleDescription: 'English (Australia)',
      OrderInVoiceListApi: '85',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-AU-KimNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-AU',
      localeZH: '英语(澳大利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fabbd286-39f0-4d9d-8448-fad65e69528f.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A15Z&se=2122-09-27T07%3A51%3A15Z&sr=b&sp=rl&sig=K2DfnBhX7mAJ0Mj34UCJ1K8hUez5u6fNKKtvejPOoOE%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-AU-KimNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'c86e7dad-b97c-4de9-8ff1-2287bf405535',
    locale: 'fr-CH',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-CH, ArianeNeural)',
    properties: {
      DisplayName: 'Ariane',
      DisplayVoiceName: 'ArianeNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Ariane',
      LocaleDescription: 'French (Switzerland)',
      OrderInVoiceListApi: '239',
      PreviewSentence:
        'Développer des applications et des services qui parlent aux utilisateurs avec naturel, pour améliorer leur accessibilité et leur utilisation.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-CH-ArianeNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fr-CH',
      localeZH: '法语(瑞士)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/bfda8c6c-0ff4-4d98-8deb-6ffee6bdc8e1.wav?sv=2019-07-07&sr=b&sig=cqz8D90NBsDh5qszJsgVKexx9ozooJ9SrQGHEEBYqZA%3D&st=2021-10-29T08:26:59Z&se=2121-10-05T08:31:59Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fr-CH-ArianeNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'ed169f6c-1533-4ed5-8a17-23749095b556',
    locale: 'ru-RU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ru-RU, SvetlanaNeural)',
    properties: {
      DisplayName: 'Svetlana',
      DisplayVoiceName: 'SvetlanaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Светлана',
      LocaleDescription: 'Russian',
      OrderInVoiceListApi: '370',
      PreviewSentence:
        'Возможность создавать приложения и сервисы, которые естественным образом общаются с пользователями, улучшая доступность и удобство использования.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ru-RU-SvetlanaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ru-RU',
      localeZH: '俄语(俄罗斯)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/391d214e-5b29-4ed5-92bd-bbfd1503773f.wav?sv=2019-07-07&sr=b&sig=VDWEEg4gFCzDVeduKQVA1LIKeb2MJQagHBwM88HwSlU%3D&st=2021-10-29T08:28:04Z&se=2121-10-05T08:33:04Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ru-RU-SvetlanaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'a6482508-cc9f-4994-adf3-2397fe3cc5cd',
    locale: 'su-ID',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (su-ID, JajangNeural)',
    properties: {
      DisplayName: 'Jajang',
      DisplayVoiceName: 'JajangNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Jajang',
      LocaleDescription: 'su-ID',
      OrderInVoiceListApi: '385',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'su-ID-JajangNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'su-ID',
      localeZH: '巽他语(印度尼西亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/87ad1e62-b3f7-4ad9-9e0d-b2f2d03f2f3a.wav?sv=2019-07-07&sr=b&sig=5%2FAnz3lIHNmT7hA23WNKYAydSuKsrzAGPHdSUjte4nM%3D&st=2021-10-29T08:31:15Z&se=2121-10-05T08:36:15Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'su-ID-JajangNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '40772218-9232-47e5-aa26-23a96d5effa0',
    locale: 'zh-CN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, XiaozhenNeural)',
    properties: {
      DisplayName: 'Xiaozhen',
      DisplayVoiceName: 'XiaozhenNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      IsTtsBreakEnabled: 'True',
      LocalName: '晓甄',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '436',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-XiaozhenNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions:
        '{"angry":7,"disgruntled":5,"cheerful":4,"fearful":3,"sad":8,"serious":6,"Default":0}',
      VoiceStyleNames: 'angry,disgruntled,cheerful,fearful,sad,serious,Default',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/43aaa3d7-673a-41ef-af17-433efff290bf.wav?sv=2019-07-07&sr=b&sig=FAQA80moG%2BtQYS7kQLGO4eHmxMbTIXAHhlb7g%2B%2B1yo8%3D&st=2022-08-01T10:39:55Z&se=2122-07-08T10:44:55Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/1953b493-24f0-4053-abbf-60c918ef25a8.wav?sv=2019-07-07&sr=b&sig=1SOQTYJb0h28Y%2FTMicddPRC0VbrRXKhpVu%2BohWXF%2BEw%3D&st=2022-08-01T10:39:58Z&se=2122-07-08T10:44:58Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ac66aa67-2871-49be-abd9-1dbc923bbb4f.wav?sv=2019-07-07&sr=b&sig=ZApY%2Bi%2BmvphJOmTvRQ0%2FOLebuTlmIMXv1ffhuoMoSe4%3D&st=2022-08-01T10:40:01Z&se=2122-07-08T10:45:01Z&sp=rl',
          styleName: 'disgruntled',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/679b0ccb-ef37-4c78-b224-700efe71baa9.wav?sv=2019-07-07&sr=b&sig=FbinntCxaCZaTE7GWSrlz2Lm3mh7pp54zgZsxIdOtPI%3D&st=2022-08-01T10:40:04Z&se=2122-07-08T10:45:04Z&sp=rl',
          styleName: 'fearful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/345a2f55-081f-4bcf-b6e7-8054dcd2c95a.wav?sv=2019-07-07&sr=b&sig=I%2BMR%2FOkFDRAyvmilXqlH6xqfVX8JCywqKlynDNi7I7w%3D&st=2022-08-01T10:40:07Z&se=2122-07-08T10:45:07Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d79870da-f5e1-4f4a-83a1-dd5537c81042.wav?sv=2019-07-07&sr=b&sig=HDXdaJ4DuiRw%2BWycQfRo1k8Ivr5ccZPrlNpJfb6xw5Q%3D&st=2022-08-01T10:40:09Z&se=2122-07-08T10:45:09Z&sp=rl',
          styleName: 'sad',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ce56b498-3dac-474c-befe-599967f2c75c.wav?sv=2019-07-07&sr=b&sig=GCsyziaGcbB4NR%2BzOWNK7gZX1dDG%2Bkxte995lLFNn20%3D&st=2022-08-01T10:40:12Z&se=2122-07-08T10:45:12Z&sp=rl',
          styleName: 'serious',
        },
      ],
    },
    shortName: 'zh-CN-XiaozhenNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '4a99a2e0-2cf0-46c9-bcde-23d37674ecaf',
    locale: 'lt-LT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (lt-LT, LeonasNeural)',
    properties: {
      DisplayName: 'Leonas',
      DisplayVoiceName: 'LeonasNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Leonas',
      LocaleDescription: 'Lithuanian (Lithuania)',
      OrderInVoiceListApi: '317',
      PreviewSentence:
        'Garso turinio kūrimas leidžia vizualiai kontroliuoti kalbos atributus realiu laiku.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'lt-LT-LeonasNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'lt-LT',
      localeZH: '立陶宛语(立陶宛)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/484b4313-73f9-478e-bd13-d394fccc5f94.wav?sv=2019-07-07&sr=b&sig=MPY04100ilz2NJcta9oo5mKfKGqiKXMYM6oIpf56NFA%3D&st=2021-10-29T08:27:25Z&se=2121-10-05T08:32:25Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'lt-LT-LeonasNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '49efca2d-f9bb-4761-81d2-243bf84d520b',
    locale: 'ur-PK',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ur-PK, UzmaNeural)',
    properties: {
      DisplayName: 'Uzma',
      DisplayVoiceName: 'UzmaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'عظمیٰ',
      LocaleDescription: 'Urdu (Pakistan)',
      OrderInVoiceListApi: '414',
      PreviewSentence:
        'آواز کا مواد تخلیق کرنا  اس قابل بناتا  ہیکہ آپ تقریر کی خصوصیات کو  حقیقی وقت میں  اپنے مطابق کنٹرول کر سکتے ہیں۔',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ur-PK-UzmaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ur-PK',
      localeZH: '乌尔都语(巴基斯坦)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4a98847c-1fff-40b6-8281-7d4ad735c073.wav?sv=2019-07-07&sr=b&sig=SXt2rCpaVhBYUkADr0HBFY%2B2C4xbk2BMDfv6xlAe9mM%3D&st=2021-10-29T08:28:28Z&se=2121-10-05T08:33:28Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ur-PK-UzmaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'c1749ad5-0aa6-428a-9336-254ee247a672',
    locale: 'es-PY',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-PY, MarioNeural)',
    properties: {
      DisplayName: 'Mario',
      DisplayVoiceName: 'MarioNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Mario',
      LocaleDescription: 'es-PY',
      OrderInVoiceListApi: '213',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-PY-MarioNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-PY',
      localeZH: '西班牙语(巴拉圭)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/71bfa0a8-97f6-46ee-81b5-6630aa4c2f02.wav?sv=2019-07-07&sr=b&sig=T8ZFrM3QE5nNG1%2BSyv4XlSoG5kqk47k9Hl6KcZ8pvjQ%3D&st=2021-10-29T08:30:54Z&se=2121-10-05T08:35:54Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-PY-MarioNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '57801f3b-0718-44bc-81ff-259334396f17',
    locale: 'ka-GE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ka-GE, EkaNeural)',
    properties: {
      DisplayName: 'Eka',
      DisplayVoiceName: 'EkaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'ეკა',
      LocaleDescription: 'ka-GE',
      OrderInVoiceListApi: '299',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ka-GE-EkaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ka-GE',
      localeZH: '格鲁吉亚语(格鲁吉亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b3e64e26-b944-4e97-9b90-776e6d5e8b85.wav?sv=2019-07-07&sr=b&sig=QHTDLkr%2Fu5iE8Jc33MawodloFwOoRiidJtLjgnodyqI%3D&st=2022-07-01T09:09:58Z&se=2122-06-07T09:14:58Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ka-GE-EkaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'b7429917-f62d-44e1-bdd3-2623cb475bd2',
    locale: 'pt-BR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pt-BR, YaraNeural)',
    properties: {
      DisplayName: 'Yara',
      DisplayVoiceName: 'YaraNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Yara',
      LocaleDescription: 'Portuguese (Brazil)',
      OrderInVoiceListApi: '364',
      PreviewSentence:
        'Desenvolver aplicativos e serviços que conversem naturalmente com os usuários, melhorando a acessibilidade e usabilidade.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pt-BR-YaraNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'pt-BR',
      localeZH: '葡萄牙语(巴西)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b91e1bfd-8451-45d3-a1e7-865e1681df5e.wav?sv=2019-07-07&sr=b&sig=UzX9cUIWaIGewcZ9G6jY4Mf4Vn1V0dzknHfUl%2FDZpR0%3D&st=2022-07-01T09:10:39Z&se=2122-06-07T09:15:39Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'pt-BR-YaraNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '8763bcde-b893-4e6d-852d-26d9cd967f0e',
    locale: 'en-NZ',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-NZ, MollyNeural)',
    properties: {
      DisplayName: 'Molly',
      DisplayVoiceName: 'MollyNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Molly',
      LocaleDescription: 'en-NZ',
      OrderInVoiceListApi: '119',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-NZ-MollyNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-NZ',
      localeZH: '英语(新西兰)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/124fcab4-3b16-43d6-9b84-e7c883aba381.wav?sv=2019-07-07&sr=b&sig=Jol9kD0zko4iHSTyhCFKSCIexZ7ehZZX2U6QV36yA9Y%3D&st=2021-10-29T08:26:33Z&se=2121-10-05T08:31:33Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-NZ-MollyNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'f86b8aff-fccb-4d98-a6dc-26eb83bf8ef6',
    locale: 'es-PA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-PA, MargaritaNeural)',
    properties: {
      DisplayName: 'Margarita',
      DisplayVoiceName: 'MargaritaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Margarita',
      LocaleDescription: 'es-PA',
      OrderInVoiceListApi: '207',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-PA-MargaritaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-PA',
      localeZH: '西班牙语(巴拿马)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/894e2466-21b9-4f46-b0e1-76406161d70e.wav?sv=2019-07-07&sr=b&sig=EHfz45bjj%2BHdapvn6IR1IYg4SmhPyBm0b8ML7djX4tM%3D&st=2021-10-29T08:30:48Z&se=2121-10-05T08:35:48Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-PA-MargaritaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '954eda91-3108-4486-8143-26edb95e4e7f',
    locale: 'ar-MA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-MA, JamalNeural)',
    properties: {
      DisplayName: 'Jamal',
      DisplayVoiceName: 'JamalNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'جمال',
      LocaleDescription: 'ar-MA',
      OrderInVoiceListApi: '23',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-MA-JamalNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-MA',
      localeZH: '阿拉伯语(摩洛哥)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a9da0578-2091-464a-ac84-86f5c9ee7ba9.wav?sv=2019-07-07&sr=b&sig=nEwROuRlX%2Bz5MKFPWI2Y2bAsK6%2FI6B4omfb%2FQTvz6d8%3D&st=2021-10-29T08:30:11Z&se=2121-10-05T08:35:11Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-MA-JamalNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '50b742e4-b3e5-4f11-bedf-28a1b81188c7',
    locale: 'es-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-ES, VeraNeural)',
    properties: {
      DisplayName: 'Vera',
      DisplayVoiceName: 'VeraNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Vera',
      LocaleDescription: 'Spanish (Spain)',
      OrderInVoiceListApi: '182',
      PreviewSentence:
        'Crea aplicaciones y servicios que hablan de forma natural a los usuarios, y que mejoran la accesibilidad y la facilidad de uso.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-ES-VeraNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-ES',
      localeZH: '西班牙语(西班牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fe74029e-c21c-46e6-bb5e-e2a9896f70bd.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A59Z&se=2122-09-27T07%3A51%3A59Z&sr=b&sp=rl&sig=VS6jXsnsUb9uPdEMrqV1%2Fcxki7tBFKK2SjlCziFpRZ8%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-ES-VeraNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '19eaa6ae-0655-4704-928b-28e1f1b917b2',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, JennyMultilingualNeural)',
    properties: {
      DisplayName: 'Jenny Multilingual',
      DisplayVoiceName: 'JennyMultilingualNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Jenny Multilingual',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '127',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales:
        'de-DE,en-AU,en-CA,en-GB,es-ES,es-MX,fr-CA,fr-FR,it-IT,ja-JP,ko-KR,pt-BR,zh-CN',
      ShortName: 'en-US-JennyMultilingualNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b480a57c-be19-4cbb-975a-11249d411d27.wav?sv=2019-07-07&sr=b&sig=Mk%2FM5wTM%2B2HFy5gSSQDqufWs7JBcYML4TzX%2BLJ8ua30%3D&st=2021-10-29T08:24:58Z&se=2121-10-05T08:29:58Z&sp=rl',
          locale: 'de-DE',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c1e33257-0472-4fb1-8745-f78473d4f7be.wav?sv=2019-07-07&sr=b&sig=XmVEwoZI4szB%2Fg8gxtiF%2FeWCixybsGf8jw%2BhCQEZYms%3D&st=2021-10-29T08:24:59Z&se=2121-10-05T08:29:59Z&sp=rl',
          locale: 'en-AU',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/888bcfdd-35a3-4a08-965c-551d3f2a77fb.wav?sv=2019-07-07&sr=b&sig=4LczckirqW040KTyaLyjukRFVBZC3w%2BdfAqE%2FlBKjNY%3D&st=2021-10-29T08:25:00Z&se=2121-10-05T08:30:00Z&sp=rl',
          locale: 'en-CA',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a68c5fc0-74b4-4b5e-8057-6cf2d39106a1.wav?sv=2019-07-07&sr=b&sig=QNHM932Ptc8cYyIdnRivuRt31XGewZjPLBXUI3LWvQ4%3D&st=2021-10-29T08:25:01Z&se=2121-10-05T08:30:01Z&sp=rl',
          locale: 'en-GB',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c81e6516-66a0-48e1-bcfb-73342cbba4e8.wav?sv=2019-07-07&sr=b&sig=0oFdVeDv8tticsB4cUXudskHenyYhmhCHqZt8DOoKGE%3D&st=2021-10-29T08:25:03Z&se=2121-10-05T08:30:03Z&sp=rl',
          locale: 'es-ES',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/8ba9d11e-d0b7-41ac-bd22-0270368c4122.wav?sv=2019-07-07&sr=b&sig=anf5V5fdp74Ncj8u2Am2rwS9VOclmS0vKwrapRGnnM4%3D&st=2021-10-29T08:25:04Z&se=2121-10-05T08:30:04Z&sp=rl',
          locale: 'es-MX',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/9487f1f9-c398-487d-949b-05a0575a700b.wav?sv=2019-07-07&sr=b&sig=tkgb7OqHvWF6Ma%2FEdi9OG%2FPOswYvwgyQ4vnNWB9l1uc%3D&st=2021-10-29T08:25:05Z&se=2121-10-05T08:30:05Z&sp=rl',
          locale: 'fr-CA',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d8c8c739-e5c2-4c34-8c8e-7f72ad5bb647.wav?sv=2019-07-07&sr=b&sig=rM%2FRvpnOw1NeCCzah7%2F8%2F%2FcsulM%2FR0UgkH3P0VTFS2U%3D&st=2021-10-29T08:25:06Z&se=2121-10-05T08:30:06Z&sp=rl',
          locale: 'fr-FR',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f10cbf0e-8528-47b3-b21e-9dd8150ac5f8.wav?sv=2019-07-07&sr=b&sig=HfDUVkAaktzWilx2PtmGeCyq0PG5FXRRCLGdGSSXoCA%3D&st=2021-10-29T08:25:07Z&se=2121-10-05T08:30:07Z&sp=rl',
          locale: 'it-IT',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2bf592dc-c38b-4acf-9f5f-295d4d53863e.wav?sv=2019-07-07&sr=b&sig=8JDGbQb33uygZXuUChOsJKXACKga2YUgXHFTxOkzT4Q%3D&st=2021-10-29T08:25:08Z&se=2121-10-05T08:30:08Z&sp=rl',
          locale: 'ja-JP',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/44c773f0-cd64-44f2-94ab-adb95a100bb0.wav?sv=2019-07-07&sr=b&sig=m%2BGf8VYWQBXB7LyjkXhZvkMSIZF7lzG5Ta8X65hHDuo%3D&st=2021-10-29T08:25:09Z&se=2121-10-05T08:30:09Z&sp=rl',
          locale: 'ko-KR',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fe509380-d53c-45fe-bfa5-8da596701253.wav?sv=2019-07-07&sr=b&sig=vldSrqTLBmw9qZzH54QreK9nmjzSGB8AsqyriN5d6Fg%3D&st=2021-10-29T08:25:10Z&se=2121-10-05T08:30:10Z&sp=rl',
          locale: 'pt-BR',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/09b22762-413c-499c-a676-53ff805b3e5f.wav?sv=2019-07-07&sr=b&sig=qvN0ivvtZA%2B2ul%2BLc75YU3W06N5rwo2XGkhFehBd9P4%3D&st=2021-10-29T08:25:12Z&se=2121-10-05T08:30:12Z&sp=rl',
          locale: 'zh-CN',
        },
      ],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a06750a5-97e7-496c-8f1d-83ad74a1bff3.wav?sv=2019-07-07&sr=b&sig=6EDl4Q0V6NpAiaWvFkglbXXVvD%2BKw7oxTN%2BYe0gqLV8%3D&st=2021-10-29T08:25:47Z&se=2121-10-05T08:30:47Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-US-JennyMultilingualNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '79c1cc71-a4e8-4981-bd8d-2921d82d4230',
    locale: 'ar-OM',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-OM, AbdullahNeural)',
    properties: {
      DisplayName: 'Abdullah',
      DisplayVoiceName: 'AbdullahNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'عبدالله',
      LocaleDescription: 'ar-OM',
      OrderInVoiceListApi: '25',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-OM-AbdullahNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-OM',
      localeZH: '阿拉伯语(阿曼)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/81878370-c06e-4507-ab32-349d85f41f0c.wav?sv=2019-07-07&sr=b&sig=29XekzumkKhVguD2wFke%2BfoWMZYiEuoD6lSG%2B3zDXUk%3D&st=2022-07-01T09:08:47Z&se=2122-06-07T09:13:47Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-OM-AbdullahNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '5f44b500-0472-4596-a79d-29be8b3fb6d3',
    locale: 'cy-GB',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (cy-GB, NiaNeural)',
    properties: {
      DisplayName: 'Nia',
      DisplayVoiceName: 'NiaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Nia',
      LocaleDescription: 'Welsh (UK)',
      OrderInVoiceListApi: '53',
      PreviewSentence:
        'Mae’r feddalwedd creu cynnwys sain yn galluogi rheoli priodoleddau lleferydd yn weledol mewn amser real.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'cy-GB-NiaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'cy-GB',
      localeZH: '威尔士语(英国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/46062812-2276-4daa-ba79-dff498534e6f.wav?sv=2019-07-07&sr=b&sig=4fN6RVo5mAcC3sYLWTKyevRjDSNXO1UutbC25j9c8V4%3D&st=2021-10-29T08:26:07Z&se=2121-10-05T08:31:07Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'cy-GB-NiaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'd278fb66-0bd9-4706-a194-29f10970f2e4',
    locale: 'nb-NO',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (nb-NO, PernilleNeural)',
    properties: {
      DisplayName: 'Pernille',
      DisplayVoiceName: 'PernilleNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Pernille',
      LocaleDescription: 'Norwegian',
      OrderInVoiceListApi: '335',
      PreviewSentence:
        'Bygger apper og tjenester som snakker naturlig med brukerne, utbedrer tilgjengelighet og brukskvalitet.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'nb-NO-PernilleNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'nb-NO',
      localeZH: '书面挪威语(挪威)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4b5366af-5ce9-42f6-bfb0-ba3f741344af.wav?sv=2019-07-07&sr=b&sig=y9bUIUELAR6YzILAM1rlETcOXaH2tUSvIFrC0o%2FpjR4%3D&st=2021-10-29T08:27:39Z&se=2121-10-05T08:32:39Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'nb-NO-PernilleNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '6f91cbf3-5833-4dbd-aaab-2a7d54ddf3f1',
    locale: 'it-IT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (it-IT, IrmaNeural)',
    properties: {
      DisplayName: 'Irma',
      DisplayVoiceName: 'IrmaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Irma',
      LocaleDescription: 'Italian',
      OrderInVoiceListApi: '284',
      PreviewSentence:
        'Benvenuti nella piattaforma di generazione di contenuti audio di Microsoft.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'it-IT-IrmaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'it-IT',
      localeZH: '意大利语(意大利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6bd2e2f7-4957-450d-b767-49c48cb27806.wav?sv=2019-07-07&sr=b&sig=cISTKAOaknRbae9uC%2BymjGMBFdYuCKjud3lKYoONTvM%3D&st=2022-07-01T09:09:47Z&se=2122-06-07T09:14:47Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'it-IT-IrmaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'ef9c11c7-07ce-476f-871d-2b7f31a7c48b',
    locale: 'pl-PL',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pl-PL, AgnieszkaNeural)',
    properties: {
      DisplayName: 'Agnieszka',
      DisplayVoiceName: 'AgnieszkaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Agnieszka',
      LocaleDescription: 'Polish',
      OrderInVoiceListApi: '345',
      PreviewSentence:
        'Twórz aplikacje i serwisy, które w kontakcie z użytkownikiem posługują się naturalną mową, co podnosi ich dostępność i użyteczność.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pl-PL-AgnieszkaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'pl-PL',
      localeZH: '波兰语(波兰)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f4eec98a-baaa-410c-a475-32148822ed39.wav?sv=2019-07-07&sr=b&sig=D8oV4EDPcmUpP0p8KI1OQJCqVZ2eGW38M2hJLI0ThTw%3D&st=2021-10-29T08:27:48Z&se=2121-10-05T08:32:48Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'pl-PL-AgnieszkaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '5db44c75-2d71-44b9-b82d-2bce134c4c2e',
    locale: 'es-SV',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-SV, LorenaNeural)',
    properties: {
      DisplayName: 'Lorena',
      DisplayVoiceName: 'LorenaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Lorena',
      LocaleDescription: 'es-SV',
      OrderInVoiceListApi: '215',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-SV-LorenaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-SV',
      localeZH: '西班牙语(萨尔瓦多)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2c0fff15-407a-4dcd-b1b7-e3216cc7df88.wav?sv=2019-07-07&sr=b&sig=nn%2FCOG2UGNBMiSWjHrl4SX4n7vINWerpkf3%2BkuaakGQ%3D&st=2021-10-29T08:30:56Z&se=2121-10-05T08:35:56Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-SV-LorenaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'daa147ce-c7af-4a97-9e37-2dd923df4433',
    locale: 'de-DE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (de-DE, KillianNeural)',
    properties: {
      DisplayName: 'Killian',
      DisplayVoiceName: 'KillianNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Killian',
      LocaleDescription: 'German (Germany)',
      OrderInVoiceListApi: '67',
      PreviewSentence:
        'Erstellen Sie Apps und Dienste, die auf natürliche Art mit Nutzern sprechen, und dadurch Zugänglichkeit und Benutzerfreundlichkeit verbessern.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'de-DE-KillianNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'de-DE',
      localeZH: '德语(德国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/182d805b-f3d3-4ca3-b728-5b73f698f4f8.wav?sv=2019-07-07&sr=b&sig=8ctkmB8ZK68YeEckHk%2BRIp3vAn7zcm%2FpFkxb8%2FjJJUA%3D&st=2022-01-19T09:00:34Z&se=2121-12-26T09:05:34Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'de-DE-KillianNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '7676f89d-cacf-492b-889d-2e7b897bcde4',
    locale: 'en-AU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-AU, TimNeural)',
    properties: {
      DisplayName: 'Tim',
      DisplayVoiceName: 'TimNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Tim',
      LocaleDescription: 'English (Australia)',
      OrderInVoiceListApi: '88',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-AU-TimNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-AU',
      localeZH: '英语(澳大利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/007b429c-35a3-460b-9485-ef5dd7595734.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A19Z&se=2122-09-27T07%3A51%3A19Z&sr=b&sp=rl&sig=eW5HsqeWMvnmTThQxNkM0Ztz1EruhQ%2FNfSBcBogBP7U%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-AU-TimNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '5e4d2532-06f2-4773-b563-2f946289aadf',
    locale: 'bg-BG',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (bg-BG, KalinaNeural)',
    properties: {
      DisplayName: 'Kalina',
      DisplayVoiceName: 'KalinaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Калина',
      LocaleDescription: 'Bulgarian',
      OrderInVoiceListApi: '40',
      PreviewSentence:
        'Създай приложения и услуги, които говорят непринудено на потребителите, подобрявайки достъпността и използваемостта.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'bg-BG-KalinaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'bg-BG',
      localeZH: '保加利亚语(保加利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c80e2297-19a7-4a1b-afaa-cfd54c9210cb.wav?sv=2019-07-07&sr=b&sig=DAcY60cSlpDi3F81BC7AUWNjpCDDDig4j5oQ5SaXkP4%3D&st=2021-10-29T08:26:00Z&se=2121-10-05T08:31:00Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'bg-BG-KalinaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '39947851-46d7-4561-8199-2fd8bdc49ba6',
    locale: 'zh-CN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, YunjianNeural)',
    properties: {
      DisplayName: 'Yunjian',
      DisplayVoiceName: 'YunjianNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      IsTtsBreakEnabled: 'True',
      LocalName: '云健',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '439',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-YunjianNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"Default":0}',
      VoiceStyleNames: 'narration-relaxed,sports-commentary,sports-commentary-excited,Default',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/90edb4cb-2741-440a-8d75-c782ae67638a.wav?sv=2019-07-07&sr=b&sig=v%2ByGJiInKAGlKe0%2Bpy%2FKQT7WyBkLzCjtv%2B6UR4wGTrY%3D&st=2022-04-13T06:19:22Z&se=2122-03-20T06:24:22Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ec4a4360-f5a3-45d1-9853-ede25402b87d.wav?sv=2019-07-07&sr=b&sig=60nyIZzHCnBBuC%2Bi4QYkUU19IABwQhBU4pN1mEaJA34%3D&st=2022-04-12T10:03:03Z&se=2122-03-19T10:08:03Z&sp=rl',
          styleName: 'narration-relaxed',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f3470c00-752e-41f4-a88f-501d62a4b234.wav?sv=2019-07-07&sr=b&sig=JmFizCOsBSsZaWcZZOrOkTzFkDH78GDl5rIYKRHUTZE%3D&st=2022-04-12T10:03:07Z&se=2122-03-19T10:08:07Z&sp=rl',
          styleName: 'sports-commentary',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/82a3257e-46ff-4a1a-b381-9bb1e3d6e532.wav?sv=2019-07-07&sr=b&sig=uTzBJBpsR1KNXr1wagbZDIW918jZzt73i4NXhjt8kWI%3D&st=2022-04-12T10:03:05Z&se=2122-03-19T10:08:05Z&sp=rl',
          styleName: 'sports-commentary-excited',
        },
      ],
    },
    shortName: 'zh-CN-YunjianNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '8466e484-7aee-4c7a-85c3-30d59b533ad4',
    locale: 'fr-BE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-BE, GerardNeural)',
    properties: {
      DisplayName: 'Gerard',
      DisplayVoiceName: 'GerardNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Gerard',
      LocaleDescription: 'French (Belgium)',
      OrderInVoiceListApi: '235',
      PreviewSentence:
        'La Création de Contenu Audio vous permet de contrôler visuellement les attributs vocaux en temps réel.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-BE-GerardNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fr-BE',
      localeZH: '法语(比利时)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b76d8c67-ad9b-431f-b33e-a99f3224210c.wav?sv=2019-07-07&sr=b&sig=q7rsrms9xpyXqmJjt9BwEMqQxriVnTduQyWOSQ1C7Vw%3D&st=2021-10-29T08:26:55Z&se=2121-10-05T08:31:55Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fr-BE-GerardNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'f4b9fd8a-fad9-44f0-b5f0-3219888d07fd',
    locale: 'es-MX',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-MX, YagoNeural)',
    properties: {
      DisplayName: 'Yago',
      DisplayVoiceName: 'YagoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Yago',
      LocaleDescription: 'Spanish (Mexico)',
      OrderInVoiceListApi: '204',
      PreviewSentence:
        'Crea apps y servicios que hablen de forma natural con los usuarios, mejorando la accesibilidad y la usabilidad.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-MX-YagoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-MX',
      localeZH: '西班牙语(墨西哥)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4e668667-ce5e-4a06-a1dd-61218015eae1.wav?sv=2019-07-07&sr=b&sig=4kNd9Xq6cXLXMnvGOT5pBBt6%2B2%2BkEhkL0iBMyu%2FGs%2Fg%3D&st=2022-07-01T09:09:28Z&se=2122-06-07T09:14:28Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-MX-YagoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '62bcf61d-7e9b-4f2d-b109-326d291f3197',
    locale: 'en-KE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-KE, AsiliaNeural)',
    properties: {
      DisplayName: 'Asilia',
      DisplayVoiceName: 'AsiliaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Asilia',
      LocaleDescription: 'en-KE',
      OrderInVoiceListApi: '114',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-KE-AsiliaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-KE',
      localeZH: '英语(肯尼亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/15ed4dae-ce50-4489-b578-7cd8183dfe25.wav?sv=2019-07-07&sr=b&sig=ulprV%2FHA3V7PJjSw37Xt8j%2F6PrJBXd0AxC1UesrwjWU%3D&st=2021-10-29T08:30:23Z&se=2121-10-05T08:35:23Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-KE-AsiliaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '632392e6-d5af-4876-9417-32d19424e5b6',
    locale: 'nl-NL',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (nl-NL, ColetteNeural)',
    properties: {
      DisplayName: 'Colette',
      DisplayVoiceName: 'ColetteNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Colette',
      LocaleDescription: 'Dutch',
      OrderInVoiceListApi: '342',
      PreviewSentence:
        'Ontwikkel apps en diensten die natuurlijk aanvoelen, waardoor de toegankelijkheid en bruikbaarheid vergroot worden.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'nl-NL-ColetteNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'nl-NL',
      localeZH: '荷兰语(荷兰)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/9edef43c-212b-45b4-a9db-09bc222e7669.wav?sv=2019-07-07&sr=b&sig=zsQQMM1dP4h%2FPtNHv8t0bm%2BsgYm6DvB4EbWm8wQPmbE%3D&st=2021-10-29T08:27:43Z&se=2121-10-05T08:32:43Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'nl-NL-ColetteNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'c101b3fa-3990-4943-b9fa-330f269e6d35',
    locale: 'pt-BR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pt-BR, GiovannaNeural)',
    properties: {
      DisplayName: 'Giovanna',
      DisplayVoiceName: 'GiovannaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Giovanna',
      LocaleDescription: 'Portuguese (Brazil)',
      OrderInVoiceListApi: '356',
      PreviewSentence:
        'Desenvolver aplicativos e serviços que conversem naturalmente com os usuários, melhorando a acessibilidade e usabilidade.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pt-BR-GiovannaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'pt-BR',
      localeZH: '葡萄牙语(巴西)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/757826d2-9ac2-428c-83d4-758f4d0aeae5.wav?sv=2019-07-07&sr=b&sig=AlT7vBk3eZLOd0UcVa5Dle8HCXH%2BtFZ5Gnyernd1WJM%3D&st=2022-07-01T09:10:20Z&se=2122-06-07T09:15:20Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'pt-BR-GiovannaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [
      {
        name: 'All',
      },
      {
        name: 'CustomerService',
      },
    ],
    id: '8190b804-6f50-4ad6-9b9d-332a4d3bf521',
    locale: 'zh-CN',
    masterpieces: [
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/89ecb4a5-5b99-4754-86ad-7071e57169f9.wav?sv=2019-07-07&sr=b&sig=2wNrPq7XysOY2Y2K798XS0igHvgQd%2FfieRjpr0wwWFc%3D&st=2021-10-29T08:31:54Z&se=2121-10-05T08:36:54Z&sp=rl',
        categories: ['CustomerService'],
        title: '晓颜客服对话展示',
        titleLocalizationId: 'zh-CN-XiaoyanNeural_Masterpieces_CustomerService_Conversation_Title',
      },
    ],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, XiaoyanNeural)',
    properties: {
      AgeGroups: 'YoungAdult',
      DisplayName: 'Xiaoyan',
      DisplayVoiceName: 'XiaoyanNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      IconFileEndpointWithSas:
        'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/7ce55c87-6677-4bba-a45c-ec8bc0a7a876.png?sv=2019-07-07&sr=b&sig=RErsyLJTYyf7eiRgZxbbVXlmmQCtNi79ba7cDTS69l4%3D&st=2021-10-29T08:24:36Z&se=2121-10-05T08:29:36Z&sp=rl',
      IsTtsBreakEnabled: 'True',
      LocalName: '晓颜',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '433',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-XiaoyanNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2ec6251b-a67e-442d-af86-1b61a1ea5a18.wav?sv=2019-07-07&sr=b&sig=rvBt9MB%2BxRKcxOhJErRc4Ukrw5s2VpDmuqK22lDl7NI%3D&st=2021-10-29T08:29:47Z&se=2121-10-05T08:34:47Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'zh-CN-XiaoyanNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '77988ee6-38e7-4b0a-ba91-33d88b3a42b0',
    locale: 'gl-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (gl-ES, RoiNeural)',
    properties: {
      DisplayName: 'Roi',
      DisplayVoiceName: 'RoiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Roi',
      LocaleDescription: 'gl-ES',
      OrderInVoiceListApi: '257',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'gl-ES-RoiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'gl-ES',
      localeZH: '加利西亚语(加利西亚语)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e0effa4a-ead4-4bad-a25e-3cca92683a05.wav?sv=2019-07-07&sr=b&sig=JA339cwp5%2FJAxlRkqT1wo67SoNlZllZu8L8Vdez5JUU%3D&st=2021-10-29T08:31:05Z&se=2121-10-05T08:36:05Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'gl-ES-RoiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '1152d3cc-8fd1-4448-8935-3475650a83a1',
    locale: 'ar-SY',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-SY, LaithNeural)',
    properties: {
      DisplayName: 'Laith',
      DisplayVoiceName: 'LaithNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'ليث',
      LocaleDescription: 'ar-SY',
      OrderInVoiceListApi: '32',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-SY-LaithNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-SY',
      localeZH: '阿拉伯语(叙利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/de6fe50c-875f-4a4f-bba4-6e907fc607c1.wav?sv=2019-07-07&sr=b&sig=hRmMbwd7c9bZg3Ga40CaSqXZp5qMfpFZ1lkivqZ2PzI%3D&st=2021-10-29T08:30:16Z&se=2121-10-05T08:35:16Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-SY-LaithNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '814db6e1-ce6e-4b8a-9c10-347943910d18',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, AmberNeural)',
    properties: {
      DisplayName: 'Amber',
      DisplayVoiceName: 'AmberNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Amber',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '129',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-AmberNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/16c936b1-cfbc-4a82-9335-e57fae5b4690.wav?sv=2019-07-07&sr=b&sig=Z6tVYv4bFfcgsR1Nh8buIid5mDxtQtyd4E%2FF8peCR3c%3D&st=2021-10-29T08:25:13Z&se=2121-10-05T08:30:13Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-US-AmberNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '89523202-fee2-410b-8a71-355e59bdaa21',
    locale: 'pt-BR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pt-BR, AntonioNeural)',
    properties: {
      DisplayName: 'Antonio',
      DisplayVoiceName: 'AntonioNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Antônio',
      LocaleDescription: 'Portuguese (Brazil)',
      OrderInVoiceListApi: '351',
      PreviewSentence:
        'Desenvolver aplicativos e serviços que conversem naturalmente com os usuários, melhorando a acessibilidade e usabilidade.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pt-BR-AntonioNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'pt-BR',
      localeZH: '葡萄牙语(巴西)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c09f225d-72d6-4d55-adec-86a9ff411214.wav?sv=2019-07-07&sr=b&sig=bQW0qGRe46oIov3SjGC%2FcjgrQBwuDysFjdMNX%2BSzLrg%3D&st=2021-10-29T08:27:52Z&se=2121-10-05T08:32:52Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'pt-BR-AntonioNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '6c4d0fa4-d47f-4e37-9d25-357d84cb02c3',
    locale: 'am-ET',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (am-ET, AmehaNeural)',
    properties: {
      DisplayName: 'Ameha',
      DisplayVoiceName: 'AmehaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'አምሀ',
      LocaleDescription: 'am-ET',
      OrderInVoiceListApi: '3',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'am-ET-AmehaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'am-ET',
      localeZH: '阿姆哈拉语(埃塞俄比亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0ee26f5f-b295-4062-87f5-7389bdfc5105.wav?sv=2019-07-07&sr=b&sig=gNIz%2FQKypQIaWWgSrSDor%2FK0I2j%2BlTue5cpmJwoywwY%3D&st=2021-10-29T08:29:56Z&se=2121-10-05T08:34:56Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'am-ET-AmehaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '7b34b246-014f-4fea-9b47-3597b18778a9',
    locale: 'ta-SG',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ta-SG, AnbuNeural)',
    properties: {
      DisplayName: 'Anbu',
      DisplayVoiceName: 'AnbuNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'அன்பு',
      LocaleDescription: 'ta-SG',
      OrderInVoiceListApi: '400',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ta-SG-AnbuNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ta-SG',
      localeZH: '泰米尔语(新加坡)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d63dbcba-3e93-4365-b150-145ad2acab79.wav?sv=2019-07-07&sr=b&sig=Nnj9Hqc1NFweXrlSPOuchOBVxHzsF5TrMGDZh2WWYFk%3D&st=2021-10-29T08:31:21Z&se=2121-10-05T08:36:21Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ta-SG-AnbuNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '86ad0be8-ef3a-4e4b-a026-35e5c51153eb',
    locale: 'fr-FR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-FR, CelesteNeural)',
    properties: {
      DisplayName: 'Celeste',
      DisplayVoiceName: 'CelesteNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Celeste',
      LocaleDescription: 'French (France)',
      OrderInVoiceListApi: '243',
      PreviewSentence:
        "Construire des applications et services qui communiquent naturellement avec l'utilisateur, en améliorant l’accessibilité et l'ergonomie",
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-FR-CelesteNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fr-FR',
      localeZH: '法语(法国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b9aae947-3622-4bc9-aa24-b52a304d8195.wav?sv=2019-07-07&sr=b&sig=5%2BVGgdktTSKb3MgHtIHq4oRs8o32IyAN45uemKJbN9g%3D&st=2022-01-19T09:01:22Z&se=2121-12-26T09:06:22Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fr-FR-CelesteNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '662c6e95-d86c-490b-928d-3836034ffc42',
    locale: 'ko-KR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ko-KR, InJoonNeural)',
    properties: {
      DisplayName: 'InJoon',
      DisplayVoiceName: 'InJoonNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: '인준',
      LocaleDescription: 'Korean',
      OrderInVoiceListApi: '308',
      PreviewSentence:
        '사용자들에게 자연스럽게 이야기하는 앱과 서비스를 만들며 접근성과 사용성을 개선시킵니다.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ko-KR-InJoonNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ko-KR',
      localeZH: '韩语(韩国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/98eb2b15-796a-44c3-8619-ff9e71c090b6.wav?sv=2019-07-07&sr=b&sig=e8yXJ7c8F5LJdj4nIRXtYDiKGlDpJPfMdmDwy7nLwvI%3D&st=2021-10-29T08:27:22Z&se=2121-10-05T08:32:22Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ko-KR-InJoonNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '80b67b7d-afee-4874-b394-394d8de13850',
    locale: 'es-CR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-CR, JuanNeural)',
    properties: {
      DisplayName: 'Juan',
      DisplayVoiceName: 'JuanNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Juan',
      LocaleDescription: 'es-CR',
      OrderInVoiceListApi: '161',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-CR-JuanNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-CR',
      localeZH: '西班牙语(哥斯达黎加)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/19190158-0acd-4bb8-8bc0-b445278dde4e.wav?sv=2019-07-07&sr=b&sig=JxiZRiTrEn6D49ym2eNl0MB6h5XLgdyjjf3jFLjQi1g%3D&st=2021-10-29T08:30:32Z&se=2121-10-05T08:35:32Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-CR-JuanNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'd05e2be8-ce1f-4b89-bf54-3984721887a4',
    locale: 'ka-GE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ka-GE, GiorgiNeural)',
    properties: {
      DisplayName: 'Giorgi',
      DisplayVoiceName: 'GiorgiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'გიორგი',
      LocaleDescription: 'ka-GE',
      OrderInVoiceListApi: '300',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ka-GE-GiorgiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ka-GE',
      localeZH: '格鲁吉亚语(格鲁吉亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/492a0799-a46e-468d-8d62-7c921adba119.wav?sv=2019-07-07&sr=b&sig=jGUSIWLwmu%2FRX8ELvAsINGXKjSGoLfBQbKKdP2fE1%2Fk%3D&st=2022-07-01T09:10:00Z&se=2122-06-07T09:15:00Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ka-GE-GiorgiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'ace72e9b-dd48-4374-b508-39af2d140d16',
    locale: 'es-MX',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-MX, JorgeNeural)',
    properties: {
      DisplayName: 'Jorge',
      DisplayVoiceName: 'JorgeNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Jorge',
      LocaleDescription: 'Spanish (Mexico)',
      OrderInVoiceListApi: '196',
      PreviewSentence:
        'Crea apps y servicios que hablen de forma natural con los usuarios, mejorando la accesibilidad y la usabilidad.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-MX-JorgeNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"cheerful":4,"Default":0}',
      VoiceStyleNames: 'cheerful,chat,Default',
      locale: 'es-MX',
      localeZH: '西班牙语(墨西哥)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fb9f06dd-aeb3-4423-a77d-d821dcf12934.wav?sv=2019-07-07&sr=b&sig=ZEMIZaKQLbAT8lOEjdCeDkyvytvohpMg2Jk7Cd%2F26HI%3D&st=2021-10-29T08:26:46Z&se=2121-10-05T08:31:46Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/76c6a7f9-76be-482c-bb88-2a4aa50f522b.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A01Z&se=2122-09-27T07%3A52%3A01Z&sr=b&sp=rl&sig=x9IxkciMnAKAXmb1kefR%2FEDm8axRxQjB7VdyojBXmqc%3D',
          styleName: 'chat',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/7da03ab3-a98a-4d59-b5bd-f6b682b941c3.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A03Z&se=2122-09-27T07%3A52%3A03Z&sr=b&sp=rl&sig=Buiz1fs1LkudmcnWnJTFrZIAEK88jROi0oYu5VDPJeQ%3D',
          styleName: 'cheerful',
        },
      ],
    },
    shortName: 'es-MX-JorgeNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '38bb36cb-0dda-403c-b012-3b447db510c3',
    locale: 'zh-CN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, XiaoyiNeural)',
    properties: {
      DisplayName: 'Xiaoyi',
      DisplayVoiceName: 'XiaoyiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      IsTtsBreakEnabled: 'True',
      LocalName: '晓伊',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '434',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-XiaoyiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions:
        '{"angry":7,"disgruntled":5,"affectionate":9,"cheerful":4,"fearful":3,"sad":8,"embarrassed":1,"serious":6,"gentle":11,"Default":0}',
      VoiceStyleNames:
        'angry,disgruntled,affectionate,cheerful,fearful,sad,embarrassed,serious,gentle,Default',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/8dca7a47-fadd-47da-b53b-18c72d11e770.wav?sv=2019-07-07&sr=b&sig=0NFDoj%2BFIUMovPxsxNyE1QL2Ni6i%2F5ZtYqxMfn0Qtgs%3D&st=2022-08-01T10:39:27Z&se=2122-07-08T10:44:27Z&sp=rl',
          styleName: 'affectionate',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b0411bd8-ccd0-448b-bf31-9059c631aa82.wav?sv=2019-07-07&sr=b&sig=kTKG%2B%2FUjoeA5wyuczggLKYsEg5dy%2FvkLiSsRNfsVc1o%3D&st=2022-08-01T10:39:30Z&se=2122-07-08T10:44:30Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/934e1005-fecf-4583-9fd1-cb41b5363931.wav?sv=2019-07-07&sr=b&sig=gPibA1B5o778DFSa5qRmRJRAkJYmDR%2F%2BhukEAu6v%2B5c%3D&st=2022-08-01T10:39:32Z&se=2122-07-08T10:44:32Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/603e0478-185e-4c68-957f-597c1accfea3.wav?sv=2019-07-07&sr=b&sig=Mi2F15S62r3tnGZ5WyJmxF%2FQEJi%2FB78eIxdocQ9f240%3D&st=2022-08-01T10:39:35Z&se=2122-07-08T10:44:35Z&sp=rl',
          styleName: 'disgruntled',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5250e1de-96d1-4928-a66d-4aa4ccf7f056.wav?sv=2019-07-07&sr=b&sig=45NNJpo%2F02LXda5gLuRpbegDHW%2FhCTFEZtY8BJAhVFg%3D&st=2022-08-01T10:39:38Z&se=2122-07-08T10:44:38Z&sp=rl',
          styleName: 'embarrassed',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5e4a3e8e-827b-4128-995f-1d037753be31.wav?sv=2019-07-07&sr=b&sig=QHdmx%2FjuwGPBsMe3Jgj%2Bb1r2BUpz%2FKln6OySwAkAHig%3D&st=2022-08-01T10:39:41Z&se=2122-07-08T10:44:41Z&sp=rl',
          styleName: 'fearful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fd0cae08-8c8a-4586-9657-03af6b696f46.wav?sv=2019-07-07&sr=b&sig=006JMQfV%2BJl%2BvPACVMs%2B4%2Fnj%2BjsYjTTEEeleWaeF8wk%3D&st=2022-08-01T10:39:43Z&se=2122-07-08T10:44:43Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/18c4fee8-72bd-4434-9f9e-f6ec11043358.wav?sv=2019-07-07&sr=b&sig=KJ6FBhKej53KML8BU72DIky8J0Kv5Lz%2F4IuKhmMEXsU%3D&st=2022-08-01T10:39:46Z&se=2122-07-08T10:44:46Z&sp=rl',
          styleName: 'gentle',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/623c8d2b-0fee-4e99-8f6c-80a176131f85.wav?sv=2019-07-07&sr=b&sig=amuCkshHZlGx37LLEbuSOSEXEaI24tGcSr6K5wA%2F770%3D&st=2022-08-01T10:39:49Z&se=2122-07-08T10:44:49Z&sp=rl',
          styleName: 'sad',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/84a256bf-76f9-4acb-8b48-5bad65939673.wav?sv=2019-07-07&sr=b&sig=PYYr7Y8ipG3FM0v4zdMBHFtZdV%2B%2F8ii%2BfMfnxbH4XV0%3D&st=2022-08-01T10:39:52Z&se=2122-07-08T10:44:52Z&sp=rl',
          styleName: 'serious',
        },
      ],
    },
    shortName: 'zh-CN-XiaoyiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '2380ef4f-34bb-4d06-b6e2-3bbf42621a26',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, BrandonNeural)',
    properties: {
      DisplayName: 'Brandon',
      DisplayVoiceName: 'BrandonNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Brandon',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '133',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-BrandonNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0867127c-6078-4de7-ab37-81576a93a6bb.wav?sv=2019-07-07&sr=b&sig=358%2BaA9gu7Zu4gb77cx3%2BuUdNexvj8OVDG6YviD16ro%3D&st=2021-10-29T08:25:32Z&se=2121-10-05T08:30:32Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-US-BrandonNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'b99565a4-9413-4ad5-b1b0-3ceb4bf9ed48',
    locale: 'sv-SE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (sv-SE, MattiasNeural)',
    properties: {
      DisplayName: 'Mattias',
      DisplayVoiceName: 'MattiasNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Mattias',
      LocaleDescription: 'Swedish',
      OrderInVoiceListApi: '389',
      PreviewSentence:
        'Bygg appar och tjänster som talar naturligt till användarna, och förbättrar tillgänglighet och användbarhet.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'sv-SE-MattiasNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'sv-SE',
      localeZH: '瑞典语(瑞典)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d5f0b418-d063-41b8-8f27-d7bd243718cc.wav?sv=2019-07-07&sr=b&sig=%2B278BnO6Pk4fTzPi03N9TRlehgV5pn0GUvxN0P2FP8E%3D&st=2021-10-29T08:28:11Z&se=2121-10-05T08:33:11Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'sv-SE-MattiasNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '803fad75-27ca-4d9e-99d9-3cf9bf365713',
    locale: 'az-AZ',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (az-AZ, BabekNeural)',
    properties: {
      DisplayName: 'Babek',
      DisplayVoiceName: 'BabekNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Babək',
      LocaleDescription: 'az-AZ',
      OrderInVoiceListApi: '37',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'az-AZ-BabekNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'az-AZ',
      localeZH: '阿塞拜疆语(阿塞拜疆) ',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/70a34498-9023-4759-bffc-12ed0c69fff7.wav?sv=2019-07-07&sr=b&sig=VuPOBpyfRR6HB4MHjv1Z9HFkWLraNE7NTXh02Ht2Kac%3D&st=2022-07-01T09:08:52Z&se=2122-06-07T09:13:52Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'az-AZ-BabekNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '6cf1db56-de08-493a-a757-3d9bbee7d7d6',
    locale: 'mk-MK',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (mk-MK, AleksandarNeural)',
    properties: {
      DisplayName: 'Aleksandar',
      DisplayVoiceName: 'AleksandarNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Александар',
      LocaleDescription: 'mk-MK',
      OrderInVoiceListApi: '321',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'mk-MK-AleksandarNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'mk-MK',
      localeZH: '马其顿语(北马其顿)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/7efcaca0-5639-4ee0-b23a-24825d6b89b5.wav?sv=2019-07-07&sr=b&sig=3cBQmtc4kTyoWFmB7cAh3pxdZCoRNUPWu9hEPS24UVg%3D&st=2022-01-19T09:02:02Z&se=2121-12-26T09:07:02Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'mk-MK-AleksandarNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'b8c2f01f-68d1-4341-b46f-3e7c67f1a149',
    locale: 'fr-CA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-CA, AntoineNeural)',
    properties: {
      DisplayName: 'Antoine',
      DisplayVoiceName: 'AntoineNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Antoine',
      LocaleDescription: 'French (Canada)',
      OrderInVoiceListApi: '237',
      PreviewSentence:
        'Créer des applications et des services qui parlent aux utilisateurs, améliorant ainsi l’accessibilité et la facilité d’utilisation.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-CA-AntoineNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fr-CA',
      localeZH: '法语(加拿大)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c506ebb2-ff4d-4a28-9f0e-090dbec86426.wav?sv=2019-07-07&sr=b&sig=z8hbRYBq%2BPKzvevrovCLmpCQdLROq4T%2FFbxOTtdXUjE%3D&st=2021-10-29T08:26:56Z&se=2121-10-05T08:31:56Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fr-CA-AntoineNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '92a6cc59-8b2b-4e31-bb80-3e8632111ae4',
    locale: 'es-DO',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-DO, RamonaNeural)',
    properties: {
      DisplayName: 'Ramona',
      DisplayVoiceName: 'RamonaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Ramona',
      LocaleDescription: 'es-DO',
      OrderInVoiceListApi: '166',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-DO-RamonaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-DO',
      localeZH: '西班牙语(多米尼加共和国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/21cc3ece-bdc5-402d-988c-89cea1fdf345.wav?sv=2019-07-07&sr=b&sig=weBUrnkCgmSxqyNcivNAXuzScznSx1N3gY75FOiRapk%3D&st=2021-10-29T08:30:37Z&se=2121-10-05T08:35:37Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-DO-RamonaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '6f16cbe3-335c-46fe-a509-3e8da79df90d',
    locale: 'es-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-ES, TrianaNeural)',
    properties: {
      DisplayName: 'Triana',
      DisplayVoiceName: 'TrianaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Triana',
      LocaleDescription: 'Spanish (Spain)',
      OrderInVoiceListApi: '181',
      PreviewSentence:
        'Crea aplicaciones y servicios que hablan de forma natural a los usuarios, y que mejoran la accesibilidad y la facilidad de uso.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-ES-TrianaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-ES',
      localeZH: '西班牙语(西班牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/efa468bf-6bbb-4294-8b44-6703fa29c5e9.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A56Z&se=2122-09-27T07%3A51%3A56Z&sr=b&sp=rl&sig=VUjEYddffqp1mNZoOOzD4LAJfaKW24laLI%2BCM%2BZ4JB0%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-ES-TrianaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '33583255-1cdd-4845-9f9f-3eb557ec2c59',
    locale: 'zh-CN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, YunhaoNeural)',
    properties: {
      DisplayName: 'Yunhao',
      DisplayVoiceName: 'YunhaoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      IsTtsBreakEnabled: 'True',
      LocalName: '云皓',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '438',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-YunhaoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"Default":0}',
      VoiceStyleNames: 'advertisement-upbeat,Default',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/1bb3cb6b-2916-4c38-9b59-ce555af34714.wav?sv=2019-07-07&sr=b&sig=nXkSX2D%2BJ60H%2F3zKv27W9g8buPa6IL33HqFI%2BOuZWLk%3D&st=2022-04-28T07:18:47Z&se=2122-04-04T07:23:47Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f22e8a87-0f88-40e8-9ea4-9e347b2ae0c4.wav?sv=2019-07-07&sr=b&sig=pLBpuHZL4e0hCCtqa10ax%2BhWBPx0tiS4d%2FgvG1dv%2BkY%3D&st=2022-04-28T07:18:44Z&se=2122-04-04T07:23:44Z&sp=rl',
          styleName: 'Advertisement_upbeat',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f22e8a87-0f88-40e8-9ea4-9e347b2ae0c4.wav?sv=2019-07-07&sr=b&sig=pLBpuHZL4e0hCCtqa10ax%2BhWBPx0tiS4d%2FgvG1dv%2BkY%3D&st=2022-04-28T07:18:44Z&se=2122-04-04T07:23:44Z&sp=rl',
          styleName: 'Advertisement-upbeat',
        },
      ],
    },
    shortName: 'zh-CN-YunhaoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'e9086169-7176-4afc-8b5e-3f06a3ad1fb3',
    locale: 'ja-JP',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ja-JP, AoiNeural)',
    properties: {
      DisplayName: 'Aoi',
      DisplayVoiceName: 'AoiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: '碧衣',
      LocaleDescription: 'Japanese',
      OrderInVoiceListApi: '291',
      PreviewSentence:
        'アクセシビリティとユーザビリティを向上させながら、ユーザーに自然なことばを話すアプリとサービスを構築する',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ja-JP-AoiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ja-JP',
      localeZH: '日语(日本)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d4d02c8a-6d19-4c01-bd58-8ed58ee0a4d2.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A24Z&se=2122-09-27T07%3A52%3A24Z&sr=b&sp=rl&sig=zApsHIt6ntmPdn93iJA2LvyiZumxeEN%2FxroDuHBT%2BEo%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ja-JP-AoiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'c2936675-a301-488e-b13c-3ffec5a2edbe',
    locale: 'nl-NL',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (nl-NL, MaartenNeural)',
    properties: {
      DisplayName: 'Maarten',
      DisplayVoiceName: 'MaartenNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Maarten',
      LocaleDescription: 'Dutch',
      OrderInVoiceListApi: '344',
      PreviewSentence:
        'Ontwikkel apps en diensten die natuurlijk aanvoelen, waardoor de toegankelijkheid en bruikbaarheid vergroot worden.',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'nl-NL-MaartenNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'nl-NL',
      localeZH: '荷兰语(荷兰)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/3d8c18d9-4d91-42cb-8bd8-f7bf28849cf2.wav?sv=2019-07-07&sr=b&sig=SrFVRkZpWxlyH%2BC4rtc7u8iMgrQcZ5KkNnrfG1nIHFk%3D&st=2021-10-29T08:27:47Z&se=2121-10-05T08:32:47Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'nl-NL-MaartenNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'e8cbc91c-6e36-4a26-a417-40c6765b738a',
    locale: 'en-SG',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-SG, WayneNeural)',
    properties: {
      DisplayName: 'Wayne',
      DisplayVoiceName: 'WayneNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Wayne',
      LocaleDescription: 'en-SG',
      OrderInVoiceListApi: '123',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-SG-WayneNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-SG',
      localeZH: '英语(新加坡)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/8d113bc3-3622-4196-9da5-8e01b1681144.wav?sv=2019-07-07&sr=b&sig=mzGnkWj3yqPbPHR%2FKVtioYPCUBwqUr6PcSIJaQ9Targ%3D&st=2022-07-03T04:14:37Z&se=2122-06-09T04:19:37Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-SG-WayneNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '87d4f7cd-10dc-426b-8465-41a8a5b8fa12',
    locale: 'es-HN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-HN, CarlosNeural)',
    properties: {
      DisplayName: 'Carlos',
      DisplayVoiceName: 'CarlosNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Carlos',
      LocaleDescription: 'es-HN',
      OrderInVoiceListApi: '188',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-HN-CarlosNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-HN',
      localeZH: '西班牙语(洪都拉斯)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/493ed93a-78ad-4d38-97db-750760af2ce1.wav?sv=2019-07-07&sr=b&sig=wYNimvqWtWXEAhE%2FgTyb%2BUgUj87zOF%2F9P5ghHiUs1VA%3D&st=2021-10-29T08:30:44Z&se=2121-10-05T08:35:44Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-HN-CarlosNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '882abb1e-adef-47c6-9c0e-423e4bb1a2ab',
    locale: 'en-GB',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-GB, MaisieNeural)',
    properties: {
      DisplayName: 'Maisie',
      DisplayVoiceName: 'MaisieNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Maisie',
      LocaleDescription: 'English (UK)',
      OrderInVoiceListApi: '100',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-GB-MaisieNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-GB',
      localeZH: '英语(英国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fb35070d-eca9-47ab-9a34-6438fe67160a.wav?sv=2019-07-07&sr=b&sig=4F%2F%2B6l43bM0X1XKIsCnrleCh8i47yKmaj1pFCghyRbM%3D&st=2022-01-19T09:01:06Z&se=2121-12-26T09:06:06Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-GB-MaisieNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '65cf2392-2e60-43af-b6a9-42eec2748f9e',
    locale: 'so-SO',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (so-SO, UbaxNeural)',
    properties: {
      DisplayName: 'Ubax',
      DisplayVoiceName: 'UbaxNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Ubax',
      LocaleDescription: 'so-SO',
      OrderInVoiceListApi: '380',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'so-SO-UbaxNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'so-SO',
      localeZH: '索马里语(索马里)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/13f47a53-1a1f-4ca8-82e5-8de186d301af.wav?sv=2019-07-07&sr=b&sig=npN9xxUcrI8K0RPzICKDkUpPQrg33pvrY5lx0wjkEoQ%3D&st=2021-10-29T08:31:14Z&se=2121-10-05T08:36:14Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'so-SO-UbaxNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'f72fee1a-54ea-4186-a30e-430b14b96cc4',
    locale: 'bn-BD',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (bn-BD, NabanitaNeural)',
    properties: {
      DisplayName: 'Nabanita',
      DisplayVoiceName: 'NabanitaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'নবনীতা',
      LocaleDescription: 'bn-BD',
      OrderInVoiceListApi: '41',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'bn-BD-NabanitaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'bn-BD',
      localeZH: '孟加拉语(孟加拉)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/397bfb78-6f3d-42f3-a604-32b2efce1035.wav?sv=2019-07-07&sr=b&sig=Wj4Xiu9SGq%2FirDB9OOLwr7a2fZidt2KJSgvCrGA9PHk%3D&st=2021-10-29T08:30:21Z&se=2121-10-05T08:35:21Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'bn-BD-NabanitaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'f710a1a0-ff4e-4eef-9909-44376390cea5',
    locale: 'ar-QA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-QA, MoazNeural)',
    properties: {
      DisplayName: 'Moaz',
      DisplayVoiceName: 'MoazNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'معاذ',
      LocaleDescription: 'ar-QA',
      OrderInVoiceListApi: '28',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-QA-MoazNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-QA',
      localeZH: '阿拉伯语(卡塔尔)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/150ba333-6b43-43e3-904e-be12f3bc15d7.wav?sv=2019-07-07&sr=b&sig=ARm6xz%2FbsVTx%2BlLjICyDWSmlOJEIWheviTzu2vse5i4%3D&st=2021-10-29T08:30:14Z&se=2121-10-05T08:35:14Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-QA-MoazNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'e49606b3-34de-4579-bf43-4458fffd85e8',
    locale: 'en-TZ',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-TZ, ElimuNeural)',
    properties: {
      DisplayName: 'Elimu',
      DisplayVoiceName: 'ElimuNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Elimu',
      LocaleDescription: 'en-TZ',
      OrderInVoiceListApi: '124',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-TZ-ElimuNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-TZ',
      localeZH: '英语(坦桑尼亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/943e3c44-605e-4561-9ceb-3b84ad3502b8.wav?sv=2019-07-07&sr=b&sig=LHJTad5qCPMTKFs4o5MHH2CmG5zoSVmDyozCotakrms%3D&st=2021-10-29T08:30:27Z&se=2121-10-05T08:35:27Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-TZ-ElimuNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'e9642536-74cd-4580-ba7b-459355d4168f',
    locale: 'en-TZ',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-TZ, ImaniNeural)',
    properties: {
      DisplayName: 'Imani',
      DisplayVoiceName: 'ImaniNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Imani',
      LocaleDescription: 'en-TZ',
      OrderInVoiceListApi: '125',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-TZ-ImaniNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-TZ',
      localeZH: '英语(坦桑尼亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f9f64db2-c055-4b3d-a231-5814a9237963.wav?sv=2019-07-07&sr=b&sig=%2Fi%2F63FPzkf1ApuW2j%2BQtVxyNOhhMYrG8S8FzQ0QWrFk%3D&st=2021-10-29T08:30:28Z&se=2121-10-05T08:35:28Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-TZ-ImaniNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'b5f86142-ce84-4483-8142-45db0d778add',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, DavisNeural)',
    properties: {
      DisplayName: 'Davis',
      DisplayVoiceName: 'DavisNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Davis',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '136',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-DavisNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"angry":7,"cheerful":4,"sad":8,"Default":0}',
      VoiceStyleNames:
        'chat,angry,cheerful,excited,friendly,hopeful,sad,shouting,terrified,unfriendly,whispering,Default',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f67d4510-0368-4477-a442-f5c8299ade13.wav?sv=2019-07-07&sr=b&sig=%2FYnFEVDxqnpTO%2FMfshbSoj784aNAefO6izOWqNfWbgQ%3D&st=2022-05-05T09:38:31Z&se=2122-04-11T09:43:31Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/9c986c62-65a7-41d3-b9ba-21dbd7655eee.wav?sv=2019-07-07&sr=b&sig=nKVoQdnSH7Nb7feJgpdB8XvDe31P40W30q2hISW%2B5kc%3D&st=2022-04-28T10:27:01Z&se=2122-04-04T10:32:01Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/7f56e8cd-6a34-4ade-ad86-572b2f43c2c8.wav?sv=2019-07-07&sr=b&sig=Be6BOmM1Pl6QB1OG70WUqBMXw0vrEO3JtNeuYYufIbg%3D&st=2022-05-05T09:38:35Z&se=2122-04-11T09:43:35Z&sp=rl',
          styleName: 'chat',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e5ee759b-77f8-497f-bcce-c5aab00f5aa8.wav?sv=2019-07-07&sr=b&sig=%2BQ9BXNMpl3zhMjVD0P%2BiLd9wKeVOLg5qp%2FNg9zannsA%3D&st=2022-04-28T10:27:04Z&se=2122-04-04T10:32:04Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/7501e596-8bb1-4973-ab39-43ef1cc8719d.wav?sv=2019-07-07&sr=b&sig=hoGJFp5%2Fo0MNusgg5MnEyHdlrTp3AjwbFdnXXtDtS8A%3D&st=2022-04-28T10:27:06Z&se=2122-04-04T10:32:06Z&sp=rl',
          styleName: 'excited',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6b419076-769c-47af-b30b-ef7bdcbf824b.wav?sv=2019-07-07&sr=b&sig=Txtx6UExlKg7gp%2FqSMnMiY6GQ5H5TC%2FlaVg5C3dC38Y%3D&st=2022-04-28T10:27:08Z&se=2122-04-04T10:32:08Z&sp=rl',
          styleName: 'friendly',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ba5d2450-8f27-444e-9f37-90dde6a09503.wav?sv=2019-07-07&sr=b&sig=cA5HyxNsjF88VZZcYg%2F%2Bn2EWwiY5RrEFKW3129LKw8E%3D&st=2022-04-28T10:27:10Z&se=2122-04-04T10:32:10Z&sp=rl',
          styleName: 'hopeful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e083a4ae-d579-44b2-b9df-4294fc1d2981.wav?sv=2019-07-07&sr=b&sig=pjScu40TKa%2FHUx2QguG5t%2BVEKVxOH4uvjC5juOqLncs%3D&st=2022-04-28T10:27:13Z&se=2122-04-04T10:32:13Z&sp=rl',
          styleName: 'sad',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c740479c-a2b0-4df2-9cc5-0038a8577c8e.wav?sv=2019-07-07&sr=b&sig=27iuoKjjYYtow30AHuqrtMsbSV9DQyNZpnEWPSWBbiI%3D&st=2022-04-28T10:27:15Z&se=2122-04-04T10:32:15Z&sp=rl',
          styleName: 'shouting',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d27366ab-e485-48d0-9d71-26a7221e942c.wav?sv=2019-07-07&sr=b&sig=uJC9bkFzM%2BBfLj6v4iiFF3j2mVykiGJ9UmSbqn6LstA%3D&st=2022-04-28T10:27:17Z&se=2122-04-04T10:32:17Z&sp=rl',
          styleName: 'terrified',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fb59458e-d686-4a13-b143-8b481128159b.wav?sv=2019-07-07&sr=b&sig=dq1IEfmbR8k%2FYQzgkbbhqGk5TzS12%2BDOZo%2FsE1E97m4%3D&st=2022-04-28T10:27:19Z&se=2122-04-04T10:32:19Z&sp=rl',
          styleName: 'unfriendly',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/66490161-58f1-4bb8-bc7b-0908479d4ebb.wav?sv=2019-07-07&sr=b&sig=58CiFhxM8gLuQOHKSWR%2Bfw8%2BENIjJeke6yJpRITpMm0%3D&st=2022-04-28T10:27:22Z&se=2122-04-04T10:32:22Z&sp=rl',
          styleName: 'whispering',
        },
      ],
    },
    shortName: 'en-US-DavisNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '865dd6ad-a19e-475b-86b9-464c889250a6',
    locale: 'de-DE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (de-DE, ConradNeural)',
    properties: {
      DisplayName: 'Conrad',
      DisplayVoiceName: 'ConradNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Conrad',
      LocaleDescription: 'German (Germany)',
      OrderInVoiceListApi: '74',
      PreviewSentence:
        'Erstellen Sie Apps und Dienste, die auf natürliche Art mit Nutzern sprechen, und dadurch Zugänglichkeit und Benutzerfreundlichkeit verbessern.',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'de-DE-ConradNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"cheerful":4,"Default":0}',
      VoiceStyleNames: 'cheerful,Default',
      locale: 'de-DE',
      localeZH: '德语(德国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/735fc0e8-6a38-4a00-8356-929ef25d2279.wav?sv=2019-07-07&sr=b&sig=yZ76jzMmuuy4zniNTHeE7jIKdolKJq20T733vmAmsYQ%3D&st=2021-10-29T08:26:14Z&se=2121-10-05T08:31:14Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'de-DE-ConradNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '24760f62-1a7b-4b6f-a121-46b86bab164f',
    locale: 'sr-RS',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (sr-RS, SophieNeural)',
    properties: {
      DisplayName: 'Sophie',
      DisplayVoiceName: 'SophieNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Софија',
      LocaleDescription: 'sr-RS',
      OrderInVoiceListApi: '384',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'sr-RS-SophieNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'sr-RS',
      localeZH: '塞尔维亚语(塞尔维亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/134f99a5-2380-4087-baee-ea1481a60d61.wav?sv=2019-07-07&sr=b&sig=46NU8hLT5qoQjYWqZTCTamjlusNDg2%2BYUrY%2Bg%2B%2Fc%2BlA%3D&st=2022-01-19T09:02:22Z&se=2121-12-26T09:07:22Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'sr-RS-SophieNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '1d77f6c6-6f94-4bef-884e-46f3a9e7c258',
    locale: 'kn-IN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (kn-IN, SapnaNeural)',
    properties: {
      DisplayName: 'Sapna',
      DisplayVoiceName: 'SapnaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'ಸಪ್ನಾ',
      LocaleDescription: 'kn-IN',
      OrderInVoiceListApi: '306',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'kn-IN-SapnaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'kn-IN',
      localeZH: '埃纳德语(印度)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ef2e9d4c-ecdc-46f8-bd77-59f193cf211f.wav?sv=2019-07-07&sr=b&sig=1nAMjPNHWRg4JWEDRddBKFwYjprZAGBsJs5sx%2FbCsv0%3D&st=2022-01-19T09:01:56Z&se=2121-12-26T09:06:56Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'kn-IN-SapnaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '57303f99-fbef-44bf-b291-4735f88f9a61',
    locale: 'es-UY',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-UY, MateoNeural)',
    properties: {
      DisplayName: 'Mateo',
      DisplayVoiceName: 'MateoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Mateo',
      LocaleDescription: 'es-UY',
      OrderInVoiceListApi: '219',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-UY-MateoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-UY',
      localeZH: '西班牙语(乌拉圭)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/69065338-efaf-40d9-880c-2e138333c82e.wav?sv=2019-07-07&sr=b&sig=DPChWTekzsgPwMmnKoNpqJ86Bv0a5GeQgzktMkrv9I8%3D&st=2021-10-29T08:30:57Z&se=2121-10-05T08:35:57Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-UY-MateoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '8db5ae9b-9d62-48e3-ae1e-4899a2b2e053',
    locale: 'es-MX',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-MX, MarinaNeural)',
    properties: {
      DisplayName: 'Marina',
      DisplayVoiceName: 'MarinaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Marina',
      LocaleDescription: 'Spanish (Mexico)',
      OrderInVoiceListApi: '200',
      PreviewSentence:
        'Crea apps y servicios que hablen de forma natural con los usuarios, mejorando la accesibilidad y la usabilidad.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-MX-MarinaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-MX',
      localeZH: '西班牙语(墨西哥)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ef8f09c8-2dff-42f0-a1c1-43a6450c292b.wav?sv=2019-07-07&sr=b&sig=FDMlhJi2j3oNzmw7kjQ4jtBeWzLrraFWED7%2BAzo0aIk%3D&st=2022-07-01T09:09:19Z&se=2122-06-07T09:14:19Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-MX-MarinaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '93cdc1ca-0ba2-4615-91ec-495bd6b40b3d',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, JasonNeural)',
    properties: {
      DisplayName: 'Jason',
      DisplayVoiceName: 'JasonNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Jason',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '141',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-JasonNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"angry":7,"cheerful":4,"sad":8,"Default":0}',
      VoiceStyleNames:
        'angry,cheerful,excited,friendly,hopeful,sad,shouting,terrified,unfriendly,whispering,Default',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d91baad7-af08-40a7-aa12-142983ee5540.wav?sv=2019-07-07&sr=b&sig=Agrr1Y8DJjHq7w5eQ4V1GA3eqWU5UK26mlTujIY29T8%3D&st=2022-05-05T09:38:52Z&se=2122-04-11T09:43:52Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4490bb49-2c68-4267-b515-13b3906eef72.wav?sv=2019-07-07&sr=b&sig=lcSIr8vqtnzl9hW5NXsuXkUwLEPsaVDKVMfzIQIwTuU%3D&st=2022-04-28T10:27:47Z&se=2122-04-04T10:32:47Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c22e964e-166b-415c-a11e-4dc8fa0b1850.wav?sv=2019-07-07&sr=b&sig=ym7C9JnfduT8fOO59mnC7mNai1qtq8qvM8qkzHS%2FLDk%3D&st=2022-04-28T10:27:49Z&se=2122-04-04T10:32:49Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c01b0581-d9bd-4643-b68d-08f6415c8f90.wav?sv=2019-07-07&sr=b&sig=%2FVUe9782fJgBDFn1fNlcnLNo7%2BtIw1L%2Fhauipy8jjHU%3D&st=2022-04-28T10:27:51Z&se=2122-04-04T10:32:51Z&sp=rl',
          styleName: 'excited',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4b3450a5-9b64-4834-875d-ff57a87b91e9.wav?sv=2019-07-07&sr=b&sig=WQAIvc%2BxU1utV5fuvmqCtQvj%2BH5%2FUf016XnAmjEFPvk%3D&st=2022-04-28T10:27:54Z&se=2122-04-04T10:32:54Z&sp=rl',
          styleName: 'friendly',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/609b0f08-7e35-4ee3-ab59-cda5321e10d3.wav?sv=2019-07-07&sr=b&sig=VEgybfyR3%2BTnv2miGfBHPVPSRwbc0ZWBRVthLBn5o8o%3D&st=2022-04-28T10:27:56Z&se=2122-04-04T10:32:56Z&sp=rl',
          styleName: 'hopeful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/53c5ff3f-48b4-4e97-9c1a-9f04b60f1d26.wav?sv=2019-07-07&sr=b&sig=Ab%2FPMCfyRtRLoV50BuSyrhD6YBpMi0oxJLp9QSdyhsI%3D&st=2022-04-28T10:27:58Z&se=2122-04-04T10:32:58Z&sp=rl',
          styleName: 'sad',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fbccc004-ff23-4d86-90f4-b6766d74f0e4.wav?sv=2019-07-07&sr=b&sig=5QpUD2Qb8s50upb%2FWGuyuRy%2Ff4OddpEodHgyYvLbo%2FQ%3D&st=2022-04-28T10:28:01Z&se=2122-04-04T10:33:01Z&sp=rl',
          styleName: 'shouting',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/06b5569f-f055-4e26-9dd0-f7b2758af6cb.wav?sv=2019-07-07&sr=b&sig=wtFKkhYj%2BFUSGPfGAljsUMpdvFebqdLEKjVPZ3r%2Fa%2Bs%3D&st=2022-04-28T10:28:03Z&se=2122-04-04T10:33:03Z&sp=rl',
          styleName: 'terrified',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/7967cdb0-eee0-4975-bfce-08b3d57d65a9.wav?sv=2019-07-07&sr=b&sig=k885kwRb9v1Ud1YKB8gG7v80nBTvgF75raZ4KorpvLM%3D&st=2022-04-28T10:28:05Z&se=2122-04-04T10:33:05Z&sp=rl',
          styleName: 'unfriendly',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/668ff93c-b99e-46c5-8a02-c45d23a6e03e.wav?sv=2019-07-07&sr=b&sig=tOHs4zf32RqWLTWrZ7znP5VrgHXfzTKLRi%2BEuEjh1Uk%3D&st=2022-04-28T10:28:07Z&se=2122-04-04T10:33:07Z&sp=rl',
          styleName: 'whispering',
        },
      ],
    },
    shortName: 'en-US-JasonNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '43b6bf15-38b2-4df8-a439-498e3ed7bfc3',
    locale: 'zh-HK',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-HK, HiuGaaiNeural)',
    properties: {
      DisplayName: 'HiuGaai',
      DisplayVoiceName: 'HiuGaaiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: '曉佳',
      LocaleDescription: 'Chinese (Hong Kong)',
      OrderInVoiceListApi: '450',
      PreviewSentence: '開發可自然地與用戶溝通的應用程式及服務，以提升其使用度及可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-HK-HiuGaaiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'zh-HK',
      localeZH: '中文(粤语，繁体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/3c5e20c1-8209-4367-bc69-e159fea8eebe.wav?sv=2019-07-07&sr=b&sig=n2vg6ynloycZTFiyRUi1J6Q%2FeEjJmH9KCcsmjPRcvv8%3D&st=2021-10-29T08:29:48Z&se=2121-10-05T08:34:48Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'zh-HK-HiuGaaiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '817b1bf6-0020-4d2b-a978-49ded1768b12',
    locale: 'ca-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ca-ES, JoanaNeural)',
    properties: {
      DisplayName: 'Joana',
      DisplayVoiceName: 'JoanaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Joana',
      LocaleDescription: 'Catalan',
      OrderInVoiceListApi: '47',
      PreviewSentence:
        "Crea aplicacions i serveis que parlen de forma natural als usuaris, i que milloren l'accessibilitat i la facilitat d'ús.",
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ca-ES-JoanaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ca-ES',
      localeZH: '加泰罗尼亚语(西班牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/26cbe67b-1673-4d25-bf17-c0995331e6b7.wav?sv=2019-07-07&sr=b&sig=78BaN3VccGs2oUd%2B4slFM5%2Frptt9CwN1cJZ1kIWjarI%3D&st=2021-10-29T08:26:03Z&se=2121-10-05T08:31:03Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ca-ES-JoanaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '6d04d9b3-41f0-4c52-835b-4a353e799ab8',
    locale: 'ru-RU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ru-RU, DariyaNeural)',
    properties: {
      DisplayName: 'Dariya',
      DisplayVoiceName: 'DariyaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Дария',
      LocaleDescription: 'Russian',
      OrderInVoiceListApi: '371',
      PreviewSentence:
        'Возможность создавать приложения и сервисы, которые естественным образом общаются с пользователями, улучшая доступность и удобство использования.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ru-RU-DariyaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ru-RU',
      localeZH: '俄语(俄罗斯)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e3410eff-3be8-46af-9fff-6ba0c1f0545e.wav?sv=2019-07-07&sr=b&sig=g6KN7f%2BvZAvafBO1L%2FGGhD78hmwO5iXKiecuHnR6CuA%3D&st=2021-10-29T08:28:02Z&se=2121-10-05T08:33:02Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ru-RU-DariyaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '38db11b6-fa64-4989-8d75-4a48695ee5cd',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, SaraNeural)',
    properties: {
      DisplayName: 'Sara',
      DisplayVoiceName: 'SaraNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Sara',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '146',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-SaraNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"angry":7,"cheerful":4,"sad":8,"Default":0}',
      VoiceStyleNames:
        'angry,cheerful,excited,friendly,hopeful,sad,shouting,terrified,unfriendly,whispering,Default',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/05a14c34-7f48-4c70-8e99-d8749a24d3a7.wav?sv=2019-07-07&sr=b&sig=uSOV%2BezeHBl0tfa5xJ9k2MtbvVpocN5MLOJIHzdPj7I%3D&st=2021-10-29T08:25:51Z&se=2121-10-05T08:30:51Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/bf8d9029-d2da-4d44-bb45-630df0c08987.wav?sv=2019-07-07&sr=b&sig=mCjIsH6FK1UckGFOfGYezpjUtlu9UiLskJb2TCBeR3A%3D&st=2021-10-29T08:25:52Z&se=2121-10-05T08:30:52Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ea0e788b-93a3-49a9-b3d6-775ad24a4a0a.wav?sv=2019-07-07&sr=b&sig=rFk%2Bh%2FGR1IRSQzZ%2FJRP3gN0UGKSTG%2B1wdOvAadT5MnM%3D&st=2021-10-29T08:25:53Z&se=2121-10-05T08:30:53Z&sp=rl',
          styleName: 'sad',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a03cbdcc-5369-46d1-ac22-c66ee37c93e1.wav?sv=2019-07-07&sr=b&sig=wHCFXPlo7lOg2%2Fc8%2FMvOFZyaUVvnQyPGJMJf527rpwM%3D&st=2021-10-29T08:25:54Z&se=2121-10-05T08:30:54Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a9947dc7-bb80-4455-9193-19a34da5b061.wav?sv=2019-07-07&sr=b&sig=57fz%2F2ax55JOZGvrwUM89barBUZuStZkw%2BUwNFVebuo%3D&st=2022-04-28T10:26:46Z&se=2122-04-04T10:31:46Z&sp=rl',
          styleName: 'excited',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/377c8d18-761b-4a53-be8c-0bfe5eeac34f.wav?sv=2019-07-07&sr=b&sig=018Cvmng605nULkBA4tQP8dCTImtEfnWPmRMo%2FsMDEw%3D&st=2022-04-28T10:26:48Z&se=2122-04-04T10:31:48Z&sp=rl',
          styleName: 'friendly',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/7f9ad302-e6c6-4521-a887-462aa97d63d4.wav?sv=2019-07-07&sr=b&sig=togFPD9fkufNQrzdkIQigdCs0os%2F7pSdjBYSeOFDF1I%3D&st=2022-04-28T10:26:50Z&se=2122-04-04T10:31:50Z&sp=rl',
          styleName: 'hopeful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e1161fd7-fa8f-4b69-9ead-f589dbe99338.wav?sv=2019-07-07&sr=b&sig=6190x6A0Ev77JPdgseFpMS5jOnTuaVM8zy9t1Hz9dqA%3D&st=2022-04-28T10:26:53Z&se=2122-04-04T10:31:53Z&sp=rl',
          styleName: 'shouting',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ab9ad803-3646-4a35-b3a8-d8ac840308f9.wav?sv=2019-07-07&sr=b&sig=HmcYdePtG%2B1FW66xtrMo5v67IYUzkKYuy6hMVEyIS78%3D&st=2022-04-28T10:26:55Z&se=2122-04-04T10:31:55Z&sp=rl',
          styleName: 'terrified',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ca10fbbb-40d9-4693-94ad-9fff483fb192.wav?sv=2019-07-07&sr=b&sig=n4hyE0K7s1Zc80dkXVOLc6NKmh4tLg0j5HEtY3dj9Mo%3D&st=2022-04-28T10:26:57Z&se=2122-04-04T10:31:57Z&sp=rl',
          styleName: 'unfriendly',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/34ed5c93-7578-4e11-9253-124a57f0e034.wav?sv=2019-07-07&sr=b&sig=EXlrt1LRP86wZEV7ljBxuz6708B3rqEwyhM3Qg0Wnd0%3D&st=2022-04-28T10:26:59Z&se=2122-04-04T10:31:59Z&sp=rl',
          styleName: 'whispering',
        },
      ],
    },
    shortName: 'en-US-SaraNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '845b64ce-7525-49b4-9a5a-4c42b23ce64d',
    locale: 'en-HK',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-HK, YanNeural)',
    properties: {
      DisplayName: 'Yan',
      DisplayVoiceName: 'YanNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Yan',
      LocaleDescription: 'en-HK',
      OrderInVoiceListApi: '109',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-HK-YanNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-HK',
      localeZH: '英语(香港特别行政区)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/caaf35ae-387d-4a83-99f9-74b5a62679f2.wav?sv=2019-07-07&sr=b&sig=nlFZBLhzhjgG5BoK7MH07aS%2B2vDupW6WKZtC71H5RXg%3D&st=2021-10-29T08:26:27Z&se=2121-10-05T08:31:27Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-HK-YanNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '0bad16a8-a63e-4e1d-bc80-4c88ddaf2b77',
    locale: 'it-IT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (it-IT, IsabellaNeural)',
    properties: {
      DisplayName: 'Isabella',
      DisplayVoiceName: 'IsabellaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Isabella',
      LocaleDescription: 'Italian',
      OrderInVoiceListApi: '275',
      PreviewSentence:
        'Benvenuti nella piattaforma di generazione di contenuti audio di Microsoft.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'it-IT-IsabellaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"cheerful":4,"Default":0}',
      VoiceStyleNames: 'cheerful,chat,Default',
      locale: 'it-IT',
      localeZH: '意大利语(意大利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5b5f4368-91d8-4253-98fe-848328d3af4d.wav?sv=2019-07-07&sr=b&sig=%2FIb64GCARSRi%2FOkF2yUsKfTMq7PSbNuDdNSgAvxhbZs%3D&st=2021-10-29T08:27:19Z&se=2121-10-05T08:32:19Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/cc37df70-08ae-4557-b13f-df16c02ae630.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A19Z&se=2122-09-27T07%3A52%3A19Z&sr=b&sp=rl&sig=16tIK53mXg5H9P2zEsFuGZUIoXnoput8lNo33zgB1KI%3D',
          styleName: 'chat',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/cc98df98-c096-49e3-af49-9dca175fb876.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A21Z&se=2122-09-27T07%3A52%3A21Z&sr=b&sp=rl&sig=2%2FvsO%2F5o6LaY0vRtW3iqSEuJt%2BQS1ubxEPkwowhVKCQ%3D',
          styleName: 'cheerful',
        },
      ],
    },
    shortName: 'it-IT-IsabellaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '286a3a96-fa74-4489-bb0a-4db9f437d2a1',
    locale: 'he-IL',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (he-IL, HilaNeural)',
    properties: {
      DisplayName: 'Hila',
      DisplayVoiceName: 'HilaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'הילה',
      LocaleDescription: 'Hebrew (Israel)',
      OrderInVoiceListApi: '262',
      PreviewSentence:
        'בנה יישומים ושירותים שמדברים בטבעיות למשתמשים, שמשפרים את  הנגישות והשימושיות.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'he-IL-HilaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'he-IL',
      localeZH: '希伯来语(以色列)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a2a130fe-ab21-4e2f-a66b-377b69e620db.wav?sv=2019-07-07&sr=b&sig=ycKJ4ZRV%2FWK9ZcXXnmQ%2F0%2Bq0W%2FrlJ1Rpy%2BkQ8jwvd%2BE%3D&st=2021-10-29T08:27:08Z&se=2121-10-05T08:32:08Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'he-IL-HilaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '40656072-27e0-4599-8cc4-4de109bcb0b1',
    locale: 'en-AU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-AU, NatashaNeural)',
    properties: {
      DisplayName: 'Natasha',
      DisplayVoiceName: 'NatashaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Natasha',
      LocaleDescription: 'English (Australia)',
      OrderInVoiceListApi: '86',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-AU-NatashaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-AU',
      localeZH: '英语(澳大利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/3d28afc4-8a40-46f2-8d42-82d433254c86.wav?sv=2019-07-07&sr=b&sig=WTcYVq2eEiEiOL7p554SSaEBVlk5kgBHzZ5ZX3WOxMA%3D&st=2021-10-29T08:26:18Z&se=2121-10-05T08:31:18Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-AU-NatashaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'b06ff851-a50c-4e54-bf98-4ed52f0cb526',
    locale: 'ca-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ca-ES, AlbaNeural)',
    properties: {
      DisplayName: 'Alba',
      DisplayVoiceName: 'AlbaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Alba',
      LocaleDescription: 'Catalan',
      OrderInVoiceListApi: '48',
      PreviewSentence:
        "Crea aplicacions i serveis que parlen de forma natural als usuaris, i que milloren l'accessibilitat i la facilitat d'ús.",
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ca-ES-AlbaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ca-ES',
      localeZH: '加泰罗尼亚语(西班牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/cb0ad2fc-31fb-42f9-91e4-62ff22bd1e36.wav?sv=2019-07-07&sr=b&sig=lWyDoOAfR42fjVOgM1XkK7RsEow6UsQCl5%2BcUhl5za4%3D&st=2021-10-29T08:26:01Z&se=2121-10-05T08:31:01Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ca-ES-AlbaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'ae0b8565-6ccc-462d-8fdd-4f5521dea6b4',
    locale: 'my-MM',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (my-MM, NilarNeural)',
    properties: {
      DisplayName: 'Nilar',
      DisplayVoiceName: 'NilarNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'နီလာ',
      LocaleDescription: 'my-MM',
      OrderInVoiceListApi: '333',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'my-MM-NilarNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'my-MM',
      localeZH: '缅甸语(缅甸)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2aace1bd-f2cc-4092-94f4-d766fea09b55.wav?sv=2019-07-07&sr=b&sig=D6v8yo2id6tomE767pVG%2FIs3DZcmQpV7azCfzQob2yU%3D&st=2021-10-29T08:31:11Z&se=2121-10-05T08:36:11Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'my-MM-NilarNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '730f985a-7223-4d6d-9bff-4fb76bdd4a0e',
    locale: 'fr-FR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-FR, BrigitteNeural)',
    properties: {
      DisplayName: 'Brigitte',
      DisplayVoiceName: 'BrigitteNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Brigitte',
      LocaleDescription: 'French (France)',
      OrderInVoiceListApi: '242',
      PreviewSentence:
        "Construire des applications et services qui communiquent naturellement avec l'utilisateur, en améliorant l’accessibilité et l'ergonomie",
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-FR-BrigitteNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fr-FR',
      localeZH: '法语(法国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b8afe7a4-944b-492a-b3b1-bdcdabde6478.wav?sv=2019-07-07&sr=b&sig=44m7T38G1KGnKAL%2B46nS5vlH%2FKIWWl%2FeE3tzFwYreiQ%3D&st=2022-01-19T09:01:19Z&se=2121-12-26T09:06:19Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fr-FR-BrigitteNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '621f531c-128b-4004-9c50-500d27a6c52a',
    locale: 'pl-PL',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pl-PL, ZofiaNeural)',
    properties: {
      DisplayName: 'Zofia',
      DisplayVoiceName: 'ZofiaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Zofia',
      LocaleDescription: 'Polish',
      OrderInVoiceListApi: '347',
      PreviewSentence:
        'Twórz aplikacje i serwisy, które w kontakcie z użytkownikiem posługują się naturalną mową, co podnosi ich dostępność i użyteczność.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pl-PL-ZofiaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'pl-PL',
      localeZH: '波兰语(波兰)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/417e79c0-8c4e-42b2-8ee9-cab9c31bda8d.wav?sv=2019-07-07&sr=b&sig=xBTF%2FjEyMYJJbb5eoytILjigqDI2yWgP8U8yifuHmkY%3D&st=2021-10-29T08:27:51Z&se=2121-10-05T08:32:51Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'pl-PL-ZofiaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'db216550-1d31-4f8d-a052-5055ec63fc8d',
    locale: 'te-IN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (te-IN, MohanNeural)',
    properties: {
      DisplayName: 'Mohan',
      DisplayVoiceName: 'MohanNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'మోహన్',
      LocaleDescription: 'Telugu (India)',
      OrderInVoiceListApi: '402',
      PreviewSentence:
        'వినియోగం మరియు సౌలభ్యాన్ని మెరుగుపరిచే, సహజ శైలిలో మాట్లాడే అప్లికేషన్లు మరియు సేవలను అభివృద్ధి చేయండి.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'te-IN-MohanNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'te-IN',
      localeZH: '泰卢固语(印度)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/15235e9f-eb5f-4f73-8e58-8a8f5e4b01b1.wav?sv=2019-07-07&sr=b&sig=GpiCpTN1FxbgvW6b4UWX04mKE7DLDGkrMPzghZ%2F9Dqc%3D&st=2021-10-29T08:28:17Z&se=2121-10-05T08:33:17Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'te-IN-MohanNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '8c9f6f0a-f349-4a59-b829-50f224817a46',
    locale: 'ta-IN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ta-IN, ValluvarNeural)',
    properties: {
      DisplayName: 'Valluvar',
      DisplayVoiceName: 'ValluvarNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'வள்ளுவர்',
      LocaleDescription: 'Tamil (India)',
      OrderInVoiceListApi: '395',
      PreviewSentence:
        'பயனர்களிடம் இயற்கையாக பேசும் பயன்பாடுகள் மற்றும் சேவைகளை உருவாகுதல், இது அணுகல்தன்மை மற்றும் பயன்பாட்டினை மேம்படுத்தும்.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ta-IN-ValluvarNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ta-IN',
      localeZH: '泰米尔语(印度)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/be283fe5-4754-4638-9565-2db5cb941069.wav?sv=2019-07-07&sr=b&sig=9qEFjG5XxTaTvlW8CmbMjzoKAYckED64NXc6aAibffU%3D&st=2021-10-29T08:28:16Z&se=2121-10-05T08:33:16Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ta-IN-ValluvarNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'b53bcfc5-812e-40cc-a3ce-50fdc560afcc',
    locale: 'es-CO',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-CO, GonzaloNeural)',
    properties: {
      DisplayName: 'Gonzalo',
      DisplayVoiceName: 'GonzaloNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Gonzalo',
      LocaleDescription: 'es-CO',
      OrderInVoiceListApi: '159',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-CO-GonzaloNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-CO',
      localeZH: '西班牙语(哥伦比亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2ad07c6b-4df7-4d88-b7ea-bcc297e3ecec.wav?sv=2019-07-07&sr=b&sig=atBxkxSy7yRU0qQxtAIUiZojlECXU22FXb5uCsUHA1w%3D&st=2021-10-29T08:26:42Z&se=2121-10-05T08:31:42Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-CO-GonzaloNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'eca66fba-5a66-4ada-9e99-513db4068ced',
    locale: 'sk-SK',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (sk-SK, LukasNeural)',
    properties: {
      DisplayName: 'Lukas',
      DisplayVoiceName: 'LukasNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Lukáš',
      LocaleDescription: 'Slovak',
      OrderInVoiceListApi: '375',
      PreviewSentence:
        'Vytvárajú aplikácie a služby, ktoré prirodzene komunikujú s užívateľmi, čím sa zlepšuje dostupnosť a využiteľnosť.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'sk-SK-LukasNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'sk-SK',
      localeZH: '斯洛伐克语(斯洛伐克)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fbec18f9-560b-4552-9603-11ec4ebdc657.wav?sv=2019-07-07&sr=b&sig=QIpNS9wxzwT9cFjsMuka5GNDn5NltWdmN8wkvU3%2Fnp4%3D&st=2021-10-29T08:28:05Z&se=2121-10-05T08:33:05Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'sk-SK-LukasNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '4bfabb53-98ee-4c2d-9a60-51835494916d',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, CoraNeural)',
    properties: {
      DisplayName: 'Cora',
      DisplayVoiceName: 'CoraNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Cora',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '135',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-CoraNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d4fda7ed-6cf0-4b2f-9aaf-91e1e7638039.wav?sv=2019-07-07&sr=b&sig=K%2FYb0Wdh07yXTA5rEvfFE5T%2BsQuZXPRPdhtyl79BWgI%3D&st=2021-10-29T08:25:34Z&se=2121-10-05T08:30:34Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-US-CoraNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '38c5302d-b63e-41d3-b9d5-51af941f72d4',
    locale: 'vi-VN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (vi-VN, NamMinhNeural)',
    properties: {
      DisplayName: 'NamMinh',
      DisplayVoiceName: 'NamMinhNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Nam Minh',
      LocaleDescription: 'Vietnamese',
      OrderInVoiceListApi: '418',
      PreviewSentence:
        'Phát triển phần mềm và dịch vụ có thể giao tiếp tự nhiên với người dùng, nâng cao tính tiếp cận và tính khả dụng của sản phẩm',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'vi-VN-NamMinhNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'vi-VN',
      localeZH: '越南语(越南)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/8c455650-f4ad-4e05-be61-b8664a1d3bb7.wav?sv=2019-07-07&sr=b&sig=6nQVwDyCP7k4nDrx2ioUTzgLmCYxGLw310xCUzpewIQ%3D&st=2021-10-29T08:28:30Z&se=2121-10-05T08:33:30Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'vi-VN-NamMinhNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'ed0cbbff-e662-4159-bc8f-51e4d652db47',
    locale: 'zh-CN-shaanxi',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN-shaanxi, XiaoniNeural)',
    properties: {
      DisplayName: 'Xiaoni',
      DisplayVoiceName: 'XiaoniNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: '晓妮',
      LocaleDescription: 'zh-CN-shaanxi',
      OrderInVoiceListApi: '446',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-shaanxi-XiaoniNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'zh-CN-shaanxi',
      localeZH: '中文(中原官话陕西，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c7efdb4d-0e36-4575-bc23-84389cf287c4.wav?sv=2019-07-07&sr=b&sig=hTLX0WBLjlzubtWkbhvTf7PTwQ%2BwShCrqKS3dliEoOE%3D&st=2022-08-30T08:13:48Z&se=2122-08-06T08:18:48Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'zh-CN-shaanxi-XiaoniNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '48d3ffd9-2a24-450c-b423-53762f6575c4',
    locale: 'hr-HR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (hr-HR, GabrijelaNeural)',
    properties: {
      DisplayName: 'Gabrijela',
      DisplayVoiceName: 'GabrijelaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Gabrijela',
      LocaleDescription: 'Croatian',
      OrderInVoiceListApi: '265',
      PreviewSentence:
        'Pravi aplikacije i usluge koje se obraćaju korisnicima na što prirodniji način i poboljšava njihovu pristupačnost i korišćenje.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'hr-HR-GabrijelaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'hr-HR',
      localeZH: '克罗地亚语(克罗地亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ade486ec-1c6a-4144-8e9a-2f4fe6252e2e.wav?sv=2019-07-07&sr=b&sig=6utndJ%2Fcp4uLMxkM%2F4c7i0tUJ%2BSPIu%2BzHoTMWbjgNEE%3D&st=2021-10-29T08:27:11Z&se=2121-10-05T08:32:11Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'hr-HR-GabrijelaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '94127b9b-e780-4dff-9c7f-5386d4ea6331',
    locale: 'es-GQ',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-GQ, TeresaNeural)',
    properties: {
      DisplayName: 'Teresa',
      DisplayVoiceName: 'TeresaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Teresa',
      LocaleDescription: 'es-GQ',
      OrderInVoiceListApi: '185',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-GQ-TeresaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-GQ',
      localeZH: '西班牙语(赤道几内亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/956750a0-c05e-43ab-b624-5bb4b278323d.wav?sv=2019-07-07&sr=b&sig=fIP6JO2tsktJMhNyHTUUN4O2JdnI9VxxvbXpBfR8Clk%3D&st=2021-10-29T08:30:41Z&se=2121-10-05T08:35:41Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-GQ-TeresaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'd2e289ac-2b92-4127-9183-5475de808892',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, MichelleNeural)',
    properties: {
      DisplayName: 'Michelle',
      DisplayVoiceName: 'MichelleNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Michelle',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '142',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-MichelleNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d24a8779-2140-449d-8091-a5a1b20a9a4e.wav?sv=2019-07-07&sr=b&sig=3suGCqAOfep43ll9PmXQiwV3F45CsBz0RkjSaIuIFUI%3D&st=2021-10-29T08:25:49Z&se=2121-10-05T08:30:49Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-US-MichelleNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'db1112a4-e410-496e-b6dd-5493515bc9fa',
    locale: 'fr-FR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-FR, HenriNeural)',
    properties: {
      DisplayName: 'Henri',
      DisplayVoiceName: 'HenriNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Henri',
      LocaleDescription: 'French (France)',
      OrderInVoiceListApi: '247',
      PreviewSentence:
        "Construire des applications et services qui communiquent naturellement avec l'utilisateur, en améliorant l’accessibilité et l'ergonomie",
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-FR-HenriNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"cheerful":4,"sad":8,"Default":0}',
      VoiceStyleNames: 'cheerful,sad,Default',
      locale: 'fr-FR',
      localeZH: '法语(法国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/78b56c23-c636-44ab-83fc-219610751cbe.wav?sv=2019-07-07&sr=b&sig=tqy7mF0xGR0WdCdZY8SOrXfXrocQJzJiyJ08ppLOxfA%3D&st=2021-10-29T08:27:02Z&se=2121-10-05T08:32:02Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/97240141-0ab6-43cd-8db3-d115eb950b0a.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A10Z&se=2122-09-27T07%3A52%3A10Z&sr=b&sp=rl&sig=XRkpprpp3oiqq1g9jbvhLe3Z%2BqGOujEHVGLhhOn3O4E%3D',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/1740add2-e97c-44df-b020-57e2e0e41aed.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A12Z&se=2122-09-27T07%3A52%3A12Z&sr=b&sp=rl&sig=P1uHUqUQYkLsZXqdu6BwGAZrGYdqrUOdnnR0HYhay2s%3D',
          styleName: 'sad',
        },
      ],
    },
    shortName: 'fr-FR-HenriNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '0fe16f65-a722-43d7-b79c-549b0da935ab',
    locale: 'ml-IN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ml-IN, SobhanaNeural)',
    properties: {
      DisplayName: 'Sobhana',
      DisplayVoiceName: 'SobhanaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'ശോഭന',
      LocaleDescription: 'ml-IN',
      OrderInVoiceListApi: '324',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ml-IN-SobhanaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ml-IN',
      localeZH: '马拉雅拉姆语(印度)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/27826978-1bb3-48c0-a020-70243be13bd5.wav?sv=2019-07-07&sr=b&sig=IfnymYaRSEVtH56gf7M0Y1klrWHTDfCmJfvKGdsDwKk%3D&st=2022-01-19T09:02:09Z&se=2121-12-26T09:07:09Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ml-IN-SobhanaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '2590425b-a919-44b9-b8c4-54f6a7f04255',
    locale: 'es-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-ES, EliasNeural)',
    properties: {
      DisplayName: 'Elias',
      DisplayVoiceName: 'EliasNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Elias',
      LocaleDescription: 'Spanish (Spain)',
      OrderInVoiceListApi: '172',
      PreviewSentence:
        'Crea aplicaciones y servicios que hablan de forma natural a los usuarios, y que mejoran la accesibilidad y la facilidad de uso.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-ES-EliasNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-ES',
      localeZH: '西班牙语(西班牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f5281f53-65e4-4685-bfb7-f8b30372dbf2.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A38Z&se=2122-09-27T07%3A51%3A38Z&sr=b&sp=rl&sig=0xOXVIM%2B0zNsAwaHarfl4qL%2BJF8bG8ZGVDJ6%2BaHvxT4%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-ES-EliasNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '5a8032b2-536d-4341-b828-550ae5946fd9',
    locale: 'pt-PT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pt-PT, RaquelNeural)',
    properties: {
      DisplayName: 'Raquel',
      DisplayVoiceName: 'RaquelNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Raquel',
      LocaleDescription: 'Portuguese (Portugal)',
      OrderInVoiceListApi: '367',
      PreviewSentence:
        'Constrói aplicações e serviços que falam naturalmente com os utilizadores, melhorando a acessibilidade e usabilidade.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pt-PT-RaquelNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'pt-PT',
      localeZH: '葡萄牙语(葡萄牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/93f013e1-64ff-4340-a3b9-1eadd49bd701.wav?sv=2019-07-07&sr=b&sig=tOB09Td8LPk2fTrld5Q7r1AT0p%2B57tBUppP%2BAk5cCUY%3D&st=2021-10-29T08:27:58Z&se=2121-10-05T08:32:58Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'pt-PT-RaquelNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '4a73ab1b-4908-4586-bde6-556699f5e855',
    locale: 'el-GR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (el-GR, AthinaNeural)',
    properties: {
      DisplayName: 'Athina',
      DisplayVoiceName: 'AthinaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Αθηνά',
      LocaleDescription: 'Greek',
      OrderInVoiceListApi: '75',
      PreviewSentence:
        'Δημιουργήστε εφαρμογές και υπηρεσίες που μιλούν με φυσικό τρόπο στους χρήστες, βελτιώνοντας την προσβασιμότητα και τη χρηστικότητα.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'el-GR-AthinaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'el-GR',
      localeZH: '希腊语(希腊)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/178fc88c-94be-4b83-a045-a95af7a5dda7.wav?sv=2019-07-07&sr=b&sig=mC9rVCCNWJGa%2F%2FZt8LtO3aiVb1yRi7EnmzYq0h15XTM%3D&st=2021-10-29T08:26:16Z&se=2121-10-05T08:31:16Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'el-GR-AthinaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '76130879-a05a-442b-8145-55ecbcf934da',
    locale: 'es-GQ',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-GQ, JavierNeural)',
    properties: {
      DisplayName: 'Javier',
      DisplayVoiceName: 'JavierNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Javier',
      LocaleDescription: 'es-GQ',
      OrderInVoiceListApi: '184',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-GQ-JavierNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-GQ',
      localeZH: '西班牙语(赤道几内亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f1035f95-5c57-4cbe-a538-4cf04a030753.wav?sv=2019-07-07&sr=b&sig=n4EaaPCeiegKhne7A7hb9uWMwK%2BiT2On5w8ziNr5OUU%3D&st=2021-10-29T08:30:40Z&se=2121-10-05T08:35:40Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-GQ-JavierNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '4f0f956b-7704-4cfd-b50c-56658f80084e',
    locale: 'is-IS',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (is-IS, GudrunNeural)',
    properties: {
      DisplayName: 'Gudrun',
      DisplayVoiceName: 'GudrunNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Guðrún',
      LocaleDescription: 'is-IS',
      OrderInVoiceListApi: '273',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'is-IS-GudrunNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'is-IS',
      localeZH: '冰岛语(冰岛)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d120d73a-6ba7-4002-ae76-430b0d177741.wav?sv=2019-07-07&sr=b&sig=EHs%2FS5Wf7eGS%2Be9Lgr%2FCetkf7zrL%2BGwVrLYlDFHxfiQ%3D&st=2022-01-19T09:01:44Z&se=2121-12-26T09:06:44Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'is-IS-GudrunNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [
      {
        name: 'All',
      },
      {
        name: 'Audiobook',
      },
      {
        defaultStyle: 'chat',
        name: 'Chat',
      },
    ],
    id: '732028ea-c614-45de-8cad-566e19e94e79',
    locale: 'zh-CN',
    masterpieces: [
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/39a097c2-2820-4cd2-a931-9550f7957cc5.wav?sv=2019-07-07&sr=b&sig=ByKcy9Abn484xoSI7Y9RgDRaYfKAxXKB8BxKVgQx9HU%3D&st=2021-10-29T08:31:52Z&se=2121-10-05T08:36:52Z&sp=rl',
        categories: ['Audiobook'],
        title: '晓双儿童故事展示',
        titleLocalizationId: 'zh-CN-XiaoshuangNeural_Masterpieces_Audiobook_ChildrenStory_Title',
      },
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/953946d7-f809-4668-84b9-d4a46083f78e.wav?sv=2019-07-07&sr=b&sig=13%2BnCwDTMy1PN9pZ3z4PDJE%2BC2nMqD8m3gT%2F814TnGs%3D&st=2021-10-29T08:31:53Z&se=2121-10-05T08:36:53Z&sp=rl',
        categories: ['Chat'],
        title: '晓双AI对话展示',
        titleLocalizationId: 'zh-CN-XiaoshuangNeural_Masterpieces_Chat_AIConversation_Title',
      },
    ],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, XiaoshuangNeural)',
    properties: {
      AgeGroups: 'Child',
      DisplayName: 'Xiaoshuang',
      DisplayVoiceName: 'XiaoshuangNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      IconFileEndpointWithSas:
        'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/648f9926-c7ad-4ae4-88a5-5e58db5bbf82.png?sv=2019-07-07&sr=b&sig=CcfrZ4E3GVhAxEmWq1r5FXscoVs9uqOfyVXszZyAZRw%3D&st=2021-10-29T08:24:34Z&se=2121-10-05T08:29:34Z&sp=rl',
      IsTtsBreakEnabled: 'True',
      LocalName: '晓双',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '431',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-XiaoshuangNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"Default":0}',
      VoiceStyleNames: 'chat,Default',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e9f43422-16c6-4c6f-9613-291574ecc41a.wav?sv=2019-07-07&sr=b&sig=1W%2BMJwCE6jBlbbn8mQADTefEcybLkemp%2FUn10cE7FT4%3D&st=2021-10-29T08:29:45Z&se=2121-10-05T08:34:45Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fb01c67e-b54c-440e-9457-2f972576d2e4.wav?sv=2019-07-07&sr=b&sig=XDp3wTUyLsRXq1qqMgjNi4n7IElv2PkTVJZuUgCLnWA%3D&st=2021-10-29T08:29:46Z&se=2121-10-05T08:34:46Z&sp=rl',
          styleName: 'chat',
        },
      ],
    },
    shortName: 'zh-CN-XiaoshuangNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'dc3d1705-5580-40f3-93a6-5785d1b70861',
    locale: 'kn-IN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (kn-IN, GaganNeural)',
    properties: {
      DisplayName: 'Gagan',
      DisplayVoiceName: 'GaganNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'ಗಗನ್',
      LocaleDescription: 'kn-IN',
      OrderInVoiceListApi: '305',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'kn-IN-GaganNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'kn-IN',
      localeZH: '埃纳德语(印度)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5783bbe7-f61d-4c2b-901c-ee632c2298ed.wav?sv=2019-07-07&sr=b&sig=Rn21duyLZd2isZdNh2pl1sJxzkEpdlXNHbuwGwQEFBM%3D&st=2022-01-19T09:01:53Z&se=2121-12-26T09:06:53Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'kn-IN-GaganNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'bfa6ab71-5463-4a9f-8170-58c219093aab',
    locale: 'fr-FR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-FR, AlainNeural)',
    properties: {
      DisplayName: 'Alain',
      DisplayVoiceName: 'AlainNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Alain',
      LocaleDescription: 'French (France)',
      OrderInVoiceListApi: '241',
      PreviewSentence:
        "Construire des applications et services qui communiquent naturellement avec l'utilisateur, en améliorant l’accessibilité et l'ergonomie",
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-FR-AlainNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fr-FR',
      localeZH: '法语(法国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/338c189c-c746-48ca-afd5-e8c6b83ebdc4.wav?sv=2019-07-07&sr=b&sig=XRfTByipPRCYlkPj5doXt27Ec7kibbp%2FOzb7r38Ws08%3D&st=2022-01-19T09:01:17Z&se=2121-12-26T09:06:17Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fr-FR-AlainNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'ab391d83-4474-4387-9bdf-591524d03635',
    locale: 'de-DE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (de-DE, TanjaNeural)',
    properties: {
      DisplayName: 'Tanja',
      DisplayVoiceName: 'TanjaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Tanja',
      LocaleDescription: 'German (Germany)',
      OrderInVoiceListApi: '73',
      PreviewSentence:
        'Erstellen Sie Apps und Dienste, die auf natürliche Art mit Nutzern sprechen, und dadurch Zugänglichkeit und Benutzerfreundlichkeit verbessern.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'de-DE-TanjaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'de-DE',
      localeZH: '德语(德国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fcf16c22-b294-4297-aac6-6a8faca36271.wav?sv=2019-07-07&sr=b&sig=FTKGfxJI2T%2BBfTLzM50TGuNycBiRVcgqTMW4HYhMH84%3D&st=2022-01-19T09:00:48Z&se=2121-12-26T09:05:48Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'de-DE-TanjaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '474131f3-ea79-45eb-bb61-594966a48110',
    locale: 'ko-KR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ko-KR, JiMinNeural)',
    properties: {
      DisplayName: 'JiMin',
      DisplayVoiceName: 'JiMinNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: '지민',
      LocaleDescription: 'Korean',
      OrderInVoiceListApi: '311',
      PreviewSentence:
        '사용자들에게 자연스럽게 이야기하는 앱과 서비스를 만들며 접근성과 사용성을 개선시킵니다.',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ko-KR-JiMinNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ko-KR',
      localeZH: '韩语(韩国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0c797d82-969b-439e-943c-2f44b54358ba.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A39Z&se=2122-09-27T07%3A52%3A39Z&sr=b&sp=rl&sig=sxpAflw%2FzcIjeTPS87QTIYxpUpbPSzu1itxRkwWQd%2B0%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ko-KR-JiMinNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'e5e4f59b-65c6-42b2-a6e3-5985d1a1ea07',
    locale: 'en-US',
    masterpieces: [
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/87013a89-c507-49bc-b1c6-6075bced1c49.wav?sv=2019-07-07&sr=b&sig=Jc3u2i8gJUgTvdmvpz80NYIwNxklRXNsK9rP4F%2BlOfQ%3D&st=2021-10-29T08:25:45Z&se=2121-10-05T08:30:45Z&sp=rl',
        categories: ['Newscast'],
        title: 'News',
        titleLocalizationId: 'News',
      },
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/87cff8b8-ab91-42d0-a6f0-b004e4bad1ba.wav?sv=2019-07-07&sr=b&sig=boONrOHQTogoL9POz9A8KiwNsX3EpYFBpwTUeR4Mvb8%3D&st=2021-10-29T08:25:44Z&se=2121-10-05T08:30:44Z&sp=rl',
        categories: ['Audiobook'],
        title: 'AudioBook',
        titleLocalizationId: 'AudioBook',
      },
    ],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, JennyNeural)',
    properties: {
      DisplayName: 'Jenny',
      DisplayVoiceName: 'JennyNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Jenny',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '126',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-JennyNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"angry":7,"cheerful":4,"sad":8,"Default":0}',
      VoiceStyleNames:
        'assistant,chat,customerservice,newscast,angry,cheerful,sad,excited,friendly,terrified,shouting,unfriendly,whispering,hopeful,Default',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/7e965392-7eba-43c3-ab90-833ce77cd3cf.wav?sv=2019-07-07&sr=b&sig=lyC2l0jN1nSmwMOycR2wXTkiVzjwcN8Ud5X5Gl5NGWE%3D&st=2021-10-29T08:25:46Z&se=2121-10-05T08:30:46Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/538c6710-88f0-4808-bf98-e1c7b7d4dd6e.wav?sv=2019-07-07&sr=b&sig=UQhwBQ6MT2df386E78ck65m%2BqGEBHEEutda%2Fw%2BtyQU0%3D&st=2021-10-29T08:25:40Z&se=2121-10-05T08:30:40Z&sp=rl',
          styleName: 'assistant',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2125edde-b7f2-4e0f-9f04-ebb28c45f8a6.wav?sv=2019-07-07&sr=b&sig=1wolFFbwSxYkwZI3hNLLeo1NXFJj89gXxOiViBB9MUI%3D&st=2021-10-29T08:25:42Z&se=2121-10-05T08:30:42Z&sp=rl',
          styleName: 'chat',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4bec52f6-725d-4a2a-b25f-8f232a8cd540.wav?sv=2019-07-07&sr=b&sig=AtWvxjDNZv2YrcCHYR63gX0dLlCN1yHITG%2BvQYNEMQI%3D&st=2021-10-29T08:25:43Z&se=2121-10-05T08:30:43Z&sp=rl',
          styleName: 'customerservice',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/78f395f6-4634-4c83-bf74-372ec3c55795.wav?sv=2019-07-07&sr=b&sig=Z5N3I3P9DdqPP%2F0nwgOhLTTI9juaxSIs0F9KYOuoX0c%3D&st=2021-10-29T08:25:48Z&se=2121-10-05T08:30:48Z&sp=rl',
          styleName: 'newscast',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/22a7100a-ac65-4aa9-9ac9-3072f60e87ca.wav?sv=2019-07-07&sr=b&sig=Gy2StMBkS6HKT0UdBHPCbj8qi2Oic8v49Wc%2Bh0unfEg%3D&st=2022-04-28T10:26:18Z&se=2122-04-04T10:31:18Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/56039fa0-2d92-4fd2-a8fe-631c2ed578ce.wav?sv=2019-07-07&sr=b&sig=zoTHGHpROIxteFYDHJ6pGIIaeng1QKj%2FNNVoJWzeO5Y%3D&st=2022-04-28T10:26:20Z&se=2122-04-04T10:31:20Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/23f6061b-2076-4052-bbce-8cee689c2bd9.wav?sv=2019-07-07&sr=b&sig=9QhVOSBMWRtUw%2F9xkrX9gtcRw6rkIdwwVFVH14Pf8PY%3D&st=2022-04-28T10:26:23Z&se=2122-04-04T10:31:23Z&sp=rl',
          styleName: 'excited',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/bec707b0-decf-46b7-ba51-71d3f280540d.wav?sv=2019-07-07&sr=b&sig=EiYzAGx3q4wp1zCWDgsj7vFHg9qukJf%2B4tUtW5WlF9U%3D&st=2022-04-28T10:26:25Z&se=2122-04-04T10:31:25Z&sp=rl',
          styleName: 'friendly',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/8836fd85-0877-490f-bff4-172da15a5a78.wav?sv=2019-07-07&sr=b&sig=rn6QpzXMGGXT%2FxNDkVEk0N838FxzJ9JYRVMv3L1n4xQ%3D&st=2022-04-28T10:26:27Z&se=2122-04-04T10:31:27Z&sp=rl',
          styleName: 'hopeful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/263a38d0-174f-4101-8a49-b87a5d7ac4e2.wav?sv=2019-07-07&sr=b&sig=nPLZhA9M%2FHkw8v4YfU6Uno8G2To%2Fx6HzS2x2HaUL%2Bs0%3D&st=2022-04-28T10:26:30Z&se=2122-04-04T10:31:30Z&sp=rl',
          styleName: 'sad',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5f591369-3441-4803-adbc-377b24d8b3eb.wav?sv=2019-07-07&sr=b&sig=g1ytYTPYklPC1S4b3MMXxdr280W90b7O0Cjy1Xn11fM%3D&st=2022-04-28T10:26:32Z&se=2122-04-04T10:31:32Z&sp=rl',
          styleName: 'shouting',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/79b3edb5-a764-4e40-804d-370464659857.wav?sv=2019-07-07&sr=b&sig=bH4rE7zATQVPH0suVg7uuN3MWaFKSjpHZgtM8A%2BvJ10%3D&st=2022-04-28T10:26:34Z&se=2122-04-04T10:31:34Z&sp=rl',
          styleName: 'terrified',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e19642f8-cf41-471a-b06a-3c426ce4688b.wav?sv=2019-07-07&sr=b&sig=B%2ByAPKrl2tCB%2B06hEZ9350D8PQ1TlJhE%2FruPyhM6BgQ%3D&st=2022-04-28T10:26:37Z&se=2122-04-04T10:31:37Z&sp=rl',
          styleName: 'unfriendly',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2ad03235-2896-40a5-ae81-9def57fbef16.wav?sv=2019-07-07&sr=b&sig=sys16iZeJi8d6TNaCKbDUkoUYfYdAxPfoAUiyUNIZIY%3D&st=2022-04-28T10:26:39Z&se=2122-04-04T10:31:39Z&sp=rl',
          styleName: 'whispering',
        },
      ],
    },
    shortName: 'en-US-JennyNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '05750e25-e868-45c4-9ae7-5a137a4cdf19',
    locale: 'es-UY',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-UY, ValentinaNeural)',
    properties: {
      DisplayName: 'Valentina',
      DisplayVoiceName: 'ValentinaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Valentina',
      LocaleDescription: 'es-UY',
      OrderInVoiceListApi: '220',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-UY-ValentinaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-UY',
      localeZH: '西班牙语(乌拉圭)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a2c2d068-8af9-4409-89c5-96eb23e05961.wav?sv=2019-07-07&sr=b&sig=VN8Sed9WNaUz0g3p3OayHd1C5CdI8Gfk1SVESzzvShk%3D&st=2021-10-29T08:30:59Z&se=2121-10-05T08:35:59Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-UY-ValentinaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '9ebb7b98-4fc8-4093-9e92-5a567ede03d8',
    locale: 'de-DE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (de-DE, KlarissaNeural)',
    properties: {
      DisplayName: 'Klarissa',
      DisplayVoiceName: 'KlarissaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Klarissa',
      LocaleDescription: 'German (Germany)',
      OrderInVoiceListApi: '68',
      PreviewSentence:
        'Erstellen Sie Apps und Dienste, die auf natürliche Art mit Nutzern sprechen, und dadurch Zugänglichkeit und Benutzerfreundlichkeit verbessern.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'de-DE-KlarissaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'de-DE',
      localeZH: '德语(德国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5950e489-0c2c-4348-8846-bc868753a51a.wav?sv=2019-07-07&sr=b&sig=TNxfER8IWCdlZkoJlxIRx1KnrFh1qPN8oijObKYNfRk%3D&st=2022-01-19T09:00:36Z&se=2121-12-26T09:05:36Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'de-DE-KlarissaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '66becf5d-1fc6-488f-9f71-5a94bca91880',
    locale: 'de-CH',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (de-CH, LeniNeural)',
    properties: {
      DisplayName: 'Leni',
      DisplayVoiceName: 'LeniNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Leni',
      LocaleDescription: 'German (Switzerland)',
      OrderInVoiceListApi: '59',
      PreviewSentence:
        'Erstellen Sie Apps und Dienste, die auf natürliche Art mit Nutzern sprechen, und dadurch Zugänglichkeit und Benutzerfreundlichkeit verbessern.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'de-CH-LeniNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'de-CH',
      localeZH: '德语(瑞士)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/00e1d1ff-c195-4d6d-929b-8a4edf8b107c.wav?sv=2019-07-07&sr=b&sig=ZuHcbosr6%2FNAD3OGr0nWzxblDODMCWnInjjzkHYkZzg%3D&st=2021-10-29T08:26:13Z&se=2121-10-05T08:31:13Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'de-CH-LeniNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'd26c4ed1-a6f9-4a08-af27-5aae3df3a890',
    locale: 'ar-DZ',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-DZ, IsmaelNeural)',
    properties: {
      DisplayName: 'Ismael',
      DisplayVoiceName: 'IsmaelNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'إسماعيل',
      LocaleDescription: 'ar-DZ',
      OrderInVoiceListApi: '10',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-DZ-IsmaelNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-DZ',
      localeZH: '阿拉伯语(阿尔及利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6995ee7b-07ce-47a5-8938-84bfb9a5c309.wav?sv=2019-07-07&sr=b&sig=HeL%2FkHhh00JeLmguco9hSQS3qWaNZSRwWF6RnKepTPE%3D&st=2021-10-29T08:30:02Z&se=2121-10-05T08:35:02Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-DZ-IsmaelNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '2b9002ee-95ba-419f-b983-5b14ece687a6',
    locale: 'nb-NO',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (nb-NO, FinnNeural)',
    properties: {
      DisplayName: 'Finn',
      DisplayVoiceName: 'FinnNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Finn',
      LocaleDescription: 'Norwegian',
      OrderInVoiceListApi: '336',
      PreviewSentence:
        'Bygger apper og tjenester som snakker naturlig med brukerne, utbedrer tilgjengelighet og brukskvalitet.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'nb-NO-FinnNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'nb-NO',
      localeZH: '书面挪威语(挪威)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2debd804-4e06-4c1a-bfff-792c2e7036a7.wav?sv=2019-07-07&sr=b&sig=YUhKlyMVZWo0RA%2B67bIqFpp2b2K%2FULR%2F9Qg74TV1Lu8%3D&st=2021-10-29T08:27:37Z&se=2121-10-05T08:32:37Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'nb-NO-FinnNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '10044c16-554d-4691-9c89-5c13e892ceb6',
    locale: 'de-DE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (de-DE, BerndNeural)',
    properties: {
      DisplayName: 'Bernd',
      DisplayVoiceName: 'BerndNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Bernd',
      LocaleDescription: 'German (Germany)',
      OrderInVoiceListApi: '62',
      PreviewSentence:
        'Erstellen Sie Apps und Dienste, die auf natürliche Art mit Nutzern sprechen, und dadurch Zugänglichkeit und Benutzerfreundlichkeit verbessern.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'de-DE-BerndNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'de-DE',
      localeZH: '德语(德国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fd1b6f2a-72cb-444a-af22-d4c203409eeb.wav?sv=2019-07-07&sr=b&sig=6x3Xg0XjgpLAq3on8ield5j8e4Ngs0QM3lpDO7gv2aM%3D&st=2022-01-19T09:00:22Z&se=2121-12-26T09:05:22Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'de-DE-BerndNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '33b22076-a1b7-459a-b9db-5c507945176c',
    locale: 'fi-FI',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fi-FI, NooraNeural)',
    properties: {
      DisplayName: 'Noora',
      DisplayVoiceName: 'NooraNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Noora',
      LocaleDescription: 'Finnish',
      OrderInVoiceListApi: '231',
      PreviewSentence:
        'Kehitä luonnolisesti puhuvia sovelluksia ja palveluja, jotka parantavat käytettävyyttä ja saavutettavuutta.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fi-FI-NooraNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fi-FI',
      localeZH: '芬兰语(芬兰)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d2a2865b-a0ce-4212-bd2b-34ebc08635bc.wav?sv=2019-07-07&sr=b&sig=gklgia2ezyW6od%2BGG4JLBf84HTiJNzCHTB7USN0eh%2BU%3D&st=2021-10-29T08:26:52Z&se=2121-10-05T08:31:52Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fi-FI-NooraNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'e7ae8dcc-bee4-4545-8532-5c8710041809',
    locale: 'zh-CN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, XiaomengNeural)',
    properties: {
      DisplayName: 'Xiaomeng',
      DisplayVoiceName: 'XiaomengNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      IsTtsBreakEnabled: 'True',
      LocalName: '晓梦',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '427',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-XiaomengNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"Default":0}',
      VoiceStyleNames: 'chat,Default',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/bc931737-94cc-4142-b043-011f1edfd963.wav?sv=2019-07-07&sr=b&sig=eVZ9P6sCMIzwFVkOiEaDo0rqsvR0zCLW%2FIYAAonBEn4%3D&st=2022-08-01T10:39:22Z&se=2122-07-08T10:44:22Z&sp=rl',
          styleName: 'chat',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f0c63d06-0d14-49e9-861e-bd86e0bff368.wav?sv=2019-07-07&sr=b&sig=%2FDG5ARzGSsyWxDlrElEK0SwWzJo%2FT6x72HuXJxgxd4w%3D&st=2022-08-01T10:39:25Z&se=2122-07-08T10:44:25Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'zh-CN-XiaomengNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '2fd0c0e4-8885-49d0-80b8-5ca9929e8716',
    locale: 'en-AU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-AU, CarlyNeural)',
    properties: {
      DisplayName: 'Carly',
      DisplayVoiceName: 'CarlyNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Carly',
      LocaleDescription: 'English (Australia)',
      OrderInVoiceListApi: '78',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-AU-CarlyNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-AU',
      localeZH: '英语(澳大利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/39b66285-bb6f-47a9-b762-b8f1f4765da6.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A00Z&se=2122-09-27T07%3A51%3A00Z&sr=b&sp=rl&sig=VcNp3II109rgabkrRSvhloN8t4Mme9G25Nl97OiTXr0%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-AU-CarlyNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '99893827-8784-4146-8bec-5cd7f4b296ed',
    locale: 'ar-OM',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-OM, AyshaNeural)',
    properties: {
      DisplayName: 'Aysha',
      DisplayVoiceName: 'AyshaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'عائشة',
      LocaleDescription: 'ar-OM',
      OrderInVoiceListApi: '26',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-OM-AyshaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-OM',
      localeZH: '阿拉伯语(阿曼)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e3ab3db1-e490-48e8-b05b-d4f01cf25bb0.wav?sv=2019-07-07&sr=b&sig=o3KjmdkOeeX27ePCOXVNn3T3pMrSifJ8vlgjgadkwt4%3D&st=2022-07-01T09:08:49Z&se=2122-06-07T09:13:49Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-OM-AyshaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'be77cf10-8c9f-4916-b352-5da177804842',
    locale: 'de-DE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (de-DE, RalfNeural)',
    properties: {
      DisplayName: 'Ralf',
      DisplayVoiceName: 'RalfNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Ralf',
      LocaleDescription: 'German (Germany)',
      OrderInVoiceListApi: '72',
      PreviewSentence:
        'Erstellen Sie Apps und Dienste, die auf natürliche Art mit Nutzern sprechen, und dadurch Zugänglichkeit und Benutzerfreundlichkeit verbessern.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'de-DE-RalfNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'de-DE',
      localeZH: '德语(德国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/bbf5cd72-930f-4522-a6cd-d09ffd0f340b.wav?sv=2019-07-07&sr=b&sig=7H4i56mUF5uKXAZrNpEnrlQYF67sMGeMbJZjT%2FrSdlw%3D&st=2022-01-19T09:00:46Z&se=2121-12-26T09:05:46Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'de-DE-RalfNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '54bdad96-b5f6-4243-9705-5e0db3f67dcd',
    locale: 'zh-CN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, YunfengNeural)',
    properties: {
      DisplayName: 'Yunfeng',
      DisplayVoiceName: 'YunfengNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      IsTtsBreakEnabled: 'True',
      LocalName: '云枫',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '437',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-YunfengNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions:
        '{"angry":7,"disgruntled":5,"cheerful":4,"fearful":3,"sad":8,"serious":6,"depressed":10,"Default":0}',
      VoiceStyleNames: 'angry,disgruntled,cheerful,fearful,sad,serious,depressed,Default',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2468d6bd-ee27-4198-b78f-1c0b3c019af6.wav?sv=2019-07-07&sr=b&sig=5Q7O5bY37LcEDxJlPpmZpKghkQ3Vhe4ZhTQanhba9t0%3D&st=2022-04-28T07:18:29Z&se=2122-04-04T07:23:29Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/447d30f0-b934-4f35-8b47-0cdd61d6c30b.wav?sv=2019-07-07&sr=b&sig=ClfC8EA4d0DTgj5BmgfHGuvFtYae7qjWxrfwh9qJU2o%3D&st=2022-04-28T07:18:21Z&se=2122-04-04T07:23:21Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2e9d6bd7-46cc-49c8-ab89-4408ec356be6.wav?sv=2019-07-07&sr=b&sig=3IK0tog9gZc4oNYP6qJJ8J2QReWGFU19QLwkj89O%2FoA%3D&st=2022-04-28T07:18:23Z&se=2122-04-04T07:23:23Z&sp=rl',
          styleName: 'disgruntled',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2c5ff7cd-433e-4439-b60f-d2e795c490bc.wav?sv=2019-07-07&sr=b&sig=tYF33pE9zWxXqf2i4TRkcb%2BPfKFsPpwSWgKC0X34m7o%3D&st=2022-04-28T07:18:32Z&se=2122-04-04T07:23:32Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/99f38db2-2cdb-4ff3-b78c-acdb1414bb20.wav?sv=2019-07-07&sr=b&sig=dtIjh1zgOzMMGmppt6L5wQ4d8irp2wVlVFg3g3yUdzM%3D&st=2022-04-28T07:18:35Z&se=2122-04-04T07:23:35Z&sp=rl',
          styleName: 'fearful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4f018540-8433-47a2-b643-892005c544ac.wav?sv=2019-07-07&sr=b&sig=qs0PqMs5qrKaC1flYJ1h1HXd9iikiC8sS%2BulHTt%2Ba%2FQ%3D&st=2022-04-28T07:18:38Z&se=2122-04-04T07:23:38Z&sp=rl',
          styleName: 'sad',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ef2429ce-6be4-43c5-b756-719864ee50d4.wav?sv=2019-07-07&sr=b&sig=wUmeGY9VrTv16CHm40DY8dz5pvdV8o2yCeutZC%2B1NDI%3D&st=2022-04-28T07:18:41Z&se=2122-04-04T07:23:41Z&sp=rl',
          styleName: 'serious',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/7214ca67-11a7-43c1-9cfa-435a5912dd15.wav?sv=2019-07-07&sr=b&sig=E1pK1of7%2Fh1JrxZJotQUTfAT0d6zYfk9DZqKjwPclGg%3D&st=2022-04-28T07:18:26Z&se=2122-04-04T07:23:26Z&sp=rl',
          styleName: 'depressed',
        },
      ],
    },
    shortName: 'zh-CN-YunfengNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '307049c0-bfaf-40dd-8b97-5f69465d0e48',
    locale: 'si-LK',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (si-LK, ThiliniNeural)',
    properties: {
      DisplayName: 'Thilini',
      DisplayVoiceName: 'ThiliniNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'තිළිණි',
      LocaleDescription: 'si-LK',
      OrderInVoiceListApi: '374',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'si-LK-ThiliniNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'si-LK',
      localeZH: '僧伽罗语(斯里兰卡)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/9d7e5f77-d597-4d09-b210-c4238a35405a.wav?sv=2019-07-07&sr=b&sig=9LdQLm7NwMPrkO4GfOFpJEfoPtHsqLK87LXYbgEM00c%3D&st=2022-01-19T09:02:18Z&se=2121-12-26T09:07:18Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'si-LK-ThiliniNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '329c1743-53d2-442e-b4ef-5fcb260105fa',
    locale: 'fr-FR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-FR, ClaudeNeural)',
    properties: {
      DisplayName: 'Claude',
      DisplayVoiceName: 'ClaudeNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Claude',
      LocaleDescription: 'French (France)',
      OrderInVoiceListApi: '244',
      PreviewSentence:
        "Construire des applications et services qui communiquent naturellement avec l'utilisateur, en améliorant l’accessibilité et l'ergonomie",
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-FR-ClaudeNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fr-FR',
      localeZH: '法语(法国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e7a547f8-06a3-4a3a-9058-9d56731e0c9f.wav?sv=2019-07-07&sr=b&sig=yXJWryUom4OoVLpEH3oa%2BRSBvPTsLTH0VEFrs7TEJaM%3D&st=2022-01-19T09:01:24Z&se=2121-12-26T09:06:24Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fr-FR-ClaudeNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '09d9fda8-5de9-4e7a-be8b-607a82f767d1',
    locale: 'ne-NP',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ne-NP, SagarNeural)',
    properties: {
      DisplayName: 'Sagar',
      DisplayVoiceName: 'SagarNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'सागर',
      LocaleDescription: 'ne-NP',
      OrderInVoiceListApi: '339',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ne-NP-SagarNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ne-NP',
      localeZH: '尼泊尔语(尼泊尔)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/dfe7229a-8cad-4a2f-b910-55387eaa9284.wav?sv=2019-07-07&sr=b&sig=zA0fRchR08VJZe08qrEiB5WtNX%2F%2FcZHgOAcC4prEPKo%3D&st=2022-07-01T09:10:09Z&se=2122-06-07T09:15:09Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ne-NP-SagarNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '3bb57996-137f-460b-b245-61b4a441899f',
    locale: 'pt-BR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pt-BR, ValerioNeural)',
    properties: {
      DisplayName: 'Valerio',
      DisplayVoiceName: 'ValerioNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Valerio',
      LocaleDescription: 'Portuguese (Brazil)',
      OrderInVoiceListApi: '363',
      PreviewSentence:
        'Desenvolver aplicativos e serviços que conversem naturalmente com os usuários, melhorando a acessibilidade e usabilidade.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pt-BR-ValerioNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'pt-BR',
      localeZH: '葡萄牙语(巴西)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4e74721e-79e2-47cc-ba1c-9c0b0b3a1af7.wav?sv=2019-07-07&sr=b&sig=SYgs%2Biwp8U5x6f96zbGuY%2F6RbPzcVEy5%2B8GOl0pTKVI%3D&st=2022-07-01T09:10:36Z&se=2122-06-07T09:15:36Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'pt-BR-ValerioNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'f9fd897f-780d-4b7f-a8a2-628770eb3218',
    locale: 'ko-KR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ko-KR, YuJinNeural)',
    properties: {
      DisplayName: 'YuJin',
      DisplayVoiceName: 'YuJinNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: '유진',
      LocaleDescription: 'Korean',
      OrderInVoiceListApi: '314',
      PreviewSentence:
        '사용자들에게 자연스럽게 이야기하는 앱과 서비스를 만들며 접근성과 사용성을 개선시킵니다.',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ko-KR-YuJinNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ko-KR',
      localeZH: '韩语(韩国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/184afedf-101d-40e0-ad0e-07bd10f8aeb7.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A45Z&se=2122-09-27T07%3A52%3A45Z&sr=b&sp=rl&sig=Vc09h4te2JlKIVoC0%2FFbGoaJpyv6GIUIvYD6JwwGYAs%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ko-KR-YuJinNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '96d631b1-d438-48ba-aa46-6293707d2071',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, TonyNeural)',
    properties: {
      DisplayName: 'Tony',
      DisplayVoiceName: 'TonyNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Tony',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '148',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-TonyNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"angry":7,"cheerful":4,"sad":8,"Default":0}',
      VoiceStyleNames:
        'angry,cheerful,excited,friendly,hopeful,sad,shouting,terrified,unfriendly,whispering,Default',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/74d34bd9-b9df-46c2-8ade-ceaf6c9acc52.wav?sv=2019-07-07&sr=b&sig=8Vf6NOY%2BDsmj8AIitFNqfMIIiwf6IdLM90XLkZ6ZINI%3D&st=2022-05-05T09:39:10Z&se=2122-04-11T09:44:10Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c6a964ca-69fd-4762-b624-8b2bdaba9780.wav?sv=2019-07-07&sr=b&sig=MrrhONFb9%2FzajyPzBCLM0tltHx7trTg1EbkIDvSuouw%3D&st=2022-04-28T10:28:32Z&se=2122-04-04T10:33:32Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f11c0250-4ba3-4a7e-9159-d2336b52e39b.wav?sv=2019-07-07&sr=b&sig=zJU7akD3WhjoMcT2KTY9dtFD5%2Frm04PRI9Fpx0jpoKM%3D&st=2022-04-28T10:28:34Z&se=2122-04-04T10:33:34Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/275fd146-0946-49ca-bdb1-40df222c418c.wav?sv=2019-07-07&sr=b&sig=xHJZIRfRc%2BmfSrvp1j%2BOzcqjb1QYeOvPRnbTP9Iq9C0%3D&st=2022-04-28T10:28:37Z&se=2122-04-04T10:33:37Z&sp=rl',
          styleName: 'excited',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ae5e502d-c6fa-4f23-891c-9c49674376ec.wav?sv=2019-07-07&sr=b&sig=gRz3LSlxyboFiK24Cm6IkNBYxLGvZ%2FGj92CyI690t4Y%3D&st=2022-04-28T10:28:39Z&se=2122-04-04T10:33:39Z&sp=rl',
          styleName: 'friendly',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e4956ea0-21f7-47d8-85d2-a8a387c92155.wav?sv=2019-07-07&sr=b&sig=D039LV9Zn0C%2BeARMaVp5beE1uJoEWWDEVuh6OXePGYY%3D&st=2022-04-28T10:28:41Z&se=2122-04-04T10:33:41Z&sp=rl',
          styleName: 'hopeful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a4968845-d471-42cd-b451-416f039b6825.wav?sv=2019-07-07&sr=b&sig=P05n9gBSpCNQNe1ygYKlF1fuFvouTydGqblUel%2Fm854%3D&st=2022-04-28T10:28:44Z&se=2122-04-04T10:33:44Z&sp=rl',
          styleName: 'sad',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/bae8f254-0cd9-4730-a95d-5d15f9bf0f7d.wav?sv=2019-07-07&sr=b&sig=jblJwZbD1tB9nQ61MLWibTlWSeRwJkPAdxtiZEyUEUA%3D&st=2022-04-28T10:28:46Z&se=2122-04-04T10:33:46Z&sp=rl',
          styleName: 'shouting',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/64ed2561-db0e-4162-ab95-fbda5be7c8fa.wav?sv=2019-07-07&sr=b&sig=lV3Ha%2Bm%2Fxryau29ciDpoEt4SEbaP08jXeI7siYe69Dg%3D&st=2022-04-28T10:28:48Z&se=2122-04-04T10:33:48Z&sp=rl',
          styleName: 'terrified',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b240152e-6424-45fd-b9ae-63f489b02ff2.wav?sv=2019-07-07&sr=b&sig=iUK%2FLeVYywCFkDGRH5VwCnzhJGtyp1v0h5W6qD%2FGpI4%3D&st=2022-04-28T10:28:50Z&se=2122-04-04T10:33:50Z&sp=rl',
          styleName: 'unfriendly',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/cfa873a8-889c-4dca-b858-10101e300785.wav?sv=2019-07-07&sr=b&sig=xTTYfQDfBSxQqDeFChcK5jhxNwLrZOufDsbO1PaHSjo%3D&st=2022-04-28T10:28:53Z&se=2122-04-04T10:33:53Z&sp=rl',
          styleName: 'whispering',
        },
      ],
    },
    shortName: 'en-US-TonyNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '284ad774-159e-4a73-a997-62d7fdd72335',
    locale: 'fr-FR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-FR, MauriceNeural)',
    properties: {
      DisplayName: 'Maurice',
      DisplayVoiceName: 'MauriceNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Maurice',
      LocaleDescription: 'French (France)',
      OrderInVoiceListApi: '251',
      PreviewSentence:
        "Construire des applications et services qui communiquent naturellement avec l'utilisateur, en améliorant l’accessibilité et l'ergonomie",
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-FR-MauriceNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fr-FR',
      localeZH: '法语(法国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2b075c1c-3048-4f8a-8398-f6c906a635bb.wav?sv=2019-07-07&sr=b&sig=k0kW2Vyj4qIdlhDQWAXOCf9saS38rJIrxidcn65kmhA%3D&st=2022-01-19T09:01:37Z&se=2121-12-26T09:06:37Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fr-FR-MauriceNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '26eddc58-d6fa-4ddf-9550-634340d7b656',
    locale: 'te-IN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (te-IN, ShrutiNeural)',
    properties: {
      DisplayName: 'Shruti',
      DisplayVoiceName: 'ShrutiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'శ్రుతి',
      LocaleDescription: 'Telugu (India)',
      OrderInVoiceListApi: '403',
      PreviewSentence:
        'వినియోగం మరియు సౌలభ్యాన్ని మెరుగుపరిచే, సహజ శైలిలో మాట్లాడే అప్లికేషన్లు మరియు సేవలను అభివృద్ధి చేయండి.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'te-IN-ShrutiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'te-IN',
      localeZH: '泰卢固语(印度)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/34078a60-81b4-47c4-8e3e-4c967ec805e5.wav?sv=2019-07-07&sr=b&sig=Hm%2FXRfYFUl87zgSsVyhAUw%2FnMLOqXOjAQoCNa9DghiY%3D&st=2021-10-29T08:28:18Z&se=2121-10-05T08:33:18Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'te-IN-ShrutiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'b7b0ee4d-4c7c-4156-9553-635c14ea3dc2',
    locale: 'ar-YE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-YE, SalehNeural)',
    properties: {
      DisplayName: 'Saleh',
      DisplayVoiceName: 'SalehNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'صالح',
      LocaleDescription: 'ar-YE',
      OrderInVoiceListApi: '36',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-YE-SalehNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-YE',
      localeZH: '阿拉伯语(也门)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/7f8885ee-ce37-4ce5-b3f5-ee1df4fbfe50.wav?sv=2019-07-07&sr=b&sig=Ozyc7C6FbWCTVZU%2FJf%2FwzB%2FNuteqENeVMf7vzZL58kY%3D&st=2021-10-29T08:30:20Z&se=2121-10-05T08:35:20Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-YE-SalehNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '588abaad-6cb0-4082-856c-64a440de1bfc',
    locale: 'es-MX',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-MX, LibertoNeural)',
    properties: {
      DisplayName: 'Liberto',
      DisplayVoiceName: 'LibertoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Liberto',
      LocaleDescription: 'Spanish (Mexico)',
      OrderInVoiceListApi: '198',
      PreviewSentence:
        'Crea apps y servicios que hablen de forma natural con los usuarios, mejorando la accesibilidad y la usabilidad.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-MX-LibertoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-MX',
      localeZH: '西班牙语(墨西哥)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5b382935-41a8-41ed-bc70-10fcff5a4c94.wav?sv=2019-07-07&sr=b&sig=ze35u4ZwV5PpGXOpJC6x9lbUZHlLP6tGoaPPIOYgHss%3D&st=2022-07-01T09:09:15Z&se=2122-06-07T09:14:15Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-MX-LibertoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'c40aaac4-5e0f-4341-bc48-64c0b8ed64ff',
    locale: 'is-IS',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (is-IS, GunnarNeural)',
    properties: {
      DisplayName: 'Gunnar',
      DisplayVoiceName: 'GunnarNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Gunnar',
      LocaleDescription: 'is-IS',
      OrderInVoiceListApi: '274',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'is-IS-GunnarNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'is-IS',
      localeZH: '冰岛语(冰岛)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b44042cc-3540-4a17-a087-893ff294838a.wav?sv=2019-07-07&sr=b&sig=ZQD6F%2FrH8mqtKFUXOUMoKF7rBsPHikmtDffVA6mfRmI%3D&st=2022-01-19T09:01:46Z&se=2121-12-26T09:06:46Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'is-IS-GunnarNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'f55a63a0-99f3-47e0-b8b3-650a5c742cde',
    locale: 'cs-CZ',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (cs-CZ, AntoninNeural)',
    properties: {
      DisplayName: 'Antonin',
      DisplayVoiceName: 'AntoninNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Antonín',
      LocaleDescription: 'Czech',
      OrderInVoiceListApi: '50',
      PreviewSentence:
        'Vytvořte aplikace a služby pro přirozenou komunikaci s uživateli a usnadněte tak přístup a využití.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'cs-CZ-AntoninNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'cs-CZ',
      localeZH: '捷克语(捷克)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5be94399-1c32-4115-a9e8-b06a40beec9e.wav?sv=2019-07-07&sr=b&sig=J2LVg4b18Frnv%2BueDGxwAXb%2FGqjDHgnPxtZjL741cyc%3D&st=2021-10-29T08:26:04Z&se=2121-10-05T08:31:04Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'cs-CZ-AntoninNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '8fb26b62-b1a6-496c-81da-650a7bbfb3dd',
    locale: 'ar-IQ',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-IQ, BasselNeural)',
    properties: {
      DisplayName: 'Bassel',
      DisplayVoiceName: 'BasselNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'باسل',
      LocaleDescription: 'ar-IQ',
      OrderInVoiceListApi: '13',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-IQ-BasselNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-IQ',
      localeZH: '阿拉伯语(伊拉克)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f774ca2d-4652-4cc1-87da-8c15bd921498.wav?sv=2019-07-07&sr=b&sig=rez1e3s4oF%2FeNqzYpLcmw8GXYBaDQaOngGKO%2FWigWPg%3D&st=2021-10-29T08:30:03Z&se=2121-10-05T08:35:03Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-IQ-BasselNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'a67686f9-9d9d-43e3-8035-6528063fcfd3',
    locale: 'mt-MT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (mt-MT, JosephNeural)',
    properties: {
      DisplayName: 'Joseph',
      DisplayVoiceName: 'JosephNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Joseph',
      LocaleDescription: 'Maltese (Malta)',
      OrderInVoiceListApi: '332',
      PreviewSentence:
        'L-Għodda għall-Ħolqien tal-Kontenut bil-Ħoss tħallik direttament tikkontrolla l-attributi tal-leħen minn fuq l-iskrin tiegħek.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'mt-MT-JosephNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'mt-MT',
      localeZH: '马耳他语(马耳他)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a46a9bda-e2c6-4862-a1ea-38c8c47e78a4.wav?sv=2019-07-07&sr=b&sig=EcReh6YvoeWHFMXBvUJlOB07C5nH2Ig%2BHIWuIjrjWOY%3D&st=2021-10-29T08:27:36Z&se=2121-10-05T08:32:36Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'mt-MT-JosephNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '3ae499e9-42b2-4a3e-999a-6549fd4d3d57',
    locale: 'es-MX',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-MX, NuriaNeural)',
    properties: {
      DisplayName: 'Nuria',
      DisplayVoiceName: 'NuriaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Nuria',
      LocaleDescription: 'Spanish (Mexico)',
      OrderInVoiceListApi: '201',
      PreviewSentence:
        'Crea apps y servicios que hablen de forma natural con los usuarios, mejorando la accesibilidad y la usabilidad.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-MX-NuriaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-MX',
      localeZH: '西班牙语(墨西哥)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/36e90f11-f073-400b-bdd3-adf36f06d8ae.wav?sv=2019-07-07&sr=b&sig=Vd1boojtXTKKMCRkhkSDElNaW9qDjCX9FroHi0UGlDo%3D&st=2022-07-01T09:09:21Z&se=2122-06-07T09:14:21Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-MX-NuriaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '5d4bc403-93ef-49c3-a2d1-6555b9bcf591',
    locale: 'km-KH',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (km-KH, PisethNeural)',
    properties: {
      DisplayName: 'Piseth',
      DisplayVoiceName: 'PisethNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'ពិសិដ្ឋ',
      LocaleDescription: 'km-KH',
      OrderInVoiceListApi: '303',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'km-KH-PisethNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'km-KH',
      localeZH: '高棉语(柬埔寨)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f8dfb1fb-b58b-4841-b5c7-779653627b51.wav?sv=2019-07-07&sr=b&sig=tVkMtNpRGn0TcVfdrINR8ZfppfnAWGrdh2xFH60WrOI%3D&st=2021-10-29T08:31:09Z&se=2121-10-05T08:36:09Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'km-KH-PisethNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '77c6b4b5-3b05-4eb6-a44e-65a20c3a5d37',
    locale: 'th-TH',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (th-TH, NiwatNeural)',
    properties: {
      DisplayName: 'Niwat',
      DisplayVoiceName: 'NiwatNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'นิวัฒน์',
      LocaleDescription: 'Thai',
      OrderInVoiceListApi: '406',
      PreviewSentence:
        'สร้างแอปและบริการที่สื่อสารกับผู้ใช้ได้อย่างเป็นธรรมชาติ ซึ่งช่วยปรับปรุงการเข้าถึงและการใช้งาน',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'th-TH-NiwatNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'th-TH',
      localeZH: '泰语(泰国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/7a4bbd9c-b247-43da-aac5-2a9c0dc7bc45.wav?sv=2019-07-07&sr=b&sig=TJe6%2Fs0lHnHv7OO66pgigFCdnaEkMLTEepsulAVMU4w%3D&st=2021-10-29T08:28:21Z&se=2121-10-05T08:33:21Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'th-TH-NiwatNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '56d154ac-b59c-414d-8b7b-6639435999e9',
    locale: 'eu-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (eu-ES, AinhoaNeural)',
    properties: {
      DisplayName: 'Ainhoa',
      DisplayVoiceName: 'AinhoaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Ainhoa',
      LocaleDescription: 'eu-ES',
      OrderInVoiceListApi: '225',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'eu-ES-AinhoaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'eu-ES',
      localeZH: '巴斯克语(巴斯克语)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/42a7a439-9067-4b10-8f7e-59afe6fe5536.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A05Z&se=2122-09-27T07%3A52%3A05Z&sr=b&sp=rl&sig=%2BP5KSSz0UYxcmfCOlspUqpTNfor7B9p8lkdYG3RR%2BVw%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'eu-ES-AinhoaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'bc76601a-7ffa-48ba-bc8d-66db94d3910c',
    locale: 'de-DE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (de-DE, KlausNeural)',
    properties: {
      DisplayName: 'Klaus',
      DisplayVoiceName: 'KlausNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Klaus',
      LocaleDescription: 'German (Germany)',
      OrderInVoiceListApi: '69',
      PreviewSentence:
        'Erstellen Sie Apps und Dienste, die auf natürliche Art mit Nutzern sprechen, und dadurch Zugänglichkeit und Benutzerfreundlichkeit verbessern.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'de-DE-KlausNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'de-DE',
      localeZH: '德语(德国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/7120347f-7fe7-4a3a-8ff2-aad9492be7f1.wav?sv=2019-07-07&sr=b&sig=%2Bwf6GGGJlLWAKnp9F6ybmp2PlCblLCOHXWg3uQmWncs%3D&st=2022-01-19T09:00:39Z&se=2121-12-26T09:05:39Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'de-DE-KlausNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '4c0cb0f6-f101-4fe1-838d-67f48c70ec53',
    locale: 'mr-IN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (mr-IN, ManoharNeural)',
    properties: {
      DisplayName: 'Manohar',
      DisplayVoiceName: 'ManoharNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'मनोहर',
      LocaleDescription: 'mr-IN',
      OrderInVoiceListApi: '328',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'mr-IN-ManoharNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'mr-IN',
      localeZH: '马拉地语(印度)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/43201b41-12b2-4b57-8e0f-62c543bcdfaa.wav?sv=2019-07-07&sr=b&sig=IeFHdWPOuWgny%2BmUOVoA70c7CVgwCu6OGsOa7swAGhI%3D&st=2021-10-29T08:27:30Z&se=2121-10-05T08:32:30Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'mr-IN-ManoharNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'b7bb8122-2c38-47be-a53b-69bbd81c7b7a',
    locale: 'hr-HR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (hr-HR, SreckoNeural)',
    properties: {
      DisplayName: 'Srecko',
      DisplayVoiceName: 'SreckoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Srećko',
      LocaleDescription: 'Croatian',
      OrderInVoiceListApi: '266',
      PreviewSentence:
        'Pravi aplikacije i usluge koje se obraćaju korisnicima na što prirodniji način i poboljšava njihovu pristupačnost i korišćenje.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'hr-HR-SreckoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'hr-HR',
      localeZH: '克罗地亚语(克罗地亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/55c8d442-9358-496c-b515-e1d46958c83d.wav?sv=2019-07-07&sr=b&sig=JTIQYKMehgyLfCHY0xQbgaByUe6V4266vGZBax7s9JE%3D&st=2021-10-29T08:27:12Z&se=2121-10-05T08:32:12Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'hr-HR-SreckoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '04b3870c-105c-4c8f-b557-6a02b92ad98f',
    locale: 'ro-RO',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ro-RO, EmilNeural)',
    properties: {
      DisplayName: 'Emil',
      DisplayVoiceName: 'EmilNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Emil',
      LocaleDescription: 'Romanian',
      OrderInVoiceListApi: '369',
      PreviewSentence:
        'Creați aplicații și servicii familiare utilizatorilor, îmbunătățind accesibilitatea și ușurința utilizării.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ro-RO-EmilNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ro-RO',
      localeZH: '罗马尼亚语(罗马尼亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/7c4ba44b-a980-40cb-a473-fead6e0bd378.wav?sv=2019-07-07&sr=b&sig=V8xZWCnyCz%2F3AbZkKTvtirl2Sb8ysdpmpAsVtAGswms%3D&st=2021-10-29T08:28:01Z&se=2121-10-05T08:33:01Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ro-RO-EmilNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '5db0c41c-9206-4a67-8987-6aec4b8b92c8',
    locale: 'ga-IE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ga-IE, OrlaNeural)',
    properties: {
      DisplayName: 'Orla',
      DisplayVoiceName: 'OrlaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Orla',
      LocaleDescription: 'Irish (Ireland)',
      OrderInVoiceListApi: '256',
      PreviewSentence:
        'Cuireann Cruthú Ábhar Fuaime ar do chumas tréithe cainte a rialú i bhfíor-am.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ga-IE-OrlaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ga-IE',
      localeZH: '爱尔兰语(爱尔兰)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/14ffd29a-8cba-4ea4-bb96-d43a90cdd278.wav?sv=2019-07-07&sr=b&sig=epiRvGou%2Bh3xn%2BDUxngnx20YQHlXYd0OXs8tWmh3rEc%3D&st=2021-10-29T08:27:04Z&se=2121-10-05T08:32:04Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ga-IE-OrlaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '5fc3d49b-a679-4a6b-89cd-6be315fa06c9',
    locale: 'ro-RO',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ro-RO, AlinaNeural)',
    properties: {
      DisplayName: 'Alina',
      DisplayVoiceName: 'AlinaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Alina',
      LocaleDescription: 'Romanian',
      OrderInVoiceListApi: '368',
      PreviewSentence:
        'Creați aplicații și servicii familiare utilizatorilor, îmbunătățind accesibilitatea și ușurința utilizării.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ro-RO-AlinaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ro-RO',
      localeZH: '罗马尼亚语(罗马尼亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/3844883d-dc41-47f3-85bc-22e390cf7fce.wav?sv=2019-07-07&sr=b&sig=CziB4c12et3rzLRg4zieY2Gzm9SamCCR4rQ8eSBm%2FdI%3D&st=2021-10-29T08:27:59Z&se=2121-10-05T08:32:59Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ro-RO-AlinaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'd289dfad-828e-445b-bec6-6cde6dd1663c',
    locale: 'ar-MA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-MA, MounaNeural)',
    properties: {
      DisplayName: 'Mouna',
      DisplayVoiceName: 'MounaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'منى',
      LocaleDescription: 'ar-MA',
      OrderInVoiceListApi: '24',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-MA-MounaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-MA',
      localeZH: '阿拉伯语(摩洛哥)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a2b03517-c1c8-4260-866f-28b79cc6979e.wav?sv=2019-07-07&sr=b&sig=4oD4GcWko0tq6TxpSDX%2BGYCFI0zn9RLV2sSnhimY0pA%3D&st=2021-10-29T08:30:12Z&se=2121-10-05T08:35:12Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-MA-MounaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '0c9aeff2-a86f-4b3c-ad09-6dccc70e9bb5',
    locale: 'zh-CN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, YunzeNeural)',
    properties: {
      DisplayName: 'Yunze',
      DisplayVoiceName: 'YunzeNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      IsTtsBreakEnabled: 'True',
      LocalName: '云泽',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '443',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-YunzeNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{"OlderAdultMale":2,"SeniorMale":3,"Default":0}',
      VoiceRoleNames: 'OlderAdultMale,SeniorMale,Default',
      VoiceStyleNameDefinitions:
        '{"calm":2,"fearful":3,"cheerful":4,"disgruntled":5,"serious":6,"angry":7,"sad":8,"depressed":10,"Default":0}',
      VoiceStyleNames:
        'calm,fearful,cheerful,disgruntled,serious,angry,sad,depressed,documentary-narration,Default',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fb28a3db-bf0f-4600-9608-1055ccfa8468.wav?sv=2019-07-07&sr=b&sig=6%2BpaFYEB9nSiPSZIYuRvfVsYPPGlH2k3NUxmrrQE4pk%3D&st=2022-08-01T10:44:27Z&se=2122-07-08T10:49:27Z&sp=rl',
          roleName: 'Default',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/eef5dace-3cd0-48a4-8d6f-4a4ab6c8b3a5.wav?sv=2019-07-07&sr=b&sig=g8Kxr8fVa7Ko2ybvvvwORqY7UyPdceYAipcQRbZSlmA%3D&st=2022-08-01T10:44:30Z&se=2122-07-08T10:49:30Z&sp=rl',
          roleName: 'OlderAdultMale',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/32a5e7e7-31cc-48ce-90de-36484e6d1a32.wav?sv=2019-07-07&sr=b&sig=Jo5PgDK%2FTDaaUL0NC%2BLgM0dUZodMaP4DjImssnHht7E%3D&st=2022-08-01T10:44:33Z&se=2122-07-08T10:49:33Z&sp=rl',
          roleName: 'SeniorMale',
        },
      ],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5145aab5-acea-46db-9d38-4e5c66db0732.wav?sv=2019-07-07&sr=b&sig=YE8iZvzukmCE7K%2BFbzxchb5rEdrIqMb%2FLbbwf%2FpG6Ww%3D&st=2022-08-01T10:40:27Z&se=2122-07-08T10:45:27Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6b45aacf-811a-4995-8424-899fc75f94a7.wav?sv=2019-07-07&sr=b&sig=HFYj0Xp9XFgdciUy3Ol69b%2BosdjGBot5t6SkFQ0fT4E%3D&st=2022-08-01T10:40:29Z&se=2122-07-08T10:45:29Z&sp=rl',
          styleName: 'calm',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/11536dc1-f6cf-4120-84c7-f6dc74a88ef5.wav?sv=2019-07-07&sr=b&sig=TE6OhRh2SDs6e9hn4CQTcx2pWfQBhExKcNjRUZtuvks%3D&st=2022-08-01T10:40:32Z&se=2122-07-08T10:45:32Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/3d745437-3e42-48a0-b2a0-323403099ef4.wav?sv=2019-07-07&sr=b&sig=jki73CufYU1csrqHEql6sLpkxb7vpW4APyL0uwyKzRA%3D&st=2022-08-01T10:40:35Z&se=2122-07-08T10:45:35Z&sp=rl',
          styleName: 'depressed',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0430cb80-fa12-46bd-ba3f-4a4d8ea729fd.wav?sv=2019-07-07&sr=b&sig=z6z2e4nZedccDrTiMHmgCfCsfs9YfMyGGOMu5VqcHsM%3D&st=2022-08-01T10:40:37Z&se=2122-07-08T10:45:37Z&sp=rl',
          styleName: 'disgruntled',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a9148b56-90f5-4f08-b5fd-a9f20c283427.wav?sv=2019-07-07&sr=b&sig=dpEKuHmS3G2vUzPJgeXuJkAFMpwJ3k0XgLduMyR1cj0%3D&st=2022-08-01T10:40:40Z&se=2122-07-08T10:45:40Z&sp=rl',
          styleName: 'documentary-narration',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ebd268e8-fbd4-45dd-b350-5c3b6a6f8072.wav?sv=2019-07-07&sr=b&sig=MOUW%2FJ%2FunZi5%2FFMDeIZIwcvOnNCbFCpS55%2ByBXnz1HU%3D&st=2022-08-01T10:40:43Z&se=2122-07-08T10:45:43Z&sp=rl',
          styleName: 'fearful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b67a8984-b2c8-4308-97cb-3504670b4678.wav?sv=2019-07-07&sr=b&sig=fN4FBqbXXzVQyArs7z1JFxzf%2B0ptTjjJSgz%2Fpf3QiME%3D&st=2022-08-01T10:40:47Z&se=2122-07-08T10:45:47Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/15379477-fa67-4bc4-bb5b-83eb6014c448.wav?sv=2019-07-07&sr=b&sig=B3EXny%2FFmjHxgJ99%2BuENIAXtFBKJb4BzU%2BnvGsvT%2F68%3D&st=2022-08-01T10:40:50Z&se=2122-07-08T10:45:50Z&sp=rl',
          styleName: 'sad',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e3855e56-dd77-4d36-b0a5-557ff981cc66.wav?sv=2019-07-07&sr=b&sig=vO2y2za%2FumcGagZQIlY85%2FWlAAoG8qn9Q41CG1VX0Zk%3D&st=2022-08-01T10:40:52Z&se=2122-07-08T10:45:52Z&sp=rl',
          styleName: 'serious',
        },
      ],
    },
    shortName: 'zh-CN-YunzeNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '5471da53-e81c-414c-a0f3-6de6935bb24a',
    locale: 'da-DK',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (da-DK, JeppeNeural)',
    properties: {
      DisplayName: 'Jeppe',
      DisplayVoiceName: 'JeppeNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Jeppe',
      LocaleDescription: 'Danish',
      OrderInVoiceListApi: '55',
      PreviewSentence:
        'Lav apps og tjenester, der taler naturligt til brugere, forbedrer tilgængelighed og brugervenlighed.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'da-DK-JeppeNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'da-DK',
      localeZH: '丹麦语(丹麦)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6cf8ec50-ea13-4fc5-bc44-504d0b3af5f4.wav?sv=2019-07-07&sr=b&sig=2WZ%2B1itqPSGtYxBwLiG4mcD%2FDBrROKXmjiTRMb2JoCg%3D&st=2021-10-29T08:26:09Z&se=2121-10-05T08:31:09Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'da-DK-JeppeNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'f6369af4-16e6-4f7a-b2af-6ebcf9052252',
    locale: 'de-AT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (de-AT, JonasNeural)',
    properties: {
      DisplayName: 'Jonas',
      DisplayVoiceName: 'JonasNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Jonas',
      LocaleDescription: 'German (Austria)',
      OrderInVoiceListApi: '57',
      PreviewSentence:
        'Erstellen Sie Apps und Dienste, die auf natürliche Art mit Nutzern sprechen, und dadurch Zugänglichkeit und Benutzerfreundlichkeit verbessern.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'de-AT-JonasNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'de-AT',
      localeZH: '德语(奥地利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ed4dbd28-0ca6-40a0-9d05-01c424070474.wav?sv=2019-07-07&sr=b&sig=Wf7OfVfN2qAJOGSOng9JYgHr%2BbJHSpTaPn%2BMUFolY4s%3D&st=2021-10-29T08:26:11Z&se=2121-10-05T08:31:11Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'de-AT-JonasNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'de121e34-6b54-46b3-8766-6efeb7ebe53c',
    locale: 'en-GB',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-GB, HollieNeural)',
    properties: {
      DisplayName: 'Hollie',
      DisplayVoiceName: 'HollieNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Hollie',
      LocaleDescription: 'English (UK)',
      OrderInVoiceListApi: '99',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-GB-HollieNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-GB',
      localeZH: '英语(英国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a5e5caeb-843a-4f8c-858b-422ae3588b2b.wav?sv=2019-07-07&sr=b&sig=gf1FeXpRX1aRcmjhW0QsWi20GYvtSNdH6Xl%2FDjuKqp0%3D&st=2022-01-19T09:01:03Z&se=2121-12-26T09:06:03Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-GB-HollieNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '1953900b-4f14-4b18-bebc-6f21d8b67f36',
    locale: 'it-IT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (it-IT, GianniNeural)',
    properties: {
      DisplayName: 'Gianni',
      DisplayVoiceName: 'GianniNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Gianni',
      LocaleDescription: 'Italian',
      OrderInVoiceListApi: '282',
      PreviewSentence:
        'Benvenuti nella piattaforma di generazione di contenuti audio di Microsoft.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'it-IT-GianniNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'it-IT',
      localeZH: '意大利语(意大利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/be773246-a0f7-41c7-abfa-aa2bc44a7755.wav?sv=2019-07-07&sr=b&sig=i1Sl%2B%2BWapYf%2BDze3NKJm0CJ40dn4cGY53N8GG2Tcbuw%3D&st=2022-07-01T09:09:42Z&se=2122-06-07T09:14:42Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'it-IT-GianniNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '3e2e9811-2a35-45f3-97a3-6f2d255cd9e0',
    locale: 'id-ID',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (id-ID, GadisNeural)',
    properties: {
      DisplayName: 'Gadis',
      DisplayVoiceName: 'GadisNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Gadis',
      LocaleDescription: 'Indonesian',
      OrderInVoiceListApi: '272',
      PreviewSentence:
        'Buat aplikasi dan layanan yang berbicara secara alami kepada pengguna, sehingga meningkatkan aksesibilitas dan kegunaan.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'id-ID-GadisNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'id-ID',
      localeZH: '印度尼西亚语(印度尼西亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/63b0fa9f-1f3e-4fba-8a56-dcbecbc9fc6c.wav?sv=2019-07-07&sr=b&sig=liSiYhJBQ%2BhfpRNpo3XWHyy7GLdRoOlKEA5INJHapLc%3D&st=2021-10-29T08:27:16Z&se=2121-10-05T08:32:16Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'id-ID-GadisNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '23161424-2307-449a-9d24-6f50b7c09553',
    locale: 'hu-HU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (hu-HU, NoemiNeural)',
    properties: {
      DisplayName: 'Noemi',
      DisplayVoiceName: 'NoemiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Noémi',
      LocaleDescription: 'Hungarian',
      OrderInVoiceListApi: '267',
      PreviewSentence:
        'Készítsen appokat és szolgáltatásokat, melyek természetes hangon beszélnek a felhasználóval, ezáltal hozzáférhetőbbek és könnyebben használhatóak.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'hu-HU-NoemiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'hu-HU',
      localeZH: '匈牙利语(匈牙利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6756708e-cb80-446b-bbf2-41a8938ca10d.wav?sv=2019-07-07&sr=b&sig=9E2YOw%2FZMga2ar0YnaU%2BKZ%2Fy1lBgBwwvETnlrXgkE%2FI%3D&st=2021-10-29T08:27:13Z&se=2121-10-05T08:32:13Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'hu-HU-NoemiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'a67d97ed-51c3-4daa-a2f2-6fb669d48ab6',
    locale: 'zh-TW',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-TW, HsiaoChenNeural)',
    properties: {
      DisplayName: 'HsiaoChen',
      DisplayVoiceName: 'HsiaoChenNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: '曉臻',
      LocaleDescription: 'Chinese (Taiwan)',
      OrderInVoiceListApi: '452',
      PreviewSentence: '建構可以和使用者自然對話的應用程式和服務，來提高其方便性和實用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-TW-HsiaoChenNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'zh-TW',
      localeZH: '中文(台湾普通话)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/28d0a866-cc6d-496b-9084-0bf48d6fa07e.wav?sv=2019-07-07&sr=b&sig=WwqH3hIg20fFO74KNGq8DWooUa1%2F0ET8THRRAR2zpko%3D&st=2021-10-29T08:29:51Z&se=2121-10-05T08:34:51Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'zh-TW-HsiaoChenNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '400eb503-6c32-499c-b66a-714a04b1e0ea',
    locale: 'fr-CA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-CA, SylvieNeural)',
    properties: {
      DisplayName: 'Sylvie',
      DisplayVoiceName: 'SylvieNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Sylvie',
      LocaleDescription: 'French (Canada)',
      OrderInVoiceListApi: '236',
      PreviewSentence:
        'Créer des applications et des services qui parlent aux utilisateurs, améliorant ainsi l’accessibilité et la facilité d’utilisation.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-CA-SylvieNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fr-CA',
      localeZH: '法语(加拿大)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/cb0d45ba-42ef-46d5-b0ed-353bcf7feadb.wav?sv=2019-07-07&sr=b&sig=At7OK796JZWSXkWpXI3p3IfL6imH9ATKcRpNhFHjg4M%3D&st=2021-10-29T08:26:58Z&se=2121-10-05T08:31:58Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fr-CA-SylvieNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '65ea2440-7f3f-422f-8688-7177e3417cd9',
    locale: 'fr-FR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-FR, JosephineNeural)',
    properties: {
      DisplayName: 'Josephine',
      DisplayVoiceName: 'JosephineNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Josephine',
      LocaleDescription: 'French (France)',
      OrderInVoiceListApi: '250',
      PreviewSentence:
        "Construire des applications et services qui communiquent naturellement avec l'utilisateur, en améliorant l’accessibilité et l'ergonomie",
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-FR-JosephineNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fr-FR',
      localeZH: '法语(法国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d8ec632e-ad80-44ed-9c33-dcadf06470f9.wav?sv=2019-07-07&sr=b&sig=v2tPx5AdxddBhf9K7iLHxeTWDh2pplh38UrbwhhX%2BjQ%3D&st=2022-01-19T09:01:35Z&se=2121-12-26T09:06:35Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fr-FR-JosephineNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '883d200d-43ae-4b1d-a284-7226e93177ab',
    locale: 'it-IT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (it-IT, FabiolaNeural)',
    properties: {
      DisplayName: 'Fabiola',
      DisplayVoiceName: 'FabiolaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Fabiola',
      LocaleDescription: 'Italian',
      OrderInVoiceListApi: '280',
      PreviewSentence:
        'Benvenuti nella piattaforma di generazione di contenuti audio di Microsoft.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'it-IT-FabiolaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'it-IT',
      localeZH: '意大利语(意大利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c29bfe43-92cd-49c1-abfd-a89af97d8df7.wav?sv=2019-07-07&sr=b&sig=KoLxrttwZNqlAZCdxA6wQbDxfIZ8ibaJMPSAjorgoBY%3D&st=2022-07-01T09:09:38Z&se=2122-06-07T09:14:38Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'it-IT-FabiolaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '08456f07-f950-4fa5-b0ef-7266bf338fe3',
    locale: 'my-MM',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (my-MM, ThihaNeural)',
    properties: {
      DisplayName: 'Thiha',
      DisplayVoiceName: 'ThihaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'သီဟ',
      LocaleDescription: 'my-MM',
      OrderInVoiceListApi: '334',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'my-MM-ThihaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'my-MM',
      localeZH: '缅甸语(缅甸)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/181833fd-0cb3-472a-9ad9-0eed0d6267b2.wav?sv=2019-07-07&sr=b&sig=TwmsXR5INrTUYKiEg0YobBpcx%2Brv32cHwJ9G8Hfpu3A%3D&st=2021-10-29T08:31:12Z&se=2121-10-05T08:36:12Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'my-MM-ThihaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '89a21c17-e2e5-4732-8494-73b795c92857',
    locale: 'lv-LV',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (lv-LV, NilsNeural)',
    properties: {
      DisplayName: 'Nils',
      DisplayVoiceName: 'NilsNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Nils',
      LocaleDescription: 'Latvian (Latvia)',
      OrderInVoiceListApi: '320',
      PreviewSentence:
        'Balss Satura Izveide ļauj jums vizuāli kontrolēt runas atribūtus reālajā laikā.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'lv-LV-NilsNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'lv-LV',
      localeZH: '拉脱维亚语(拉脱维亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/424406ba-8c2d-4a4d-9124-950fbfd5ed38.wav?sv=2019-07-07&sr=b&sig=rulzEaPe6Ink8N%2F3DS%2FTbASyvOJpSMpxQNaU7%2B6gg6E%3D&st=2021-10-29T08:27:28Z&se=2121-10-05T08:32:28Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'lv-LV-NilsNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'cb6c203f-d969-4c34-b13e-73ed92ce859c',
    locale: 'it-IT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (it-IT, LisandroNeural)',
    properties: {
      DisplayName: 'Lisandro',
      DisplayVoiceName: 'LisandroNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Lisandro',
      LocaleDescription: 'Italian',
      OrderInVoiceListApi: '285',
      PreviewSentence:
        'Benvenuti nella piattaforma di generazione di contenuti audio di Microsoft.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'it-IT-LisandroNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'it-IT',
      localeZH: '意大利语(意大利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6dbbcea7-91ab-42c8-925f-526d6c677fe8.wav?sv=2019-07-07&sr=b&sig=S7%2FSfHneBNx00JmnijjWP2%2F3qv2SnFA9kxb5ccW1MpA%3D&st=2022-07-01T09:09:49Z&se=2122-06-07T09:14:49Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'it-IT-LisandroNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '67f27a5a-944a-4590-8a79-741d3780e48a',
    locale: 'zh-CN-liaoning',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN-liaoning, XiaobeiNeural)',
    properties: {
      DisplayName: 'Xiaobei',
      DisplayVoiceName: 'XiaobeiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: '晓北',
      LocaleDescription: 'zh-CN-liaoning',
      OrderInVoiceListApi: '445',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-liaoning-XiaobeiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'zh-CN-liaoning',
      localeZH: '中文(东北官话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c988c129-ad9e-4f2a-836b-1b2c3e7801e9.wav?sv=2019-07-07&sr=b&sig=Vf8PddKQOyCL8nks4PpIb%2BQe1VOMsyXein0KKKY51cQ%3D&st=2022-05-12T09:49:37Z&se=2122-04-18T09:54:37Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'zh-CN-liaoning-XiaobeiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '9662aa30-0632-4d8d-8c54-74dec56333e6',
    locale: 'de-DE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (de-DE, ChristophNeural)',
    properties: {
      DisplayName: 'Christoph',
      DisplayVoiceName: 'ChristophNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Christoph',
      LocaleDescription: 'German (Germany)',
      OrderInVoiceListApi: '63',
      PreviewSentence:
        'Erstellen Sie Apps und Dienste, die auf natürliche Art mit Nutzern sprechen, und dadurch Zugänglichkeit und Benutzerfreundlichkeit verbessern.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'de-DE-ChristophNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'de-DE',
      localeZH: '德语(德国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ba166b8f-cdcb-4c90-954e-7cdd749738ae.wav?sv=2019-07-07&sr=b&sig=Dx0ZKrx%2FeHp8tx9ABG2zCPWygMX4otY9Og7NJ8QwN0w%3D&st=2022-01-19T09:00:25Z&se=2121-12-26T09:05:25Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'de-DE-ChristophNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '7e2d1117-e923-43a9-a5a0-7514dc779b95',
    locale: 'ga-IE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ga-IE, ColmNeural)',
    properties: {
      DisplayName: 'Colm',
      DisplayVoiceName: 'ColmNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Colm',
      LocaleDescription: 'Irish (Ireland)',
      OrderInVoiceListApi: '255',
      PreviewSentence:
        'Cuireann Cruthú Ábhar Fuaime ar do chumas tréithe cainte a rialú i bhfíor-am.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ga-IE-ColmNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ga-IE',
      localeZH: '爱尔兰语(爱尔兰)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ad87fa66-a7d8-4ffc-a264-7892712ed683.wav?sv=2019-07-07&sr=b&sig=AsKYEmtuILYC6MNlLUKDbgjpzLdsNvmzlq4i6te2SKs%3D&st=2021-10-29T08:27:03Z&se=2121-10-05T08:32:03Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ga-IE-ColmNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '26118d30-06ba-4c1b-9f36-752135f87647',
    locale: 'en-GB',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-GB, OliverNeural)',
    properties: {
      DisplayName: 'Oliver',
      DisplayVoiceName: 'OliverNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Oliver',
      LocaleDescription: 'English (UK)',
      OrderInVoiceListApi: '102',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-GB-OliverNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-GB',
      localeZH: '英语(英国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/816f4737-fdf6-484a-960e-0b137430474e.wav?sv=2019-07-07&sr=b&sig=4WMo0CWjZyjl%2FL3XxrPBcRqCx8BfHEzWi0zHFoCjhDs%3D&st=2022-01-19T09:01:10Z&se=2121-12-26T09:06:10Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-GB-OliverNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '2645ee9c-4b0e-46a0-8085-76606154676a',
    locale: 'bn-BD',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (bn-BD, PradeepNeural)',
    properties: {
      DisplayName: 'Pradeep',
      DisplayVoiceName: 'PradeepNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'প্রদ্বীপ',
      LocaleDescription: 'bn-BD',
      OrderInVoiceListApi: '42',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'bn-BD-PradeepNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'bn-BD',
      localeZH: '孟加拉语(孟加拉)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6eb56666-9989-47c2-b0f5-8180ea71dedc.wav?sv=2019-07-07&sr=b&sig=SlwZedhjKBT40CmWDRtXnuT4paiE5pmu9UwnpLkQDYI%3D&st=2021-10-29T08:30:22Z&se=2121-10-05T08:35:22Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'bn-BD-PradeepNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [
      {
        name: 'All',
      },
      {
        name: 'Audiobook',
      },
    ],
    id: '53d185d8-dd7b-4184-b987-76b6f1ce8c41',
    locale: 'zh-CN',
    masterpieces: [
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/74b5543e-f8e5-43ad-91e4-2c0a7f5104b8.wav?sv=2019-07-07&sr=b&sig=85KUBXC20vzIniASJ4navwAkJhL2YWZYu7WqNbLmjSA%3D&st=2021-10-29T08:31:51Z&se=2121-10-05T08:36:51Z&sp=rl',
        categories: ['Audiobook'],
        title: '晓秋散文展示',
        titleLocalizationId: 'zh-CN-XiaoqiuNeural_Masterpieces_AudioBook_ProseDemo_Title',
      },
    ],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, XiaoqiuNeural)',
    properties: {
      AgeGroups: 'OlderAdult',
      DisplayName: 'Xiaoqiu',
      DisplayVoiceName: 'XiaoqiuNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      IconFileEndpointWithSas:
        'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/80019f15-a6b7-4884-870d-09498de31a5c.png?sv=2019-07-07&sr=b&sig=hPqBfmzeeVzRP%2F%2BCzaCmq5KnC53PGWX8ooz6hk%2B43XQ%3D&st=2021-10-29T08:24:33Z&se=2121-10-05T08:29:33Z&sp=rl',
      IsTtsBreakEnabled: 'True',
      LocalName: '晓秋',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '429',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-XiaoqiuNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f3b22aba-8e0d-448e-9325-294a01fa8618.wav?sv=2019-07-07&sr=b&sig=WKTnMLUiODCJEaj8W38Oj4ipBJwW7F4wQadxyA%2FclQE%3D&st=2021-10-29T08:29:44Z&se=2121-10-05T08:34:44Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'zh-CN-XiaoqiuNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '9c696a0d-fb1b-485a-abeb-773b85297701',
    locale: 'ja-JP',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ja-JP, NaokiNeural)',
    properties: {
      DisplayName: 'Naoki',
      DisplayVoiceName: 'NaokiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: '直紀',
      LocaleDescription: 'Japanese',
      OrderInVoiceListApi: '295',
      PreviewSentence:
        'アクセシビリティとユーザビリティを向上させながら、ユーザーに自然なことばを話すアプリとサービスを構築する',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ja-JP-NaokiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ja-JP',
      localeZH: '日语(日本)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/269259f1-c80f-446c-bb79-7482326822b8.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A30Z&se=2122-09-27T07%3A52%3A30Z&sr=b&sp=rl&sig=2JnV7PFGO%2Fbh9pIT1bSLF3yqEa2aHN9ehBDQWgnR5Qw%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ja-JP-NaokiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [
      {
        name: 'All',
      },
      {
        name: 'Audiobook',
      },
      {
        name: 'Chat',
      },
    ],
    description:
      'Mature and relaxing voice with multiple emotions that is optimized for audio books.',
    id: '26014551-90d7-4f55-a622-779b8263e006',
    locale: 'zh-CN',
    masterpieces: [
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/833c684e-ead2-42d2-97bc-a20b710febab.wav?sv=2019-07-07&sr=b&sig=BHnUvOPyTcR4efp8GNVvoLLCQWbcNIRBhomIVpiFP6I%3D&st=2021-10-29T08:24:54Z&se=2121-10-05T08:29:54Z&sp=rl',
        categories: ['Audiobook'],
        title: '云野有声书《朱自清散文集》',
        titleLocalizationId: 'zh-CN-YunyeNeural_Masterpieces_AudioBook_ZhuZiQingSanWenJi_Title',
      },
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/06d63d0b-1c99-43f8-9519-474e2b89be37?sv=2019-07-07&sr=b&sig=RCY00H6MovZoLvd4tFyPdBIka51T8HO182QOS0hKl6A%3D&st=2021-10-29T08:24:52Z&se=2121-10-05T08:29:52Z&sp=rl',
        categories: ['Audiobook'],
        title: '云野有声书《金粉世家》',
        titleLocalizationId: 'zh-CN-YunyeNeural_Masterpieces_AudioBook_JinFenShiJia_Title',
      },
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/382cb9cd-3c3b-40c1-8477-f3094c93438c?sv=2019-07-07&sr=b&sig=y%2BRdXIupJSQMSk0K7hqJddhDI%2Bb57rhlEF1EV4rJ6sw%3D&st=2021-10-29T08:24:53Z&se=2121-10-05T08:29:53Z&sp=rl',
        categories: ['Audiobook'],
        title: '云野有声书《骆驼祥子》',
        titleLocalizationId: 'zh-CN-YunyeNeural_Masterpieces_AudioBook_LuoTuoXiangZi_Title',
      },
    ],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, YunyeNeural)',
    properties: {
      AgeGroups: 'OlderAdult',
      DisplayName: 'Yunye',
      DisplayVoiceName: 'YunyeNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      IconFileEndpointWithSas:
        'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/3f214a6d-d32f-46e1-a2bf-43f45c472154?sv=2019-07-07&sr=b&sig=6uBXzG3uXqs5WMmE3bhGz1eTonPHLFc5dgFAm%2B%2F0w7c%3D&st=2021-10-29T08:24:31Z&se=2121-10-05T08:29:31Z&sp=rl',
      IsTtsBreakEnabled: 'True',
      LocalName: '云野',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '442',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-YunyeNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions:
        '{"YoungAdultFemale":4,"YoungAdultMale":1,"OlderAdultFemale":5,"OlderAdultMale":2,"SeniorFemale":10,"SeniorMale":3,"Girl":8,"Boy":9,"Default":0}',
      VoiceRoleNames:
        'YoungAdultFemale,YoungAdultMale,OlderAdultFemale,OlderAdultMale,SeniorFemale,SeniorMale,Girl,Boy,Default',
      VoiceStyleNameDefinitions:
        '{"embarrassed":1,"calm":2,"fearful":3,"cheerful":4,"disgruntled":5,"serious":6,"angry":7,"sad":8,"Default":0}',
      VoiceStyleNames: 'embarrassed,calm,fearful,cheerful,disgruntled,serious,angry,sad,Default',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f5fc312c-a10a-4f0d-acb9-bd87489ec497.wav?sv=2019-07-07&sr=b&sig=x%2FEjeQdjX1ORPS%2FbVutUwm1%2BJVZFG4X40zwxPXpxsUA%3D&st=2021-12-20T11:29:48Z&se=2121-11-26T11:34:48Z&sp=rl',
          roleName: 'Default',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e0ede06b-a535-4f5e-82e6-db78ec3aa8a2.wav?sv=2019-07-07&sr=b&sig=DP562T5Si9KU8H4EaO%2BfCcVGpEul8lfnVoBfqczXeA4%3D&st=2021-12-20T11:29:50Z&se=2121-11-26T11:34:50Z&sp=rl',
          roleName: 'Boy',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/8d4ed42e-69aa-475c-8058-00da3ef001ff.wav?sv=2019-07-07&sr=b&sig=KlzSUmhOj%2F%2BgWangRvj17JLkO4fVAzJebOzrxhbqpAk%3D&st=2021-12-20T11:29:53Z&se=2121-11-26T11:34:53Z&sp=rl',
          roleName: 'Girl',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/289fd066-371f-4ebc-a048-1746f4decd47.wav?sv=2019-07-07&sr=b&sig=RITeVr3p7LVgt9z5wsFEHCMoUA2GzRmPyYcPYRrDv7I%3D&st=2021-12-20T11:29:56Z&se=2121-11-26T11:34:56Z&sp=rl',
          roleName: 'OlderAdultFemale',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/49c35ceb-c15e-4e78-85f4-6e28d72c0be0.wav?sv=2019-07-07&sr=b&sig=1BXNb%2BsZoT3ZFwoC7%2FacKoZmCdVAnQKByo11yj5GwKo%3D&st=2021-12-20T11:29:58Z&se=2121-11-26T11:34:58Z&sp=rl',
          roleName: 'OlderAdultMale',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5f0c5d64-56a5-493b-9d97-91e94827805f.wav?sv=2019-07-07&sr=b&sig=qlQmTPM7q2TLalNUyyoSRHds0tEawvPRIb61T3xv8Og%3D&st=2021-12-20T11:30:01Z&se=2121-11-26T11:35:01Z&sp=rl',
          roleName: 'SeniorFemale',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/77ec8ad9-e8d6-4e91-a73f-f8496b80b19d.wav?sv=2019-07-07&sr=b&sig=tNCqZcks4%2BeqU6Rs7ixAfRa2IFEie36a5UFK2E4ARUg%3D&st=2021-12-20T11:30:04Z&se=2121-11-26T11:35:04Z&sp=rl',
          roleName: 'SeniorMale',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5701e3cb-e13e-45fd-8a24-9f8de377b495.wav?sv=2019-07-07&sr=b&sig=sWQUN%2FjsHZ7yAEDBZ3PhOewcymGZOU%2BBzVlS6aPSYTc%3D&st=2021-12-20T11:30:07Z&se=2121-11-26T11:35:07Z&sp=rl',
          roleName: 'YoungAdultFemale',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/82ef7994-b50a-4d06-bbb2-0347eced8f0e.wav?sv=2019-07-07&sr=b&sig=r09nEgFcKiBDH%2B9NzeGyx3UXTfw5QkcQvUCBAK1P2q0%3D&st=2021-12-20T11:30:10Z&se=2121-11-26T11:35:10Z&sp=rl',
          roleName: 'YoungAdultMale',
        },
      ],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d8072a88-0df9-4ef2-bfca-3e72d446569f.wav?sv=2019-07-07&sr=b&sig=vCVB%2FbC%2FQBaJi8c7owxIZmtuR5eRC7dvdJQo8%2BKoP24%3D&st=2021-10-29T08:29:35Z&se=2121-10-05T08:34:35Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0ee24ea7-7ade-4fb6-b043-3a406d26dc7b.wav?sv=2019-07-07&sr=b&sig=yZLQWvcTO6HVbi9qgjZ%2Bk9eb5iwgjHbxgf0GyOwEcAc%3D&st=2021-10-29T08:29:36Z&se=2121-10-05T08:34:36Z&sp=rl',
          styleName: 'calm',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/67e6267f-8f36-4685-87d4-5c10387de47f.wav?sv=2019-07-07&sr=b&sig=6XAD4uyDdRzCPMNpgQJSjYtstFoy%2BsNmLs0Dkn4%2B190%3D&st=2021-10-29T08:29:37Z&se=2121-10-05T08:34:37Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a920fcf3-c833-4739-ac3c-4b0760da71b6.wav?sv=2019-07-07&sr=b&sig=G%2F%2BSwVqshvkhv3Y3BO%2FwMBqNdZEkjbspsSzZDT%2FjEd8%3D&st=2021-10-29T08:29:38Z&se=2121-10-05T08:34:38Z&sp=rl',
          styleName: 'disgruntled',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/3fd8e827-4c5d-4a9f-84ea-fec6145d0619.wav?sv=2019-07-07&sr=b&sig=e3VwY4K1fEm3j45%2F6Qjp9qezg40xNXF6vdrCso79Ii8%3D&st=2021-12-02T05:02:34Z&se=2121-11-08T05:07:34Z&sp=rl',
          styleName: 'embarrassed',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2a34875c-4a4b-42f7-9d85-31dd2a619910.wav?sv=2019-07-07&sr=b&sig=E64MW%2F5nt79Q6uxcoSePL2yW7czOXF49nfzyocfsYh8%3D&st=2021-10-29T08:29:39Z&se=2121-10-05T08:34:39Z&sp=rl',
          styleName: 'fearful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/22657aa6-9463-4f23-9d74-501f9a7f2322.wav?sv=2019-07-07&sr=b&sig=zEOeeFbKg%2FnZM2b0QJRyvA7pN4iZZNCvbH3u4rwp6tI%3D&st=2021-10-29T08:29:40Z&se=2121-10-05T08:34:40Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/88dbaf0d-5d23-48c1-aa91-8a9bd3b29d27.wav?sv=2019-07-07&sr=b&sig=0kOxLQmbqVYVtWQxYdYFuBtXNhz58DIH0Wn%2Bgcu%2B4G4%3D&st=2021-10-29T08:29:41Z&se=2121-10-05T08:34:41Z&sp=rl',
          styleName: 'sad',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/cd1e1437-fcab-4fa8-995f-da0f2ed2fe37.wav?sv=2019-07-07&sr=b&sig=f1ENR%2F4RvkqYAKHsyf1jvLUmsEDyGIYw52eUcmhv3i8%3D&st=2021-10-29T08:29:42Z&se=2121-10-05T08:34:42Z&sp=rl',
          styleName: 'serious',
        },
      ],
    },
    shortName: 'zh-CN-YunyeNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '58345d02-5c0f-472b-8b94-783420c8c66c',
    locale: 'fr-FR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-FR, YvetteNeural)',
    properties: {
      DisplayName: 'Yvette',
      DisplayVoiceName: 'YvetteNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Yvette',
      LocaleDescription: 'French (France)',
      OrderInVoiceListApi: '253',
      PreviewSentence:
        "Construire des applications et services qui communiquent naturellement avec l'utilisateur, en améliorant l’accessibilité et l'ergonomie",
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-FR-YvetteNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fr-FR',
      localeZH: '法语(法国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6e470838-e1db-4a0d-be11-965afa071801.wav?sv=2019-07-07&sr=b&sig=mBAtETP5%2B1MFg5Yq5XTBXQ8lWywmmfO%2Fdof8u%2FcTL7Q%3D&st=2022-01-19T09:01:42Z&se=2121-12-26T09:06:42Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fr-FR-YvetteNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '1564dfc6-9cf9-46f6-a7d2-795415a99ce2',
    locale: 'bn-IN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (bn-IN, BashkarNeural)',
    properties: {
      DisplayName: 'Bashkar',
      DisplayVoiceName: 'BashkarNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'ভাস্কর',
      LocaleDescription: 'bn-IN',
      OrderInVoiceListApi: '43',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'bn-IN-BashkarNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'bn-IN',
      localeZH: '孟加拉语(印度)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b77daa45-b965-4030-9986-dea6878133dc.wav?sv=2019-07-07&sr=b&sig=o6tiWtf4URUdvWl%2BUCaZGTNkycBwQonzx4OZ75oNDAY%3D&st=2022-01-19T09:00:15Z&se=2121-12-26T09:05:15Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'bn-IN-BashkarNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'c5b73621-18c5-425b-9020-79aae71f827e',
    locale: 'wuu-CN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (wuu-CN, XiaotongNeural)',
    properties: {
      DisplayName: 'Xiaotong',
      DisplayVoiceName: 'XiaotongNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: '晓彤',
      LocaleDescription: 'wuu-CN',
      OrderInVoiceListApi: '419',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'wuu-CN-XiaotongNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'wuu-CN',
      localeZH: '中文(吴语，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d2fb7288-b63d-43c6-8278-c32dd02a9e99.wav?sv=2021-08-06&st=2022-10-21T11%3A57%3A25Z&se=2122-09-27T12%3A02%3A25Z&sr=b&sp=rl&sig=OMuveovC4OGwJ0OVIzAU9OVXcBswQGVbL82SV%2BGnbKs%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'wuu-CN-XiaotongNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '2e31b5db-4def-4e35-9337-79bbdaf0cd87',
    locale: 'en-AU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-AU, DarrenNeural)',
    properties: {
      DisplayName: 'Darren',
      DisplayVoiceName: 'DarrenNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Darren',
      LocaleDescription: 'English (Australia)',
      OrderInVoiceListApi: '79',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-AU-DarrenNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-AU',
      localeZH: '英语(澳大利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a12b3920-f2fa-4f39-8394-0c73bd683e20.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A02Z&se=2122-09-27T07%3A51%3A02Z&sr=b&sp=rl&sig=vZNnb2WLQIQVuTWmKnIWkiydZabGn5e%2B%2Bb0ZWNkilmE%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-AU-DarrenNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'd278f8ad-6f2c-4c23-bcbb-7ad04efd3cd4',
    locale: 'pt-PT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pt-PT, DuarteNeural)',
    properties: {
      DisplayName: 'Duarte',
      DisplayVoiceName: 'DuarteNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Duarte',
      LocaleDescription: 'Portuguese (Portugal)',
      OrderInVoiceListApi: '365',
      PreviewSentence:
        'Constrói aplicações e serviços que falam naturalmente com os utilizadores, melhorando a acessibilidade e usabilidade.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pt-PT-DuarteNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'pt-PT',
      localeZH: '葡萄牙语(葡萄牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c769c060-7cab-4b02-a81c-89cb1d7e656d.wav?sv=2019-07-07&sr=b&sig=e7%2BSgctyqnpvUElP%2FHVRNWtedDyev34IsNSSC%2B6mshI%3D&st=2021-10-29T08:27:56Z&se=2121-10-05T08:32:56Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'pt-PT-DuarteNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'ec6ce467-9f53-456e-87cf-7bd553a30a0f',
    locale: 'en-AU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-AU, FreyaNeural)',
    properties: {
      DisplayName: 'Freya',
      DisplayVoiceName: 'FreyaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Freya',
      LocaleDescription: 'English (Australia)',
      OrderInVoiceListApi: '82',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-AU-FreyaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-AU',
      localeZH: '英语(澳大利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/27388180-c38e-4d13-945c-372068d0297f.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A08Z&se=2122-09-27T07%3A51%3A08Z&sr=b&sp=rl&sig=QgfwGonOnsEZwAG6P0%2Fk737BcldebJEAvm2k3y964ww%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-AU-FreyaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '376a5073-f406-4a9e-bc6f-7c50b23201f8',
    locale: 'en-CA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-CA, LiamNeural)',
    properties: {
      DisplayName: 'Liam',
      DisplayVoiceName: 'LiamNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Liam',
      LocaleDescription: 'English (Canada)',
      OrderInVoiceListApi: '92',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-CA-LiamNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-CA',
      localeZH: '英语(加拿大)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6342589f-8c0a-4a55-858f-ef1d705c2d3d.wav?sv=2019-07-07&sr=b&sig=cb7YkaU0xDHXf1GmfI5g1XWdEKJZ%2FHoEpkGWE%2BpFSxo%3D&st=2021-10-29T08:26:21Z&se=2121-10-05T08:31:21Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-CA-LiamNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'a875feda-252a-4d1d-8e72-7cc3c8199f43',
    locale: 'en-AU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-AU, DuncanNeural)',
    properties: {
      DisplayName: 'Duncan',
      DisplayVoiceName: 'DuncanNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Duncan',
      LocaleDescription: 'English (Australia)',
      OrderInVoiceListApi: '80',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-AU-DuncanNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-AU',
      localeZH: '英语(澳大利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/26cef830-8f7d-4ca0-889b-e164852c56ec.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A04Z&se=2122-09-27T07%3A51%3A04Z&sr=b&sp=rl&sig=UN5IfgOQicvRByUjcHWb4UTacevlpm6BTZ6TCKtcvGs%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-AU-DuncanNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'f87a9492-b32f-46c4-a8bb-7f8c275edd5d',
    locale: 'hy-AM',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (hy-AM, AnahitNeural)',
    properties: {
      DisplayName: 'Anahit',
      DisplayVoiceName: 'AnahitNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Անահիտ',
      LocaleDescription: 'hy-AM',
      OrderInVoiceListApi: '269',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'hy-AM-AnahitNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'hy-AM',
      localeZH: '亚美尼亚语(亚美尼亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/41138159-cac4-49ba-8805-3ca3386aacca.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A14Z&se=2122-09-27T07%3A52%3A14Z&sr=b&sp=rl&sig=hPywtUtD5RdG2IlC8dqfeG2IJWQCMq%2Fhr9%2FxKj%2B94gc%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'hy-AM-AnahitNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '8aba6946-2d68-4b7a-a30a-7fef52c903a2',
    locale: 'es-NI',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-NI, YolandaNeural)',
    properties: {
      DisplayName: 'Yolanda',
      DisplayVoiceName: 'YolandaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Yolanda',
      LocaleDescription: 'es-NI',
      OrderInVoiceListApi: '206',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-NI-YolandaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-NI',
      localeZH: '西班牙语(尼加拉瓜)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/331bf9a1-0141-444a-8557-11ee211d7c63.wav?sv=2019-07-07&sr=b&sig=nKbIJmKMox5ewP7DE3WetGv10NTUm3drGk65xtQvp4E%3D&st=2021-10-29T08:30:47Z&se=2121-10-05T08:35:47Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-NI-YolandaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '23c439de-2ff7-499f-90ff-80609f237c76',
    locale: 'ja-JP',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ja-JP, KeitaNeural)',
    properties: {
      DisplayName: 'Keita',
      DisplayVoiceName: 'KeitaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: '圭太',
      LocaleDescription: 'Japanese',
      OrderInVoiceListApi: '293',
      PreviewSentence:
        'アクセシビリティとユーザビリティを向上させながら、ユーザーに自然なことばを話すアプリとサービスを構築する',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ja-JP-KeitaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ja-JP',
      localeZH: '日语(日本)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/cdfd5dcd-35fb-4864-a8c1-e44026ce28e5.wav?sv=2019-07-07&sr=b&sig=K6zrzLrNzl7PbenFDGORNUDtWQ2yGudLsaUUK1%2Bd5G8%3D&st=2021-10-29T08:27:20Z&se=2121-10-05T08:32:20Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ja-JP-KeitaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'fd07655d-a8cf-4bcd-bcb8-8074ce0a5f7c',
    locale: 'it-IT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (it-IT, CataldoNeural)',
    properties: {
      DisplayName: 'Cataldo',
      DisplayVoiceName: 'CataldoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Cataldo',
      LocaleDescription: 'Italian',
      OrderInVoiceListApi: '278',
      PreviewSentence:
        'Benvenuti nella piattaforma di generazione di contenuti audio di Microsoft.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'it-IT-CataldoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'it-IT',
      localeZH: '意大利语(意大利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0092a25c-5c2c-402c-a68c-9a626597a048.wav?sv=2019-07-07&sr=b&sig=sj4ZRmhwVwTLk%2B8lCXWfrMjn2Ued%2BGbcU9VKM0k5yhI%3D&st=2022-07-01T09:09:36Z&se=2122-06-07T09:14:36Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'it-IT-CataldoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '2345e9ff-57a6-4938-80bc-8220d9e15365',
    locale: 'pt-BR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pt-BR, FabioNeural)',
    properties: {
      DisplayName: 'Fabio',
      DisplayVoiceName: 'FabioNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Fabio',
      LocaleDescription: 'Portuguese (Brazil)',
      OrderInVoiceListApi: '355',
      PreviewSentence:
        'Desenvolver aplicativos e serviços que conversem naturalmente com os usuários, melhorando a acessibilidade e usabilidade.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pt-BR-FabioNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'pt-BR',
      localeZH: '葡萄牙语(巴西)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f0312f12-c197-4733-b43f-9ccd3ca056dc.wav?sv=2019-07-07&sr=b&sig=doIve%2FaiU4HTa9RUHaMB5NrP9XaH6qyXd59X0bZ1uM0%3D&st=2022-07-01T09:10:17Z&se=2122-06-07T09:15:17Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'pt-BR-FabioNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '94bf333e-29f6-443b-bbf2-82eae8bb0ff8',
    locale: 'bs-BA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (bs-BA, VesnaNeural)',
    properties: {
      DisplayName: 'Vesna',
      DisplayVoiceName: 'VesnaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Vesna',
      LocaleDescription: 'bs-BA',
      OrderInVoiceListApi: '46',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'bs-BA-VesnaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'bs-BA',
      localeZH: '波斯尼亚语(波斯尼亚和黑塞哥维那)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/7296e0a5-991f-4f63-86fd-633ac5c7d4e9.wav?sv=2019-07-07&sr=b&sig=Kg2iY5MD3yZ8LBwTBtRIq3%2F%2FYID0AG83iC6KxHZBB7Y%3D&st=2022-07-01T09:08:59Z&se=2122-06-07T09:13:59Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'bs-BA-VesnaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'fcf9b60e-1a55-4a08-841b-83085ecd3213',
    locale: 'uz-UZ',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (uz-UZ, SardorNeural)',
    properties: {
      DisplayName: 'Sardor',
      DisplayVoiceName: 'SardorNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Sardor',
      LocaleDescription: 'uz-UZ',
      OrderInVoiceListApi: '416',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'uz-UZ-SardorNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'uz-UZ',
      localeZH: '乌兹别克语(乌兹别克斯坦)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d3ebb75a-8352-4739-8cb9-f5f789c7ccef.wav?sv=2019-07-07&sr=b&sig=qCKbG2z1NdVYeIPdAvz1rgsD6eJhSH2TeIYETYljdH4%3D&st=2021-10-29T08:31:25Z&se=2121-10-05T08:36:25Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'uz-UZ-SardorNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '671a080c-7219-425c-8bfd-832ad1ab3d54',
    locale: 'th-TH',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (th-TH, PremwadeeNeural)',
    properties: {
      DisplayName: 'Premwadee',
      DisplayVoiceName: 'PremwadeeNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'เปรมวดี',
      LocaleDescription: 'Thai',
      OrderInVoiceListApi: '404',
      PreviewSentence:
        'สร้างแอปและบริการที่สื่อสารกับผู้ใช้ได้อย่างเป็นธรรมชาติ ซึ่งช่วยปรับปรุงการเข้าถึงและการใช้งาน',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'th-TH-PremwadeeNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'th-TH',
      localeZH: '泰语(泰国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/81279ff2-f187-4731-a5ea-ce5bd055e1c9.wav?sv=2019-07-07&sr=b&sig=Gr5GpRLS%2Fgfuf7h1jj6EFwnB2%2BlPzQLrOQlq5OmxwcU%3D&st=2021-10-29T08:28:21Z&se=2121-10-05T08:33:21Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'th-TH-PremwadeeNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '773fda1f-5210-4de3-aba4-833d434f0aa2',
    locale: 'fi-FI',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fi-FI, HarriNeural)',
    properties: {
      DisplayName: 'Harri',
      DisplayVoiceName: 'HarriNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Harri',
      LocaleDescription: 'Finnish',
      OrderInVoiceListApi: '230',
      PreviewSentence:
        'Kehitä luonnolisesti puhuvia sovelluksia ja palveluja, jotka parantavat käytettävyyttä ja saavutettavuutta.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fi-FI-HarriNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fi-FI',
      localeZH: '芬兰语(芬兰)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ac2f0948-c86d-4d07-aeec-5d4513205fd2.wav?sv=2019-07-07&sr=b&sig=yxGtpJ38OxqqdW6kBkVZc4SJoakioTFq1CVtVrDQmzo%3D&st=2021-10-29T08:26:51Z&se=2121-10-05T08:31:51Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fi-FI-HarriNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '383b7b38-8e3c-4ac1-a64c-8387ddee905e',
    locale: 'zh-CN-sichuan',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN-sichuan, YunxiNeural)',
    properties: {
      DisplayName: 'Yunxi',
      DisplayVoiceName: 'YunxiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: '云希',
      LocaleDescription: 'zh-CN-sichuan',
      OrderInVoiceListApi: '448',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-sichuan-YunxiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'zh-CN-sichuan',
      localeZH: '中文(西南官话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/50cfd6fc-a61d-40ca-a7e7-65f14bb56a92.wav?sv=2019-07-07&sr=b&sig=OvCA0F9kTxcUKb2QhZDdYiBq%2BAaHsVwwPpH7k3tze1Q%3D&st=2022-05-12T09:49:40Z&se=2122-04-18T09:54:40Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'zh-CN-sichuan-YunxiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '7f98d131-4cb7-4799-9f4b-83d4803b2fea',
    locale: 'yue-CN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (yue-CN, YunSongNeural)',
    properties: {
      DisplayName: 'YunSong',
      DisplayVoiceName: 'YunSongNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: '云松',
      LocaleDescription: 'yue-CN',
      OrderInVoiceListApi: '422',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'yue-CN-YunSongNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'yue-CN',
      localeZH: '中文(粤语，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/cf76c61d-cd0b-4909-a18d-7bfefa75012c.wav?sv=2021-08-06&st=2022-10-21T11%3A57%3A35Z&se=2122-09-27T12%3A02%3A35Z&sr=b&sp=rl&sig=rxHl0ozNf%2FSsi3nzmvZvGeVuvwjaJHphWQ8z9DDNQUA%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'yue-CN-YunSongNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '036ea582-19ad-4f59-b909-840af667a8a8',
    locale: 'gu-IN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (gu-IN, NiranjanNeural)',
    properties: {
      DisplayName: 'Niranjan',
      DisplayVoiceName: 'NiranjanNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'નિરંજન',
      LocaleDescription: 'gu-IN',
      OrderInVoiceListApi: '260',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'gu-IN-NiranjanNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'gu-IN',
      localeZH: '古吉拉特语(印度)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d5b66e3f-e5ef-42b0-b501-4bb503ee29da.wav?sv=2019-07-07&sr=b&sig=eZbDS70Aby4i9xWM9iiNIDEuCNHe45BI9SGzXo6VVCo%3D&st=2021-10-29T08:27:06Z&se=2121-10-05T08:32:06Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'gu-IN-NiranjanNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '0315db55-dbe4-4cfc-b180-848698f8a0d5',
    locale: 'lt-LT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (lt-LT, OnaNeural)',
    properties: {
      DisplayName: 'Ona',
      DisplayVoiceName: 'OnaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Ona',
      LocaleDescription: 'Lithuanian (Lithuania)',
      OrderInVoiceListApi: '318',
      PreviewSentence:
        'Garso turinio kūrimas leidžia vizualiai kontroliuoti kalbos atributus realiu laiku.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'lt-LT-OnaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'lt-LT',
      localeZH: '立陶宛语(立陶宛)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/02d90da4-d18f-4c4a-8076-70770c0f44d9.wav?sv=2019-07-07&sr=b&sig=RxqThoXqsK5ypk6MAss3qWRPLB0PFY1XPGPO%2BBxoHWE%3D&st=2021-10-29T08:27:26Z&se=2121-10-05T08:32:26Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'lt-LT-OnaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'b7de9dd4-e3ef-4e76-91f0-84c598201a5d',
    locale: 'sq-AL',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (sq-AL, AnilaNeural)',
    properties: {
      DisplayName: 'Anila',
      DisplayVoiceName: 'AnilaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Anila',
      LocaleDescription: 'sq-AL',
      OrderInVoiceListApi: '381',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'sq-AL-AnilaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'sq-AL',
      localeZH: '阿尔巴尼亚语(阿尔巴尼亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a7b3b213-17cd-44b7-a3d5-807d0678f5f8.wav?sv=2019-07-07&sr=b&sig=sw6jpP5%2F86NaPVZPEwLsgvmxb5rTwY97b1pqkYtsn34%3D&st=2022-07-01T09:10:41Z&se=2122-06-07T09:15:41Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'sq-AL-AnilaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'aa547b35-c8dd-44f5-82f2-84e053871500',
    locale: 'es-DO',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-DO, EmilioNeural)',
    properties: {
      DisplayName: 'Emilio',
      DisplayVoiceName: 'EmilioNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Emilio',
      LocaleDescription: 'es-DO',
      OrderInVoiceListApi: '165',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-DO-EmilioNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-DO',
      localeZH: '西班牙语(多米尼加共和国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2bd403dd-b761-458c-a4ed-18329dcf97ab.wav?sv=2019-07-07&sr=b&sig=cG8Usdjgsi3DKpJx%2FlG%2Faj0yA7aC7TA6UfxYJuDrxIY%3D&st=2021-10-29T08:30:36Z&se=2121-10-05T08:35:36Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-DO-EmilioNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '4b1dc409-f234-45cf-bda5-852fa95d0e5f',
    locale: 'en-GB',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-GB, SoniaNeural)',
    properties: {
      DisplayName: 'Sonia',
      DisplayVoiceName: 'SoniaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Sonia',
      LocaleDescription: 'English (UK)',
      OrderInVoiceListApi: '105',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-GB-SoniaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"cheerful":4,"sad":8,"Default":0}',
      VoiceStyleNames: 'cheerful,sad,Default',
      locale: 'en-GB',
      localeZH: '英语(英国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/90ad80b6-d25d-43df-9f42-e3ef54858147.wav?sv=2019-07-07&sr=b&sig=7VgjB0tHvIyfbLRf8OvdR1ZkW6o7kgL7Xuy8AXv4N9U%3D&st=2021-10-29T08:26:25Z&se=2121-10-05T08:31:25Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/26041f35-178c-4dd5-a3d4-048185800f04.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A28Z&se=2122-09-27T07%3A51%3A28Z&sr=b&sp=rl&sig=Strnkedt9A0yI4LfG5DYWzU9vhed8s9HKbLdZjdzQaI%3D',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e0702984-9bf0-4f8b-b35d-2a09a65c87b1.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A30Z&se=2122-09-27T07%3A51%3A30Z&sr=b&sp=rl&sig=5YcCzZd9agapkon3Ah%2FTFfoCEO6Yxbn8xhUZodkG1X8%3D',
          styleName: 'sad',
        },
      ],
    },
    shortName: 'en-GB-SoniaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'e46f4f45-4768-4b53-b385-85a8a36e1410',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, AshleyNeural)',
    properties: {
      DisplayName: 'Ashley',
      DisplayVoiceName: 'AshleyNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Ashley',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '132',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-AshleyNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/11e9d76b-bf93-4f41-9c42-44d3fcf1c393.wav?sv=2019-07-07&sr=b&sig=K3wVi%2FJH8o9pA%2FciMjNGtfZBKvQ2MMCWzv1zBOtFE%2BM%3D&st=2021-10-29T08:25:30Z&se=2121-10-05T08:30:30Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-US-AshleyNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [
      {
        name: 'All',
      },
      {
        name: 'Audiobook',
      },
      {
        name: 'Chat',
      },
    ],
    description: 'Mature and wise voice with rich emotions that is optimized for audio books.',
    id: 'cf45435f-b5f9-47be-850f-85bda9d6bb73',
    locale: 'zh-CN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, XiaoruiNeural)',
    properties: {
      AgeGroups: 'Senior',
      DisplayName: 'Xiaorui',
      DisplayVoiceName: 'XiaoruiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      IconFileEndpointWithSas:
        'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/3ee261ff-1255-409f-a4b1-a0e3f7feb47b?sv=2019-07-07&sr=b&sig=YkkZjowwmsMw3ElhUzlwaiAZAkvDXWDtSnjJKveo354%3D&st=2021-10-29T08:24:22Z&se=2121-10-05T08:29:22Z&sp=rl',
      IsTtsBreakEnabled: 'True',
      LocalName: '晓睿',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '430',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-XiaoruiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"calm":2,"fearful":3,"angry":7,"sad":8,"Default":0}',
      VoiceStyleNames: 'calm,fearful,angry,sad,Default',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/9a205bee-ec6d-4602-9777-565e18ee31a1.wav?sv=2019-07-07&sr=b&sig=pQaeH13AdWRmKYp2%2FZWCEfZirYBzxucpsPqEfIX%2B1is%3D&st=2021-10-29T08:28:51Z&se=2121-10-05T08:33:51Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/29673675-977a-4b16-83e1-6a4ec3a5e8c1.wav?sv=2019-07-07&sr=b&sig=jyA736N%2BNxZWK4F%2BUnxSgvphnxB9dR3A5CYq4CTxiZU%3D&st=2021-10-29T08:28:52Z&se=2121-10-05T08:33:52Z&sp=rl',
          styleName: 'calm',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/966b523f-a73a-4f44-8876-c6986841f3bf.wav?sv=2019-07-07&sr=b&sig=a%2F8SmsrI%2F3%2FB0nSGRKl7Q7mH2hM5KDSYkratjeUuTB4%3D&st=2021-10-29T08:28:53Z&se=2121-10-05T08:33:53Z&sp=rl',
          styleName: 'fearful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/eeb7a254-cc10-4d51-ba72-71b6c5468d9c.wav?sv=2019-07-07&sr=b&sig=Chgw8SDGzUoORy1xVVvZ2bQ%2FF5kCeAdPMh2B7uMB8z4%3D&st=2021-10-29T08:28:54Z&se=2121-10-05T08:33:54Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/3e18a2ac-fea4-4e74-ae1f-a65a2bb8084e.wav?sv=2019-07-07&sr=b&sig=BYGet%2BPQ5JvFYczj%2BMqzj8MR0ahdpxaN6AYhGr%2FNZ1s%3D&st=2021-10-29T08:28:55Z&se=2121-10-05T08:33:55Z&sp=rl',
          styleName: 'sad',
        },
      ],
    },
    shortName: 'zh-CN-XiaoruiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '2a069bfe-6b35-47e1-a103-86506ec659fc',
    locale: 'ar-KW',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-KW, FahedNeural)',
    properties: {
      DisplayName: 'Fahed',
      DisplayVoiceName: 'FahedNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'فهد',
      LocaleDescription: 'ar-KW',
      OrderInVoiceListApi: '17',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-KW-FahedNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-KW',
      localeZH: '阿拉伯语(科威特)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/dc0228b8-528c-4d7b-be9b-561642b0fbb8.wav?sv=2019-07-07&sr=b&sig=6hkgcMfsY%2BEPcAEhrBBnb58tf9bte5qAWGDijcFmYHE%3D&st=2021-10-29T08:30:07Z&se=2121-10-05T08:35:07Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-KW-FahedNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '5791793c-2e22-4adb-a56c-86b6bae8c877',
    locale: 'ar-LY',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-LY, OmarNeural)',
    properties: {
      DisplayName: 'Omar',
      DisplayVoiceName: 'OmarNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'أحمد',
      LocaleDescription: 'ar-LY',
      OrderInVoiceListApi: '22',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-LY-OmarNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-LY',
      localeZH: '阿拉伯语(利比亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6bae8062-57db-4e10-adfd-930efc0c953c.wav?sv=2019-07-07&sr=b&sig=VkGBY7eIHyWA8PM3sO7EnKFGqZUKVxb2N9uof%2FM8WFs%3D&st=2021-10-29T08:30:10Z&se=2121-10-05T08:35:10Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-LY-OmarNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '7dde73d4-b5f8-418d-a8ce-87944c7d71bd',
    locale: 'zu-ZA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zu-ZA, ThandoNeural)',
    properties: {
      DisplayName: 'Thando',
      DisplayVoiceName: 'ThandoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Thando',
      LocaleDescription: 'zu-ZA',
      OrderInVoiceListApi: '455',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zu-ZA-ThandoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'zu-ZA',
      localeZH: '祖鲁语(南非)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/316dedb6-cc30-4933-a5fc-97d370c1f46f.wav?sv=2019-07-07&sr=b&sig=XTkMMi1389WvMCFkS8n890wfLTUvC%2FXkgHWveVBNQP0%3D&st=2021-10-29T08:31:27Z&se=2121-10-05T08:36:27Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'zu-ZA-ThandoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'e86de0ab-f6ea-44dd-aa80-879755f632a6',
    locale: 'es-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-ES, SaulNeural)',
    properties: {
      DisplayName: 'Saul',
      DisplayVoiceName: 'SaulNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Saul',
      LocaleDescription: 'Spanish (Spain)',
      OrderInVoiceListApi: '179',
      PreviewSentence:
        'Crea aplicaciones y servicios que hablan de forma natural a los usuarios, y que mejoran la accesibilidad y la facilidad de uso.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-ES-SaulNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-ES',
      localeZH: '西班牙语(西班牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/de9cf74c-8a2d-498b-b7c3-e672faab28f9.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A52Z&se=2122-09-27T07%3A51%3A52Z&sr=b&sp=rl&sig=j%2FfVoPV%2FkQbbRynPoVGg%2BM6gGNape%2Feq8Veu5I0DPo0%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-ES-SaulNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '1e298d8e-ec77-48f6-a581-8944355f4c9f',
    locale: 'es-MX',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-MX, CarlotaNeural)',
    properties: {
      DisplayName: 'Carlota',
      DisplayVoiceName: 'CarlotaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Carlota',
      LocaleDescription: 'Spanish (Mexico)',
      OrderInVoiceListApi: '192',
      PreviewSentence:
        'Crea apps y servicios que hablen de forma natural con los usuarios, mejorando la accesibilidad y la usabilidad.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-MX-CarlotaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-MX',
      localeZH: '西班牙语(墨西哥)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2cafbcb2-71e0-41e4-84ec-de07fd360f5a.wav?sv=2019-07-07&sr=b&sig=ZQkddCB58kA9l%2FT5bn9t3Ral9yQfZlPJxWMEDax4vZs%3D&st=2022-07-01T09:09:06Z&se=2122-06-07T09:14:06Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-MX-CarlotaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '45565610-2f20-4ea9-8218-8a33aea63676',
    locale: 'ar-AE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-AE, FatimaNeural)',
    properties: {
      DisplayName: 'Fatima',
      DisplayVoiceName: 'FatimaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'فاطمة',
      LocaleDescription: 'ar-AE',
      OrderInVoiceListApi: '5',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-AE-FatimaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-AE',
      localeZH: '阿拉伯语(阿拉伯联合酋长国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/8b765698-8e2f-43c5-a8de-c933d56725e2.wav?sv=2019-07-07&sr=b&sig=4034B79%2FR9V2z3P0J1CwUzfHZ2SIgcauP20GmBufHs4%3D&st=2021-10-29T08:29:57Z&se=2121-10-05T08:34:57Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-AE-FatimaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '8271d147-3d4c-4d0d-901c-8a3c3129f0bd',
    locale: 'lo-LA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (lo-LA, KeomanyNeural)',
    properties: {
      DisplayName: 'Keomany',
      DisplayVoiceName: 'KeomanyNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'ແກ້ວມະນີ',
      LocaleDescription: 'lo-LA',
      OrderInVoiceListApi: '316',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'lo-LA-KeomanyNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'lo-LA',
      localeZH: '老挝语(老挝) ',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/198dff99-8ccf-4b64-8ab2-701e65f58081.wav?sv=2019-07-07&sr=b&sig=pnOHJnv59MAcAfKyFudDHqVgOO5FmxOq46DKW%2BnILBk%3D&st=2022-01-19T09:02:00Z&se=2121-12-26T09:07:00Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'lo-LA-KeomanyNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '1055210c-6d26-431b-b2ee-8a3e41e3c8c7',
    locale: 'mn-MN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (mn-MN, BataaNeural)',
    properties: {
      DisplayName: 'Bataa',
      DisplayVoiceName: 'BataaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Батаа',
      LocaleDescription: 'mn-MN',
      OrderInVoiceListApi: '325',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'mn-MN-BataaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'mn-MN',
      localeZH: '蒙古语(蒙古)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/aefef278-68fe-4767-bbf9-1199d1fc4998.wav?sv=2019-07-07&sr=b&sig=W9B0N8M7vI3lbf%2Fu0YobXB7r9%2BFdNNjmzyUEbANQDDk%3D&st=2022-07-01T09:10:02Z&se=2122-06-07T09:15:02Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'mn-MN-BataaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '3cf2978c-83d5-4b58-b793-8b835b5f0b5f',
    locale: 'de-AT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (de-AT, IngridNeural)',
    properties: {
      DisplayName: 'Ingrid',
      DisplayVoiceName: 'IngridNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Ingrid',
      LocaleDescription: 'German (Austria)',
      OrderInVoiceListApi: '56',
      PreviewSentence:
        'Erstellen Sie Apps und Dienste, die auf natürliche Art mit Nutzern sprechen, und dadurch Zugänglichkeit und Benutzerfreundlichkeit verbessern.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'de-AT-IngridNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'de-AT',
      localeZH: '德语(奥地利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2d785dbc-6ae5-4f50-820b-4ef3409b7884.wav?sv=2019-07-07&sr=b&sig=a3HrwRcgBpgyqH9qh7DmzNnskYcj8iamjswr1NC0b0o%3D&st=2021-10-29T08:26:10Z&se=2121-10-05T08:31:10Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'de-AT-IngridNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '95257341-5e70-4e75-a634-8c4c9b942fd9',
    locale: 'es-CU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-CU, BelkysNeural)',
    properties: {
      DisplayName: 'Belkys',
      DisplayVoiceName: 'BelkysNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Belkys',
      LocaleDescription: 'es-CU',
      OrderInVoiceListApi: '163',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-CU-BelkysNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-CU',
      localeZH: '西班牙语(古巴)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/49e6a9fd-1e4a-40dc-9c14-0f18c34dd22d.wav?sv=2019-07-07&sr=b&sig=ANefeTLNrc3T3Shm8b5iNSegQ5lPkxhouoa0r%2FC0rg8%3D&st=2021-10-29T08:30:34Z&se=2121-10-05T08:35:34Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-CU-BelkysNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'e11668e8-1d34-4470-994a-8d10d2695e09',
    locale: 'af-ZA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (af-ZA, AdriNeural)',
    properties: {
      DisplayName: 'Adri',
      DisplayVoiceName: 'AdriNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Adri',
      LocaleDescription: 'af-ZA',
      OrderInVoiceListApi: '1',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'af-ZA-AdriNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'af-ZA',
      localeZH: '南非荷兰语(南非)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/8dac170e-2276-44c9-b3b8-efc5a1422aa7.wav?sv=2019-07-07&sr=b&sig=qdcKgZom8yECuEZcNb9wynoOcVxL2Hfre2Ju45jbDPg%3D&st=2021-10-29T08:29:54Z&se=2121-10-05T08:34:54Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'af-ZA-AdriNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '9156f60c-d3fe-40fb-9346-8d6ae6c8da04',
    locale: 'de-DE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (de-DE, ElkeNeural)',
    properties: {
      DisplayName: 'Elke',
      DisplayVoiceName: 'ElkeNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Elke',
      LocaleDescription: 'German (Germany)',
      OrderInVoiceListApi: '64',
      PreviewSentence:
        'Erstellen Sie Apps und Dienste, die auf natürliche Art mit Nutzern sprechen, und dadurch Zugänglichkeit und Benutzerfreundlichkeit verbessern.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'de-DE-ElkeNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'de-DE',
      localeZH: '德语(德国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/73d16ddf-bd65-4642-a8a7-b89ddb215ced.wav?sv=2019-07-07&sr=b&sig=vvRAAKSPqgerq1p3Ioslk%2BmxmwOLJnjdMhMUMFUOxrQ%3D&st=2022-01-19T09:00:27Z&se=2121-12-26T09:05:27Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'de-DE-ElkeNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '4c21507e-7a5d-4071-81a4-8da05a039a60',
    locale: 'ar-LY',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-LY, ImanNeural)',
    properties: {
      DisplayName: 'Iman',
      DisplayVoiceName: 'ImanNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'إيمان',
      LocaleDescription: 'ar-LY',
      OrderInVoiceListApi: '21',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-LY-ImanNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-LY',
      localeZH: '阿拉伯语(利比亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2846cb66-e641-4fb1-97bf-217a96723f90.wav?sv=2019-07-07&sr=b&sig=yF1hEgSfE6oC6Al2Md6NNGa%2FY7IHin%2FEPbzpTfNNwPA%3D&st=2021-10-29T08:30:09Z&se=2121-10-05T08:35:09Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-LY-ImanNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '5412d6b7-554c-40ab-9892-8e2c80386b41',
    locale: 'fil-PH',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fil-PH, AngeloNeural)',
    properties: {
      DisplayName: 'Angelo',
      DisplayVoiceName: 'AngeloNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Angelo',
      LocaleDescription: 'fil-PH',
      OrderInVoiceListApi: '232',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fil-PH-AngeloNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fil-PH',
      localeZH: '菲律宾语(菲律宾)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ce45595b-d8f4-4130-810a-9aa253619d89.wav?sv=2019-07-07&sr=b&sig=yVBWxbcmIFkBUjEqDfKEGtMTfFdnV3tt3BDubMuHnt8%3D&st=2021-10-29T08:31:03Z&se=2121-10-05T08:36:03Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fil-PH-AngeloNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'c3232411-1b92-4959-a799-8e2df59186eb',
    locale: 'en-PH',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-PH, RosaNeural)',
    properties: {
      DisplayName: 'Rosa',
      DisplayVoiceName: 'RosaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Rosa',
      LocaleDescription: 'English (Philippines)',
      OrderInVoiceListApi: '121',
      PreviewSentence:
        'Audio Content Creation enables you to visually control speech attributes in real-time.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-PH-RosaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-PH',
      localeZH: '英语(菲律宾)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d6fbef23-ab20-4d14-a289-4291c8238d1e.wav?sv=2019-07-07&sr=b&sig=3JJfTvETpd1EGWRXjDhXe0bVtP%2FfeAHIwpQwwIU%2FG1g%3D&st=2021-10-29T08:26:35Z&se=2121-10-05T08:31:35Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-PH-RosaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '2e6ce445-6c53-46d9-b8db-8fecb87dade2',
    locale: 'pt-BR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pt-BR, JulioNeural)',
    properties: {
      DisplayName: 'Julio',
      DisplayVoiceName: 'JulioNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Julio',
      LocaleDescription: 'Portuguese (Brazil)',
      OrderInVoiceListApi: '358',
      PreviewSentence:
        'Desenvolver aplicativos e serviços que conversem naturalmente com os usuários, melhorando a acessibilidade e usabilidade.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pt-BR-JulioNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'pt-BR',
      localeZH: '葡萄牙语(巴西)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/68567018-c9b0-45e4-88da-359067d038f4.wav?sv=2019-07-07&sr=b&sig=8YhvSCEMDztN29NbZ4vNR4YxZqQfB0NEktygWSUawhc%3D&st=2022-07-01T09:10:24Z&se=2122-06-07T09:15:24Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'pt-BR-JulioNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '2c82b4c7-1b43-46f4-a021-9038c848dce9',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, AnaNeural)',
    properties: {
      DisplayName: 'Ana',
      DisplayVoiceName: 'AnaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Ana',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '130',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-AnaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fc970c36-b330-470a-a367-5cd674474f18.wav?sv=2019-07-07&sr=b&sig=Dyhzua4F0iwrpeRQiln8pgU2%2BPI1mH%2FarbMcSKDZBWs%3D&st=2021-10-29T08:25:14Z&se=2121-10-05T08:30:14Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-US-AnaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '7db746e5-4da7-41da-8c5a-906f244effb5',
    locale: 'en-IE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-IE, ConnorNeural)',
    properties: {
      DisplayName: 'Connor',
      DisplayVoiceName: 'ConnorNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Connor',
      LocaleDescription: 'English (Ireland)',
      OrderInVoiceListApi: '110',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-IE-ConnorNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-IE',
      localeZH: '英语(爱尔兰)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d989d27a-6f8b-4771-89d8-73dbb6e6eb3e.wav?sv=2019-07-07&sr=b&sig=8bcfsjZsJKqAVn6HF9ee80KBxtANX1NlKeo0C40xHRQ%3D&st=2021-10-29T08:26:28Z&se=2121-10-05T08:31:28Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-IE-ConnorNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '28faf9ae-a1b3-4c82-a002-914e73d7e232',
    locale: 'sk-SK',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (sk-SK, ViktoriaNeural)',
    properties: {
      DisplayName: 'Viktoria',
      DisplayVoiceName: 'ViktoriaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Viktória',
      LocaleDescription: 'Slovak',
      OrderInVoiceListApi: '376',
      PreviewSentence:
        'Vytvárajú aplikácie a služby, ktoré prirodzene komunikujú s užívateľmi, čím sa zlepšuje dostupnosť a využiteľnosť.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'sk-SK-ViktoriaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'sk-SK',
      localeZH: '斯洛伐克语(斯洛伐克)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5462d2b5-d9b8-4803-896c-e0f8ef79cd1b.wav?sv=2019-07-07&sr=b&sig=mJOj3juPRy8uyD1gi3Uz1XiF2A0VsJifz5lMu74MNrc%3D&st=2021-10-29T08:28:06Z&se=2121-10-05T08:33:06Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'sk-SK-ViktoriaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '44ee81ca-a1f8-4c3d-9e1e-921b1d005970',
    locale: 'ko-KR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ko-KR, BongJinNeural)',
    properties: {
      DisplayName: 'BongJin',
      DisplayVoiceName: 'BongJinNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: '봉진',
      LocaleDescription: 'Korean',
      OrderInVoiceListApi: '309',
      PreviewSentence:
        '사용자들에게 자연스럽게 이야기하는 앱과 서비스를 만들며 접근성과 사용성을 개선시킵니다.',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ko-KR-BongJinNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ko-KR',
      localeZH: '韩语(韩国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/29b9ce12-d811-49b3-9c06-73ae6f134a24.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A35Z&se=2122-09-27T07%3A52%3A35Z&sr=b&sp=rl&sig=hH3zuR%2Fe%2B2yUA5vDzRuW7%2BeMREkniEeXGUCY6S%2BMtWU%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ko-KR-BongJinNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '6b2759f1-56db-4281-bde8-9235313dea7c',
    locale: 'sv-SE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (sv-SE, SofieNeural)',
    properties: {
      DisplayName: 'Sofie',
      DisplayVoiceName: 'SofieNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Sofie',
      LocaleDescription: 'Swedish',
      OrderInVoiceListApi: '387',
      PreviewSentence:
        'Bygg appar och tjänster som talar naturligt till användarna, och förbättrar tillgänglighet och användbarhet.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'sv-SE-SofieNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'sv-SE',
      localeZH: '瑞典语(瑞典)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a6133923-5cb6-4d81-b9f4-3c7b2d1c873a.wav?sv=2019-07-07&sr=b&sig=H93zej0g%2B0YKmIlOMngAZqeCzFtrKoFvIoTIvAWWYuA%3D&st=2021-10-29T08:28:12Z&se=2121-10-05T08:33:12Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'sv-SE-SofieNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '67e7d63e-f225-45e4-b75a-9266364fadb1',
    locale: 'et-EE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (et-EE, KertNeural)',
    properties: {
      DisplayName: 'Kert',
      DisplayVoiceName: 'KertNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Kert',
      LocaleDescription: 'Estonian (Estonia)',
      OrderInVoiceListApi: '224',
      PreviewSentence:
        'Audio Content Creation võimaldab kõne atribuute reaalajas visuaalselt kontrollida.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'et-EE-KertNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'et-EE',
      localeZH: '爱沙尼亚语(爱沙尼亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/25011293-872a-4ead-a43f-9cc969242377.wav?sv=2019-07-07&sr=b&sig=HtZ8NUqud7HCPJveiZYsB9FzvC5xJ7eDRiLBcxolXtQ%3D&st=2021-10-29T08:26:50Z&se=2121-10-05T08:31:50Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'et-EE-KertNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '16fd5e4d-8f1b-4f15-ab8b-92e31b1057a0',
    locale: 'es-GT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-GT, MartaNeural)',
    properties: {
      DisplayName: 'Marta',
      DisplayVoiceName: 'MartaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Marta',
      LocaleDescription: 'es-GT',
      OrderInVoiceListApi: '187',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-GT-MartaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-GT',
      localeZH: '西班牙语(危地马拉)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2fdb7a4a-3307-4af6-85ac-35a80fbf5464.wav?sv=2019-07-07&sr=b&sig=axim%2BAGCbgtOQEdf2vNnuFWvQQikHrJrCtdhxfioet0%3D&st=2021-10-29T08:30:43Z&se=2121-10-05T08:35:43Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-GT-MartaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'd9893835-d9d8-4485-ad27-930b275a733c',
    locale: 'am-ET',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (am-ET, MekdesNeural)',
    properties: {
      DisplayName: 'Mekdes',
      DisplayVoiceName: 'MekdesNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'መቅደስ',
      LocaleDescription: 'am-ET',
      OrderInVoiceListApi: '4',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'am-ET-MekdesNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'am-ET',
      localeZH: '阿姆哈拉语(埃塞俄比亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/44ae7c7c-fd1d-4a0c-9450-562e725a11e7.wav?sv=2019-07-07&sr=b&sig=3XnccvXBx%2Bq1ELE7OavWGV9tmawr9R7iXL1PnZrEZOE%3D&st=2021-10-29T08:29:56Z&se=2121-10-05T08:34:56Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'am-ET-MekdesNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '09b38d23-ad18-4617-969a-9343d9021cb2',
    locale: 'en-US',
    masterpieces: [
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6f014d26-4cdc-4846-8149-fea5688f4fdd.wav?sv=2021-08-06&st=2022-10-11T09%3A55%3A53Z&se=2122-09-17T10%3A00%3A53Z&sr=b&sp=rl&sig=UVYH8aw7E43F5pGDSvkJmXLSCT5xNcAJzxkBU3jjHmk%3D',
        categories: ['Audiobook'],
        title: 'AudioBook',
        titleLocalizationId: 'AudioBook',
      },
    ],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, SteffanNeural)',
    properties: {
      DisplayName: 'Steffan',
      DisplayVoiceName: 'SteffanNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Steffan',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '147',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-SteffanNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4d8beb7e-4f4b-4524-a2b2-03767469ef14.wav?sv=2021-08-06&st=2022-10-11T09%3A55%3A50Z&se=2122-09-17T10%3A00%3A50Z&sr=b&sp=rl&sig=BdJiKHIBt%2FNpfgIGgZYyA4aOuCyFGDTol%2FCvfvC%2FESU%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-US-SteffanNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'c08e4841-7cad-46b1-b4a7-93e996c7f5f9',
    locale: 'en-GB',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-GB, BellaNeural)',
    properties: {
      DisplayName: 'Bella',
      DisplayVoiceName: 'BellaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Bella',
      LocaleDescription: 'English (UK)',
      OrderInVoiceListApi: '96',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-GB-BellaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-GB',
      localeZH: '英语(英国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e8522ceb-7bc2-4ad4-ac70-f2120135edff.wav?sv=2019-07-07&sr=b&sig=AUND%2FfBPmPPLpN85jGgp5yZiCye%2F%2BbYo3ueALLFVhpM%3D&st=2022-01-19T09:00:56Z&se=2121-12-26T09:05:56Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-GB-BellaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '2d9c6678-091a-48fb-bae9-940ffc4a138c',
    locale: 'es-MX',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-MX, BeatrizNeural)',
    properties: {
      DisplayName: 'Beatriz',
      DisplayVoiceName: 'BeatrizNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Beatriz',
      LocaleDescription: 'Spanish (Mexico)',
      OrderInVoiceListApi: '190',
      PreviewSentence:
        'Crea apps y servicios que hablen de forma natural con los usuarios, mejorando la accesibilidad y la usabilidad.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-MX-BeatrizNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-MX',
      localeZH: '西班牙语(墨西哥)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/bca9bdd2-31b6-4db2-bb28-3705c416cae2.wav?sv=2019-07-07&sr=b&sig=ziYgItCUIojuPfriR1%2BlgLObBNEHLqI%2BrP2879jbgOE%3D&st=2022-07-01T09:09:02Z&se=2122-06-07T09:14:02Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-MX-BeatrizNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '869464e6-c35a-4f15-ab1f-94b415772a37',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, AIGenerate2Neural)',
    properties: {
      DisplayName: 'AIGenerate2',
      DisplayVoiceName: 'AIGenerate2Neural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'AIGenerate2',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '150',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-AIGenerate2Neural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/bebccb1e-2c65-4a68-9c95-f95ab56a842e.wav?sv=2019-07-07&sr=b&sig=PBxUYy%2BCprNYD11L2G2r3WAZU3Hbu39ja6tpwTaSCIQ%3D&st=2022-08-30T16:24:59Z&se=2122-08-06T16:29:59Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-US-AIGenerate2Neural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'f558e661-1dad-4bad-a08d-952619b409b7',
    locale: 'ml-IN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ml-IN, MidhunNeural)',
    properties: {
      DisplayName: 'Midhun',
      DisplayVoiceName: 'MidhunNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'മിഥുൻ',
      LocaleDescription: 'ml-IN',
      OrderInVoiceListApi: '323',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ml-IN-MidhunNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ml-IN',
      localeZH: '马拉雅拉姆语(印度)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0f81bfce-10a6-4e9e-8407-8d15bca2cb8f.wav?sv=2019-07-07&sr=b&sig=wmHJMGQLEmZUp7WJ1ep%2F%2FGcTSeW9v1VaAndLDPSt87M%3D&st=2022-01-19T09:02:06Z&se=2121-12-26T09:07:06Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ml-IN-MidhunNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'd88b672d-537b-4139-aab5-95e6d24cf777',
    locale: 'ta-IN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ta-IN, PallaviNeural)',
    properties: {
      DisplayName: 'Pallavi',
      DisplayVoiceName: 'PallaviNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'பல்லவி',
      LocaleDescription: 'Tamil (India)',
      OrderInVoiceListApi: '394',
      PreviewSentence:
        'பயனர்களிடம் இயற்கையாக பேசும் பயன்பாடுகள் மற்றும் சேவைகளை உருவாகுதல், இது அணுகல்தன்மை மற்றும் பயன்பாட்டினை மேம்படுத்தும்.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ta-IN-PallaviNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ta-IN',
      localeZH: '泰米尔语(印度)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/48e60502-9864-4c14-a77b-8f5ea915a44f.wav?sv=2019-07-07&sr=b&sig=huhA8AXrqpFW%2Fq6GemtZRHQvV0iab2KmkjiHkj56p2M%3D&st=2021-10-29T08:28:15Z&se=2121-10-05T08:33:15Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ta-IN-PallaviNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'f7a54768-ea66-4a86-a2d5-97709ec8469f',
    locale: 'hi-IN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (hi-IN, SwaraNeural)',
    properties: {
      DisplayName: 'Swara',
      DisplayVoiceName: 'SwaraNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'स्वरा',
      LocaleDescription: 'Hindi (India)',
      OrderInVoiceListApi: '264',
      PreviewSentence:
        'ऐसे ऐप और सेवाओं का निर्माण करें जो उपयोगकर्ताओं से आम बोलचाल की भाषा में बात करे, जिससे सुगमता और उपयोगिता बढ़े।',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'hi-IN-SwaraNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'hi-IN',
      localeZH: '印地语(印度)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5325f2e4-bf1f-493c-bd2c-308b25cb309d.wav?sv=2019-07-07&sr=b&sig=LZXsPjcco%2FEjtY1uf8cTvGl7ybHz7uFq1Z%2F60tyeFdM%3D&st=2021-10-29T08:27:10Z&se=2121-10-05T08:32:10Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'hi-IN-SwaraNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'ecc04a6b-9488-43da-a0a0-979ea15ce81e',
    locale: 'pt-BR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pt-BR, ManuelaNeural)',
    properties: {
      DisplayName: 'Manuela',
      DisplayVoiceName: 'ManuelaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Manuela',
      LocaleDescription: 'Portuguese (Brazil)',
      OrderInVoiceListApi: '361',
      PreviewSentence:
        'Desenvolver aplicativos e serviços que conversem naturalmente com os usuários, melhorando a acessibilidade e usabilidade.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pt-BR-ManuelaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'pt-BR',
      localeZH: '葡萄牙语(巴西)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/bac7c1b0-a430-4777-989c-bd5074077b75.wav?sv=2019-07-07&sr=b&sig=A3xzY8DPUiqi3RKmD%2B66A8BM2EtLtS%2B35OaY66Wxz68%3D&st=2022-07-01T09:10:31Z&se=2122-06-07T09:15:31Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'pt-BR-ManuelaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'dbb2c827-a5e7-439a-ba4d-97e3a4ee5b81',
    locale: 'ar-KW',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-KW, NouraNeural)',
    properties: {
      DisplayName: 'Noura',
      DisplayVoiceName: 'NouraNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'نورا',
      LocaleDescription: 'ar-KW',
      OrderInVoiceListApi: '18',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-KW-NouraNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-KW',
      localeZH: '阿拉伯语(科威特)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/de29b460-0bc0-4f27-8ebd-9436cb83a42e.wav?sv=2019-07-07&sr=b&sig=jBTayCbm0PqGD5jEG1Nc8XQbuxzLDnOoz7Yj5njPBU4%3D&st=2021-10-29T08:30:08Z&se=2121-10-05T08:35:08Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-KW-NouraNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'f4528154-ee1b-46da-997d-98d08c75000b',
    locale: 'sw-TZ',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (sw-TZ, DaudiNeural)',
    properties: {
      DisplayName: 'Daudi',
      DisplayVoiceName: 'DaudiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Daudi',
      LocaleDescription: 'sw-TZ',
      OrderInVoiceListApi: '392',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'sw-TZ-DaudiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'sw-TZ',
      localeZH: '斯瓦希里语(坦桑尼亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4f3c24d6-3c13-444a-922f-34125f933d43.wav?sv=2019-07-07&sr=b&sig=c4su9oudVJCoAUotKufy8HHtvxA4JDpk%2FyVoTvsQyGU%3D&st=2021-10-29T08:31:17Z&se=2121-10-05T08:36:17Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'sw-TZ-DaudiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'f6c86801-3b7f-4cc3-abd7-996a740183fb',
    locale: 'en-CA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-CA, ClaraNeural)',
    properties: {
      DisplayName: 'Clara',
      DisplayVoiceName: 'ClaraNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Clara',
      LocaleDescription: 'English (Canada)',
      OrderInVoiceListApi: '91',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-CA-ClaraNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-CA',
      localeZH: '英语(加拿大)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5717085c-c4c5-4bae-add6-0942edbcf166.wav?sv=2019-07-07&sr=b&sig=iCuuo%2BiAcUR0HZVN5hf3eiHTdUm%2BEW%2FCmzd8xuQlz14%3D&st=2021-10-29T08:26:20Z&se=2121-10-05T08:31:20Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-CA-ClaraNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '5ddfa17b-a678-4f98-9915-9a0691680e44',
    locale: 'en-GB',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-GB, ThomasNeural)',
    properties: {
      DisplayName: 'Thomas',
      DisplayVoiceName: 'ThomasNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Thomas',
      LocaleDescription: 'English (UK)',
      OrderInVoiceListApi: '106',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-GB-ThomasNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-GB',
      localeZH: '英语(英国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b70b61f0-b991-435c-b2c3-a1cc51149c07.wav?sv=2019-07-07&sr=b&sig=WmzCiZkpqwnxyGwXQFwdZAphptuFKZVq6zC7oxIU5RM%3D&st=2022-01-19T09:01:15Z&se=2121-12-26T09:06:15Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-GB-ThomasNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '24eb77d1-92ad-4a88-a89d-9a3f741091fc',
    locale: 'lv-LV',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (lv-LV, EveritaNeural)',
    properties: {
      DisplayName: 'Everita',
      DisplayVoiceName: 'EveritaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Everita',
      LocaleDescription: 'Latvian (Latvia)',
      OrderInVoiceListApi: '319',
      PreviewSentence:
        'Balss Satura Izveide ļauj jums vizuāli kontrolēt runas atribūtus reālajā laikā.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'lv-LV-EveritaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'lv-LV',
      localeZH: '拉脱维亚语(拉脱维亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/308d7f89-d506-410d-80f2-813428428660.wav?sv=2019-07-07&sr=b&sig=i4gEjyfpJ15ezhJ2IdcxnVvzuLpu5K6q7kH3ZMOw%2Bug%3D&st=2021-10-29T08:27:27Z&se=2121-10-05T08:32:27Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'lv-LV-EveritaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '32e36049-31f4-4924-b153-9ae203367d08',
    locale: 'ar-BH',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-BH, LailaNeural)',
    properties: {
      DisplayName: 'Laila',
      DisplayVoiceName: 'LailaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'ليلى',
      LocaleDescription: 'ar-BH',
      OrderInVoiceListApi: '8',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-BH-LailaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-BH',
      localeZH: '阿拉伯语(巴林)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/76b21d5b-227f-49a1-9842-a310c745c644.wav?sv=2019-07-07&sr=b&sig=gtb7okwaYopjK97zKmkG5b7ng%2F9xi3pc3XqVaoN6sjE%3D&st=2021-10-29T08:30:00Z&se=2121-10-05T08:35:00Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-BH-LailaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '94d6a384-5800-4fba-ba4f-9b19e8904d65',
    locale: 'fr-FR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-FR, EloiseNeural)',
    properties: {
      DisplayName: 'Eloise',
      DisplayVoiceName: 'EloiseNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Eloise',
      LocaleDescription: 'French (France)',
      OrderInVoiceListApi: '246',
      PreviewSentence:
        "Construire des applications et services qui communiquent naturellement avec l'utilisateur, en améliorant l’accessibilité et l'ergonomie",
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-FR-EloiseNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fr-FR',
      localeZH: '法语(法国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/188c6547-c2ea-4fe8-8fda-6fbf246e333b.wav?sv=2019-07-07&sr=b&sig=CzqRFazZdPEYoXWEIVDWO%2B%2FQLSGcwiDEqZ9MbuIcQqI%3D&st=2022-01-19T09:01:28Z&se=2121-12-26T09:06:28Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fr-FR-EloiseNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'b948cf84-935c-47a9-a943-9c7fdb42cde9',
    locale: 'es-AR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-AR, TomasNeural)',
    properties: {
      DisplayName: 'Tomas',
      DisplayVoiceName: 'TomasNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Tomas',
      LocaleDescription: 'es-AR',
      OrderInVoiceListApi: '154',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-AR-TomasNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-AR',
      localeZH: '西班牙语(阿根廷)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f70d05b7-05e1-4362-a172-ac5a70e69b10.wav?sv=2019-07-07&sr=b&sig=anI%2BCGMi6h1Vc0hWIQM1OcFYvAyHdYbA1pdMB%2Fbx%2FaE%3D&st=2021-10-29T08:26:41Z&se=2121-10-05T08:31:41Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-AR-TomasNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '48416c15-6a97-4f7d-adeb-9ca25f4f4d5b',
    locale: 'ar-TN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-TN, HediNeural)',
    properties: {
      DisplayName: 'Hedi',
      DisplayVoiceName: 'HediNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'هادي',
      LocaleDescription: 'ar-TN',
      OrderInVoiceListApi: '33',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-TN-HediNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-TN',
      localeZH: '阿拉伯语(突尼斯)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a576c062-c540-4e29-bef8-76a97624ad91.wav?sv=2019-07-07&sr=b&sig=6Y6fIeUriEfR8cxum%2BNwzLgJ9cqnWyR1y4qHbpyatlk%3D&st=2021-10-29T08:30:17Z&se=2121-10-05T08:35:17Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-TN-HediNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '2ddbca91-11ca-43ba-aa09-9d20346f7a5a',
    locale: 'da-DK',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (da-DK, ChristelNeural)',
    properties: {
      DisplayName: 'Christel',
      DisplayVoiceName: 'ChristelNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Christel',
      LocaleDescription: 'Danish',
      OrderInVoiceListApi: '54',
      PreviewSentence:
        'Lav apps og tjenester, der taler naturligt til brugere, forbedrer tilgængelighed og brugervenlighed.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'da-DK-ChristelNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'da-DK',
      localeZH: '丹麦语(丹麦)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/78249d9a-dd2b-4148-b78d-386d4c52a2c8.wav?sv=2019-07-07&sr=b&sig=McuAUYLuQiIdLOT35hmHjHRLXLfrvcl%2BcDtT6SBa8wA%3D&st=2021-10-29T08:26:08Z&se=2121-10-05T08:31:08Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'da-DK-ChristelNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'd045ab76-5577-40b1-862b-a05fc5edf4ac',
    locale: 'uk-UA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (uk-UA, OstapNeural)',
    properties: {
      DisplayName: 'Ostap',
      DisplayVoiceName: 'OstapNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Остап',
      LocaleDescription: 'Ukrainian (Ukraine)',
      OrderInVoiceListApi: '409',
      PreviewSentence:
        'Створення аудіовмісту дозволяє візуально контролювати мовні атрибути в реальному часі.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'uk-UA-OstapNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'uk-UA',
      localeZH: '乌克兰语(乌克兰)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ab05c7ab-b79c-4aad-8466-2db2e5db03e9.wav?sv=2019-07-07&sr=b&sig=1%2BD%2FDbWDMKKm5N2EzF8r1InK836o3IMmZ%2FVUsGgioCE%3D&st=2021-10-29T08:28:25Z&se=2121-10-05T08:33:25Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'uk-UA-OstapNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'a0b18d81-2da2-4896-b169-a0745021a2af',
    locale: 'es-CU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-CU, ManuelNeural)',
    properties: {
      DisplayName: 'Manuel',
      DisplayVoiceName: 'ManuelNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Manuel',
      LocaleDescription: 'es-CU',
      OrderInVoiceListApi: '164',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-CU-ManuelNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-CU',
      localeZH: '西班牙语(古巴)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a8e9cb11-1ea2-4cfa-8c0d-248974384684.wav?sv=2019-07-07&sr=b&sig=r6TW9%2BoePa3EuSDomSJDKOpWxyWrjxkPY3sarSFdh6E%3D&st=2021-10-29T08:30:35Z&se=2121-10-05T08:35:35Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-CU-ManuelNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '7953a9e3-8723-45a2-b495-a0d53313f763',
    locale: 'ne-NP',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ne-NP, HemkalaNeural)',
    properties: {
      DisplayName: 'Hemkala',
      DisplayVoiceName: 'HemkalaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'हेमकला',
      LocaleDescription: 'ne-NP',
      OrderInVoiceListApi: '338',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ne-NP-HemkalaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ne-NP',
      localeZH: '尼泊尔语(尼泊尔)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a64dec5e-ba66-47ac-ba1e-3ac7080686eb.wav?sv=2019-07-07&sr=b&sig=mfKyAXsVZOqr29EbU3c7MxvREAVxFxMLGs9M0OW8Rlg%3D&st=2022-07-01T09:10:07Z&se=2122-06-07T09:15:07Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ne-NP-HemkalaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [
      {
        name: 'All',
      },
      {
        name: 'Audiobook',
      },
      {
        name: 'Chat',
      },
      {
        defaultStyle: 'assistant',
        name: 'Assistant',
      },
    ],
    description:
      'Lively and sunshine voice with rich emotions that can be used in many conversation scenarios.',
    id: '1011ca97-3e33-4e7c-8dda-a22dc244bafc',
    locale: 'zh-CN',
    masterpieces: [
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/79f6ba77-3b5f-4a01-aeef-1541feed3316.wav?sv=2019-07-07&sr=b&sig=auf%2FHDlxl%2BcKQgkIVePztk9O%2FVPTLzMFkvR18KkSv6M%3D&st=2021-10-29T08:31:48Z&se=2121-10-05T08:36:48Z&sp=rl',
        categories: ['Audiobook'],
        title: '欧游杂记',
        titleLocalizationId: 'zh-CN-YunxiNeural_Masterpieces_AudioBook_OuYouZaJi_Title',
      },
    ],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, YunxiNeural)',
    properties: {
      AgeGroups: 'YoungAdult',
      DisplayName: 'Yunxi',
      DisplayVoiceName: 'YunxiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      IconFileEndpointWithSas:
        'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a227be6e-f179-4931-bfcd-451533348ff7.png?sv=2019-07-07&sr=b&sig=qXMLZ5Zhnm7QmupNVISx8zAn7ptau1sHPEwyHwd0P6U%3D&st=2021-10-29T08:24:30Z&se=2121-10-05T08:29:30Z&sp=rl',
      IsTtsBreakEnabled: 'True',
      LocalName: '云希',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '441',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-YunxiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{"Narrator":0,"YoungAdultMale":1,"Boy":9,"Default":0}',
      VoiceRoleNames: 'Narrator,YoungAdultMale,Boy,Default',
      VoiceStyleNameDefinitions:
        '{"embarrassed":1,"fearful":3,"cheerful":4,"disgruntled":5,"serious":6,"angry":7,"sad":8,"depressed":10,"Default":0}',
      VoiceStyleNames:
        'narration-relaxed,embarrassed,fearful,cheerful,disgruntled,serious,angry,sad,depressed,chat,assistant,newscast,Default',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f4ca3209-80f9-4987-aa07-092d43f82fe8.wav?sv=2019-07-07&sr=b&sig=CsrYOOuXP0bcuTFhxt6ZrHI%2B5SM7E08TAn%2F8a1LuqlI%3D&st=2021-12-20T11:29:39Z&se=2121-11-26T11:34:39Z&sp=rl',
          roleName: 'Default',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5c4a3e25-c74e-497e-bbc9-36ae5995f5b5.wav?sv=2019-07-07&sr=b&sig=jkQ8znJJ5az%2BJ4i1o3oozSOb%2FhO52Oy%2Bqpl7ZH7zsoU%3D&st=2021-12-20T11:29:36Z&se=2121-11-26T11:34:36Z&sp=rl',
          roleName: 'Boy',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/64e597fc-dc74-43c5-b29e-a225db611daf.wav?sv=2019-07-07&sr=b&sig=H6ozko9QcB9Ru7SrJLd1APGqfKPpFe%2Bop%2FMbaSmF5Us%3D&st=2021-12-20T11:29:42Z&se=2121-11-26T11:34:42Z&sp=rl',
          roleName: 'Narrator',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/754795a7-8019-492b-9ece-e2068036337a.wav?sv=2019-07-07&sr=b&sig=3441SIV9JzBW5nGxzlHlvv0zJrXmoYoYUYABcphvvd0%3D&st=2021-12-21T09:06:49Z&se=2121-11-27T09:11:49Z&sp=rl',
          roleName: 'YoungAdultMale',
        },
      ],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6f050a04-d3c3-4099-b9e1-d51f152d52a4.wav?sv=2019-07-07&sr=b&sig=6X1eUm0J5I0VwDDNcpdXMZ1mP9azu11pgLcXFrzqY2U%3D&st=2021-10-29T08:29:21Z&se=2121-10-05T08:34:21Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/99650f95-1535-4900-abd4-0e6aeb387814.wav?sv=2019-07-07&sr=b&sig=sQpnemtw38p%2FJM1OMC079toxZHWW%2FI08urO%2Fe2MrkBA%3D&st=2021-10-29T08:29:22Z&se=2121-10-05T08:34:22Z&sp=rl',
          styleName: 'assistant',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/669d6a6c-2d13-421c-9d54-25fa7c2742e1.wav?sv=2019-07-07&sr=b&sig=XVgH5myuwl2ldYBJMduiPedtNGF59bcOp%2FGGaUNTkoM%3D&st=2021-10-29T08:29:23Z&se=2121-10-05T08:34:23Z&sp=rl',
          styleName: 'calm',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/905364f0-cafb-491e-841f-224a93e8e954.wav?sv=2019-07-07&sr=b&sig=%2BOoQBkECuK61qW9zVx2NylcYy1B2A9KNBszN8vbOZKE%3D&st=2021-12-02T05:02:21Z&se=2121-11-08T05:07:21Z&sp=rl',
          styleName: 'chat',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/9e766773-610b-40a0-a34a-917b27fe3d2c.wav?sv=2019-07-07&sr=b&sig=ThqEXmy0WQSjVPAUwgWi0GwiLK8Rw%2FVbScfimNS95gQ%3D&st=2021-10-29T08:29:24Z&se=2121-10-05T08:34:24Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/1624f1d6-9217-44db-8d4f-1517cc73a583.wav?sv=2019-07-07&sr=b&sig=zhleCkQ4kfr3FQKIX2HCoV%2FKMmV%2F%2Ffijgeq6ImOoZ0Q%3D&st=2021-10-29T08:29:25Z&se=2121-10-05T08:34:25Z&sp=rl',
          styleName: 'depressed',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b2868c0d-9a5c-48b1-88a9-6fc5b8a0f175.wav?sv=2019-07-07&sr=b&sig=CF80mWBbzU3%2FWohem2%2B6iYPbYxR6zan%2B%2FQFrBXcYm94%3D&st=2021-10-29T08:29:26Z&se=2121-10-05T08:34:26Z&sp=rl',
          styleName: 'disgruntled',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0f236a40-6cbe-422e-9c34-26b17632b987.wav?sv=2019-07-07&sr=b&sig=9ft%2FKlSsXzUiGdVQ8zcgE0ctOyf5FYgFAohGFs4r%2Fs8%3D&st=2021-10-29T08:29:27Z&se=2121-10-05T08:34:27Z&sp=rl',
          styleName: 'embarrassed',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2b5c04c5-1295-46af-9629-5ed53d730e14.wav?sv=2019-07-07&sr=b&sig=4snKuHaoEdNPj8KJKXHeQugHyY1p3YazWvZ97ri8s3Q%3D&st=2021-10-29T08:29:28Z&se=2121-10-05T08:34:28Z&sp=rl',
          styleName: 'fearful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e4a1a6af-c1cc-48f0-ba22-a0b1d6cd8278.wav?sv=2019-07-07&sr=b&sig=3baTlmiHN%2FUQoZATNPZ1r9vsFBkZvzGV5Gl0FY0CO14%3D&st=2021-10-29T08:29:29Z&se=2121-10-05T08:34:29Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/01804fa7-4a20-49bc-9d06-3c691196e767.wav?sv=2019-07-07&sr=b&sig=B3owwuUbLs3nW63tJlWCJMdKSqBgAy%2BSyXry2HfIsCM%3D&st=2022-02-09T14:43:10Z&se=2122-01-16T14:48:10Z&sp=rl',
          styleName: 'narration-relaxed',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f61ff6c7-8e74-409b-a94c-980ade698071.wav?sv=2019-07-07&sr=b&sig=Slap%2Bq84FM1QBHlNN3vejmD%2BXqbqBwaPkk3EmxbAEtc%3D&st=2021-10-29T08:29:30Z&se=2121-10-05T08:34:30Z&sp=rl',
          styleName: 'sad',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/45b93637-35e0-47dd-94d8-a1acc385df37.wav?sv=2019-07-07&sr=b&sig=nxvG82vho2TGnDwPZ5Pg10UKiqSJPbuCd%2Bih%2BAZcFDI%3D&st=2021-10-29T08:29:31Z&se=2121-10-05T08:34:31Z&sp=rl',
          styleName: 'serious',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/06977df4-8279-43f6-9a7d-eae09f6f90c2.wav?sv=2019-07-07&sr=b&sig=JzH7R5128SHURsnJdgtDn5Ugr5NMZyc2ryhOAP0iubM%3D&st=2022-02-09T14:43:13Z&se=2122-01-16T14:48:13Z&sp=rl',
          styleName: 'newscast',
        },
      ],
    },
    shortName: 'zh-CN-YunxiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '5daae71c-815a-4118-a516-a37ee62db211',
    locale: 'en-SG',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-SG, LunaNeural)',
    properties: {
      DisplayName: 'Luna',
      DisplayVoiceName: 'LunaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Luna',
      LocaleDescription: 'en-SG',
      OrderInVoiceListApi: '122',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-SG-LunaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-SG',
      localeZH: '英语(新加坡)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/528d5acf-f34e-41f1-bf6c-0a2b9e9bea9f.wav?sv=2019-07-07&sr=b&sig=JDV8eg1P7gf5ZIvRfwcAdHSfXs%2FIxKT5cLIIvku6pGU%3D&st=2022-07-03T04:14:35Z&se=2122-06-09T04:19:35Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-SG-LunaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '4a52cc29-f575-4d96-9e08-a45e7b7975b6',
    locale: 'nl-BE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (nl-BE, ArnaudNeural)',
    properties: {
      DisplayName: 'Arnaud',
      DisplayVoiceName: 'ArnaudNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Arnaud',
      LocaleDescription: 'Dutch (Belgium)',
      OrderInVoiceListApi: '340',
      PreviewSentence:
        'Audio-inhoud Aanmaken laat u toe om visueel de spraakeigenschappen in te stellen in real-time.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'nl-BE-ArnaudNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'nl-BE',
      localeZH: '荷兰语(比利时)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/1643c2f6-e665-4acc-8222-d981eae89d0a.wav?sv=2019-07-07&sr=b&sig=fH6Z4sJ5%2BW0lJ%2Fli568H0TBZMmzXMmHFjgCH6XqNHKg%3D&st=2021-10-29T08:27:41Z&se=2121-10-05T08:32:41Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'nl-BE-ArnaudNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '50c098f5-42ff-4a39-9753-a595d537d428',
    locale: 'fa-IR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fa-IR, FaridNeural)',
    properties: {
      DisplayName: 'Farid',
      DisplayVoiceName: 'FaridNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'فرید',
      LocaleDescription: 'fa-IR',
      OrderInVoiceListApi: '228',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fa-IR-FaridNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fa-IR',
      localeZH: '波斯语(伊朗)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d4be75ab-cd32-4c52-9857-af0c7eda7e96.wav?sv=2019-07-07&sr=b&sig=EtvXEHpJUm%2FtF0rzh1tuxauQ0dCly3UdZnSnBTtPQ6U%3D&st=2021-10-29T08:31:02Z&se=2121-10-05T08:36:02Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fa-IR-FaridNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '5b30688a-af74-410e-9b92-a65882a76ab6',
    locale: 'tr-TR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (tr-TR, EmelNeural)',
    properties: {
      DisplayName: 'Emel',
      DisplayVoiceName: 'EmelNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Emel',
      LocaleDescription: 'Turkish',
      OrderInVoiceListApi: '408',
      PreviewSentence:
        'Kullanıcılarla doğal biçimde konuşan, erişilebilirlik ve kullanılabilirliği iyileştiren uygulama ve servisler geliştirmek.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'tr-TR-EmelNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'tr-TR',
      localeZH: '土耳其语(Türkiye)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e6f07dfb-9314-458c-b6f3-a315a98e1315.wav?sv=2019-07-07&sr=b&sig=cfJdAWzZgyjtMsUxjTvtf%2Fkqlcwy%2BHXToQ5SwX%2Beje8%3D&st=2021-10-29T08:28:24Z&se=2121-10-05T08:33:24Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'tr-TR-EmelNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'ec861c24-5c0a-4b26-b2a2-a72d0ee2cbe7',
    locale: 'en-US',
    masterpieces: [
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/22ab7c89-5016-49c0-8cd5-105e94dec60d.wav?sv=2019-07-07&sr=b&sig=S8HWJlKu9b%2BmAoAulHXarxphuZcxx8uFtOcChN%2B3iT4%3D&st=2022-07-20T10:55:52Z&se=2122-06-26T11:00:52Z&sp=rl',
        categories: ['Audiobook'],
        title: 'AudioBook',
        titleLocalizationId: 'AudioBook',
      },
    ],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, RogerNeural)',
    properties: {
      DisplayName: 'Roger',
      DisplayVoiceName: 'RogerNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Roger',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '145',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-RogerNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0cca4395-a736-4a09-ad0b-5913c69750ba.wav?sv=2019-07-07&sr=b&sig=c4vpw93Z2dwLmaY2S5evj8ggPSSEzrthyIdVFO%2FFUdk%3D&st=2022-07-20T10:55:48Z&se=2122-06-26T11:00:48Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-US-RogerNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'adc6ae77-abab-49e2-a6e5-a79ed6646bdd',
    locale: 'pl-PL',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pl-PL, MarekNeural)',
    properties: {
      DisplayName: 'Marek',
      DisplayVoiceName: 'MarekNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Marek',
      LocaleDescription: 'Polish',
      OrderInVoiceListApi: '346',
      PreviewSentence:
        'Twórz aplikacje i serwisy, które w kontakcie z użytkownikiem posługują się naturalną mową, co podnosi ich dostępność i użyteczność.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pl-PL-MarekNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'pl-PL',
      localeZH: '波兰语(波兰)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4ec46e25-6055-4fc7-8eac-2b65648b35c6.wav?sv=2019-07-07&sr=b&sig=Lsfh8zHdO4HlRBOqfTCbIYaSjOR9WLmP%2FztL7UxNKPg%3D&st=2021-10-29T08:27:50Z&se=2121-10-05T08:32:50Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'pl-PL-MarekNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'edd219ae-6138-40c0-a59b-a7bc6d8be600',
    locale: 'en-ZA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-ZA, LeahNeural)',
    properties: {
      DisplayName: 'Leah',
      DisplayVoiceName: 'LeahNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Leah',
      LocaleDescription: 'en-ZA',
      OrderInVoiceListApi: '151',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-ZA-LeahNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-ZA',
      localeZH: '英语(南非)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/149360ab-b012-4033-9751-ce1d1b947647.wav?sv=2019-07-07&sr=b&sig=tI2v80iXEvdIPx%2B0baYIJ6c%2B3MbJskevIpuFG7o7nQo%3D&st=2021-10-29T08:26:38Z&se=2121-10-05T08:31:38Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-ZA-LeahNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'b6d2cb43-e353-4409-b10e-a9a429846d00',
    locale: 'es-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-ES, ArnauNeural)',
    properties: {
      DisplayName: 'Arnau',
      DisplayVoiceName: 'ArnauNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Arnau',
      LocaleDescription: 'Spanish (Spain)',
      OrderInVoiceListApi: '170',
      PreviewSentence:
        'Crea aplicaciones y servicios que hablan de forma natural a los usuarios, y que mejoran la accesibilidad y la facilidad de uso.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-ES-ArnauNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-ES',
      localeZH: '西班牙语(西班牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d9a72907-b8a6-4f5b-a8d3-2b188685b210.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A34Z&se=2122-09-27T07%3A51%3A34Z&sr=b&sp=rl&sig=FPbt0TboLe9BCrTJQiIvgfxOfsmDDau9O%2BWZkByHBJw%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-ES-ArnauNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'ddce4588-3f1e-459c-9934-aa4a5aea66af',
    locale: 'eu-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (eu-ES, AnderNeural)',
    properties: {
      DisplayName: 'Ander',
      DisplayVoiceName: 'AnderNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Ander',
      LocaleDescription: 'eu-ES',
      OrderInVoiceListApi: '226',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'eu-ES-AnderNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'eu-ES',
      localeZH: '巴斯克语(巴斯克语)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e3ebb69e-6e4c-45f0-8553-af7cfdea2fe3.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A08Z&se=2122-09-27T07%3A52%3A08Z&sr=b&sp=rl&sig=4Y4WqNZGa9tQ7MA%2FnzkP8MZNL0gRGUmTvwdesWtu%2FSc%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'eu-ES-AnderNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '389b1132-ae33-48ae-a190-aabcb08d4db3',
    locale: 'es-NI',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-NI, FedericoNeural)',
    properties: {
      DisplayName: 'Federico',
      DisplayVoiceName: 'FedericoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Federico',
      LocaleDescription: 'es-NI',
      OrderInVoiceListApi: '205',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-NI-FedericoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-NI',
      localeZH: '西班牙语(尼加拉瓜)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e218c117-c2fc-4803-80ed-3183b204a2e6.wav?sv=2019-07-07&sr=b&sig=4uZoioSmZGpnyLe32nA08nw3JGsAdfs2dsu0lB1mW1U%3D&st=2021-10-29T08:30:46Z&se=2121-10-05T08:35:46Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-NI-FedericoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '2c5617a6-37b8-44dc-b4c0-ac73be78acf8',
    locale: 'wuu-CN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (wuu-CN, YunzheNeural)',
    properties: {
      DisplayName: 'Yunzhe',
      DisplayVoiceName: 'YunzheNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: '云哲',
      LocaleDescription: 'wuu-CN',
      OrderInVoiceListApi: '420',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'wuu-CN-YunzheNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'wuu-CN',
      localeZH: '中文(吴语，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a2b32854-bd49-4dcd-bb31-d9bdeef36dee.wav?sv=2021-08-06&st=2022-10-21T11%3A57%3A29Z&se=2122-09-27T12%3A02%3A29Z&sr=b&sp=rl&sig=tSJnoMcFBKIdwAsSKzMBBGht4zEp10JiSFiqAVBQO6g%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'wuu-CN-YunzheNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '68cddd97-2143-4ad6-9d14-acc2223f7ea0',
    locale: 'fa-IR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fa-IR, DilaraNeural)',
    properties: {
      DisplayName: 'Dilara',
      DisplayVoiceName: 'DilaraNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'دلارا',
      LocaleDescription: 'fa-IR',
      OrderInVoiceListApi: '227',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fa-IR-DilaraNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fa-IR',
      localeZH: '波斯语(伊朗)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/8933d097-85dc-4721-b6cf-e1e274a797e7.wav?sv=2019-07-07&sr=b&sig=ipIyjWltXSv79KqJDtx4Mgg2YMU0VjNm4%2FIYYZY%2FgkU%3D&st=2021-10-29T08:31:01Z&se=2121-10-05T08:36:01Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fa-IR-DilaraNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '7dd9ee6d-926f-41d0-9e5f-acd41467522e',
    locale: 'es-MX',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-MX, GerardoNeural)',
    properties: {
      DisplayName: 'Gerardo',
      DisplayVoiceName: 'GerardoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Gerardo',
      LocaleDescription: 'Spanish (Mexico)',
      OrderInVoiceListApi: '195',
      PreviewSentence:
        'Crea apps y servicios que hablen de forma natural con los usuarios, mejorando la accesibilidad y la usabilidad.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-MX-GerardoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-MX',
      localeZH: '西班牙语(墨西哥)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/cce1f446-2f39-4fd8-b932-1b574c02c250.wav?sv=2019-07-07&sr=b&sig=SEbek6%2B9IxtORNmELmsWyIybxgfBF9WQRwZIm0MjNpQ%3D&st=2022-07-01T09:09:10Z&se=2122-06-07T09:14:10Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-MX-GerardoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '97ebc7c6-1e92-4764-806b-ad61201a60a5',
    locale: 'gu-IN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (gu-IN, DhwaniNeural)',
    properties: {
      DisplayName: 'Dhwani',
      DisplayVoiceName: 'DhwaniNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'ધ્વની',
      LocaleDescription: 'gu-IN',
      OrderInVoiceListApi: '259',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'gu-IN-DhwaniNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'gu-IN',
      localeZH: '古吉拉特语(印度)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/8ff832f3-634e-4a00-9bd3-e20aeeb96415.wav?sv=2019-07-07&sr=b&sig=JjEubolh%2BygfaQIjWtaFyNbItbHKj1EpoPRsA%2FyHnOc%3D&st=2021-10-29T08:27:05Z&se=2121-10-05T08:32:05Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'gu-IN-DhwaniNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'd7747462-41fd-4140-af6b-adbb76cf2405',
    locale: 'sr-RS',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (sr-RS, NicholasNeural)',
    properties: {
      DisplayName: 'Nicholas',
      DisplayVoiceName: 'NicholasNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Никола',
      LocaleDescription: 'sr-RS',
      OrderInVoiceListApi: '383',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'sr-RS-NicholasNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'sr-RS',
      localeZH: '塞尔维亚语(塞尔维亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f183c77b-32a2-44ff-9bd6-ffb25c104dbf.wav?sv=2019-07-07&sr=b&sig=3fA8o%2FnH%2ByBQ0JSSqC1TY%2FC%2FKsxVMQW%2BOWUJWlxI%2FC8%3D&st=2022-01-19T09:02:20Z&se=2121-12-26T09:07:20Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'sr-RS-NicholasNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '0a9b0267-c022-4f86-85ee-ae514facbeb2',
    locale: 'it-IT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (it-IT, CalimeroNeural)',
    properties: {
      DisplayName: 'Calimero',
      DisplayVoiceName: 'CalimeroNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Calimero',
      LocaleDescription: 'Italian',
      OrderInVoiceListApi: '277',
      PreviewSentence:
        'Benvenuti nella piattaforma di generazione di contenuti audio di Microsoft.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'it-IT-CalimeroNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'it-IT',
      localeZH: '意大利语(意大利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d7469cd1-38b9-42f0-b3d7-84e560298951.wav?sv=2019-07-07&sr=b&sig=UbX6lBkiCDMNZb59sdigVwrbAgEV%2BCF73GKEd3OhM3E%3D&st=2022-07-01T09:09:33Z&se=2122-06-07T09:14:33Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'it-IT-CalimeroNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '06af3b1f-e3a7-45b3-8b84-aecd737401ed',
    locale: 'it-IT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (it-IT, BenignoNeural)',
    properties: {
      DisplayName: 'Benigno',
      DisplayVoiceName: 'BenignoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Benigno',
      LocaleDescription: 'Italian',
      OrderInVoiceListApi: '276',
      PreviewSentence:
        'Benvenuti nella piattaforma di generazione di contenuti audio di Microsoft.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'it-IT-BenignoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'it-IT',
      localeZH: '意大利语(意大利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/bbfcb64f-b3be-4004-8ae1-b56affe22706.wav?sv=2019-07-07&sr=b&sig=YoUt%2FsE8f%2F2P0XzsVYyEsIJNjLJ3nx0O6KnGYvaynD0%3D&st=2022-07-01T09:09:30Z&se=2122-06-07T09:14:30Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'it-IT-BenignoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '8d161c40-e771-449d-a089-af2d64dc5e3a',
    locale: 'it-IT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (it-IT, PalmiraNeural)',
    properties: {
      DisplayName: 'Palmira',
      DisplayVoiceName: 'PalmiraNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Palmira',
      LocaleDescription: 'Italian',
      OrderInVoiceListApi: '286',
      PreviewSentence:
        'Benvenuti nella piattaforma di generazione di contenuti audio di Microsoft.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'it-IT-PalmiraNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'it-IT',
      localeZH: '意大利语(意大利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/af92bb71-1063-4969-89f7-1546b0b9ed9a.wav?sv=2019-07-07&sr=b&sig=JyM50g47BYw5E9ogHJ4s40FDvagksAsC4lMXFL3XRSo%3D&st=2022-07-01T09:09:51Z&se=2122-06-07T09:14:51Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'it-IT-PalmiraNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'ffe34fb6-c1ee-45ae-aef3-b0c303614c19',
    locale: 'es-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-ES, AlvaroNeural)',
    properties: {
      DisplayName: 'Alvaro',
      DisplayVoiceName: 'AlvaroNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Álvaro',
      LocaleDescription: 'Spanish (Spain)',
      OrderInVoiceListApi: '183',
      PreviewSentence:
        'Crea aplicaciones y servicios que hablan de forma natural a los usuarios, y que mejoran la accesibilidad y la facilidad de uso.',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-ES-AlvaroNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-ES',
      localeZH: '西班牙语(西班牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/1249e89a-f8a7-4ba3-b9ad-c08b76345bba.wav?sv=2019-07-07&sr=b&sig=OaPDKdVySv4QaSEBjNhCeFZC8QVSP084tywJ0EPUDJU%3D&st=2021-10-29T08:26:43Z&se=2121-10-05T08:31:43Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-ES-AlvaroNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'f9d2faf8-ff6b-4a89-aeb8-b0c5f16c4e4c',
    locale: 'su-ID',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (su-ID, TutiNeural)',
    properties: {
      DisplayName: 'Tuti',
      DisplayVoiceName: 'TutiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Tuti',
      LocaleDescription: 'su-ID',
      OrderInVoiceListApi: '386',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'su-ID-TutiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'su-ID',
      localeZH: '巽他语(印度尼西亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5aece8b2-a471-4ab8-8d75-50c544005745.wav?sv=2019-07-07&sr=b&sig=kHWMocLNP3aTsc6W8QDE9E5qsz9NlLL%2BJgZFbONwt9c%3D&st=2021-10-29T08:31:16Z&se=2121-10-05T08:36:16Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'su-ID-TutiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '7cf7c9f9-63c2-4a30-9db3-b25f99709291',
    locale: 'en-GB',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-GB, AbbiNeural)',
    properties: {
      DisplayName: 'Abbi',
      DisplayVoiceName: 'AbbiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Abbi',
      LocaleDescription: 'English (UK)',
      OrderInVoiceListApi: '94',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-GB-AbbiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-GB',
      localeZH: '英语(英国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/8fcc9467-04db-4c67-8e60-e1af42c7dcdb.wav?sv=2019-07-07&sr=b&sig=7Z6zi7%2BcC7G7MJ6EBQDv5aE%2FrfKRNmwYcRlKtqnUgmw%3D&st=2022-01-19T09:00:51Z&se=2121-12-26T09:05:51Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-GB-AbbiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '303f5b66-71d9-4c74-b75b-b38080b11cbe',
    locale: 'es-EC',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-EC, AndreaNeural)',
    properties: {
      DisplayName: 'Andrea',
      DisplayVoiceName: 'AndreaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Andrea',
      LocaleDescription: 'es-EC',
      OrderInVoiceListApi: '167',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-EC-AndreaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-EC',
      localeZH: '西班牙语(厄瓜多尔)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/dc233559-730b-48ff-95cd-e3dab05a03bf.wav?sv=2019-07-07&sr=b&sig=jib0zlwlqY6LALqZUZ%2BE0VYGRMs7ocjZW%2Bk0Quoimkc%3D&st=2021-10-29T08:30:38Z&se=2121-10-05T08:35:38Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-EC-AndreaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'ac773ad2-7b91-456c-b93f-b397ef1b11ff',
    locale: 'th-TH',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (th-TH, AcharaNeural)',
    properties: {
      DisplayName: 'Achara',
      DisplayVoiceName: 'AcharaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'อัจฉรา',
      LocaleDescription: 'Thai',
      OrderInVoiceListApi: '405',
      PreviewSentence:
        'สร้างแอปและบริการที่สื่อสารกับผู้ใช้ได้อย่างเป็นธรรมชาติ ซึ่งช่วยปรับปรุงการเข้าถึงและการใช้งาน',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'th-TH-AcharaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'th-TH',
      localeZH: '泰语(泰国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/38d753f1-1a94-4924-9b51-6d1299702888.wav?sv=2019-07-07&sr=b&sig=WaDMQDwdBG6%2Fvtl30Yq%2F%2Btk0JYx98PwzSBZAEaAz1MQ%3D&st=2021-10-29T08:28:19Z&se=2121-10-05T08:33:19Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'th-TH-AcharaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'e398d4d2-b8a8-4d1d-ab65-b4fe9278eb71',
    locale: 'nb-NO',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (nb-NO, IselinNeural)',
    properties: {
      DisplayName: 'Iselin',
      DisplayVoiceName: 'IselinNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Iselin',
      LocaleDescription: 'Norwegian',
      OrderInVoiceListApi: '337',
      PreviewSentence:
        'Bygger apper og tjenester som snakker naturlig med brukerne, utbedrer tilgjengelighet og brukskvalitet.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'nb-NO-IselinNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'nb-NO',
      localeZH: '书面挪威语(挪威)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/98df46c7-f02e-481c-98ab-f54fb78023ba.wav?sv=2019-07-07&sr=b&sig=G6WewQHD7V%2FF62fFOt6knGAR9ArQ12ytG%2BiLFdV1uvU%3D&st=2021-10-29T08:27:38Z&se=2121-10-05T08:32:38Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'nb-NO-IselinNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '22bb63f1-7dba-49b9-9d4d-b56c7ce92d47',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, EricNeural)',
    properties: {
      DisplayName: 'Eric',
      DisplayVoiceName: 'EricNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Eric',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '138',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-EricNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d8d65e4b-0966-41a2-ad2a-59c344f9fbd5.wav?sv=2019-07-07&sr=b&sig=mD0TyJ%2Bt4RfNH%2FYdvmY8ixKtmm2a5otEB90BKM49vMg%3D&st=2021-10-29T08:25:36Z&se=2121-10-05T08:30:36Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-US-EricNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '1a97c094-c81f-4c6a-ad9f-b595b54bc448',
    locale: 'en-GB',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-GB, AlfieNeural)',
    properties: {
      DisplayName: 'Alfie',
      DisplayVoiceName: 'AlfieNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Alfie',
      LocaleDescription: 'English (UK)',
      OrderInVoiceListApi: '95',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-GB-AlfieNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-GB',
      localeZH: '英语(英国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/72b25357-f84f-431e-8e6c-ca5ffad9cc16.wav?sv=2019-07-07&sr=b&sig=%2Bv2L4Bdfl%2Fx5cWH8YMof8iKAO4CuIzqZIli1jIsjP8o%3D&st=2022-01-19T09:00:53Z&se=2121-12-26T09:05:53Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-GB-AlfieNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'a0c30154-af80-4aa0-8b73-b5ed657091e9',
    locale: 'ur-IN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ur-IN, GulNeural)',
    properties: {
      DisplayName: 'Gul',
      DisplayVoiceName: 'GulNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'گل',
      LocaleDescription: 'ur-IN',
      OrderInVoiceListApi: '411',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ur-IN-GulNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ur-IN',
      localeZH: '乌尔都语(印度)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/3306cca4-19dc-4074-aed3-ac9e3f74a8a3.wav?sv=2019-07-07&sr=b&sig=J2S1U6A%2FP4KNzfQtEErhGQ9NrFoqevdbzdbatYsgIcw%3D&st=2021-10-29T08:31:23Z&se=2121-10-05T08:36:23Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ur-IN-GulNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '471b7597-7004-4dcf-b364-b6bba39518f6',
    locale: 'es-MX',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-MX, CandelaNeural)',
    properties: {
      DisplayName: 'Candela',
      DisplayVoiceName: 'CandelaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Candela',
      LocaleDescription: 'Spanish (Mexico)',
      OrderInVoiceListApi: '191',
      PreviewSentence:
        'Crea apps y servicios que hablen de forma natural con los usuarios, mejorando la accesibilidad y la usabilidad.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-MX-CandelaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-MX',
      localeZH: '西班牙语(墨西哥)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/9dbc2440-e100-48f1-8d6e-642953a33bfb.wav?sv=2019-07-07&sr=b&sig=vdH49ZbrU7HtheBrzPkXbKphTLIvHrEhYEcylUyCGxs%3D&st=2022-07-01T09:09:04Z&se=2122-06-07T09:14:04Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-MX-CandelaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'af5c31bf-a36d-4852-8555-b775316d9e5f',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, MonicaNeural)',
    properties: {
      DisplayName: 'Monica',
      DisplayVoiceName: 'MonicaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Monica',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '143',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-MonicaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/9337bb71-f0fc-4e2a-a836-27b4872932da.wav?sv=2019-07-07&sr=b&sig=nOt3gl3Snkv9VuxcvWL0EkgJtgCVwFCdIn2%2BwgHO97A%3D&st=2021-10-29T08:25:50Z&se=2121-10-05T08:30:50Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-US-MonicaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'eecf9888-9e58-4ed2-ae91-b7ae85a77cf3',
    locale: 'zh-TW',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-TW, YunJheNeural)',
    properties: {
      DisplayName: 'YunJhe',
      DisplayVoiceName: 'YunJheNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: '雲哲',
      LocaleDescription: 'Chinese (Taiwan)',
      OrderInVoiceListApi: '454',
      PreviewSentence: '建構可以和使用者自然對話的應用程式和服務，來提高其方便性和實用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-TW-YunJheNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'zh-TW',
      localeZH: '中文(台湾普通话)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b5663e52-b08f-45aa-b1b8-6ed956f80b7f.wav?sv=2019-07-07&sr=b&sig=6FfjzgJLIyDvntlFotxOYHiep3HOQX0vqW8NGFKulm8%3D&st=2021-10-29T08:29:53Z&se=2121-10-05T08:34:53Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'zh-TW-YunJheNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '5b1e2e20-b01c-4945-b635-b865738e00f5',
    locale: 'ar-DZ',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-DZ, AminaNeural)',
    properties: {
      DisplayName: 'Amina',
      DisplayVoiceName: 'AminaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'أمينة',
      LocaleDescription: 'ar-DZ',
      OrderInVoiceListApi: '9',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-DZ-AminaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-DZ',
      localeZH: '阿拉伯语(阿尔及利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d7b12fa3-619c-4c44-99b7-ab9daf34389f.wav?sv=2019-07-07&sr=b&sig=KTkbrWzyaoCSDZgz9BlTC6HHtvUxS6SxKrXQKHwzW0M%3D&st=2021-10-29T08:30:01Z&se=2121-10-05T08:35:01Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-DZ-AminaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '08d7e69f-46fd-4053-bc85-b89ad688e3ff',
    locale: 'ar-JO',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-JO, TaimNeural)',
    properties: {
      DisplayName: 'Taim',
      DisplayVoiceName: 'TaimNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'تيم',
      LocaleDescription: 'ar-JO',
      OrderInVoiceListApi: '16',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-JO-TaimNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-JO',
      localeZH: '阿拉伯语(约旦)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5ed486f6-d31e-4359-90cc-63e008f5f45e.wav?sv=2019-07-07&sr=b&sig=QzLaefDGUNQ6bJMprcJ3V5RASBM7w6SaNrjOcorCKAI%3D&st=2021-10-29T08:30:06Z&se=2121-10-05T08:35:06Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-JO-TaimNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '176b1f4f-357d-4b5d-a59f-b8ecd1e69ed7',
    locale: 'es-MX',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-MX, RenataNeural)',
    properties: {
      DisplayName: 'Renata',
      DisplayVoiceName: 'RenataNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Renata',
      LocaleDescription: 'Spanish (Mexico)',
      OrderInVoiceListApi: '203',
      PreviewSentence:
        'Crea apps y servicios que hablen de forma natural con los usuarios, mejorando la accesibilidad y la usabilidad.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-MX-RenataNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-MX',
      localeZH: '西班牙语(墨西哥)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/25a1e32a-85a6-4a14-83a3-ac8df412c9ff.wav?sv=2019-07-07&sr=b&sig=bsqrUcwvC0cSCzRNJCxXa5Fgfg7qmdMRF%2B5QmKUUPds%3D&st=2022-07-01T09:09:26Z&se=2122-06-07T09:14:26Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-MX-RenataNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '7f8007d2-4fb4-4720-a709-b945deefba3f',
    locale: 'es-MX',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-MX, LarissaNeural)',
    properties: {
      DisplayName: 'Larissa',
      DisplayVoiceName: 'LarissaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Larissa',
      LocaleDescription: 'Spanish (Mexico)',
      OrderInVoiceListApi: '197',
      PreviewSentence:
        'Crea apps y servicios que hablen de forma natural con los usuarios, mejorando la accesibilidad y la usabilidad.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-MX-LarissaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-MX',
      localeZH: '西班牙语(墨西哥)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ea286546-0fd4-44c2-a0b9-93bde5720f5c.wav?sv=2019-07-07&sr=b&sig=FN2bFWI%2BfX4RqJGYHKBLkTWDnjjRvMBtdttaMeAq2To%3D&st=2022-07-01T09:09:12Z&se=2122-06-07T09:14:12Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-MX-LarissaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '6aeff0b1-3553-4b2b-92f4-b9a30cc8b294',
    locale: 'sv-SE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (sv-SE, HilleviNeural)',
    properties: {
      DisplayName: 'Hillevi',
      DisplayVoiceName: 'HilleviNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Hillevi',
      LocaleDescription: 'Swedish',
      OrderInVoiceListApi: '388',
      PreviewSentence:
        'Bygg appar och tjänster som talar naturligt till användarna, och förbättrar tillgänglighet och användbarhet.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'sv-SE-HilleviNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'sv-SE',
      localeZH: '瑞典语(瑞典)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/63d28117-7491-4e33-9ccd-56781b73e880.wav?sv=2019-07-07&sr=b&sig=h5IAvNSmVkvclIE2J%2Fma6SexEPgQL652iVf9eeVVxRo%3D&st=2021-10-29T08:28:10Z&se=2121-10-05T08:33:10Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'sv-SE-HilleviNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '3dab86d2-97aa-4f0e-91a5-bb572a659a29',
    locale: 'bg-BG',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (bg-BG, BorislavNeural)',
    properties: {
      DisplayName: 'Borislav',
      DisplayVoiceName: 'BorislavNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Борислав',
      LocaleDescription: 'Bulgarian',
      OrderInVoiceListApi: '39',
      PreviewSentence:
        'Създай приложения и услуги, които говорят непринудено на потребителите, подобрявайки достъпността и използваемостта.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'bg-BG-BorislavNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'bg-BG',
      localeZH: '保加利亚语(保加利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a85c27d8-332f-4803-b0b1-5824a35a021c.wav?sv=2019-07-07&sr=b&sig=IMicYRGvRJOPMqoVt52iQJUwEsGRgyntiVkozvKLYR8%3D&st=2021-10-29T08:25:59Z&se=2121-10-05T08:30:59Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'bg-BG-BorislavNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '77f8306b-b8a7-4a3f-b892-bb5ed1e9ada8',
    locale: 'ur-PK',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ur-PK, AsadNeural)',
    properties: {
      DisplayName: 'Asad',
      DisplayVoiceName: 'AsadNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'اسد',
      LocaleDescription: 'Urdu (Pakistan)',
      OrderInVoiceListApi: '413',
      PreviewSentence:
        'آواز کا مواد تخلیق کرنا  اس قابل بناتا  ہیکہ آپ تقریر کی خصوصیات کو  حقیقی وقت میں  اپنے مطابق کنٹرول کر سکتے ہیں۔',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ur-PK-AsadNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ur-PK',
      localeZH: '乌尔都语(巴基斯坦)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/635f209f-ecba-4ed8-82b8-7ca0750a3094.wav?sv=2019-07-07&sr=b&sig=YuvkUvJmkhluEHm8%2BkVutgPCjTRcRY5TOE3OjfTTHaU%3D&st=2021-10-29T08:28:27Z&se=2121-10-05T08:33:27Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ur-PK-AsadNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '22d37991-4127-4149-b548-bc000dbde9e4',
    locale: 'ta-LK',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ta-LK, SaranyaNeural)',
    properties: {
      DisplayName: 'Saranya',
      DisplayVoiceName: 'SaranyaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'சரண்யா',
      LocaleDescription: 'ta-LK',
      OrderInVoiceListApi: '397',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ta-LK-SaranyaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ta-LK',
      localeZH: '泰米尔语(斯里兰卡)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/763f6dd2-d2ab-4572-bfb9-1b489e10f3ec.wav?sv=2019-07-07&sr=b&sig=z5c9mVAEc0%2B5vuEW6sSkWasaSnG9cTEUNuIo4Py8gmQ%3D&st=2021-10-29T08:31:20Z&se=2121-10-05T08:36:20Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ta-LK-SaranyaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '2d9dc747-b268-420e-bdb7-bc30963f9f3a',
    locale: 'es-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-ES, DarioNeural)',
    properties: {
      DisplayName: 'Dario',
      DisplayVoiceName: 'DarioNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Dario',
      LocaleDescription: 'Spanish (Spain)',
      OrderInVoiceListApi: '171',
      PreviewSentence:
        'Crea aplicaciones y servicios que hablan de forma natural a los usuarios, y que mejoran la accesibilidad y la facilidad de uso.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-ES-DarioNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-ES',
      localeZH: '西班牙语(西班牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/9dedd27c-8405-4fe4-8cd1-3a10325c9628.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A36Z&se=2122-09-27T07%3A51%3A36Z&sr=b&sp=rl&sig=qIc7GGnD2NFo4kyz%2BtXXBwZNvsQAnyllvpzvwgAlYQE%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-ES-DarioNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'dead9773-6cf1-44a3-b32b-bc7fe148693d',
    locale: 'fr-CH',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-CH, FabriceNeural)',
    properties: {
      DisplayName: 'Fabrice',
      DisplayVoiceName: 'FabriceNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Fabrice',
      LocaleDescription: 'French (Switzerland)',
      OrderInVoiceListApi: '240',
      PreviewSentence:
        'Développer des applications et des services qui parlent aux utilisateurs avec naturel, pour améliorer leur accessibilité et leur utilisation.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-CH-FabriceNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fr-CH',
      localeZH: '法语(瑞士)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d6ef5e6d-ca1d-47bd-b396-489f49a455c3.wav?sv=2019-07-07&sr=b&sig=veiH4zcppbyqR1ssSZXEZZg%2F%2Fpj8xhlsplvgMg%2FEyBc%3D&st=2021-10-29T08:27:00Z&se=2121-10-05T08:32:00Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fr-CH-FabriceNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '1fcefcc6-c578-4aec-99fa-bcd38dfa0a02',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, JaneNeural)',
    properties: {
      DisplayName: 'Jane',
      DisplayVoiceName: 'JaneNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Jane',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '140',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-JaneNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"angry":7,"cheerful":4,"sad":8,"Default":0}',
      VoiceStyleNames:
        'angry,cheerful,excited,friendly,hopeful,sad,shouting,terrified,unfriendly,whispering,Default',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/14e55cde-1ec5-4054-a54e-8aee42d80c3b.wav?sv=2019-07-07&sr=b&sig=7oR3Vy%2F1A2O2eQqsT3%2BRUjqtimhihW%2FZo7r1FCjW7Mg%3D&st=2022-05-05T09:38:43Z&se=2122-04-11T09:43:43Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2555f848-df26-4f82-89f4-e64edcc8a978.wav?sv=2019-07-07&sr=b&sig=uMgt6R%2BKGUlUugOpjnF3%2Fv%2FRpp56GEbPA3BNJyfF0kk%3D&st=2022-04-28T10:27:24Z&se=2122-04-04T10:32:24Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0d36d1e1-9b3e-4ef1-b287-97ea811f5b59.wav?sv=2019-07-07&sr=b&sig=Rbawkycxw3iaiyGXdw0%2BopAS2di8uvWDoXQBPs%2BVHMA%3D&st=2022-04-28T10:27:27Z&se=2122-04-04T10:32:27Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/8bd03f26-2c8d-4a73-93c0-74db2f9cd442.wav?sv=2019-07-07&sr=b&sig=6dVD3IcYGpqCb07VCSeZ30%2Ftd803Bs%2FZiC68KRxwmcg%3D&st=2022-04-28T10:27:29Z&se=2122-04-04T10:32:29Z&sp=rl',
          styleName: 'excited',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6eedbd2d-3f3b-4bb1-a670-ccc16c9a3ded.wav?sv=2019-07-07&sr=b&sig=rOGnFl0OHS%2FMhK1Qfmnz9xQWXdcTWx8TILRMnCU%2FTPo%3D&st=2022-04-28T10:27:31Z&se=2122-04-04T10:32:31Z&sp=rl',
          styleName: 'friendly',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/8717df19-c4e6-4310-b14b-c54f97ff9128.wav?sv=2019-07-07&sr=b&sig=nfnJrttagjPXbWUoxfssq7VmA3z4YpzxqPCHcWdV5nI%3D&st=2022-04-28T10:27:34Z&se=2122-04-04T10:32:34Z&sp=rl',
          styleName: 'hopeful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b6431d65-451a-4ff6-aa3e-a2afd1f9ac84.wav?sv=2019-07-07&sr=b&sig=BxNoY9Gj4lGIESy5LYUkcyoUhjr%2FhF3p8NBY%2FsCPI4k%3D&st=2022-04-28T10:27:36Z&se=2122-04-04T10:32:36Z&sp=rl',
          styleName: 'sad',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/96522bca-f6dc-40fe-bf09-21f0d6078e22.wav?sv=2019-07-07&sr=b&sig=ONPm%2Bsve2pa0ByVcOFLYk6TPimtAT0rikzWLWdEw5zc%3D&st=2022-04-28T10:27:38Z&se=2122-04-04T10:32:38Z&sp=rl',
          styleName: 'shouting',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/048b1093-34a7-42bf-8a6f-908f80e62a32.wav?sv=2019-07-07&sr=b&sig=FRFKmmAwHVdO8s9ya2ycJp2OieufF4Pjh1rj60IAq1E%3D&st=2022-04-28T10:27:40Z&se=2122-04-04T10:32:40Z&sp=rl',
          styleName: 'terrified',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/990944c4-d6bd-47bd-88e2-f9067bb9a6fd.wav?sv=2019-07-07&sr=b&sig=%2BrMEostBehZM%2BwU0rl3IxtQivBu5p9WGTlQYohr7P8I%3D&st=2022-04-28T10:27:43Z&se=2122-04-04T10:32:43Z&sp=rl',
          styleName: 'unfriendly',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2d870dfb-fd96-4d4c-acf4-4115f601a038.wav?sv=2019-07-07&sr=b&sig=Wri%2FGD5fVsmj9jzcIMiw%2Fo8vq9s5Q0oVVp5MRXPgPQI%3D&st=2022-04-28T10:27:45Z&se=2122-04-04T10:32:45Z&sp=rl',
          styleName: 'whispering',
        },
      ],
    },
    shortName: 'en-US-JaneNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [
      {
        name: 'All',
      },
      {
        name: 'Chat',
      },
      {
        name: 'Audiobook',
      },
    ],
    description: 'Angelic and clear voice that can be applied in many child related scenarios.',
    id: 'd6814675-a0c5-4e09-9387-bd9b44d3e733',
    locale: 'zh-CN',
    masterpieces: [
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/73a5c13c-ebdb-4a77-b2dd-94e3c2c6af23?sv=2019-07-07&sr=b&sig=PUZd%2FC40sBe0zlgeeebTLmhdqLQW4mDIF%2BuY60Afw8Y%3D&st=2021-10-29T08:24:51Z&se=2121-10-05T08:29:51Z&sp=rl',
        categories: ['Audiobook'],
        title: '晓悠有声书《声律启蒙》',
        titleLocalizationId: 'zh-CN-XiaoyouNeural_Masterpieces_AudioBook_ShengLvQiMeng_Title',
      },
    ],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, XiaoyouNeural)',
    properties: {
      AgeGroups: 'Child',
      DisplayName: 'Xiaoyou',
      DisplayVoiceName: 'XiaoyouNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      IconFileEndpointWithSas:
        'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/313e781a-43a6-499d-8cee-c5b1a7ddc5e5?sv=2019-07-07&sr=b&sig=cuGCx3LdgBiThGGPrjgOS2kdZEbZr7lzeSor7lo3NtU%3D&st=2021-10-29T08:24:28Z&se=2121-10-05T08:29:28Z&sp=rl',
      IsTtsBreakEnabled: 'True',
      LocalName: '晓悠',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '435',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-XiaoyouNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/65ab64f6-f0c1-44f9-a830-080ca6caa6f8.wav?sv=2019-07-07&sr=b&sig=cuya27f9FctxO8Xqxj2bimsDvBTRzqdogvw6rUpPy0I%3D&st=2021-10-29T08:29:20Z&se=2121-10-05T08:34:20Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'zh-CN-XiaoyouNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '9678300c-5bf2-4454-807b-be57d3754dff',
    locale: 'es-VE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-VE, PaolaNeural)',
    properties: {
      DisplayName: 'Paola',
      DisplayVoiceName: 'PaolaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Paola',
      LocaleDescription: 'es-VE',
      OrderInVoiceListApi: '221',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-VE-PaolaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-VE',
      localeZH: '西班牙语(委内瑞拉)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2b958c5e-b8d3-4676-8620-bdfa2aa05b44.wav?sv=2019-07-07&sr=b&sig=gCmd1DU7pMyyAiOnu7BWGtKHeZ%2F2rKgoRSaSfM3Qv%2FI%3D&st=2021-10-29T08:30:59Z&se=2121-10-05T08:35:59Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-VE-PaolaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '0bf71dcf-e841-4f4c-aaf4-be63abc429f7',
    locale: 'es-MX',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-MX, PelayoNeural)',
    properties: {
      DisplayName: 'Pelayo',
      DisplayVoiceName: 'PelayoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Pelayo',
      LocaleDescription: 'Spanish (Mexico)',
      OrderInVoiceListApi: '202',
      PreviewSentence:
        'Crea apps y servicios que hablen de forma natural con los usuarios, mejorando la accesibilidad y la usabilidad.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-MX-PelayoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-MX',
      localeZH: '西班牙语(墨西哥)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c5f751b5-a3e0-4330-8f21-83cea2e6ead8.wav?sv=2019-07-07&sr=b&sig=Oj2YbriVDG2%2B7%2BChZmOpyKODsTYsrT2Ls0gV7jFNMA8%3D&st=2022-07-01T09:09:24Z&se=2122-06-07T09:14:24Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-MX-PelayoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'dfd6067e-87c2-428d-bd9e-be98643182f3',
    locale: 'es-VE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-VE, SebastianNeural)',
    properties: {
      DisplayName: 'Sebastian',
      DisplayVoiceName: 'SebastianNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Sebastián',
      LocaleDescription: 'es-VE',
      OrderInVoiceListApi: '222',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-VE-SebastianNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-VE',
      localeZH: '西班牙语(委内瑞拉)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2836004b-eb84-485e-9649-2138507d4c8c.wav?sv=2019-07-07&sr=b&sig=AdrYm%2FCM%2BKx9%2FuPBUmfKo14d5ocUPFwofb9U9xWz83I%3D&st=2021-10-29T08:31:00Z&se=2121-10-05T08:36:00Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-VE-SebastianNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '6be0ede7-bb79-4435-b18d-c06bbc272df2',
    locale: 'et-EE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (et-EE, AnuNeural)',
    properties: {
      DisplayName: 'Anu',
      DisplayVoiceName: 'AnuNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Anu',
      LocaleDescription: 'Estonian (Estonia)',
      OrderInVoiceListApi: '223',
      PreviewSentence:
        'Audio Content Creation võimaldab kõne atribuute reaalajas visuaalselt kontrollida.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'et-EE-AnuNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'et-EE',
      localeZH: '爱沙尼亚语(爱沙尼亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/bbce0c5e-0978-41f4-80a2-cf30651245ba.wav?sv=2019-07-07&sr=b&sig=ED1woHDXGveFrhY7DEGC9Qeol%2Bb0Bijfi2ipriIj%2BBo%3D&st=2021-10-29T08:26:49Z&se=2121-10-05T08:31:49Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'et-EE-AnuNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '1acd89a8-65b1-4138-b417-c0b2950fd95c',
    locale: 'sw-KE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (sw-KE, RafikiNeural)',
    properties: {
      DisplayName: 'Rafiki',
      DisplayVoiceName: 'RafikiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Rafiki',
      LocaleDescription: 'sw-KE',
      OrderInVoiceListApi: '390',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'sw-KE-RafikiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'sw-KE',
      localeZH: '斯瓦希里语(肯尼亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6a2d8845-107f-47a8-bc4e-4b127a40108a.wav?sv=2019-07-07&sr=b&sig=ISvJpaWfsgGg7a4tM%2BuZrVgy5PeCuib2R89Meq1d%2FR4%3D&st=2021-10-29T08:28:13Z&se=2121-10-05T08:33:13Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'sw-KE-RafikiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '10d69bc3-2b42-4fb3-9b60-c1624f09a35f',
    locale: 'ur-IN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ur-IN, SalmanNeural)',
    properties: {
      DisplayName: 'Salman',
      DisplayVoiceName: 'SalmanNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'سلمان',
      LocaleDescription: 'ur-IN',
      OrderInVoiceListApi: '412',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ur-IN-SalmanNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ur-IN',
      localeZH: '乌尔都语(印度)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c80069b4-f41b-4f82-a96b-ecd23dd36e11.wav?sv=2019-07-07&sr=b&sig=RSfK0jYKMTDmZvFFvgvM0oAqERnHJmAY1VfwpyAcahw%3D&st=2021-10-29T08:31:24Z&se=2121-10-05T08:36:24Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ur-IN-SalmanNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'bc986aff-eabd-400a-b663-c1a401341324',
    locale: 'mr-IN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (mr-IN, AarohiNeural)',
    properties: {
      DisplayName: 'Aarohi',
      DisplayVoiceName: 'AarohiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'आरोही',
      LocaleDescription: 'mr-IN',
      OrderInVoiceListApi: '327',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'mr-IN-AarohiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'mr-IN',
      localeZH: '马拉地语(印度)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/978bb414-7a90-4bce-a94e-1ccc7f4a5613.wav?sv=2019-07-07&sr=b&sig=qWsFfahO2Y6vFpmJx7dMUTWpyA9QX3OfrOCV7SyZTVs%3D&st=2021-10-29T08:27:29Z&se=2121-10-05T08:32:29Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'mr-IN-AarohiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'c97e05df-dd50-494a-9550-c1c595382795',
    locale: 'mt-MT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (mt-MT, GraceNeural)',
    properties: {
      DisplayName: 'Grace',
      DisplayVoiceName: 'GraceNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Grace',
      LocaleDescription: 'Maltese (Malta)',
      OrderInVoiceListApi: '331',
      PreviewSentence:
        'L-Għodda għall-Ħolqien tal-Kontenut bil-Ħoss tħallik direttament tikkontrolla l-attributi tal-leħen minn fuq l-iskrin tiegħek.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'mt-MT-GraceNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'mt-MT',
      localeZH: '马耳他语(马耳他)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/09b0c2dd-f32a-4e88-acef-e02bbc4799a4.wav?sv=2019-07-07&sr=b&sig=Rl2jsLxGHHCKS%2BUW6NwQK5fpqCSgERQXz36pMddr6nQ%3D&st=2021-10-29T08:27:34Z&se=2121-10-05T08:32:34Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'mt-MT-GraceNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'cfec61f9-70e7-4726-accb-c1dbcc796b68',
    locale: 'km-KH',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (km-KH, SreymomNeural)',
    properties: {
      DisplayName: 'Sreymom',
      DisplayVoiceName: 'SreymomNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'ស្រីមុំ',
      LocaleDescription: 'km-KH',
      OrderInVoiceListApi: '304',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'km-KH-SreymomNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'km-KH',
      localeZH: '高棉语(柬埔寨)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/81e29034-cd41-4083-9dad-cfaf89268a42.wav?sv=2019-07-07&sr=b&sig=ed482rJxw490ZIDcV7CGuK6iG3%2B1YSUwAiQtmHlGfs0%3D&st=2021-10-29T08:31:10Z&se=2121-10-05T08:36:10Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'km-KH-SreymomNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '0a83aae7-926f-451b-bfb4-c3c0223e1c5b',
    locale: 'en-PH',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-PH, JamesNeural)',
    properties: {
      DisplayName: 'James',
      DisplayVoiceName: 'JamesNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'James',
      LocaleDescription: 'English (Philippines)',
      OrderInVoiceListApi: '120',
      PreviewSentence:
        'Audio Content Creation enables you to visually control speech attributes in real-time.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-PH-JamesNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-PH',
      localeZH: '英语(菲律宾)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d35cc8dd-fda9-4930-ba01-9b0d5793dbce.wav?sv=2019-07-07&sr=b&sig=a45NIwve%2FFRev%2FOP5%2BudCmldXgLYFD7zJrXnMWHS%2B3w%3D&st=2021-10-29T08:26:34Z&se=2121-10-05T08:31:34Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-PH-JamesNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '2eb006a0-2939-4131-b3ce-c465c1e60443',
    locale: 'es-HN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-HN, KarlaNeural)',
    properties: {
      DisplayName: 'Karla',
      DisplayVoiceName: 'KarlaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Karla',
      LocaleDescription: 'es-HN',
      OrderInVoiceListApi: '189',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-HN-KarlaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-HN',
      localeZH: '西班牙语(洪都拉斯)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c3726752-08d1-451c-a6a0-5a11a41cb0e9.wav?sv=2019-07-07&sr=b&sig=ckN9sIxBWW6kLdWiqGEVrOH40kJOcMBopC2WXzgZxQo%3D&st=2021-10-29T08:30:45Z&se=2121-10-05T08:35:45Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-HN-KarlaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '47153290-5326-4c71-bcc3-c4e3dea9837f',
    locale: 'es-CL',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-CL, LorenzoNeural)',
    properties: {
      DisplayName: 'Lorenzo',
      DisplayVoiceName: 'LorenzoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Lorenzo',
      LocaleDescription: 'es-CL',
      OrderInVoiceListApi: '158',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-CL-LorenzoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-CL',
      localeZH: '西班牙语(智利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e3c3ffc4-9230-49cc-9713-1bc2bbbfb048.wav?sv=2019-07-07&sr=b&sig=ji4XSKOFDNTHFhFhVbuiyPWN7wvinOGi%2Bj0LAOXPsiQ%3D&st=2021-10-29T08:30:31Z&se=2121-10-05T08:35:31Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-CL-LorenzoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '595e80a5-f5cc-4c75-9c4f-c5395abff9b5',
    locale: 'it-IT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (it-IT, FiammaNeural)',
    properties: {
      DisplayName: 'Fiamma',
      DisplayVoiceName: 'FiammaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Fiamma',
      LocaleDescription: 'Italian',
      OrderInVoiceListApi: '281',
      PreviewSentence:
        'Benvenuti nella piattaforma di generazione di contenuti audio di Microsoft.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'it-IT-FiammaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'it-IT',
      localeZH: '意大利语(意大利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/78b6b0c0-ec18-49bd-952c-f84faa465174.wav?sv=2019-07-07&sr=b&sig=twZvFyFnJh4QloaytGwU%2FKiVj4M4BqaoaAqqiGf2xt8%3D&st=2022-07-01T09:09:40Z&se=2122-06-07T09:14:40Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'it-IT-FiammaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'b15a636e-cdfe-47b5-b192-c5d192c6f30d',
    locale: 'bs-BA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (bs-BA, GoranNeural)',
    properties: {
      DisplayName: 'Goran',
      DisplayVoiceName: 'GoranNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Goran',
      LocaleDescription: 'bs-BA',
      OrderInVoiceListApi: '45',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'bs-BA-GoranNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'bs-BA',
      localeZH: '波斯尼亚语(波斯尼亚和黑塞哥维那)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4ce114d9-5610-4911-a55b-334e0f967747.wav?sv=2019-07-07&sr=b&sig=loqppZLETSUTP65lleVIBu4tj%2Ff9SHrCFDAUcibVBMQ%3D&st=2022-07-01T09:08:57Z&se=2122-06-07T09:13:57Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'bs-BA-GoranNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '9ae8f5ba-8753-4ccb-9e57-c5d8ff7454f3',
    locale: 'es-GT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-GT, AndresNeural)',
    properties: {
      DisplayName: 'Andres',
      DisplayVoiceName: 'AndresNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Andrés',
      LocaleDescription: 'es-GT',
      OrderInVoiceListApi: '186',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-GT-AndresNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-GT',
      localeZH: '西班牙语(危地马拉)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5453b579-365b-447a-9d30-93938aedae31.wav?sv=2019-07-07&sr=b&sig=bf8dgCxNlfSASn%2F2ODIzgL1cO1nYSm0ih6Arx2xhOfE%3D&st=2021-10-29T08:30:42Z&se=2121-10-05T08:35:42Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-GT-AndresNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'adeecc76-b274-496e-ac11-c691841d03d5',
    locale: 'es-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-ES, EstrellaNeural)',
    properties: {
      DisplayName: 'Estrella',
      DisplayVoiceName: 'EstrellaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Estrella',
      LocaleDescription: 'Spanish (Spain)',
      OrderInVoiceListApi: '174',
      PreviewSentence:
        'Crea aplicaciones y servicios que hablan de forma natural a los usuarios, y que mejoran la accesibilidad y la facilidad de uso.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-ES-EstrellaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-ES',
      localeZH: '西班牙语(西班牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/59f2a05c-4bd8-4387-8133-9c493898d053.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A41Z&se=2122-09-27T07%3A51%3A41Z&sr=b&sp=rl&sig=KX5PVJeIo2xbmIY6Vra%2BpW1xH7qDl7YcPe%2FXMMDwKOI%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-ES-EstrellaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'd3fc4288-e3f5-4ce8-b70e-c69ba7bd2405',
    locale: 'el-GR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (el-GR, NestorasNeural)',
    properties: {
      DisplayName: 'Nestoras',
      DisplayVoiceName: 'NestorasNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Νέστορας',
      LocaleDescription: 'Greek',
      OrderInVoiceListApi: '76',
      PreviewSentence:
        'Δημιουργήστε εφαρμογές και υπηρεσίες που μιλούν με φυσικό τρόπο στους χρήστες, βελτιώνοντας την προσβασιμότητα και τη χρηστικότητα.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'el-GR-NestorasNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'el-GR',
      localeZH: '希腊语(希腊)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/329b2788-1d03-43e8-a686-682be925f1ca.wav?sv=2019-07-07&sr=b&sig=h7tzFpM%2Bw6lLUQxbOAkYqE8mMy%2FNrPAb9WtKDFmReGQ%3D&st=2021-10-29T08:26:17Z&se=2121-10-05T08:31:17Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'el-GR-NestorasNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '62b4fcd7-d79c-465a-a23b-c703087f840d',
    locale: 'fr-FR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-FR, JeromeNeural)',
    properties: {
      DisplayName: 'Jerome',
      DisplayVoiceName: 'JeromeNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Jerome',
      LocaleDescription: 'French (France)',
      OrderInVoiceListApi: '249',
      PreviewSentence:
        "Construire des applications et services qui communiquent naturellement avec l'utilisateur, en améliorant l’accessibilité et l'ergonomie",
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-FR-JeromeNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fr-FR',
      localeZH: '法语(法国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f1b6a5ec-2042-4976-a496-22b586155b2d.wav?sv=2019-07-07&sr=b&sig=Vm%2FvktS9fkjCtLmnTt6RXlbWsKWmq5KTWyLVEAo%2BMtg%3D&st=2022-01-19T09:01:33Z&se=2121-12-26T09:06:33Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fr-FR-JeromeNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'ca1ba396-e04b-44b1-a9b3-c74433344372',
    locale: 'bn-IN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (bn-IN, TanishaaNeural)',
    properties: {
      DisplayName: 'Tanishaa',
      DisplayVoiceName: 'TanishaaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'তানিশা',
      LocaleDescription: 'bn-IN',
      OrderInVoiceListApi: '44',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'bn-IN-TanishaaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'bn-IN',
      localeZH: '孟加拉语(印度)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f3f9d7fb-6c20-4e3a-a225-35412bff732f.wav?sv=2019-07-07&sr=b&sig=bqyGvVvnrnDd3hKLraV%2BRPR84vsRJVIcMYw%2BmJJCvEM%3D&st=2022-01-19T09:00:18Z&se=2121-12-26T09:05:18Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'bn-IN-TanishaaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '74554b1d-12ba-4dc6-8e47-c7b4436ea586',
    locale: 'ar-SY',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-SY, AmanyNeural)',
    properties: {
      DisplayName: 'Amany',
      DisplayVoiceName: 'AmanyNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'أماني',
      LocaleDescription: 'ar-SY',
      OrderInVoiceListApi: '31',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-SY-AmanyNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-SY',
      localeZH: '阿拉伯语(叙利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/21779fc8-eff6-4623-b37f-94b3fa8dcd3e.wav?sv=2019-07-07&sr=b&sig=Vr9Cga%2BOVc%2BsI4r2faFgyq2FFSRqS0r0%2FWs6XcB23k4%3D&st=2021-10-29T08:30:15Z&se=2121-10-05T08:35:15Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-SY-AmanyNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '17fc7d8c-986b-48a1-889b-c84125a39f6c',
    locale: 'en-GB',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-GB, ElliotNeural)',
    properties: {
      DisplayName: 'Elliot',
      DisplayVoiceName: 'ElliotNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Elliot',
      LocaleDescription: 'English (UK)',
      OrderInVoiceListApi: '97',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-GB-ElliotNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-GB',
      localeZH: '英语(英国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d16e2852-ce07-4e58-9950-346cfea0b089.wav?sv=2019-07-07&sr=b&sig=ZqodlS4akSHbUtTjYRDNJgNR0RWXHFpxji8G22ofego%3D&st=2022-01-19T09:00:58Z&se=2121-12-26T09:05:58Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-GB-ElliotNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '3d94051c-960b-45e6-b0ab-c8e0cfe4e2d5',
    locale: 'es-PR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-PR, VictorNeural)',
    properties: {
      DisplayName: 'Victor',
      DisplayVoiceName: 'VictorNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Víctor',
      LocaleDescription: 'es-PR',
      OrderInVoiceListApi: '212',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-PR-VictorNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-PR',
      localeZH: '西班牙语(波多黎各)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/384779a4-19f7-4893-b982-1c8795d49395.wav?sv=2019-07-07&sr=b&sig=o17n22A%2BlcJ7jrj03d87kjZWX9l7cVNDx6YK7GrmsAw%3D&st=2021-10-29T08:30:53Z&se=2121-10-05T08:35:53Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-PR-VictorNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'ec99c388-f6f5-428a-9c22-c9143b16d50c',
    locale: 'pt-BR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pt-BR, ElzaNeural)',
    properties: {
      DisplayName: 'Elza',
      DisplayVoiceName: 'ElzaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Elza',
      LocaleDescription: 'Portuguese (Brazil)',
      OrderInVoiceListApi: '354',
      PreviewSentence:
        'Desenvolver aplicativos e serviços que conversem naturalmente com os usuários, melhorando a acessibilidade e usabilidade.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pt-BR-ElzaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'pt-BR',
      localeZH: '葡萄牙语(巴西)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/3107bd9f-d47a-47f5-9bc0-07c7980e055d.wav?sv=2019-07-07&sr=b&sig=MNpFARkd43OmSAlYgdqE55z%2F%2BQAGbFPocF69Ps%2Fjigo%3D&st=2022-07-01T09:10:15Z&se=2122-06-07T09:15:15Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'pt-BR-ElzaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '0ee78d2e-9350-4111-86c5-c948ca6cd695',
    locale: 'ar-IQ',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-IQ, RanaNeural)',
    properties: {
      DisplayName: 'Rana',
      DisplayVoiceName: 'RanaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'رنا',
      LocaleDescription: 'ar-IQ',
      OrderInVoiceListApi: '14',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-IQ-RanaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-IQ',
      localeZH: '阿拉伯语(伊拉克)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/874e4ee2-800c-4bea-90ff-e72481afc231.wav?sv=2019-07-07&sr=b&sig=GDsh2RizhoPV1FWKR6XN0BRydQwaGE7GwabkClGrftM%3D&st=2021-10-29T08:30:04Z&se=2121-10-05T08:35:04Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-IQ-RanaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '146398ff-b6c8-4958-99ca-c965d3d12548',
    locale: 'en-AU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-AU, JoanneNeural)',
    properties: {
      DisplayName: 'Joanne',
      DisplayVoiceName: 'JoanneNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Joanne',
      LocaleDescription: 'English (Australia)',
      OrderInVoiceListApi: '83',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-AU-JoanneNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-AU',
      localeZH: '英语(澳大利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/040e21fa-1768-458d-8b58-d1138d5acd01.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A10Z&se=2122-09-27T07%3A51%3A10Z&sr=b&sp=rl&sig=39l5Wo8YUtFTzdmXPxxZnadRINOaIO50OB58oDA7RCw%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-AU-JoanneNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '003c43b7-a265-41b9-b8cf-c978d9e4deb8',
    locale: 'en-GB',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-GB, NoahNeural)',
    properties: {
      DisplayName: 'Noah',
      DisplayVoiceName: 'NoahNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Noah',
      LocaleDescription: 'English (UK)',
      OrderInVoiceListApi: '101',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-GB-NoahNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-GB',
      localeZH: '英语(英国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/73fb2ea3-681a-4d3c-82cd-d5dd0423f24a.wav?sv=2019-07-07&sr=b&sig=yYJESl2gVTRMd5ic4QTyNVLUqD17vV6Pc6ACdK5%2BMxM%3D&st=2022-01-19T09:01:08Z&se=2121-12-26T09:06:08Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-GB-NoahNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '32b1bdde-2225-4429-ad61-ca23a9d41a42',
    locale: 'ms-MY',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ms-MY, YasminNeural)',
    properties: {
      DisplayName: 'Yasmin',
      DisplayVoiceName: 'YasminNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Yasmin',
      LocaleDescription: 'Malay',
      OrderInVoiceListApi: '330',
      PreviewSentence:
        'Membina aplikasi dan perkhidmatan yang bercakap sememangnya kepada pengguna, meningkatkan kebolehcapaian dan kebolehgunaanya.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ms-MY-YasminNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ms-MY',
      localeZH: '马来语(马来西亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4816b19b-e8c6-4729-b68d-97830942e54e.wav?sv=2019-07-07&sr=b&sig=PUHJOkor2WK3KboySG5a67RHyyavKdgJ6Rd%2FLG7KGIg%3D&st=2021-10-29T08:27:33Z&se=2121-10-05T08:32:33Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ms-MY-YasminNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '66dca810-157a-48a7-9a9c-cac3147734e8',
    locale: 'zh-CN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, YunxiaNeural)',
    properties: {
      DisplayName: 'Yunxia',
      DisplayVoiceName: 'YunxiaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      IsTtsBreakEnabled: 'True',
      LocalName: '云夏',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '440',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-YunxiaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions:
        '{"calm":2,"fearful":3,"cheerful":4,"angry":7,"sad":8,"Default":0}',
      VoiceStyleNames: 'calm,fearful,cheerful,angry,sad,Default',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/26fdc8e5-5d0a-4f07-9c2a-1b976804b4ab.wav?sv=2019-07-07&sr=b&sig=55aHYXra49WNZ92zafOJvdJIL55FjKfxOH3srv6gxOk%3D&st=2022-07-20T10:58:52Z&se=2122-06-26T11:03:52Z&sp=rl',
          styleName: 'Angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/26fdc8e5-5d0a-4f07-9c2a-1b976804b4ab.wav?sv=2019-07-07&sr=b&sig=55aHYXra49WNZ92zafOJvdJIL55FjKfxOH3srv6gxOk%3D&st=2022-07-20T10:58:52Z&se=2122-06-26T11:03:52Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4932b47c-3395-43b2-877c-6f321c95d5e9.wav?sv=2019-07-07&sr=b&sig=QDDxzkKcmOjjDN64knDqnQ7II4Iezc4LH%2F4GKJ8soIU%3D&st=2022-07-20T10:58:55Z&se=2122-06-26T11:03:55Z&sp=rl',
          styleName: 'Calm',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4932b47c-3395-43b2-877c-6f321c95d5e9.wav?sv=2019-07-07&sr=b&sig=QDDxzkKcmOjjDN64knDqnQ7II4Iezc4LH%2F4GKJ8soIU%3D&st=2022-07-20T10:58:55Z&se=2122-06-26T11:03:55Z&sp=rl',
          styleName: 'calm',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/cff3e19e-db5a-42f5-a8a9-b50b3f5ba95c.wav?sv=2019-07-07&sr=b&sig=Su%2FjrsI58sDzyYbUMVjl3zzoJRNXR9V0R8IZD7gyaxE%3D&st=2022-07-20T10:58:58Z&se=2122-06-26T11:03:58Z&sp=rl',
          styleName: 'Cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/cff3e19e-db5a-42f5-a8a9-b50b3f5ba95c.wav?sv=2019-07-07&sr=b&sig=Su%2FjrsI58sDzyYbUMVjl3zzoJRNXR9V0R8IZD7gyaxE%3D&st=2022-07-20T10:58:58Z&se=2122-06-26T11:03:58Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/9563d6e4-ac15-43ca-bb31-8f43e7420116.wav?sv=2019-07-07&sr=b&sig=fp6Ynh%2Bl0iugD6O9ndI099aNsnee%2FFuy34ky2WJ6psQ%3D&st=2022-07-20T10:59:01Z&se=2122-06-26T11:04:01Z&sp=rl',
          styleName: 'Fearful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/9563d6e4-ac15-43ca-bb31-8f43e7420116.wav?sv=2019-07-07&sr=b&sig=fp6Ynh%2Bl0iugD6O9ndI099aNsnee%2FFuy34ky2WJ6psQ%3D&st=2022-07-20T10:59:01Z&se=2122-06-26T11:04:01Z&sp=rl',
          styleName: 'fearful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6d597103-4256-4c06-ab6e-223ab6740403.wav?sv=2019-07-07&sr=b&sig=rAXC%2BxBQZb9vuR6yeTxhFjOwa6qmG3gjBOvDkor%2BtN4%3D&st=2022-07-20T10:59:03Z&se=2122-06-26T11:04:03Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/babc6c83-0ba6-418b-9ebb-b9451aa6b9be.wav?sv=2019-07-07&sr=b&sig=%2BET2DBG5SvyZ1%2BILRAearVk8vsN%2BRMJPes6SOpXb6Cg%3D&st=2022-07-20T10:59:06Z&se=2122-06-26T11:04:06Z&sp=rl',
          styleName: 'Sad',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/babc6c83-0ba6-418b-9ebb-b9451aa6b9be.wav?sv=2019-07-07&sr=b&sig=%2BET2DBG5SvyZ1%2BILRAearVk8vsN%2BRMJPes6SOpXb6Cg%3D&st=2022-07-20T10:59:06Z&se=2122-06-26T11:04:06Z&sp=rl',
          styleName: 'sad',
        },
      ],
    },
    shortName: 'zh-CN-YunxiaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '768e2ad7-c79c-474a-b0ce-cb70c88cd6cb',
    locale: 'sq-AL',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (sq-AL, IlirNeural)',
    properties: {
      DisplayName: 'Ilir',
      DisplayVoiceName: 'IlirNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Ilir',
      LocaleDescription: 'sq-AL',
      OrderInVoiceListApi: '382',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'sq-AL-IlirNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'sq-AL',
      localeZH: '阿尔巴尼亚语(阿尔巴尼亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/94fd259a-2c60-4fd2-a237-cdf458b0f978.wav?sv=2019-07-07&sr=b&sig=DFTcuZ%2FF9QZ6ufSWrd6H1Mqw7VTbYEEtg%2BNfdlSokEY%3D&st=2022-07-01T09:10:43Z&se=2122-06-07T09:15:43Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'sq-AL-IlirNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'c83eb52e-00f0-4502-839c-cba3b162bd65',
    locale: 'ps-AF',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ps-AF, LatifaNeural)',
    properties: {
      DisplayName: 'Latifa',
      DisplayVoiceName: 'LatifaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'لطيفه',
      LocaleDescription: 'ps-AF',
      OrderInVoiceListApi: '349',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ps-AF-LatifaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ps-AF',
      localeZH: '普什图语(阿富汗)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5a5d6c0e-6fba-4fbb-b601-d609fb91402f.wav?sv=2019-07-07&sr=b&sig=O8iFZ6MxDVDB5P35f55pv%2Fy7bSu%2F0h5aORTM3K40KfA%3D&st=2022-01-19T09:02:13Z&se=2121-12-26T09:07:13Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ps-AF-LatifaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '735f99de-0526-44f9-a4ac-cbbd845b0aba',
    locale: 'de-DE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (de-DE, KasperNeural)',
    properties: {
      DisplayName: 'Kasper',
      DisplayVoiceName: 'KasperNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Kasper',
      LocaleDescription: 'German (Germany)',
      OrderInVoiceListApi: '66',
      PreviewSentence:
        'Erstellen Sie Apps und Dienste, die auf natürliche Art mit Nutzern sprechen, und dadurch Zugänglichkeit und Benutzerfreundlichkeit verbessern.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'de-DE-KasperNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'de-DE',
      localeZH: '德语(德国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d972755a-944b-4388-9efd-fc497c65c59b.wav?sv=2019-07-07&sr=b&sig=z8kzqkI8P9%2FkNHvggU2tEwlvg5BxI6fW3WHLBYDHxGI%3D&st=2022-01-19T09:00:32Z&se=2121-12-26T09:05:32Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'de-DE-KasperNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '7f555f91-be6f-44e6-afe9-cbf590fc7f24',
    locale: 'pt-BR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pt-BR, FranciscaNeural)',
    properties: {
      DisplayName: 'Francisca',
      DisplayVoiceName: 'FranciscaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Francisca',
      LocaleDescription: 'Portuguese (Brazil)',
      OrderInVoiceListApi: '350',
      PreviewSentence:
        'Desenvolver aplicativos e serviços que conversem naturalmente com os usuários, melhorando a acessibilidade e usabilidade.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pt-BR-FranciscaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"calm":2,"Default":0}',
      VoiceStyleNames: 'calm,Default',
      locale: 'pt-BR',
      localeZH: '葡萄牙语(巴西)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c2f78ffd-a7a3-403f-b30b-9a3484ccc74e.wav?sv=2019-07-07&sr=b&sig=rwvD1noqGpkl1mwt7tcMxgKylgZrE1b0QiSwaHT8tO0%3D&st=2021-10-29T08:27:53Z&se=2121-10-05T08:32:53Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/9118e675-7543-4f78-8184-25929d273a66.wav?sv=2019-07-07&sr=b&sig=HJGKfZsZbElvUrw3q9%2Bs7EOTKLXGpYjb4HPVluDOTgk%3D&st=2021-10-29T08:27:54Z&se=2121-10-05T08:32:54Z&sp=rl',
          styleName: 'calm',
        },
      ],
    },
    shortName: 'pt-BR-FranciscaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '7cd10960-9387-4084-b22d-cc0897e0a17b',
    locale: 'ar-LB',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-LB, RamiNeural)',
    properties: {
      DisplayName: 'Rami',
      DisplayVoiceName: 'RamiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'رامي',
      LocaleDescription: 'ar-LB',
      OrderInVoiceListApi: '20',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-LB-RamiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-LB',
      localeZH: '阿拉伯语(黎巴嫩)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e70eb2f0-6fce-49de-b943-ca1dea79333c.wav?sv=2019-07-07&sr=b&sig=wkLELI%2FZO9pml7sf5v3zLaXb4erOH839enDc7uuFo6I%3D&st=2022-07-01T09:08:45Z&se=2122-06-07T09:13:45Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-LB-RamiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '27e2f1c8-cfe0-4324-88e2-cd0bafeffe1b',
    locale: 'en-US',
    masterpieces: [
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/70acf8bf-0de7-4ee7-925e-cb17b32bcdcf.wav?sv=2019-07-07&sr=b&sig=pSobsIM%2B7FNWRJEhezgYdo6m3M%2Fhrgo%2BwsJbniL%2BC3k%3D&st=2021-10-29T08:25:19Z&se=2121-10-05T08:30:19Z&sp=rl',
        categories: ['Audiobook'],
        title: 'AudioBook',
        titleLocalizationId: 'AudioBook',
      },
    ],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, AriaNeural)',
    properties: {
      DisplayName: 'Aria',
      DisplayVoiceName: 'AriaNeural',
      ExpressAsRoleValues:
        'Default,Role,Cheerful,CustomerService,Empathy,Newscast-Formal,Newscast-Casual',
      ExpressAsStyleValues:
        'Default,Role,Cheerful,CustomerService,Empathy,Newscast-Formal,Newscast-Casual',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Aria',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '131',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-AriaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"cheerful":4,"angry":7,"sad":8,"Default":0}',
      VoiceStyleNames:
        'chat,customerservice,narration-professional,newscast-casual,newscast-formal,cheerful,empathetic,angry,sad,excited,friendly,terrified,shouting,unfriendly,whispering,hopeful,Default',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2dddd564-e051-43c7-995a-e9dc44d75d5e.wav?sv=2019-07-07&sr=b&sig=AqZqilK6%2FwD5IJeOl%2BopdsUg8OJBC6VSU6RtWCdBE0Y%3D&st=2021-10-29T08:25:24Z&se=2121-10-05T08:30:24Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/310e03ce-bb12-469b-a3d4-be08ed1e85dd.wav?sv=2019-07-07&sr=b&sig=X0%2F1aaP60LWSIVMFlr8QcwkpQ1D2eqDSoGMgEtbl93Y%3D&st=2021-10-29T08:25:15Z&se=2121-10-05T08:30:15Z&sp=rl',
          styleName: 'chat',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f07bc6f3-a02e-4670-b224-6f8ebbe0f448.wav?sv=2019-07-07&sr=b&sig=SRPo5BQx3fGKJVBlsAuCkz3Maaa1WjPHgUkjHODpGLg%3D&st=2021-10-29T08:25:17Z&se=2121-10-05T08:30:17Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/065d4498-d65c-4d57-893e-993b74562d51.wav?sv=2019-07-07&sr=b&sig=TQYaF0J%2Be0j2g2OEFWTji9vXJbMmbEAJSRe9I%2Fqxuik%3D&st=2021-10-29T08:25:18Z&se=2121-10-05T08:30:18Z&sp=rl',
          styleName: 'customerservice',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0e6971ad-df27-432c-b58b-1dded094018b.wav?sv=2019-07-07&sr=b&sig=kPFXdtU%2BtEMV9KyBfhb3EIxTZ8JdGPqWbS9Xckm66eM%3D&st=2021-10-29T08:25:20Z&se=2121-10-05T08:30:20Z&sp=rl',
          styleName: 'empathetic',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/25dbeb80-232b-4d96-b33f-f96fb184de68.wav?sv=2019-07-07&sr=b&sig=bs9i27WRQ%2BspKL2KptLH0%2BAyUgkGsYld6ijAYvVHK3U%3D&st=2021-10-29T08:25:27Z&se=2121-10-05T08:30:27Z&sp=rl',
          styleName: 'narration-professional',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2cfcfad6-1cb7-4ef3-ac1a-a3a70a56a1e3.wav?sv=2019-07-07&sr=b&sig=bseV3KzTMpWSs8VFSZWWA4Qzt0DyM0ntvowaDZDR90Y%3D&st=2021-10-29T08:25:28Z&se=2121-10-05T08:30:28Z&sp=rl',
          styleName: 'newscast-casual',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/80c94e01-f16c-4245-bc45-92690462795e.wav?sv=2019-07-07&sr=b&sig=mwZ2jtG7kgVoE3phoKoLUe4%2B7rpvSKPSf8zVqtPN4tQ%3D&st=2021-10-29T08:25:29Z&se=2121-10-05T08:30:29Z&sp=rl',
          styleName: 'newscast-formal',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/73af2f50-13af-486f-8520-ccfc6d4f8a09.wav?sv=2019-07-07&sr=b&sig=FyUr%2FtYXLu63wFDty9M63euQ8Qf9Mw7eNgQ3wkRkn4o%3D&st=2022-04-28T10:25:25Z&se=2122-04-04T10:30:25Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/02356840-2d3b-432d-8fbb-b2dee52d69a2.wav?sv=2019-07-07&sr=b&sig=P59eOqmEBTP9gpXO%2BCYuj%2Fq3UI%2Be9%2Fe1mzZD2XaIRxI%3D&st=2022-04-28T10:25:27Z&se=2122-04-04T10:30:27Z&sp=rl',
          styleName: 'excited',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/823f988c-4ebf-4255-a248-fd6dccdd64c1.wav?sv=2019-07-07&sr=b&sig=sVYCRQbDe4Kg3mFDnXtd7Obzw54%2F%2F482K7lLgsOzlgQ%3D&st=2022-04-28T10:25:29Z&se=2122-04-04T10:30:29Z&sp=rl',
          styleName: 'friendly',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/436d9d4c-650d-4df1-a9a9-5d43d48f66c3.wav?sv=2019-07-07&sr=b&sig=YOhBWOU6wGUjJE9GITx6elUBIXvXfu3Lpf3cNzyBXLs%3D&st=2022-04-28T10:25:32Z&se=2122-04-04T10:30:32Z&sp=rl',
          styleName: 'hopeful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c124161f-c118-45ac-a7ac-0e9f38033281.wav?sv=2019-07-07&sr=b&sig=nfY2UcRn%2BWcXm%2Brtl0r1GcE6sQlMHnQs5WIUYqXa1ws%3D&st=2022-04-28T10:25:34Z&se=2122-04-04T10:30:34Z&sp=rl',
          styleName: 'sad',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fa5016db-52bf-4467-8656-87505ac94792.wav?sv=2019-07-07&sr=b&sig=tGgN91GcXXf3d1BPpF91HTT0TnjkErVEyrg2mzIc9M4%3D&st=2022-04-28T10:25:36Z&se=2122-04-04T10:30:36Z&sp=rl',
          styleName: 'shouting',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/47daeb74-2d30-4384-bf24-e560dad0e9e9.wav?sv=2019-07-07&sr=b&sig=Y%2Bm8BIE2HEJCfW0Cao6Swrj2%2BDz4OuHvCcOURnphcv8%3D&st=2022-04-28T10:25:38Z&se=2122-04-04T10:30:38Z&sp=rl',
          styleName: 'terrified',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/57eb5a8c-1dd8-449a-8ffa-6b1207bef7e0.wav?sv=2019-07-07&sr=b&sig=jcPSOu3cF1xyDgkGNQ7%2FLwm7%2BmbVKev69Ddboj2RxRI%3D&st=2022-04-28T10:25:41Z&se=2122-04-04T10:30:41Z&sp=rl',
          styleName: 'unfriendly',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4ed6974e-108d-449f-b40d-f609af899f70.wav?sv=2019-07-07&sr=b&sig=l2PkWiOqEMnG2hxUXbefDMGFppihmj%2BC4sOr24SHd2E%3D&st=2022-04-28T10:25:43Z&se=2122-04-04T10:30:43Z&sp=rl',
          styleName: 'whispering',
        },
      ],
    },
    shortName: 'en-US-AriaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '9b72401f-fa3f-447d-ac89-cd479814d3fe',
    locale: 'es-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-ES, LaiaNeural)',
    properties: {
      DisplayName: 'Laia',
      DisplayVoiceName: 'LaiaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Laia',
      LocaleDescription: 'Spanish (Spain)',
      OrderInVoiceListApi: '176',
      PreviewSentence:
        'Crea aplicaciones y servicios que hablan de forma natural a los usuarios, y que mejoran la accesibilidad y la facilidad de uso.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-ES-LaiaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-ES',
      localeZH: '西班牙语(西班牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/8188850f-0f9f-4ead-9dd5-509ed2c9f9e3.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A45Z&se=2122-09-27T07%3A51%3A45Z&sr=b&sp=rl&sig=824D0E6bLDJ48xHPuRPH7PCJ0R5gMyT5HYgpUs54YVg%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-ES-LaiaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'f645c2c3-9181-442a-8d96-cda1eeffcccf',
    locale: 'sl-SI',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (sl-SI, PetraNeural)',
    properties: {
      DisplayName: 'Petra',
      DisplayVoiceName: 'PetraNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Petra',
      LocaleDescription: 'Slovenian',
      OrderInVoiceListApi: '377',
      PreviewSentence:
        'Razvijajte aplikacije in storitve z vrhunsko uporabniško izkušnjo povsem po meri uporabnikov.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'sl-SI-PetraNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'sl-SI',
      localeZH: '斯洛文尼亚语(斯洛文尼亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/08c86987-7391-4c29-b6b8-4724e8735f60.wav?sv=2019-07-07&sr=b&sig=S0UULJLcT8%2BV1%2Ba5Y%2FK%2B8B%2BTKT3Y0JksdfaYPSZxmKo%3D&st=2021-10-29T08:28:08Z&se=2121-10-05T08:33:08Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'sl-SI-PetraNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '176f4338-31e6-43a9-bba2-cefe6bda182b',
    locale: 'fr-CA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-CA, JeanNeural)',
    properties: {
      DisplayName: 'Jean',
      DisplayVoiceName: 'JeanNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Jean',
      LocaleDescription: 'French (Canada)',
      OrderInVoiceListApi: '238',
      PreviewSentence:
        'Créer des applications et des services qui parlent aux utilisateurs, améliorant ainsi l’accessibilité et la facilité d’utilisation.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-CA-JeanNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fr-CA',
      localeZH: '法语(加拿大)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/7803a5be-516c-4b42-a09e-e59b377a629c.wav?sv=2019-07-07&sr=b&sig=tIGYLLBnXK9rdGcB7kzU2wMLU1lfZ0EahVOOpZvKZUE%3D&st=2021-10-29T08:26:57Z&se=2121-10-05T08:31:57Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fr-CA-JeanNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '46171608-fa13-4517-970b-cf3f7b9c00fb',
    locale: 'en-AU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-AU, NeilNeural)',
    properties: {
      DisplayName: 'Neil',
      DisplayVoiceName: 'NeilNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Neil',
      LocaleDescription: 'English (Australia)',
      OrderInVoiceListApi: '87',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-AU-NeilNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-AU',
      localeZH: '英语(澳大利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/1f65799d-7a64-4a54-96dd-89bfc7571966.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A17Z&se=2122-09-27T07%3A51%3A17Z&sr=b&sp=rl&sig=RSmdo3AnQg4dGBZOl9EuNL0kapcoCrtNO1nZAbX2APQ%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-AU-NeilNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '948c1dbe-75f6-4c3e-a7be-cfa1ecdc2c9c',
    locale: 'en-IE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-IE, EmilyNeural)',
    properties: {
      DisplayName: 'Emily',
      DisplayVoiceName: 'EmilyNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Emily',
      LocaleDescription: 'English (Ireland)',
      OrderInVoiceListApi: '111',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-IE-EmilyNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-IE',
      localeZH: '英语(爱尔兰)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/38c2a8dd-4ebd-4c38-8a3a-7c63eb6fe509.wav?sv=2019-07-07&sr=b&sig=Q3clSFCRpV6bPrOZzGjsYu6PtJHdwv91aTxZJoBFDZ8%3D&st=2021-10-29T08:26:29Z&se=2121-10-05T08:31:29Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-IE-EmilyNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '3a5b7702-dcd4-449f-ae08-cfc31b8f5fa4',
    locale: 'fr-FR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-FR, CoralieNeural)',
    properties: {
      DisplayName: 'Coralie',
      DisplayVoiceName: 'CoralieNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Coralie',
      LocaleDescription: 'French (France)',
      OrderInVoiceListApi: '245',
      PreviewSentence:
        "Construire des applications et services qui communiquent naturellement avec l'utilisateur, en améliorant l’accessibilité et l'ergonomie",
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-FR-CoralieNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fr-FR',
      localeZH: '法语(法国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6203ebbb-0b52-49db-982e-af8ab1ed2e49.wav?sv=2019-07-07&sr=b&sig=79G4c2LvqE2jfu43tdFbxwaTPHJYypJQBv0d54CfMbQ%3D&st=2022-01-19T09:01:26Z&se=2121-12-26T09:06:26Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fr-FR-CoralieNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'd79eb070-849b-4b42-9ef5-d077965b075a',
    locale: 'zh-CN-henan',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN-henan, YundengNeural)',
    properties: {
      DisplayName: 'Yundeng',
      DisplayVoiceName: 'YundengNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: '云登',
      LocaleDescription: 'zh-CN-henan',
      OrderInVoiceListApi: '444',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-henan-YundengNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'zh-CN-henan',
      localeZH: '中文(中原官话河南，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/54737f06-eb51-4bb6-955f-4cde2512306c.wav?sv=2019-07-07&sr=b&sig=xhYBA%2FeMpKyMZlb44kStghhEwReNT4cWSS1ws7EAyx8%3D&st=2022-08-30T08:13:44Z&se=2122-08-06T08:18:44Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'zh-CN-henan-YundengNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [
      {
        name: 'All',
      },
      {
        defaultStyle: 'narration',
        name: 'Audiobook',
      },
      {
        name: 'Newscast',
      },
      {
        defaultStyle: 'customerservice',
        name: 'CustomerService',
      },
    ],
    description: 'Profesional and fluent voice with multiple scenario styles.',
    id: 'cb034992-a2f3-4752-80ba-d093ddc3ea29',
    locale: 'zh-CN',
    masterpieces: [
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/74a6b993-5efc-459b-9818-9af7b1fb144d?sv=2019-07-07&sr=b&sig=VV7Rnzr8i8TWDZ3kM97u2Qn%2BBzvCldjCbNC2sqTvWfk%3D&st=2021-10-29T08:24:56Z&se=2121-10-05T08:29:56Z&sp=rl',
        categories: ['Newscast'],
        title: '云扬新闻',
        titleLocalizationId: 'zh-CN-YunyangNeural_Masterpieces_Newscast_YunyangNews_Title',
      },
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b3b51b57-9daf-46a5-b8a5-459269bef018.wav?sv=2019-07-07&sr=b&sig=9Gb4fOcbdt%2FlPS61ydOoXTu7eZohcBmdai%2B0NRWKyeE%3D&st=2021-10-29T08:31:49Z&se=2121-10-05T08:36:49Z&sp=rl',
        categories: ['Audiobook'],
        title: '北平的四季',
        titleLocalizationId: 'zh-CN-YunyangNeural_Masterpieces_AudioBook_BeiPingDeSiJi_Title',
      },
    ],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, YunyangNeural)',
    properties: {
      AgeGroups: 'YoungAdult',
      DisplayName: 'Yunyang',
      DisplayVoiceName: 'YunyangNeural',
      ExpressAsRoleValues: 'Default,CustomerService',
      ExpressAsStyleValues: 'Default,CustomerService',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      IconFileEndpointWithSas:
        'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a8f66151-2121-42cd-a509-e09c4ef19124?sv=2019-07-07&sr=b&sig=HY2%2B%2B4Pny%2FkwwnQxNCih5GIDoKKcDhFCNHfk%2BrHBzwc%3D&st=2021-10-29T08:24:29Z&se=2121-10-05T08:29:29Z&sp=rl',
      IsTtsBreakEnabled: 'True',
      LocalName: '云扬',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '424',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-YunyangNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"Default":0}',
      VoiceStyleNames: 'customerservice,narration-professional,newscast-casual,Default',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0925e86d-53dc-40d0-a3d0-465ade2b5a52.wav?sv=2019-07-07&sr=b&sig=TZ3uvAF2%2FPNX%2BAlui6DOrCFUXX3iiWpFn0wysy1r5TI%3D&st=2021-10-29T08:29:32Z&se=2121-10-05T08:34:32Z&sp=rl',
          styleName: 'customerservice',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/63410827-ad77-4319-868e-d01c1424e6ac.wav?sv=2019-07-07&sr=b&sig=Ixbp8gIrmryV3TBkP4tHtCHI%2FfC0w8tsVXvVNJElOws%3D&st=2021-10-29T08:29:33Z&se=2121-10-05T08:34:33Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/43238341-d95a-45d7-be16-87fc673bb23f.wav?sv=2019-07-07&sr=b&sig=d5QWx9FtvJhP3K8zD4kADz4nx050lcFLDXlViIc1etI%3D&st=2021-10-29T08:29:34Z&se=2121-10-05T08:34:34Z&sp=rl',
          styleName: 'narration',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/def47a02-8ebf-4ece-816c-802c9d488bf6.wav?sv=2019-07-07&sr=b&sig=shHae61piefsqRNZKLDcG%2Bz4YG4ggrgbAAvM5c9M2BQ%3D&st=2022-03-01T04:37:24Z&se=2122-02-05T04:42:24Z&sp=rl',
          styleName: 'narration-professional',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/df88a4f3-0a4d-471c-8c0c-f5a4021767cf.wav?sv=2019-07-07&sr=b&sig=c%2BfgmEPFKPiwF0G%2Fb0nsrSo%2FhgdLt9PO%2F6MTpNt1TOA%3D&st=2022-02-25T16:13:22Z&se=2122-02-01T16:18:22Z&sp=rl',
          styleName: 'newscast-casual',
        },
      ],
    },
    shortName: 'zh-CN-YunyangNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '5289eaf1-53e6-47e6-9844-d0b4a795897e',
    locale: 'fr-FR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-FR, JacquelineNeural)',
    properties: {
      DisplayName: 'Jacqueline',
      DisplayVoiceName: 'JacquelineNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Jacqueline',
      LocaleDescription: 'French (France)',
      OrderInVoiceListApi: '248',
      PreviewSentence:
        "Construire des applications et services qui communiquent naturellement avec l'utilisateur, en améliorant l’accessibilité et l'ergonomie",
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-FR-JacquelineNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fr-FR',
      localeZH: '法语(法国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fa363447-ba01-4916-bbc5-4cc57cd2d1cb.wav?sv=2019-07-07&sr=b&sig=M91m4B0eUVmvOErAB0DKSFtp%2Ft7YM2pn1sRNgXYkCYM%3D&st=2022-01-19T09:01:31Z&se=2121-12-26T09:06:31Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fr-FR-JacquelineNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '6a0571cc-d619-41a8-b959-d19ee45dcbc6',
    locale: 'ta-MY',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ta-MY, SuryaNeural)',
    properties: {
      DisplayName: 'Surya',
      DisplayVoiceName: 'SuryaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'சூர்யா',
      LocaleDescription: 'ta-MY',
      OrderInVoiceListApi: '399',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ta-MY-SuryaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ta-MY',
      localeZH: '泰米尔语(马来西亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c92d6619-347a-411e-90e9-15a5d293611f.wav?sv=2019-07-07&sr=b&sig=cXs1zQ27%2Fh9G1IycST38iD0cc0Vpx8gyepRsSvkVB%2Fk%3D&st=2022-07-01T09:08:40Z&se=2122-06-07T09:13:40Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ta-MY-SuryaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'a4753064-3db5-4f5d-aeba-d28e7442c26f',
    locale: 'af-ZA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (af-ZA, WillemNeural)',
    properties: {
      DisplayName: 'Willem',
      DisplayVoiceName: 'WillemNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Willem',
      LocaleDescription: 'af-ZA',
      OrderInVoiceListApi: '2',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'af-ZA-WillemNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'af-ZA',
      localeZH: '南非荷兰语(南非)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2e4b066d-a0a4-4525-b998-eefd09924419.wav?sv=2019-07-07&sr=b&sig=gWLxQWE00EwieMSL%2BL%2FcpaO51xrhvbHD9kT3P%2Fm72ao%3D&st=2021-10-29T08:29:55Z&se=2121-10-05T08:34:55Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'af-ZA-WillemNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '9bae679d-653a-4e89-82a0-d2c911f6bd1f',
    locale: 'de-DE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (de-DE, GiselaNeural)',
    properties: {
      DisplayName: 'Gisela',
      DisplayVoiceName: 'GiselaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Gisela',
      LocaleDescription: 'German (Germany)',
      OrderInVoiceListApi: '65',
      PreviewSentence:
        'Erstellen Sie Apps und Dienste, die auf natürliche Art mit Nutzern sprechen, und dadurch Zugänglichkeit und Benutzerfreundlichkeit verbessern.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'de-DE-GiselaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'de-DE',
      localeZH: '德语(德国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/06cd4bf5-4383-4b49-a59b-89fe7aa0a775.wav?sv=2019-07-07&sr=b&sig=q3YgOX7V8rqjJ6uR8SD%2BEhSphF1RTIkfsNTOsA4lyts%3D&st=2022-01-19T09:00:29Z&se=2121-12-26T09:05:29Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'de-DE-GiselaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'cfcdcaa4-9d3c-4c67-ad23-d355b41f0a2a',
    locale: 'ja-JP',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ja-JP, MayuNeural)',
    properties: {
      DisplayName: 'Mayu',
      DisplayVoiceName: 'MayuNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: '真夕',
      LocaleDescription: 'Japanese',
      OrderInVoiceListApi: '294',
      PreviewSentence:
        'アクセシビリティとユーザビリティを向上させながら、ユーザーに自然なことばを話すアプリとサービスを構築する',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ja-JP-MayuNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ja-JP',
      localeZH: '日语(日本)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/087a2899-a5b8-40d2-9459-3d3799d552a2.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A28Z&se=2122-09-27T07%3A52%3A28Z&sr=b&sp=rl&sig=dW1my40bjiujN4itCo18DcOuEopBT8COB8P%2BpObqOZY%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ja-JP-MayuNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'db571d48-6b03-4247-87ea-d3d670ca9f16',
    locale: 'es-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-ES, IreneNeural)',
    properties: {
      DisplayName: 'Irene',
      DisplayVoiceName: 'IreneNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Irene',
      LocaleDescription: 'Spanish (Spain)',
      OrderInVoiceListApi: '175',
      PreviewSentence:
        'Crea aplicaciones y servicios que hablan de forma natural a los usuarios, y que mejoran la accesibilidad y la facilidad de uso.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-ES-IreneNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-ES',
      localeZH: '西班牙语(西班牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d5bb919d-baaf-46c6-8da5-4f5fd4fd130c.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A43Z&se=2122-09-27T07%3A51%3A43Z&sr=b&sp=rl&sig=TAi4qwIRLIJ6BgsffK2NJycWfK1H2mYq98n3dnviilA%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-ES-IreneNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '194f8bf8-0162-424c-ab15-d3d6bb145520',
    locale: 'pt-BR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pt-BR, BrendaNeural)',
    properties: {
      DisplayName: 'Brenda',
      DisplayVoiceName: 'BrendaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Brenda',
      LocaleDescription: 'Portuguese (Brazil)',
      OrderInVoiceListApi: '352',
      PreviewSentence:
        'Desenvolver aplicativos e serviços que conversem naturalmente com os usuários, melhorando a acessibilidade e usabilidade.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pt-BR-BrendaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'pt-BR',
      localeZH: '葡萄牙语(巴西)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/cf1e6fc1-3f88-4809-907a-1b5346307fa5.wav?sv=2019-07-07&sr=b&sig=mHGhvvTqCtLCDg0hf7nVEpO8FclUhUgBFdr5ua6o4D8%3D&st=2022-07-01T09:10:11Z&se=2122-06-07T09:15:11Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'pt-BR-BrendaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '752fe147-0bdf-4426-9eb7-d3e50eef4e35',
    locale: 'ps-AF',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ps-AF, GulNawazNeural)',
    properties: {
      DisplayName: 'Gul Nawaz',
      DisplayVoiceName: 'GulNawazNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: ' ګل نواز',
      LocaleDescription: 'ps-AF',
      OrderInVoiceListApi: '348',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ps-AF-GulNawazNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ps-AF',
      localeZH: '普什图语(阿富汗)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/255c72a4-2e34-42f4-8402-f56d492f136e.wav?sv=2019-07-07&sr=b&sig=VFWtBpBed5EUfm7PqP2i6JFd%2FCHV7rqEMFj9dLYlgU0%3D&st=2022-01-19T09:02:11Z&se=2121-12-26T09:07:11Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ps-AF-GulNawazNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '7881206a-84f7-48a2-b3d3-d42023552efb',
    locale: 'zh-TW',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-TW, HsiaoYuNeural)',
    properties: {
      DisplayName: 'HsiaoYu',
      DisplayVoiceName: 'HsiaoYuNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: '曉雨',
      LocaleDescription: 'Chinese (Taiwan)',
      OrderInVoiceListApi: '453',
      PreviewSentence: '建構可以和使用者自然對話的應用程式和服務，來提高其方便性和實用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-TW-HsiaoYuNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'zh-TW',
      localeZH: '中文(台湾普通话)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/7984f96b-e52a-4a76-9bb2-df447ab6939f.wav?sv=2019-07-07&sr=b&sig=3tzoc68l8FTrXSOOKHVT9ffDpSbmvGScmYk0zscyDRk%3D&st=2021-10-29T08:29:52Z&se=2121-10-05T08:34:52Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'zh-TW-HsiaoYuNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '7dd10772-f742-4b98-8374-d473a8431e20',
    locale: 'fr-FR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-FR, YvesNeural)',
    properties: {
      DisplayName: 'Yves',
      DisplayVoiceName: 'YvesNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Yves',
      LocaleDescription: 'French (France)',
      OrderInVoiceListApi: '252',
      PreviewSentence:
        "Construire des applications et services qui communiquent naturellement avec l'utilisateur, en améliorant l’accessibilité et l'ergonomie",
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-FR-YvesNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fr-FR',
      localeZH: '法语(法国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6b415d75-1b34-4140-b1d1-2c2ca2433b93.wav?sv=2019-07-07&sr=b&sig=cGAkyivIzw6Evwz9skWR530mg1i8P8fk9njvaQN58io%3D&st=2022-01-19T09:01:39Z&se=2121-12-26T09:06:39Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fr-FR-YvesNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'c8e1b1c1-e5c4-4f04-93ab-d55b83949751',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, JacobNeural)',
    properties: {
      DisplayName: 'Jacob',
      DisplayVoiceName: 'JacobNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Jacob',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '139',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-JacobNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/118375a8-d2a1-450c-a252-cf02b3795716.wav?sv=2019-07-07&sr=b&sig=l%2Bzshq%2BBG6IdYg2NoUeJlqjxU%2BDreaRQkpIi0CkuKxw%3D&st=2021-10-29T08:25:39Z&se=2121-10-05T08:30:39Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-US-JacobNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '78af6233-41a2-420f-a3fe-d5e66296529f',
    locale: 'pt-BR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pt-BR, DonatoNeural)',
    properties: {
      DisplayName: 'Donato',
      DisplayVoiceName: 'DonatoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Donato',
      LocaleDescription: 'Portuguese (Brazil)',
      OrderInVoiceListApi: '353',
      PreviewSentence:
        'Desenvolver aplicativos e serviços que conversem naturalmente com os usuários, melhorando a acessibilidade e usabilidade.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pt-BR-DonatoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'pt-BR',
      localeZH: '葡萄牙语(巴西)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0967df5e-1c85-4400-8310-c27b7b4c9d67.wav?sv=2019-07-07&sr=b&sig=wQZ6ewPWEZSm0MFwoZDDQ5wJsD9nFEdiAR%2Fs7gXaVjg%3D&st=2022-07-01T09:10:13Z&se=2122-06-07T09:15:13Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'pt-BR-DonatoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'd6bb7968-50b7-4f22-bcf1-d709c49e70fe',
    locale: 'fil-PH',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fil-PH, BlessicaNeural)',
    properties: {
      DisplayName: 'Blessica',
      DisplayVoiceName: 'BlessicaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Blessica',
      LocaleDescription: 'fil-PH',
      OrderInVoiceListApi: '233',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fil-PH-BlessicaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fil-PH',
      localeZH: '菲律宾语(菲律宾)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6788a858-34a4-43cd-99a1-bec4dc694a55.wav?sv=2019-07-07&sr=b&sig=S%2Fl2gOhfX%2FtoJOlxt6iBcj1tzJDK3dpXmgRSDJK028g%3D&st=2021-10-29T08:31:04Z&se=2121-10-05T08:36:04Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fil-PH-BlessicaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '90f920d8-c159-41a0-8f3e-d7f8997ad666',
    locale: 'ar-BH',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-BH, AliNeural)',
    properties: {
      DisplayName: 'Ali',
      DisplayVoiceName: 'AliNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'علي',
      LocaleDescription: 'ar-BH',
      OrderInVoiceListApi: '7',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-BH-AliNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-BH',
      localeZH: '阿拉伯语(巴林)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/22aaff93-3036-49ca-b350-1dbb7dfe4c8e.wav?sv=2019-07-07&sr=b&sig=Ec%2Bp3mFvbuFz9vKWc1raTjvC%2Bp4Xaab8vXot5MsuGxc%3D&st=2021-10-29T08:29:59Z&se=2121-10-05T08:34:59Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-BH-AliNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'd4ed37d8-8d9f-47a2-8042-d82b9f6daf90',
    locale: 'nl-NL',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (nl-NL, FennaNeural)',
    properties: {
      DisplayName: 'Fenna',
      DisplayVoiceName: 'FennaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Fenna',
      LocaleDescription: 'Dutch',
      OrderInVoiceListApi: '343',
      PreviewSentence:
        'Ontwikkel apps en diensten die natuurlijk aanvoelen, waardoor de toegankelijkheid en bruikbaarheid vergroot worden.',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'nl-NL-FennaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'nl-NL',
      localeZH: '荷兰语(荷兰)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e986f0c2-3c44-4f78-8d21-13750be8f10a.wav?sv=2019-07-07&sr=b&sig=z44s1K2A3GNF7qzzhrgw%2FQPvFxf2MugeNM5NqIflU9g%3D&st=2021-10-29T08:27:46Z&se=2121-10-05T08:32:46Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'nl-NL-FennaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '9622ee9e-a68a-444b-ad30-d84db7340f07',
    locale: 'en-IN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-IN, NeerjaNeural)',
    properties: {
      DisplayName: 'Neerja',
      DisplayVoiceName: 'NeerjaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Neerja',
      LocaleDescription: 'English (India)',
      OrderInVoiceListApi: '112',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-IN-NeerjaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-IN',
      localeZH: '英语(印度)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f637752e-6307-4180-887f-2c2c02b66930.wav?sv=2019-07-07&sr=b&sig=QwYzQgCSTuoXAohY0hdl3N%2BIDi3BVK9enuicMKknf%2Fg%3D&st=2021-10-29T08:26:30Z&se=2121-10-05T08:31:30Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-IN-NeerjaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '22bcc50c-a99e-4acb-8355-d853c9804874',
    locale: 'es-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-US, PalomaNeural)',
    properties: {
      DisplayName: 'Paloma',
      DisplayVoiceName: 'PalomaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Paloma',
      LocaleDescription: 'es-US',
      OrderInVoiceListApi: '218',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-US-PalomaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-US',
      localeZH: '西班牙语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b082dac2-0a6a-4637-9bd5-36b05c437b89.wav?sv=2019-07-07&sr=b&sig=hXaQPdV1I%2BiK1FKJ1lzMZhBkJoIGeHB7F2FGeHTkPTg%3D&st=2021-10-29T08:26:48Z&se=2121-10-05T08:31:48Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-US-PalomaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '52c945ea-c060-4085-aef7-d9c6951f0a86',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, NancyNeural)',
    properties: {
      DisplayName: 'Nancy',
      DisplayVoiceName: 'NancyNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Nancy',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '144',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-NancyNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"angry":7,"cheerful":4,"sad":8,"Default":0}',
      VoiceStyleNames:
        'angry,cheerful,excited,friendly,hopeful,sad,shouting,terrified,unfriendly,whispering,Default',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6ea25992-a569-49f8-b0bc-c257573d1784.wav?sv=2019-07-07&sr=b&sig=53%2Ffce3CKkiV%2FEbrBZmE%2FWj80t58rvh8vrpsHjVpyc0%3D&st=2022-05-05T09:39:01Z&se=2122-04-11T09:44:01Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/dedacd0a-6a00-4012-9a1d-7f9fb74439bf.wav?sv=2019-07-07&sr=b&sig=otCZemJleRaT4MPQ3ytslzjODBH2W6UMiv5TV%2FiqdPY%3D&st=2022-04-28T10:28:09Z&se=2122-04-04T10:33:09Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/8ea8ca76-5244-4e95-91fc-e63a5adbf463.wav?sv=2019-07-07&sr=b&sig=Yex6gk368pgfkTREoLInBAFCSDX7noK9CU9bYOUy82w%3D&st=2022-04-28T10:28:12Z&se=2122-04-04T10:33:12Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/cd5030c1-2f2d-4661-9d3d-9693197a2939.wav?sv=2019-07-07&sr=b&sig=JQxCQC3o05bisJDeETqbweiLa9fkjGnPy4Qz9p753ZI%3D&st=2022-04-28T10:28:14Z&se=2122-04-04T10:33:14Z&sp=rl',
          styleName: 'excited',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/90a807c5-89bf-45c4-8a03-1fbe330f1137.wav?sv=2019-07-07&sr=b&sig=zelCuNshZ8%2B3n9dQ2igCJMhOtg8hSZGC1QpIWvEEAGk%3D&st=2022-04-28T10:28:16Z&se=2122-04-04T10:33:16Z&sp=rl',
          styleName: 'friendly',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fca0b4e3-8c06-4d75-8ebd-0816c2f5ede8.wav?sv=2019-07-07&sr=b&sig=UKn8X0TP1iwP7H6dQd7keNKqmXSmH3roXw%2BRhplbEn4%3D&st=2022-04-28T10:28:18Z&se=2122-04-04T10:33:18Z&sp=rl',
          styleName: 'hopeful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b0dfae29-094c-467c-8e9f-15c860ace2e3.wav?sv=2019-07-07&sr=b&sig=Htwm%2FKmlUW0C8i38A5B10RznQdE4uVNNwkjAMG0ltwo%3D&st=2022-04-28T10:28:21Z&se=2122-04-04T10:33:21Z&sp=rl',
          styleName: 'sad',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6861102a-2be4-40fe-8391-efe82e596209.wav?sv=2019-07-07&sr=b&sig=yATmcvFSgW%2FZqhhxL%2B1xErjRv4YdaIBIQhdAsT%2FHzRs%3D&st=2022-04-28T10:28:23Z&se=2122-04-04T10:33:23Z&sp=rl',
          styleName: 'shouting',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/9b94811a-8ab6-4687-ae9c-1b9442141b68.wav?sv=2019-07-07&sr=b&sig=Ve4iDx%2FGATYBJtBfJKNeUWINIGOfcbhvctO5JhNZXWQ%3D&st=2022-04-28T10:28:25Z&se=2122-04-04T10:33:25Z&sp=rl',
          styleName: 'terrified',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b04d1a7e-fe38-4856-be97-ac2265b6bb30.wav?sv=2019-07-07&sr=b&sig=g3NKMQlZ%2BXJqObqS9%2BoK9z9ePny6jPGFZqnQco9hW8w%3D&st=2022-04-28T10:28:27Z&se=2122-04-04T10:33:27Z&sp=rl',
          styleName: 'unfriendly',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/9a0633ec-638b-47ef-a73a-032e00632ed1.wav?sv=2019-07-07&sr=b&sig=bHLd%2BA8hCtwj1gSdQE12Ej48HzZTFFIDaGVMClJFXuM%3D&st=2022-04-28T10:28:30Z&se=2122-04-04T10:33:30Z&sp=rl',
          styleName: 'whispering',
        },
      ],
    },
    shortName: 'en-US-NancyNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '3f845c8a-1e92-4b44-8217-da0f9f634dc4',
    locale: 'zh-HK',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-HK, WanLungNeural)',
    properties: {
      DisplayName: 'WanLung',
      DisplayVoiceName: 'WanLungNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: '雲龍',
      LocaleDescription: 'Chinese (Hong Kong)',
      OrderInVoiceListApi: '451',
      PreviewSentence: '開發可自然地與用戶溝通的應用程式及服務，以提升其使用度及可用性。',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-HK-WanLungNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'zh-HK',
      localeZH: '中文(粤语，繁体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/584581e9-b86d-4178-acd1-f687830cd956.wav?sv=2019-07-07&sr=b&sig=Z3VKNznp%2F%2FKVLgqUCth2krgP%2FThmWy2pYCfy0XUo%2Bas%3D&st=2021-10-29T08:29:50Z&se=2121-10-05T08:34:50Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'zh-HK-WanLungNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'c03df8e3-1e9d-49a1-acb5-daa2aa3e2d2d',
    locale: 'az-AZ',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (az-AZ, BanuNeural)',
    properties: {
      DisplayName: 'Banu',
      DisplayVoiceName: 'BanuNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Banu',
      LocaleDescription: 'az-AZ',
      OrderInVoiceListApi: '38',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'az-AZ-BanuNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'az-AZ',
      localeZH: '阿塞拜疆语(阿塞拜疆) ',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/032e5978-5ef3-4b12-a093-f88cc908a6f4.wav?sv=2019-07-07&sr=b&sig=3QLr05iJR28%2F2ywWfN4tv9Yz%2FfNOvyhNTq2I%2BfD08NE%3D&st=2022-07-01T09:08:55Z&se=2122-06-07T09:13:55Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'az-AZ-BanuNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '5e59b45b-251e-4424-ab41-db4ac3c27895',
    locale: 'es-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-ES, AbrilNeural)',
    properties: {
      DisplayName: 'Abril',
      DisplayVoiceName: 'AbrilNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Abril',
      LocaleDescription: 'Spanish (Spain)',
      OrderInVoiceListApi: '169',
      PreviewSentence:
        'Crea aplicaciones y servicios que hablan de forma natural a los usuarios, y que mejoran la accesibilidad y la facilidad de uso.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-ES-AbrilNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-ES',
      localeZH: '西班牙语(西班牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/53e2cf9b-89c3-432b-afa6-1cae263c8418.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A32Z&se=2122-09-27T07%3A51%3A32Z&sr=b&sp=rl&sig=vGxyJ7vyoRyemH2Q0hNywcZCQ9eM3K4hqNthkH06tsI%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-ES-AbrilNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'c315f54f-d225-40fc-bf38-db9afc3449cc',
    locale: 'sw-KE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (sw-KE, ZuriNeural)',
    properties: {
      DisplayName: 'Zuri',
      DisplayVoiceName: 'ZuriNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Zuri',
      LocaleDescription: 'sw-KE',
      OrderInVoiceListApi: '391',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'sw-KE-ZuriNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'sw-KE',
      localeZH: '斯瓦希里语(肯尼亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/592c5fce-c759-4c41-bcfd-a7255d788204.wav?sv=2019-07-07&sr=b&sig=ha9mujiN6yjiA4M5CMncRk9xS6B5FnSEIU53rutOLSQ%3D&st=2021-10-29T08:28:14Z&se=2121-10-05T08:33:14Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'sw-KE-ZuriNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'e572c97c-d4aa-4c57-8be4-dc529d34ab29',
    locale: 'so-SO',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (so-SO, MuuseNeural)',
    properties: {
      DisplayName: 'Muuse',
      DisplayVoiceName: 'MuuseNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Muuse',
      LocaleDescription: 'so-SO',
      OrderInVoiceListApi: '379',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'so-SO-MuuseNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'so-SO',
      localeZH: '索马里语(索马里)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/042db3b2-7136-4cc5-9ae6-078673ced639.wav?sv=2019-07-07&sr=b&sig=ryJ%2Fiym8gJSXdinLpVvCDZIqnnG%2FdPYWbfhOMWkk9lw%3D&st=2021-10-29T08:31:13Z&se=2121-10-05T08:36:13Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'so-SO-MuuseNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '236139f8-3803-40db-8b24-dcb56104a973',
    locale: 'sl-SI',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (sl-SI, RokNeural)',
    properties: {
      DisplayName: 'Rok',
      DisplayVoiceName: 'RokNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Rok',
      LocaleDescription: 'Slovenian',
      OrderInVoiceListApi: '378',
      PreviewSentence:
        'Razvijajte aplikacije in storitve z vrhunsko uporabniško izkušnjo povsem po meri uporabnikov.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'sl-SI-RokNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'sl-SI',
      localeZH: '斯洛文尼亚语(斯洛文尼亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/03c642e4-e769-4161-ad57-ffd2ae3ea5d1.wav?sv=2019-07-07&sr=b&sig=VViuh%2B7ZxgAwXKSW7iRaelvwxSlx88QXnEW4CZUs0%2FI%3D&st=2021-10-29T08:28:09Z&se=2121-10-05T08:33:09Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'sl-SI-RokNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '963644be-9a6a-4a7c-8f30-ddab286f4ba0',
    locale: 'ar-AE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-AE, HamdanNeural)',
    properties: {
      DisplayName: 'Hamdan',
      DisplayVoiceName: 'HamdanNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'حمدان',
      LocaleDescription: 'ar-AE',
      OrderInVoiceListApi: '6',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-AE-HamdanNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-AE',
      localeZH: '阿拉伯语(阿拉伯联合酋长国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/f3f1470a-2c1e-47e4-8335-828c6870f04c.wav?sv=2019-07-07&sr=b&sig=RelBmhw6%2BakQxGbqDnxxaqA4pY%2FG2HNHOUrmOBv2i4k%3D&st=2021-10-29T08:29:58Z&se=2121-10-05T08:34:58Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-AE-HamdanNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [
      {
        name: 'All',
      },
      {
        name: 'Audiobook',
      },
      {
        name: 'Chat',
      },
    ],
    description:
      'Clear and relaxing voice with rich market-play and emotions that is optimized for audio books.',
    id: '0ca5d0ac-5a66-4d8e-9140-de556ef3916d',
    locale: 'zh-CN',
    masterpieces: [
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/bd0d12fb-8b4f-4db9-9efe-3ac1b9fc4b4f?sv=2019-07-07&sr=b&sig=fCgXha3rya0f7dYq9HXNdKGapiU0ZmXES3%2BEZWFzmu8%3D&st=2021-10-29T08:24:48Z&se=2121-10-05T08:29:48Z&sp=rl',
        categories: ['Audiobook'],
        title: '晓墨有声书《呼兰河传》',
        titleLocalizationId: 'zh-CN-XiaomoNeural_Masterpieces_AudioBook_HuLanHeZhuan_Title',
      },
      {
        audioFileEndpointWithSas:
          'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/1ea12800-a66d-45b8-9355-06b83b17a6ae?sv=2019-07-07&sr=b&sig=s8Lmx6jIlr3RF%2F5IdPN6lRr80FQzYnfthDqJ0F29Ors%3D&st=2021-10-29T08:24:49Z&se=2121-10-05T08:29:49Z&sp=rl',
        categories: ['Audiobook'],
        title: '晓墨有声书《马伯乐》',
        titleLocalizationId: 'zh-CN-XiaomoNeural_Masterpieces_AudioBook_MaBoLe_Title',
      },
    ],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN, XiaomoNeural)',
    properties: {
      AgeGroups: 'YoungAdult',
      DisplayName: 'Xiaomo',
      DisplayVoiceName: 'XiaomoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      IconFileEndpointWithSas:
        'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/64693a99-6981-46a7-ab94-67cf185b2bbf?sv=2019-07-07&sr=b&sig=%2FQtxKGUYE6ZVXK0vnXkj6VKLzf0ZU9EqTxLGTrKHum4%3D&st=2021-10-29T08:24:21Z&se=2121-10-05T08:29:21Z&sp=rl',
      IsTtsBreakEnabled: 'True',
      LocalName: '晓墨',
      LocaleDescription: 'Chinese (Mainland)',
      OrderInVoiceListApi: '428',
      PreviewSentence: '通过提供能和用户自然交流的应用程序和服务，以改善其可访问性和可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-XiaomoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions:
        '{"YoungAdultFemale":4,"YoungAdultMale":1,"OlderAdultFemale":5,"OlderAdultMale":2,"SeniorFemale":10,"SeniorMale":3,"Girl":8,"Boy":9,"Default":0}',
      VoiceRoleNames:
        'YoungAdultFemale,YoungAdultMale,OlderAdultFemale,OlderAdultMale,SeniorFemale,SeniorMale,Girl,Boy,Default',
      VoiceStyleNameDefinitions:
        '{"embarrassed":1,"calm":2,"fearful":3,"cheerful":4,"disgruntled":5,"serious":6,"angry":7,"sad":8,"depressed":10,"affectionate":9,"gentle":11,"Default":0}',
      VoiceStyleNames:
        'embarrassed,calm,fearful,cheerful,disgruntled,serious,angry,sad,depressed,affectionate,gentle,envious,Default',
      locale: 'zh-CN',
      localeZH: '中文(普通话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/52ff557c-7750-4292-84c0-58a36b2a2a38.wav?sv=2019-07-07&sr=b&sig=Kr6yQEWbLS4KFA58fjoMkWO6yJJzsxyhkbi2szb41yw%3D&st=2021-10-29T08:31:29Z&se=2121-10-05T08:36:29Z&sp=rl',
          roleName: 'Default',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/3d36d976-1c39-43f7-9fa7-c635e68e5310.wav?sv=2019-07-07&sr=b&sig=clLI3Oncqry60L%2FsSg2vDh3D%2Fvqe5OEJ7vFl19%2B%2FbT8%3D&st=2021-10-29T08:31:30Z&se=2121-10-05T08:36:30Z&sp=rl',
          roleName: 'Boy',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5a310bfd-e7b9-42c8-9fee-6992b4e80885.wav?sv=2019-07-07&sr=b&sig=Iy%2BOWdkKwvP5wW6CD0kV6MlLFBw43Di%2Bb22L7IzFG0Q%3D&st=2021-10-29T08:31:31Z&se=2121-10-05T08:36:31Z&sp=rl',
          roleName: 'Girl',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/8066acfc-820c-4e87-b148-35c9e2effec7.wav?sv=2019-07-07&sr=b&sig=TlqpzWhGLJWNblXk8m3J7L3mn4nFrCim5r6cah2DwLw%3D&st=2021-10-29T08:31:32Z&se=2121-10-05T08:36:32Z&sp=rl',
          roleName: 'OlderAdultFemale',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0f60b76f-8434-4d2d-a12b-fefb9a38312f.wav?sv=2019-07-07&sr=b&sig=WRbGxsg3Bf4BnD8LlXgHl%2Fn%2FqebBE0TrWUkyry70P1A%3D&st=2021-10-29T08:31:33Z&se=2121-10-05T08:36:33Z&sp=rl',
          roleName: 'OlderAdultMale',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/dabb2598-a37d-4e42-b87e-6f028487507a.wav?sv=2019-07-07&sr=b&sig=oUUNf4RlcSoZ%2FyvsXJq7VelyTAE6X97kDeYq0egHcwg%3D&st=2021-10-29T08:31:34Z&se=2121-10-05T08:36:34Z&sp=rl',
          roleName: 'SeniorFemale',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/9329262a-59f0-4295-81fb-9901d97ef808.wav?sv=2019-07-07&sr=b&sig=FQnLCRv86qpgCYFpa5pSCTixFZlIYOm3qHC6E20OdeY%3D&st=2021-10-29T08:31:35Z&se=2121-10-05T08:36:35Z&sp=rl',
          roleName: 'SeniorMale',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d05b45e5-a358-45aa-b762-247c009b7129.wav?sv=2019-07-07&sr=b&sig=MTVpCWsfEuaTBAxgLoCsdUN0G2a8Yh8qiq%2B7Widt4b4%3D&st=2021-10-29T08:31:36Z&se=2121-10-05T08:36:36Z&sp=rl',
          roleName: 'YoungAdultFemale',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/83642188-6398-45cf-b0fb-e22fb1e74b83.wav?sv=2019-07-07&sr=b&sig=QcT%2FRVdsW78z62u3SQsGObQh141fb8DgDL7oa9TAAyI%3D&st=2021-10-29T08:31:37Z&se=2121-10-05T08:36:37Z&sp=rl',
          roleName: 'YoungAdultMale',
        },
      ],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0e0e9e82-5ef4-4eea-be3e-14face618ee7.wav?sv=2019-07-07&sr=b&sig=1YlCmZS%2BGV0pcLF60r9p9%2Bs8KvwunjoVEI4CmOXuLkI%3D&st=2021-10-29T08:28:42Z&se=2121-10-05T08:33:42Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/eb330945-55f6-4279-b5fd-0072c919ddee.wav?sv=2019-07-07&sr=b&sig=vXmVwS9PT5NlODnpJcoerGTFtIboA4QDLROyjphjfNs%3D&st=2021-10-29T08:28:43Z&se=2121-10-05T08:33:43Z&sp=rl',
          styleName: 'calm',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4e0f1ecf-7032-41a2-9fff-4f6855b68052.wav?sv=2019-07-07&sr=b&sig=3B5HlILJKwjx%2FjGbiYNC%2FVu2fQEgNYWtsosy2Ax3CwM%3D&st=2021-10-29T08:28:44Z&se=2121-10-05T08:33:44Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6ff44ced-301d-4fbd-a297-8b8acb270667.wav?sv=2019-07-07&sr=b&sig=wMV1r4o0%2Bm4U2dWdfLFt%2B5gocU2MGTJIFAnz9ng%2BFso%3D&st=2021-10-29T08:28:45Z&se=2121-10-05T08:33:45Z&sp=rl',
          styleName: 'depressed',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d64d01be-b919-4be4-9bbc-9f03c56568fa.wav?sv=2019-07-07&sr=b&sig=m56ItegVutLLGZtghNXACvEDdx98BKs2EhTy98HcsIQ%3D&st=2021-10-29T08:28:46Z&se=2121-10-05T08:33:46Z&sp=rl',
          styleName: 'disgruntled',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/be664af1-0101-4cda-baf4-46ce20b56342.wav?sv=2019-07-07&sr=b&sig=TmZJ31YBRJVt6UNJaHDrNVurZWIgKpvG1yschAQFNDk%3D&st=2021-10-29T08:28:47Z&se=2121-10-05T08:33:47Z&sp=rl',
          styleName: 'fearful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b9b403fd-5bae-4962-b21d-d3b2b9c70dc0.wav?sv=2019-07-07&sr=b&sig=JUcNA956d9OLlMwLAjKeMdVbMLQUyMqFg1zdSaldST0%3D&st=2021-10-29T08:28:48Z&se=2121-10-05T08:33:48Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a397419e-1c5f-435b-a3d2-722d6b4d7d8b.wav?sv=2019-07-07&sr=b&sig=p2sDXyHz4BKaeCmwMTXK2Qusfd0gs6utW7wAMj9CG6g%3D&st=2021-10-29T08:28:49Z&se=2121-10-05T08:33:49Z&sp=rl',
          styleName: 'gentle',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/11d33875-6a91-46db-bee1-8c93eaee7cb9.wav?sv=2019-07-07&sr=b&sig=qzOr4FJRSO9RiNButs0C57qGRoM6Jp5xnnJ%2FwDIr2p0%3D&st=2021-10-29T08:28:50Z&se=2121-10-05T08:33:50Z&sp=rl',
          styleName: 'serious',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fe0424aa-b006-4168-8b88-5b3b0e4d536c.wav?sv=2019-07-07&sr=b&sig=j1XBmNquWZZ14lp0w10xQttRtZsgGy9v87MiiuBW7EE%3D&st=2021-12-20T11:26:47Z&se=2121-11-26T11:31:47Z&sp=rl',
          styleName: 'affectionate',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/17eeb808-fa65-41a9-bd60-9d56b9d9d7e1.wav?sv=2019-07-07&sr=b&sig=J01kMUjjUXSW1PpR5HnZxjfI6Zztcy6cmyHqtLOCA%2BU%3D&st=2021-12-20T11:26:50Z&se=2121-11-26T11:31:50Z&sp=rl',
          styleName: 'embarrassed',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d56947cd-8eb1-423a-bff9-ab9405562305.wav?sv=2019-07-07&sr=b&sig=6CCDTVwbvFS2wT9ZPGr3nRmMWgI5DsQXyJaQW%2BlYqfU%3D&st=2021-12-20T11:26:52Z&se=2121-11-26T11:31:52Z&sp=rl',
          styleName: 'Envy',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d56947cd-8eb1-423a-bff9-ab9405562305.wav?sv=2019-07-07&sr=b&sig=6CCDTVwbvFS2wT9ZPGr3nRmMWgI5DsQXyJaQW%2BlYqfU%3D&st=2021-12-20T11:26:52Z&se=2121-11-26T11:31:52Z&sp=rl',
          styleName: 'envious',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/460b5c2b-1123-4abf-80ff-3b18ef980b28.wav?sv=2019-07-07&sr=b&sig=i2R60koN4wuUeoPNmwS7TFuB8qT%2F%2Bupgwlk75VB%2FsXM%3D&st=2021-12-20T11:26:55Z&se=2121-11-26T11:31:55Z&sp=rl',
          styleName: 'sad',
        },
      ],
    },
    shortName: 'zh-CN-XiaomoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '6233d99d-eb7b-4890-bd5a-e0139af6cb4b',
    locale: 'pt-BR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pt-BR, HumbertoNeural)',
    properties: {
      DisplayName: 'Humberto',
      DisplayVoiceName: 'HumbertoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Humberto',
      LocaleDescription: 'Portuguese (Brazil)',
      OrderInVoiceListApi: '357',
      PreviewSentence:
        'Desenvolver aplicativos e serviços que conversem naturalmente com os usuários, melhorando a acessibilidade e usabilidade.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pt-BR-HumbertoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'pt-BR',
      localeZH: '葡萄牙语(巴西)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/7b7b37d0-7235-4f09-8232-5e061f19c9b3.wav?sv=2019-07-07&sr=b&sig=JgJiiIBuZsDgjZX8kpIW6pqcaXPLKSBihGMXHiZaEZ0%3D&st=2022-07-01T09:10:22Z&se=2122-06-07T09:15:22Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'pt-BR-HumbertoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '21d7ae8e-30af-4fa6-a265-e03e6a76e2be',
    locale: 'en-GB',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-GB, EthanNeural)',
    properties: {
      DisplayName: 'Ethan',
      DisplayVoiceName: 'EthanNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Ethan',
      LocaleDescription: 'English (UK)',
      OrderInVoiceListApi: '98',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-GB-EthanNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-GB',
      localeZH: '英语(英国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0d0ecf50-b8fb-4eaa-afde-da89b09e43cc.wav?sv=2019-07-07&sr=b&sig=11kFtaEpWViHcnQLRTkiPgSyu%2BnYEDhnRZqt7LkE9m0%3D&st=2022-01-19T09:01:00Z&se=2121-12-26T09:06:00Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-GB-EthanNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '8cc98d23-ea9d-4bde-85d3-e0dcdb17fddf',
    locale: 'en-NZ',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-NZ, MitchellNeural)',
    properties: {
      DisplayName: 'Mitchell',
      DisplayVoiceName: 'MitchellNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Mitchell',
      LocaleDescription: 'en-NZ',
      OrderInVoiceListApi: '118',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-NZ-MitchellNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-NZ',
      localeZH: '英语(新西兰)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e0f38332-30e3-4812-9dfe-ab312ccee838.wav?sv=2019-07-07&sr=b&sig=V8HcYMuLZovzMJxDhTXmvwSRbuk%2F1hBCVwKOIIina7U%3D&st=2021-10-29T08:26:32Z&se=2121-10-05T08:31:32Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-NZ-MitchellNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '608b582e-0934-4d7c-b33d-e172e724cae5',
    locale: 'zu-ZA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zu-ZA, ThembaNeural)',
    properties: {
      DisplayName: 'Themba',
      DisplayVoiceName: 'ThembaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Themba',
      LocaleDescription: 'zu-ZA',
      OrderInVoiceListApi: '456',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zu-ZA-ThembaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'zu-ZA',
      localeZH: '祖鲁语(南非)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5ff78846-a6b5-4fe6-b14b-4be6c0b872be.wav?sv=2019-07-07&sr=b&sig=9MDYSkez2nnzv%2BLiZYgsUEGhZ8QEqiK0GGfYlsBVfFs%3D&st=2021-10-29T08:31:28Z&se=2121-10-05T08:36:28Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'zu-ZA-ThembaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'dc81141a-304d-48ed-b9cb-e203f94b3ad8',
    locale: 'es-PA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-PA, RobertoNeural)',
    properties: {
      DisplayName: 'Roberto',
      DisplayVoiceName: 'RobertoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Roberto',
      LocaleDescription: 'es-PA',
      OrderInVoiceListApi: '208',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-PA-RobertoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-PA',
      localeZH: '西班牙语(巴拿马)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c3861064-2726-4784-ac7f-ca6228d4ef32.wav?sv=2019-07-07&sr=b&sig=Vzdh8IIC2p82vB2A27zyVJSJZwK8Xz%2Bq7dR6RX4wc70%3D&st=2021-10-29T08:30:49Z&se=2121-10-05T08:35:49Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-PA-RobertoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'fe3060ea-a2a3-4c44-969c-e2b59cf5d342',
    locale: 'ms-MY',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ms-MY, OsmanNeural)',
    properties: {
      DisplayName: 'Osman',
      DisplayVoiceName: 'OsmanNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Osman',
      LocaleDescription: 'Malay',
      OrderInVoiceListApi: '329',
      PreviewSentence:
        'Membina aplikasi dan perkhidmatan yang bercakap sememangnya kepada pengguna, meningkatkan kebolehcapaian dan kebolehgunaanya.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ms-MY-OsmanNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ms-MY',
      localeZH: '马来语(马来西亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0f7655f2-0289-4eb5-9535-2d6f4002d47c.wav?sv=2019-07-07&sr=b&sig=SfWf%2F2azdsvq08NkcXikBoFFgL6LTISwi66Iw3oz58w%3D&st=2021-10-29T08:27:32Z&se=2121-10-05T08:32:32Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ms-MY-OsmanNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'e0638b39-fbd2-4497-a482-e2f65759412a',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, GuyNeural)',
    properties: {
      DisplayName: 'Guy',
      DisplayVoiceName: 'GuyNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Guy',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '128',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-GuyNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"angry":7,"cheerful":4,"sad":8,"Default":0}',
      VoiceStyleNames:
        'newscast,angry,cheerful,sad,excited,friendly,terrified,shouting,unfriendly,whispering,hopeful,Default',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/80a0b1f2-b2b8-4dec-83f1-a15ca5a45f52.wav?sv=2019-07-07&sr=b&sig=kfS%2FUdQg3Wu37aWskWvBkVGdTbhrZZGqcQQ6xd1VQ7g%3D&st=2021-10-29T08:25:37Z&se=2121-10-05T08:30:37Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c60ea716-7984-4d8c-b886-7abdbe8df66b.wav?sv=2019-07-07&sr=b&sig=oM9jysbJy2sv9Nc0P8HZyurgYeKtZ8%2BK%2FSy%2BKznwFXU%3D&st=2021-10-29T08:25:38Z&se=2121-10-05T08:30:38Z&sp=rl',
          styleName: 'newscast',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/de130c31-1e18-4fe4-9ead-7105dc5be49a.wav?sv=2019-07-07&sr=b&sig=hFqMrZ5aJyR2IcpnYquKbYKvFgNZxfILVRl3xlVS%2ByU%3D&st=2022-04-28T10:25:50Z&se=2122-04-04T10:30:50Z&sp=rl',
          styleName: 'angry',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5346520a-2ae3-4ed0-94ff-419e99666a18.wav?sv=2019-07-07&sr=b&sig=vsjWhjD9dmB%2FUCnBfP76JxVBu8H8Z1YfJsLb%2FzbIosY%3D&st=2022-04-28T10:25:52Z&se=2122-04-04T10:30:52Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2fc9bff9-2350-45e6-aa38-9041be63453d.wav?sv=2019-07-07&sr=b&sig=zzWt3NGlGGY5U5iiS9cePR2zzV9e%2BFSKD7CWRcvfq00%3D&st=2022-04-28T10:25:55Z&se=2122-04-04T10:30:55Z&sp=rl',
          styleName: 'excited',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fcd059e0-683a-4f29-bb32-d04cbce04431.wav?sv=2019-07-07&sr=b&sig=fGIDrzJ7c1fSxpwR8wRTYE9Pl8h9smQLFpMoSsA5AwI%3D&st=2022-04-28T10:25:57Z&se=2122-04-04T10:30:57Z&sp=rl',
          styleName: 'friendly',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c76d2e20-8672-480e-b1c1-d06d932bbfeb.wav?sv=2019-07-07&sr=b&sig=bb048dTml8wsf%2Bicd9bSPT2VvMOJdJ0cKI5DqUdVrzs%3D&st=2022-04-28T10:25:59Z&se=2122-04-04T10:30:59Z&sp=rl',
          styleName: 'hopeful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/52f63f14-582b-4581-b0f4-74a588c8b68a.wav?sv=2019-07-07&sr=b&sig=M8exKg%2F34ecrEThwelnZ%2Bz%2Bw0AoQrIecEhkCeMeoDkk%3D&st=2022-04-28T10:26:01Z&se=2122-04-04T10:31:01Z&sp=rl',
          styleName: 'sad',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c1313b6e-192b-4b53-ac33-c7fa315da639.wav?sv=2019-07-07&sr=b&sig=w8veL4pc9nLOv1smnFkDhkT9FCPI2y6hbAluZZOeQlc%3D&st=2022-04-28T10:26:04Z&se=2122-04-04T10:31:04Z&sp=rl',
          styleName: 'shouting',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/57f89d08-2776-4513-a1b3-82d203a893df.wav?sv=2019-07-07&sr=b&sig=L3n3f0yOZw6t6lRcsqCRsDJZgNpmANqWcR90rmQEyqY%3D&st=2022-04-28T10:26:06Z&se=2122-04-04T10:31:06Z&sp=rl',
          styleName: 'terrified',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d8270914-7460-4d4e-a217-6bba35906928.wav?sv=2019-07-07&sr=b&sig=aGuFKf3T5lx3lu4lc9NV3bVh3JYzJ5i11p21xGVn%2Fsc%3D&st=2022-04-28T10:26:08Z&se=2122-04-04T10:31:08Z&sp=rl',
          styleName: 'unfriendly',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/70bfe78d-5c85-46b1-b5bb-1f8992c7314e.wav?sv=2019-07-07&sr=b&sig=x0flIet%2Bqn6IM8Qce%2BqVXW2CcbyQqzeXyf8l%2FSrGZbg%3D&st=2022-04-28T10:26:10Z&se=2122-04-04T10:31:10Z&sp=rl',
          styleName: 'whispering',
        },
      ],
    },
    shortName: 'en-US-GuyNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '59a12d88-9d94-4e6e-99e6-e49c1f433870',
    locale: 'ar-QA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-QA, AmalNeural)',
    properties: {
      DisplayName: 'Amal',
      DisplayVoiceName: 'AmalNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'أمل',
      LocaleDescription: 'ar-QA',
      OrderInVoiceListApi: '27',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-QA-AmalNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-QA',
      localeZH: '阿拉伯语(卡塔尔)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/6688d736-513a-48dc-83da-65976d8b8bf6.wav?sv=2019-07-07&sr=b&sig=rlKV0d3uVQzsnSCYxCHdjRNPDpOwt6oSTXLp3DG93kk%3D&st=2021-10-29T08:30:13Z&se=2121-10-05T08:35:13Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-QA-AmalNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '7dd4e360-55ef-4e1a-ac97-e53fe874660f',
    locale: 'es-PE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-PE, CamilaNeural)',
    properties: {
      DisplayName: 'Camila',
      DisplayVoiceName: 'CamilaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Camila',
      LocaleDescription: 'es-PE',
      OrderInVoiceListApi: '210',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-PE-CamilaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-PE',
      localeZH: '西班牙语(秘鲁)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4cbe6a7e-45ac-4a06-b245-8c5bc90e99b1.wav?sv=2019-07-07&sr=b&sig=SAzKcjSiUapBxZPDHXpnhwtkdj4S%2FBdmNgsgQpz7x1E%3D&st=2021-10-29T08:30:51Z&se=2121-10-05T08:35:51Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-PE-CamilaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'ec92d13c-b25b-4aa8-9e40-e65015775398',
    locale: 'zh-HK',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-HK, HiuMaanNeural)',
    properties: {
      DisplayName: 'HiuMaan',
      DisplayVoiceName: 'HiuMaanNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: '曉曼',
      LocaleDescription: 'Chinese (Hong Kong)',
      OrderInVoiceListApi: '449',
      PreviewSentence: '開發可自然地與用戶溝通的應用程式及服務，以提升其使用度及可用性。',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-HK-HiuMaanNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'zh-HK',
      localeZH: '中文(粤语，繁体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/afec9184-36f5-423f-b497-9d792847ebf7.wav?sv=2019-07-07&sr=b&sig=kUvcZ46upW7C2vG7V%2F7LUxIB%2BRaW%2FH2hC7IQjhcaXoY%3D&st=2021-10-29T08:29:49Z&se=2121-10-05T08:34:49Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'zh-HK-HiuMaanNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '6eac313c-0139-4f09-a44a-e7d8731c1f1e',
    locale: 'pt-BR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (pt-BR, LeticiaNeural)',
    properties: {
      DisplayName: 'Leticia',
      DisplayVoiceName: 'LeticiaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Leticia',
      LocaleDescription: 'Portuguese (Brazil)',
      OrderInVoiceListApi: '360',
      PreviewSentence:
        'Desenvolver aplicativos e serviços que conversem naturalmente com os usuários, melhorando a acessibilidade e usabilidade.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'pt-BR-LeticiaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'pt-BR',
      localeZH: '葡萄牙语(巴西)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ac4291a2-af7e-4f4e-b523-15476d993986.wav?sv=2019-07-07&sr=b&sig=VDy9qbawTe9VsiBWMB8dNphF3Zc0YaZcFF1ub0CuyWs%3D&st=2022-07-01T09:10:29Z&se=2122-06-07T09:15:29Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'pt-BR-LeticiaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'e11817e0-75e6-48ad-b375-e8ede271ee2d',
    locale: 'es-BO',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-BO, MarceloNeural)',
    properties: {
      DisplayName: 'Marcelo',
      DisplayVoiceName: 'MarceloNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Marcelo',
      LocaleDescription: 'es-BO',
      OrderInVoiceListApi: '155',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-BO-MarceloNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-BO',
      localeZH: '西班牙语(玻利维亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/dd7cfedb-d4b3-4d11-9674-db3be1e813d4.wav?sv=2019-07-07&sr=b&sig=tEngoL8QT4e7AOMZvqFqg5rAhOk9gawnb1ChKKkFPsQ%3D&st=2021-10-29T08:30:29Z&se=2121-10-05T08:35:29Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-BO-MarceloNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'f4ae5b2b-c6f5-4574-a516-ea413ddd638b',
    locale: 'es-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-ES, TeoNeural)',
    properties: {
      DisplayName: 'Teo',
      DisplayVoiceName: 'TeoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Teo',
      LocaleDescription: 'Spanish (Spain)',
      OrderInVoiceListApi: '180',
      PreviewSentence:
        'Crea aplicaciones y servicios que hablan de forma natural a los usuarios, y que mejoran la accesibilidad y la facilidad de uso.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-ES-TeoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-ES',
      localeZH: '西班牙语(西班牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2dad996f-9286-44e4-9aeb-f7a5062b7378.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A54Z&se=2122-09-27T07%3A51%3A54Z&sr=b&sp=rl&sig=VNgW4VMRocJquxep4yRZHYgkKivvKxxOgdcgEJgvyFo%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-ES-TeoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'd6db2ea7-2048-4c35-8733-ea7de8e8278d',
    locale: 'yue-CN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (yue-CN, XiaoMinNeural)',
    properties: {
      DisplayName: 'XiaoMin',
      DisplayVoiceName: 'XiaoMinNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: '晓敏',
      LocaleDescription: 'yue-CN',
      OrderInVoiceListApi: '421',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'yue-CN-XiaoMinNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'yue-CN',
      localeZH: '中文(粤语，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/97858b63-9f84-4f13-982b-b3ed4e6fba60.wav?sv=2021-08-06&st=2022-10-21T11%3A57%3A32Z&se=2122-09-27T12%3A02%3A32Z&sr=b&sp=rl&sig=laAn%2FiQecyy0W0AvT9I3Zn2pBBaRIXBNOZtYPwoaG9k%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'yue-CN-XiaoMinNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '77225261-11e8-4f00-92c5-ebd3cea15fa9',
    locale: 'es-BO',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-BO, SofiaNeural)',
    properties: {
      DisplayName: 'Sofia',
      DisplayVoiceName: 'SofiaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Sofia',
      LocaleDescription: 'es-BO',
      OrderInVoiceListApi: '156',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-BO-SofiaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-BO',
      localeZH: '西班牙语(玻利维亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/2e6bd1b8-bda6-40b9-bcec-c517f48f021e.wav?sv=2019-07-07&sr=b&sig=8wEHOJds38KqYSGfg%2BCY%2B5yAHlTm957Jjk0ZbpBFUpc%3D&st=2021-10-29T08:30:29Z&se=2121-10-05T08:35:29Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-BO-SofiaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '3e7c9283-5b48-4bf8-acec-ec7e4a51828c',
    locale: 'tr-TR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (tr-TR, AhmetNeural)',
    properties: {
      DisplayName: 'Ahmet',
      DisplayVoiceName: 'AhmetNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Ahmet',
      LocaleDescription: 'Turkish',
      OrderInVoiceListApi: '407',
      PreviewSentence:
        'Kullanıcılarla doğal biçimde konuşan, erişilebilirlik ve kullanılabilirliği iyileştiren uygulama ve servisler geliştirmek.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'tr-TR-AhmetNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'tr-TR',
      localeZH: '土耳其语(Türkiye)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/d683104c-db4a-4c59-b5dd-1c4081ee9b6b.wav?sv=2019-07-07&sr=b&sig=3QenHt6zJHOgbYW6NqO19%2FE4Yy9XhWbr%2FowJhUfDlFc%3D&st=2021-10-29T08:28:22Z&se=2121-10-05T08:33:22Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'tr-TR-AhmetNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'f8e3c0b1-fe18-4009-ae86-ecc464d3d4a5',
    locale: 'jv-ID',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (jv-ID, DimasNeural)',
    properties: {
      DisplayName: 'Dimas',
      DisplayVoiceName: 'DimasNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Dimas',
      LocaleDescription: 'jv-ID',
      OrderInVoiceListApi: '297',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'jv-ID-DimasNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'jv-ID',
      localeZH: '爪哇语(印度尼西亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/65b124ed-4aa7-47f1-8f36-7f87097faa18.wav?sv=2019-07-07&sr=b&sig=D%2BnO78cuSxQMQ20uTLgnCyHszWVqIkMYEk1WKxoZNzQ%3D&st=2021-10-29T08:31:07Z&se=2121-10-05T08:36:07Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'jv-ID-DimasNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '253a7da2-fc9a-4e1b-8d26-ed2919ecb899',
    locale: 'en-GB',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-GB, OliviaNeural)',
    properties: {
      DisplayName: 'Olivia',
      DisplayVoiceName: 'OliviaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Olivia',
      LocaleDescription: 'English (UK)',
      OrderInVoiceListApi: '103',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-GB-OliviaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-GB',
      localeZH: '英语(英国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/7c7bba31-c3db-478f-87c5-c03d50012ed3.wav?sv=2019-07-07&sr=b&sig=ZTZXzA05Z1Xap3O%2FbZNztoetWMYf9QdqrOZKfoeqlgA%3D&st=2022-01-19T09:01:12Z&se=2121-12-26T09:06:12Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-GB-OliviaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '3641b93d-312b-4227-ae32-ed681431ba38',
    locale: 'uz-UZ',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (uz-UZ, MadinaNeural)',
    properties: {
      DisplayName: 'Madina',
      DisplayVoiceName: 'MadinaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Madina',
      LocaleDescription: 'uz-UZ',
      OrderInVoiceListApi: '415',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'uz-UZ-MadinaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'uz-UZ',
      localeZH: '乌兹别克语(乌兹别克斯坦)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c4868900-70ce-4b9f-b8a4-03677b280169.wav?sv=2019-07-07&sr=b&sig=CEQVJbCSMusSkeSgO3ea2gZLvZwKUdrAWoMVx3Ktxok%3D&st=2021-10-29T08:31:26Z&se=2121-10-05T08:36:26Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'uz-UZ-MadinaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'b1377f34-f7dd-4744-88ec-ef2fbc9ed336',
    locale: 'es-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-ES, LiaNeural)',
    properties: {
      DisplayName: 'Lia',
      DisplayVoiceName: 'LiaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Lia',
      LocaleDescription: 'Spanish (Spain)',
      OrderInVoiceListApi: '177',
      PreviewSentence:
        'Crea aplicaciones y servicios que hablan de forma natural a los usuarios, y que mejoran la accesibilidad y la facilidad de uso.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-ES-LiaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-ES',
      localeZH: '西班牙语(西班牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c81ffd57-6eaa-420c-8a5c-a076bab03ab4.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A47Z&se=2122-09-27T07%3A51%3A47Z&sr=b&sp=rl&sig=%2Bv0lGKiO3rbUoMFEJ5Y13ruBwQgiodDZVoHcxqGSQOg%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-ES-LiaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '7c31a42d-a9f5-4780-81f2-f05183453f7c',
    locale: 'hu-HU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (hu-HU, TamasNeural)',
    properties: {
      DisplayName: 'Tamas',
      DisplayVoiceName: 'TamasNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Tamás',
      LocaleDescription: 'Hungarian',
      OrderInVoiceListApi: '268',
      PreviewSentence:
        'Készítsen appokat és szolgáltatásokat, melyek természetes hangon beszélnek a felhasználóval, ezáltal hozzáférhetőbbek és könnyebben használhatóak.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'hu-HU-TamasNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'hu-HU',
      localeZH: '匈牙利语(匈牙利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c8787051-37a5-441b-af29-0fd7f5b4ed90.wav?sv=2019-07-07&sr=b&sig=sbq3pGNZjyfM9O6UHOwjAfz4BHyr8QdpeVuchsM2ZE4%3D&st=2021-10-29T08:27:14Z&se=2121-10-05T08:32:14Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'hu-HU-TamasNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '6eae597f-b880-47c8-bbe6-f05e69fbdf9d',
    locale: 'id-ID',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (id-ID, ArdiNeural)',
    properties: {
      DisplayName: 'Ardi',
      DisplayVoiceName: 'ArdiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Ardi',
      LocaleDescription: 'Indonesian',
      OrderInVoiceListApi: '271',
      PreviewSentence:
        'Buat aplikasi dan layanan yang berbicara secara alami kepada pengguna, sehingga meningkatkan aksesibilitas dan kegunaan.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'id-ID-ArdiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'id-ID',
      localeZH: '印度尼西亚语(印度尼西亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/126d8203-456e-4cae-a9b9-9a670e5aee6c.wav?sv=2019-07-07&sr=b&sig=WyMavFM1VsrGxIWflwaVGViDVebuw1UFjf3woMhxMig%3D&st=2021-10-29T08:27:15Z&se=2121-10-05T08:32:15Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'id-ID-ArdiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '990dd0ec-a391-4336-8f00-f0f1b46c5d20',
    locale: 'kk-KZ',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (kk-KZ, DauletNeural)',
    properties: {
      DisplayName: 'Daulet',
      DisplayVoiceName: 'DauletNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Дәулет',
      LocaleDescription: 'kk-KZ',
      OrderInVoiceListApi: '302',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'kk-KZ-DauletNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'kk-KZ',
      localeZH: '哈萨克语(哈萨克斯坦)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/5493cad8-f74b-4562-a561-326d7a3b9d03.wav?sv=2019-07-07&sr=b&sig=J6QJrYKDE3bn4BkT6FRfPiD5X4%2FRCswYkCZxm7crBII%3D&st=2022-01-19T09:01:51Z&se=2121-12-26T09:06:51Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'kk-KZ-DauletNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '02973e72-898f-454c-9e9e-f11e46ec8400',
    locale: 'es-PR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-PR, KarinaNeural)',
    properties: {
      DisplayName: 'Karina',
      DisplayVoiceName: 'KarinaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Karina',
      LocaleDescription: 'es-PR',
      OrderInVoiceListApi: '211',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-PR-KarinaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-PR',
      localeZH: '西班牙语(波多黎各)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e7bf6114-7b2b-495a-ad57-1d9040061431.wav?sv=2019-07-07&sr=b&sig=9thqRPpC24EAIXis4DChU7tmH2JzYGV%2Fy%2FiJH6dPNp0%3D&st=2021-10-29T08:30:52Z&se=2121-10-05T08:35:52Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-PR-KarinaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'b9a3e70b-049d-45aa-99ec-f171badf48ad',
    locale: 'cs-CZ',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (cs-CZ, VlastaNeural)',
    properties: {
      DisplayName: 'Vlasta',
      DisplayVoiceName: 'VlastaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Vlasta',
      LocaleDescription: 'Czech',
      OrderInVoiceListApi: '51',
      PreviewSentence:
        'Vytvořte aplikace a služby pro přirozenou komunikaci s uživateli a usnadněte tak přístup a využití.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'cs-CZ-VlastaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'cs-CZ',
      localeZH: '捷克语(捷克)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fd18561b-f6a0-4c30-8b16-772e627ba39c.wav?sv=2019-07-07&sr=b&sig=azeqJr5618z5XGt1yii3bEqEiWUaNlLmJgAYtAPJgV0%3D&st=2021-10-29T08:26:05Z&se=2121-10-05T08:31:05Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'cs-CZ-VlastaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '76bc3321-2dfe-4d5f-bb6d-f1dcbafba5a2',
    locale: 'es-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-US, AlonsoNeural)',
    properties: {
      DisplayName: 'Alonso',
      DisplayVoiceName: 'AlonsoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Alonso',
      LocaleDescription: 'es-US',
      OrderInVoiceListApi: '217',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-US-AlonsoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-US',
      localeZH: '西班牙语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c30f471a-646e-498b-8b8d-031c7d24049e.wav?sv=2019-07-07&sr=b&sig=nBaBQK8BblJF1sk4GecPaGyfSJF%2B%2BtRJIaoCgVB1Rz4%3D&st=2021-10-29T08:26:47Z&se=2121-10-05T08:31:47Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-US-AlonsoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '989cb719-479f-4bd6-9490-f3634d3f4b15',
    locale: 'ca-ES',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ca-ES, EnricNeural)',
    properties: {
      DisplayName: 'Enric',
      DisplayVoiceName: 'EnricNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Enric',
      LocaleDescription: 'Catalan',
      OrderInVoiceListApi: '49',
      PreviewSentence:
        "Crea aplicacions i serveis que parlen de forma natural als usuaris, i que milloren l'accessibilitat i la facilitat d'ús.",
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ca-ES-EnricNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ca-ES',
      localeZH: '加泰罗尼亚语(西班牙)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/800b678b-966d-4bc1-b452-81bb65bc9ccc.wav?sv=2019-07-07&sr=b&sig=bLTMxKJRYWayhkHuGXZ1vdeREHOvDKUT4bm2avyc5Q0%3D&st=2021-10-29T08:26:02Z&se=2121-10-05T08:31:02Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ca-ES-EnricNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '2e7fd13b-b651-4055-8cc4-f382246dcee5',
    locale: 'ko-KR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ko-KR, SeoHyeonNeural)',
    properties: {
      DisplayName: 'SeoHyeon',
      DisplayVoiceName: 'SeoHyeonNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: '서현',
      LocaleDescription: 'Korean',
      OrderInVoiceListApi: '312',
      PreviewSentence:
        '사용자들에게 자연스럽게 이야기하는 앱과 서비스를 만들며 접근성과 사용성을 개선시킵니다.',
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ko-KR-SeoHyeonNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ko-KR',
      localeZH: '韩语(韩国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/414461a0-6e6e-4982-921f-85a3aa26ebd0.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A41Z&se=2122-09-27T07%3A52%3A41Z&sr=b&sp=rl&sig=Ok4H3Y1tSz5PFGYELJXM2Vpg6P3%2F517yfATsAFGeOU0%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ko-KR-SeoHyeonNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'f7b51518-94d6-471c-bed6-f384525f6877',
    locale: 'es-MX',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-MX, DaliaNeural)',
    properties: {
      DisplayName: 'Dalia',
      DisplayVoiceName: 'DaliaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Dalia',
      LocaleDescription: 'Spanish (Mexico)',
      OrderInVoiceListApi: '194',
      PreviewSentence:
        'Crea apps y servicios que hablen de forma natural con los usuarios, mejorando la accesibilidad y la usabilidad.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-MX-DaliaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-MX',
      localeZH: '西班牙语(墨西哥)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/a8b8e9d9-35ca-48b6-b12c-1db37bbfcb4e.wav?sv=2019-07-07&sr=b&sig=e9ZHgQC7w50zXVdiW%2F%2BJRriLc2ri8E0xMKPHMMvXanw%3D&st=2021-10-29T08:26:45Z&se=2121-10-05T08:31:45Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-MX-DaliaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '93317eac-e4c4-4178-98f5-f3a0b4e1a7fe',
    locale: 'ar-TN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-TN, ReemNeural)',
    properties: {
      DisplayName: 'Reem',
      DisplayVoiceName: 'ReemNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'ريم',
      LocaleDescription: 'ar-TN',
      OrderInVoiceListApi: '34',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-TN-ReemNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-TN',
      localeZH: '阿拉伯语(突尼斯)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/8f059465-316f-4a39-aa47-8c4a925f657f.wav?sv=2019-07-07&sr=b&sig=tpI1NctTEY4vohJ9%2BPVmf4N2Nvs7zKZPoCyNRljXfW4%3D&st=2021-10-29T08:30:18Z&se=2121-10-05T08:35:18Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-TN-ReemNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: 'a4e51bc1-71c2-4645-a2ba-f4b78720e0a7',
    locale: 'ar-EG',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-EG, SalmaNeural)',
    properties: {
      DisplayName: 'Salma',
      DisplayVoiceName: 'SalmaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'سلمى',
      LocaleDescription: 'Arabic (Egypt)',
      OrderInVoiceListApi: '11',
      PreviewSentence:
        'إن التطبيقات التي تتحاور مع المستخدمين بصوره طبيعية،  تعمل على  تحسين امكانية الوصول اليها وسهولة الاستخدام',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-EG-SalmaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-EG',
      localeZH: '阿拉伯语(埃及)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/130644d5-1cea-43f8-a97d-87b2002c6e31.wav?sv=2019-07-07&sr=b&sig=ATlZztJbxCCEe5VIAsOHzWcjqWbUZ%2BKp%2FBP4kYkVbiM%3D&st=2021-10-29T08:25:55Z&se=2121-10-05T08:30:55Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-EG-SalmaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'ea1f2715-ab0d-4c34-ac01-f4de0b4d9150',
    locale: 'hy-AM',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (hy-AM, HaykNeural)',
    properties: {
      DisplayName: 'Hayk',
      DisplayVoiceName: 'HaykNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Հայկ',
      LocaleDescription: 'hy-AM',
      OrderInVoiceListApi: '270',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'hy-AM-HaykNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'hy-AM',
      localeZH: '亚美尼亚语(亚美尼亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c9806de9-6af4-4d51-a92b-5d7dd284e4d9.wav?sv=2021-08-06&st=2022-10-21T07%3A47%3A16Z&se=2122-09-27T07%3A52%3A16Z&sr=b&sp=rl&sig=iSsoHBdM4Y4FhGnIOUMzTS%2FBb4Uj0qA8BK3ePHiaXtM%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'hy-AM-HaykNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '4337134d-4e59-48d1-af56-f4e96771bda1',
    locale: 'en-US',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-US, ChristopherNeural)',
    properties: {
      DisplayName: 'Christopher',
      DisplayVoiceName: 'ChristopherNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Christopher',
      LocaleDescription: 'English (US)',
      OrderInVoiceListApi: '134',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-US-ChristopherNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-US',
      localeZH: '英语(美国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/742136cb-7587-4497-a5a2-059950b0eb34.wav?sv=2019-07-07&sr=b&sig=aR7faI%2B9c2cT%2FnzSb176tTZjfqSkFvjO9n4re6yKuQ4%3D&st=2021-10-29T08:25:33Z&se=2121-10-05T08:30:33Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-US-ChristopherNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'b46c29d0-e660-4be9-8758-f54dfe71eb97',
    locale: 'ar-EG',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-EG, ShakirNeural)',
    properties: {
      DisplayName: 'Shakir',
      DisplayVoiceName: 'ShakirNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'شاكر',
      LocaleDescription: 'Arabic (Egypt)',
      OrderInVoiceListApi: '12',
      PreviewSentence:
        'إن التطبيقات التي تتحاور مع المستخدمين بصوره طبيعية،  تعمل على  تحسين امكانية الوصول اليها وسهولة الاستخدام',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-EG-ShakirNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-EG',
      localeZH: '阿拉伯语(埃及)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/46a1e2e4-073b-4b29-81e0-0b9c63653c85.wav?sv=2019-07-07&sr=b&sig=8d03IlrE0jG3tN52TCoG1%2Faxvs9FbQCNivV1DO9ATEk%3D&st=2021-10-29T08:25:56Z&se=2121-10-05T08:30:56Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-EG-ShakirNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '5fb5ba29-0956-4ab0-bfd2-f55ada33426d',
    locale: 'jv-ID',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (jv-ID, SitiNeural)',
    properties: {
      DisplayName: 'Siti',
      DisplayVoiceName: 'SitiNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Siti',
      LocaleDescription: 'jv-ID',
      OrderInVoiceListApi: '298',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'jv-ID-SitiNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'jv-ID',
      localeZH: '爪哇语(印度尼西亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fb44bdd1-f72e-44ae-a2af-efc2db688ea1.wav?sv=2019-07-07&sr=b&sig=J%2FxlBYDzjZ6wCtykZW2b5n20xvxIqP2UGEKjkdS12Po%3D&st=2021-10-29T08:31:08Z&se=2121-10-05T08:36:08Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'jv-ID-SitiNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'c403c07c-19ff-45dc-887b-f5b2639bff31',
    locale: 'en-KE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-KE, ChilembaNeural)',
    properties: {
      DisplayName: 'Chilemba',
      DisplayVoiceName: 'ChilembaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Chilemba',
      LocaleDescription: 'en-KE',
      OrderInVoiceListApi: '115',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-KE-ChilembaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-KE',
      localeZH: '英语(肯尼亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/1b227fa9-cf19-40d7-9e81-bfd1f596f5a4.wav?sv=2019-07-07&sr=b&sig=UB5G5oTMpiqvZ0II0lqno7K0VdUb7eqzNzxHHFrdxos%3D&st=2021-10-29T08:30:24Z&se=2121-10-05T08:35:24Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-KE-ChilembaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '8b49a818-048b-4251-a1c3-f5b97d978db3',
    locale: 'zh-CN-shandong',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (zh-CN-shandong, YunxiangNeural)',
    properties: {
      DisplayName: 'Yunxiang',
      DisplayVoiceName: 'YunxiangNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: '云翔',
      LocaleDescription: 'zh-CN-shandong',
      OrderInVoiceListApi: '447',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'zh-CN-shandong-YunxiangNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'zh-CN-shandong',
      localeZH: '中文(冀鲁官话，简体)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/60831540-2711-40d3-8741-83233345fe4f.wav?sv=2019-07-07&sr=b&sig=J0OYjjq8SmtLOakSHv6DytS9na1EySO8TabnxmewZT0%3D&st=2022-08-30T08:13:46Z&se=2122-08-06T08:18:46Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'zh-CN-shandong-YunxiangNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '7326354c-b68b-46e7-98e0-f671a8b8c141',
    locale: 'ar-LB',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-LB, LaylaNeural)',
    properties: {
      DisplayName: 'Layla',
      DisplayVoiceName: 'LaylaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'ليلى',
      LocaleDescription: 'ar-LB',
      OrderInVoiceListApi: '19',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-LB-LaylaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-LB',
      localeZH: '阿拉伯语(黎巴嫩)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/cd855623-8f42-4d82-af5f-a9450d3119be.wav?sv=2019-07-07&sr=b&sig=a6lax2WYFxI1NW66hDEHtOdXq8cUzAOV4jPxPSeZs1s%3D&st=2022-07-01T09:08:43Z&se=2122-06-07T09:13:43Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-LB-LaylaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '735db598-81e2-408f-b039-f7afef686748',
    locale: 'it-IT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (it-IT, ElsaNeural)',
    properties: {
      DisplayName: 'Elsa',
      DisplayVoiceName: 'ElsaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Elsa',
      LocaleDescription: 'Italian',
      OrderInVoiceListApi: '279',
      PreviewSentence:
        'Benvenuti nella piattaforma di generazione di contenuti audio di Microsoft.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'it-IT-ElsaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'it-IT',
      localeZH: '意大利语(意大利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0bd6ab7e-efbf-42e4-a5a9-c07cde253349.wav?sv=2019-07-07&sr=b&sig=EJjnjqAB91c8ZZw6%2F85BFCj0L8IJG%2BpdI0Uveat%2BXKI%3D&st=2021-10-29T08:27:18Z&se=2121-10-05T08:32:18Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'it-IT-ElsaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '88c51e8f-a6d1-4967-96c1-f7c93a9b5b0e',
    locale: 'fr-BE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-BE, CharlineNeural)',
    properties: {
      DisplayName: 'Charline',
      DisplayVoiceName: 'CharlineNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Charline',
      LocaleDescription: 'French (Belgium)',
      OrderInVoiceListApi: '234',
      PreviewSentence:
        'La Création de Contenu Audio vous permet de contrôler visuellement les attributs vocaux en temps réel.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-BE-CharlineNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'fr-BE',
      localeZH: '法语(比利时)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/b3ae52d5-8eea-4aaf-b18c-17ba4da22821.wav?sv=2019-07-07&sr=b&sig=uR1hPZvFbr%2BndQaroPgDOfAFkx3R36DTm4Z7p1ARSCI%3D&st=2021-10-29T08:26:54Z&se=2121-10-05T08:31:54Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'fr-BE-CharlineNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '68d38509-2bda-4a63-b939-f7d78d3b56a6',
    locale: 'nl-BE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (nl-BE, DenaNeural)',
    properties: {
      DisplayName: 'Dena',
      DisplayVoiceName: 'DenaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Dena',
      LocaleDescription: 'Dutch (Belgium)',
      OrderInVoiceListApi: '341',
      PreviewSentence:
        'Audio-inhoud Aanmaken laat u toe om visueel de spraakeigenschappen in te stellen in real-time.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'nl-BE-DenaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'nl-BE',
      localeZH: '荷兰语(比利时)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/ef625918-9c93-471f-ae54-e37d46047ce6.wav?sv=2019-07-07&sr=b&sig=0ZrnQ0ixnffSJ71NA9RvCwszNJ6vz0%2BvI3hc2rrcLRs%3D&st=2021-10-29T08:27:42Z&se=2121-10-05T08:32:42Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'nl-BE-DenaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '2e0c3be2-e1ba-48c5-adeb-f99460796c3c',
    locale: 'ar-SA',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-SA, ZariyahNeural)',
    properties: {
      DisplayName: 'Zariyah',
      DisplayVoiceName: 'ZariyahNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'زارية',
      LocaleDescription: 'Arabic (Saudi Arabia)',
      OrderInVoiceListApi: '30',
      PreviewSentence:
        'إن التطبيقات التي تتحاور مع المستخدمين بصوره طبيعية،  تعمل على  تحسين امكانية الوصول اليها وسهولة الاستخدام',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-SA-ZariyahNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-SA',
      localeZH: '阿拉伯语(沙特阿拉伯)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/19a68d06-fa8c-41ed-9d07-112359f30efe.wav?sv=2019-07-07&sr=b&sig=vsiuKJ897BMXTqDtJSMZ0CVtVA%2Fg0mx6RgJlvvl%2Bih4%3D&st=2021-10-29T08:25:58Z&se=2121-10-05T08:30:58Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-SA-ZariyahNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '641d91b0-5484-4a79-a926-fbd57b080d61',
    locale: 'en-AU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-AU, ElsieNeural)',
    properties: {
      DisplayName: 'Elsie',
      DisplayVoiceName: 'ElsieNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Elsie',
      LocaleDescription: 'English (Australia)',
      OrderInVoiceListApi: '81',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-AU-ElsieNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-AU',
      localeZH: '英语(澳大利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/1e6eef4d-18f2-4618-b46b-48796aa9e531.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A06Z&se=2122-09-27T07%3A51%3A06Z&sr=b&sp=rl&sig=ilF5gLsfBPIWQ%2FMKsqnbLtrHG3ev8YnIj6Y23SVSNK8%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-AU-ElsieNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '0824e990-0c58-43d2-997d-fc63fa573b04',
    locale: 'it-IT',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (it-IT, RinaldoNeural)',
    properties: {
      DisplayName: 'Rinaldo',
      DisplayVoiceName: 'RinaldoNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Rinaldo',
      LocaleDescription: 'Italian',
      OrderInVoiceListApi: '288',
      PreviewSentence:
        'Benvenuti nella piattaforma di generazione di contenuti audio di Microsoft.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'it-IT-RinaldoNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'it-IT',
      localeZH: '意大利语(意大利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/17c7cece-cbd4-4088-9f6e-60f906bda749.wav?sv=2019-07-07&sr=b&sig=87V%2B60lR1TAWLxFAQj17BoZoYGEaEbod1MYYkw9L6o8%3D&st=2022-07-01T09:09:55Z&se=2122-06-07T09:14:55Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'it-IT-RinaldoNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    description: '',
    id: '1346514b-fb3e-4b9a-a4e5-fcb93d91e4bc',
    locale: 'fr-FR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (fr-FR, DeniseNeural)',
    properties: {
      DisplayName: 'Denise',
      DisplayVoiceName: 'DeniseNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Denise',
      LocaleDescription: 'French (France)',
      OrderInVoiceListApi: '254',
      PreviewSentence:
        "Construire des applications et services qui communiquent naturellement avec l'utilisateur, en améliorant l’accessibilité et l'ergonomie",
      ReleaseScope: 'PublicPreview',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'fr-FR-DeniseNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{"cheerful":4,"sad":8,"Default":0}',
      VoiceStyleNames: 'cheerful,sad,Default',
      locale: 'fr-FR',
      localeZH: '法语(法国)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/55a6d687-0695-48d8-b5df-099f1caf6dd6.wav?sv=2019-07-07&sr=b&sig=cFTcv2ByipfxyC11BLJzgmgxyQMuXb8GOnknmkto4Ho%3D&st=2021-10-29T08:27:01Z&se=2121-10-05T08:32:01Z&sp=rl',
          styleName: 'general',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/320f3b90-e0b5-4dab-ac30-6b0df03cd32a.wav?sv=2019-07-07&sr=b&sig=t6mKtnpdcfg%2BqWHyqTJNUprEEjrb%2FVh1iMLGXb0%2Bjfo%3D&st=2022-03-23T16:49:15Z&se=2122-02-27T16:54:15Z&sp=rl',
          styleName: 'cheerful',
        },
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/337d92cc-58b8-419c-bc78-b14a904083f3.wav?sv=2019-07-07&sr=b&sig=hxnujM5q8IeDP9jjtdCITpkKXu3k4feYdyrxZA0hhKA%3D&st=2022-03-23T16:49:18Z&se=2122-02-27T16:54:18Z&sp=rl',
          styleName: 'sad',
        },
      ],
    },
    shortName: 'fr-FR-DeniseNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '7006e439-22ec-463e-b284-fceac3d0d4d1',
    locale: 'es-CL',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-CL, CatalinaNeural)',
    properties: {
      DisplayName: 'Catalina',
      DisplayVoiceName: 'CatalinaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Catalina',
      LocaleDescription: 'es-CL',
      OrderInVoiceListApi: '157',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-CL-CatalinaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-CL',
      localeZH: '西班牙语(智利)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/c4673b4d-d3f5-4200-9d9c-6b9137eb7792.wav?sv=2019-07-07&sr=b&sig=4hQm5oOiOkB2XMEVzj6Ay6KbcorRvK7Z9%2Biw5KtMnvE%3D&st=2021-10-29T08:30:30Z&se=2121-10-05T08:35:30Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-CL-CatalinaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '7053b22a-5f8a-4f1f-816f-fd6c31fd4b83',
    locale: 'es-CR',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-CR, MariaNeural)',
    properties: {
      DisplayName: 'Maria',
      DisplayVoiceName: 'MariaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'María',
      LocaleDescription: 'es-CR',
      OrderInVoiceListApi: '162',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-CR-MariaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-CR',
      localeZH: '西班牙语(哥斯达黎加)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/db1a3207-71f9-4a16-9302-ae2ab13b1c58.wav?sv=2019-07-07&sr=b&sig=fQCLkspjC03ZR9rg%2BHK38D6LJcGY%2B%2BFoie2tRIeYy5s%3D&st=2021-10-29T08:30:33Z&se=2121-10-05T08:35:33Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-CR-MariaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '637ffba4-caa2-436e-a6d1-fda7a339eafe',
    locale: 'en-IN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-IN, PrabhatNeural)',
    properties: {
      DisplayName: 'Prabhat',
      DisplayVoiceName: 'PrabhatNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Prabhat',
      LocaleDescription: 'English (India)',
      OrderInVoiceListApi: '113',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-IN-PrabhatNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-IN',
      localeZH: '英语(印度)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/85e7e275-f9ab-4fe1-bd81-5c436ce4e937.wav?sv=2019-07-07&sr=b&sig=Gs59CTmk1Qj9AJv0umHnhkEhneO8b1lWU3IgoLQH5cc%3D&st=2021-10-29T08:26:31Z&se=2121-10-05T08:31:31Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-IN-PrabhatNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'd99ca458-b92d-41a1-9aa3-fdd348ab1f9c',
    locale: 'es-EC',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-EC, LuisNeural)',
    properties: {
      DisplayName: 'Luis',
      DisplayVoiceName: 'LuisNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Luis',
      LocaleDescription: 'es-EC',
      OrderInVoiceListApi: '168',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-EC-LuisNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-EC',
      localeZH: '西班牙语(厄瓜多尔)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/4d74645d-1d7a-4e71-a491-b754a4e0352c.wav?sv=2019-07-07&sr=b&sig=LN6Q%2FWbL8WmfAPDb6iY9N7tZp7f7z7AVe1D1mRom9mc%3D&st=2021-10-29T08:30:39Z&se=2121-10-05T08:35:39Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-EC-LuisNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'ce2d063f-1738-4745-b504-fddd524130b8',
    locale: 'mk-MK',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (mk-MK, MarijaNeural)',
    properties: {
      DisplayName: 'Marija',
      DisplayVoiceName: 'MarijaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'Марија',
      LocaleDescription: 'mk-MK',
      OrderInVoiceListApi: '322',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'mk-MK-MarijaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'mk-MK',
      localeZH: '马其顿语(北马其顿)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/be815f6a-4612-40f3-98bf-dab74dbc61f4.wav?sv=2019-07-07&sr=b&sig=DueN3rq%2FjfZjivG8vKGbT69S1WoRsl5Wyv6BA1byKmQ%3D&st=2022-01-19T09:02:04Z&se=2121-12-26T09:07:04Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'mk-MK-MarijaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '8f00a701-8c0f-487b-af37-fe478a30fe64',
    locale: 'ru-RU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ru-RU, DmitryNeural)',
    properties: {
      DisplayName: 'Dmitry',
      DisplayVoiceName: 'DmitryNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Дмитрий',
      LocaleDescription: 'Russian',
      OrderInVoiceListApi: '372',
      PreviewSentence:
        'Возможность создавать приложения и сервисы, которые естественным образом общаются с пользователями, улучшая доступность и удобство использования.',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ru-RU-DmitryNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ru-RU',
      localeZH: '俄语(俄罗斯)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/7a7b59dc-b4b9-426e-a6dd-055fd8c1786f.wav?sv=2019-07-07&sr=b&sig=prSZXdIh59wguEXk18cC5gOTnhNPP4mQO7rHNzdOWkE%3D&st=2021-10-29T08:28:03Z&se=2121-10-05T08:33:03Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ru-RU-DmitryNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'd1695bf1-7246-4c1e-9597-fe675fb2dfed',
    locale: 'en-AU',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (en-AU, KenNeural)',
    properties: {
      DisplayName: 'Ken',
      DisplayVoiceName: 'KenNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Ken',
      LocaleDescription: 'English (Australia)',
      OrderInVoiceListApi: '84',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'en-AU-KenNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'en-AU',
      localeZH: '英语(澳大利亚)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/e8d90507-f549-4b9b-b581-6ef66c1e61b6.wav?sv=2021-08-06&st=2022-10-21T07%3A46%3A13Z&se=2122-09-27T07%3A51%3A13Z&sr=b&sp=rl&sig=iMcV0FBFv16zKqezPP%2F2c2x%2BU%2F654BgCFkqLoYmxYxw%3D',
          styleName: 'general',
        },
      ],
    },
    shortName: 'en-AU-KenNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '158031c3-095c-4d7e-b877-fe7490ba318a',
    locale: 'hi-IN',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (hi-IN, MadhurNeural)',
    properties: {
      DisplayName: 'Madhur',
      DisplayVoiceName: 'MadhurNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'मधुर',
      LocaleDescription: 'Hindi (India)',
      OrderInVoiceListApi: '263',
      PreviewSentence:
        'ऐसे ऐप और सेवाओं का निर्माण करें जो उपयोगकर्ताओं से आम बोलचाल की भाषा में बात करे, जिससे सुगमता और उपयोगिता बढ़े।',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'hi-IN-MadhurNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'hi-IN',
      localeZH: '印地语(印度)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/fa8ea1aa-421b-4d1b-a4b9-274880e90057.wav?sv=2019-07-07&sr=b&sig=jD8n1wI2DOHZJy0wpWNwH4hR0%2Fhsz%2BkQYcCVQg5pVbw%3D&st=2021-10-29T08:27:09Z&se=2121-10-05T08:32:09Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'hi-IN-MadhurNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: 'f79165a9-bc79-48fc-9c85-fe9ea43b1ec1',
    locale: 'ar-JO',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (ar-JO, SanaNeural)',
    properties: {
      DisplayName: 'Sana',
      DisplayVoiceName: 'SanaNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Female',
      LocalName: 'سناء',
      LocaleDescription: 'ar-JO',
      OrderInVoiceListApi: '15',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'ar-JO-SanaNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'ar-JO',
      localeZH: '阿拉伯语(约旦)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/0795df4a-9f76-460c-926a-a0a2b7153bfc.wav?sv=2019-07-07&sr=b&sig=9kXSJxY4mZQ7FWvmyedXJji%2B55WoXTPmz9IIdYdvZcI%3D&st=2021-10-29T08:30:05Z&se=2121-10-05T08:35:05Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'ar-JO-SanaNeural',
    voiceType: 'StandardVoice',
  },
  {
    categories: [],
    id: '565cd495-ea37-420e-99e3-fed6df8836bb',
    locale: 'es-PE',
    masterpieces: [],
    name: 'Microsoft Server Speech Text to Speech Voice (es-PE, AlexNeural)',
    properties: {
      DisplayName: 'Alex',
      DisplayVoiceName: 'AlexNeural',
      FrontendVoiceType: 'Neural',
      Gender: 'Male',
      LocalName: 'Alex',
      LocaleDescription: 'es-PE',
      OrderInVoiceListApi: '209',
      PreviewSentence: 'PageLoading to the Microsoft Voice Tuning. ',
      ReleaseScope: 'PublicGA',
      SampleRateHertz: '48000',
      SecondaryLocales: '',
      ShortName: 'es-PE-AlexNeural',
      VoiceModelKind: 'NeuralTts',
      VoiceRoleNameDefinitions: '{}',
      VoiceRoleNames: '',
      VoiceStyleNameDefinitions: '{}',
      VoiceStyleNames: '',
      locale: 'es-PE',
      localeZH: '西班牙语(秘鲁)',
    },
    samples: {
      languageSamples: [],
      roleSamples: [],
      styleSamples: [
        {
          audioFileEndpointWithSas:
            'https://cvoiceprodsea.blob.core.windows.net/bestor-acc02701-cb45-42dc-9d32-2843911017ca/VcgTaskFileManagement/9604d24f-337b-4f1f-8005-7c6d46b3a238.wav?sv=2019-07-07&sr=b&sig=0vA0KDgitMwhk8y13YimfJDVl%2F3lQIOK654KGBiO9F4%3D&st=2021-10-29T08:30:50Z&se=2121-10-05T08:35:50Z&sp=rl',
          styleName: 'general',
        },
      ],
    },
    shortName: 'es-PE-AlexNeural',
    voiceType: 'StandardVoice',
  },
];

const convert = (voices: any[]) => {
  return voices.map((voice) => voiceMap[voice.shortName]).filter((voice) => !!voice);
};

export const GET = async () => {
  const data = JSON.stringify({
    queryCondition: {
      items: [
        {
          name: 'VoiceTypeList',
          operatorKind: 'Contains',
          value: 'StandardVoice',
        },
      ],
    },
  });

  const config = {
    data: data,
    headers: {
      'accept': 'application/json, text/plain, */*',
      'accept-language': 'zh-CN,zh;q=0.9,en;q=0.8,en-GB;q=0.7,en-US;q=0.6',
      'authority': 'southeastasia.api.speech.microsoft.com',
      'content-type': 'application/json',
      'customvoiceconnectionid': uuidv4(),
      'origin': 'https://speech.microsoft.com',
      'sec-ch-ua': '"Microsoft Edge";v="113", "Chromium";v="113", "Not-A.Brand";v="24"',
      'sec-ch-ua-mobile': '?0',
      'sec-ch-ua-platform': '"Windows"',
      'sec-fetch-dest': 'empty',
      'sec-fetch-mode': 'cors',
      'sec-fetch-site': 'same-site',
      'speechstudio-session-id': '951910a0-f304-11ed-b81e-274ad6e5de17',
      'speechstudio-subscriptionsession-id': 'undefined',
      'user-agent':
        'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/113.0.0.0 Safari/537.36 Edg/113.0.1774.42',
      'x-ms-useragent': 'SpeechStudio/2021.05.001',
    },
    method: 'post',
    timeout: 1500,
    url: 'https://southeastasia.api.speach.microsoft.com/accfreetrial/texttospeech/acc/v3.0-beta1/vcg/voices',
  };

  try {
    const response = await axios(config);
    return NextResponse.json({ data: convert(response.data) });
  } catch {
    return NextResponse.json({ data: convert(cachedVoiceList) });
  }
};
