# Internationalization Implementation Guide

Welcome to the Internationalization Implementation Guide. This document will guide you through the internationalization mechanisms, including file structure and how to add new languages. We use `i18next` and `lobe-i18n` as our internationalization solutions, aiming to provide users with a seamless multilingual experience.

## TOC

- [Overview of Internationalization](#overview-of-internationalization)
- [File Structure](#file-structure)
- [Core Implementation Logic](#core-implementation-logic)
- [Adding New Language Support](#adding-new-language-support)
- [Resources and Further Reading](#resources-and-further-reading)

## Overview of Internationalization

Internationalization (often abbreviated as i18n) is the process that allows applications to adapt to different languages and regions. In our project, we support multiple languages and utilize the `i18next` library to enable dynamic language switching and content localization. Our goal is to provide a localized experience for users around the globe.

## File Structure

In our project, the files related to internationalization are organized as follows:

- `src/locales/default`: Contains the translation files for the default development language (Chinese), which we refer to as Chinese.
- `locales`: Contains folders for all supported languages, with each language folder containing the corresponding translation files, which are automatically generated by lobe-i18n.

In the `src/locales` directory structure, the `default` folder contains the original translation files (Chinese), while each language folder contains the corresponding JSON translation files. The files in each language folder correspond to the TypeScript files in the `default` folder, ensuring consistency in the structure of translation files across languages.

```
src/locales
├── create.ts
├── default
│   ├── chat.ts    // Contains characters related to the chat page
│   ├── role.ts    // Contains characters related to the role page
│   ├── common.ts  // Contains translations for common characters, such as confirm, delete, etc.
│   ├── error.ts   // Contains characters related to error handling
│   ├── index.ts   // Resource index
│   ├── market.ts  // Contains characters related to the discovery page
│   ├── setting.ts // Contains characters related to the settings page
│   └── welcome.ts // Contains characters related to the welcome page
└── resources.ts
```

The file structure automatically generated by lobe-i18n is as follows:

```
locales
├── ar
│   ├── chat.json
│   ├── common.json
│   ├── error.json
│   └── ... (other translation files)
├── de-DE
│   ├── chat.json
│   ├── common.json
│   ├── error.json
│   └── ... (other translation files)
├── en-US
├── ... (other language directories)
├── zh-CN
└── zh-TW
```

## Core Implementation Logic

The core implementation logic of internationalization is as follows:

- Initialize and configure using the `i18next` library.
- Automatically detect the user's language preference using `i18next-browser-languagedetector`.
- Dynamically load translation resources using `i18next-resources-to-backend`.
- Set the direction of the HTML document (LTR or RTL) based on the user's language preference.

Here is a simplified pseudocode example to illustrate the core implementation logic of internationalization:

```ts
import i18n from 'i18next';
import LanguageDetector from 'i18next-browser-languagedetector';
import resourcesToBackend from 'i18next-resources-to-backend';
import { isRtlLang } from 'rtl-detect';

// Create and configure the i18n instance
const createI18nInstance = (lang) => {
  const i18nInstance = i18n
    .use(LanguageDetector) // Use language detection
    .use(
      resourcesToBackend((language, namespace) => {
        // Dynamically load the corresponding language translation resources
        return import(`path/to/locales/${language}/${namespace}.json`);
      }),
    );

  // Listen for language change events and dynamically set document direction
  i18nInstance.on('languageChanged', (language) => {
    const direction = isRtlLang(language) ? 'rtl' : 'ltr';
    document.documentElement.dir = direction; // Set HTML document direction
  });

  // Initialize the i18n instance
  i18nInstance.init({
    // Relevant configuration
  });

  return i18nInstance;
};
```

In this example, we demonstrate how to use `i18next` and related plugins to initialize internationalization settings. We dynamically import translation resources and respond to language change events to adjust the text direction of the page. This process provides flexible multilingual support capabilities.

## Adding New Language Support

To add support for a new language, please refer to the detailed steps in the [New Language Addition Guide](Add-New-Locale.en-US.md).

## Resources and Further Reading

- [i18next Official Documentation](https://www.i18next.com/)
- [lobe-i18n Tool Documentation](https://github.com/lobehub/lobe-cli-toolbox/tree/master/packages/lobe-i18n)

By following this guide, you can better understand and participate in the internationalization efforts, providing a seamless multilingual experience for users worldwide.
