<?php

namespace Ignite\Application\Concerns;

use Ignite\Support\Facades\Route;

trait ManagesAssets
{
    /**
     * Included css files..
     *
     * @var array
     */
    protected $styles = [];

    protected $urls = [];

    /**
     * Included scripts.
     *
     * @var array
     */
    protected $scripts = [];

    /**
     * Included login scripts.
     *
     * @var array
     */
    protected $loginScripts = [];

    /**
     * Add script to the application.
     *
     * @param  string  $src
     * @return $this
     */
    public function script($src)
    {
        if (in_array($src, $this->scripts)) {
            return;
        }

        $this->scripts[] = $this->resolveAssetPath($src);

        return $this;
    }

    /**
     * Add script to the application.
     *
     * @param  string  $src
     * @return $this
     */
    public function loginScript($src)
    {
        if (in_array($src, $this->loginScripts)) {
            return;
        }

        $this->loginScripts[] = $this->resolveAssetPath($src);

        return $this;
    }

    /**
     * Add css file to the application.
     *
     * @param  string  $path
     * @return $this
     */
    public function style($path)
    {
        if (in_array($path, $this->styles)) {
            return;
        }

        $this->styles[] = $this->resolveAssetPath($path);

        return $this;
    }

    /**
     * Get styles.
     *
     * @return array
     */
    public function getStyles()
    {
        return $this->styles;
    }

    /**
     * Get scripts.
     *
     * @return array
     */
    public function getScripts()
    {
        return $this->scripts;
    }

    /**
     * Get login scripts.
     *
     * @return array
     */
    public function getLoginScripts()
    {
        return $this->loginScripts;
    }

    /**
     * Resolve path to asset.
     *
     * @param  string  $path
     * @return void
     */
    protected function resolveAssetPath($path)
    {
        if (! file_exists($path)) {
            return $path;
        }

        $info = pathinfo($path);

        $uri = $this->generateUriFromPath($path);

        $route = Route::public()->get($uri, function () use ($path, $info) {
            return response(app('files')->get($path), 200)
                ->header('Content-Type', [
                    'js'  => 'application/javascript; charset=utf-8',
                    'css' => 'text/css',
                ][$info['extension'] ?? 'plain'] ?? 'text/'.$info['extension'] ?? 'plain');
        });

        return url($route->uri);
    }

    /**
     * Generate unique uri for the given path.
     *
     * @param  string  $path
     * @return string
     */
    protected function generateUriFromPath($path)
    {
        $info = pathinfo($path);

        $uri = implode('/', [
            $info['extension'] ?? 'dist',
            $info['basename'],
        ]);

        $i = 1;
        while (in_array($uri, $this->urls)) {
            $uri = implode('/', [
                $info['extension'] ?? 'dist',
                $i++.'-'.$info['basename'],
            ]);
        }

        return $this->urls[] = $uri;
    }
}
