package it.unive.lisa.program.cfg.statement;

import it.unive.lisa.program.cfg.CFG;
import it.unive.lisa.program.cfg.CodeLocation;
import it.unive.lisa.program.cfg.statement.call.Call;
import it.unive.lisa.program.cfg.statement.call.UnresolvedCall;
import it.unive.lisa.symbolic.value.Identifier;
import it.unive.lisa.type.Type;
import it.unive.lisa.type.Untyped;
import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 * An expression that is part of a statement of the program.
 * 
 * @author <a href="mailto:luca.negrini@unive.it">Luca Negrini</a>
 */
public abstract class Expression extends Statement {

	private static final Logger LOG = LogManager.getLogger(Expression.class);

	/**
	 * The static type of this expression.
	 */
	private final Type staticType;

	/**
	 * The statement (or expression) that contains this expression.
	 */
	private Statement parent;

	/**
	 * The collection of meta variables that are generated by the evaluation of
	 * this expression. These should be removed as soon as the values computed
	 * by those gets out of scope (e.g., popped from the stack).
	 */
	private final Collection<Identifier> metaVariables;

	/**
	 * Builds an untyped expression happening at the given source location, that
	 * is its type is {@link Untyped#INSTANCE}.
	 * 
	 * @param cfg      the cfg that this expression belongs to
	 * @param location the location where the expression is defined within the
	 *                     program
	 */
	protected Expression(
			CFG cfg,
			CodeLocation location) {
		this(cfg, location, Untyped.INSTANCE);
	}

	/**
	 * Builds a typed expression happening at the given source location.
	 * 
	 * @param cfg        the cfg that this expression belongs to
	 * @param location   the location where this expression is defined within
	 *                       the program
	 * @param staticType the static type of this expression
	 */
	protected Expression(
			CFG cfg,
			CodeLocation location,
			Type staticType) {
		super(cfg, location);
		Objects.requireNonNull(staticType, "The expression type of a CFG cannot be null");
		this.staticType = staticType;
		this.metaVariables = new HashSet<>();
	}

	/**
	 * Yields the static type of this expression.
	 * 
	 * @return the static type of this expression
	 */
	public final Type getStaticType() {
		return staticType;
	}

	/**
	 * Yields the meta variables that are generated by the evaluation of this
	 * expression. These should be removed as soon as the values computed by
	 * those gets out of scope (e.g., popped from the stack). The returned
	 * collection will be filled while evaluating this expression semantics,
	 * thus invoking this method before computing the semantics will yield an
	 * empty collection. Variables added here should represent stack values that
	 * cannot be re-computed at a later time (e.g., call return values).
	 * 
	 * @return the meta variables
	 */
	public Collection<Identifier> getMetaVariables() {
		return metaVariables;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		result = prime * result + ((staticType == null) ? 0 : staticType.hashCode());
		// we ignore parent: just syntactic info...
		// we ignore meta variables and runtime types: those might change during
		// execution...
		return result;
	}

	@Override
	public boolean equals(
			Object obj) {
		if (this == obj)
			return true;
		if (!super.equals(obj))
			return false;
		if (getClass() != obj.getClass())
			return false;
		Expression other = (Expression) obj;
		if (staticType == null) {
			if (other.staticType != null)
				return false;
		} else if (!staticType.equals(other.staticType))
			return false;
		// we ignore parent: just syntactic info...
		// we ignore meta variables and runtime types: those might change during
		// execution...
		return true;
	}

	/**
	 * Sets the {@link Statement} that contains this expression.
	 * 
	 * @param st the containing statement
	 */
	public final void setParentStatement(
			Statement st) {
		if (this.parent == null)
			this.parent = st;
		else
			// this usually happens either with a bad code parsing process
			// or when constructing resolved calls or similar constructs
			// inside semantic functions. Either way, the syntactic structure
			// of the code should not change once set
			LOG.trace("Attempt to change the parent of " + this + " ignored");
	}

	/**
	 * Yields the {@link Statement} that contains this expression, if any. If
	 * this method returns {@code null}, than this expression is used as a
	 * command: it is the root statement of a node in the cfg, and its returned
	 * value is discarded.
	 * 
	 * @return the statement that contains this expression, if any
	 */
	public final Statement getParentStatement() {
		if (this instanceof Call) {
			Call original = (Call) this;
			while (original.getSource() != null)
				original = original.getSource();
			if (original != this)
				return original.getParentStatement();
		}

		return parent;
	}

	/**
	 * Yields the outer-most {@link Statement} containing this expression, that
	 * is used as a node in the cfg. If this expression is used a command, then
	 * this method return {@code this}. If this expression is a {@link Call}
	 * built as a resolved version of an {@link UnresolvedCall} {@code uc}, then
	 * {@code uc.getRootStatement()} is returned.
	 * 
	 * @return the outer-most statement containing this expression, or
	 *             {@code this}
	 */
	public final Statement getRootStatement() {
		if (this instanceof Call) {
			Call original = (Call) this;
			while (original.getSource() != null)
				original = original.getSource();
			if (original != this)
				return original.getRootStatement();
		}

		if (parent == null)
			return this;

		if (!(parent instanceof Expression))
			return parent;

		return ((Expression) parent).getRootStatement();
	}

	@Override
	public Statement getStatementEvaluatedBefore(
			Statement other) {
		if (this instanceof Call) {
			Call original = (Call) this;
			while (original.getSource() != null)
				original = original.getSource();
			if (original != this)
				return original.getStatementEvaluatedBefore(other);
		}

		if (other != this || parent == null)
			return null;

		return parent.getStatementEvaluatedBefore(this);
	}

	@Override
	public Statement getStatementEvaluatedAfter(
			Statement other) {
		if (this instanceof Call) {
			Call original = (Call) this;
			while (original.getSource() != null)
				original = original.getSource();
			if (original != this)
				return original.getStatementEvaluatedAfter(other);
		}

		if (other != this || parent == null)
			return null;

		return parent.getStatementEvaluatedAfter(this);
	}
}
