package it.unive.lisa.analysis.dataflow;

import it.unive.lisa.analysis.ScopedObject;
import it.unive.lisa.analysis.SemanticEvaluator;
import it.unive.lisa.analysis.SemanticException;
import it.unive.lisa.analysis.SemanticOracle;
import it.unive.lisa.program.cfg.ProgramPoint;
import it.unive.lisa.symbolic.SymbolicExpression;
import it.unive.lisa.symbolic.value.Identifier;
import it.unive.lisa.symbolic.value.PushInv;
import it.unive.lisa.symbolic.value.ValueExpression;
import it.unive.lisa.type.Type;
import it.unive.lisa.util.representation.StructuredObject;
import java.util.Collection;
import java.util.Set;

/**
 * An element of the dataflow domain, that contains a collection of
 * {@link Identifier}s in its definition. A domain element implements standard
 * dataflow gen (
 * {@link #gen(Identifier, ValueExpression, ProgramPoint, DataflowDomain)},
 * {@link #gen(ValueExpression, ProgramPoint, DataflowDomain)}) and kill
 * ({@link #kill(Identifier, ValueExpression, ProgramPoint, DataflowDomain)},
 * {@link #kill(ValueExpression, ProgramPoint, DataflowDomain)}) operations.
 * 
 * @author <a href="mailto:luca.negrini@unive.it">Luca Negrini</a>
 * 
 * @param <D> the concrete type of {@link DataflowDomain} that contains
 *                instances of this element
 * @param <E> the concrete type of {@link DataflowElement}
 */
public interface DataflowElement<D extends DataflowDomain<D, E>, E extends DataflowElement<D, E>>
		extends
		SemanticEvaluator,
		StructuredObject,
		ScopedObject<E> {

	/**
	 * Yields all the {@link Identifier}s that are involved in the definition of
	 * this element.
	 * 
	 * @return the identifiers
	 */
	Collection<Identifier> getInvolvedIdentifiers();

	/**
	 * The dataflow <i>gen</i> operation, yielding the dataflow elements that
	 * are generated by the assignment of the given {@code expression} to the
	 * given {@code id}.
	 * 
	 * @param id         the {@link Identifier} being assigned
	 * @param expression the expressions that is being assigned to {@code id}
	 * @param pp         the program point where this operation happens
	 * @param domain     the {@link DataflowDomain} that is being used to track
	 *                       instances of this element
	 * 
	 * @return the collection of dataflow elements that are generated by the
	 *             assignment
	 * 
	 * @throws SemanticException if an error occurs during the computation
	 */
	Collection<E> gen(
			Identifier id,
			ValueExpression expression,
			ProgramPoint pp,
			D domain)
			throws SemanticException;

	/**
	 * The dataflow <i>gen</i> operation, yielding the dataflow elements that
	 * are generated by evaluating the given non-assigning {@code expression}.
	 * 
	 * @param expression the expressions that is being evaluated
	 * @param pp         the program point where this operation happens
	 * @param domain     the {@link DataflowDomain} that is being used to track
	 *                       instances of this element
	 * 
	 * @return the collection of dataflow elements that are generated by the
	 *             expression
	 * 
	 * @throws SemanticException if an error occurs during the computation
	 */
	Collection<E> gen(
			ValueExpression expression,
			ProgramPoint pp,
			D domain)
			throws SemanticException;

	/**
	 * The dataflow <i>kill</i> operation, yielding the dataflow elements that
	 * are killed by the assignment of the given {@code expression} to the given
	 * {@code id}.
	 * 
	 * @param id         the {@link Identifier} being assigned
	 * @param expression the expressions that is being assigned to {@code id}
	 * @param pp         the program point where this operation happens
	 * @param domain     the {@link DataflowDomain} that is being used to track
	 *                       instances of this element
	 * 
	 * @return the collection of dataflow elements that are killed by the
	 *             assignment
	 * 
	 * @throws SemanticException if an error occurs during the computation
	 */
	Collection<E> kill(
			Identifier id,
			ValueExpression expression,
			ProgramPoint pp,
			D domain)
			throws SemanticException;

	/**
	 * The dataflow <i>kill</i> operation, yielding the dataflow elements that
	 * are killed by evaluating the given non-assigning {@code expression}.
	 * 
	 * @param expression the expressions that is being evaluated
	 * @param pp         the program point where this operation happens
	 * @param domain     the {@link DataflowDomain} that is being used to track
	 *                       instances of this element
	 * 
	 * @return the collection of dataflow elements that are killed by the
	 *             expression
	 * 
	 * @throws SemanticException if an error occurs during the computation
	 */
	Collection<E> kill(
			ValueExpression expression,
			ProgramPoint pp,
			D domain)
			throws SemanticException;

	@Override
	default boolean canProcess(
			SymbolicExpression expression,
			ProgramPoint pp,
			SemanticOracle oracle) {
		if (expression instanceof PushInv)
			// the type approximation of a pushinv is bottom, so the below check
			// will always fail regardless of the kind of value we are tracking
			return expression.getStaticType().isValueType();

		Set<Type> rts = null;
		try {
			rts = oracle.getRuntimeTypesOf(expression, pp, oracle);
		} catch (SemanticException e) {
			return false;
		}

		if (rts == null || rts.isEmpty())
			// if we have no runtime types, either the type domain has no type
			// information for the given expression (thus it can be anything,
			// also something that we can track) or the computation returned
			// bottom (and the whole state is likely going to go to bottom
			// anyway).
			return true;

		return rts.stream().anyMatch(Type::isValueType);
	}
}
