package it.unive.lisa;

import it.unive.lisa.checks.semantic.SemanticCheck;
import it.unive.lisa.checks.syntactic.SyntacticCheck;
import it.unive.lisa.checks.warnings.Warning;
import it.unive.lisa.program.Application;
import it.unive.lisa.program.Global;
import it.unive.lisa.program.Program;
import it.unive.lisa.program.Unit;
import it.unive.lisa.program.cfg.CFG;
import it.unive.lisa.program.cfg.CodeMember;
import it.unive.lisa.program.cfg.edge.Edge;
import it.unive.lisa.program.cfg.statement.Expression;
import it.unive.lisa.program.cfg.statement.Statement;
import it.unive.lisa.util.datastructures.graph.GraphVisitor;
import it.unive.lisa.util.file.FileManager;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import org.joda.time.DateTime;
import org.joda.time.Period;
import org.joda.time.format.PeriodFormatter;
import org.joda.time.format.PeriodFormatterBuilder;

/**
 * A collector of the information about a competed analysis.
 * 
 * @author <a href="mailto:luca.negrini@unive.it">Luca Negrini</a>
 */
public class LiSARunInfo {

	private static class StatementCounter
			implements
			GraphVisitor<CFG, Statement, Edge, Void> {

		private int statements = 0;

		private int expressions = 0;

		@Override
		public boolean visit(
				Void tool,
				CFG graph,
				Statement node) {
			if (node instanceof Expression) {
				if (((Expression) node).getRootStatement() == node)
					statements++;
				else
					expressions++;
			} else
				statements++;
			return true;
		}
	}

	/**
	 * A {@link PeriodFormatter}, producing timestamps as (where italics words
	 * represent portions of the timestamp): <i>year</i>Y <i>month</i>M
	 * <i>day</i>D <i>hours</i>h <i>minutes</i>m <i>seconds</i>s
	 * <i>milliseconds</i>ms. Note that zeros are omitted.
	 */
	public static final PeriodFormatter PERIOD_FORMAT = new PeriodFormatterBuilder()
			.appendYears().appendSuffix("Y ")
			.appendMonths().appendSuffix("M ")
			.appendDays().appendSuffix("D ")
			.appendHours().appendSuffix("h ")
			.appendMinutes().appendSuffix("m ")
			.appendSeconds().appendSuffix("s ")
			.appendMillis().appendSuffix("ms")
			.printZeroNever()
			.toFormatter();

	/**
	 * The version of LiSA used to run the analysis.
	 */
	public final String version;

	/**
	 * The number of {@link Warning}s that were generated by the
	 * {@link SyntacticCheck}s and {@link SemanticCheck}s executed during the
	 * analysis.
	 */
	public final int warnings;

	/**
	 * The number of files that were created by the {@link FileManager} of the
	 * analysis.
	 */
	public final int files;

	/**
	 * The number of {@link Unit}s that were submitted to the analysis.
	 */
	public final int units;

	/**
	 * The number of {@link Global}s that were submitted to the analysis.
	 */
	public final int globals;

	/**
	 * The number of {@link CodeMember}s that were submitted to the analysis.
	 */
	public final int members;

	/**
	 * The number of {@link CFG}s that were submitted to the analysis.
	 */
	public final int cfgs;

	/**
	 * The total number of top-level {@link Statement}s analyzed.
	 */
	public final int statements;

	/**
	 * The total number of inner (i.e., non top-level) {@link Expression}s
	 * analyzed.
	 */
	public final int expressions;

	/**
	 * The number of {@link Program}s analyzed.
	 */
	public final int programs;

	/**
	 * The timestamp of the analysis start, formatted through
	 * {@link #PERIOD_FORMAT}.
	 */
	public final String start;

	/**
	 * The timestamp of the analysis conclusion, formatted through
	 * {@link #PERIOD_FORMAT}.
	 */
	public final String end;

	/**
	 * The analysis duration, formatted through {@link #PERIOD_FORMAT}.
	 */
	public final String duration;

	/**
	 * Builds the run info.
	 * 
	 * @param warnings the warnings generated by the analysis
	 * @param files    the files generated by the analysis
	 * @param app      the {@link Application} under analysis
	 * @param start    the start time
	 * @param end      the end time
	 */
	public LiSARunInfo(
			Collection<Warning> warnings,
			Collection<String> files,
			Application app,
			DateTime start,
			DateTime end) {
		this.version = VersionInfo.VERSION;
		this.warnings = warnings.size();
		this.files = files.size();
		this.programs = app.getPrograms().length;
		int units = 0, globals = 0;
		for (Program p : app.getPrograms()) {
			units += p.getUnits().size();
			globals += p.getGlobalsRecursively().size();
		}
		this.units = units;
		this.globals = globals;
		this.members = app.getAllCodeCodeMembers().size();
		this.cfgs = app.getAllCFGs().size();
		this.start = start.toString();
		this.end = end.toString();
		this.duration = PERIOD_FORMAT.print(new Period(start, end)).trim();

		StatementCounter counter = new StatementCounter();
		app.getAllCFGs().forEach(c -> c.accept(counter, null));
		this.statements = counter.statements;
		this.expressions = counter.expressions;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		try {
			for (Field field : LiSARunInfo.class.getFields())
				if (!Modifier.isStatic(field.getModifiers()))
					result = prime * result + Objects.hashCode(field.get(this));
		} catch (IllegalArgumentException | IllegalAccessException e) {
			throw new IllegalStateException("Cannot access one of this class' public fields", e);
		}
		return result;
	}

	@Override
	public boolean equals(
			Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		LiSARunInfo other = (LiSARunInfo) obj;
		try {
			for (Field field : LiSARunInfo.class.getFields())
				if (!Modifier.isStatic(field.getModifiers())) {
					Object value = field.get(this);
					Object ovalue = field.get(other);
					if (!Objects.equals(value, ovalue))
						return false;
				}
		} catch (IllegalArgumentException | IllegalAccessException e) {
			throw new IllegalStateException("Cannot access one of this class' public fields", e);
		}
		return true;
	}

	/**
	 * Checks whether the given run information match this one in terms of
	 * analyzed code and analysis results. This corresponds to calling
	 * {@link #equals(Object)}, but ignoring {@link #version},
	 * {@link #duration}, {@link #end}, and {@link #start}.
	 * 
	 * @param other the other run info
	 * 
	 * @return whether or not the two info contain the same code metrics and
	 *             results
	 */
	public boolean sameCodeAndResults(
			LiSARunInfo other) {
		if (this == other)
			return true;
		if (other == null)
			return false;
		if (cfgs != other.cfgs)
			return false;
		if (expressions != other.expressions)
			return false;
		if (globals != other.globals)
			return false;
		if (members != other.members)
			return false;
		if (statements != other.statements)
			return false;
		if (units != other.units)
			return false;
		if (warnings != other.warnings)
			return false;
		if (files != other.files)
			return false;
		if (programs != other.programs)
			return false;
		return true;
	}

	@Override
	public String toString() {
		return "Version " + version +
				"\nDuration: " + duration + " (started: " + start + ", ended: " + end + ")" +
				"\nPrograms: " + programs +
				"\nUnits: " + units +
				"\nGlobals: " + globals +
				"\nCode Members: " + members +
				"\nCFGs: " + cfgs +
				"\nStatements: " + statements +
				"\nExpressions: " + expressions +
				"\nGenerated Warnings: " + warnings +
				"\nGenerated Files: " + files;
	}

	/**
	 * Converts this configuration to a property bag, that is, a map from keys
	 * (fields of this class) to values (their values).
	 * 
	 * @return the property bag
	 */
	public Map<String, String> toPropertyBag() {
		Map<String, String> bag = new TreeMap<>();
		try {
			for (Field field : LiSARunInfo.class.getFields())
				if (!Modifier.isStatic(field.getModifiers())) {
					Object value = field.get(this);
					String key = field.getName();
					String val = String.valueOf(value);
					bag.put(key, val);
				}
		} catch (IllegalArgumentException | IllegalAccessException e) {
			throw new IllegalStateException("Cannot access one of this class' public fields", e);
		}
		return bag;
	}
}
