package it.unive.lisa;

import it.unive.lisa.checks.warnings.Warning;
import it.unive.lisa.conf.LiSAConfiguration;
import it.unive.lisa.util.file.FileManager;
import java.util.Collection;

/**
 * A structured report containing the information about an analysis that has
 * successfully completed.
 * 
 * @author <a href="mailto:luca.negrini@unive.it">Luca Negrini</a>
 */
public class LiSAReport {

	private final LiSARunInfo info;

	private final LiSAConfiguration configuration;

	private final Collection<Warning> warnings;

	private final Collection<String> createdFiles;

	/**
	 * Builds the report.
	 * 
	 * @param configuration the configuration used for the analysis
	 * @param info          the information of the analysis run
	 * @param warnings      the warnings generated by the analysis
	 * @param createdFiles  the files generated by the analysis
	 */
	public LiSAReport(
			LiSAConfiguration configuration,
			LiSARunInfo info,
			Collection<Warning> warnings,
			Collection<String> createdFiles) {
		this.info = info;
		this.configuration = configuration;
		this.warnings = warnings;
		this.createdFiles = createdFiles;
	}

	/**
	 * The {@link LiSARunInfo} containing information about the analysis.
	 * 
	 * @return the information
	 */
	public LiSARunInfo getInfo() {
		return info;
	}

	/**
	 * Yields the {@link LiSAConfiguration} used to run the analysis.
	 * 
	 * @return the configuration
	 */
	public LiSAConfiguration getConfiguration() {
		return configuration;
	}

	/**
	 * Yields the {@link Warning}s that have been generated during the analysis.
	 * 
	 * @return the generated warnings
	 */
	public Collection<Warning> getWarnings() {
		return warnings;
	}

	/**
	 * Yields the names of the files created during the analysis, as reported by
	 * {@link FileManager#createdFiles()}.
	 * 
	 * @return the names of the created files
	 */
	public Collection<String> getCreatedFiles() {
		return createdFiles;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((configuration == null) ? 0 : configuration.hashCode());
		result = prime * result + ((createdFiles == null) ? 0 : createdFiles.hashCode());
		result = prime * result + ((info == null) ? 0 : info.hashCode());
		result = prime * result + ((warnings == null) ? 0 : warnings.hashCode());
		return result;
	}

	@Override
	public boolean equals(
			Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		LiSAReport other = (LiSAReport) obj;
		if (configuration == null) {
			if (other.configuration != null)
				return false;
		} else if (!configuration.equals(other.configuration))
			return false;
		if (createdFiles == null) {
			if (other.createdFiles != null)
				return false;
		} else if (!createdFiles.equals(other.createdFiles))
			return false;
		if (info == null) {
			if (other.info != null)
				return false;
		} else if (!info.equals(other.info))
			return false;
		if (warnings == null) {
			if (other.warnings != null)
				return false;
		} else if (!warnings.equals(other.warnings))
			return false;
		return true;
	}

	@Override
	public String toString() {
		return "LiSAReport [info=" + info + ", configuration=" + configuration + ", warnings=" + warnings
				+ ", createdFiles=" + createdFiles + "]";
	}
}
